/**
 * FGraphInfo.java
 *
 * Created on Jul 3, 2006
 * 
 * Copyright (c) 2006 Eureka Team
 */
package graphe;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;

import maths.CoordSyst;

/**
 * Cette classe implemente <CODE>GraphInfo</CODE>.
 * Represente le syst. de coordonees avec precision 1 en dessinant aussi le cercle unit.
 *
 * @author Nicolae
 */
public class FGraphInfo implements GraphInfo {

	private static final int ARROW_LEN = 6;
	private static final int ARROW_SPACE = 10;
	private static final int TEXT_H = 9;
	private static final int TEXT_W = 6;
	
	private CoordSyst pr;
	private Color col, col1, col2;
	private double bigStep = 1;
	private int bigSmallRatio = 4;
	private boolean circle = true, axes = true, big = true, small = true, arrows = true, text = true;
	private boolean under = false;
	
	/**
	 * Constructeur standard.
	 */
	public FGraphInfo() {
		this(0x60000000, 0x60FF0000);		
	}

	/**
	 * Constructeur qui initialise la couleur.
	 *
	 * @param c la couleur des axes. 
	 * @param c1 la couleur des paralelles & cercle unite.
	 */
	public FGraphInfo(int c, int c1) {
		col = new Color((c >> 16) & 0xFF, (c >> 8) & 0xFF, c & 0xFF, (c >> 24) & 0xFF);
		col1 = new Color((c1 >> 16) & 0xFF, (c1 >> 8) & 0xFF, c1 & 0xFF, (c1 >> 24) & 0xFF);
		col2 = new Color(col1.getRed(), col1.getGreen(), col1.getBlue(), 64);
	}

	/**
	 * Constructeur qui initialise la couleur.
	 *
	 * @param c la couleur.
	 * @param c1 la couleur des paralelles & cercle unite.
	 */
	public FGraphInfo(Color c, Color c1) {
		col = new Color(c.getRed(), c.getGreen(), c.getBlue(), 0x70);
		col1 = new Color(c1.getRed(), c1.getGreen(), c1.getBlue(), 0x70);
		col2 = new Color(col1.getRed(), col1.getGreen(), col1.getBlue(), 0x38);
	}
	
	public void paint(Graphics g) {
		if(pr == null)
			return;			

		int x0 = pr.getX(0);
		int y0 = pr.getY(0);
		int wd = pr.getWidth();
		int ht = pr.getHeigth();
		
		double left = pr.getRe(0);
		double right = pr.getRe(wd);
		double top = pr.getIm(0);
		double bottom = pr.getIm(ht);

		g.setColor(col);
		
		if(axes) {
			vLine(g, pr.getX(0));
			hLine(g, pr.getY(0));
		}
		
		if(text) {
			g.setFont(g.getFont().deriveFont(Font.BOLD));
			
			int x = wd - ARROW_SPACE - ARROW_LEN;
			int y = y0 + ARROW_LEN + TEXT_H;
			g.drawString("x", x, y);
			
			x = x0 - ARROW_LEN - TEXT_W;
			y = ARROW_SPACE + ARROW_LEN;
			g.drawString("y", x, y);
			
			for (int i = (int) (left / bigStep - 1); i < right / bigStep + 1; i++) 
				if(i != 0) 
					g.drawString("" + i * bigStep, 
							pr.getX(i * bigStep) + ARROW_LEN / 2, y0 - ARROW_LEN / 2);

			for (int i = (int) (bottom / bigStep - 1); i < top / bigStep + 1; i++) 
				if(i != 0) 
					g.drawString("" + i * bigStep, x0 + ARROW_LEN / 2,
							pr.getY(i * bigStep) - ARROW_LEN / 2);

			g.drawString("0", x0 + ARROW_LEN / 2, y0 - ARROW_LEN / 2);
		}
		
		if(arrows) {
			g.drawLine(x0 - ARROW_LEN / 2, ARROW_SPACE + ARROW_LEN, x0, ARROW_SPACE);
			g.drawLine(x0 + ARROW_LEN / 2, ARROW_SPACE + ARROW_LEN, x0, ARROW_SPACE);
			g.drawLine(wd - ARROW_SPACE - ARROW_LEN, y0 - ARROW_LEN / 2, wd - ARROW_SPACE, y0);
			g.drawLine(wd - ARROW_SPACE - ARROW_LEN, y0 + ARROW_LEN / 2, wd - ARROW_SPACE, y0);
		}
		
		if(big) {
			g.setColor(col1);

			for (int i = (int) (left / bigStep - 1); i < right / bigStep + 1; i++) 
				if(i != 0) 
					vLine(g, pr.getX(i * bigStep));

			for (int i = (int) (bottom / bigStep - 1); i < top / bigStep + 1; i++) 
				if(i != 0) 
					hLine(g, pr.getY(i * bigStep));
		}
		
		g.setColor(col2);

		double ss = bigStep / bigSmallRatio;
		if(small) {
			for (int i = (int) (left / ss - 1); i < right / ss + 1; i++) 
				if(i % bigSmallRatio != 0) 
					vLine(g, pr.getX(i * ss));

			for (int i = (int) (bottom / ss - 1); i < top / ss + 1; i++) 
				if(i % bigSmallRatio != 0) 
					hLine(g, pr.getY(i * ss));
		}
		
		if(circle)
			g.drawOval(pr.getX(-1), pr.getY(1), 
					   pr.getX(1) - pr.getX(-1), pr.getY(-1) - pr.getY(1));		
	}

	private void hLine(Graphics g, int y) {
		if(y >= 0 && y < pr.getHeigth())
			g.drawLine(0, y, pr.getWidth(), y);
	}

	private void vLine(Graphics g, int x) {
		if(x >= 0 && x < pr.getWidth())
			g.drawLine(x, 0, x, pr.getHeigth());
	}

	public void setPoint(double re, double im) {
		// pas de point
	}

	public void setCoords(CoordSyst princ, CoordSyst sec) {
		pr = princ;
	}

	/**
	 * Renvoie la valeur de la propriete <code>arrows</code>.
	 * <code>true</code> pour dessiner les fleches.
	 *
	 * @return la valeur de la propriete <code>arrows</code>.
	 */
	public boolean isArrows() {
		return arrows;
	}

	/**
	 * Etablie la valeur de la propriete <code>arrows</code>.
	 * <code>true</code> pour dessiner les fleches.
	 *
	 * @param arrows la nouvelle valeur de la propriete <code>arrows</code>.
	 */
	public void setArrows(boolean arrows) {
		this.arrows = arrows;
	}

	/**
	 * Renvoie la valeur de la propriete <code>big</code>.
	 * <code>true</code> pour afficher la grille pricipale.
	 *
	 * @return la valeur de la propriete <code>big</code>.
	 */
	public boolean isBig() {
		return big;
	}

	/**
	 * Etablie la valeur de la propriete <code>big</code>.
	 * <code>true</code> pour afficher la grille pricipale.
	 *
	 * @param big la nouvelle valeur de la propriete <code>big</code>.
	 */
	public void setBig(boolean big) {
		this.big = big;
	}

	/**
	 * Renvoie la valeur de la propriete <code>bigSmallRatio</code>.
	 * Le rapport des distances entre axes pricipales / secondaires.
	 *
	 * @return la valeur de la propriete <code>bigSmallRatio</code>.
	 */
	public int getBigSmallRatio() {
		return bigSmallRatio;
	}

	/**
	 * Etablie la valeur de la propriete <code>bigSmallRatio</code>.
	 * Le rapport des distances entre axes pricipales / secondaires.
	 *
	 * @param bigSmallRatio la nouvelle valeur de la propriete <code>bigSmallRatio</code>.
	 */
	public void setBigSmallRatio(int bigSmallRatio) {
		this.bigSmallRatio = bigSmallRatio;
	}

	/**
	 * Renvoie la valeur de la propriete <code>bigStep</code>.
	 * La distance entre axes principales.
	 *
	 * @return la valeur de la propriete <code>bigStep</code>.
	 */
	public double getBigStep() {
		return bigStep;
	}

	/**
	 * Etablie la valeur de la propriete <code>bigStep</code>.
	 * La distance entre axes principales.
	 *
	 * @param bigStep la nouvelle valeur de la propriete <code>bigStep</code>.
	 */
	public void setBigStep(double bigStep) {
		this.bigStep = bigStep;
	}

	/**
	 * Renvoie la valeur de la propriete <code>circle</code>.
	 * <code>true</code> pour representer le cercle unite.
	 *
	 * @return la valeur de la propriete <code>circle</code>.
	 */
	public boolean isCircle() {
		return circle;
	}

	/**
	 * Etablie la valeur de la propriete <code>circle</code>.
	 * <code>true</code> pour representer le cercle unite.
	 *
	 * @param circle la nouvelle valeur de la propriete <code>circle</code>.
	 */
	public void setCircle(boolean circle) {
		this.circle = circle;
	}

	/**
	 * Renvoie la valeur de la propriete <code>small</code>.
	 * <code>true</code> pour afficher les axes secondaires.
	 *
	 * @return la valeur de la propriete <code>small</code>.
	 */
	public boolean isSmall() {
		return small;
	}

	/**
	 * Etablie la valeur de la propriete <code>small</code>.
	 * <code>true</code> pour afficher les axes secondaires.
	 *
	 * @param small la nouvelle valeur de la propriete <code>small</code>.
	 */
	public void setSmall(boolean small) {
		this.small = small;
	}

	/**
	 * Renvoie la valeur de la propriete <code>text</code>.
	 * <code>true</code> pour ecrire les valeurs sur les axes.
	 *
	 * @return la valeur de la propriete <code>text</code>.
	 */
	public boolean isText() {
		return text;
	}

	/**
	 * Etablie la valeur de la propriete <code>text</code>.
	 * <code>true</code> pour ecrire les valeurs sur les axes.
	 *
	 * @param text la nouvelle valeur de la propriete <code>text</code>.
	 */
	public void setText(boolean text) {
		this.text = text;
	}

	/**
	 * Renvoie la valeur de la propriete <code>axes</code>.
	 * <code>true</code> pour representer les axes.
	 *
	 * @return la valeur de la propriete <code>axes</code>.
	 */
	public boolean isAxes() {
		return axes;
	}

	/**
	 * Etablie la valeur de la propriete <code>axes</code>.
	 * <code>true</code> pour representer les axes.
	 *
	 * @param axes la nouvelle valeur de la propriete <code>axes</code>.
	 */
	public void setAxes(boolean axes) {
		this.axes = axes;
	}
	
	public boolean isUnder() {
		return under;
	}

	public void setUnder(boolean under) {
		this.under = under;
	}
}
