﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

#nullable disable

using System;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.Test.Utilities;
using Microsoft.VisualStudio.IntegrationTest.Utilities;
using Microsoft.VisualStudio.IntegrationTest.Utilities.Common;
using Roslyn.Test.Utilities;
using Xunit;
using Xunit.Abstractions;
using ProjectUtils = Microsoft.VisualStudio.IntegrationTest.Utilities.Common.ProjectUtils;

namespace Roslyn.VisualStudio.IntegrationTests.CSharp
{
    [Collection(nameof(SharedIntegrationHostFixture))]
    public class CSharpGoToDefinition : AbstractEditorTest
    {
        protected override string LanguageName => LanguageNames.CSharp;

        public CSharpGoToDefinition(VisualStudioInstanceFactory instanceFactory)
            : base(instanceFactory, nameof(CSharpGoToDefinition))
        {
        }

        [WpfFact, Trait(Traits.Feature, Traits.Features.GoToDefinition), Trait(Traits.Editor, Traits.Editors.LanguageServerProtocol)]
        public void GoToClassDeclaration()
        {
            var project = new ProjectUtils.Project(ProjectName);
            VisualStudio.SolutionExplorer.AddFile(project, "FileDef.cs");
            VisualStudio.SolutionExplorer.OpenFile(project, "FileDef.cs");
            VisualStudio.Editor.SetText(
@"class SomeClass
{
}");
            VisualStudio.SolutionExplorer.AddFile(project, "FileConsumer.cs");
            VisualStudio.SolutionExplorer.OpenFile(project, "FileConsumer.cs");
            VisualStudio.Editor.SetText(
@"class SomeOtherClass
{
    SomeClass sc;
}");
            VisualStudio.Editor.PlaceCaret("SomeClass");
            VisualStudio.Editor.GoToDefinition("FileDef.cs");
            VisualStudio.Editor.Verify.TextContains(@"class SomeClass$$", assertCaretPosition: true);
            Assert.False(VisualStudio.Shell.IsActiveTabProvisional());
        }

        [WpfFact, Trait(Traits.Feature, Traits.Features.GoToDefinition), Trait(Traits.Editor, Traits.Editors.LanguageServerProtocol)]
        public void GoToDefinitionOpensProvisionalTabIfDocumentNotAlreadyOpen()
        {
            var project = new ProjectUtils.Project(ProjectName);
            VisualStudio.SolutionExplorer.AddFile(project, "FileDef.cs");
            VisualStudio.SolutionExplorer.OpenFile(project, "FileDef.cs");
            VisualStudio.Editor.SetText(
@"class SomeClass
{
}
");
            VisualStudio.SolutionExplorer.CloseCodeFile(project, "FileDef.cs", saveFile: true);
            VisualStudio.SolutionExplorer.AddFile(project, "FileConsumer.cs");
            VisualStudio.SolutionExplorer.OpenFile(project, "FileConsumer.cs");
            VisualStudio.Editor.SetText(
@"class SomeOtherClass
{
    SomeClass sc;
}");
            VisualStudio.Editor.PlaceCaret("SomeClass");
            VisualStudio.Editor.GoToDefinition("FileDef.cs");
            VisualStudio.Editor.Verify.TextContains(@"class SomeClass$$", assertCaretPosition: true);
            Assert.True(VisualStudio.Shell.IsActiveTabProvisional());
        }

        [WpfFact, Trait(Traits.Feature, Traits.Features.GoToDefinition), Trait(Traits.Editor, Traits.Editors.LanguageServerProtocol)]
        public virtual void GoToDefinitionWithMultipleResults()
        {
            SetUpEditor(
@"partial class /*Marker*/ $$PartialClass { }

partial class PartialClass { int i = 0; }");

            var declarationWindowName = VisualStudio.IsUsingLspEditor ? "'PartialClass' references" : "'PartialClass' declarations";

            VisualStudio.Editor.GoToDefinition(declarationWindowName);

            var results = VisualStudio.FindReferencesWindow.GetContents(declarationWindowName);

            var activeWindowCaption = VisualStudio.Shell.GetActiveWindowCaption();
            Assert.Equal(expected: declarationWindowName, actual: activeWindowCaption);

            Assert.Collection(
                results,
                new Action<Reference>[]
                {
                    reference =>
                    {
                        Assert.Equal(expected: "partial class /*Marker*/ PartialClass { }", actual: reference.Code);
                        Assert.Equal(expected: 0, actual: reference.Line);
                        Assert.Equal(expected: 25, actual: reference.Column);
                    },
                    reference =>
                    {
                        Assert.Equal(expected: "partial class PartialClass { int i = 0; }", actual: reference.Code);
                        Assert.Equal(expected: 2, actual: reference.Line);
                        Assert.Equal(expected: 14, actual: reference.Column);
                    }
                });
        }
    }
}
