/*
 *  @(#)PathImpl.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.mbtf.v1.engine;


import net.sourceforge.groboutils.mbtf.v1.IPath;
import net.sourceforge.groboutils.mbtf.v1.IState;
import net.sourceforge.groboutils.mbtf.v1.ITransition;
import net.sourceforge.groboutils.mbtf.v1.IPathIterator;

import org.apache.log4j.Logger;


/**
 * An immutable implementation of IPath
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:26 $
 * @since      June 12, 2002
 */
public class PathImpl implements IPath
{
    private static final Logger LOG = Logger.getLogger( PathImpl.class );
    
    private ITransition[] trans;
    private IState start;
    
    
    public PathImpl( IState startState, ITransition[] trans )
    {
        if (startState == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        
        LOG.debug("Path starting at "+startState);
        if (trans == null)
        {
            LOG.debug("-- Path has no transitions");
            this.trans = new ITransition[0];
        }
        else
        {
            int len = trans.length;
            ITransition[] t = new ITransition[ len ];
            for (int i = 0; i < len; ++i)
            {
                if (trans[i] == null)
                {
                    throw new IllegalArgumentException(
                        "no nulls allowed in ITransition array");
                }
                // else
                LOG.debug("-- transition "+i+"="+trans[i]);
                t[i] = trans[i];
            }
            this.trans = t;
        }
        
        this.start = startState;
    }
    
    
    /**
     * Generate an iterator for this path.  The iterator will only return
     * transition elements.
     *
     * @return a new iterator for the path.
     */
    public IPathIterator iterator()
    {
        return new PathIteratorImpl( this.trans );
    }
    
    
    /**
     * Retrieve the starting state for this path.
     *
     * @return the start state for the path, which can never be <tt>null</tt>.
     */
    public IState getStartState()
    {
        return this.start;
    }
    
    
    /**
     * Returns the number of transitions in the path.  This is named "size" to
     * correspond to the <tt>java.util</tt>
     * container classes terminology.
     *
     * @return the path transition count.
     */
    public int size()
    {
        return this.trans.length;
    }
    
    
    /**
     * Returns the number of states visited in the path, which should
     * always equal <tt>size() + 2</tt>, due to the start and final states,
     * unless there are no transitions, in which case the depth is 1
     * (there must always be a start state).
     *
     * @return the depth of the path.
     */
    public int getDepth()
    {
        int size = size();
        int depth;
        if (size == 0)
        {
            depth = 1;
        }
        else
        {
            depth = size + 2;
        }
        return depth;
    }
}

