#include "osl/piece.h"
#include "osl/move.h"
#include "osl/state/simpleState.h"
#include "osl/squareCompressor.h"
#include "move-phash.c"
#include <boost/static_assert.hpp>
#include <iostream>

namespace osl
{
  BOOST_STATIC_ASSERT(sizeof(Move) == 4);
} //namespace osl

bool osl::Move::isValid() const
{
  if (! isNormal())
    return false;
  const Square from = this->from();
  if (! from.isValid())
    return false;
  const Square to = this->to();
  if (! to.isOnBoard())
    return false;
  return osl::isValid(ptype())
    && osl::isValid(capturePtype())
    && capturePtype()!=KING
    && osl::isValid(player());
}

const osl::Move osl::Move::rotate180() const
{
  if (isPass())
    return Move::PASS(alt(player()));
  if (! isNormal())
    return *this;
  return Move(from().rotate180Safe(), to().rotate180(), ptype(),
	      capturePtype(), isPromotion(), alt(player()));
}

std::ostream& osl::operator<<(std::ostream& os,const Move move)
{
  if (move == Move::DeclareWin())
    return os << "MOVE_DECLARE_WIN";
  if (move.isInvalid())
    return os << "MOVE_INVALID";
  if (move.isPass())
    return os << "MOVE_PASS";
  const Player turn = move.player();
  if (move.isValid())
  {
    if (move.from().isPieceStand()) 
    {
      os << "Drop(" << turn << "," << move.ptype() << "," << move.to() << ")";
    }
    else
    {
      const Ptype capture_ptype=move.capturePtype();
      os << "Move(" << turn << "," << move.ptype() << "," 
	 << move.from() << "->" << move.to() ;
      if (move.promoteMask())
	os << ",promote";
      if (capture_ptype != PTYPE_EMPTY)
	os << ",capture=" << capture_ptype;
      os << ")";
    }
  }
  else
  {
    os << "InvalidMove " << move.from() << " " << move.to() 
       << " " << move.ptypeO() << " " << move.oldPtypeO()
       << " " << move.promoteMask()
       << " " << move.capturePtype() << "\n";
  }
  return os;
}

unsigned int osl::Move::hash() const
{
  assert(capturePtype() == PTYPE_EMPTY);
  return move_phash(intValue());
}

const osl::Move osl::Move::
fromMove16(Move16 move16, const SimpleState& state)
{
  if (move16==MOVE16_NONE)
    return Move();
  Player turn=state.turn();
  Square to=SquareCompressor::melt((move16>>8)&0x7f);
  if((move16&0x80)!=0){
    Ptype ptype=(Ptype)(move16-0x80);
    return Move(to,ptype,turn);
  }
  Square from=SquareCompressor::melt(move16&0x7f);
  Ptype ptype=state[from].ptype();
  Ptype capture_ptype=state[to].ptype();
  bool is_promote=(move16&0x8000)!=0;
  if(is_promote)
    return Move(from,to,::osl::promote(ptype),capture_ptype,true,turn);
  else
    return Move(from,to,ptype,capture_ptype,false,turn);
}
osl::Move16 osl::Move::toMove16() const
{
  if (isInvalid())
    return MOVE16_NONE;
  if (isDrop())
    return Move16(0x80+(uint16_t)ptype()+((SquareCompressor::compress(to()))<<8));
  if (isPromotion())
    return Move16(SquareCompressor::compress(from())+(SquareCompressor::compress(to())<<8)+0x8000);
  return Move16(SquareCompressor::compress(from())+(SquareCompressor::compress(to())<<8));
}


// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
