#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <getopt.h>

#include "dialog.h"

#include "tui.h"
#include "util.h"
#include <libintl.h>

#define N_(String) gettext(String)
#define PACKAGE "deepin-fix-init"
#define LOCALEDIR "/usr/share/locale/"
#define BLKID_BIN   "blkid"
#define DF_BIN      "df"
#define FIX_BIN_EXT4   "fsck.ext4"
#define FIX_BIN       "fsck"

#define EXIT_FILE     "/tmp/deepin-fix-init.txt"

#define FIX_ROOT_TITLE    N_("hint")
#define FIX_ROOT_CONTENT  N_("Repairing the file system may result in data loss, are you sure to perform file repair?")
#define FIX_ROOT_IN       N_("The file system is being repaired, the repair process may take a few minutes, please wait......")
#define FIX_ROOT_SUC      N_("The file system was repaired successfully!")
#define FIX_ROOT_SUC_OK   N_("continue to boot the system")
#define FIX_ROOT_SUC_CANCEL  N_("reboot")

typedef enum TUI_EXIT_STATUS {
    TUI_NOT_EXIT = 0,
    TUI_EXIT_OK,
    TUI_EXIT_REBOOT,
    TUI_EXIT_SHUTDOWN
}TUI_EXIT_STATUS;

char g_fix_bin[256] = {0};
tui_cfg g_cfg = {0};
TUI_EXIT_STATUS g_exit_status = TUI_NOT_EXIT;

menu_args main_fix_menu;
menu_args main_live_menu;
menu_args main_fix_failed_menu;
menu_args *main_menu = NULL;
menu_args *bak_main_menu = NULL;

int do_fix_root(void *data);
int do_enter_cmd_mode(void *data);
int do_reboot(void *data);
int do_shutdown(void *data);
int do_enter_live(void *data);

menu_item_t tui_all_menu[] = {
    {0},
    {MENU_ITEM_FIX_ROOT, 0,do_fix_root},
    {MENU_ITEM_ENTER_CMD_MODE, 0,do_enter_cmd_mode},
    {MENU_ITEM_REBOOT, 0,do_reboot},
    {MENU_ITEM_SHUTDOWN, 0,do_shutdown},
    {MENU_ITEM_ENTER_LIVE, 0,do_enter_live}
};


void init_tui() {
    init_tui_dialog();
    const char *MENU_ITEM_FIX_ROOT_LABEL = N_("Automatically repair file system");
    const char *MENU_ITEM_ENTER_CMD_MODE_LABEL  = N_("Enter command line management mode");
    const char *MENU_ITEM_REBOOT_LABEL = N_("Reboot");
    const char *MENU_ITEM_SHUTDOWN_LABEL = N_("Poweroff");
    const char *MENU_ITEM_ENTER_LIVE_LABEL = N_("Enter the Live system");

    tui_all_menu[MENU_ITEM_FIX_ROOT].item_str = strdup(MENU_ITEM_FIX_ROOT_LABEL);
    tui_all_menu[MENU_ITEM_ENTER_CMD_MODE].item_str = strdup(MENU_ITEM_ENTER_CMD_MODE_LABEL);
    tui_all_menu[MENU_ITEM_REBOOT].item_str = strdup(MENU_ITEM_REBOOT_LABEL);
    tui_all_menu[MENU_ITEM_SHUTDOWN].item_str = strdup(MENU_ITEM_SHUTDOWN_LABEL);
    tui_all_menu[MENU_ITEM_ENTER_LIVE].item_str = strdup(MENU_ITEM_ENTER_LIVE_LABEL);

    /*init fix menu*/
    memset(&main_fix_menu,0,sizeof(main_fix_menu));    
    sprintf(main_fix_menu.title,"%s",N_("system repair"));
    main_fix_menu.height = 20;
    main_fix_menu.width = 80;
    main_fix_menu.menu_height = 4;
    sprintf(main_fix_menu.content,"%s",N_("The file system is abnormal, causing the system to fail to start, please choose:"));
    main_fix_menu.menus[0] = MENU_ITEM_FIX_ROOT;
    main_fix_menu.menus[1] = MENU_ITEM_ENTER_CMD_MODE;
    main_fix_menu.menus[2] = MENU_ITEM_REBOOT;
    main_fix_menu.menus[3] = MENU_ITEM_SHUTDOWN;

    /*init fix failed menu*/    
    memset(&main_fix_failed_menu,0,sizeof(main_fix_failed_menu));    
    sprintf(main_fix_failed_menu.title,"%s","system repair");
    main_fix_failed_menu.height = 20;
    main_fix_failed_menu.width = 80;
    main_fix_failed_menu.menu_height = 4;
    sprintf(main_fix_failed_menu.content,"%s",N_("Failed to repair system files, failure reason:"));    
    main_fix_failed_menu.menus[0] = MENU_ITEM_ENTER_CMD_MODE;
    main_fix_failed_menu.menus[1] = MENU_ITEM_REBOOT;
    main_fix_failed_menu.menus[2] = MENU_ITEM_SHUTDOWN;

    /*init live menu*/    
    memset(&main_live_menu,0,sizeof(main_live_menu));    
    sprintf(main_live_menu.title,"%s",N_("system repair"));
    main_live_menu.height = 20;
    main_live_menu.width = 80;
    main_live_menu.menu_height = 4;
    sprintf(main_live_menu.content,"%s",N_("The file system is abnormal, causing the system to fail to start, please choose:"));    
    /* 1060需求，暂时不支持
    main_live_menu.menus[0] = MENU_ITEM_ENTER_LIVE;*/
    main_live_menu.menus[0] = MENU_ITEM_ENTER_CMD_MODE;
    main_live_menu.menus[1] = MENU_ITEM_REBOOT;
    main_live_menu.menus[2] = MENU_ITEM_SHUTDOWN;
}

int do_fix_root(void *data) {
    int ret,ret1=1,ret2;
    FILE *fptmp;
    char strtmp[MAX_CMD_LEN]={0},resbuf[MAX_LINE_LEN];

    ret = create_yesno(FIX_ROOT_TITLE,FIX_ROOT_CONTENT,10,80,0,0);
    TUI_LOG(("fix root user choose ret:%d \n",ret));
    if (!ret) {
        /*ok,fix root*/
        create_infobox(FIX_ROOT_TITLE,FIX_ROOT_IN,10,80);
        //sleep(2);
        snprintf(strtmp,MAX_CMD_LEN,"%s -y %s >/dev/null 2>&1",g_fix_bin,g_cfg.root_dev);
        ret1 = excute_shell_command(strtmp,resbuf,MAX_LINE_LEN);
        TUI_LOG(("fix root cmd:%s ret:%d errno %d res:%s\n",strtmp,ret1,errno,resbuf));
        end_infobox();
        if (ret1==0 || ret1==1) {
            /*fix success*/
            ret2 = create_yesno(FIX_ROOT_TITLE,FIX_ROOT_SUC,10,80,FIX_ROOT_SUC_OK,FIX_ROOT_SUC_CANCEL);
            if (!ret2) {
                /*user choose yes,continue boot*/
                /*Now just touch exit file*/
                fptmp = fopen(EXIT_FILE,"w+");
                fclose(fptmp);
                g_exit_status = TUI_EXIT_OK;
            }
            else {
                /*user choose no,reboot*/
                g_exit_status = TUI_EXIT_REBOOT;
            }
        }
        else {
            /*fix failed*/
            bak_main_menu = main_menu;
            main_menu = &main_fix_failed_menu;            
        }
    }
    else {
        /*cancel do nothing*/
    }
    return OK;
}

int do_enter_cmd_mode(void *data) {
    g_exit_status = TUI_EXIT_OK;
    return 0;
}

int do_reboot(void *data) {
    g_exit_status = TUI_EXIT_REBOOT;
    return 0;
}

int do_shutdown(void *data) {
    g_exit_status = TUI_EXIT_SHUTDOWN;
    return 0;
}

int do_enter_live(void *data) {
    return 0;
}

int main_loop() {
    char cmd[MAX_CMD_LEN]={0};
    char resbuf[MAX_LINE_LEN]={0};
    int ret=0;
    if (!g_cfg.root_dev)
        return TERROR;
    
    snprintf(cmd,MAX_CMD_LEN,"%s |grep %s",DF_BIN,g_cfg.root_dev);
    ret = excute_shell_command(cmd,resbuf,MAX_LINE_LEN);
    TUI_LOG(("ret:%d res:%s\n",ret,resbuf));
    if (ret)
        main_menu = &main_fix_menu;
    else
        main_menu = &main_live_menu;

    while (g_exit_status == TUI_NOT_EXIT) {        
        ret = create_menu(main_menu);
        TUI_LOG(("main menu:%d root=%s\n",ret,g_cfg.root_dev));
        if (ret>0 && ret<MENU_ITEM_MAX) {
            tui_all_menu[ret].callback(NULL);
        }
        else if (ret == DLG_EXIT_ESC && bak_main_menu) {
            main_menu = bak_main_menu;
            bak_main_menu = NULL;
        }
    }
    return 0;
}

int check_cfg() {
    char tmp[MAX_FILE_NAME_LEN],tmp1[MAX_FILE_NAME_LEN];

    if (!g_cfg.root_dev || !g_cfg.root_type) {
        TUI_LOG(("root cfg is null\n"));
        return TERROR;
    }
    TUI_LOG(("root dev is %s root type is %s\n",g_cfg.root_dev,g_cfg.root_type));
    snprintf(g_fix_bin,256,"%s.%s",FIX_BIN,g_cfg.root_type);
    snprintf(tmp,MAX_FILE_NAME_LEN,"/bin/%s",g_fix_bin);
    snprintf(tmp1,MAX_FILE_NAME_LEN,"/sbin/%s",g_fix_bin);

    if (access(tmp,F_OK)!=0 && access(tmp1,F_OK)!=0) {
        TUI_LOG(("cann't find %s\n",g_fix_bin));
        snprintf(g_fix_bin,256,"%s",FIX_BIN);
    }   
    
    return TOK;
}

void showUsage(const char *cmd) {
    printf(N_("Usage: %s [options]\n"),cmd);
    printf(N_("options:\n"));
    printf(N_("\t-h --help:\tno arg,show help\n"));
    printf(N_("\t-v --version:\tno arg,show version\n"));
    printf(N_("\t-r --root:\trequire one arg,input root dev, example:-r /dev/sda1\n"));
    printf(N_("\t-t --type:\trequire one arg,input root type, example:-t ext4\n"));
    printf("\n\n");
    printf(N_("If a command-line option is used, use the value in the command-line option;\n \
    otherwise read it from the config file %s\n"),CONFIG_FILE);
}

int main(int argc,char *argv[]) {
    int c;

    setlocale (LC_ALL, "");
    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);

    static const struct option longopts[] = {
		{ "root",	      required_argument, NULL, 'r' },
        { "type",	      required_argument, NULL, 't' },
		{ "version",	  no_argument,	     NULL, 'v' },
		{ "help",	      no_argument,       NULL, 'h' },
        { "debug",	      no_argument,       NULL, 'd' },
		{ NULL, 0, NULL, 0 }
	};
    init_log("/tmp/deepin-fix-init.log");    

    while ((c = getopt_long (argc, argv,
			    "r:t:vhd", longopts, NULL)) != -1) {
        switch (c) {
		case 'r':
			g_cfg.root_dev = strdup(optarg);
			break;
        case 't':
            g_cfg.root_type = strdup(optarg);
            break;
        case 'v':
            printf("%s version 1.0\n",argv[0]);
            return TOK;
        case 'h':
            showUsage(argv[0]);
            return TOK;
        case 'd':
            g_cfg.b_debug = true;
        default:
            break;
        }
    }
    if (!g_cfg.root_dev||!g_cfg.root_type)
        read_config();
    
    init_tui();
    
    if (check_cfg() == TOK)
        main_loop();
    else
        showUsage(argv[0]);

    if (g_cfg.root_dev)
        free(g_cfg.root_dev);
    if (g_cfg.root_type)
        free(g_cfg.root_type);
    
    for (c=0;c<MENU_ITEM_MAX;c++) {
        if (tui_all_menu[c].item_str)
            free(tui_all_menu[c].item_str);
    }
    
    close_log();

    switch (g_exit_status) {
        case TUI_EXIT_OK:
            return TUI_EXIT_OK;
        case TUI_EXIT_REBOOT:
	    system("reboot -f");
            break;
        case TUI_EXIT_SHUTDOWN:
	    system("poweroff -f");
            break;
        default:
            break;
    }

    return 0;
}
