package main

import (
	"pkg.deepin.io/daemon/sync/infrastructure/log"
	"pkg.deepin.io/daemon/sync/infrastructure/storage"

	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
)

func putMachineData(data []byte, mid string, token storage.TokenSource) error {
	if len(data) == 0 {
		return fmt.Errorf("invalid hardware data")
	}

	req, err := http.NewRequest(http.MethodPut,
		storage.SyncServerAPI()+"/machines/"+mid,
		bytes.NewBuffer(data))
	if err != nil {
		return err
	}

	req.Header.Set("Access-Token", token.Token())

	var cli = new(http.Client)
	resp, err := cli.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode == http.StatusOK {
		return nil
	}

	data, _ = ioutil.ReadAll(resp.Body)
	return fmt.Errorf("Put machine failure: %s", string(data))
}

func getMachineData(mid string, token storage.TokenSource) ([]byte, error) {
	req, err := http.NewRequest(http.MethodGet,
		storage.SyncServerAPI()+"/machines/"+mid, nil)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Access-Token", token.Token())

	var cli = new(http.Client)
	resp, err := cli.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()

	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf(string(data))
	}
	return data, nil
}

func getMachineLastSyncTime(mid string, token storage.TokenSource) (int64, error) {
	req, err := http.NewRequest(http.MethodGet,
		storage.SyncServerAPI()+"/machine/lastSyncTime", nil)
	if err != nil {
		return 0, err
	}
	req.Header.Set("Access-Token", token.Token())
	req.Header.Set("X-Machine-ID", mid)
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return 0, err
	}
	defer resp.Body.Close()

	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return 0, err
	}
	var info = struct {
		Timestamp int64 `json:"timestamp"`
	}{}
	err = json.Unmarshal(data, &info)
	if err != nil {
		return 0, err
	}
	return info.Timestamp, nil
}

func validToken(mid, token string) (bool, error) {
	req, err := http.NewRequest(http.MethodGet,
		storage.SyncServerAPI()+"/sessions/current/status", nil)
	if err != nil {
		return false, err
	}

	req.Header.Set("X-Machine-ID", mid)
	req.Header.Set("Access-Token", token)

	var cli = new(http.Client)
	resp, err := cli.Do(req)
	if err != nil {
		return false, err
	}
	defer resp.Body.Close()

	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return false, err
	}

	log.Debug("[Valid Token] result:", string(data))
	if resp.StatusCode == http.StatusOK {
		var ret = struct {
			IsValid bool `json:"isValid"`
		}{}
		err = json.Unmarshal(data, &ret)
		if err != nil {
			return false, err
		}
		return ret.IsValid, nil
	}

	var ret = struct {
		Error struct {
			Code    int    `json:"code"`
			Message string `json:"msg"`
		} `json:"error"`
	}{}
	err = json.Unmarshal(data, &ret)
	if err != nil {
		return false, err
	}
	return false, fmt.Errorf(ret.Error.Message)
}
