package hardware

import (
	"bufio"
	"os"
	"strings"
)

const (
	fileOS = "/etc/deepin-version"

	kfSectionRelease = "Release"
	kfKeyVersion     = "Version"
	kfKeyType        = "Type"
)

type iniValue map[string]string
type iniInfo map[string]iniValue

func getOSFromDeepin(filename string) (string, error) {
	// TODO: fallback to /etc/lsb-release or /etc/os-release
	kf, err := loadIniFile(filename)
	if err != nil {
		return "", err
	}

	ver := kf.GetString(kfSectionRelease, kfKeyVersion)
	ty := kf.GetString(kfSectionRelease, kfKeyType)
	kf = nil
	var ret = "Deepin"
	if len(ty) != 0 && ty != "Desktop" {
		ret += " " + ty
	}
	ret += " " + ver
	return ret, nil
}

func (kf iniInfo) GetString(group, key string) string {
	values, ok := kf[group]
	if !ok {
		return ""
	}
	if len(values) == 0 {
		return ""
	}
	return values[key]
}

func loadIniFile(filename string) (iniInfo, error) {
	fr, err := os.Open(filename)
	if err != nil {
		return nil, err
	}
	defer fr.Close()

	var kf = make(iniInfo)
	var scanner = bufio.NewScanner(fr)
	var group string
	var values = make(iniValue)
	for scanner.Scan() {
		line := scanner.Text()
		if len(line) == 0 {
			continue
		}
		line = strings.TrimSpace(line)
		if line[0] == '#' {
			continue
		}
		if line[0] == '[' {
			// group
			if len(group) != 0 {
				kf[group] = values
				values = make(iniValue)
			}
			line = strings.TrimLeft(line, "[")
			line = strings.TrimRight(line, "]\n")
			group = line
			continue
		}
		if len(group) == 0 {
			continue
		}
		items := strings.SplitN(line, "=", 2)
		if len(items) != 2 {
			continue
		}
		// TODO(jouyouyun): ignore translation
		values[items[0]] = items[1]
	}
	return kf, nil
}
