/* Ergo, version 3.5, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2016 Elias Rudberg, Emanuel H. Rubensson, Pawel Salek,
 * and Anastasia Kruchinina.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Kohn−Sham Density Functional Theory Electronic Structure Calculations 
 * with Linearly Scaling Computational Time and Memory Usage,
 * Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek,
 * J. Chem. Theory Comput. 7, 340 (2011),
 * <http://dx.doi.org/10.1021/ct100611z>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */

/*-*-mode: C; c-indentation-style: "bsd"; c-basic-offset: 4; -*-*/
/** @file fun-pw91c.c  PW91C implementation.

   Automatically generated code implementing pw91c functional and its
   derivatives. Generated by func-codegen.pl being a part of a 
   "Automatic code generation framework for analytical functional
    derivative evaluation", Pawel Salek, 2004
    
    This functional is connected by making following changes:
    1. add "extern Functional pw91cFunctional;" to 'functionals.h'
    2. add "&pw91cFunctional," to 'functionals.c'
    3. add "fun-pw91c}.c" to 'Makefile.am', 'Makefile.in' or 'Makefile'.

    This functional has been generated from following input:
    ------ cut here -------
 PW92C parameters 
c:1.709921;
T:[0.031091,0.015545,0.016887];
U:[0.21370,0.20548,0.11125];
V:[7.5957,14.1189,10.357];
W:[3.5876,6.1977,3.6231];
X:[1.6382,3.3662,0.88026];
Y:[0.49294,0.62517,0.49671];
P:[1,1,1];

 PW92C helper functions  
r(a,b):=(3/(4*%PI*(a+b)))^(1/3);
zet(a,b):=(a-b)/(a+b);
omega(z):=((1+z)^(4/3)+(1-z)^(4/3)-2)/(2^(4/3)-2);
en(r,t,u,v,w,x,y,p):=-2*t*(1+u*r)*log(1+1/(2*t*(v*sqrt(r)+w*r+x*r^(3/2)+y*r^(p+1))));

eps(a,b):=en(r(a,b),T[1],U[1],V[1],W[1],X[1],Y[1],P[1]);

 pw91c parameters  
iota:0.09;
nu:16/%PI*(3*%PI^2)^(1/3);
kappa:0.004235;
lamda:nu*kappa;
Z:-0.001667;
Xi:23.266;
Phi:0.007389;
Lamda:8.723;
Upsilon:0.472;

 pw91c functions (closed shell!)  
theta(r):=10^(-3)*(2.568+Xi*r+Phi*r^2)/(1+Lamda*r+Upsilon*r^2+10*Phi*r^3);
phi(r):=theta(r)-Z;
sigma(ga,gb,gab):=ga*ga+gb*gb+2*gab;
d(a,b,ga,gb,gab):=sqrt(sigma(ga,gb,gab))/4/(3/%PI)^(1/6)/(a+b)^(7/6);
A(a,b):=2*iota/lamda*(1/(exp(-2*iota*eps(a,b)/lamda^2)-1));
L(d,a,b):=lamda^2/(2*iota)*log((1+2*(iota*(d^2+A(a,b)*d^4))/(lamda*(1+A(a,b)*d^2+(A(a,b))^2*d^4))));
J(d,a,b):=nu*(phi(r(a,b))-kappa-3*Z/7)*d^2*exp(-100*4/(3*%PI^5*(a+b))^(1/3)*d^2);
H(d,a,b):=L(d,a,b)+J(d,a,b);


 kernel  
K(rhoa,grada,rhob,gradb,gradab):=(rhoa+rhob)*(eps(rhoa,rhob)+H(d(rhoa,rhob,grada,gradb,gradab),rhoa,rhob));

    ------ cut here -------
*/

#include <math.h>
#include <stddef.h>
 
#define __CVERSION__
 
#include "functionals.h"
 
/* INTERFACE PART */
static int pw91c_isgga(void) { return 1; } /* FIXME: detect! */
static int pw91c_read(const char *conf_line);
static real pw91c_energy(const FunDensProp *dp);
static void pw91c_first(FunFirstFuncDrv *ds,   real factor,
                         const FunDensProp *dp);
static void pw91c_second(FunSecondFuncDrv *ds, real factor,
                          const FunDensProp *dp);
static void pw91c_third(FunThirdFuncDrv *ds,   real factor,
                         const FunDensProp *dp);
 
Functional Pw91cFunctional = {
  "PW91c",       /* name */
  pw91c_isgga,   /* gga-corrected */
  pw91c_read,
  NULL,
  pw91c_energy,
  pw91c_first,
  pw91c_second,
  pw91c_third
};
 
/* IMPLEMENTATION PART */
static int
pw91c_read(const char *conf_line)
{
    fun_set_hf_weight(0);
    return 1;
}

static real
pw91c_energy(const FunDensProp *dp)
{
    real res;
    real rhoa = dp->rhoa, rhob = dp->rhob;
    real grada = dp->grada, gradb = dp->gradb, gradab = dp->gradab;

    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;
    real t19, t20, t21, t22;

    t1 = rhob+rhoa;
    t2 = POW(gradb,2.0)+2.0*gradab+POW(grada,2.0);
    t3 = 1/POW(t1,2.333333333333334);
    t4 = POW(3.0,0.66666666666667);
    t5 = 1/POW(4.0,0.66666666666667);
    t6 = POW(3.141592653589793,0.66666666666667);
    t7 = 1/t6;
    t8 = 1/POW(t1,0.66666666666667);
    t9 = POW(3.0,0.33333333333333);
    t10 = 1/POW(4.0,0.33333333333333);
    t11 = POW(3.141592653589793,0.33333333333333);
    t12 = 1/t11;
    t13 = 1/POW(t1,0.33333333333333);
    t14 = 1/t4;
    t15 = POW(3.141592653589793,1.333333333333333);
    t16 = 0.2137*t9*t10*t12*t13+1.0;
    t17 = LOG(16.0818243221511/(0.49294*t4*t5*t7*t8+3.5876*
        t9*t10*t12*t13+7.5957*POW(3.0,0.16666666666667)/(POW(2.0,0.33333333333333)*
        POW(3.141592653589793,0.16666666666667)*POW(t1,0.16666666666667))+
        0.80042863499936/SQRT(t1))+1.0);
    t18 = 1/t9;
    t19 = POW(t2,2.0);
    t20 = 1/POW(t1,4.666666666666667);
    t21 = POW(2.718281828459045,2.437756914117331*t14*t6*
        t16*t17)-1.0;
    t22 = 1/t21;

   /* code */
    res = t1*(t2*t3*(0.001*(23.266*t9*t10*t12*t13+0.007389*
        t4*t5*t7*t8+2.568)/(0.472*t4*t5*t7*t8+8.723000000000001*t9*
        t10*t12*t13+0.01763993811759/t1+1.0)-0.00185357142857)/POW(2.718281828459045,
        25.0*t14*t2/(POW(t1,2.666666666666667)*t15))+0.02550787555556*
        LOG(2.656434474616293*t11*t18*(0.0625*t11*t18*t2*t3+0.01086645186224*
        t19*t20*t22)/(0.16602715466352*t14*t6*t2*t3*t22+0.00918833869522*
        t15*t18*t19*t20/POW(t21,2.0)+1.0)+1.0)*t4*t7-0.062182*t16*
        t17);

    return res;
}

static void
pw91c_first(FunFirstFuncDrv *ds, real factor, const FunDensProp *dp)
{
    real dfdra, dfdrb, dfdga, dfdgb, dfdgab;
    real rhoa = dp->rhoa, rhob = dp->rhob;
    real grada = dp->grada, gradb = dp->gradb, gradab = dp->gradab;

    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;
    real t19, t20, t21, t22, t23, t24, t25, t26;
    real t27, t28, t29, t30, t31, t32, t33, t34;
    real t35, t36, t37, t38, t39, t40, t41, t42;
    real t43, t44, t45, t46, t47, t48, t49, t50;
    real t51, t52, t53, t54, t55;

    t1 = POW(gradb,2.0)+2.0*gradab+POW(grada,2.0);
    t2 = rhob+rhoa;
    t3 = 1/POW(t2,2.333333333333334);
    t4 = 0.31830988618379;
    t5 = POW(3.0,0.66666666666667);
    t6 = 1/POW(4.0,0.66666666666667);
    t7 = POW(3.141592653589793,0.66666666666667);
    t8 = 1/t7;
    t9 = 1/POW(t2,0.66666666666667);
    t10 = POW(3.0,0.33333333333333);
    t11 = 1/POW(4.0,0.33333333333333);
    t12 = POW(3.141592653589793,0.33333333333333);
    t13 = 1/t12;
    t14 = 1/POW(t2,0.33333333333333);
    t15 = 0.472*t5*t6*t8*t9+0.0554175*t4/t2+8.723000000000001*
        t10*t11*t13*t14+1.0;
    t16 = 1/t15;
    t17 = 23.266*t10*t11*t13*t14+0.007389*t5*t6*t8*t9+2.568;
    t18 = 0.001*
        t16*t17-0.00185357142857;
    t19 = 1/t5;
    t20 = POW(3.141592653589793,1.333333333333333);
    t21 = 1/t20;
    t22 = 1/POW(2.718281828459045,25.0*t1*t19*t21/POW(t2,
        2.666666666666667));
    t23 = 0.2137*t10*t11*t13*t14+1.0;
    t24 = 1.732050807568877;
    t25 = 0.56418958354776;
    t26 = SQRT(t2);
    t27 = 1/POW(2.0,0.33333333333333);
    t28 = POW(3.0,0.16666666666667);
    t29 = 1/POW(3.141592653589793,0.16666666666667);
    t30 = 0.49294*t5*t6*t8*t9+7.5957*t27*t28*t29/POW(t2,0.16666666666667)+
        0.8191*t24*t25/t26+3.5876*t10*t11*t13*t14;
    t31 = 16.0818243221511/t30+1.0;
    t32 = LOG(t31);
    t33 = 1/POW(t2,1.666666666666667);
    t34 = 1/POW(t2,1.333333333333333);
    t35 = -1.195866666666667*t10*t11*t13*t34-0.32862666666667*
        t5*t6*t8*t33-1.26595*t27*t28*t29/POW(t2,1.166666666666667)-
        0.40955*t24*t25/POW(t26,3.0);
    t36 = 1/POW(t30,2.0);
    t37 = 1/t31;
    t38 = POW(t1,2.0);
    t39 = 1/POW(t2,3.333333333333334);
    t40 = 1/t10;
    t41 = 1/POW(t2,4.666666666666667);
    t42 = POW(2.718281828459045,2.437756914117331*t19*t7*
        t23*t32);
    t43 = t42-1.0;
    t44 = 1/t43;
    t45 = 0.01086645186224*t38*t41*t44+0.0625*t1*t12*t3*t40;
    t46 = 1/
        POW(t43,2.0);
    t47 = 0.00918833869522*t40*t20*t38*t41*t46+0.16602715466352*
        t19*t7*t1*t3*t44+1.0;
    t48 = 1/t47;
    t49 = 2.656434474616293*t40*t12*t45*t48+1.0;
    t50 = 1/t49;
    t51 = 1/POW(t2,5.666666666666667);
    t52 = -0.17364955084896*t40*t11*t12*t34*t32-39.20357843294411*
        t19*t7*t35*t23*t36*t37;
    t53 = 1/POW(t47,2.0);
    t54 = t2*(0.02550787555556*t5*t50*(2.656434474616293*
        t12*t40*t48*(-0.01086645186224*t38*t41*t42*t46*t52-0.05071010869043*
        t38*t44*t51-0.14583333333333*t1*t12*t39*t40)-2.656434474616293*
        t12*t40*t45*(-0.01837667739044*t20*t38*t40*t41*t42*t52/POW(t43,
        3.0)-0.04287891391103*t40*t20*t38*t51*t46-0.38739669421488*
        t19*t7*t1*t39*t44-0.16602715466352*t19*t7*t1*t3*t52*t46*t42)*
        t53)*t8+t1*t22*t3*(0.001*t16*(-7.755333333333333*t10*t11*t13*
        t34-0.004926*t5*t6*t8*t33)-0.001*t17*(-0.0554175*t4/POW(t2,
        2.0)-2.907666666666667*t10*t11*t13*t34-0.31466666666667*t5*
        t6*t8*t33)/POW(t15,2.0))-2.333333333333334*t1*t18*t22*t39+
        66.66666666666667*t18*t19*t21*t22*t38/POW(t2,6.0)+1.0*t35*
        t23*t36*t37+0.00442943113333*t10*t11*t13*t34*t32)+0.02550787555556*
        t5*t8*LOG(t49)-0.062182*t23*t32+t1*t3*t18*t22;
    t55 = 1/POW(t2,5.0);

   /* code */
    dfdra = t54;
    dfdrb = t54;
    dfdga = t2*(0.02550787555556*t5*t50*t8*(2.656434474616293*
        t12*t40*t48*(0.04346580744894*t1*t41*t44*grada+0.125*t12*t3*
        t40*grada)-2.656434474616293*t40*t12*t45*(0.33205430932704*
        t19*t7*grada*t3*t44+0.03675335478089*t40*t20*grada*t1*t41*
        t46)*t53)-50.0*t1*t18*t19*t21*t22*t55*grada+2.0*t18*t22*t3*
        grada);
    dfdgb = t2*(0.02550787555556*t5*t50*t8*(2.656434474616293*
        t12*t40*t48*(0.04346580744894*t1*t41*t44*gradb+0.125*t12*t3*
        t40*gradb)-2.656434474616293*t40*t12*t45*(0.33205430932704*
        t19*t7*gradb*t3*t44+0.03675335478089*t40*t20*gradb*t1*t41*
        t46)*t53)-50.0*t1*t18*t19*t21*t22*t55*gradb+2.0*t18*t22*t3*
        gradb);
    dfdgab = t2*(0.02550787555556*t5*t50*(2.656434474616293*
        t12*t40*(0.04346580744894*t1*t41*t44+0.125*t12*t3*t40)*t48-
        2.656434474616293*t40*t12*t45*(0.33205430932704*t19*t7*t3*
        t44+0.03675335478089*t40*t20*t1*t41*t46)*t53)*t8-50.0*t1*t18*
        t19*t21*t22*t55+2.0*t18*t22*t3);


   /* Final assignment */
    ds->df1000 += factor*dfdra;
    ds->df0100 += factor*dfdrb;
    ds->df0010 += factor*dfdga;
    ds->df0001 += factor*dfdgb;
    ds->df00001+= factor*dfdgab;
}

static void
pw91c_second(FunSecondFuncDrv *ds, real factor, const FunDensProp *dp)
{
    real dfdra, dfdrb, dfdga, dfdgb, dfdgab;
    real d2fdrara, d2fdrarb, d2fdraga, d2fdragb, d2fdrbrb, d2fdraab, 
         d2fdgaga, d2fdgbgb, d2fdrbga, d2fdrbgb,
         d2fdrbgab, d2fdgagb, d2fdgagab, d2fdgbgab, d2fdgabgab;
    real rhoa = dp->rhoa, rhob = dp->rhob;
    real grada = dp->grada, gradb = dp->gradb, gradab = dp->gradab;

    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;
    real t19, t20, t21, t22, t23, t24, t25, t26;
    real t27, t28, t29, t30, t31, t32, t33, t34;
    real t35, t36, t37, t38, t39, t40, t41, t42;
    real t43, t44, t45, t46, t47, t48, t49, t50;
    real t51, t52, t53, t54, t55, t56, t57, t58;
    real t59, t60, t61, t62, t63, t64, t65, t66;
    real t67, t68, t69, t70, t71, t72, t73, t74;
    real t75, t76, t77, t78, t79, t80, t81, t82;
    real t83, t84, t85, t86, t87, t88, t89, t90;
    real t91, t92, t93, t94, t95, t96, t97, t98;
    real t99, t100, t101, t102, t103, t104, t105;
    real t106, t107, t108, t109;

    t1 = POW(grada,2.0);
    t2 = POW(gradb,2.0);
    t3 = 2.0*gradab+t2+t1;
    t4 = rhob+rhoa;
    t5 = 1/POW(t4,2.333333333333334);
    t6 = 0.31830988618379;
    t7 = POW(3.0,0.66666666666667);
    t8 = 1/POW(4.0,0.66666666666667);
    t9 = POW(3.141592653589793,0.66666666666667);
    t10 = 1/t9;
    t11 = 1/POW(t4,0.66666666666667);
    t12 = POW(3.0,0.33333333333333);
    t13 = 1/POW(4.0,0.33333333333333);
    t14 = POW(3.141592653589793,0.33333333333333);
    t15 = 1/t14;
    t16 = 1/POW(t4,0.33333333333333);
    t17 = 0.0554175*t6/t4+8.723000000000001*t12*t13*t15*t16+
        0.472*t7*t8*t10*t11+1.0;
    t18 = 1/t17;
    t19 = 23.266*t12*t13*t15*t16+0.007389*t7*t8*t10*t11+2.568;
    t20 = 0.001*
        t18*t19-0.00185357142857;
    t21 = 1/t7;
    t22 = POW(3.141592653589793,1.333333333333333);
    t23 = 1/t22;
    t24 = 1/POW(t4,2.666666666666667);
    t25 = 1/POW(2.718281828459045,25.0*t21*t23*t24*t3);
    t26 = 0.2137*
        t12*t13*t15*t16+1.0;
    t27 = 1.732050807568877;
    t28 = 0.56418958354776;
    t29 = SQRT(t4);
    t30 = 1/POW(2.0,0.33333333333333);
    t31 = POW(3.0,0.16666666666667);
    t32 = 1/POW(3.141592653589793,0.16666666666667);
    t33 = 7.5957*t30*t31*t32/POW(t4,0.16666666666667)+0.8191*
        t27*t28/t29+3.5876*t12*t13*t15*t16+0.49294*t7*t8*t10*t11;
    t34 = 16.0818243221511/
        t33+1.0;
    t35 = LOG(t34);
    t36 = 1/POW(t4,1.666666666666667);
    t37 = 1/POW(t4,1.333333333333333);
    t38 = -1.26595*t30*t31*t32/POW(t4,1.166666666666667)-
        1.195866666666667*t12*t13*t15*t37-0.32862666666667*t7*t8*t10*
        t36-0.40955*t27*t28/POW(t29,3.0);
    t39 = 1/POW(t33,2.0);
    t40 = 1/t34;
    t41 = -7.755333333333333*t12*t13*t15*t37-0.004926*t7*
        t8*t10*t36;
    t42 = -0.0554175*t6/POW(t4,2.0)-2.907666666666667*t12*
        t13*t15*t37-0.31466666666667*t7*t8*t10*t36;
    t43 = 1/POW(t17,2.0);
    t44 = 0.001*t18*t41-0.001*t19*t42*t43;
    t45 = POW(t3,2.0);
    t46 = 1/POW(t4,6.0);
    t47 = 1/POW(t4,3.333333333333334);
    t48 = 1/t12;
    t49 = 1/POW(t4,4.666666666666667);
    t50 = POW(2.718281828459045,2.437756914117331*t21*t9*
        t26*t35);
    t51 = t50-1.0;
    t52 = 1/t51;
    t53 = 0.01086645186224*t45*t49*t52+0.0625*t14*t3*t48*
        t5;
    t54 = 1/POW(t51,2.0);
    t55 = 0.00918833869522*t48*t22*t45*t49*t54+0.16602715466352*
        t21*t9*t3*t5*t52+1.0;
    t56 = 1/t55;
    t57 = 2.656434474616293*t48*t14*t53*t56+1.0;
    t58 = 1/t57;
    t59 = 1/POW(t4,5.666666666666667);
    t60 = -0.17364955084896*t48*t13*t14*t37*t35-39.20357843294411*
        t21*t9*t38*t26*t39*t40;
    t61 = -0.01086645186224*t45*t49*t50*t54*t60-0.05071010869043*
        t45*t52*t59-0.14583333333333*t14*t3*t47*t48;
    t62 = 1/POW(t55,2.0);
    t63 = 1/POW(t51,3.0);
    t64 = -0.16602715466352*t21*t9*t3*t5*t60*t54*t50-0.01837667739044*
        t48*t22*t45*t49*t60*t63*t50-0.38739669421488*t21*t9*t3*t47*
        t52-0.04287891391103*t48*t22*t45*t59*t54;
    t65 = 2.656434474616293*t48*t14*t56*t61-2.656434474616293*
        t48*t14*t53*t62*t64;
    t66 = t4*(0.02550787555556*t7*t10*t58*t65-2.333333333333334*
        t20*t25*t3*t47+66.66666666666667*t20*t21*t23*t25*t45*t46+1.0*
        t38*t26*t39*t40+0.00442943113333*t12*t13*t15*t37*t35+t3*t5*
        t44*t25)+0.02550787555556*t7*t10*LOG(t57)-0.062182*t26*t35+
        t3*t5*t20*t25;
    t67 = 1/POW(t4,5.0);
    t68 = -50.0*t20*t21*t23*t25*t3*t67*grada;
    t69 = 2.0*t20*t25*t5*grada;
    t70 = 0.33205430932704*t21*t9*grada*t5*t52+0.03675335478089*
        t48*t22*grada*t3*t49*t54;
    t71 = 0.04346580744894*t3*t49*t52*grada+0.125*t14*t48*
        t5*grada;
    t72 = 2.656434474616293*t48*t14*t71*t56-2.656434474616293*
        t48*t14*t53*t70*t62;
    t73 = 0.02550787555556*t7*t10*t72*t58;
    t74 = -50.0*t20*t21*t23*t25*t3*t67*gradb;
    t75 = 2.0*t20*t25*t5*gradb;
    t76 = 0.33205430932704*t21*t9*gradb*t5*t52+0.03675335478089*
        t48*t22*gradb*t3*t49*t54;
    t77 = 0.04346580744894*t3*t49*t52*gradb+0.125*t14*t48*
        t5*gradb;
    t78 = 2.656434474616293*t48*t14*t77*t56-2.656434474616293*
        t48*t14*t53*t76*t62;
    t79 = 0.02550787555556*t7*t10*t78*t58;
    t80 = -50.0*t20*t21*t23*t25*t3*t67;
    t81 = 2.0*t20*t25*t5;
    t82 = 0.03675335478089*t48*t22*t3*t49*t54;
    t83 = 0.33205430932704*t21*t9*t5*t52;
    t84 = t83+t82;
    t85 = 0.125*t14*t48*t5;
    t86 = 0.04346580744894*t3*t49*t52;
    t87 = t86+t85;
    t88 = 2.656434474616293*t48*t14*t87*t56-2.656434474616293*
        t48*t14*t53*t84*t62;
    t89 = 0.02550787555556*t7*t10*t88*t58;
    t90 = POW(t38,2.0);
    t91 = 1/POW(t33,4.0);
    t92 = 1/POW(t34,2.0);
    t93 = 1/POW(t33,3.0);
    t94 = 1.594488888888889*t12*t13*t15*t5+1.476941666666667*
        t30*t31*t32/POW(t4,2.166666666666667)+0.614325*t27*t28/POW(t29,
        5.0)+0.54771111111111*t7*t8*t10*t24;
    t95 = 1/POW(3.141592653589793,2.666666666666667);
    t96 = 1/POW(t4,4.333333333333333);
    t97 = 1/POW(t57,2.0);
    t98 = 1/POW(t55,3.0);
    t99 = 1/POW(t4,6.666666666666667);
    t100 = 0.23153273446528*t48*t13*t14*t5*t35-39.20357843294411*
        t21*t9*t94*t26*t39*t40+5.58520314074677*t48*t13*t14*t37*t38*
        t39*t40+78.40715686588821*t21*t9*t90*t26*t93*t40-630.4650611582789*
        t21*t9*t90*t26*t91*t92;
    t101 = POW(t60,2.0);
    t102 = POW(2.718281828459045,4.875513828234661*t21*t9*
        t26*t35);
    t103 = t4*(0.02550787555556*t10*t58*t7*(2.656434474616293*
        t14*t48*t56*(0.28735728257913*t45*t52*t99+0.48611111111111*
        t14*t3*t48*t96+0.02173290372447*t101*t102*t45*t49*t63+0.10142021738087*
        t45*t50*t54*t59*t60-0.01086645186224*t101*t45*t49*t50*t54-
        0.01086645186224*t100*t45*t49*t50*t54)+5.312868949232586*t14*
        t48*t53*POW(t64,2.0)*t98-5.312868949232586*t48*t14*t62*t61*
        t64-2.656434474616293*t14*t48*t53*(0.24298051216252*t48*t22*
        t45*t99*t54+1.291322314049587*t21*t9*t3*t96*t52+0.05513003217133*
        t101*t102*t22*t45*t48*t49/POW(t51,4.0)+0.17151565564413*t48*
        t22*t45*t59*t60*t63*t50-0.01837667739044*t48*t22*t45*t49*t101*
        t63*t50-0.01837667739044*t48*t22*t45*t49*t100*t63*t50+0.77479338842975*
        t21*t9*t3*t47*t60*t54*t50-0.16602715466352*t21*t9*t3*t5*t101*
        t54*t50-0.16602715466352*t21*t9*t3*t5*t100*t54*t50+0.33205430932704*
        t21*t9*t3*t5*t101*t63*t102)*t62)-0.02550787555556*t10*POW(t65,
        2.0)*t7*t97+7.777777777777778*t20*t25*t3*t96+1481.481481481482*
        t20*t25*POW(t3,3.0)*t48*t95/POW(t4,9.666666666666666)+16.0818243221511*
        t90*t26*t91*t92+t25*t3*t5*(-0.001*t19*t43*(0.110835*t6/POW(t4,
        3.0)+3.876888888888889*t12*t13*t15*t5+0.52444444444444*t7*
        t8*t10*t24)+0.001*t18*(10.34044444444444*t12*t13*t15*t5+0.00821*
        t7*t8*t10*t24)-0.002*t41*t42*t43+0.002*t19*POW(t42,2.0)/POW(t17,
        3.0))-4.666666666666667*t25*t3*t44*t47+133.3333333333333*t21*
        t23*t25*t44*t45*t46-555.5555555555555*t20*t21*t23*t25*t45/
        POW(t4,7.0)-2.0*t90*t26*t93*t40-0.14246666666667*t12*t13*t15*
        t37*t38*t39*t40+1.0*t94*t26*t39*t40-0.00590590817778*t12*t13*
        t15*t5*t35)+0.05101575111111*t7*t10*t58*t65+2.0*t25*t3*t44*
        t5-4.666666666666667*t20*t25*t3*t47+133.3333333333333*t20*
        t21*t23*t25*t45*t46+2.0*t38*t26*t39*t40+0.00885886226667*t12*
        t13*t15*t37*t35;
    t104 = 1/POW(t4,8.666666666666666);
    t105 = t4*(0.02550787555556*t10*t58*t7*(2.656434474616293*
        t14*t48*t56*(-0.04346580744894*t3*t49*t50*t54*t60*grada-0.20284043476174*
        t3*t52*t59*grada-0.29166666666667*t14*t47*t48*grada)+5.312868949232586*
        t48*t14*t53*t70*t98*t64-2.656434474616293*t48*t14*t71*t62*
        t64-2.656434474616293*t48*t14*t70*t62*t61-2.656434474616293*
        t48*t14*t53*t62*(-0.33205430932704*t21*t9*grada*t5*t60*t54*
        t50-0.07350670956177*t48*t22*grada*t3*t49*t60*t63*t50-0.77479338842975*
        t21*t9*grada*t47*t52-0.17151565564413*t48*t22*grada*t3*t59*
        t54))-1111.111111111111*t104*t20*t25*t45*t48*t95*grada-50.0*
        t21*t23*t25*t3*t44*t67*grada+2.0*t25*t44*t5*grada-4.666666666666667*
        t20*t25*t47*grada+383.3333333333333*t20*t21*t23*t25*t3*t46*
        grada-0.02550787555556*t7*t10*t72*t97*t65)+t73+t69+t68;
    t106 = t4*(0.02550787555556*t10*t58*t7*(2.656434474616293*
        t14*t48*t56*(-0.04346580744894*t3*t49*t50*t54*t60*gradb-0.20284043476174*
        t3*t52*t59*gradb-0.29166666666667*t14*t47*t48*gradb)+5.312868949232586*
        t48*t14*t53*t76*t98*t64-2.656434474616293*t48*t14*t77*t62*
        t64-2.656434474616293*t48*t14*t76*t62*t61-2.656434474616293*
        t48*t14*t53*t62*(-0.33205430932704*t21*t9*gradb*t5*t60*t54*
        t50-0.07350670956177*t48*t22*gradb*t3*t49*t60*t63*t50-0.77479338842975*
        t21*t9*gradb*t47*t52-0.17151565564413*t48*t22*gradb*t3*t59*
        t54))-1111.111111111111*t104*t20*t25*t45*t48*t95*gradb-50.0*
        t21*t23*t25*t3*t44*t67*gradb+2.0*t25*t44*t5*gradb-4.666666666666667*
        t20*t25*t47*gradb+383.3333333333333*t20*t21*t23*t25*t3*t46*
        gradb-0.02550787555556*t7*t10*t78*t97*t65)+t79+t75+t74;
    t107 = t4*(-1111.111111111111*t104*t20*t25*t45*t48*t95+
        0.02550787555556*t10*t58*(5.312868949232586*t48*t14*t53*t84*
        t98*t64-2.656434474616293*t48*t14*t87*t62*t64-2.656434474616293*
        t48*t14*t84*t62*t61+2.656434474616293*t14*t48*t56*(-0.04346580744894*
        t3*t49*t50*t54*t60-0.20284043476174*t3*t52*t59-0.29166666666667*
        t14*t47*t48)-2.656434474616293*t48*t14*t53*t62*(-0.33205430932704*
        t21*t9*t5*t60*t54*t50-0.07350670956177*t48*t22*t3*t49*t60*
        t63*t50-0.77479338842975*t21*t9*t47*t52-0.17151565564413*t48*
        t22*t3*t59*t54))*t7-50.0*t21*t23*t25*t3*t44*t67-0.02550787555556*
        t7*t10*t88*t97*t65+2.0*t25*t44*t5-4.666666666666667*t20*t25*
        t47+383.3333333333333*t20*t21*t23*t25*t3*t46)+t89+t81+t80;
    t108 = 1/
        POW(t4,7.666666666666667);
    t109 = POW(3.141592653589793,1.666666666666667);

   /* code */
    dfdra = t66;
    dfdrb = t66;
    dfdga = t4*(t73+t69+t68);
    dfdgb = t4*(t79+t75+t74);
    dfdgab = t4*(t89+t81+t80);
    d2fdrara = t103;
    d2fdrarb = t103;
    d2fdraga = t105;
    d2fdragb = t106;
    d2fdrbrb = t103;
    d2fdraab = t107;
    d2fdgaga = t4*(0.02550787555556*t10*t58*t7*(5.312868949232586*
        t14*t48*t53*POW(t70,2.0)*t98+2.656434474616293*t14*t48*t56*
        (t86+t85+0.08693161489789*t1*t49*t52)-2.656434474616293*t48*
        t14*t53*(t83+t82+0.07350670956177*t48*t22*t1*t49*t54)*t62-
        5.312868949232586*t48*t14*t71*t70*t62)-0.02550787555556*t10*
        t7*POW(t72,2.0)*t97+833.3333333333334*t1*t108*t20*t25*t3*t48*
        t95+t81+t80-200.0*t1*t20*t21*t23*t25*t67);
    d2fdgbgb = t4*(0.02550787555556*t10*t58*t7*(5.312868949232586*
        t14*t48*t53*POW(t76,2.0)*t98+2.656434474616293*t14*t48*t56*
        (t86+t85+0.08693161489789*t2*t49*t52)-2.656434474616293*t48*
        t14*t53*(t83+t82+0.07350670956177*t48*t22*t2*t49*t54)*t62-
        5.312868949232586*t48*t14*t77*t76*t62)-0.02550787555556*t10*
        t7*POW(t78,2.0)*t97+833.3333333333334*t108*t2*t20*t25*t3*t48*
        t95+t81+t80-200.0*t2*t20*t21*t23*t25*t67);
    d2fdrbga = t105;
    d2fdrbgb = t106;
    d2fdgagb = t4*(833.3333333333334*t108*t20*t25*t3*t48*
        t95*grada*gradb-200.0*t20*t21*t23*t25*t67*grada*gradb-0.02550787555556*
        t7*t10*t72*t78*t97+0.02550787555556*t7*t10*t58*(0.07350670956177*
        t48*t22*grada*gradb*t49*t56*t52-0.19526575739549*t21*t109*
        grada*gradb*t49*t53*t62*t54-2.656434474616293*t48*t14*t71*
        t76*t62-2.656434474616293*t48*t14*t77*t70*t62+5.312868949232586*
        t48*t14*t53*t70*t76*t98));
    d2fdrbgab = t107;
    d2fdgagab = t4*(833.3333333333334*t108*t20*t25*t3*t48*
        t95*grada-200.0*t20*t21*t23*t25*t67*grada-0.02550787555556*
        t7*t10*t88*t72*t97+0.02550787555556*t7*t10*t58*(0.07350670956177*
        t48*t22*grada*t49*t56*t52-0.19526575739549*t21*t109*grada*
        t49*t53*t62*t54-2.656434474616293*t48*t14*t87*t70*t62-2.656434474616293*
        t48*t14*t71*t84*t62+5.312868949232586*t48*t14*t53*t84*t70*
        t98));
    d2fdgbgab = t4*(833.3333333333334*t108*t20*t25*t3*t48*
        t95*gradb-200.0*t20*t21*t23*t25*t67*gradb-0.02550787555556*
        t7*t10*t88*t78*t97+0.02550787555556*t7*t10*t58*(0.07350670956177*
        t48*t22*gradb*t49*t56*t52-0.19526575739549*t21*t109*gradb*
        t49*t53*t62*t54-2.656434474616293*t48*t14*t87*t76*t62-2.656434474616293*
        t48*t14*t77*t84*t62+5.312868949232586*t48*t14*t53*t84*t76*
        t98));
    d2fdgabgab = t4*(0.02550787555556*t10*t58*t7*(5.312868949232586*
        t14*t48*t53*POW(t84,2.0)*t98-5.312868949232586*t48*t14*t87*
        t84*t62-0.19526575739549*t21*t109*t49*t53*t62*t54+0.07350670956177*
        t48*t22*t49*t56*t52)-0.02550787555556*t10*t7*POW(t88,2.0)*
        t97+833.3333333333334*t108*t20*t25*t3*t48*t95-200.0*t20*t21*
        t23*t25*t67);


    ds->df1000 += factor*dfdra;
    ds->df0100 += factor*dfdrb;
    ds->df0010 += factor*dfdga;
    ds->df0001 += factor*dfdgb;
    ds->df00001+= factor*dfdgab;

    ds->df2000 += factor*d2fdrara;
    ds->df1100 += factor*d2fdrarb;
    ds->df1010 += factor*d2fdraga;
    ds->df1001 += factor*d2fdragb;
    ds->df10001+= factor*d2fdraab;
    ds->df0200 += factor*d2fdrbrb;
    ds->df0110 += factor*d2fdrbga;
    ds->df0101 += factor*d2fdrbgb;
    ds->df01001+= factor*d2fdrbgab;
    ds->df0020 += factor*d2fdgaga;
    ds->df0011 += factor*d2fdgagb;
    ds->df00101+= factor*d2fdgagab;
    ds->df0002 += factor*d2fdgbgb;
    ds->df00011+= factor*d2fdgbgab;
    ds->df00002+= factor*d2fdgabgab;
}

static void
pw91c_third(FunThirdFuncDrv *ds, real factor, const FunDensProp *dp)
{
    real dfdra, dfdrb, dfdga, dfdgb, dfdgab;
    real d2fdrara, d2fdrarb, d2fdraga, d2fdragb, d2fdrbrb, d2fdraab, 
         d2fdrbab, d2fdgaga, d2fdgbgb, d2fdrbga, d2fdrbgb,
         d2fdgagb;
    real d3fdrararb, d3fdraraga, d3fdraragb, d3fdrbrbab, d3fdraraab,
         d3fdrarbrb, d3fdrarbga, d3fdrarbgb, d3fdrarbab, d3fdragaga,
         d3fdragbgb, d3fdrarara, d3fdrbrbrb, d3fdrbrbga, d3fdrbrbgb,
         d3fdrbgaga, d3fdrbgbgb, d3fdgagaga,
         d3fdragagb, d3fdrbgagb, d3fdgagagb, d3fdgagbgb, d3fdgbgbgb;
    real rhoa = dp->rhoa, rhob = dp->rhob;
    real grada = dp->grada, gradb = dp->gradb, gradab = dp->gradab;

    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;
    real t19, t20, t21, t22, t23, t24, t25, t26;
    real t27, t28, t29, t30, t31, t32, t33, t34;
    real t35, t36, t37, t38, t39, t40, t41, t42;
    real t43, t44, t45, t46, t47, t48, t49, t50;
    real t51, t52, t53, t54, t55, t56, t57, t58;
    real t59, t60, t61, t62, t63, t64, t65, t66;
    real t67, t68, t69, t70, t71, t72, t73, t74;
    real t75, t76, t77, t78, t79, t80, t81, t82;
    real t83, t84, t85, t86, t87, t88, t89, t90;
    real t91, t92, t93, t94, t95, t96, t97, t98;
    real t99, t100, t101, t102, t103, t104, t105;
    real t106, t107, t108, t109, t110, t111, t112;
    real t113, t114, t115, t116, t117, t118, t119;
    real t120, t121, t122, t123, t124, t125, t126;
    real t127, t128, t129, t130, t131, t132, t133;
    real t134, t135, t136, t137, t138, t139, t140;
    real t141, t142, t143, t144, t145, t146, t147;
    real t148, t149, t150, t151, t152, t153, t154;
    real t155, t156, t157, t158, t159, t160, t161;
    real t162, t163, t164, t165, t166, t167, t168;
    real t169, t170, t171, t172, t173, t174, t175;
    real t176, t177, t178, t179, t180, t181, t182;
    real t183, t184, t185, t186, t187, t188, t189;
    real t190, t191, t192, t193, t194, t195, t196;
    real t197, t198, t199, t200, t201;

    t1 = POW(grada,2.0);
    t2 = POW(gradb,2.0);
    t3 = 2.0*gradab+t2+t1;
    t4 = rhob+rhoa;
    t5 = 1/POW(t4,2.333333333333334);
    t6 = 0.31830988618379;
    t7 = POW(3.0,0.66666666666667);
    t8 = 1/POW(4.0,0.66666666666667);
    t9 = POW(3.141592653589793,0.66666666666667);
    t10 = 1/t9;
    t11 = 1/POW(t4,0.66666666666667);
    t12 = POW(3.0,0.33333333333333);
    t13 = 1/POW(4.0,0.33333333333333);
    t14 = POW(3.141592653589793,0.33333333333333);
    t15 = 1/t14;
    t16 = 1/POW(t4,0.33333333333333);
    t17 = 0.0554175*t6/t4+8.723000000000001*t12*t13*t15*t16+
        0.472*t7*t8*t10*t11+1.0;
    t18 = 1/t17;
    t19 = 23.266*t12*t13*t15*t16+0.007389*t7*t8*t10*t11+2.568;
    t20 = 0.001*
        t18*t19-0.00185357142857;
    t21 = 1/t7;
    t22 = POW(3.141592653589793,1.333333333333333);
    t23 = 1/t22;
    t24 = 1/POW(t4,2.666666666666667);
    t25 = 1/POW(2.718281828459045,25.0*t21*t23*t24*t3);
    t26 = 0.2137*
        t12*t13*t15*t16+1.0;
    t27 = 1.732050807568877;
    t28 = 0.56418958354776;
    t29 = SQRT(t4);
    t30 = 1/POW(2.0,0.33333333333333);
    t31 = POW(3.0,0.16666666666667);
    t32 = 1/POW(3.141592653589793,0.16666666666667);
    t33 = 7.5957*t30*t31*t32/POW(t4,0.16666666666667)+0.8191*
        t27*t28/t29+3.5876*t12*t13*t15*t16+0.49294*t7*t8*t10*t11;
    t34 = 16.0818243221511/
        t33+1.0;
    t35 = LOG(t34);
    t36 = 1/POW(t4,1.666666666666667);
    t37 = 1/POW(t4,1.333333333333333);
    t38 = -1.26595*t30*t31*t32/POW(t4,1.166666666666667)-
        1.195866666666667*t12*t13*t15*t37-0.32862666666667*t7*t8*t10*
        t36-0.40955*t27*t28/POW(t29,3.0);
    t39 = 1/POW(t33,2.0);
    t40 = 1/t34;
    t41 = -7.755333333333333*t12*t13*t15*t37-0.004926*t7*
        t8*t10*t36;
    t42 = -0.0554175*t6/POW(t4,2.0)-2.907666666666667*t12*
        t13*t15*t37-0.31466666666667*t7*t8*t10*t36;
    t43 = 1/POW(t17,2.0);
    t44 = 0.001*t18*t41-0.001*t19*t42*t43;
    t45 = POW(t3,2.0);
    t46 = 1/POW(t4,6.0);
    t47 = 1/POW(t4,3.333333333333334);
    t48 = 1/t12;
    t49 = 1/POW(t4,4.666666666666667);
    t50 = POW(2.718281828459045,2.437756914117331*t21*t9*
        t26*t35);
    t51 = t50-1.0;
    t52 = 1/t51;
    t53 = 0.01086645186224*t45*t49*t52+0.0625*t14*t3*t48*
        t5;
    t54 = 1/POW(t51,2.0);
    t55 = 0.00918833869522*t48*t22*t45*t49*t54+0.16602715466352*
        t21*t9*t3*t5*t52+1.0;
    t56 = 1/t55;
    t57 = 2.656434474616293*t48*t14*t53*t56+1.0;
    t58 = 1/t57;
    t59 = 1/POW(t4,5.666666666666667);
    t60 = -0.17364955084896*t48*t13*t14*t37*t35-39.20357843294411*
        t21*t9*t38*t26*t39*t40;
    t61 = -0.01086645186224*t45*t49*t50*t54*t60-0.05071010869043*
        t45*t52*t59-0.14583333333333*t14*t3*t47*t48;
    t62 = 1/POW(t55,2.0);
    t63 = 1/POW(t51,3.0);
    t64 = -0.16602715466352*t21*t9*t3*t5*t60*t54*t50-0.01837667739044*
        t48*t22*t45*t49*t60*t63*t50-0.38739669421488*t21*t9*t3*t47*
        t52-0.04287891391103*t48*t22*t45*t59*t54;
    t65 = 2.656434474616293*t48*t14*t56*t61-2.656434474616293*
        t48*t14*t53*t62*t64;
    t66 = t4*(0.02550787555556*t7*t10*t58*t65-2.333333333333334*
        t20*t25*t3*t47+66.66666666666667*t20*t21*t23*t25*t45*t46+1.0*
        t38*t26*t39*t40+0.00442943113333*t12*t13*t15*t37*t35+t3*t5*
        t44*t25)+0.02550787555556*t7*t10*LOG(t57)-0.062182*t26*t35+
        t3*t5*t20*t25;
    t67 = 1/POW(t4,5.0);
    t68 = -50.0*t20*t21*t23*t25*t3*t67*grada;
    t69 = 2.0*t20*t25*t5*grada;
    t70 = 0.33205430932704*t21*t9*grada*t5*t52+0.03675335478089*
        t48*t22*grada*t3*t49*t54;
    t71 = 0.04346580744894*t3*t49*t52*grada+0.125*t14*t48*
        t5*grada;
    t72 = 2.656434474616293*t48*t14*t71*t56-2.656434474616293*
        t48*t14*t53*t70*t62;
    t73 = 0.02550787555556*t7*t10*t72*t58;
    t74 = -50.0*t20*t21*t23*t25*t3*t67*gradb;
    t75 = 2.0*t20*t25*t5*gradb;
    t76 = 0.33205430932704*t21*t9*gradb*t5*t52+0.03675335478089*
        t48*t22*gradb*t3*t49*t54;
    t77 = 0.04346580744894*t3*t49*t52*gradb+0.125*t14*t48*
        t5*gradb;
    t78 = 2.656434474616293*t48*t14*t77*t56-2.656434474616293*
        t48*t14*t53*t76*t62;
    t79 = 0.02550787555556*t7*t10*t78*t58;
    t80 = -50.0*t20*t21*t23*t25*t3*t67;
    t81 = 2.0*t20*t25*t5;
    t82 = 0.03675335478089*t48*t22*t3*t49*t54;
    t83 = 0.33205430932704*t21*t9*t5*t52;
    t84 = t83+t82;
    t85 = 0.125*t14*t48*t5;
    t86 = 0.04346580744894*t3*t49*t52;
    t87 = t86+t85;
    t88 = 2.656434474616293*t48*t14*t87*t56-2.656434474616293*
        t48*t14*t53*t84*t62;
    t89 = 0.02550787555556*t7*t10*t88*t58;
    t90 = POW(t38,2.0);
    t91 = 1/POW(t33,4.0);
    t92 = 1/POW(t34,2.0);
    t93 = 1/POW(t33,3.0);
    t94 = 1.594488888888889*t12*t13*t15*t5+1.476941666666667*
        t30*t31*t32/POW(t4,2.166666666666667)+0.614325*t27*t28/POW(t29,
        5.0)+0.54771111111111*t7*t8*t10*t24;
    t95 = 10.34044444444444*t12*t13*t15*t5+0.00821*t7*t8*
        t10*t24;
    t96 = POW(t42,2.0);
    t97 = 1/POW(t17,3.0);
    t98 = 0.110835*t6/POW(t4,3.0)+3.876888888888889*t12*t13*
        t15*t5+0.52444444444444*t7*t8*t10*t24;
    t99 = -0.001*t19*t43*t98+0.002*t19*t96*t97+0.001*t18*
        t95-0.002*t41*t42*t43;
    t100 = 1/POW(3.141592653589793,2.666666666666667);
    t101 = POW(t3,3.0);
    t102 = 1/POW(t4,9.666666666666666);
    t103 = 1/POW(t4,7.0);
    t104 = 1/POW(t4,4.333333333333333);
    t105 = 1/POW(t57,2.0);
    t106 = POW(t65,2.0);
    t107 = 1/POW(t55,3.0);
    t108 = POW(t64,2.0);
    t109 = 1/POW(t4,6.666666666666667);
    t110 = 0.23153273446528*t48*t13*t14*t5*t35-39.20357843294411*
        t21*t9*t94*t26*t39*t40+5.58520314074677*t48*t13*t14*t37*t38*
        t39*t40+78.40715686588821*t21*t9*t90*t26*t93*t40-630.4650611582789*
        t21*t9*t90*t26*t91*t92;
    t111 = POW(t60,2.0);
    t112 = POW(2.718281828459045,4.875513828234661*t21*t9*
        t26*t35);
    t113 = 0.02173290372447*t111*t112*t45*t49*t63+0.10142021738087*
        t45*t50*t54*t59*t60-0.01086645186224*t111*t45*t49*t50*t54-
        0.01086645186224*t110*t45*t49*t50*t54+0.28735728257913*t109*
        t45*t52+0.48611111111111*t104*t14*t3*t48;
    t114 = 1/POW(t51,4.0);
    t115 = 0.33205430932704*t21*t9*t3*t5*t111*t63*t112+0.05513003217133*
        t48*t22*t45*t49*t111*t114*t112-0.16602715466352*t21*t9*t3*
        t5*t111*t54*t50+0.77479338842975*t21*t9*t3*t47*t60*t54*t50-
        0.16602715466352*t21*t9*t3*t5*t110*t54*t50-0.01837667739044*
        t48*t22*t45*t49*t111*t63*t50+0.17151565564413*t48*t22*t45*
        t59*t60*t63*t50-0.01837667739044*t48*t22*t45*t49*t110*t63*
        t50+1.291322314049587*t21*t9*t3*t104*t52+0.24298051216252*
        t48*t22*t45*t109*t54;
    t116 = -2.656434474616293*t48*t14*t53*t62*t115+2.656434474616293*
        t48*t14*t56*t113+5.312868949232586*t48*t14*t53*t107*t108-5.312868949232586*
        t48*t14*t62*t61*t64;
    t117 = t4*(16.0818243221511*t90*t26*t91*t92+1481.481481481482*
        t100*t101*t102*t20*t25*t48-4.666666666666667*t25*t3*t44*t47+
        133.3333333333333*t21*t23*t25*t44*t45*t46-555.5555555555555*
        t103*t20*t21*t23*t25*t45-2.0*t90*t26*t93*t40-0.14246666666667*
        t12*t13*t15*t37*t38*t39*t40+1.0*t94*t26*t39*t40-0.00590590817778*
        t12*t13*t15*t5*t35+7.777777777777778*t104*t20*t25*t3+t3*t5*
        t99*t25+0.02550787555556*t7*t10*t58*t116-0.02550787555556*
        t7*t10*t105*t106)+0.05101575111111*t7*t10*t58*t65+2.0*t25*
        t3*t44*t5-4.666666666666667*t20*t25*t3*t47+133.3333333333333*
        t20*t21*t23*t25*t45*t46+2.0*t38*t26*t39*t40+0.00885886226667*
        t12*t13*t15*t37*t35;
    t118 = 1/POW(t4,8.666666666666666);
    t119 = -0.04346580744894*t3*t49*t50*t54*t60*grada-0.20284043476174*
        t3*t52*t59*grada-0.29166666666667*t14*t47*t48*grada;
    t120 = -0.33205430932704*t21*t9*grada*t5*t60*t54*t50-
        0.07350670956177*t48*t22*grada*t3*t49*t60*t63*t50-0.77479338842975*
        t21*t9*grada*t47*t52-0.17151565564413*t48*t22*grada*t3*t59*
        t54;
    t121 = -2.656434474616293*t48*t14*t71*t62*t64+5.312868949232586*
        t48*t14*t53*t70*t107*t64-2.656434474616293*t48*t14*t53*t62*
        t120-2.656434474616293*t48*t14*t70*t62*t61+2.656434474616293*
        t48*t14*t56*t119;
    t122 = t4*(-50.0*t21*t23*t25*t3*t44*t67*grada+2.0*t25*
        t44*t5*grada-1111.111111111111*t100*t118*t20*t25*t45*t48*grada-
        4.666666666666667*t20*t25*t47*grada+383.3333333333333*t20*
        t21*t23*t25*t3*t46*grada-0.02550787555556*t7*t10*t72*t105*
        t65+0.02550787555556*t7*t10*t58*t121)+t73+t69+t68;
    t123 = -0.04346580744894*t3*t49*t50*t54*t60*gradb-0.20284043476174*
        t3*t52*t59*gradb-0.29166666666667*t14*t47*t48*gradb;
    t124 = -0.33205430932704*t21*t9*gradb*t5*t60*t54*t50-
        0.07350670956177*t48*t22*gradb*t3*t49*t60*t63*t50-0.77479338842975*
        t21*t9*gradb*t47*t52-0.17151565564413*t48*t22*gradb*t3*t59*
        t54;
    t125 = -2.656434474616293*t48*t14*t77*t62*t64+5.312868949232586*
        t48*t14*t53*t76*t107*t64-2.656434474616293*t48*t14*t53*t62*
        t124-2.656434474616293*t48*t14*t76*t62*t61+2.656434474616293*
        t48*t14*t56*t123;
    t126 = t4*(-50.0*t21*t23*t25*t3*t44*t67*gradb+2.0*t25*
        t44*t5*gradb-1111.111111111111*t100*t118*t20*t25*t45*t48*gradb-
        4.666666666666667*t20*t25*t47*gradb+383.3333333333333*t20*
        t21*t23*t25*t3*t46*gradb-0.02550787555556*t7*t10*t78*t105*
        t65+0.02550787555556*t7*t10*t58*t125)+t79+t75+t74;
    t127 = -50.0*t21*t23*t25*t3*t44*t67;
    t128 = 2.0*t25*t44*t5;
    t129 = -1111.111111111111*t100*t118*t20*t25*t45*t48;
    t130 = 383.3333333333333*
        t20*t21*t23*t25*t3*t46;
    t131 = -4.666666666666667*t20*t25*t47;
    t132 = -0.29166666666667*t14*t47*t48;
    t133 = -0.20284043476174*t3*t52*t59;
    t134 = -0.04346580744894*t3*t49*t50*t54*t60;
    t135 = t134+t133+t132;
    t136 = -0.17151565564413*t48*t22*t3*t59*t54;
    t137 = -0.77479338842975*t21*t9*t47*t52;
    t138 = -0.07350670956177*t48*t22*t3*t49*t60*t63*t50;
    t139 = -
        0.33205430932704*t21*t9*t5*t60*t54*t50;
    t140 = t139+t138+t137+t136;
    t141 = -2.656434474616293*t48*t14*t87*t62*t64+5.312868949232586*
        t48*t14*t53*t84*t107*t64-2.656434474616293*t48*t14*t53*t62*
        t140-2.656434474616293*t48*t14*t84*t62*t61+2.656434474616293*
        t48*t14*t56*t135;
    t142 = t4*(-0.02550787555556*t7*t10*t88*t105*t65+0.02550787555556*
        t7*t10*t58*t141+t131+t130+t129+t128+t127)+t89+t81+t80;
    t143 = 1/POW(t4,7.666666666666667);
    t144 = 833.3333333333334*t1*t100*t143*t20*t25*t3*t48;
    t145 = -
        200.0*t1*t20*t21*t23*t25*t67;
    t146 = POW(t72,2.0);
    t147 = -0.02550787555556*t7*t10*t146*t105;
    t148 = POW(t70,2.0);
    t149 = t83+t82+0.07350670956177*t48*t22*t1*t49*t54;
    t150 = t86+
        t85+0.08693161489789*t1*t49*t52;
    t151 = 2.656434474616293*t48*t14*t150*t56-5.312868949232586*
        t48*t14*t71*t70*t62-2.656434474616293*t48*t14*t53*t149*t62+
        5.312868949232586*t48*t14*t53*t148*t107;
    t152 = 0.02550787555556*t7*t10*t151*t58;
    t153 = 833.3333333333334*t100*t143*t2*t20*t25*t3*t48;
    t154 = -
        200.0*t2*t20*t21*t23*t25*t67;
    t155 = POW(t78,2.0);
    t156 = -0.02550787555556*t7*t10*t155*t105;
    t157 = POW(t76,2.0);
    t158 = t83+t82+0.07350670956177*t48*t22*t2*t49*t54;
    t159 = t86+
        t85+0.08693161489789*t2*t49*t52;
    t160 = 2.656434474616293*t48*t14*t159*t56-5.312868949232586*
        t48*t14*t77*t76*t62-2.656434474616293*t48*t14*t53*t158*t62+
        5.312868949232586*t48*t14*t53*t157*t107;
    t161 = 0.02550787555556*t7*t10*t160*t58;
    t162 = 833.3333333333334*t100*t143*t20*t25*t3*t48*grada*
        gradb;
    t163 = -200.0*t20*t21*t23*t25*t67*grada*gradb;
    t164 = -0.02550787555556*t7*t10*t72*t78*t105;
    t165 = POW(3.141592653589793,1.666666666666667);
    t166 = 0.07350670956177*t48*t22*grada*gradb*t49*t56*t52-
        0.19526575739549*t21*t165*grada*gradb*t49*t53*t62*t54-2.656434474616293*
        t48*t14*t71*t76*t62-2.656434474616293*t48*t14*t77*t70*t62+
        5.312868949232586*t48*t14*t53*t70*t76*t107;
    t167 = 0.02550787555556*t7*t10*t58*t166;
    t168 = 833.3333333333334*t100*t143*t20*t25*t3*t48*grada;
    t169 = -
        200.0*t20*t21*t23*t25*t67*grada;
    t170 = -0.19526575739549*t21*t165*grada*t49*t53*t62*t54;
    t171 = 0.07350670956177*
        t48*t22*grada*t49*t56*t52;
    t172 = 833.3333333333334*t100*t143*t20*t25*t3*t48*gradb;
    t173 = -
        200.0*t20*t21*t23*t25*t67*gradb;
    t174 = -0.19526575739549*t21*t165*gradb*t49*t53*t62*t54;
    t175 = 0.07350670956177*
        t48*t22*gradb*t49*t56*t52;
    t176 = POW(t38,3.0);
    t177 = 1/POW(t33,6.0);
    t178 = 1/POW(t34,3.0);
    t179 = 1/POW(t33,5.0);
    t180 = 1/POW(t4,3.666666666666667);
    t181 = -3.720474074074074*t12*t13*t15*t47-3.200040277777777*
        t30*t31*t32/POW(t4,3.166666666666667)-1.5358125*t27*t28/POW(t29,
        7.0)-1.460562962962963*t7*t8*t10*t180;
    t182 = 1/POW(3.141592653589793,4.0);
    t183 = 1/POW(t4,5.333333333333333);
    t184 = 1/POW(t57,3.0);
    t185 = 1/POW(t55,4.0);
    t186 = -0.54024304708565*t48*t13*t14*t47*t35-39.20357843294411*
        t21*t9*t181*t26*t39*t40-11.17040628149354*t48*t13*t14*t5*t38*
        t39*t40+8.377804711120154*t48*t13*t14*t37*t94*t39*t40+235.2214705976646*
        t21*t9*t94*t38*t26*t93*t40-16.75560942224031*t48*t13*t14*t37*
        t90*t93*t40-235.2214705976646*t21*t9*t176*t26*t91*t40-1891.395183474837*
        t21*t9*t94*t38*t26*t91*t92+134.7303835695242*t48*t13*t14*t37*
        t90*t91*t92+3782.790366949674*t21*t9*t176*t26*t179*t92-20278.05670960339*
        t21*t9*t176*t26*t177*t178;
    t187 = POW(t60,3.0);
    t188 = -0.34729910169792*t48*t13*t14*t37*t35-78.40715686588821*
        t21*t9*t38*t26*t39*t40;
    t189 = POW(2.718281828459045,7.313270742351992*t21*t9*
        t26*t35);
    t190 = t4*(-7.0*t25*t3*t47*t99+200.0*t21*t23*t25*t45*
        t46*t99+t25*t3*t5*(0.006*t19*t42*t97*t98-0.003*t41*t43*t98+
        0.006*t41*t96*t97-0.003*t42*t43*t95-0.001*t19*t43*(-0.332505*
        t6/POW(t4,4.0)-9.046074074074074*t12*t13*t15*t47-1.398518518518519*
        t7*t8*t10*t180)+0.001*t18*(-24.1277037037037*t12*t13*t15*t47-
        0.02189333333333*t7*t8*t10*t180)-0.006*t19*POW(t42,3.0)/POW(t17,
        4.0))-3.43668585764369*t12*t13*t15*t37*t90*t91*t92+48.2454729664533*
        t94*t38*t26*t91*t92-96.4909459329066*t176*t26*t179*t92+0.05101575111111*
        t10*t184*POW(t65,3.0)*t7+0.02550787555556*t10*t58*(-15.93860684769776*
        t14*t185*t48*t53*POW(t64,3.0)+2.656434474616293*t14*t48*t56*
        (0.06519871117342*t110*t112*t45*t49*t60*t63-0.30426065214261*
        t111*t112*t45*t59*t63+0.02173290372447*t111*t112*t188*t45*
        t49*t63+0.02173290372447*t112*t187*t45*t49*t63-0.03259935558671*
        t110*t45*t49*t50*t54*t60-0.86207184773739*t109*t45*t50*t54*
        t60+0.1521303260713*t111*t45*t50*t54*t59+0.1521303260713*t110*
        t45*t50*t54*t59-0.01086645186224*t187*t45*t49*t50*t54-0.01086645186224*
        t186*t45*t49*t50*t54-1.91571521719419*t143*t45*t52-0.06519871117342*
        t114*t187*t189*t45*t49-2.106481481481481*t14*t183*t3*t48)-
        2.656434474616293*t14*t48*t53*(-1.619870081083455*t48*t22*
        t45*t143*t54-5.59573002754821*t21*t9*t3*t183*t52-0.22052012868531*
        t187*t189*t22*t45*t48*t49/POW(t51,5.0)-0.05513003217133*t48*
        t22*t45*t49*t110*t60*t63*t50-1.457883072975109*t48*t22*t45*
        t109*t60*t63*t50-0.01837667739044*t48*t22*t45*t49*t187*t63*
        t50-0.01837667739044*t48*t22*t45*t49*t186*t63*t50+0.2572734834662*
        t48*t22*t45*t59*t111*t63*t50+0.2572734834662*t48*t22*t45*t59*
        t110*t63*t50-0.49808146399055*t21*t9*t3*t5*t110*t60*t54*t50-
        3.873966942148761*t21*t9*t3*t104*t60*t54*t50-0.16602715466352*
        t21*t9*t3*t5*t187*t54*t50-0.16602715466352*t21*t9*t3*t5*t186*
        t54*t50+1.162190082644628*t21*t9*t3*t47*t111*t54*t50+1.162190082644628*
        t21*t9*t3*t47*t110*t54*t50-0.99616292798111*t21*t9*t3*t5*t187*
        t114*t189+0.99616292798111*t21*t9*t3*t5*t110*t60*t63*t112+
        0.33205430932704*t21*t9*t3*t5*t187*t63*t112-2.324380165289256*
        t21*t9*t3*t47*t111*t63*t112+0.33205430932704*t21*t9*t3*t5*
        t188*t111*t63*t112+0.16539009651398*t48*t22*t45*t49*t110*t60*
        t114*t112+0.05513003217133*t48*t22*t45*t49*t187*t114*t112-
        0.77182045039859*t48*t22*t45*t59*t111*t114*t112+0.05513003217133*
        t48*t22*t45*t49*t188*t111*t114*t112)*t62+15.93860684769776*
        t48*t14*t53*t107*t64*t115-7.969303423848879*t48*t14*t62*t61*
        t115-7.969303423848879*t48*t14*t62*t64*t113+15.93860684769776*
        t48*t14*t107*t61*t108)*t7+4444.444444444444*t100*t101*t102*
        t25*t44*t48-26666.66666666667*t100*t101*t20*t25*t48/POW(t4,
        10.66666666666667)-1666.666666666667*t103*t21*t23*t25*t44*
        t45+4407.407407407408*t20*t21*t23*t25*t45/POW(t4,8.0)+23.33333333333333*
        t104*t25*t3*t44+0.4274*t12*t13*t15*t37*t90*t93*t40-5.999999999999999*
        t94*t38*t26*t93*t40+5.999999999999999*t176*t26*t91*t40-0.2137*
        t12*t13*t15*t37*t94*t39*t40+0.28493333333333*t12*t13*t15*t5*
        t38*t39*t40+1.0*t181*t26*t39*t40+32921.81069958848*t182*t20*
        t25*POW(t3,4.0)/POW(t4,13.33333333333333)+0.01378045241481*
        t12*t13*t15*t47*t35-33.7037037037037*t183*t20*t25*t3+517.2501470570614*
        t176*t26*t177*t178-0.07652362666667*t7*t10*t105*t65*t116)+
        3.0*t25*t3*t5*t99+48.2454729664533*t90*t26*t91*t92+4444.444444444444*
        t100*t101*t102*t20*t25*t48-14.0*t25*t3*t44*t47+400.0*t21*t23*
        t25*t44*t45*t46-1666.666666666667*t103*t20*t21*t23*t25*t45-
        5.999999999999999*t90*t26*t93*t40-0.4274*t12*t13*t15*t37*t38*
        t39*t40+3.0*t94*t26*t39*t40-0.01771772453333*t12*t13*t15*t5*
        t35+23.33333333333333*t104*t20*t25*t3+0.07652362666667*t7*
        t10*t58*t116-0.07652362666667*t7*t10*t105*t106;
    t191 = 1/POW(t4,12.33333333333333);
    t192 = t4*(0.02550787555556*t10*t58*t7*(2.656434474616293*
        t14*t48*t56*(0.08693161489789*t111*t112*t3*t49*t63*grada+0.40568086952348*
        t3*t50*t54*t59*t60*grada-0.04346580744894*t111*t3*t49*t50*
        t54*grada-0.04346580744894*t110*t3*t49*t50*t54*grada+1.149429130316514*
        t109*t3*t52*grada+0.97222222222222*t104*t14*t48*grada)+10.62573789846517*
        t48*t14*t70*t107*t61*t64+10.62573789846517*t48*t14*t53*t107*
        t120*t64-5.312868949232586*t48*t14*t62*t119*t64-5.312868949232586*
        t48*t14*t62*t61*t120-2.656434474616293*t48*t14*t71*t62*t115+
        5.312868949232586*t48*t14*t53*t70*t107*t115-2.656434474616293*
        t48*t14*t70*t62*t113-2.656434474616293*t48*t14*t53*t62*(0.66410861865407*
        t21*t9*grada*t5*t111*t63*t112+0.22052012868531*t48*t22*grada*
        t3*t49*t111*t114*t112-0.33205430932704*t21*t9*grada*t5*t111*
        t54*t50+1.549586776859504*t21*t9*grada*t47*t60*t54*t50-0.33205430932704*
        t21*t9*grada*t5*t110*t54*t50-0.07350670956177*t48*t22*grada*
        t3*t49*t111*t63*t50+0.68606262257652*t48*t22*grada*t3*t59*
        t60*t63*t50-0.07350670956177*t48*t22*grada*t3*t49*t110*t63*
        t50+2.582644628099174*t21*t9*grada*t104*t52+0.97192204865007*
        t48*t22*grada*t3*t109*t54)-15.93860684769776*t48*t14*t53*t70*
        t185*t108+5.312868949232586*t48*t14*t71*t107*t108)-50.0*t21*
        t23*t25*t3*t67*t99*grada+2.0*t25*t5*t99*grada-2222.222222222222*
        t100*t118*t25*t44*t45*t48*grada+18148.14814814815*t100*t102*
        t20*t25*t45*t48*grada-9.333333333333334*t25*t44*t47*grada+
        766.6666666666666*t21*t23*t25*t3*t44*t46*grada-2611.111111111111*
        t103*t20*t21*t23*t25*t3*grada-24691.35802469136*t101*t182*
        t191*t20*t25*grada+15.55555555555556*t104*t20*t25*grada-0.05101575111111*
        t7*t10*t105*t121*t65-0.02550787555556*t7*t10*t72*t105*t116+
        0.05101575111111*t7*t10*t72*t184*t106)-100.0*t21*t23*t25*t3*
        t44*t67*grada+4.0*t25*t44*t5*grada-2222.222222222222*t100*
        t118*t20*t25*t45*t48*grada-9.333333333333334*t20*t25*t47*grada+
        766.6666666666666*t20*t21*t23*t25*t3*t46*grada-0.05101575111111*
        t7*t10*t72*t105*t65+0.05101575111111*t7*t10*t58*t121;
    t193 = t4*(0.02550787555556*t10*t58*t7*(2.656434474616293*
        t14*t48*t56*(0.08693161489789*t111*t112*t3*t49*t63*gradb+0.40568086952348*
        t3*t50*t54*t59*t60*gradb-0.04346580744894*t111*t3*t49*t50*
        t54*gradb-0.04346580744894*t110*t3*t49*t50*t54*gradb+1.149429130316514*
        t109*t3*t52*gradb+0.97222222222222*t104*t14*t48*gradb)+10.62573789846517*
        t48*t14*t76*t107*t61*t64+10.62573789846517*t48*t14*t53*t107*
        t124*t64-5.312868949232586*t48*t14*t62*t123*t64-5.312868949232586*
        t48*t14*t62*t61*t124-2.656434474616293*t48*t14*t77*t62*t115+
        5.312868949232586*t48*t14*t53*t76*t107*t115-2.656434474616293*
        t48*t14*t76*t62*t113-2.656434474616293*t48*t14*t53*t62*(0.66410861865407*
        t21*t9*gradb*t5*t111*t63*t112+0.22052012868531*t48*t22*gradb*
        t3*t49*t111*t114*t112-0.33205430932704*t21*t9*gradb*t5*t111*
        t54*t50+1.549586776859504*t21*t9*gradb*t47*t60*t54*t50-0.33205430932704*
        t21*t9*gradb*t5*t110*t54*t50-0.07350670956177*t48*t22*gradb*
        t3*t49*t111*t63*t50+0.68606262257652*t48*t22*gradb*t3*t59*
        t60*t63*t50-0.07350670956177*t48*t22*gradb*t3*t49*t110*t63*
        t50+2.582644628099174*t21*t9*gradb*t104*t52+0.97192204865007*
        t48*t22*gradb*t3*t109*t54)-15.93860684769776*t48*t14*t53*t76*
        t185*t108+5.312868949232586*t48*t14*t77*t107*t108)-50.0*t21*
        t23*t25*t3*t67*t99*gradb+2.0*t25*t5*t99*gradb-2222.222222222222*
        t100*t118*t25*t44*t45*t48*gradb+18148.14814814815*t100*t102*
        t20*t25*t45*t48*gradb-9.333333333333334*t25*t44*t47*gradb+
        766.6666666666666*t21*t23*t25*t3*t44*t46*gradb-2611.111111111111*
        t103*t20*t21*t23*t25*t3*gradb-24691.35802469136*t101*t182*
        t191*t20*t25*gradb+15.55555555555556*t104*t20*t25*gradb-0.05101575111111*
        t7*t10*t105*t125*t65-0.02550787555556*t7*t10*t78*t105*t116+
        0.05101575111111*t7*t10*t78*t184*t106)-100.0*t21*t23*t25*t3*
        t44*t67*gradb+4.0*t25*t44*t5*gradb-2222.222222222222*t100*
        t118*t20*t25*t45*t48*gradb-9.333333333333334*t20*t25*t47*gradb+
        766.6666666666666*t20*t21*t23*t25*t3*t46*gradb-0.05101575111111*
        t7*t10*t78*t105*t65+0.05101575111111*t7*t10*t58*t125;
    t194 = t4*(-50.0*t21*t23*t25*t3*t67*t99+2.0*t25*t5*t99+
        0.02550787555556*t10*t58*(10.62573789846517*t48*t14*t84*t107*
        t61*t64+10.62573789846517*t48*t14*t53*t107*t140*t64-5.312868949232586*
        t48*t14*t62*t135*t64+2.656434474616293*t14*t48*t56*(0.08693161489789*
        t111*t112*t3*t49*t63+0.40568086952348*t3*t50*t54*t59*t60-0.04346580744894*
        t111*t3*t49*t50*t54-0.04346580744894*t110*t3*t49*t50*t54+1.149429130316514*
        t109*t3*t52+0.97222222222222*t104*t14*t48)-5.312868949232586*
        t48*t14*t62*t61*t140-2.656434474616293*t48*t14*t87*t62*t115+
        5.312868949232586*t48*t14*t53*t84*t107*t115-2.656434474616293*
        t48*t14*t84*t62*t113-2.656434474616293*t48*t14*t53*t62*(0.66410861865407*
        t21*t9*t5*t111*t63*t112+0.22052012868531*t48*t22*t3*t49*t111*
        t114*t112-0.33205430932704*t21*t9*t5*t111*t54*t50+1.549586776859504*
        t21*t9*t47*t60*t54*t50-0.33205430932704*t21*t9*t5*t110*t54*
        t50-0.07350670956177*t48*t22*t3*t49*t111*t63*t50+0.68606262257652*
        t48*t22*t3*t59*t60*t63*t50-0.07350670956177*t48*t22*t3*t49*
        t110*t63*t50+2.582644628099174*t21*t9*t104*t52+0.97192204865007*
        t48*t22*t3*t109*t54)-15.93860684769776*t48*t14*t53*t84*t185*
        t108+5.312868949232586*t48*t14*t87*t107*t108)*t7-0.05101575111111*
        t7*t10*t105*t141*t65-2222.222222222222*t100*t118*t25*t44*t45*
        t48+18148.14814814815*t100*t102*t20*t25*t45*t48-9.333333333333334*
        t25*t44*t47+766.6666666666666*t21*t23*t25*t3*t44*t46-2611.111111111111*
        t103*t20*t21*t23*t25*t3-24691.35802469136*t101*t182*t191*t20*
        t25+15.55555555555556*t104*t20*t25-0.02550787555556*t7*t10*
        t88*t105*t116+0.05101575111111*t7*t10*t88*t184*t106)-100.0*
        t21*t23*t25*t3*t44*t67-0.05101575111111*t7*t10*t88*t105*t65+
        4.0*t25*t44*t5-2222.222222222222*t100*t118*t20*t25*t45*t48-
        9.333333333333334*t20*t25*t47+766.6666666666666*t20*t21*t23*
        t25*t3*t46+0.05101575111111*t7*t10*t58*t141;
    t195 = 1/POW(t4,11.33333333333333);
    t196 = t81+t80+t4*(0.02550787555556*t10*t58*(-2.656434474616293*
        t48*t14*t150*t62*t64-15.93860684769776*t48*t14*t53*t148*t185*
        t64+10.62573789846517*t48*t14*t71*t70*t107*t64+5.312868949232586*
        t48*t14*t53*t149*t107*t64-2.656434474616293*t48*t14*t149*t62*
        t61+5.312868949232586*t48*t14*t148*t107*t61+2.656434474616293*
        t14*t48*t56*(-0.08693161489789*t1*t49*t50*t54*t60-0.40568086952348*
        t1*t52*t59+t134+t133+t132)-2.656434474616293*t48*t14*t53*t62*
        (t139+t138-0.14701341912354*t48*t22*t1*t49*t60*t63*t50+t137+
        t136-0.34303131128826*t48*t22*t1*t59*t54)-5.312868949232586*
        t48*t14*t71*t62*t120+10.62573789846517*t48*t14*t53*t70*t107*
        t120-5.312868949232586*t48*t14*t70*t62*t119)*t7-200.0*t1*t21*
        t23*t25*t44*t67+0.05101575111111*t7*t10*t146*t184*t65-0.02550787555556*
        t7*t10*t151*t105*t65+833.3333333333334*t1*t100*t143*t25*t3*
        t44*t48-10833.33333333333*t1*t100*t118*t20*t25*t3*t48+1000.0*
        t1*t20*t21*t23*t25*t46+18518.51851851852*t1*t182*t195*t20*
        t25*t45+t131+t130+t129+t128+t127-0.05101575111111*t7*t10*t72*
        t105*t121)+t152+t147+t145+t144;
    t197 = t4*(0.02550787555556*t10*t58*t7*(2.656434474616293*
        t14*t48*t56*(-0.08693161489789*t49*t50*t54*t60*grada*gradb-
        0.40568086952348*t52*t59*grada*gradb)+0.39053151479099*t21*
        t165*grada*gradb*t49*t53*t107*t54*t64-0.07350670956177*t48*
        t22*grada*gradb*t49*t62*t52*t64-15.93860684769776*t48*t14*
        t53*t70*t76*t185*t64+5.312868949232586*t48*t14*t71*t76*t107*
        t64+5.312868949232586*t48*t14*t77*t70*t107*t64-0.19526575739549*
        t21*t165*grada*gradb*t49*t62*t54*t61+5.312868949232586*t48*
        t14*t70*t76*t107*t61-2.656434474616293*t48*t14*t53*t62*(-0.14701341912354*
        t48*t22*grada*gradb*t49*t60*t63*t50-0.34303131128826*t48*t22*
        grada*gradb*t59*t54)-2.656434474616293*t48*t14*t71*t62*t124+
        5.312868949232586*t48*t14*t53*t70*t107*t124-2.656434474616293*
        t48*t14*t70*t62*t123-2.656434474616293*t48*t14*t77*t62*t120+
        5.312868949232586*t48*t14*t53*t76*t107*t120-2.656434474616293*
        t48*t14*t76*t62*t119)-200.0*t21*t23*t25*t44*t67*grada*gradb+
        833.3333333333334*t100*t143*t25*t3*t44*t48*grada*gradb-10833.33333333333*
        t100*t118*t20*t25*t3*t48*grada*gradb+1000.0*t20*t21*t23*t25*
        t46*grada*gradb+18518.51851851852*t182*t195*t20*t25*t45*grada*
        gradb+0.05101575111111*t7*t10*t72*t78*t184*t65-0.02550787555556*
        t7*t10*t105*t166*t65-0.02550787555556*t7*t10*t72*t105*t125-
        0.02550787555556*t7*t10*t78*t105*t121)+t167+t164+t163+t162;
    t198 = t81+
        t80+t4*(0.02550787555556*t10*t58*(-2.656434474616293*t48*t14*
        t159*t62*t64-15.93860684769776*t48*t14*t53*t157*t185*t64+10.62573789846517*
        t48*t14*t77*t76*t107*t64+5.312868949232586*t48*t14*t53*t158*
        t107*t64-2.656434474616293*t48*t14*t158*t62*t61+5.312868949232586*
        t48*t14*t157*t107*t61+2.656434474616293*t14*t48*t56*(-0.08693161489789*
        t2*t49*t50*t54*t60-0.40568086952348*t2*t52*t59+t134+t133+t132)-
        2.656434474616293*t48*t14*t53*t62*(t139+t138-0.14701341912354*
        t48*t22*t2*t49*t60*t63*t50+t137+t136-0.34303131128826*t48*
        t22*t2*t59*t54)-5.312868949232586*t48*t14*t77*t62*t124+10.62573789846517*
        t48*t14*t53*t76*t107*t124-5.312868949232586*t48*t14*t76*t62*
        t123)*t7-200.0*t2*t21*t23*t25*t44*t67+0.05101575111111*t7*
        t10*t155*t184*t65-0.02550787555556*t7*t10*t160*t105*t65+833.3333333333334*
        t100*t143*t2*t25*t3*t44*t48-10833.33333333333*t100*t118*t2*
        t20*t25*t3*t48+1000.0*t2*t20*t21*t23*t25*t46+18518.51851851852*
        t182*t195*t2*t20*t25*t45+t131+t130+t129+t128+t127-0.05101575111111*
        t7*t10*t78*t105*t125)+t161+t156+t154+t153;
    t199 = POW(grada,3.0);
    t200 = 1/POW(t4,10.33333333333333);
    t201 = POW(gradb,3.0);

   /* code */
    dfdra = t66;
    dfdrb = t66;
    dfdga = t4*(t73+t69+t68);
    dfdgb = t4*(t79+t75+t74);
    dfdgab = t4*(t89+t81+t80);
    d2fdrara = t117;
    d2fdrarb = t117;
    d2fdraga = t122;
    d2fdragb = t126;
    d2fdrbrb = t117;
    d2fdraab = t142;
    d2fdrbab = t142;
    d2fdgaga = t4*(t152+t147+t81+t80+t145+t144);
    d2fdgbgb = t4*(t161+t156+t81+t80+t154+t153);
    d2fdrbga = t122;
    d2fdrbgb = t126;
    d2fdgagb = t4*(t167+t164+t163+t162);
    d3fdrararb = t190;
    d3fdraraga = t192;
    d3fdraragb = t193;
    d3fdrbrbab = t194;
    d3fdraraab = t194;
    d3fdrarbrb = t190;
    d3fdrarbga = t192;
    d3fdrarbgb = t193;
    d3fdrarbab = t194;
    d3fdragaga = t196;
    d3fdragagb = t197;
    d3fdrbgagb = t197;
    d3fdragbgb = t198;
    d3fdrarara = t190;
    d3fdrbrbrb = t190;
    d3fdrbrbga = t192;
    d3fdrbrbgb = t193;
    d3fdrbgaga = t196;
    d3fdrbgbgb = t198;
    d3fdgagaga = t4*(-600.0*t20*t21*t23*t25*t67*grada+2500.0*
        t100*t143*t20*t25*t3*t48*grada+0.05101575111111*t10*t184*t7*
        POW(t72,3.0)+0.02550787555556*t10*t58*t7*(-15.93860684769776*
        t14*t185*t48*t53*POW(t70,3.0)-7.969303423848879*t48*t14*t150*
        t70*t62-7.969303423848879*t48*t14*t71*t149*t62-0.58579727218648*
        t21*t165*grada*t49*t53*t62*t54+0.22052012868531*t48*t22*grada*
        t49*t56*t52+15.93860684769776*t48*t14*t53*t149*t70*t107+15.93860684769776*
        t48*t14*t71*t148*t107)+5000.0*t100*t143*t199*t20*t25*t48-13888.88888888889*
        t182*t199*t20*t200*t25*t3-0.07652362666667*t7*t10*t151*t72*
        t105);
    d3fdgagagb = t4*(5000.0*t1*t100*t143*t20*t25*t48*gradb-
        13888.88888888889*t1*t182*t20*t200*t25*t3*gradb+0.05101575111111*
        t7*t10*t146*t78*t184+0.02550787555556*t7*t10*t58*(t175-0.14701341912354*
        t48*t22*grada*gradb*t49*t70*t62*t52+t174-0.39053151479099*
        t21*t165*grada*gradb*t49*t71*t62*t54+0.78106302958197*t21*
        t165*grada*gradb*t49*t53*t70*t107*t54-2.656434474616293*t48*
        t14*t150*t76*t62-2.656434474616293*t48*t14*t77*t149*t62+10.62573789846517*
        t48*t14*t71*t70*t76*t107+5.312868949232586*t48*t14*t53*t149*
        t76*t107+5.312868949232586*t48*t14*t77*t148*t107-15.93860684769776*
        t48*t14*t53*t148*t76*t185)+t173+t172-0.05101575111111*t7*t10*
        t72*t105*t166-0.02550787555556*t7*t10*t151*t78*t105);
    d3fdgagbgb = t4*(5000.0*t100*t143*t2*t20*t25*t48*grada-
        13888.88888888889*t182*t2*t20*t200*t25*t3*grada+0.05101575111111*
        t7*t10*t72*t155*t184+0.02550787555556*t7*t10*t58*(t171-0.14701341912354*
        t48*t22*grada*gradb*t49*t76*t62*t52+t170-0.39053151479099*
        t21*t165*grada*gradb*t49*t77*t62*t54+0.78106302958197*t21*
        t165*grada*gradb*t49*t53*t76*t107*t54-2.656434474616293*t48*
        t14*t159*t70*t62-2.656434474616293*t48*t14*t71*t158*t62+5.312868949232586*
        t48*t14*t71*t157*t107+10.62573789846517*t48*t14*t77*t70*t76*
        t107+5.312868949232586*t48*t14*t53*t158*t70*t107-15.93860684769776*
        t48*t14*t53*t70*t157*t185)+t169+t168-0.05101575111111*t7*t10*
        t78*t105*t166-0.02550787555556*t7*t10*t160*t72*t105);
    d3fdgbgbgb = t4*(-600.0*t20*t21*t23*t25*t67*gradb+2500.0*
        t100*t143*t20*t25*t3*t48*gradb+0.05101575111111*t10*t184*t7*
        POW(t78,3.0)+0.02550787555556*t10*t58*t7*(-15.93860684769776*
        t14*t185*t48*t53*POW(t76,3.0)-7.969303423848879*t48*t14*t159*
        t76*t62-7.969303423848879*t48*t14*t77*t158*t62-0.58579727218648*
        t21*t165*gradb*t49*t53*t62*t54+0.22052012868531*t48*t22*gradb*
        t49*t56*t52+15.93860684769776*t48*t14*t53*t158*t76*t107+15.93860684769776*
        t48*t14*t77*t157*t107)+5000.0*t100*t143*t20*t201*t25*t48-13888.88888888889*
        t182*t20*t200*t201*t25*t3-0.07652362666667*t7*t10*t160*t78*
        t105);


    ds->df1000 += factor*dfdra;
    ds->df0100 += factor*dfdrb;
    ds->df0010 += factor*dfdga;
    ds->df0001 += factor*dfdgb;
    ds->df00001+= factor*dfdgab;

    ds->df2000 += factor*d2fdrara;
    ds->df1100 += factor*d2fdrarb;
    ds->df1010 += factor*d2fdraga;
    ds->df1001 += factor*d2fdragb;
    ds->df0200 += factor*d2fdrbrb;
    ds->df10001+= factor*d2fdraab;
    ds->df01001+= factor*d2fdrbab;
    ds->df0020 += factor*d2fdgaga;
    ds->df0011 += factor*d2fdgagb;
    ds->df0002 += factor*d2fdgbgb;
    ds->df0110 += factor*d2fdrbga;
    ds->df0101 += factor*d2fdrbgb;

    ds->df2010 += factor*d3fdraraga;
    ds->df2001 += factor*d3fdraragb;
    ds->df1101 += factor*d3fdrarbgb;
    ds->df11001 += factor*d3fdrarbab;
    ds->df1020 += factor*d3fdragaga;
    ds->df1011 += factor*d3fdragagb;
    ds->df0111 += factor*d3fdrbgagb;
    ds->df1002 += factor*d3fdragbgb;
    ds->df3000 += factor*d3fdrarara;
    ds->df2100 += factor*d3fdrararb;
    ds->df20001 += factor*d3fdraraab;
    ds->df02001 += factor*d3fdrbrbab;
    ds->df1200 += factor*d3fdrarbrb;
    ds->df1110 += factor*d3fdrarbga;
    ds->df0300 += factor*d3fdrbrbrb;
    ds->df0210 += factor*d3fdrbrbga;
    ds->df0201 += factor*d3fdrbrbgb;
    ds->df0120 += factor*d3fdrbgaga;
    ds->df0102 += factor*d3fdrbgbgb;
    ds->df0030 += factor*d3fdgagaga;
    ds->df0021 += factor*d3fdgagagb;
    ds->df0012 += factor*d3fdgagbgb;
    ds->df0003 += factor*d3fdgbgbgb;
}
