/**
 * @copyright 2020-2020 Uniontech Technology Co., Ltd.
 *
 * @file devicelistwidget.cpp
 *
 * @brief 主要实现磁盘树结构展示以及右键磁盘信息、硬盘健康检测、分区表错误处理、隐藏分区、显示分区、删除分区等操作
 *
 * @date 2020-09-03 14:54
 *
 * Author: yuandandan  <yuandandan@uniontech.com>
 *
 * Maintainer: yuandandan  <yuandandan@uniontech.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "devicelistwidget.h"
#include "customcontrol/dmdiskinfobox.h"
#include "diskinfodisplaydialog.h"
#include "diskhealthdetectiondialog.h"
#include "messagebox.h"
#include "partitiontableerrorsinfodialog.h"
#include "diskbadsectorsdialog.h"
#include "createpartitiontabledialog.h"

#include <DPalette>
#include <DMenu>
#include <DMessageManager>

#include <QVBoxLayout>
#include <QDebug>
#include <QApplication>

DeviceListWidget::DeviceListWidget(QWidget *parent)
    : DWidget(parent)
{
    m_curChooseDevicePath = "";

    setAutoFillBackground(true);
    auto plt = this->palette();
    plt.setBrush(QPalette::Background, QBrush(Qt::white));
    setPalette(plt);
    setMaximumWidth(360);
    setMinimumWidth(100);
    initUi();
    initConnection();
}

DeviceListWidget::~DeviceListWidget()
{
    //    delete m_box;
    //    delete m_childbox;
}

void DeviceListWidget::initUi()
{
    QVBoxLayout *layout = new QVBoxLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    setLayout(layout);
    m_treeView = new DmTreeview(this);
    m_treeView->setContextMenuPolicy(Qt::CustomContextMenu);
    layout->addWidget(m_treeView);
}

void DeviceListWidget::initConnection()
{
    connect(DMDbusHandler::instance(), &DMDbusHandler::updateDeviceInfo, this, &DeviceListWidget::onUpdateDeviceInfo);
    connect(m_treeView, &DmTreeview::curSelectChanged, DMDbusHandler::instance(), &DMDbusHandler::onSetCurSelect);
    connect(m_treeView, &QTreeView::customContextMenuRequested, this, &DeviceListWidget::treeMenu);
    connect(DMDbusHandler::instance(), &DMDbusHandler::unmountPartitionMessage, this, &DeviceListWidget::onUnmountPartition);
    connect(DMDbusHandler::instance(), &DMDbusHandler::deletePartitionMessage, this, &DeviceListWidget::onDeletePartition);
    connect(DMDbusHandler::instance(), &DMDbusHandler::hidePartitionMessage, this, &DeviceListWidget::onHidePartition);
    connect(DMDbusHandler::instance(), &DMDbusHandler::showPartitionMessage, this, &DeviceListWidget::onShowPartition);
    connect(DMDbusHandler::instance(), &DMDbusHandler::createPartitionTableMessage, this, &DeviceListWidget::onCreatePartitionTableMessage);
    connect(DMDbusHandler::instance(), &DMDbusHandler::updateUsb, this, &DeviceListWidget::onUpdateUsb);
}

void DeviceListWidget::treeMenu(const QPoint &pos)
{
    QModelIndex curIndex = m_treeView->indexAt(pos);      //当前点击的元素的index
    QModelIndex index = curIndex.sibling(curIndex.row(),0); //该行的第1列元素的index

    if (!index.isValid()) {
        return;
    }

    m_curDiskInfoData = m_treeView->getCurItem()->data().value<DiskInfoData>();
    if (m_curDiskInfoData.m_level == 0) {
        QMenu *menu = new QMenu(this);
        menu->setObjectName("treeMenu");
        menu->setAccessibleName("menu");

        QAction *actionInfo = new QAction(this);
        actionInfo->setText(tr("Disk info")); // 磁盘信息
        actionInfo->setObjectName("Disk info");
        menu->addAction(actionInfo);
        connect(actionInfo, &QAction::triggered, this, &DeviceListWidget::onDiskInfoClicked);

        //        menu->addSeparator();    //添加一个分隔线
        QMenu *itemChildMenu = new QMenu(this);
        itemChildMenu->setTitle(tr("Health management")); // 健康管理
        itemChildMenu->setObjectName("Health management");
        itemChildMenu->setAccessibleName("Health management");
        menu->addMenu(itemChildMenu);

        QAction *actionHealthDetection = new QAction(this);
        actionHealthDetection->setText(tr("Check health")); // 硬盘健康检测
        actionHealthDetection->setObjectName("Check health");
        itemChildMenu->addAction(actionHealthDetection);
        connect(actionHealthDetection, &QAction::triggered, this, &DeviceListWidget::onDiskCheckHealthClicked);

        QAction *actionCheckError = new QAction(this);
        actionCheckError->setText(tr("Check partition table error")); // 分区表错误检测
        actionCheckError->setObjectName("Check partition table error");
        itemChildMenu->addAction(actionCheckError);
        connect(actionCheckError, &QAction::triggered, this, &DeviceListWidget::onPartitionErrorCheckClicked);

        QAction *actionVerifyRepair = new QAction(this);
        actionVerifyRepair->setText(tr("Verify or repair bad sectors")); // 坏道检测与修复
        actionVerifyRepair->setObjectName("Verify or repair bad sectors");
        itemChildMenu->addAction(actionVerifyRepair);
        connect(actionVerifyRepair, &QAction::triggered, this, &DeviceListWidget::onDiskBadSectorsClicked);

        QAction *createPartitionTable = new QAction(this);
        createPartitionTable->setText(tr("Create partition table")); // 新建分区表
        createPartitionTable->setObjectName("Create partition table");
        menu->addAction(createPartitionTable);
        connect(createPartitionTable, &QAction::triggered, this, &DeviceListWidget::onCreatePartitionTableClicked);

        menu->exec(QCursor::pos());  //显示菜单
        delete menu;
    } else {
        QMenu *menu = new QMenu(this);
        menu->setObjectName("treeMenu");
        menu->setAccessibleName("menu");

//        QAction *actionHidePartition = new QAction(this);
//        actionHidePartition->setText(tr("Hide partition")); // 隐藏分区
//        actionHidePartition->setObjectName("Hide partition");
//        menu->addAction(actionHidePartition);
//        connect(actionHidePartition, &QAction::triggered, this, &DeviceListWidget::onHidePartitionClicked);

//        QAction *actionShowPartition = new QAction(this);
//        actionShowPartition->setText(tr("Unhide partition")); // 显示分区
//        actionShowPartition->setObjectName("Unhide partition");
//        menu->addAction(actionShowPartition);
//        connect(actionShowPartition, &QAction::triggered, this, &DeviceListWidget::onShowPartitionClicked);
//        //            actionShowPartition->setDisabled(true); // 将按钮置为不可选

        QAction *actionDelete = new QAction(this);
        actionDelete->setText(tr("Delete partition")); // 删除分区
        actionDelete->setObjectName("Delete partition");
        menu->addAction(actionDelete);
        connect(actionDelete, &QAction::triggered, this, &DeviceListWidget::onDeletePartitionClicked);

        if (!m_curDiskInfoData.m_mountpoints.isEmpty()) {
            actionDelete->setDisabled(true);
        }

        if (m_curDiskInfoData.m_fstype == "unallocated" || m_curDiskInfoData.m_fstype == "linux-swap") {
//            actionHidePartition->setDisabled(true);
//            actionShowPartition->setDisabled(true);
            actionDelete->setDisabled(true);
        } /*else {
            if (1 == m_curDiskInfoData.m_flag) {
                actionHidePartition->setDisabled(true);
                actionShowPartition->setDisabled(false);
            } else {
                actionHidePartition->setDisabled(false);
                actionShowPartition->setDisabled(true);
            }
        }*/

        menu->exec(QCursor::pos());  //显示菜单
        delete menu;
    }
}

void DeviceListWidget::onDiskInfoClicked()
{
    setCurDevicePath(m_curDiskInfoData.m_diskPath);

    DiskInfoDisplayDialog diskInfoDisplayDialog(m_curDiskInfoData.m_diskPath, this);
    diskInfoDisplayDialog.setObjectName("diskInfoDisplayDialog");
    diskInfoDisplayDialog.setAccessibleName("diskInfoDialog");
    diskInfoDisplayDialog.exec();

    setCurDevicePath("");
}

void DeviceListWidget::onDiskCheckHealthClicked()
{
    setCurDevicePath(m_curDiskInfoData.m_diskPath);

    HardDiskStatusInfoList hardDiskStatusInfoList = DMDbusHandler::instance()->getDeviceHardStatusInfo(m_curDiskInfoData.m_diskPath);
    if (hardDiskStatusInfoList.count() < 1) {
        MessageBox warningBox(this);
        warningBox.setObjectName("messageBox");
        warningBox.setAccessibleName("warningMessageBox");
        // 获取不到硬件相应信息  关闭
        warningBox.setWarings(tr("Failed to get hardware information"), "", tr("Close"), "close");
        warningBox.exec();

        setCurDevicePath("");

        return;
    }

    DiskHealthDetectionDialog diskHealthDetectionDialog(m_curDiskInfoData.m_diskPath, hardDiskStatusInfoList, this);
    diskHealthDetectionDialog.setObjectName("diskHealthDetectionDialog");
    diskHealthDetectionDialog.setAccessibleName("diskHealthInfoDialog");
    diskHealthDetectionDialog.exec();

    setCurDevicePath("");
}

void DeviceListWidget::onDiskBadSectorsClicked()
{
    setCurDevicePath(m_curDiskInfoData.m_diskPath);

    DiskBadSectorsDialog diskBadSectorsDialog(this);
    diskBadSectorsDialog.setObjectName("diskBadSectorsDialog");
    diskBadSectorsDialog.setAccessibleName("diskBadSectorsDialog");
    diskBadSectorsDialog.exec();

    setCurDevicePath("");
}

bool DeviceListWidget::isExistMountPartition()
{
    bool isExist = false;

    DeviceInfo info = DMDbusHandler::instance()->getCurDeviceInfo();
    for (int i = 0; i < info.m_partition.size(); i++) {
        PartitionInfo partitionInfo = info.m_partition.at(i);

        QString mountpoints;
        for (int j = 0; j < partitionInfo.m_mountPoints.size(); j++) {
            mountpoints += partitionInfo.m_mountPoints[j];
        }

        if (!mountpoints.isEmpty()) {
            isExist = true;
            break;
        }
    }

    return isExist;
}

void DeviceListWidget::onCreatePartitionTableClicked()
{
    setCurDevicePath(m_curDiskInfoData.m_diskPath);

    if (isExistMountPartition()) {
        MessageBox warningBox(this);
        warningBox.setObjectName("messageBox");
        warningBox.setAccessibleName("messageBox");
        // 请先卸载当前磁盘中的所有分区  确定
        warningBox.setWarings(tr("Please unmount all partitions in the disk first"), "", tr("OK"), "ok");
        warningBox.exec();

        setCurDevicePath("");
        
        return;
    }

    MessageBox messageBox(this);
    messageBox.setObjectName("messageBox");
    messageBox.setAccessibleName("messageBox");
    // 新建分区表之后将会合并当前磁盘所有分区，丢失所有数据，请谨慎使用  继续  取消
    messageBox.setWarings(tr("All partitions in this disk will be merged and all data\n will be lost if creating a new partition table,\n please take it carefully"), "", tr("Proceed"), "Proceed", tr("Cancel"), "cancelBtn");
    if (messageBox.exec() == DDialog::Accepted) {
        CreatePartitionTableDialog createPartitionTableDialog(this);
        createPartitionTableDialog.setObjectName("createPartitionTable");
        createPartitionTableDialog.setAccessibleName("createPartitionTableDialog");
        createPartitionTableDialog.exec();
    }

    setCurDevicePath("");
}

void DeviceListWidget::onPartitionErrorCheckClicked()
{
    bool result = DMDbusHandler::instance()->detectionPartitionTableError(m_curDiskInfoData.m_diskPath);
    if (result) {
        QString deviceInfo = QString("%1(%2)").arg(m_curDiskInfoData.m_diskPath).arg(m_curDiskInfoData.m_diskSize);
        setCurDevicePath(m_curDiskInfoData.m_diskPath);

        PartitionTableErrorsInfoDialog partitionTableErrorsInfoDialog(deviceInfo, this);
        partitionTableErrorsInfoDialog.setObjectName("partitionErrorCheck");
        partitionTableErrorsInfoDialog.setAccessibleName("partitionErrorCheckDialog");
        partitionTableErrorsInfoDialog.exec();

        setCurDevicePath("");
    } else {
        // 分区表检测正常
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/ok.svg"), tr("No errors found in the partition table"));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    }
}

void DeviceListWidget::onHidePartitionClicked()
{
    setCurDevicePath(m_curDiskInfoData.m_diskPath);

    MessageBox messageBox(this);
    messageBox.setObjectName("messageBox");
    messageBox.setAccessibleName("hideMessageBox");
    // 您是否要隐藏该分区？ 隐藏  取消
    messageBox.setWarings(tr("Do you want to hide this partition?"), "", tr("Hide"), "hide", tr("Cancel"), "cancel");
    if (messageBox.exec() == DDialog::Accepted) {
        if (m_curDiskInfoData.m_mountpoints == "/boot/efi" || m_curDiskInfoData.m_mountpoints == "/boot"
                || m_curDiskInfoData.m_mountpoints == "/" || m_curDiskInfoData.m_mountpoints == "/data/home/opt/root/var"
                || m_curDiskInfoData.m_mountpoints == "/deepin/userdata/home/opt/root/var") {
            isHideSuccess = false;
            // 隐藏分区失败！无法锁定该分区
            DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("Failed to hide the partition: unable to lock it"));
            DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
        } else {
            if (m_curDiskInfoData.m_mountpoints.isEmpty()) {
                DMDbusHandler::instance()->hidePartition();
            } else {
                isHideSuccess = false;
                // 只有处于卸载状态的分区才能被隐藏
                DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("You can only hide the unmounted partition"));
                DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
            }
        }
    }

    setCurDevicePath("");
}

void DeviceListWidget::onShowPartitionClicked()
{
    setCurDevicePath(m_curDiskInfoData.m_diskPath);

    MessageBox messageBox(this);
    messageBox.setObjectName("messageBox");
    messageBox.setAccessibleName("unhideMessageBox");
    // 您是否要显示该隐藏分区？ 显示  取消
    messageBox.setWarings(tr("Do you want to unhide this partition?"), "", tr("Unhide"), "unhide", tr("Cancel"), "cancel");
    if (messageBox.exec() == DDialog::Accepted) {
        DMDbusHandler::instance()->unhidePartition();
    }

    setCurDevicePath("");
}

void DeviceListWidget::onDeletePartitionClicked()
{
    setCurDevicePath(m_curDiskInfoData.m_diskPath);

    MessageBox messageBox(this);
    messageBox.setObjectName("messageBox");
    messageBox.setAccessibleName("deleteMessageBox");
    // 您确定要删除该分区吗？ 该分区内所有文件将会丢失  删除  取消
    messageBox.setWarings(tr("Are you sure you want to delete this partition?"), tr("You will lose all data in it"), tr("Delete"), DDialog::ButtonWarning, "delete", tr("Cancel"), "cancel");
    if (messageBox.exec() == DDialog::Accepted) {
        if (m_curDiskInfoData.m_mountpoints == "/boot/efi" || m_curDiskInfoData.m_mountpoints == "/boot"
                || m_curDiskInfoData.m_mountpoints == "/" || m_curDiskInfoData.m_mountpoints == "/data/home/opt/root/var"
                || m_curDiskInfoData.m_fstype == "linux-swap" || m_curDiskInfoData.m_mountpoints == "/deepin/userdata/home/opt/root/var") {
            isDeleteSuccess = false;
            // 删除分区失败！无法锁定该分区
            DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("Failed to delete the partition: unable to lock it"));
            DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
        } else {
            DMDbusHandler::instance()->deletePartition();
        }
    }

    setCurDevicePath("");
}

void DeviceListWidget::onHidePartition(const QString &hideMessage)
{
    if ("1" == hideMessage) {
        // 隐藏分区成功
        isHideSuccess = true;
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/ok.svg"), tr("Hide the partition successfully"));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    } else {
        isHideSuccess = false;
        // 隐藏分区失败
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("Failed to hide the partition"));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    }
}

void DeviceListWidget::onShowPartition(const QString &showMessage)
{
    if ("1" == showMessage) {
        // 显示分区成功
        isShowSuccess = true;
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/ok.svg"), tr("Unhide the partition successfully"));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    } else {
        isShowSuccess = false;
        // 显示分区失败
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("Failed to unhide the partition"));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    }
}

void DeviceListWidget::onDeletePartition(const QString &deleteMessage)
{
    QStringList infoList = deleteMessage.split(":");

    if (infoList.count() <= 1) {
        return;
    }

    if ("1" == infoList.at(0)) {
        // 删除分区成功
        isDeleteSuccess = true;
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/ok.svg"), tr("Delete the partition successfully"));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    } else {
        isDeleteSuccess = false;

        QString reason;
        switch (infoList.at(1).toInt()) {
        case 1: {
            reason = tr("Failed to find the disk");
            break;
        }
        case 2: {
            reason = tr("Failed to get the partition info");
            break;
        }
        case 3: {
            reason = tr("Failed to delete the partition");
            break;
        }
        case 4: {
            reason = tr("Failed to submit the request to the kernel");
            break;
        }
        default:
            break;
        }
        // 删除分区失败
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("Failed to delete the partition: %1").arg(reason));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    }
}

void DeviceListWidget::onUnmountPartition(const QString &unmountMessage)
{
    if ("1" == unmountMessage) {
        isUnmountSuccess = true;
        // 卸载成功
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/ok.svg"), tr("Unmounting successful"));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    } else {
        isUnmountSuccess = false;
        // 卸载失败
        DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("Unmounting failed"));
        DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
    }
}

void DeviceListWidget::onCreatePartitionTableMessage(const bool &flag)
{
    if (flag) {
        isCreateTableSuccess = true;
        if (DMDbusHandler::instance()->getCurCreatePartitionTableType() == "create") {
            // 新建分区表成功
            DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/ok.svg"), tr("Creating partition table successful"));
            DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
        } else {
            // 替换分区表成功
            DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/ok.svg"), tr("Replacing partition table successful"));
            DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
        }

    } else {
        isCreateTableSuccess = false;
        if (DMDbusHandler::instance()->getCurCreatePartitionTableType() == "create") {
            // 新建分区表失败
            DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("Creating partition table failed"));
            DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
        } else {
            // 替换分区表失败
            DMessageManager::instance()->sendMessage(this->parentWidget()->parentWidget(), QIcon::fromTheme("://icons/deepin/builtin/warning.svg"), tr("Replacing partition table failed"));
            DMessageManager::instance()->setContentMargens(this->parentWidget()->parentWidget(), QMargins(0, 0, 0, 20));
        }
    }
}

void DeviceListWidget::onUpdateUsb()
{
    if (m_curChooseDevicePath == "")
        return;

    QStringList deviceNameList = DMDbusHandler::instance()->getDeviceNameList();
    if (deviceNameList.indexOf(m_curChooseDevicePath) != -1)
        return;

    QWidgetList widgetList = QApplication::topLevelWidgets();
    for (int i = 0; i < widgetList.count(); i++) {
        QWidget *widget = widgetList.at(i);
        if (widget->objectName() == "diskBadSectorsDialog") {
            DiskBadSectorsDialog *diskBadSectorsDialog = static_cast<DiskBadSectorsDialog *>(widget);
            diskBadSectorsDialog->stopCheckRepair();

            diskBadSectorsDialog->close();
        } else if (widget->objectName() == "diskInfoDisplayDialog" || widget->objectName() == "diskHealthDetectionDialog" ||
                   widget->objectName() == "partitionErrorCheck" || widget->objectName() == "createPartitionTable" ||
                   widget->objectName() == "messageBox") {
            widget->close();
        } else if (widget->objectName() == "exitMessageBox") {
            MessageBox *messageBox = static_cast<MessageBox *>(widget);
            messageBox->accept();
            break;
        }
    }

    setCurDevicePath("");
}

void DeviceListWidget::setCurDevicePath(const QString &devPath)
{
    m_curChooseDevicePath = devPath;
}

void DeviceListWidget::onUpdateDeviceInfo()
{
    //更新DmTreeview  lx
    //设置当前选项
    auto handler = DMDbusHandler::instance();
    m_num = handler->getCurPartititonInfo().m_partitionNumber;
    m_deviceNum = m_treeView->getCurrentTopNum();
    m_devicePath = handler->getCurPartititonInfo().m_devicePath;
    m_treeView->m_model->clear();
    DeviceInfoMap infoMap = DMDbusHandler::instance()->probDeviceInfo();
    QMap<int, int> countMap;
    int deviceCount = 0;

    for (auto devInfo = infoMap.begin(); devInfo != infoMap.end(); devInfo++) {
        DeviceInfo info = devInfo.value();

        if (info.m_path.isEmpty()) {
            continue;
        }

        QString diskSize = Utils::formatSize(info.m_length, info.m_sectorSize);
        auto diskinfoBox = new DmDiskinfoBox(0, this, info.m_path, diskSize);
        int partitionCount = 0;

        for (auto it = info.m_partition.begin(); it != info.m_partition.end(); it++) {
            QString partitionSize = Utils::formatSize(it->m_sectorEnd - it->m_sectorStart + 1, it->m_sectorSize);
            QString partitionPath = it->m_path;
            if ("unallocated" != partitionPath) {
                partitionPath = partitionPath.remove(0, 5);
            }

            QString used = Utils::formatSize(it->m_sectorsUsed, it->m_sectorSize);
            QString unused = Utils::formatSize(it->m_sectorsUnused, it->m_sectorSize);

            FSType fstype = static_cast<FSType>(it->m_fileSystemType);
            QString fstypeName = Utils::fileSystemTypeToString(fstype);
            QString mountpoints;

            for (int i = 0; i < it->m_mountPoints.size(); i++) {
                mountpoints += it->m_mountPoints[i];
            }

            QString fileSystemLabel = it->m_fileSystemLabel;
            auto childDiskinfoBox = new DmDiskinfoBox(1, this, it->m_devicePath, "", partitionPath, partitionSize, used, unused,
                                                      it->m_sectorsUnallocated, it->m_sectorStart, it->m_sectorEnd, fstypeName,
                                                      mountpoints, fileSystemLabel, it->m_flag);
            diskinfoBox->m_childs.append(childDiskinfoBox);

            partitionCount ++;
        }

        m_treeView->addTopItem(diskinfoBox, m_addItem);

        countMap[deviceCount] = partitionCount;
        deviceCount ++;
    }

    QStringList deviceNameList = DMDbusHandler::instance()->getDeviceNameList();

    if (deviceNameList.indexOf(DMDbusHandler::instance()->getCurDevicePath()) == -1) {
        m_flag = 0;
    } else {
        m_deviceNum = deviceNameList.indexOf(DMDbusHandler::instance()->getCurDevicePath());
    }

    m_addItem = 1;

    if (m_flag == 0) {
        for (int i = 0; i < deviceNameList.count(); i++) {
            if (countMap.value(i) != 0) {
                m_treeView->setRefreshItem(i, 0);
                break;
            }
        }
//        m_treeView->setDefaultdmItem();
    } else {
        int index = m_treeView->getCurrentNum();
        if (countMap.value(m_deviceNum) == index) {
            index = m_treeView->getCurrentNum() - 1;
        }

        if (countMap.value(m_deviceNum) < index) {
            index = countMap.value(m_deviceNum) - 1;
        }

        if (m_treeView->getCurrentTopNum() == -1) {
            if (deviceNameList.indexOf(DMDbusHandler::instance()->getCurDevicePath()) == -1) {
                m_treeView->setDefaultdmItem();
            } else {
                m_treeView->setRefreshItem(m_treeView->getCurrentTopNum(), deviceNameList.indexOf(DMDbusHandler::instance()->getCurDevicePath()));
            }

        } else {
            m_treeView->setRefreshItem(m_deviceNum, index);
        }
    }

    m_flag += 1;
}
