/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Multimedia Card (MMC) / Secure Degital (SD) Card Controller */

/*
 * For most of the comments, infos, ... see:
 * TI AM1808/AM1810 ARM Microprocessor - Technical Reference Manual
 * (SPRUH82A - December 2011), Chapter 26, pages 1225...
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	/* Control Register, 26.4.1, 1256 */
	uint8_t permdx;
	uint8_t permdr;
	uint8_t width;
	uint8_t dateg;
	uint8_t cmdrst;
	uint8_t datrst;

	/* Memory Clock Control Register, 26.4.2, 1257 */
	uint8_t div4;
	uint8_t clken;
	uint8_t clkrt;

	/* Status Register 0, 26.4.3, 1258 */
	/* Bit 31-14: Reserved */
	uint8_t ccs;
	uint8_t trndne;
	uint8_t dated;
	uint8_t drrdy;
	uint8_t dxrdy;
	/* Bit 8: Reserved */
	uint8_t crcrs;
	uint8_t crcrd;
	uint8_t crcwr;
	uint8_t toutrs;
	uint8_t toutrd;
	uint8_t rspdne;
	uint8_t bsydne;
	uint8_t datdne;

	/* Interrupt Mask Register, 26.4.5, 1261 */
	/* Bit 31-14: Reserved */
	uint8_t eccs;
	uint8_t etrndne;
	uint8_t edated;
	uint8_t edrrdy;
	uint8_t edxrdy;
	/* Bit 8: Reserved */
	uint8_t ecrcrs;
	uint8_t ecrcrd;
	uint8_t ecrcwr;
	uint8_t etoutrs;
	uint8_t etoutrd;
	uint8_t erspdne;
	uint8_t ebsydne;
	uint8_t edatdne;

	/* Response Time-Out Register, 26.4.6, 1263 */
	uint32_t tod;
	uint8_t tor;

	/* Data Read Time-Out Register, 26.4.7, 1264 */
	/* tod - See above. */

	/* Block Length Register, 26.4.8, 1265 */
	/* Bit 31-12: Reserved */
	uint16_t blen;

	/* Number of Blocks Register, 26.4.9, 1266 */
	/* Bit 31-16: Reserved */
	uint16_t nblk;

	/* FIXME */

	/* Command Register, 26.4.13, 1268 */
	/* Bit 31-17: Reserved */
	uint8_t dmatrig;
	uint8_t dclr;
	uint8_t initck;
	uint8_t wdatx;
	uint8_t strmtp;
	uint8_t dtrw;
	uint8_t rspfmt;
	uint8_t bsyexp;
	uint8_t pplen;
	/* Bit 6: Reserved */
	uint8_t cmd;

	/* Argument Register, 26.4.14, 1270 */
	uint32_t arg;

	/* FIFO Control Register, 26.4.22, 1277 */
	/* Bit 31-5: Reserved */
	uint8_t accwd;
	uint8_t fifolev;
	uint8_t fifodir;
	uint8_t fiforst;

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

static void
NAME_(cmdreset)(struct cpssp *cpssp)
{
	/* FIXME */
}

static void
NAME_(datreset)(struct cpssp *cpssp)
{
	/* FIXME */
}

static void
NAME_(fiforeset)(struct cpssp *cpssp)
{
	/* FIXME */
}

static void
NAME_(cmd)(struct cpssp *cpssp)
{
	fprintf(stderr, "%s: cmd 0x%02x\n", __FUNCTION__, cpssp->NAME.cmd);

	cpssp->NAME.toutrs = 1;
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	assert(bs == 0b1111);

	addr &= 0xfff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		/* Control Register, 26.4.1, 1256 */
		/* Bit 31-11: Reserved */
		cpssp->NAME.permdx = (val >> 10) & 1;
			assert(cpssp->NAME.permdx == 0);
		cpssp->NAME.permdr = (val >> 9) & 1;
			assert(cpssp->NAME.permdr == 0);
		cpssp->NAME.width &= ~0b10;
		cpssp->NAME.width |= ((val >> 8) & 1) << 1;
		cpssp->NAME.dateg = (val >> 6) & 0x3;
		/* Bit 5-3: Reserved */
		cpssp->NAME.width &= ~0b01;
		cpssp->NAME.width |= ((val >> 2) & 1) << 0;
		cpssp->NAME.cmdrst = (val >> 1) & 1;
		cpssp->NAME.datrst = (val >> 0) & 1;
		if (cpssp->NAME.cmdrst) {
			NAME_(cmdreset)(cpssp);
		}
		if (cpssp->NAME.datrst) {
			NAME_(datreset)(cpssp);
		}
		break;
	case 0x004:
		/* Memory Clock Control Register, 26.4.2, 1257 */
		/* Bit 31-10: Reserved */
		cpssp->NAME.div4 = (val >> 9) & 1;
		cpssp->NAME.clken = (val >> 8) & 1;
		cpssp->NAME.clkrt = (val >> 0) & 0xff;
		break;
	case 0x008:
		/* Status Register 0, 26.4.3, 1258 */
		goto read_only;
	case 0x00c:
		/* Status Register 1, 26.4.4, 1260 */
		goto read_only;
	case 0x010:
		/* Interrupt Mask Register, 26.4.5, 1261 */
		/* Bit 31-14: Reserved */
		cpssp->NAME.eccs = (val >> 13) & 1;
		cpssp->NAME.etrndne = (val >> 12) & 1;
		cpssp->NAME.edated = (val >> 11) & 1;
		cpssp->NAME.edrrdy = (val >> 10) & 1;
		cpssp->NAME.edxrdy = (val >> 9) & 1;
		/* Bit 8: Reserved */
		cpssp->NAME.ecrcrs = (val >> 7) & 1;
		cpssp->NAME.ecrcrd = (val >> 6) & 1;
		cpssp->NAME.ecrcwr = (val >> 5) & 1;
		cpssp->NAME.etoutrs = (val >> 4) & 1;
		cpssp->NAME.etoutrd = (val >> 3) & 1;
		cpssp->NAME.erspdne = (val >> 2) & 1;
		cpssp->NAME.ebsydne = (val >> 1) & 1;
		cpssp->NAME.edatdne = (val >> 0) & 1;
		break;
	case 0x014:
		/* Response Time-Out Register, 26.4.6, 1263 */
		/* Bit 31-18: Reserved */
		cpssp->NAME.tod &= ~(0x03ff << 16);
		cpssp->NAME.tod |= ((val >> 8) & 0x3ff) << 16;
		cpssp->NAME.tor = (val >> 0) & 0xff;
		break;
	case 0x018:
		/* Data Read Time-Out Register, 26.4.7, 1264 */
		/* Bit 31-16: Reserved */
		cpssp->NAME.tod &= ~0xffff;
		cpssp->NAME.tod |= (val >> 0) & 0xffff;
		break;
	case 0x01c:
		/* Block Length Register, 26.4.8, 1265 */
		/* Bit 31-12: Reserved */
		cpssp->NAME.blen = (val >> 0) & 0x3ff;
		break;
	case 0x020:
		/* Number of Blocks Register, 26.4.9, 1266 */
		/* Bit 31-16: Reserved */
		cpssp->NAME.nblk = (val >> 0) & 0xffff;
		break;
	/* FIXME */
	case 0x030:
		/* Command Register, 26.4.13, 1268 */
		/* Bit 31-17: Reserved */
		cpssp->NAME.dmatrig = (val >> 16) & 1;
		cpssp->NAME.dclr = (val >> 15) & 1;
		cpssp->NAME.initck = (val >> 14) & 1;
		cpssp->NAME.wdatx = (val >> 13) & 1;
		cpssp->NAME.strmtp = (val >> 12) & 1;
		cpssp->NAME.dtrw = (val >> 11) & 1;
		cpssp->NAME.rspfmt = (val >> 9) & 3;
		cpssp->NAME.bsyexp = (val >> 8) & 1;
		cpssp->NAME.pplen = (val >> 7) & 1;
		/* Bit 6: Reserved */
		cpssp->NAME.cmd = (val >> 16) & 0x3f;
		NAME_(cmd)(cpssp);
		break;
	case 0x034:
		/* Argument Register, 26.4.14, 1270 */
		cpssp->NAME.arg = (val >> 0) & 0xffffffff;
		break;
	/* FIXME */
	case 0x074:
		/* FIFO Control Register, 26.4.22, 1277 */
		/* Bit 31-5: Reserved */
		cpssp->NAME.accwd = (val >> 3) & 0x3;
		cpssp->NAME.fifolev = (val >> 2) & 1;
		cpssp->NAME.fifodir = (val >> 1) & 1;
		cpssp->NAME.fiforst = (val >> 0) & 1;
		if (cpssp->NAME.fiforst) {
			NAME_(fiforeset)(cpssp);
		}
		break;
	default:
		cpssp->NAME.reg[addr >> 2] = val;
	read_only:;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	assert(bs == 0b1111);

	addr &= 0xfff;

	switch (addr) {
	case 0x000:
		/* Control Register, 26.4.1, 1256 */
		*valp = 0;
		/* Bit 31-11: Reserved */
		*valp |= cpssp->NAME.permdx << 10;
		*valp |= cpssp->NAME.permdr << 9;
		*valp |= ((cpssp->NAME.width >> 1) & 1) << 8;
		*valp |= cpssp->NAME.dateg << 6;
		/* Bit 5-3: Reserved */
		*valp |= ((cpssp->NAME.width >> 0) & 1) << 2;
		*valp |= cpssp->NAME.cmdrst << 1;
		*valp |= cpssp->NAME.datrst << 0;
		break;
	case 0x004:
		/* Memory Clock Control Register, 26.4.2, 1257 */
		*valp = 0;
		/* Bit 31-10: Reserved */
		*valp |= cpssp->NAME.div4 << 9;
		*valp |= cpssp->NAME.clken << 8;
		*valp |= cpssp->NAME.clkrt << 0;
		break;
	case 0x008:
		/* Status Register 0, 26.4.3, 1258 */
		*valp = 0;
		/* Bit 31-14: Reserved */
		*valp |= cpssp->NAME.ccs << 13;
			cpssp->NAME.ccs = 0;
		*valp |= cpssp->NAME.trndne << 12;
			cpssp->NAME.trndne = 0;
		*valp |= cpssp->NAME.dated << 11;
			cpssp->NAME.dated = 0;
		*valp |= cpssp->NAME.drrdy << 10;
		*valp |= cpssp->NAME.dxrdy << 9;
		/* Bit 8: Reserved */
		*valp |= cpssp->NAME.crcrs << 7;
			cpssp->NAME.crcrs = 0;
		*valp |= cpssp->NAME.crcrd << 6;
			cpssp->NAME.crcrd = 0;
		*valp |= cpssp->NAME.crcwr << 5;
			cpssp->NAME.crcwr = 0;
		*valp |= cpssp->NAME.toutrs << 4;
			cpssp->NAME.toutrs = 0;
		*valp |= cpssp->NAME.toutrd << 3;
			cpssp->NAME.toutrd = 0;
		*valp |= cpssp->NAME.rspdne << 2;
			cpssp->NAME.rspdne = 0;
		*valp |= cpssp->NAME.bsydne << 1;
			cpssp->NAME.bsydne = 0;
		*valp |= cpssp->NAME.datdne << 0;
			cpssp->NAME.datdne = 0;
		break;
	case 0x00c:
		/* Status Register 1, 26.4.4, 1260 */
		*valp = 0;
		/* FIXME */
		goto warn;
	case 0x010:
		/* Interrupt Mask Register, 26.4.5, 1261 */
		*valp = 0;
		/* Bit 31-14: Reserved */
		*valp |= cpssp->NAME.eccs << 13;
		*valp |= cpssp->NAME.etrndne << 12;
		*valp |= cpssp->NAME.edated << 11;
		*valp |= cpssp->NAME.edrrdy << 10;
		*valp |= cpssp->NAME.edxrdy << 9;
		/* Bit 8: Reserved */
		*valp |= cpssp->NAME.ecrcrs << 7;
		*valp |= cpssp->NAME.ecrcrd << 6;
		*valp |= cpssp->NAME.ecrcwr << 5;
		*valp |= cpssp->NAME.etoutrs << 4;
		*valp |= cpssp->NAME.etoutrd << 3;
		*valp |= cpssp->NAME.erspdne << 2;
		*valp |= cpssp->NAME.ebsydne << 1;
		*valp |= cpssp->NAME.edatdne << 0;
		goto warn;
	case 0x014:
		/* Response Time-Out Register, 26.4.6, 1263 */
		*valp = 0;
		/* Bit 31-18: Reserved */
		*valp |= ((cpssp->NAME.tod >> 16) & 0x3ff) << 8;
		*valp |= cpssp->NAME.tor << 0;
		break;
	case 0x018:
		/* Data Read Time-Out Register, 26.4.7, 1264 */
		*valp = 0;
		/* Bit 31-16: Reserved */
		*valp |= ((cpssp->NAME.tod >> 0) & 0xffff) << 0;
		break;
	case 0x01c:
		/* Block Length Register, 26.4.8, 1265 */
		*valp = 0;
		/* Bit 31-12: Reserved */
		*valp |= cpssp->NAME.blen << 0;
		break;
	case 0x020:
		/* Number of Blocks Register, 26.4.9, 1266 */
		*valp = 0;
		/* Bit 31-16: Reserved */
		*valp |= cpssp->NAME.nblk << 0;
		break;
	/* FIXME */
	case 0x030:
		/* Command Register, 26.4.13, 1268 */
		*valp = 0;
		/* Bit 31-17: Reserved */
		*valp |= 0 << 16; /* Trigger only */
		*valp |= cpssp->NAME.dclr << 15;
		*valp |= cpssp->NAME.initck << 14;
		*valp |= cpssp->NAME.wdatx << 13;
		*valp |= cpssp->NAME.strmtp << 12;
		*valp |= cpssp->NAME.dtrw << 11;
		*valp |= cpssp->NAME.rspfmt << 9;
		*valp |= cpssp->NAME.bsyexp << 8;
		*valp |= cpssp->NAME.pplen << 7;
		/* Bit 6: Reserved */
		*valp |= cpssp->NAME.cmd << 0;
		break;
	case 0x034:
		/* Argument Register, 26.4.14, 1270 */
		*valp = cpssp->NAME.arg << 0;
		break;
	/* FIXME */
	case 0x074:
		/* FIFO Control Register, 26.4.22, 1277 */
		*valp = 0;
		/* Bit 31-5: Reserved */
		*valp |= cpssp->NAME.accwd << 3;
		*valp |= cpssp->NAME.fifolev << 2;
		*valp |= cpssp->NAME.fifodir << 1;
		*valp |= cpssp->NAME.fiforst << 0;
		break;
	default:
		*valp = cpssp->NAME.reg[addr >> 2];
	warn:	;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* Control Register, 26.4.1, 1256 */
	/* FIXME */
	cpssp->NAME.permdx = 0;
	cpssp->NAME.permdr = 0;
	cpssp->NAME.width = 0;
	cpssp->NAME.dateg = 0;
	cpssp->NAME.cmdrst = 0;
	cpssp->NAME.datrst = 0;

	/* Memory Clock Control Register, 26.4.2, 1257 */
	cpssp->NAME.div4 = 0;
	cpssp->NAME.clken = 0;
	cpssp->NAME.clkrt = 0xff;

	/* Status Register 0, 26.4.3, 1258 */
	/* Bit 31-14: Reserved */
	cpssp->NAME.ccs = 0;
	cpssp->NAME.trndne = 0;
	cpssp->NAME.dated = 0;
	cpssp->NAME.drrdy = 0;
	cpssp->NAME.dxrdy = 0;
	/* Bit 8: Reserved */
	cpssp->NAME.crcrs = 0;
	cpssp->NAME.crcrd = 0;
	cpssp->NAME.crcwr = 0;
	cpssp->NAME.toutrs = 0;
	cpssp->NAME.toutrd = 0;
	cpssp->NAME.rspdne = 0;
	cpssp->NAME.bsydne = 0;
	cpssp->NAME.datdne = 0;

	/* Interrupt Mask Register, 26.4.5, 1261 */
	/* Bit 31-14: Reserved */
	cpssp->NAME.eccs = 0;
	cpssp->NAME.etrndne = 0;
	cpssp->NAME.edated = 0;
	cpssp->NAME.edrrdy = 0;
	cpssp->NAME.edxrdy = 0;
	/* Bit 8: Reserved */
	cpssp->NAME.ecrcrs = 0;
	cpssp->NAME.ecrcrd = 0;
	cpssp->NAME.ecrcwr = 0;
	cpssp->NAME.etoutrs = 0;
	cpssp->NAME.etoutrd = 0;
	cpssp->NAME.erspdne = 0;
	cpssp->NAME.ebsydne = 0;
	cpssp->NAME.edatdne = 0;

	/* Response Time-Out Register, 26.4.6, 1263 */
	cpssp->NAME.tod = 0x0000000;
	cpssp->NAME.tor = 0x00;

	/* Data Read Time-Out Register, 26.4.7, 1264 */
	/* tod - See above. */

	/* Block Length Register, 26.4.8, 1265 */
	/* Bit 31-12: Reserved */
	cpssp->NAME.blen = 0x200;

	/* Number of Blocks Register, 26.4.9, 1266 */
	/* Bit 31-16: Reserved */
	cpssp->NAME.nblk = 0;

	/* Command Register, 26.4.13, 1268 */
	/* Bit 31-17: Reserved */
	cpssp->NAME.dmatrig = 0;
	cpssp->NAME.dclr = 0;
	cpssp->NAME.initck = 0;
	cpssp->NAME.wdatx = 0;
	cpssp->NAME.strmtp = 0;
	cpssp->NAME.dtrw = 0;
	cpssp->NAME.rspfmt = 0x0;
	cpssp->NAME.bsyexp = 0;
	cpssp->NAME.pplen = 0;
	/* Bit 6: Reserved */
	cpssp->NAME.cmd = 0x00;

	/* Argument Register, 26.4.14, 1270 */
	cpssp->NAME.arg = 0x00000000;

	/* FIFO Control Register, 26.4.22, 1277 */
	/* Bit 31-5: Reserved */
	cpssp->NAME.accwd = 0;
	cpssp->NAME.fifolev = 0;
	cpssp->NAME.fifodir = 0;
	cpssp->NAME.fiforst = 0;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
