// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/cert/ev_root_ca_metadata.h"

#if defined(OS_WIN)
#include <stdlib.h>
#endif

#include <algorithm>

#include "base/lazy_instance.h"
#include "base/logging.h"
#include "base/strings/string_piece.h"
#include "net/der/input.h"
#if defined(PLATFORM_USES_CHROMIUM_EV_METADATA)
#include "third_party/boringssl/src/include/openssl/bytestring.h"
#include "third_party/boringssl/src/include/openssl/mem.h"
#endif

namespace net {

namespace {
#if defined(PLATFORM_USES_CHROMIUM_EV_METADATA)
// Raw metadata.
struct EVMetadata {
  // kMaxOIDsPerCA is the number of OIDs that we can support per root CA. At
  // least one CA has different EV policies for business vs government
  // entities and, in the case of cross-signing, we might need to list another
  // CA's policy OID under the cross-signing root.
  static const size_t kMaxOIDsPerCA = 2;

  // The SHA-256 fingerprint of the root CA certificate, used as a unique
  // identifier for a root CA certificate.
  SHA256HashValue fingerprint;

  // The EV policy OIDs of the root CA.
  const base::StringPiece policy_oids[kMaxOIDsPerCA];
};

// These certificates may be found in net/data/ssl/ev_roots.
static const EVMetadata kEvRootCaMetadata[] = {
    // AC Camerfirma S.A. Chambers of Commerce Root - 2008
    // https://www.camerfirma.com
    {
        {{0x06, 0x3e, 0x4a, 0xfa, 0xc4, 0x91, 0xdf, 0xd3, 0x32, 0xf3, 0x08,
          0x9b, 0x85, 0x42, 0xe9, 0x46, 0x17, 0xd8, 0x93, 0xd7, 0xfe, 0x94,
          0x4e, 0x10, 0xa7, 0x93, 0x7e, 0xe2, 0x9d, 0x96, 0x93, 0xc0}},
        {
            // AC Camerfirma uses the last two arcs to track how the private key
            // is managed - the effective verification policy is the same.
            "1.3.6.1.4.1.17326.10.14.2.1.2",
            "1.3.6.1.4.1.17326.10.14.2.2.2",
        },
    },
    // AddTrust External CA Root
    // https://addtrustexternalcaroot-ev.comodoca.com
    {
        {{0x68, 0x7f, 0xa4, 0x51, 0x38, 0x22, 0x78, 0xff, 0xf0, 0xc8, 0xb1,
          0x1f, 0x8d, 0x43, 0xd5, 0x76, 0x67, 0x1c, 0x6e, 0xb2, 0xbc, 0xea,
          0xb4, 0x13, 0xfb, 0x83, 0xd9, 0x65, 0xd0, 0x6d, 0x2f, 0xf2}},
        {
            "1.3.6.1.4.1.6449.1.2.1.5.1",
            // This is the Network Solutions EV OID. However, this root
            // cross-certifies NetSol and so we need it here too.
            "1.3.6.1.4.1.782.1.2.1.8.1",
        },
    },
    // Actalis Authentication Root CA
    // https://ssltest-a.actalis.it:8443
    {
        {{0x55, 0x92, 0x60, 0x84, 0xec, 0x96, 0x3a, 0x64, 0xb9, 0x6e, 0x2a,
          0xbe, 0x01, 0xce, 0x0b, 0xa8, 0x6a, 0x64, 0xfb, 0xfe, 0xbc, 0xc7,
          0xaa, 0xb5, 0xaf, 0xc1, 0x55, 0xb3, 0x7f, 0xd7, 0x60, 0x66}},
        {"1.3.159.1.17.1", ""},
    },
    // AffirmTrust Commercial
    // https://commercial.affirmtrust.com/
    {
        {{0x03, 0x76, 0xab, 0x1d, 0x54, 0xc5, 0xf9, 0x80, 0x3c, 0xe4, 0xb2,
          0xe2, 0x01, 0xa0, 0xee, 0x7e, 0xef, 0x7b, 0x57, 0xb6, 0x36, 0xe8,
          0xa9, 0x3c, 0x9b, 0x8d, 0x48, 0x60, 0xc9, 0x6f, 0x5f, 0xa7}},
        {"1.3.6.1.4.1.34697.2.1", ""},
    },
    // AffirmTrust Networking
    // https://networking.affirmtrust.com:4431
    {
        {{0x0a, 0x81, 0xec, 0x5a, 0x92, 0x97, 0x77, 0xf1, 0x45, 0x90, 0x4a,
          0xf3, 0x8d, 0x5d, 0x50, 0x9f, 0x66, 0xb5, 0xe2, 0xc5, 0x8f, 0xcd,
          0xb5, 0x31, 0x05, 0x8b, 0x0e, 0x17, 0xf3, 0xf0, 0xb4, 0x1b}},
        {"1.3.6.1.4.1.34697.2.2", ""},
    },
    // AffirmTrust Premium
    // https://premium.affirmtrust.com:4432/
    {
        {{0x70, 0xa7, 0x3f, 0x7f, 0x37, 0x6b, 0x60, 0x07, 0x42, 0x48, 0x90,
          0x45, 0x34, 0xb1, 0x14, 0x82, 0xd5, 0xbf, 0x0e, 0x69, 0x8e, 0xcc,
          0x49, 0x8d, 0xf5, 0x25, 0x77, 0xeb, 0xf2, 0xe9, 0x3b, 0x9a}},
        {"1.3.6.1.4.1.34697.2.3", ""},
    },
    // AffirmTrust Premium ECC
    // https://premiumecc.affirmtrust.com:4433/
    {
        {{0xbd, 0x71, 0xfd, 0xf6, 0xda, 0x97, 0xe4, 0xcf, 0x62, 0xd1, 0x64,
          0x7a, 0xdd, 0x25, 0x81, 0xb0, 0x7d, 0x79, 0xad, 0xf8, 0x39, 0x7e,
          0xb4, 0xec, 0xba, 0x9c, 0x5e, 0x84, 0x88, 0x82, 0x14, 0x23}},
        {"1.3.6.1.4.1.34697.2.4", ""},
    },
    // Amazon Root CA 1
    // https://good.sca1a.amazontrust.com/
    {
        {{0x8e, 0xcd, 0xe6, 0x88, 0x4f, 0x3d, 0x87, 0xb1, 0x12, 0x5b, 0xa3,
          0x1a, 0xc3, 0xfc, 0xb1, 0x3d, 0x70, 0x16, 0xde, 0x7f, 0x57, 0xcc,
          0x90, 0x4f, 0xe1, 0xcb, 0x97, 0xc6, 0xae, 0x98, 0x19, 0x6e}},
        {"2.23.140.1.1", ""},
    },
    // Amazon Root CA 2
    // https://good.sca2a.amazontrust.com/
    {
        {{0x1b, 0xa5, 0xb2, 0xaa, 0x8c, 0x65, 0x40, 0x1a, 0x82, 0x96, 0x01,
          0x18, 0xf8, 0x0b, 0xec, 0x4f, 0x62, 0x30, 0x4d, 0x83, 0xce, 0xc4,
          0x71, 0x3a, 0x19, 0xc3, 0x9c, 0x01, 0x1e, 0xa4, 0x6d, 0xb4}},
        {"2.23.140.1.1", ""},
    },
    // Amazon Root CA 3
    // https://good.sca3a.amazontrust.com/
    {
        {{0x18, 0xce, 0x6c, 0xfe, 0x7b, 0xf1, 0x4e, 0x60, 0xb2, 0xe3, 0x47,
          0xb8, 0xdf, 0xe8, 0x68, 0xcb, 0x31, 0xd0, 0x2e, 0xbb, 0x3a, 0xda,
          0x27, 0x15, 0x69, 0xf5, 0x03, 0x43, 0xb4, 0x6d, 0xb3, 0xa4}},
        {"2.23.140.1.1", ""},
    },
    // Amazon Root CA 4
    // https://good.sca4a.amazontrust.com/
    {
        {{0xe3, 0x5d, 0x28, 0x41, 0x9e, 0xd0, 0x20, 0x25, 0xcf, 0xa6, 0x90,
          0x38, 0xcd, 0x62, 0x39, 0x62, 0x45, 0x8d, 0xa5, 0xc6, 0x95, 0xfb,
          0xde, 0xa3, 0xc2, 0x2b, 0x0b, 0xfb, 0x25, 0x89, 0x70, 0x92}},
        {"2.23.140.1.1", ""},
    },
    // Autoridad de Certificacion Firmaprofesional CIF A62634068
    // https://publifirma.firmaprofesional.com/
    {
        {{0x04, 0x04, 0x80, 0x28, 0xbf, 0x1f, 0x28, 0x64, 0xd4, 0x8f, 0x9a,
          0xd4, 0xd8, 0x32, 0x94, 0x36, 0x6a, 0x82, 0x88, 0x56, 0x55, 0x3f,
          0x3b, 0x14, 0x30, 0x3f, 0x90, 0x14, 0x7f, 0x5d, 0x40, 0xef}},
        {"1.3.6.1.4.1.13177.10.1.3.10", ""},
    },
    // Baltimore CyberTrust Root
    // https://secure.omniroot.com/repository/
    {
        {{0x16, 0xaf, 0x57, 0xa9, 0xf6, 0x76, 0xb0, 0xab, 0x12, 0x60, 0x95,
          0xaa, 0x5e, 0xba, 0xde, 0xf2, 0x2a, 0xb3, 0x11, 0x19, 0xd6, 0x44,
          0xac, 0x95, 0xcd, 0x4b, 0x93, 0xdb, 0xf3, 0xf2, 0x6a, 0xeb}},
        {"1.3.6.1.4.1.6334.1.100.1", ""},
    },
    // Buypass Class 3 Root CA
    // https://valid.evident.ca23.ssl.buypass.no/
    {
        {{0xed, 0xf7, 0xeb, 0xbc, 0xa2, 0x7a, 0x2a, 0x38, 0x4d, 0x38, 0x7b,
          0x7d, 0x40, 0x10, 0xc6, 0x66, 0xe2, 0xed, 0xb4, 0x84, 0x3e, 0x4c,
          0x29, 0xb4, 0xae, 0x1d, 0x5b, 0x93, 0x32, 0xe6, 0xb2, 0x4d}},
        {"2.16.578.1.26.1.3.3", ""},
    },
    // Certum Trusted Network CA
    // https://juice.certum.pl/
    {
        {{0x5c, 0x58, 0x46, 0x8d, 0x55, 0xf5, 0x8e, 0x49, 0x7e, 0x74, 0x39,
          0x82, 0xd2, 0xb5, 0x00, 0x10, 0xb6, 0xd1, 0x65, 0x37, 0x4a, 0xcf,
          0x83, 0xa7, 0xd4, 0xa3, 0x2d, 0xb7, 0x68, 0xc4, 0x40, 0x8e}},
        {"1.2.616.1.113527.2.5.1.1", ""},
    },
    // CFCA EV ROOT
    // https://www.erenepu.com/
    {
        {{0x5c, 0xc3, 0xd7, 0x8e, 0x4e, 0x1d, 0x5e, 0x45, 0x54, 0x7a, 0x04,
          0xe6, 0x87, 0x3e, 0x64, 0xf9, 0x0c, 0xf9, 0x53, 0x6d, 0x1c, 0xcc,
          0x2e, 0xf8, 0x00, 0xf3, 0x55, 0xc4, 0xc5, 0xfd, 0x70, 0xfd}},
        {"2.16.156.112554.3", ""},
    },
    // COMODO Certification Authority
    // https://secure.comodo.com/
    {
        {{0x0c, 0x2c, 0xd6, 0x3d, 0xf7, 0x80, 0x6f, 0xa3, 0x99, 0xed, 0xe8,
          0x09, 0x11, 0x6b, 0x57, 0x5b, 0xf8, 0x79, 0x89, 0xf0, 0x65, 0x18,
          0xf9, 0x80, 0x8c, 0x86, 0x05, 0x03, 0x17, 0x8b, 0xaf, 0x66}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // COMODO Certification Authority (reissued certificate with NotBefore of
    // Jan 1 00:00:00 2011 GMT)
    // https://secure.comodo.com/
    {
        {{0x1a, 0x0d, 0x20, 0x44, 0x5d, 0xe5, 0xba, 0x18, 0x62, 0xd1, 0x9e,
          0xf8, 0x80, 0x85, 0x8c, 0xbc, 0xe5, 0x01, 0x02, 0xb3, 0x6e, 0x8f,
          0x0a, 0x04, 0x0c, 0x3c, 0x69, 0xe7, 0x45, 0x22, 0xfe, 0x6e}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // COMODO ECC Certification Authority
    // https://comodoecccertificationauthority-ev.comodoca.com/
    {
        {{0x17, 0x93, 0x92, 0x7a, 0x06, 0x14, 0x54, 0x97, 0x89, 0xad, 0xce,
          0x2f, 0x8f, 0x34, 0xf7, 0xf0, 0xb6, 0x6d, 0x0f, 0x3a, 0xe3, 0xa3,
          0xb8, 0x4d, 0x21, 0xec, 0x15, 0xdb, 0xba, 0x4f, 0xad, 0xc7}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // COMODO RSA Certification Authority
    // https://comodorsacertificationauthority-ev.comodoca.com/
    {
        {{0x52, 0xf0, 0xe1, 0xc4, 0xe5, 0x8e, 0xc6, 0x29, 0x29, 0x1b, 0x60,
          0x31, 0x7f, 0x07, 0x46, 0x71, 0xb8, 0x5d, 0x7e, 0xa8, 0x0d, 0x5b,
          0x07, 0x27, 0x34, 0x63, 0x53, 0x4b, 0x32, 0xb4, 0x02, 0x34}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // Cybertrust Global Root
    // https://evup.cybertrust.ne.jp/ctj-ev-upgrader/evseal.gif
    {
        {{0x96, 0x0a, 0xdf, 0x00, 0x63, 0xe9, 0x63, 0x56, 0x75, 0x0c, 0x29,
          0x65, 0xdd, 0x0a, 0x08, 0x67, 0xda, 0x0b, 0x9c, 0xbd, 0x6e, 0x77,
          0x71, 0x4a, 0xea, 0xfb, 0x23, 0x49, 0xab, 0x39, 0x3d, 0xa3}},
        {"1.3.6.1.4.1.6334.1.100.1", ""},
    },
    // DigiCert Assured ID Root CA
    // https://assured-id-root-ca.chain-demos.digicert.com/
    {
        {{0x3e, 0x90, 0x99, 0xb5, 0x01, 0x5e, 0x8f, 0x48, 0x6c, 0x00, 0xbc,
          0xea, 0x9d, 0x11, 0x1e, 0xe7, 0x21, 0xfa, 0xba, 0x35, 0x5a, 0x89,
          0xbc, 0xf1, 0xdf, 0x69, 0x56, 0x1e, 0x3d, 0xc6, 0x32, 0x5c}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // DigiCert Assured ID Root G2
    // https://assured-id-root-g2.chain-demos.digicert.com/
    {
        {{0x7d, 0x05, 0xeb, 0xb6, 0x82, 0x33, 0x9f, 0x8c, 0x94, 0x51, 0xee,
          0x09, 0x4e, 0xeb, 0xfe, 0xfa, 0x79, 0x53, 0xa1, 0x14, 0xed, 0xb2,
          0xf4, 0x49, 0x49, 0x45, 0x2f, 0xab, 0x7d, 0x2f, 0xc1, 0x85}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // DigiCert Assured ID Root G3
    // https://assured-id-root-g3.chain-demos.digicert.com/
    {
        {{0x7e, 0x37, 0xcb, 0x8b, 0x4c, 0x47, 0x09, 0x0c, 0xab, 0x36, 0x55,
          0x1b, 0xa6, 0xf4, 0x5d, 0xb8, 0x40, 0x68, 0x0f, 0xba, 0x16, 0x6a,
          0x95, 0x2d, 0xb1, 0x00, 0x71, 0x7f, 0x43, 0x05, 0x3f, 0xc2}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // DigiCert Global Root CA
    // https://global-root-ca.chain-demos.digicert.com/
    {
        {{0x43, 0x48, 0xa0, 0xe9, 0x44, 0x4c, 0x78, 0xcb, 0x26, 0x5e, 0x05,
          0x8d, 0x5e, 0x89, 0x44, 0xb4, 0xd8, 0x4f, 0x96, 0x62, 0xbd, 0x26,
          0xdb, 0x25, 0x7f, 0x89, 0x34, 0xa4, 0x43, 0xc7, 0x01, 0x61}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // DigiCert Global Root G2
    // https://global-root-g2.chain-demos.digicert.com/
    {
        {{0xcb, 0x3c, 0xcb, 0xb7, 0x60, 0x31, 0xe5, 0xe0, 0x13, 0x8f, 0x8d,
          0xd3, 0x9a, 0x23, 0xf9, 0xde, 0x47, 0xff, 0xc3, 0x5e, 0x43, 0xc1,
          0x14, 0x4c, 0xea, 0x27, 0xd4, 0x6a, 0x5a, 0xb1, 0xcb, 0x5f}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // DigiCert Global Root G3
    // https://global-root-g3.chain-demos.digicert.com/
    {
        {{0x31, 0xad, 0x66, 0x48, 0xf8, 0x10, 0x41, 0x38, 0xc7, 0x38, 0xf3,
          0x9e, 0xa4, 0x32, 0x01, 0x33, 0x39, 0x3e, 0x3a, 0x18, 0xcc, 0x02,
          0x29, 0x6e, 0xf9, 0x7c, 0x2a, 0xc9, 0xef, 0x67, 0x31, 0xd0}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // DigiCert High Assurance EV Root CA
    // https://www.digicert.com
    {
        {{0x74, 0x31, 0xe5, 0xf4, 0xc3, 0xc1, 0xce, 0x46, 0x90, 0x77, 0x4f,
          0x0b, 0x61, 0xe0, 0x54, 0x40, 0x88, 0x3b, 0xa9, 0xa0, 0x1e, 0xd0,
          0x0b, 0xa6, 0xab, 0xd7, 0x80, 0x6e, 0xd3, 0xb1, 0x18, 0xcf}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // DigiCert Trusted Root G4
    // https://trusted-root-g4.chain-demos.digicert.com/
    {
        {{0x55, 0x2f, 0x7b, 0xdc, 0xf1, 0xa7, 0xaf, 0x9e, 0x6c, 0xe6, 0x72,
          0x01, 0x7f, 0x4f, 0x12, 0xab, 0xf7, 0x72, 0x40, 0xc7, 0x8e, 0x76,
          0x1a, 0xc2, 0x03, 0xd1, 0xd9, 0xd2, 0x0a, 0xc8, 0x99, 0x88}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // D-TRUST Root Class 3 CA 2 EV 2009
    // https://certdemo-ev-valid.ssl.d-trust.net/
    {
        {{0xee, 0xc5, 0x49, 0x6b, 0x98, 0x8c, 0xe9, 0x86, 0x25, 0xb9, 0x34,
          0x09, 0x2e, 0xec, 0x29, 0x08, 0xbe, 0xd0, 0xb0, 0xf3, 0x16, 0xc2,
          0xd4, 0x73, 0x0c, 0x84, 0xea, 0xf1, 0xf3, 0xd3, 0x48, 0x81}},
        {"1.3.6.1.4.1.4788.2.202.1", ""},
    },
    // emSign Root CA - G1
    // https://testevg1.emsign.com/
    {
        {{0x40, 0xf6, 0xaf, 0x03, 0x46, 0xa9, 0x9a, 0xa1, 0xcd, 0x1d, 0x55,
          0x5a, 0x4e, 0x9c, 0xce, 0x62, 0xc7, 0xf9, 0x63, 0x46, 0x03, 0xee,
          0x40, 0x66, 0x15, 0x83, 0x3d, 0xc8, 0xc8, 0xd0, 0x03, 0x67}},
        {"2.23.140.1.1", ""},
    },
    // Entrust Root Certification Authority
    // https://www.entrust.net/
    {
        {{0x73, 0xc1, 0x76, 0x43, 0x4f, 0x1b, 0xc6, 0xd5, 0xad, 0xf4, 0x5b,
          0x0e, 0x76, 0xe7, 0x27, 0x28, 0x7c, 0x8d, 0xe5, 0x76, 0x16, 0xc1,
          0xe6, 0xe6, 0x14, 0x1a, 0x2b, 0x2c, 0xbc, 0x7d, 0x8e, 0x4c}},
        {"2.16.840.1.114028.10.1.2", ""},
    },
    // Entrust Root Certification Authority – G2
    // https://validg2.entrust.net
    {
        {{0x43, 0xdf, 0x57, 0x74, 0xb0, 0x3e, 0x7f, 0xef, 0x5f, 0xe4, 0x0d,
          0x93, 0x1a, 0x7b, 0xed, 0xf1, 0xbb, 0x2e, 0x6b, 0x42, 0x73, 0x8c,
          0x4e, 0x6d, 0x38, 0x41, 0x10, 0x3d, 0x3a, 0xa7, 0xf3, 0x39}},
        {"2.16.840.1.114028.10.1.2", ""},
    },
    // Entrust Root Certification Authority – EC1
    // https://validec.entrust.net
    {
        {{0x02, 0xed, 0x0e, 0xb2, 0x8c, 0x14, 0xda, 0x45, 0x16, 0x5c, 0x56,
          0x67, 0x91, 0x70, 0x0d, 0x64, 0x51, 0xd7, 0xfb, 0x56, 0xf0, 0xb2,
          0xab, 0x1d, 0x3b, 0x8e, 0xb0, 0x70, 0xe5, 0x6e, 0xdf, 0xf5}},
        {"2.16.840.1.114028.10.1.2", ""},
    },
    // E-Tugra Certification Authority
    // https://sslev.e-tugra.com.tr
    {
        {{0xb0, 0xbf, 0xd5, 0x2b, 0xb0, 0xd7, 0xd9, 0xbd, 0x92, 0xbf, 0x5d,
          0x4d, 0xc1, 0x3d, 0xa2, 0x55, 0xc0, 0x2c, 0x54, 0x2f, 0x37, 0x83,
          0x65, 0xea, 0x89, 0x39, 0x11, 0xf5, 0x5e, 0x55, 0xf2, 0x3c}},
        {"2.16.792.3.0.4.1.1.4", ""},
    },
    // GlobalSign Root CA
    {
        {{0xeb, 0xd4, 0x10, 0x40, 0xe4, 0xbb, 0x3e, 0xc7, 0x42, 0xc9, 0xe3,
          0x81, 0xd3, 0x1e, 0xf2, 0xa4, 0x1a, 0x48, 0xb6, 0x68, 0x5c, 0x96,
          0xe7, 0xce, 0xf3, 0xc1, 0xdf, 0x6c, 0xd4, 0x33, 0x1c, 0x99}},
        {"1.3.6.1.4.1.4146.1.1", ""},
    },
    // GlobalSign Root CA - R3
    // https://2029.globalsign.com/
    {
        {{0xcb, 0xb5, 0x22, 0xd7, 0xb7, 0xf1, 0x27, 0xad, 0x6a, 0x01, 0x13,
          0x86, 0x5b, 0xdf, 0x1c, 0xd4, 0x10, 0x2e, 0x7d, 0x07, 0x59, 0xaf,
          0x63, 0x5a, 0x7c, 0xf4, 0x72, 0x0d, 0xc9, 0x63, 0xc5, 0x3b}},
        {"1.3.6.1.4.1.4146.1.1", ""},
    },
    // GlobalSign ECC Root CA - R5
    // https://2038r5.globalsign.com/
    {
        {{0x17, 0x9f, 0xbc, 0x14, 0x8a, 0x3d, 0xd0, 0x0f, 0xd2, 0x4e, 0xa1,
          0x34, 0x58, 0xcc, 0x43, 0xbf, 0xa7, 0xf5, 0x9c, 0x81, 0x82, 0xd7,
          0x83, 0xa5, 0x13, 0xf6, 0xeb, 0xec, 0x10, 0x0c, 0x89, 0x24}},
        {"1.3.6.1.4.1.4146.1.1", ""},
    },
    // Go Daddy Class 2 Certification Authority
    // https://www.godaddy.com/
    {
        {{0xc3, 0x84, 0x6b, 0xf2, 0x4b, 0x9e, 0x93, 0xca, 0x64, 0x27, 0x4c,
          0x0e, 0xc6, 0x7c, 0x1e, 0xcc, 0x5e, 0x02, 0x4f, 0xfc, 0xac, 0xd2,
          0xd7, 0x40, 0x19, 0x35, 0x0e, 0x81, 0xfe, 0x54, 0x6a, 0xe4}},
        {"2.16.840.1.114413.1.7.23.3", ""},
    },
    // Go Daddy Root Certificate Authority - G2
    // https://valid.gdig2.catest.godaddy.com/
    {
        {{0x45, 0x14, 0x0b, 0x32, 0x47, 0xeb, 0x9c, 0xc8, 0xc5, 0xb4, 0xf0,
          0xd7, 0xb5, 0x30, 0x91, 0xf7, 0x32, 0x92, 0x08, 0x9e, 0x6e, 0x5a,
          0x63, 0xe2, 0x74, 0x9d, 0xd3, 0xac, 0xa9, 0x19, 0x8e, 0xda}},
        {"2.16.840.1.114413.1.7.23.3", ""},
    },
    // Hongkong Post Root CA 3
    // https://valid-ev.ecert.gov.hk/
    {
        {{0x5a, 0x2f, 0xc0, 0x3f, 0x0c, 0x83, 0xb0, 0x90, 0xbb, 0xfa, 0x40,
          0x60, 0x4b, 0x09, 0x88, 0x44, 0x6c, 0x76, 0x36, 0x18, 0x3d, 0xf9,
          0x84, 0x6e, 0x17, 0x10, 0x1a, 0x44, 0x7f, 0xb8, 0xef, 0xd6}},
        {"2.23.140.1.1", ""},
    },
    // Izenpe.com - SHA256 root
    // The first OID is for businesses and the second for government entities.
    // These are the test sites, respectively:
    // https://servicios.izenpe.com
    // https://servicios1.izenpe.com
    {
        {{0x25, 0x30, 0xcc, 0x8e, 0x98, 0x32, 0x15, 0x02, 0xba, 0xd9, 0x6f,
          0x9b, 0x1f, 0xba, 0x1b, 0x09, 0x9e, 0x2d, 0x29, 0x9e, 0x0f, 0x45,
          0x48, 0xbb, 0x91, 0x4f, 0x36, 0x3b, 0xc0, 0xd4, 0x53, 0x1f}},
        {"1.3.6.1.4.1.14777.6.1.1", "1.3.6.1.4.1.14777.6.1.2"},
    },
    // Izenpe.com - SHA1 root
    // Windows XP finds this, SHA1, root instead. The policy OIDs are the same
    // as for the SHA256 root, above.
    {
        {{0x23, 0x80, 0x42, 0x03, 0xca, 0x45, 0xd8, 0xcd, 0xe7, 0x16, 0xb8,
          0xc1, 0x3b, 0xf3, 0xb4, 0x48, 0x45, 0x7f, 0xa0, 0x6c, 0xc1, 0x02,
          0x50, 0x99, 0x7f, 0xa0, 0x14, 0x58, 0x31, 0x7c, 0x41, 0xe5}},
        {"1.3.6.1.4.1.14777.6.1.1", "1.3.6.1.4.1.14777.6.1.2"},
    },
    // LuxTrust Global Root 2
    // https://ltsslca5.trustme.lu/
    {
        {{0x54, 0x45, 0x5f, 0x71, 0x29, 0xc2, 0x0b, 0x14, 0x47, 0xc4, 0x18,
          0xf9, 0x97, 0x16, 0x8f, 0x24, 0xc5, 0x8f, 0xc5, 0x02, 0x3b, 0xf5,
          0xda, 0x5b, 0xe2, 0xeb, 0x6e, 0x1d, 0xd8, 0x90, 0x2e, 0xd5}},
        {"1.3.171.1.1.10.5.2", ""},
    },
    // Network Solutions Certificate Authority
    // https://www.networksolutions.com/website-packages/index.jsp
    {
        {{0x15, 0xf0, 0xba, 0x00, 0xa3, 0xac, 0x7a, 0xf3, 0xac, 0x88, 0x4c,
          0x07, 0x2b, 0x10, 0x11, 0xa0, 0x77, 0xbd, 0x77, 0xc0, 0x97, 0xf4,
          0x01, 0x64, 0xb2, 0xf8, 0x59, 0x8a, 0xbd, 0x83, 0x86, 0x0c}},
        {"1.3.6.1.4.1.782.1.2.1.8.1", ""},
    },
    // Network Solutions Certificate Authority (reissued certificate with
    // NotBefore of Jan  1 00:00:00 2011 GMT).
    // https://www.networksolutions.com/website-packages/index.jsp
    {
        {{0x00, 0x16, 0x86, 0xcd, 0x18, 0x1f, 0x83, 0xa1, 0xb1, 0x21, 0x7d,
          0x30, 0x5b, 0x36, 0x5c, 0x41, 0xe3, 0x47, 0x0a, 0x78, 0xa1, 0xd3,
          0x7b, 0x13, 0x4a, 0x98, 0xcd, 0x54, 0x7b, 0x92, 0xda, 0xb3}},
        {"1.3.6.1.4.1.782.1.2.1.8.1", ""},
    },
    // OISTE WISeKey Global Root GB CA
    // https://goodevssl.wisekey.com
    {
        {{0x6b, 0x9c, 0x08, 0xe8, 0x6e, 0xb0, 0xf7, 0x67, 0xcf, 0xad, 0x65,
          0xcd, 0x98, 0xb6, 0x21, 0x49, 0xe5, 0x49, 0x4a, 0x67, 0xf5, 0x84,
          0x5e, 0x7b, 0xd1, 0xed, 0x01, 0x9f, 0x27, 0xb8, 0x6b, 0xd6}},
        {"2.16.756.5.14.7.4.8", ""},
    },
    // QuoVadis Root CA 2
    // https://www.quovadis.bm/
    {
        {{0x85, 0xa0, 0xdd, 0x7d, 0xd7, 0x20, 0xad, 0xb7, 0xff, 0x05, 0xf8,
          0x3d, 0x54, 0x2b, 0x20, 0x9d, 0xc7, 0xff, 0x45, 0x28, 0xf7, 0xd6,
          0x77, 0xb1, 0x83, 0x89, 0xfe, 0xa5, 0xe5, 0xc4, 0x9e, 0x86}},
        {"1.3.6.1.4.1.8024.0.2.100.1.2", ""},
    },
    // QuoVadis Root CA 2 G3
    // https://evsslicag3-v.quovadisglobal.com/
    {
        {{0x8f, 0xe4, 0xfb, 0x0a, 0xf9, 0x3a, 0x4d, 0x0d, 0x67, 0xdb, 0x0b,
          0xeb, 0xb2, 0x3e, 0x37, 0xc7, 0x1b, 0xf3, 0x25, 0xdc, 0xbc, 0xdd,
          0x24, 0x0e, 0xa0, 0x4d, 0xaf, 0x58, 0xb4, 0x7e, 0x18, 0x40}},
        {"1.3.6.1.4.1.8024.0.2.100.1.2", ""},
    },
    // SecureTrust CA, SecureTrust Corporation
    // https://www.securetrust.com
    // https://www.trustwave.com/
    {
        {{0xf1, 0xc1, 0xb5, 0x0a, 0xe5, 0xa2, 0x0d, 0xd8, 0x03, 0x0e, 0xc9,
          0xf6, 0xbc, 0x24, 0x82, 0x3d, 0xd3, 0x67, 0xb5, 0x25, 0x57, 0x59,
          0xb4, 0xe7, 0x1b, 0x61, 0xfc, 0xe9, 0xf7, 0x37, 0x5d, 0x73}},
        {"2.16.840.1.114404.1.1.2.4.1", ""},
    },
    // Secure Global CA, SecureTrust Corporation
    {
        {{0x42, 0x00, 0xf5, 0x04, 0x3a, 0xc8, 0x59, 0x0e, 0xbb, 0x52, 0x7d,
          0x20, 0x9e, 0xd1, 0x50, 0x30, 0x29, 0xfb, 0xcb, 0xd4, 0x1c, 0xa1,
          0xb5, 0x06, 0xec, 0x27, 0xf1, 0x5a, 0xde, 0x7d, 0xac, 0x69}},
        {"2.16.840.1.114404.1.1.2.4.1", ""},
    },
    // Security Communication RootCA1
    // https://www.secomtrust.net/contact/form.html
    {
        {{0xe7, 0x5e, 0x72, 0xed, 0x9f, 0x56, 0x0e, 0xec, 0x6e, 0xb4, 0x80,
          0x00, 0x73, 0xa4, 0x3f, 0xc3, 0xad, 0x19, 0x19, 0x5a, 0x39, 0x22,
          0x82, 0x01, 0x78, 0x95, 0x97, 0x4a, 0x99, 0x02, 0x6b, 0x6c}},
        {"1.2.392.200091.100.721.1", ""},
    },
    // Security Communication EV RootCA2
    // https://www.secomtrust.net/contact/form.html
    {
        {{0x51, 0x3b, 0x2c, 0xec, 0xb8, 0x10, 0xd4, 0xcd, 0xe5, 0xdd, 0x85,
          0x39, 0x1a, 0xdf, 0xc6, 0xc2, 0xdd, 0x60, 0xd8, 0x7b, 0xb7, 0x36,
          0xd2, 0xb5, 0x21, 0x48, 0x4a, 0xa4, 0x7a, 0x0e, 0xbe, 0xf6}},
        {"1.2.392.200091.100.721.1", ""},
    },
    // SSL.com EV Root Certification Authority ECC
    // https://test-ev-ecc.ssl.com/
    {
        {{0x22, 0xa2, 0xc1, 0xf7, 0xbd, 0xed, 0x70, 0x4c, 0xc1, 0xe7, 0x01,
          0xb5, 0xf4, 0x08, 0xc3, 0x10, 0x88, 0x0f, 0xe9, 0x56, 0xb5, 0xde,
          0x2a, 0x4a, 0x44, 0xf9, 0x9c, 0x87, 0x3a, 0x25, 0xa7, 0xc8}},
        {"2.23.140.1.1", ""},
    },
    // SSL.com EV Root Certification Authority RSA R2
    // https://test-ev-rsa.ssl.com/
    {
        {{0x2e, 0x7b, 0xf1, 0x6c, 0xc2, 0x24, 0x85, 0xa7, 0xbb, 0xe2, 0xaa,
          0x86, 0x96, 0x75, 0x07, 0x61, 0xb0, 0xae, 0x39, 0xbe, 0x3b, 0x2f,
          0xe9, 0xd0, 0xcc, 0x6d, 0x4e, 0xf7, 0x34, 0x91, 0x42, 0x5c}},
        {"2.23.140.1.1", ""},
    },
    // Staat der Nederlanden EV Root CA
    // https://pkioevssl-v.quovadisglobal.com/
    {
        {{0x4d, 0x24, 0x91, 0x41, 0x4c, 0xfe, 0x95, 0x67, 0x46, 0xec, 0x4c,
          0xef, 0xa6, 0xcf, 0x6f, 0x72, 0xe2, 0x8a, 0x13, 0x29, 0x43, 0x2f,
          0x9d, 0x8a, 0x90, 0x7a, 0xc4, 0xcb, 0x5d, 0xad, 0xc1, 0x5a}},
        {"2.16.528.1.1003.1.2.7", ""},
    },
    // Starfield Class 2 Certification Authority
    // https://www.starfieldtech.com/
    {
        {{0x14, 0x65, 0xfa, 0x20, 0x53, 0x97, 0xb8, 0x76, 0xfa, 0xa6, 0xf0,
          0xa9, 0x95, 0x8e, 0x55, 0x90, 0xe4, 0x0f, 0xcc, 0x7f, 0xaa, 0x4f,
          0xb7, 0xc2, 0xc8, 0x67, 0x75, 0x21, 0xfb, 0x5f, 0xb6, 0x58}},
        {"2.16.840.1.114414.1.7.23.3", ""},
    },
    // Starfield Root Certificate Authority - G2
    // https://valid.sfig2.catest.starfieldtech.com/
    {
        {{0x2c, 0xe1, 0xcb, 0x0b, 0xf9, 0xd2, 0xf9, 0xe1, 0x02, 0x99, 0x3f,
          0xbe, 0x21, 0x51, 0x52, 0xc3, 0xb2, 0xdd, 0x0c, 0xab, 0xde, 0x1c,
          0x68, 0xe5, 0x31, 0x9b, 0x83, 0x91, 0x54, 0xdb, 0xb7, 0xf5}},
        {"2.16.840.1.114414.1.7.23.3", ""},
    },
    // Starfield Services Root Certificate Authority - G2
    // https://valid.sfsg2.catest.starfieldtech.com/
    {
        {{0x56, 0x8d, 0x69, 0x05, 0xa2, 0xc8, 0x87, 0x08, 0xa4, 0xb3, 0x02,
          0x51, 0x90, 0xed, 0xcf, 0xed, 0xb1, 0x97, 0x4a, 0x60, 0x6a, 0x13,
          0xc6, 0xe5, 0x29, 0x0f, 0xcb, 0x2a, 0xe6, 0x3e, 0xda, 0xb5}},
        {"2.16.840.1.114414.1.7.24.3", "2.23.140.1.1"},
    },
    // SwissSign Gold CA - G2
    // https://testevg2.swisssign.net/
    {
        {{0x62, 0xdd, 0x0b, 0xe9, 0xb9, 0xf5, 0x0a, 0x16, 0x3e, 0xa0, 0xf8,
          0xe7, 0x5c, 0x05, 0x3b, 0x1e, 0xca, 0x57, 0xea, 0x55, 0xc8, 0x68,
          0x8f, 0x64, 0x7c, 0x68, 0x81, 0xf2, 0xc8, 0x35, 0x7b, 0x95}},
        {"2.16.756.1.89.1.2.1.1", ""},
    },
    // TWCA Global Root CA
    // https://evssldemo3.twca.com.tw/index.html
    {
        {{0x59, 0x76, 0x90, 0x07, 0xf7, 0x68, 0x5d, 0x0f, 0xcd, 0x50, 0x87,
          0x2f, 0x9f, 0x95, 0xd5, 0x75, 0x5a, 0x5b, 0x2b, 0x45, 0x7d, 0x81,
          0xf3, 0x69, 0x2b, 0x61, 0x0a, 0x98, 0x67, 0x2f, 0x0e, 0x1b}},
        {"1.3.6.1.4.1.40869.1.1.22.3", ""},
    },
    // TWCA Root Certification Authority
    // https://evssldemo.twca.com.tw/index.html
    {
        {{0xbf, 0xd8, 0x8f, 0xe1, 0x10, 0x1c, 0x41, 0xae, 0x3e, 0x80, 0x1b,
          0xf8, 0xbe, 0x56, 0x35, 0x0e, 0xe9, 0xba, 0xd1, 0xa6, 0xb9, 0xbd,
          0x51, 0x5e, 0xdc, 0x5c, 0x6d, 0x5b, 0x87, 0x11, 0xac, 0x44}},
        {"1.3.6.1.4.1.40869.1.1.22.3", ""},
    },
    // T-TeleSec GlobalRoot Class 3
    // http://www.telesec.de/ / https://root-class3.test.telesec.de/
    {
        {{0xfd, 0x73, 0xda, 0xd3, 0x1c, 0x64, 0x4f, 0xf1, 0xb4, 0x3b, 0xef,
          0x0c, 0xcd, 0xda, 0x96, 0x71, 0x0b, 0x9c, 0xd9, 0x87, 0x5e, 0xca,
          0x7e, 0x31, 0x70, 0x7a, 0xf3, 0xe9, 0x6d, 0x52, 0x2b, 0xbd}},
        {"1.3.6.1.4.1.7879.13.24.1", ""},
    },
    // UCA Extended Validation Root
    // https://rsaevg1.good.sheca.com/
    {
        {{0xd4, 0x3a, 0xf9, 0xb3, 0x54, 0x73, 0x75, 0x5c, 0x96, 0x84, 0xfc,
          0x06, 0xd7, 0xd8, 0xcb, 0x70, 0xee, 0x5c, 0x28, 0xe7, 0x73, 0xfb,
          0x29, 0x4e, 0xb4, 0x1e, 0xe7, 0x17, 0x22, 0x92, 0x4d, 0x24}},
        {"2.23.140.1.1", ""},
    },
    // USERTrust ECC Certification Authority
    // https://usertrustecccertificationauthority-ev.comodoca.com/
    {
        {{0x4f, 0xf4, 0x60, 0xd5, 0x4b, 0x9c, 0x86, 0xda, 0xbf, 0xbc, 0xfc,
          0x57, 0x12, 0xe0, 0x40, 0x0d, 0x2b, 0xed, 0x3f, 0xbc, 0x4d, 0x4f,
          0xbd, 0xaa, 0x86, 0xe0, 0x6a, 0xdc, 0xd2, 0xa9, 0xad, 0x7a}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // USERTrust RSA Certification Authority
    // https://usertrustrsacertificationauthority-ev.comodoca.com/
    {
        {{0xe7, 0x93, 0xc9, 0xb0, 0x2f, 0xd8, 0xaa, 0x13, 0xe2, 0x1c, 0x31,
          0x22, 0x8a, 0xcc, 0xb0, 0x81, 0x19, 0x64, 0x3b, 0x74, 0x9c, 0x89,
          0x89, 0x64, 0xb1, 0x74, 0x6d, 0x46, 0xc3, 0xd4, 0xcb, 0xd2}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // XRamp Global Certification Authority
    {
        {{0xce, 0xcd, 0xdc, 0x90, 0x50, 0x99, 0xd8, 0xda, 0xdf, 0xc5, 0xb1,
          0xd2, 0x09, 0xb7, 0x37, 0xcb, 0xe2, 0xc1, 0x8c, 0xfb, 0x2c, 0x10,
          0xc0, 0xff, 0x0b, 0xcf, 0x0d, 0x32, 0x86, 0xfc, 0x1a, 0xa2}},
        {"2.16.840.1.114404.1.1.2.4.1", ""},
    }};

#endif  // defined(PLATFORM_USES_CHROMIUM_EV_METADATA)
}  // namespace

static base::LazyInstance<EVRootCAMetadata>::Leaky g_ev_root_ca_metadata =
    LAZY_INSTANCE_INITIALIZER;

// static
EVRootCAMetadata* EVRootCAMetadata::GetInstance() {
  return g_ev_root_ca_metadata.Pointer();
}

#if defined(OS_WIN)

namespace {

bool ConvertBytesToDottedString(const der::Input& policy_oid,
                                std::string* dotted) {
  CBS cbs;
  CBS_init(&cbs, policy_oid.UnsafeData(), policy_oid.Length());
  bssl::UniquePtr<char> text(CBS_asn1_oid_to_text(&cbs));
  if (!text)
    return false;
  dotted->assign(text.get());
  return true;
}

}  // namespace

bool EVRootCAMetadata::IsEVPolicyOID(PolicyOID policy_oid) const {
  for (const auto& ev_root : kEvRootCaMetadata) {
    if (std::find(std::begin(ev_root.policy_oids),
                  std::end(ev_root.policy_oids),
                  policy_oid) != std::end(ev_root.policy_oids)) {
      return true;
    }
  }

  for (const auto& ca : extra_cas_) {
    if (ca.second == policy_oid)
      return true;
  }

  return false;
}

bool EVRootCAMetadata::IsEVPolicyOIDGivenBytes(
    const der::Input& policy_oid) const {
  std::string dotted;
  return ConvertBytesToDottedString(policy_oid, &dotted) &&
         IsEVPolicyOID(dotted.c_str());
}

bool EVRootCAMetadata::HasEVPolicyOID(const SHA256HashValue& fingerprint,
                                      PolicyOID policy_oid) const {
  for (const auto& ev_root : kEvRootCaMetadata) {
    if (fingerprint != ev_root.fingerprint)
      continue;
    return std::find(std::begin(ev_root.policy_oids),
                     std::end(ev_root.policy_oids),
                     policy_oid) != std::end(ev_root.policy_oids);
  }

  auto it = extra_cas_.find(fingerprint);
  return it != extra_cas_.end() && it->second == policy_oid;
}

bool EVRootCAMetadata::HasEVPolicyOIDGivenBytes(
    const SHA256HashValue& fingerprint,
    const der::Input& policy_oid) const {
  std::string dotted;
  return ConvertBytesToDottedString(policy_oid, &dotted) &&
         HasEVPolicyOID(fingerprint, dotted.c_str());
}

// static
bool EVRootCAMetadata::IsCaBrowserForumEvOid(PolicyOID policy_oid) {
  return strcmp(policy_oid, "2.23.140.1.1") == 0;
}

bool EVRootCAMetadata::AddEVCA(const SHA256HashValue& fingerprint,
                               const char* policy) {
  for (const auto& ev_root : kEvRootCaMetadata) {
    if (fingerprint == ev_root.fingerprint)
      return false;
  }
  if (extra_cas_.find(fingerprint) != extra_cas_.end())
    return false;

  extra_cas_[fingerprint] = policy;
  return true;
}

bool EVRootCAMetadata::RemoveEVCA(const SHA256HashValue& fingerprint) {
  auto it = extra_cas_.find(fingerprint);
  if (it == extra_cas_.end())
    return false;

  extra_cas_.erase(it);
  return true;
}

#elif defined(PLATFORM_USES_CHROMIUM_EV_METADATA)

namespace {

std::string OIDStringToDER(base::StringPiece policy) {
  uint8_t* der;
  size_t len;
  bssl::ScopedCBB cbb;
  if (!CBB_init(cbb.get(), 32) ||
      !CBB_add_asn1_oid_from_text(cbb.get(), policy.data(), policy.size()) ||
      !CBB_finish(cbb.get(), &der, &len)) {
    return std::string();
  }
  bssl::UniquePtr<uint8_t> delete_der(der);
  return std::string(reinterpret_cast<const char*>(der), len);
}

}  // namespace

bool EVRootCAMetadata::IsEVPolicyOID(PolicyOID policy_oid) const {
  return policy_oids_.find(policy_oid.AsString()) != policy_oids_.end();
}

bool EVRootCAMetadata::IsEVPolicyOIDGivenBytes(
    const der::Input& policy_oid) const {
  // This implementation uses DER bytes already, so the two functions are the
  // same.
  return IsEVPolicyOID(policy_oid);
}

bool EVRootCAMetadata::HasEVPolicyOID(const SHA256HashValue& fingerprint,
                                      PolicyOID policy_oid) const {
  PolicyOIDMap::const_iterator iter = ev_policy_.find(fingerprint);
  if (iter == ev_policy_.end())
    return false;
  for (const std::string& ev_oid : iter->second) {
    if (der::Input(&ev_oid) == policy_oid)
      return true;
  }
  return false;
}

bool EVRootCAMetadata::HasEVPolicyOIDGivenBytes(
    const SHA256HashValue& fingerprint,
    const der::Input& policy_oid) const {
  // The implementation uses DER bytes already, so the two functions are the
  // same.
  return HasEVPolicyOID(fingerprint, policy_oid);
}

// static
bool EVRootCAMetadata::IsCaBrowserForumEvOid(PolicyOID policy_oid) {
  const uint8_t kCabEvOid[] = {0x67, 0x81, 0x0c, 0x01, 0x01};
  return der::Input(kCabEvOid) == policy_oid;
}

bool EVRootCAMetadata::AddEVCA(const SHA256HashValue& fingerprint,
                               const char* policy) {
  if (ev_policy_.find(fingerprint) != ev_policy_.end())
    return false;

  std::string der_policy = OIDStringToDER(policy);
  if (der_policy.empty())
    return false;

  ev_policy_[fingerprint].push_back(der_policy);
  policy_oids_.insert(der_policy);
  return true;
}

bool EVRootCAMetadata::RemoveEVCA(const SHA256HashValue& fingerprint) {
  PolicyOIDMap::iterator it = ev_policy_.find(fingerprint);
  if (it == ev_policy_.end())
    return false;
  std::string oid = it->second[0];
  ev_policy_.erase(it);
  policy_oids_.erase(oid);
  return true;
}

#else

// These are just stub functions for platforms where we don't use this EV
// metadata.
//

bool EVRootCAMetadata::IsEVPolicyOID(PolicyOID policy_oid) const {
  LOG(WARNING) << "Not implemented";
  return false;
}

bool EVRootCAMetadata::IsEVPolicyOIDGivenBytes(
    const der::Input& policy_oid) const {
  LOG(WARNING) << "Not implemented";
  return false;
}

bool EVRootCAMetadata::HasEVPolicyOID(const SHA256HashValue& fingerprint,
                                      PolicyOID policy_oid) const {
  LOG(WARNING) << "Not implemented";
  return false;
}

bool EVRootCAMetadata::HasEVPolicyOIDGivenBytes(
    const SHA256HashValue& fingerprint,
    const der::Input& policy_oid) const {
  LOG(WARNING) << "Not implemented";
  return false;
}

bool EVRootCAMetadata::AddEVCA(const SHA256HashValue& fingerprint,
                               const char* policy) {
  LOG(WARNING) << "Not implemented";
  return true;
}

bool EVRootCAMetadata::RemoveEVCA(const SHA256HashValue& fingerprint) {
  LOG(WARNING) << "Not implemented";
  return true;
}

#endif

EVRootCAMetadata::EVRootCAMetadata() {
// Constructs the object from the raw metadata in kEvRootCaMetadata.
#if defined(PLATFORM_USES_CHROMIUM_EV_METADATA) && !defined(OS_WIN)
  for (const auto& ev_root : kEvRootCaMetadata) {
    for (const auto& policy : ev_root.policy_oids) {
      if (policy.empty())
        break;

      std::string policy_der = OIDStringToDER(policy.data());
      if (policy_der.empty()) {
        LOG(ERROR) << "Failed to register OID: " << policy;
        continue;
      }

      ev_policy_[ev_root.fingerprint].push_back(policy_der);
      policy_oids_.insert(policy_der);
    }
  }
#endif
}

EVRootCAMetadata::~EVRootCAMetadata() = default;

}  // namespace net
