/*
 * nb_common.c - Contains common in use functions for NetBEUI services.
 *
 * Copyright (c) 1997 by Procom Technology,Inc.
 *
 * This program can be redistributed or modified under the terms of the 
 * GNU General Public License as published by the Free Software Foundation.
 * This program is distributed without any warranty or implied warranty
 * of merchantability or fitness for a particular purpose.
 *
 * See the GNU General Public License for more details.
 *
 */
 

#include <linux/netbeui.h>


/*
 * Function: NB_FUNCADDR
 *	Returns functional address of a network device.
 *
 * Parameters:
 *	dev : pointer to device that need to its functional address.
 *
 * Returns: char *
 *	
 */
inline char *
NB_FUNCADDR (struct device *dev)
{
	switch (dev->type) {
#ifdef CONFIG_TR
		case ARPHRD_IEEE802 :
			return (NetBIOS_FUNC_ADDR_4TR);
#endif

#ifdef CONFIG_NET_ETHERNET
		case ARPHRD_ETHER :
		case ARPHRD_EETHER :
#endif
		case ARPHRD_LOOPBACK :
			return (NetBIOS_FUNC_ADDR_4ETH);

		default :
			return ("\x00\x00\x00\x00\x00\x00");
	}
}


/*
 * Function: nbcm_dev_supported
 *	Indicates that a network device is supported by us or not.
 *
 * Parameters:
 *	dev_type : device type that must check it.
 *
 * Returns: unsigned char
 *	0 : device not supported by us.
 *	1 : device supported by us.
 */

static inline unsigned char
nbcm_dev_supported (unsigned int dev_type)
{
	return ( 
#ifdef CONFIG_NET_ETHERNET
	         (dev_type == ARPHRD_ETHER) ||
	         (dev_type == ARPHRD_EETHER) ||
#endif
#ifdef CONFIG_TR
	         (dev_type == ARPHRD_IEEE802) ||
#endif
	         (dev_type == ARPHRD_LOOPBACK)
	       );
}

/*
 * Function: nbcm_apt_dev
 *
 * Parameters:
 *
 * Returns: unsigned char
 */

unsigned char
nbcm_apt_dev (struct device *dev)
{
	return (  (dev->flags & IFF_LOOPBACK )     ||
	         ( (dev->flags & IFF_MULTICAST) &&
	           nbcm_dev_supported(dev->type)  ) );
}


/*
 * Function: MAC_HEADLEN
 *	Calculates the MAC header length of a device.
 *
 * Parameters:
 *	dev : pointer to device that we need to its MAC header length.
 *
 * Returns: int
 *	Always positive : MAC header length of the device.
 *
 * Note:
 *	- if we do not recognize device type, the function returns ethernet
 *	  MAC header length.
 *	
 */

inline int
MAC_HEADLEN (struct device *dev)
{
	switch (dev->type) {
#ifdef CONFIG_TR
			case ARPHRD_IEEE802:
				return (sizeof(struct trh_hdr));
#endif

#ifdef CONFIG_NET_ETHERNET
			case ARPHRD_ETHER:
			case ARPHRD_EETHER:
#endif
			default:
				return (sizeof(struct ethhdr));
	}
}

/*
 * Function: LLCMAC_I_HEADLEN
 *	Calculates lengths of LLC header and MAC header together for 'I frames'
 *	on a specific device.
 *
 * Parameters:
 *	dev : pointer to device that calculation is performed for it.
 *
 * Returns: int
 *       Always positive : I frame's MAC+LLC header length of the device.
 */

inline int
LLCMAC_I_HEADLEN (struct device *dev)
{
	return (MAC_HEADLEN(dev) + LLC_I_HEADLEN);
}


/*
 * Function: LLCMAC_UI_HEADLEN
 *       Calculates lengths of LLC header and MAC header together for 'UI frames'.
 *
 * Parameters:
 *	mac_hlen : MAC header length for UI frames.
 *
 * Returns: int
 *       Always positive : UI frame's MAC+LLC header length of the device.
 */

inline int
LLCMAC_UI_HEADLEN (int mac_hlen)
{
	return (mac_hlen + LLC_UI_HEADLEN);
}


/*
 * Function: LLCMAC_UIB_HEADLEN
 *       Calculates lengths of LLC header and MAC header together for broadcasted
 *	'UI frames'.
 *
 * Parameters: none
 *
 * Returns: int
 *       Always positive : broadcasted UI frame's MAC+LLC header length.
 */

inline int
LLCMAC_UIB_HEADLEN (void)
{
	return (MAC_B_HEADLEN + LLC_UI_HEADLEN);
}


/*
 * Function: CALC_DG_SKBLEN
 *	Calculates appropriate sk_buff length for sending UI frames.
 *
 * Parameters:
 *	mac_hlen     : MAC header length.
 *	user_datalen : length of data that must put into sk_buff after
 *	               MAC+LLC headers (including NetBIOS header).
 *
 * Returns: int
 *       Always positive : appropriate sk_buff length for sending UI frames.
 */

inline int
CALC_DG_SKBLEN (int mac_hlen, int user_datalen)
{
	return (LLCMAC_UI_HEADLEN(mac_hlen) + user_datalen);
}
