/*
 *	MINIXEMU	An emulator for Minix binaries.
 *
 *	System calls are mostly pretty easy as the emulator is tightly
 *	bound to the minix task.
 *
 *
 *	Copyright (C) 1995 Alan Cox (alan@lxorguk.ukuu.org.uk)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/vm86.h>
#include <sys/wait.h>
#include <sys/times.h>
#include <utime.h>
#include <termios.h>
#include <time.h>
#include <signal.h>
#include <errno.h>
#include "minix.h" 

#define dbprintf(x)
 
/*
 *	Compress a host stat into a minix one. Lose upper bits with wild
 *	abandon. For SYS5.3 this isn't a problem, but machines with 32
 *	bit inodes (BSD, SYS5 with veritas, newest SCO) you lose the top
 *	bits which can confuse a few programs which use inode numbers 
 *	(eg gnu tar).
 */
 
static void squash_stat(struct stat *s, struct minix_stat *ms)
{
	ms->st_dev=s->st_dev;
	ms->st_inode=(unsigned short)s->st_ino;	/* Bits lost */
	ms->st_mode=s->st_mode;
	ms->st_nlink=s->st_nlink;
	ms->st_uid=s->st_uid;
	ms->st_gid=s->st_gid;
	ms->st_rdev=s->st_rdev;
	ms->st_size=s->st_size;
	ms->st_atime=s->st_atime;
	ms->st_mtime=s->st_mtime;
	ms->st_ctime=s->st_ctime;
}

/*
 *	Implementation of Minix syscalls.
 */
 
 
static int minix_exit(minix_msg *m)
{
	dbprintf(("exit(%d)\n",
		m->msg.m1.m1i1));
	exit(m->msg.m1.m1i1);
}

static int minix_fork(minix_msg *m)
{
	dbprintf(("fork()\n"));
	/* This is fun 8) - fork the emulator (its easier that way) */
	return fork();
}

static int minix_read(minix_msg *m)
{
	dbprintf(("read(%d, %d, %d)\n",
		m->msg.m1.m1i1,m->msg.m1.m1p1,m->msg.m1.m1i2));
	/* Fixme: Emulate raw reads on directory files if needed */
	return read(m->msg.m1.m1i1, (void *)MINIX_DSEG(m->msg.m1.m1p1),
		m->msg.m1.m1i2);
}

static int minix_write(minix_msg *m)
{
	dbprintf(("write(%d, %d, %d)\n",
		m->msg.m1.m1i1,m->msg.m1.m1p1,m->msg.m1.m1i2));
	return write(m->msg.m1.m1i1,
		(void *)MINIX_DSEG(m->msg.m1.m1p1),
		m->msg.m1.m1i2);
}

static int minix_open(minix_msg *m)
{
	char *dp=(void *)MINIX_DSEG(m->msg.m3.m3p1);
	/* 
	 *	"" is not valid in posix, but in non POSIX systems
	 *	equates to current directory.
	 */
	if(m->msg.m1.m1i2&O_CREAT)
	{
		dp=(void *)MINIX_DSEG(m->msg.m1.m1p1);
		dbprintf(("open(%s, %d, %d)\n",
			dp,
			m->msg.m1.m1i2,
			m->msg.m1.m1i3));
		return open(dp, m->msg.m1.m1i2,m->msg.m1.m1i3);
	}
	else
	{		
		if(*dp==0)
			dp=".";
		dbprintf(("open(%s, %d)\n",dp,m->msg.m3.m3i2));
		return open(dp,m->msg.m3.m3i2);
	}
}

static int minix_close(minix_msg *m)
{
	dbprintf(("close(%d)\n",m->msg.m1.m1i1));
	close(m->msg.m1.m1i1);
	return 0;
}

static int minix_wait(minix_msg *m)
{
	int val;
	int r;
	dbprintf(("wait([unknown])\n"));
	r=wait(&val);
	m->msg.m2.m2i1=val;
	return r;
}

static int minix_creat(minix_msg *m)
{
	int r;
	dbprintf(("creat(%s,%d)\n",
		(void *)MINIX_DSEG(m->msg.m3.m3p1),m->msg.m3.m3i2));
	r=creat((void *)MINIX_DSEG(m->msg.m3.m3p1),m->msg.m3.m3i2);
	return r;
}

static int minix_link(minix_msg *m)
{
	dbprintf(("link(%s,%s)\n",
		(void *)MINIX_DSEG(m->msg.m1.m1p1),
		(void *)MINIX_DSEG(m->msg.m1.m1p2)));
	return link((void *)MINIX_DSEG(m->msg.m1.m1p1),MINIX_DSEG(m->msg.m1.m1p2));
}

static int minix_unlink(minix_msg *m)
{
	dbprintf(("unlink(%s)\n",(void *)MINIX_DSEG(m->msg.m3.m3p1)));
	return unlink((void *)MINIX_DSEG(m->msg.m3.m3p1));
}

static int minix_chdir(minix_msg *m)
{
	dbprintf(("chdir(%s)\n",(void *)MINIX_DSEG(m->msg.m3.m3p1)));
	return chdir((void *)MINIX_DSEG(m->msg.m3.m3p1));
}

static int minix_time(minix_msg *m)
{
	time(&m->msg.m2.m2l1);
	return 0;
}

static int minix_mknod(minix_msg *m)
{
	dbprintf(("mknod(%s,%d,%d)\n",
		m->msg.m1.m1p1,
		m->msg.m1.m1i2,
		m->msg.m1.m1i3));
	return mknod((void *)MINIX_DSEG(m->msg.m1.m1p1), m->msg.m1.m1i2,m->msg.m1.m1i3);
}

static int minix_chmod(minix_msg *m)
{
	dbprintf(("chmod(%s,%d)\n",
		(void *)MINIX_DSEG(m->msg.m3.m3p1),
		m->msg.m3.m3i2));
	return chmod((void *)MINIX_DSEG(m->msg.m3.m3p1), m->msg.m3.m3i2);
}

static int minix_chown(minix_msg *m)
{
	dbprintf(("chown(%s,%d,%d)\n",
		(void *)MINIX_DSEG(m->msg.m1.m1p1),
		m->msg.m1.m1i2,
		m->msg.m1.m1i3));
	return chown((void *)MINIX_DSEG(m->msg.m1.m1p1),m->msg.m1.m1i2,
			m->msg.m1.m1i3);
}

static int minix_brk(minix_msg *m)
{
	dbprintf(("brk(%d)\n",m->msg.m3.m3p1));	
	if(m->msg.m3.m3p1>=minix_cpu.regs.esp)
	{
		errno= 1;	/* Really return -1 */
		return -1;
	}
	return m->msg.m3.m3p1;
}

static int minix_stat(minix_msg *m)
{
	struct stat s;
	dbprintf(("stat(%s,%d)\n",
		(void *)MINIX_DSEG(m->msg.m1.m1p1),
		m->msg.m1.m1p2));
	if(stat((void *)MINIX_DSEG(m->msg.m1.m1p1),&s)==-1)
		return -1;
	squash_stat(&s,(void *)MINIX_DSEG(m->msg.m1.m1p2));
	return 0;
}

static int minix_lseek(minix_msg *m)
{
	dbprintf(("lseek(%d,%ld,%d)\n",
		m->msg.m2.m2i1,
		m->msg.m2.m2l1,
		m->msg.m2.m2i2));
	return lseek(m->msg.m2.m2i1,m->msg.m2.m2l1,m->msg.m2.m2i2);
}

static int minix_getpid(minix_msg *m)
{
	m->msg.m2.m2i1=getppid();
	return getpid();
}

/*
 *	Not too hard to do but very undesirable I expect
 */
 
static int minix_mount(minix_msg *m)
{
	errno=EINVAL;
	return -1;
}


static int minix_umount(minix_msg *m)
{
	errno=EINVAL;
	return -1;
}

static int minix_setuid(minix_msg *m)
{
	return setuid(m->msg.m1.m1i1);
}

static int minix_getuid(minix_msg *m)
{
	m->msg.m2.m2i2=geteuid();
	return getuid();
}

static int minix_stime(minix_msg *m)
{
	dbprintf(("stime(&%ld)\n",
		m->msg.m2.m2l1));
	stime(&m->msg.m2.m2l1);
	return 0;
}

static int minix_ptrace(minix_msg *m)
{
	errno=EINVAL;
	return -1;
}

static int minix_alarm(minix_msg *m)
{
	dbprintf(("alarm(%d)\n",m->msg.m1.m1i1));
	alarm(m->msg.m1.m1i1);
	return 0;
}

static int minix_fstat(minix_msg *m)
{
	struct stat s;
	int err;
	dbprintf(("fstat(%d,%d)\n",
		m->msg.m1.m1i1, m->msg.m1.m1p1));
	err=fstat(m->msg.m1.m1i1,&s);
	squash_stat(&s,(void *)MINIX_DSEG(m->msg.m1.m1p1));
	return err;
}

static int minix_pause(minix_msg *m)
{
	dbprintf(("pause()\n"));
	return pause();
}

static int minix_utime(minix_msg *m)
{
	struct utimbuf u;
	u.actime=m->msg.m2.m2l1;
	u.modtime=m->msg.m2.m2l2;
	return utime(MINIX_DSEG(m->msg.m2.m2p1),&u);
	
}

static int minix_stty(minix_msg *m)
{
	errno=EINVAL;
	return -1;
}

int minix_gtty(minix_msg *m)
{
	errno=EINVAL;
	return -1;
}

static int minix_access(minix_msg *m)
{
	dbprintf(("access(%s,%d)\n",(void *)MINIX_DSEG(m->msg.m3.m3p1),m->msg.m3.m3i1));
	return access((void *)MINIX_DSEG(m->msg.m3.m3p1),m->msg.m3.m3i1);
}

static int minix_nice(minix_msg *m)
{
	return 0;
}

static int minix_ftime(minix_msg *m)
{
	/* FIXME */
	return 0;
}

static int minix_sync(minix_msg *m)
{
	dbprintf(("sync()\n"));
	sync();
	return 0;
}

static int minix_kill(minix_msg *m)
{
	dbprintf(("kill(%d,%d)\n",m->msg.m1.m1i1,m->msg.m1.m1i2));
	return kill(m->msg.m1.m1i1,m->msg.m1.m1i2);
}

/*
 *	Obsolete. Minix 1.7.x uses fcntl like other modern systems
 */
 
static int minix_dup(minix_msg *m)
{
	int fd=0;
	if(m->msg.m1.m1i1&0100)	/* DUP2 flag */
	{
		if(dup2(m->msg.m1.m1i2, fd)==-1)
			return -1;
		m->msg.m1.m1i3=fd;
	 	return 0;
	}
	else
	{
		return dup(m->msg.m1.m1i1);
	}
}

static int minix_pipe(minix_msg *m)
{
	int p[2];
	int err=pipe(p);
	if(err==-1)
		return err;
	m->msg.m1.m1i1=p[0];
	m->msg.m1.m1i2=p[1];
	return 0;
}

static int minix_times(minix_msg *m)
{
	struct tms t;
	times(&t);
	m->msg.m4.m4l1=(t.tms_utime*MINIX_HZ)/HZ;
	m->msg.m4.m4l2=(t.tms_stime*MINIX_HZ)/HZ;
	m->msg.m4.m4l3=(t.tms_cutime*MINIX_HZ)/HZ;
	m->msg.m4.m4l4=(t.tms_cstime*MINIX_HZ)/HZ;
	return 0;
}

static int minix_prof(minix_msg *m)
{
	errno=EINVAL;
	return -1;
}

static int minix_setgid(minix_msg *m)
{
	return setgid(m->msg.m1.m1i1);
}

static int minix_getgid(minix_msg *m)
{
	m->msg.m2.m2i2=getegid();
	return getgid();
}

static int minix_signal(minix_msg *m)
{	
#if 0
	dbprintf("signal(%d,%d)\n",m->m_ushort[1],m->m_ushort[5]));
	void *p=(void *)m->m_ushort[5];
	if(m->m_ushort[5]==MINIX_SIG_IGN)
		p=SIG_IGN;
	else if(m->m_ushort[5]==MINIX_SIG_DFL)
		p=SIG_DFL;
	else
	{
		minix_vector[m->m_short[0]]=m->m_ushort[5];	/* Vector for faked signal */
		p=minix_dosig;
	}
	signal(m->m_short[0],(void *)p);
#endif	
	return 0;
}

static int minix_acct(minix_msg *m)
{
	return acct(m->msg.m3.m3p1?(void *)MINIX_DSEG(m->msg.m3.m3p1):NULL);
}

static int minix_plock(minix_msg *m)
{
	errno=EINVAL;
	return -1;
}


static int minix_ioctl(minix_msg *m)
{
	struct termios t;
	switch(m->msg.m2.m2i3)
	{
		/*
		 *	This is tricky. The minix program doesn't save enough state
		 *	to restore everything properly with GETP/SETP. Do the best we
		 *	can not to damage anything. We potentially screw up VMIN/VTIME
		 *	but normally applications using VMIN/VTIME have saved their state
		 *	before calling another program 8) 
		 *
		 *	m2l1 holds packed data, m2i1 the file handle.
		 */
		 
		case MINIX_TIOCGETP:
		
			if(tcgetattr(m->msg.m2.m2i1,&t)==-1)
				return -1;
			m->msg.m2.m2l1=((t.c_cflag&CBAUD)<<8)|(t.c_cflag&CBAUD);
			m->msg.m2.m2l1<<=16;
			m->msg.m2.m2l1|=t.c_cc[VERASE]<<8;
			m->msg.m2.m2l1|=t.c_cc[VKILL];
			m->msg.m2.m2l2=0L;
			if(t.c_lflag&ICANON)
				m->msg.m2.m2l2|=MINIX_COOKED;
			else if(t.c_lflag&ISIG)
				m->msg.m2.m2l2|=MINIX_CBREAK;
			else
				m->msg.m2.m2l2|=MINIX_RAW;
			if(t.c_lflag&ECHO)
				m->msg.m2.m2l2|=MINIX_ECHO;
			if(t.c_oflag&OCRNL)
				m->msg.m2.m2l2|=MINIX_CRMOD;
			/* Deliberately leave XTAB alone. */
			return 0;

		case MINIX_TIOCSETP:
			if(tcgetattr(m->msg.m2.m2i1, &t)==-1)
				return -1;
			/* We need to unpack the control chars from l1 */
			t.c_cc[VKILL]=m->msg.m2.m2l1&0xFF;
			t.c_cc[VERASE]=(m->msg.m2.m2l1&0xFF00)>>8;
			/* The upper 16 hold ispeed/ospeed */
			/* FIXME: Set the speeds */
			if(m->msg.m2.m2l2&MINIX_COOKED)
				t.c_lflag|=ICANON|ISIG;
			else if(m->msg.m2.m2l2&MINIX_CBREAK)
			{
				t.c_lflag&=~ICANON;
				t.c_lflag|=ISIG;
				t.c_cc[VMIN]=1;
				t.c_cc[VTIME]=0;
			}
			else
			{
				t.c_lflag&=~(ICANON|ISIG);
				t.c_cc[VMIN]=1;
				t.c_cc[VTIME]=0;
			}
			if(m->msg.m2.m2l2&MINIX_ECHO)
				t.c_lflag|=ECHO;
			else
				t.c_lflag&=~ECHO;
			if(m->msg.m2.m2l2&MINIX_CRMOD)
				t.c_oflag|=ONLCR;
			else
				t.c_oflag&=~ONLCR;				

			return tcsetattr(m->msg.m2.m2i1,0,&t);			
			
		/*
		 *	While this pair are easy 8)
		 */
		case MINIX_TIOCGETC:
			if(tcgetattr(m->msg.m2.m2i1,&t)==-1)
				return -1;
			m->msg.m2.m2l1=t.c_cc[VINTR]<<24;
			m->msg.m2.m2l1|=t.c_cc[VQUIT]<<16;
			m->msg.m2.m2l1|=t.c_cc[VSTART]<<8;
			m->msg.m2.m2l1|=t.c_cc[VSTOP];
			m->msg.m2.m2l2=t.c_cc[VEOF]<<8;
			m->msg.m2.m2l2|=0;		/* brk char not used */
			return 0;
			
		case MINIX_TIOCSETC:
			if(tcgetattr(m->msg.m2.m2i1,&t)==-1)
				return -1;
			t.c_cc[VINTR]=(m->msg.m2.m2l1>>24)&0xFF;
			t.c_cc[VQUIT]=(m->msg.m2.m2l1>>16)&0xFF;
			t.c_cc[VSTART]=(m->msg.m2.m2l1>>8)&0xFF;
			t.c_cc[VSTOP]=(m->msg.m2.m2l1)&0xFF;
			t.c_cc[VEOF]=(m->msg.m2.m2l2>>8)&0xFF;
			return tcsetattr(m->msg.m2.m2i1,0,&t);
		
		case MINIX_TIOCFLUSH:
			/* Minix only does TCIOFLUSH */
			return tcflush(m->msg.m2.m2i1,TCIOFLUSH);
	}
	return -1;
}

/*
 *	Exec is fun. The Minix user library builds a complete minix stack
 *	image. Great except that we need to unpack it all again and do a
 *	real exec. If its another minix image then our kernel side binary
 *	loader will load minixemu again and we'll take the Unix args and turn
 *	them back intoa minix stack image.
 *
 *	For now we run minixemu ourselves and do token attempts at binary
 *	checking.
 */
	
int minix_exec(minix_msg *m)
{
	/* ushort 3 is the name , ushort 4 the stack segment */
	int fd;
	int arg_ct,env_ct;
	int ct;
	char **argp, **envp;
	unsigned short *bp=(unsigned short *)MINIX_DSEG(m->msg.m1.m1p2);
	unsigned char *base=(unsigned char *)bp;
	unsigned short *tmp=bp;
	struct minix_exec_hdr mh;
	
	dbprintf(("exec(%s,%d)\n",(void *)MINIX_DSEG(m->msg.m1.m1p1),
		m->msg.m1.m1p2));
	fd=open((void *)MINIX_DSEG(m->msg.m1.m1p1),O_RDONLY);
	if(fd==-1)
		return -ENOENT;
	if(read(fd, &mh, sizeof(mh))!=sizeof(mh))
	{
		close(fd);
		return -ENOEXEC;
	}
	close(fd);
	if(mh.hlen!=EXEC_HEADER_SIZE)
		return -ENOEXEC;
	if(mh.type!=MINIX_COMBID && mh.type!=MINIX_SPLITID)
		return -ENOEXEC;

	while(*tmp++)
		arg_ct++;
	while(*tmp++)
		env_ct++;
	arg_ct+=2;	/* minixemu-path progname arg0...argn */
	argp=malloc(sizeof(char *)*(arg_ct+1));
	envp=malloc(sizeof(char *)*(env_ct+1));
	if(!argp||!envp)
		return -ENOMEM;
	argp[0]="minixemu";
	argp[1]=(void *)MINIX_DSEG(m->msg.m1.m1p1);
	ct=2;
	while(*base)
		argp[ct++]=(void *)MINIX_DSEG(*base++);
	argp[ct]=0;
	base++;
	ct=0;
	while(*base)
		envp[ct++]=(void *)MINIX_DSEG(*base++);
	execve(argp[0],argp,envp);
	perror("minixemu");
	exit(1);
}

int minix_umask(minix_msg *m)
{
	return umask(m->msg.m1.m1i1);
}

int minix_chroot(minix_msg *m)
{
	dbprintf(("chroot(%s)\n",
		MINIX_DSEG(m->msg.m3.m3p1)));
	return chroot((void *)MINIX_DSEG(m->msg.m3.m3p1));
}


int minix_noemulation(minix_msg *m)
{
	errno = ENOSYS;
	return -1;
}

int minix_fcntl(minix_msg *m)
{
	dbprintf(("fcntl(%d,%d,...)\n",
		m->msg.m1.m1i1,m->msg.m1.m1i2));
	switch(m->msg.m1.m1i2)
	{
		case MINIX_F_GETFD:
			return fcntl(m->msg.m1.m1i1,F_GETFD,0);
		case MINIX_F_GETFL:
			return fcntl(m->msg.m1.m1i1,F_GETFL,0);
		case MINIX_F_DUPFD:
			return fcntl(m->msg.m1.m1i1,F_DUPFD,m->msg.m1.m1i3);
		case MINIX_F_SETFD:
			return fcntl(m->msg.m1.m1i1,F_SETFD,m->msg.m1.m1i3);
		case MINIX_F_SETFL:
			return fcntl(m->msg.m1.m1i1,F_SETFL,m->msg.m1.m1i3);
		/*
		 *	Fixme: Unpack and process minix file locks 
		 */
		case MINIX_F_GETLK:
		case MINIX_F_SETLK:
		case MINIX_F_SETLKW:
			errno = EINVAL;
			return -1;
	}
	errno = EINVAL;
	return -1;
}

int minix_rename(minix_msg *m)
{
	dbprintf(("rename(%s,%s)\n",
		(void *)MINIX_DSEG(m->msg.m1.m1p1),
		(void *)MINIX_DSEG(m->msg.m1.m1p2)));
	return rename((void *)MINIX_DSEG(m->msg.m1.m1p1),
		      (void *)MINIX_DSEG(m->msg.m1.m1p2));
}

int minix_mkdir(minix_msg *m)
{
	dbprintf(("mkdir(%s,%d)\n",
		(void *)MINIX_DSEG(m->msg.m1.m1p1),
		m->msg.m1.m1i2));
	return mkdir((void *)MINIX_DSEG(m->msg.m1.m1p1),
			m->msg.m1.m1i2);
}

int minix_rmdir(minix_msg *m)
{
	dbprintf(("rmdir(%s)\n",
		(void *)MINIX_DSEG(m->msg.m3.m3p1)));
	return rmdir((void *)MINIX_DSEG(m->msg.m3.m3p1));
}

/*
 *	m is a pointer into the minix object to the message. On exit error is set for errors and the return is -1 so
 *	we can do error mapping. Other returns are the function return.
 */
 
int minix_syscall(minix_msg *m, int syscall, int fs)
{
	int r;	
	errno = 0;
	
	if(!fs)
	{
		int retind=0;
		
		switch(syscall)
		{
			case EXIT:
				r=minix_exit(m);break;
			case FORK:
				r=minix_fork(m);break;
			case WAIT:
				r=minix_wait(m);break;
			case BRK:
				retind=1;
				r=minix_brk(m);break;
			case EXEC:
				r=minix_exec(m);break;
			case GETPID:
				r=minix_getpid(m);break;
			case SETUID:
				r=minix_setuid(m);break;
			case GETUID:
				r=minix_getuid(m);break;
			case ALARM:
				r=minix_alarm(m);break;
			case PAUSE:
				r=minix_pause(m);break;
			case NICE:
				r=minix_nice(m);break;
			case KILL:
				r=minix_kill(m);break;
			case SETGID:
				r=minix_setgid(m);break;
			case GETGID:
				r=minix_getgid(m);break;
			case SIGNAL:
				r=minix_signal(m);break;
			default:
				r=minix_noemulation(m);break;
		}
		m->msg.m1.m1i1=r;
		m->msg.m1.m1p1=r;
		return retind?0:r;
	}
	else
	{
		switch(syscall)
		{
			case READ:
				r=minix_read(m);break;
			case WRITE:
				r=minix_write(m);break;
			case OPEN:
				r=minix_open(m);break;
			case CLOSE:
				r=minix_close(m);break;
			case CREAT:
				r=minix_creat(m);break;
			case LINK:
				r=minix_link(m);break;
			case UNLINK:
				r=minix_unlink(m);break;
			case CHDIR:
				r=minix_chdir(m);break;
			case TIME:
				r=minix_time(m);break;
			case MKNOD:
				r=minix_mknod(m);break;
			case CHMOD:
				r=minix_chmod(m);break;
			case CHOWN:
				r=minix_chown(m);break;
			case STAT:
				r=minix_stat(m);break;
			case LSEEK:
				r=minix_lseek(m);break;
			case MOUNT:
				r=minix_mount(m);break;
			case UMOUNT:
				r=minix_umount(m);break;
			case STIME:
				r=minix_stime(m);break;
			case PTRACE:
				r=minix_ptrace(m);break;
			case FSTAT:
				r=minix_fstat(m);break;
			case UTIME:
				r=minix_utime(m);break;
			case STTY:
				r=minix_stty(m);break;
			case GTTY:
				r=minix_gtty(m);break;
			case ACCESS:
				r=minix_access(m);break;
			case FTIME:
				r=minix_ftime(m);break;
			case SYNC:
				r=minix_sync(m);break;
			case RENAME:
				r=minix_rename(m);break;
			case MKDIR:
				r=minix_mkdir(m);break;
			case RMDIR:
				r=minix_rmdir(m);break;
			case DUP:
				r=minix_dup(m);break;
			case PIPE:
				r=minix_pipe(m);break;
			case TIMES:
				r=minix_times(m);break;
			case PROF:
				r=minix_prof(m);break;
			case ACCT:
				r=minix_acct(m);break;
			case PLOCK:
				r=minix_plock(m);break;
			case IOCTL:
				r=minix_ioctl(m);break;
			case FCNTL:
				r=minix_fcntl(m);break;
			case UMASK:
				r=minix_umask(m);break;
			case CHROOT:
				r=minix_chroot(m);break;
			default:
				r=minix_noemulation(m);break;
		}
		if(errno==0)
			return r;
		else
			return -errno;
	}
}
			
