/*
 * oFono - Open Source Telephony
 * Copyright (C) 2008-2011  Intel Corporation
 *
 * SPDX-License-Identifier: GPL-2.0-only
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dlfcn.h>

#include <glib.h>

#include "ofono.h"

static GSList *plugins = NULL;

struct ofono_plugin {
	void *handle;
	gboolean active;
	struct ofono_plugin_desc *desc;
};

static gint compare_priority(gconstpointer a, gconstpointer b)
{
	const struct ofono_plugin *plugin1 = a;
	const struct ofono_plugin *plugin2 = b;

	return plugin2->desc->priority - plugin1->desc->priority;
}

static gboolean add_plugin(void *handle, struct ofono_plugin_desc *desc)
{
	struct ofono_plugin *plugin;

	if (desc->init == NULL)
		return FALSE;

	if (g_str_equal(desc->version, OFONO_VERSION) == FALSE) {
		ofono_error("Invalid version %s for %s", desc->version,
							desc->description);
		return FALSE;
	}

	plugin = g_try_new0(struct ofono_plugin, 1);
	if (plugin == NULL)
		return FALSE;

	plugin->handle = handle;
	plugin->active = FALSE;
	plugin->desc = desc;

	__ofono_log_enable(desc->debug_start, desc->debug_stop);

	plugins = g_slist_insert_sorted(plugins, plugin, compare_priority);

	return TRUE;
}

static gboolean check_plugin(struct ofono_plugin_desc *desc,
				char **patterns, char **excludes)
{
	if (excludes) {
		for (; *excludes; excludes++)
			if (g_pattern_match_simple(*excludes, desc->name))
				break;
		if (*excludes) {
			ofono_info("Excluding %s", desc->description);
			return FALSE;
		}
	}

	if (patterns) {
		for (; *patterns; patterns++)
			if (g_pattern_match_simple(*patterns, desc->name))
				break;
		if (*patterns == NULL) {
			ofono_info("Ignoring %s", desc->description);
			return FALSE;
		}
	}

	return TRUE;
}

#include "builtin.h"

int __ofono_plugin_init(const char *pattern, const char *exclude)
{
	gchar **patterns = NULL;
	gchar **excludes = NULL;
	GSList *list;
	GDir *dir;
	const gchar *file;
	gchar *filename;
	unsigned int i;

	DBG("");

	if (pattern)
		patterns = g_strsplit_set(pattern, ":, ", -1);

	if (exclude)
		excludes = g_strsplit_set(exclude, ":, ", -1);

	for (i = 0; __ofono_builtin[i]; i++) {
		if (check_plugin(__ofono_builtin[i],
					patterns, excludes) == FALSE)
			continue;

		add_plugin(NULL, __ofono_builtin[i]);
	}

	dir = g_dir_open(PLUGINDIR, 0, NULL);
	if (dir != NULL) {
		while ((file = g_dir_read_name(dir)) != NULL) {
			void *handle;
			struct ofono_plugin_desc *desc;

			if (g_str_has_prefix(file, "lib") == TRUE ||
					g_str_has_suffix(file, ".so") == FALSE)
				continue;

			filename = g_build_filename(PLUGINDIR, file, NULL);

			handle = dlopen(filename, RTLD_NOW);
			if (handle == NULL) {
				ofono_error("Can't load %s: %s",
							filename, dlerror());
				g_free(filename);
				continue;
			}

			g_free(filename);

			desc = dlsym(handle, "ofono_plugin_desc");
			if (desc == NULL) {
				ofono_error("Can't load symbol: %s",
								dlerror());
				dlclose(handle);
				continue;
			}

			if (check_plugin(desc, patterns, excludes) == FALSE) {
				dlclose(handle);
				continue;
			}

			if (add_plugin(handle, desc) == FALSE)
				dlclose(handle);
		}

		g_dir_close(dir);
	}

	for (list = plugins; list; list = list->next) {
		struct ofono_plugin *plugin = list->data;

		if (plugin->desc->init() < 0)
			continue;

		plugin->active = TRUE;
	}

	g_strfreev(patterns);
	g_strfreev(excludes);

	return 0;
}

void __ofono_plugin_cleanup(void)
{
	GSList *list;

	DBG("");

	/*
	 * Terminate the plugins but don't unload the libraries yet.
	 * Plugins may reference data structures allocated by each other.
	 */
	for (list = plugins; list; list = list->next) {
		struct ofono_plugin *plugin = list->data;

		if (plugin->active == TRUE && plugin->desc->exit)
			plugin->desc->exit();
	}

	/* Second pass - unload the libraries */
	for (list = plugins; list; list = list->next) {
		struct ofono_plugin *plugin = list->data;

		if (plugin->handle)
			dlclose(plugin->handle);
	}

	/* Finally, free the memory */
	g_slist_free_full(plugins, g_free);
	plugins = NULL;
}
