// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.share.long_screenshots.bitmap_generation;

import android.content.Context;
import android.graphics.Rect;

import androidx.annotation.VisibleForTesting;

import org.chromium.chrome.browser.share.long_screenshots.bitmap_generation.LongScreenshotsEntry.EntryStatus;
import org.chromium.chrome.browser.tab.Tab;
import org.chromium.components.paintpreview.player.CompositorStatus;
import org.chromium.content_public.browser.RenderCoordinates;
import org.chromium.ui.display.DisplayAndroid;

import java.util.ArrayList;
import java.util.List;

/**
 * Entry manager responsible for managing all the of the {@LongScreenshotEntry}. This should be used
 * to generate and retrieve the needed bitmaps. The first bitmap can be generated by calling
 * {@link generateInitialEntry}.
 */
public class EntryManager {
    private static final int NUM_VIEWPORTS_CAPTURE_ABOVE = 4;
    private static final int NUM_VIEWPORTS_CAPTURE_BELOW = 6;

    private Context mContext;
    private Tab mTab;

    // List of all entries in correspondence of the webpage.
    private List<LongScreenshotsEntry> mEntries;
    // List of entries that are queued to generate the bitmap. Entries should only be queued
    // while the capture is in progress.
    private List<LongScreenshotsEntry> mQueuedEntries;
    private int mDisplayHeightPx;
    private BitmapGenerator mGenerator;
    private @EntryStatus int mGeneratorStatus;

    /**
     * @param context An instance of current Android {@link Context}.
     * @param tab Tab to generate the bitmap for.
     */
    public EntryManager(Context context, Tab tab) {
        mContext = context;
        mTab = tab;
        mEntries = new ArrayList<LongScreenshotsEntry>();
        mQueuedEntries = new ArrayList<LongScreenshotsEntry>();

        calculateDisplayHeight();

        mGenerator = new BitmapGenerator(
                context, tab, calculateCaptureSize(), createBitmapGeneratorCallback());
        mGenerator.captureTab();
        updateGeneratorStatus(EntryStatus.CAPTURE_IN_PROGRESS);
    }

    /**
     * Generates the first bitmap of the page that is the height of the phone display. Callers of
     * this function should add a listener to the returned entry to get that status of the
     * generation and retrieve the bitmap.
     */
    public LongScreenshotsEntry generateInitialEntry() {
        RenderCoordinates coords = RenderCoordinates.fromWebContents(mTab.getWebContents());
        int startY = coords.getScrollYPixInt() / coords.getPageScaleFactorInt();

        LongScreenshotsEntry entry = new LongScreenshotsEntry(
                mContext, mTab, startY, mDisplayHeightPx, mGenerator, false);
        processEntry(entry, false);
        return entry;
    }

    /**
     * Calculates the height of the phone used to determine the height of the bitmaps.
     */
    private void calculateDisplayHeight() {
        DisplayAndroid displayAndroid = DisplayAndroid.getNonMultiDisplay(mContext);
        mDisplayHeightPx = displayAndroid.getDisplayHeight();
    }

    /**
     * Creates the entry to generate the bitmap before the one passed in.
     *
     * @param relativeToId Id to base the new entry off of.
     * @return The new entry that generates the bitmap.
     */
    public LongScreenshotsEntry getPreviousEntry(int relativeToId) {
        int found = -1;
        for (int i = 0; i < mEntries.size(); i++) {
            if (mEntries.get(i).getId() == relativeToId) {
                found = i;
            }
        }

        if (found == -1) {
            return null;
        }

        if (found > 0) {
            return mEntries.get(found - 1);
        }

        // found = 0
        LongScreenshotsEntry newEntry = new LongScreenshotsEntry(
                mContext, mTab, mEntries.get(0).getId(), mDisplayHeightPx, mGenerator, true);
        processEntry(newEntry, true);
        return newEntry;
    }

    /**
     * Creates the entry to generate the bitmap after the one passed in.
     *
     * @param relativeToId Id to base the new entry off of.
     * @return The new entry that generates the bitmap.
     */
    public LongScreenshotsEntry getNextEntry(int relativeToId) {
        int found = -1;
        for (int i = 0; i < mEntries.size(); i++) {
            if (mEntries.get(i).getId() == relativeToId) {
                found = i;
            }
        }

        if (found == -1) {
            return null;
        }

        if (found < mEntries.size() - 1) {
            return mEntries.get(found + 1);
        }

        // found = last entry in the arraylist
        int newStartY = mEntries.get(mEntries.size() - 1).getEndYAxis() + 1;

        LongScreenshotsEntry newEntry = new LongScreenshotsEntry(
                mContext, mTab, newStartY, mDisplayHeightPx, mGenerator, false);
        processEntry(newEntry, false);
        return newEntry;
    }

    private void processEntry(LongScreenshotsEntry entry, boolean addToBeginningOfList) {
        if (mGeneratorStatus == EntryStatus.CAPTURE_COMPLETE) {
            entry.generateBitmap();
        } else if (mGeneratorStatus == EntryStatus.CAPTURE_IN_PROGRESS) {
            mQueuedEntries.add(entry);
        } else {
            entry.updateStatus(mGeneratorStatus);
        }

        // Add to the list of all entries
        if (addToBeginningOfList) {
            mEntries.add(0, entry);
        } else {
            mEntries.add(entry);
        }
    }

    /**
     * Defines the bounds of the capture.
     */
    private Rect calculateCaptureSize() {
        RenderCoordinates coords = RenderCoordinates.fromWebContents(mTab.getWebContents());

        // The current position the user has scrolled to.
        int currYAxisPosition = coords.getScrollYPixInt() / coords.getPageScaleFactorInt();

        int startYAxis = (int) Math.floor(currYAxisPosition
                - (NUM_VIEWPORTS_CAPTURE_ABOVE * mDisplayHeightPx * coords.getPageScaleFactor()));
        startYAxis = startYAxis < 0 ? 0 : startYAxis;

        int endYAxis = (int) Math.floor(currYAxisPosition
                + (NUM_VIEWPORTS_CAPTURE_BELOW * mDisplayHeightPx * coords.getPageScaleFactor()));
        int maxY = coords.getContentHeightPixInt() / coords.getPageScaleFactorInt();
        endYAxis = endYAxis > maxY ? maxY : endYAxis;

        return new Rect(0, startYAxis, 0, endYAxis);
    }

    /**
     * Updates based on the generator status. If the capture is complete, generates the bitmap for
     * all the queued entries.
     *
     * @param status New status from the generator.
     */
    private void updateGeneratorStatus(@EntryStatus int status) {
        mGeneratorStatus = status;
        if (status == EntryStatus.CAPTURE_COMPLETE) {
            for (LongScreenshotsEntry entry : mQueuedEntries) {
                entry.generateBitmap();
            }
            mQueuedEntries.clear();
        } else {
            for (LongScreenshotsEntry entry : mQueuedEntries) {
                entry.updateStatus(status);
            }
        }
    }

    /**
     * Creates the default BitmapGenerator to be used to retrieve the state of the generation. This
     * is the default implementation and should only be overridden for tests.
     */
    @VisibleForTesting
    public BitmapGenerator.GeneratorCallBack createBitmapGeneratorCallback() {
        return new BitmapGenerator.GeneratorCallBack() {
            @Override
            public void onCompositorResult(@CompositorStatus int status) {
                // TODO(tgupta): Add metrics logging here.
                if (status == CompositorStatus.STOPPED_DUE_TO_MEMORY_PRESSURE
                        || status == CompositorStatus.SKIPPED_DUE_TO_MEMORY_PRESSURE) {
                    updateGeneratorStatus(EntryStatus.INSUFFICIENT_MEMORY);
                } else if (status == CompositorStatus.OK) {
                    updateGeneratorStatus(EntryStatus.CAPTURE_COMPLETE);
                } else {
                    updateGeneratorStatus(EntryStatus.GENERATION_ERROR);
                }
            }

            @Override
            public void onCaptureResult(@Status int status) {
                // TODO(tgupta): Add metrics logging here.
                if (status == Status.LOW_MEMORY_DETECTED) {
                    updateGeneratorStatus(EntryStatus.INSUFFICIENT_MEMORY);
                } else if (status != Status.OK) {
                    updateGeneratorStatus(EntryStatus.GENERATION_ERROR);
                }
            }
        };
    }
}
