/***************************************************************
 * Name:      dxcalendarctrl.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2020
 * License:   GPL3
 **************************************************************/

#include <wx/tokenzr.h>
#include "dxcalendarctrl.h"
#include "dxsettings.h"
#include "dxdefs.h"
#include "dxutils.h"
#include "eventdialog.h"

// the constants used for the layout
#define VERT_MARGIN    5           // distance between combo and calendar
#define HORZ_MARGIN    5           //                            spin

BEGIN_EVENT_TABLE(dxCalendarCtrl, wxGenericCalendarCtrl)
    EVT_MOTION(dxCalendarCtrl::OnMouseMove)
    EVT_PAINT(dxCalendarCtrl::OnPaint)
    EVT_LEFT_DOWN(dxCalendarCtrl::OnClick)
    EVT_CHAR(dxCalendarCtrl::OnChar)
    EVT_RIGHT_DOWN(dxCalendarCtrl::OnRightClick)
END_EVENT_TABLE()

dxCalendarCtrl::dxCalendarCtrl(wxWindow *parent, const wxWindowID id, bool smaller, bool markDays)
#if defined (__WXGTK__) || defined (__WXMAC__)
    : wxGenericCalendarCtrl(parent, id, wxDateTime::Today(), wxDefaultPosition, wxDefaultSize, wxBORDER_SUNKEN|wxCAL_MONDAY_FIRST|wxCAL_SHOW_HOLIDAYS|wxCAL_SEQUENTIAL_MONTH_SELECTION), m_smaller(smaller)
#else
    : wxGenericCalendarCtrl(parent, id, wxDateTime::Today(), wxDefaultPosition, wxDefaultSize, wxCAL_MONDAY_FIRST|wxCAL_SHOW_HOLIDAYS|wxCAL_SEQUENTIAL_MONTH_SELECTION), m_smaller(smaller)
#endif
{
    m_eventID = -1;
    m_calendarWeekWidth = 0;
    m_markDays = markDays;
    m_showhidden = dxsettings.hidden();
    m_lowdate = wxDefaultDateTime;
    m_highdate = wxDefaultDateTime;
    wxDateTime::WeekDay wd;
    for(wd = wxDateTime::Sun; wd < wxDateTime::Inv_WeekDay; wxNextWDay(wd))
    {
        m_weekdays[wd] = wxDateTime::GetWeekDayName(wd, wxDateTime::Name_Abbr);
    }
    m_datum = wxDateTime::Today();
    if(smaller) SetFont(dxutils::fontFromString(dxsettings.calMiniFont()));
    else SetFont(dxutils::fontFromString(dxsettings.calFont()));
    SetBackgroundColour(dxsettings.calBackground());
}

void dxCalendarCtrl::updateColors()
{
    SetBackgroundColour(dxsettings.calBackground());
    Refresh();
}

void dxCalendarCtrl::updateFonts()
{
    if(m_smaller) SetFont(dxutils::fontFromString(dxsettings.calMiniFont()));
    else SetFont(dxutils::fontFromString(dxsettings.calFont()));
}

bool dxCalendarCtrl::SetDate(const wxDateTime &date)
{
    bool retval = true;
    bool sameMonth = m_datum.GetMonth() == date.GetMonth(), sameYear = m_datum.GetYear() == date.GetYear();
    if(isDateInRange(date))
    {
        if(sameMonth && sameYear)
        {
            // just change the day
            changeDay(date);
        }
        else
        {
            if(AllowMonthChange())
            {
                // change everything
                m_datum = date;
                // update the calendar
                markDays(true, m_showhidden);
            }
            else
            {
                // forbidden
                retval = false;
            }
        }
    }
    return retval;
}

wxCalendarHitTestResult dxCalendarCtrl::HitTest(const wxPoint &pos, wxDateTime *date, wxDateTime::WeekDay *wd)
{
    recalcGeometry();
    // the position where the calendar really begins
    wxCoord x0 = m_calendarWeekWidth+(GetClientSize().x-7*m_widthCol-m_calendarWeekWidth)/2;
    if(m_todayRect.Contains(pos))
    {
        //return this for click on Today text
        return wxCAL_HITTEST_WEEK;
    }
    if(HasFlag(wxCAL_SEQUENTIAL_MONTH_SELECTION))
    {
        // Header: month
        // we need to find out if the hit is on left arrow, on month or on right arrow
        // left arrow?
        if(m_leftArrowRect.Contains(pos))
        {
            if(date)
            {
                if(isDateInRange(m_datum - wxDateSpan::Month()))
                {
                    *date = m_datum - wxDateSpan::Month();
                }
                else
                {
                    *date = GetLowerDateLimit();
                }
            }
            return wxCAL_HITTEST_DECMONTH;
        }
        if(m_rightArrowRect.Contains(pos))
        {
            if(date)
            {
                if(isDateInRange(m_datum + wxDateSpan::Month()))
                {
                    *date = m_datum + wxDateSpan::Month();
                }
                else
                {
                    *date = GetUpperDateLimit();
                }
            }
            return wxCAL_HITTEST_INCMONTH;
        }
    }
    if(pos.x - x0 < 0)
    {
        if(pos.x >= 0 && pos.y > m_rowOffset + m_heightRow && pos.y <= m_rowOffset + m_heightRow * 7)
        {
            if(date)
            {
                *date = startDate();
                *date += wxDateSpan::Week() * (( pos.y - m_rowOffset ) / m_heightRow - 1 );
            }
            if ( wd )
                *wd = wxDateTime::Mon;
            //return wxCAL_HITTEST_WEEK;
            return wxCAL_HITTEST_NOWHERE;
        }
        else    // early exit -> the rest of the function checks for clicks on days
            return wxCAL_HITTEST_NOWHERE;
    }
    // header: week days
    int wday = (pos.x - x0) / m_widthCol;
    if(wday > 6)
        return wxCAL_HITTEST_NOWHERE;
    if(pos.y < (m_heightRow + m_rowOffset))
    {
        if(pos.y > m_rowOffset)
        {
            if(wd)
            {
                if(HasFlag(wxCAL_MONDAY_FIRST))
                {
                    wday = wday == 6 ? 0 : wday + 1;
                }
                *wd = (wxDateTime::WeekDay)wday;
            }
            return wxCAL_HITTEST_HEADER;
        }
        else
        {
            return wxCAL_HITTEST_NOWHERE;
        }
    }
    int week = (pos.y - (m_heightRow + m_rowOffset)) / m_heightRow;
    if(week >= 6 || wday >= 7)
    {
        return wxCAL_HITTEST_NOWHERE;
    }
    wxDateTime dt = startDate() + wxDateSpan::Days(7*week + wday);
    if(dt.GetMonth() == m_datum.GetMonth())
    {
        if(date)
            *date = dt;
        if(dt.GetMonth() == m_datum.GetMonth())
        {
            return wxCAL_HITTEST_DAY;
        }
        else
        {
            return wxCAL_HITTEST_SURROUNDING_WEEK;
        }
    }
    else
    {
        return wxCAL_HITTEST_NOWHERE;
    }
}

wxSize dxCalendarCtrl::DoGetBestSize() const
{
    // calc the size of the calendar
    const_cast<dxCalendarCtrl *>(this)->recalcGeometry();
    wxCoord width = wxMax(8*m_widthCol+m_calendarWeekWidth,m_widthToday),
            height = 8*m_heightRow + m_rowOffset + 2*VERT_MARGIN;
    wxSize best(width, height);
    CacheBestSize(best);
    return best;
}

bool dxCalendarCtrl::adjustDateToRange(wxDateTime *date) const
{
    if(m_lowdate.IsValid() && *date < m_lowdate)
    {
        *date = m_lowdate;
        return true;
    }
    if(m_highdate.IsValid() && *date > m_highdate)
    {
        *date = m_highdate;
        return true;
    }
    return false;
}

void dxCalendarCtrl::markDays(bool refresh, bool showhidden)
{
    m_showhidden = showhidden;
    if(m_markDays)
    {
        m_calendardays = iengine->updateCalendarDays(m_datum.GetMonth(), m_datum.GetYear(), showhidden);
    }
    if(refresh) Refresh();
}

void dxCalendarCtrl::Resize()
{
    InvalidateBestSize();
    DoGetBestSize();
}

void dxCalendarCtrl::setEventID(long id)
{
    m_eventID = id;
    Refresh();
}

wxUint8 dxCalendarCtrl::daytypeForDay() const
{
    return daytypeForDay(m_datum.GetDay());
}

bool dxCalendarCtrl::hasEventForDay() const
{
    return hasEventForDay(m_datum.GetDay());
}

bool dxCalendarCtrl::hasEventForDay(long id) const
{
    if(!m_markDays) return false;
    for(Calendarday cday : m_calendardays)
    {
        if(cday.day() == m_datum.GetDay())
        {
            wxStringTokenizer tok(cday.ids(), "|");
            while(tok.HasMoreTokens())
            {
                if(id == wxAtol(tok.GetNextToken())) return true;
            }
        }
    }
    return false;
}

void dxCalendarCtrl::OnMouseMove(wxMouseEvent &event)
{
    if(!m_markDays)
    {
        SetToolTip(wxEmptyString);
        return;
    }
    if(HitTest(event.GetPosition(), &m_datumForMove) == wxCAL_HITTEST_DAY)
    {
        for(Calendarday day : m_calendardays)
        {
            if(day.day() == m_datumForMove.GetDay())
            {
                SetToolTip(day.tooltip());
            }
        }
    }
    else
    {
        SetToolTip(wxEmptyString);
    }
}

void dxCalendarCtrl::OnPaint(wxPaintEvent &/*event*/)
{
    wxPaintDC dc(this);
    dc.SetFont(GetFont());
    recalcGeometry();
    wxColour colHeaderFg = *wxBLUE;
    wxColour colHeaderBg = *wxLIGHT_GREY;
    wxCoord y = 0;
    wxCoord x0 = m_calendarWeekWidth+(GetClientSize().x-7*m_widthCol-m_calendarWeekWidth)/2;
    // draw the sequential month-selector
    dc.SetBackgroundMode(wxBRUSHSTYLE_TRANSPARENT);
    dc.SetTextForeground(*wxBLACK);
    dc.SetBrush(wxBrush(colHeaderBg, wxBRUSHSTYLE_SOLID));
    dc.SetPen(wxPen(colHeaderBg, 1, wxPENSTYLE_SOLID));
    dc.DrawRectangle(0, y, GetClientSize().x, m_heightRow);
    // Get extent of month-name + year
    wxCoord monthw, monthh;
    wxString headertext = headerText();
    dc.GetTextExtent(headertext, &monthw, &monthh);
    // draw month-name centered above weekdays
    wxCoord monthx = ((m_widthCol * 7) - monthw) / 2 + x0;
    wxCoord monthy = ((m_heightRow - monthh) / 2) + y;
    dc.DrawText(headertext, monthx,  monthy);
    // calculate the "month-arrows"
    wxPoint leftarrow[3];
    wxPoint rightarrow[3];
    int arrowheight = monthh / 2;
    leftarrow[0] = wxPoint(0, arrowheight / 2);
    leftarrow[1] = wxPoint(arrowheight / 2, 0);
    leftarrow[2] = wxPoint(arrowheight / 2, arrowheight - 1);
    rightarrow[0] = wxPoint(0,0);
    rightarrow[1] = wxPoint(arrowheight / 2, arrowheight / 2);
    rightarrow[2] = wxPoint(0, arrowheight - 1);
    // draw the "month-arrows"
    wxCoord arrowy = (m_heightRow - arrowheight) / 2;
    wxCoord larrowx = (m_widthCol - (arrowheight / 2)) / 2 + x0;
    wxCoord rarrowx = ((m_widthCol - (arrowheight / 2)) / 2) + m_widthCol*6 + x0;
    m_leftArrowRect = m_rightArrowRect = m_todayRect = wxRect(0,0,0,0);
    wxDateTime ldpm = wxDateTime(1,m_datum.GetMonth(), m_datum.GetYear()) - wxDateSpan::Day(); // last day prev month
    // Check if range permits change
    if(isDateInRange(ldpm))
    {
        m_leftArrowRect = wxRect(larrowx - 3, arrowy - 3, (arrowheight / 2) + 8, (arrowheight + 6));
        dc.SetBrush(*wxBLACK_BRUSH);
        dc.SetPen(*wxBLACK_PEN);
        dc.DrawPolygon(3, leftarrow, larrowx , arrowy, wxWINDING_RULE);
        dc.SetBrush(*wxTRANSPARENT_BRUSH);
        dc.DrawRectangle(m_leftArrowRect);
    }
    wxDateTime fdnm = wxDateTime(1,m_datum.GetMonth(), m_datum.GetYear()) + wxDateSpan::Month(); // first day next month
    if(isDateInRange(fdnm))
    {
        m_rightArrowRect = wxRect(rarrowx - 4, arrowy - 3, (arrowheight / 2) + 8, (arrowheight + 6));
        dc.SetBrush(*wxBLACK_BRUSH);
        dc.SetPen(*wxBLACK_PEN);
        dc.DrawPolygon(3, rightarrow, rarrowx , arrowy, wxWINDING_RULE);
        dc.SetBrush(*wxTRANSPARENT_BRUSH);
        dc.DrawRectangle(m_rightArrowRect);
    }
    y += m_heightRow;
    // first draw the week days
    if(IsExposed(x0, y, x0 + 7*m_widthCol, m_heightRow))
    {
        dc.SetBackgroundMode(wxBRUSHSTYLE_TRANSPARENT);
        dc.SetTextForeground(colHeaderFg);
        dc.SetBrush(wxBrush(colHeaderBg, wxBRUSHSTYLE_SOLID));
        dc.SetPen(wxPen(colHeaderBg, 1, wxPENSTYLE_SOLID));
        dc.DrawRectangle(0, y, GetClientSize().x, m_heightRow);
        bool startOnMonday = HasFlag(wxCAL_MONDAY_FIRST);
        for ( int wd = 0; wd < 7; wd++ )
        {
            size_t n;
            if ( startOnMonday )
                n = wd == 6 ? 0 : wd + 1;
            else
                n = wd;
            wxCoord dayw, dayh;
            dc.GetTextExtent(m_weekdays[n], &dayw, &dayh);
            dc.DrawText(m_weekdays[n], x0 + (wd*m_widthCol) + ((m_widthCol- dayw) / 2), y); // center the day-name
        }
    }
    // then the calendar itself
    y += m_heightRow;
    // draw column with calendar week nr
    if(dxsettings.showWeekNumber())
    {
        dc.SetBackgroundMode(wxTRANSPARENT);
        dc.SetTextForeground(*wxBLACK);
        dc.SetBrush(wxBrush(colHeaderBg, wxBRUSHSTYLE_SOLID));
        dc.SetPen(wxPen(colHeaderBg, 1, wxPENSTYLE_SOLID));
        wxDateTime::wxDateTime_t weeks = dxutils::weeksofmonth(m_datum);
        dc.DrawRectangle(0, y, m_calendarWeekWidth, m_heightRow * weeks);
        wxDateTime date = startDate();
        for(wxDateTime::wxDateTime_t i = 0; i < weeks; ++i)
        {
            const int weekNr = date.GetWeekOfYear();
            wxString text = wxString::Format(wxT("%d"), weekNr);
            dc.DrawText(text, m_calendarWeekWidth - dc.GetTextExtent(text).GetWidth() - 2, y + m_heightRow * i);
            date += wxDateSpan::Week();
        }
    }
    dc.SetTextForeground(dxsettings.calForeground());
    wxDateTime date = startDate();
    dc.SetBackgroundMode(wxBRUSHSTYLE_SOLID);
    for(size_t nWeek = 1; nWeek <= 6; nWeek++, y += m_heightRow)
    {
        // if the update region doesn't intersect this row, don't paint it
        if(!IsExposed(x0, y, x0 + 7*m_widthCol, m_heightRow - 1))
        {
            date += wxDateSpan::Week();
            continue;
        }
        for(int wd = 0; wd < 7; wd++)
        {
            dc.SetTextBackground(dxsettings.calBackground());
            if(date.GetMonth() == m_datum.GetMonth())
            {
                // don't use wxDate::Format() which prepends 0s
                unsigned int day = date.GetDay();
                wxString dayStr = wxString::Format(wxT("%u"), day);
                wxCoord width;
                dc.GetTextExtent(dayStr, &width, nullptr);
                bool changedColours = false;
                bool isSel = false;
                isSel = date.IsSameDate(m_datum);
                if(isSel)
                {
                    dc.SetTextForeground(dxsettings.calHighlightFg());
                    dc.SetTextBackground(dxsettings.calHighlightBg());
                    changedColours = true;
                }
                else
                {
                    if(date.GetWeekDay()==wxDateTime::Sat || date.GetWeekDay()==wxDateTime::Sun)
                    {
                        dc.SetTextForeground(dxsettings.calWeekendFg());
                        dc.SetTextBackground(dxsettings.calWeekendBg());
                        changedColours = true;
                    }
                    if(iengine->isHoliday(date))
                    {
                        dc.SetTextForeground(dxsettings.calHolidayFg());
                        dc.SetTextBackground(dxsettings.calHolidayBg());
                        changedColours = true;
                    }
                }
                wxCoord x = wd*m_widthCol + (m_widthCol - width) / 2 + x0;
                dc.DrawText(dayStr, x, y + 1);
                if(daytypeForDay(date.GetDay()) == 1)
                {
                    wxPen pen(dxsettings.calReminder(), hasEventForDay(date.GetDay())?2:1, wxPENSTYLE_SOLID);
                    dc.SetPen(pen);
                    dc.SetBrush(*wxTRANSPARENT_BRUSH);
                    dc.DrawEllipse(x - (m_widthCol - width) / 2, y, m_widthCol, m_heightRow);
                }
                else if(daytypeForDay(date.GetDay()) == 2)
                {
                    wxPen pen(dxsettings.calRecurrence(), hasEventForDay(date.GetDay())?2:1, wxPENSTYLE_SOLID);
                    dc.SetPen(pen);
                    dc.SetBrush(*wxTRANSPARENT_BRUSH);
                    dc.DrawEllipse(x - (m_widthCol - width) / 2, y, m_widthCol, m_heightRow);
                }
                if(!isSel)
                {
#if !defined (__WXGTK__)
                    if(date.IsSameDate(wxDateTime::Today()))
                    {
                        wxPen pen(dxsettings.calHighlightBg(), 1, wxPENSTYLE_SOLID);
                        dc.SetPen(pen);
                        dc.SetBrush(*wxTRANSPARENT_BRUSH);
                        dc.DrawRectangle(x - 2, y, width + 4, m_heightRow);
                    }
#endif
                }
                if(changedColours)
                {
                    dc.SetTextForeground(dxsettings.calForeground());
                    dc.SetTextBackground(dxsettings.calBackground());
                }
            }
            date += wxDateSpan::Day();
        }
    }
    //draw today text
    dc.SetTextForeground(dxsettings.calForeground());
    dc.SetTextBackground(dxsettings.calBackground());
    dc.SetBackgroundMode(wxBRUSHSTYLE_SOLID);
    wxCoord todayw, todayh;
    wxString todaytext = todayText();
    dc.GetTextExtent(todaytext, &todayw, &todayh);
#if defined (__WXGTK__)
    wxCoord todayx = ((m_widthCol * 7) - todayw) / 2 + x0;
    wxCoord todayy = ((m_heightRow - todayh) / 2) + y;
    dc.DrawText(todaytext, todayx, todayy);
    m_todayRect = wxRect(todayx, todayy, todayw, todayh);
#else
    int width = m_widthCol/2;
    wxCoord todayx = ((m_widthCol * 7) - todayw) / 2 + x0 + (width+2)/2;
    wxCoord todayy = ((m_heightRow - todayh) / 2) + y;
    dc.DrawText(todaytext, todayx, todayy);
    wxPen pen(dxsettings.calHighlightBg(), 1, wxPENSTYLE_SOLID);
    dc.SetPen(pen);
    dc.SetBrush(*wxTRANSPARENT_BRUSH);
    dc.DrawRectangle(todayx-2, y, -width, m_heightRow);
    m_todayRect = wxRect(todayx-2-width, y+1, todayw+2+width, m_heightRow-2);
#endif
}

void dxCalendarCtrl::OnClick(wxMouseEvent &event)
{
    wxDateTime date;
    wxDateTime::WeekDay wday;
    switch(HitTest(event.GetPosition(), &date, &wday))
    {
        case wxCAL_HITTEST_DAY:
        if(isDateInRange(date))
        {
            changeDay(date);
            GenerateEvent(wxEVT_CALENDAR_SEL_CHANGED);
            // we know that the month/year never change when the user
            // clicks on the control so there is no need to call
            // GenerateAllChangeEvents() here, we know which event to send
            GenerateEvent(wxEVT_CALENDAR_DAY_CHANGED);
        }
        break;
        case wxCAL_HITTEST_WEEK:
        {
              setDateAndNotify(wxDateTime::Today());
//            wxCalendarEvent send(this, date, wxEVT_CALENDAR_WEEK_CLICKED);
//            HandleWindowEvent(send);
        }
        break;
        case wxCAL_HITTEST_HEADER:
        {
            wxCalendarEvent eventWd(this, GetDate(), wxEVT_CALENDAR_WEEKDAY_CLICKED);
            eventWd.SetWeekDay(wday);
            (void)GetEventHandler()->ProcessEvent(eventWd);
        }
        break;
        case wxCAL_HITTEST_DECMONTH:
        case wxCAL_HITTEST_INCMONTH:
        case wxCAL_HITTEST_SURROUNDING_WEEK:
        {
            setDateAndNotify(date);
        }
        break;
        default:
            wxFAIL_MSG(wxT("unknown hittest code"));
            // fall through
        case wxCAL_HITTEST_NOWHERE:
            event.Skip();
            break;
    }
    // as we don't (always) skip the message, we're not going to receive the
    // focus on click by default if we don't do it ourselves
    SetFocus();
}

void dxCalendarCtrl::OnRightClick(wxMouseEvent &event)
{
    wxDateTime date;
    wxDateTime::WeekDay wday;
    if(HitTest(event.GetPosition(), &date, &wday) == wxCAL_HITTEST_DAY)
    {
        if(dxsettings.rightclickNewevent())
        {
            EventDialog dialog(this, date);
            dialog.ShowModal();
        }
        else
        {
            event.Skip();
        }
    }
    else
    {
        event.Skip();
    }
}

void dxCalendarCtrl::OnChar(wxKeyEvent& event)
{
    switch ( event.GetKeyCode() )
    {
        case wxT('+'):
        case WXK_ADD:
            setDateAndNotify(m_datum + wxDateSpan::Year());
            break;

        case wxT('-'):
        case WXK_SUBTRACT:
            setDateAndNotify(m_datum - wxDateSpan::Year());
            break;

        case WXK_PAGEUP:
            setDateAndNotify(m_datum - wxDateSpan::Month());
            break;

        case WXK_PAGEDOWN:
            setDateAndNotify(m_datum + wxDateSpan::Month());
            break;

        case WXK_RIGHT:
            setDateAndNotify(m_datum + wxDateSpan::Day());
            break;

        case WXK_LEFT:
            setDateAndNotify(m_datum - wxDateSpan::Day());
            break;

        case WXK_UP:
            setDateAndNotify(m_datum - wxDateSpan::Week());
            break;

        case WXK_DOWN:
            setDateAndNotify(m_datum + wxDateSpan::Week());
            break;

        case WXK_HOME:
            if ( event.ControlDown() )
                setDateAndNotify(wxDateTime::Today());
            else
                setDateAndNotify(wxDateTime(1, m_datum.GetMonth(), m_datum.GetYear()));
            break;

        case WXK_END:
            setDateAndNotify(wxDateTime(m_datum).SetToLastMonthDay());
            break;

        case WXK_RETURN:
            GenerateEvent(wxEVT_CALENDAR_DOUBLECLICKED);
            break;

        default:
            event.Skip();
    }
}

wxDateTime::WeekDay dxCalendarCtrl::GetWeekStart() const
{
    return HasFlag(wxCAL_MONDAY_FIRST) ? wxDateTime::Mon
                                       : wxDateTime::Sun;
}

wxDateTime::WeekDay dxCalendarCtrl::GetWeekEnd() const
{
    return HasFlag(wxCAL_MONDAY_FIRST) ? wxDateTime::Sun
                                       : wxDateTime::Sat;
}

wxString dxCalendarCtrl::tooltipForDay(wxDateTime::wxDateTime_t day) const
{
    if(!m_markDays) return wxEmptyString;
    for(Calendarday cday : m_calendardays)
    {
        if(cday.day() == day)
        {
            return cday.tooltip();
        }
    }
    return wxEmptyString;
}

wxUint8 dxCalendarCtrl::daytypeForDay(wxDateTime::wxDateTime_t day) const
{
    if(!m_markDays) return 0;
    for(Calendarday cday : m_calendardays)
    {
        if(cday.day() == day)
        {
            return cday.daytype();
        }
    }
    return 0;
}

bool dxCalendarCtrl::hasEventForDay(wxDateTime::wxDateTime_t day) const
{
    if(!m_markDays) return false;
    for(Calendarday cday : m_calendardays)
    {
        if(cday.day() == day)
        {
            wxStringTokenizer tok(cday.ids(), "|");
            while(tok.HasMoreTokens())
            {
                if(m_eventID == wxAtol(tok.GetNextToken())) return true;
            }
        }
    }
    return false;
}

wxDateTime dxCalendarCtrl::startDate() const
{
    wxDateTime::Tm tm = m_datum.GetTm();
    wxDateTime date = wxDateTime(1, tm.mon, tm.year);
    // rewind back
    date.SetToPrevWeekDay(wxDateTime::Mon);
    return date;
}

void dxCalendarCtrl::recalcGeometry()
{
    wxClientDC dc(this);
    if(m_smaller) dc.SetFont(dxutils::fontFromString(dxsettings.calMiniFont()));
    else dc.SetFont(dxutils::fontFromString(dxsettings.calFont()));
    // determine the column width (weekday names are not necessarily wider
    // than the numbers (in some languages), so let's not assume that they are)
    m_widthCol = 0;
    for(int day = 10; day <= 31; day++)
    {
        wxCoord width;
        dc.GetTextExtent(wxString::Format(wxT("%d"), day), &width, &m_heightRow);
        if(width > m_widthCol)
        {
            // 1.5 times the width gives nice margins even if the weekday
            // names are short
            m_widthCol = width+width/2;
        }
    }
    wxDateTime::WeekDay wd;
    for(wd = wxDateTime::Sun; wd < wxDateTime::Inv_WeekDay; wxNextWDay(wd))
    {
        wxCoord width;
        dc.GetTextExtent(m_weekdays[wd], &width, &m_heightRow);
        if(width > m_widthCol)
        {
            m_widthCol = width+2;
        }
    }
    m_calendarWeekWidth = dxsettings.showWeekNumber() ? dc.GetTextExtent(wxString::Format(wxT( "%d" ), 42)).GetWidth() + 4 : 0;
    // leave some margins
    m_widthCol += 2;
    m_heightRow += 2;
    m_rowOffset = HasFlag(wxCAL_SEQUENTIAL_MONTH_SELECTION) ? m_heightRow : 0;
    wxSize tsize = dc.GetTextExtent(todayText());
#if defined (__WXGTK__)
    m_widthToday = tsize.x;
#else
    m_widthToday = 2*m_widthCol+2+m_widthCol/2+tsize.x;
#endif
}

bool dxCalendarCtrl::isDateInRange(const wxDateTime &date) const
{
    return ((m_lowdate.IsValid())?(date >= m_lowdate):true) && ((m_highdate.IsValid())?(date <= m_highdate):true);
}

void dxCalendarCtrl::changeDay(const wxDateTime &date)
{
    if(m_datum != date)
    {
        // we need to refresh the row containing the old date and the one
        // containing the new one
        wxDateTime dateOld = m_datum;
        m_datum = date;
        refreshDate(dateOld);
        // if the date is in the same row, it was already drawn correctly
        if(week(m_datum) != week(dateOld))
        {
            refreshDate(m_datum);
        }
    }
}

void dxCalendarCtrl::refreshDate(const wxDateTime& /*date*/)
{
    recalcGeometry();
    wxRect rect;
    rect.x = 0;
    rect.y = m_rowOffset;
    rect.width = GetClientSize().x;
    rect.height = 7*m_heightRow;
//    rect.y = (m_heightRow * GetWeek(date)) + m_rowOffset;
//    rect.width = 7*m_widthCol;
//    rect.height = m_heightRow;
//#ifdef __WXMSW__
//    // VZ: for some reason, the selected date seems to occupy more space under
//    //     MSW - this is probably some bug in the font size calculations, but I
//    //     don't know where exactly. This fix is ugly and leads to more
//    //     refreshes than really needed, but without it the selected days
//    //     leaves even more ugly underscores on screen.
//    rect.Inflate(0, 1);
//#endif // MSW
    Refresh(true, &rect);
}

size_t dxCalendarCtrl::week(const wxDateTime& date) const
{
    return date.GetWeekOfMonth(HasFlag(wxCAL_MONDAY_FIRST) ? wxDateTime::Monday_First : wxDateTime::Sunday_First);
}

void dxCalendarCtrl::setDateAndNotify(const wxDateTime& date)
{
    const wxDateTime dateOld = GetDate();
    if(date != dateOld && SetDate(date))
    {
        GenerateAllChangeEvents(dateOld);
    }
}

wxString dxCalendarCtrl::todayText()
{
    return _("Today:")+" "+dxutils::formatDate(wxDateTime::Today(), dxsettings.dateFormat(), true);
}

wxString dxCalendarCtrl::headerText()
{
    if(dxsettings.locale().StartsWith("cs"))
    {
        wxString text;
        switch (m_datum.GetMonth()) {
        case wxDateTime::Jan: text = "Leden"; break;
        case wxDateTime::Feb: text = wxString::FromUTF8("Únor"); break;
        case wxDateTime::Mar: text = wxString::FromUTF8("Březen"); break;
        case wxDateTime::Apr: text = "Duben"; break;
        case wxDateTime::May: text = wxString::FromUTF8("Květen"); break;
        case wxDateTime::Jun: text = wxString::FromUTF8("Červen"); break;
        case wxDateTime::Jul: text = wxString::FromUTF8("Červenec"); break;
        case wxDateTime::Aug: text = "Srpen"; break;
        case wxDateTime::Sep: text = wxString::FromUTF8("Září"); break;
        case wxDateTime::Oct: text = wxString::FromUTF8("Říjen"); break;
        case wxDateTime::Nov: text = "Listopad"; break;
        case wxDateTime::Dec: text = "Prosinec"; break;
        default: text = "Prd";
        }
        text << " " << m_datum.Format("%Y");
        return text;
    }
    return m_datum.Format("%B %Y");
}

BEGIN_EVENT_TABLE(dxCalendar, wxPanel)
END_EVENT_TABLE()

dxCalendar::dxCalendar(wxWindow *parent, const wxWindowID id, bool smaller, bool markDays)
    : wxPanel(parent)
{
    m_mainSizer = new wxBoxSizer(wxHORIZONTAL);
    m_calendar = new dxCalendarCtrl(this, id, smaller, markDays);
    m_mainSizer->Add(m_calendar);
    this->SetSizer(m_mainSizer);
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
}

void dxCalendar::updateCalendar()
{
    m_calendar->updateColors();
    m_calendar->updateFonts();
    m_calendar->Resize();
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
}

wxDateTime dxCalendar::getValue() const
{
    return m_calendar->GetDate();
}

void dxCalendar::setValue(const wxDateTime &date)
{
    m_calendar->SetDate(date);
}

void dxCalendar::markDays(bool showhidden)
{
    m_calendar->markDays(true, showhidden);
}

void dxCalendar::refreshFrame()
{
    m_calendar->Refresh();
}

void dxCalendar::setEventID(long id)
{
    m_calendar->setEventID(id);
}

wxUint8 dxCalendar::daytypeForDay() const
{
    return m_calendar->daytypeForDay();
}

bool dxCalendar::hasEventForDay() const
{
    return m_calendar->hasEventForDay();
}

bool dxCalendar::hasEventForDay(long id) const
{
    return m_calendar->hasEventForDay(id);
}
