"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var bolt_protocol_util_1 = require("./bolt-protocol-util");
var packstream_1 = require("../packstream");
var request_message_1 = __importDefault(require("./request-message"));
var stream_observers_1 = require("./stream-observers");
var neo4j_driver_core_1 = require("neo4j-driver-core");
var Bookmark = neo4j_driver_core_1.internal.bookmark.Bookmark, _a = neo4j_driver_core_1.internal.constants, ACCESS_MODE_WRITE = _a.ACCESS_MODE_WRITE, BOLT_PROTOCOL_V1 = _a.BOLT_PROTOCOL_V1, Logger = neo4j_driver_core_1.internal.logger.Logger, TxConfig = neo4j_driver_core_1.internal.txConfig.TxConfig;
var BoltProtocol = /** @class */ (function () {
    /**
     * @callback CreateResponseHandler Creates the response handler
     * @param {BoltProtocol} protocol The bolt protocol
     * @returns {ResponseHandler} The response handler
     */
    /**
     * @callback OnProtocolError Handles protocol error
     * @param {string} error The description
     */
    /**
     * @constructor
     * @param {Object} server the server informatio.
     * @param {Chunker} chunker the chunker.
     * @param {Object} packstreamConfig Packstream configuration
     * @param {boolean} packstreamConfig.disableLosslessIntegers if this connection should convert all received integers to native JS numbers.
     * @param {boolean} packstreamConfig.useBigInt if this connection should convert all received integers to native BigInt numbers.
     * @param {CreateResponseHandler} createResponseHandler Function which creates the response handler
     * @param {Logger} log the logger
     * @param {OnProtocolError} onProtocolError handles protocol errors
     */
    function BoltProtocol(server, chunker, _a, createResponseHandler, log, onProtocolError) {
        var _b = _a === void 0 ? {} : _a, disableLosslessIntegers = _b.disableLosslessIntegers, useBigInt = _b.useBigInt;
        if (createResponseHandler === void 0) { createResponseHandler = function () { return null; }; }
        this._server = server || {};
        this._chunker = chunker;
        this._packer = this._createPacker(chunker);
        this._unpacker = this._createUnpacker(disableLosslessIntegers, useBigInt);
        this._responseHandler = createResponseHandler(this);
        this._log = log;
        this._onProtocolError = onProtocolError;
        this._fatalError = null;
    }
    Object.defineProperty(BoltProtocol.prototype, "version", {
        /**
         * Returns the numerical version identifier for this protocol
         */
        get: function () {
            return BOLT_PROTOCOL_V1;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Get the packer.
     * @return {Packer} the protocol's packer.
     */
    BoltProtocol.prototype.packer = function () {
        return this._packer;
    };
    /**
     * Get the unpacker.
     * @return {Unpacker} the protocol's unpacker.
     */
    BoltProtocol.prototype.unpacker = function () {
        return this._unpacker;
    };
    /**
     * Transform metadata received in SUCCESS message before it is passed to the handler.
     * @param {Object} metadata the received metadata.
     * @return {Object} transformed metadata.
     */
    BoltProtocol.prototype.transformMetadata = function (metadata) {
        return metadata;
    };
    /**
     * Perform initialization and authentication of the underlying connection.
     * @param {Object} param
     * @param {string} param.userAgent the user agent.
     * @param {Object} param.authToken the authentication token.
     * @param {function(err: Error)} param.onError the callback to invoke on error.
     * @param {function()} param.onComplete the callback to invoke on completion.
     * @returns {StreamObserver} the stream observer that monitors the corresponding server response.
     */
    BoltProtocol.prototype.initialize = function (_a) {
        var _this = this;
        var _b = _a === void 0 ? {} : _a, userAgent = _b.userAgent, authToken = _b.authToken, onError = _b.onError, onComplete = _b.onComplete;
        var observer = new stream_observers_1.LoginObserver({
            onError: function (error) { return _this._onLoginError(error, onError); },
            onCompleted: function (metadata) { return _this._onLoginCompleted(metadata, onComplete); }
        });
        this.write(request_message_1.default.init(userAgent, authToken), observer, true);
        return observer;
    };
    /**
     * Perform protocol related operations for closing this connection
     */
    BoltProtocol.prototype.prepareToClose = function () {
        // no need to notify the database in this protocol version
    };
    /**
     * Begin an explicit transaction.
     * @param {Object} param
     * @param {Bookmark} param.bookmark the bookmark.
     * @param {TxConfig} param.txConfig the configuration.
     * @param {string} param.database the target database name.
     * @param {string} param.mode the access mode.
     * @param {string} param.impersonatedUser the impersonated user
     * @param {function(err: Error)} param.beforeError the callback to invoke before handling the error.
     * @param {function(err: Error)} param.afterError the callback to invoke after handling the error.
     * @param {function()} param.beforeComplete the callback to invoke before handling the completion.
     * @param {function()} param.afterComplete the callback to invoke after handling the completion.
     * @returns {StreamObserver} the stream observer that monitors the corresponding server response.
     */
    BoltProtocol.prototype.beginTransaction = function (_a) {
        var _b = _a === void 0 ? {} : _a, bookmark = _b.bookmark, txConfig = _b.txConfig, database = _b.database, mode = _b.mode, impersonatedUser = _b.impersonatedUser, beforeError = _b.beforeError, afterError = _b.afterError, beforeComplete = _b.beforeComplete, afterComplete = _b.afterComplete;
        return this.run('BEGIN', bookmark ? bookmark.asBeginTransactionParameters() : {}, {
            bookmark: bookmark,
            txConfig: txConfig,
            database: database,
            mode: mode,
            impersonatedUser: impersonatedUser,
            beforeError: beforeError,
            afterError: afterError,
            beforeComplete: beforeComplete,
            afterComplete: afterComplete,
            flush: false
        });
    };
    /**
     * Commit the explicit transaction.
     * @param {Object} param
     * @param {function(err: Error)} param.beforeError the callback to invoke before handling the error.
     * @param {function(err: Error)} param.afterError the callback to invoke after handling the error.
     * @param {function()} param.beforeComplete the callback to invoke before handling the completion.
     * @param {function()} param.afterComplete the callback to invoke after handling the completion.
     * @returns {StreamObserver} the stream observer that monitors the corresponding server response.
     */
    BoltProtocol.prototype.commitTransaction = function (_a) {
        var _b = _a === void 0 ? {} : _a, beforeError = _b.beforeError, afterError = _b.afterError, beforeComplete = _b.beforeComplete, afterComplete = _b.afterComplete;
        // WRITE access mode is used as a place holder here, it has
        // no effect on behaviour for Bolt V1 & V2
        return this.run('COMMIT', {}, {
            bookmark: Bookmark.empty(),
            txConfig: TxConfig.empty(),
            mode: ACCESS_MODE_WRITE,
            beforeError: beforeError,
            afterError: afterError,
            beforeComplete: beforeComplete,
            afterComplete: afterComplete
        });
    };
    /**
     * Rollback the explicit transaction.
     * @param {Object} param
     * @param {function(err: Error)} param.beforeError the callback to invoke before handling the error.
     * @param {function(err: Error)} param.afterError the callback to invoke after handling the error.
     * @param {function()} param.beforeComplete the callback to invoke before handling the completion.
     * @param {function()} param.afterComplete the callback to invoke after handling the completion.
     * @returns {StreamObserver} the stream observer that monitors the corresponding server response.
     */
    BoltProtocol.prototype.rollbackTransaction = function (_a) {
        var _b = _a === void 0 ? {} : _a, beforeError = _b.beforeError, afterError = _b.afterError, beforeComplete = _b.beforeComplete, afterComplete = _b.afterComplete;
        // WRITE access mode is used as a place holder here, it has
        // no effect on behaviour for Bolt V1 & V2
        return this.run('ROLLBACK', {}, {
            bookmark: Bookmark.empty(),
            txConfig: TxConfig.empty(),
            mode: ACCESS_MODE_WRITE,
            beforeError: beforeError,
            afterError: afterError,
            beforeComplete: beforeComplete,
            afterComplete: afterComplete
        });
    };
    /**
     * Send a Cypher query through the underlying connection.
     * @param {string} query the cypher query.
     * @param {Object} parameters the query parameters.
     * @param {Object} param
     * @param {Bookmark} param.bookmark the bookmark.
     * @param {TxConfig} param.txConfig the transaction configuration.
     * @param {string} param.database the target database name.
     * @param {string} param.impersonatedUser the impersonated user
     * @param {string} param.mode the access mode.
     * @param {function(keys: string[])} param.beforeKeys the callback to invoke before handling the keys.
     * @param {function(keys: string[])} param.afterKeys the callback to invoke after handling the keys.
     * @param {function(err: Error)} param.beforeError the callback to invoke before handling the error.
     * @param {function(err: Error)} param.afterError the callback to invoke after handling the error.
     * @param {function()} param.beforeComplete the callback to invoke before handling the completion.
     * @param {function()} param.afterComplete the callback to invoke after handling the completion.
     * @param {boolean} param.flush whether to flush the buffered messages.
     * @returns {StreamObserver} the stream observer that monitors the corresponding server response.
     */
    BoltProtocol.prototype.run = function (query, parameters, _a) {
        var _b = _a === void 0 ? {} : _a, bookmark = _b.bookmark, txConfig = _b.txConfig, database = _b.database, mode = _b.mode, impersonatedUser = _b.impersonatedUser, beforeKeys = _b.beforeKeys, afterKeys = _b.afterKeys, beforeError = _b.beforeError, afterError = _b.afterError, beforeComplete = _b.beforeComplete, afterComplete = _b.afterComplete, _c = _b.flush, flush = _c === void 0 ? true : _c;
        var observer = new stream_observers_1.ResultStreamObserver({
            server: this._server,
            beforeKeys: beforeKeys,
            afterKeys: afterKeys,
            beforeError: beforeError,
            afterError: afterError,
            beforeComplete: beforeComplete,
            afterComplete: afterComplete
        });
        // bookmark and mode are ignored in this version of the protocol
        (0, bolt_protocol_util_1.assertTxConfigIsEmpty)(txConfig, this._onProtocolError, observer);
        // passing in a database name on this protocol version throws an error
        (0, bolt_protocol_util_1.assertDatabaseIsEmpty)(database, this._onProtocolError, observer);
        // passing impersonated user on this protocol version throws an error
        (0, bolt_protocol_util_1.assertImpersonatedUserIsEmpty)(impersonatedUser, this._onProtocolError, observer);
        this.write(request_message_1.default.run(query, parameters), observer, false);
        this.write(request_message_1.default.pullAll(), observer, flush);
        return observer;
    };
    Object.defineProperty(BoltProtocol.prototype, "currentFailure", {
        get: function () {
            return this._responseHandler.currentFailure;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Send a RESET through the underlying connection.
     * @param {Object} param
     * @param {function(err: Error)} param.onError the callback to invoke on error.
     * @param {function()} param.onComplete the callback to invoke on completion.
     * @returns {StreamObserver} the stream observer that monitors the corresponding server response.
     */
    BoltProtocol.prototype.reset = function (_a) {
        var _b = _a === void 0 ? {} : _a, onError = _b.onError, onComplete = _b.onComplete;
        var observer = new stream_observers_1.ResetObserver({
            onProtocolError: this._onProtocolError,
            onError: onError,
            onComplete: onComplete
        });
        this.write(request_message_1.default.reset(), observer, true);
        return observer;
    };
    BoltProtocol.prototype._createPacker = function (chunker) {
        return new packstream_1.v1.Packer(chunker);
    };
    BoltProtocol.prototype._createUnpacker = function (disableLosslessIntegers, useBigInt) {
        return new packstream_1.v1.Unpacker(disableLosslessIntegers, useBigInt);
    };
    /**
     * Write a message to the network channel.
     * @param {RequestMessage} message the message to write.
     * @param {StreamObserver} observer the response observer.
     * @param {boolean} flush `true` if flush should happen after the message is written to the buffer.
     */
    BoltProtocol.prototype.write = function (message, observer, flush) {
        var _this = this;
        var queued = this.queueObserverIfProtocolIsNotBroken(observer);
        if (queued) {
            if (this._log.isDebugEnabled()) {
                this._log.debug("C: " + message);
            }
            this.packer().packStruct(message.signature, message.fields.map(function (field) { return _this.packer().packable(field); }));
            this._chunker.messageBoundary();
            if (flush) {
                this._chunker.flush();
            }
        }
    };
    /**
     * Notifies faltal erros to the observers and mark the protocol in the fatal error state.
     * @param {Error} error The error
     */
    BoltProtocol.prototype.notifyFatalError = function (error) {
        this._fatalError = error;
        return this._responseHandler._notifyErrorToObservers(error);
    };
    /**
     * Updates the the current observer with the next one on the queue.
     */
    BoltProtocol.prototype.updateCurrentObserver = function () {
        return this._responseHandler._updateCurrentObserver();
    };
    /**
     * Checks if exist an ongoing observable requests
     * @return {boolean}
     */
    BoltProtocol.prototype.hasOngoingObservableRequests = function () {
        return this._responseHandler.hasOngoingObservableRequests();
    };
    /**
     * Enqueue the observer if the protocol is not broken.
     * In case it's broken, the observer will be notified about the error.
     *
     * @param {StreamObserver} observer The observer
     * @returns {boolean} if it was queued
     */
    BoltProtocol.prototype.queueObserverIfProtocolIsNotBroken = function (observer) {
        if (this.isBroken()) {
            this.notifyFatalErrorToObserver(observer);
            return false;
        }
        return this._responseHandler._queueObserver(observer);
    };
    /**
     * Veritfy the protocol is not broken.
     * @returns {boolean}
     */
    BoltProtocol.prototype.isBroken = function () {
        return !!this._fatalError;
    };
    /**
     * Notifies the current fatal error to the observer
     *
     * @param {StreamObserver} observer The observer
     */
    BoltProtocol.prototype.notifyFatalErrorToObserver = function (observer) {
        if (observer && observer.onError) {
            observer.onError(this._fatalError);
        }
    };
    /**
     * Reset current failure on the observable response handler to null.
     */
    BoltProtocol.prototype.resetFailure = function () {
        this._responseHandler._resetFailure();
    };
    BoltProtocol.prototype._onLoginCompleted = function (metadata, onCompleted) {
        if (metadata) {
            var serverVersion = metadata.server;
            if (!this._server.version) {
                this._server.version = serverVersion;
            }
        }
        if (onCompleted) {
            onCompleted(metadata);
        }
    };
    BoltProtocol.prototype._onLoginError = function (error, onError) {
        this._onProtocolError(error.message);
        if (onError) {
            onError(error);
        }
    };
    return BoltProtocol;
}());
exports.default = BoltProtocol;
