"use strict";
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectivityVerifier = void 0;
var connection_holder_1 = require("./connection-holder");
var constants_1 = require("./constants");
var error_1 = require("../error");
/**
 * Verifies connectivity using the given connection provider.
 */
var ConnectivityVerifier = /** @class */ (function () {
    /**
     * @constructor
     * @param {ConnectionProvider} connectionProvider the provider to obtain connections from.
     */
    function ConnectivityVerifier(connectionProvider) {
        this._connectionProvider = connectionProvider;
    }
    /**
     * Try to obtain a working connection from the connection provider.
     * @returns {Promise<object>} promise resolved with server info or rejected with error.
     */
    ConnectivityVerifier.prototype.verify = function (_a) {
        var _b = _a === void 0 ? {} : _a, _c = _b.database, database = _c === void 0 ? '' : _c;
        return acquireAndReleaseDummyConnection(this._connectionProvider, database);
    };
    return ConnectivityVerifier;
}());
exports.ConnectivityVerifier = ConnectivityVerifier;
/**
 * @private
 * @param {ConnectionProvider} connectionProvider the provider to obtain connections from.
 * @param {string|undefined} database The database name
 * @return {Promise<object>} promise resolved with server info or rejected with error.
 */
function acquireAndReleaseDummyConnection(connectionProvider, database) {
    var connectionHolder = new connection_holder_1.ConnectionHolder({
        mode: constants_1.ACCESS_MODE_READ,
        database: database,
        connectionProvider: connectionProvider
    });
    connectionHolder.initializeConnection();
    return connectionHolder
        .getConnection()
        .then(function (connection) {
        // able to establish a connection
        if (!connection) {
            throw (0, error_1.newError)('Unexpected error acquiring transaction');
        }
        return connectionHolder.close().then(function () { return connection.server; });
    })
        .catch(function (error) {
        // failed to establish a connection
        return connectionHolder
            .close()
            .catch(function (ignoredError) {
            // ignore connection release error
        })
            .then(function () {
            return Promise.reject(error);
        });
    });
}
