/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import Connection from '../connection';
import { Bookmark } from './bookmark';
import ConnectionProvider from '../connection-provider';
/**
 * @private
 */
interface ConnectionHolderInterface {
    /**
     * Returns the assigned access mode.
     * @returns {string} access mode
     */
    mode(): string | undefined;
    /**
     * Returns the target database name
     * @returns {string} the database name
     */
    database(): string | undefined;
    /**
     * Returns the bookmark
     */
    bookmark(): Bookmark;
    /**
     * Make this holder initialize new connection if none exists already.
     * @return {boolean}
     */
    initializeConnection(): boolean;
    /**
     * Get the current connection promise.
     * @return {Promise<Connection>} promise resolved with the current connection.
     */
    getConnection(): Promise<Connection | void>;
    /**
     * Notify this holder that single party does not require current connection any more.
     * @return {Promise<Connection>} promise resolved with the current connection, never a rejected promise.
     */
    releaseConnection(): Promise<Connection | void>;
    /**
     * Closes this holder and releases current connection (if any) despite any existing users.
     * @return {Promise<Connection>} promise resolved when current connection is released to the pool.
     */
    close(): Promise<Connection | void>;
}
/**
 * Utility to lazily initialize connections and return them back to the pool when unused.
 * @private
 */
declare class ConnectionHolder implements ConnectionHolderInterface {
    private _mode;
    private _database?;
    private _bookmark;
    private _connectionProvider?;
    private _referenceCount;
    private _connectionPromise;
    private _impersonatedUser?;
    private _onDatabaseNameResolved?;
    /**
     * @constructor
     * @param {object} params
     * @property {string} params.mode - the access mode for new connection holder.
     * @property {string} params.database - the target database name.
     * @property {Bookmark} params.bookmark - the last bookmark
     * @property {ConnectionProvider} params.connectionProvider - the connection provider to acquire connections from.
     * @property {string?} params.impersonatedUser - the user which will be impersonated
     * @property {function(databaseName:string)} params.onDatabaseNameResolved - callback called when the database name is resolved
     */
    constructor({ mode, database, bookmark, connectionProvider, impersonatedUser, onDatabaseNameResolved }?: {
        mode?: string;
        database?: string;
        bookmark?: Bookmark;
        connectionProvider?: ConnectionProvider;
        impersonatedUser?: string;
        onDatabaseNameResolved?: (databaseName?: string) => void;
    });
    mode(): string | undefined;
    database(): string | undefined;
    setDatabase(database?: string): void;
    bookmark(): Bookmark;
    connectionProvider(): ConnectionProvider | undefined;
    referenceCount(): number;
    initializeConnection(): boolean;
    getConnection(): Promise<Connection | void>;
    releaseConnection(): Promise<void | Connection>;
    close(): Promise<void | Connection>;
    /**
     * Return the current pooled connection instance to the connection pool.
     * We don't pool Session instances, to avoid users using the Session after they've called close.
     * The `Session` object is just a thin wrapper around Connection anyway, so it makes little difference.
     * @return {Promise} - promise resolved then connection is returned to the pool.
     * @private
     */
    private _releaseConnection;
}
/**
 * Provides a interaction with a ConnectionHolder without change it state by
 * releasing or initilizing
 */
export default class ReadOnlyConnectionHolder extends ConnectionHolder {
    private _connectionHolder;
    /**
     * Contructor
     * @param {ConnectionHolder} connectionHolder the connection holder which will treat the requests
     */
    constructor(connectionHolder: ConnectionHolder);
    /**
     * Return the true if the connection is suppose to be initilized with the command.
     *
     * @return {boolean}
     */
    initializeConnection(): boolean;
    /**
     * Get the current connection promise.
     * @return {Promise<Connection>} promise resolved with the current connection.
     */
    getConnection(): Promise<Connection | void>;
    /**
     * Get the current connection promise, doesn't performs the release
     * @return {Promise<Connection>} promise with the resolved current connection
     */
    releaseConnection(): Promise<Connection | void>;
    /**
     * Get the current connection promise, doesn't performs the connection close
     * @return {Promise<Connection>} promise with the resolved current connection
     */
    close(): Promise<Connection | void>;
}
declare class EmptyConnectionHolder extends ConnectionHolder {
    mode(): undefined;
    database(): undefined;
    initializeConnection(): boolean;
    getConnection(): Promise<Connection>;
    releaseConnection(): Promise<void>;
    close(): Promise<void>;
}
/**
 * Connection holder that does not manage any connections.
 * @type {ConnectionHolder}
 * @private
 */
declare const EMPTY_CONNECTION_HOLDER: EmptyConnectionHolder;
export { ConnectionHolder, ReadOnlyConnectionHolder, EMPTY_CONNECTION_HOLDER };
