/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015 - 2023 by Kay F. Jahnke                    */
/*                                                                      */
/*    The git repository for this software is at                        */
/*                                                                      */
/*    https://bitbucket.org/kfj/vspline                                 */
/*                                                                      */
/*    Please direct questions, bug reports, and contributions to        */
/*                                                                      */
/*    kfjahnke+vspline@gmail.com                                        */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/*! \file poles.h

    \brief precalculated prefilter poles and basis function values

    The contents of this file below the comments can be generated
    using bootstrap.cc

    both the precalculated basis function values and the prefilter poles
    can be generated in arbitrary precision, so the constants below are
    given as 'xlf_type' - concrete splines will downcast them to whatever
    precision they use for prefiltering (usually the same type as the data
    type's elementary type).
    
    The values defined here are used in several places in vspline. They are
    precomputed because calculating them when needed can be (potentially very)
    expensive, and providing them by definitions evaluated at compile time
    slows compilation. Great care is taken to provide very exact values.
    The code in bootstrap.cc goes beyond simply calculating polynomial
    roots with gsl/blas, adding a polishing stage done in arbitrary precision
    arithmetic. The basis function values are done as GMP fractions, and only
    divided out before the result is printed out.
    
    The set of values provided here is sufficient to calculate the b-spline
    basis function for all spline degrees for arbitrary arguments (including
    the spline's derivatives) - see basis.h. The poles are needed for prefiltering.
    
    Note: I have calculated the basis function values and prefilter poles
    using arbitrary-precision library GNU GMP. The precision of these values
    is way beyond long double and well sufficient to initialize quad floats
    (__float128) up to their precision, which is only 80bit mantissa or so.
    So now I express the literals below with a macro which adds a suffix to them
    so they can be interpreted as literals of the precision wanted for the
    given compile.
*/

#include "common.h" // to have xlf_type and macro XLF()

#ifndef VSPLINE_POLES_H

// if 'USE_QUAD' is defined, the literals in poles.h will be interpreted
// as quad literals (with 'Q' suffix). otherwise, the suffix used will be L

// TODO: on my system, I tried to compile with
// g++ -std=gnu++11 -fext-numeric-literals -DUSE_QUAD but I got
// 'error: unable to find numeric literal operator ‘operator""Q’'
// The quad literals can be enabled with -std=gnu++11 for g++; clang++
// accepts them without extra compiler arguments. The literals in poles.h
// are precise enough to be used for even higher precision, with their
// 64 correct postcomma digits. should be reasonably future-proof ;)

// I tried defining xlf_type as __float128 when USE_QUAD is defined, but
// the resulting code does not compile. The problem is that __float128 is
// not a standard type and the standard maths functions like 'pow' don't work
// with it. But using quad literals does work (on g++ it needs -std=gnu++11)
// There is not much point in using quad literals in vspline as it stands,
// but I leave the code in here for now. Maybe higher precision floats become
// standard at some point, and then I can just up xlf_type (in common.h).

#ifdef USE_QUAD

#define XLF(arg) (arg##Q)

#else

#define XLF(arg) (arg##L)

#endif // #ifdef USE_QUAD

namespace vspline_constants
{

using vspline::xlf_type ;
  
/// to allow code to inquire about the maximal spline degree possible

static const int max_degree { 45 } ;

// For long doubles, the number of postcomma digits I provide is clearly
// total overkill, but it leaves enough room to manoevre if the need for
// more precise values should arise - and when assigned to 'lesser' types
// the value is rounded to what is actually used.
// I have refrained from giving the basis function values exact (as fractions
// of integral values) - if this should be needed, the way to obtain these
// values can easily be gleaned by looking at bootstrap.cc, which does
// initially compute such fractions and only casts them to floating point
// before echoing them.
// For most real applications, the number of degrees covered is also overkill.

// TODO how to communicate K0[-.5]==1?

// Note that the basis function values are given for positive x only, in half
// unit steps so, x € [0.0, 0.5, 1.0, 1.5 ...]
// The standard reconstruction kernel for a cubic b-spline would therefore
// be { K3[2] , K3[0], K3[2] }

const xlf_type K0[] = {
  XLF(1.0) ,
} ;
const xlf_type K1[] = {
  XLF(1.0) ,
  XLF(0.5) ,
} ;
const xlf_type K2[] = {
  XLF(0.75) ,
  XLF(0.5) ,
  XLF(0.125) ,
} ;
const xlf_type K3[] = {
  XLF(0.6666666666666666666666666666666666666666666666666666666666666667) ,
  XLF(0.4791666666666666666666666666666666666666666666666666666666666667) ,
  XLF(0.1666666666666666666666666666666666666666666666666666666666666667) ,
  XLF(0.02083333333333333333333333333333333333333333333333333333333333333) ,
} ;
const xlf_type K4[] = {
  XLF(0.5989583333333333333333333333333333333333333333333333333333333333) ,
  XLF(0.4583333333333333333333333333333333333333333333333333333333333333) ,
  XLF(0.1979166666666666666666666666666666666666666666666666666666666667) ,
  XLF(0.04166666666666666666666666666666666666666666666666666666666666667) ,
  XLF(0.002604166666666666666666666666666666666666666666666666666666666667) ,
} ;
const xlf_type K5[] = {
  XLF(0.55) ,
  XLF(0.4380208333333333333333333333333333333333333333333333333333333333) ,
  XLF(0.2166666666666666666666666666666666666666666666666666666666666667) ,
  XLF(0.06171875) ,
  XLF(0.008333333333333333333333333333333333333333333333333333333333333333) ,
  XLF(0.0002604166666666666666666666666666666666666666666666666666666666667) ,
} ;
const xlf_type K6[] = {
  XLF(0.5110243055555555555555555555555555555555555555555555555555555556) ,
  XLF(0.4194444444444444444444444444444444444444444444444444444444444444) ,
  XLF(0.2287977430555555555555555555555555555555555555555555555555555556) ,
  XLF(0.07916666666666666666666666666666666666666666666666666666666666667) ,
  XLF(0.01566840277777777777777777777777777777777777777777777777777777778) ,
  XLF(0.001388888888888888888888888888888888888888888888888888888888888889) ,
  XLF(2.170138888888888888888888888888888888888888888888888888888888889e-05) ,
} ;
const xlf_type K7[] = {
  XLF(0.4793650793650793650793650793650793650793650793650793650793650794) ,
  XLF(0.4025964161706349206349206349206349206349206349206349206349206349) ,
  XLF(0.2363095238095238095238095238095238095238095238095238095238095238) ,
  XLF(0.09402436755952380952380952380952380952380952380952380952380952381) ,
  XLF(0.02380952380952380952380952380952380952380952380952380952380952381) ,
  XLF(0.003377666170634920634920634920634920634920634920634920634920634921) ,
  XLF(0.0001984126984126984126984126984126984126984126984126984126984126984) ,
  XLF(1.550099206349206349206349206349206349206349206349206349206349206e-06) ,
} ;
const xlf_type K8[] = {
  XLF(0.4529209681919642857142857142857142857142857142857142857142857143) ,
  XLF(0.3873759920634920634920634920634920634920634920634920634920634921) ,
  XLF(0.2407776847718253968253968253968253968253968253968253968253968254) ,
  XLF(0.1064732142857142857142857142857142857142857142857142857142857143) ,
  XLF(0.03212696862599206349206349206349206349206349206349206349206349206) ,
  XLF(0.006125992063492063492063492063492063492063492063492063492063492063) ,
  XLF(0.000634765625) ,
  XLF(2.48015873015873015873015873015873015873015873015873015873015873e-05) ,
  XLF(9.68812003968253968253968253968253968253968253968253968253968254e-08) ,
} ;
const xlf_type K9[] = {
  XLF(0.430417768959435626102292768959435626102292768959435626102292769) ,
  XLF(0.3736024025676532186948853615520282186948853615520282186948853616) ,
  XLF(0.2431492504409171075837742504409171075837742504409171075837742504) ,
  XLF(0.1168385769744819223985890652557319223985890652557319223985890653) ,
  XLF(0.04025573192239858906525573192239858906525573192239858906525573192) ,
  XLF(0.009453129305831128747795414462081128747795414462081128747795414462) ,
  XLF(0.001383377425044091710758377425044091710758377425044091710758377425) ,
  XLF(0.0001058857697448192239858906525573192239858906525573192239858906526) ,
  XLF(2.755731922398589065255731922398589065255731922398589065255731922e-06) ,
  XLF(5.382288910934744268077601410934744268077601410934744268077601411e-09) ,
} ;
const xlf_type K10[] = {
  XLF(0.4109626428244185405643738977072310405643738977072310405643738977) ,
  XLF(0.3610984347442680776014109347442680776014109347442680776014109347) ,
  XLF(0.2440661561888571979717813051146384479717813051146384479717813051) ,
  XLF(0.1254387125220458553791887125220458553791887125220458553791887125) ,
  XLF(0.04798334892044201940035273368606701940035273368606701940035273369) ,
  XLF(0.01318342151675485008818342151675485008818342151675485008818342152) ,
  XLF(0.00245328523074087852733686067019400352733686067019400352733686067) ,
  XLF(0.0002791556437389770723104056437389770723104056437389770723104056437) ,
  XLF(1.58879786361882716049382716049382716049382716049382716049382716e-05) ,
  XLF(2.755731922398589065255731922398589065255731922398589065255731922e-07) ,
  XLF(2.691144455467372134038800705467372134038800705467372134038800705e-10) ,
} ;
const xlf_type K11[] = {
  XLF(0.3939255651755651755651755651755651755651755651755651755651755652) ,
  XLF(0.3497022318874430690836940836940836940836940836940836940836940837) ,
  XLF(0.2439602873977873977873977873977873977873977873977873977873977874) ,
  XLF(0.132561165432106594215969215969215969215969215969215969215969216) ,
  XLF(0.0552020202020202020202020202020202020202020202020202020202020202) ,
  XLF(0.01716314960753132139850889850889850889850889850889850889850889851) ,
  XLF(0.003823878667628667628667628667628667628667628667628667628667628668) ,
  XLF(0.0005712862612632713544171877505210838544171877505210838544171877505) ,
  XLF(5.100609267275933942600609267275933942600609267275933942600609267e-05) ,
  XLF(2.16679942326914983164983164983164983164983164983164983164983165e-06) ,
  XLF(2.505210838544171877505210838544171877505210838544171877505210839e-08) ,
  XLF(1.22324747975789642456309122975789642456309122975789642456309123e-11) ,
} ;
const xlf_type K12[] = {
  XLF(0.3788440845447299915073352573352573352573352573352573352573352573) ,
  XLF(0.3392729502364919031585698252364919031585698252364919031585698252) ,
  XLF(0.243130918010144694715007215007215007215007215007215007215007215) ,
  XLF(0.1384514665504248837582170915504248837582170915504248837582170916) ,
  XLF(0.06186766800904132548826559243225909892576559243225909892576559243) ,
  XLF(0.0212685824013949013949013949013949013949013949013949013949013949) ,
  XLF(0.005458186925696725230145369034257923146812035700924589813478702368) ,
  XLF(0.0009984747441344663566885789108011330233552455774677996900219122441) ,
  XLF(0.0001209139205918753716062743840521618299396077173854951632729410507) ,
  XLF(8.523979878146544813211479878146544813211479878146544813211479878e-06) ,
  XLF(2.708616506969914087969643525199080754636310191865747421302976859e-07) ,
  XLF(2.087675698786809897921009032120143231254342365453476564587675699e-09) ,
  XLF(5.096864498991235102346213457324568435679546790657901769012880124e-13) ,
} ;
const xlf_type K13[] = {
  XLF(0.3653708694854528187861521194854528187861521194854528187861521195) ,
  XLF(0.3296898795859100119354025604025604025604025604025604025604025604) ,
  XLF(0.241788417986334653001319667986334653001319667986334653001319668) ,
  XLF(0.1433150174717420836602151706318372985039651706318372985039651706) ,
  XLF(0.06797496725882142548809215475882142548809215475882142548809215476) ,
  XLF(0.02540440629498498879873662859773970885081996193107304218415329526) ,
  XLF(0.007312236695917251472807028362583918139473695029250584806140361696) ,
  XLF(0.001567173108165633054413436357880802325246769691214135658580103025) ,
  XLF(0.000237629847004847004847004847004847004847004847004847004847004847) ,
  XLF(2.349228542020798693975777309110642443975777309110642443975777309e-05) ,
  XLF(1.313308604975271641938308604975271641938308604975271641938308605e-06) ,
  XLF(3.125375747123929632610188165743721299276854832410387965943521499e-08) ,
  XLF(1.605904383682161459939237717015494793272571050348828126605904384e-10) ,
  XLF(1.960332499612013500902389791278680167569056457945346834235723125e-14) ,
} ;
const xlf_type K14[] = {
  XLF(0.3532391566991892985022170290027432884575741718598861456004313147) ,
  XLF(0.3208502450206319253938301557349176396795444414492033539652587272) ,
  XLF(0.2400829904155873420494246852133756895661657566419471181375943281) ,
  XLF(0.1473218009462429105286248143391000533857676714819571962429105286) ,
  XLF(0.07354103256406706097994152929668802684675700548716421732294748168) ,
  XLF(0.0294998002323771172977522183871390220596569802919009268215617422) ,
  XLF(0.009341081854512256904689707889112651017412922174826936731698636461) ,
  XLF(0.00227591965005159449603894048338492782937227381671826116270560715) ,
  XLF(0.0004110905114937219671610172602236094299586363078426570490062553555) ,
  XLF(5.204637459101744816030530316244601958887673173387459101744816031e-05) ,
  XLF(4.22295610849360418239076473203457330441457425584409711393838378e-06) ,
  XLF(1.877646346892378638410384442130473876505622537368569114600860633e-07) ,
  XLF(3.348635775124742293641103164912688722212531736341260150783960308e-09) ,
  XLF(1.147074559772972471385169797868210566623265035963448661861360274e-11) ,
  XLF(7.001187498614333931794249254566714884175201635519095836556154016e-16) ,
} ;
const xlf_type K15[] = {
  XLF(0.3422402613553407204200854994505788156581807375458169108962759756) ,
  XLF(0.3126666062517608097457825027889512016496143480270464397448524433) ,
  XLF(0.2381231949107073115009622946130882638819146755654692162628670565) ,
  XLF(0.1506119498039969868420469988136985491482845980200477554974909472) ,
  XLF(0.07859525386674857574328473799373270272741172212071682971153870625) ,
  XLF(0.03350380257164983552592802293116446555599994753433906873060312214) ,
  XLF(0.01150227448749687506301262915019528776142532756289370045983802598) ,
  XLF(0.003117493948498863912897025599571895868192164488460784757081053377) ,
  XLF(0.0006485490063532391574719617047659375701703744031786359828687871016) ,
  XLF(9.944024943894646248956299799553767807736061704315672569640823609e-05) ,
  XLF(1.057200426826749578072329395080717832040583363334686086008837332e-05) ,
  XLF(7.068397902798796317960711148541836372524203212033899864587695276e-07) ,
  XLF(2.504599065445626292187138747985308831869678430524991371552218113e-08) ,
  XLF(3.348642542939324287497237232686968136703586439036174485909935645e-10) ,
  XLF(7.647163731819816475901131985788070444155100239756324412409068494e-13) ,
  XLF(2.333729166204777977264749751522238294725067211839698612185384672e-17) ,
} ;
const xlf_type K16[] = {
  XLF(0.332208269142495860354893909213260651752715244778736842228905721) ,
  XLF(0.3050644278149432229293340404451515562626673737784848895960007071) ,
  XLF(0.2359883168766360905058009537279543893300507057120813734570348327) ,
  XLF(0.1533009314401523086212239651393090546529699968853408006847160286) ,
  XLF(0.08317297504551898046847185831664171611261558351505441452531399621) ,
  XLF(0.0373810339101848175095529592884090238587593084947582302079656577) ,
  XLF(0.01375763090948818939943249503228008519013810019101024392029683035) ,
  XLF(0.004080872532107702825295417888010480603073195665788258380850973444) ,
  XLF(0.0009544828678894823993222595420636095239269842444445619048793651968) ,
  XLF(0.0001707270050562771296898281025265152249279233406217533201660185787) ,
  XLF(2.234895063781818710891984081468208452335436462420589404716388843e-05) ,
  XLF(2.004166042122804688942255079821217387354953492519630085767651905e-06) ,
  XLF(1.107471879616850687316905488201784498080794377090673386969683266e-07) ,
  XLF(3.131465753406890973028539166105303671441237578803716369853935992e-09) ,
  XLF(3.139354644805746280407325380870354415327960301505275050248595222e-11) ,
  XLF(4.779477332387385297438207491117544027596937649847702757755667809e-14) ,
  XLF(7.2929036443899311789523429735069946710158350369990581630793271e-19) ,
} ;
const xlf_type K17[] = {
  XLF(0.3230093941569987066310595722360428242781183957654545889840007487) ,
  XLF(0.2979799587081916278130416052193107199137388991108256592227958552) ,
  XLF(0.2337367492306511099792301161737825292135905270047815955067744672) ,
  XLF(0.155484036150159998453674215604964467009284936455804803143738718) ,
  XLF(0.08731164077018230311492060636162721441277308068468977276944908442) ,
  XLF(0.04110806430911691477285533952525836084815232029673361761752673673) ,
  XLF(0.01607392199096478464475663355215175943467259993870638128341209574) ,
  XLF(0.005152823873576680687553350997255034564402756123889021492507331287) ,
  XLF(0.001330812572133536283084991457229241231731116573942593239200771102) ,
  XLF(0.0002704049258301891954659168111932029999256890013192534200937562282) ,
  XLF(4.182154969498986967085317536609164371380368890484047658028361421e-05) ,
  XLF(4.695715379871064022225232694663522550232789884206006210363508838e-06) ,
  XLF(3.56439418392324554789540783938543042184498767131820353108868515e-07) ,
  XLF(1.63149746984798566178052651461459429090926134187883331985417264e-08) ,
  XLF(3.684527190109978781002746144358344794074641569194946101264184053e-10) ,
  XLF(2.770019512081012202911548587318895442144741864629819811892641024e-12) ,
  XLF(2.81145725434552076319894558301032001623349273520453103397392224e-15) ,
  XLF(2.14497166011468564086833616867852784441642206970560534208215503e-20) ,
} ;
const xlf_type K18[] = {
  XLF(0.3145344008586467182470994721759390932422799490614270847351734027) ,
  XLF(0.2913584466510833033039909617701228172577206017742726253290973006) ,
  XLF(0.2314117793664616011208018282342371123813484708459982520223900727) ,
  XLF(0.1572401134620674563387759777426755329027348946427689035843408704) ,
  XLF(0.0910485005933913615716885011987613695735215948239429236766443254) ,
  XLF(0.04467047496015852986341799150873397856141570911429472137706048554) ,
  XLF(0.01842292869049824747877647746554818208711247408207673795390363499) ,
  XLF(0.006319116410195815530480260569827260701141117505193999989449033264) ,
  XLF(0.001777277655743294328907405968667273922400664455511085521114223953) ,
  XLF(0.0004021980309109744217120500967373049539249352507989296037848792284) ,
  XLF(7.138389106911010043993841802455747648022088662542374891512077251e-05) ,
  XLF(9.590710558658019278069767401302484471554568729241067312098193587e-06) ,
  XLF(9.271047742986201030757942245048661689392055612617911611583154274e-07) ,
  XLF(5.973408326006386835656815253592235990133420710589849850492379109e-08) ,
  XLF(2.268507892674943235881177967614201058195940099318734723513919143e-09) ,
  XLF(4.094184626640664611408464505258778523640023795641598318224524262e-11) ,
  XLF(2.308349801939754902465172240520651489977491983232551064622164667e-13) ,
  XLF(1.561920696858622646221636435005733342351940408446961685541067911e-16) ,
  XLF(5.95825461142968233574537824632924401226783908251557039467265286e-22) ,
} ;
const xlf_type K19[] = {
  XLF(0.3066931017379824245305168018632871760607585281834448687674708427) ,
  XLF(0.2851526574476310860587296937963637403055693926298178395378548994) ,
  XLF(0.2290456456811837762874976005842387482762931187915216614415592252) ,
  XLF(0.1586346253388907509263807001987774318598400180350699349721960716) ,
  XLF(0.09441929511676010574059072210769536873467724560598357290523435734) ,
  XLF(0.04806054542535070027749314354263487094159567827397532378940146726) ,
  XLF(0.02078114937124501636527206989309959152867076270355710786854004375) ,
  XLF(0.007565383412672267476436688637003756922033686930678865854495560569) ,
  XLF(0.00229186688915413342562369814596241229272609466012363233629907941) ,
  XLF(0.0005689522908994850139668336783697942095877293348094534065998675234) ,
  XLF(0.0001134132006807759156700682628792664229841049886806981067687299498) ,
  XLF(1.766303335855916124091566797843863525155535136915603332189459818e-05) ,
  XLF(2.069399345620689421367587370830773454981032748982033959182713421e-06) ,
  XLF(1.727524784354898381135431169596139951972343408450014835258991401e-07) ,
  XLF(9.468329535090553581770731398613740634369548703313398275203660126e-09) ,
  XLF(2.98700491781092245304329419497705988715647939961447040691994034e-10) ,
  XLF(4.309815999477244092049078371092556883832458389766693205027603114e-12) ,
  XLF(1.822381480598601402708885037931828837924869579896762959721736475e-14) ,
  XLF(8.220635246624329716955981236872280749220738991826114134426673217e-18) ,
  XLF(1.5679617398499164041435205911392747400704839690830448407033297e-23) ,
} ;
const xlf_type K20[] = {
  XLF(0.2994102903200126403616661784861819273208478622613087315147476444) ,
  XLF(0.2793216559936422892233820812529748995823404794270593481765929952) ,
  XLF(0.2266624218574869476605655071675697999645534718693286863894938184) ,
  XLF(0.1597221176265887627737283535275246579651382508197748913918422658) ,
  XLF(0.09745755665987275681714501229862720287892930658613929973155924745) ,
  XLF(0.05127546513801330293366313427359288198750689399970554927664477137) ,
  XLF(0.02312933833806029315065469365846561252547111803094898087281005361) ,
  XLF(0.008877709102343649125781813164758545639943033208331530389398370899) ,
  XLF(0.002871240020020613564967878223844321801612052020207485122495961139) ,
  XLF(0.0007726199672568219644396606409481735050559071395476132809662712852) ,
  XLF(0.0001701507308502417288025889042349833499565809643836956126394318825) ,
  XLF(3.000881964669053045461113561648122451001107336936015385952865818e-05) ,
  XLF(4.116703300385090395649698366640374477637672389257233721335151381e-06) ,
  XLF(4.219279492289648548180225958549763705354206661942231803704657954e-07) ,
  XLF(3.04930466565191773845113337099929418996422015953415257776378443e-08) ,
  XLF(1.424128264663112556948453880326044396350881518047799765165073862e-09) ,
  XLF(3.735441850133206772601623462381411800620679753879492514387668031e-11) ,
  XLF(4.309894095512087023181389452914307170499575986787115553111880168e-13) ,
  XLF(1.366786125736578015568348718848197392051580900609790918350989553e-15) ,
  XLF(4.110317623312164858477990618436140374610369495913057067213336609e-19) ,
  XLF(3.91990434962479101035880147784818685017620992270761210175832425e-25) ,
} ;
const xlf_type K21[] = {
  XLF(0.2926226872314347791864002755983546567053090736854907457088117093) ,
  XLF(0.2738298047486301248520951526919982350695841657257629321135251989) ,
  XLF(0.2242800938788327640723124311838116614954792287670424846889541496) ,
  XLF(0.1605482126616445458566516653249882445547559197659697032068896127) ,
  XLF(0.1001942907349272287191036155859252089297063752082926744344267819) ,
  XLF(0.05431596662727297097521766518702938064594091425714263283338996795) ,
  XLF(0.0254519832636627386300119265950172711837741722340136104602254935) ,
  XLF(0.01024300084884529025342592555639213143163848283452980432486984946) ,
  XLF(0.003511107772631327302241314369882496288354278073882614854299174551) ,
  XLF(0.001014304593252987379463396974791920649276444908537901680577854361) ,
  XLF(0.0002436124246613323939957972388358447915474104767235924832166707734) ,
  XLF(4.779783924441349999725995220676069541153413802538429647232882627e-05) ,
  XLF(7.486517779540240704998193438663843849959881817719169683997200449e-06) ,
  XLF(9.075615794391424945072662376098362682196792514083823070055639757e-07) ,
  XLF(8.158790979427597358557916920313210863219040629055967225023021481e-08) ,
  XLF(5.115081906671036385653379444072183706279168158960145063661627315e-09) ,
  XLF(2.038368377509909826756860133780319225084092261199188007164825789e-10) ,
  XLF(4.448223742037239646968769969502515945936285698788057322980930717e-12) ,
  XLF(4.104700189226971566504737364590944316098401990935309222554811715e-14) ,
  XLF(9.762758079241290190873069759561138655553116160837698662271853038e-17) ,
  XLF(1.957294106339126123084757437350543035528747379006217651053969814e-20) ,
  XLF(9.333105594344740500854289232971873452800499815970505004186486309e-27) ,
} ;
const xlf_type K22[] = {
  XLF(0.2862766140553860396180994778143617912091107187132976108459581625) ,
  XLF(0.2686459402768988972690069184448927800774613888974958190483808453) ,
  XLF(0.2219120730968715060979611327194697511438673971452332762399152158) ,
  XLF(0.1611512144701082551850668779297792877745897802444649824789586209) ,
  XLF(0.1026578895342640133547104227551038619086537991021880783507370858) ,
  XLF(0.05718529010480106360418634148198039440637253582503748256149990664) ,
  XLF(0.02773678312000349827109209430261071096587526237665429735429216112) ,
  XLF(0.01164920375903508266244159674128980062706852495365218665892234392) ,
  XLF(0.004206555798261862785562595217373443445503341697332318567521846069) ,
  XLF(0.001294343327409118609982816200479590485601750642943444714161861556) ,
  XLF(0.0003355292819853291234757667976163198042994184174052364461613399537) ,
  XLF(7.222478864637174799817057824874336794244458414739912654218077492e-05) ,
  XLF(1.267138379474814743812758619524345270260373391087546913501845064e-05) ,
  XLF(1.768235057909007774978699647226163509326310745537541932749552096e-06) ,
  XLF(1.899389146704343362916947994981635729797600564672037756910353634e-07) ,
  XLF(1.501020632247148740968885050303232004801915456629623391884245583e-08) ,
  XLF(8.177057743781069783137599576662700924055839148857670382097192615e-10) ,
  XLF(2.783324787685537919728904489122790710856578903445197563608903178e-11) ,
  XLF(5.055709418408792537232297228662272681029592107174143198035496601e-13) ,
  XLF(3.731564309831898297571845231403669542545704584347989807127407075e-15) ,
  XLF(6.656425972240051051886725780081104470099637347626192089883802417e-18) ,
  XLF(8.896791392450573286748897442502468343312488086391898413881680971e-22) ,
  XLF(2.121160362351077386557793007493607602909204503629660228224201434e-28) ,
} ;
const xlf_type K23[] = {
  XLF(0.2803261985498075449763550453338011618199597101539086807461365343) ,
  XLF(0.2637426945803405774709851371242840212262224230225364555533677855) ,
  XLF(0.219568310050317182059345457216563100953119235697847773119941364) ,
  XLF(0.1615634033143354345355296782934058488164101720828038460110461054) ,
  XLF(0.1048741828768824974917077200019790608450049262606597643761123871) ,
  XLF(0.05988840460067647181741671232875399987553923460575104591170602834) ,
  XLF(0.02997415944907547010323047932422480691579916072739652795379714429) ,
  XLF(0.01308540310404733080350013162789040941805397941348749849165530364) ,
  XLF(0.004952309709166372133011210136434428382007661300709243856433414621) ,
  XLF(0.001612408766944430496916070273520439244009368113790382200785911076) ,
  XLF(0.0004473141173413978254716788362428595936623396709178216587108740898) ,
  XLF(0.0001044647630135970383808572583879495282713819589814938091265883733) ,
  XLF(2.022508534437359251907130709315352829663222166539437017576419945e-05) ,
  XLF(3.182890469239906353526003385415435476715047719759058699690679684e-06) ,
  XLF(3.967986612900868319859820172038448533714746810629277438350333539e-07) ,
  XLF(3.785523385292728692998890943114036283525344088881258468933689831e-08) ,
  XLF(2.634673489018393792291355778693645144967301480255390225569200172e-09) ,
  XLF(1.248841049839614108173689591804653470615501898226039837952169645e-10) ,
  XLF(3.63383071656837423722244101841492166896047073476640150844420743e-12) ,
  XLF(5.495958555480875197416042920738434408710892039115259837560028922e-14) ,
  XLF(3.244847040262982602784748395818135908922506285291204207144054213e-16) ,
  XLF(4.341147375275081475269570778207256944352205707819564464344538756e-19) ,
  XLF(3.868170170630684037716911931522812323179342646257347136470296074e-23) ,
  XLF(4.611218179024081275125636972812190441106966312238391800487394422e-30) ,
} ;
const xlf_type K24[] = {
  XLF(0.2747319735211881015322761845044625221106483573151421412014247766) ,
  XLF(0.2590959338854922461036563119925389272595861077466218841463698393) ,
  XLF(0.2172561221840602086310824889120935501301289661615425483338688315) ,
  XLF(0.1618120821179101653140295045587458734297658566802317585660386843) ,
  XLF(0.1068665667295971209989834979519872671153177378605346062014883966) ,
  XLF(0.06243142585437320943606393291179844634112652806323106512783670979) ,
  XLF(0.03215681632579833790498795030647768103322747540265379844924435315) ,
  XLF(0.01454184959951421604405223650420725484809645947324652722029620552) ,
  XLF(0.005742944626624392292869428264589821982483891620507710145501567236) ,
  XLF(0.001967617402838947504046175887816835006180044367136538293731340687) ,
  XLF(0.0005800499627008823707723137113830171281174768806501670069004012335) ,
  XLF(0.0001456354315661878934852098075573658610406565702667437614224947591) ,
  XLF(3.074601805288829237565846842299406208680046317480218105279591668e-05) ,
  XLF(5.3704036096147168717567292035747592497438062055234103411437846e-06) ,
  XLF(7.601697767063152933136584314038388415508418565002778437392003242e-07) ,
  XLF(8.486194900961675149065571673304571540652997645498943848693961555e-08) ,
  XLF(7.204528187097666671112756504305465515558427620459167198742416257e-09) ,
  XLF(4.422918500467296261727955990067205806215573252688189722480887101e-10) ,
  XLF(1.826149993888722191784315859831614492138743775698015517284728258e-11) ,
  XLF(4.545262838830708863913937292381485377450434049219518822745141337e-13) ,
  XLF(5.725363811192343702863351231362992772745712226696837081626895761e-15) ,
  XLF(2.704042907215565690059493141889047633463898284612706169244384294e-17) ,
  XLF(2.713217109998441035406256361235587543807988881412284908288679253e-20) ,
  XLF(1.611737571096118349048713304801171801324726102607227973529290031e-24) ,
  XLF(9.606704539633502656511743693358730085639513150496649584348738378e-32) ,
} ;
const xlf_type K25[] = {
  XLF(0.2694597712409119359478025644722404843499695520564867595122246328) ,
  XLF(0.2546842927399865634269226362647617781255938203848040467010015573) ,
  XLF(0.214980814251065970705611552309316374205270211459308289187239186) ,
  XLF(0.1619204249078340261497083737117156479867436123934196438304429325) ,
  XLF(0.1086561716445043983998113417529272521137728937772406128457590469) ,
  XLF(0.06482118414842576032066559832985081442903448465106988964315662552) ,
  XLF(0.03427935408543838204261900452741202163923234528817020347212425545) ,
  XLF(0.01600993365737546731718924377109080130106580912254353210674388857) ,
  XLF(0.006573045689755602078610204745230059549517155580021595839043945655) ,
  XLF(0.002358636146942911039116225207928651463726756967427738354300813724) ,
  XLF(0.0007344950796361184846041273455426906219268869280757477622182252465) ,
  XLF(0.0001967670421694020475896813800479207443794754069444037160491893484) ,
  XLF(4.485942758183979499839386031077925812118913239088604505756780885e-05) ,
  XLF(8.586897119581881946455855366473450438977777073518783791843514591e-06) ,
  XLF(1.356786425515225450414139582244378792263737470489610032664059996e-06) ,
  XLF(1.731450639888094511993173152423750268639431190788376427255928527e-07) ,
  XLF(1.734391495596260318411629164977478836902810344422369563407631763e-08) ,
  XLF(1.312519963625023011649641287169535677432910168153653029222283589e-09) ,
  XLF(7.116667913729384256767172232280486361266481023934235321609576606e-11) ,
  XLF(2.561762818874284177830619219872486982489712426981248877572083767e-12) ,
  XLF(5.456803126071488957131579484548320376763307645485266276969874438e-14) ,
  XLF(5.725618853600683556320679419460948917974830177651689836408274897e-16) ,
  XLF(2.163235873040520804321209600276010715896047899427223438334362024e-18) ,
  XLF(1.627930266093210325732162142775167614479708883686638173848074718e-21) ,
  XLF(6.446950284384473396194853219204687205298904410428911894117160124e-26) ,
  XLF(1.921340907926700531302348738671746017127902630099329916869747676e-33) ,
} ;
const xlf_type K26[] = {
  XLF(0.2644798424607552774048811991980218465150397365534503561895016172) ,
  XLF(0.2504887856017991829692305796334444436701302745063324585492411089) ,
  XLF(0.2127461469389779008464347835434384277837578897890244392039977688) ,
  XLF(0.1619081671415522163255580290002194335449936901603964870358944591) ,
  XLF(0.1102620477207957917573831950691314452114456174852272766738085161) ,
  XLF(0.06706490590217548057999341660464585074920074446770653834066683091) ,
  XLF(0.03633793611173714208079274248793968357586338228300777388516995113) ,
  XLF(0.01748212759923957906779090484396273956669673452854112169173037041) ,
  XLF(0.007437327012175687796070452190927462037513208984390345239012737754) ,
  XLF(0.002783781640048098131937159035647652582320706343290685471973983282) ,
  XLF(0.0009110999049672569889575776845492335081806818942925863368641122921) ,
  XLF(0.0002587803754286116916077885427017819569122188053858775751476765515) ,
  XLF(6.319989654239854057224998192329376409254354634679400593499494798e-05) ,
  XLF(1.312122006089165669219383899309009164206148754407671215600984084e-05) ,
  XLF(2.283242503425108707751887109405619842464091727653394858571442243e-06) ,
  XLF(3.271131064294833603596646006965978195896928844333214540115370249e-07) ,
  XLF(3.771219350601499909025817390412337172371133552111194519688614565e-08) ,
  XLF(3.395586220032056940656547543845217263254582424860923459274449807e-09) ,
  XLF(2.293838269129721133240221047965400579059274725499429512017556937e-10) ,
  XLF(1.099699158723737741272412125279482954374287698907042400415715905e-11) ,
  XLF(3.453701950140752395678661871611322180258167055904558670237635628e-13) ,
  XLF(6.298308132426832354355811291032932136926182096697855448908317608e-15) ,
  XLF(5.50555614496803912589673189552975085038562422030989531437433847e-17) ,
  XLF(1.664028214545620271061833942025125474216883509069442109191037529e-19) ,
  XLF(9.391905381495421852886054145042158902137280188333533760921711497e-23) ,
  XLF(2.479596263224797460074943545847956617422655542472658420814292355e-27) ,
  XLF(3.694886361397501021735286035907203879092120442498711378595668607e-35) ,
} ;
const xlf_type K27[] = {
  XLF(0.2597661480314954490051280085087572008430980624510114384955092981) ,
  XLF(0.2464924816235324373051555759464130418821305868671271917413422029) ,
  XLF(0.2105546933313582453141988137125210841069110711477136765436500482) ,
  XLF(0.1617921694966503160303657154764636202990511367919936399442223744) ,
  XLF(0.111701351856793862414318185691739511649153192348446757699311215) ,
  XLF(0.06916998128269683146455507386466727403456927439813859077237415293) ,
  XLF(0.03833000496707419001971751722216559003240935837518411075950931248) ,
  XLF(0.01895190932912276030665024849868878752603876559994120990655582954) ,
  XLF(0.008330716500491168779732515225232783042545928128320131681956866784) ,
  XLF(0.003241109439169010679940166036221174416952522236893375074170632817) ,
  XLF(0.001110031921984315012147496801191212237859944977479549747243396445) ,
  XLF(0.0003324721175851279980480846283767671562718664909351654436695317025) ,
  XLF(8.639508906098987654467352301763175881996222940624277201117071219e-05) ,
  XLF(1.928909982919273269632550185805623842461446214880924589308061762e-05) ,
  XLF(3.656219098565212867144808502454044285400517162282842060233746793e-06) ,
  XLF(5.796995715793529289195487759287104247435699979055916074980528412e-07) ,
  XLF(7.545857538583723758712709815348821101221326370173589037678740909e-08) ,
  XLF(7.873266681060161760970757549281507251233174574018126480922714969e-09) ,
  XLF(6.381800706172868290287182410385321257695925294464161869802912479e-10) ,
  XLF(3.853123780337797326251645655491543954445482663170811086233181724e-11) ,
  XLF(1.634782434967694578792630536376448464676212456540831279533423549e-12) ,
  XLF(4.482016828906594474345838349549509452032233138891005403463682158e-14) ,
  XLF(6.999660914006244856942625354797725638690403147728237019258492748e-16) ,
  XLF(5.097825872595305906666325308594472481857801870896126218906203584e-18) ,
  XLF(1.232613731031951474285558334642804766260180573344194541212320174e-20) ,
  XLF(5.217725211978165654780042590607119790058318957517126604855104851e-24) ,
  XLF(9.183689863795546148425716836473913397861687194343179336349230946e-29) ,
  XLF(6.842382150736113003213492659087414590911334152775391441843830754e-37) ,
} ;
const xlf_type K28[] = {
  XLF(0.2552957845386585957803397036587849362350638221123817343035329959) ,
  XLF(0.2426802311575467844923133687432291811931085336260666173961386035) ,
  XLF(0.2084081117541345643603667451843486492872876979636470895586988051) ,
  XLF(0.1615868801077271067040598410475216528491533929462652113663010738) ,
  XLF(0.1129895289240223810551689343651001884038761513381859731088197573) ,
  XLF(0.07114379666863527068953628646706032751217133573426753483226403184) ,
  XLF(0.04025404278609506668602723922038312261082946619919874883025106342) ,
  XLF(0.02041367684452346886614564155350041375436891601559955673677950141) ,
  XLF(0.009248413306443417171382881460940142704893882149353933674678318474) ,
  XLF(0.003728491840093345465290271124105745105171365566261170072042614296) ,
  XLF(0.001331205213036271336477472414194575589583941350847242477006317142) ,
  XLF(0.0004185067528242368800078907739669123327549553351943143987647850803) ,
  XLF(0.0001150514123561442501315497189033097755719807731071248504285418619) ,
  XLF(2.742647548420673152026532723902103573403959627349578069122408501e-05) ,
  XLF(5.611849625282199399793270093496037939878400752537127277118313152e-06) ,
  XLF(9.733436553016039034618019884483946657168682834792172860687696625e-07) ,
  XLF(1.408998469853445599142467531569804169280696933679197605199323055e-07) ,
  XLF(1.66939140692578936639436824451719823202135789424172469486168269e-08) ,
  XLF(1.578873244078938287750296544646028668252755306533744151797847501e-09) ,
  XLF(1.153619689830592433940933692166319782857211337924664602003663716e-10) ,
  XLF(6.231738008510106975983528031884253848778379177204099828726492132e-12) ,
  XLF(2.341653175769926402611742396033138633143419940725944896672085867e-13) ,
  XLF(5.607163698981499532240154768914356781050554778889037154993249419e-15) ,
  XLF(7.50078126347122058737893518430830293159410211166978784413665367e-17) ,
  XLF(4.551679625430850781448451765219582961899621865247389196828763436e-19) ,
  XLF(8.804384678655461682324509559928443276128653174835511891751722965e-22) ,
  XLF(2.79520993499502322610163864166768456782141743800024859864385343e-25) ,
  XLF(3.279889237069837910152041727312111927807745426551135477267582481e-30) ,
  XLF(1.221853955488591607716695117694181176948452527281319900329255492e-38) ,
} ;
const xlf_type K29[] = {
  XLF(0.2510485149905656391299793469757543253721812416821378800649709691) ,
  XLF(0.2390384347586081167724348363244753668745305082436574150193468938) ,
  XLF(0.2063073557906651272468049868677777580099990951001784836347157803) ,
  XLF(0.1613047150319719278025944302418227542773220834341139401840675815) ,
  XLF(0.1141404822333535168577549932261312777498588557166895117210483621) ,
  XLF(0.07299361587265322173086511606622265519786095000735526787550484744) ,
  XLF(0.04210937045603605820224334432992177164567919024455180824997239336) ,
  XLF(0.02186265993825160291585850200213098448500260969151149601825641797) ,
  XLF(0.01018592380177695590924542098091426580332117317001921915425462714) ,
  XLF(0.004243684185236661081407964393678128358561124789990949375370535456) ,
  XLF(0.001574312188186834215622776783461920610579750426430965127438684295) ,
  XLF(0.0005174139130050184422149226611160799482852775407498228608739696901) ,
  XLF(0.0001497419572616025269653996150938500946623941220169738270137507417) ,
  XLF(3.788247489554460045564579026511372096790421336517618845174653651e-05) ,
  XLF(8.304322906561694415113181436483895466140961118086518131838124573e-06) ,
  XLF(1.560659267130577520569657849905253238964468060165746523796062949e-06) ,
  XLF(2.48185021058764099258735124668197181151137596552762678699985333e-07) ,
  XLF(3.28604319531239548001633056661915304736687157446559114119589449e-08) ,
  XLF(3.549385230039337373280011570076933151728923478121471611603784632e-09) ,
  XLF(3.047062215049192483978699114598042032650077405440376993342055281e-10) ,
  XLF(2.009186130533520803368952420942216337970566271772921252563834733e-11) ,
  XLF(9.719242659523968837930344858650502902558446704924179198869497398e-13) ,
  XLF(3.236591287711976132404880970900605558688215320135903805201068541e-14) ,
  XLF(6.771425826704358661176337961509682975353784732432598490982058341e-16) ,
  XLF(7.760248611819620254065873507071065336023055541850541076076115164e-18) ,
  XLF(3.923888252362185971296499037279863195198161983482075172037369412e-20) ,
  XLF(6.071989750234451635932287573251231103327119357186118350972889505e-23) ,
  XLF(1.445798242240005904457103602959834287178467094260233099196045485e-26) ,
  XLF(1.130996288644771693155876457693831699244050147086598440437097407e-31) ,
  XLF(2.106644750842399323649474340852036511980090564278137759188371538e-40) ,
} ;
const xlf_type K30[] = {
  XLF(0.2470063825838950539981826642019578791036815251851126621866584569) ,
  XLF(0.2355548472503042207632856664840253002914234715611641313043339007) ,
  XLF(0.2042528367342451500647704408564989421752168915398637510272548352) ,
  XLF(0.1609563726345440522679034900331040111295862626195406823381281027) ,
  XLF(0.115166731023435080187505477980783454956880491716308512676874906) ,
  XLF(0.07472649790807482555970650366260995001234635162396320669577105959) ,
  XLF(0.04389598024219399751930654126224021343152700514616345082605181085) ,
  XLF(0.02329483325687316202341943768654774366704175243883289543101688787) ,
  XLF(0.01113908103006694415399426929004099415231573149524019056765580827) ,
  XLF(0.004784380186109439977138505548643177868270930446627690441519152815) ,
  XLF(0.001838855638719593980672984689549999556824666662675877902976900031) ,
  XLF(0.0006295900994787331740833719917196686031235926942222033880558536176) ,
  XLF(0.0001909968461267294703612716590434234836506692408158201743338420864) ,
  XLF(5.101242397662600066070288424824321840722160809168892806747318052e-05) ,
  XLF(1.190386233741890326952688396254449420346287816525756328750856589e-05) ,
  XLF(2.404761291228170986795211978641323296520128455513522888826821975e-06) ,
  XLF(4.15905488479258144735875718581496675278824175708083761555050244e-07) ,
  XLF(6.074934643107642639232887167864088878998191131147513565261293873e-08) ,
  XLF(7.36860367073920759289364332203367170196131139945307826038070591e-09) ,
  XLF(7.266202637623134813507435841899050998288727478890686660921222158e-10) ,
  XLF(5.668894290196139955750022974728273001196888706965880010984158742e-11) ,
  XLF(3.376694008716038465429096336651499144792908319062713253918133816e-12) ,
  XLF(1.463867825075517509173590827330242322011996182252275675164125428e-13) ,
  XLF(4.322439274033272501435167247357171370386707843502203893903226559e-15) ,
  XLF(7.903593695386417108512749412545470012508347169656223475053435653e-17) ,
  XLF(7.7608153308629754695518938539112354509260327396572117804555393e-19) ,
  XLF(3.269920612051789589469841540217447447757529848339124782245583539e-21) ,
  XLF(4.047993276152608992949455387235544979288477165048926452353108912e-24) ,
  XLF(7.228991211202101056290513040800760085660839973871279251701767593e-28) ,
  XLF(3.769987628815905643852921525646105664146833823621994801456991357e-33) ,
  XLF(3.511074584737332206082457234753394186633484273796896265313952563e-42) ,
} ;
const xlf_type K31[] = {
  XLF(0.2431533907099914536911335912093164390105016480631371677980220911) ,
  XLF(0.2322184108440457520335659215891154732902626564766773598659003504) ,
  XLF(0.2022445497916713612904401396071983126959457592886985022359092341) ,
  XLF(0.1605510943727957606200166857123530684930985010439813987658227731) ,
  XLF(0.1160795541686762448975924492384011372914981292227389442986345971) ,
  XLF(0.07634924204183750118188693439006518033552052365723189552966999136) ,
  XLF(0.04561439692534073582971464076317214451078373765965634516884982723) ,
  XLF(0.02470683332624936709658774248173555012465304285432989843209470635) ,
  XLF(0.01210405170337634657496139881327214456031998832898543722588986164) ,
  XLF(0.005348257175468436592733234934556336247307631573552183501260265991) ,
  XLF(0.002124179810847007625944334608425419265050828435211963419544632482) ,
  XLF(0.0007553037547832393680591509648293096853273673753431242493366146834) ,
  XLF(0.0002392959458797775404957801087309044140708000748464254934852563273) ,
  XLF(6.717119163857597716117534355063787787933137246672860742388524365e-05) ,
  XLF(1.659423630606070769814889786267546456403088475167778642097663e-05) ,
  XLF(3.579245446687846586847470744138495890273768791413846978721499703e-06) ,
  XLF(6.67615311102296842831535120948966700100664306954309237621847946e-07) ,
  XLF(1.064458694686305377917729726048725813319107563579751170658775524e-07) ,
  XLF(1.430357843908363941226034488242689835566436122185653762001914924e-08) ,
  XLF(1.591661029154995646533062498444341150904959055378106712750824158e-09) ,
  XLF(1.434682460258225125227618712868506315076726485041949206824066056e-10) ,
  XLF(1.018285276442702641824391223556387603932776350858361068204237623e-11) ,
  XLF(5.483927710960996995059429419011657855149994421641330895072921354e-13) ,
  XLF(2.131980675968432722041796381893738868254316378984450049288697217e-14) ,
  XLF(5.584350767438550431308198158780528045422394924412856699547800563e-16) ,
  XLF(8.926418454063486283911165158806053151292040017375660860747233774e-18) ,
  XLF(7.510845132464196988794721581869678922845836089467580493560580742e-20) ,
  XLF(2.637039630855982909636361260502286768269702361395398347060564958e-22) ,
  XLF(2.611608601743498983992283900018947009019394920807376649472923183e-25) ,
  XLF(3.497898973162652407156165950814192025144470885817368416211664337e-29) ,
  XLF(1.216125041553517949629974685692292149724785104394191871437739147e-34) ,
  XLF(5.663023523769890654971705217344184171989490764188542363409600907e-44) ,
} ;
const xlf_type K32[] = {
  XLF(0.2394752361829221817846148566387753318305833644915735273617097364) ,
  XLF(0.2290191124318211375311131197709823231249720086536896632118378261) ,
  XLF(0.2002821724877073427303301182686708917071342169893179236351048113) ,
  XLF(0.1600968819347263383597895681374748488026922723543680790910644143) ,
  XLF(0.1168891201931103844017234446576664165424080610248912609188539078) ,
  XLF(0.07786835312321691904158976449493395836830827577036924307465722115) ,
  XLF(0.04726556304458340388559170602061634881125816890725198778363723928) ,
  XLF(0.02609588106552989054017244706833377405770588612985128849102640583) ,
  XLF(0.01307733402110062621432431553687810217587404634177948413053185257) ,
  XLF(0.00593301238963447872013649414175623560280960178397738995378486318) ,
  XLF(0.002429499632678958350928248359100875783705505052053977550788446081) ,
  XLF(0.0008947027727710059305092138463987346570348973260610299522395811407) ,
  XLF(0.0002950637886591241515858603237686594983819472913001311964084960049) ,
  XLF(8.670709043241161506297580431720561955252222180477637570421955511e-05) ,
  XLF(2.256995589261363055694004144432232791484630015725459920396453305e-05) ,
  XLF(5.167840444406296672228028864589199433709184566625109359116063148e-06) ,
  XLF(1.03223469058837950512818559856239354093621400805587780272646086e-06) ,
  XLF(1.780784984311083039987121746766376893655288589431527294211019469e-07) ,
  XLF(2.621660553931816945077782463271484560429397276865484684903836435e-08) ,
  XLF(3.245475733445526912856694463451450153401563044190775977433644264e-09) ,
  XLF(3.317388214926496219546365598165831311226381122792237286235774999e-10) ,
  XLF(2.736300253045405521947599022351360203921690237386253492222301629e-11) ,
  XLF(1.768499527819998884340719103144440512658519884419705703587108157e-12) ,
  XLF(8.617500117591645121054305201095045019069312239403192019672482871e-14) ,
  XLF(3.006047917016258807842884543507631924570501881087358079766333467e-15) ,
  XLF(6.98724516269948371765634291490922945330182269447215086925703998e-17) ,
  XLF(9.765701205041633482272657912981146179840167400869040449293483398e-19) ,
  XLF(7.041662149991598130024800759618450766450066902149682404273932533e-21) ,
  XLF(2.060190545541195443806482805488083407487478935663968003320338075e-23) ,
  XLF(1.632255387870898205044882574552221648281202025963466104739310744e-26) ,
  XLF(1.639640143670049061242264899305287389509703959539834466648927639e-30) ,
  XLF(3.800390754854743592593670892788412967889953451231849598242934836e-36) ,
  XLF(8.848474255890454148393289402100287768733579319044597442827501418e-46) ,
} ;
const xlf_type K33[] = {
  XLF(0.2359590855358157174562983658246484541287590392189529863394692754) ,
  XLF(0.2259478610773664999159673395224707145478628579219402080328043893) ,
  XLF(0.1983651416283094633628491558427352863166064557829593950008655963) ,
  XLF(0.1596006787009698764773333502827644705301554391604520043437249051) ,
  XLF(0.1176046041927883799147590620444202406375224643925011152964985884) ,
  XLF(0.07929002188392354392709176408903581338437730844097991045770629785) ,
  XLF(0.04885074439501923264805168619078033025183132070855136887471906705) ,
  XLF(0.02745971056164966029845868181164611525931524940164791696227290024) ,
  XLF(0.01405574891012765545894267238985515183058115627732096786099470917) ,
  XLF(0.006536391435344104006788220725201366865093755693678911181018174177) ,
  XLF(0.002753927565956844700389110130965060293590392805487010254384519492) ,
  XLF(0.001047823667292070134738082224680451825036579459177990719795705759) ,
  XLF(0.0003586664721341373449105998730206609235482388390357935186615891611) ,
  XLF(0.000109956477102946179083018920409469679969979836737783655866132563) ,
  XLF(3.003330530272325062918881017946051460406795901959898126245397172e-05) ,
  XLF(7.263767603007417823229422451086083543756881657311748442559435002e-06) ,
  XLF(1.544318983649526595455153459036931428409723714339539468714306576e-06) ,
  XLF(2.86136918522843609532860973209532535177797435578050300540163503e-07) ,
  XLF(4.572758656116515550678701374129815814886883365739945487218455533e-08) ,
  XLF(6.22471576710428745038277435092426651536226381031244788413068292e-09) ,
  XLF(7.108221576497256933612943327331823736021150810130218268162282208e-10) ,
  XLF(6.681624474900825794013658830740189292106839474656048917884543101e-11) ,
  XLF(5.048209551989393695477307478708734086080024899097215251480193726e-12) ,
  XLF(2.97346053595301734542347796266193646347639959633920335606134405e-13) ,
  XLF(1.311822142353833655547986665151745234833100425913807650714268743e-14) ,
  XLF(4.107886195493344683050207511099284255090238472341113644931489811e-16) ,
  XLF(8.475789587044821110610741230883567625911103326846985613406358096e-18) ,
  XLF(1.035946599963927813153270074787598493449497657993800505301130876e-19) ,
  XLF(6.401664378528798293490292785591958632745808448205180571913414177e-22) ,
  XLF(1.56075197839983065762749997783473979708302251355375532084224701e-24) ,
  XLF(9.892456933039535835227104986073303495410077421741767434607697583e-28) ,
  XLF(7.452909743954855604256754524042161704409432621954545709412661676e-32) ,
  XLF(1.151633562077195028058688149329822111481804076130863514619071162e-37) ,
  XLF(1.340677917559159719453528697287922389202057472582514764064772942e-47) ,
} ;
const xlf_type K34[] = {
  XLF(0.232593386403171396972319320096661029681623530213761978857298636) ,
  XLF(0.2229963824534834569790693242408311433496417609063373493466634828) ,
  XLF(0.1964927136395438224542390612456443969072827288288345739088289167) ,
  XLF(0.159068521932821489181941431539051445681724415176025632666097727) ,
  XLF(0.118234292547104182116969303797619048653316671223121009448617907) ,
  XLF(0.08062011619977089269654175454358559454704304058995717961593706373) ,
  XLF(0.0503714524656091006782715751891107252026303995625879705695098757) ,
  XLF(0.0287965043718514712267799919664083179990541991689252790978516421) ,
  XLF(0.0150364267188873132404452632367368218705962239510689255700757323) ,
  XLF(0.007156210067138532481612210704392597066368931568408435523217372001) ,
  XLF(0.003096497807525672885778223797656683878735146588308564145827369317) ,
  XLF(0.001214601754428449862870974077972233081314535498931393352406787783) ,
  XLF(0.0004304102760522953458217320651014039255063291159966120350148450428) ,
  XLF(0.000137239132962966729679915589633362426750831124878356007516952238) ,
  XLF(3.919133281925342991296527983385413783843623195149606539520876673e-05) ,
  XLF(9.968853753484431505243145243250836227968314207484655863011687693e-06) ,
  XLF(2.244184813261852393051984355945908038609535598579467437296729466e-06) ,
  XLF(4.437584736304715706550836320178274990458265618867129719148693418e-07) ,
  XLF(7.638584633154218524895475860530940034318759392313845363296626096e-08) ,
  XLF(1.132390855134893640514856049747591733171199342489844789122101778e-08) ,
  XLF(1.427141617172937734447781288540751497663128218613934487747010919e-09) ,
  XLF(1.505030873824642022647772040745917991066190077725257298457365579e-10) ,
  XLF(1.302293980576528753074543341740290655611330307947676203609618036e-11) ,
  XLF(9.020495803887933509697870883866591363112278975911263449290676557e-13) ,
  XLF(4.845651644267249725876208798887670421954517592588141562018491537e-14) ,
  XLF(1.936628847214794923844636926309228493248010423396935061885753761e-15) ,
  XLF(5.446201250299691432949149569772882618222436118854065387530408424e-17) ,
  XLF(9.97735397286844815210376024446164869864880156281687603335245999e-19) ,
  XLF(1.066560216690277439027528801249886079443918516197079765141183346e-20) ,
  XLF(5.648620498296544748711060477686891248220376019733246309499455876e-23) ,
  XLF(1.147612461233895126293182743538844060708855946161761893333227377e-25) ,
  XLF(5.819092324730288005588719381789210564324789565406785112215850338e-29) ,
  XLF(3.288048416450684799733932358209954191125149434803487303836565891e-33) ,
  XLF(3.387157535521161847231435733323006210240600223914304454761974007e-39) ,
  XLF(1.971585172881117234490483378364591748826555106738992300095254327e-49) ,
} ;
const xlf_type K35[] = {
  XLF(0.2293677076664401271784713049334263188739172397893755593279967251) ,
  XLF(0.2201571275539160046405431638495996960632326360592649784947446026) ,
  XLF(0.1946640119552236303743161631810316258541906806786349131299753149) ,
  XLF(0.1585056708491715606221670267030486856488525747007608615355350375) ,
  XLF(0.1187856764263017623097685141570438577670986129890158204278944259) ,
  XLF(0.08186418028643146819193118572113900345086661699975511551795800029) ,
  XLF(0.05182938099444126532030017569852453131959382261133681586554115543) ,
  XLF(0.03010483529164026270012888570948420530445598881730021351470061263) ,
  XLF(0.01601679093379909462201110065789581678419643665342684253944871924) ,
  XLF(0.007790370324980181962172031118377785214988280616467805385061091304) ,
  XLF(0.003456187749275864831628318370013289220943726481849477397348055858) ,
  XLF(0.001394881830894281477115385742874044020959659045136498132734027856) ,
  XLF(0.0005105417212643599961934189453390712919369820852644932688368087745) ,
  XLF(0.0001688544522478030145662162315598734720104135004792197715672380794) ,
  XLF(5.025290875513556611743286048709307428452429225282949940747905173e-05) ,
  XLF(1.339244878115252143168460112377397435108924552155671732287881863e-05) ,
  XLF(3.177893081406759311127532196142053635853560933825745597140099423e-06) ,
  XLF(6.669708758220990183740329281293378707395731264419459907312153098e-07) ,
  XLF(1.228449083874475833952789663312084931248189394129335668656226159e-07) ,
  XLF(1.967217395115326749466941238800001626008087355310171583695017196e-08) ,
  XLF(2.708724424500013968702921305539883115105179417623380101869964739e-09) ,
  XLF(3.164204546646098200852744147557605448377911650677027536037520867e-10) ,
  XLF(3.084802998595784065804469297386730596283881895536636494065996765e-11) ,
  XLF(2.459387884929520217685108537394021582548944363040355857012017924e-12) ,
  XLF(1.562970385071201129581160468031209049332802199499618891837422442e-13) ,
  XLF(7.662055194744004023149742130217217211601608874531379310120394546e-15) ,
  XLF(2.775449723825676230998487510943992450630246686236973607180617552e-16) ,
  XLF(7.011857792335530053600154348919240912429555990886714930425367196e-18) ,
  XLF(1.140785470773381187188883324896434081331128898700875729188578235e-19) ,
  XLF(1.066666780704534872146398882504643257820984338534937643031278574e-21) ,
  XLF(4.8417295785532429558963902459429350824599287935015421006877345e-24) ,
  XLF(8.197235013088449504892470770612714720305125406655615407146614939e-27) ,
  XLF(3.325195617421974751985539726904372177699371498751888853068667378e-30) ,
  XLF(1.40916360705029542901992713633411150824439414572764697353427631e-34) ,
  XLF(9.677592958631890992089816380922874886401714925469441299319925734e-41) ,
  XLF(2.816550246973024620700690540520845355466507295341417571564649038e-51) ,
} ;
const xlf_type K36[] = {
  XLF(0.2262726033193025603250026961787552431761002092831334501195986193) ,
  XLF(0.2174231934762547573979025163678131841933370346973006504825964454) ,
  XLF(0.1928780642708993198150489552243734458683110093806103867666934382) ,
  XLF(0.157916714771245471182187362700511522079533717536586386993540802) ,
  XLF(0.1192655351356437735611207013301014689985786980849315023181796423) ,
  XLF(0.08302743954733596579673888656060324672180689118506461833396319663) ,
  XLF(0.05322635425583203902854734503976569354261767564078219114151144933) ,
  XLF(0.03138361431833886281913185694309933164017302626515118816252747652) ,
  XLF(0.01699454011224549731390943174863947622921800421448274214241759549) ,
  XLF(0.008436871980737005995433520325579085751679328395292049324757982058) ,
  XLF(0.003831936733701335311153721807101114858191493774528418856043732978) ,
  XLF(0.001588428945859191186661394263841957524409889286399751239378003798) ,
  XLF(0.0005992488046180470089115172071984013979514130229429753072936855875) ,
  XLF(0.0002050784270569641418682458015623939544543581202637404642007967107) ,
  XLF(6.342593457692011011165122198874392427979252823085343513212910521e-05) ,
  XLF(1.765020045619630470502992506826992254616554997390538552799755984e-05) ,
  XLF(4.397095566982974972766671788751505118362104606307141684572355592e-06) ,
  XLF(9.748817594591299406374403903434212686918255750668517299103556906e-07) ,
  XLF(1.910335584435182091849089349416308394216157978863551138184076437e-07) ,
  XLF(3.28180127603619067133664581537775879262865410669404651289711821e-08) ,
  XLF(4.895318376187337599364453503292758714960124261272058696993983225e-09) ,
  XLF(6.267885629331358091307407372378115776045489191859073721733015827e-10) ,
  XLF(6.793625979416651380725746701448132563801743324215169860583319942e-11) ,
  XLF(6.128475584914402444307120339476799135774170535723983646114778913e-12) ,
  XLF(4.50547609318932598268313128556465705875610761956542226098244126e-13) ,
  XLF(2.628850347740500761291087755840699461657319678964183203568648277e-14) ,
  XLF(1.176727141431960870111444071616268076459732772878764381276960252e-15) ,
  XLF(3.864931598386980375828467172532402262153841543317582238676978407e-17) ,
  XLF(8.774451871078550729040403481616106947725606652775722961197963593e-19) ,
  XLF(1.267983237181790921925271974546360414972702047584821784902094414e-20) ,
  XLF(1.037113427733002920991891662926963035643493168492612420479337339e-22) ,
  XLF(4.034806053419644781897427290605422221120507821982106629523391065e-25) ,
  XLF(5.692525665093213357152890845911224209290976933277455910624610011e-28) ,
  XLF(1.847330899508640844303400361400050024644930335484549425014344225e-31) ,
  XLF(5.871515029376233731681189944235854475310369731921032363643067532e-36) ,
  XLF(2.688220266286636386691615661367465246222698590408178138699979371e-42) ,
  XLF(3.911875343018089750973181306278951882592371243529746627173123664e-53) ,
} ;
const xlf_type K37[] = {
  XLF(0.223299496002640021111359341134510837820183981581011478874017971) ,
  XLF(0.2147882544510711919569190407076284917078356366124289559233269325) ,
  XLF(0.1911338318377701366704322310440775069478578641968416067177716772) ,
  XLF(0.1573056647303090663450359755052328491126028641865345288091842659) ,
  XLF(0.1196800103136548186980730211805774069327031868380628530784708314) ,
  XLF(0.0841148093577975973749259321703145040444007714777888350754718989) ,
  XLF(0.05456428507461703877753305507345506063872207667011783872754096296) ,
  XLF(0.03263204441867352147312016385029071527210754235832393678102225306) ,
  XLF(0.01796762892789281297789050869283267424030324181267526910560393879) ,
  XLF(0.009093820131609218671994573589787800121382734184092367680596834041) ,
  XLF(0.004222662227863207362593047213251734624576971362369803251126590269) ,
  XLF(0.001794938962651626443754254485735834725280940135628014853656370057) ,
  XLF(0.0006966631614214483506298451478109539642888517224267642626198369499) ,
  XLF(0.0002461613889037040653848938175862969606358749258560528091091989737) ,
  XLF(7.891476585255874661485750460874987674736761466179528632804046175e-05) ,
  XLF(2.286273730161162440708934743979648688795161747897172320704465225e-05) ,
  XLF(5.958757095231239463582163953790311412500009195669303716636561131e-06) ,
  XLF(1.389808819203188863963109445750963562889095481299993332919349836e-06) ,
  XLF(2.883168095103089944418558035605401391582237540687256843976959461e-07) ,
  XLF(5.281987510634492186050058937268477291572198976666560146322165234e-08) ,
  XLF(8.474026572116705375272692020645806137489480763752851350101794184e-09) ,
  XLF(1.178765563824872479781404055538260761715745148687999298197863157e-09) ,
  XLF(1.404908857030410417939226894077512593218815262042967158586447575e-10) ,
  XLF(1.414226082541827831896255548728600395444153723123816966053366058e-11) ,
  XLF(1.181466829789199252598650799476978669825215212176940062592154956e-12) ,
  XLF(8.015206393319374746358231956413350088107977121792447426077867749e-14) ,
  XLF(4.296426999349023767506226347179792876306173244100341041364744046e-15) ,
  XLF(1.756896282826159479860087487893169396786118506068844647742479667e-16) ,
  XLF(5.23418957804375702150428914535730843363873159031302981413153004e-18) ,
  XLF(1.068102073504933348068514703729203635850250998882033941661537675e-19) ,
  XLF(1.371163454806844964034306222557904730661078152161827312395247462e-21) ,
  XLF(9.811063214056637673871466472915012329695801698454977145693506043e-24) ,
  XLF(3.271481842389301931924819592115923152733077307704881580486499648e-26) ,
  XLF(3.846301688411045627417622124756889385663724613074295840142706519e-29) ,
  XLF(9.985572432392002660729702221646166722654454485430460385339547514e-33) ,
  XLF(2.380343930828203250096293275123389410318035510729455592345974374e-37) ,
  XLF(7.265460179153071315382745030722879043845131325427508482972917218e-44) ,
  XLF(5.286318031105526690504299062539124165665366545310468415098815762e-55) ,
} ;
const xlf_type K38[] = {
  XLF(0.2204405769366256970084169101999345046475155217864402442370986939) ,
  XLF(0.2122465016001463982769597921694014751984382351046329008674151314) ,
  XLF(0.1894302324820040239809536776565368553527188945566814085571006764) ,
  XLF(0.1566760313070161487559819369364610702802160126090123377780994119) ,
  XLF(0.12003467194665939485786939771061412359013596605360037390219396) ,
  XLF(0.08513090649930807292034127820225861189231157534183318300894487683) ,
  XLF(0.05584514089010038394293372545899290553565874822225800937679702699) ,
  XLF(0.03384957964566861365610554160816927573017178390561622655446173683) ,
  XLF(0.01893424898900674267434828997472419446237266210211030685789132954) ,
  XLF(0.009759429668081820299226072723961361699852524733868534595028874955) ,
  XLF(0.004627273591928908384628961893485554013932438920848044612716714737) ,
  XLF(0.002014048690147923995843389202126266680297339266680167160033373356) ,
  XLF(0.0008028629319168792278420323133127458233554605905344091357011991999) ,
  XLF(0.0002923264476485619992151126852992078508207007992830161278575508514) ,
  XLF(9.691789209971075958155369544692944501260932727351080996978080146e-05) ,
  XLF(2.915430610005127433934232847519307708173451642696354306014928135e-05) ,
  XLF(7.924768986753194064224026455679239399760361230103279829902516351e-06) ,
  XLF(1.937347334574533840625678052404961827208693849112806843319412077e-06) ,
  XLF(4.236410274253766669641293678291219325903573489999973588777629044e-07) ,
  XLF(8.233986488669669474477542379484552025182717060094972455605030243e-08) ,
  XLF(1.412006309587132883758702680733905355710456275446366297888007001e-08) ,
  XLF(2.117920150530041569243997601790652447817415069997488516491460273e-09) ,
  XLF(2.750222696759650938650076951167747261893237013000199528770556179e-10) ,
  XLF(3.054085677231035555741946552752095824577983897925944391603986484e-11) ,
  XLF(2.857677531698228695719000242655664077264882684938181405084657439e-12) ,
  XLF(2.212566703290411454934409168275948437057448428697313513544409605e-13) ,
  XLF(1.386048232855116781112711477638211990175193152792717967895327533e-14) ,
  XLF(6.82928690846568068072289463812514432530187305449536216633021178e-16) ,
  XLF(2.552292361843761159576520112101936843190693471537156236744552547e-17) ,
  XLF(6.899359854653215220078582141671976560272667002299645052441865425e-19) ,
  XLF(1.265748459889749896396263637556466326468388038567136764366364157e-20) ,
  XLF(1.443631273124267397845631204497068288354676101501317357403726348e-22) ,
  XLF(9.036864917446741520986280502909692592017688570218437206565926792e-25) ,
  XLF(2.582758282954911159732641421914675627473535126092313760820233202e-27) ,
  XLF(2.530461865750828637062695157115597074788538499675973951738962501e-30) ,
  XLF(5.255564438808480417827905481574723449125361636074636673970335044e-34) ,
  XLF(9.39609446379553966679306862932193423523228450345089368676370266e-39) ,
  XLF(1.911963205040281925100722376506020801011876664586186442887609794e-45) ,
  XLF(6.955681619875693013821446134919900217980745454355879493551073371e-57) ,
} ;
const xlf_type K39[] = {
  XLF(0.2176887195898937418225228637634886925112187026714183598640155194) ,
  XLF(0.2097925901575713739300943625860955993681356413653218881549740614) ,
  XLF(0.1877661586628236074650732494072900129398682751737765182005378242) ,
  XLF(0.1560308909684679854638932073698496277234159697295131819795777603) ,
  XLF(0.1203345770900273918424405893668202237608908431918603104154506847) ,
  XLF(0.08608006228445636022381341750674583429178721130612337831452121556) ,
  XLF(0.05707091647022090650605715862631512164008327719590608517704315008) ,
  XLF(0.03503588912636704584373310985844619259497761295043203502250249204) ,
  XLF(0.01989280990678619245331062592578925878151818297699242192697643047) ,
  XLF(0.0104320272392785324839950784871441304489261799011150474181762851) ,
  XLF(0.0050446836480750103688327133567327459616723423762828768699042938) ,
  XLF(0.002245345431970502356079276190897834658912425882331386662686485725) ,
  XLF(0.0009178761359213730236256507345524907088589992439713015273016294646) ,
  XLF(0.0003437685569620565308917592092068127925162273114092964050298815497) ,
  XLF(0.0001176258995931459076143258404543567985096395470309544955069792735) ,
  XLF(3.66514050867204443983482543748153114397699108909416748703738261e-05) ,
  XLF(1.036147175817185229768274172229886810622199038201848995730089524e-05) ,
  XLF(2.646374682802178634539845928806185337847931758786862635176799508e-06) ,
  XLF(6.076583785136942665251524581155153893999646227607286484266408107e-07) ,
  XLF(1.247377602383245949700557859749303172341881221988884698998326298e-07) ,
  XLF(2.27419552149453383211557478462096096398152348233185102834678675e-08) ,
  XLF(3.654584067587039982563063830020067639519685104027061714413099011e-09) ,
  XLF(5.130268696079998755250251755761287624609207855983189592261635454e-10) ,
  XLF(6.224887524344096158378753631118560489854090424681339265936255029e-11) ,
  XLF(6.446335067410568233721688039965402588841603764254024476196436652e-12) ,
  XLF(5.611037734208879018398495653884589609306040490177947148128439037e-13) ,
  XLF(4.029059843849294268462748641279427903112077131199882447769170313e-14) ,
  XLF(2.332003925046160275917548212411228928304011211397220903910402942e-15) ,
  XLF(1.056674350919289574405642041578659311663089974488470793070879336e-16) ,
  XLF(3.61058366974535455105039274668111557072101884122690397960658425e-18) ,
  XLF(8.858288812262929604953207128337764151655320173089941376850886551e-20) ,
  XLF(1.461301578602453365903368332969905489737926269998600765347940926e-21) ,
  XLF(1.480885868071521374205135530487221137601178174012990426299692481e-23) ,
  XLF(8.110243802472972916898592216388787503844668669824849218839263716e-26) ,
  XLF(1.986742126782860787327043648460079156127530036439742245799075204e-28) ,
  XLF(1.622091029930926662817814204494632381165097453948872758655363976e-31) ,
  XLF(2.695161250857309449633620024278902923416156711931740123200877163e-35) ,
  XLF(3.613882486075207632816242386973610867942077679382692031385706149e-40) ,
  XLF(4.90246975651354339769415993975902769490224785791329857150669178e-47) ,
  XLF(8.917540538302170530540315557589615664077878787635742940450094065e-59) ,
} ;
const xlf_type K40[] = {
  XLF(0.215037404911510658278346721650747989352339032399454935358848413) ,
  XLF(0.2074215930929292648304248878205716030490401958019418519872901174) ,
  XLF(0.1861404915973675869777636007220157795933022088952077557895729011) ,
  XLF(0.1553729427643562790592521176202138792149273944630670168737533431) ,
  XLF(0.1205843221079231459029456557560999846112101923595942469674728979) ,
  XLF(0.08696633666088934757008113142520029563544876382398313866375261941) ,
  XLF(0.05824361211119029703111157220103844065220625255480779858832324588) ,
  XLF(0.03619082544391560073706066797165748196594630259445553935642919706) ,
  XLF(0.02084192094868450755698689338998483434539555053148618099041525259) ,
  XLF(0.01111005124276135846184469621827366973865248717597376681448075581) ,
  XLF(0.005473818268101626589548631485465720144015566211668339872005945125) ,
  XLF(0.002488375856377021353668940486233898696385477899687424818960169327) ,
  XLF(0.001041684388076669555794528095299978538897754976153786533556147602) ,
  XLF(0.0004006541297978540459086476994000625870946564296358498190227813222) ,
  XLF(0.0001412197289718143846998331579899848440890685313381599381709670285) ,
  XLF(4.548144759849271658303381735327516718998824605247315400987889054e-05) ,
  XLF(1.333910605109669115159346971640419187814474853155491302455529031e-05) ,
  XLF(3.54899385187398403253555804882340908918157146610707525729958316e-06) ,
  XLF(8.52826819481390746220621846688289393591494120772903254164456423e-07) ,
  XLF(1.841625205024749270352837368664239393148516973766892610299271236e-07) ,
  XLF(3.553028241409532416634397998880950984910814196902885790594605331e-08) ,
  XLF(6.083084627682534483820831472623902200861022314668324764267243622e-09) ,
  XLF(9.169847053061317531059603444748247282435261743007927038923590235e-10) ,
  XLF(1.20588113715728426580108014936601293624780459771024985406843947e-10) ,
  XLF(1.368378280513567575679972923800506394669605794227480414589997761e-11) ,
  XLF(1.322506757193870324459155284283635797968995389183203925433382986e-12) ,
  XLF(1.071600108036426408260527401382304594019428685803816815344331944e-13) ,
  XLF(7.140672046564404583634289695773184871937497627431314301007072792e-15) ,
  XLF(3.820647662351564131368825482608371630291487006026066015678511571e-16) ,
  XLF(1.59276252908966442501279711860528451112733336688415988831106353e-17) ,
  XLF(4.977521597409959281316682420641641820962825002333230553751516319e-19) ,
  XLF(1.108618898814130569855935513019659017980756081992854363490030542e-20) ,
  XLF(1.644704335674735695419956371130889152814892879764547378507652624e-22) ,
  XLF(1.481069641718248788827963282024703694923119970541361102457428896e-24) ,
  XLF(7.096615398197907326660907359421860437649819313983050773305479692e-27) ,
  XLF(1.490059155490333904939259888284082432053424772678683019502423444e-29) ,
  XLF(1.013806928490448092735007343665478212849190512660543138218013287e-32) ,
  XLF(1.347580625476453804942817060266969521120728875991166978215093243e-36) ,
  XLF(1.355205932278202871112162176688497474386840742888254980046545109e-41) ,
  XLF(1.225617439128385849423539984939756923725561964478324642876672945e-48) ,
  XLF(1.114692567287771316317539444698701958009734848454467867556261758e-60) ,
} ;
const xlf_type K41[] = {
  XLF(0.2124806563390982712897035436210733494648704444800379947186874374) ,
  XLF(0.2051289602446188198714152490088700742189987233033120225446880449) ,
  XLF(0.1845521122603517911198059596111253025294928437616173514870335585) ,
  XLF(0.1547045569165107007044553041119892038005273756474980925528002872) ,
  XLF(0.1207880891639810803960403965259431830414376112865825734114054577) ,
  XLF(0.08779353277096463852281991654176227001546915838269455696887426559) ,
  XLF(0.05936521635487762619392479017008255825904363441703974245126458241) ,
  XLF(0.0373143969558195276387958878432390525091658636031840953551422071) ,
  XLF(0.0217803735028194920506377746579272911435951054513436180346662255) ,
  XLF(0.01179205028024111835911643521497867391149813468700157690434154313) ,
  XLF(0.005913624198780104648412380295962441022547859029450080837186743792) ,
  XLF(0.002742654132673340374794116647565075451537360564608161247690977095) ,
  XLF(0.001174226813419448281574819394568540738959460782816568717024820381) ,
  XLF(0.00046312112619113668629917242016034809820650809185437324478696911) ,
  XLF(0.0001678692280470183587084393823778678268449966074236864824047694499) ,
  XLF(5.577148447745731442117467121420280155437928063937741918311904044e-05) ,
  XLF(1.693121074353050859812123924410868383063689685851844352658813387e-05) ,
  XLF(4.680419922278284546546993071520900867994108127005781922068112032e-06) ,
  XLF(1.173482971647855029304517285167649445112790451575258071182751503e-06) ,
  XLF(2.656385862845341699661132347944882457731702614376528680249415485e-07) ,
  XLF(5.400886217037518942406260685809766587048927714386097412247519296e-08) ,
  XLF(9.803731306466928145596330261791630057446501589397023858171777293e-09) ,
  XLF(1.577796729651918402165165151322201546453736533155025540870522989e-09) ,
  XLF(2.233189668676868565025027920181951096727102570483283371151014633e-10) ,
  XLF(2.753501820558423316897864045879930790136587770200659995611944297e-11) ,
  XLF(2.924439370867689570743182080918872291143453136750698087111056626e-12) ,
  XLF(2.639716318813207890613855639981798699853131227511779534073488868e-13) ,
  XLF(1.992393452347396835240531704760310555217261328106469607956926397e-14) ,
  XLF(1.232735883198346698394987972242052284937976830986064782145326385e-15) ,
  XLF(6.100222395608102538630270210460291190619230689451770435748415619e-17) ,
  XLF(2.340606208511047622095511314658637754002757907379683435841956653e-18) ,
  XLF(6.691804998538269243932824810018367675456897829986383908524114408e-20) ,
  XLF(1.353310841889026919463010007132143047213413647281774354519328097e-21) ,
  XLF(1.805812375929423132838366222036288874471008936873952889591323503e-23) ,
  XLF(1.445084094963922312730665865086917739272356142336356660823689391e-25) ,
  XLF(6.058181514478084016055824081903371400216174277387329759921315331e-28) ,
  XLF(1.090288468789131969021370473619215204229611387531381810745972042e-30) ,
  XLF(6.181749694528669650796418946570979295899089073003822413734402267e-34) ,
  XLF(6.573564026833981237684803791489952784485482997393552280510535295e-38) ,
  XLF(4.958070483944644661421336981824926933332428939907622926851814145e-43) ,
  XLF(2.989310827142404510789121914487212009086736498727621080187007183e-50) ,
  XLF(1.359381179619233312582365176461831656109432742017643740922270437e-62) ,
} ;
const xlf_type K42[] = {
  XLF(0.210012983107585934630258469223366980748022502429581356414799665) ,
  XLF(0.2029104822106857881361787030354118332002648070675328481382660588) ,
  XLF(0.1829999098960994422193870701714665144524128518996810367762883662) ,
  XLF(0.1540278165709190636072630550552190300129267127338987194810237309) ,
  XLF(0.1209496876188006587862653921170286385589002154076079736559083534) ,
  XLF(0.08856521158649294181378482138273575895248480572509577889635842081) ,
  XLF(0.0604376924209624819005397073376656852800176925799490390505989997) ,
  XLF(0.03840674361805439458944263308356353160125388228072156702355757431) ,
  XLF(0.02270712449697595837705707464853903818222869160891099713751661843) ,
  XLF(0.01247668044562414489808485849713779561969907446282246686562336124) ,
  XLF(0.006363075336662427925177744576371967119177601840420054571082056064) ,
  XLF(0.00300766931292397090421710019973213467411121870493449163701566788) ,
  XLF(0.00131540404825459606386990594265876767141685860184337532311551894) ,
  XLF(0.000531279537776386625415680800597819291334567065478851244350139293) ,
  XLF(0.0001977319913661670336271780529030911254355756388121587737173796369) ,
  XLF(6.76470072433486136015158878897452715932957741149677527604244805e-05) ,
  XLF(2.121398638459245568592843707146677134737967805662537442078848454e-05) ,
  XLF(6.078814971793625301302657826867675551230080302571369237645911558e-06) ,
  XLF(1.585886331194210690852473025124836960619239156224302059538383753e-06) ,
  XLF(3.751445330251402672333368627288766900794917383419711202854702148e-07) ,
  XLF(8.008717329585359979039491912816788507621482872854795794807711412e-08) ,
  XLF(1.534730902911877315557080857812820890623575280103408411083532996e-08) ,
  XLF(2.623739170026251627740304964033415634941698810541414796834391884e-09) ,
  XLF(3.973005451738919932435225392467236339352961737075350763632659882e-10) ,
  XLF(5.284521200397839850937316961627704199152507433429208544075438806e-11) ,
  XLF(6.114052365100547746402258888217425873614132035514272619545829735e-12) ,
  XLF(6.082169141294098395113102649036362289587930195375634703616457364e-13) ,
  XLF(5.130759073720258206987878788128359023465081359747466415794946471e-14) ,
  XLF(3.609406854202277120101276756708016547951533964350966645272861127e-15) ,
  XLF(2.074622144260672524561989098338589750883318357516219542549427413e-16) ,
  XLF(9.498975346166503173452453446085134085518851488626431153577578976e-18) ,
  XLF(3.355645179099090411569809347194218051610343749816077682421179776e-19) ,
  XLF(8.779201299061389097407613140101757868525143071442353507200548542e-21) ,
  XLF(1.612391792620475595738911087130487029399119807083273792358992977e-22) ,
  XLF(1.935354307515590038123530831294007793597291295959424315504410827e-24) ,
  XLF(1.376371807704408812473948046388662488337874789949181876714511546e-26) ,
  XLF(5.048542733282435223487283510764330822485428091636675669216653849e-29) ,
  XLF(7.787781037602396763468110701855148115880828468541296165854353164e-32) ,
  XLF(3.67961296121988779208313241761704666753744368630307811918922243e-35) ,
  XLF(3.130268584235839099829153849505299897015462306466585894635453108e-39) ,
  XLF(1.770739458551658808993675563846954439813204498137627562650444547e-44) ,
  XLF(7.117406731291439311402671224969552402587467854113383524254779007e-52) ,
  XLF(1.618310928118134895931387114835513876320753264306718739193179091e-64) ,
} ;
const xlf_type K43[] = {
  XLF(0.2076293306341901087905084403153051316467825932784057050717141067) ,
  XLF(0.2007622583598915126624829341200089921223668132553214532647600241) ,
  XLF(0.1814827885478032561355070472096206090116947753790200083480728615) ,
  XLF(0.1533445537653919545808063175177961988932236866516802008099478561) ,
  XLF(0.1210725909184700436203743445183729957004225486530296065844901562) ,
  XLF(0.08928470634605101518361390641988064514555785275946760332627647403) ,
  XLF(0.06146296768824943626041807984560622581694321780918276685162207798) ,
  XLF(0.03946811591769053148313813928568745700760834764549506161494779685) ,
  XLF(0.02362128085374899697581799340534247104499408876945051964954055185) ,
  XLF(0.0131627017469449677119932334330997743452839198122575324660515188) ,
  XLF(0.006821177651733899480960565112653724691532259788400307230581864508) ,
  XLF(0.003282891962370498895857097649610548103032419579479621437407690603) ,
  XLF(0.001465082234058659534397968967731467324259241328194411651954757821) ,
  XLF(0.0006052121977181860336827468981150929298110514891795079664938572844) ,
  XLF(0.0002309524715512304459460272734365093545633507774771356626814976588) ,
  XLF(8.123084821290463791462722466751374589682668059178490149570383787e-05) ,
  XLF(2.626564071071370580140237593633637369402426155085324687849581567e-05) ,
  XLF(7.785077890544687856535682040258347453221219721611250406896866182e-06) ,
  XLF(2.108234305978987841887627778927324640894309019116951274842940161e-06) ,
  XLF(5.196819790406284192803105294557616369218896873896054981300982793e-07) ,
  XLF(1.161127508193833476227513541220144875803126732496917240691249566e-07) ,
  XLF(2.34017213006551064067698018908143415467924591097883138906977802e-08) ,
  XLF(4.230960867698719546239886933826794834615303594040083319515730399e-09) ,
  XLF(6.818506020327655451473768210347940714791611046717125634657299252e-10) ,
  XLF(9.722984260819391990262563243410764695478702851820548913249344402e-11) ,
  XLF(1.216309296686649826284151896497226392601317593836792445150333638e-11) ,
  XLF(1.321447371202488839303862501623296709784650390593955042359086021e-12) ,
  XLF(1.232087840331201040374356413149777333395818632977616003251436123e-13) ,
  XLF(9.719287572487420410405894507076456860008987322485016363180782707e-15) ,
  XLF(6.376107908523757155021067494440053838083242514410956878291172113e-16) ,
  XLF(3.406166018942180003991164333538677749789025865431798063278040823e-17) ,
  XLF(1.443547902297606342292877509123422964556795989331416064064518231e-18) ,
  XLF(4.696544642491655940013240628947969027157960244550146627589934974e-20) ,
  XLF(1.1246539136203951221444099583154531994637623862285437703662776e-21) ,
  XLF(1.876122514686610505363535775124035100162817848162132737468407321e-23) ,
  XLF(2.025834548418558542667247957630183019037835482770606024804462627e-25) ,
  XLF(1.280418312106776592038681606972726548668868463473364920628995432e-27) ,
  XLF(4.109313625770570392479318989427458457927682229580851189981887357e-30) ,
  XLF(5.433338592304275734266195675603773795202805388144033723387262208e-33) ,
  XLF(2.139309878264048141884109678437244783370283364652484179228976727e-36) ,
  XLF(1.455938876395714257658077149794997910040882961168381198664693588e-40) ,
  XLF(6.176998111226716777158826721443346489513130862817522654446481297e-46) ,
  XLF(1.655210867742195188698295633713849395950573919561251982384832327e-53) ,
  XLF(1.881756893160621972013240831204085902698550307333393882782766385e-66) ,
} ;
const xlf_type K44[] = {
  XLF(0.2053250369589799561320848189863728328524206044656696681116863883) ,
  XLF(0.1986806684216285746479056311990451568976862928600542205359860491) ,
  XLF(0.1799996720050995103384620805738273637414009800296339991869571622) ,
  XLF(0.15265638048970794149396891499643147013763026024163933484857486) ,
  XLF(0.1211599694924723850205697776454976791676648402037818272294932878) ,
  XLF(0.0899551366039917449754077019660821718052734413247763932495535244) ,
  XLF(0.06244292567384325792955667561301597122800888622203959854644454689) ,
  XLF(0.04049885655168666387089116694557778494754016923591332002429258704) ,
  XLF(0.02452208501752991992740627869914459033615405241969280114152044295) ,
  XLF(0.01384897390827948253524225316676852376093783209402085565721455171) ,
  XLF(0.007286972944016492149907994782807639178996821253413554720104842512) ,
  XLF(0.003567780059843608139533471318445327382620617581642017026696020809) ,
  XLF(0.001623096932138216675945463586701231640901133875068857517325041535) ,
  XLF(0.000684975850452823533700052145758232919783750768135170923841808119) ,
  XLF(0.000267661342884376598649183637465354452636992844539546092608592071) ,
  XLF(9.664218851340608780073819044631208021070430701551437151312134892e-05) ,
  XLF(3.216573124428932339514620590697338438243713795659145929768318336e-05) ,
  XLF(9.842596214439556916321584642169461263274164575103885132251649189e-06) ,
  XLF(2.760648496866660937967306573212367776852863440784646674641040665e-06) ,
  XLF(7.073330455442524788288001013172655439590005180116023123447777432e-07) ,
  XLF(1.649222881881624136187250040829201491316903639417770210925887806e-07) ,
  XLF(3.484033451060586173861210268819222910241038858180962359939814961e-08) ,
  XLF(6.635495230127985669097496482932512026878754017478817741343782382e-09) ,
  XLF(1.132872368031011551939273358519418254027979511455906889663132079e-09) ,
  XLF(1.7225131883411210032080656875356547399951468565742443886288703e-10) ,
  XLF(2.314884281986423428186117072495208759978029656483734813198869207e-11) ,
  XLF(2.725529432236534027052804441679959473424552899126495888655097588e-12) ,
  XLF(2.782481975779987883752130617469187007959585695217058175069609628e-13) ,
  XLF(2.433062404880983177122795108027311204307840166383121807657917007e-14) ,
  XLF(1.795785954702453861016451974091385582897587230683861277019525835e-15) ,
  XLF(1.099139540392949569114041710777765622666661758047305854800181482e-16) ,
  XLF(5.459461552047714307649693972425174334444359897117157276089705294e-18) ,
  XLF(2.142354668320187826615059907431225958590574723300909852631470497e-19) ,
  XLF(6.421008325686980329497414015663102620875934400768000668704744939e-21) ,
  XLF(1.407636038949642381372225090947189049994521039639530597911932824e-22) ,
  XLF(2.133121419700336280269507636829106001592901071514672078869216497e-24) ,
  XLF(2.072254486341352391593538621256032205987663723584251671898187384e-26) ,
  XLF(1.16406727620667973714996166861672862486388117475326672452241832e-28) ,
  XLF(3.268792379899305529926251057624490851910317652339230055234100853e-31) ,
  XLF(3.704550429967297287454197543348943301045372349122849563037004777e-34) ,
  XLF(1.215516982252809619869050113413210346490893407286821957965207522e-37) ,
  XLF(6.617903983633058913744194861139542025697126078953067875417046725e-42) ,
  XLF(2.105794810645471628762910075156793248203512739954559397578088938e-47) ,
  XLF(3.761842881232261792496126440258748627160395271730118141783709834e-55) ,
  XLF(2.138360105864343150015046399095552162157443531060674866798598165e-68) ,
} ;
const xlf_type K45[] = {
  XLF(0.2030957943865536540845257563368017159398570993680554254367857391) ,
  XLF(0.1966623471932641667983503849039629286022752584705325891868097122) ,
  XLF(0.1785495074895293164024261743620744163344938819715674864034997716) ,
  XLF(0.1519647155705602867840198729922662546677313256806619055476036564) ,
  XLF(0.1212147201196368976835231058683803370671571444043074636235313371) ,
  XLF(0.0905794217617485993361194593303246597611412405075452715808655188) ,
  XLF(0.06337940005385973689225165288681479654981140503620609301383728336) ,
  XLF(0.04149938452454414239339159022180317961898345934336958670902134228) ,
  XLF(0.02540890155567983648885495572152729012330188182268702629347449) ,
  XLF(0.01453445175077237895065524472136411108091694564962643502046691919) ,
  XLF(0.007759541600547815856473283420406724320227961555074486190607789186) ,
  XLF(0.003861784201916173064174124909335973053291704163871103263235520616) ,
  XLF(0.001789256904010516018874900547567984892850650470307427694338773093) ,
  XLF(0.0007706024221193263291833570330177395206146666189456168615070366984) ,
  XLF(0.0003079750947254968705160662232338019805080253666806259094858348971) ,
  XLF(0.0001139956803924147031692067924639040497675049288453111987558336487) ,
  XLF(3.89945180077492795870453773467096556071569712680208002621471413e-05) ,
  XLF(1.229696734874433352834644761571785907813785997233983400150219866e-05) ,
  XLF(3.565133432434886136689417516278332335389473791729459241034577254e-06) ,
  XLF(9.473050905292266912259378082458193296509687395266774065147374299e-07) ,
  XLF(2.298902362427839036033022379074399029299233234966788364280499911e-07) ,
  XLF(5.075150427914039311441841518254980037881261797461673684427503261e-08) ,
  XLF(1.014670388089610519176178947662148821924168814158258483202942858e-08) ,
  XLF(1.827797014361082281237534137238264381379753952359723270360513332e-09) ,
  XLF(2.949290121563638949154921759797094191177796667952062882425346021e-10) ,
  XLF(4.234211428005720025175207843537946751781057394048771582461266176e-11) ,
  XLF(5.366785851365562204447097277164665538365721647803442016669722563e-12) ,
  XLF(5.951243863117473803700324980086574130467914267207011200448570399e-13) ,
  XLF(5.712617463391066418298947286141443497179639673845900499583046935e-14) ,
  XLF(4.687379504252380687547005346616401632248142127449838902364969469e-15) ,
  XLF(3.238610483688320895960512036374053619531018560346964931699470108e-16) ,
  XLF(1.85022826837276644437333169717096541542348895826708420896170505e-17) ,
  XLF(8.548144486452398308088501545241573631849929049211122879712760469e-19) ,
  XLF(3.106868215026607054902687175421196155180782062782594555716017684e-20) ,
  XLF(8.580305513535532317376725422100395547848738321659686865640719621e-22) ,
  XLF(1.722309076642825683274042629250139267867581279132874335063737376e-23) ,
  XLF(2.371195505407584175171078450441524132294766060086633063981472089e-25) ,
  XLF(2.072555941638609771992409597742457575699562119545522956447725639e-27) ,
  XLF(1.034761043543283905574648833503052256683148578811495568393626852e-29) ,
  XLF(2.542405553137625577589049585847897342916852810257136640722792557e-32) ,
  XLF(2.469700919022356627683983453344582265142041799362836141984489058e-35) ,
  XLF(6.75287214398273661328983748589263265146507588422741180825633819e-39) ,
  XLF(2.941290659396148874703513705370237112829197922636809220006601327e-43) ,
  XLF(7.019316035484905429421160305435895872179863721536869261295865877e-49) ,
  XLF(8.359650847182803983324725422797219171467545048289151426186021854e-57) ,
  XLF(2.375955673182603500016718221217280180174937256734083185331775739e-70) ,
} ;
const xlf_type Poles_2[] = {
  XLF(-0.171572875253809902396622551580603842860656249246103853646640524) ,
} ;
const xlf_type Poles_3[] = {
  XLF(-0.2679491924311227064725536584941276330571947461896193719441930205) ,
} ;
const xlf_type Poles_4[] = {
  XLF(-0.3613412259002201770922128413256752554300293066372075233002101795) ,
  XLF(-0.01372542929733912136033122693912820409948234934137640212780201923) ,
} ;
const xlf_type Poles_5[] = {
  XLF(-0.4305753470999737918514347834935201100399844094093466703345548961) ,
  XLF(-0.04309628820326465382271237682255018245930966932524242612443881562) ,
} ;
const xlf_type Poles_6[] = {
  XLF(-0.4882945893030447551301180388837890621122791612393776083939970451) ,
  XLF(-0.08167927107623751259793776573705908065337961039814817852536794476) ,
  XLF(-0.001414151808325817751087243976558592527864169055346698516527091711) ,
} ;
const xlf_type Poles_7[] = {
  XLF(-0.5352804307964381655424037816816460718339231523426924148811983969) ,
  XLF(-0.122554615192326690515272264359357343605486549427295558490763347) ,
  XLF(-0.009148694809608276928593021651647853415692563954599448264800304012) ,
} ;
const xlf_type Poles_8[] = {
  XLF(-0.5746869092487654305301393041287454242906615780412521120018827786) ,
  XLF(-0.1630352692972809352405518968607370522347681455082985484897311685) ,
  XLF(-0.02363229469484485002340391929636132061266592085462943706514568421) ,
  XLF(-0.0001538213106416909117393525301840216076296405407004300196299400714) ,
} ;
const xlf_type Poles_9[] = {
  XLF(-0.6079973891686257790077208239542897694396347185399082955021963142) ,
  XLF(-0.2017505201931532387960646850559704346808988657574706493188670608) ,
  XLF(-0.04322260854048175213332114297942968826585238023149706938143476332) ,
  XLF(-0.002121306903180818420304896557848623422054856098862394663415165309) ,
} ;
const xlf_type Poles_10[] = {
  XLF(-0.6365506639694238587579920549134977331378795901012886043233946858) ,
  XLF(-0.2381827983775732848874561622001619786665434940597287872519238159) ,
  XLF(-0.06572703322830855153820180394968425220512169439225586310303406759) ,
  XLF(-0.007528194675548690643769834031814883165056756744131408609363634153) ,
  XLF(-1.698276282327466423072746793996887861144001323413620950069296025e-05) ,
} ;
const xlf_type Poles_11[] = {
  XLF(-0.6612660689007347069101312629224816696181628671680088080242089952) ,
  XLF(-0.2721803492947858856862952802582877681512352595653351762441917173) ,
  XLF(-0.08975959979371330994414267655614154254756196601701854440621445969) ,
  XLF(-0.01666962736623465609658583608981508371547272055193351560536097831) ,
  XLF(-0.0005105575344465020571359195284074939241798925253401410628961016927) ,
} ;
const xlf_type Poles_12[] = {
  XLF(-0.6828648841977233230185698902607230174299001013684859476841835817) ,
  XLF(-0.3037807932882541599892583738286603768367541154892904874450630646) ,
  XLF(-0.1143505200271358850105800857719841802152524363943309096223167966) ,
  XLF(-0.02883619019866380371021569628068064589471421148796348420674179784) ,
  XLF(-0.002516166217261335592723139872534647136514907382952945705857963505) ,
  XLF(-1.883305645063902643439348865811401007447714099792616803257191584e-06) ,
} ;
const xlf_type Poles_13[] = {
  XLF(-0.7018942518168078624531414012287862764084947122814778574940273876) ,
  XLF(-0.3331072329306235924822781708234713524225941280939732447311497427) ,
  XLF(-0.1389011131943194302117942814286554135390464688327444862031836343) ,
  XLF(-0.04321386674036366964776175493557673701243101733056994259431228149) ,
  XLF(-0.006738031415244913998475856269761420409518027617620529866093864119) ,
  XLF(-0.0001251001132144187159609064792562718948551304769602578038425767285) ,
} ;
const xlf_type Poles_14[] = {
  XLF(-0.7187837872399444505020456522129282595964750174549937757131229127) ,
  XLF(-0.3603190719169610581080084136755666203625002934242563261809150911) ,
  XLF(-0.1630335147992986909768156526491246798608320766128086498946800539) ,
  XLF(-0.05908948219483101879561625313720803556201246447376003919538272219) ,
  XLF(-0.01324675673484791465503043810667202647754110845913635657121884712) ,
  XLF(-0.0008640240409533379359827592769202302559672760211495479400001004579) ,
  XLF(-2.091309677527532850132717832806851684714105981958539758162531259e-07) ,
} ;
const xlf_type Poles_15[] = {
  XLF(-0.7338725716848373495693848190690037113259546621452360440203438709) ,
  XLF(-0.3855857342784352080140541786889509307194303153524667924992133235) ,
  XLF(-0.1865201084509643457413145787278752891979317998887406861617926908) ,
  XLF(-0.07590759204766819930382221724156900984700061847619931049627371902) ,
  XLF(-0.02175206579654047146207140882909654431610437531706224317179848268) ,
  XLF(-0.002801151482076454872535129646713807156117939572215494561939408094) ,
  XLF(-3.09356804514744232400754981719555295636107606371219648458073457e-05) ,
} ;
const xlf_type Poles_16[] = {
  XLF(-0.7474323877664685085574427283736589844157583318318004439330736865) ,
  XLF(-0.4090736047572509102690812680965741156739840772723126299135932808) ,
  XLF(-0.2092287193395396948898543031587591726843472742877748490017783853) ,
  XLF(-0.09325471898024062584258002426076364675726611549020054157303989561) ,
  XLF(-0.03186770612045390040617688956292344467322144721011000028899027708) ,
  XLF(-0.006258406785125984912475394080646091554877289512409984227223665675) ,
  XLF(-0.0003015653633069595807102978730177732059471914699696332281898335403) ,
  XLF(-2.323248636421231704917740718046885239052877562647066064799116983e-08) ,
} ;
const xlf_type Poles_17[] = {
  XLF(-0.7596832240719327792045776593358745445069777141400539152276006722) ,
  XLF(-0.4309396531803965785693965637300692642342545429704656904137548696) ,
  XLF(-0.2310898435992718341019926513706630467054625898374489974155007084) ,
  XLF(-0.1108289933162472532114988831064802678108300620508942380785689952) ,
  XLF(-0.04321391145668415769660776100790405761679462981400693616624507606) ,
  XLF(-0.0112581836894716022181896293491206662733281119456807152725245783) ,
  XLF(-0.001185933125152176738908831907655560107159111098519587692500186341) ,
  XLF(-7.687562581254682728955673310667501934979112121779492801030611799e-06) ,
} ;
const xlf_type Poles_18[] = {
  XLF(-0.7708050514072084875653763894746947171002896449375376904119489378) ,
  XLF(-0.4513287333778174878316722609027777273866881227917307297897186109) ,
  XLF(-0.252074574682638763584472896601959892330780571622277331198978582) ,
  XLF(-0.1284128367967145725789350332258624076307639057064238630350586344) ,
  XLF(-0.05546296713852201088203462904233467368282674158008500152656817855) ,
  XLF(-0.01766237768478522019536068887024688998012080586932492506861471418) ,
  XLF(-0.003011930728994830197965802954651925043296476246417439420870245334) ,
  XLF(-0.0001063373558871366640189637199846302574071259717460536890338053772) ,
  XLF(-2.581240396257155753009061408465308128783279309333058163208326594e-09) ,
} ;
const xlf_type Poles_19[] = {
  XLF(-0.7809464448517323032921331480485990736133673529544175911917193925) ,
  XLF(-0.4703728194676430247813348528935411602052750451796672897113815149) ,
  XLF(-0.2721803762830344869914311059697431889279401637302705823402891798) ,
  XLF(-0.1458508937575645126564144435615800230623243319927531764237235694) ,
  XLF(-0.06834590612488046381791503567280010798024149861224965610323129877) ,
  XLF(-0.02526507334485559573805086677780212256467747396675298316895380627) ,
  XLF(-0.005936659591082969962210417817284936502087769359983070616251584598) ,
  XLF(-0.0005084101946808165632250947968083231925294339604915334694819054562) ,
  XLF(-1.915478656212247986385955601447744124330070942089945639492293327e-06) ,
} ;
const xlf_type Poles_20[] = {
  XLF(-0.7902311174807248549332936375457893542860119537154342460770408151) ,
  XLF(-0.488191260639127932238629607901884051326457227480964476948377784) ,
  XLF(-0.2914216014747825537725469531269362807164011247774040738675675161) ,
  XLF(-0.1630335348502630805820942942606579464754200839866649328459354433) ,
  XLF(-0.08164811561958893557999175584335413995558571097225276455879102418) ,
  XLF(-0.03384947955391195844315351136577024243837182069205517957617886762) ,
  XLF(-0.009973029020058727414581506443127589654186608606974154953533863969) ,
  XLF(-0.001468321757104340435450206390769993910611464623851653381355200669) ,
  XLF(-3.774657319751902549594020029881206053580025869672360111786256136e-05) ,
  XLF(-2.867994488150105473662242044890074684469234296770066865401193409e-10) ,
} ;
const xlf_type Poles_21[] = {
  XLF(-0.7987628856657315757203801383612220144086561675303372460663002769) ,
  XLF(-0.5048915374483703361482062601022197153890772443749564975705740451) ,
  XLF(-0.3098231964157903198162558904555100588543835587935743155364140584) ,
  XLF(-0.1798846667985358388109211672203553781384692593141807071537453683) ,
  XLF(-0.09520081246073120058708495270851148142240478223201471931334499159) ,
  XLF(-0.04321391844071573523188109232276280721250761578908178194021699031) ,
  XLF(-0.01504549998729455474001276889463311669647492626006472344511038256) ,
  XLF(-0.00317200396388417259157655496824079224784953069498238455598383488) ,
  XLF(-0.0002199029576316323303215070397334940614688205773541255587462877769) ,
  XLF(-4.779764689425363162589596178024910556810143850138812709242039102e-07) ,
} ;
const xlf_type Poles_22[] = {
  XLF(-0.8066294994690704381784577435835683355058440289815194105900519751) ,
  XLF(-0.5205702359699174392369736031713993379968280185938642109865886427) ,
  XLF(-0.3274164741105744655612120945481846353114397459808033338881750675) ,
  XLF(-0.1963528261275153599272899298935673605843126849556521483006046854) ,
  XLF(-0.1088724519840775961967031236282505637337642412423072789898409219) ,
  XLF(-0.05318160458588924115751824694892656984029762515468591039270267612) ,
  XLF(-0.02103566093067810131733673269896154114203893543044501644112959873) ,
  XLF(-0.005706613645975125961234826367939161186678825260186810986162279501) ,
  XLF(-0.0007225479650742529170027747868538226466859987928309007310242102331) ,
  XLF(-1.345815498330488082616558794512746781444517394742113720906213664e-05) ,
  XLF(-3.186643260385932993403597953003403529026022997206807142846413846e-11) ,
} ;
const xlf_type Poles_23[] = {
  XLF(-0.8139056235618003511160832839269542710216365904829182798424493152) ,
  XLF(-0.5353140836684856561212427588911268204417052058992365104857921705) ,
  XLF(-0.3442362769193780815082593918406731698782468720833165382304015715) ,
  XLF(-0.2124046605401930352247334461921198933385766551087644460716790274) ,
  XLF(-0.1225611609919387285989759258209244090173671464974510953899350814) ,
  XLF(-0.06360248015411367696237081559934765507739077553650250339847155377) ,
  XLF(-0.02781166203794071880492535613955159051479715041944045901584377782) ,
  XLF(-0.009079595335295149302724717051209901706254818465340811597141835918) ,
  XLF(-0.001711271446781703660793369164854098249027950784315701693774921441) ,
  XLF(-9.57339435007398175334370890753995602326869462772976757522717198e-05) ,
  XLF(-1.193691881606649475531914423897375002871006593953056103637512138e-07) ,
} ;
const xlf_type Poles_24[] = {
  XLF(-0.8206551810729532156848190473086285634380511372977395325255480247) ,
  XLF(-0.5492009639573297506029461059136924670588956118496358242197130164) ,
  XLF(-0.3603190713767427976422377445566057753967064903947126273791578148) ,
  XLF(-0.2280201472279068744951961560602261536599135945349852345747884729) ,
  XLF(-0.1361885000680127106266852561836016842025719709716392129723196932) ,
  XLF(-0.07435149727736939900226484262517080976959261781922548817880852866) ,
  XLF(-0.03524412666583196627738452246600390373343076990931685283556371198) ,
  XLF(-0.01324637532530081046738011746362276892101808265885329670404363107) ,
  XLF(-0.003297682623486454663018723727298904531875022240846464545776918981) ,
  XLF(-0.0003580715441086799517180636275229816571893919159569076806601582924) ,
  XLF(-4.812675563310482672752116792028140409899187591565777090081223269e-06) ,
  XLF(-3.540708807231918732476161328832045170720301334926365154550694474e-12) ,
} ;
const xlf_type Poles_25[] = {
  XLF(-0.8269332126349767039536046203302337317844718793583579210028891483) ,
  XLF(-0.562300867680010075615339941705359654958070477846209224585448454) ,
  XLF(-0.3757016770096310556705494107763768501576330898118634499426110874) ,
  XLF(-0.2431890684152279829614631308632634458341487203556387167850475983) ,
  XLF(-0.149694464390990002697418020451765628593153014908409895366739785) ,
  XLF(-0.08532560756216579704117185688473574146181888947184193945606167234) ,
  XLF(-0.0432139182679600323290365545980944075204028971941545119308763697) ,
  XLF(-0.01813436080558409621046323364959820919088282979160784076616637111) ,
  XLF(-0.005534084441487329840365165647931103570866714934686263509692316399) ,
  XLF(-0.0009299873306092404593718224645733852844290861912017824499403961315) ,
  XLF(-4.187875052377163435464804985213326332107893175912333792888710719e-05) ,
  XLF(-2.982478285796372981678270626577953206842887466839166029601480287e-08) ,
} ;
const xlf_type Poles_26[] = {
  XLF(-0.8327873634529700580425956057636158354428269078244656773169746416) ,
  XLF(-0.5746767631908141696308575162797660334002247637147295013301378323) ,
  XLF(-0.3904204283620341355956031741141048059108648997510733921739160726) ,
  XLF(-0.2579084013447033805967347937013898442368925540218292437269621744) ,
  XLF(-0.1630335348216207883671433265453213548141587465724585702337761475) ,
  XLF(-0.09644049675858785528252082983409831400920777167932652742564346496) ,
  XLF(-0.05161516247228332846551007467288419671247505561613920738425427913) ,
  XLF(-0.02365917403431692583041444393488441803059901802519424862659885329) ,
  XLF(-0.008424934725813554238984216777947408817561185396254943927068045688) ,
  XLF(-0.001920011836343276892788589768295800806467265571069546675768503616) ,
  XLF(-0.0001784141661193633141006905730926794065324105614061195213247225048) ,
  XLF(-1.72453039221046446691317823363963175837149616659442073510620035e-06) ,
  XLF(-3.934118864515371781248316568521890530388409112936934377274810852e-13) ,
} ;
const xlf_type Poles_27[] = {
  XLF(-0.8382590821014597701203030396072828509655577771509714233091817271) ,
  XLF(-0.5863853797684231139786093581652022714013749014396205771002823535) ,
  XLF(-0.4045106347246844323157228276106909233093664941429080876157738678) ,
  XLF(-0.272180376229601012226312104374117418470314796210649818535048906) ,
  XLF(-0.176171577593237769260878301446307942921591427559888820575898633) ,
  XLF(-0.1076275243906049218438449209355682072315081677169028900962096741) ,
  XLF(-0.06035576309668649760051296356543150268415454102548685960527181155) ,
  XLF(-0.02973460374678660315296461597390414378869303198628675252413705228) ,
  XLF(-0.01194239633191651155429095081546769400433414825855333988737411513) ,
  XLF(-0.003398965898172871421994175583610690257655645339749817682769112609) ,
  XLF(-0.0005082738135987045967033201076290063904583491673040769265276190783) ,
  XLF(-1.83863717083283226335666550100440075767671437179135903626413165e-05) ,
  XLF(-7.453737087428511227281352008850694658452951597264931151361779818e-09) ,
} ;
const xlf_type Poles_28[] = {
  XLF(-0.8433845940335697327620979814421339557729383849789504154891521657) ,
  XLF(-0.5974779061705426952715719545925882255856081976422230541781315342) ,
  XLF(-0.4180062448978233890294706967407767049597049821465408946420339303) ,
  XLF(-0.2860110251917676037727830882739365545362954330924689512958222167) ,
  XLF(-0.1890834149624572449013119889044253081757892779575779157818425185) ,
  XLF(-0.1188310346590514029276147642028441774360383590515184634504142881) ,
  XLF(-0.06935667734278621035033686384669203272534727947051358856607541783) ,
  XLF(-0.03627817198644646968927939247374571890234386194695592481681508351) ,
  XLF(-0.01603960247911933790027956028994526888519143240744611981000577388) ,
  XLF(-0.005399939151044706317754683424710989393682567643539954358821435295) ,
  XLF(-0.001124473968853076633238431246782069906092193168667140497266094101) ,
  XLF(-8.92772318754021921924487165064105370760868416010510825241999918e-05) ,
  XLF(-6.188192321062351389282815986762183605802895405571164085405977482e-07) ,
  XLF(-4.371242485810543843473735883734189077880275959202918768787132822e-14) ,
} ;
const xlf_type Poles_29[] = {
  XLF(-0.8481956975631381369279712504506719747780655943730513988835529113) ,
  XLF(-0.6080006097327870264529380052457492228776122471012456793760593247) ,
  XLF(-0.4309396531921757538439504082304478699802287742339303845026397798) ,
  XLF(-0.299409094278988118709694298044146784157754689464631775211890117) ,
  XLF(-0.2017509205638139981207049644225579330514883488697396176604902006) ,
  XLF(-0.1300060717335343394702785641868817666903514694464950426411133639) ,
  XLF(-0.07855065547112690760891665743680039037354134185450452866157544345) ,
  XLF(-0.04321391826348333423277774691288523219177010041751862435576842583) ,
  XLF(-0.02066024751435230681809652075146407276551297215303958926398979035) ,
  XLF(-0.00792582821369559765627187731651237889770310786172227205028825455) ,
  XLF(-0.002100199856436503618329184465359189752696089460776825340850275862) ,
  XLF(-0.0002790426068554639276105549448921194704648004130002938449591201897) ,
  XLF(-8.094582441984409990310718354798851107543606137925062869408782119e-06) ,
  XLF(-1.863088870511326052369351682118102095185217108464409117916298728e-09) ,
} ;
const xlf_type Poles_30[] = {
  XLF(-0.8527204188873781146594681300124141875559652045399938128715825351) ,
  XLF(-0.6179953830117889231405877079394802978883853957072802322233160192) ,
  XLF(-0.4433416024105959222699191024048622397707713367332537381488443229) ,
  XLF(-0.312385225626562158841058646135479643676536592509814188657404174) ,
  XLF(-0.2141615233206583148231085471027137107279207520033109007938207413) ,
  XLF(-0.1411164738701920750459492747398355136725916523234446788894915012) ,
  XLF(-0.08788081801854613176014562004502497336032216679963396567234973739) ,
  XLF(-0.05047351681146979841967981172475549486973658960072458590729594154) ,
  XLF(-0.02574493385099440911538076059724497153820987481170045943673148698) ,
  XLF(-0.01095839629287172842698413310327426362928130402503158742791216183) ,
  XLF(-0.003482288669069040241834549094690772217582796376804567159145023786) ,
  XLF(-0.0006616482677279396550449026566993868687720784853285375490860328414) ,
  XLF(-4.482596596998705327314716498792059075387831881897735585689104483e-05) ,
  XLF(-2.22269051335366943425960529138102939881992253252418536823699629e-07) ,
  XLF(-4.85693585632446413022984934041620945754435531034158717983414012e-15) ,
} ;
const xlf_type Poles_31[] = {
  XLF(-0.8569835543745790006692660176613903978235236113685321985797685177) ,
  XLF(-0.6275002254374090295625305245199459527611076546929184941211088856) ,
  XLF(-0.4552411529474754870125291859636648372320011901586959086311909406) ,
  XLF(-0.3249513417693065471598202874462884925973577281876529518062349127) ,
  XLF(-0.2263070292125006212432694033802036554671095695492104461360681813) ,
  XLF(-0.1521332995713027103088557046255432838293121337329509714701607771) ,
  XLF(-0.09729925878028641707609812795297238388135853571553997308037946095) ,
  XLF(-0.05799644242639943959494223261278172620094142936286773950122429818) ,
  XLF(-0.03123508345332988342320715140725308412210179149409314511539688198) ,
  XLF(-0.01446609949032905075952120070199286833657619175444924733096165779) ,
  XLF(-0.005292679268065068164838563470991247284500582212313629199311876432) ,
  XLF(-0.001303973183898104304083796237137082634737593376712824510513537086) ,
  XLF(-0.0001537511493585799622361256146865309659915332469334283743248429426) ,
  XLF(-3.5711513081322671069286563588876077644083947779678621581936046e-06) ,
  XLF(-4.657236728778484597058209578183411059230343300563713079312932431e-10) ,
} ;
const xlf_type Poles_32[] = {
  XLF(-0.8610071220775073984545025884174647651873342531478458248801207525) ,
  XLF(-0.6365496672844416072579682630981124647076099496259688957518003072) ,
  XLF(-0.4666656963925545842292615356620524236211323834923337799059180971) ,
  XLF(-0.3371201820284790043011923387339551881746442456200868620563921785) ,
  XLF(-0.2381826906635858373340295751961314788024671940456938940165961336) ,
  XLF(-0.1630335348215802295105412986273882397126963886458718842033727226) ,
  XLF(-0.1067657601912722166551080865931935785675861772344540943455026145) ,
  XLF(-0.06572962575614472922119768199720706395464914974888004512385744442) ,
  XLF(-0.03707517947165689966943488764775622423088672123093473823346986939) ,
  XLF(-0.01841002303519408058508537427084853973752496338830604506274024818) ,
  XLF(-0.007533364010056747507177074686484737587952268224142799783978323192) ,
  XLF(-0.002256727280342519009243907948704147629674856107478718283783368275) ,
  XLF(-0.0003907975105276824894206940606710036419613843135574836014341309415) ,
  XLF(-2.25689522301406632236836294375985204961356448661504774692761827e-05) ,
  XLF(-7.988927839078504108724438276285537087646234418866499870958201165e-08) ,
  XLF(-5.396595313947431360582308041198731074375556607940834970394538893e-16) ,
} ;
const xlf_type Poles_33[] = {
  XLF(-0.8648107396863688016557219577627117613740501334459417369612973419) ,
  XLF(-0.6451751428015154579730662039203667631320950694797143070152669499) ,
  XLF(-0.4776409984675163764629779248436891401227532476094464300149541618) ,
  XLF(-0.3489049538475770937469889124693515126882768453750585940131312858) ,
  XLF(-0.2497864693298141795853141466596027633821329905647261273822420384) ,
  XLF(-0.1737990337412069900543274825289003508247496408633109766001367353) ,
  XLF(-0.1162466535813227634735432582332078210348909125051792336777833672) ,
  XLF(-0.07362686063059345521691441309339694564214463979560969267774880714) ,
  XLF(-0.04321391826377283621523198782502830748757882697443220936696418131) ,
  XLF(-0.02274804078697487978955811906660017817649880159892115509883731691) ,
  XLF(-0.01019192053500615854898240748305761043473774820921200129062596151) ,
  XLF(-0.003552018463170819032019002147056237136233449836891093400369306572) ,
  XLF(-0.0008127973866455559381973382214548059560731448019530608139739299597) ,
  XLF(-8.49681467828939693602893765540687251950218465964645214688094231e-05) ,
  XLF(-1.578071702213531896356749679837082743345320035321838037161612955e-06) ,
  XLF(-1.164240937862773374270370353664588526443504516547867172644971029e-10) ,
} ;
const xlf_type Poles_34[] = {
  XLF(-0.868411942510091296418192858342830867706147953079911531537804392) ,
  XLF(-0.6534053187195726880325209718789722564773514329072110885304557506) ,
  XLF(-0.4881912606388469230758719263538286201213595461658537171317618765) ,
  XLF(-0.3603190713769044598971420551578760292660424066730323100982338825) ,
  XLF(-0.2611184504267236207918374987104792017107442476619846966422450009) ,
  XLF(-0.1844156510074104335265741903507421423784375784641641435814659504) ,
  XLF(-0.125713828924540961029769529606132339915978500886858165075893666) ,
  XLF(-0.08164811550866748008243902441268118591133748932536970143016310113) ,
  XLF(-0.0496046783509799933361487402166251061242552782544855823840231184) ,
  XLF(-0.02743756763176555494239788327004066705311239344350355316655813824) ,
  XLF(-0.01324637525145965067990480147017405091904582300386826800688010616) ,
  XLF(-0.005204974773903469112731368122962788298472575120242691293903018908) ,
  XLF(-0.00146839633965869992758814019288581419517739426256657781805763164) ,
  XLF(-0.0002315423225987300653263686287845986665627360964885761526307610157) ,
  XLF(-1.138845488453173720559090869349695013665749123053966886063934007e-05) ,
  XLF(-2.872789784985329839346948561380530010617370484875594835519237793e-08) ,
  XLF(-5.996216987387188258386427309101063013984749098381037237664635103e-17) ,
} ;
const xlf_type Poles_35[] = {
  XLF(-0.8718264522846033999342394061195880256470727705830912665217989902) ,
  XLF(-0.661266383704797300961590243559889585396787970468597759497540677) ,
  XLF(-0.4983391929337384452379841548311697188141004659525090030081443309) ,
  XLF(-0.3713759605149176482720350542079580293398141128093010991800163526) ,
  XLF(-0.2721803762296094393575708923396865462233310560831404983487216165) ,
  XLF(-0.1948725309279406466291289892705231229598703096700427955532161037) ,
  XLF(-0.1351438852882478430178858011566228749007004654435698322990999038) ,
  XLF(-0.08975883392391288140125772247222514252144984921722775882971425236) ,
  XLF(-0.05620557742615392424089724950105651030992598653610383642011580072) ,
  XLF(-0.03243728423609834830880510161503194535520176987756301783559450961) ,
  XLF(-0.0166690219153114058357659185710515802172279551695696752297456289) ,
  XLF(-0.00721702343640738546865426044583219398433188989439217014848981244) ,
  XLF(-0.002393632153391934976738453336174253298399748468472159832875633091) ,
  XLF(-0.0005082814595413963602179947181220570751322338012057876016538467729) ,
  XLF(-4.707210567991254277165743888852309796308110097300718701320097425e-05) ,
  XLF(-6.982167361691684787303327536741801670656721991854478901847667883e-07) ,
  XLF(-2.910506393324535687374866149601235708803686266342619866044464595e-11) ,
} ;
const xlf_type Poles_36[] = {
  XLF(-0.8750684054438948847452437373406664292611236884066242889288642594) ,
  XLF(-0.6687823036766544255379859722251323680047749665524258229415040719) ,
  XLF(-0.5081060927404221239947201115147411334358304319098157872485907861) ,
  XLF(-0.3820889147120881724540180154136545429612621860660454125962716) ,
  XLF(-0.2829752736425094565610373986802070821391575930451883961728558898) ,
  XLF(-0.2051615241418207572727997969528817223652452129707106074838495989) ,
  XLF(-0.14451740754635553297283498330242538411493124191880429538112964) ,
  XLF(-0.0979292687537774729071245716398629162926615109445190283416048203) ,
  XLF(-0.06297929294836479990958242477558204390841384178857692323889463793) ,
  XLF(-0.03770814705533074950721322074860584689824799331481849682475799349) ,
  XLF(-0.02042922248740917225652419002908239423559601850491465766821897678) ,
  XLF(-0.009579426581121988184135814928911638479896594280206909496592497805) ,
  XLF(-0.003611220147146754138028542854031671084901914644652113492134872149) ,
  XLF(-0.0009586466637195032820566510915883769629669327808512967663043218798) ,
  XLF(-0.0001375423203276128227751325930784359970998592216808835620834600289) ,
  XLF(-5.757252845898398275684359808988308159656730828735893183687005237e-06) ,
  XLF(-1.03338827926791787654999609776125053513014082753969839763193209e-08) ,
  XLF(-6.66246330967919499434053724033473943586232453420054571477522732e-18) ,
} ;
const xlf_type Poles_37[] = {
  XLF(-0.8781505478013884659147982039146191429554172009001990824695919527) ,
  XLF(-0.6759750473095329121930507307346648240013232615900152212905613459) ,
  XLF(-0.51751192597684673535943933018009947621992684254357013566134864) ,
  XLF(-0.3924709896295193646603292631813403487513963808345868441557056082) ,
  XLF(-0.2935071562988579964793774104326865572720481250740885379204084173) ,
  XLF(-0.2152767082291064937186817220600191746368374476873378239389736579) ,
  XLF(-0.1538183533518710859528029818294132876109657645325606260063974945) ,
  XLF(-0.1061338716626476411426469605322070807913023243648144140725098819) ,
  XLF(-0.06989275733426103265981254390513881080482550264003339404594638648) ,
  XLF(-0.04321391826377260296705031812585452479799546997916006826834407192) ,
  XLF(-0.0244953631883323823647994032926557267827645892519567084880868072) ,
  XLF(-0.01227642795169495998563028454995526435467331188779107959837355095) ,
  XLF(-0.005131933739471125785686386431120910226126867368175623514253818484) ,
  XLF(-0.00161854045004396671227841390207027781540938842753041362501455449) ,
  XLF(-0.0003187154552604863395905178506217389398462053401287750680304108978) ,
  XLF(-2.613169289577131021924859793304113754318711850182299727934643579e-05) ,
  XLF(-3.092266885451066382879815399764181243826377944052970487313410121e-07) ,
  XLF(-7.276131065451551553705333358570156901754325759020523687046512448e-12) ,
} ;
const xlf_type Poles_38[] = {
  XLF(-0.8810844012627599734510052333679306661258475871582915975183354138) ,
  XLF(-0.6828647854893554114611749196595961022046669812516270865341674094) ,
  XLF(-0.5265754081532769568125705273304797865876876752529184222498852699) ,
  XLF(-0.4025349275748515179303532349494372237459313802116942382392916943) ,
  XLF(-0.3037807859294119250547972933652790371407468852466401107412552645) ,
  XLF(-0.2252139929681739289389520458526953685645416021161753009872430301) ,
  XLF(-0.1630335348215804651084659583007473631897139758517007075874474948) ,
  XLF(-0.1143507456252265693234335556374301447233223914711100984308679374) ,
  XLF(-0.07691679639577583589865395330466435228808416915862999472469522428) ,
  XLF(-0.04892138275231939194828291450254472790783561299918000372996496771) ,
  XLF(-0.02883616221709854417423820419081887870825812026619930751611767068) ,
  XLF(-0.01528779400217048844947648824603569566787484444193002536183233717) ,
  XLF(-0.006956830335738254573880471095541693737462538024945412076596100204) ,
  XLF(-0.002514189750356696521640250002282235456229817645538615794802163848) ,
  XLF(-0.0006276117067541021362760836858935686241283161765630151995535965938) ,
  XLF(-8.188256383484340050684230163387384565153875154394489527208557087e-05) ,
  XLF(-2.9148971463249452196845816069833301908545518066006530514321514e-06) ,
  XLF(-3.718130253557185189659320961860860784155151645390455257007783541e-09) ,
  XLF(-7.402737007448795866796676541208151168640091132168127867274461559e-19) ,
} ;
const xlf_type Poles_39[] = {
  XLF(-0.8838804071427821403538573400038413542684225267333774944945296725) ,
  XLF(-0.6894700680083400659184881413107846576898799429348201246959462546) ,
  XLF(-0.5353140836684927401585843610161871270364923529126416793493216935) ,
  XLF(-0.4122931047661460010046659312530330357059065205259364136473437579) ,
  XLF(-0.3138014810230074140709270572250097647882280987948234857948216725) ,
  XLF(-0.2349707946459764815730416069548419384943178603174648217070090128) ,
  XLF(-0.1721521807622996270283830813993210037412921179142107951912987013) ,
  XLF(-0.122561160993944562864220405676381709565252994327507482588781223) ,
  XLF(-0.08402575235660301965392963042072378981949820562143619688073452448) ,
  XLF(-0.05480036756601220303752327387292872471060304044883900751548376047) ,
  XLF(-0.03342150293797134126219152488193779932419906557898733772011352821) ,
  XLF(-0.01859074722483924127966095754884896879187571795331881586252931603) ,
  XLF(-0.009079595195884276359663059407364285645767778417547513531325443767) ,
  XLF(-0.003662104378870138633637104472977441640395396108077576162517066551) ,
  XLF(-0.001097586856737514722051463876016155621381043342811812777686957317) ,
  XLF(-0.0002003068879448743003281295502194398955674433171563053669149427904) ,
  XLF(-1.45321316247905172365968562238955363512217736902440957179898996e-05) ,
  XLF(-1.370545029060038177725089857260695152348902198174582425063614265e-07) ,
  XLF(-1.819013794625787766003677964763074328080221008080626474426009181e-12) ,
} ;
const xlf_type Poles_40[] = {
  XLF(-0.8865480498249417239427008642652960486190151746361567239906664105) ,
  XLF(-0.6958079803507162016170415352239725903926397620664406565492174346) ,
  XLF(-0.5437444022594512304292991538755948156506013584800660408864804425) ,
  XLF(-0.4217574960836135760735709524382669355715274973575640737089458263) ,
  XLF(-0.3235749633481036665976933546361683677573000672674848853480301136) ,
  XLF(-0.2445457668079241226571226562971627843396378500156285307769417512) ,
  XLF(-0.1811655669715627651723204505658858064383274888853659588702144339) ,
  XLF(-0.1307491316833214492154291590394745734318271118026811489274112894) ,
  XLF(-0.09119711540837702671005955403236223294722560143602328150342637692) ,
  XLF(-0.06082364166735568921630568955938537615700163020351593484887383573) ,
  XLF(-0.03822293082901784992510790132362458922098823926664849012758909777) ,
  XLF(-0.02216137502477028067168835200751675594489780033002192164202868613) ,
  XLF(-0.0114886482426543975194893315944088434725923778733718568173041969) ,
  XLF(-0.005070168042357454064418051147786571569953137135000204986554756488) ,
  XLF(-0.001755563044393195285060352517031713953284630161352714167548947187) ,
  XLF(-0.0004118650546227838083044596471981422158427970594272683285924695645) ,
  XLF(-4.883749869313317942908248899528742439137511496965237578555703579e-05) ,
  XLF(-1.477664068328894179909487136579716007242828522498197961890553758e-06) ,
  XLF(-1.33800265813914219921316176928755813141341326695004689445150919e-09) ,
  XLF(-8.22526334047607454535486271035257115822466847965542839458599925e-20) ,
} ;
const xlf_type Poles_41[] = {
  XLF(-0.8890959638357837873469022953836399246288449125671096352859832732) ,
  XLF(-0.701894283045822693090333392530384140994025066196092798504051686) ,
  XLF(-0.5518817919367211668766568069591515221233174524575627655885217399) ,
  XLF(-0.4309396531921752276110843193597059893489616747715943193038832713) ,
  XLF(-0.3331072348755986375573819476643568411736978618085493378158771037) ,
  XLF(-0.2539385772414767539599720891513983314085613177649191147268291957) ,
  XLF(-0.1900667038736958640880944899109356184028768960348825827085321728) ,
  XLF(-0.1389010462253641359590036042388613572070161268938288702402679418) ,
  XLF(-0.09841117686633293944432583376322831792096348808502462072703927122) ,
  XLF(-0.06696674766202966506890676491489340355301803471998128364424764985) ,
  XLF(-0.04321391826377224185679885786371453233952356822996869493387370648) ,
  XLF(-0.02597561948079389846050245809589723368039309202890154753255867357) ,
  XLF(-0.01416888539677724491675220290304631912898640250999730674086817242) ,
  XLF(-0.006739186449046984038080911435844015053701108547912994938306338808) ,
  XLF(-0.00262104180821929193652757454556567953998534980504767823799715708) ,
  XLF(-0.0007461328733157922833797184262927803175879006213308751009054405432) ,
  XLF(-0.0001261350888067020144430598696989766129159126927479792117631999536) ,
  XLF(-8.093502330497814676653757956531789181149791636306257030476742624e-06) ,
  XLF(-6.078089086680648675357094830545843973084443584869234951774044295e-08) ,
  XLF(-4.547507808420938896720625846449621216033643220349188723154739851e-13) ,
} ;
const xlf_type Poles_42[] = {
  XLF(-0.8915320268698522159651647309409898698731488935948280938605312513) ,
  XLF(-0.7077435357376586388858410940687095792858920465042904375498401819) ,
  XLF(-0.5597407280298025835090097476707528895889058210710629541672853992) ,
  XLF(-0.4398506928520543702522960970484893773080325900543098559008872809) ,
  XLF(-0.3424044791801646434681043343247679811771476619885264677273003911) ,
  XLF(-0.2631497229248989413568644924188030250797685093594147906749133287) ,
  XLF(-0.1988500723607262106739088511038080175064902350929060755680797828) ,
  XLF(-0.1470053478138674280323752622112899295178614489003852780125198287) ,
  XLF(-0.1056507102479482771048677935189569652285606137594146388606492182) ,
  XLF(-0.0732077991194162305895413659749245559880295171793736680711999188) ,
  XLF(-0.04836997283028642285462458648811236312434105938659972391425244394) ,
  XLF(-0.0300099472525122720946746368368032003662655931894695709023281059) ,
  XLF(-0.01710304346825772288069577457560739794564483583185968574040843086) ,
  XLF(-0.008664489321413533350337112095657616565847803017696510225767787322) ,
  XLF(-0.003706304925360371606803185866509402690510381270201288741985858992) ,
  XLF(-0.001228913565528939125078519304902651811485016604792305852587191118) ,
  XLF(-0.0002708325035954437436783014078654001809518243058307399683824956403) ,
  XLF(-2.917476017494444073456549303554553293863110238727813211098795334e-05) ,
  XLF(-7.498609369472629308544215472257229740522463123374387877904055657e-07) ,
  XLF(-4.815479655866042128205442074833506699433411427882331039523924082e-10) ,
  XLF(-9.139181489029075064260505998730771341799704346795797559508345866e-21) ,
} ;
const xlf_type Poles_43[] = {
  XLF(-0.893863440867893358608828901893563497810908751596147384916450371) ,
  XLF(-0.7133692078362500762254293013499779757058403109135527663592579498) ,
  XLF(-0.5673347981712786697944014046050991485878993432456727187652533242) ,
  XLF(-0.4485012929513466523033762513202365413857949704604126689088403276) ,
  XLF(-0.3514729825996737699863520720668652637190478996596308842905246751) ,
  XLF(-0.2721803762296094398715259352720103937021937668858261345535146447) ,
  XLF(-0.2075114001378054455728203676821808935160404127100836552303700095) ,
  XLF(-0.1550522574708870535187643983914013179243763048323631791625734408) ,
  XLF(-0.1129006825209351018462604631594150605652417220648747297441159382) ,
  XLF(-0.07952726494797824527452900402745299894178991587219038642874328136) ,
  XLF(-0.0536686430180897847565647778831917630437506181627560030120559346) ,
  XLF(-0.03424178287301239611979002915701822580782004283452107789848164312) ,
  XLF(-0.02027272969155223317872331706788925401264853242898531741842110359) ,
  XLF(-0.0108373831470151738767331253445189914819894105048839802154604759) ,
  XLF(-0.005017257442672953433759649112858697950215891223015279326312800931) ,
  XLF(-0.001880716222005879490408415736977615798879384426669420484772851697) ,
  XLF(-0.0005082812655905478544414023088409706846704423690510655413901375747) ,
  XLF(-7.956174663838349201171982112646408318851297885982006145797839986e-05) ,
  XLF(-4.513325250611668622670878692265663961000697908990863106300239453e-06) ,
  XLF(-2.696759817112634783679825031765393335855971365215612678489021744e-08) ,
  XLF(-1.136873200559593259398193993017687665952388539500605071159134504e-13) ,
} ;
const xlf_type Poles_44[] = {
  XLF(-0.8960968028992146665627409301695607302205014963225443894685342127) ,
  XLF(-0.7187837773708819352136947096079319017090596410656272550377272326) ,
  XLF(-0.5746767631908139827528406251451784714430500492740761209250364974) ,
  XLF(-0.4569016943466847335506940555269725780044767461717720240889695593) ,
  XLF(-0.3603190713769044571789712579438782147170830151763222752366173198) ,
  XLF(-0.2810322569159787547930034286249012421326484907559484425330827152) ,
  XLF(-0.2160474721108212555687295004555083560208818252478624520061428472) ,
  XLF(-0.1630335348215804648625873336213043131376530422779831554484065536) ,
  XLF(-0.1201479953744980079922840269252211154399742809514526337471659156) ,
  XLF(-0.08590775414808882023294983578453695403030749604255062341580882368) ,
  XLF(-0.05908945890328844788754841773552247793962609244906350553590671065) ,
  XLF(-0.03864977109894073793200650011114251143377588126854134896148539826) ,
  XLF(-0.02365917402130976673278605911930246830033721946504547153497426121) ,
  XLF(-0.01324637525169005773742919987891981609058182960094684094561128642) ,
  XLF(-0.00655452780820816870084755830167570427328585406514796089829324295) ,
  XLF(-0.002716322197289291416048524927342344422154897427943894833746072091) ,
  XLF(-0.0008621057366132465648673765980953159889884901490970504618348817264) ,
  XLF(-0.0001784049129824874983750749912243393871433082536567270149530538835) ,
  XLF(-1.745255401918100730688816665717770286170547015610198053753240676e-05) ,
  XLF(-3.808577626111630594159094735088815707265727307233566729871581692e-07) ,
  XLF(-1.733235166498085104752211857283419959035264593098427257293460137e-10) ,
  XLF(-1.015464609878786614079491151475195110882861668111464956725349865e-21) ,
} ;
const xlf_type Poles_45[] = {
  XLF(-0.898238167312102371349291053804821640312192793555562471215476875) ,
  XLF(-0.7239988194534050450565699116216409286927740874829315191220161918) ,
  XLF(-0.5817786139867798902425713772156262872378141049843191723750717028) ,
  XLF(-0.4650617070250951283723883102723573446594725565665736785548979181) ,
  XLF(-0.368949061752944921465166012473919422035541674539231445258506868) ,
  XLF(-0.28970752546932497793372776574871614478649833989202096321840715) ,
  XLF(-0.2244559694078480600273876525332568970050856529133041456485020925) ,
  XLF(-0.1709422714709099403943210263664855143389875812409323306553227399) ,
  XLF(-0.127381255033133456657313390242003823329182271006023449160273537) ,
  XLF(-0.09233380888114754628108377637591134087471387005282428247572939459) ,
  XLF(-0.06461383375765931115919652801084614876607520051980950747351586994) ,
  XLF(-0.0432139182637722494695217347313226126861928859040447373499280064) ,
  XLF(-0.02724376234303231214714221804415821981624772806286889997360655287) ,
  XLF(-0.01587815723560460470494461070583947317801425002006642339597852799) ,
  XLF(-0.008314592328776581400858429423285613326485676365826471754960988952) ,
  XLF(-0.003745054905481998368479047563539474655222912170792398409464371879) ,
  XLF(-0.001352465418015474711809167945824158348426517243049641995552176944) ,
  XLF(-0.0003468798403437999650432616403275257135294579726410741017702179129) ,
  XLF(-5.025781166635491597024308150332417618545992094836478580674841117e-05) ,
  XLF(-2.51960760849922105778664437501872178787878497284421041128478423e-06) ,
  XLF(-1.196948365983150080846074019812402782497101718879224902675681104e-08) ,
  XLF(-2.84217772584202427130585839790790701423165122912176737813096462e-14) ,
} ;
const xlf_type* const precomputed_poles[] = {
  0, 
  0, 
  Poles_2, 
  Poles_3, 
  Poles_4, 
  Poles_5, 
  Poles_6, 
  Poles_7, 
  Poles_8, 
  Poles_9, 
  Poles_10, 
  Poles_11, 
  Poles_12, 
  Poles_13, 
  Poles_14, 
  Poles_15, 
  Poles_16, 
  Poles_17, 
  Poles_18, 
  Poles_19, 
  Poles_20, 
  Poles_21, 
  Poles_22, 
  Poles_23, 
  Poles_24, 
  Poles_25, 
  Poles_26, 
  Poles_27, 
  Poles_28, 
  Poles_29, 
  Poles_30, 
  Poles_31, 
  Poles_32, 
  Poles_33, 
  Poles_34, 
  Poles_35, 
  Poles_36, 
  Poles_37, 
  Poles_38, 
  Poles_39, 
  Poles_40, 
  Poles_41, 
  Poles_42, 
  Poles_43, 
  Poles_44, 
  Poles_45, 
} ;
const xlf_type* const precomputed_basis_function_values[] = {
  K0, 
  K1, 
  K2, 
  K3, 
  K4, 
  K5, 
  K6, 
  K7, 
  K8, 
  K9, 
  K10, 
  K11, 
  K12, 
  K13, 
  K14, 
  K15, 
  K16, 
  K17, 
  K18, 
  K19, 
  K20, 
  K21, 
  K22, 
  K23, 
  K24, 
  K25, 
  K26, 
  K27, 
  K28, 
  K29, 
  K30, 
  K31, 
  K32, 
  K33, 
  K34, 
  K35, 
  K36, 
  K37, 
  K38, 
  K39, 
  K40, 
  K41, 
  K42, 
  K43, 
  K44, 
  K45, 
} ;


} ; // end of namespace vspline_constants

#define VSPLINE_POLES_H
#endif
