/******************************************************************************
 *
 * Name:	skgi.c
 * Project:	SysKonnect SK-9Dxx Gigabit Ethernet
 * Version:	$Revision: 1.10 $
 * Date:	$Date: 2002/02/18 10:14:02 $
 * Purpose:	The main driver source module
 *
 ******************************************************************************/

/******************************************************************************
 *
 *	(C)Copyright 2001 SysKonnect GmbH.
 *
 *	Driver for
 *		SysKonnect SK-9D21 Gigabit Ethernet 10/100/1000Base-T
 *		SysKonnect SK-9D41 Gigabit Ethernet 1000Base-SX with SC connector
 *		SysKonnect SK-9D51 Gigabit Ethernet 1000Base-SX with MTRJ connector
 *		SysKonnect SK-9D61 Gigabit Ethernet 1000Base-SX with VF-45 connector
 *
 *	Created 12-Apr-2001, based on SysKonnect's skge.c
 *	Author: Ralf Assmann (linux@syskonnect.de)
 *
 *	Address all questions to: linux@syskonnect.de
 *
 *	The technical manual for the adapters may be made available on
 *	SysKonnect's web pages: http://www.syskonnect.com
 *	Goto "Support" and search the "Knowledge Base" for "manual".
 *	
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	The information in this file is provided "AS IS" without warranty.
 *
 ******************************************************************************/

/******************************************************************************
 *
 * History:
 *
 *	$Log: skgi.c,v $
 *	Revision 1.10  2002/02/18 10:14:02  mlindner
 *	add: New driver info
 *	add: New boot messages (autoneg, flowctrl)
 *	fix: VLAN Error
 *	fix: FlowCtrl init
 *	fix: Module parameters
 *	
 *	Revision 1.9  2001/10/22 15:57:47  rschmidt
 *	Changes for proc-file (Kernel 2.2.x and 2.4.x)
 *	
 *	Revision 1.8  2001/10/05 13:25:50  rschmidt
 *	Fixed ShowDriverStatus() and Link up/down message
 *	Editorial changes.
 *	
 *	Revision 1.7  2001/08/23 08:12:19  mlindner
 *	Added support for LBFO
 *	Send and receive ring structures removed
 *	Physical address of buffer changed
 *	
 *	Revision 1.6  2001/08/14 13:52:30  rschmidt
 *	Renamed Rx and Tx descriptor structures.
 *	Added ShowDriverStatus() and counter for Rx errors.
 *	Moved some structure defines into HWAC.
 *	Eliminated define for SK_JUMBO.
 *	Editorial changes.
 *	
 *	Revision 1.5  2001/08/08 15:18:48  mlindner
 *	Changed VER_STRING, BOOT_STRING and SysKonnectFileId[] for
 *	initial release process
 *	
 *	Revision 1.4  2001/08/08 12:48:35  rschmidt
 *	Added SkErrorLog(), changed DMA descriptors (address field) and
 *	moved some register values into HW-structure.
 *	
 *	Revision 1.3  2001/06/05 15:13:49  rassmann
 *	Kernel 2.4 support.
 *	
 *	Revision 1.2  2001/06/05 08:48:07  rassmann
 *	Added SysKonnectFileId.
 *	
 *	Revision 1.1  2001/06/05 08:28:23  rassmann
 *	First public version.
 *	
 *
 ******************************************************************************/

/******************************************************************************
 *
 * Possible compiler options (#define xxx / -Dxxx):
 *
 *	- Debugging can be enable by defining DEBUGin the Makefile.
 *	  Then specify the parameters DebugComponents and DebugCategories
 *	  on the command line or below for each instance.
 *
 ******************************************************************************/

/******************************************************************************
 *
 * Description:
 *
 *	This is the main module of the
 *	SysKonnect SK-9Dxx Gigabit Ethernet Linux driver.
 */

#include	"h/skversion.h"

#include	"h/skdrv1st.h"
#include	<linux/module.h>
#include 	<linux/proc_fs.h>
#include	"h/skdrv2nd.h"


/* defines ******************************************************************/

/* Number of adapters that can be configured via command line params. */
#define SK_MAX_CARD_PARAM	16

/************************************************************
 * Use these defines for a compiled-in version of the driver
 * instead of command line parameters.
 */
// #define AUTO_NEG		{"Sense", }		// Sense, On, Off
// #define DUP_CAP		{"Both", }		// Both, Full, Half
// #define FLOW_CTRL	{"SymOrRem", }	// SymOrRem, Sym, LocSend, None
// #define ROLE			{"Auto", }		// Auto, Master, Slave (copper1000 only)
// #define SPEED		{"Auto", }		// Auto, 1000, 100, 10 (copper only)
//		0..RX_COPY_SIZE_MAX
// #define RX_COPY_SIZE							{"RX_COPY_SIZE_DEF", }
//		0, TASK_OFFLOAD_CAP_DEF
// #define TASK_OFFLOAD_CAP						{"TASK_OFFLOAD_CAP_DEF", }
//		0..RX_COAL_TICKS_MAX
// #define RX_COALESCING_TICKS					{"RX_COAL_TICKS_DEF", }
//		0..TX_COAL_TICKS_MAX
// #define TX_COALESCING_TICKS					{"TX_COAL_TICKS_DEF", }
//		0..RX_COAL_FRAMES_MAX
// #define RX_MAX_COALESCED_FRAMES				{"RX_COAL_FRAMES_DEF", }
//		0..TX_COAL_FRAMES_MAX
// #define TX_MAX_COALESCED_FRAMES				{"TX_COAL_FRAMES_DEF", }
//		0..RX_COAL_TICKS_INT_MAX
// #define RX_COALESCING_TICKS_DURING_INT		{"RX_COAL_TICKS_INT_DEF", }
//		0..TX_COAL_TICKS_INT_MAX
// #define TX_COALESCING_TICKS_DURING_INT		{"TX_COAL_TICKS_INT_DEF", }
//		0..RX_COAL_FRAMES_INT_MAX
// #define RX_MAX_COALESCED_FRAMES_DURING_INT	{"RX_COAL_TICKS_INT_DEF", }
//		0..TX_COAL_FRAMES_INT_MAX
// #define TX_MAX_COALESCED_FRAMES_DURING_INT	{"RX_COAL_FRAMES_INT_DEF", }
//		0..STATS_COAL_TICKS_INT_MAX
// #define STATS_COALESCING_TICKS				{"STATS_COAL_TICKS_INT_DEF", }
#ifdef DEBUG
// #define DBG_COMP		{0xFFFFFFFF, }		// See Debug Components
//		**** possible DebugComponents *****************
//		SK_DBGMOD_MERR        0x00000001L	general module error indication
//		SK_DBGMOD_HWM         0x00000002L	Hardware init module
//		SK_DBGMOD_RLMT        0x00000004L	RLMT module
//		SK_DBGMOD_VPD         0x00000008L	VPD module
//		SK_DBGMOD_I2C         0x00000010L	I2C module
//		SK_DBGMOD_PNMI        0x00000020L	PNMI module
//		SK_DBGMOD_CSUM        0x00000040L	CSUM module
//		SK_DBGMOD_ADDR        0x00000080L	ADDR module
//		SK_DBGMOD_DRV         0x00010000L	DRV module

// #define DBG_CAT		{0xFFFFFFFF, }
//		**** possible DebugCategories **************
//		*** modules ***
//		SK_DBGCAT_INIT        0x00000001L	module/driver initialization
//		SK_DBGCAT_CTRL        0x00000002L	controlling: add/remove MCA/MAC
//											  and other controls (IOCTL)
//		SK_DBGCAT_ERR         0x00000004L	error handling paths
//		SK_DBGCAT_TX          0x00000008L	transmit path
//		SK_DBGCAT_RX          0x00000010L	receive path
//		SK_DBGCAT_IRQ         0x00000020L	general IRQ handling
//		SK_DBGCAT_QUEUE       0x00000040L	any queue management
//		SK_DBGCAT_DUMP        0x00000080L	large data output e.g. hex dump
//		SK_DBGCAT_FATAL       0x00000100L	large data output e.g. hex dump
//		*** driver ***
//		SK_DBGCAT_DRV_ENTRY			0x00010000
//		SK_DBGCAT_DRV_SAP			0x00020000
//		SK_DBGCAT_DRV_MCA			0x00040000
//		SK_DBGCAT_DRV_TX_PROGRESS	0x00080000
//		SK_DBGCAT_DRV_RX_PROGRESS	0x00100000
//		SK_DBGCAT_DRV_PROGRESS		0x00200000
//		SK_DBGCAT_DRV_MSG			0x00400000
//		SK_DBGCAT_DRV_PROM			0x00800000
//		SK_DBGCAT_DRV_TX_FRAME		0x01000000
//		SK_DBGCAT_DRV_ERROR			0x02000000
//		SK_DBGCAT_DRV_INT_SRC		0x04000000
//		SK_DBGCAT_DRV_EVENT			0x08000000
#endif	/* DEBUG */
/*
 * End of defines for a compiled-in version of the driver.
 ************************************************************/

/* For debuging on x86 only. */
/* #define BREAKPOINT() asm(" int $3"); */

#ifdef SK_KERNEL_22

#define	netif_stop_queue(dev)   	set_bit(0, (void *)&(dev)->tbusy);
#define	netif_start_queue(dev)  	(dev)->tbusy = 0;
#define netif_running(dev)			((dev)->start != 0)
#define	pci_resource_start(pdev,nr) (pdev)->base_address[nr]
#define	pci_map_single(pPciDev,pMem,Len,Direction)	virt_to_bus(pMem)
#define	pci_unmap_single(pPciDev,Phys,Len,Direction)
#if 0
#define	pci_dma_sync_single(pPciDev,Phys,Len,Direction)
#endif	/* 0 */

// #define DEV_KFREE_SKB(skb)		dev_kfree_skb(skb)
// #define DEV_KFREE_SKB_IRQ(skb)	dev_kfree_skb(skb)
#define DEV_KFREE_SKB_ANY(skb)	dev_kfree_skb(skb)

#define	SK_FREE_CONSISTENT_IF_ALLOCED(pPciDev,AllocLength,pMem,PhysAddr)	\
	if (pMem != NULL) {														\
		kfree((SK_U8 *)pMem);												\
		pMem = NULL;														\
	}

#else  /* !SK_KERNEL_22 */

// #define DEV_KFREE_SKB(skb)		dev_kfree_skb(skb)
// #define DEV_KFREE_SKB_IRQ(skb)	dev_kfree_skb_irq(skb)
#define DEV_KFREE_SKB_ANY(skb)	dev_kfree_skb_any(skb)

#define	SK_FREE_CONSISTENT_IF_ALLOCED(pPciDev,AllocLength,pMem,PhysAddr)	\
	if (pMem != NULL) {														\
		pci_free_consistent(pPciDev, AllocLength, (SK_U8 *)pMem, PhysAddr);	\
		pMem = NULL;														\
	}

#endif /* !SK_KERNEL_22 */

#define	SK_FREE_IF_ALLOCED(pMem)	\
	if (pMem != NULL) {				\
		kfree((SK_U8 *)pMem);		\
		pMem = NULL;				\
	}

/* external Proc function */
extern int proc_read(
	char	*buffer,
	char	**buffer_location,
	off_t	offset,
	int		buffer_length,
	int		*eof,
	void	*data);

/* function prototypes ******************************************************/
int			init_module(void);
void		cleanup_module(void);

static int	SkGiOpen(
	NET_DEV			*dev);
static int	SkGiClose(
	NET_DEV			*dev);
static int	SkGiXmit(
	struct sk_buff	*skb,
	NET_DEV			*dev);
static int	SkGiSetMacAddr(
	NET_DEV			*dev,
	void			*p);
static void	SkGiSetRxMode(
	NET_DEV			*dev);
static NET_STATS *SkGiStats(
	NET_DEV			*dev);
static int	SkGiIoctl(
	NET_DEV			*dev,
	struct ifreq	*rq,
	int				cmd);
static int	SkGiChangeMtu(
	NET_DEV			*dev,
	int				new_mtu);
static void	SkGiIsr(
	int				irq,
	void			*dev_id,
	struct pt_regs	*ptregs);
static void SkGiTimer(
	ulong			Data);

static void GetProductStr(
	SK_AC			*pAC);
static void	FreeResources(
	NET_DEV			*dev);
static SK_BOOL	AllocDriverMem(
	SK_AC			*pAC);
static void GetConfiguration(
	SK_AC			*pAC);
static void	FreeSndRings(
	SK_AC			*pAC);
static void	FreeRecRings(
	SK_AC			*pAC);
static void	FillRecPrdRing(
	SK_AC			*pAC,
	SK_IOC			IoC,
	P_REC_RING		pRecPrdRing);
static void	FreeSbds(
	SK_AC			*pAC,
	int 			RingIdx);

#ifdef SK_KERNEL_22
/* Proc functions */
static struct dentry *SkGiProcLookup(
	struct inode	*dir,
	struct dentry	*dentry);
static int	SkGiProcFileOpen(
	struct inode	*inode,
	struct file		*file);
static int	SkGiProcFileRelease(
	struct inode	*inode,
	struct file		*file);
#endif /* !SK_KERNEL_22 */

#ifdef DEBUG
#if defined(SK_DUMP_TX) || defined(SK_DUMP_RX)
static void DumpMsg(
	struct sk_buff	*skb,
	char			*str);
static void DumpData(
	char			*p,
	int				size);
#if 0
static void DumpLong(
	char			*pc,
	int				size);
#endif	/* 0 */
#endif	/* defined(SK_DUMP_TX) || defined(SK_DUMP_RX) */
#endif	/* DEBUG */


/* global variables *********************************************************/
SK_AC	*pACList = NULL;

/* local variables **********************************************************/
const char SK_Root_Dir_entry[8];

#ifdef SK_KERNEL_22
static struct inode_operations 	SkInodeOps;
static struct file_operations	SkFileOps;  /* with open/release */

static struct proc_dir_entry	Our_Proc_Dir = {
	0,
	sizeof(SK_Root_Dir_entry) - 1,
	SK_Root_Dir_entry,
	S_IFDIR | S_IRUGO,
	2, 0, 0, 0, NULL,
	NULL
};
#endif /* SK_KERNEL_22 */

static struct proc_dir_entry	*pSkRootDir;


static const char				*BootString = BOOT_STRING;
static int 						Probed __initdata = 0;

#ifdef AUTO_NEG
static char *AutoNeg[SK_MAX_CARD_PARAM] = AUTO_NEG;
#else	/* AUTO_NEG */
static char *AutoNeg[SK_MAX_CARD_PARAM] = {"", };
#endif	/* AUTO_NEG */

#ifdef DUP_CAP
static char *DupCap[SK_MAX_CARD_PARAM] = DUP_CAP;
#else	/* DUP_CAP */
static char *DupCap[SK_MAX_CARD_PARAM] = {"", };
#endif	/* DUP_CAP */

#ifdef FLOW_CTRL
static char *FlowCtrl[SK_MAX_CARD_PARAM] = FLOW_CTRL;
#else	/* FLOW_CTRL */
static char *FlowCtrl[SK_MAX_CARD_PARAM] = {"", };
#endif	/* FLOW_CTRL */

#ifdef ROLE
static char *Role[SK_MAX_CARD_PARAM] = ROLE;
#else	/* ROLE */
static char *Role[SK_MAX_CARD_PARAM] = {"", };
#endif	/* ROLE */

#ifdef SPEED
static char *Speed[SK_MAX_CARD_PARAM] = SPEED;
#else	/* SPEED */
static char *Speed[SK_MAX_CARD_PARAM] = {"", };
#endif	/* SPEED */

#ifdef RX_COPY_SIZE
static char *RxCopySize[SK_MAX_CARD_PARAM] = RX_COPY_SIZE;
#else	/* RX_COPY_SIZE */
static char *RxCopySize[SK_MAX_CARD_PARAM] = {"", };
#endif	/* RX_COPY_SIZE */

#ifdef TASK_OFFLOAD_CAP
static char *TaskOffloadCap[SK_MAX_CARD_PARAM] = TASK_OFFLOAD_CAP;
#else	/* TASK_OFFLOAD_CAP */
static char *TaskOffloadCap[SK_MAX_CARD_PARAM] = {"", };
#endif	/* TASK_OFFLOAD_CAP */

#ifdef RX_COALESCING_TICKS
static char *RxCoalescingTicks[SK_MAX_CARD_PARAM] = RX_COALESCING_TICKS;
#else	/* RX_COALESCING_TICKS */
static char *RxCoalescingTicks[SK_MAX_CARD_PARAM] = {"", };
#endif	/* RX_COALESCING_TICKS */

#ifdef TX_COALESCING_TICKS
static char *TxCoalescingTicks[SK_MAX_CARD_PARAM] = TX_COALESCING_TICKS;
#else	/* TX_COALESCING_TICKS */
static char *TxCoalescingTicks[SK_MAX_CARD_PARAM] = {"", };
#endif	/* TX_COALESCING_TICKS */

#ifdef RX_MAX_COALESCED_FRAMES
static char *RxMaxCoalescedFrames[SK_MAX_CARD_PARAM] = RX_MAX_COALESCED_FRAMES;
#else	/* RX_MAX_COALESCED_FRAMES */
static char *RxMaxCoalescedFrames[SK_MAX_CARD_PARAM] = {"", };
#endif	/* RX_MAX_COALESCED_FRAMES */

#ifdef TX_MAX_COALESCED_FRAMES
static char *TxMaxCoalescedFrames[SK_MAX_CARD_PARAM] = TX_MAX_COALESCED_FRAMES;
#else	/* TX_MAX_COALESCED_FRAMES */
static char *TxMaxCoalescedFrames[SK_MAX_CARD_PARAM] = {"", };
#endif	/* TX_MAX_COALESCED_FRAMES */

#ifdef RX_COALESCING_TICKS_DURING_INT
static char *RxCoalescingTicksDuringInt[SK_MAX_CARD_PARAM] =
	RX_COALESCING_TICKS_DURING_INT;
#else	/* RX_COALESCING_TICKS_DURING_INT */
static char *RxCoalescingTicksDuringInt[SK_MAX_CARD_PARAM] = {"", };
#endif	/* RX_COALESCING_TICKS_DURING_INT */

#ifdef TX_COALESCING_TICKS_DURING_INT
static char *TxCoalescingTicksDuringInt[SK_MAX_CARD_PARAM] =
	TX_COALESCING_TICKS_DURING_INT;
#else	/* TX_COALESCING_TICKS_DURING_INT */
static char *TxCoalescingTicksDuringInt[SK_MAX_CARD_PARAM] = {"", };
#endif	/* TX_COALESCING_TICKS_DURING_INT */

#ifdef RX_MAX_COALESCED_FRAMES_DURING_INT
static char *RxMaxCoalescedFramesDuringInt[SK_MAX_CARD_PARAM] =
	RX_MAX_COALESCED_FRAMES_DURING_INT;
#else	/* RX_MAX_COALESCED_FRAMES_DURING_INT */
static char *RxMaxCoalescedFramesDuringInt[SK_MAX_CARD_PARAM] = {"", };
#endif	/* RX_MAX_COALESCED_FRAMES_DURING_INT */

#ifdef TX_MAX_COALESCED_FRAMES_DURING_INT
static char *TxMaxCoalescedFramesDuringInt[SK_MAX_CARD_PARAM] =
	TX_MAX_COALESCED_FRAMES_DURING_INT;
#else	/* TX_MAX_COALESCED_FRAMES_DURING_INT */
static char *TxMaxCoalescedFramesDuringInt[SK_MAX_CARD_PARAM] = {"", };
#endif	/* TX_MAX_COALESCED_FRAMES_DURING_INT */

#ifdef STATS_COALESCING_TICKS
static char *StatsCoalescingTicks[SK_MAX_CARD_PARAM] = STATS_COALESCING_TICKS;
#else	/* STATS_COALESCING_TICKS */
static char *StatsCoalescingTicks[SK_MAX_CARD_PARAM] = {"", };
#endif	/* STATS_COALESCING_TICKS */

#ifdef DEBUG
#ifdef DBG_COMP
static int	DebugComponents[SK_MAX_CARD_PARAM] = DBG_COMP;
#else	/* DBG_COMP */
static int	DebugComponents[SK_MAX_CARD_PARAM] = {};
#endif	/* DBG_COMP */

#ifdef DBG_CAT
static int	DebugCategories[SK_MAX_CARD_PARAM] = DBG_CAT;
#else	/* DBG_CAT */
static int	DebugCategories[SK_MAX_CARD_PARAM] = {};
#endif	/* DBG_CAT */
#endif	/* DEBUG */

#ifdef MODULE
static int debug = 0;							/* Not used. */
static int options[SK_MAX_CARD_PARAM] = {0, };	/* Not used. */
#endif	/* MODULE */


#ifdef SK_KERNEL_22
void *pci_alloc_consistent(
struct pci_dev	*PciDev,
int				AllocLength,
dma_addr_t		*PhysAddr)
{
	void	*pMem;

	pMem = kmalloc(AllocLength, GFP_KERNEL);
	if (pMem != NULL) {
		SK_MEMSET(pMem, 0, AllocLength);
		*PhysAddr = virt_to_bus(pMem);
	}
	return (pMem);
}

/*****************************************************************************
 *
 * 	SkGiProcLookup - ???
 *
 * Description:
 *	This function ???
 *
 * Returns:
 *	???
 */
static struct dentry *SkGiProcLookup(
struct inode	*dir,
struct dentry	*dentry)
{
	return (NULL);
}	/* SkGiProcLookup */


/*****************************************************************************
 *
 * 	SkGiProcFileOpen - ???
 *
 * Description:
 *	This function ???
 *
 * Returns:
 *	???
 */
static int SkGiProcFileOpen(
struct inode	*inode,
struct file		*file)
{
	MOD_INC_USE_COUNT;
	return (0);
}	/* SkGiProcFileOpen */


/*****************************************************************************
 *
 * 	SkGiProcFileRelease - ???
 *
 * Description:
 *	This function ???
 *
 * Returns:
 *	???
 */
static int SkGiProcFileRelease(
struct inode	*inode,
struct file		*file)
{
	MOD_DEC_USE_COUNT;
	return (0);
}	/* SkGiProcFileRelease */

#endif /* SK_KERNEL_22 */

/*****************************************************************************
 *
 * 	PushRpd - put a Receive Packet Descriptor into a queue
 *
 * Description:
 *	This function queues a Receive Packet Descriptor.
 *
 * Returns: N/A
 *
 */
__inline static void PushRpd(
SK_RX_PACKET_QUEUE	*RpdQ,
P_SK_RX_PACKET		pRpd)
{
	pRpd->pNext = *RpdQ;
	*RpdQ = pRpd;
	return;
}	/* PushRpd */


/*****************************************************************************
 *
 * 	PopRpd - get a Receive Packet Descriptor from a queue
 *
 * Description:
 *	This function dequeues a Receive Packet Descriptor.
 *
 * Returns:
 *	Pointer to first Receive Packet Descriptor in queue (NULL if empty)
 */
__inline static P_SK_RX_PACKET PopRpd(
SK_RX_PACKET_QUEUE	*RpdQ)
{
	P_SK_RX_PACKET	pRet;

	pRet = *RpdQ;
#ifdef SK_CHECK_QUEUE
	if (*RpdQ != NULL) {
		*RpdQ = (*RpdQ)->pNext;
	}
#else	/* !SK_CHECK_QUEUE */
	*RpdQ = (*RpdQ)->pNext;
#endif	/* !SK_CHECK_QUEUE */

	return (pRet);
}	/* PopRpd */


/*****************************************************************************
 *
 * 	IsEmpty - check if a queue ist empty
 *
 * Description:
 *	This function checks if a queue is empty.
 *
 * Returns:
 *	SK_TRUE if queue is empty
 *	SK_FALSE else
 */
__inline static SK_BOOL IsEmpty(
SK_RX_PACKET_QUEUE	*RpdQ)
{
	return (*RpdQ == NULL);
}	/* IsEmpty */


uint power(
uint base,
uint exp)
{
	uint res;
	for (res = 1; exp > 0; exp--) {
		res *= base;
	}
	return (res);
}

/*****************************************************************************
 *
 * 	StrToU32 - Convert a string to a SK_U32 value
 *
 * Description:
 *	This function converts a zero-terminated string containing
 *	exactly one number to a SK_U32 value.
 *
 * Returns:
 *	SK_TRUE if string contains one single non-negative number,
 *			eventually surrounded by white space
 *	SK_FALSE otherwise
 *
 */
SK_BOOL	StrToU32(
const char	*Str,
SK_U32		*Num)
{
	SK_U32	Number;

#define	SK_IS_DIGIT(c)	(c >= '0' && c <= '9')
#define	SK_IS_SPACE(c)	(c >= 0x09 && c <= 0x0D)

	/* Skip leading white space. */
	while (*Str != 0 && SK_IS_SPACE(*Str)) {
		Str++;
	}

	if (!SK_IS_DIGIT(*Str)) {
		*Num = 0;
		return (SK_FALSE);
	}
	Number = 0;

	/* Concatenate number while isdigit. */
	while (SK_IS_DIGIT(*Str)) {
		Number = 10 * Number + (*Str - '0');
		Str++;
	}

	/* Skip trailing white space. */
	while (*Str != 0 && SK_IS_SPACE(*Str)) {
		Str++;
	}

	*Num = Number;

	/* If we reached the end of the dting, return true. */
	return (*Str == 0);
}	/* StrToU32 */


/*****************************************************************************
 *
 *	SkErrorLog - log errors
 *
 * Description:
 *	This function logs errors to the system buffer and to the console
 *
 * Returns:
 *	0 if everything ok
 *	< 0  on error
 *	
 */
void SkErrorLog(
const SK_AC	*pAC,
int		ErrClass,
int		ErrNum,
char	*pErrorMsg)
{
char	ClassStr[80];

	switch (ErrClass) {
	case SK_ERRCL_OTHER:
		strcpy(ClassStr, "Other error");
		break;
	case SK_ERRCL_CONFIG:
		strcpy(ClassStr, "Configuration error");
		break;
	case SK_ERRCL_INIT:
		strcpy(ClassStr, "Initialization error");
		break;
	case SK_ERRCL_NORES:
		strcpy(ClassStr, "Out of resources error");
		break;
	case SK_ERRCL_SW:
		strcpy(ClassStr, "Internal Software error");
		break;
	case SK_ERRCL_HW:
		strcpy(ClassStr, "Hardware failure");
		break;
	case SK_ERRCL_COMM:
		strcpy(ClassStr, "Communication error");
		break;
	}
	printk(KERN_INFO "%s: -- ERROR --\n        Class: %s\n"
		"        Nr: 0x%x\n        Msg: %s\n",
		pAC->dev->name, ClassStr, ErrNum, pErrorMsg);

} /* SkErrorLog */

static void ShowDriverStatus(
SK_AC	*pAC)
{
	printk("%s: ", pAC->dev->name);

	if (pAC->Hw.LinkStatus == STATUS_LINK_DOWN) {
		printk("NIC Link is down");
	}
	else if (pAC->Hw.LinkStatus == STATUS_LINK_WRONG_SET) {
		printk("NIC Link wrong set");
	}
	else if (pAC->Hw.LinkStatus == STATUS_LINK_UP) {
		printk("Network connection up using\n");
		printk("      speed        = %u Mbps\n",
			power(10, pAC->Hw.LineSpeed));
		printk("      duplex mode  = %s\n",
			(pAC->Hw.DuplexMode == DUPLEX_MODE_FULL) ? "full" : "half");
		printk("      card         = %s\n",
			pAC->Hw.IsCopper ? "copper" : "fiber");
		
		printk("      flowctrl     = ");
		if (pAC->Config.FlowCtrl == 4)
			printk("remote send\n");
		else if (pAC->Config.FlowCtrl == 3)
			printk("symmetric\n");
		else if (pAC->Config.FlowCtrl == 2)
			printk("local send\n");
		else
			printk("none\n");

		printk("      autoneg      = ");
		if (pAC->Config.AutoNeg == 0)
			printk("no\n");
		else if (pAC->Config.AutoNeg == 1)
			printk("yes\n");
		else if (pAC->Config.AutoNeg == 2)
			printk("sense\n");
		 
	}
	return;
}

/*****************************************************************************
 *
 * 	GetConfiguration - read configuration information
 *
 * Description:
 *	This function stores configuration parameters in pAC and ensures that
 *	each parameters has a valid value.
 *	Inter-parameter dependencies (e.g. between AutoNeg and FlowCtrl are
 *	checked in the init function of the affected module (in this case
 *	HWAC, which checks it at the end of the SK_INIT_IO phase, as it
 *	knows the adapter type then).
 *
 * Returns: N/A
 *
 */
static void GetConfiguration(
SK_AC	*pAC)	/* Pointer to the adapter context structure */
{
	SK_U32	Val32;

	/* Read Configuration. */

#ifdef DEBUG
	pAC->DbgComp = DebugComponents[pAC->Index];
	printk("%s: DebugComponents=%08x\n",
		pAC->dev->name, DebugComponents[pAC->Index]);
	pAC->DbgCat = DebugCategories[pAC->Index];
	printk("%s: DebugCategories=%08x\n",
		pAC->dev->name, DebugCategories[pAC->Index]);
#endif	/* DEBUG */

//	pAC->Config.TxRingNum = SK_NUM_TX_RINGS;
	pAC->Config.TxRingNum = 1;

//	pAC->Config.RxRingNum = SK_NUM_RX_RINGS;
	pAC->Config.RxRingNum = 1;

	pAC->Config.TaskOffloadCap = TASK_OFFLOAD_CAP_RX;
	if (TaskOffloadCap != NULL && TaskOffloadCap[pAC->Index] != NULL &&
		strlen(TaskOffloadCap[pAC->Index]) != 0) {

		if (strcmp(TaskOffloadCap[pAC->Index], "Both") == 0) {
			printk("%s: Linux does not support transmit checksum offloading, "
				"using RX checksumming only.\n",
				pAC->dev->name);
			pAC->Config.TaskOffloadCap = TASK_OFFLOAD_CAP_RX;
		}
		else if (strcmp(TaskOffloadCap[pAC->Index], "RX") == 0) {
			pAC->Config.TaskOffloadCap = TASK_OFFLOAD_CAP_RX;
		}
		else if (strcmp(TaskOffloadCap[pAC->Index], "TX") == 0) {
			printk("%s: Linux does not support transmit checksum offloading.\n",
				pAC->dev->name);
			pAC->Config.TaskOffloadCap = TASK_OFFLOAD_CAP_NONE;
		}
		else if (strcmp(TaskOffloadCap[pAC->Index], "None") == 0) {
			pAC->Config.TaskOffloadCap = TASK_OFFLOAD_CAP_NONE;
		}
		else {
			printk("%s: Illegal value for TaskOffloadCap.\n", pAC->dev->name);
		}
	}

	pAC->Config.RxCopySize = RX_COPY_SIZE_DEF;
	if (RxCopySize != NULL && RxCopySize[pAC->Index] != NULL &&
		strlen(RxCopySize[pAC->Index]) != 0) {

		if (StrToU32(RxCopySize[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= RX_COPY_SIZE_MAX) {

			pAC->Config.RxCopySize = Val32;
		}
		else {
			printk("%s: Illegal value for RxCopySize, using default.\n",
				pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("RxCopySize=%u\n", pAC->Config.RxCopySize));

	pAC->Config.MbufDescLow = MBUF_DESC_LOW_DEF;
	pAC->Config.MbufRxLow = MBUF_RX_LOW_DEF;
	pAC->Config.MbufHigh = MBUF_HIGH_DEF;

	pAC->Config.AutoNeg = SK_AUTONEG_SENS;	/* Default: do AutoSense. */
	if (AutoNeg != NULL && AutoNeg[pAC->Index] != NULL &&
		strlen(AutoNeg[pAC->Index]) != 0) {

		pAC->Config.AutoNegSet = SK_TRUE;
		if (strcmp(AutoNeg[pAC->Index], "On") == 0) {
			pAC->Config.AutoNeg = SK_AUTONEG_ON;
		}
		else if (strcmp(AutoNeg[pAC->Index], "Off") == 0) {
			pAC->Config.AutoNeg = SK_AUTONEG_OFF;
		}
		else if (strcmp(AutoNeg[pAC->Index], "Sense") == 0) {
			pAC->Config.AutoNeg = SK_AUTONEG_SENS;
		}
		else {
			printk("%s: Illegal value for AutoNeg.\n", pAC->dev->name);
			pAC->Config.AutoNegSet = SK_FALSE;
		}
	}	/* AutoNeg */

	pAC->Config.DupCap = SK_DUPCAP_BOTH;	/* Default: both. */
	if (DupCap != NULL && DupCap[pAC->Index] != NULL &&
		strlen(DupCap[pAC->Index]) != 0) {

		pAC->Config.DupCapSet = SK_TRUE;
		if (strcmp(DupCap[pAC->Index], "Both") == 0) {
			pAC->Config.DupCap = SK_DUPCAP_BOTH;
		}
		else if (strcmp(DupCap[pAC->Index], "Full") == 0) {
			pAC->Config.DupCap = SK_DUPCAP_FULL;
		}
		else if (strcmp(DupCap[pAC->Index], "Half") == 0) {
			pAC->Config.DupCap = SK_DUPCAP_HALF;
		}
		else {
			printk("%s: Illegal value for DupCap.\n", pAC->dev->name);
			pAC->Config.DupCapSet = SK_FALSE;
		}
	}	/* DupCap */

	pAC->Config.LinkSpeed = SK_LSPEED_AUTO;		/* Default: do auto select. */
	if (Speed != NULL && Speed[pAC->Index] != NULL &&
		strlen(Speed[pAC->Index]) != 0) {

		pAC->Config.LinkSpeedSet = SK_TRUE;
		if (strcmp(Speed[pAC->Index], "Auto") == 0) {
			pAC->Config.LinkSpeed = SK_LSPEED_AUTO;
		}
		else if (strcmp(Speed[pAC->Index], "1000") == 0) {
			pAC->Config.LinkSpeed = SK_LSPEED_1000MBPS;
		}
		else if (strcmp(Speed[pAC->Index], "100") == 0) {
			pAC->Config.LinkSpeed = SK_LSPEED_100MBPS;
		}
		else if (strcmp(Speed[pAC->Index], "10") == 0) {
			pAC->Config.LinkSpeed = SK_LSPEED_10MBPS;
		}
		else {
			printk("%s: Illegal value for Speed.\n", pAC->dev->name);
			pAC->Config.LinkSpeedSet = SK_FALSE;
		}
	}	/* Speed */

	pAC->Config.FlowCtrl = SK_FLOW_MODE_SYM_OR_REM;	/* Default: SymOrRem. */

	if (FlowCtrl != NULL && FlowCtrl[pAC->Index] != NULL &&
		strlen(FlowCtrl[pAC->Index]) != 0) {

		pAC->Config.FlowCtrlSet = SK_TRUE;
		if (strcmp(FlowCtrl[pAC->Index], "SymOrRem") == 0) {
			pAC->Config.FlowCtrl = SK_FLOW_MODE_SYM_OR_REM;
		}
		else if (strcmp(FlowCtrl[pAC->Index], "Sym") == 0) {
			pAC->Config.FlowCtrl = SK_FLOW_MODE_SYMMETRIC;
		}
		else if (strcmp(FlowCtrl[pAC->Index], "LocSend") == 0) {
			pAC->Config.FlowCtrl = SK_FLOW_MODE_LOC_SEND;
		}
		else if (strcmp(FlowCtrl[pAC->Index], "None") == 0) {
			pAC->Config.FlowCtrl = SK_FLOW_MODE_NONE;
		}
		else {
			pAC->Config.FlowCtrlSet = SK_FALSE;
			printk("%s: Illegal value for FlowCtrl.\n", pAC->dev->name);
		}
	}	/* FlowCtrl */

	pAC->Config.MsMode = SK_MS_MODE_AUTO; /* Default: do auto select. */
	if (Role != NULL && Role[pAC->Index] != NULL &&
		strlen(Role[pAC->Index]) != 0) {

		pAC->Config.MsModeSet = SK_TRUE;
		if (strcmp(Role[pAC->Index], "Auto") == 0) {
			pAC->Config.MsMode = SK_MS_MODE_AUTO;
		}
		else if (strcmp(Role[pAC->Index], "Master") == 0) {
			pAC->Config.MsMode = SK_MS_MODE_MASTER;
		}
		else if (strcmp(Role[pAC->Index], "Slave") == 0) {
			pAC->Config.MsMode = SK_MS_MODE_SLAVE;
		}
		else {
			printk("%s: Illegal value for Role.\n", pAC->dev->name);
			pAC->Config.MsModeSet = SK_FALSE;
		}
	}

	pAC->Config.RxCoalTicks = RX_COAL_TICKS_DEF;
	if (RxCoalescingTicks != NULL &&
		RxCoalescingTicks[pAC->Index] != NULL &&
		strlen(RxCoalescingTicks[pAC->Index]) != 0) {

		if (StrToU32(RxCoalescingTicks[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= RX_COAL_TICKS_MAX) {

			pAC->Config.RxCoalTicks = Val32;
		}
		else {
			printk("%s: Illegal value for RxCoalescingTicks, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("RxCoalescingTicks=%u\n", pAC->Config.RxCoalTicks));

	pAC->Config.TxCoalTicks = TX_COAL_TICKS_DEF;
	if (TxCoalescingTicks != NULL &&
		TxCoalescingTicks[pAC->Index] != NULL &&
		strlen(TxCoalescingTicks[pAC->Index]) != 0) {

		if (StrToU32(TxCoalescingTicks[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= TX_COAL_TICKS_MAX) {

			pAC->Config.TxCoalTicks = Val32;
		}
		else {
			printk("%s: Illegal value for TxCoalescingTicks, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("TxCoalescingTicks=%u\n", pAC->Config.TxCoalTicks));

	pAC->Config.RxCoalFrames = RX_COAL_FRAMES_DEF;
	if (RxMaxCoalescedFrames != NULL &&
		RxMaxCoalescedFrames[pAC->Index] != NULL &&
		strlen(RxMaxCoalescedFrames[pAC->Index]) != 0) {

		if (StrToU32(RxMaxCoalescedFrames[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= RX_COAL_FRAMES_MAX) {

			pAC->Config.RxCoalFrames = Val32;
		}
		else {
			printk("%s: Illegal value for RxMaxCoalescedFrames, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("RxMaxCoalescedFrames=%u\n", pAC->Config.RxCoalFrames));

	pAC->Config.TxCoalFrames = TX_COAL_FRAMES_DEF;
	if (TxMaxCoalescedFrames != NULL &&
		TxMaxCoalescedFrames[pAC->Index] != NULL &&
		strlen(TxMaxCoalescedFrames[pAC->Index]) != 0) {

		if (StrToU32(TxMaxCoalescedFrames[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= TX_COAL_FRAMES_MAX) {

			pAC->Config.TxCoalFrames = Val32;
		}
		else {
			printk("%s: Illegal value for TxMaxCoalescedFrames, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("TxMaxCoalescedFrames=%u\n", pAC->Config.TxCoalFrames));

	pAC->Config.RxCoalTicksInt = RX_COAL_TICKS_INT_DEF;
	if (RxCoalescingTicksDuringInt != NULL &&
		RxCoalescingTicksDuringInt[pAC->Index] != NULL &&
		strlen(RxCoalescingTicksDuringInt[pAC->Index]) != 0) {

		if (StrToU32(RxCoalescingTicksDuringInt[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= RX_COAL_TICKS_INT_MAX) {

			pAC->Config.RxCoalTicksInt = Val32;
		}
		else {
			printk("%s: Illegal value for RxCoalescingTicksDuringInt, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("RxCoalescingTicksDuringInt=%u\n", pAC->Config.RxCoalTicksInt));

	pAC->Config.TxCoalTicksInt = TX_COAL_TICKS_INT_DEF;
	if (TxCoalescingTicksDuringInt != NULL &&
		TxCoalescingTicksDuringInt[pAC->Index] != NULL &&
		strlen(TxCoalescingTicksDuringInt[pAC->Index]) != 0) {

		if (StrToU32(TxCoalescingTicksDuringInt[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= TX_COAL_TICKS_INT_MAX) {

			pAC->Config.TxCoalTicksInt = Val32;
		}
		else {
			printk("%s: Illegal value for TxCoalescingTicksDuringInt, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("TxCoalescingTicksDuringInt=%u\n", pAC->Config.TxCoalTicksInt));

	pAC->Config.RxCoalFramesInt = RX_COAL_FRAMES_INT_DEF;
	if (RxMaxCoalescedFramesDuringInt != NULL &&
		RxMaxCoalescedFramesDuringInt[pAC->Index] != NULL &&
		strlen(RxMaxCoalescedFramesDuringInt[pAC->Index]) != 0) {

		if (StrToU32(RxMaxCoalescedFramesDuringInt[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= RX_COAL_TICKS_INT_MAX) {

			pAC->Config.RxCoalFramesInt = Val32;
		}
		else {
			printk("%s: Illegal value for RxMaxCoalescedFramesDuringInt, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("RxMaxCoalescedFramesDuringInt=%u\n", pAC->Config.RxCoalFramesInt));

	pAC->Config.TxCoalFramesInt = TX_COAL_FRAMES_INT_DEF;
	if (TxMaxCoalescedFramesDuringInt != NULL &&
		TxMaxCoalescedFramesDuringInt[pAC->Index] != NULL &&
		strlen(TxMaxCoalescedFramesDuringInt[pAC->Index]) != 0) {

		if (StrToU32(TxMaxCoalescedFramesDuringInt[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= TX_COAL_FRAMES_INT_MAX) {

			pAC->Config.TxCoalFramesInt = Val32;
		}
		else {
			printk("%s: Illegal value for TxMaxCoalescedFramesDuringInt, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("TxMaxCoalescedFramesDuringInt=%u\n", pAC->Config.TxCoalFramesInt));

	pAC->Config.StatsCoalTicks = STATS_COAL_TICKS_DEF;
	if (StatsCoalescingTicks != NULL &&
		StatsCoalescingTicks[pAC->Index] != NULL &&
		strlen(StatsCoalescingTicks[pAC->Index]) != 0) {

		if (StrToU32(StatsCoalescingTicks[pAC->Index], &Val32) &&
			Val32 >= 0 && Val32 <= STATS_COAL_TICKS_MAX) {

			pAC->Config.StatsCoalTicks = Val32;
		}
		else {
			printk("%s: Illegal value for StatsCoalescingTicks, "
				"using default.\n", pAC->dev->name);
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("StatsCoalescingTicks=%u\n", pAC->Config.StatsCoalTicks));
} /* GetConfiguration */


/*****************************************************************************
 *
 * 	GetProductStr - return a adapter identification string from vpd
 *
 * Description:
 *	This function reads the product name string from the vpd area
 *	and puts it into the field pAC->DeviceString.
 *
 * Returns: N/A
 *
 */
static void GetProductStr(
SK_AC	*pAC)	/* Pointer to adapter context */
{
//RARA
#if 0
	int			StrLen;					/* Length of pAC->DeviceString. */
	char		Keyword[] = VPD_NAME;	/* VPD productname identifier. */
	int			ReturnCode;				/* Return code from VpdRead(). */
	unsigned	Flags;

	StrLen = 80;
	
	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	ReturnCode = VpdRead(pAC, pAC->IoBase, Keyword, pAC->DeviceStr, &StrLen);
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);
	if (ReturnCode != 0) {
		/* There was an error reading the VPD data. */
		SK_DBG_MSG(NULL, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ERROR,
			("Error reading VPD data: %d\n", ReturnCode));
		pAC->DeviceStr[0] = '\0';
	}
#else	/* !0 */
	sprintf(pAC->DeviceStr, "SysKonnect SK-9D%02x Gigabit Ethernet",
		 (SK_U8)(pAC->Hw.SubsystemVendorId >> 16) & 0xFF);
#endif	/* !0 */
} /* GetProductStr */


/*****************************************************************************
 *
 * 	FreeResources - release all resources allocated by the driver
 *
 * Description:
 *	This function releases all resources allocated by the driver.
 *
 * Returns: N/A
 *	
 */
static void FreeResources(
NET_DEV *dev)
{
	SK_AC	*pAC;	/* Pointer to Adapter Context */
	int		i;

	netif_stop_queue(dev);
#ifdef SK_KERNEL_22
	remove_proc_entry(dev->name, pSkRootDir);
#endif /* SK_KERNEL_22 */

	if (dev->priv != NULL) {
		pAC = (SK_AC *)dev->priv;

		if (pAC->AllocFlags & SK_ALLOC_IRQ) {
			free_irq(dev->irq, dev);
		}

		if (pAC->IoBase) {
			iounmap(pAC->IoBase);
		}

		/* Free driver memory. */
		for (i = 0; i < pAC->Config.TxRingNum; i++) {
			SK_FREE_IF_ALLOCED(pAC->SndPrdRing[i].pDescriptors);
			SK_FREE_CONSISTENT_IF_ALLOCED(&pAC->PciDev,
				sizeof(TX_DESC) * pAC->SndPrdRing[i].NumDescs,
				pAC->SndPrdRing[i].pSbdRing, pAC->SndPrdRing[i].Phys);
		}

		for (i = 0; i < pAC->Config.RxRingNum; i++) {
			SK_FREE_CONSISTENT_IF_ALLOCED(&pAC->PciDev,
				sizeof(RX_DESC) * pAC->RecRetRing[i].NumDescs,
				pAC->RecRetRing[i].pRbdRing, pAC->RecRetRing[i].Phys);
		}
		SK_FREE_IF_ALLOCED(pAC->pRpds);

		SK_FREE_CONSISTENT_IF_ALLOCED(&pAC->PciDev,
			sizeof(RX_DESC) * pAC->RecJumPrdRing.NumDescs,
			pAC->RecJumPrdRing.pRbdRing, pAC->RecJumPrdRing.Phys);
		
		SK_FREE_CONSISTENT_IF_ALLOCED(&pAC->PciDev,
			sizeof(RX_DESC) * pAC->RecStdPrdRing.NumDescs,
			pAC->RecStdPrdRing.pRbdRing, pAC->RecStdPrdRing.Phys);
#ifdef SK_EXT_MEM
		SK_FREE_CONSISTENT_IF_ALLOCED(&pAC->PciDev,
			sizeof(RX_DESC) * pAC->RecMinPrdRing.NumDescs,
			pAC->RecMinPrdRing.pRbdRing, pAC->RecMinPrdRing.Phys);
#endif	/* SK_EXT_MEM */
		SK_FREE_CONSISTENT_IF_ALLOCED(&pAC->PciDev,
			sizeof(PNMI_STAT_BLOCK),
			pAC->pPnmiStatBlock, pAC->StatisticsPhys);
		SK_FREE_CONSISTENT_IF_ALLOCED(&pAC->PciDev,
			sizeof(STATUS_BLOCK),
			pAC->pStatusBlock, pAC->StatusBlockPhys);
	}

	/* Otherwise unregister_netdev() calls get_stats with invalid I/O ... */
	dev->get_stats = NULL;
	unregister_netdev(dev);

	/* This also frees pAC, because it is embedded in the dev struct. */
	kfree(dev);
}	/* FreeResources */


/*****************************************************************************
 *
 * 	AllocDriverMem - allocate and initialize memory for DMA areas
 *
 * Description:
 *	This function allocates and initializes memory that is later used for DMA.
 *
 * Returns:
 *	SK_TRUE if everything is ok
 *	SK_FALSE on error
 */
static SK_BOOL	AllocDriverMem(
SK_AC	*pAC)
{
	int		i;

	pAC->pStatusBlock = pci_alloc_consistent(&pAC->PciDev,
		sizeof(STATUS_BLOCK), &pAC->StatusBlockPhys);
	if (pAC->pStatusBlock == NULL) {
		printk(KERN_ERR "%s status block!\n", SK_MEM_ALLOC_ERR);
		return (SK_FALSE);
	}

	pAC->pPnmiStatBlock = pci_alloc_consistent(&pAC->PciDev,
		sizeof(PNMI_STAT_BLOCK), &pAC->StatisticsPhys);
	if (pAC->pPnmiStatBlock == NULL) {
		printk(KERN_ERR "%s statistics block!\n", SK_MEM_ALLOC_ERR);
		return (SK_FALSE);
	}

	pAC->NumRpds = 0;
#ifdef SK_EXT_MEM
	pAC->RecMinPrdRing.NumDescs = RX_MIN_PRD_RING_ENTRIES;
	pAC->RecMinPrdRing.FreeBds = pAC->RecMinPrdRing.NumDescs - 1;
	pAC->RecMinPrdRing.BufferSize = RX_MIN_PRD_RING_BUF_SIZE;
	pAC->RecMinPrdRing.Mailbox = MIN_RX_RING_PROD_IDX + 4;
	pAC->RecMinPrdRing.pStatusIdx = &pAC->pStatusBlock->RxMinConsumerIdx;
	pAC->NumRpds += pAC->RecMinPrdRing.NumDescs;
	pAC->RecMinPrdRing.pRbdRing = pci_alloc_consistent(&pAC->PciDev,
		sizeof(RX_DESC) * pAC->RecMinPrdRing.NumDescs, &pAC->RecMinPrdRing.Phys);
	if (pAC->RecMinPrdRing.pRbdRing == NULL) {
		printk(KERN_ERR "%s mini RX producer ring!\n", SK_MEM_ALLOC_ERR);
		return (SK_FALSE);
	}
#endif	/* SK_EXT_MEM */

	pAC->RecStdPrdRing.NumDescs = RX_STD_PRD_RING_ENTRIES;
	pAC->RecStdPrdRing.FreeBds = pAC->RecStdPrdRing.NumDescs - 1;
	pAC->RecStdPrdRing.BufferSize = RX_STD_PRD_RING_BUF_SIZE;
	pAC->RecStdPrdRing.Mailbox = STD_RX_RING_PROD_IDX + 4;
	pAC->RecStdPrdRing.pStatusIdx = &pAC->pStatusBlock->RxStdConsumerIdx;
	pAC->NumRpds += pAC->RecStdPrdRing.NumDescs;
	pAC->RecStdPrdRing.pRbdRing = pci_alloc_consistent(&pAC->PciDev,
		sizeof(RX_DESC) * pAC->RecStdPrdRing.NumDescs, &pAC->RecStdPrdRing.Phys);
	if (pAC->RecStdPrdRing.pRbdRing == NULL) {
		printk(KERN_ERR "%s standard RX producer ring!\n", SK_MEM_ALLOC_ERR);
		return (SK_FALSE);
	}

	pAC->RecJumPrdRing.NumDescs = RX_JMB_PRD_RING_ENTRIES;
	pAC->RecJumPrdRing.FreeBds = pAC->RecJumPrdRing.NumDescs - 1;
	pAC->RecJumPrdRing.BufferSize = pAC->dev->mtu + 36;
	pAC->RecJumPrdRing.Mailbox = JUM_RX_RING_PROD_IDX + 4;
	pAC->RecJumPrdRing.pStatusIdx = &pAC->pStatusBlock->RxJumConsumerIdx;
	pAC->NumRpds += pAC->RecJumPrdRing.NumDescs;
	pAC->RecJumPrdRing.pRbdRing = pci_alloc_consistent(&pAC->PciDev,
		sizeof(RX_DESC) * pAC->RecJumPrdRing.NumDescs, &pAC->RecJumPrdRing.Phys);
	if (pAC->RecJumPrdRing.pRbdRing == NULL) {
		printk(KERN_ERR "%s jumbo RX producer ring!\n", SK_MEM_ALLOC_ERR);
		return (SK_FALSE);
	}
	
	pAC->NumRpds *= 2;

	pAC->pRpds = kmalloc(sizeof(SK_RX_PACKET) * pAC->NumRpds, GFP_KERNEL);
	if (pAC->pRpds == NULL) {
		printk(KERN_ERR "%s RPDs!\n", SK_MEM_ALLOC_ERR);
		return (SK_FALSE);
	}
	SK_MEMSET(pAC->pRpds, 0, sizeof(SK_RX_PACKET) * pAC->NumRpds);
	for (i = 0; i < pAC->NumRpds; i++) {
		PushRpd(&pAC->RpdQ, &pAC->pRpds[i]);
	}

	for (i = 0; i < pAC->Config.RxRingNum; i++) {
		pAC->RecRetRing[i].NumDescs = RX_RETURN_RING_ENTRIES;
		pAC->RecRetRing[i].FreeBds = pAC->RecRetRing[i].NumDescs - 1;
		pAC->RecRetRing[i].Mailbox = RX_RET_CONSUM_IDX1 + 8 * i + 4;
		pAC->RecRetRing[i].pStatusIdx = &pAC->pStatusBlock->ConProdIdx[i].ProdRx;
		pAC->RecRetRing[i].pRbdRing = pci_alloc_consistent(&pAC->PciDev,
			sizeof(RX_DESC) * pAC->RecRetRing[i].NumDescs, &pAC->RecRetRing[i].Phys);
		if (pAC->RecRetRing[i].pRbdRing == NULL) {
			printk(KERN_ERR "%s return ring!\n", SK_MEM_ALLOC_ERR);
			return (SK_FALSE);
		}
	}

	for (i = 0; i < pAC->Config.TxRingNum; i++) {
		pAC->SndPrdRing[i].NumDescs = TX_RING_ENTRIES;
		pAC->SndPrdRing[i].FreeBds = pAC->SndPrdRing[i].NumDescs - 1;
		pAC->SndPrdRing[i].Mailbox = TX_HOST_PROD_IDX1 + 8 * i + 4;
		pAC->SndPrdRing[i].pSbdRing = pci_alloc_consistent(&pAC->PciDev,
			sizeof(TX_DESC) * pAC->SndPrdRing[i].NumDescs, &pAC->SndPrdRing[i].Phys);
		if (pAC->SndPrdRing[i].pSbdRing == NULL) {
			printk(KERN_ERR "%s send ring!\n", SK_MEM_ALLOC_ERR);
			return (SK_FALSE);
		}

		pAC->SndPrdRing[i].pDescriptors =
			kmalloc(sizeof(SK_TX_PACKET) * pAC->SndPrdRing[i].NumDescs, GFP_KERNEL);
		if (pAC->SndPrdRing[i].pDescriptors == NULL) {
			printk(KERN_ERR "%s TX descriptors!\n", SK_MEM_ALLOC_ERR);
			return (SK_FALSE);
		}
		SK_MEMSET(pAC->SndPrdRing[i].pDescriptors, 0,
			sizeof(SK_TX_PACKET) * pAC->SndPrdRing[i].NumDescs);
	}

	return (SK_TRUE);
}	/* AllocDriverMem */


/*****************************************************************************
 *
 * 	skgi_probe - find all SysKonnect SK-9Dxx Gigabit Ethernet adapters
 *
 * Description:
 *	This function scans the PCI bus for SysKonnect SK-9Dxx Gigabit Ethernet
 *	adapters. Resources for each adapter are allocated and the adapter
 *	is brought into INIT_IO state.
 *
 * Returns:
 *	== 0 if everything is ok
 *	!= 0 on error
 */
INITFUNC(int, skgi_probe, (NET_DEV *dev))
{
	struct proc_dir_entry	*pProcFile;
	struct pci_dev			*pdev;
#ifndef SK_KERNEL_22
	NET_DEV	*dev;
#endif /* !SK_KERNEL_22 */
	SK_AC	*pAC;			/* Pointer to Adapter Context */
	SK_IOC	IoC;			/* I/O Context */
	char	*DescrStr;
	char	*VerStr;
	uint	BoardsFound;
	int		Ret;
	int		i;
	ulong	BaseAddr;
	SK_U32	Data32;
	SK_U16	Data16;
	SK_BOOL	DeviceFound;

	if (Probed != 0) {
		return (-ENODEV);
	}
	Probed++;
	
	DescrStr = "sk9dlin: Driver for Linux"; /* This is given to PNMI. */
	VerStr	= VER_STRING;
	BoardsFound = 0;
	DeviceFound = SK_FALSE;
	pdev = NULL;

	if (!pci_present()) {		/* is PCI support present? */
		return (-ENODEV);
	}

#ifdef SYSKONNECT_ONLY
	while ((pdev = pci_find_device(PCI_VENDOR_ID_SYSKONNECT,
		PCI_DEVICE_ID_SYSKONNECT_SK9DXX, pdev)) != NULL) {
#else	/* SYSKONNECT_ONLY */
	while ((pdev = pci_find_class(PCI_CLASS_NETWORK_ETHERNET << 8, pdev))
		!= NULL) {

		if (!((pdev->vendor == PCI_VENDOR_ID_SYSKONNECT &&
			  (pdev->device == PCI_DEVICE_ID_SYSKONNECT_SK9DXX ||
			   pdev->device == PCI_DEVICE_ID_SYSKONNECT_SK9MXX)) ||
		      (pdev->vendor == PCI_VENDOR_ID_BROADCOM &&
			   (pdev->device == 0x1644 || pdev->device == 0x1645)))) {

			continue;
		}
#endif	/* SYSKONNECT_ONLY */

#ifdef SYSKONNECT_ONLY
		/* Check also PCI subsytem ID */
		pci_read_config_word(pdev, PCI_SUBSYSTEM_ID, &Data16);

//		printk("PCI_SUBSYSTEM_ID=0x%04X\n", Data16);

		if (Data16 != PCI_SUB_DEVICE_ID_SYSKONNECT_SK9D21 &&
			Data16 != PCI_SUB_DEVICE_ID_SYSKONNECT_SK9D41 &&
			Data16 != PCI_SUB_DEVICE_ID_SYSKONNECT_SK9D11) {

			continue;
		}
#endif	/* SYSKONNECT_ONLY */
			
#ifndef SK_KERNEL_22
		if (pci_enable_device(pdev) != 0) {
			continue;
		}
#endif /* !SK_KERNEL_22 */

		if (BoardsFound >= SK_MAX_CARD_PARAM) {
			printk(KERN_ERR "More than %u boards found in system.\n"
				"Please recompile driver with higher value "
				"of SK_MAX_CARD_PARAM!\n", SK_MAX_CARD_PARAM);
			break;
		}

		BaseAddr = pci_resource_start(pdev, 0);
		if ((BaseAddr & PCI_BASE_ADDRESS_MEM_MASK) == 0) {
			printk(KERN_ERR "VendorId=0x%04X, DeviceId=0x%04X --> Base Address is 0 !\n",
				pdev->vendor, pdev->device);
			break;
		}
		
		dev = NULL;
		dev = init_etherdev(dev, sizeof(SK_AC));

		if (dev == NULL) {
			printk(KERN_ERR "%s etherdev structure!\n", SK_MEM_ALLOC_ERR);
			continue;
		}

		/* Display driver info. */
		if (!DeviceFound) {
			/* Set flag to TRUE so this string is displayed only ONCE */
			DeviceFound = SK_TRUE;
			printk("%s\n", BootString);

			SK_MEMCPY(&SK_Root_Dir_entry, BootString,
				sizeof(SK_Root_Dir_entry) - 1);

#ifdef SK_KERNEL_22
	       	/* Create proc (directory) */
			pSkRootDir = &Our_Proc_Dir;
			pSkRootDir->namelen = sizeof(SK_Root_Dir_entry) - 1;
			proc_register(proc_net, pSkRootDir);
			
			SK_MEMCPY(&SkInodeOps, &proc_dir_inode_operations,
				sizeof(SkInodeOps));
			SkInodeOps.lookup = &SkGiProcLookup;
			SkInodeOps.default_file_ops = &SkFileOps;

			SK_MEMCPY(&SkFileOps, proc_net_inode_operations.default_file_ops,
				sizeof(SkFileOps));
			SkFileOps.open = &SkGiProcFileOpen;
			SkFileOps.release = &SkGiProcFileRelease;
#else /* !SK_KERNEL_22 */
			pSkRootDir = create_proc_entry(SK_Root_Dir_entry,
				S_IFDIR | S_IWUSR | S_IRUGO | S_IXUGO, proc_net);

			pSkRootDir->owner = THIS_MODULE;
#endif /* !SK_KERNEL_22 */
		}
		pAC = dev->priv;
		SK_MEMSET(pAC, 0, sizeof(SK_AC));

		/* ****** SK_INIT_DATA phase ****** */
		pAC->Index = BoardsFound;
		
		pAC->PciDev = *pdev;
		pAC->PciDevId = pdev->device;
		pAC->dev = dev;
		pAC->pNext = pACList;
		pACList = pAC;
		sprintf(pAC->Name, "SysKonnect SK-9Dxx");

		dev->irq = pdev->irq;

		dev->open =					&SkGiOpen;
		dev->stop =					&SkGiClose;
		dev->hard_start_xmit =		&SkGiXmit;
		dev->get_stats =			&SkGiStats;
		dev->set_multicast_list = 	&SkGiSetRxMode;
		dev->set_mac_address =		&SkGiSetMacAddr;
		dev->do_ioctl =				&SkGiIoctl;
		dev->change_mtu =			&SkGiChangeMtu;
		dev->flags &= 				~IFF_RUNNING;

		pProcFile = create_proc_entry(dev->name,
			S_IFREG | S_IXUSR | S_IWGRP | S_IROTH,
			pSkRootDir);

#ifdef SK_KERNEL_22
		pProcFile->ops = &SkInodeOps;
#endif /* !SK_KERNEL_22 */
		
		pProcFile->read_proc = proc_read;
		pProcFile->write_proc = NULL;
		pProcFile->nlink = 1;
		pProcFile->size = sizeof(dev->name + 1);
		pProcFile->data = (void *)pProcFile;

		/* Dummy value. */
		dev->base_addr = 42;

		GetConfiguration(pAC);

		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("mod_probe: pAC =0x%08lX\n", (ulong)pAC));

		IoC = NULL;
		(void)SkGiHwacInit(pAC, IoC, SK_INIT_DATA);
		SkEventInit(pAC, IoC, SK_INIT_DATA);
		SkPnmiInit( pAC, IoC, SK_INIT_DATA);
		SkAddrInit(pAC, IoC, SK_INIT_DATA);
		SkRlmtInit(pAC, IoC, SK_INIT_DATA);
		SkTimerInit(pAC, IoC, SK_INIT_DATA);

		pci_set_master(pdev);

#ifdef PCI_CONF_16BIT
		SK_PCI_IN16(pAC, PCI_COMMAND, &Data16);
		if ((Data16 & (BUS_MASTER | MEM_SPACE)) !=
			(SK_U16)(BUS_MASTER | MEM_SPACE)) {

			SK_PCI_OUT16(pAC, PCI_COMMAND, Data16 | BUS_MASTER | MEM_SPACE);
		}
#else  /* PCI_CONF_16BIT */
		SK_PCI_IN32(pAC, PCI_COMMAND, &Data32);
		if ((Data32 & (BUS_MASTER | MEM_SPACE)) != (BUS_MASTER | MEM_SPACE)) {
			SK_PCI_OUT32(pAC, PCI_COMMAND, Data32 | BUS_MASTER | MEM_SPACE);
		}
#endif /* PCI_CONF_16BIT */

		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("mod_probe: base=0x%08lX\n", BaseAddr));

		BaseAddr &= 0xffffc000L;

		/*
		 * Remap the regs into kernel space.
		 */

		IoC = pAC->IoBase = (char *)ioremap_nocache(BaseAddr, 0x10000);
		if (IoC == NULL) {
			printk(KERN_ERR "%s: Unable to map I/O register, SysKonnect "
			       "SK-9Dxx #%u at base address 0x%08lx will be disabled.\n",
			       dev->name, BoardsFound, BaseAddr);
			pACList = pAC->pNext;
			FreeResources(dev);
			continue;
		}
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("mod_probe: IoC =0x%08lX\n", (ulong)IoC));

		SK_PCI_IN32(pAC, PCI_VENDOR_ID, &Data32);
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("Found '%s' VendorId=0x%04X DeviceId=0x%04X\n",
			 pAC->Name, (SK_U16)Data32, (SK_U16)(Data32 >> 16)));

		/*
		 * Write the Driver Version and Description for OID queries
		 */
		SK_PNMI_SET_DRIVER_DESCR(pAC, DescrStr);
		SK_PNMI_SET_DRIVER_VER(pAC, VerStr);

		if (!AllocDriverMem(pAC)) {
			printk(KERN_ERR "%s DMA!\n", SK_MEM_ALLOC_ERR);
			pACList = pAC->pNext;
			FreeResources(dev);
			continue;
		}

		Ret = request_irq(dev->irq, SkGiIsr, SA_SHIRQ, dev->name, dev);
		if (Ret != 0) {
			printk(KERN_WARNING "%s: Requested IRQ %d is busy!\n",
				dev->name, dev->irq);
			pACList = pAC->pNext;
			FreeResources(dev);
			continue;
		}
		pAC->AllocFlags |= SK_ALLOC_IRQ;

		/* Initialize the mutexes */
		spin_lock_init(&pAC->SlowPathLock);
		for (i = 0; i < pAC->Config.TxRingNum; i++) {
			spin_lock_init(&pAC->SndPrdRing[0].RingLock);
		}

		/* ****** SK_INIT_IO phase ****** */
		(void)SkGiHwacInit(pAC, IoC, SK_INIT_IO);
		SkEventInit(pAC, IoC, SK_INIT_IO);
		SkPnmiInit( pAC, IoC, SK_INIT_IO);
		SkAddrInit(pAC, IoC, SK_INIT_IO);
		SkRlmtInit(pAC, IoC, SK_INIT_IO);
		SkTimerInit(pAC, IoC, SK_INIT_IO);

		SK_MEMCPY((caddr_t)&dev->dev_addr,
			(caddr_t)&pAC->Addr.Net.CurrentMacAddress.a[0], 6);
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("Mac Addr: %02x:%02x:%02x:%02x:%02x:%02x\n",
			dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2],
			dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5]));

		GetProductStr(pAC);
		printk("%s: %s\n", dev->name, pAC->DeviceStr);
		
		BoardsFound++;

		/* ****** SK_INIT_RUN phase ****** */

		/* Executed in SkGiOpen(). */

#ifndef MODULE
		/*
		 * This is bollocks, but we need to tell the net-init
		 * code that it shall go for the next device.
		 */
		dev->base_addr = 0;
#endif	/* MODULE */
	}	/* while */

#ifndef MODULE
	/*
	 * If we're at this point we're going through skgi_probe() for
	 * the first time. Return success (0) if we've initialized one
	 * or more boards. Otherwise, return failure (-ENODEV).
	 */
	if (BoardsFound > 0) {
		return (0);
	}
	else {
		return (-ENODEV);
	}
#else	/* MODULE */
	return (BoardsFound);
#endif	/* MODULE */
} /* skgi_probe */

#ifdef MODULE

#ifndef SK_KERNEL_22
static struct pci_device_id skgi_pci_tbl[] __initdata = {
	{PCI_VENDOR_ID_SYSKONNECT, PCI_DEVICE_ID_SYSKONNECT_SK9DXX,
	 PCI_ANY_ID, PCI_ANY_ID,},
	{ }			/* Terminating entry */
};

MODULE_DEVICE_TABLE(pci, skgi_pci_tbl);
#endif /* !SK_KERNEL_22 */

MODULE_AUTHOR("Ralf Assmann <linux@syskonnect.de>");
MODULE_DESCRIPTION("SysKonnect SK-9Dxx Gigabit Ethernet driver");
MODULE_LICENSE("GPL");

MODULE_PARM(AutoNeg,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(DupCap,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(FlowCtrl,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(Role,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(Speed,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(RxCopySize,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(TaskOffloadCap,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(RxCoalescingTicks,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(TxCoalescingTicks,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(RxMaxCoalescedFrames,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(TxMaxCoalescedFrames,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(RxCoalescingTicksDuringInt,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(TxCoalescingTicksDuringInt,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(RxMaxCoalescedFramesDuringInt,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(TxMaxCoalescedFramesDuringInt,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");
MODULE_PARM(StatsCoalescingTicks,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "s");

#ifdef DEBUG
MODULE_PARM(DebugComponents,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "i");
MODULE_PARM(DebugCategories,
	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "i");
#endif	/* DEBUG */

/* Not used, just there because every driver should have them. */
MODULE_PARM(options,	"1-" __MODULE_STRING(SK_MAX_CARD_PARAM) "i");
MODULE_PARM(debug,		"i");

/*****************************************************************************
 *
 * 	(skgi_)init_module - module initialization function
 *
 * Description:
 *	Very simple, only call skgi_probe and return approriate result.
 *
 * Returns:
 *	== 0 if everything is ok
 *	!= 0 on error
 */
INITMOD(int, skgi_, init_module, (void))
{
	int Cards;

	pACList = NULL;

	/* Just to avoid warnings ... */
	options[0] = 0;
	debug = 0;
	
#ifdef SK_KERNEL_22
	Cards = skgi_probe(NULL);
#else  /* !SK_KERNEL_22 */
	Cards = skgi_probe();
#endif /* !SK_KERNEL_22 */

#ifdef DEBUG
	if (Cards == 0) {
		printk("No adapter found.\n");
	}
#endif /* DEBUG */

	return ((Cards > 0) ? 0 : -ENODEV);
} /* (skgi_)init_module */


/*****************************************************************************
 *
 * 	(skgi_)cleanup_module - module unload function
 *
 * Description:
 *	Disable adapter if it is still running, free resources,
 *	free device struct.
 *
 * Returns: N/A
 */
EXITMOD(void, skgi_, cleanup_module, (void))
{
	NET_DEV *dev;

	while (pACList != NULL) {
		dev = pACList->dev;
		pACList = pACList->pNext;
		FreeResources(dev);
	}

#ifdef SK_KERNEL_22
	/* clear proc-dir */
	proc_unregister(proc_net, pSkRootDir->low_ino);
#else  /* !SK_KERNEL_22 */
	remove_proc_entry(pSkRootDir->name, proc_net);
#endif /* !SK_KERNEL_22 */

} /* (skgi_)cleanup_module */

#ifndef SK_KERNEL_22
module_init(skgi_init_module);
module_exit(skgi_cleanup_module);
#endif /* !SK_KERNEL_22 */

#endif /* MODULE */

/****************************************************************************
 *
 *	SkGiOpen - handle start of initialized adapter
 *
 * Description:
 *	This function starts the initialized adapter.
 *	The board level variable is set and the adapter is
 *	brought to full functionality.
 *	The device flags are set for operation.
 *	Do all necessary level 2 initialization, enable interrupts and
 *	give start command to RLMT.
 *
 * Returns:
 *	== 0 on success
 *	!= 0 on error
 */
static int SkGiOpen(
NET_DEV *dev)
{
	SK_AC		*pAC;		/* Pointer to Adapter Context */
	SK_IOC		IoC;		/* I/O Context */
	SK_EVPARA	Para;
	unsigned	Flags;

	pAC = (SK_AC *)dev->priv;
	IoC = pAC->IoBase;

	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiOpen: pAC=0x%lX.\n", (ulong)pAC));

	if (dev->mtu > SK_NORM_MTU) {
		pAC->Config.UseJumbo = SK_TRUE;
		pAC->RecJumPrdRing.BufferSize = dev->mtu + 36;
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("Use Jumbo.\n"));
	}
	else {
		pAC->Config.UseJumbo = SK_FALSE;
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("Don't use Jumbo.\n"));
	}

	if (SkGiHwacDeinit(pAC, IoC) != SK_OK) {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("SkGiOpen: SkGiHwacDeinit failed !\n"));
		return (-1);
	}

	if (SkGiHwacInit(pAC, IoC, SK_INIT_RUN) != SK_OK) {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
			("SkGiOpen: SkGiHwacInit failed !\n"));
		return (-1);
	}
	SkEventInit(pAC, IoC, SK_INIT_RUN);
	SkPnmiInit( pAC, IoC, SK_INIT_RUN);
	SkAddrInit(pAC, IoC, SK_INIT_RUN);
	SkRlmtInit(pAC, IoC, SK_INIT_RUN);
	SkTimerInit(pAC, IoC, SK_INIT_RUN);

//	SkAddrMcUpdate(pAC, IoC);		/* RS */

	ShowDriverStatus(pAC);
	
	pAC->Timer.expires = RUN_AT(1);	/* 1/100 sec in x86 Linux. */
	add_timer(&pAC->Timer);


#ifdef SK_EXT_MEM
	FillRecPrdRing(pAC, IoC, &pAC->RecMinPrdRing);
#endif	/* SK_EXT_MEM */
	FillRecPrdRing(pAC, IoC, &pAC->RecStdPrdRing);
	
	if (pAC->Config.UseJumbo) {
		FillRecPrdRing(pAC, IoC, &pAC->RecJumPrdRing);
	}

	Para.Para32[0] = 0;		/* NetNr */
	Para.Para32[1] = -1;
	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	SkEventQueue(pAC, SKGI_RLMT, SK_RLMT_START, Para);
	SkEventDispatcher(pAC, IoC);
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);

	/* Enable interrupts. */
	SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost & ~MASK_PCI_INT);
	// SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 0);
	// SK_OUT32(IoC, INTERRUPT_MAILBOX0, 0);
	// SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);
	// SK_OUT32(IoC, HOST_COAL_MODE, COALESC_NOW);
	// pAC->pStatusBlock->Status |= STATUS_WORD_UPDATED;
	// SK_OUT32(IoC, MISC_LOCAL, pAC->Hw.MiscLocal | SET_INT);

	netif_start_queue(pAC->dev);
#ifdef SK_KERNEL_22
	dev->interrupt = 0;
	dev->start = 1;
#endif /* SK_KERNEL_22 */

	MOD_INC_USE_COUNT;

	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiOpen suceeded.\n"));
	return (0);
} /* SkGiOpen */


/****************************************************************************
 *
 *	SkGiClose - Stop initialized adapter
 *
 * Description:
 *	Close initialized adapter.
 *
 * Returns:
 *	0 on success
 *	error code on error
 */
static int SkGiClose(
NET_DEV	*dev)
{
	SK_EVPARA	Para;
	SK_AC		*pAC;	/* Pointer to Adapter Context */
	SK_IOC		IoC;	/* I/O Context */
	unsigned	Flags;	/* for spin lock */
	int			Ret;
	int			i;


#ifdef SK_KERNEL_22
	(dev)->start = 0;
#endif /* SK_KERNEL_22 */
	netif_stop_queue(dev);
	
	pAC = (SK_AC *)dev->priv;
	IoC = pAC->IoBase;
	
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiClose: pAC = 0x%lX.\n", (ulong)pAC));

#if 0
	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	for (i = 0; i < pAC->Config.TxRingNum; i++) {
		spin_lock_irqsave(&pAC->SndPrdRing[i].RingLock, Flags);
	}

	/* Disable interrupts. */
	SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost);
	SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);

	del_timer(&pAC->Timer);
	Ret = SkGiHwacDeinit(pAC, IoC);

	FreeSndRings(pAC);
	FreeRecRings(pAC);

	for (i = 0; i < pAC->Config.TxRingNum; i++) {
		spin_unlock_irqrestore(&pAC->SndPrdRing[i].RingLock, Flags);
	}
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);
#else	/* !0 */
	Para.Para32[0] = 0;		/* NetNr */
	Para.Para32[1] = -1;
	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	
	/* Disable interrupts. */
	SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost);
	SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);

	del_timer(&pAC->Timer);

	SkEventQueue(pAC, SKGI_RLMT, SK_RLMT_STOP, Para);
	SkEventDispatcher(pAC, IoC);

	for (i = 0; i < pAC->Config.TxRingNum; i++) {
		spin_lock_irqsave(&pAC->SndPrdRing[i].RingLock, Flags);
	}
	FreeSndRings(pAC);
	FreeRecRings(pAC);
	for (i = 0; i < pAC->Config.TxRingNum; i++) {
		spin_unlock_irqrestore(&pAC->SndPrdRing[i].RingLock, Flags);
	}
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);
	Ret = SK_OK;
#endif	/* !0 */
	
	MOD_DEC_USE_COUNT;
	
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiClose: %s\n", (Ret != SK_OK) ? "failed" : "done"));

	return (0);
} /* SkGiClose */


/*****************************************************************************
 *
 * 	FreeSbds - release descriptors from the descriptor ring
 *
 * Description:
 *	This function releases descriptors from a send ring whose messages
 *	have been transferred to the adapter.
 *	If a descriptor is sent, it can be freed and the message can
 *	be freed, too.
 *	The tx descriptor ring lock must be held while calling this function!!!
 *
 * Returns: N/A
 */
static void FreeSbds(
SK_AC	*pAC,		/* pointer to the adapter context */
int		RingIdx)	/* pointer to destination port structure */
{
	struct sk_buff	*pMsg;
	P_SND_RING		pSndPrdRing;
	volatile SK_U16	*pTxConsumerIdx;

	pSndPrdRing = &pAC->SndPrdRing[RingIdx];
	pTxConsumerIdx = &pAC->pStatusBlock->ConProdIdx[RingIdx].ConTx;

	//SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_TX_PROGRESS,
	//	("FreeSbds: %d, %d.\n",
	//	pSndPrdRing->NextUsedBd, *pTxConsumerIdx));

	while (pSndPrdRing->NextUsedBd != *pTxConsumerIdx) {
		pMsg = pSndPrdRing->pDescriptors[pSndPrdRing->NextUsedBd].pMsg;
		//SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_TX_PROGRESS,
		//	("Freeing buffer %x of descriptor #%d.\n",
		//	(unsigned)pMsg, pSndPrdRing->NextUsedBd));
		pci_unmap_single(&pAC->PciDev,
			pSndPrdRing->pDescriptors[pSndPrdRing->NextUsedBd].Phys,
			pSndPrdRing->pSbdRing[pSndPrdRing->NextUsedBd].Length,
			PCI_DMA_TODEVICE);
		pSndPrdRing->NextUsedBd++;
		if (pSndPrdRing->NextUsedBd >= pSndPrdRing->NumDescs) {
			pSndPrdRing->NextUsedBd %= pSndPrdRing->NumDescs;
		}
		pSndPrdRing->FreeBds++;
		netif_start_queue(pAC->dev);
		DEV_KFREE_SKB_ANY(pMsg);
	}	/* while */
	
	return;
}	/* FreeSbds */


/*****************************************************************************
 *
 * 	FreeSndRings - free all send rings
 *
 * Description:
 *	This function frees buffer descriptors and messages in all send rings.
 *	This function is either called during an IRQ or with the IRQs disabled.
 *
 * Returns:	N/A
 */
static void	FreeSndRings(
SK_AC	*pAC)
{
	struct sk_buff	*pMsg;
	P_SND_RING		pSndPrdRing;
	int				i;

	for (i = 0; i < pAC->Config.TxRingNum; i++) {
		pSndPrdRing = &pAC->SndPrdRing[i];
		while (pSndPrdRing->NextUsedBd != pSndPrdRing->NextFreeBd) {
			pMsg = pSndPrdRing->pDescriptors[pSndPrdRing->NextUsedBd].pMsg;
			//SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_TX_PROGRESS,
			//	("Freeing buffer %x of descriptor #%d.\n",
			//	(unsigned)pMsg, pSndPrdRing->NextUsedBd));
			pci_unmap_single(&pAC->PciDev,
				pSndPrdRing->pDescriptors[pSndPrdRing->NextUsedBd].Phys,
				pSndPrdRing->pSbdRing[pSndPrdRing->NextUsedBd].Length,
				PCI_DMA_TODEVICE);
			pSndPrdRing->NextUsedBd++;
			if (pSndPrdRing->NextUsedBd >= pSndPrdRing->NumDescs) {
				pSndPrdRing->NextUsedBd %= pSndPrdRing->NumDescs;
			}
			DEV_KFREE_SKB_ANY(pMsg);
		}	/* while */

		pSndPrdRing->FreeBds = pSndPrdRing->NumDescs - 1;
		pSndPrdRing->NextFreeBd = 0;
		pSndPrdRing->NextUsedBd = 0;
	}	/* for */
}	/* FreeSndRings */


/*****************************************************************************
 *
 * 	XmitFrame - fill one socket buffer into the transmit ring
 *
 * Description:
 *	This function puts a message into the transmit descriptor ring
 *	if there is a descriptor left.
 *	Linux skb's consist of only one continuous buffer.
 *	The first step locks the ring. It is held locked all time.
 *	Then the descriptor is allocated.
 *	The second part is linking the buffer to the descriptor.
 *	At the very last, the Control field of the descriptor
 *	is made valid for the BMU and a start TX command is given
 *	if necessary.
 *
 * Returns:
 *	> 0 - on success: the number of bytes in the message
 *	= 0 - on resource shortage: this frame was sent, now
 *        the ring is full (-> set tbusy)
 *	< 0 - on resource shortage: this frame could not be sent
 */
static int XmitFrame(
SK_AC 			*pAC,		/* pointer to adapter context */
SK_IOC			IoC,		/* I/O Context */
int				RingIdx,	/* index of ring to send to */
struct sk_buff	*pMsg,		/* pointer to send message */
SK_BOOL			SlowPathLocked)
{
	SK_U64		PhysAddr;
	P_SND_RING	pSndPrdRing;	/* pointer to ring to send to */
	P_TX_DESC	pSbd;			/* pointer to Tx buffer descriptor to fill */
	unsigned	Flags;
	int			BdIndex;
	int 		Ret;

	pSndPrdRing = &pAC->SndPrdRing[RingIdx];

#ifdef	SK_WA_B2_3
	if (!SlowPathLocked) {
		/* Device may not be read (accessed?) more than once. */
		spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	}
#endif	/* SK_WA_B2_3 */
	spin_lock_irqsave(&pSndPrdRing->RingLock, Flags);

	if (pSndPrdRing->FreeBds == 0) {
		/* Currently not enough free descriptors in the ring. */
		FreeSbds(pAC, RingIdx);
		if (pSndPrdRing->FreeBds == 0) {
			spin_unlock_irqrestore(&pSndPrdRing->RingLock, Flags);
#ifdef	SK_WA_B2_3
			if (!SlowPathLocked) {
				/* Device may not be read (accessed?) more than once. */
				spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);
			}
#endif	/* SK_WA_B2_3 */
			// SK_PNMI_CNT_NO_TX_BUF(pAC, RingIdx);
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_TX_PROGRESS,
				("XmitFrame failed\n"));
			/* Frame could not be sent. */
			/* Do not free message buffer - else system will hang.*/
			/* DEV_KFREE_SKB(pMsg); */
			return (SK_NOT_OK);
		}
	}

#ifdef	SK_WA_B2_5_DISABLED
	/* DMA read may not end on 4 GB boundary. */

/*
 * This can't happen on IA32 with kernel 2.2.
 * This code was not tested.
 */

/* The following define may not be available in kernel 2.2. */
#if BITS_PER_LONG > 32
	{
		struct sk_buff	*pBadMsgs;
		struct sk_buff	*pMsgX;

		pBadMsgs = NULL;

		/* Maybe we should not use virt_to_bus() in kernel 2.4 ? */
		while ((virt_to_bus(pMsg->data) >> 32) !=
			(virt_to_bus(pMsg->data + pMsg->len - 1) >> 32)) {

			/* Mark message buffer as bad. */
			pMsg->next = pBadMsgs;
			pBadMsgs = pMsg;
			pMsgX = alloc_skb(pMsg->len);
			if (pMsgX == NULL) {
				Ret = pMsg->len;
				pMsg = NULL;
				break;
			}

			/* Copy message to new buffer. */
			skb_reserve(pMsgX, 2);	/* To align IP frames. */
			skb_put(pMsgX, pMsg->len);
			eth_copy_and_sum(pMsgX, pMsg->data, pMsg->len, 0);
			pMsg = pMsgX;
		}

		/* Free bad message buffers. */
		while (pBadMsgs != NULL) {
			pMsgX = pBadMsgs;
			pBadMsgs = pBadMsgs->next;
			DEV_KFREE_SKB_ANY(pMsgX);
		}

		if (pMsg == NULL) {
			/* Could not find good buffer. */
			printk("%s: Could not get suitable buffer for sending, "
				"trashing message.\n", pAC->dev->name);
//RARA
			/* Count error? */

			spin_unlock_irqrestore(&pSndPrdRing->RingLock, Flags);
#ifdef	SK_WA_B2_3
			if (!SlowPathLocked) {
				/* Device may not be read (accessed?) more than once. */
				spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);
			}
#endif	/* SK_WA_B2_3 */

			/* Return message length even if we did not really send. */
			return (Ret);
		}
	}
#endif	/* BITS_PER_LONG > 32 */
#endif	/* SK_WA_B2_5 */

	/* Reserve next buffer descriptor. */
	pSndPrdRing->FreeBds--;
	BdIndex = pSndPrdRing->NextFreeBd++;
	if (pSndPrdRing->NextFreeBd >= pSndPrdRing->NumDescs) {
		pSndPrdRing->NextFreeBd %= pSndPrdRing->NumDescs;
	}

	// pSbd = pSndPrdRing->pDescriptors[BdIndex].pSbd;
	pSbd = &pSndPrdRing->pSbdRing[BdIndex];

	/*
	 * Everything allocated ok, so add buffer to descriptor.
	 */
	pSndPrdRing->pDescriptors[BdIndex].pMsg = pMsg;

#ifdef SK_DUMP_TX
	DumpMsg(pMessage, "XmitFrame");
#endif	/* SK_DUMP_TX */

	/* Set up buffer descriptor. */
	if (pMsg->len <= ETH_MIN_FRAME_SIZE) {
		
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_TX_PROGRESS,
			("XmitFrame: len=%u\n", pMsg->len));

		pSbd->Length = (SK_U16)ETH_MIN_FRAME_SIZE;
	}
	else {
		pSbd->Length = (SK_U16)pMsg->len;
	}
	PhysAddr = (SK_U64)
#ifndef SK_KERNEL_22
		pSndPrdRing->pDescriptors[BdIndex].Phys =
#endif /* !SK_KERNEL_22 */
		pci_map_single(&pAC->PciDev,
			pMsg->data, pSbd->Length, PCI_DMA_TODEVICE);

#ifdef SK_LITTLE_ENDIAN
	pSbd->Buffer.High = (SK_U32)HIGH32(PhysAddr);
	pSbd->Buffer.Low  = (SK_U32)LOW32(PhysAddr);
#else
	pSbd->Buffer = PhysAddr;
#endif	/* SK_LITTLE_ENDIAN */
	
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_TX_PROGRESS,
		("XmitFrame: PhysAddr=0x%08X %08X\n",
		 pSbd->Buffer.High, pSbd->Buffer.Low));
	
	pSbd->Flags = TX_PACKET_END | TX_COAL_NOW;
	/* pSbd->VlanTag	= 0; */
	/* pSbd->Reserved	= 0; */

	if (pSndPrdRing->FreeBds != 0) {
		Ret = pMsg->len;
	}
	else {
		Ret = 0;
	}

	/* Write new producer index to mailbox. */
	SK_OUT32(IoC, pSndPrdRing->Mailbox, pSndPrdRing->NextFreeBd);

#ifdef SK_WA_BX_5
	/* Write twice to send producer index. */
	SK_OUT32(IoC, pSndPrdRing->Mailbox, pSndPrdRing->NextFreeBd);
#endif	/* SK_WA_BX_5 */

	/* After releasing the lock, the buffer may be freed immediately. */
	spin_unlock_irqrestore(&pSndPrdRing->RingLock, Flags);
#ifdef	SK_WA_B2_3
	if (!SlowPathLocked) {
		/* Device may not be read (accessed?) more than once. */
		spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);
	}
#endif	/* SK_WA_B2_3 */

	return (Ret);
} /* XmitFrame */


/*****************************************************************************
 *
 * 	SkGiXmit - Linux frame transmit function
 *
 * Description:
 *	The system calls this function to send frames onto the wire.
 *	It puts the frame into the tx descriptor ring. If the ring is
 *	full then, the 'tbusy' flag is set.
 *
 * Returns:
 *	== 0 if everything is ok
 *	!= 0 on error
 *
 * WARNING:	Returning -EBUSY in 'tbusy' case caused system crashes
 *			(double allocated skb's)!!!
 */
static int SkGiXmit(
struct sk_buff	*skb,
NET_DEV			*dev)
{
	SK_AC	*pAC;
	int		Ret;	/* Return code of XmitFrame */

	pAC = (SK_AC *)dev->priv;

	Ret = XmitFrame(pAC, pAC->IoBase, 0, skb, SK_FALSE);

//	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_TX_PROGRESS,
//		("XmitFrame, Ret=%d, jif=%ld\n", Ret, jiffies));

	if (Ret <= 0) {
		/* Transmitter is out of resources. */
		netif_stop_queue(dev);
		if (Ret == 0) {
			return (0);
		}
		else {
#ifdef SK_KERNEL_22
			return (-EBUSY);
#else  /* !SK_KERNEL_22 */
			return (1);
#endif /* !SK_KERNEL_22 */
		}	/* if (Ret == 0) */
	}	/* if (Ret <= 0) */

	dev->trans_start = jiffies;
	return (0);
} /* SkGiXmit */
	

/*****************************************************************************
 *
 * 	SkGiSetMacAddr - Set the hardware MAC address
 *
 * Description:
 *	This function sets the MAC address used by the adapter.
 *
 * Returns:
 *	== 0 if everything is ok
 *	!= 0 on error
 */
static int SkGiSetMacAddr(
NET_DEV	*dev,
void	*p)
{
	SK_AC			*pAC;
	struct sockaddr	*pAddr;
	unsigned int	Flags;
	
	pAC = (SK_AC *)dev->priv;
	pAddr = p;

	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiSetMacAddr starts now ...\n"));

	if (netif_running(dev)) {
		return (-EBUSY);
	}
	
	SK_MEMCPY(dev->dev_addr, pAddr->sa_data, dev->addr_len);
	
	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	SkAddrOverride(pAC, pAC->IoBase, (SK_MAC_ADDR *)dev->dev_addr,
		SK_ADDR_LOGICAL_ADDRESS);
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);

	return (0);
} /* SkGiSetMacAddr */


/*****************************************************************************
 *
 * 	SkGiSetRxMode - set receive mode
 *
 * Description:
 *	This function sets the receive mode of an adapter. The adapter
 *	supports promiscuous mode, allmulticast mode and a number of
 *	multicast addresses. If more multicast addresses than available
 *	are selected, a hash function in the hardware is used.
 *
 * Returns:
 *	== 0 if everything is ok
 *	!= 0 on error
 */
static void SkGiSetRxMode(
NET_DEV *dev)
{
	SK_AC				*pAC;	/* Pointer to Adapter Context */
	SK_IOC				IoC;
	struct dev_mc_list	*pMcList;
	int					i;
	unsigned int		Flags;

	pAC = (SK_AC *)dev->priv;
	IoC = pAC->IoBase;

	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiSetRxMode(%x) called.\n", dev->flags));

	if (!netif_running(dev)) {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("Driver not started.\n"));
		return;
	}

	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	if (dev->flags & IFF_PROMISC) {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("PROMISCUOUS mode.\n"));
		SkAddrPromiscuousChange(pAC, IoC, SK_PROM_MODE_LLC);
	} else if (dev->flags & IFF_ALLMULTI) {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("ALLMULTI mode\n"));
		SkAddrPromiscuousChange(pAC, IoC, SK_PROM_MODE_ALL_MC);
	} else {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY, ("Normal RX.\n"));
		SkAddrPromiscuousChange(pAC, IoC, SK_PROM_MODE_NONE);
		SkAddrMcClear(pAC, IoC, 0);

		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("Number of MC entries: %d.\n", dev->mc_count));
		
		pMcList = dev->mc_list;
		for (i = 0; i < dev->mc_count; i++, pMcList = pMcList->next) {
			SkAddrMcAdd(pAC, IoC, (SK_MAC_ADDR *)pMcList->dmi_addr, 0);
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_MCA,
				("MC %d: %02x:%02x:%02x:%02x:%02x:%02x\n", i,
				pMcList->dmi_addr[0], pMcList->dmi_addr[1],
				pMcList->dmi_addr[2], pMcList->dmi_addr[3],
				pMcList->dmi_addr[4], pMcList->dmi_addr[5]));
		}
		(void)SkAddrMcUpdate(pAC, IoC);
	}	/* else */
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);

	return;
} /* SkGiSetRxMode */


/*****************************************************************************
 *
 * 	SkGiChangeMtu - set the MTU to another value
 *
 * Description:
 *	This function is called whenever the MTU size is changed
 *	(ifconfig mtu xxx dev ethX). If the MTU is bigger than standard
 *	ethernet MTU size, long frame support is activated.
 *
 * Returns:
 *	== 0 if everything is ok
 *	!= 0 on error
 */
static int SkGiChangeMtu(
NET_DEV	*dev,
int		NewMtu)
{
	SK_EVPARA	Para;
	SK_AC		*pAC;
	SK_IOC		IoC;
	unsigned	Flags;
	int			i;

	pAC = (SK_AC *)dev->priv;
	IoC = pAC->IoBase;

	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiChangeMtu starts now ...\n"));

	if ((NewMtu < SK_MIN_MTU) || (NewMtu > SK_MAX_MTU)) {
		return (-EINVAL);
	}

	dev->mtu = NewMtu;
	
	if (NewMtu > SK_NORM_MTU) {
		pAC->Config.UseJumbo = SK_TRUE;
		pAC->RecJumPrdRing.BufferSize = dev->mtu + 36;
	}
	else {
		pAC->Config.UseJumbo = SK_FALSE;
	}

	if (!netif_running(dev)) {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
			("Driver not started.\n"));
		return (0);
	}

	/* Prevent reconfiguration while changing the MTU. */
	netif_stop_queue(dev);

	spin_lock_irqsave(&pAC->SlowPathLock, Flags);

	/* Disable interrupts. */
	SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost);
	SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);

	Para.Para32[0] = 0;	/* NetNumber */
	Para.Para32[1] = -1;
	SkEventQueue(pAC, SKGI_RLMT, SK_RLMT_STOP, Para);
	SkEventDispatcher(pAC, IoC);

	for (i = 0; i < pAC->Config.TxRingNum; i++) {
		spin_lock_irqsave(&pAC->SndPrdRing[i].RingLock, Flags);
	}
	FreeSndRings(pAC);
	FreeRecRings(pAC);
	for (i = 0; i < pAC->Config.TxRingNum; i++) {
		spin_unlock_irqrestore(&pAC->SndPrdRing[i].RingLock, Flags);
	}

	if (SkGiHwacInit(pAC, IoC, SK_INIT_RUN) != SK_OK) {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
			("SkGiHwacInit failed.\n"));
	}
	SkAddrMcUpdate(pAC, IoC);

#ifdef SK_EXT_MEM
	FillRecPrdRing(pAC, IoC, &pAC->RecMinPrdRing);
#endif	/* SK_EXT_MEM */
	FillRecPrdRing(pAC, IoC, &pAC->RecStdPrdRing);
	
	if (pAC->Config.UseJumbo) {
		FillRecPrdRing(pAC, IoC, &pAC->RecJumPrdRing);
	}

	Para.Para32[0] = 0;		/* NetNr */
	Para.Para32[1] = -1;
	SkEventQueue(pAC, SKGI_RLMT, SK_RLMT_START, Para);
	SkEventDispatcher(pAC, IoC);
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);

	/* Enable interrupts. */
	SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost & ~MASK_PCI_INT);
	// SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 0);
	// SK_OUT32(IoC, INTERRUPT_MAILBOX0, 0);
	// SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);
	// SK_OUT32(IoC, HOST_COAL_MODE, COALESC_NOW);
	// pAC->pStatusBlock->Status |= STATUS_WORD_UPDATED;
	// SK_OUT32(IoC, MISC_LOCAL, pAC->Hw.MiscLocal | SET_INT);

	netif_start_queue(dev);

	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiChangeMtu suceeded.\n"));

	return (0);
} /* SkGiChangeMtu */


/*****************************************************************************
 *
 * 	SkGiStats - return ethernet device statistics
 *
 * Description:
 *	This function return statistic data about the ethernet device
 *	to the operating system.
 *
 * Returns:
 *	pointer to the statistics structure.
 */
static NET_STATS *SkGiStats(
NET_DEV *dev)
{
	SK_AC				*pAC;	/* Pointer to Adapter Context */
	SK_PNMI_STRUCT_DATA	*pPnmiStruct;
	SK_PNMI_STAT		*pPnmiStat;
	unsigned int		Size;
	unsigned			Flags;

	pAC = (SK_AC *)dev->priv;

	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiStats starts now ...\n"));

	pPnmiStruct = &pAC->PnmiStruct;
	SK_MEMSET(pPnmiStruct, 0, sizeof(SK_PNMI_STRUCT_DATA));

	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	Size = SK_PNMI_STRUCT_SIZE;
	SkPnmiGetStruct(pAC, pAC->IoBase, pPnmiStruct, &Size);
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);

	pPnmiStat = &pPnmiStruct->Stat[0];

//	pAC->Stats.rx_packets	= (ulong)pPnmiStat->COSIfHCInPkts;
	pAC->Stats.rx_packets	=
		(ulong)pPnmiStat->ifHCInUcastPkts +
		(ulong)pPnmiStat->ifHCInMulticastPkts +
		(ulong)pPnmiStat->ifHCInBroadcastPkts;
	pAC->Stats.tx_packets	= (ulong)pPnmiStat->COSIfHCOutPkts;
	pAC->Stats.rx_bytes		= (ulong)pPnmiStat->ifHCInOctets;
	pAC->Stats.tx_bytes		= (ulong)pPnmiStat->ifHCOutOctets;
	pAC->Stats.rx_errors	= (ulong)pPnmiStat->ifInErrors;
	pAC->Stats.tx_errors	=
		(ulong)pPnmiStat->dot3StatsInternalMacTransmitErrors +
		(ulong)pPnmiStat->dot3StatsCarrierSenseErrors +
		(ulong)pPnmiStat->ifOutDiscards +
		(ulong)pPnmiStat->ifOutErrors;
	pAC->Stats.rx_dropped	= (ulong)pPnmiStat->ifInDiscards;
	pAC->Stats.tx_dropped	= (ulong)0;
	pAC->Stats.multicast	= (ulong)pPnmiStat->ifHCInMulticastPkts;
	pAC->Stats.collisions	= (ulong)pPnmiStat->dot3StatsSingleCollisionFrames;

	/* detailed rx_errors: */
	pAC->Stats.rx_length_errors	= (ulong)pPnmiStat->etherStatsUndersizePkts;
	pAC->Stats.rx_over_errors	= (ulong)pPnmiStat->nicNoMoreRxBDs;
	pAC->Stats.rx_crc_errors	= (ulong)pPnmiStat->dot3StatsFCSErrors;
	pAC->Stats.rx_frame_errors	= (ulong)pPnmiStat->dot3StatsAlignmentErrors;
	pAC->Stats.rx_fifo_errors	= (ulong)pPnmiStat->nicNoMoreRxBDs;
	pAC->Stats.rx_missed_errors	= (ulong)0;

	/* detailed tx_errors */
	pAC->Stats.tx_aborted_errors = (ulong)0;
	pAC->Stats.tx_carrier_errors =
		(ulong)pPnmiStat->dot3StatsCarrierSenseErrors;
	pAC->Stats.tx_fifo_errors =
		(ulong)pPnmiStat->dot3StatsInternalMacTransmitErrors;
	pAC->Stats.tx_heartbeat_errors =
		(ulong)pPnmiStat->dot3StatsCarrierSenseErrors;
	pAC->Stats.tx_window_errors = (ulong)0;

	return (&pAC->Stats);
} /* SkGiStats */


/*****************************************************************************
 *
 * 	SkGiIocMib - handle a GetMib, SetMib, or PresetMib ioctl message
 *
 * Description:
 *	This function reads/writes the MIB data using PNMI (Private Network
 *	Management Interface).
 *	The destination for the data must be provided with the ioctl call and is
 *	given to the driver in the form of a user space address.
 *	Copying from the user-provided data area into kernel messages and back
 *	is done by copy_from_user and copy_to_user calls in SkGeIoctl.
 *
 * Returns:
 *	size returned from PNMI call
 */
static int SkGiIocMib(
SK_AC		*pAC,	/* pointer to the adapter context */
unsigned	Size,	/* length of ioctl data */
int			Mode)	/* flag for set/preset */
{
	unsigned	Flags;	/* for spin lock */
	
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiIocMib starts now...\n"));

	/* access MIB */
	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	switch (Mode) {
	case SK_IOCTL_GETMIB:
		SkPnmiGetStruct(pAC, pAC->IoBase, &pAC->PnmiStruct, &Size);
		break;

	case SK_IOCTL_PRESETMIB:
		SkPnmiPreSetStruct(pAC, pAC->IoBase, &pAC->PnmiStruct, &Size);
		break;

	case SK_IOCTL_SETMIB:
		SkPnmiSetStruct(pAC, pAC->IoBase, &pAC->PnmiStruct, &Size);
		break;

	default:
		break;
	}
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);

	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("MIB data access succeeded\n"));
	return (Size);
} /* SkGiIocMib */


/*****************************************************************************
 *
 * 	SkGiIoctl - I/O-control function
 *
 * Description:
 *	This function is called if an ioctl is issued on the device.
 *	There are three subfunction for reading, writing and test-writing
 *	the private MIB data structure (usefull for SysKonnect-internal tools).
 *
 * Returns:
 *	== 0 if everything is ok
 *	!= 0 on error
 */
static int SkGiIoctl(
NET_DEV			*dev,
struct ifreq	*rq,
int				cmd)
{
	unsigned	Err;
	SK_AC		*pAC;	/* Pointer to Adapter Context */
	SK_GI_IOCTL	Ioctl;
	int		   	Size;

	Err = 0;
	pAC = (SK_AC *)dev->priv;
	
	SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
		("SkGiIoctl starts now...\n"));

	if (copy_from_user(&Ioctl, rq->ifr_data, sizeof(SK_GI_IOCTL))) {
		return (-EFAULT);
	}

	switch (cmd) {
	case SK_IOCTL_SETMIB:
	case SK_IOCTL_PRESETMIB:
		if (!capable(CAP_NET_ADMIN)) {
			return (-EPERM);
		}
 	case SK_IOCTL_GETMIB:
		if (copy_from_user(&pAC->PnmiStruct, Ioctl.pData,
			Ioctl.Len < sizeof(pAC->PnmiStruct)?
			Ioctl.Len : sizeof(pAC->PnmiStruct))) {

			return (-EFAULT);
		}
		Size = SkGiIocMib(pAC, Ioctl.Len, cmd);
		if (copy_to_user(Ioctl.pData, &pAC->PnmiStruct,
			Ioctl.Len < Size ? Ioctl.Len : Size)) {

			return (-EFAULT);
		}
		Ioctl.Len = Size;
		if (copy_to_user(rq->ifr_data, &Ioctl, sizeof(SK_GI_IOCTL))) {
			return (-EFAULT);
		}
		break;

#if 0
//RARA Haeh?
	case SK_IOCTL_SETROM:
		if (!capable(CAP_NET_ADMIN)) {
			return (-EPERM);
		}
		Size = Ioctl.Len<sizeof(pAC->Data) ? Ioctl.Len : sizeof(pAC->Data);
printk ("Data Size is %08x bytes.\n", Size);
		if (copy_from_user(&pAC->Data, Ioctl.pData, Size)) {
			return (-EFAULT);
		}
		break;

 	case SK_IOCTL_GETROM:
		Size = Ioctl.Len<sizeof(pAC->Data) ? Ioctl.Len : sizeof(pAC->Data);
printk ("Data Size is %08x bytes.\n", Size);
		// EReadLine(pAC->IoBase, 0, &pAC->Data[0], Size);
		if (copy_to_user(Ioctl.pData, &pAC->Data, Size)) {
			return (-EFAULT);
		}
		break;
#endif	/* 0 */

	default:
		Err = -EOPNOTSUPP;
		break;
	}

	return (Err);
} /* SkGiIoctl */


/*****************************************************************************
 *
 * 	FreeRecRings - free all receive rings
 *
 * Description:
 *	This function frees buffer descriptors and messages in all receive rings.
 *	This function is either called during an IRQ or with the IRQs disabled.
 *
 * Returns:	N/A
 */
static void	FreeRecRings(
SK_AC	*pAC)
{
	P_SK_RX_PACKET	pRpd;
	int	i;

	while (!IsEmpty(&pAC->RpdQ)) {
		pRpd = PopRpd(&pAC->RpdQ);
		pRpd->pMsg = NULL;	/* Did not point to real message. */
	}

	for (i = 0; i < pAC->NumRpds; i++) {
		if (pAC->pRpds[i].pMsg != NULL) {
			//SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_RX_PROGRESS,
			//	("freeing message %x from Rpd #%d\n",
			//	(unsigned)pAC->pRpds[i].pMsg, i));

			pci_unmap_single(&pAC->PciDev, pAC->pRpds[i].Phys,
				pAC->pRpds[i].pRecPrdRing->BufferSize, PCI_DMA_FROMDEVICE);
			DEV_KFREE_SKB_ANY(pAC->pRpds[i].pMsg);
			pAC->pRpds[i].pMsg = NULL;
		}
	}

	/* Re-fill RPD Queue */
	for (i = 0; i < pAC->NumRpds; i++) {
		PushRpd(&pAC->RpdQ, &pAC->pRpds[i]);
	}

#if SK_EXT_MEM
	pAC->RecMinPrdRing.FreeBds = pAC->RecMinPrdRing.NumDescs - 1;
	pAC->RecMinPrdRing.NextFreeBd = 0;
	pAC->RecMinPrdRing.NextUsedBd = 0;
#endif	/* SK_EXT_MEM */

	pAC->RecStdPrdRing.FreeBds = pAC->RecStdPrdRing.NumDescs - 1;
	pAC->RecStdPrdRing.NextFreeBd = 0;
	pAC->RecStdPrdRing.NextUsedBd = 0;

	pAC->RecJumPrdRing.FreeBds = pAC->RecJumPrdRing.NumDescs - 1;
	pAC->RecJumPrdRing.NextFreeBd = 0;
	pAC->RecJumPrdRing.NextUsedBd = 0;

	for (i = 0; i < pAC->Config.RxRingNum; i++) {
		pAC->RecRetRing[i].FreeBds = pAC->RecRetRing[i].NumDescs - 1;
		pAC->RecRetRing[i].NextFreeBd = 0;
		pAC->RecRetRing[i].NextUsedBd = 0;
	}
}	/* FreeRecRings */


/*****************************************************************************
 *
 * 	FillRecPrdRing - fill a receive producer ring with valid descriptors
 *
 * Description:
 *	This function fills a receive ring's descriptors with data segments.
 *
 * Returns:	N/A
 */
static void	FillRecPrdRing(
SK_AC		*pAC,
SK_IOC		IoC,
P_REC_RING	pRecPrdRing)
{
	SK_U64			PhysAddr;
	P_SK_RX_PACKET	pRpd;
	P_RX_DESC		pRbd;
	struct sk_buff	*pMsg;			/* pointer to a new message block */
	volatile SK_U16	*pConsumerIdx;
	int				ProducerIdx;

	pConsumerIdx = pRecPrdRing->pStatusIdx;
	ProducerIdx = pRecPrdRing->NextFreeBd;

	while (pRecPrdRing->FreeBds > 0) {
		if (IsEmpty(&pAC->RpdQ)) {
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_RX_PROGRESS,
				("RpdQ empty.\n"));
			break;
		}
		pMsg = alloc_skb(pRecPrdRing->BufferSize + 2, GFP_ATOMIC);
		if (pMsg == NULL) {
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_ENTRY,
				("%s: Allocation of rx buffer failed !\n", pAC->dev->name));
			// SK_PNMI_CNT_NO_RX_BUF(pAC, pRecRetRing->PortIndex);
			break;
		}

		//SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_RX_PROGRESS,
		//	("Filling RX descriptor #%d.\n", ProducerIdx));

		skb_reserve(pMsg, 2);	/* To align IP frames. */

		pRbd = &pRecPrdRing->pRbdRing[ProducerIdx];
		pRbd->Length = pRecPrdRing->BufferSize;

		pRpd = PopRpd(&pAC->RpdQ);
		PhysAddr = (SK_U64)
#ifndef SK_KERNEL_22
			pRpd->Phys =
#endif /* !SK_KERNEL_22 */
			pci_map_single(&pAC->PciDev,
				pMsg->data, pRbd->Length, PCI_DMA_FROMDEVICE);

		pRpd->pRbd = pRbd;
		pRpd->pRecPrdRing = pRecPrdRing;
		pRpd->pMsg = pMsg;

#ifdef SK_LITTLE_ENDIAN
		pRbd->Buffer.High = (SK_U32)HIGH32(PhysAddr);
		pRbd->Buffer.Low  = (SK_U32)LOW32(PhysAddr);
#else		
		pRbd->Buffer = PhysAddr;
#endif	/* SK_LITTLE_ENDIAN */
		
		pRbd->Index = 0;
		pRbd->Flags = 0 | RX_PACKET_END;
		/* Store offset of Receive Packet Descriptor in field Opaque. */
		pRbd->Opaque = (SK_U32)((SK_UPTR)pRpd - (SK_UPTR)pAC->pRpds);

		ProducerIdx++;
		if (ProducerIdx >= pRecPrdRing->NumDescs) {
			ProducerIdx %= pRecPrdRing->NumDescs;
		}
		pRecPrdRing->FreeBds--;
	}	/* while */

	if (ProducerIdx != pRecPrdRing->NextFreeBd) {
		pRecPrdRing->NextFreeBd = ProducerIdx;

//		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_RX_PROGRESS,
//			("Writing %d to 0x%x.\n", ProducerIdx, pRecPrdRing->Mailbox));
		
		SK_OUT32(IoC, pRecPrdRing->Mailbox, ProducerIdx);
	}
	else {
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_RX_PROGRESS,
			("Put nothing into RX producer ring with Mailbox 0x%x.\n",
			pRecPrdRing->Mailbox));
	}
}	/* FillRecPrdRing */


/*****************************************************************************
 *
 * 	ReceiveIrq - handle a receive IRQ
 *
 * Description:
 *	This function is called when a receive IRQ is set. It walks the
 *	receive descriptor ring and indicates all frames that are complete.
 *
 * Returns:	N/A
 */
static void ReceiveIrq(
SK_AC	*pAC,		/* Pointer to Adapter Context */
SK_IOC	IoC,		/* I/O Context */
int		RingIdx,	/* Index of receive return ring */
SK_BOOL	SlowPathLocked)
{
	P_REC_RING		pRecRetRing;
	P_RX_DESC		pRbd;
	P_SK_RX_PACKET	pRpd;
	struct sk_buff	*pMsg;
//RARA	struct sk_buff	*pNewMsg;
	volatile SK_U16	*pProducerIdx;
	int				ConsumerIdx;
	int				Length;

	pRecRetRing = &pAC->RecRetRing[RingIdx];
	pProducerIdx = pRecRetRing->pStatusIdx;
	ConsumerIdx = pRecRetRing->NextFreeBd;

	while (ConsumerIdx != *pProducerIdx) {
		pRbd = &pRecRetRing->pRbdRing[ConsumerIdx];
		pRpd = (P_SK_RX_PACKET)((SK_UPTR)pAC->pRpds + pRbd->Opaque);
		pMsg = pRpd->pMsg;

		Length = pRbd->Length - 4;	/* Without CRC */

		if ((pRbd->Flags & RX_ERROR_FRAME) != 0) {
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_RX_PROGRESS,
				("%s: Error in received frame, dropped!\n"
				"Flags: %04x; ErrorFlags: %04x\n",
				pAC->dev->name, pRbd->Flags, pRbd->ErrFlags));
#ifdef SK_DUMP_RX
			if (Length <= pAC->dev->mtu + 18) {	/* Header, CRC */
				/* Set length in message. */
				skb_put(pMsg, Length);
				DumpMsg(pMsg, "RxFrame");
			}
#endif	/* SK_DUMP_RX */
			pci_unmap_single(&pAC->PciDev, pRpd->Phys,
				pRpd->pRecPrdRing->BufferSize, PCI_DMA_FROMDEVICE);
			DEV_KFREE_SKB_ANY(pMsg);
		}
		else {
#if 0
			if (Length <= pAC->Config.RxCopySize) {
				/* Copy data of short frames to reduce memory waste. */
				pNewMsg = alloc_skb(Length + 2, GFP_ATOMIC);
				if (pNewMsg == NULL) {
					/* Use original pMsg - set length in message */
					skb_put(pMsg, Length);
					pci_unmap_single(&pAC->PciDev, pRpd->Phys,
						pRpd->pRecPrdRing->BufferSize, PCI_DMA_FROMDEVICE);
				}
				else {
					/* Copy data. */
					skb_reserve(pNewMsg, 2);
					skb_put(pNewMsg, Length);
					pci_dma_sync_single(&pAC->PciDev,pRpd->Phys,
						pRpd->pRecPrdRing->BufferSize, PCI_DMA_FROMDEVICE);
					eth_copy_and_sum(pNewMsg, pMsg->data, Length, 0);
//RARA ???
					ReQueueRxBuffer(pAC, pRxPort, pMsg,
						pRxd->VDataHigh, pRxd->VDataLow);
					pMsg = pNewMsg;
				}
			}
			else {
#else	/* !0 */
			{
#endif	/* !0 */
				/* Set length in message. */
				skb_put(pMsg, Length);
				pci_unmap_single(&pAC->PciDev, pRpd->Phys,
					pRpd->pRecPrdRing->BufferSize, PCI_DMA_FROMDEVICE);

				/* Check HW checksumming. */
				// Type = ntohs(*((short*)&pMsg->data[12]));
				pMsg->pkt_type = 0;
				pMsg->dev = pAC->dev;
				pMsg->protocol = eth_type_trans(pMsg, pAC->dev);

				if ((pAC->Config.TaskOffloadCap & TASK_OFFLOAD_CAP_RX) != 0 &&
					(pRbd->Flags & RX_TCP_UDP_CKSUM) != 0 &&
					pRbd->TcpUdpChksum == 0xffff &&
					pMsg->protocol == __constant_htons(ETH_P_IP) &&
					(((struct iphdr *)pMsg->data)->frag_off &
					__constant_htons(IP_MF | IP_OFFSET)) != 0) {

					SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_RX_PROGRESS,
						("Checksum OK.\n"));
					pMsg->ip_summed = CHECKSUM_UNNECESSARY;
				}
				else {
					pMsg->ip_summed = CHECKSUM_NONE;
				}
			}

			if ((pRbd->Flags & RX_VLAN_TAG) != 0) {
				DEV_KFREE_SKB_ANY(pMsg);
			}
			else {
				netif_rx(pMsg);
			}
		}

		pRpd->pRecPrdRing->FreeBds++;

		PushRpd(&pAC->RpdQ, pRpd);
		ConsumerIdx++;
		if (ConsumerIdx >= pRecRetRing->NumDescs) {
			ConsumerIdx %= pRecRetRing->NumDescs;
		}
	}	/* while */

	pRecRetRing->NextFreeBd = ConsumerIdx;
	SK_OUT32(IoC, pAC->RecRetRing[0].Mailbox, ConsumerIdx);

	return;
}	/* ReceiveIrq */


/****************************************************************************
 *
 *	SkGiIsr - handle adapter interrupts
 *
 * Description:
 *	The interrupt routine is called when the network adapter
 *	generates an interrupt. It may also be called if another device
 *	shares this interrupt vector with the driver.
 *
 * Returns: N/A
 *
 */
static void SkGiIsr(
int				irq,
void			*dev_id,
struct pt_regs	*ptregs)
{
	NET_DEV	*dev;
	SK_AC	*pAC;	/* Pointer to Adapter Context */
	SK_IOC	IoC;	/* I/O Context */
	SK_U32	Data32;
	int		i;

	dev = (NET_DEV *)dev_id;
	pAC = (SK_AC *)dev->priv;
	IoC = pAC->IoBase;

	/* Check and process if it's our interrupt. */
	while ((pAC->pStatusBlock->Status & STATUS_WORD_UPDATED) != 0) {
		/* Disable controller interrupts. */
		SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_INT_SRC,
			("%ld: in ISR, Status: %x\n", jiffies, pAC->pStatusBlock->Status));
	
		pAC->pStatusBlock->Status &= ~STATUS_WORD_UPDATED;

#ifdef	SK_WA_BX_10
		/* Link state update in status block. */

#ifdef	SK_WA_B2_3
		/* Device may not be read (accessed?) more than once. */
		spin_lock(&pAC->SlowPathLock);
#endif	/* SK_WA_B2_3 */
		SK_IN32(IoC, ETH_MAC_STATUS, &Data32);

		if ((Data32 & (LINK_STATE_CHANGED | MI_INTERRUPT)) != 0) {
			
//			pAC->pStatusBlock->Status &= ~STATUS_WORD_LINK_CHANGED;
#ifdef DEBUG
			SK_IN32(IoC, US_TIMER_REG, &Data32);
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
				("\nLink state changed, Myst=%10u\n", Data32));
#endif /* DEBUG */
			pAC->dev->flags &= ~IFF_RUNNING;
			
			(void)SkGiInitPhy(pAC, IoC);
		}
#ifdef	SK_WA_B2_3
		/* Device may not be read (accessed?) more than once. */
		spin_unlock(&pAC->SlowPathLock);
#endif	/* SK_WA_B2_3 */

#else	/* !SK_WA_BX_10 */

		if ((pAC->pStatusBlock->Status & STATUS_WORD_LINK_CHANGED) != 0) {
			
			pAC->pStatusBlock->Status &= ~STATUS_WORD_LINK_CHANGED;
			
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
				("Link changed, Status=0x%02X\n", pAC->pStatusBlock->Status));
			
			pAC->dev->flags &= ~IFF_RUNNING;
			
			(void)SkGiInitPhy(pAC, IoC);
		}
#endif	/* !SK_WA_BX_10 */

		for (i = 0; i < pAC->Config.RxRingNum; i++) {
			if (pAC->RecRetRing[i].NextFreeBd !=
				pAC->pStatusBlock->ConProdIdx[i].ProdRx) {

				ReceiveIrq(pAC, IoC, i, SK_FALSE);
				// SK_PNMI_CNT_RX_INTR(pAC, i);
			}
		}	/* for */

		for (i = 0; i < pAC->Config.TxRingNum; i++) {
			if (pAC->SndPrdRing[i].NextUsedBd !=
				pAC->pStatusBlock->ConProdIdx[i].ConTx) {

				// SK_PNMI_CNT_TX_INTR(pAC, 0);
				spin_lock(&pAC->SndPrdRing[i].RingLock);
				FreeSbds(pAC, i);
				spin_unlock(&pAC->SndPrdRing[i].RingLock);
			}
		}	/* for */

		if ((pAC->pStatusBlock->Status & STATUS_WORD_ERROR) != 0) {
			pAC->pStatusBlock->Status &= ~STATUS_WORD_ERROR;
			/* Ignored in original sources ... */
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_INT_SRC,
				("Error IRQ!\n"));
		}

#ifdef SK_EXT_MEM
		if (pAC->RecMinPrdRing.FreeBds > pAC->RecMinPrdRing.NumDescs / 8) {
			FillRecPrdRing(pAC, IoC, &pAC->RecMinPrdRing);
		}
#endif	/* SK_EXT_MEM */
		if (pAC->RecStdPrdRing.FreeBds > pAC->RecStdPrdRing.NumDescs / 8) {
			FillRecPrdRing(pAC, IoC, &pAC->RecStdPrdRing);
		}
		if (pAC->Config.UseJumbo &&
			pAC->RecJumPrdRing.FreeBds > pAC->RecJumPrdRing.NumDescs / 8) {

			FillRecPrdRing(pAC, IoC, &pAC->RecJumPrdRing);
		}

		/* Enable controller interrupts. */
		SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 0);
#ifdef	SK_WA_B2_3
		/* Device may not be read (accessed?) more than once. */
		spin_lock(&pAC->SlowPathLock);
#endif	/* SK_WA_B2_3 */

		SK_IN32(IoC, INTERRUPT_MAILBOX0 + 4, &Data32);

#ifdef	SK_WA_B2_3
		/* Device may not be read (accessed?) more than once. */
		spin_unlock(&pAC->SlowPathLock);
#endif	/* SK_WA_B2_3 */
	}	/* WHILE */

	if ((pAC->pStatusBlock->Status & STATUS_WORD_UPDATED) != 0) {
		/* Force an interrupt. */
		SK_OUT32(IoC, MISC_LOCAL, pAC->Hw.MiscLocal | SET_INT);
	}
	else {
		SK_OUT32(IoC, MISC_LOCAL, pAC->Hw.MiscLocal | CLEAR_INT);
	}

	if (pAC->Event.CheckQueue) {
		spin_lock(&pAC->SlowPathLock);
		SkEventDispatcher(pAC, IoC);
		spin_unlock(&pAC->SlowPathLock);
	}

	return;
}	/* SkGiIsr */

#ifdef SK_OSTIMER_STOPPABLE

/*****************************************************************************
 *
 *	SkOsTimerStart - start the OS timer
 *
 * Description:
 *	This routine starts the OS timer to fire after Delta "TICKs".
 *
 *
 * Returns: N/A
 *
 */
void		SkOsTimerStart(
SK_AC		*pAC,	/* Pointer to Adapter Context */
SK_IOC		IoC,	/* I/O Context */
SK_U32		Delta)	/* Time in TICKs when the timer should fire the next time */
{
	pAC->Timer.expires = RUN_AT(Delta);
	add_timer(&pAC->Timer);
}	/* SkOsTimerStart */


/*****************************************************************************
 *
 *	SkOsTimerStop - stop the OS timer
 *
 * Description:
 *	This routine stops the OS timer.
 *
 *
 * Returns: N/A
 *
 */
void		SkOsTimerStop(
SK_AC		*pAC,
SK_IOC		IoC)
{
	del_timer(&pAC->Timer);
}	/* SkOsTimerStop */

#endif	/* SK_OSTIMER_STOPPABLE */

/*****************************************************************************
 *
 *	SkOsTimerInit - initialize the OS timer
 *
 * Description:
 *	This routine initializes the OS timer.
 *
 *
 * Returns: N/A
 *
 */
void		SkOsTimerInit(
SK_AC		*pAC,	/* Pointer to Adapter Context */
SK_IOC		IoC)	/* I/O Context */
{
	init_timer(&pAC->Timer);
	pAC->Timer.data = (ulong)pAC->dev;
	pAC->Timer.function = &SkGiTimer;
}	/* SkOsTimerInit */


/*****************************************************************************
 *
 *	SkGiTimer - OS timer fired
 *
 * Description:
 *	This routine calls the timer module as the OS timer has fired.
 *	As a workaround, it also forces an interrupt.
 *
 *
 * Returns: N/A
 *
 */
void SkGiTimer(
ulong Data)
{
	NET_DEV		*dev;
	SK_AC		*pAC;
	SK_IOC		IoC;
	unsigned	Flags;

	dev = (NET_DEV *)Data;
	pAC = (SK_AC *)dev->priv;
	IoC = pAC->IoBase;

	spin_lock_irqsave(&pAC->SlowPathLock, Flags);
	SkTimerDone(pAC, IoC);
	if (pAC->Event.CheckQueue) {
		SkEventDispatcher(pAC, IoC);
	}
	spin_unlock_irqrestore(&pAC->SlowPathLock, Flags);

#ifdef	SK_WA_BX_1
	/* Missed status block updates. */
	pAC->Timer.expires = RUN_AT(1);		/* 1/100 sec in x86 Linux. */
	add_timer(&pAC->Timer);

	/* Possibility 1. */
//	SK_OUT32(IoC, HOST_COAL_MODE, COALESC_NOW | COAL_ENG_ENABLE);

	/* Possibility 2. */
#if 0 /* RS */
	if ((pAC->pStatusBlock->Status & STATUS_WORD_UPDATED) != 0) {
		/* Force an interrupt. */
		SK_OUT32(IoC, MISC_LOCAL, pAC->Hw.MiscLocal | SET_INT);
	}
#endif
	pAC->pStatusBlock->Status |= STATUS_WORD_UPDATED;

	SK_OUT32(IoC, MISC_LOCAL, pAC->Hw.MiscLocal | SET_INT);
#endif	/* SK_WA_BX_1 */
}	/* SkGiTimer */

#if 0

/*****************************************************************************
 *
 *	SkOsGetTime - provide a time value
 *
 * Description:
 *	This routine provides a time value. The unit is 1/HZ (defined by Linux).
 *	It is not used for absolute time, but only for time differences.
 *
 *
 * Returns:
 *	Time value
 */
SK_U64 SkOsGetTime(
SK_AC	*pAC)	/* Pointer to Adapter Context */
{
	return (jiffies);
} /* SkOsGetTime */

#endif	/* 0 */

/*****************************************************************************
 *
 *	SkDrvAllocRlmtMbuf - allocate an mbuf for RLMT
 *
 * Description:
 *	This routine allocates an mbuf for RLMT.
 *
 *
 * Returns:
 *	Pointer to mbuf, NULL if allocation failed
 */
SK_MBUF	*SkDrvAllocRlmtMbuf(
SK_AC		*pAC,
SK_IOC		IoC,
unsigned	Length)
{
	return (NULL);
}	/* SkDrvAllocRlmtMbuf */


/*****************************************************************************
 *
 *	SkDrvFreeRlmtMbuf - free an RLMT mbuf
 *
 * Description:
 *	This routine frees an RLMT mbuf.
 *
 *
 * Returns: N/A
 *
 */
void SkDrvFreeRlmtMbuf(
SK_AC		*pAC,
SK_IOC		IoC,
SK_MBUF		*pMb)
{
}	/* SkDrvFreeRlmtMbuf */


/*****************************************************************************
 *
 *	SkGiDrvEvent - handle DRV events
 *
 * Description:
 *	This routine handles events for DRV.
 *
 *
 * Returns:
 *	0
 */
int	SkGiDrvEvent(
SK_AC		*pAC,	/* Pointer to Adapter Context */
SK_IOC		IoC,	/* I/O Context */
SK_U32		Event,
SK_EVPARA	Para)
{
	struct sk_buff	*pMsg;
	SK_MBUF			*pRlmtMbuf;
	SK_EVPARA		NewPara;
	unsigned		Flags;
#if 0
	int				i;
	SK_U32			Dummy;
#endif	/* 0 */

	/* Just to make the compiler happy. */
	Flags = 0;

	switch (Event) {
	case SK_DRV_PORT_RESET:
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
			("PORT RESET \n"));
		NewPara.Para64 = 0;	/* PortNumber */
		SkPnmiEvent(pAC, IoC, SK_PNMI_EVT_MAC_RESET, NewPara);

#if 0
		SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);
		SK_IN32(IoC, INTERRUPT_MAILBOX0 + 4, &Dummy);
		
		/* Disable interrupts. */
		SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost);

		for (i = 0; i < pAC->Config.TxRingNum; i++) {
			spin_lock_irqsave(&pAC->SndPrdRing[i].RingLock, Flags);
		}

		if (SkGiHwacDeinit(pAC, IoC) != SK_OK) {
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
				("SkGiHwacDeinit failed!\n"));
		}

		for (i = 0; i < pAC->Config.RxRingNum; i++) {
			ReceiveIrq(pAC, IoC, i, SK_TRUE);
		}
		FreeSndRings(pAC);
		FreeRecRings(pAC);
		
		if (SkGiHwacInit(pAC, IoC, SK_INIT_RUN) != SK_OK) {
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
				("SkGiHwacInit failed.\n"));
		}
		SkAddrMcUpdate(pAC, IoC);

		for (i = 0; i < pAC->Config.TxRingNum; i++) {
			spin_unlock_irqrestore(&pAC->SndPrdRing[i].RingLock, Flags);
		}

#ifdef SK_EXT_MEM
		FillRecPrdRing(pAC, IoC, &pAC->RecMinPrdRing);
#endif	/* SK_EXT_MEM */
		FillRecPrdRing(pAC, IoC, &pAC->RecStdPrdRing);
		
		if (pAC->Config.UseJumbo) {
			FillRecPrdRing(pAC, IoC, &pAC->RecJumPrdRing);
		}

	  	/* Inform the world that link has been closed */
	 	pAC->dev->flags &= ~IFF_RUNNING;

		/* Enable interrupts. */
		SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost & ~MASK_PCI_INT);
		// SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 0);
		// SK_OUT32(IoC, INTERRUPT_MAILBOX0, 0);
		// SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);
		// SK_OUT32(IoC, HOST_COAL_MODE, COALESC_NOW);
		// pAC->pStatusBlock->Status |= STATUS_WORD_UPDATED;
		// SK_OUT32(IoC, MISC_LOCAL, pAC->Hw.MiscLocal | SET_INT);
#else	/* !0 */
		SkAddrMcUpdate(pAC, IoC);
#endif	/* !0 */
		break;

	case SK_DRV_NET_UP:
		if ((pAC->dev->flags & IFF_RUNNING) == 0) {
			SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
				("NET_UP "));
			/* Inform the world that link protocol is up. */
			pAC->dev->flags |= IFF_RUNNING;
			
			ShowDriverStatus(pAC);
		}
		break;

	case SK_DRV_NET_DOWN:	 /* SK_U32 Reason */
		/* action list 7 */
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
			("NET_DOWN "));
		printk("%s: network connection down\n", pAC->dev->name);
	  	/* Inform the world that link has been closed */
	 	pAC->dev->flags &= ~IFF_RUNNING;
		break;

	case SK_DRV_RLMT_SEND:	 /* SK_MBUF *pMb */
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT, ("RLS "));
		pRlmtMbuf = (SK_MBUF *)Para.pParaPtr;
		pMsg = pRlmtMbuf->pOs;
		skb_put(pMsg, pRlmtMbuf->Length);
		if (XmitFrame(pAC, IoC, 0, pMsg, SK_TRUE) < 0) {
			netif_stop_queue(pAC->dev);
			DEV_KFREE_SKB_ANY(pMsg);
		}
		break;		

	case SK_DRV_ADAP_FAIL:
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
			("ADAPTER FAIL EVENT!\n"));
		printk("%s: Adapter failed.\n", pAC->dev->name);
		
		/* Disable interrupts. */
		SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost);
		SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 1);
		break;

	case SK_DRV_PORT_FAIL:
		SK_DBG_MSG(pAC, SK_DBGMOD_DRV, SK_DBGCAT_DRV_EVENT,
			("PORT FAIL EVENT!\n"))
		printk("%s: Port failed.\n", pAC->dev->name);
		break;

	default:
		printk("SkGiDrvEvent: %d(%d).\n", Event, Para.Para32[0]);
		break;
	}

	return (0);
} /* SkGiDrvEvent */


#ifdef DEBUG

/****************************************************************************/
/* "debug only" section *****************************************************/
/****************************************************************************/

#if defined(SK_DUMP_TX) || defined(SK_DUMP_RX)

/*****************************************************************************
 *
 *	DumpMsg - print a frame
 *
 * Description:
 *	This function prints frames to the system logfile/to the console.
 *
 * Returns: N/A
 *	
 */
static void DumpMsg(
struct sk_buff	*skb,
char			*str)
{
	int	msglen;

	if (skb == NULL) {
		printk("DumpMsg(): NULL-Message\n");
		return;
	}

	if (skb->data == NULL) {
		printk("DumpMsg(): Message empty\n");
		return;
	}

	msglen = skb->len;
	if (msglen > 64) {
		msglen = 64;
	}

	printk("--- Begin of message from %s , len %d (from %d) ----\n",
		str, msglen, skb->len);

	DumpData((char *)skb->data, msglen);

	printk("------- End of message ---------\n");
} /* DumpMsg */


/*****************************************************************************
 *
 *	DumpData - print a data area
 *
 * Description:
 *	This function prints a area of data to the system logfile/to the
 *	console.
 *
 * Returns: N/A
 *	
 */
static void DumpData(
char	*p,
int		size)
{
	register int    i;
	int				haddr, addr;
	char			hex_buffer[180];
	char			asc_buffer[180];
	char			HEXCHAR[] = "0123456789ABCDEF";

	addr = 0;
	haddr = 0;
	hex_buffer[0] = 0;
	asc_buffer[0] = 0;
	for (i = 0; i < size; ) {
		if (*p >= '0' && *p <='z') {
			asc_buffer[addr] = *p;
		}
		else {
			asc_buffer[addr] = '.';
		}
		addr++;
		asc_buffer[addr] = 0;
		hex_buffer[haddr] = HEXCHAR[(*p & 0xf0) >> 4];
		haddr++;
		hex_buffer[haddr] = HEXCHAR[*p & 0x0f];
		haddr++;
		hex_buffer[haddr] = ' ';
		haddr++;
		hex_buffer[haddr] = 0;
		p++;
		i++;
		if (i % 16 == 0) {
			printk("%s  %s\n", hex_buffer, asc_buffer);
			addr = 0;
			haddr = 0;
		}
	}
} /* DumpData */

#if 0

/*****************************************************************************
 *
 *	DumpLong - print a data area as long values
 *
 * Description:
 *	This function prints a area of data to the system logfile/to the
 *	console.
 *
 * Returns: N/A
 *	
 */
static void DumpLong(
char	*pc,
int		size)
{
	register int	i;
	int				haddr, addr;
	char			hex_buffer[180];
	char			asc_buffer[180];
	char			HEXCHAR[] = "0123456789ABCDEF";
	long			*p;
	int				l;

	addr = 0;
	haddr = 0;
	hex_buffer[0] = 0;
	asc_buffer[0] = 0;
	p = (long *)pc;
	for (i = 0; i < size; ) {
		l = (long) *p;
		hex_buffer[haddr] = HEXCHAR[(l >> 28) & 0xf];
		haddr++;
		hex_buffer[haddr] = HEXCHAR[(l >> 24) & 0xf];
		haddr++;
		hex_buffer[haddr] = HEXCHAR[(l >> 20) & 0xf];
		haddr++;
		hex_buffer[haddr] = HEXCHAR[(l >> 16) & 0xf];
		haddr++;
		hex_buffer[haddr] = HEXCHAR[(l >> 12) & 0xf];
		haddr++;
		hex_buffer[haddr] = HEXCHAR[(l >> 8) & 0xf];
		haddr++;
		hex_buffer[haddr] = HEXCHAR[(l >> 4) & 0xf];
		haddr++;
		hex_buffer[haddr] = HEXCHAR[l & 0x0f];
		haddr++;
		hex_buffer[haddr] = ' ';
		haddr++;
		hex_buffer[haddr] = 0;
		p++;
		i++;
		if (i % 8 == 0) {
			printk("%4x %s\n", (i - 8) * 4, hex_buffer);
			haddr = 0;
		}
	}
	printk("------------------------\n");
} /* DumpLong */

#endif	/* 0 */
#endif	/* defined(SK_DUMP_TX) || defined(SK_DUMP_RX) */
#endif	/* DEBUG */

/*
 * Local variables:
 * compile-command: "make"
 * End:
 */

