// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * Copyright (C) 2016 IBM Corp.
 */
#include <linux/bitops.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/mutex.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/pinctrl/pinctrl.h>
#include <linux/pinctrl/pinmux.h>
#include <linux/pinctrl/pinconf.h>
#include <linux/pinctrl/pinconf-generic.h>
#include <linux/string.h>
#include <linux/types.h>

#include "../core.h"
#include "../pinctrl-utils.h"
#include "pinmux-aspeed.h"
#include "pinctrl-aspeed.h"

/* Wrap some of the common macros for clarity */
#define SIG_EXPR_DECL_SINGLE(sig, func, ...) \
	SIG_EXPR_DECL(sig, func, func, __VA_ARGS__)

#define SIG_EXPR_LIST_DECL_SINGLE SIG_EXPR_LIST_DECL_SESG
#define SIG_EXPR_LIST_DECL_DUAL SIG_EXPR_LIST_DECL_DESG

/*
 * The "Multi-function Pins Mapping and Control" table in the SoC datasheet
 * references registers by the device/offset mnemonic. The register macros
 * below are named the same way to ease transcription and verification (as
 * opposed to naming them e.g. PINMUX_CTRL_[0-9]). Further, signal expressions
 * reference registers beyond those dedicated to pinmux, such as the system
 * reset control and MAC clock configuration registers.
 */
#define SCU2C           0x2C /* Misc. Control Register */
#define SCU3C           0x3C /* System Reset Control/Status Register */
#define SCU48           0x48 /* MAC Interface Clock Delay Setting */
#define HW_STRAP1       0x70 /* AST2400 strapping is 33 bits, is split */
#define HW_REVISION_ID  0x7C /* Silicon revision ID register */
#define SCU80           0x80 /* Multi-function Pin Control #1 */
#define SCU84           0x84 /* Multi-function Pin Control #2 */
#define SCU88           0x88 /* Multi-function Pin Control #3 */
#define SCU8C           0x8C /* Multi-function Pin Control #4 */
#define SCU90           0x90 /* Multi-function Pin Control #5 */
#define SCU94           0x94 /* Multi-function Pin Control #6 */
#define SCUA0           0xA0 /* Multi-function Pin Control #7 */
#define SCUA4           0xA4 /* Multi-function Pin Control #8 */
#define SCUA8           0xA8 /* Multi-function Pin Control #9 */
#define SCUAC           0xAC /* Multi-function Pin Control #10 */
#define HW_STRAP2       0xD0 /* Strapping */

/*
 * Uses undefined macros for symbol naming and references, eg GPIOA0, MAC1LINK,
 * TIMER3 etc.
 *
 * Pins are defined in GPIO bank order:
 *
 * GPIOA0: 0
 * ...
 * GPIOA7: 7
 * GPIOB0: 8
 * ...
 * GPIOZ7: 207
 * GPIOAA0: 208
 * ...
 * GPIOAB3: 219
 *
 * Not all pins have their signals defined (yet).
 */

#define D6 0
SSSF_PIN_DECL(D6, GPIOA0, MAC1LINK, SIG_DESC_SET(SCU80, 0));

#define B5 1
SSSF_PIN_DECL(B5, GPIOA1, MAC2LINK, SIG_DESC_SET(SCU80, 1));

#define A4 2
SSSF_PIN_DECL(A4, GPIOA2, TIMER3, SIG_DESC_SET(SCU80, 2));

#define E6 3
SSSF_PIN_DECL(E6, GPIOA3, TIMER4, SIG_DESC_SET(SCU80, 3));

#define I2C9_DESC	SIG_DESC_SET(SCU90, 22)

#define C5 4
SIG_EXPR_LIST_DECL_SINGLE(C5, SCL9, I2C9, I2C9_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C5, TIMER5, TIMER5, SIG_DESC_SET(SCU80, 4));
PIN_DECL_2(C5, GPIOA4, SCL9, TIMER5);

FUNC_GROUP_DECL(TIMER5, C5);

#define B4 5
SIG_EXPR_LIST_DECL_SINGLE(B4, SDA9, I2C9, I2C9_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B4, TIMER6, TIMER6, SIG_DESC_SET(SCU80, 5));
PIN_DECL_2(B4, GPIOA5, SDA9, TIMER6);

FUNC_GROUP_DECL(TIMER6, B4);
FUNC_GROUP_DECL(I2C9, C5, B4);

#define MDIO2_DESC	SIG_DESC_SET(SCU90, 2)

#define A3 6
SIG_EXPR_LIST_DECL_SINGLE(A3, MDC2, MDIO2, MDIO2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A3, TIMER7, TIMER7, SIG_DESC_SET(SCU80, 6));
PIN_DECL_2(A3, GPIOA6, MDC2, TIMER7);

FUNC_GROUP_DECL(TIMER7, A3);

#define D5 7
SIG_EXPR_LIST_DECL_SINGLE(D5, MDIO2, MDIO2, MDIO2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D5, TIMER8, TIMER8, SIG_DESC_SET(SCU80, 7));
PIN_DECL_2(D5, GPIOA7, MDIO2, TIMER8);

FUNC_GROUP_DECL(TIMER8, D5);
FUNC_GROUP_DECL(MDIO2, A3, D5);

#define J21 8
SSSF_PIN_DECL(J21, GPIOB0, SALT1, SIG_DESC_SET(SCU80, 8));

#define J20 9
SSSF_PIN_DECL(J20, GPIOB1, SALT2, SIG_DESC_SET(SCU80, 9));

#define H18 10
SSSF_PIN_DECL(H18, GPIOB2, SALT3, SIG_DESC_SET(SCU80, 10));

#define F18 11
SSSF_PIN_DECL(F18, GPIOB3, SALT4, SIG_DESC_SET(SCU80, 11));

#define E19 12
SIG_EXPR_DECL_SINGLE(LPCRST, LPCRST, SIG_DESC_SET(SCU80, 12));
SIG_EXPR_DECL_SINGLE(LPCRST, LPCRSTS, SIG_DESC_SET(HW_STRAP1, 14));
SIG_EXPR_LIST_DECL_DUAL(E19, LPCRST, LPCRST, LPCRSTS);
PIN_DECL_1(E19, GPIOB4, LPCRST);

FUNC_GROUP_DECL(LPCRST, E19);

#define H19 13
#define H19_DESC        SIG_DESC_SET(SCU80, 13)
SIG_EXPR_LIST_DECL_SINGLE(H19, LPCPD, LPCPD, H19_DESC);
SIG_EXPR_LIST_DECL_SINGLE(H19, LPCSMI, LPCSMI, H19_DESC);
PIN_DECL_2(H19, GPIOB5, LPCPD, LPCSMI);

FUNC_GROUP_DECL(LPCPD, H19);
FUNC_GROUP_DECL(LPCSMI, H19);

#define H20 14
SSSF_PIN_DECL(H20, GPIOB6, LPCPME, SIG_DESC_SET(SCU80, 14));

#define E18 15
SIG_EXPR_LIST_DECL_SINGLE(E18, EXTRST, EXTRST,
		SIG_DESC_SET(SCU80, 15),
		SIG_DESC_BIT(SCU90, 31, 0),
		SIG_DESC_SET(SCU3C, 3));
SIG_EXPR_LIST_DECL_SINGLE(E18, SPICS1, SPICS1,
		SIG_DESC_SET(SCU80, 15),
		SIG_DESC_SET(SCU90, 31));
PIN_DECL_2(E18, GPIOB7, EXTRST, SPICS1);

FUNC_GROUP_DECL(EXTRST, E18);
FUNC_GROUP_DECL(SPICS1, E18);

#define SD1_DESC	SIG_DESC_SET(SCU90, 0)
#define I2C10_DESC	SIG_DESC_SET(SCU90, 23)

#define C4 16
SIG_EXPR_LIST_DECL_SINGLE(C4, SD1CLK, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C4, SCL10, I2C10, I2C10_DESC);
PIN_DECL_2(C4, GPIOC0, SD1CLK, SCL10);

#define B3 17
SIG_EXPR_LIST_DECL_SINGLE(B3, SD1CMD, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B3, SDA10, I2C10, I2C10_DESC);
PIN_DECL_2(B3, GPIOC1, SD1CMD, SDA10);

FUNC_GROUP_DECL(I2C10, C4, B3);

#define I2C11_DESC	SIG_DESC_SET(SCU90, 24)

#define A2 18
SIG_EXPR_LIST_DECL_SINGLE(A2, SD1DAT0, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A2, SCL11, I2C11, I2C11_DESC);
PIN_DECL_2(A2, GPIOC2, SD1DAT0, SCL11);

#define E5 19
SIG_EXPR_LIST_DECL_SINGLE(E5, SD1DAT1, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E5, SDA11, I2C11, I2C11_DESC);
PIN_DECL_2(E5, GPIOC3, SD1DAT1, SDA11);

FUNC_GROUP_DECL(I2C11, A2, E5);

#define I2C12_DESC	SIG_DESC_SET(SCU90, 25)

#define D4 20
SIG_EXPR_LIST_DECL_SINGLE(D4, SD1DAT2, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D4, SCL12, I2C12, I2C12_DESC);
PIN_DECL_2(D4, GPIOC4, SD1DAT2, SCL12);

#define C3 21
SIG_EXPR_LIST_DECL_SINGLE(C3, SD1DAT3, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C3, SDA12, I2C12, I2C12_DESC);
PIN_DECL_2(C3, GPIOC5, SD1DAT3, SDA12);

FUNC_GROUP_DECL(I2C12, D4, C3);

#define I2C13_DESC	SIG_DESC_SET(SCU90, 26)

#define B2 22
SIG_EXPR_LIST_DECL_SINGLE(B2, SD1CD, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B2, SCL13, I2C13, I2C13_DESC);
PIN_DECL_2(B2, GPIOC6, SD1CD, SCL13);

#define A1 23
SIG_EXPR_LIST_DECL_SINGLE(A1, SD1WP, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A1, SDA13, I2C13, I2C13_DESC);
PIN_DECL_2(A1, GPIOC7, SD1WP, SDA13);

FUNC_GROUP_DECL(I2C13, B2, A1);
FUNC_GROUP_DECL(SD1, C4, B3, A2, E5, D4, C3, B2, A1);

#define SD2_DESC	SIG_DESC_SET(SCU90, 1)
#define GPID_DESC       SIG_DESC_SET(HW_STRAP1, 21)
#define GPID0_DESC	SIG_DESC_SET(SCU8C, 8)

#define A18 24
SIG_EXPR_LIST_DECL_SINGLE(A18, SD2CLK, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID0IN, GPID0, GPID0_DESC);
SIG_EXPR_DECL_SINGLE(GPID0IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(A18, GPID0IN, GPID0, GPID);
PIN_DECL_2(A18, GPIOD0, SD2CLK, GPID0IN);

#define D16 25
SIG_EXPR_LIST_DECL_SINGLE(D16, SD2CMD, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID0OUT, GPID0, GPID0_DESC);
SIG_EXPR_DECL_SINGLE(GPID0OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(D16, GPID0OUT, GPID0, GPID);
PIN_DECL_2(D16, GPIOD1, SD2CMD, GPID0OUT);

FUNC_GROUP_DECL(GPID0, A18, D16);

#define GPID2_DESC	SIG_DESC_SET(SCU8C, 9)

#define B17 26
SIG_EXPR_LIST_DECL_SINGLE(B17, SD2DAT0, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID2IN, GPID2, GPID2_DESC);
SIG_EXPR_DECL_SINGLE(GPID2IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(B17, GPID2IN, GPID2, GPID);
PIN_DECL_2(B17, GPIOD2, SD2DAT0, GPID2IN);

#define A17 27
SIG_EXPR_LIST_DECL_SINGLE(A17, SD2DAT1, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID2OUT, GPID2, GPID2_DESC);
SIG_EXPR_DECL_SINGLE(GPID2OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(A17, GPID2OUT, GPID2, GPID);
PIN_DECL_2(A17, GPIOD3, SD2DAT1, GPID2OUT);

FUNC_GROUP_DECL(GPID2, B17, A17);

#define GPID4_DESC	SIG_DESC_SET(SCU8C, 10)

#define C16 28
SIG_EXPR_LIST_DECL_SINGLE(C16, SD2DAT2, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID4IN, GPID4, GPID4_DESC);
SIG_EXPR_DECL_SINGLE(GPID4IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(C16, GPID4IN, GPID4, GPID);
PIN_DECL_2(C16, GPIOD4, SD2DAT2, GPID4IN);

#define B16 29
SIG_EXPR_LIST_DECL_SINGLE(B16, SD2DAT3, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID4OUT, GPID4, GPID4_DESC);
SIG_EXPR_DECL_SINGLE(GPID4OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(B16, GPID4OUT, GPID4, GPID);
PIN_DECL_2(B16, GPIOD5, SD2DAT3, GPID4OUT);

FUNC_GROUP_DECL(GPID4, C16, B16);

#define GPID6_DESC	SIG_DESC_SET(SCU8C, 11)

#define A16 30
SIG_EXPR_LIST_DECL_SINGLE(A16, SD2CD, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID6IN, GPID6, GPID6_DESC);
SIG_EXPR_DECL_SINGLE(GPID6IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(A16, GPID6IN, GPID6, GPID);
PIN_DECL_2(A16, GPIOD6, SD2CD, GPID6IN);

#define E15 31
SIG_EXPR_LIST_DECL_SINGLE(E15, SD2WP, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID6OUT, GPID6, GPID6_DESC);
SIG_EXPR_DECL_SINGLE(GPID6OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(E15, GPID6OUT, GPID6, GPID);
PIN_DECL_2(E15, GPIOD7, SD2WP, GPID6OUT);

FUNC_GROUP_DECL(GPID6, A16, E15);
FUNC_GROUP_DECL(SD2, A18, D16, B17, A17, C16, B16, A16, E15);
FUNC_GROUP_DECL(GPID, A18, D16, B17, A17, C16, B16, A16, E15);

#define GPIE_DESC       SIG_DESC_SET(HW_STRAP1, 22)
#define GPIE0_DESC      SIG_DESC_SET(SCU8C, 12)
#define GPIE2_DESC      SIG_DESC_SET(SCU8C, 13)
#define GPIE4_DESC      SIG_DESC_SET(SCU8C, 14)
#define GPIE6_DESC      SIG_DESC_SET(SCU8C, 15)

#define D15 32
SIG_EXPR_LIST_DECL_SINGLE(D15, NCTS3, NCTS3, SIG_DESC_SET(SCU80, 16));
SIG_EXPR_DECL_SINGLE(GPIE0IN, GPIE0, GPIE0_DESC);
SIG_EXPR_DECL_SINGLE(GPIE0IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(D15, GPIE0IN, GPIE0, GPIE);
PIN_DECL_2(D15, GPIOE0, NCTS3, GPIE0IN);

FUNC_GROUP_DECL(NCTS3, D15);

#define C15 33
SIG_EXPR_LIST_DECL_SINGLE(C15, NDCD3, NDCD3, SIG_DESC_SET(SCU80, 17));
SIG_EXPR_DECL_SINGLE(GPIE0OUT, GPIE0, GPIE0_DESC);
SIG_EXPR_DECL_SINGLE(GPIE0OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(C15, GPIE0OUT, GPIE0, GPIE);
PIN_DECL_2(C15, GPIOE1, NDCD3, GPIE0OUT);

FUNC_GROUP_DECL(NDCD3, C15);
FUNC_GROUP_DECL(GPIE0, D15, C15);

#define B15 34
SIG_EXPR_LIST_DECL_SINGLE(B15, NDSR3, NDSR3, SIG_DESC_SET(SCU80, 18));
SIG_EXPR_DECL_SINGLE(GPIE2IN, GPIE2, GPIE2_DESC);
SIG_EXPR_DECL_SINGLE(GPIE2IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(B15, GPIE2IN, GPIE2, GPIE);
PIN_DECL_2(B15, GPIOE2, NDSR3, GPIE2IN);

FUNC_GROUP_DECL(NDSR3, B15);

#define A15 35
SIG_EXPR_LIST_DECL_SINGLE(A15, NRI3, NRI3, SIG_DESC_SET(SCU80, 19));
SIG_EXPR_DECL_SINGLE(GPIE2OUT, GPIE2, GPIE2_DESC);
SIG_EXPR_DECL_SINGLE(GPIE2OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(A15, GPIE2OUT, GPIE2, GPIE);
PIN_DECL_2(A15, GPIOE3, NRI3, GPIE2OUT);

FUNC_GROUP_DECL(NRI3, A15);
FUNC_GROUP_DECL(GPIE2, B15, A15);

#define E14 36
SIG_EXPR_LIST_DECL_SINGLE(E14, NDTR3, NDTR3, SIG_DESC_SET(SCU80, 20));
SIG_EXPR_DECL_SINGLE(GPIE4IN, GPIE4, GPIE4_DESC);
SIG_EXPR_DECL_SINGLE(GPIE4IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(E14, GPIE4IN, GPIE4, GPIE);
PIN_DECL_2(E14, GPIOE4, NDTR3, GPIE4IN);

FUNC_GROUP_DECL(NDTR3, E14);

#define D14 37
SIG_EXPR_LIST_DECL_SINGLE(D14, NRTS3, NRTS3, SIG_DESC_SET(SCU80, 21));
SIG_EXPR_DECL_SINGLE(GPIE4OUT, GPIE4, GPIE4_DESC);
SIG_EXPR_DECL_SINGLE(GPIE4OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(D14, GPIE4OUT, GPIE4, GPIE);
PIN_DECL_2(D14, GPIOE5, NRTS3, GPIE4OUT);

FUNC_GROUP_DECL(NRTS3, D14);
FUNC_GROUP_DECL(GPIE4, E14, D14);

#define C14 38
SIG_EXPR_LIST_DECL_SINGLE(C14, TXD3, TXD3, SIG_DESC_SET(SCU80, 22));
SIG_EXPR_DECL_SINGLE(GPIE6IN, GPIE6, GPIE6_DESC);
SIG_EXPR_DECL_SINGLE(GPIE6IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(C14, GPIE6IN, GPIE6, GPIE);
PIN_DECL_2(C14, GPIOE6, TXD3, GPIE6IN);

FUNC_GROUP_DECL(TXD3, C14);

#define B14 39
SIG_EXPR_LIST_DECL_SINGLE(B14, RXD3, RXD3, SIG_DESC_SET(SCU80, 23));
SIG_EXPR_DECL_SINGLE(GPIE6OUT, GPIE6, GPIE6_DESC);
SIG_EXPR_DECL_SINGLE(GPIE6OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(B14, GPIE6OUT, GPIE6, GPIE);
PIN_DECL_2(B14, GPIOE7, RXD3, GPIE6OUT);

FUNC_GROUP_DECL(RXD3, B14);
FUNC_GROUP_DECL(GPIE6, C14, B14);

#define D18 40
SSSF_PIN_DECL(D18, GPIOF0, NCTS4, SIG_DESC_SET(SCU80, 24));

#define ACPI_DESC       SIG_DESC_BIT(HW_STRAP1, 19, 0)

#define B19 41
SIG_EXPR_LIST_DECL_SINGLE(B19, NDCD4, NDCD4, SIG_DESC_SET(SCU80, 25));
SIG_EXPR_DECL_SINGLE(SIOPBI, SIOPBI, SIG_DESC_SET(SCUA4, 12));
SIG_EXPR_DECL_SINGLE(SIOPBI, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(B19, SIOPBI, SIOPBI, ACPI);
PIN_DECL_2(B19, GPIOF1, NDCD4, SIOPBI);
FUNC_GROUP_DECL(NDCD4, B19);
FUNC_GROUP_DECL(SIOPBI, B19);

#define A20 42
SIG_EXPR_LIST_DECL_SINGLE(A20, NDSR4, NDSR4, SIG_DESC_SET(SCU80, 26));
SIG_EXPR_DECL_SINGLE(SIOPWRGD, SIOPWRGD, SIG_DESC_SET(SCUA4, 12));
SIG_EXPR_DECL_SINGLE(SIOPWRGD, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(A20, SIOPWRGD, SIOPWRGD, ACPI);
PIN_DECL_2(A20, GPIOF2, NDSR4, SIOPWRGD);
FUNC_GROUP_DECL(NDSR4, A20);
FUNC_GROUP_DECL(SIOPWRGD, A20);

#define D17 43
SIG_EXPR_LIST_DECL_SINGLE(D17, NRI4, NRI4, SIG_DESC_SET(SCU80, 27));
SIG_EXPR_DECL_SINGLE(SIOPBO, SIOPBO, SIG_DESC_SET(SCUA4, 14));
SIG_EXPR_DECL_SINGLE(SIOPBO, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(D17, SIOPBO, SIOPBO, ACPI);
PIN_DECL_2(D17, GPIOF3, NRI4, SIOPBO);
FUNC_GROUP_DECL(NRI4, D17);
FUNC_GROUP_DECL(SIOPBO, D17);

#define B18 44
SSSF_PIN_DECL(B18, GPIOF4, NDTR4, SIG_DESC_SET(SCU80, 28));

#define A19 45
SIG_EXPR_LIST_DECL_SINGLE(A19, NDTS4, NDTS4, SIG_DESC_SET(SCU80, 29));
SIG_EXPR_DECL_SINGLE(SIOSCI, SIOSCI, SIG_DESC_SET(SCUA4, 15));
SIG_EXPR_DECL_SINGLE(SIOSCI, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(A19, SIOSCI, SIOSCI, ACPI);
PIN_DECL_2(A19, GPIOF5, NDTS4, SIOSCI);
FUNC_GROUP_DECL(NDTS4, A19);
FUNC_GROUP_DECL(SIOSCI, A19);

#define E16 46
SSSF_PIN_DECL(E16, GPIOF6, TXD4, SIG_DESC_SET(SCU80, 30));

#define C17 47
SSSF_PIN_DECL(C17, GPIOF7, RXD4, SIG_DESC_SET(SCU80, 31));

#define A14 48
SSSF_PIN_DECL(A14, GPIOG0, SGPSCK, SIG_DESC_SET(SCU84, 0));

#define E13 49
SSSF_PIN_DECL(E13, GPIOG1, SGPSLD, SIG_DESC_SET(SCU84, 1));

#define D13 50
SSSF_PIN_DECL(D13, GPIOG2, SGPSI0, SIG_DESC_SET(SCU84, 2));

#define C13 51
SSSF_PIN_DECL(C13, GPIOG3, SGPSI1, SIG_DESC_SET(SCU84, 3));

#define B13 52
SIG_EXPR_LIST_DECL_SINGLE(B13, OSCCLK, OSCCLK, SIG_DESC_SET(SCU2C, 1));
SIG_EXPR_LIST_DECL_SINGLE(B13, WDTRST1, WDTRST1, SIG_DESC_SET(SCU84, 4));
PIN_DECL_2(B13, GPIOG4, OSCCLK, WDTRST1);

FUNC_GROUP_DECL(OSCCLK, B13);
FUNC_GROUP_DECL(WDTRST1, B13);

#define Y21 53
SIG_EXPR_LIST_DECL_SINGLE(Y21, USBCKI, USBCKI, SIG_DESC_SET(HW_STRAP1, 23));
SIG_EXPR_LIST_DECL_SINGLE(Y21, WDTRST2, WDTRST2, SIG_DESC_SET(SCU84, 5));
PIN_DECL_2(Y21, GPIOG5, USBCKI, WDTRST2);

FUNC_GROUP_DECL(USBCKI, Y21);
FUNC_GROUP_DECL(WDTRST2, Y21);

#define AA22 54
SSSF_PIN_DECL(AA22, GPIOG6, FLBUSY, SIG_DESC_SET(SCU84, 6));

#define U18 55
SSSF_PIN_DECL(U18, GPIOG7, FLWP, SIG_DESC_SET(SCU84, 7));

#define UART6_DESC	SIG_DESC_SET(SCU90, 7)
#define ROM16_DESC	SIG_DESC_SET(SCU90, 6)
#define FLASH_WIDE	SIG_DESC_SET(HW_STRAP1, 4)
#define BOOT_SRC_NOR	{ ASPEED_IP_SCU, HW_STRAP1, GENMASK(1, 0), 0, 0 }

#define A8 56
SIG_EXPR_DECL_SINGLE(ROMD8, ROM16, ROM16_DESC);
SIG_EXPR_DECL_SINGLE(ROMD8, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(A8, ROMD8, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(A8, NCTS6, NCTS6, UART6_DESC);
PIN_DECL_2(A8, GPIOH0, ROMD8, NCTS6);

#define C7 57
SIG_EXPR_DECL_SINGLE(ROMD9, ROM16, ROM16_DESC);
SIG_EXPR_DECL_SINGLE(ROMD9, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(C7, ROMD9, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(C7, NDCD6, NDCD6, UART6_DESC);
PIN_DECL_2(C7, GPIOH1, ROMD9, NDCD6);

#define B7 58
SIG_EXPR_DECL_SINGLE(ROMD10, ROM16, ROM16_DESC);
SIG_EXPR_DECL_SINGLE(ROMD10, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(B7, ROMD10, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(B7, NDSR6, NDSR6, UART6_DESC);
PIN_DECL_2(B7, GPIOH2, ROMD10, NDSR6);

#define A7 59
SIG_EXPR_DECL_SINGLE(ROMD11, ROM16, ROM16_DESC);
SIG_EXPR_DECL_SINGLE(ROMD11, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(A7, ROMD11, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(A7, NRI6, NRI6, UART6_DESC);
PIN_DECL_2(A7, GPIOH3, ROMD11, NRI6);

#define D7 60
SIG_EXPR_DECL_SINGLE(ROMD12, ROM16, ROM16_DESC);
SIG_EXPR_DECL_SINGLE(ROMD12, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(D7, ROMD12, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(D7, NDTR6, NDTR6, UART6_DESC);
PIN_DECL_2(D7, GPIOH4, ROMD12, NDTR6);

#define B6 61
SIG_EXPR_DECL_SINGLE(ROMD13, ROM16, ROM16_DESC);
SIG_EXPR_DECL_SINGLE(ROMD13, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(B6, ROMD13, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(B6, NRTS6, NRTS6, UART6_DESC);
PIN_DECL_2(B6, GPIOH5, ROMD13, NRTS6);

#define A6 62
SIG_EXPR_DECL_SINGLE(ROMD14, ROM16, ROM16_DESC);
SIG_EXPR_DECL_SINGLE(ROMD14, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(A6, ROMD14, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(A6, TXD6, TXD6, UART6_DESC);
PIN_DECL_2(A6, GPIOH6, ROMD14, TXD6);

#define E7 63
SIG_EXPR_DECL_SINGLE(ROMD15, ROM16, ROM16_DESC);
SIG_EXPR_DECL_SINGLE(ROMD15, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(E7, ROMD15, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(E7, RXD6, RXD6, UART6_DESC);
PIN_DECL_2(E7, GPIOH7, ROMD15, RXD6);

FUNC_GROUP_DECL(UART6, A8, C7, B7, A7, D7, B6, A6, E7);

#define SPI1_DESC \
	{ ASPEED_IP_SCU, HW_STRAP1, GENMASK(13, 12), 1, 0 }
#define SPI1DEBUG_DESC \
	{ ASPEED_IP_SCU, HW_STRAP1, GENMASK(13, 12), 2, 0 }
#define SPI1PASSTHRU_DESC \
	{ ASPEED_IP_SCU, HW_STRAP1, GENMASK(13, 12), 3, 0 }

#define C22 64
SIG_EXPR_DECL_SINGLE(SYSCS, SPI1DEBUG, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SYSCS, SPI1PASSTHRU, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL_DUAL(C22, SYSCS, SPI1DEBUG, SPI1PASSTHRU);
PIN_DECL_1(C22, GPIOI0, SYSCS);

#define G18 65
SIG_EXPR_DECL_SINGLE(SYSCK, SPI1DEBUG, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SYSCK, SPI1PASSTHRU, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL_DUAL(G18, SYSCK, SPI1DEBUG, SPI1PASSTHRU);
PIN_DECL_1(G18, GPIOI1, SYSCK);

#define D19 66
SIG_EXPR_DECL_SINGLE(SYSDO, SPI1DEBUG, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SYSDO, SPI1PASSTHRU, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL_DUAL(D19, SYSDO, SPI1DEBUG, SPI1PASSTHRU);
PIN_DECL_1(D19, GPIOI2, SYSDO);

#define C20 67
SIG_EXPR_DECL_SINGLE(SYSDI, SPI1DEBUG, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SYSDI, SPI1PASSTHRU, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL_DUAL(C20, SYSDI, SPI1DEBUG, SPI1PASSTHRU);
PIN_DECL_1(C20, GPIOI3, SYSDI);

#define VB_DESC	SIG_DESC_SET(HW_STRAP1, 5)

#define B22 68
SIG_EXPR_DECL_SINGLE(SPI1CS0, SPI1, SPI1_DESC);
SIG_EXPR_DECL_SINGLE(SPI1CS0, SPI1DEBUG, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SPI1CS0, SPI1PASSTHRU, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL(SPI1CS0, SPI1,
			    SIG_EXPR_PTR(SPI1CS0, SPI1),
			    SIG_EXPR_PTR(SPI1CS0, SPI1DEBUG),
			    SIG_EXPR_PTR(SPI1CS0, SPI1PASSTHRU));
SIG_EXPR_LIST_ALIAS(B22, SPI1CS0, SPI1);
SIG_EXPR_LIST_DECL_SINGLE(B22, VBCS, VGABIOS_ROM, VB_DESC);
PIN_DECL_2(B22, GPIOI4, SPI1CS0, VBCS);

#define G19 69
SIG_EXPR_DECL_SINGLE(SPI1CK, SPI1, SPI1_DESC);
SIG_EXPR_DECL_SINGLE(SPI1CK, SPI1DEBUG, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SPI1CK, SPI1PASSTHRU, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL(SPI1CK, SPI1,
			    SIG_EXPR_PTR(SPI1CK, SPI1),
			    SIG_EXPR_PTR(SPI1CK, SPI1DEBUG),
			    SIG_EXPR_PTR(SPI1CK, SPI1PASSTHRU));
SIG_EXPR_LIST_ALIAS(G19, SPI1CK, SPI1);
SIG_EXPR_LIST_DECL_SINGLE(G19, VBCK, VGABIOS_ROM, VB_DESC);
PIN_DECL_2(G19, GPIOI5, SPI1CK, VBCK);

#define C18 70
SIG_EXPR_DECL_SINGLE(SPI1DO, SPI1, SPI1_DESC);
SIG_EXPR_DECL_SINGLE(SPI1DO, SPI1DEBUG, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SPI1DO, SPI1PASSTHRU, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL(SPI1DO, SPI1,
			    SIG_EXPR_PTR(SPI1DO, SPI1),
			    SIG_EXPR_PTR(SPI1DO, SPI1DEBUG),
			    SIG_EXPR_PTR(SPI1DO, SPI1PASSTHRU));
SIG_EXPR_LIST_ALIAS(C18, SPI1DO, SPI1);
SIG_EXPR_LIST_DECL_SINGLE(C18, VBDO, VGABIOS_ROM, VB_DESC);
PIN_DECL_2(C18, GPIOI6, SPI1DO, VBDO);

#define E20 71
SIG_EXPR_DECL_SINGLE(SPI1DI, SPI1, SPI1_DESC);
SIG_EXPR_DECL_SINGLE(SPI1DI, SPI1DEBUG, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SPI1DI, SPI1PASSTHRU, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL(SPI1DI, SPI1,
			    SIG_EXPR_PTR(SPI1DI, SPI1),
			    SIG_EXPR_PTR(SPI1DI, SPI1DEBUG),
			    SIG_EXPR_PTR(SPI1DI, SPI1PASSTHRU));
SIG_EXPR_LIST_ALIAS(E20, SPI1DI, SPI1);
SIG_EXPR_LIST_DECL_SINGLE(E20, VBDI, VGABIOS_ROM, VB_DESC);
PIN_DECL_2(E20, GPIOI7, SPI1DI, VBDI);

FUNC_GROUP_DECL(SPI1, B22, G19, C18, E20);
FUNC_GROUP_DECL(SPI1DEBUG, C22, G18, D19, C20, B22, G19, C18, E20);
FUNC_GROUP_DECL(SPI1PASSTHRU, C22, G18, D19, C20, B22, G19, C18, E20);
FUNC_GROUP_DECL(VGABIOS_ROM, B22, G19, C18, E20);

#define J5 72
SSSF_PIN_DECL(J5, GPIOJ0, SGPMCK, SIG_DESC_SET(SCU84, 8));

#define J4 73
SSSF_PIN_DECL(J4, GPIOJ1, SGPMLD, SIG_DESC_SET(SCU84, 9));

#define K5 74
SSSF_PIN_DECL(K5, GPIOJ2, SGPMO, SIG_DESC_SET(SCU84, 10));

#define J3 75
SSSF_PIN_DECL(J3, GPIOJ3, SGPMI, SIG_DESC_SET(SCU84, 11));

#define T4 76
SSSF_PIN_DECL(T4, GPIOJ4, VGAHS, SIG_DESC_SET(SCU84, 12));

#define U2 77
SSSF_PIN_DECL(U2, GPIOJ5, VGAVS, SIG_DESC_SET(SCU84, 13));

#define T2 78
SSSF_PIN_DECL(T2, GPIOJ6, DDCCLK, SIG_DESC_SET(SCU84, 14));

#define T1 79
SSSF_PIN_DECL(T1, GPIOJ7, DDCDAT, SIG_DESC_SET(SCU84, 15));

#define I2C5_DESC	SIG_DESC_SET(SCU90, 18)

#define E3 80
SIG_EXPR_LIST_DECL_SINGLE(E3, SCL5, I2C5, I2C5_DESC);
PIN_DECL_1(E3, GPIOK0, SCL5);

#define D2 81
SIG_EXPR_LIST_DECL_SINGLE(D2, SDA5, I2C5, I2C5_DESC);
PIN_DECL_1(D2, GPIOK1, SDA5);

FUNC_GROUP_DECL(I2C5, E3, D2);

#define I2C6_DESC	SIG_DESC_SET(SCU90, 19)

#define C1 82
SIG_EXPR_LIST_DECL_SINGLE(C1, SCL6, I2C6, I2C6_DESC);
PIN_DECL_1(C1, GPIOK2, SCL6);

#define F4 83
SIG_EXPR_LIST_DECL_SINGLE(F4, SDA6, I2C6, I2C6_DESC);
PIN_DECL_1(F4, GPIOK3, SDA6);

FUNC_GROUP_DECL(I2C6, C1, F4);

#define I2C7_DESC	SIG_DESC_SET(SCU90, 20)

#define E2 84
SIG_EXPR_LIST_DECL_SINGLE(E2, SCL7, I2C7, I2C7_DESC);
PIN_DECL_1(E2, GPIOK4, SCL7);

#define D1 85
SIG_EXPR_LIST_DECL_SINGLE(D1, SDA7, I2C7, I2C7_DESC);
PIN_DECL_1(D1, GPIOK5, SDA7);

FUNC_GROUP_DECL(I2C7, E2, D1);

#define I2C8_DESC	SIG_DESC_SET(SCU90, 21)

#define G5 86
SIG_EXPR_LIST_DECL_SINGLE(G5, SCL8, I2C8, I2C8_DESC);
PIN_DECL_1(G5, GPIOK6, SCL8);

#define F3 87
SIG_EXPR_LIST_DECL_SINGLE(F3, SDA8, I2C8, I2C8_DESC);
PIN_DECL_1(F3, GPIOK7, SDA8);

FUNC_GROUP_DECL(I2C8, G5, F3);

#define U1 88
SSSF_PIN_DECL(U1, GPIOL0, NCTS1, SIG_DESC_SET(SCU84, 16));

#define VPI18_DESC	{ ASPEED_IP_SCU, SCU90, GENMASK(5, 4), 1, 0 }
#define VPI24_DESC	{ ASPEED_IP_SCU, SCU90, GENMASK(5, 4), 2, 0 }
#define VPI30_DESC	{ ASPEED_IP_SCU, SCU90, GENMASK(5, 4), 3, 0 }

#define T5 89
#define T5_DESC         SIG_DESC_SET(SCU84, 17)
SIG_EXPR_DECL_SINGLE(VPIDE, VPI18, VPI18_DESC, T5_DESC);
SIG_EXPR_DECL_SINGLE(VPIDE, VPI24, VPI24_DESC, T5_DESC);
SIG_EXPR_DECL_SINGLE(VPIDE, VPI30, VPI30_DESC, T5_DESC);
SIG_EXPR_LIST_DECL(VPIDE, VPI,
		   SIG_EXPR_PTR(VPIDE, VPI18),
		   SIG_EXPR_PTR(VPIDE, VPI24),
		   SIG_EXPR_PTR(VPIDE, VPI30));
SIG_EXPR_LIST_ALIAS(T5, VPIDE, VPI);
SIG_EXPR_LIST_DECL_SINGLE(T5, NDCD1, NDCD1, T5_DESC);
PIN_DECL_2(T5, GPIOL1, VPIDE, NDCD1);
FUNC_GROUP_DECL(NDCD1, T5);

#define U3 90
#define U3_DESC         SIG_DESC_SET(SCU84, 18)
SIG_EXPR_DECL_SINGLE(VPIODD, VPI18, VPI18_DESC, U3_DESC);
SIG_EXPR_DECL_SINGLE(VPIODD, VPI24, VPI24_DESC, U3_DESC);
SIG_EXPR_DECL_SINGLE(VPIODD, VPI30, VPI30_DESC, U3_DESC);
SIG_EXPR_LIST_DECL(VPIODD, VPI,
		SIG_EXPR_PTR(VPIODD, VPI18),
		SIG_EXPR_PTR(VPIODD, VPI24),
		SIG_EXPR_PTR(VPIODD, VPI30));
SIG_EXPR_LIST_ALIAS(U3, VPIODD, VPI);
SIG_EXPR_LIST_DECL_SINGLE(U3, NDSR1, NDSR1, U3_DESC);
PIN_DECL_2(U3, GPIOL2, VPIODD, NDSR1);
FUNC_GROUP_DECL(NDSR1, U3);

#define V1 91
#define V1_DESC         SIG_DESC_SET(SCU84, 19)
SIG_EXPR_DECL_SINGLE(VPIHS, VPI18, VPI18_DESC, V1_DESC);
SIG_EXPR_DECL_SINGLE(VPIHS, VPI24, VPI24_DESC, V1_DESC);
SIG_EXPR_DECL_SINGLE(VPIHS, VPI30, VPI30_DESC, V1_DESC);
SIG_EXPR_LIST_DECL(VPIHS, VPI,
		SIG_EXPR_PTR(VPIHS, VPI18),
		SIG_EXPR_PTR(VPIHS, VPI24),
		SIG_EXPR_PTR(VPIHS, VPI30));
SIG_EXPR_LIST_ALIAS(V1, VPIHS, VPI);
SIG_EXPR_LIST_DECL_SINGLE(V1, NRI1, NRI1, V1_DESC);
PIN_DECL_2(V1, GPIOL3, VPIHS, NRI1);
FUNC_GROUP_DECL(NRI1, V1);

#define U4 92
#define U4_DESC         SIG_DESC_SET(SCU84, 20)
SIG_EXPR_DECL_SINGLE(VPIVS, VPI18, VPI18_DESC, U4_DESC);
SIG_EXPR_DECL_SINGLE(VPIVS, VPI24, VPI24_DESC, U4_DESC);
SIG_EXPR_DECL_SINGLE(VPIVS, VPI30, VPI30_DESC, U4_DESC);
SIG_EXPR_LIST_DECL(VPIVS, VPI,
		SIG_EXPR_PTR(VPIVS, VPI18),
		SIG_EXPR_PTR(VPIVS, VPI24),
		SIG_EXPR_PTR(VPIVS, VPI30));
SIG_EXPR_LIST_ALIAS(U4, VPIVS, VPI);
SIG_EXPR_LIST_DECL_SINGLE(U4, NDTR1, NDTR1, U4_DESC);
PIN_DECL_2(U4, GPIOL4, VPIVS, NDTR1);
FUNC_GROUP_DECL(NDTR1, U4);

#define V2 93
#define V2_DESC         SIG_DESC_SET(SCU84, 21)
SIG_EXPR_DECL_SINGLE(VPICLK, VPI18, VPI18_DESC, V2_DESC);
SIG_EXPR_DECL_SINGLE(VPICLK, VPI24, VPI24_DESC, V2_DESC);
SIG_EXPR_DECL_SINGLE(VPICLK, VPI30, VPI30_DESC, V2_DESC);
SIG_EXPR_LIST_DECL(VPICLK, VPI,
		SIG_EXPR_PTR(VPICLK, VPI18),
		SIG_EXPR_PTR(VPICLK, VPI24),
		SIG_EXPR_PTR(VPICLK, VPI30));
SIG_EXPR_LIST_ALIAS(V2, VPICLK, VPI);
SIG_EXPR_LIST_DECL_SINGLE(V2, NRTS1, NRTS1, V2_DESC);
PIN_DECL_2(V2, GPIOL5, VPICLK, NRTS1);
FUNC_GROUP_DECL(NRTS1, V2);

#define W1 94
#define W1_DESC         SIG_DESC_SET(SCU84, 22)
SIG_EXPR_LIST_DECL_SINGLE(W1, VPIB0, VPI30, VPI30_DESC, W1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(W1, TXD1, TXD1, W1_DESC);
PIN_DECL_2(W1, GPIOL6, VPIB0, TXD1);
FUNC_GROUP_DECL(TXD1, W1);

#define U5 95
#define U5_DESC         SIG_DESC_SET(SCU84, 23)
SIG_EXPR_LIST_DECL_SINGLE(U5, VPIB1, VPI30, VPI30_DESC, U5_DESC);
SIG_EXPR_LIST_DECL_SINGLE(U5, RXD1, RXD1, U5_DESC);
PIN_DECL_2(U5, GPIOL7, VPIB1, RXD1);
FUNC_GROUP_DECL(RXD1, U5);

#define V3 96
#define V3_DESC		SIG_DESC_SET(SCU84, 24)
SIG_EXPR_DECL_SINGLE(VPIOB2, VPI18, VPI18_DESC, V3_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB2, VPI24, VPI24_DESC, V3_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB2, VPI30, VPI30_DESC, V3_DESC);
SIG_EXPR_LIST_DECL(VPIOB2, VPI,
		SIG_EXPR_PTR(VPIOB2, VPI18),
		SIG_EXPR_PTR(VPIOB2, VPI24),
		SIG_EXPR_PTR(VPIOB2, VPI30));
SIG_EXPR_LIST_ALIAS(V3, VPIOB2, VPI);
SIG_EXPR_LIST_DECL_SINGLE(V3, NCTS2, NCTS2, V3_DESC);
PIN_DECL_2(V3, GPIOM0, VPIOB2, NCTS2);
FUNC_GROUP_DECL(NCTS2, V3);

#define W2 97
#define W2_DESC		SIG_DESC_SET(SCU84, 25)
SIG_EXPR_DECL_SINGLE(VPIOB3, VPI18, VPI18_DESC, W2_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB3, VPI24, VPI24_DESC, W2_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB3, VPI30, VPI30_DESC, W2_DESC);
SIG_EXPR_LIST_DECL(VPIOB3, VPI,
		SIG_EXPR_PTR(VPIOB3, VPI18),
		SIG_EXPR_PTR(VPIOB3, VPI24),
		SIG_EXPR_PTR(VPIOB3, VPI30));
SIG_EXPR_LIST_ALIAS(W2, VPIOB3, VPI);
SIG_EXPR_LIST_DECL_SINGLE(W2, NDCD2, NDCD2, W2_DESC);
PIN_DECL_2(W2, GPIOM1, VPIOB3, NDCD2);
FUNC_GROUP_DECL(NDCD2, W2);

#define Y1 98
#define Y1_DESC		SIG_DESC_SET(SCU84, 26)
SIG_EXPR_DECL_SINGLE(VPIOB4, VPI18, VPI18_DESC, Y1_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB4, VPI24, VPI24_DESC, Y1_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB4, VPI30, VPI30_DESC, Y1_DESC);
SIG_EXPR_LIST_DECL(VPIOB4, VPI,
		SIG_EXPR_PTR(VPIOB4, VPI18),
		SIG_EXPR_PTR(VPIOB4, VPI24),
		SIG_EXPR_PTR(VPIOB4, VPI30));
SIG_EXPR_LIST_ALIAS(Y1, VPIOB4, VPI);
SIG_EXPR_LIST_DECL_SINGLE(Y1, NDSR2, NDSR2, Y1_DESC);
PIN_DECL_2(Y1, GPIOM2, VPIOB4, NDSR2);
FUNC_GROUP_DECL(NDSR2, Y1);

#define V4 99
#define V4_DESC		SIG_DESC_SET(SCU84, 27)
SIG_EXPR_DECL_SINGLE(VPIOB5, VPI18, VPI18_DESC, V4_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB5, VPI24, VPI24_DESC, V4_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB5, VPI30, VPI30_DESC, V4_DESC);
SIG_EXPR_LIST_DECL(VPIOB5, VPI,
		SIG_EXPR_PTR(VPIOB5, VPI18),
		SIG_EXPR_PTR(VPIOB5, VPI24),
		SIG_EXPR_PTR(VPIOB5, VPI30));
SIG_EXPR_LIST_ALIAS(V4, VPIOB5, VPI);
SIG_EXPR_LIST_DECL_SINGLE(V4, NRI2, NRI2, V4_DESC);
PIN_DECL_2(V4, GPIOM3, VPIOB5, NRI2);
FUNC_GROUP_DECL(NRI2, V4);

#define W3 100
#define W3_DESC		SIG_DESC_SET(SCU84, 28)
SIG_EXPR_DECL_SINGLE(VPIOB6, VPI18, VPI18_DESC, W3_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB6, VPI24, VPI24_DESC, W3_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB6, VPI30, VPI30_DESC, W3_DESC);
SIG_EXPR_LIST_DECL(VPIOB6, VPI,
		SIG_EXPR_PTR(VPIOB6, VPI18),
		SIG_EXPR_PTR(VPIOB6, VPI24),
		SIG_EXPR_PTR(VPIOB6, VPI30));
SIG_EXPR_LIST_ALIAS(W3, VPIOB6, VPI);
SIG_EXPR_LIST_DECL_SINGLE(W3, NDTR2, NDTR2, W3_DESC);
PIN_DECL_2(W3, GPIOM4, VPIOB6, NDTR2);
FUNC_GROUP_DECL(NDTR2, W3);

#define Y2 101
#define Y2_DESC		SIG_DESC_SET(SCU84, 29)
SIG_EXPR_DECL_SINGLE(VPIOB7, VPI18, VPI18_DESC, Y2_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB7, VPI24, VPI24_DESC, Y2_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB7, VPI30, VPI30_DESC, Y2_DESC);
SIG_EXPR_LIST_DECL(VPIOB7, VPI,
		SIG_EXPR_PTR(VPIOB7, VPI18),
		SIG_EXPR_PTR(VPIOB7, VPI24),
		SIG_EXPR_PTR(VPIOB7, VPI30));
SIG_EXPR_LIST_ALIAS(Y2, VPIOB7, VPI);
SIG_EXPR_LIST_DECL_SINGLE(Y2, NRTS2, NRTS2, Y2_DESC);
PIN_DECL_2(Y2, GPIOM5, VPIOB7, NRTS2);
FUNC_GROUP_DECL(NRTS2, Y2);

#define AA1 102
#define AA1_DESC	SIG_DESC_SET(SCU84, 30)
SIG_EXPR_DECL_SINGLE(VPIOB8, VPI18, VPI18_DESC, AA1_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB8, VPI24, VPI24_DESC, AA1_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB8, VPI30, VPI30_DESC, AA1_DESC);
SIG_EXPR_LIST_DECL(VPIOB8, VPI,
		SIG_EXPR_PTR(VPIOB8, VPI18),
		SIG_EXPR_PTR(VPIOB8, VPI24),
		SIG_EXPR_PTR(VPIOB8, VPI30));
SIG_EXPR_LIST_ALIAS(AA1, VPIOB8, VPI);
SIG_EXPR_LIST_DECL_SINGLE(AA1, TXD2, TXD2, AA1_DESC);
PIN_DECL_2(AA1, GPIOM6, VPIOB8, TXD2);
FUNC_GROUP_DECL(TXD2, AA1);

#define V5 103
#define V5_DESC		SIG_DESC_SET(SCU84, 31)
SIG_EXPR_DECL_SINGLE(VPIOB9, VPI18, VPI18_DESC, V5_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB9, VPI24, VPI24_DESC, V5_DESC);
SIG_EXPR_DECL_SINGLE(VPIOB9, VPI30, VPI30_DESC, V5_DESC);
SIG_EXPR_LIST_DECL(VPIOB9, VPI,
		SIG_EXPR_PTR(VPIOB9, VPI18),
		SIG_EXPR_PTR(VPIOB9, VPI24),
		SIG_EXPR_PTR(VPIOB9, VPI30));
SIG_EXPR_LIST_ALIAS(V5, VPIOB9, VPI);
SIG_EXPR_LIST_DECL_SINGLE(V5, RXD2, RXD2, V5_DESC);
PIN_DECL_2(V5, GPIOM7, VPIOB9, RXD2);
FUNC_GROUP_DECL(RXD2, V5);

#define W4 104
#define W4_DESC         SIG_DESC_SET(SCU88, 0)
SIG_EXPR_LIST_DECL_SINGLE(W4, VPIG0, VPI30, VPI30_DESC, W4_DESC);
SIG_EXPR_LIST_DECL_SINGLE(W4, PWM0, PWM0, W4_DESC);
PIN_DECL_2(W4, GPION0, VPIG0, PWM0);
FUNC_GROUP_DECL(PWM0, W4);

#define Y3 105
#define Y3_DESC         SIG_DESC_SET(SCU88, 1)
SIG_EXPR_LIST_DECL_SINGLE(Y3, VPIG1, VPI30, VPI30_DESC, Y3_DESC);
SIG_EXPR_LIST_DECL_SINGLE(Y3, PWM1, PWM1, Y3_DESC);
PIN_DECL_2(Y3, GPION1, VPIG1, PWM1);
FUNC_GROUP_DECL(PWM1, Y3);

#define AA2 106
#define AA2_DESC        SIG_DESC_SET(SCU88, 2)
SIG_EXPR_DECL_SINGLE(VPIG2, VPI18, VPI18_DESC, AA2_DESC);
SIG_EXPR_DECL_SINGLE(VPIG2, VPI24, VPI24_DESC, AA2_DESC);
SIG_EXPR_DECL_SINGLE(VPIG2, VPI30, VPI30_DESC, AA2_DESC);
SIG_EXPR_LIST_DECL(VPIG2, VPI,
		SIG_EXPR_PTR(VPIG2, VPI18),
		SIG_EXPR_PTR(VPIG2, VPI24),
		SIG_EXPR_PTR(VPIG2, VPI30));
SIG_EXPR_LIST_ALIAS(AA2, VPIG2, VPI);
SIG_EXPR_LIST_DECL_SINGLE(AA2, PWM2, PWM2, AA2_DESC);
PIN_DECL_2(AA2, GPION2, VPIG2, PWM2);
FUNC_GROUP_DECL(PWM2, AA2);

#define AB1 107
#define AB1_DESC        SIG_DESC_SET(SCU88, 3)
SIG_EXPR_DECL_SINGLE(VPIG3, VPI18, VPI18_DESC, AB1_DESC);
SIG_EXPR_DECL_SINGLE(VPIG3, VPI24, VPI24_DESC, AB1_DESC);
SIG_EXPR_DECL_SINGLE(VPIG3, VPI30, VPI30_DESC, AB1_DESC);
SIG_EXPR_LIST_DECL(VPIG3, VPI,
		SIG_EXPR_PTR(VPIG3, VPI18),
		SIG_EXPR_PTR(VPIG3, VPI24),
		SIG_EXPR_PTR(VPIG3, VPI30));
SIG_EXPR_LIST_ALIAS(AB1, VPIG3, VPI);
SIG_EXPR_LIST_DECL_SINGLE(AB1, PWM3, PWM3, AB1_DESC);
PIN_DECL_2(AB1, GPION3, VPIG3, PWM3);
FUNC_GROUP_DECL(PWM3, AB1);

#define W5 108
#define W5_DESC         SIG_DESC_SET(SCU88, 4)
SIG_EXPR_DECL_SINGLE(VPIG4, VPI18, VPI18_DESC, W5_DESC);
SIG_EXPR_DECL_SINGLE(VPIG4, VPI24, VPI24_DESC, W5_DESC);
SIG_EXPR_DECL_SINGLE(VPIG4, VPI30, VPI30_DESC, W5_DESC);
SIG_EXPR_LIST_DECL(VPIG4, VPI,
		SIG_EXPR_PTR(VPIG4, VPI18),
		SIG_EXPR_PTR(VPIG4, VPI24),
		SIG_EXPR_PTR(VPIG4, VPI30));
SIG_EXPR_LIST_ALIAS(W5, VPIG4, VPI);
SIG_EXPR_LIST_DECL_SINGLE(W5, PWM4, PWM4, W5_DESC);
PIN_DECL_2(W5, GPION4, VPIG4, PWM4);
FUNC_GROUP_DECL(PWM4, W5);

#define Y4 109
#define Y4_DESC         SIG_DESC_SET(SCU88, 5)
SIG_EXPR_DECL_SINGLE(VPIG5, VPI18, VPI18_DESC, Y4_DESC);
SIG_EXPR_DECL_SINGLE(VPIG5, VPI24, VPI24_DESC, Y4_DESC);
SIG_EXPR_DECL_SINGLE(VPIG5, VPI30, VPI30_DESC, Y4_DESC);
SIG_EXPR_LIST_DECL(VPIG5, VPI,
		SIG_EXPR_PTR(VPIG5, VPI18),
		SIG_EXPR_PTR(VPIG5, VPI24),
		SIG_EXPR_PTR(VPIG5, VPI30));
SIG_EXPR_LIST_ALIAS(Y4, VPIG5, VPI);
SIG_EXPR_LIST_DECL_SINGLE(Y4, PWM5, PWM5, Y4_DESC);
PIN_DECL_2(Y4, GPION5, VPIG5, PWM5);
FUNC_GROUP_DECL(PWM5, Y4);

#define AA3 110
#define AA3_DESC        SIG_DESC_SET(SCU88, 6)
SIG_EXPR_LIST_DECL_SINGLE(AA3, VPIG6, VPI30, VPI30_DESC, AA3_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AA3, PWM6, PWM6, AA3_DESC);
PIN_DECL_2(AA3, GPION6, VPIG6, PWM6);
FUNC_GROUP_DECL(PWM6, AA3);

#define AB2 111
#define AB2_DESC        SIG_DESC_SET(SCU88, 7)
SIG_EXPR_LIST_DECL_SINGLE(AB2, VPIG7, VPI30, VPI30_DESC, AB2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AB2, PWM7, PWM7, AB2_DESC);
PIN_DECL_2(AB2, GPION7, VPIG7, PWM7);
FUNC_GROUP_DECL(PWM7, AB2);

#define V6 112
SIG_EXPR_LIST_DECL_SINGLE(V6, VPIG8, VPI24, VPI24_DESC, SIG_DESC_SET(SCU88, 8));
PIN_DECL_1(V6, GPIOO0, VPIG8);

#define Y5 113
SIG_EXPR_LIST_DECL_SINGLE(Y5, VPIG9, VPI24, VPI24_DESC, SIG_DESC_SET(SCU88, 9));
PIN_DECL_1(Y5, GPIOO1, VPIG9);

#define AA4 114
SIG_EXPR_LIST_DECL_SINGLE(AA4, VPIR0, VPI30, VPI30_DESC,
			  SIG_DESC_SET(SCU88, 10));
PIN_DECL_1(AA4, GPIOO2, VPIR0);

#define AB3 115
SIG_EXPR_LIST_DECL_SINGLE(AB3, VPIR1, VPI30, VPI30_DESC,
			  SIG_DESC_SET(SCU88, 11));
PIN_DECL_1(AB3, GPIOO3, VPIR1);

#define W6 116
SIG_EXPR_LIST_DECL_SINGLE(W6, VPIR2, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 12));
PIN_DECL_1(W6, GPIOO4, VPIR2);

#define AA5 117
SIG_EXPR_LIST_DECL_SINGLE(AA5, VPIR3, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 13));
PIN_DECL_1(AA5, GPIOO5, VPIR3);

#define AB4 118
SIG_EXPR_LIST_DECL_SINGLE(AB4, VPIR4, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 14));
PIN_DECL_1(AB4, GPIOO6, VPIR4);

#define V7 119
SIG_EXPR_LIST_DECL_SINGLE(V7, VPIR5, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 15));
PIN_DECL_1(V7, GPIOO7, VPIR5);

#define Y6 120
SIG_EXPR_LIST_DECL_SINGLE(Y6, VPIR6, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 16));
PIN_DECL_1(Y6, GPIOP0, VPIR6);

#define AB5 121
SIG_EXPR_LIST_DECL_SINGLE(AB5, VPIR7, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 17));
PIN_DECL_1(AB5, GPIOP1, VPIR7);

#define W7 122
SIG_EXPR_LIST_DECL_SINGLE(W7, VPIR8, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 18));
PIN_DECL_1(W7, GPIOP2, VPIR8);

#define AA6 123
SIG_EXPR_LIST_DECL_SINGLE(AA6, VPIR9, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 19));
PIN_DECL_1(AA6, GPIOP3, VPIR9);

FUNC_GROUP_DECL(VPI18, T5, U3, V1, U4, V2, V3, W2, Y1, V4, W3, Y2, AA1, V5,
		AA22, W5, Y4, AA3, AB2);
FUNC_GROUP_DECL(VPI24, T5, U3, V1, U4, V2, V3, W2, Y1, V4, W3, Y2, AA1, V5,
		AA22, W5, Y4, AA3, AB2, V6, Y5, W6, AA5, AB4, V7, Y6, AB5, W7,
		AA6);
FUNC_GROUP_DECL(VPI30, T5, U3, V1, U4, V2, W1, U5, V3, W2, Y1, V4, W3, Y2, AA1,
		V5, W4, Y3, AA22, W5, Y4, AA3, AB2, AA4, AB3);

#define AB6 124
SIG_EXPR_LIST_DECL_SINGLE(AB6, GPIOP4, GPIOP4);
PIN_DECL_(AB6, SIG_EXPR_LIST_PTR(AB6, GPIOP4));

#define Y7 125
SIG_EXPR_LIST_DECL_SINGLE(Y7, GPIOP5, GPIOP5);
PIN_DECL_(Y7, SIG_EXPR_LIST_PTR(Y7, GPIOP5));

#define AA7 126
SSSF_PIN_DECL(AA7, GPIOP6, BMCINT, SIG_DESC_SET(SCU88, 22));

#define AB7 127
SSSF_PIN_DECL(AB7, GPIOP7, FLACK, SIG_DESC_SET(SCU88, 23));

#define I2C3_DESC	SIG_DESC_SET(SCU90, 16)

#define D3 128
SIG_EXPR_LIST_DECL_SINGLE(D3, SCL3, I2C3, I2C3_DESC);
PIN_DECL_1(D3, GPIOQ0, SCL3);

#define C2 129
SIG_EXPR_LIST_DECL_SINGLE(C2, SDA3, I2C3, I2C3_DESC);
PIN_DECL_1(C2, GPIOQ1, SDA3);

FUNC_GROUP_DECL(I2C3, D3, C2);

#define I2C4_DESC	SIG_DESC_SET(SCU90, 17)

#define B1 130
SIG_EXPR_LIST_DECL_SINGLE(B1, SCL4, I2C4, I2C4_DESC);
PIN_DECL_1(B1, GPIOQ2, SCL4);

#define F5 131
SIG_EXPR_LIST_DECL_SINGLE(F5, SDA4, I2C4, I2C4_DESC);
PIN_DECL_1(F5, GPIOQ3, SDA4);

FUNC_GROUP_DECL(I2C4, B1, F5);

#define I2C14_DESC	SIG_DESC_SET(SCU90, 27)

#define H4 132
SIG_EXPR_LIST_DECL_SINGLE(H4, SCL14, I2C14, I2C14_DESC);
PIN_DECL_1(H4, GPIOQ4, SCL14);

#define H3 133
SIG_EXPR_LIST_DECL_SINGLE(H3, SDA14, I2C14, I2C14_DESC);
PIN_DECL_1(H3, GPIOQ5, SDA14);

FUNC_GROUP_DECL(I2C14, H4, H3);

/*
 * There are several opportunities to document USB port 4 in the datasheet, but
 * it is only mentioned in one location. Particularly, the Multi-function Pins
 * Mapping and Control table in the datasheet elides the signal names,
 * suggesting that port 4 may not actually be functional. As such we define the
 * signal names and control bit, but don't export the capability's function or
 * group.
 */
#define USB11H3_DESC	SIG_DESC_SET(SCU90, 28)

#define H2 134
SIG_EXPR_LIST_DECL_SINGLE(H2, USB11HDP3, USB11H3, USB11H3_DESC);
PIN_DECL_1(H2, GPIOQ6, USB11HDP3);

#define H1 135
SIG_EXPR_LIST_DECL_SINGLE(H1, USB11HDN3, USB11H3, USB11H3_DESC);
PIN_DECL_1(H1, GPIOQ7, USB11HDN3);

#define V20 136
SSSF_PIN_DECL(V20, GPIOR0, ROMCS1, SIG_DESC_SET(SCU88, 24));

#define W21 137
SSSF_PIN_DECL(W21, GPIOR1, ROMCS2, SIG_DESC_SET(SCU88, 25));

#define Y22 138
SSSF_PIN_DECL(Y22, GPIOR2, ROMCS3, SIG_DESC_SET(SCU88, 26));

#define U19 139
SSSF_PIN_DECL(U19, GPIOR3, ROMCS4, SIG_DESC_SET(SCU88, 27));

#define VPOOFF0_DESC	{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 0, 0 }
#define VPO12_DESC	{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 1, 0 }
#define VPO24_DESC	{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 2, 0 }
#define VPOOFF1_DESC	{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 3, 0 }
#define VPO_OFF_12      { ASPEED_IP_SCU, SCU94, 0x2, 0, 0 }
#define VPO_24_OFF      SIG_DESC_SET(SCU94, 1)

#define V21 140
#define V21_DESC	SIG_DESC_SET(SCU88, 28)
SIG_EXPR_DECL_SINGLE(ROMA24, ROM8, V21_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA24, ROM16, V21_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA24, ROM16S, V21_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL(ROMA24, ROM,
		SIG_EXPR_PTR(ROMA24, ROM8),
		SIG_EXPR_PTR(ROMA24, ROM16),
		SIG_EXPR_PTR(ROMA24, ROM16S));
SIG_EXPR_LIST_ALIAS(V21, ROMA24, ROM);
SIG_EXPR_LIST_DECL_SINGLE(V21, VPOR6, VPO24, V21_DESC, VPO_24_OFF);
PIN_DECL_2(V21, GPIOR4, ROMA24, VPOR6);

#define W22 141
#define W22_DESC	SIG_DESC_SET(SCU88, 29)
SIG_EXPR_DECL_SINGLE(ROMA25, ROM8, W22_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA25, ROM16, W22_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA25, ROM16S, W22_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL(ROMA25, ROM,
		SIG_EXPR_PTR(ROMA25, ROM8),
		SIG_EXPR_PTR(ROMA25, ROM16),
		SIG_EXPR_PTR(ROMA25, ROM16S));
SIG_EXPR_LIST_ALIAS(W22, ROMA25, ROM);
SIG_EXPR_LIST_DECL_SINGLE(W22, VPOR7, VPO24, W22_DESC, VPO_24_OFF);
PIN_DECL_2(W22, GPIOR5, ROMA25, VPOR7);

#define C6 142
SIG_EXPR_LIST_DECL_SINGLE(C6, MDC1, MDIO1, SIG_DESC_SET(SCU88, 30));
PIN_DECL_1(C6, GPIOR6, MDC1);

#define A5 143
SIG_EXPR_LIST_DECL_SINGLE(A5, MDIO1, MDIO1, SIG_DESC_SET(SCU88, 31));
PIN_DECL_1(A5, GPIOR7, MDIO1);

FUNC_GROUP_DECL(MDIO1, C6, A5);

#define U21 144
#define U21_DESC        SIG_DESC_SET(SCU8C, 0)
SIG_EXPR_DECL_SINGLE(ROMD4, ROM8, U21_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMD4, ROM16, U21_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMD4, ROM16S, U21_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL(ROMD4, ROM,
		SIG_EXPR_PTR(ROMD4, ROM8),
		SIG_EXPR_PTR(ROMD4, ROM16),
		SIG_EXPR_PTR(ROMD4, ROM16S));
SIG_EXPR_LIST_ALIAS(U21, ROMD4, ROM);
SIG_EXPR_DECL_SINGLE(VPODE, VPO12, U21_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPODE, VPO24, U21_DESC, VPO12_DESC);
SIG_EXPR_LIST_DECL_DUAL(U21, VPODE, VPO12, VPO24);
PIN_DECL_2(U21, GPIOS0, ROMD4, VPODE);

#define T19 145
#define T19_DESC        SIG_DESC_SET(SCU8C, 1)
SIG_EXPR_DECL_SINGLE(ROMD5, ROM8, T19_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMD5, ROM16, T19_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMD5, ROM16S, T19_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL(ROMD5, ROM,
		SIG_EXPR_PTR(ROMD5, ROM8),
		SIG_EXPR_PTR(ROMD5, ROM16),
		SIG_EXPR_PTR(ROMD5, ROM16S));
SIG_EXPR_LIST_ALIAS(T19, ROMD5, ROM);
SIG_EXPR_DECL_SINGLE(VPOHS, VPO12, T19_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOHS, VPO24, T19_DESC, VPO24_DESC);
SIG_EXPR_LIST_DECL_DUAL(T19, VPOHS, VPO12, VPO24);
PIN_DECL_2(T19, GPIOS1, ROMD5, VPOHS);

#define V22 146
#define V22_DESC        SIG_DESC_SET(SCU8C, 2)
SIG_EXPR_DECL_SINGLE(ROMD6, ROM8, V22_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMD6, ROM16, V22_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMD6, ROM16S, V22_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL(ROMD6, ROM,
		SIG_EXPR_PTR(ROMD6, ROM8),
		SIG_EXPR_PTR(ROMD6, ROM16),
		SIG_EXPR_PTR(ROMD6, ROM16S));
SIG_EXPR_LIST_ALIAS(V22, ROMD6, ROM);
SIG_EXPR_DECL_SINGLE(VPOVS, VPO12, V22_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOVS, VPO24, V22_DESC, VPO24_DESC);
SIG_EXPR_LIST_DECL_DUAL(V22, VPOVS, VPO12, VPO24);
PIN_DECL_2(V22, GPIOS2, ROMD6, VPOVS);

#define U20 147
#define U20_DESC        SIG_DESC_SET(SCU8C, 3)
SIG_EXPR_DECL_SINGLE(ROMD7, ROM8, U20_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMD7, ROM16, U20_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMD7, ROM16S, U20_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL(ROMD7, ROM,
		SIG_EXPR_PTR(ROMD7, ROM8),
		SIG_EXPR_PTR(ROMD7, ROM16),
		SIG_EXPR_PTR(ROMD7, ROM16S));
SIG_EXPR_LIST_ALIAS(U20, ROMD7, ROM);
SIG_EXPR_DECL_SINGLE(VPOCLK, VPO12, U20_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOCLK, VPO24, U20_DESC, VPO24_DESC);
SIG_EXPR_LIST_DECL_DUAL(U20, VPOCLK, VPO12, VPO24);
PIN_DECL_2(U20, GPIOS3, ROMD7, VPOCLK);

#define R18 148
#define ROMOE_DESC      SIG_DESC_SET(SCU8C, 4)
SIG_EXPR_LIST_DECL_SINGLE(R18, GPIOS4, GPIOS4);
SIG_EXPR_DECL_SINGLE(ROMOE, ROM8, ROMOE_DESC);
SIG_EXPR_DECL_SINGLE(ROMOE, ROM16, ROMOE_DESC);
SIG_EXPR_DECL_SINGLE(ROMOE, ROM16S, ROMOE_DESC);
SIG_EXPR_LIST_DECL(ROMOE, ROM,
		SIG_EXPR_PTR(ROMOE, ROM8),
		SIG_EXPR_PTR(ROMOE, ROM16),
		SIG_EXPR_PTR(ROMOE, ROM16S));
SIG_EXPR_LIST_ALIAS(R18, ROMOE, ROM);
PIN_DECL_(R18, SIG_EXPR_LIST_PTR(R18, ROMOE), SIG_EXPR_LIST_PTR(R18, GPIOS4));

#define N21 149
#define ROMWE_DESC      SIG_DESC_SET(SCU8C, 5)
SIG_EXPR_LIST_DECL_SINGLE(N21, GPIOS5, GPIOS5);
SIG_EXPR_DECL_SINGLE(ROMWE, ROM8, ROMWE_DESC);
SIG_EXPR_DECL_SINGLE(ROMWE, ROM16, ROMWE_DESC);
SIG_EXPR_DECL_SINGLE(ROMWE, ROM16S, ROMWE_DESC);
SIG_EXPR_LIST_DECL(ROMWE, ROM,
		SIG_EXPR_PTR(ROMWE, ROM8),
		SIG_EXPR_PTR(ROMWE, ROM16),
		SIG_EXPR_PTR(ROMWE, ROM16S));
SIG_EXPR_LIST_ALIAS(N21, ROMWE, ROM);
PIN_DECL_(N21, SIG_EXPR_LIST_PTR(N21, ROMWE), SIG_EXPR_LIST_PTR(N21, GPIOS5));

#define L22 150
#define L22_DESC        SIG_DESC_SET(SCU8C, 6)
SIG_EXPR_DECL_SINGLE(ROMA22, ROM8, L22_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA22, ROM16, L22_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA22, ROM16S, L22_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL(ROMA22, ROM,
		SIG_EXPR_PTR(ROMA22, ROM8),
		SIG_EXPR_PTR(ROMA22, ROM16),
		SIG_EXPR_PTR(ROMA22, ROM16S));
SIG_EXPR_LIST_ALIAS(L22, ROMA22, ROM);
SIG_EXPR_LIST_DECL_SINGLE(L22, VPOR4, VPO24, L22_DESC, VPO_24_OFF);
PIN_DECL_2(L22, GPIOS6, ROMA22, VPOR4);

#define K18 151
#define K18_DESC	SIG_DESC_SET(SCU8C, 7)
SIG_EXPR_DECL_SINGLE(ROMA23, ROM8, K18_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA23, ROM16, K18_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA23, ROM16S, K18_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL(ROMA23, ROM,
		SIG_EXPR_PTR(ROMA23, ROM8),
		SIG_EXPR_PTR(ROMA23, ROM16),
		SIG_EXPR_PTR(ROMA23, ROM16S));
SIG_EXPR_LIST_ALIAS(K18, ROMA23, ROM);
SIG_EXPR_LIST_DECL_SINGLE(K18, VPOR5, VPO24, K18_DESC, VPO_24_OFF);
PIN_DECL_2(K18, GPIOS7, ROMA23, VPOR5);

#define RMII1_DESC      SIG_DESC_BIT(HW_STRAP1, 6, 0)

#define A12 152
SIG_EXPR_LIST_DECL_SINGLE(A12, GPIOT0, GPIOT0, SIG_DESC_SET(SCUA0, 0));
SIG_EXPR_LIST_DECL_SINGLE(A12, RMII1TXEN, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A12, RGMII1TXCK, RGMII1);
PIN_DECL_(A12, SIG_EXPR_LIST_PTR(A12, GPIOT0),
	  SIG_EXPR_LIST_PTR(A12, RMII1TXEN),
	  SIG_EXPR_LIST_PTR(A12, RGMII1TXCK));

#define B12 153
SIG_EXPR_LIST_DECL_SINGLE(B12, GPIOT1, GPIOT1, SIG_DESC_SET(SCUA0, 1));
SIG_EXPR_LIST_DECL_SINGLE(B12, DASHB12, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B12, RGMII1TXCTL, RGMII1);
PIN_DECL_(B12, SIG_EXPR_LIST_PTR(B12, GPIOT1), SIG_EXPR_LIST_PTR(B12, DASHB12),
		SIG_EXPR_LIST_PTR(B12, RGMII1TXCTL));

#define C12 154
SIG_EXPR_LIST_DECL_SINGLE(C12, GPIOT2, GPIOT2, SIG_DESC_SET(SCUA0, 2));
SIG_EXPR_LIST_DECL_SINGLE(C12, RMII1TXD0, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C12, RGMII1TXD0, RGMII1);
PIN_DECL_(C12, SIG_EXPR_LIST_PTR(C12, GPIOT2),
	  SIG_EXPR_LIST_PTR(C12, RMII1TXD0),
	  SIG_EXPR_LIST_PTR(C12, RGMII1TXD0));

#define D12 155
SIG_EXPR_LIST_DECL_SINGLE(D12, GPIOT3, GPIOT3, SIG_DESC_SET(SCUA0, 3));
SIG_EXPR_LIST_DECL_SINGLE(D12, RMII1TXD1, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D12, RGMII1TXD1, RGMII1);
PIN_DECL_(D12, SIG_EXPR_LIST_PTR(D12, GPIOT3),
	  SIG_EXPR_LIST_PTR(D12, RMII1TXD1),
	  SIG_EXPR_LIST_PTR(D12, RGMII1TXD1));

#define E12 156
SIG_EXPR_LIST_DECL_SINGLE(E12, GPIOT4, GPIOT4, SIG_DESC_SET(SCUA0, 4));
SIG_EXPR_LIST_DECL_SINGLE(E12, DASHE12, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E12, RGMII1TXD2, RGMII1);
PIN_DECL_(E12, SIG_EXPR_LIST_PTR(E12, GPIOT4), SIG_EXPR_LIST_PTR(E12, DASHE12),
		SIG_EXPR_LIST_PTR(E12, RGMII1TXD2));

#define A13 157
SIG_EXPR_LIST_DECL_SINGLE(A13, GPIOT5, GPIOT5, SIG_DESC_SET(SCUA0, 5));
SIG_EXPR_LIST_DECL_SINGLE(A13, DASHA13, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A13, RGMII1TXD3, RGMII1);
PIN_DECL_(A13, SIG_EXPR_LIST_PTR(A13, GPIOT5), SIG_EXPR_LIST_PTR(A13, DASHA13),
		SIG_EXPR_LIST_PTR(A13, RGMII1TXD3));

#define RMII2_DESC      SIG_DESC_BIT(HW_STRAP1, 7, 0)

#define D9 158
SIG_EXPR_LIST_DECL_SINGLE(D9, GPIOT6, GPIOT6, SIG_DESC_SET(SCUA0, 6));
SIG_EXPR_LIST_DECL_SINGLE(D9, RMII2TXEN, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D9, RGMII2TXCK, RGMII2);
PIN_DECL_(D9, SIG_EXPR_LIST_PTR(D9, GPIOT6), SIG_EXPR_LIST_PTR(D9, RMII2TXEN),
		SIG_EXPR_LIST_PTR(D9, RGMII2TXCK));

#define E9 159
SIG_EXPR_LIST_DECL_SINGLE(E9, GPIOT7, GPIOT7, SIG_DESC_SET(SCUA0, 7));
SIG_EXPR_LIST_DECL_SINGLE(E9, DASHE9, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E9, RGMII2TXCTL, RGMII2);
PIN_DECL_(E9, SIG_EXPR_LIST_PTR(E9, GPIOT7), SIG_EXPR_LIST_PTR(E9, DASHE9),
		SIG_EXPR_LIST_PTR(E9, RGMII2TXCTL));

#define A10 160
SIG_EXPR_LIST_DECL_SINGLE(A10, GPIOU0, GPIOU0, SIG_DESC_SET(SCUA0, 8));
SIG_EXPR_LIST_DECL_SINGLE(A10, RMII2TXD0, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A10, RGMII2TXD0, RGMII2);
PIN_DECL_(A10, SIG_EXPR_LIST_PTR(A10, GPIOU0),
	  SIG_EXPR_LIST_PTR(A10, RMII2TXD0),
	  SIG_EXPR_LIST_PTR(A10, RGMII2TXD0));

#define B10 161
SIG_EXPR_LIST_DECL_SINGLE(B10, GPIOU1, GPIOU1, SIG_DESC_SET(SCUA0, 9));
SIG_EXPR_LIST_DECL_SINGLE(B10, RMII2TXD1, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B10, RGMII2TXD1, RGMII2);
PIN_DECL_(B10, SIG_EXPR_LIST_PTR(B10, GPIOU1),
	  SIG_EXPR_LIST_PTR(B10, RMII2TXD1),
	  SIG_EXPR_LIST_PTR(B10, RGMII2TXD1));

#define C10 162
SIG_EXPR_LIST_DECL_SINGLE(C10, GPIOU2, GPIOU2, SIG_DESC_SET(SCUA0, 10));
SIG_EXPR_LIST_DECL_SINGLE(C10, DASHC10, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C10, RGMII2TXD2, RGMII2);
PIN_DECL_(C10, SIG_EXPR_LIST_PTR(C10, GPIOU2), SIG_EXPR_LIST_PTR(C10, DASHC10),
		SIG_EXPR_LIST_PTR(C10, RGMII2TXD2));

#define D10 163
SIG_EXPR_LIST_DECL_SINGLE(D10, GPIOU3, GPIOU3, SIG_DESC_SET(SCUA0, 11));
SIG_EXPR_LIST_DECL_SINGLE(D10, DASHD10, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D10, RGMII2TXD3, RGMII2);
PIN_DECL_(D10, SIG_EXPR_LIST_PTR(D10, GPIOU3), SIG_EXPR_LIST_PTR(D10, DASHD10),
		SIG_EXPR_LIST_PTR(D10, RGMII2TXD3));

#define E11 164
SIG_EXPR_LIST_DECL_SINGLE(E11, GPIOU4, GPIOU4, SIG_DESC_SET(SCUA0, 12));
SIG_EXPR_LIST_DECL_SINGLE(E11, RMII1RCLK, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E11, RGMII1RXCK, RGMII1);
PIN_DECL_(E11, SIG_EXPR_LIST_PTR(E11, GPIOU4),
	  SIG_EXPR_LIST_PTR(E11, RMII1RCLK),
	  SIG_EXPR_LIST_PTR(E11, RGMII1RXCK));

#define D11 165
SIG_EXPR_LIST_DECL_SINGLE(D11, GPIOU5, GPIOU5, SIG_DESC_SET(SCUA0, 13));
SIG_EXPR_LIST_DECL_SINGLE(D11, DASHD11, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D11, RGMII1RXCTL, RGMII1);
PIN_DECL_(D11, SIG_EXPR_LIST_PTR(D11, GPIOU5), SIG_EXPR_LIST_PTR(D11, DASHD11),
		SIG_EXPR_LIST_PTR(D11, RGMII1RXCTL));

#define C11 166
SIG_EXPR_LIST_DECL_SINGLE(C11, GPIOU6, GPIOU6, SIG_DESC_SET(SCUA0, 14));
SIG_EXPR_LIST_DECL_SINGLE(C11, RMII1RXD0, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C11, RGMII1RXD0, RGMII1);
PIN_DECL_(C11, SIG_EXPR_LIST_PTR(C11, GPIOU6),
	  SIG_EXPR_LIST_PTR(C11, RMII1RXD0),
	  SIG_EXPR_LIST_PTR(C11, RGMII1RXD0));

#define B11 167
SIG_EXPR_LIST_DECL_SINGLE(B11, GPIOU7, GPIOU7, SIG_DESC_SET(SCUA0, 15));
SIG_EXPR_LIST_DECL_SINGLE(B11, RMII1RXD1, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B11, RGMII1RXD1, RGMII1);
PIN_DECL_(B11, SIG_EXPR_LIST_PTR(B11, GPIOU7),
	  SIG_EXPR_LIST_PTR(B11, RMII1RXD1),
	  SIG_EXPR_LIST_PTR(B11, RGMII1RXD1));

#define A11 168
SIG_EXPR_LIST_DECL_SINGLE(A11, GPIOV0, GPIOV0, SIG_DESC_SET(SCUA0, 16));
SIG_EXPR_LIST_DECL_SINGLE(A11, RMII1CRSDV, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A11, RGMII1RXD2, RGMII1);
PIN_DECL_(A11, SIG_EXPR_LIST_PTR(A11, GPIOV0),
	  SIG_EXPR_LIST_PTR(A11, RMII1CRSDV),
	  SIG_EXPR_LIST_PTR(A11, RGMII1RXD2));

#define E10 169
SIG_EXPR_LIST_DECL_SINGLE(E10, GPIOV1, GPIOV1, SIG_DESC_SET(SCUA0, 17));
SIG_EXPR_LIST_DECL_SINGLE(E10, RMII1RXER, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E10, RGMII1RXD3, RGMII1);
PIN_DECL_(E10, SIG_EXPR_LIST_PTR(E10, GPIOV1),
	  SIG_EXPR_LIST_PTR(E10, RMII1RXER),
	  SIG_EXPR_LIST_PTR(E10, RGMII1RXD3));

#define C9 170
SIG_EXPR_LIST_DECL_SINGLE(C9, GPIOV2, GPIOV2, SIG_DESC_SET(SCUA0, 18));
SIG_EXPR_LIST_DECL_SINGLE(C9, RMII2RCLK, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C9, RGMII2RXCK, RGMII2);
PIN_DECL_(C9, SIG_EXPR_LIST_PTR(C9, GPIOV2), SIG_EXPR_LIST_PTR(C9, RMII2RCLK),
		SIG_EXPR_LIST_PTR(C9, RGMII2RXCK));

#define B9 171
SIG_EXPR_LIST_DECL_SINGLE(B9, GPIOV3, GPIOV3, SIG_DESC_SET(SCUA0, 19));
SIG_EXPR_LIST_DECL_SINGLE(B9, DASHB9, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B9, RGMII2RXCTL, RGMII2);
PIN_DECL_(B9, SIG_EXPR_LIST_PTR(B9, GPIOV3), SIG_EXPR_LIST_PTR(B9, DASHB9),
		SIG_EXPR_LIST_PTR(B9, RGMII2RXCTL));

#define A9 172
SIG_EXPR_LIST_DECL_SINGLE(A9, GPIOV4, GPIOV4, SIG_DESC_SET(SCUA0, 20));
SIG_EXPR_LIST_DECL_SINGLE(A9, RMII2RXD0, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A9, RGMII2RXD0, RGMII2);
PIN_DECL_(A9, SIG_EXPR_LIST_PTR(A9, GPIOV4), SIG_EXPR_LIST_PTR(A9, RMII2RXD0),
		SIG_EXPR_LIST_PTR(A9, RGMII2RXD0));

#define E8 173
SIG_EXPR_LIST_DECL_SINGLE(E8, GPIOV5, GPIOV5, SIG_DESC_SET(SCUA0, 21));
SIG_EXPR_LIST_DECL_SINGLE(E8, RMII2RXD1, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E8, RGMII2RXD1, RGMII2);
PIN_DECL_(E8, SIG_EXPR_LIST_PTR(E8, GPIOV5), SIG_EXPR_LIST_PTR(E8, RMII2RXD1),
		SIG_EXPR_LIST_PTR(E8, RGMII2RXD1));

#define D8 174
SIG_EXPR_LIST_DECL_SINGLE(D8, GPIOV6, GPIOV6, SIG_DESC_SET(SCUA0, 22));
SIG_EXPR_LIST_DECL_SINGLE(D8, RMII2CRSDV, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D8, RGMII2RXD2, RGMII2);
PIN_DECL_(D8, SIG_EXPR_LIST_PTR(D8, GPIOV6), SIG_EXPR_LIST_PTR(D8, RMII2CRSDV),
		SIG_EXPR_LIST_PTR(D8, RGMII2RXD2));

#define C8 175
SIG_EXPR_LIST_DECL_SINGLE(C8, GPIOV7, GPIOV7, SIG_DESC_SET(SCUA0, 23));
SIG_EXPR_LIST_DECL_SINGLE(C8, RMII2RXER, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C8, RGMII2RXD3, RGMII2);
PIN_DECL_(C8, SIG_EXPR_LIST_PTR(C8, GPIOV7), SIG_EXPR_LIST_PTR(C8, RMII2RXER),
		SIG_EXPR_LIST_PTR(C8, RGMII2RXD3));

FUNC_GROUP_DECL(RMII1, A12, B12, C12, D12, E12, A13, E11, D11, C11, B11, A11,
		E10);
FUNC_GROUP_DECL(RGMII1, A12, B12, C12, D12, E12, A13, E11, D11, C11, B11, A11,
		E10);

FUNC_GROUP_DECL(RMII2, D9, E9, A10, B10, C10, D10, C9, B9, A9, E8, D8, C8);
FUNC_GROUP_DECL(RGMII2, D9, E9, A10, B10, C10, D10, C9, B9, A9, E8, D8, C8);

#define L5 176
SIG_EXPR_LIST_DECL_SINGLE(L5, GPIOW0, GPIOW0, SIG_DESC_SET(SCUA0, 24));
SIG_EXPR_LIST_DECL_SINGLE(L5, ADC0, ADC0);
PIN_DECL_(L5, SIG_EXPR_LIST_PTR(L5, GPIOW0), SIG_EXPR_LIST_PTR(L5, ADC0));
FUNC_GROUP_DECL(ADC0, L5);

#define L4 177
SIG_EXPR_LIST_DECL_SINGLE(L4, GPIOW1, GPIOW1, SIG_DESC_SET(SCUA0, 25));
SIG_EXPR_LIST_DECL_SINGLE(L4, ADC1, ADC1);
PIN_DECL_(L4, SIG_EXPR_LIST_PTR(L4, GPIOW1), SIG_EXPR_LIST_PTR(L4, ADC1));
FUNC_GROUP_DECL(ADC1, L4);

#define L3 178
SIG_EXPR_LIST_DECL_SINGLE(L3, GPIOW2, GPIOW2, SIG_DESC_SET(SCUA0, 26));
SIG_EXPR_LIST_DECL_SINGLE(L3, ADC2, ADC2);
PIN_DECL_(L3, SIG_EXPR_LIST_PTR(L3, GPIOW2), SIG_EXPR_LIST_PTR(L3, ADC2));
FUNC_GROUP_DECL(ADC2, L3);

#define L2 179
SIG_EXPR_LIST_DECL_SINGLE(L2, GPIOW3, GPIOW3, SIG_DESC_SET(SCUA0, 27));
SIG_EXPR_LIST_DECL_SINGLE(L2, ADC3, ADC3);
PIN_DECL_(L2, SIG_EXPR_LIST_PTR(L2, GPIOW3), SIG_EXPR_LIST_PTR(L2, ADC3));
FUNC_GROUP_DECL(ADC3, L2);

#define L1 180
SIG_EXPR_LIST_DECL_SINGLE(L1, GPIOW4, GPIOW4, SIG_DESC_SET(SCUA0, 28));
SIG_EXPR_LIST_DECL_SINGLE(L1, ADC4, ADC4);
PIN_DECL_(L1, SIG_EXPR_LIST_PTR(L1, GPIOW4), SIG_EXPR_LIST_PTR(L1, ADC4));
FUNC_GROUP_DECL(ADC4, L1);

#define M5 181
SIG_EXPR_LIST_DECL_SINGLE(M5, GPIOW5, GPIOW5, SIG_DESC_SET(SCUA0, 29));
SIG_EXPR_LIST_DECL_SINGLE(M5, ADC5, ADC5);
PIN_DECL_(M5, SIG_EXPR_LIST_PTR(M5, GPIOW5), SIG_EXPR_LIST_PTR(M5, ADC5));
FUNC_GROUP_DECL(ADC5, M5);

#define M4 182
SIG_EXPR_LIST_DECL_SINGLE(M4, GPIOW6, GPIOW6, SIG_DESC_SET(SCUA0, 30));
SIG_EXPR_LIST_DECL_SINGLE(M4, ADC6, ADC6);
PIN_DECL_(M4, SIG_EXPR_LIST_PTR(M4, GPIOW6), SIG_EXPR_LIST_PTR(M4, ADC6));
FUNC_GROUP_DECL(ADC6, M4);

#define M3 183
SIG_EXPR_LIST_DECL_SINGLE(M3, GPIOW7, GPIOW7, SIG_DESC_SET(SCUA0, 31));
SIG_EXPR_LIST_DECL_SINGLE(M3, ADC7, ADC7);
PIN_DECL_(M3, SIG_EXPR_LIST_PTR(M3, GPIOW7), SIG_EXPR_LIST_PTR(M3, ADC7));
FUNC_GROUP_DECL(ADC7, M3);

#define M2 184
SIG_EXPR_LIST_DECL_SINGLE(M2, GPIOX0, GPIOX0, SIG_DESC_SET(SCUA4, 0));
SIG_EXPR_LIST_DECL_SINGLE(M2, ADC8, ADC8);
PIN_DECL_(M2, SIG_EXPR_LIST_PTR(M2, GPIOX0), SIG_EXPR_LIST_PTR(M2, ADC8));
FUNC_GROUP_DECL(ADC8, M2);

#define M1 185
SIG_EXPR_LIST_DECL_SINGLE(M1, GPIOX1, GPIOX1, SIG_DESC_SET(SCUA4, 1));
SIG_EXPR_LIST_DECL_SINGLE(M1, ADC9, ADC9);
PIN_DECL_(M1, SIG_EXPR_LIST_PTR(M1, GPIOX1), SIG_EXPR_LIST_PTR(M1, ADC9));
FUNC_GROUP_DECL(ADC9, M1);

#define N5 186
SIG_EXPR_LIST_DECL_SINGLE(N5, GPIOX2, GPIOX2, SIG_DESC_SET(SCUA4, 2));
SIG_EXPR_LIST_DECL_SINGLE(N5, ADC10, ADC10);
PIN_DECL_(N5, SIG_EXPR_LIST_PTR(N5, GPIOX2), SIG_EXPR_LIST_PTR(N5, ADC10));
FUNC_GROUP_DECL(ADC10, N5);

#define N4 187
SIG_EXPR_LIST_DECL_SINGLE(N4, GPIOX3, GPIOX3, SIG_DESC_SET(SCUA4, 3));
SIG_EXPR_LIST_DECL_SINGLE(N4, ADC11, ADC11);
PIN_DECL_(N4, SIG_EXPR_LIST_PTR(N4, GPIOX3), SIG_EXPR_LIST_PTR(N4, ADC11));
FUNC_GROUP_DECL(ADC11, N4);

#define N3 188
SIG_EXPR_LIST_DECL_SINGLE(N3, GPIOX4, GPIOX4, SIG_DESC_SET(SCUA4, 4));
SIG_EXPR_LIST_DECL_SINGLE(N3, ADC12, ADC12);
PIN_DECL_(N3, SIG_EXPR_LIST_PTR(N3, GPIOX4), SIG_EXPR_LIST_PTR(N3, ADC12));
FUNC_GROUP_DECL(ADC12, N3);

#define N2 189
SIG_EXPR_LIST_DECL_SINGLE(N2, GPIOX5, GPIOX5, SIG_DESC_SET(SCUA4, 5));
SIG_EXPR_LIST_DECL_SINGLE(N2, ADC13, ADC13);
PIN_DECL_(N2, SIG_EXPR_LIST_PTR(N2, GPIOX5), SIG_EXPR_LIST_PTR(N2, ADC13));
FUNC_GROUP_DECL(ADC13, N2);

#define N1 190
SIG_EXPR_LIST_DECL_SINGLE(N1, GPIOX6, GPIOX6, SIG_DESC_SET(SCUA4, 6));
SIG_EXPR_LIST_DECL_SINGLE(N1, ADC14, ADC14);
PIN_DECL_(N1, SIG_EXPR_LIST_PTR(N1, GPIOX6), SIG_EXPR_LIST_PTR(N1, ADC14));
FUNC_GROUP_DECL(ADC14, N1);

#define P5 191
SIG_EXPR_LIST_DECL_SINGLE(P5, GPIOX7, GPIOX7, SIG_DESC_SET(SCUA4, 7));
SIG_EXPR_LIST_DECL_SINGLE(P5, ADC15, ADC15);
PIN_DECL_(P5, SIG_EXPR_LIST_PTR(P5, GPIOX7), SIG_EXPR_LIST_PTR(P5, ADC15));
FUNC_GROUP_DECL(ADC15, P5);

#define C21 192
SIG_EXPR_DECL_SINGLE(SIOS3, SIOS3, SIG_DESC_SET(SCUA4, 8));
SIG_EXPR_DECL_SINGLE(SIOS3, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(C21, SIOS3, SIOS3, ACPI);
PIN_DECL_1(C21, GPIOY0, SIOS3);
FUNC_GROUP_DECL(SIOS3, C21);

#define F20 193
SIG_EXPR_DECL_SINGLE(SIOS5, SIOS5, SIG_DESC_SET(SCUA4, 9));
SIG_EXPR_DECL_SINGLE(SIOS5, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(F20, SIOS5, SIOS5, ACPI);
PIN_DECL_1(F20, GPIOY1, SIOS5);
FUNC_GROUP_DECL(SIOS5, F20);

#define G20 194
SIG_EXPR_DECL_SINGLE(SIOPWREQ, SIOPWREQ, SIG_DESC_SET(SCUA4, 10));
SIG_EXPR_DECL_SINGLE(SIOPWREQ, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(G20, SIOPWREQ, SIOPWREQ, ACPI);
PIN_DECL_1(G20, GPIOY2, SIOPWREQ);
FUNC_GROUP_DECL(SIOPWREQ, G20);

#define K20 195
SIG_EXPR_DECL_SINGLE(SIOONCTRL, SIOONCTRL, SIG_DESC_SET(SCUA4, 11));
SIG_EXPR_DECL_SINGLE(SIOONCTRL, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(K20, SIOONCTRL, SIOONCTRL, ACPI);
PIN_DECL_1(K20, GPIOY3, SIOONCTRL);
FUNC_GROUP_DECL(SIOONCTRL, K20);

FUNC_GROUP_DECL(ACPI, B19, A20, D17, A19, C21, F20, G20, K20);

#define R22 200
#define R22_DESC	SIG_DESC_SET(SCUA4, 16)
SIG_EXPR_DECL_SINGLE(ROMA2, ROM8, R22_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA2, ROM16, R22_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(R22, ROMA2, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOB0, VPO12, R22_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOB0, VPO24, R22_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOB0, VPOOFF1, R22_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOB0, VPO,
		SIG_EXPR_PTR(VPOB0, VPO12),
		SIG_EXPR_PTR(VPOB0, VPO24),
		SIG_EXPR_PTR(VPOB0, VPOOFF1));
SIG_EXPR_LIST_ALIAS(R22, VPOB0, VPO);
PIN_DECL_2(R22, GPIOZ0, ROMA2, VPOB0);

#define P18 201
#define P18_DESC	SIG_DESC_SET(SCUA4, 17)
SIG_EXPR_DECL_SINGLE(ROMA3, ROM8, P18_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA3, ROM16, P18_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(P18, ROMA3, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOB1, VPO12, P18_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOB1, VPO24, P18_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOB1, VPOOFF1, P18_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOB1, VPO,
		SIG_EXPR_PTR(VPOB1, VPO12),
		SIG_EXPR_PTR(VPOB1, VPO24),
		SIG_EXPR_PTR(VPOB1, VPOOFF1));
SIG_EXPR_LIST_ALIAS(P18, VPOB1, VPO);
PIN_DECL_2(P18, GPIOZ1, ROMA3, VPOB1);

#define P19 202
#define P19_DESC	SIG_DESC_SET(SCUA4, 18)
SIG_EXPR_DECL_SINGLE(ROMA4, ROM8, P19_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA4, ROM16, P19_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(P19, ROMA4, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOB2, VPO12, P19_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOB2, VPO24, P19_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOB2, VPOOFF1, P19_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOB2, VPO,
		SIG_EXPR_PTR(VPOB2, VPO12),
		SIG_EXPR_PTR(VPOB2, VPO24),
		SIG_EXPR_PTR(VPOB2, VPOOFF1));
SIG_EXPR_LIST_ALIAS(P19, VPOB2, VPO);
PIN_DECL_2(P19, GPIOZ2, ROMA4, VPOB2);

#define P20 203
#define P20_DESC	SIG_DESC_SET(SCUA4, 19)
SIG_EXPR_DECL_SINGLE(ROMA5, ROM8, P20_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA5, ROM16, P20_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(P20, ROMA5, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOB3, VPO12, P20_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOB3, VPO24, P20_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOB3, VPOOFF1, P20_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOB3, VPO,
		SIG_EXPR_PTR(VPOB3, VPO12),
		SIG_EXPR_PTR(VPOB3, VPO24),
		SIG_EXPR_PTR(VPOB3, VPOOFF1));
SIG_EXPR_LIST_ALIAS(P20, VPOB3, VPO);
PIN_DECL_2(P20, GPIOZ3, ROMA5, VPOB3);

#define P21 204
#define P21_DESC	SIG_DESC_SET(SCUA4, 20)
SIG_EXPR_DECL_SINGLE(ROMA6, ROM8, P21_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA6, ROM16, P21_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(P21, ROMA6, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOB4, VPO12, P21_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOB4, VPO24, P21_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOB4, VPOOFF1, P21_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOB4, VPO,
		SIG_EXPR_PTR(VPOB4, VPO12),
		SIG_EXPR_PTR(VPOB4, VPO24),
		SIG_EXPR_PTR(VPOB4, VPOOFF1));
SIG_EXPR_LIST_ALIAS(P21, VPOB4, VPO);
PIN_DECL_2(P21, GPIOZ4, ROMA6, VPOB4);

#define P22 205
#define P22_DESC	SIG_DESC_SET(SCUA4, 21)
SIG_EXPR_DECL_SINGLE(ROMA7, ROM8, P22_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA7, ROM16, P22_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(P22, ROMA7, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOB5, VPO12, P22_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOB5, VPO24, P22_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOB5, VPOOFF1, P22_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOB5, VPO,
		SIG_EXPR_PTR(VPOB5, VPO12),
		SIG_EXPR_PTR(VPOB5, VPO24),
		SIG_EXPR_PTR(VPOB5, VPOOFF1));
SIG_EXPR_LIST_ALIAS(P22, VPOB5, VPO);
PIN_DECL_2(P22, GPIOZ5, ROMA7, VPOB5);

#define M19 206
#define M19_DESC	SIG_DESC_SET(SCUA4, 22)
SIG_EXPR_DECL_SINGLE(ROMA8, ROM8, M19_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA8, ROM16, M19_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(M19, ROMA8, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOB6, VPO12, M19_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOB6, VPO24, M19_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOB6, VPOOFF1, M19_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOB6, VPO,
		SIG_EXPR_PTR(VPOB6, VPO12),
		SIG_EXPR_PTR(VPOB6, VPO24),
		SIG_EXPR_PTR(VPOB6, VPOOFF1));
SIG_EXPR_LIST_ALIAS(M19, VPOB6, VPO);
PIN_DECL_2(M19, GPIOZ6, ROMA8, VPOB6);

#define M20 207
#define M20_DESC	SIG_DESC_SET(SCUA4, 23)
SIG_EXPR_DECL_SINGLE(ROMA9, ROM8, M20_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA9, ROM16, M20_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(M20, ROMA9, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOB7, VPO12, M20_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOB7, VPO24, M20_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOB7, VPOOFF1, M20_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOB7, VPO,
		SIG_EXPR_PTR(VPOB7, VPO12),
		SIG_EXPR_PTR(VPOB7, VPO24),
		SIG_EXPR_PTR(VPOB7, VPOOFF1));
SIG_EXPR_LIST_ALIAS(M20, VPOB7, VPO);
PIN_DECL_2(M20, GPIOZ7, ROMA9, VPOB7);

#define M21 208
#define M21_DESC	SIG_DESC_SET(SCUA4, 24)
SIG_EXPR_DECL_SINGLE(ROMA10, ROM8, M21_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA10, ROM16, M21_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(M21, ROMA10, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOG0, VPO12, M21_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOG0, VPO24, M21_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOG0, VPOOFF1, M21_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOG0, VPO,
		SIG_EXPR_PTR(VPOG0, VPO12),
		SIG_EXPR_PTR(VPOG0, VPO24),
		SIG_EXPR_PTR(VPOG0, VPOOFF1));
SIG_EXPR_LIST_ALIAS(M21, VPOG0, VPO);
PIN_DECL_2(M21, GPIOAA0, ROMA10, VPOG0);

#define M22 209
#define M22_DESC	SIG_DESC_SET(SCUA4, 25)
SIG_EXPR_DECL_SINGLE(ROMA11, ROM8, M22_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA11, ROM16, M22_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(M22, ROMA11, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOG1, VPO12, M22_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOG1, VPO24, M22_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOG1, VPOOFF1, M22_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOG1, VPO,
		SIG_EXPR_PTR(VPOG1, VPO12),
		SIG_EXPR_PTR(VPOG1, VPO24),
		SIG_EXPR_PTR(VPOG1, VPOOFF1));
SIG_EXPR_LIST_ALIAS(M22, VPOG1, VPO);
PIN_DECL_2(M22, GPIOAA1, ROMA11, VPOG1);

#define L18 210
#define L18_DESC	SIG_DESC_SET(SCUA4, 26)
SIG_EXPR_DECL_SINGLE(ROMA12, ROM8, L18_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA12, ROM16, L18_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(L18, ROMA12, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOG2, VPO12, L18_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOG2, VPO24, L18_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOG2, VPOOFF1, L18_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOG2, VPO,
		SIG_EXPR_PTR(VPOG2, VPO12),
		SIG_EXPR_PTR(VPOG2, VPO24),
		SIG_EXPR_PTR(VPOG2, VPOOFF1));
SIG_EXPR_LIST_ALIAS(L18, VPOG2, VPO);
PIN_DECL_2(L18, GPIOAA2, ROMA12, VPOG2);

#define L19 211
#define L19_DESC	SIG_DESC_SET(SCUA4, 27)
SIG_EXPR_DECL_SINGLE(ROMA13, ROM8, L19_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL_SINGLE(ROMA13, ROM16, L19_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL_DUAL(L19, ROMA13, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOG3, VPO12, L19_DESC, VPO12_DESC);
SIG_EXPR_DECL_SINGLE(VPOG3, VPO24, L19_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOG3, VPOOFF1, L19_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL(VPOG3, VPO,
		SIG_EXPR_PTR(VPOG3, VPO12),
		SIG_EXPR_PTR(VPOG3, VPO24),
		SIG_EXPR_PTR(VPOG3, VPOOFF1));
SIG_EXPR_LIST_ALIAS(L19, VPOG3, VPO);
PIN_DECL_2(L19, GPIOAA3, ROMA13, VPOG3);

#define L20 212
#define L20_DESC	SIG_DESC_SET(SCUA4, 28)
SIG_EXPR_DECL_SINGLE(ROMA14, ROM8, L20_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA14, ROM16, L20_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL_DUAL(L20, ROMA14, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOG4, VPO24, L20_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOG4, VPOOFF1, L20_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL_DUAL(L20, VPOG4, VPO24, VPOOFF1);
PIN_DECL_2(L20, GPIOAA4, ROMA14, VPOG4);

#define L21 213
#define L21_DESC	SIG_DESC_SET(SCUA4, 29)
SIG_EXPR_DECL_SINGLE(ROMA15, ROM8, L21_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA15, ROM16, L21_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL_DUAL(L21, ROMA15, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOG5, VPO24, L21_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOG5, VPOOFF1, L21_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL_DUAL(L21, VPOG5, VPO24, VPOOFF1);
PIN_DECL_2(L21, GPIOAA5, ROMA15, VPOG5);

#define T18 214
#define T18_DESC	SIG_DESC_SET(SCUA4, 30)
SIG_EXPR_DECL_SINGLE(ROMA16, ROM8, T18_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA16, ROM16, T18_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL_DUAL(T18, ROMA16, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOG6, VPO24, T18_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOG6, VPOOFF1, T18_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL_DUAL(T18, VPOG6, VPO24, VPOOFF1);
PIN_DECL_2(T18, GPIOAA6, ROMA16, VPOG6);

#define N18 215
#define N18_DESC	SIG_DESC_SET(SCUA4, 31)
SIG_EXPR_DECL_SINGLE(ROMA17, ROM8, N18_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA17, ROM16, N18_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL_DUAL(N18, ROMA17, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOG7, VPO24, N18_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOG7, VPOOFF1, N18_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL_DUAL(N18, VPOG7, VPO24, VPOOFF1);
PIN_DECL_2(N18, GPIOAA7, ROMA17, VPOG7);

#define N19 216
#define N19_DESC	SIG_DESC_SET(SCUA8, 0)
SIG_EXPR_DECL_SINGLE(ROMA18, ROM8, N19_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA18, ROM16, N19_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL_DUAL(N19, ROMA18, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOR0, VPO24, N19_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOR0, VPOOFF1, N19_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL_DUAL(N19, VPOR0, VPO24, VPOOFF1);
PIN_DECL_2(N19, GPIOAB0, ROMA18, VPOR0);

#define M18 217
#define M18_DESC	SIG_DESC_SET(SCUA8, 1)
SIG_EXPR_DECL_SINGLE(ROMA19, ROM8, M18_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA19, ROM16, M18_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL_DUAL(M18, ROMA19, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOR1, VPO24, M18_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOR1, VPOOFF1, M18_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL_DUAL(M18, VPOR1, VPO24, VPOOFF1);
PIN_DECL_2(M18, GPIOAB1, ROMA19, VPOR1);

#define N22 218
#define N22_DESC	SIG_DESC_SET(SCUA8, 2)
SIG_EXPR_DECL_SINGLE(ROMA20, ROM8, N22_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA20, ROM16, N22_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL_DUAL(N22, ROMA20, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOR2, VPO24, N22_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOR2, VPOOFF1, N22_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL_DUAL(N22, VPOR2, VPO24, VPOOFF1);
PIN_DECL_2(N22, GPIOAB2, ROMA20, VPOR2);

#define N20 219
#define N20_DESC	SIG_DESC_SET(SCUA8, 3)
SIG_EXPR_DECL_SINGLE(ROMA21, ROM8, N20_DESC, VPO_OFF_12);
SIG_EXPR_DECL_SINGLE(ROMA21, ROM16, N20_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL_DUAL(N20, ROMA21, ROM8, ROM16);
SIG_EXPR_DECL_SINGLE(VPOR3, VPO24, N20_DESC, VPO24_DESC);
SIG_EXPR_DECL_SINGLE(VPOR3, VPOOFF1, N20_DESC, VPOOFF1_DESC);
SIG_EXPR_LIST_DECL_DUAL(N20, VPOR3, VPO24, VPOOFF1);
PIN_DECL_2(N20, GPIOAB3, ROMA21, VPOR3);

FUNC_GROUP_DECL(ROM8, V20, U21, T19, V22, U20, R18, N21, L22, K18, W21, Y22,
		U19, R22, P18, P19, P20, P21, P22, M19, M20, M21, M22, L18,
		L19, L20, L21, T18, N18, N19, M18, N22, N20);
FUNC_GROUP_DECL(ROM16, V20, U21, T19, V22, U20, R18, N21, L22, K18,
		A8, C7, B7, A7, D7, B6, A6, E7, W21, Y22, U19, R22, P18, P19,
		P20, P21, P22, M19, M20, M21, M22, L18, L19, L20, L21, T18,
		N18, N19, M18, N22, N20);
FUNC_GROUP_DECL(VPO12, U21, T19, V22, U20, R22, P18, P19, P20, P21, P22, M19,
		M20, M21, M22, L18, L19, L20, L21, T18, N18, N19, M18, N22,
		N20);
FUNC_GROUP_DECL(VPO24, U21, T19, V22, U20, L22, K18, V21, W22, R22, P18, P19,
		P20, P21, P22, M19, M20, M21, M22, L18, L19);

#define USB11H2_DESC	SIG_DESC_SET(SCU90, 3)
#define USB11D1_DESC	SIG_DESC_BIT(SCU90, 3, 0)

#define K4 220
SIG_EXPR_LIST_DECL_SINGLE(K4, USB11HDP2, USB11H2, USB11H2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(K4, USB11DP1, USB11D1, USB11D1_DESC);
PIN_DECL_(K4, SIG_EXPR_LIST_PTR(K4, USB11HDP2),
	  SIG_EXPR_LIST_PTR(K4, USB11DP1));

#define K3 221
SIG_EXPR_LIST_DECL_SINGLE(K3, USB11HDN1, USB11H2, USB11H2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(K3, USB11DDN1, USB11D1, USB11D1_DESC);
PIN_DECL_(K3, SIG_EXPR_LIST_PTR(K3, USB11HDN1),
	  SIG_EXPR_LIST_PTR(K3, USB11DDN1));

FUNC_GROUP_DECL(USB11H2, K4, K3);
FUNC_GROUP_DECL(USB11D1, K4, K3);

#define USB2H1_DESC	SIG_DESC_SET(SCU90, 29)
#define USB2D1_DESC	SIG_DESC_BIT(SCU90, 29, 0)

#define AB21 222
SIG_EXPR_LIST_DECL_SINGLE(AB21, USB2HDP1, USB2H1, USB2H1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AB21, USB2DDP1, USB2D1, USB2D1_DESC);
PIN_DECL_(AB21, SIG_EXPR_LIST_PTR(AB21, USB2HDP1),
	  SIG_EXPR_LIST_PTR(AB21, USB2DDP1));

#define AB20 223
SIG_EXPR_LIST_DECL_SINGLE(AB20, USB2HDN1, USB2H1, USB2H1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AB20, USB2DDN1, USB2D1, USB2D1_DESC);
PIN_DECL_(AB20, SIG_EXPR_LIST_PTR(AB20, USB2HDN1),
	  SIG_EXPR_LIST_PTR(AB20, USB2DDN1));

FUNC_GROUP_DECL(USB2H1, AB21, AB20);
FUNC_GROUP_DECL(USB2D1, AB21, AB20);

/* Note we account for GPIOY4-GPIOY7 even though they're not valid, thus 216
 * pins becomes 220. Four additional non-GPIO-capable pins are present for USB.
 */
#define ASPEED_G4_NR_PINS 224

/* Pins, groups and functions are sort(1):ed alphabetically for sanity */

static struct pinctrl_pin_desc aspeed_g4_pins[ASPEED_G4_NR_PINS] = {
	ASPEED_PINCTRL_PIN(A1),
	ASPEED_PINCTRL_PIN(A10),
	ASPEED_PINCTRL_PIN(A11),
	ASPEED_PINCTRL_PIN(A12),
	ASPEED_PINCTRL_PIN(A13),
	ASPEED_PINCTRL_PIN(A14),
	ASPEED_PINCTRL_PIN(A15),
	ASPEED_PINCTRL_PIN(A16),
	ASPEED_PINCTRL_PIN(A17),
	ASPEED_PINCTRL_PIN(A18),
	ASPEED_PINCTRL_PIN(A19),
	ASPEED_PINCTRL_PIN(A2),
	ASPEED_PINCTRL_PIN(A20),
	ASPEED_PINCTRL_PIN(A3),
	ASPEED_PINCTRL_PIN(A4),
	ASPEED_PINCTRL_PIN(A5),
	ASPEED_PINCTRL_PIN(A6),
	ASPEED_PINCTRL_PIN(A7),
	ASPEED_PINCTRL_PIN(A8),
	ASPEED_PINCTRL_PIN(A9),
	ASPEED_PINCTRL_PIN(AA1),
	ASPEED_PINCTRL_PIN(AA2),
	ASPEED_PINCTRL_PIN(AA22),
	ASPEED_PINCTRL_PIN(AA3),
	ASPEED_PINCTRL_PIN(AA4),
	ASPEED_PINCTRL_PIN(AA5),
	ASPEED_PINCTRL_PIN(AA6),
	ASPEED_PINCTRL_PIN(AA7),
	ASPEED_PINCTRL_PIN(AB1),
	ASPEED_PINCTRL_PIN(AB2),
	ASPEED_PINCTRL_PIN(AB3),
	ASPEED_PINCTRL_PIN(AB4),
	ASPEED_PINCTRL_PIN(AB5),
	ASPEED_PINCTRL_PIN(AB6),
	ASPEED_PINCTRL_PIN(AB7),
	ASPEED_PINCTRL_PIN(AB20),
	ASPEED_PINCTRL_PIN(AB21),
	ASPEED_PINCTRL_PIN(B1),
	ASPEED_PINCTRL_PIN(B10),
	ASPEED_PINCTRL_PIN(B11),
	ASPEED_PINCTRL_PIN(B12),
	ASPEED_PINCTRL_PIN(B13),
	ASPEED_PINCTRL_PIN(B14),
	ASPEED_PINCTRL_PIN(B15),
	ASPEED_PINCTRL_PIN(B16),
	ASPEED_PINCTRL_PIN(B17),
	ASPEED_PINCTRL_PIN(B18),
	ASPEED_PINCTRL_PIN(B19),
	ASPEED_PINCTRL_PIN(B2),
	ASPEED_PINCTRL_PIN(B22),
	ASPEED_PINCTRL_PIN(B3),
	ASPEED_PINCTRL_PIN(B4),
	ASPEED_PINCTRL_PIN(B5),
	ASPEED_PINCTRL_PIN(B6),
	ASPEED_PINCTRL_PIN(B7),
	ASPEED_PINCTRL_PIN(B9),
	ASPEED_PINCTRL_PIN(C1),
	ASPEED_PINCTRL_PIN(C10),
	ASPEED_PINCTRL_PIN(C11),
	ASPEED_PINCTRL_PIN(C12),
	ASPEED_PINCTRL_PIN(C13),
	ASPEED_PINCTRL_PIN(C14),
	ASPEED_PINCTRL_PIN(C15),
	ASPEED_PINCTRL_PIN(C16),
	ASPEED_PINCTRL_PIN(C17),
	ASPEED_PINCTRL_PIN(C18),
	ASPEED_PINCTRL_PIN(C2),
	ASPEED_PINCTRL_PIN(C20),
	ASPEED_PINCTRL_PIN(C21),
	ASPEED_PINCTRL_PIN(C22),
	ASPEED_PINCTRL_PIN(C3),
	ASPEED_PINCTRL_PIN(C4),
	ASPEED_PINCTRL_PIN(C5),
	ASPEED_PINCTRL_PIN(C6),
	ASPEED_PINCTRL_PIN(C7),
	ASPEED_PINCTRL_PIN(C8),
	ASPEED_PINCTRL_PIN(C9),
	ASPEED_PINCTRL_PIN(D1),
	ASPEED_PINCTRL_PIN(D10),
	ASPEED_PINCTRL_PIN(D11),
	ASPEED_PINCTRL_PIN(D12),
	ASPEED_PINCTRL_PIN(D13),
	ASPEED_PINCTRL_PIN(D14),
	ASPEED_PINCTRL_PIN(D15),
	ASPEED_PINCTRL_PIN(D16),
	ASPEED_PINCTRL_PIN(D17),
	ASPEED_PINCTRL_PIN(D18),
	ASPEED_PINCTRL_PIN(D19),
	ASPEED_PINCTRL_PIN(D2),
	ASPEED_PINCTRL_PIN(D3),
	ASPEED_PINCTRL_PIN(D4),
	ASPEED_PINCTRL_PIN(D5),
	ASPEED_PINCTRL_PIN(D6),
	ASPEED_PINCTRL_PIN(D7),
	ASPEED_PINCTRL_PIN(D8),
	ASPEED_PINCTRL_PIN(D9),
	ASPEED_PINCTRL_PIN(E10),
	ASPEED_PINCTRL_PIN(E11),
	ASPEED_PINCTRL_PIN(E12),
	ASPEED_PINCTRL_PIN(E13),
	ASPEED_PINCTRL_PIN(E14),
	ASPEED_PINCTRL_PIN(E15),
	ASPEED_PINCTRL_PIN(E16),
	ASPEED_PINCTRL_PIN(E18),
	ASPEED_PINCTRL_PIN(E19),
	ASPEED_PINCTRL_PIN(E2),
	ASPEED_PINCTRL_PIN(E20),
	ASPEED_PINCTRL_PIN(E3),
	ASPEED_PINCTRL_PIN(E5),
	ASPEED_PINCTRL_PIN(E6),
	ASPEED_PINCTRL_PIN(E7),
	ASPEED_PINCTRL_PIN(E8),
	ASPEED_PINCTRL_PIN(E9),
	ASPEED_PINCTRL_PIN(F18),
	ASPEED_PINCTRL_PIN(F20),
	ASPEED_PINCTRL_PIN(F3),
	ASPEED_PINCTRL_PIN(F4),
	ASPEED_PINCTRL_PIN(F5),
	ASPEED_PINCTRL_PIN(G18),
	ASPEED_PINCTRL_PIN(G19),
	ASPEED_PINCTRL_PIN(G20),
	ASPEED_PINCTRL_PIN(G5),
	ASPEED_PINCTRL_PIN(H1),
	ASPEED_PINCTRL_PIN(H18),
	ASPEED_PINCTRL_PIN(H19),
	ASPEED_PINCTRL_PIN(H2),
	ASPEED_PINCTRL_PIN(H20),
	ASPEED_PINCTRL_PIN(H3),
	ASPEED_PINCTRL_PIN(H4),
	ASPEED_PINCTRL_PIN(J20),
	ASPEED_PINCTRL_PIN(J21),
	ASPEED_PINCTRL_PIN(J3),
	ASPEED_PINCTRL_PIN(J4),
	ASPEED_PINCTRL_PIN(J5),
	ASPEED_PINCTRL_PIN(K18),
	ASPEED_PINCTRL_PIN(K20),
	ASPEED_PINCTRL_PIN(K3),
	ASPEED_PINCTRL_PIN(K4),
	ASPEED_PINCTRL_PIN(K5),
	ASPEED_PINCTRL_PIN(L1),
	ASPEED_PINCTRL_PIN(L18),
	ASPEED_PINCTRL_PIN(L19),
	ASPEED_PINCTRL_PIN(L2),
	ASPEED_PINCTRL_PIN(L20),
	ASPEED_PINCTRL_PIN(L21),
	ASPEED_PINCTRL_PIN(L22),
	ASPEED_PINCTRL_PIN(L3),
	ASPEED_PINCTRL_PIN(L4),
	ASPEED_PINCTRL_PIN(L5),
	ASPEED_PINCTRL_PIN(M1),
	ASPEED_PINCTRL_PIN(M18),
	ASPEED_PINCTRL_PIN(M19),
	ASPEED_PINCTRL_PIN(M2),
	ASPEED_PINCTRL_PIN(M20),
	ASPEED_PINCTRL_PIN(M21),
	ASPEED_PINCTRL_PIN(M22),
	ASPEED_PINCTRL_PIN(M3),
	ASPEED_PINCTRL_PIN(M4),
	ASPEED_PINCTRL_PIN(M5),
	ASPEED_PINCTRL_PIN(N1),
	ASPEED_PINCTRL_PIN(N18),
	ASPEED_PINCTRL_PIN(N19),
	ASPEED_PINCTRL_PIN(N2),
	ASPEED_PINCTRL_PIN(N20),
	ASPEED_PINCTRL_PIN(N21),
	ASPEED_PINCTRL_PIN(N22),
	ASPEED_PINCTRL_PIN(N3),
	ASPEED_PINCTRL_PIN(N4),
	ASPEED_PINCTRL_PIN(N5),
	ASPEED_PINCTRL_PIN(P18),
	ASPEED_PINCTRL_PIN(P19),
	ASPEED_PINCTRL_PIN(P20),
	ASPEED_PINCTRL_PIN(P21),
	ASPEED_PINCTRL_PIN(P22),
	ASPEED_PINCTRL_PIN(P5),
	ASPEED_PINCTRL_PIN(R18),
	ASPEED_PINCTRL_PIN(R22),
	ASPEED_PINCTRL_PIN(T1),
	ASPEED_PINCTRL_PIN(T18),
	ASPEED_PINCTRL_PIN(T19),
	ASPEED_PINCTRL_PIN(T2),
	ASPEED_PINCTRL_PIN(T4),
	ASPEED_PINCTRL_PIN(T5),
	ASPEED_PINCTRL_PIN(U1),
	ASPEED_PINCTRL_PIN(U18),
	ASPEED_PINCTRL_PIN(U19),
	ASPEED_PINCTRL_PIN(U2),
	ASPEED_PINCTRL_PIN(U20),
	ASPEED_PINCTRL_PIN(U21),
	ASPEED_PINCTRL_PIN(U3),
	ASPEED_PINCTRL_PIN(U4),
	ASPEED_PINCTRL_PIN(U5),
	ASPEED_PINCTRL_PIN(V1),
	ASPEED_PINCTRL_PIN(V2),
	ASPEED_PINCTRL_PIN(V20),
	ASPEED_PINCTRL_PIN(V21),
	ASPEED_PINCTRL_PIN(V22),
	ASPEED_PINCTRL_PIN(V3),
	ASPEED_PINCTRL_PIN(V4),
	ASPEED_PINCTRL_PIN(V5),
	ASPEED_PINCTRL_PIN(V6),
	ASPEED_PINCTRL_PIN(V7),
	ASPEED_PINCTRL_PIN(W1),
	ASPEED_PINCTRL_PIN(W2),
	ASPEED_PINCTRL_PIN(W21),
	ASPEED_PINCTRL_PIN(W22),
	ASPEED_PINCTRL_PIN(W3),
	ASPEED_PINCTRL_PIN(W4),
	ASPEED_PINCTRL_PIN(W5),
	ASPEED_PINCTRL_PIN(W6),
	ASPEED_PINCTRL_PIN(W7),
	ASPEED_PINCTRL_PIN(Y1),
	ASPEED_PINCTRL_PIN(Y2),
	ASPEED_PINCTRL_PIN(Y21),
	ASPEED_PINCTRL_PIN(Y22),
	ASPEED_PINCTRL_PIN(Y3),
	ASPEED_PINCTRL_PIN(Y4),
	ASPEED_PINCTRL_PIN(Y5),
	ASPEED_PINCTRL_PIN(Y6),
	ASPEED_PINCTRL_PIN(Y7),
};

static const struct aspeed_pin_group aspeed_g4_groups[] = {
	ASPEED_PINCTRL_GROUP(ACPI),
	ASPEED_PINCTRL_GROUP(ADC0),
	ASPEED_PINCTRL_GROUP(ADC1),
	ASPEED_PINCTRL_GROUP(ADC10),
	ASPEED_PINCTRL_GROUP(ADC11),
	ASPEED_PINCTRL_GROUP(ADC12),
	ASPEED_PINCTRL_GROUP(ADC13),
	ASPEED_PINCTRL_GROUP(ADC14),
	ASPEED_PINCTRL_GROUP(ADC15),
	ASPEED_PINCTRL_GROUP(ADC2),
	ASPEED_PINCTRL_GROUP(ADC3),
	ASPEED_PINCTRL_GROUP(ADC4),
	ASPEED_PINCTRL_GROUP(ADC5),
	ASPEED_PINCTRL_GROUP(ADC6),
	ASPEED_PINCTRL_GROUP(ADC7),
	ASPEED_PINCTRL_GROUP(ADC8),
	ASPEED_PINCTRL_GROUP(ADC9),
	ASPEED_PINCTRL_GROUP(BMCINT),
	ASPEED_PINCTRL_GROUP(DDCCLK),
	ASPEED_PINCTRL_GROUP(DDCDAT),
	ASPEED_PINCTRL_GROUP(EXTRST),
	ASPEED_PINCTRL_GROUP(FLACK),
	ASPEED_PINCTRL_GROUP(FLBUSY),
	ASPEED_PINCTRL_GROUP(FLWP),
	ASPEED_PINCTRL_GROUP(GPID),
	ASPEED_PINCTRL_GROUP(GPID0),
	ASPEED_PINCTRL_GROUP(GPID2),
	ASPEED_PINCTRL_GROUP(GPID4),
	ASPEED_PINCTRL_GROUP(GPID6),
	ASPEED_PINCTRL_GROUP(GPIE0),
	ASPEED_PINCTRL_GROUP(GPIE2),
	ASPEED_PINCTRL_GROUP(GPIE4),
	ASPEED_PINCTRL_GROUP(GPIE6),
	ASPEED_PINCTRL_GROUP(I2C10),
	ASPEED_PINCTRL_GROUP(I2C11),
	ASPEED_PINCTRL_GROUP(I2C12),
	ASPEED_PINCTRL_GROUP(I2C13),
	ASPEED_PINCTRL_GROUP(I2C14),
	ASPEED_PINCTRL_GROUP(I2C3),
	ASPEED_PINCTRL_GROUP(I2C4),
	ASPEED_PINCTRL_GROUP(I2C5),
	ASPEED_PINCTRL_GROUP(I2C6),
	ASPEED_PINCTRL_GROUP(I2C7),
	ASPEED_PINCTRL_GROUP(I2C8),
	ASPEED_PINCTRL_GROUP(I2C9),
	ASPEED_PINCTRL_GROUP(LPCPD),
	ASPEED_PINCTRL_GROUP(LPCPME),
	ASPEED_PINCTRL_GROUP(LPCRST),
	ASPEED_PINCTRL_GROUP(LPCSMI),
	ASPEED_PINCTRL_GROUP(MAC1LINK),
	ASPEED_PINCTRL_GROUP(MAC2LINK),
	ASPEED_PINCTRL_GROUP(MDIO1),
	ASPEED_PINCTRL_GROUP(MDIO2),
	ASPEED_PINCTRL_GROUP(NCTS1),
	ASPEED_PINCTRL_GROUP(NCTS2),
	ASPEED_PINCTRL_GROUP(NCTS3),
	ASPEED_PINCTRL_GROUP(NCTS4),
	ASPEED_PINCTRL_GROUP(NDCD1),
	ASPEED_PINCTRL_GROUP(NDCD2),
	ASPEED_PINCTRL_GROUP(NDCD3),
	ASPEED_PINCTRL_GROUP(NDCD4),
	ASPEED_PINCTRL_GROUP(NDSR1),
	ASPEED_PINCTRL_GROUP(NDSR2),
	ASPEED_PINCTRL_GROUP(NDSR3),
	ASPEED_PINCTRL_GROUP(NDSR4),
	ASPEED_PINCTRL_GROUP(NDTR1),
	ASPEED_PINCTRL_GROUP(NDTR2),
	ASPEED_PINCTRL_GROUP(NDTR3),
	ASPEED_PINCTRL_GROUP(NDTR4),
	ASPEED_PINCTRL_GROUP(NDTS4),
	ASPEED_PINCTRL_GROUP(NRI1),
	ASPEED_PINCTRL_GROUP(NRI2),
	ASPEED_PINCTRL_GROUP(NRI3),
	ASPEED_PINCTRL_GROUP(NRI4),
	ASPEED_PINCTRL_GROUP(NRTS1),
	ASPEED_PINCTRL_GROUP(NRTS2),
	ASPEED_PINCTRL_GROUP(NRTS3),
	ASPEED_PINCTRL_GROUP(OSCCLK),
	ASPEED_PINCTRL_GROUP(PWM0),
	ASPEED_PINCTRL_GROUP(PWM1),
	ASPEED_PINCTRL_GROUP(PWM2),
	ASPEED_PINCTRL_GROUP(PWM3),
	ASPEED_PINCTRL_GROUP(PWM4),
	ASPEED_PINCTRL_GROUP(PWM5),
	ASPEED_PINCTRL_GROUP(PWM6),
	ASPEED_PINCTRL_GROUP(PWM7),
	ASPEED_PINCTRL_GROUP(RGMII1),
	ASPEED_PINCTRL_GROUP(RGMII2),
	ASPEED_PINCTRL_GROUP(RMII1),
	ASPEED_PINCTRL_GROUP(RMII2),
	ASPEED_PINCTRL_GROUP(ROM16),
	ASPEED_PINCTRL_GROUP(ROM8),
	ASPEED_PINCTRL_GROUP(ROMCS1),
	ASPEED_PINCTRL_GROUP(ROMCS2),
	ASPEED_PINCTRL_GROUP(ROMCS3),
	ASPEED_PINCTRL_GROUP(ROMCS4),
	ASPEED_PINCTRL_GROUP(RXD1),
	ASPEED_PINCTRL_GROUP(RXD2),
	ASPEED_PINCTRL_GROUP(RXD3),
	ASPEED_PINCTRL_GROUP(RXD4),
	ASPEED_PINCTRL_GROUP(SALT1),
	ASPEED_PINCTRL_GROUP(SALT2),
	ASPEED_PINCTRL_GROUP(SALT3),
	ASPEED_PINCTRL_GROUP(SALT4),
	ASPEED_PINCTRL_GROUP(SD1),
	ASPEED_PINCTRL_GROUP(SD2),
	ASPEED_PINCTRL_GROUP(SGPMCK),
	ASPEED_PINCTRL_GROUP(SGPMI),
	ASPEED_PINCTRL_GROUP(SGPMLD),
	ASPEED_PINCTRL_GROUP(SGPMO),
	ASPEED_PINCTRL_GROUP(SGPSCK),
	ASPEED_PINCTRL_GROUP(SGPSI0),
	ASPEED_PINCTRL_GROUP(SGPSI1),
	ASPEED_PINCTRL_GROUP(SGPSLD),
	ASPEED_PINCTRL_GROUP(SIOONCTRL),
	ASPEED_PINCTRL_GROUP(SIOPBI),
	ASPEED_PINCTRL_GROUP(SIOPBO),
	ASPEED_PINCTRL_GROUP(SIOPWREQ),
	ASPEED_PINCTRL_GROUP(SIOPWRGD),
	ASPEED_PINCTRL_GROUP(SIOS3),
	ASPEED_PINCTRL_GROUP(SIOS5),
	ASPEED_PINCTRL_GROUP(SIOSCI),
	ASPEED_PINCTRL_GROUP(SPI1),
	ASPEED_PINCTRL_GROUP(SPI1DEBUG),
	ASPEED_PINCTRL_GROUP(SPI1PASSTHRU),
	ASPEED_PINCTRL_GROUP(SPICS1),
	ASPEED_PINCTRL_GROUP(TIMER3),
	ASPEED_PINCTRL_GROUP(TIMER4),
	ASPEED_PINCTRL_GROUP(TIMER5),
	ASPEED_PINCTRL_GROUP(TIMER6),
	ASPEED_PINCTRL_GROUP(TIMER7),
	ASPEED_PINCTRL_GROUP(TIMER8),
	ASPEED_PINCTRL_GROUP(TXD1),
	ASPEED_PINCTRL_GROUP(TXD2),
	ASPEED_PINCTRL_GROUP(TXD3),
	ASPEED_PINCTRL_GROUP(TXD4),
	ASPEED_PINCTRL_GROUP(UART6),
	ASPEED_PINCTRL_GROUP(USB11D1),
	ASPEED_PINCTRL_GROUP(USB11H2),
	ASPEED_PINCTRL_GROUP(USB2D1),
	ASPEED_PINCTRL_GROUP(USB2H1),
	ASPEED_PINCTRL_GROUP(USBCKI),
	ASPEED_PINCTRL_GROUP(VGABIOS_ROM),
	ASPEED_PINCTRL_GROUP(VGAHS),
	ASPEED_PINCTRL_GROUP(VGAVS),
	ASPEED_PINCTRL_GROUP(VPI18),
	ASPEED_PINCTRL_GROUP(VPI24),
	ASPEED_PINCTRL_GROUP(VPI30),
	ASPEED_PINCTRL_GROUP(VPO12),
	ASPEED_PINCTRL_GROUP(VPO24),
	ASPEED_PINCTRL_GROUP(WDTRST1),
	ASPEED_PINCTRL_GROUP(WDTRST2),
};

static const struct aspeed_pin_function aspeed_g4_functions[] = {
	ASPEED_PINCTRL_FUNC(ACPI),
	ASPEED_PINCTRL_FUNC(ADC0),
	ASPEED_PINCTRL_FUNC(ADC1),
	ASPEED_PINCTRL_FUNC(ADC10),
	ASPEED_PINCTRL_FUNC(ADC11),
	ASPEED_PINCTRL_FUNC(ADC12),
	ASPEED_PINCTRL_FUNC(ADC13),
	ASPEED_PINCTRL_FUNC(ADC14),
	ASPEED_PINCTRL_FUNC(ADC15),
	ASPEED_PINCTRL_FUNC(ADC2),
	ASPEED_PINCTRL_FUNC(ADC3),
	ASPEED_PINCTRL_FUNC(ADC4),
	ASPEED_PINCTRL_FUNC(ADC5),
	ASPEED_PINCTRL_FUNC(ADC6),
	ASPEED_PINCTRL_FUNC(ADC7),
	ASPEED_PINCTRL_FUNC(ADC8),
	ASPEED_PINCTRL_FUNC(ADC9),
	ASPEED_PINCTRL_FUNC(BMCINT),
	ASPEED_PINCTRL_FUNC(DDCCLK),
	ASPEED_PINCTRL_FUNC(DDCDAT),
	ASPEED_PINCTRL_FUNC(EXTRST),
	ASPEED_PINCTRL_FUNC(FLACK),
	ASPEED_PINCTRL_FUNC(FLBUSY),
	ASPEED_PINCTRL_FUNC(FLWP),
	ASPEED_PINCTRL_FUNC(GPID),
	ASPEED_PINCTRL_FUNC(GPID0),
	ASPEED_PINCTRL_FUNC(GPID2),
	ASPEED_PINCTRL_FUNC(GPID4),
	ASPEED_PINCTRL_FUNC(GPID6),
	ASPEED_PINCTRL_FUNC(GPIE0),
	ASPEED_PINCTRL_FUNC(GPIE2),
	ASPEED_PINCTRL_FUNC(GPIE4),
	ASPEED_PINCTRL_FUNC(GPIE6),
	ASPEED_PINCTRL_FUNC(I2C10),
	ASPEED_PINCTRL_FUNC(I2C11),
	ASPEED_PINCTRL_FUNC(I2C12),
	ASPEED_PINCTRL_FUNC(I2C13),
	ASPEED_PINCTRL_FUNC(I2C14),
	ASPEED_PINCTRL_FUNC(I2C3),
	ASPEED_PINCTRL_FUNC(I2C4),
	ASPEED_PINCTRL_FUNC(I2C5),
	ASPEED_PINCTRL_FUNC(I2C6),
	ASPEED_PINCTRL_FUNC(I2C7),
	ASPEED_PINCTRL_FUNC(I2C8),
	ASPEED_PINCTRL_FUNC(I2C9),
	ASPEED_PINCTRL_FUNC(LPCPD),
	ASPEED_PINCTRL_FUNC(LPCPME),
	ASPEED_PINCTRL_FUNC(LPCRST),
	ASPEED_PINCTRL_FUNC(LPCSMI),
	ASPEED_PINCTRL_FUNC(MAC1LINK),
	ASPEED_PINCTRL_FUNC(MAC2LINK),
	ASPEED_PINCTRL_FUNC(MDIO1),
	ASPEED_PINCTRL_FUNC(MDIO2),
	ASPEED_PINCTRL_FUNC(NCTS1),
	ASPEED_PINCTRL_FUNC(NCTS2),
	ASPEED_PINCTRL_FUNC(NCTS3),
	ASPEED_PINCTRL_FUNC(NCTS4),
	ASPEED_PINCTRL_FUNC(NDCD1),
	ASPEED_PINCTRL_FUNC(NDCD2),
	ASPEED_PINCTRL_FUNC(NDCD3),
	ASPEED_PINCTRL_FUNC(NDCD4),
	ASPEED_PINCTRL_FUNC(NDSR1),
	ASPEED_PINCTRL_FUNC(NDSR2),
	ASPEED_PINCTRL_FUNC(NDSR3),
	ASPEED_PINCTRL_FUNC(NDSR4),
	ASPEED_PINCTRL_FUNC(NDTR1),
	ASPEED_PINCTRL_FUNC(NDTR2),
	ASPEED_PINCTRL_FUNC(NDTR3),
	ASPEED_PINCTRL_FUNC(NDTR4),
	ASPEED_PINCTRL_FUNC(NDTS4),
	ASPEED_PINCTRL_FUNC(NRI1),
	ASPEED_PINCTRL_FUNC(NRI2),
	ASPEED_PINCTRL_FUNC(NRI3),
	ASPEED_PINCTRL_FUNC(NRI4),
	ASPEED_PINCTRL_FUNC(NRTS1),
	ASPEED_PINCTRL_FUNC(NRTS2),
	ASPEED_PINCTRL_FUNC(NRTS3),
	ASPEED_PINCTRL_FUNC(OSCCLK),
	ASPEED_PINCTRL_FUNC(PWM0),
	ASPEED_PINCTRL_FUNC(PWM1),
	ASPEED_PINCTRL_FUNC(PWM2),
	ASPEED_PINCTRL_FUNC(PWM3),
	ASPEED_PINCTRL_FUNC(PWM4),
	ASPEED_PINCTRL_FUNC(PWM5),
	ASPEED_PINCTRL_FUNC(PWM6),
	ASPEED_PINCTRL_FUNC(PWM7),
	ASPEED_PINCTRL_FUNC(RGMII1),
	ASPEED_PINCTRL_FUNC(RGMII2),
	ASPEED_PINCTRL_FUNC(RMII1),
	ASPEED_PINCTRL_FUNC(RMII2),
	ASPEED_PINCTRL_FUNC(ROM16),
	ASPEED_PINCTRL_FUNC(ROM8),
	ASPEED_PINCTRL_FUNC(ROMCS1),
	ASPEED_PINCTRL_FUNC(ROMCS2),
	ASPEED_PINCTRL_FUNC(ROMCS3),
	ASPEED_PINCTRL_FUNC(ROMCS4),
	ASPEED_PINCTRL_FUNC(RXD1),
	ASPEED_PINCTRL_FUNC(RXD2),
	ASPEED_PINCTRL_FUNC(RXD3),
	ASPEED_PINCTRL_FUNC(RXD4),
	ASPEED_PINCTRL_FUNC(SALT1),
	ASPEED_PINCTRL_FUNC(SALT2),
	ASPEED_PINCTRL_FUNC(SALT3),
	ASPEED_PINCTRL_FUNC(SALT4),
	ASPEED_PINCTRL_FUNC(SD1),
	ASPEED_PINCTRL_FUNC(SD2),
	ASPEED_PINCTRL_FUNC(SGPMCK),
	ASPEED_PINCTRL_FUNC(SGPMI),
	ASPEED_PINCTRL_FUNC(SGPMLD),
	ASPEED_PINCTRL_FUNC(SGPMO),
	ASPEED_PINCTRL_FUNC(SGPSCK),
	ASPEED_PINCTRL_FUNC(SGPSI0),
	ASPEED_PINCTRL_FUNC(SGPSI1),
	ASPEED_PINCTRL_FUNC(SGPSLD),
	ASPEED_PINCTRL_FUNC(SIOONCTRL),
	ASPEED_PINCTRL_FUNC(SIOPBI),
	ASPEED_PINCTRL_FUNC(SIOPBO),
	ASPEED_PINCTRL_FUNC(SIOPWREQ),
	ASPEED_PINCTRL_FUNC(SIOPWRGD),
	ASPEED_PINCTRL_FUNC(SIOS3),
	ASPEED_PINCTRL_FUNC(SIOS5),
	ASPEED_PINCTRL_FUNC(SIOSCI),
	ASPEED_PINCTRL_FUNC(SPI1),
	ASPEED_PINCTRL_FUNC(SPI1DEBUG),
	ASPEED_PINCTRL_FUNC(SPI1PASSTHRU),
	ASPEED_PINCTRL_FUNC(SPICS1),
	ASPEED_PINCTRL_FUNC(TIMER3),
	ASPEED_PINCTRL_FUNC(TIMER4),
	ASPEED_PINCTRL_FUNC(TIMER5),
	ASPEED_PINCTRL_FUNC(TIMER6),
	ASPEED_PINCTRL_FUNC(TIMER7),
	ASPEED_PINCTRL_FUNC(TIMER8),
	ASPEED_PINCTRL_FUNC(TXD1),
	ASPEED_PINCTRL_FUNC(TXD2),
	ASPEED_PINCTRL_FUNC(TXD3),
	ASPEED_PINCTRL_FUNC(TXD4),
	ASPEED_PINCTRL_FUNC(UART6),
	ASPEED_PINCTRL_FUNC(USB11D1),
	ASPEED_PINCTRL_FUNC(USB11H2),
	ASPEED_PINCTRL_FUNC(USB2D1),
	ASPEED_PINCTRL_FUNC(USB2H1),
	ASPEED_PINCTRL_FUNC(USBCKI),
	ASPEED_PINCTRL_FUNC(VGABIOS_ROM),
	ASPEED_PINCTRL_FUNC(VGAHS),
	ASPEED_PINCTRL_FUNC(VGAVS),
	ASPEED_PINCTRL_FUNC(VPI18),
	ASPEED_PINCTRL_FUNC(VPI24),
	ASPEED_PINCTRL_FUNC(VPI30),
	ASPEED_PINCTRL_FUNC(VPO12),
	ASPEED_PINCTRL_FUNC(VPO24),
	ASPEED_PINCTRL_FUNC(WDTRST1),
	ASPEED_PINCTRL_FUNC(WDTRST2),
};

static const struct aspeed_pin_config aspeed_g4_configs[] = {
	/* GPIO banks ranges [A, B], [D, J], [M, R] */
	{ PIN_CONFIG_BIAS_PULL_DOWN, { D6,  D5  }, SCU8C, 16 },
	{ PIN_CONFIG_BIAS_DISABLE,   { D6,  D5  }, SCU8C, 16 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { J21, E18 }, SCU8C, 17 },
	{ PIN_CONFIG_BIAS_DISABLE,   { J21, E18 }, SCU8C, 17 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { A18, E15 }, SCU8C, 19 },
	{ PIN_CONFIG_BIAS_DISABLE,   { A18, E15 }, SCU8C, 19 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { D15, B14 }, SCU8C, 20 },
	{ PIN_CONFIG_BIAS_DISABLE,   { D15, B14 }, SCU8C, 20 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { D18, C17 }, SCU8C, 21 },
	{ PIN_CONFIG_BIAS_DISABLE,   { D18, C17 }, SCU8C, 21 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { A14, U18 }, SCU8C, 22 },
	{ PIN_CONFIG_BIAS_DISABLE,   { A14, U18 }, SCU8C, 22 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { A8,  E7  }, SCU8C, 23 },
	{ PIN_CONFIG_BIAS_DISABLE,   { A8,  E7  }, SCU8C, 23 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { C22, E20 }, SCU8C, 24 },
	{ PIN_CONFIG_BIAS_DISABLE,   { C22, E20 }, SCU8C, 24 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { J5,  T1  }, SCU8C, 25 },
	{ PIN_CONFIG_BIAS_DISABLE,   { J5,  T1  }, SCU8C, 25 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { U1,  U5  }, SCU8C, 26 },
	{ PIN_CONFIG_BIAS_DISABLE,   { U1,  U5  }, SCU8C, 26 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { V3,  V5  }, SCU8C, 27 },
	{ PIN_CONFIG_BIAS_DISABLE,   { V3,  V5  }, SCU8C, 27 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { W4,  AB2 }, SCU8C, 28 },
	{ PIN_CONFIG_BIAS_DISABLE,   { W4,  AB2 }, SCU8C, 28 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { V6,  V7  }, SCU8C, 29 },
	{ PIN_CONFIG_BIAS_DISABLE,   { V6,  V7  }, SCU8C, 29 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { Y6,  AB7 }, SCU8C, 30 },
	{ PIN_CONFIG_BIAS_DISABLE,   { Y6,  AB7 }, SCU8C, 30 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { V20, A5  }, SCU8C, 31 },
	{ PIN_CONFIG_BIAS_DISABLE,   { V20, A5  }, SCU8C, 31 },

	/* GPIOs T[0-5] (RGMII1 Tx pins) */
	{ PIN_CONFIG_DRIVE_STRENGTH, { A12, A13 }, SCU90, 9  },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { A12, A13 }, SCU90, 12 },
	{ PIN_CONFIG_BIAS_DISABLE,   { A12, A13 }, SCU90, 12 },

	/* GPIOs T[6-7], U[0-3] (RGMII2 TX pins) */
	{ PIN_CONFIG_DRIVE_STRENGTH, { D9,  D10 }, SCU90, 11 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { D9,  D10 }, SCU90, 14 },
	{ PIN_CONFIG_BIAS_DISABLE,   { D9,  D10 }, SCU90, 14 },

	/* GPIOs U[4-7], V[0-1] (RGMII1 Rx pins) */
	{ PIN_CONFIG_BIAS_PULL_DOWN, { E11, E10 }, SCU90, 13 },
	{ PIN_CONFIG_BIAS_DISABLE,   { E11, E10 }, SCU90, 13 },

	/* GPIOs V[2-7] (RGMII2 Rx pins) */
	{ PIN_CONFIG_BIAS_PULL_DOWN, { C9,  C8  }, SCU90, 15 },
	{ PIN_CONFIG_BIAS_DISABLE,   { C9,  C8  }, SCU90, 15 },

	/* ADC pull-downs (SCUA8[19:4]) */
	{ PIN_CONFIG_BIAS_PULL_DOWN, { L5,  L5  }, SCUA8, 4 },
	{ PIN_CONFIG_BIAS_DISABLE,   { L5,  L5  }, SCUA8, 4 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { L4,  L4  }, SCUA8, 5 },
	{ PIN_CONFIG_BIAS_DISABLE,   { L4,  L4  }, SCUA8, 5 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { L3,  L3  }, SCUA8, 6 },
	{ PIN_CONFIG_BIAS_DISABLE,   { L3,  L3  }, SCUA8, 6 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { L2,  L2  }, SCUA8, 7 },
	{ PIN_CONFIG_BIAS_DISABLE,   { L2,  L2  }, SCUA8, 7 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { L1,  L1  }, SCUA8, 8 },
	{ PIN_CONFIG_BIAS_DISABLE,   { L1,  L1  }, SCUA8, 8 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { M5,  M5  }, SCUA8, 9 },
	{ PIN_CONFIG_BIAS_DISABLE,   { M5,  M5  }, SCUA8, 9 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { M4,  M4  }, SCUA8, 10 },
	{ PIN_CONFIG_BIAS_DISABLE,   { M4,  M4  }, SCUA8, 10 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { M3,  M3  }, SCUA8, 11 },
	{ PIN_CONFIG_BIAS_DISABLE,   { M3,  M3  }, SCUA8, 11 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { M2,  M2  }, SCUA8, 12 },
	{ PIN_CONFIG_BIAS_DISABLE,   { M2,  M2  }, SCUA8, 12 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { M1,  M1  }, SCUA8, 13 },
	{ PIN_CONFIG_BIAS_DISABLE,   { M1,  M1  }, SCUA8, 13 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { N5,  N5  }, SCUA8, 14 },
	{ PIN_CONFIG_BIAS_DISABLE,   { N5,  N5  }, SCUA8, 14 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { N4,  N4  }, SCUA8, 15 },
	{ PIN_CONFIG_BIAS_DISABLE,   { N4,  N4  }, SCUA8, 15 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { N3,  N3  }, SCUA8, 16 },
	{ PIN_CONFIG_BIAS_DISABLE,   { N3,  N3  }, SCUA8, 16 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { N2,  N2  }, SCUA8, 17 },
	{ PIN_CONFIG_BIAS_DISABLE,   { N2,  N2  }, SCUA8, 17 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { N1,  N1  }, SCUA8, 18 },
	{ PIN_CONFIG_BIAS_DISABLE,   { N1,  N1  }, SCUA8, 18 },
	{ PIN_CONFIG_BIAS_PULL_DOWN, { P5,  P5  }, SCUA8, 19 },
	{ PIN_CONFIG_BIAS_DISABLE,   { P5,  P5  }, SCUA8, 19 },

	/*
	 * Debounce settings for GPIOs D and E passthrough mode are in
	 * SCUA8[27:20] and so are managed by pinctrl. Normal GPIO debounce for
	 * banks D and E is handled by the GPIO driver - GPIO passthrough is
	 * treated like any other non-GPIO mux function. There is a catch
	 * however, in that the debounce period is configured in the GPIO
	 * controller. Due to this tangle between GPIO and pinctrl we don't yet
	 * fully support pass-through debounce.
	 */
	{ PIN_CONFIG_INPUT_DEBOUNCE, { A18, D16 }, SCUA8, 20 },
	{ PIN_CONFIG_INPUT_DEBOUNCE, { B17, A17 }, SCUA8, 21 },
	{ PIN_CONFIG_INPUT_DEBOUNCE, { C16, B16 }, SCUA8, 22 },
	{ PIN_CONFIG_INPUT_DEBOUNCE, { A16, E15 }, SCUA8, 23 },
	{ PIN_CONFIG_INPUT_DEBOUNCE, { D15, C15 }, SCUA8, 24 },
	{ PIN_CONFIG_INPUT_DEBOUNCE, { B15, A15 }, SCUA8, 25 },
	{ PIN_CONFIG_INPUT_DEBOUNCE, { E14, D14 }, SCUA8, 26 },
	{ PIN_CONFIG_INPUT_DEBOUNCE, { C14, B14 }, SCUA8, 27 },
};

static int aspeed_g4_sig_expr_set(struct aspeed_pinmux_data *ctx,
				  const struct aspeed_sig_expr *expr,
				  bool enable)
{
	int ret;
	int i;

	for (i = 0; i < expr->ndescs; i++) {
		const struct aspeed_sig_desc *desc = &expr->descs[i];
		u32 pattern = enable ? desc->enable : desc->disable;
		u32 val = (pattern << __ffs(desc->mask));

		if (!ctx->maps[desc->ip])
			return -ENODEV;

		/*
		 * Strap registers are configured in hardware or by early-boot
		 * firmware. Treat them as read-only despite that we can write
		 * them. This may mean that certain functions cannot be
		 * deconfigured and is the reason we re-evaluate after writing
		 * all descriptor bits.
		 *
		 * Port D and port E GPIO loopback modes are the only exception
		 * as those are commonly used with front-panel buttons to allow
		 * normal operation of the host when the BMC is powered off or
		 * fails to boot. Once the BMC has booted, the loopback mode
		 * must be disabled for the BMC to control host power-on and
		 * reset.
		 */
		if (desc->ip == ASPEED_IP_SCU && desc->reg == HW_STRAP1 &&
		    !(desc->mask & (BIT(21) | BIT(22))))
			continue;

		if (desc->ip == ASPEED_IP_SCU && desc->reg == HW_STRAP2)
			continue;

		ret = regmap_update_bits(ctx->maps[desc->ip], desc->reg,
					 desc->mask, val);

		if (ret)
			return ret;
	}

	ret = aspeed_sig_expr_eval(ctx, expr, enable);
	if (ret < 0)
		return ret;

	if (!ret)
		return -EPERM;

	return 0;
}

static const struct aspeed_pinmux_ops aspeed_g4_ops = {
	.set = aspeed_g4_sig_expr_set,
};

static struct aspeed_pinctrl_data aspeed_g4_pinctrl_data = {
	.pins = aspeed_g4_pins,
	.npins = ARRAY_SIZE(aspeed_g4_pins),
	.pinmux = {
		.ops = &aspeed_g4_ops,
		.groups = aspeed_g4_groups,
		.ngroups = ARRAY_SIZE(aspeed_g4_groups),
		.functions = aspeed_g4_functions,
		.nfunctions = ARRAY_SIZE(aspeed_g4_functions),
	},
	.configs = aspeed_g4_configs,
	.nconfigs = ARRAY_SIZE(aspeed_g4_configs),
};

static const struct pinmux_ops aspeed_g4_pinmux_ops = {
	.get_functions_count = aspeed_pinmux_get_fn_count,
	.get_function_name = aspeed_pinmux_get_fn_name,
	.get_function_groups = aspeed_pinmux_get_fn_groups,
	.set_mux = aspeed_pinmux_set_mux,
	.gpio_request_enable = aspeed_gpio_request_enable,
	.strict = true,
};

static const struct pinctrl_ops aspeed_g4_pinctrl_ops = {
	.get_groups_count = aspeed_pinctrl_get_groups_count,
	.get_group_name = aspeed_pinctrl_get_group_name,
	.get_group_pins = aspeed_pinctrl_get_group_pins,
	.pin_dbg_show = aspeed_pinctrl_pin_dbg_show,
	.dt_node_to_map = pinconf_generic_dt_node_to_map_all,
	.dt_free_map = pinctrl_utils_free_map,
};

static const struct pinconf_ops aspeed_g4_conf_ops = {
	.is_generic = true,
	.pin_config_get = aspeed_pin_config_get,
	.pin_config_set = aspeed_pin_config_set,
	.pin_config_group_get = aspeed_pin_config_group_get,
	.pin_config_group_set = aspeed_pin_config_group_set,
};

static struct pinctrl_desc aspeed_g4_pinctrl_desc = {
	.name = "aspeed-g4-pinctrl",
	.pins = aspeed_g4_pins,
	.npins = ARRAY_SIZE(aspeed_g4_pins),
	.pctlops = &aspeed_g4_pinctrl_ops,
	.pmxops = &aspeed_g4_pinmux_ops,
	.confops = &aspeed_g4_conf_ops,
};

static int aspeed_g4_pinctrl_probe(struct platform_device *pdev)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(aspeed_g4_pins); i++)
		aspeed_g4_pins[i].number = i;

	return aspeed_pinctrl_probe(pdev, &aspeed_g4_pinctrl_desc,
			&aspeed_g4_pinctrl_data);
}

static const struct of_device_id aspeed_g4_pinctrl_of_match[] = {
	{ .compatible = "aspeed,ast2400-pinctrl", },
	/*
	 * The aspeed,g4-pinctrl compatible has been removed the from the
	 * bindings, but keep the match in case of old devicetrees.
	 */
	{ .compatible = "aspeed,g4-pinctrl", },
	{ },
};

static struct platform_driver aspeed_g4_pinctrl_driver = {
	.probe = aspeed_g4_pinctrl_probe,
	.driver = {
		.name = "aspeed-g4-pinctrl",
		.of_match_table = aspeed_g4_pinctrl_of_match,
	},
};

static int aspeed_g4_pinctrl_init(void)
{
	return platform_driver_register(&aspeed_g4_pinctrl_driver);
}

arch_initcall(aspeed_g4_pinctrl_init);
