; -*- fundamental -*- (asm-mode sucks)
; $Id: isolinux.asm,v 1.45 2001/12/14 08:21:08 hpa Exp $
; ****************************************************************************
;
;  isolinux.asm
;
;  A program to boot Linux kernels off a CD-ROM using the El Torito
;  boot standard in "no emulation" mode, making the entire filesystem
;  available.  It is based on the SYSLINUX boot loader for MS-DOS
;  floppies.
;
;   Copyright (C) 1994-2001  H. Peter Anvin
;
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
;  USA; either version 2 of the License, or (at your option) any later
;  version; incorporated herein by reference.
; 
; ****************************************************************************

; Note: The Makefile builds one version with DEBUG_MESSAGES automatically.
; %define DEBUG_TRACERS			; Uncomment to get debugging tracers
; %define DEBUG_MESSAGES		; Uncomment to get debugging messages

%ifdef DEBUG_TRACERS

%macro TRACER	1
	call debug_tracer
	db %1
%endmacro

%else	; DEBUG_TRACERS

%macro	TRACER	1
%endmacro

%endif	; DEBUG_TRACERS

;
; Some semi-configurable constants... change on your own risk.  Most are imposed
; by the kernel.
;
max_cmd_len	equ 255			; Must be odd; 255 is the kernel limit
FILENAME_MAX_LG2 equ 8			; log2(Max filename size Including final null)
FILENAME_MAX	equ (1 << FILENAME_MAX_LG2)
HIGHMEM_MAX	equ 037FFFFFFh		; DEFAULT highest address for an initrd
HIGHMEM_SLOP	equ 128*1024		; Avoid this much memory near the top
DEFAULT_BAUD	equ 9600		; Default baud rate for serial port
BAUD_DIVISOR	equ 115200		; Serial port parameter
MAX_OPEN_LG2	equ 6			; log2(Max number of open files)
MAX_OPEN	equ (1 << MAX_OPEN_LG2)
SECTORSIZE_LG2	equ 11			; 2048 bytes/sector (El Torito requirement)
SECTORSIZE	equ (1 << SECTORSIZE_LG2)

;
; Should be updated with every release to avoid bootsector/SYS file mismatch
;
%define	version_str	VERSION		; Must be 4 characters long!
%define date		DATE_STR	; Defined from the Makefile
%define	year		'2001'
;
; Debgging stuff
;
; %define debug 1			; Uncomment to enable debugging
;
; ID for SYSLINUX (reported to kernel)
;
syslinux_id	equ 033h		; SYSLINUX (3) 3 = ISOLINUX

;
; Segments used by Linux
;
real_mode_seg	equ 5000h
fake_setup_seg	equ real_mode_seg+020h

		struc real_mode_seg_t
		resb 20h-($-$$)		; org 20h
kern_cmd_magic	resw 1			; 0020 Magic # for command line
kern_cmd_offset resw 1			; 0022 Offset for kernel command line
		resb 497-($-$$)		; org 497d
bs_setupsecs	resb 1			; 01F1 Sectors for setup code (0 -> 4)
bs_rootflags	resw 1			; 01F2 Root readonly flag
bs_syssize	resw 1			; 01F4
bs_swapdev	resw 1			; 01F6 Swap device (obsolete)
bs_ramsize	resw 1			; 01F8 Ramdisk flags, formerly ramdisk size
bs_vidmode	resw 1			; 01FA Video mode
bs_rootdev	resw 1			; 01FC Root device
bs_bootsign	resw 1			; 01FE Boot sector signature (0AA55h)
su_jump		resb 1			; 0200 0EBh
su_jump2	resb 1			; 0201 Size of following header
su_header	resd 1			; 0202 New setup code: header
su_version	resw 1			; 0206 See linux/arch/i386/boot/setup.S
su_switch	resw 1			; 0208
su_setupseg	resw 1			; 020A
su_startsys	resw 1			; 020C
su_kver		resw 1			; 020E Kernel version pointer
su_loader	resb 1			; 0210 Loader ID
su_loadflags	resb 1			; 0211 Load high flag
su_movesize	resw 1			; 0212
su_code32start	resd 1			; 0214 Start of code loaded high
su_ramdiskat	resd 1			; 0218 Start of initial ramdisk
su_ramdisklen	equ $			; Length of initial ramdisk
su_ramdisklen1	resw 1			; 021C
su_ramdisklen2	resw 1			; 021E
su_bsklugeoffs	resw 1			; 0220
su_bsklugeseg	resw 1			; 0222
su_heapend	resw 1			; 0224
su_pad1		resw 1			; 0226
su_cmd_line_ptr	resd 1			; 0228
su_ramdisk_max	resd 1			; 022C
		resb (9000h-12)-($-$$)	; Were bootsect.S puts it...
linux_stack	equ $			; 8FF4
linux_fdctab	equ $
		resb 9000h-($-$$)
cmd_line_here	equ $			; 9000 Should be out of the way
		endstruc

;
; Kernel command line signature
;
CMD_MAGIC	equ 0A33Fh		; Command line magic

;
; Magic number of su_header field
;
HEADER_ID       equ 'HdrS'		; HdrS (in littleendian hex)

;
; Flags for the su_loadflags field
;
LOAD_HIGH	equ 01h			; Large kernel, load high
CAN_USE_HEAP    equ 80h                 ; Boot loader reports heap size

;
; The following structure is used for "virtual kernels"; i.e. LILO-style
; option labels.  The options we permit here are `kernel' and `append
; Since there is no room in the bottom 64K for all of these, we
; stick them at vk_seg:0000 and copy them down before we need them.
;
; Note: this structure can be added to, but it must 
;
%define vk_power	6		; log2(max number of vkernels)
%define	max_vk		(1 << vk_power)	; Maximum number of vkernels
%define vk_shift	(16-vk_power)	; Number of bits to shift
%define vk_size		(1 << vk_shift)	; Size of a vkernel buffer

		struc vkernel
vk_vname:	resb FILENAME_MAX	; Virtual name **MUST BE FIRST!**
vk_rname:	resb FILENAME_MAX	; Real name
vk_appendlen:	resw 1
		alignb 4
vk_append:	resb max_cmd_len+1	; Command line
		alignb 4
vk_end:		equ $			; Should be <= vk_size
		endstruc

%if (vk_end > vk_size) || (vk_size*max_vk > 65536)
%error "Too many vkernels defined, reduce vk_power"
%endif

;
; Segment assignments in the bottom 640K
; 0000h - main code/data segment (and BIOS segment)
; 5000h - real_mode_seg
;
vk_seg          equ 4000h		; Virtual kernels
xfer_buf_seg	equ 3000h		; Bounce buffer for I/O to high mem
comboot_seg	equ 2000h		; COMBOOT image loading zone

;
; File structure.  This holds the information for each currently open file.
;
		struc open_file_t
file_sector	resd 1			; Sector pointer (0 = structure free)
file_left	resd 1			; Number of sectors left
		endstruc

%if (open_file_t_size & (open_file_t_size-1))
%error "open_file_t is not a power of 2"
%endif

;
; For our convenience: define macros for jump-over-unconditinal jumps
;
%macro	jmpz	1
	jnz %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpnz	1
	jz %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpe	1
	jne %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpne	1
	je %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpc	1
	jnc %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpnc	1
	jc %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpb	1
	jnb %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpnb	1
	jb %%skip
	jmp %1
%%skip:
%endmacro

;
; Macros similar to res[bwd], but which works in the code segment (after
; section .text)
;
%macro	zb	1
	times %1 db 0
%endmacro

%macro	zw	1
	times %1 dw 0
%endmacro

%macro	zd	1
	times %1 dd 0
%endmacro

; ---------------------------------------------------------------------------
;   BEGIN THE BIOS/CODE/DATA SEGMENT
; ---------------------------------------------------------------------------

		absolute 0400h
serial_base	resw 4			; Base addresses for 4 serial ports
		absolute 0413h
BIOS_fbm	resw 1			; Free Base Memory (kilobytes)
		absolute 046Ch
BIOS_timer	resw 1			; Timer ticks
		absolute 0472h
BIOS_magic	resw 1			; BIOS reset magic
                absolute 0484h
BIOS_vidrows    resb 1			; Number of screen rows

;
; Memory below this point is reserved for the BIOS and the MBR
;
 		absolute 1000h
trackbuf	resb 8192		; Track buffer goes here
trackbufsize	equ $-trackbuf
;		trackbuf ends at 3000h


;
; Constants for the xfer_buf_seg
;
; The xfer_buf_seg is also used to store message file buffers.  We
; need two trackbuffers (text and graphics), plus a work buffer
; for the graphics decompressor.
;
xbs_textbuf	equ 0			; Also hard-coded, do not change
xbs_vgabuf	equ trackbufsize
xbs_vgatmpbuf	equ 2*trackbufsize

		struc dir_t
dir_lba		resd 1			; Directory start (LBA)
dir_len		resd 1			; Length in bytes
dir_clust	resd 1			; Length in clusters
		endstruc

                absolute 5000h          ; Here we keep our BSS stuff
VKernelBuf:	resb vk_size		; "Current" vkernel
		alignb 4
AppendBuf       resb max_cmd_len+1	; append=
KbdMap		resb 256		; Keyboard map
FKeyName	resb 10*FILENAME_MAX	; File names for F-key help
NumBuf		resb 15			; Buffer to load number
NumBufEnd	resb 1			; Last byte in NumBuf
ISOFileName	resb 64			; ISO filename canonicalization buffer
ISOFileNameEnd	equ $
		alignb 32
KernelName      resb FILENAME_MAX       ; Mangled name for kernel
KernelCName     resb FILENAME_MAX	; Unmangled kernel name
InitRDCName     resb FILENAME_MAX       ; Unmangled initrd name
MNameBuf	resb FILENAME_MAX
InitRD		resb FILENAME_MAX
PartInfo	resb 16			; Partition table entry
E820Buf		resd 5			; INT 15:E820 data buffer
InitRDat	resd 1			; Load address (linear) for initrd
HiLoadAddr      resd 1			; Address pointer for high load loop
HighMemSize	resd 1			; End of memory pointer (bytes)
RamdiskMax	resd 1			; Highest address for a ramdisk
KernelSize	resd 1			; Size of kernel (bytes)
RootDir		resb dir_t_size		; Root directory
CurDir		resb dir_t_size		; Current directory
SavedSSSP	resd 1			; Our SS:SP while running a COMBOOT image
KernelClust	resd 1			; Kernel size in clusters
InitRDClust	resd 1			; Ramdisk size in clusters
InitStack	resd 1			; Initial stack pointer (SS:SP)
FirstSecSum	resd 1			; Checksum of bytes 64-2048
ImageDwords	resd 1			; isolinux.bin size, dwords
FBytes		equ $			; Used by open/getc
FBytes1		resw 1
FBytes2		resw 1
FClust		resw 1			; Number of clusters in open/getc file
FNextClust	resw 1			; Pointer to next cluster in d:o
FPtr		resw 1			; Pointer to next char in buffer
CmdOptPtr       resw 1			; Pointer to first option on cmd line
KernelCNameLen  resw 1			; Length of unmangled kernel name
InitRDCNameLen  resw 1			; Length of unmangled initrd name
NextCharJump    resw 1			; Routine to interpret next print char
SetupSecs	resw 1			; Number of setup sectors
A20Test		resw 1			; Counter for testing status of A20
CmdLineLen	resw 1			; Length of command line including null
GraphXSize	resw 1			; Width of splash screen file
VGAPos		resw 1			; Pointer into VGA memory
VGACluster	resw 1			; Cluster pointer for VGA image file
VGAFilePtr	resw 1			; Pointer into VGAFileBuf
ConfigFile	resw 1			; Socket for config file
PktTimeout	resw 1			; Timeout for current packet
KernelExtPtr	resw 1			; During search, final null pointer
LocalBootType	resw 1			; Local boot return code
ImageSectors	resw 1			; isolinux.bin size, sectors
TextAttrBX      equ $
TextAttribute   resb 1			; Text attribute for message file
TextPage        resb 1			; Active display page
CursorDX        equ $
CursorCol       resb 1			; Cursor column for message file
CursorRow       resb 1			; Cursor row for message file
ScreenSize      equ $
VidCols         resb 1			; Columns on screen-1
VidRows         resb 1			; Rows on screen-1
FlowControl	equ $
FlowOutput	resb 1			; Outputs to assert for serial flow
FlowInput	resb 1			; Input bits for serial flow
FlowIgnore	resb 1			; Ignore input unless these bits set
RetryCount      resb 1			; Used for disk access retries
KbdFlags	resb 1			; Check for keyboard escapes
LoadFlags	resb 1			; Loadflags from kernel
A20Tries	resb 1			; Times until giving up on A20
FuncFlag	resb 1			; == 1 if <Ctrl-F> pressed
DisplayMask	resb 1			; Display modes mask
ISOFlags	resb 1			; Flags for ISO directory search
DiskError	resb 1			; Error code for disk I/O
DriveNo		resb 1			; CD-ROM BIOS drive number
TextColorReg	resb 17			; VGA color registers for text mode
VGAFileBuf	resb FILENAME_MAX	; Unmangled VGA image name
VGAFileBufEnd	equ $
VGAFileMBuf	resb FILENAME_MAX	; Mangled VGA image name

		alignb open_file_t_size
Files		resb MAX_OPEN*open_file_t_size

		section .text
                org 7C00h
;;
;; Primary entry point.  Because BIOSes are buggy, we only load the first
;; CD-ROM sector (2K) of the file, so the number one priority is actually
;; loading the rest.
;;
bootsec		equ $
_start:		; Far jump makes sure we canonicalize the address
		cli
		jmp 0:_start1
		times 8-($-$$) nop		; Pad to file offset 8

		; This table gets filled in by mkisofs using the
		; -boot-info-table option
bi_pvd:		dd 0xdeadbeef			; LBA of primary volume descriptor
bi_file:	dd 0xdeadbeef			; LBA of boot file
bi_length:	dd 0xdeadbeef			; Length of boot file
bi_csum:	dd 0xdeadbeef			; Checksum of boot file
bi_reserved:	times 10 dd 0xdeadbeef		; Reserved

_start1:	mov [cs:InitStack],sp		; Save initial stack pointer
		mov ax,ss
		mov [cs:InitStack+2],ax
		xor ax,ax
		mov ss,ax
		mov sp,_start			; Set up stack
		mov ds,ax
		mov es,ax
		mov fs,ax
		mov gs,ax
		sti

		cld
		; Show signs of life
		mov si,isolinux_banner
		call writestr
%ifdef DEBUG_MESSAGES
		mov si,copyright_str
		call writestr
%endif

		;
		; Before modifying any memory, get the checksum of bytes
		; 64-2048
		;
initial_csum:	xor edi,edi
		mov si,_start1
		mov cx,(SECTORSIZE-64) >> 2
.loop:		lodsd
		add edi,eax
		loop .loop
		mov [FirstSecSum],edi

		; Set up boot file sizes
		mov eax,[bi_length]
		sub eax,SECTORSIZE-3
		shr eax,2			; bytes->dwords
		mov [ImageDwords],eax		; boot file dwords
		add eax,(2047 >> 2)
		shr eax,9			; dwords->sectors
		mov [ImageSectors],ax		; boot file sectors

		mov [DriveNo],dl
%ifdef DEBUG_MESSAGES
		mov si,startup_msg
		call writemsg
		mov al,dl
		call writehex2
		call crlf
%endif

		; Now figure out what we're actually doing
		; Note: use passed-in DL value rather than 7Fh because
		; at least some BIOSes will get the wrong value otherwise
		mov ax,4B01h			; Get disk emulation status
		mov dl,[DriveNo]
		mov si,spec_packet
		int 13h
		jc near spec_query_failed	; Shouldn't happen (BIOS bug)
		mov dl,[DriveNo]
		cmp [sp_drive],dl		; Should contain the drive number
		jne near spec_query_failed

%ifdef DEBUG_MESSAGES
		mov si,spec_ok_msg
		call writemsg
		mov al,byte [sp_drive]
		call writehex2
		call crlf
%endif

found_drive:
		; Get drive information
		mov ah,48h
		mov dl,[DriveNo]
		mov si,drive_params
		int 13h
		jnc params_ok

		mov si,nosecsize_msg
		call writemsg

params_ok:
		; Check for the sector size (should be 2048, but
		; some BIOSes apparently think we're 512-byte media)
		;
		; FIX: We need to check what the proper behaviour
		; is for getlinsec when the BIOS thinks the sector
		; size is 512!!!  For that, we need such a BIOS, though...
%ifdef DEBUG_MESSAGES
		mov si,secsize_msg
		call writemsg
		mov ax,[dp_secsize]
		call writehex4
		call crlf
%endif

load_image:
		; Some BIOSes apparently have limitations on the size 
		; that may be loaded (despite the El Torito spec being very
		; clear on the fact that it must all be loaded.)  Therefore,
		; we load it ourselves, and *bleep* the BIOS.

		mov eax,[bi_file]		; Address of code to load
		inc eax				; Don't reload bootstrap code
%ifdef DEBUG_MESSAGES
		mov si,offset_msg
		call writemsg
		call writehex8
		call crlf
%endif

		; Just in case some BIOSes have problems with
		; segment wraparound, use the normalized address
		mov bx,((7C00h+2048) >> 4)
		mov es,bx
		xor bx,bx
		mov bp,[ImageSectors]
%ifdef DEBUG_MESSAGES
		push ax
		mov si,size_msg
		call writemsg
		mov ax,bp
		call writehex4
		call crlf
		pop ax
%endif
		call getlinsec

		push ds
		pop es

%ifdef DEBUG_MESSAGES
		mov si,loaded_msg
		call writemsg
%endif

		; Verify the checksum on the loaded image.
verify_image:
		mov si,7C00h+2048
		mov bx,es
		mov ecx,[ImageDwords]
		mov edi,[FirstSecSum]		; First sector checksum
.loop		es lodsd
		add edi,eax
		dec ecx
		jz .done
		and si,si
		jnz .loop
		; SI wrapped around, advance ES
		add bx,1000h
		mov es,bx
		jmp short .loop
.done:		mov ax,ds
		mov es,ax
		cmp [bi_csum],edi
		je integrity_ok

		mov si,checkerr_msg
		call writemsg
		jmp kaboom

integrity_ok:
%ifdef DEBUG_MESSAGES
		mov si,allread_msg
		call writemsg
%endif
		jmp all_read			; Jump to main code

		; INT 13h, AX=4B01h, DL=<passed in value> failed.
		; Try to scan the entire 80h-FFh from the end.
spec_query_failed:
		mov si,spec_err_msg
		call writemsg

		mov dl,0FFh
.test_loop:	pusha
		mov ax,4B01h
		mov si,spec_packet
		mov byte [si],13		; Size of buffer
		int 13h
		popa
		jc .still_broken

		mov si,maybe_msg
		call writemsg
		mov al,dl
		call writehex2
		call crlf

		cmp byte [sp_drive],dl
		jne .maybe_broken

		; Okay, good enough...
		mov si,alright_msg
		call writemsg
		mov [DriveNo],dl
.found_drive:	jmp found_drive

		; Award BIOS 4.51 apparently passes garbage in sp_drive,
		; but if this was the drive number originally passed in
		; DL then consider it "good enough"
.maybe_broken:
		cmp byte [DriveNo],dl
		je .found_drive

.still_broken:	dec dx
		cmp dl, 80h
		jnb .test_loop

fatal_error:
		mov si,nothing_msg
		call writemsg

.norge:		jmp short .norge

		; Information message (DS:SI) output
		; Prefix with "isolinux: "
		;
writemsg:	push ax
		push si
		mov si,isolinux_str
		call writestr
		pop si
		call writestr
		pop ax				
		ret

;
; crlf: Print a newline
;
crlf:		mov si,crlf_msg
		; Fall through

;
; cwritestr: write a null-terminated string to the console, saving
;            registers on entry.
;
; Note: writestr and cwritestr are distinct in SYSLINUX, not in ISOLINUX
;
cwritestr:
		pushfd
                pushad
.top:		lodsb
		and al,al
                jz .end
		call writechr
                jmp short .top
.end:		popad
		popfd
                ret

writestr	equ cwritestr

;
; writehex[248]: Write a hex number in (AL, AX, EAX) to the console
;
writehex2:
		pushfd
		pushad
		shl eax,24
		mov cx,2
		jmp short writehex_common
writehex4:
		pushfd
		pushad
		shl eax,16
		mov cx,4
		jmp short writehex_common
writehex8:
		pushfd
		pushad
		mov cx,8
writehex_common:
.loop:		rol eax,4
		push eax
		and al,0Fh
		cmp al,10
		jae .high
.low:		add al,'0'
		jmp short .ischar
.high:		add al,'A'-10
.ischar:	call writechr
		pop eax
		loop .loop
		popad
		popfd
		ret

;
; Write a character to the screen.  There is a more "sophisticated"
; version of this in the subsequent code, so we patch the pointer
; when appropriate.
;

writechr:
		jmp near writechr_simple	; NOT "short"!!!!

writechr_simple:
		pushfd
		pushad
		mov ah,0Eh
		xor bx,bx
		int 10h
		popad
		popfd
		ret

;
; Get one sector.  Convenience entry point.
;
getonesec:
		mov bp,1
		; Fall through to getlinsec

;
; Get linear sectors - EBIOS LBA addressing, 2048-byte sectors.
;
; Note that we can't always do this as a single request, because at least
; Phoenix BIOSes has a 127-sector limit.  To be on the safe side, stick
; to 32 sectors (64K) per request.
;
; Input:
;	EAX	- Linear sector number
;	ES:BX	- Target buffer
;	BP	- Sector count
;
getlinsec:
		mov si,dapa			; Load up the DAPA
		mov [si+4],bx
		mov bx,es
		mov [si+6],bx
		mov [si+8],eax
.loop:
		push bp				; Sectors left
		cmp bp,byte 32
		jbe .bp_ok
		mov bp,32
.bp_ok:
		mov [si+2],bp
		push si
		mov dl,[DriveNo]
		mov ah,42h			; Extended Read
		call xint13
		pop si
		pop bp
		movzx eax,word [si+2]		; Sectors we read
		add [si+8],eax			; Advance sector pointer
		sub bp,ax			; Sectors left
		shl ax,SECTORSIZE_LG2-4		; 2048-byte sectors -> segment
		add [si+6],ax			; Advance buffer pointer
		and bp,bp
		jnz .loop
		mov eax,[si+8]			; Next sector
		ret

		; INT 13h with retry
xint13:		mov byte [RetryCount], 6
.try:		pushad
		int 13h
		jc .error
		add sp,byte 8*4			; Clean up stack
		ret
.error:		mov [DiskError],ah		; Save error code
		popad
		dec byte [RetryCount]
		jnz .try

.real_error:	mov si,diskerr_msg
		call writemsg
		mov al,[DiskError]
		call writehex2
		mov si,ondrive_str
		call writestr
		mov al,dl
		call writehex2
		call crlf
		; Fall through to kaboom

;
; kaboom: write a message and bail out.  Wait for a user keypress,
;	  then do a hard reboot.
;
kaboom:
		lss sp,[cs:Stack]
		mov ax,cs
		mov ds,ax
		mov es,ax
		mov fs,ax
		mov gs,ax
		sti
		mov si,err_bootfailed
		call cwritestr
		call getchar
		cli
		mov word [BIOS_magic],0	; Cold reboot
		jmp 0F000h:0FFF0h	; Reset vector address

;
; Data that needs to be in the first sector
;
isolinux_banner	db CR, LF, 'ISOLINUX ', version_str, ' ', date, ' ', 0
copyright_str   db ' Copyright (C) 1994-', year, ' H. Peter Anvin'
		db CR, LF, 0
isolinux_str	db 'isolinux: ', 0
%ifdef DEBUG_MESSAGES
startup_msg:	db 'Starting up, DL = ', 0
spec_ok_msg:	db 'Loaded spec packet OK, drive = ', 0
secsize_msg:	db 'Sector size appears to be ', 0
offset_msg:	db 'Loading main image from LBA = ', 0
size_msg:	db 'Sectors to load = ', 0
loaded_msg:	db 'Loaded boot image, verifying...', CR, LF, 0
verify_msg:	db 'Image checksum verified.', CR, LF, 0
allread_msg	db 'Main image read, jumping to main code...', CR, LF, 0
%endif
spec_err_msg:	db 'Loading spec packet failed, trying to wing it...', CR, LF, 0
maybe_msg:	db 'Found something at drive = ', 0
alright_msg:	db 'Looks like it might be right, continuing...', CR, LF, 0
nosecsize_msg:	db 'Failed to get sector size, assuming 0800', CR, LF, 0
diskerr_msg:	db 'Disk error ', 0
ondrive_str:	db ', drive ', 0
nothing_msg:	db 'Failed to locate CD-ROM device; boot failed.', CR, LF, 0
checkerr_msg:	db 'Image checksum error, sorry...', CR, LF, 0

err_bootfailed	db CR, LF, 'Boot failed: press a key to retry...'
bailmsg		equ err_bootfailed
crlf_msg	db CR, LF, 0

;
; El Torito spec packet
;
		align 8, db 0
spec_packet:	db 13h				; Size of packet
sp_media:	db 0				; Media type
sp_drive:	db 0				; Drive number
sp_controller:	db 0				; Controller index
sp_lba:		dd 0				; LBA for emulated disk image
sp_devspec:	dw 0				; IDE/SCSI information
sp_buffer:	dw 0				; User-provided buffer
sp_loadseg:	dw 0				; Load segment
sp_sectors:	dw 0				; Sector count
sp_chs:		db 0,0,0			; Simulated CHS geometry
sp_dummy:	db 0				; Scratch, safe to overwrite

;
; Spec packet for disk image emulation
;
		align 8, db 0
dspec_packet:	db 13h				; Size of packet
dsp_media:	db 0				; Media type
dsp_drive:	db 0				; Drive number
dsp_controller:	db 0				; Controller index
dsp_lba:	dd 0				; LBA for emulated disk image
dsp_devspec:	dw 0				; IDE/SCSI information
dsp_buffer:	dw 0				; User-provided buffer
dsp_loadseg:	dw 0				; Load segment
dsp_sectors:	dw 1				; Sector count
dsp_chs:	db 0,0,0			; Simulated CHS geometry
dsp_dummy:	db 0				; Scratch, safe to overwrite

;
; EBIOS drive parameter packet
;
		align 8, db 0
drive_params:	dw 30				; Buffer size
dp_flags:	dw 0				; Information flags
dp_cyl:		dd 0				; Physical cylinders
dp_head:	dd 0				; Physical heads
dp_sec:		dd 0				; Physical sectors/track
dp_totalsec:	dd 0,0				; Total sectors
dp_secsize:	dw 0				; Bytes per sector
dp_dpte:	dd 0				; Device Parameter Table
dp_dpi_key:	dw 0				; 0BEDDh if rest valid
dp_dpi_len:	db 0				; DPI len
		db 0
		dw 0
dp_bus:		times 4 db 0			; Host bus type
dp_interface:	times 8 db 0			; Interface type
db_i_path:	dd 0,0				; Interface path
db_d_path:	dd 0,0				; Device path
		db 0
db_dpi_csum:	db 0				; Checksum for DPI info

;
; EBIOS disk address packet
;
		align 8, db 0
dapa:		dw 16				; Packet size
.count:		dw 0				; Block count
.off:		dw 0				; Offset of buffer
.seg:		dw 0				; Segment of buffer
.lba:		dd 0				; LBA (LSW)
		dd 0				; LBA (MSW)

		alignb 4, db 0
Stack		dw _start, 0		; SS:SP for stack reset

rl_checkpt	equ $				; Must be <= 800h

rl_checkpt_off	equ ($-$$)
%if rl_checkpt_off > 0x800
%error "Sector 0 overflow"
%endif

; ----------------------------------------------------------------------------
;  End of code and data that have to be in the first sector
; ----------------------------------------------------------------------------

all_read:
;
; Initialize screen (if we're using one)
;
		; Now set up screen parameters
		call adjust_screen

		; Patch the writechr routine to point to the full code
		mov word [writechr+1], writechr_full-(writechr+3)

; Tell the user we got this far...
%ifndef DEBUG_MESSAGES			; Gets messy with debugging on
		mov si,copyright_str
		call writestr
%endif

; Test tracers
		TRACER 'T'
		TRACER '>'

;
; Clear Files structures
;
		mov di,Files
		mov cx,(MAX_OPEN*open_file_t_size)/4
		xor eax,eax
		rep stosd

; 
; Check that no moron is trying to boot Linux on a 286 or so.  According
; to Intel, the way to check is to see if the high 4 bits of the FLAGS
; register are either all stuck at 1 (8086/8088) or all stuck at 0
; (286 in real mode), if not it is a 386 or higher.  They didn't
; say how to check for a 186/188, so I *hope* it falls out as a 8086
; or 286 in this test.
;
; Also, provide an escape route in case it doesn't work.
;
check_escapes:
		mov ah,02h			; Check keyboard flags
		int 16h
		mov [KbdFlags],al		; Save for boot prompt check
		test al,04h			; Ctrl->skip 386 check
		jnz skip_checks
test_8086:
		pushf				; Get flags
		pop ax
		and ax,0FFFh			; Clear top 4 bits
		push ax				; Load into FLAGS
		popf
		pushf				; And load back
		pop ax
		and ax,0F000h			; Get top 4 bits
		cmp ax,0F000h			; If set -> 8086/8088
		je not_386
test_286:
		pushf				; Get flags
		pop ax
		or ax,0F000h			; Set top 4 bits
		push ax
		popf
		pushf
		pop ax
		and ax,0F000h			; Get top 4 bits
		jnz is_386			; If not clear -> 386
not_386:
		mov si,err_not386
		call writestr
		jmp kaboom
is_386:
		; Now we know it's a 386 or higher
;
; Now check that there is at least 384K of low (DOS) memory
;
		int 12h
		cmp ax,384
		jae enough_ram
		mov si,err_noram
		call writestr
		jmp kaboom
enough_ram:
skip_checks:

;
; Check if we're 386 (as opposed to 486+); if so we need to blank out
; the WBINVD instruction
;
; We check for 486 by setting EFLAGS.AC
;
		pushfd				; Save the good flags
		pushfd
		pop eax
		mov ebx,eax
		xor eax,(1 << 18)		; AC bit
		push eax
		popfd
		pushfd
		pop eax
		popfd				; Restore the original flags
		xor eax,ebx
		jnz is_486
;
; 386 - Looks like we better blot out the WBINVD instruction
;
		mov byte [try_wbinvd],0c3h		; Near RET		
is_486:

;
; Now we're all set to start with our *real* business.	First load the
; configuration file (if any) and parse it.
;
; In previous versions I avoided using 32-bit registers because of a
; rumour some BIOSes clobbered the upper half of 32-bit registers at
; random.  I figure, though, that if there are any of those still left
; they probably won't be trying to install Linux on them...
;
; The code is still ripe with 16-bitisms, though.  Not worth the hassle
; to take'm out.  In fact, we may want to put them back if we're going
; to boot ELKS at some point.
;
		mov si,linuxauto_cmd		; Default command: "linux auto"
		mov di,default_cmd
                mov cx,linuxauto_len
		rep movsb

		mov di,KbdMap			; Default keymap 1:1
		xor al,al
		mov cx,256
mkkeymap:	stosb
		inc al
		loop mkkeymap

;
; Now, we need to sniff out the actual filesystem data structures.
; mkisofs gave us a pointer to the primary volume descriptor
; (which will be at 16 only for a single-session disk!); from the PVD
; we should be able to find the rest of what we need to know.
; 
get_fs_structures:
		mov eax,[bi_pvd]
		mov bx,trackbuf
		call getonesec

		mov eax,[trackbuf+156+2]
		mov [RootDir+dir_lba],eax
		mov [CurDir+dir_lba],eax
%ifdef DEBUG_MESSAGES
		mov si,dbg_rootdir_msg
		call writemsg
		call writehex8
		call crlf
%endif
		mov eax,[trackbuf+156+10]
		mov [RootDir+dir_len],eax		
		mov [CurDir+dir_len],eax
		add eax,SECTORSIZE-1
		shr eax,SECTORSIZE_LG2
		mov [RootDir+dir_clust],eax
		mov [CurDir+dir_clust],eax

		; Look for an "isolinux" directory, and if found,
		; make it the current directory instead of the root
		; directory.
		mov di,isolinux_dir
		mov al,02h			; Search for a directory
		call searchdir_iso
		jz .no_isolinux_dir
		mov [CurDir+dir_len],eax
		mov eax,[si+file_left]
		mov [CurDir+dir_clust],eax
		xor eax,eax			; Free this file pointer entry
		xchg eax,[si+file_sector]
		mov [CurDir+dir_lba],eax
%ifdef DEBUG_MESSAGES
		push si
		mov si,dbg_isodir_msg
		call writemsg
		pop si
		call writehex8
		call crlf
%endif
.no_isolinux_dir:

;
; Locate the configuration file
;
load_config:
%ifdef DEBUG_MESSAGES
		mov si,dbg_config_msg
		call writemsg
%endif

		mov di,isolinux_cfg
		call open
		jz near no_config_file		; Not found or empty

%ifdef DEBUG_MESSAGES
		mov si,dbg_configok_msg
		call writemsg
%endif

;
; Now we have the config file open
;
parse_config:
		call getkeyword
                jc near end_config_file		; Config file loaded
		cmp ax,'de'			; DEfault
		je pc_default
		cmp ax,'ap'			; APpend
		je pc_append
		cmp ax,'ti'			; TImeout
		je near pc_timeout
		cmp ax,'pr'			; PRompt
		je near pc_prompt
		cmp ax,'fo'			; FOnt
		je near pc_font
		cmp ax,'kb'			; KBd
		je near pc_kbd
		cmp ax,'di'			; DIsplay
		je near pc_display
		cmp ax,'la'			; LAbel
		je near pc_label
		cmp ax,'ke'			; KErnel
		je near pc_kernel
                cmp ax,'im'                     ; IMplicit
                je near pc_implicit
		cmp ax,'se'			; SErial
		je near pc_serial
		cmp ax,'sa'			; SAy
		je near pc_say
		cmp ax,'lo'			; LOcalboot
		je pc_localboot
		cmp al,'f'			; F-key
		jne parse_config
		jmp pc_fkey

pc_default:	mov di,default_cmd		; "default" command
		call getline
		xor al,al
		stosb				; null-terminate
		jmp short parse_config

pc_append:      cmp word [VKernelCtr],byte 0	; "append" command
		ja pc_append_vk
                mov di,AppendBuf
		call getline
                sub di,AppendBuf
pc_app1:        mov [AppendLen],di
                jmp short parse_config_2
pc_append_vk:	mov di,VKernelBuf+vk_append	; "append" command (vkernel)
		call getline
		sub di,VKernelBuf+vk_append
                cmp di,byte 2
                jne pc_app2
                cmp byte [VKernelBuf+vk_append],'-'
                jne pc_app2
                mov di,0                        ; If "append -" -> null string
pc_app2:        mov [VKernelBuf+vk_appendlen],di
		jmp short parse_config_2	

pc_localboot:	call getint			; "localboot" command
		cmp word [VKernelCtr],byte 0	; ("label" section only)
		je parse_config_2
		mov [VKernelBuf+vk_rname], byte 0	; Null kernel name
		mov [VKernelBuf+vk_rname+1], bx	; Return type
		jmp short parse_config_2

pc_kernel:	cmp word [VKernelCtr],byte 0	; "kernel" command
		je parse_config_2		; ("label" section only)
		mov di,trackbuf
		push di
		call getline
		pop si
		mov di,VKernelBuf+vk_rname
		call mangle_name
		jmp short parse_config_2

pc_timeout:	call getint			; "timeout" command
		jc parse_config_2
		mov ax,0D215h			; There are approx 1.D215h
		mul bx				; clock ticks per 1/10 s
		add bx,dx
		mov [KbdTimeOut],bx
parse_config_2:	jmp parse_config

pc_display:	call pc_getfile			; "display" command
		jz parse_config_2		; File not found?
		call get_msg_file		; Load and display file
		jmp short parse_config_2

pc_prompt:	call getint			; "prompt" command
		jc parse_config_2
		mov [ForcePrompt],bx
		jmp short parse_config_2

pc_implicit:    call getint                     ; "implicit" command
                jc parse_config_2
                mov [AllowImplicit],bx
                jmp short parse_config_2

pc_serial:	call getint			; "serial" command
		jc parse_config_2
		push bx				; Serial port #
		call skipspace
		jc parse_config_2
		call ungetc
		call getint
		mov [FlowControl], word 0	; Default to no flow control
		jc .nobaud
.valid_baud:	
		push ebx
		call skipspace
		jc .no_flow
		call ungetc
		call getint			; Hardware flow control?
		jnc .valid_flow
.no_flow:
		xor bx,bx			; Default -> no flow control
.valid_flow:
		and bh,0Fh			; FlowIgnore
		shl bh,4
		mov [FlowIgnore],bh
		mov bh,bl
		and bx,0F003h			; Valid bits
		mov [FlowControl],bx
		pop ebx				; Baud rate
		jmp short .parse_baud
.nobaud:
		mov ebx,DEFAULT_BAUD		; No baud rate given
.parse_baud:
		pop di				; Serial port #
		cmp ebx,byte 75
		jb parse_config_2		; < 75 baud == bogus
		mov eax,BAUD_DIVISOR
		cdq
		div ebx
		push ax				; Baud rate divisor
		cmp di,3
		ja .port_is_io			; If port > 3 then port is I/O addr
		shl di,1
		mov di,[di+serial_base]		; Get the I/O port from the BIOS
.port_is_io:
		mov [SerialPort],di
		lea dx,[di+3]			; DX -> LCR
		mov al,83h			; Enable DLAB
		call slow_out
		pop ax				; Divisor
		mov dx,di			; DX -> LS
		call slow_out
		inc dx				; DX -> MS
		mov al,ah
		call slow_out
		mov al,03h			; Disable DLAB
		add dx,byte 2			; DX -> LCR
		call slow_out
		in al,dx			; Read back LCR (detect missing hw)
		cmp al,03h			; If nothing here we'll read 00 or FF
		jne .serial_port_bad		; Assume serial port busted
		sub dx,byte 2			; DX -> IER
		xor al,al			; IRQ disable
		call slow_out

		add dx,byte 3			; DX -> MCR
		in al,dx
		or al,[FlowOutput]		; Assert bits
		call slow_out

		; Show some life
		mov si,isolinux_banner
		call write_serial_str
		mov si,copyright_str
		call write_serial_str

		jmp short parse_config_3

.serial_port_bad:
		mov [SerialPort], word 0
		jmp short parse_config_3

pc_fkey:	sub ah,'1'
		jnb pc_fkey1
		mov ah,9			; F10
pc_fkey1:	xor cx,cx
		mov cl,ah
		push cx
		mov ax,1
		shl ax,cl
		or [FKeyMap], ax		; Mark that we have this loaded
		mov di,trackbuf
		push di
		call getline			; Get filename to display
		pop si
		pop di
		shl di,FILENAME_MAX_LG2		; Convert to offset
		add di,FKeyName
		call mangle_name		; Mangle file name
		jmp short parse_config_3

pc_label:	call commit_vk			; Commit any current vkernel
		mov di,trackbuf			; Get virtual filename
		push di
		call getline
		pop si
		mov di,VKernelBuf+vk_vname
		call mangle_name		; Mangle virtual name
		inc word [VKernelCtr]		; One more vkernel
		mov si,VKernelBuf+vk_vname 	; By default, rname == vname
		mov di,VKernelBuf+vk_rname
		mov cx,FILENAME_MAX
		rep movsb
                mov si,AppendBuf         	; Default append==global append
                mov di,VKernelBuf+vk_append
                mov cx,[AppendLen]
                mov [VKernelBuf+vk_appendlen],cx
                rep movsb
		jmp near parse_config_3

pc_font:	call pc_getfile			; "font" command
		jz parse_config_3		; File not found?
		call loadfont			; Load and install font
		jmp short parse_config_3

pc_kbd:		call pc_getfile			; "kbd" command
		jz parse_config_3
		call loadkeys
parse_config_3:	jmp parse_config

pc_say:		mov di,trackbuf			; "say" command
		push di
		call getline
		xor al,al
		stosb				; Null-terminate
		pop si
		call writestr
		call crlf
		jmp short parse_config_3

;
; pc_getfile:	For command line options that take file argument, this
; 		routine decodes the file argument and runs it through searchdir
;
pc_getfile:	mov di,trackbuf
		push di
		call getline
		pop si
		mov di,MNameBuf
		push di
		call mangle_name
		pop di
		jmp searchdir			; Tailcall

;
; commit_vk: Store the current VKernelBuf into buffer segment
;
commit_vk:
		cmp word [VKernelCtr],byte 0
		je cvk_ret			; No VKernel = return
		cmp word [VKernelCtr],max_vk	; Above limit?
		ja cvk_overflow
		mov di,[VKernelCtr]
		dec di
		shl di,vk_shift
		mov si,VKernelBuf
		mov cx,(vk_size >> 2)
		push es
		push word vk_seg
		pop es
		rep movsd			; Copy to buffer segment
		pop es
cvk_ret:	ret
cvk_overflow:	mov word [VKernelCtr],max_vk	; No more than max_vk, please
		ret

;
; End of configuration file
;
end_config_file:
		call commit_vk			; Commit any current vkernel
no_config_file:
;
; Check whether or not we are supposed to display the boot prompt.
;
check_for_key:
		cmp word [ForcePrompt],byte 0	; Force prompt?
		jnz enter_command
		test byte [KbdFlags],5Bh	; Caps, Scroll, Shift, Alt
		jz near auto_boot		; If neither, default boot

enter_command:
		mov si,boot_prompt
		call cwritestr

		mov byte [FuncFlag],0		; <Ctrl-F> not pressed
		mov di,command_line
;
; get the very first character -- we can either time
; out, or receive a character press at this time.  Some dorky BIOSes stuff
; a return in the buffer on bootup, so wipe the keyboard buffer first.
;
clear_buffer:	mov ah,1			; Check for pending char
		int 16h
		jz get_char_time
		xor ax,ax			; Get char
		int 16h
		jmp short clear_buffer
get_char_time:	
		call vgashowcursor
		mov cx,[KbdTimeOut]
		and cx,cx
		jz get_char			; Timeout == 0 -> no timeout
		inc cx				; The first loop will happen
						; immediately as we don't
						; know the appropriate DX value
time_loop:	push cx
tick_loop:	push dx
		call pollchar
		jnz get_char_pop
		mov dx,[BIOS_timer]		; Get time "of day"
		pop ax
		cmp dx,ax			; Has the timer advanced?
		je tick_loop
		pop cx
		loop time_loop			; If so, decrement counter
		call vgahidecursor
		jmp command_done		; Timeout!

get_char_pop:	pop eax				; Clear stack
get_char:
		call vgashowcursor
		call getchar
		call vgahidecursor
		and al,al
		jz func_key

got_ascii:	cmp al,7Fh			; <DEL> == <BS>
		je backspace
		cmp al,' '			; ASCII?
		jb not_ascii
		ja enter_char
		cmp di,command_line		; Space must not be first
		je get_char
enter_char:	test byte [FuncFlag],1
		jz .not_ctrl_f
		mov byte [FuncFlag],0
		cmp al,'0'
		jb .not_ctrl_f
		je ctrl_f_0
		cmp al,'9'
		jbe ctrl_f
.not_ctrl_f:	cmp di,max_cmd_len+command_line ; Check there's space
		jnb get_char
		stosb				; Save it
		call writechr			; Echo to screen
get_char_2:	jmp short get_char
not_ascii:	mov byte [FuncFlag],0
		cmp al,0Dh			; Enter
		je near command_done
		cmp al,06h			; <Ctrl-F>
		je set_func_flag
		cmp al,08h			; Backspace
		jne get_char
backspace:	cmp di,command_line		; Make sure there is anything
		je get_char			; to erase
		dec di				; Unstore one character
		mov si,wipe_char		; and erase it from the screen
		call cwritestr
		jmp short get_char_2

set_func_flag:
		mov byte [FuncFlag],1
		jmp short get_char_2

ctrl_f_0:	add al,10			; <Ctrl-F>0 == F10
ctrl_f:		push di
		sub al,'1'
		xor ah,ah
		jmp short show_help

func_key:
		push di
		cmp ah,68			; F10
		ja get_char_2
		sub ah,59			; F1
		jb get_char_2
		shr ax,8
show_help:	; AX = func key # (0 = F1, 9 = F10)
		mov cl,al
		shl ax,FILENAME_MAX_LG2		; Convert to offset
		mov bx,1
		shl bx,cl
		and bx,[FKeyMap]
		jz get_char_2			; Undefined F-key
		mov di,ax
		add di,FKeyName
		call searchdir
		jz fk_nofile
		push si
		call crlf
		pop si
		call get_msg_file
		jmp short fk_wrcmd
fk_nofile:
		call crlf
fk_wrcmd:
		mov si,boot_prompt
		call cwritestr
		pop di				; Command line write pointer
		push di
		mov byte [di],0			; Null-terminate command line
		mov si,command_line
		call cwritestr			; Write command line so far
		pop di
		jmp short get_char_2
auto_boot:
		mov si,default_cmd
		mov di,command_line
		mov cx,(max_cmd_len+4) >> 2
		rep movsd
		jmp short load_kernel
command_done:
		call crlf
		cmp di,command_line		; Did we just hit return?
		je auto_boot
		xor al,al			; Store a final null
		stosb

load_kernel:					; Load the kernel now
;
; First we need to mangle the kernel name the way DOS would...
;
		mov si,command_line
                mov di,KernelName
                push si
                push di
		call mangle_name
		pop di
                pop si
;
; Fast-forward to first option (we start over from the beginning, since
; mangle_name doesn't necessarily return a consistent ending state.)
;
clin_non_wsp:   lodsb
                cmp al,' '
                ja clin_non_wsp
clin_is_wsp:    and al,al
                jz clin_opt_ptr
                lodsb
                cmp al,' '
                jbe clin_is_wsp
clin_opt_ptr:   dec si                          ; Point to first nonblank
                mov [CmdOptPtr],si		; Save ptr to first option
;
; Now check if it is a "virtual kernel"
;
		mov cx,[VKernelCtr]
		push ds
		push word vk_seg
		pop ds
		cmp cx,byte 0
		je not_vk
		xor si,si			; Point to first vkernel
vk_check:	pusha
		mov cx,FILENAME_MAX
		repe cmpsb			; Is this it?
		je near vk_found
		popa
		add si,vk_size
		loop vk_check
not_vk:		pop ds
;
; Not a "virtual kernel" - check that's OK and construct the command line
;
                cmp word [AllowImplicit],byte 0
                je bad_implicit
                push es
                push si
                push di
                mov di,real_mode_seg
                mov es,di
                mov si,AppendBuf
                mov di,cmd_line_here
                mov cx,[AppendLen]
                rep movsb
                mov [CmdLinePtr],di
                pop di
                pop si
                pop es
;
; Find the kernel on disk
;
get_kernel:     mov byte [KernelName+FILENAME_MAX],0	; Zero-terminate filename/extension
		mov di,KernelName
		xor al,al
		mov cx,FILENAME_MAX-5		; Need 4 chars + null
		repne scasb			; Scan for final null
		jne .no_skip
		dec di				; Point to final null 
.no_skip:	mov [KernelExtPtr],di
		mov bx,exten_table
.search_loop:	push bx
                mov di,KernelName	      	; Search on disk
                call searchdir
		pop bx
                jnz near kernel_good
		mov eax,[bx]			; Try a different extension
		mov si,[KernelExtPtr]
		mov [si],eax
		mov byte [si+4],0
		add bx,byte 4
		cmp bx,exten_table_end
		jna .search_loop		; allow == case (final case)
bad_kernel:     
		mov si,KernelName
                mov di,KernelCName
		push di
                call unmangle_name              ; Get human form
		mov si,err_notfound		; Complain about missing kernel
		call cwritestr
		pop si				; KernelCName
                call cwritestr
                mov si,crlf_msg
                jmp abort_load                  ; Ask user for clue
;
; bad_implicit: The user entered a nonvirtual kernel name, with "implicit 0"
;
bad_implicit:   mov si,KernelName		; For the error message
                mov di,KernelCName
                call unmangle_name
                jmp short bad_kernel
;
; vk_found: We *are* using a "virtual kernel"
;
vk_found:	popa
		push di
		mov di,VKernelBuf
		mov cx,vk_size >> 2
		rep movsd
		push es				; Restore old DS
		pop ds
		push es
		push word real_mode_seg
		pop es
		mov di,cmd_line_here
		mov si,VKernelBuf+vk_append
		mov cx,[VKernelBuf+vk_appendlen]
		rep movsb
		mov [CmdLinePtr],di		; Where to add rest of cmd
		pop es
                pop di                          ; DI -> KernelName
		push di	
		mov si,VKernelBuf+vk_rname
		mov cx,FILENAME_MAX		; We need ECX == CX later
		rep movsb
		pop di
		xor bx,bx			; Try only one version

		; Is this a "localboot" pseudo-kernel?
		cmp byte [VKernelBuf+vk_rname], 0
		jne near get_kernel		; No, it's real, go get it

		mov ax, [VKernelBuf+vk_rname+1]
		jmp local_boot
;
; kernel_corrupt: Called if the kernel file does not seem healthy
;
kernel_corrupt: mov si,err_notkernel
                jmp abort_load
;
; This is it!  We have a name (and location on the disk)... let's load
; that sucker!!  First we have to decide what kind of file this is; base
; that decision on the file extension.  The following extensions are
; recognized; case insensitive:
;
; .com 	- COMBOOT image
; .cbt	- COMBOOT image
; .bs	- Boot sector	  (SYSLINUX only)
; .bss	- DOS boot sector (SYSLINUX only)
; .img	- Disk image	  (ISOLINUX only)
;
; Anything else is assumed to be a Linux kernel.
;
kernel_good:
		pusha
		mov si,KernelName
		mov di,KernelCName
		call unmangle_name
		sub di,KernelCName
		mov [KernelCNameLen],di
		popa
		
		push di
		push ax
		mov di,KernelName
		xor al,al
		mov cx,FILENAME_MAX
		repne scasb
		jne .one_step
		dec di
.one_step:	mov ecx,[di-4]			; 4 bytes before end
		pop ax
		pop di

;
; At this point, DX:AX contains the size of the kernel, and SI contains
; the file handle/cluster pointer.
;
		or ecx,20202000h		; Force lower case

		cmp ecx,'.com'
		je near is_comboot_image
		cmp ecx,'.cbt'
		je near is_comboot_image
		cmp ecx,'.img'
		je near is_disk_image
		cmp ecx,'.bss'
		je near is_bss_sector
		and ecx, 00ffffffh
		cmp ecx,'.bs'
		je near is_bootsector
		; Otherwise Linux kernel
;
; A Linux kernel consists of three parts: boot sector, setup code, and
; kernel code.	The boot sector is never executed when using an external
; booting utility, but it contains some status bytes that are necessary.
;
; First check that our kernel is at least 1K and less than 8M (if it is
; more than 8M, we need to change the logic for loading it anyway...)
;
; We used to require the kernel to be 64K or larger, but it has gotten
; popular to use the Linux kernel format for other things, which may
; not be so large.
;
is_linux_kernel:
                cmp dx,80h			; 8 megs
		ja kernel_corrupt
		and dx,dx
		jnz kernel_sane
		cmp ax,1024			; Bootsect + 1 setup sect
		jb near kernel_corrupt
kernel_sane:	push ax
		push dx
		push si
		mov si,loading_msg
                call cwritestr
;
; Now start transferring the kernel
;
		push word real_mode_seg
		pop es

		movzx eax,ax			; Fix this by using a 32-bit
		shl edx,16			; register for the kernel size
		or eax,edx
		mov [KernelSize],eax
		xor edx,edx
		div dword [ClustSize]		; # of clusters total
		; Round up...
		add edx,byte -1			; Sets CF if EDX >= 1
		adc eax,byte 0			; Add 1 to EAX if CF set
                mov [KernelClust],eax

;
; Now, if we transfer these straight, we'll hit 64K boundaries.	 Hence we
; have to see if we're loading more than 64K, and if so, load it step by
; step.
;

;
; Start by loading the bootsector/setup code, to see if we need to
; do something funky.  It should fit in the first 32K (loading 64K won't
; work since we might have funny stuff up near the end of memory).
; If we have larger than 32K clusters, yes, we're hosed.
;
		call abort_check		; Check for abort key
		mov ecx,[ClustPerMoby]
		shr ecx,1			; Half a moby
		cmp ecx,[KernelClust]
		jna .normalkernel
		mov ecx,[KernelClust]
.normalkernel:
		sub [KernelClust],ecx
		xor bx,bx
                pop si                          ; Cluster pointer on stack
		call getfssec
                cmp word [es:bs_bootsign],0AA55h
		jne near kernel_corrupt		; Boot sec signature missing
;
; Get the BIOS' idea of what the size of high memory is.
;
		push si				; Save our cluster pointer!
;
; First, try INT 15:E820 (get BIOS memory map)
;
get_e820:
		push es
		xor ebx,ebx			; Start with first record
		mov es,bx			; Need ES = DS = 0 for now
		jmp short .do_e820		; Skip "at end" check first time!
.int_loop:	and ebx,ebx			; If we're back at beginning...
		jz no_e820			; ... bail; nothing found
.do_e820:	mov eax,0000E820h
		mov edx,534D4150h		; "SMAP" backwards
		mov ecx,20
		mov di,E820Buf
		int 15h
		jc no_e820
		cmp eax,534D4150h
		jne no_e820
;
; Look for a memory block starting at <= 1 MB and continuing upward
;
		cmp dword [E820Buf+4], byte 0
		ja .int_loop			; Start >= 4 GB?
		mov edx, (1 << 20)
		sub edx, [E820Buf]
		jb .int_loop			; Start >= 1 MB?
		mov eax, 0FFFFFFFFh
		cmp dword [E820Buf+12], byte 0
		ja .huge			; Size >= 4 GB
		mov eax, [E820Buf+8]
.huge:		sub eax, edx			; Adjust size to start at 1 MB
		jbe .int_loop			; Completely below 1 MB?

		; Now EAX contains the size of memory 1 MB...up
		cmp dword [E820Buf+16], byte 1
		jne near err_nohighmem		; High memory isn't usable memory!!!!

		; We're good!
		pop es
		jmp short got_highmem_add1mb	; Still need to add low 1 MB

;
; INT 15:E820 failed.  Try INT 15:E801.
;
no_e820:	pop es

		mov ax,0e801h			; Query high memory (semi-recent)
		int 15h
		jc no_e801
		cmp ax,3c00h
		ja no_e801			; > 3C00h something's wrong with this call
		jb e801_hole			; If memory hole we can only use low part

		mov ax,bx
		shl eax,16			; 64K chunks
		add eax,(16 << 20)		; Add first 16M
		jmp short got_highmem				

;
; INT 15:E801 failed.  Try INT 15:88.
;
no_e801:
		mov ah,88h			; Query high memory (oldest)
		int 15h
		cmp ax,14*1024			; Don't trust memory >15M
		jna e801_hole
		mov ax,14*1024
e801_hole:
		and eax,0ffffh
		shl eax,10			; Convert from kilobytes
got_highmem_add1mb:
		add eax,(1 << 20)		; First megabyte
got_highmem:
		sub eax,HIGHMEM_SLOP
		mov [HighMemSize],eax

;
; Construct the command line (append options have already been copied)
;
construct_cmdline:
		mov di,[CmdLinePtr]
                mov si,boot_image        	; BOOT_IMAGE=
                mov cx,boot_image_len
                rep movsb
                mov si,KernelCName       	; Unmangled kernel name
                mov cx,[KernelCNameLen]
                rep movsb
                mov al,' '                      ; Space
                stosb

.noipappend:
                mov si,[CmdOptPtr]              ; Options from user input
		mov cx,(kern_cmd_len+3) >> 2
		rep movsd
;
; Scan through the command line for anything that looks like we might be
; interested in.  The original version of this code automatically assumed
; the first option was BOOT_IMAGE=, but that is no longer certain.
;
		mov si,cmd_line_here
                mov byte [initrd_flag],0
                push es				; Set DS <- real_mode_seg
                pop ds
get_next_opt:   lodsb
		and al,al
		jz near cmdline_end
		cmp al,' '
		jbe get_next_opt
		dec si
                mov eax,[si]
                cmp eax,'vga='
		je is_vga_cmd
                cmp eax,'mem='
		je is_mem_cmd
                push es                         ; Save ES -> real_mode_seg
                push cs
                pop es                          ; Set ES <- normal DS
                mov di,initrd_cmd
		mov cx,initrd_cmd_len
		repe cmpsb
                jne not_initrd
		mov di,InitRD
                push si                         ; mangle_dir mangles si
                call mangle_name                ; Mangle ramdisk name
                pop si
		cmp byte [es:InitRD],0		; Null filename?
                seta byte [es:initrd_flag]	; Set flag if not
not_initrd:	pop es                          ; Restore ES -> real_mode_seg
skip_this_opt:  lodsb                           ; Load from command line
                cmp al,' '
                ja skip_this_opt
                dec si
                jmp short get_next_opt
is_vga_cmd:
                add si,byte 4
                mov eax,[si]
                mov bx,-1
                cmp eax, 'norm'                 ; vga=normal
                je vc0
                and eax,0ffffffh		; 3 bytes
                mov bx,-2
                cmp eax, 'ext'                  ; vga=ext
                je vc0
                mov bx,-3
                cmp eax, 'ask'                  ; vga=ask
                je vc0
                call parseint                   ; vga=<number>
		jc skip_this_opt		; Not an integer
vc0:		mov [bs_vidmode],bx		; Set video mode
		jmp short skip_this_opt
is_mem_cmd:
                add si,byte 4
                call parseint
		jc skip_this_opt		; Not an integer
		sub ebx,HIGHMEM_SLOP
		mov [cs:HighMemSize],ebx
		jmp short skip_this_opt
cmdline_end:
                push cs                         ; Restore standard DS
                pop ds
		sub si,cmd_line_here
		mov [CmdLineLen],si		; Length including final null
;
; Now check if we have a large kernel, which needs to be loaded high
;
		mov dword [RamdiskMax], HIGHMEM_MAX	; Default initrd limit
		cmp dword [es:su_header],HEADER_ID	; New setup code ID
		jne near old_kernel		; Old kernel, load low
		cmp word [es:su_version],0200h	; Setup code version 2.0
		jb near old_kernel		; Old kernel, load low
                cmp word [es:su_version],0201h	; Version 2.01+?
                jb new_kernel                   ; If 2.00, skip this step
                mov word [es:su_heapend],linux_stack	; Set up the heap
                or byte [es:su_loadflags],80h	; Let the kernel know we care
		cmp word [es:su_version],0203h	; Version 2.03+?
		jb new_kernel			; Not 2.03+
		mov eax,[es:su_ramdisk_max]
		mov [RamdiskMax],eax		; Set the ramdisk limit

;
; We definitely have a new-style kernel.  Let the kernel know who we are,
; and that we are clueful
;
new_kernel:
		mov byte [es:su_loader],syslinux_id	; Show some ID
		movzx ax,byte [es:bs_setupsecs]	; Variable # of setup sectors
		mov [SetupSecs],ax
;
; About to load the kernel.  This is a modern kernel, so use the boot flags
; we were provided.
;
                mov al,[es:su_loadflags]
		mov [LoadFlags],al
;
; Load the kernel.  We always load it at 100000h even if we're supposed to
; load it "low"; for a "low" load we copy it down to low memory right before
; jumping to it.
;
read_kernel:
                mov si,KernelCName		; Print kernel name part of
                call cwritestr                  ; "Loading" message
                mov si,dotdot_msg		; Print dots
                call cwritestr

                mov eax,[HighMemSize]
		sub eax,100000h			; Load address
		cmp eax,[KernelSize]
		jb near no_high_mem		; Not enough high memory
;
; Move the stuff beyond the setup code to high memory at 100000h
;
		movzx esi,word [SetupSecs]	; Setup sectors
		inc esi				; plus 1 boot sector
                shl esi,9			; Convert to bytes
                mov ecx,108000h			; 108000h = 1M + 32K
                sub ecx,esi			; Adjust pointer to 2nd block
                mov [HiLoadAddr],ecx
		sub ecx,100000h			; Turn into a counter
		shr ecx,2			; Convert to dwords
		add esi,(real_mode_seg << 4)	; Pointer to source
                mov edi,100000h                 ; Copy to address 100000h
                call bcopy			; Transfer to high memory

                push word xfer_buf_seg		; Transfer buffer segment
                pop es
high_load_loop: 
                mov si,dot_msg			; Progress report
                call cwritestr
                call abort_check
                mov ecx,[KernelClust]
		and ecx,ecx
		jz high_load_done		; Zero left (tiny kernel?)
		cmp ecx,[ClustPerMoby]
		jna high_last_moby
		mov ecx,[ClustPerMoby]
high_last_moby:
		sub [KernelClust],ecx
		xor bx,bx			; Load at offset 0
                pop si                          ; Restore cluster pointer
		call getfssec
                push si                         ; Save cluster pointer
                pushf                           ; Save EOF
                xor bx,bx
		mov esi,(xfer_buf_seg << 4)
                mov edi,[HiLoadAddr]		; Destination address
                mov ecx,4000h			; Cheating - transfer 64K
                call bcopy			; Transfer to high memory
		mov [HiLoadAddr],edi		; Point to next target area
                popf                            ; Restore EOF
                jc high_load_done               ; If EOF we are done
                cmp dword [KernelClust],byte 0	; Are we done?
		jne high_load_loop		; Apparently not
high_load_done:
		pop si				; No longer needed
                mov ax,real_mode_seg		; Set to real mode seg
                mov es,ax

                mov si,dot_msg
                call cwritestr

		call crlf
;
; Now see if we have an initial RAMdisk; if so, do requisite computation
; We know we have a new kernel; the old_kernel code already will have objected
; if we tried to load initrd using an old kernel
;
load_initrd:
                test byte [initrd_flag],1
                jz near nk_noinitrd
                push es                         ; ES->real_mode_seg
                push ds
                pop es                          ; We need ES==DS
                mov si,InitRD
                mov di,InitRDCName
                call unmangle_name              ; Create human-readable name
                sub di,InitRDCName
                mov [InitRDCNameLen],di
                mov di,InitRD
                call searchdir                  ; Look for it in directory
                pop es
		jz initrd_notthere
		mov [initrd_ptr],si		; Save cluster pointer
		mov [es:su_ramdisklen1],ax	; Ram disk length
		mov [es:su_ramdisklen2],dx
		movzx eax,ax
		shl edx,16
		or eax,edx
		xor edx,edx
		div dword [ClustSize]
		; Round up...
		add edx,byte -1			; Sets CF if EDX >= 1
		adc eax,byte 0			; Add 1 to EAX if CF set
		mov [InitRDClust],eax		; Ramdisk clusters
		mov edx,[HighMemSize]		; End of memory
		dec edx
		mov eax,[RamdiskMax]		; Highest address allowed by kernel
		cmp edx,eax
		jna memsize_ok
		mov edx,eax			; Adjust to fit inside limit
memsize_ok:
		inc edx
		sub edx,[es:su_ramdisklen]	; Subtract size of ramdisk
                xor dx,dx			; Round down to 64K boundary
                mov [InitRDat],edx		; Load address
		call loadinitrd			; Load initial ramdisk
		jmp short initrd_end

initrd_notthere:
                mov si,err_noinitrd
                call cwritestr
                mov si,InitRDCName
                call cwritestr
                mov si,crlf_msg
                jmp abort_load

no_high_mem:    mov si,err_nohighmem		; Error routine
                jmp abort_load

initrd_end:
nk_noinitrd:
;
; Abandon hope, ye that enter here!  We do no longer permit aborts.
;
                call abort_check        	; Last chance!!

		mov si,ready_msg
		call cwritestr

		call vgaclearmode		; We can't trust ourselves after this

		cli
		xor ax,ax
		mov ss,ax
		mov sp,7C00h			; Set up a more normal stack
		
;
; Now, if we were supposed to load "low", copy the kernel down to 10000h
; and the real mode stuff to 90000h.  We assume that all bzImage kernels are
; capable of starting their setup from a different address.
;
		mov bx,real_mode_seg		; Real mode segment
		mov fs,bx			; FS -> real_mode_seg
;
; Copy command line.  Unfortunately, the kernel boot protocol requires
; the command line to exist in the 9xxxxh range even if the rest of the
; setup doesn't.
;
		cli				; In case of hooked interrupts
		test byte [LoadFlags],LOAD_HIGH
		jz need_high_cmdline
		cmp word [fs:su_version],0202h	; Support new cmdline protocol?
		jb need_high_cmdline
		; New cmdline protocol
		; Store 32-bit (flat) pointer to command line
		mov dword [fs:su_cmd_line_ptr],(real_mode_seg << 4) + cmd_line_here
		jmp short in_proper_place

need_high_cmdline:
;
; Copy command line up to 90000h
;
		mov ax,9000h
		mov es,ax
		mov si,cmd_line_here
		mov di,si
		mov [fs:kern_cmd_magic],word CMD_MAGIC ; Store magic
		mov [fs:kern_cmd_offset],di	; Store pointer

		mov cx,[CmdLineLen]
		add cx,byte 3
		shr cx,2			; Convert to dwords
		fs rep movsd

		test byte [LoadFlags],LOAD_HIGH
		; Note bx -> real_mode_seg still
		jnz in_proper_place		; If high load, we're done

;
; Loading low; we can't assume it's safe to run in place.
;
; Copy real_mode stuff up to 90000h
;
		mov ax,real_mode_seg
		mov fs,ax
		mov ax,9000h
		mov es,ax
		mov cx,[SetupSecs]
		inc cx				; Setup + boot sector
		shl cx,7			; Sectors -> dwords
		xor si,si
		xor di,di
		fs rep movsd			; Copy setup + boot sector
;
; Some kernels in the 1.2 ballpark but pre-bzImage have more than 4
; setup sectors, but the boot protocol had not yet been defined.  They
; rely on a signature to figure out if they need to copy stuff from
; the "protected mode" kernel area.  Unfortunately, we used that area
; as a transfer buffer, so it's going to find the signature there.
; Hence, zero the low 32K beyond the setup area.
;
		mov di,[SetupSecs]
		inc di				; Setup + boot sector
		mov cx,32768/512		; Sectors/32K
		sub cx,di			; Remaining sectors
		shl di,9			; Sectors -> bytes
		shl cx,7			; Sectors -> dwords
		xor eax,eax
		rep stosd			; Clear region
;
		mov ecx,[KernelSize]
		add ecx,3			; Round upwards
		shr ecx,2			; Bytes -> dwords
		mov esi,100000h
		mov edi,10000h
		call bcopy

		mov bx,9000h			; Real mode segment

;
; Now everything is where it needs to be...
;
in_proper_place:
		mov es,bx			; Real mode segment
;
; If the default root device is set to FLOPPY (0000h), change to
; /dev/fd0 (0200h)
;
		cmp word [es:bs_rootdev],byte 0
		jne root_not_floppy
		mov word [es:bs_rootdev],0200h
root_not_floppy:
;
; Copy the disk table to high memory, then re-initialize the floppy
; controller
;
; This needs to be moved before the copy
;
%if 0
		push ds
		push bx
		lds si,[fdctab]
		mov di,linux_fdctab
		mov cx,3			; 12 bytes
		push di
		rep movsd
		pop di
		mov [fdctab1],di		; Save new floppy tab pos
		mov [fdctab2],es
		xor ax,ax
		xor dx,dx
		int 13h
		pop bx
		pop ds
%endif
;
; Linux wants the floppy motor shut off before starting the kernel,
; at least bootsect.S seems to imply so
;
kill_motor:
		mov dx,03F2h
		xor al,al
		call slow_out
;
; If we're debugging, wait for a keypress so we can read any debug messages
;
%ifdef debug
                xor ax,ax
                int 16h
%endif
;
; Set up segment registers and the Linux real-mode stack
; Note: bx == the real mode segment
;
		cli
		; es is already == real mode segment
		mov ds,bx
		mov fs,bx
		mov gs,bx
		mov ss,bx
		mov sp,linux_stack
;
; We're done... now RUN THAT KERNEL!!!!
; Setup segment == real mode segment + 020h; we need to jump to offset
; zero in the real mode segment.
;
		add bx,020h
		push bx
		push word 0h
		retf

;
; Load an older kernel.  Older kernels always have 4 setup sectors, can't have
; initrd, and are always loaded low.
;
old_kernel:
                test byte [initrd_flag],1	; Old kernel can't have initrd
                jz load_old_kernel
                mov si,err_oldkernel
                jmp abort_load
load_old_kernel:
		mov word [SetupSecs],4		; Always 4 setup sectors
		mov byte [LoadFlags],0		; Always low
		jmp read_kernel

;
; Load a COMBOOT image.  A COMBOOT image is basically a DOS .COM file,
; except that it may, of course, not contain any DOS system calls.  We
; do, however, allow the execution of INT 20h to return to SYSLINUX.
;
is_comboot_image:
		and dx,dx
		jnz near comboot_too_large
		cmp ax,0ff00h		; Max size in bytes
		jae comboot_too_large

		;
		; Set up the DOS vectors in the IVT (INT 20h-3fh)
		;
		mov dword [4*0x20],comboot_return	; INT 20h vector
		mov eax,comboot_bogus
		mov di,4*0x21
		mov cx,31		; All remaining DOS vectors
		rep stosd
	
		mov cx,comboot_seg
		mov es,cx

		mov bx,100h		; Load at <seg>:0100h

		mov cx,[ClustPerMoby]	; Absolute maximum # of clusters
		call getfssec

		xor di,di
		mov cx,64		; 256 bytes (size of PSP)
		xor eax,eax		; Clear PSP
		rep stosd

		mov word [es:0], 020CDh	; INT 20h instruction
		; First non-free paragraph
		mov word [es:02h], comboot_seg+1000h

		; Copy the command line from high memory
		mov cx,125		; Max cmdline len (minus space and CR)
		mov si,[CmdOptPtr]
		mov di,081h		; Offset in PSP for command line
		mov al,' '		; DOS command lines begin with a space
		stosb

comboot_cmd_cp:	lodsb
		and al,al
		jz comboot_end_cmd
		stosb
		loop comboot_cmd_cp
comboot_end_cmd: mov al,0Dh		; CR after last character
		stosb
		mov al,126		; Include space but not CR
		sub al,cl
		mov [es:80h], al	; Store command line length

		mov [SavedSSSP],sp
		mov ax,ss		; Save away SS:SP
		mov [SavedSSSP+2],ax

		call vgaclearmode	; Reset video

		mov ax,es
		mov ds,ax
		mov ss,ax
		xor sp,sp
		push word 0		; Return to address 0 -> exit

		jmp comboot_seg:100h	; Run it

; Looks like a COMBOOT image but too large
comboot_too_large:
		mov si,err_comlarge
		call cwritestr
cb_enter:	jmp enter_command

; Proper return vector
comboot_return:	cli			; Don't trust anyone
		xor ax,ax
		mov ds,ax
		mov es,ax
		lss sp,[SavedSSSP]
		sti
		cld
		jmp short cb_enter

; Attempted to execute DOS system call
comboot_bogus:	cli			; Don't trust anyone
		xor ax,ax
		mov ds,ax
		mov es,ax
		lss sp,[SavedSSSP]
		sti
		cld
		mov si,KernelCName
		call cwritestr
		mov si,err_notdos
		call cwritestr
		jmp short cb_enter

;
; Load a boot sector
;
is_bootsector:
is_bss_sector:
		; Can't load these from the network, dang it!
.badness:	jmp short .badness

;
; Enable disk emulation.  The kind of disk we emulate is dependent on the size of
; the file: 1200K, 1440K or 2880K floppy, otherwise harddisk.
;
is_disk_image:
		TRACER CR
		TRACER LF
		TRACER 'D'
		TRACER ':'

		shl edx,16
		mov dx,ax			; Set EDX <- file size
		mov di,img_table
		mov cx,img_table_count
		mov eax,[si+file_sector]	; Starting LBA of file
		mov [dsp_lba],eax		; Location of file
		mov byte [dsp_drive], 0		; 00h floppy, 80h hard disk
.search_table:
		TRACER 't'
		mov eax,[di+4]
		cmp edx,[di]
		je near .type_found
		add di,8
		loop .search_table

		; Hard disk image.  Need to examine the partition table
		; in order to deduce the C/H/S geometry.  Sigh.
.hard_disk_image:
		TRACER 'h'
		cmp edx,512
		jb near .bad_image

		mov bx,trackbuf
		mov cx,1			; Load 1 sector
		call getfssec
		
		cmp word [trackbuf+510],0aa55h	; Boot signature
		jne near .bad_image		; Image not bootable

		mov cx,4			; 4 partition entries
		mov di,trackbuf+446		; Start of partition table

		xor ax,ax			; Highest sector(al) head(ah)

.part_scan:
		cmp byte [di+4], 0
		jz .part_loop
		lea si,[di+1]
		call .hs_check
		add si,byte 4
		call .hs_check
.part_loop:
		add di,byte 16
		loop .part_scan
		
		push eax			; H/S
		push edx			; File size
		mov bl,ah
		xor bh,bh
		inc bx				; # of heads in BX
		xor ah,ah			; # of sectors in AX
		cwde				; EAX[31:16] <- 0
		mul bx
		shl eax,9			; Convert to bytes
		; Now eax contains the number of bytes per cylinder
		pop ebx				; File size
		xor edx,edx
		div ebx
		and edx,edx
		jz .no_remainder
		inc eax				; Fractional cylinder...
		; Now (e)ax contains the number of cylinders
.no_remainder:	cmp eax,1024
		jna .ok_cyl
		mov ax,1024			; Max possible #
.ok_cyl:	dec ax				; Convert to max cylinder no
		pop ebx				; S(bl) H(bh)
		shl ah,6
		or bl,ah
		xchg ax,bx
		shl eax,16
		mov ah,bl
		mov al,4			; Hard disk boot
		mov byte [dsp_drive], 80h	; Drive 80h = hard disk

.type_found:
		TRACER 'T'
		mov bl,[sp_media]
		and bl,0F0h			; Copy controller info bits
		or al,bl
		mov [dsp_media],al		; Emulation type
		shr eax,8
		mov [dsp_chs],eax		; C/H/S geometry
		mov ax,[sp_devspec]		; Copy device spec
		mov [dsp_devspec],ax
		mov al,[sp_controller]		; Copy controller index
		mov [dsp_controller],al

		TRACER 'V'
		call vgaclearmode		; Reset video

		mov ax,4C00h			; Enable emulation and boot
		mov si,dspec_packet
		mov dl,[DriveNo]
		lss sp,[InitStack]
		TRACER 'X'

		int 13h

		; If this returns, we have problems
.bad_image:
		mov si,err_disk_image
		call cwritestr
		jmp enter_command

;
; Look for the highest seen H/S geometry
; We compute cylinders separately
;
.hs_check:
		mov bl,[si]			; Head #
		cmp bl,ah
		jna .done_track
		mov ah,bl			; New highest head #
.done_track:	mov bl,[si+1]
		and bl,3Fh			; Sector #
		cmp bl,al
		jna .done_sector
		mov al,bl
.done_sector:	ret

;
; Boot a specified local disk.  AX specifies the BIOS disk number; or
; 0xFFFF in case we should execute INT 18h ("next device.")
;
local_boot:
		call vgaclearmode
		lss sp,[cs:Stack]		; Restore stack pointer
		xor dx,dx
		mov ds,dx
		mov es,dx
		mov fs,dx
		mov gs,dx
		mov si,localboot_msg
		call writestr
		cmp ax,-1
		je .int18
		
		; Load boot sector from the specified BIOS device and jump to it.
		mov dl,al
		xor dh,dh
		push dx
		xor ax,ax			; Reset drive
		call xint13
		mov ax,0201h			; Read one sector
		mov cx,0001h			; C/H/S = 0/0/1 (first sector)
		mov bx,trackbuf
		call xint13
		pop dx
		cli				; Abandon hope, ye who enter here
		mov si,trackbuf
		mov di,07C00h
		mov cx,512			; Probably overkill, but should be safe
		rep movsd
		lss sp,[cs:InitStack]
		jmp 0:07C00h			; Jump to new boot sector

.int18:
		int 18h				; Hope this does the right thing...
		jmp kaboom			; If we returned, oh boy...

;
; 32-bit bcopy routine for real mode
;
; We enter protected mode, set up a flat 32-bit environment, run rep movsd
; and then exit.  IMPORTANT: This code assumes cs == ss == 0.
;
; This code is probably excessively anal-retentive in its handling of
; segments, but this stuff is painful enough as it is without having to rely
; on everything happening "as it ought to."
;
		align 4
bcopy_gdt:	dw bcopy_gdt_size-1	; Null descriptor - contains GDT
		dd bcopy_gdt		; pointer for LGDT instruction
		dw 0
		dd 0000ffffh		; Code segment, use16, readable,
		dd 00009b00h		; present, dpl 0, cover 64K
		dd 0000ffffh		; Data segment, use16, read/write,
		dd 008f9300h		; present, dpl 0, cover all 4G
		dd 0000ffffh		; Data segment, use16, read/write,
		dd 00009300h		; present, dpl 0, cover 64K
bcopy_gdt_size:	equ $-bcopy_gdt

bcopy:		push eax
		pushf			; Saves, among others, the IF flag
		push gs
		push fs
		push ds
		push es
		mov [cs:SavedSSSP],sp
		mov ax,ss
		mov [cs:SavedSSSP+2],ax

		cli
		call enable_a20

		o32 lgdt [bcopy_gdt]
		mov eax,cr0
		or al,1
		mov cr0,eax		; Enter protected mode
		jmp 08h:.in_pm

.in_pm:		mov ax,10h		; Data segment selector
		mov es,ax
		mov ds,ax

		mov al,18h		; "Real-mode-like" data segment
		mov ss,ax
		mov fs,ax
		mov gs,ax	
	
		a32 rep movsd		; Do our business
		
		mov es,ax		; Set to "real-mode-like"
		mov ds,ax
	
		mov eax,cr0
		and al,~1
		mov cr0,eax		; Disable protected mode
		jmp 0:.in_rm

.in_rm:		; Back in real mode
		lss sp,[cs:SavedSSSP]
		pop es
		pop ds
		pop fs
		pop gs
		call disable_a20

		popf			; Re-enables interrupts
		pop eax
		ret

;
; Routines to enable and disable (yuck) A20.  These routines are gathered
; from tips from a couple of sources, including the Linux kernel and
; http://www.x86.org/.  The need for the delay to be as large as given here
; is indicated by Donnie Barnes of RedHat, the problematic system being an
; IBM ThinkPad 760EL.
;
; We typically toggle A20 twice for every 64K transferred.
; 
%define	io_delay	call _io_delay
%define IO_DELAY_PORT	80h		; Invalid port (we hope!)
%define disable_wait 	32		; How long to wait for a disable

%define A20_DUNNO	0		; A20 type unknown
%define A20_NONE	1		; A20 always on?
%define A20_BIOS	2		; A20 BIOS enable
%define A20_KBC		3		; A20 through KBC
%define A20_FAST	4		; A20 through port 92h

slow_out:	out dx, al		; Fall through

_io_delay:	out IO_DELAY_PORT,al
		out IO_DELAY_PORT,al
		ret

enable_a20:
		pushad
		mov byte [cs:A20Tries],255 ; Times to try to make this work

try_enable_a20:
;
; Flush the caches
;
;		call try_wbinvd

;
; If the A20 type is known, jump straight to type
;
		mov bp,[cs:A20Type]
		add bp,bp			; Convert to word offset
		jmp word [cs:bp+A20List]

;
; First, see if we are on a system with no A20 gate
;
a20_dunno:
a20_none:
		mov byte [cs:A20Type], A20_NONE
		call a20_test
		jnz a20_done

;
; Next, try the BIOS (INT 15h AX=2401h)
;
a20_bios:
		mov byte [cs:A20Type], A20_BIOS
		mov ax,2401h
		pushf				; Some BIOSes muck with IF
		int 15h
		popf

		call a20_test
		jnz a20_done

;
; Enable the keyboard controller A20 gate
;
a20_kbc:
		mov dl, 1			; Allow early exit
		call empty_8042
		jnz a20_done			; A20 live, no need to use KBC

		mov byte [cs:A20Type], A20_KBC	; Starting KBC command sequence

		mov al,0D1h			; Command write
		out 064h, al
		call empty_8042_uncond

		mov al,0DFh			; A20 on
		out 060h, al
		call empty_8042_uncond

		; Verify that A20 actually is enabled.  Do that by
		; observing a word in low memory and the same word in
		; the HMA until they are no longer coherent.  Note that
		; we don't do the same check in the disable case, because
		; we don't want to *require* A20 masking (SYSLINUX should
		; work fine without it, if the BIOS does.)
.kbc_wait:	push cx
		xor cx,cx
.kbc_wait_loop:
		call a20_test
		jnz a20_done_pop
		loop .kbc_wait_loop

		pop cx
;
; Running out of options here.  Final attempt: enable the "fast A20 gate"
;
a20_fast:
		mov byte [cs:A20Type], A20_FAST	; Haven't used the KBC yet
		in al, 092h
		or al,02h
		and al,~01h			; Don't accidentally reset the machine!
		out 092h, al

.fast_wait:	push cx
		xor cx,cx
.fast_wait_loop:
		call a20_test
		jnz a20_done_pop
		loop .fast_wait_loop

		pop cx

;
; Oh bugger.  A20 is not responding.  Try frobbing it again; eventually give up
; and report failure to the user.
;


		dec byte [cs:A20Tries]
		jnz try_enable_a20

		mov si, err_a20
		jmp abort_load
;
; A20 unmasked, proceed...
;
a20_done_pop:	pop cx
a20_done:	popad
		ret

;
; This routine tests if A20 is enabled (ZF = 0).  This routine
; must not destroy any register contents.
;
a20_test:
		push es
		push cx
		push ax
		mov cx,0FFFFh		; HMA = segment 0FFFFh
		mov es,cx
		mov cx,32		; Loop count
		mov ax,[cs:A20Test]
.a20_wait:	inc ax
		mov [cs:A20Test],ax
		io_delay		; Serialize, and fix delay
		cmp ax,[es:A20Test+10h]
		loopz .a20_wait
.a20_done:	pop ax
		pop cx
		pop es
		ret

disable_a20:
		pushad
;
; Flush the caches
;
;		call try_wbinvd

		mov bp,[cs:A20Type]
		add bp,bp			; Convert to word offset
		jmp word [cs:bp+A20DList]

a20d_bios:
		mov ax,2400h
		pushf				; Some BIOSes muck with IF
		int 15h
		popf
		jmp short a20d_snooze

;
; Disable the "fast A20 gate"
;
a20d_fast:
		in al, 092h
		and al,~03h
		out 092h, al
		jmp short a20d_snooze

;
; Disable the keyboard controller A20 gate
;
a20d_kbc:
		call empty_8042_uncond
		mov al,0D1h
		out 064h, al		; Command write
		call empty_8042_uncond
		mov al,0DDh		; A20 off
		out 060h, al
		call empty_8042_uncond
		; Wait a bit for it to take effect
a20d_snooze:
		push cx
		mov cx, disable_wait
.delayloop:	call a20_test
		jz .disabled
		loop .delayloop
.disabled:	pop cx
a20d_dunno:
a20d_none:
		popad
		ret

;
; Routine to empty the 8042 KBC controller.  If dl != 0
; then we will test A20 in the loop and exit if A20 is
; suddenly enabled.
;
empty_8042_uncond:
		xor dl,dl
empty_8042:
		call a20_test
		jz .a20_on
		and dl,dl
		jnz .done
.a20_on:	io_delay
		in al, 064h		; Status port
		test al,1
		jz .no_output
		io_delay
		in al, 060h		; Read input
		jmp short empty_8042
.no_output:
		test al,2
		jnz empty_8042
		io_delay
.done:		ret	

;
; WBINVD instruction; gets auto-eliminated on 386 CPUs
;
try_wbinvd:
		wbinvd
		ret

;
; Load RAM disk into high memory
;
loadinitrd:
                push es                         ; Save ES on entry
                mov ax,real_mode_seg
                mov es,ax
                mov si,[initrd_ptr]
                mov edi,[InitRDat]		; initrd load address
		mov [es:su_ramdiskat],edi	; Offset for ram disk
		push si
                mov si,loading_msg
                call cwritestr
                mov si,InitRDCName		; Write ramdisk name
                call cwritestr
                mov si,dotdot_msg		; Write dots
                call cwritestr
rd_load_loop:	
		mov si,dot_msg			; Progress report
                call cwritestr
		pop si				; Restore cluster pointer
                call abort_check
                mov ecx,[InitRDClust]
		cmp ecx,[ClustPerMoby]
		jna rd_last_moby
		mov ecx,[ClustPerMoby]
rd_last_moby:
		sub [InitRDClust],ecx
		xor bx,bx			; Load at offset 0
                push word xfer_buf_seg		; Bounce buffer segment
		pop es
		push cx
		call getfssec
		pop cx
                push si				; Save cluster pointer
		mov esi,(xfer_buf_seg << 4)
		mov edi,[InitRDat]
		mov ecx,4000h			; Copy 64K
		call bcopy			; Does not change flags!!
                jc rd_load_done                 ; EOF?
                add dword [InitRDat],10000h	; Point to next 64K
		cmp dword [InitRDClust],byte 0	; Are we done?
		jne rd_load_loop		; Apparently not
rd_load_done:
                pop si                          ; Clean up the stack
		call crlf
                pop es                          ; Restore original ES
                ret

;
; abort_check: let the user abort with <ESC> or <Ctrl-C>
;
abort_check:
		call pollchar
		jz ac_ret1
		pusha
		call getchar
		cmp al,27			; <ESC>
		je ac_kill
		cmp al,3			; <Ctrl-C>
		jne ac_ret2
ac_kill:	mov si,aborted_msg

;
; abort_load: Called by various routines which wants to print a fatal
;             error message and return to the command prompt.  Since this
;             may happen at just about any stage of the boot process, assume
;             our state is messed up, and just reset the segment registers
;             and the stack forcibly.
;
;             SI    = offset (in _text) of error message to print
;
abort_load:
                mov ax,cs                       ; Restore CS = DS = ES
                mov ds,ax
                mov es,ax
                cli
		lss sp,[cs:Stack]		; Reset the stack
                sti
                call cwritestr                  ; Expects SI -> error msg
al_ok:          jmp enter_command               ; Return to command prompt
;
; End of abort_check
;
ac_ret2:	popa
ac_ret1:	ret


;
; searchdir:
;
;	Open a file
;
;	     On entry:
;		DS:DI	= filename
;	     If successful:
;		ZF clear
;		SI		= file pointer
;		DX:AX or EAX	= file length in bytes
;	     If unsuccessful
;		ZF set
;

;
; searchdir_iso is a special entry point for ISOLINUX only.  In addition
; to the above, searchdir_iso passes a file flag mask in AL.  This is useful
; for searching forNow evp:rNow det the ly enabled.
;		ms' .disabCa A20 actually is enabled
;	    d lib
  b*h Kger nn of INbCa A2isabCa A20 		jmp combi    If unbp			; ,]
.a20_wwl on	; nters
            D     <Ethe command pr(
      		pop si		;
		cl		; H/S
		push any stage of,bytrd			call for     eStbCa A2       pus  registers
;            ush any stage of,bytrd			call  Now evl ES
               2isabCa l on	lr .bad_image

		mov bx,trackbuf0sSP],sp
	NNO	  uA until mov [KernelC	; , [dsp_contrev [Kernel, searestage o ailure test b; Now eax csh any st    pus  registers
;        xfint		; nc.k-0 xferd, p0xthe a l on es   cond
		mov care
		cal versmbi    If unsne ac_ret2
ac_ki       ersmb   c	jne high ersehigll cy_8 	trl- dist       ca1:rnelClw sage         		mot to_
;             t 0FFFFh
		mal 0ger -ars    t 0FFF      	cmp eax,534De_here
		m
acTcmp h hig00h	; 255 ; Tito mac_ret		ret

disable_a20:pect; Restore cluster_test
y	trl;	call a20_test
		jnz a2v di,[CmdLinePtr]
            the 're supposee           sp_con       	mov di,0 eax,53eing :csh an:
on
		G53einInit4;
loadinitxor edxl oneha[q01ta:
; Di    theo enS vecto.con    20_tesehigll2
ac_ki       e    	movEc_ki    ers mode
		jmp 08h:ef INbG_	jnz a2vll2 e  st reset thbss mode
			mov ,4000h			; Copy 64K
		cde
		jmre	bmand                   mov edi,[InitRDat]		; initrd load addre6 enS; sv3c rd0 eax,53ea              ndon hope,u:eax,ac_ret2
ac_     nd1done           D     <Ethe commant4;
loa
Rerror msg
a    bne      ore cluster poilt,B
)y:M	; FraS; Expects:
; Di    re.
  bne      ore cluseax cturn to ernes en.
;		    h			re cluse,err_disk_il empty 08hsr;    :
		byte    def,bytrd			:ac_reuence

		mov er...
		; e<s(_ki      call cwr    nd1a		; f un
uga=nosv3c rd0sSP]ere ret
:buffer segmetry pointe pointer
;		DX:AX orhttp:; 256 bytes (sizeword.movea=nes,ax
		mo1Dbytes (sr
;	A  cond
		byte    def,bytrd			:s  def,bytrd		bl		 S ow somemov es to pr0_done:	popad
		ret

;
; Thiis r:1 M.ter   ,[sp_devs msg
a  i                hiis r:ll .hs_check
		trd load'byte     r:ll .hs_check
		trd load'byte     r:l TryE0000h even if we're          eg+1n     push es    ez a20 rei
a  disk name
                call cw:ll .h      c(u cluseax cturn tmov Gi				;call empty .hFr     eSteg
   
		cmp al,27	Ls],255 avehere   t 0Ls]          egcontrollers],255cdxxuTegcEr    _sp_nte pomodeStlw sage     ush ds
_	jere   t 0Ls]  a [dsp_Imov cx,[Ck nax cturn tmo	; wollers],255cdxxuT
;
; searc	; wollers],255cd;  a [dsp_dcdxxu2eabormov cx,xone:
     [dO the HMA uolleSmp shcdxxp0_dcdxxu2ead
         cdxx	; Can'5cdxxuT
Sse pomode s-lboot_msg
		cat'5cdxxuT
S0;     lleh
		v:dremaining DOS vecOr msg
abeck
		trdc(u e:	pgmenly is edinitavehere 1 st restr
	wr    nd1aCS.s bacitreBcall cwr.of error message to print	cmps
		mmpty 08hsrhdir_iso posehe comE0201h			;-  ndahe size of
M           Retur:iS		moQmemov eses after		; End of memordl			;re    entry:
;		DS:DI	= fnd 	;re    entry:
;		DS:Dt If erro	 c8 -		;-  ndah ersehigtword [SetupSecs]n    If successuT
;
;tr]
       jz near nk_noinitrd
                push es                         ; Ed,my 64K
	v bp,[cs:A20Type]
		add bp,bd [SetuCopyj
itupScx,pe]
		add bp,bdment cliehiglld
		mov al,0DD]              ; Ed,my 6
s5u[sp_devdir_p si       ee			; R limuccessuT
;
;t	ef    push es                         ; Ed,e s-lboot_m	A  cond
		byte    def,bytrds     	rurn to command p;
; 	byte    coal,' '
O    c+loading_my 6
s5u[s;bCa A20 actuxxuTee forayloop   mdxxuT
rmode
		lss spgDD]        aDcorayloop   md,ecx
st "low"; forn],si		;0).kere    coal, si,       pop es_x,31		; u[eck
	c,[Cm     r   pSl .h;bCa A2   ;orn],si		;0).kere    coa
; 	byte    coa3 to enabl) c(u cl esiheo en24	ointehigck
	c,[Cm       xfinov sp,7 ret:popDur0
	all 	;
		cl		; H/S
		push 4 		mo;	mo1he comE0201h			;-  nda4sabCa A8t dode
		lss s; su		DS:Dte
		lssBeoKiy that 		mov esOl,18h		d s],255cdxttp:; 2*,0D1h
		;bCS vecOr msax cturn tn if DS vecning D1dxxus
.delaylon tn if0*,0D1hWBINVD ihdist     3e ceg
		mov es,cx

		mov bx,  dec s,cx

		mov btdxx,  sadd d(nitxor edxl oneh',tage of,bast_wa_devdir_p si   ; tod1sAal d can't have
; iiis r:ll .hs_check
		trd load'p
;	A  co havea:		jz neeode
	movfchar
	es after		           cmp eax,'vga='
		je is_vga_cmd
       urn to commay st_vg    'carem     ee			; R limuccessuT
;
;t	ddition
; toDAyl parmd
  er -fort	mov di,si
	sg
a  ,ame       	; Unmangled kernel name
                mov an't hav	repd load'p
;	A  yte    limuccel crlf
(ence

		_place

al d can't have
; iiis r:ll pSec:2:h  ; l d can d xfers mode
rlf
(ence

.v	repduhiglld
		mov at mo d3lld
		ax cerneds
		pu  eecs    Fte  s  registe			; Rmovbortestr
 or commn d [Cm   ]up a flat 32-bit env-          00sh	; Starting KBC commann d int
;
abort   1ics    Ftheck
		t KBC ceal_mode   e  ]up seg	,ot_wait:	push cx
		xor cx,cx
.fast_wait3ar_iest byte [LoadFlaxr]
dd;  a /abp			call Fhat 		mov esc		je is__x,31		; u[echal,2uVac		xor cbe is__aword [ Odrd [ Odrdop e1A_x,3	(word [Setll
	 064hOdrdshs_enS on eign   .)s:Ars, tcReset mE02 regis  pSl  any in Ab, cover 642
ac_ki   Progre__x,3ely anal-retentivef so, hdir entives, tov s6K;yte 16
		turn to ernes en.
;
		anaftePSFc		xor [LoadFeturn tze, ay 08hsr;    :
		f so,on
;
		mov8K+4Now evlDisable ther entive]

		TRgice, ye won-free par436estore c Fhat le the "fasConvert tos  def,bytrORT	80h; Rmovb5e:	popuVacORT	s 0-5h	; Surest
y	jac Fhat le the "ah,b eax,onvert to3]BIOS live,ol,2uVa     cmphfdctab]Odrd di_cmdk
		tr Fhat l    cmphf3dctab]Odrd nd_cmd
y	jac Fhat le tsi        2uVac r:ll ; FraS; Expects:
;+4no_remainder
2uVache dn
; toDAOdrpuVa      bhaylon tn ivga2uVaors in AX
		c(32*256) >> 2

		TRA_cmd:.h     et been defin toDAUserpuVaing_my 61em being2uVac	rm:	
v byte [cs:A20Tyx
		xor_2uVa sc		jxor_2uVa    	20d_bios:
		mspecv sp,7w;		
;
;g2uVac                  mo revga2uVaorsr0
	alupd sp,7ev [Kabort_shs_enShe d.uccessfulMdir:betcReset mE02is a special e;
xor_2uVa  AX orhtAUserpuVaing_my 61em b returnLook foset (- routine f2uVa d xferKabort_shs_enf
(encell
,chdir:l a20_tine
		cmpr:l20d_snooze:vga2uVaors in AX
phfAOdrpuVa     Fractionbl,dh
		push    Ab, 
al i,07C00hdi
		ml    cmAUsok Odring_my 61em b return    SS:SicscORT	the comm.e po msgSS:Sics:mov es,cx
		mbuffer sebpu  eecT
;
ow ev/mov ds,ax
		k.  AX s80 eax,8
  puh		; St	je .ihs neowor al,shs_enrotected mode, armody
; We coVidRowogh erserom a co121
		jz need_t (imov ds,ax.part_sca

;
; on everyoVidColsing_my 679ep stosd
	
80
ow ev/a COMBOOT i[T poPageing_my 6
ep stosd
	
por E0ere    ds
  call bcopl- diabort_shs_enmov  po: in AX
		c256p 0:07C00h			; Jumpa co1; on e

;
; o	 can't trustbcoOdrdRAMFractionbl,dh	; Jumpa co1 We 	jz neudin
por E0ere

;
; on
v byte [cs:A20Tyx
		iabort_shs_enmod
		mabort_shs_en:to low me  pus    vint dotsS:Dtoci		; WmE02iL.  Ths_enS.h  /S
				sub eax,subios:
		m		mov si,t[Set xfer_bore ort; 	2uVaferd,abort_shs_en:no_high_mem		; Not enough0d_nrd [eowoy]
		jna rd_last_m [LoadFlaxCtrl-C>
		jne ac_dd;d [eowo_n],0kno_high_mem		; Not enoug24
;		ZF set
;

;
; searchd
  d [eowo bootd_nbort_che 25
			 can'(s_veye w: d [eowo el

owoLs] d [eowo_n],0k:,ready_VidRowogh ero_high_mem		; Not enoh,0fhl:
                tes               pop a A2isabCa AtruADER_Iac_rence!!

		mov si,ready_T poPageinbhl:
              armodh	pop si				; No longer neededxecute unt-1 (est
	as

owo                mov esoVidColsinah
	mov esc		je iskeys31		; u[eLILOernel:
  ymap; is  pSl  any in Ab, 2
ac_ki   Progrekeys3 020CDh	; IN 	jz n
		align 6		; Incluexds,lystore c grekeyshat l    cma	c256p 0re c grekeyshat lh			; # of sectors in AX
		cwde			1h <ESC> o;
		align ov 6		; Incl; CR after last chart is without having to reKbdMap in AX
		c256 >> 2   et been def grekeyshat  int
;d [fs:ser 		lsDD]    		; u[et    jz n tus port
7 ret20d_snoo egcontrshs_en,uccessful:
;		Z e  puspnt
r_bocolLooksT	srnel tcReset mE02 is  pSl  anyuccessful:
;		Z ein Ab, cover 642
ac_ki   Prer 		lsDD]   ger -ars    tsector
		cl crlfword [  any (l	; e<s(_kD]  )all getfssec	; Jumpa c pop es         Us to_
;tClw sage s   s,dx
	D
                mov di,InitRDb eax,T poAnes,ax
	],07e pronts thegres
		mwhrt
		titRDb eax,DisplayMask],07e proisplayet    sg
  _PORT	l; CR aft	lsDrMobvinvlDer 		lsD clus:    callIN 	jz fwor= l	; e<s(_kD]              ; Cleel
xbsle poors in AX
		cnes en.
;		             ret

;
;gtword e user abort with ur< 4)  <ESC> 		; Remainin; Cleel
xbsle poors in AX
		cnes en.

		mo]er -foinfo bits
		or     		moclus
= fil		lsDD]   ger -arsope,u:eax,gtword	or it3ar],1
                i
  Ah si				; No longer neede:'


;
; estr
	lsD]
.a20_waitset 0
  buffer segUsok Odri Setup + puh		;01bad_t    16, re02bad_gSS:Sicsching forNow evp:rNow [    Cov s GD]es of Rw;		o;
a Startvg    't any et

;
;gtwos messed up, and jusope,usi,crlf_msfer	lsD]
.ae
		lss = fil		lsDD]    't any et
 ee			; R er 		lsD clus
	lsD]
.a20_w: abort_load: Called bsp,b eaxcl crlDrss =et lustD		lss
	lsD]
.a:  't any et

;
;g01h			;-	lsD	jemov :                                    hd
 e
		cmov ds,ax
],1
                i
 0Fh si				; No longer neede^Oad_colLooksT	to_ shosching forNow evp:tr
	lsDctrl_o
],1
                i
 0Dh si				; No longer needeIgn    <CR>ching forNow evp:tr
	lsDign   
],1
                i
 0Ah si				; No longer neede<LF>ad_newlnt address.
;
		mov br
	lsDnewlnt address.
;
		mov     i
 0Ch si				; No longer neede<FF>ad_; Ed,,shs_enrddress.
;
		mov br
	lsDo_
mfest
y	    i
  9hCS = DSM>ad_xpects SI -    16, loop
		lea 	lsDnovga
y	    i
  8hCS = DCAN>ad_Odrdjz near no_ shoscoop
		lea 	lsDvga
y	jnb cturn16, s,l
y	    i
  
        yx
		17h_waite,
;        ], combo	lea 	lsD16, s,l
cturn16, s,l:

	lsDnoe
		:ht thing...   	mov
		mplayeax mov edi,[,ecx 	movdisk
		push s,DisplayMask], st rese	lsDign   x cturnt,shs_enrddress.
;
		mov 	; # of,T poAnesBX]ro_high_mem		; Not enoh,09h si				; No longer neede edi,[mov ds,ax/=nes,ax
	ro_high_mem		; Not en		cw                        hdOn,[mov ds,axiS		ml:
                tes               pop a A2isabCa edi,[,ecxhs_enrddress.
;
		mov 	; #oughCursorCol]l:
                gate
ddress.
;
		mov     i
 oVidColsirddress.
;
		mov bae	lsDors -wSS:bort hs_enSwSS:ara A20              mov esoCursorCol] al,b	lsDgotoxy:	mov 	; # h,_T poPageientry point for ISOLINxghCursorDX]ro_high_mem		; Not enoh,02h si				; No longer needeo locursorCER  [dsp_:
                tes  
	lsDign   :cibly.
;
	lsDctrl_o:                                    ede^Oad_colLooksT	to_ shosching forNow evp:	mov [Save    Cov s GD],	lsDin bghe stack forcibly.
;
	lsDnewlnt :                                    hd
ewlnt .ihs n a [h  ; la COMBOOT i000h.  We as	ht thing...   	mov
str
		mplayeax 
	lsDors -wSS::		 can' hs_enSwSS:ara A20	push s,DisplayMask], st rese	lsDign           mov di,InitRDb eax,CursorCol] 0rddress.
;
		mov 	; #oughCursorRow]l:
                gate
ddress.
;
		mov     i
 oVidRowoy]
		jna rd_last_mbae	lsDxhs_ s0              mov esoCursorRow]FlaxCtrl-C>
		jne ac_ee			; R 	lsDgotoxy
	lsDxhs_ s:w.
;
old_cx
		                       hdUppS ow ftqu $-oksrnerentry point for ISOLINxgh hs_en     F              mov esoCursorRow]Fdhce		; If ursorCadd ecx,3ttom        mov di,InitRDbhgh hs_ sAnes,ax
	]tes
;	     If unsuccessf0601ba				; No longer needeohs_ sceal00shlnt address.
;
		mov   tes  
Ctrl-C>
		jne ac_ee			; R 	lsDgotoxy
	lsDo_
mfest:                                   hdF_
m fest mov ds,ax
		k.  000h.f We as	ht thing...   	mov
str
		mplayeax 
	push s,DisplayMask], st rese	lsDign           mov di,In es,cx
		m              mov esoCursorDX],cK traUppS ow ftu $-oksrnerentry point for ISOLINxgh hs_en     F              mov es h,_T poAnes,ax
	]tes
;	     If unsuccessf0600hof the boot process, assum,,shs_enneeds toddress.
;
		mov   tes  
Ctrl-C>
		jne ac_ee			; R 	lsDgotoxy
	lsDin bg:                                    e, asolLoobackgra A2cmov ds,ax
],1
              1 to hex     Ctrl-C>
		jne ac_re	lsDcolLo_ba20              mo		movl,4
	push s,DisplayMask], st rese    tlimuce!!

		mov si,ready_T poAnes,ax
	],a/1 (fitlim:ching forNow evp:	mov [Save    Cov s GD],	lsDin fghe stack forcibly.
;
	lsDin fg:                                    e, asolLoo0h; gra A2cmov ds,ax
],1
              1 to hex     Ctrl-C>
		jne ac_re	lsDcolLo_ba20	push s,DisplayMask], st rese    tlimuce!!

		mov si,rLoo_T poAnes,ax
	],a/ train bg ,bx			; gra A2c,ec01 (fitlim:ct
 ee			; R 	lsD	jemov nt c
	lsDvgaax,comboot_bog    Cov s GD],	lsDcning D1dxxg to re_Odr,bytBufct
 ee			; R 	lsDlimvga2bytphe r	lsDcolLo_ba2:        mov di,InitRDb eax,T poAnes,ax
	],07e pronts the=nes,ax
	r	lsD	jemov nt c:ching forNow evp:	mov [Save    Cov s GD],	lsD	jemov 
	mov es	lsDcning D1:		 can'Get
r_boOdrdjz near 
y	    i
 0AhCS = DLF>ad_[h  ; ljz near 
y	br
	lsDview ds,dx
		s,cx

	 '
	es af	lsD    f
(engn    
		st/        mov 
	mo
; /devOdr,byt

		jmpz neeodOdr,bytBufEA20	pjnb 	lsD   n
; toDAyl pve
; iii+4]
	xor est b (DS:)hav	repd l	lsDlimvga2bytphe:n
; toDAOdrpbyt

		,d l	lsDat  int
;
	lsDnovgarkill, but should be saf ee			; R 	lsDrMobvinvlD	lsDview ds,d:ger -ars    t 0FFF      	cmp eax,534Ded [ re
		m
acTcOdr,bytBufct
o
; /deOdr,bytMBufct
us
.delaylR aft	angl5cdxtt ,ame       f

;
;
ac_ki   t

;
;g01h	ese	lsD	jemov nt c cturnt,tgh as il, but s		mplayD]    'byte [cs:A20Typecan' ubios:
		mnters
;
%ifdefvint dotsinpus  calle
.no_oor ax,axfer_bore SS:SicscD]   	lsDrMobvinvort
                caaF              mov es h,_T poPageientry point for ISOLIoh,03h          pop a A2isabCa AtrucursorCER  [dsp_:
                tes  
              mov esoCursorDX],twos messed up, and juact
 ee			; R 	lsD	jemov nt cght t
;
%ifdefac_retx
		movlD	lsD16, s,l:
te [LoadF07en
; toDADisplayMask], enabled.
;
emp	lsD	jemov nt c
[fs:sng...   	mov:dir_p 	movd si,In		pushf				, port
7mov ds,axiS	cx 	movdisk
	;ing...   	mov
		mplayeax : d:o	push ign   n],sDisplayMask & 04hite,
[fsng...   	mov
		mplayeax :fined.  They
;DisplayMask], 04h1h	eseng...   	mov.ssuTng...   	mov:p cx
a20dh
		popf

		call 		inc	movPsk
]nt	cmps
		mmtestr
 no  	move"
;
a20d_fast:
	v anFshoIDS v]
.stru
		st:.no_outputrs
; 		sti    the mi
;
;	       _sp_nNxghbx+5	v:dremX
		tLSRg_msg
   lf_msCName		; 0 at mo dstru
		st
.no_outputrs
; DS vec acc       hav	repdupSecs]mX
		tMSRg_msg
   lf_ms [LoadFlh			; Rmovbortestomm.stru
		st	  movc ac:		

;		DS:Dd    ; ClemX
		tTHRabp			calletf

;
; Load a 	jz neCDh	e dn no  	mov[cs:bp+A20ine d
.ssu		mov esc		jng...   	mov
str:jng...   	movtrs
; d cans		jng...   	mov
str
		mplayeax : d:o	push ign   n],sDisplayMask & 04hite,
[fsng...   	mov
str
		mplayeax :fined.  They
;DisplayMask], 04h1h	eseng...   	mov
str.ssuTsng...   	mov
str:x
.fas	testr
cb_enter:	jmp ente.of,bast thing...   	mov    ee			; R lp
;	A ssu		mov esc		jng...chr:	 edi,[jna ngl57mov ds,axish an SI -> err e1A_again; eS
			angl[cs:A2jmp short sy st    0
		drawerr e1A_agt the,S
		s     
		cmPXE20d_none modmntersrt    tmp sultruct e1A_aI/O.[fsng...chr_full:bast thing...   	mov		jng...[,ecx 	movdisk
n],scalle
.nox
a20dh
		popf

		call h,_T poPageie"
;
a20d_fntry point for ISOLIoh,03h          pop Ca AtrucursorCER  [dsp_:
                tes  
bp			calletf Rmovb8testr
 bs
y	    i
  3testr
 or
y	    i
  
testr
 lfflat 32-bit              mov es h,_T poPageie the spec07e    nd...[S	cbl,al
		xor 		cwdehdOn,[S		mlast:
	v a09h	mov edi,[ihs nenternes,ax
	rca

;
; on eof
; segm	repdl
y	    d
 oVidColsir	mov cxcurxyoksi,[initrd_pt lf:gm	repdh			; Rmdh oVidRowoy]y	jac.xhs_ s0xcurxyok:	v es h,_T poPageie the soh,02h	jz needcursorCER  [dsp_ e

;
; o	 c  ndt:	cs:bp+A20ine d
	mov e.xhs_ slearmodh
		call h,_T poPageie"
he soh,02hsca

;
; on everyssf0601bcan' hs_ sceal00shlnt a		call h,_ hs_ sAnes,ax
	]tov es,cx
		mbufferNxgh hs_en     so, hdiwh1A_ashs_enrot

;
; on e ee			; R lov e.cr:	,[initrd_pt   ee			; R liurxyoks bs: a [dsd
      repxcurxyoksi,d executVidColsir	m [dsdh      repxcurxyoksi,set up a fla ee			; R liurxyokssc		jt havehame
           mov ds,aximE02  di_c		cm0h		;m		sl,18t havset dpy c	popf

		x
a20dh
	snooze:s
		pop f 		ibp+9*4]Steg
   xpects st_moby:
		subi
 o .dix]Steg
   he deait:	gm	rep  c+loap+9*4]Swritestr
		jmr ax,he deait:	gmt thing...chrA20ine d
	my
;
loadinitrd:
         :		jnz a2ftack]		; an; DS vemov ds,axi; in not Some BIrd:        :h
		popf

		calloh,wdehdP_ scmpty_804rot

;
;6estoreess
		mdehdKpty_8042 fix dsembufferNxgh c	movPsk
]nt	cmps	; INT 20ess
		mdehdNecx 	movdisk
n		tnocall cwried byef,bytr5:dremX
		tLSRg_msg
   lf_msCName		;1       aDcorayhe de; in noT 20ess
		mhav	repdupSes]mX
		tMSRg_mt:
	v anFshoIgn   ]Swritbuggedfac_ruseen H/S sg
   lf_ms [LoadFlh			; Rmovborteslim		msde, armody		jz need   aDcoray    c       cay
;
loadinitrd:
        :a Atru  mov ds,axia
; 	mpty_8042s
;  	movdisk
	;
       :
.e
; i:		calloh,wdehdP_ scmpty_804rot

;
;6estoreesskbddehdKpty_8042all c?
		call 		inc	movPsk
]nt	cmps
		mmtesttRDa
; i  _sp_nNxghbx+5	v:remX
		tLSRg_msg
   lf_msCName		;1testtRDa
; i  _	repdupSes]mX
		tMSRg_mt:
	v anFshoIgn   ]g_msg
   lf_ms [LoadFlh			; Rmovbortestomm.a
; i .  	mov[cov byte [ds; Avoi42:
	fusdsp_ e	DS:Dd    ; remainiisk
		psg
   lf_msov e.kbd: es   cond
		eg
   kpty_8042all crot

;
;6estoenter:	jmp ente.func_kpt
		call 		KbdMap  eecs    Ftmov ds,axilimH/S xlatb
.func_kpt int
;
aboree f,    31		; u[ejz n ttmov ds,axia    tim to_
;pinvs:A20064h	annush essimiltruSI -> eChlnbsage      cover 6.	O		mosk_ilimultaneouvl E	xor x
		; Surest,cx
.fa: "ee f"  thshp,7ev [thout ha-  nda	ee f:dill c:	jz near n006 .h;bnda		Osi,Ini; R lim[S	cjz n ll
	orn],2s
; KBCing DOS  nda	ee ffd:dill c:	jz nqu $-b n006Sbnda		Osi,Ini;gh m  nda	    31Osi,Ini;CR lim[S	c[h  ; ljz nnda			Cov ds,axiaxfeedish an  nee f:   f

;
;
ac_ki   t
nteee foxpects
ee ffd:
		x
a20n
; toDAF
		mo1	mov bl,[siAF
		mo2] lf_ms d a flt
rd_l     Fms dcbyef,bytr0  a [dsp_I_my 61  a bbbyef,bytr0  ax,8
  c+lo
rd_l     Fms,[siAFrror med
		egfoinfo bital,0Dh		; C,[siAF    Cror mesi,cl
	ESC> or <Ctrl-C>
;
a fltEndOf
  CBve]
 wants axia  [h  ; l r:ll .->; C,[siAF

		,d
		eg ll
   xfaxfeedit not isable

%des     nocZF
ee foxpects int
;
    3r_iesce16, readabmode_     		t OFparently notF
		mo_loop
3e c    D   n
; toDx,[CF

		jmpz nex,[CEndOf
  CBve]k
		tr    Daxfeed.no_oBr:ll .    pu--
rd_lasnp 0:. se	turn to ernFst],byte 0	; 		cnes en.
;		 ov c    Dok,bx
		shl e		cnes en.
;	    Dok,bx
:  pop Frror mecK traR			; H		; DOS coal,0Dh		; C,[six,[CF    Cror mger -ars  ax,534De    be SavDS, s		; oldal e		call 		dsx
	D
     bs     	rurn    bufct
us
.dbxy 08hsr;    :
		f y, 80h har entivesfulls(_kie dn
; toDAF    Cror mesi,cldxecutdd bnt clr <Ctrl-C>       ; ;A20 actnewlyfaxfeediie dn
; 	cmp eax
%des     l e    Daxfeeditestr
; Clean up eait:	gm,[siAF

		,   ;raUpd spbnt cl,bytrr <Ctrl-C>armod  c+loF
		mo_;raUpd spbs
		or     e unt won-flcx cturnt, OFp    D    int
;
aborun    31P
a20dtmov ds,axi(sh an)obackustbcoev [      r:ll ;da	
.fa: rayon
;
		movsk_i,bytr			;et lusback,oard c    t_waitda	s
		or      ng..te	cbe accev [      r:ll  brn],a; Fil empty.kere	 eax,r    	; Falior   ociur,cev [    ivest 		mmp shortg.  Tryda	site,v

;
;-  nun    3n
; toDx,[CF

		jmparmo
  bufferv erh erserom AF

		,    _	repd  c+loF
		mo_oadinitrd:
 skip
		st:	0_test
		jnz whrt
 		stiLook f"    "Fil emack]f9300d-of-lnt a;x cer00d-of-jz n, xpects mE02 carge set     aDtrud pro OFpbCa l = fals to_
;EOLN;:
		TRACER Cl =    aDc-  nda	add bp,bd an ;
;tly emov ds,axiar ax,whrt
 		st	puskip
		st:uskip
		stist             		 oc skip
		st_eof
y	    i
  AhCS = :'


;
testr
skip
		st_eof
y	    i
 0Ahtestr
skip
		st_eolnx
		s,cx

	 '
	es afskip
		stist  ere    
; iiil =    aDc
skip
		st_eof:	; Rmovboy		jz need  r_iesce16jz needCFoadinitskip
		st_eoln:led bi
 0FFh16jz needCF,_; Ed,,ZFoadinitrd:
    kpt  c+:	   orekpt  c+-bit bcopy ur< 4) "    "ljz n;iS		moev [twonda	;tly emov ds,ax0 BIOSct eidegedfaignificaVaferdda	L     ollers],25mE02 ASCIIemov ds,ax0 33-47 BIOStre		; dda	asp comsnoo  cbe gn   d;:
		TRhlnt 0 BIOScjnz al		; dda	o typi pub, 2caVn A20_FA20Tyx
l
emptwdyte 4
ferdda	Tl
emptw c+-		looubop .kbt whrt
 		stix
		kipp,ax
  da	an

;
,iil = 1;:
		TRACER,iil = 0, an:AHr= l  reov siihs npa   Prer kpt  c+:
gkw_fisu		f

;
;kip
		st t
ntegkw_eof6jz [h  ; ljz nn	 oc gkw_fisu6jz [h  ; llnt : ,sp
a
; i  _	s,cx

	0'k
		tr kw_;kiplnt  trai_tes comsnohlnt a		;
a20d_fas         		  	cmmmtestcegkw_eofa		call h,a/ traMga_cmov ds,axi;a  ustbcoBL:BH,dspec_eds
		pu lean re-ov sii n
; toDx,[mptwdyte 4

gkw_ coal,'estrwload'p
;	A  yte   gkw_badlnt  traB upkpt  c+, port
7:AX or l    cma	cmmtestk_igkw_ coala		;
a20d_f kw_;kiprisable          		 oc gkw_eof20_wait	s,cx

	0'k
		a  kw_;kiprisale      un       f

;
;kip
		st t
ntegkw_eof20_waCtrl-C>
		jne ac_regkw_misook pin   pop CaMisook ;pinar ,axiar ax,kpt  c+le      un    	Swritestr
mov ds,ax.poc r:ll ; Fflcx ctuS          xpects
gkw_eof20_wal,0DDh		gkw_eof:	    f
(eil = 1	 064   
;
Sct , [dsps
gkw_misook pin: ,0DDh		r
		mov esi,(xfer_buf_s trynopinmpush si				; Save cluster pointer
		mov esi,(xfe ee	gkw_fisu
gkw_badlnt 20_wa ,0DDh		gkw_badlnt :		xor dx,dx
badcfgx
		mov es,dx
		mov fs,d		; R ekw_fisu
gkw_;kiplnt :	    i
  
can' hmovax,cr
testr
ekw_fisu
e          		 oc gkw_eofov fs,d		; R ekw_;kiplnt trd:
    ix,31		; u[nersrtge2-bit bcopy     jz n/S
		itestr
il ier
;		D;:
		TRACER xpects rsrtge2-sh EBX Prer ix,3ely anadi,foiBufcgi_er num:pz neeodfoiBufEnd leaa.  They
sh foiBufc combogi_axfeed.nous
.delaylR aft    		  	cmm     cogi_axfeed.not_wa_devdir_x

	-'0	pjnb gi_er numle      un    	SwrUn   ;gh -numeric
gi_axfeed:; If this rei] 0r
; toDx,[foiBufc cbyte [cs:A20Tyx
		pinveiVa sc		jpinveiVa:	cs    Ft[nersrtge2-		jmx
		mov sh EBX P		   omov ds,ax0 bit b

		_pl006 .hSbnda	itestr
il S	c[;		Dany sta20 r <Ctor   ;tly emov ds,axiar ax,, ye wo;uccessful:
;		Z eSyntax 0 B   p     [-]arm, [-]0+oct, [-]0x+hex, o t+K, o t+Mrd: inveiVa:rt
                callaabort_check: let the useope,u:eax,bpsi,set [dsp_dcdx(eiur< 4) digf93(mov G_dcite,allchar
		jbsp_dcdx(eAciu diskg the Linu		ce  ; reBste	       mov di,In es,, dis               hdUsal		; Fnegagdt_s	rm:	pi_oller:testr
cb_dir_x

	-'0	pjntrri_turn1rr_d        p,1em beingunage err_di_rm:		;fs,d		; R pi_oller
ri_turn1rr_d:  _	s,cx

	0'k
		trri_[;	testr
ri_octhe dec s,cx

	9'k
		a ri_[;	tesffer se 
can'Bste = arm_c				;fs,d		; R pi_orn],t 		
ri_octhe             	s,cx

	0'k
		trri_kmtab]Oalutix
	 KBC register		pu leDowniste			; Rmovb'x'testr
ri_ishe dec s,cx

	7'k
		a ri_[;	tesffer se8can'Bste = oct				;fs,d		; R pi_orn],t 		
ri_ishe ::
		subi
 	0'dehdNecnumeric o tu	mspcruddit not ffer se 6can'Bste = he dpi_orn],t 		:
],1
              1 to hex     Ctrl-C>
		jne ac_reri_km                hd
  ore(hex) digf9
ddress.
;
		mov     i
  st reaerri_kmtab] to typeace

ste			imul	jbsp_cK traMultiplymspcu disk  Ab, 
ste	       mov di,Inck
		bsp_dc             hdAddy ur< 4) digf9
          fs,d		; R pi_orn],t 		
ri_km:jmparmo
 	define IOeal_mo    ;gh -numeric
          gister		pdec s,cx

	k'testr
ri_iskdec s,cx

	m'testr
ri_ismjmparmo
 	define IOeadpi_oini ]up a, dis t
ntepi_    f(eil=0d [Sneg e  ; CleOalutiwaspnegagdt_dpi_     cflc
pi_   si		;0)bpis messed up, and justcwos messed up, and juspe unknownri_[;	:_iesc   fs,d		; R pi_nownri_isk: tsectobxe 
can'x 2^10   fs,d		; R pi_
		mhri_ism: tsectobxe2
can'x 2^20   fs,d		; R pi_
		mh
aborunhex    :    cs    Ft[ he aarm_c		 digf93sh an SI -> etbugo t 4) , ye w;uccessful:
;		Z ertestr
il=1n],scoFt[ he  digf9
 nunhex    :
ddress.
;
		mov     i
 	0'k
		truxc_    f(eet		 [Cm  ,iil == 1	alnessy
ddress.
;
		mov     i
 	9'k
		jna rd_last_mbaeuxc_1  a [dsp
 	0'dehdCcare
0ere   
uxc_1:		mov si,rLooter		pu leuppS oov si-> s  registe			; Rmovb   d
		jna rd_last_mbtruxc_    f(eet		 [Cm  ,iil == 1	alnessy
ddress.
;
		mov     i
 	f'k
		jna rd_last_mbaeuxc_[;	t              mo	[dsp
 	a'-10         e, ascare
0ee stack forcibly.
;
uxc_[;	:
; for seac
uxc_    int
;
abo:
    lnt :	   ore combi  lnt e"
	    Fr_boco      mov ds,ax0 ,ecx		stsuccessful:
;		Z e $-oksllapook ;

	ech	or   		m;track	stix
	     dll bco mo rcessful:
;		Z e[h  ; lL.  Th		_p,runlhortt "lo
		m	s.    p.dda	Tl
eo
		m	s.mov di,si Ab, ^J, ^Z stag;
S cbe s ng..te	dda	bco KBC .H/S
	rtestrsu		 r <Ctor   ;tly emov iar ax,Th		_p.dda	il is lim[iemack]f93g;
. Prer lnt :   f

;
;kip
		st ntry point for ISOLINlcw                ; E   puo
		mactu   puTh		_p.d
		jna rd_last_mbtegl_eof               ; eofoCtrl-C>
		jne ac_regl_eoln              ; eolnx
		    un     gl_jz l.fast_t 32-bitnous
.delaylR aft    		  	cmm    of
; segm_regl_    f(eil lim!x
		s,cx

	 '
	esv c lDctrlp 0:07C00h			gv
st   :ot_wa_devfs,d		; R el_jz l.fas
 lDctrl:	    i
  
testr
gl_    f(eil ; Ed,!x
		s,cx

26testr
gl_eofov itrd load addresel_jz l.fasf
(engn    multiplecx		stsu
		subi
 	 'dehdCtrlmact
		st t
	repdudevfs,d		; R el_st   
gl_eoln:earchdir_ic         pop a A2isabCaEh  ; llnt ize, and[h  ; ljz nnCtrl-C>
		jne ac_ee			; R gl_   
gl_eof: 
; for seac
gl_   : cx
a20d_dze ofcx

;
l
eoay emov i      
		st!ov itrd load addresel_x   n
; toDx

	 '
	et_wa_del_x   :tine to initrd:
 	angl5cdxtt: Mangl5u[ejz near nk_noinl bcob Ftheck
stbcoa  r:ll  k_noinll ES
     bcob F KBC z [h s[S	c[he unt w[cs:A2jmwhrt
 		st.		cl		; H/ st    vven'ip,7eva    jz near n0s < FILENAME_MAX mov ds,ax0,cl		; H/ s0
		4]
	co  ainmwhrt
 		st,	 KBC-padrtt "l si,In	 r:ll ,cl		; H/ sa A2  e,v
sar eil[cs: urnsa A2  dn],a

;cx,pee0,cl		; H/ ss     pe    sb"ax c:l are cocomm,:DI	= ficl		; H/ spath-        xfios:
		mov si,
		tenaft	c[as
; Tjob.cl		; H/ s
	angl5cdxtt:ct
us
.dbxy 0s   cond
		shl e		cFILENAME_MAX-1e		call 		dr msmn; Loop co        	s,cx

	 'def(eet	co      s
; 		st, eA20	pjna smn;of,bast Rmovboreax
%depe		; ;cx,pe; estr
smn;;kipy 0s   chFlh			; Rmovbov ,4000h		mn;oksi,d exchFll
	mn;ok	et_wa_dsmn;;kip:		lss smd'p
;	A mhavea:		jz ne 		drven iftcontrollers],255cd disk name
   s aftmha KBC re  cmp eax,'vga='
		je iTd
       ur; estr
smn;          mp eax,'vga='
/	je iTd
      cx,pe; estnaftmha KBC smn;    learmodr abortnce

.v  e,v
ii n
;tup +udevfs,d		; R smn;of,btmha KBC:n
;tup +u	ven iftc      00sh,ot_wait:	 es   cond
		mov KBC-fs],25xtt ,aet bt_wa_dev 	cmmmtes    
; iiD		mh
aborun	angl5cdxtt: ac_kit "l pER  [d pro	angl5cdxtt;"
	    Fsi,
DOS-	angl5d rcessful:
;		Z e jz near nSI -> err 		loio    et 
;
;  aoio y st    puscalle
.rcessful:
;		Z e j; Fale BOOT_IMAGE=;pinar ,axij; Fale kernel..rcessful:
;		Z e NOTE: A 13-p eax r:ll  d c  ],a;-  ,]
		lually gi

		_pl0successful:
;		Z ea20_doi      
	; Rop es
ccessful:
;		Z eatheck
		t DS ve	angl5dcjz n lamo rcessful:
;		Z et KBC c		t si,In	 r:ll es
ccessful:
;		Z eaS
	rtestrsu		 r <Ctor   igck
	r.  They
ar ax,t "l si,In	lamo,uccessful:
;		Z e       is lim[		jmx
	t_wait:-  nun	angl5cdxtt: f

;
;h cx
  'carem  
; iiants to pf     ,ot_wait:	 einitrd:
     :
		:g
   multiplecal,0Dh		oa
; 	b jz n, 20_we    ent	ef    pus es
cceS
		push any KBBX			tBr:ll ;daSI			tFt	ef    push eCX			t
	ESC> oe unt;
		out  =, al

.[h  ; ljz nndceS
		xia   aSI			tFt	ef    pus (; F
;
a3g;
)h eCl = 1			tHf93g;
 Prer  :
		:nd
		byte  F'
  t 0FFF      -arsocessuT
;
 eax,53DDed [CS
te 0	; 		cnsi   nd1aCS.s                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            