!{\src2tex{textfont=tt}}
!!****f* ABINIT/getshell
!! NAME
!! getshell
!!
!! FUNCTION
!! For each k-point, set up the shells of first neighbours and find
!! the weigths required for the finite difference expression
!! of Marzari and Vanderbilt (see PRB 56, 12847 (1997)).
!!
!! COPYRIGHT
!! Copyright (C) 1999-2007 ABINIT group (MVeithen)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! gmet(3,3) = metric tensor of reciprocal space
!! kptopt = option for the generation of k points
!! kptrlatt = k-point lattice specification
!! kpt2(3,nkpt2) = reduced coordinates of the k-points in the
!!                 reduced part of the BZ (see below)
!! mkmem = number of k points which can fit in memory
!! mpi_enreg = informations about MPI parallelization
!! nkpt2 = number of k-points in the reduced BZ
!! nkpt3 = number of k-points in the full BZ
!! nshiftk = number of kpoint grid shifts
!! rmet(3,3) = metric tensor of real space
!! rprimd(3,3) = dimensional primitive translations (bohr)
!! shiftk = shift vectors for k point generation
!! wtk2 = weight assigned to each k point
!!
!! OUTPUT
!! kneigh(30,nkpt2) = for each k-point in the reduced part of the BZ
!!                    kneigh stores the index (ikpt) of the neighbouring
!!                    k-points
!! kptindex(2,nkpt3)
!!   kptindex(1,ikpt) = ikpt_rbz
!!     ikpt_rbz = index of the k-point in the reduced BZ
!!     ikpt = index of the k-point in the full BZ
!!   kptindex(2,ikpt) = 1: use time-reversal symmetry to transform the
!!                         wavefunction at ikpt_rbz to the wavefunction at ikpt
!!                      0: ikpt belongs already to the reduced BZ
!!                         (no transformation required)
!! kpt3(3,nkpt3) = reduced coordinates of the k-points in the full BZ
!! mvwtk(30,nkpt2) = weights required to evaluate the finite difference
!!                   formula of Marzari and Vanderbilt, computed for each
!!                   k-point in the reduced part of the BZ
!! mkmem_max = maximal number of k-points on each processor (MPI //)
!! nneigh = total number of neighbours required to evaluate the finite
!!          difference formula
!!
!! COMMENTS
!! The array kpt2 holds the reduced coordinates of the k-points in the
!! reduced part of the BZ. For example, in case time-reversal symmetry is
!! used (kptopt = 2) kpt2 samples half the BZ. Since some of the neighbours
!! of these k-points may lie outside the reduced BZ, getshell also needs the
!! coordinates of the k-points in the full BZ.
!! The coordinates of the k-points in the full BZ are stored in kpt3.
!! The weights mvwtk are computed for the k-points kpt2.
!!
!! In case no symmetry is used to reduce the number of k-points,
!! the arrays kpt2 and kpt3 are equal.
!!
!!
!! PARENTS
!!      nonlinear
!!
!! CHILDREN
!!      dgetrf,dgetri,getkgrid,leave_new,wrtout,xcomm_world,xmax_mpi_intv
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine getshell(gmet,kneigh,kptindex,kptopt,kptrlatt,kpt2,&
& kpt3,mkmem,mkmem_max,mpi_enreg,mvwtk,&
& nkpt2,nkpt3,nneigh,nshiftk,rmet,rprimd,shiftk,wtk2)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_13recipspace
 use interfaces_lib01hidempi
#endif
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: kptopt,mkmem,nkpt2,nkpt3
 integer,intent(inout) :: nshiftk
 integer,intent(out) :: mkmem_max,nneigh
 type(MPI_type),intent(inout) :: mpi_enreg
!arrays
 integer,intent(inout) :: kptrlatt(3,3)
 integer,intent(out) :: kneigh(30,nkpt2),kptindex(2,nkpt3)
 real(dp),intent(in) :: gmet(3,3),kpt2(3,nkpt2),rmet(3,3),rprimd(3,3)
 real(dp),intent(in) :: shiftk(3,nshiftk),wtk2(nkpt2)
 real(dp),intent(out) :: kpt3(3,nkpt3),mvwtk(30,nkpt2)

!Local variables-------------------------------
!scalars
 integer :: bis,flag,ier,ii,ikpt,ikpt1,ikpt2,ikpt3,ineigh,info1,info2,is1,is2
 integer :: ishell,jj,kk,kptopt_used,mkmem_cp,ndiff,nkpt_computed,nshell,nsym1
 integer :: orig,spaceComm,wtkflg
 real(dp) :: dist_,dtm,kptrlen,norm2dk,s1,scalpdk
 character(len=500) :: message
!arrays
 integer :: dsifkpt(3),idiff(2,6),neigh(6,nkpt2),symafm_dummy(1),vacuum(3)
 integer,allocatable :: ipiv(:),symrel1(:,:,:)
 real(dp) :: dist(0:10),dk(3),dk_(3),resdk(3),rvec(6),shiftk_(3,8)
 real(dp),allocatable :: dk_save(:,:),mat(:,:),tnons1(:,:),work(:,:),wtk3(:)

!************************************************************************

#ifdef VMS
!DEC$ ATTRIBUTES ALIAS:'DGETRI' :: dgetri
!DEC$ ATTRIBUTES ALIAS:'DGETRF' :: dgetrf
#endif

!DEBUG
!write(6,*)'getshell : enter'
!write(6,*)'nkpt2 : ',nkpt2
!write(6,*)'nkpt3 : ',nkpt3
!stop
!ENDDEBUG

! In case of MPI //: compute maximum number of k-points per processor
if (mpi_enreg%paral_compil_kpt == 1) then
!BEGIN TF_CHANGES
   call xcomm_world(mpi_enreg,spaceComm)
!END TF_CHANGES
   mkmem_cp=mkmem
   call xmax_mpi_intv(mkmem_cp,mkmem_max,spaceComm,ier)
else
   mkmem_max = mkmem
end if

! ------------- In case kptopt = 2 set up the whole k-point grid -------------

! kpt3(3,nkpt3) = reduced coordinates of k-points in the full BZ

if (kptopt == 3) then

 allocate(wtk3(nkpt3))
 kpt3(:,:) = kpt2(:,:)
 wtk3(:) = wtk2(:)
 do ikpt = 1,nkpt3
  kptindex(1,ikpt) = ikpt
  kptindex(2,ikpt) = 0
 end do

else if (kptopt == 2) then

 allocate(wtk3(nkpt3))
 dsifkpt(:) = 1 ; ii = 5 ; kptopt_used = 3
 symafm_dummy(1) = 1
 shiftk_(:,:) = 0._dp
 shiftk_(:,1:nshiftk) = shiftk(:,1:nshiftk)

 nsym1 = 1
 allocate(symrel1(3,3,nsym1),tnons1(3,nsym1))
 symrel1(:,:,1) = 0
 symrel1(1,1,1) = 1 ; symrel1(2,2,1) = 1 ; symrel1(3,3,1) = 1
 tnons1(:,:) = 0._dp
 vacuum(:) = 0

 call getkgrid(dsifkpt,ab_out,ii,kpt3,kptopt_used,kptrlatt,&
&  kptrlen,nsym1,nkpt3,nkpt_computed,nshiftk,nsym1,&
&  rprimd,shiftk_,symafm_dummy,symrel1,tnons1,&
&  vacuum,wtk3)

 if (nkpt_computed /= nkpt3) then
 write(message,'(a,a,a,a,i4,a,a,i4)') ch10,&
& ' mv_3dte: BUG - ',ch10,&
& ' The number of k-points in the whole BZ, nkpt_computed= ',nkpt_computed,&
& ch10,&
& ' is not twice the number of k-points in half the BZ, nkpt3=',nkpt3
 call wrtout(ab_out,message,'COLL')
 call wrtout(06,  message,'COLL')
 call leave_new('COLL')
 end if

 kptindex(:,:) = 0
 do ikpt3 = 1, nkpt3

  flag = 1
  do ikpt2 = 1, nkpt2

! In case, the k-points differ only by one reciprocal lattice
! vector, apply shift of one g-vector to kpt(:,ikpt3)

   dk_(:) = kpt3(:,ikpt3) - kpt2(:,ikpt2)
   dk(:) = dk_(:) - nint(dk_(:))
   if (dk(1)*dk(1) + dk(2)*dk(2) + dk(3)*dk(3) < tol10) then
    do ii = 1, 3
     if ((dk(ii)*dk(ii) < tol10).and.(dk_(ii)*dk_(ii) > tol10)) then
      kpt3(ii,ikpt3) = -1._dp*kpt3(ii,ikpt3)
     end if
    end do
   end if

   dk_(:) = kpt3(:,ikpt3) + kpt2(:,ikpt2)
   dk(:) = dk_(:) - nint(dk_(:))
   if (dk(1)*dk(1) + dk(2)*dk(2) + dk(3)*dk(3) < tol10) then
    do ii = 1, 3
     if ((dk(ii)*dk(ii) < tol10).and.(dk_(ii)*dk_(ii) > tol10)) then
      kpt3(ii,ikpt3) = -1._dp*kpt3(ii,ikpt3)
     end if
    end do
   end if

   dk(:) = kpt3(:,ikpt3) - kpt2(:,ikpt2)
   if (dk(1)*dk(1) + dk(2)*dk(2) + dk(3)*dk(3) < tol10) then
    kptindex(1,ikpt3) = ikpt2
    kptindex(2,ikpt3) = 0       ! no use of time-reversal symmetry
    flag = 0
    exit
   end if

   dk(:) = kpt3(:,ikpt3) + kpt2(:,ikpt2)
   if (dk(1)*dk(1) + dk(2)*dk(2) + dk(3)*dk(3) < tol10) then
    kptindex(1,ikpt3) = ikpt2
    kptindex(2,ikpt3) = 1       ! use time-reversal symmetry
    flag = 0
    exit
   end if

  end do     ! ikpt2

  if (flag == 1) then
   write(message,'(a,a,a,a,i4)') ch10,&
&   ' mv_3dte: BUG - ',ch10,&
&   ' Could not find a symmetric k-point for ikpt3=  ',&
&   ikpt3
   call wrtout(ab_out,message,'COLL')
   call wrtout(06,  message,'COLL')
   call leave_new('COLL')
  end if
 end do    ! ikpt3

else

   write(message,'(a,a,a,a)') ch10,&
&   ' mv_3dte: ERROR - ',ch10,&
&   ' the only values for kptopt that are allowed are 2 and 3 '
   call wrtout(ab_out,message,'COLL')
   call wrtout(06,  message,'COLL')
   call leave_new('COLL')

end if   ! condition on kptopt


! --------- Compute the weights required for the Marzari-Vanderbilt ---------
! --------- finite difference formula ---------------------------------------


! Initialize distance between k-points

dk(:) = 10._dp
dist(:) = 0._dp
dist_ = 0._dp
do ii = 1,3
do jj = 1,3
 dist_ = dist_ + dk(ii)*gmet(ii,jj)*dk(jj)
end do
end do
dist(1:10) = dist_


! Examine the metric tensor rmet
! get the independent elements

rvec(:) = 0._dp
ndiff = 0

do ii = 1, 3

 flag = 1
 do kk = 1, 6
  if (abs(rmet(ii,ii) - rvec(kk)) < tol6) flag = 0
 end do

 if (flag == 1) then
  ndiff = ndiff + 1
  rvec(ndiff) = rmet(ii,ii)
  idiff(1,ndiff) = ii
  idiff(2,ndiff) = ii
 end if

end do

do ii = 1, 3

 if (ii < 3) then

  do jj = ii+1, 3
   flag = 1
   do kk = 1, 6
    if (abs(rmet(jj,ii) - rvec(kk)) < tol6) flag = 0
   end do

   if (flag == 1) then
    ndiff = ndiff + 1
    rvec(ndiff) = rmet(jj,ii)
    idiff(1,ndiff) = ii
    idiff(2,ndiff) = jj
   end if
  end do

 end if

end do

!DEBUG
!write(6,*) 'Analysis of the metric tensor rmet'
!write(6,'(3(2x,f16.9))') rmet(1,:)
!write(6,'(3(2x,f16.9))') rmet(2,:)
!write(6,'(3(2x,f16.9))') rmet(3,:)
!write(6,*)
!write(6,*) 'Number of different elements : ', ndiff
!do ii = 1, ndiff
! write(6,'(2(3x,i3))')idiff(:,ii)
!end do
!stop
!ENDDEBUG

ishell = 0
wtkflg = 0
kneigh(:,:) = 0
neigh(:,:) = 0

allocate (dk_save(3,ndiff))

! For some structures, it may happen that the first shell
! is not enough to find the weights.
! In this case, include more shells (max = ndiff)

do while ((wtkflg == 0).and.(ishell < ndiff))

ishell = ishell + 1

! Find the distance between the k-point and the shell number ishell

do ikpt = 2,nkpt3
 dk(:) = kpt3(:,1) - kpt3(:,ikpt)
 dist_ = 0._dp
 do ii = 1,3
 do jj = 1,3
  dist_ = dist_ + dk(ii)*gmet(ii,jj)*dk(jj)
 end do
 end do
 s1 = 1.1_dp
 if (ishell > 1) s1 = dist_/dist(ishell - 1)
!
!  check if the new dk is linearly independent from the existing ones
!
 resdk(:) = dk(:)
 do ii=ishell-1,1,-1
  scalpdk=resdk(1)*dk_save(1,ii)*gmet(1,1) &
&        +resdk(2)*dk_save(2,ii)*gmet(2,2) &
&        +resdk(3)*dk_save(3,ii)*gmet(3,3) &
&    +two*resdk(2)*dk_save(3,ii)*gmet(2,3) &
&    +two*resdk(1)*dk_save(3,ii)*gmet(1,3) &
&    +two*resdk(1)*dk_save(2,ii)*gmet(1,2)
  norm2dk=dk_save(1,ii)*dk_save(1,ii)*gmet(1,1) &
&        +dk_save(2,ii)*dk_save(2,ii)*gmet(2,2) &
&        +dk_save(3,ii)*dk_save(3,ii)*gmet(3,3) &
&    +two*dk_save(2,ii)*dk_save(3,ii)*gmet(2,3) &
&    +two*dk_save(1,ii)*dk_save(3,ii)*gmet(1,3) &
&    +two*dk_save(1,ii)*dk_save(2,ii)*gmet(1,2)
  resdk(:) = resdk(:) - scalpdk/norm2dk*dk_save(:,ii)
 end do
 s1     =resdk(1)*resdk(1)*gmet(1,1) &
&       +resdk(2)*resdk(2)*gmet(2,2) &
&       +resdk(3)*resdk(3)*gmet(3,3) &
&   +two*resdk(2)*resdk(3)*gmet(2,3) &
&   +two*resdk(1)*resdk(3)*gmet(1,3) &
&   +two*resdk(1)*resdk(2)*gmet(1,2)

 if ((dist_ < dist(ishell)).and.(dist_ - dist(ishell-1)>tol8).and.&
&  (abs(s1) > tol8)) then
   dist(ishell) = dist_
   dk_save(:,ishell) = dk(:)
 end if
end do

!DEBUG
!write(6,*)'ishell, dist = ',ishell,dist(ishell)
!ENDDEBUG

! For each k-point in halft the BZ get the shells of nearest neighbours.
! These neighbours can be out of the zone sampled by kpt2.

do ikpt2 = 1, nkpt2              ! k-points in half the BZ
 orig = sum(neigh(:,ikpt2))
 nneigh = 0
 do ikpt3 = 1, nkpt3             ! whole k-point grid
   dk(:) = kpt3(:,ikpt3) - kpt2(:,ikpt2)
   dk_(:) = dk(:) - nint(dk(:))
   dist_ = 0._dp
    do ii = 1,3
    do jj = 1,3
     dist_ = dist_ + dk_(ii)*gmet(ii,jj)*dk_(jj)
    end do
    end do
    if (abs(dist_ - dist(ishell)) < tol8) then
     nneigh = nneigh + 1
     kneigh(orig+nneigh,ikpt2) = ikpt3
    end if
 end do
 neigh(ishell,ikpt2) = nneigh
end do


! Check if the number of points in shell number ishell
! is the same for each k-point

flag = 1
do ikpt = 1,nkpt2
 if (neigh(ishell,ikpt) /= nneigh) flag = 0
end do

if (flag == 0) then
 write(message,'(a,a,a,a,i2,a,a)') ch10,&
& ' getshell: BUG - ',ch10,&
& ' The number of points in shell number',ishell,' is not the same',&
& ' for each k-point.'
 call wrtout(ab_out,message,'COLL')
 call wrtout(06,  message,'COLL')
 call leave_new('COLL')
end if

nneigh = sum(neigh(:,1))
!DEBUG
!write(6,*)'ishell = ',ishell,'nneigh = ',nneigh
!ENDDEBUG

allocate(mat(ishell,ishell),ipiv(ishell),work(ishell,ishell))

! Find the weights needed to compute the finite difference expression
! of the ddk
!**********************************************************************

mvwtk(:,:) = 0._dp

 ikpt = 1

  mat(:,:) = 0._dp
  do is1 = 1, ishell

   ii = idiff(1,is1)
   jj = idiff(2,is1)
   do is2 = 1, ishell

    orig = sum(neigh(1:is2-1,ikpt))
    bis = sum(neigh(1:is2,ikpt))
    do ineigh = orig+1, bis
     dk_(:) = kpt3(:,kneigh(ineigh,ikpt)) - kpt2(:,ikpt)
     dk(:) = dk_(:) - nint(dk_(:))
     mat(is2,is1) = mat(is2,is1) + dk(ii)*dk(jj)
    end do

   end do

  end do

!DEBUG
!do ii = 1, ishell
! write(6,*)mat(ii,:)
!end do
!ENDDEBUG

  call dgetrf(ishell,ishell,mat,ishell,ipiv,info1)

  if (info1 /= 0) then

   write(message,'(a,a,a,a,a,a,a,a)') 'getshell : COMMENT -', ch10,&
&   ' The linear system of equations that define the weighting factors',ch10,&
&   ' cannot be solved (info1 /= 0).',ch10,&
&   ' The programm will try to include one more shell of nearest neighbours',ch10
   call wrtout(06,message,'COLL')
   write (*,*) 'info1 = ',info1

  else

   call dgetri(ishell,mat,ishell,ipiv,work,ishell,info2)

  end if

  if (info2 /= 0) then

   write(message,'(a,a,a,a,a,a,a,a)') 'getshell : COMMENT -', ch10,&
&   ' The linear system of equations that define the weighting factors',ch10,&
&   ' cannot be solved (info2 /= 0).',ch10,&
&   ' The programm will try to include one more shell of nearest neighbours',ch10
   call wrtout(06,message,'COLL')
   write (*,*) 'info2 = ',info2

  else

   do is1 = 1, ishell
    s1 = 0._dp
    do is2 = 1, ishell
     s1 = s1 + mat(is2,is1)*rvec(is2)
    end do
    if (is1 == 1) then
     orig = 0
    else
     orig = sum(neigh(1:is1-1,ikpt))
    end if
    bis = sum(neigh(1:is1,ikpt))
    mvwtk(orig+1:bis,ikpt) = s1
   end do

  end if

 do ikpt = 2,nkpt2
  mvwtk(1:nneigh,ikpt) = mvwtk(1:nneigh,1)
 end do  ! ikpt

! Check computed weighting factors

wtkflg = 1
do ikpt = 1, nkpt2
 do ii = 1,3
  do jj = 1,3
   s1 = 0._dp
   do ineigh = 1, nneigh
    dk_(:) = kpt3(:,kneigh(ineigh,ikpt)) - kpt2(:,ikpt)
    dk(:) = dk_(:) - nint(dk_(:))
    s1 = s1 + dk(ii)*dk(jj)*mvwtk(ineigh,ikpt)
   end do
   if (abs(s1 - rmet(ii,jj)) > tol6) wtkflg = 0
  end do
 end do
end do

deallocate(mat,ipiv,work)

end do   ! flag and nshell

deallocate (dk_save)

if (wtkflg == 0) then

   write(message,'(a,a,a,a)') ch10,&
&    ' getshell : BUG -',ch10,&
&    ' There is a problem with the finite difference expression of the ddk'
   call wrtout(ab_out,message,'COLL')
   call wrtout(06,  message,'COLL')
   call leave_new('COLL')

else

 nshell = ishell

 write(message,'(a,a,a,a,a,a,a,i3,a,a,f16.7)') ch10,&
& ' getshell : finite difference formula of Marzari and Vanderbilt',ch10,&
& '            (see Marzari and Vanderbilt, PRB 56, 12847 (1997), Appendix B)',&
&    ch10,ch10,&
& '            number of first neighbours  : ', neigh(1,1),ch10,&
& '            weight : ',mvwtk(1,1)
 call wrtout(ab_out,message,'COLL')
 call wrtout(06,  message,'COLL')

if (nshell > 1) then
is1 = neigh(1,1) + 1
write(message,'(a,a,i3,a,a,f16.7)')ch10,&
& '            number of second neighbours  : ', neigh(2,1),ch10,&
& '            weight : ',mvwtk(is1,1)
   call wrtout(ab_out,message,'COLL')
   call wrtout(06,  message,'COLL')
end if

if (nshell > 2) then
is1 = sum(neigh(1:2,1)) + 1
write(message,'(a,a,i3,a,a,f16.7)')ch10,&
& '            number of third neighbours  : ', neigh(3,1),ch10,&
& '            weight : ',mvwtk(is1,1)
   call wrtout(ab_out,message,'COLL')
   call wrtout(06,  message,'COLL')
end if

if (nshell > 3) then
is1 = sum(neigh(1:3,1)) + 1
write(message,'(a,a,i3,a,a,f16.7)')ch10,&
& '            number of fourth neighbours  : ', neigh(4,1),ch10,&
& '            weight : ',mvwtk(is1,1)
   call wrtout(ab_out,message,'COLL')
   call wrtout(06,  message,'COLL')
end if

if (nshell > 4) then
is1 = sum(neigh(1:4,1)) + 1
write(message,'(a,a,i3,a,a,f16.7)')ch10,&
& '            number of fifth neighbours  : ', neigh(5,1),ch10,&
& '            weight : ',mvwtk(is1,1)
   call wrtout(ab_out,message,'COLL')
   call wrtout(06,  message,'COLL')
end if

if (nshell > 5) then
is1 = sum(neigh(1:5,1)) + 1
write(message,'(a,a,i3,a,a,f16.7)')ch10,&
& '            number of sixth neighbours  : ', neigh(6,1),ch10,&
& '            weight : ',mvwtk(is1,1)
   call wrtout(ab_out,message,'COLL')
   call wrtout(06,  message,'COLL')
end if

end if



! ----------------------------------------------------------------------------

deallocate(wtk3)


end subroutine getshell
!!***
