/*
 *  dp_util.c -- Translate Otter clauses for MACE.
 *
 */

#include "header.h"

#define FUNCTION 0
#define RELATION 1

/* The following is used to collect information on symbols. */

 struct {
     int type;
     char *name;
     int arity;
     char *properties;
     } Syms[100];

 int Num_syms;

 int Dp_sn, Eq_sn, Lt_sn;

/*************
 *
 *   kludgey_e_subsume(c, d)
 *
 *   Does c subsume d (mod equality flipping)?  This works by
 *   inserting flips into (a copy of) d then doing an ordinary
 *   subsumption check.
 *
 *************/

int kludgey_e_subsume(c, d)
struct clause *c, *d;
{
    struct clause *d1;
    struct literal *l1, *l2;
    struct term *t;
    int rc;
    
    d1 = cl_copy(d);
    for (l1 = d1->first_lit; l1; l1 = l1->next_lit) {
	if (l1->atom->sym_num == Eq_sn) {
	    l2 = get_literal();
	    l2->next_lit = d1->first_lit;
	    d1->first_lit = l2;
	    l2->sign = l1->sign;
	    l2->atom = copy_term(l1->atom);
	    t = l2->atom->farg->argval;
	    l2->atom->farg->argval = l2->atom->farg->narg->argval;
	    l2->atom->farg->narg->argval = t;
	    }
	}
    rc = subsume(c, d1);
    cl_del_non(d1);
    return(rc);
}  /* kludgey_e_subsume */

/*************
 *
 *   check_for_bad_things()
 *
 *************/

void check_for_bad_things(c)
struct clause *c;
{
    struct literal *lit;
    for (lit = c->first_lit; lit; lit = lit->next_lit) {
	if (answer_lit(lit))
	    abend("answer literals not allowed in MACE input");
	if (lit->atom->varnum == EVALUABLE)
	    abend("evaluable literals not allowed in MACE input");
	}
}  /* check_for_bad_things */

/*************
 *
 *   int_term(t)
 *
 *   Is t a constant with a symbol representing a positive integer?
 *
 *************/

int int_term(t)
struct term *t;
{
    int i;
    return(t->type == NAME && str_int(sn_to_str(t->sym_num),&i) && i>=0);
}  /* int_term */

/*************
 *
 *   domain_element(t)
 *
 *   Is t a domain element?
 *
 *************/

int domain_element(t)
struct term *t;
{
    return(Flags[DP_INT_DOMAIN].val && int_term(t));
}  /* int_term */

/*************
 *
 *   first_nonvar_arg_term()
 *
 *   Return the first subterm (including the term itself) that
 *   is not a variable and is not an integer.  If the argument
 *   `check' is 0, don't check the term---check only the subterms.
 *   (check is intended to be 0 for atoms and for args of
 *   Dp_sn atoms.)
 *
 *************/

static struct term *first_nonvar_arg_term(t, check)
struct term *t;
int check;
{
    struct rel *r;
    struct term *t1;
    int check_args, i;

    check_args = (t->sym_num != Dp_sn);

    if (check && t->type != VARIABLE)
	return(t);
    else {
	for (r = t->farg; r; r = r->narg) {
	    if (t1 = first_nonvar_arg_term(r->argval, check_args))
		return(t1);
	    }
	return((struct term *) NULL);
	}
}  /* first_nonvar_arg_term */

/*************
 *
 *   first_nonvar_arg()
 *
 *   Given a clause, return the first subterm that satisfies
 *   the preceding routine first_nonvar_arg_term().
 *
 *************/

static struct term *first_nonvar_arg(c)
struct clause *c;
{
    struct literal *lit;
    struct term *t;

    for (lit = c->first_lit; lit; lit = lit->next_lit) {
	if (t = first_nonvar_arg_term(lit->atom, 0))
	    return(t);
	}
    return((struct term *) NULL);
}  /* first_nonvar_arg */

/*************
 *
 *    build_binary_term
 *
 *************/

static struct term *build_binary_term(sn, t1, t2)
int sn;
struct term *t1, *t2;
{
    struct term *t;
    struct rel *r1, *r2;

    t = get_term(); t->type = COMPLEX; t->sym_num = sn;
    r1 = get_rel(); r1->argval = t1;
    r2 = get_rel(); r2->argval = t2;
    t->farg = r1; r1->narg = r2;
    return(t);
}  /* build_binary_term */

/*************
 *
 *   replace_term()
 *
 *************/

static struct term *replace_term(t, ct, vt)
struct term *t, *ct, *vt;
{
    if (term_ident(t, ct))
	return(copy_term(vt));
    else {
	struct rel *r;
	for (r = t->farg; r; r= r->narg)
	    r->argval = replace_term(r->argval, ct, vt);
	return(t);
	}
}  /* replace_term */

/*************
 *
 *   flatten_clause()
 *
 *************/

static void flatten_clause(d)
struct clause *d;
{
    struct term *t, *t1, *t2;
    struct literal *lit;
    int vnum = MAX_VARS;

    t = first_nonvar_arg(d);
    while (t) {
        lit = get_literal();
        lit->next_lit = d->first_lit;
        d->first_lit = lit;
        lit->sign = 0;
        t1 = copy_term(t);
        t2 = get_term();
        t2->type = VARIABLE;
        t2->varnum = ++vnum;
        lit->atom = build_binary_term(Dp_sn, t1, t2);
        for (lit = lit->next_lit; lit; lit = lit->next_lit)
            lit->atom = replace_term(lit->atom, t, t2);
        t = first_nonvar_arg(d);
        }
    
}  /* flatten_clause */

/*************
 *
 *   dp_p_term()
 *
 *************/

void dp_p_term(t)
struct term *t;
{
    if (t->type == VARIABLE)
	printf("v%d", t->varnum);
    else {
	struct rel *r;
	printf("%s", sn_to_str(t->sym_num));
	
	for (r = t->farg; r; r = r->narg) {
	    printf(" ");
	    dp_p_term(r->argval);
	    }
	}
}  /* dp_p_term */

/*************
 *
 *   dp_p_clause()
 *
 *************/

static void dp_p_clause(c)
struct clause *c;    
{
    struct literal *lit;
    struct term *atom, *t1;
    struct rel *r;

    for (lit = c->first_lit; lit; lit = lit->next_lit) {
	if (lit != c->first_lit)
	    printf("   ");
	if (!lit->sign)
	    printf("-");
	atom = lit->atom;
	if (atom->sym_num == Dp_sn) {
	    t1 = atom->farg->argval;
	    printf("%s", sn_to_str(t1->sym_num));
	    for (r = t1->farg; r; r = r->narg) {
		printf(" ");
		dp_p_term(r->argval);
		}
	    printf(" ");
	    dp_p_term(atom->farg->narg->argval);
	    }
	else
	    dp_p_term(atom);
	}
    printf(" .\n");
}  /* dp_p_clause */

/*************
 *
 *   collect_symbols()
 *
 *************/

static void collect_symbols(t, type)
struct term *t;
int type;
{
    int i;
    struct rel *r;

    if (t->type == VARIABLE || domain_element(t))
	return;

    for (i = 0; i < Num_syms; i++) {
	if (str_ident(Syms[i].name, sn_to_str(t->sym_num)))
	    break;
	}
    if (i < Num_syms) {
	if (Syms[i].arity != sn_to_arity(t->sym_num))
	    abend("collect_symbols, multiple arity");
	}
    else {
	Syms[i].arity = sn_to_arity(t->sym_num);
	Syms[i].type = type;
	Syms[i].name = sn_to_str(t->sym_num);

	if (t->sym_num == Eq_sn)
	    Syms[i].properties = "equality";
	else if (t->sym_num == Lt_sn)
	    Syms[i].properties = "order";
	else
	    Syms[i].properties = "-----";

	Num_syms++;
	}
    for (r = t->farg; r; r = r->narg)
	collect_symbols(r->argval, FUNCTION);
}  /* collect_symbols */

/*************
 *
 *   occurrences_cl()
 *
 *   Return the number of occurrnces of term t in clause c.
 *
 *************/

static int occurrences_cl(t, c)
struct term *t;
struct clause *c;
{
    struct literal *l;
    int n = 0;

    for (l = c->first_lit; l; l = l->next_lit)
	n += occurrences(t, l->atom);
    return(n);
}  /* occurrences_cl */

/*************
 *
 *   clause_to_pair()
 *
 *   This routine does the following type of transformation:
 *   a!=x | b!=y | x=y
 *   is made into the pair of clauses
 *   a!=x | b=x   and   a=x | b!=x.
 *   There can be other literals, and a and b need not be constants.
 *
 *   If this can be done, two new clauses are created, and "returned".
 *
 *************/

int clause_to_pair(c, dp, ep)
struct clause *c, **dp, **ep;
{
    struct clause *d, *e;
    struct literal *curr, *prev;
    struct term *t;
    int lits[MAX_VARS];  /* lits[i] is index of Dp_eq literal with right side var i */
    int v, v1, v2, l1, l2, i;

    for (i = 0; i < MAX_VARS; i++)
	lits[i] = -1;

    /* Assume all Dp_eq lits first, and in Dp_lit, var is second arg. */

    for (curr = c->first_lit, i = 0; curr; prev = curr, curr = curr->next_lit, i++) {
	if (curr->atom->sym_num == Dp_sn) {
	    t = curr->atom->farg->narg->argval;
	    if (occurrences_cl(t, c) == 2)
		lits[t->varnum] = i;
	    }
	else if (curr->atom->sym_num == Eq_sn && curr->sign) {
            struct term *a1 = curr->atom->farg->argval;
            struct term *a2 = curr->atom->farg->narg->argval;

	    if (a1->type == VARIABLE && a2->type == VARIABLE) {
		v1 = a1->varnum;
		v2 = a2->varnum;
		if (lits[v1] >= 0 && lits[v2] >= 0) {
		    l1 = lits[v1]; l2 = lits[v2];
		    prev->next_lit = curr->next_lit;
		    
		    for (curr = c->first_lit, i = 0; i < l1; curr = curr->next_lit, i++);
		    curr->atom->farg->narg->argval->varnum = v1;
		    for (curr = c->first_lit, i = 0; i < l2; curr = curr->next_lit, i++);
		    curr->atom->farg->narg->argval->varnum = v1;
		    
		    d = cl_copy(c);
		    e = cl_copy(c);
		    for (curr = d->first_lit, i = 0; i < l1; curr = curr->next_lit, i++);
		    curr->sign = 1;
		    for (curr = e->first_lit, i = 0; i < l2; curr = curr->next_lit, i++);
		    curr->sign = 1;
		    *dp = d; *ep = e;
		    return(1);
		    }
		}
	    }
	}

    return(0);
}  /* clause_to_pair */

/*************
 *
 *   clause_to_clauses()
 *
 *************/

static struct list *clause_to_clauses(c)
struct clause *c;
{
    struct list *l;
    struct clause *d, *e, *f;
    int subsumed;
    l = get_list();
    append_cl(l, c);
    d = l->first_cl;
    while (d) {
	if (clause_to_pair(d, &e, &f)) {
	    insert_after_cl(d, e);
	    insert_after_cl(e, f);
	    rem_from_list(d);
	    d = e;
	    }
	else
	    d = d->next_cl;
	}

    /* Now delete any clauses subsumed by previous clauses in the list. */
    for (d = l->first_cl; d; d = d->next_cl) {
	for (e = l->first_cl, subsumed=0; e != d && !subsumed; e = e->next_cl)
	    subsumed = subsume(e,d);
	if (subsumed) {
	    f = d;
	    d = d->prev_cl;
	    rem_from_list(f);
	    }
	}

    return(l);
}  /* clause_to_clauses */

/*************
 *
 *   process_negative_equalities()
 *
 *   Simplify literals x!=t, where x does not occur in t, by,
 *   in effect, rewolving with x=x.
 *
 *************/

static void process_negative_equalities(c)
struct clause *c;
{
    struct literal *curr, *prev, *l;
    struct term *t1, *t2, *v, *t;

    curr = c->first_lit;
    prev = NULL;
    while (curr) {
	if (curr->atom->sym_num == Eq_sn && !curr->sign) {
	    t1 = curr->atom->farg->argval;
	    t2 = curr->atom->farg->narg->argval;
	    v = NULL;
	    if (t1->type == VARIABLE && !occurs_in(t1, t2)) {
		v = t1; t = t2;
		}
	    else if (t2->type == VARIABLE && !occurs_in(t2, t1)) {
		v = t2; t = t1;
		}

	    if (v) {
		if (prev)
		    prev->next_lit = curr->next_lit;
		else
		    c->first_lit = curr->next_lit;
		for (l = c->first_lit; l; l = l->next_lit)
		    l->atom = replace_term(l->atom, v, t);
		}
	    else
		prev = curr;
	    }
	else
	    prev = curr; 

	curr = curr->next_lit;
	}
}  /* process_negative_equalities */

/*************
 *
 *   check_transformed_clause()
 *
 *   This checks that the transformed clause is equality-equivalent
 *   to the original.
 *
 *************/

static void check_transformed_clause(c, orig)
struct clause *c, *orig;
{
    struct clause *d;
    struct literal *l;

    d = cl_copy(c);
    for (l = d->first_lit; l; l = l->next_lit) {
	if (l->atom->sym_num == Dp_sn)
	    l->atom->sym_num = Eq_sn;
	}
    process_negative_equalities(d);
    if (!kludgey_e_subsume(d, orig) || !kludgey_e_subsume(d, orig)) {
	printf("%% WARNING: possible error in check_transformed_clause:\n");
	printf("%%    original:    "); p_clause(orig);
	printf("%%    transformed: "); p_clause(d);
	}
}  /* check_transformed_clause */

/*************
 *
 *   lone_variable_process()
 *
 *   If a pos eq has a var on the right that does not occur in
 *   another literal, and the left is not a variable,
 *   then replace Eq_sn with Dp_sn.  This
 *   simplifies the result, and I think it is complete.
 *
 *************/

static void lone_variable_process(c)
struct clause *c;
{
    struct literal *l1, *l2;
    struct term *v;
    int ok;

    for (l1 = c->first_lit; l1; l1 = l1->next_lit) {
	if (l1->sign && l1->atom->sym_num == Eq_sn &&
	    l1->atom->farg->narg->argval->type == VARIABLE &&
	    l1->atom->farg->argval->type != VARIABLE) {
	    v = l1->atom->farg->narg->argval;
	    ok = 1;
	    for (l2 = c->first_lit, ok = 1; l2 && ok; l2 = l2->next_lit)
		ok = (l1 == l2 || !occurs_in(v, l2->atom));
	    if (ok)
		l1->atom->sym_num = Dp_sn;
	    }
	}
}  /* lone_variable_process */

/*************
 *
 *   domain_element_process()
 *
 *   If Flag[DP_INT_DOMAIN].val, then simplify literals "-<int> <var>"
 *
 *************/

static void domain_element_process(c)
struct clause *c;
{
    struct literal *curr, *prev, *l;
    struct term *t1, *t2, *v, *t;

    curr = c->first_lit;
    prev = NULL;
    while (curr) {
	if (curr->atom->sym_num == Dp_sn && !curr->sign) {
	    t1 = curr->atom->farg->argval;
	    t2 = curr->atom->farg->narg->argval;
	    if (domain_element(t1) && t2->type == VARIABLE) {
		if (prev)
		    prev->next_lit = curr->next_lit;
		else
		    c->first_lit = curr->next_lit;
		for (l = c->first_lit; l; l = l->next_lit)
		    l->atom = replace_term(l->atom, t2, t1);
		}
	    else
		prev = curr;
	    }
	else
	    prev = curr; 

	curr = curr->next_lit;
	}
}  /* domain_element_process */

/*************
 *
 *   dp_transform()
 *
 *************/

void dp_transform()
{
    struct clause *c, *d, *e;
    struct term *t, *a, *t1, *t2;
    struct rel *r;
    struct literal *lit, *l2;
    struct list *l;
    int i, j, k;
    char *s;
    int vnum = MAX_VARS;

    Parms[STATS_LEVEL].val = 0;

    append_lists(Usable,Sos);
    append_lists(Usable,Demodulators);

    Dp_sn = str_to_sn("DP_EQUALITY", 2);
    if (Flags[TPTP_EQ].val)
      Eq_sn = str_to_sn("equal", 2);
    else
      Eq_sn = str_to_sn("=", 2);
    Lt_sn = str_to_sn("<", 2);

    printf("%% =======START OF DP INPUT=======.\n\n");

    for (c = Usable->first_cl; c; c = c->next_cl) {
	for (lit = c->first_lit; lit; lit = lit->next_lit) {
	    collect_symbols(lit->atom, RELATION);
	    }
	}

    for (c = Passive->first_cl; c; c = c->next_cl) {
	a = c->first_lit->atom;
	if (sn_to_arity(a->sym_num) != 2)
	    abend("dp_transform, wrong arity in passive list.");

	for (i = 0; i < Num_syms; i++) {
	    if (str_ident(Syms[i].name, sn_to_str(a->farg->argval->sym_num)))
		break;
	    }

	if (i == Num_syms)
	    abend("dp_transform, symbol not found");

	if (is_symbol(a, "properties", 2))
	    Syms[i].properties = sn_to_str(a->farg->narg->argval->sym_num);
	else if (!is_symbol(a, "assign", 2))  /* handle assigns below */
	    abend("dp_transform, passive command not understood");
	}

    for (i = 0; i < Num_syms; i++) {
	if (Syms[i].type == RELATION)
	    { s = "relation"; j = Syms[i].arity; }
	else
	    { s = "function"; j = Syms[i].arity + 1; }
	printf("%s %s %d %s\n", s, Syms[i].name, j, Syms[i].properties);
	}

    printf("end_of_symbols\n\n");
	       
    for (c = Usable->first_cl; c; c = c->next_cl) {
	printf("\n%% "); p_clause(c);

	check_for_bad_things(c);

	d = cl_copy(c);

	lone_variable_process(d);

	flatten_clause(d);

	process_negative_equalities(d);

	if (Flags[DP_INT_DOMAIN].val)
	    domain_element_process(d);

	if (renumber_vars(d) == 0)
	    abend("dp_transform, too many variables");

	l = clause_to_clauses(cl_copy(d));

	for (e = l->first_cl; e; e = e->next_cl) {
	    renumber_vars(e);
	    printf("%d relational %d      ", c->id, num_literals(e)); dp_p_clause(e);
	    check_transformed_clause(e, c);
	    }
	printf("%d ordinary   %d      ", c->id, num_literals(c)); dp_p_clause(c);

	}
    printf("end_of_clauses\n\n");

    for (c = Passive->first_cl; c; c = c->next_cl) {
	a = c->first_lit->atom;

	if (is_symbol(a, "assign", 2)) {

	    for (i = 0; i < Num_syms; i++) {
		if (str_ident(Syms[i].name, sn_to_str(a->farg->argval->sym_num)))
		    break;
		}

	    if (Syms[i].type == FUNCTION) {
		if (!str_int(sn_to_str(a->farg->narg->argval->sym_num), &j))
		    abend("dp_transform, bad integer in passive assignment");
		}
	    else {
		s = sn_to_str(a->farg->narg->argval->sym_num);
		if (!str_ident(s, "T") && !str_ident(s, "F"))
		    abend("dp_transform, bad truth value in passive assignment");
		}
	    printf("%s%s",
		   (Syms[i].type == RELATION && str_ident(s, "F") ? "-" : ""),
		   Syms[i].name);
	    for (r = a->farg->argval->farg; r; r = r->narg) {
		if (!str_int(sn_to_str(r->argval->sym_num), &k))
		    abend("dp_transform, bad integer in passive assignment");
		printf(" %d", k);
		}
	    if (Syms[i].type == FUNCTION)
		printf(" %d\n", j);
	    else
		printf("\n");
	    }
	}

    printf("end_of_assignments\n");
    printf("\n%% =======END OF DP INPUT=======.\n");

    exit(0);
}  /* dp_transform */

/**************************************************************
 *
 *   ICGNS
 *
 **************************************************************/

/*************
 *
 *    print_clause_bare(fp, clause)
 *
 *************/

void print_clause_bare(fp, cl)
     FILE *fp;
     struct clause *cl;
{
    struct term *t;

    t = clause_to_term(cl);
    t = term_fixup_2(t);  /* Change -(=(a,b)) to !=(a,b). */
    print_term(fp, t);
    fprintf(fp, ".\n");
    zap_term(t);
}  /* print_clause_bare */

/*************
 *
 *   instances_recurse()
 *
 *************/

static void instances_recurse(struct clause *c,
		       struct context *s,
		       int n,
		       int nvars,
		       int nextvar,
		       struct term *terms[])
{
  if (nextvar == nvars) {
    print_clause_bare(stdout, apply_clause(cl_copy(c), s));
  }
  else {
    int i;
    for (i = 0; i < n; i++) {
      s->terms[nextvar] = terms[i];
      instances_recurse(c, s, n, nvars, nextvar+1, terms);
    }
  }
}  /* instances_recurse */

/*************
 *
 *   sym_lessthan()
 *
 *************/

int sym_lessthan(n1, n2)
{
  int a1 = sn_to_arity(n1);
  int a2 = sn_to_arity(n2);
  if (a1 < a2)
    return 1;
  else if (a2 < a1)
    return 0;
  else
    return (n1 < n2);
}  /* sym_lessthan */

/*************
 *
 *   sym_insert()
 *
 *************/

struct int_ptr *sym_insert(int sn, struct int_ptr *p)
{
  if (!p || sym_lessthan(sn, p->i)) {
    struct int_ptr *q = get_int_ptr();
    q->i = sn;
    q->next = p;
    return q;
  }
  else if (p->i == sn)
    return p;
  else {
    p->next = sym_insert(sn, p->next);
    return p;
  }
}  /* sym_insert */

/*************
 *
 *   ilist_member()
 *
 *************/

int ilist_member(int i, struct int_ptr *p)
{
  if (!p)
    return 0;
  else if (i == p->i)
    return 1;
  else
    return ilist_member(i, p->next);
}  /* ilist_member */

/*************
 *
 *   zap_ilist()
 *
 *************/

int zap_ilist(struct int_ptr *p)
{
  if (p) {
    zap_ilist(p->next);
    free_int_ptr(p);
  }
}  /* zap_ilist */

/*************
 *
 *   fsyms_in_term()
 *
 *************/

struct int_ptr *fsyms_in_term(struct term *t, struct int_ptr *p)
{
  if (t->type == VARIABLE)
    return p;
  else {
    struct rel *r;
    p = sym_insert(t->sym_num, p);
    for (r = t->farg; r; r = r->narg)
      p = fsyms_in_term(r->argval, p);
    return p;
  }
}  /* fsyms_in_term */

/*************
 *
 *   fsyms_in_clause()
 *
 *************/

struct int_ptr *fsyms_in_clause(struct clause *c, struct int_ptr *p)
{
  struct literal *lit;
  for (lit = c->first_lit; lit; lit = lit->next_lit) {
    struct rel *r;
    for (r = lit->atom->farg; r; r = r->narg)
      p = fsyms_in_term(r->argval, p);
  }
  return p;
}  /* fsyms_in_clause */

/*************
 *
 *   fsyms_in_clist()
 *
 *************/

struct int_ptr *fsyms_in_clist(struct list *lst, struct int_ptr *p)
{
  struct clause *c;
  for (c = lst->first_cl; c; c = c->next_cl)
    p = fsyms_in_clause(c, p);
  return p;
}  /* fsyms_in_clist */

/*************
 *
 *   rsyms_in_clause()
 *
 *************/

struct int_ptr *rsyms_in_clause(struct clause *c, struct int_ptr *p)
{
  struct literal *lit;
  for (lit = c->first_lit; lit; lit = lit->next_lit) {
    p = sym_insert(lit->atom->sym_num, p);
  }
  return p;
}  /* rsyms_in_clause */

/*************
 *
 *   rsyms_in_clist()
 *
 *************/

struct int_ptr *rsyms_in_clist(struct list *lst, struct int_ptr *p)
{
  struct clause *c;
  for (c = lst->first_cl; c; c = c->next_cl)
    p = rsyms_in_clause(c, p);
  return p;
}  /* rsyms_in_clist */

/*************
 *
 *   subst_axioms_for_rsym()
 *
 *************/

struct clause_ptr *subst_axioms_for_rsym(int symnum)
{
  int arity = sn_to_arity(symnum);
  struct clause *c = get_clause();
  struct literal *l1 = get_literal();
  struct literal *l2 = get_literal();
  struct literal *l3 = get_literal();
  struct term *a1 = get_term();
  struct term *a2 = get_term();
  struct term *a3 = get_term();
  struct rel *r1 = get_rel();
  struct rel *r2 = get_rel();
  struct term *t1 = get_term();
  struct term *t2 = get_term();

  struct rel *p1, *p2;
  int i, j;
  struct clause_ptr *cp1 = NULL;
  struct clause_ptr *cp2, *cp3;

  c->first_lit = l1; l1->next_lit = l2; l2->next_lit = l3;
  l1->sign = 1; l1->atom = a1;
  l2->sign = 0; l2->atom = a2;
  l3->sign = 0; l3->atom = a3;
  a3->farg = r1; r1->narg = r2; r1->argval = t1; r2->argval = t2;
  a3->sym_num = str_to_sn("=", 2);  a3->type = COMPLEX;
  t1->type = VARIABLE; t1->varnum = arity;
  t2->type = VARIABLE; t2->varnum = arity+1;

  a1->sym_num = symnum; a2->sym_num = symnum;
  if (arity == 0) {
    a1->type = NAME; a2->type = NAME;
  }
  else {
    a1->type = COMPLEX; a2->type = COMPLEX;
  }

  p1 = NULL; p2 = NULL;
  for (i = 0; i < arity; i++) {
    r1 = get_rel(); t1 = get_term(); r1->argval = t1;
    t1->type = VARIABLE;  t1->varnum = i;
    r2 = get_rel(); t2 = get_term(); r2->argval = t2;
    t2->type = VARIABLE;  t2->varnum = i;
    if (p1) {
      p1->narg = r1;
      p2->narg = r2;
    }
    else {
      a1->farg = r1;
      a2->farg = r2;
    }
    p1 = r1; p2 = r2;
  }

  for (i = 0; i < arity; i++) {
    struct clause *d = cl_copy(c);
    r1 = d->first_lit->atom->farg;
    r2 = d->first_lit->next_lit->atom->farg;
    for (j = 0; j < i; j++) {
      r1 = r1->narg;  r2 = r2->narg;
    }
    r1->argval->varnum = arity;
    r2->argval->varnum = arity+1;
    cp3 = get_clause_ptr();
    cp3->c = d;
    if (cp1)
      cp2->next = cp3;
    else
      cp1 = cp3;
    cp2 = cp3;
  }
  cl_del_non(c);
  return(cp1);
}  /* subst_axioms_for_rsym */

/*************
 *
 *   subst_axioms_for_fsym()
 *
 *************/

struct clause_ptr *subst_axioms_for_fsym(int symnum)
{
  int arity = sn_to_arity(symnum);
  struct clause *c = get_clause();
  struct literal *l1 = get_literal();
  struct literal *l2 = get_literal();
  struct literal *l3 = get_literal();
  struct term *a1 = get_term();
  struct term *a2 = get_term();
  struct term *a3 = get_term();
  struct rel *r1 = get_rel();
  struct rel *r2 = get_rel();
  struct rel *r3 = get_rel();
  struct rel *r4 = get_rel();
  struct rel *r5 = get_rel();
  struct rel *r6 = get_rel();
  struct term *t1 = get_term();
  struct term *t2 = get_term();
  struct term *t3 = get_term();
  struct term *t4 = get_term();
  struct term *t5 = get_term();
  struct term *t6 = get_term();

  struct rel *p1, *p2;
  int i, j;
  struct clause_ptr *cp1 = NULL;
  struct clause_ptr *cp2, *cp3;

  c->first_lit = l1; l1->next_lit = l2; l2->next_lit = l3;
  l1->sign = 1; l1->atom = a1;
  l2->sign = 0; l2->atom = a2;
  l3->sign = 0; l3->atom = a3;
  a3->farg = r1; r1->narg = r2; r1->argval = t1; r2->argval = t2;
  a3->sym_num = str_to_sn("=", 2);  a3->type = COMPLEX;
  t1->type = VARIABLE; t1->varnum = arity+1;
  t2->type = VARIABLE; t2->varnum = arity+2;

  a1->sym_num = str_to_sn("=", 2);  a1->type = COMPLEX;
  a2->sym_num = str_to_sn("=", 2);  a2->type = COMPLEX;
  a1->farg = r3; r3->narg = r4; r3->argval = t3; r4->argval = t4;
  a2->farg = r5; r5->narg = r6; r5->argval = t5; r6->argval = t6;
  t4->type = VARIABLE; t4->varnum = arity;
  t6->type = VARIABLE; t6->varnum = arity;

  t3->sym_num = symnum; t5->sym_num = symnum;
  if (arity == 0) {
    t3->type = NAME; t5->type = NAME;
  }
  else {
    t3->type = COMPLEX; t5->type = COMPLEX;
  }

  p1 = NULL; p2 = NULL;
  for (i = 0; i < arity; i++) {
    r1 = get_rel(); t1 = get_term(); r1->argval = t1;
    t1->type = VARIABLE;  t1->varnum = i;
    r2 = get_rel(); t2 = get_term(); r2->argval = t2;
    t2->type = VARIABLE;  t2->varnum = i;
    if (p1) {
      p1->narg = r1;
      p2->narg = r2;
    }
    else {
      t3->farg = r1;
      t5->farg = r2;
    }
    p1 = r1; p2 = r2;
  }

  for (i = 0; i < arity; i++) {
    struct clause *d = cl_copy(c);
    r1 = d->first_lit->atom->farg->argval->farg;
    r2 = d->first_lit->next_lit->atom->farg->argval->farg;
    for (j = 0; j < i; j++) {
      r1 = r1->narg;  r2 = r2->narg;
    }
    r1->argval->varnum = arity+1;
    r2->argval->varnum = arity+2;
    cp3 = get_clause_ptr();
    cp3->c = d;
    if (cp1)
      cp2->next = cp3;
    else
      cp1 = cp3;
    cp2 = cp3;
  }
  cl_del_non(c);
  return(cp1);
}  /* subst_axioms_for_fsym */

/*************
 *
 *   icgns_transform()
 *
 *************/

#define MAX_DOMAIN 100

void icgns_transform()
{
  struct clause *c;
  int i, j, n;
  struct term *terms[MAX_DOMAIN];
  struct context *s = get_context();
  struct int_ptr *fsyms, *rsyms, *p;
  struct clause_ptr *cp;

  n = Parms[ICGNS].val;

  printf("%% =======START OF ICGNS INPUT=======.\n\n");

  /* First, print the equality substitution axioms for relation
   * symbols and function symbols.  The simple negative equality
   * must be last (for restricted UR resolution).
   */

  printf("list(usable).  %% Substitution axioms for negative propagation.\n\n");

  rsyms = rsyms_in_clist(Usable, NULL);

  for (p = rsyms; p; p = p->next) {
    int symnum = p->i;
    if (!is_eq(symnum)) {
      struct clause_ptr *cp = subst_axioms_for_rsym(symnum);
      struct clause_ptr *cp2;
      for (cp2 = cp; cp2; cp2 = cp2->next)
	print_clause_bare(stdout, cp2->c);
    }
  }

  fsyms = fsyms_in_clist(Usable, NULL);

  for (p = fsyms; p; p = p->next) {
    int symnum = p->i;
    if (!is_eq(symnum)) {
      struct clause_ptr *cp = subst_axioms_for_fsym(symnum);
      struct clause_ptr *cp2;
      for (cp2 = cp; cp2; cp2 = cp2->next)
	print_clause_bare(stdout, cp2->c);
    }
  }
  printf("\nend_of_list.\n\n");
  
  /* Print reflexivity and the unique names axioms.
   */

  printf("list(usable).\n\n");
  printf("x=x.\n\n");
  for (i = 0; i < n; i++) {
    for (j = 0; j < i; j++)
      printf("      ");
    for (j = i+1; j < n; j++)
      printf("%d!=%d. ", i, j);
    printf("\n");
  }
  printf("end_of_list.\n\n");

  /* Print the instances of all the clauses in Usable, Sos, and Demodulators. */

  append_lists(Usable,Sos);
  append_lists(Usable,Demodulators);

  for (i = 0; i < MAX_DOMAIN; i++) {
    char name[10];
    sprintf(name, "%d", i);
    terms[i] = get_term();
    terms[i]->type = NAME;
    terms[i]->sym_num = str_to_sn(name, 0);
  }

  printf("list(usable).  %% instances of axioms\n\n");

  for (c = Usable->first_cl; c; c = c->next_cl) {
    int nvars = biggest_var_clause(c) + 1;
    instances_recurse(c, s, n, nvars, 0, terms);
    printf("\n");
  }
  printf("end_of_list.\n\n");

  /* Print the range clauses for constant and function symbols. */

  fsyms = fsyms_in_clist(Usable, NULL);

  printf("list(sos).  %% range clauses for constants and function symbols\n\n");

  for (p = fsyms; p; p = p->next) {
    struct clause *c = get_clause();
    struct literal *prevlit = NULL;
    int symnum = p->i;
    int arity = sn_to_arity(symnum);
    int i, j;
    for (i = 0; i < n; i++) {
      struct rel *prevr;
      struct literal *lit = get_literal();
      struct term *atom = get_term();
      struct rel *r1 = get_rel();
      struct rel *r2 = get_rel();
      struct term *t = get_term();
      if (prevlit)
	prevlit->next_lit = lit;
      else
	c->first_lit = lit;
      prevlit = lit;
      lit->sign = 1;
      lit->atom = atom;
      atom->type = COMPLEX;
      atom->sym_num = str_to_sn("=", 2);
      atom->farg = r1;
      r1->narg = r2;
      r2->argval = copy_term(terms[i]);
      r1->argval = t;
      t->sym_num = symnum;
      t->type = (arity == 0 ? NAME : COMPLEX);
      prevr = NULL;
      for (j = 0; j < arity; j++) {
	struct rel *r3 = get_rel();
	struct term *v = get_term();
	v->type = VARIABLE;
	v->varnum = j;
	if (prevr)
	  prevr->narg = r3;
	else
	  t->farg = r3;
	prevr = r3;
	r3->argval = v;
      }
    }
    instances_recurse(c, s, n, arity, 0, terms);
    printf("\n");
    cl_del_non(c);
  }
  printf("end_of_list.\n\n");

  printf("\n%% =======END OF ICGNS INPUT=======.\n");

  exit(0);
}  /* icgns_transform */
