/*
 * Copyright (c) 2022-2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>
#include "arm_gemm.hpp"

using arm_gemm::Requantize32;

namespace arm_conv {
namespace depthwise {

void sme2_s8q_planar_5x5_s2_4rows_dot_za_impl(
  const int8_t *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const int8_t *weights,
  int8_t **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  const arm_gemm::Requantize32 &qp
)
{
  struct Args
  {
    const int8_t *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const int8_t *weights;
    long unsigned int input_cols, output_cols;
    int8_t **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
  };

  Args args = { inptr, ld_in_vl, pad_top, 11u - std::min(11u, pad_top + valid_input_rows), pad_left, weights, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels };

  __asm__ __volatile__(
    ".inst 0xd503477f  // SMSTART ZA\n"
    "ldr x3, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "ptrue p2.b\n"
    "mov x20, #0xb\n"
    "ldr x4, [%x[args], %[offsetof_Args_pad_top]]\n"
    "ld1rh { z7.h }, p2/Z, [%x[qp], %[offsetof_Requantize32_a_offset]]\n"
    "sub x20, x20, x3\n"
    ".inst 0x25207812  // ptrue pn10.b\n"
    "ldr x5, [%x[args], %[offsetof_Args_n_channels]]\n"
    "whilelt p1.s, XZR, x5\n"
    "whilelt p9.s, XZR, x20\n"
    "ld1rw { z10.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_c_offset]]\n"
    "whilelt p8.s, XZR, x4\n"
    "addvl SP, SP, #-15\n"
    "ldr x6, [%x[args], %[offsetof_Args_current_channel]]\n"
    "neg z7.h, p2/M, z7.h\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "ld1rw { z6.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_per_layer_mul]]\n"
    "ld1rw { z4.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_per_layer_right_shift]]\n"
    "ld1rw { z5.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_minval]]\n"
    "ld1rw { z21.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_maxval]]\n"
    "1:"  // Channel loop
    "ldr x20, [%x[qp], %[offsetof_Requantize32_bias]]\n"
    "mov z12.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z12.s }, p1/Z, [x20, x6, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x22, [%x[args], %[offsetof_Args_weights]]\n"
    "mov x20, x22\n"
    "ld1sb { z13.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "ld1rh { z28.h }, p2/Z, [%x[qp], %[offsetof_Requantize32_b_offset]]\n"
    "sub z13.h, z13.h, z28.h\n"
    "incw x22\n"
    "mov z26.h, #0x0\n"
    "ld1sb { z22.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z22.h, z22.h, z28.h\n"
    "trn1 z17.h, z13.h, z22.h\n"
    "ld1sb { z20.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z20.h, z20.h, z28.h\n"
    "addvl x21, SP, #15\n"
    "ld1sb { z1.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z1.h, z1.h, z28.h\n"
    "trn1 z29.h, z20.h, z1.h\n"
    "ld1sb { z27.s }, p2/Z, [x20]\n"
    "mov x20, x22\n"
    "sub z27.h, z27.h, z28.h\n"
    "incw x22\n"
    "ld1sb { z14.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z14.h, z14.h, z28.h\n"
    "addvl x21, x21, #-3\n"
    "ld1sb { z18.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z18.h, z18.h, z28.h\n"
    "trn1 z22.h, z27.h, z26.h\n"
    "ld1sb { z23.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z23.h, z23.h, z28.h\n"
    "st1h { z17.h }, p2, [x21]\n"
    "ld1sb { z30.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z30.h, z30.h, z28.h\n"
    "trn1 z8.h, z14.h, z18.h\n"
    "ld1sb { z15.s }, p2/Z, [x20]\n"
    "mov x20, x22\n"
    "st1h { z29.h }, p2, [x21, #1, MUL VL]\n"
    "sub z15.h, z15.h, z28.h\n"
    "ld1sb { z20.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z23.h, z23.h, z30.h\n"
    "sub z20.h, z20.h, z28.h\n"
    "ld1sb { z24.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z24.h, z24.h, z28.h\n"
    "st1h { z22.h }, p2, [x21, #2, MUL VL]\n"
    "ld1sb { z16.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z0.h, z15.h, z26.h\n"
    "incw x22\n"
    "ld1sb { z13.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z16.h, z16.h, z28.h\n"
    "sub z13.h, z13.h, z28.h\n"
    "ld1sb { z11.s }, p2/Z, [x20]\n"
    "addvl x21, x21, #-3\n"
    "mov x20, x22\n"
    "st1h { z8.h }, p2, [x21]\n"
    "trn1 z27.h, z20.h, z24.h\n"
    "ld1sb { z22.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z11.h, z11.h, z28.h\n"
    "ld1sb { z3.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z23.h }, p2, [x21, #1, MUL VL]\n"
    "trn1 z20.h, z16.h, z13.h\n"
    "ld1sb { z13.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z22.h, z22.h, z28.h\n"
    "sub z3.h, z3.h, z28.h\n"
    "ld1sb { z15.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z0.h }, p2, [x21, #2, MUL VL]\n"
    "trn1 z29.h, z11.h, z26.h\n"
    "ld1sb { z16.s }, p2/Z, [x20]\n"
    "incw x22\n"
    "sub z13.h, z13.h, z28.h\n"
    "sub z15.h, z15.h, z28.h\n"
    "addvl x21, x21, #-3\n"
    "mov x20, x22\n"
    "st1h { z27.h }, p2, [x21]\n"
    "sub z16.h, z16.h, z28.h\n"
    "trn1 z19.h, z22.h, z3.h\n"
    "ld1sb { z17.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z20.h }, p2, [x21, #1, MUL VL]\n"
    "ld1sb { z0.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z31.h, z13.h, z15.h\n"
    "st1h { z29.h }, p2, [x21, #2, MUL VL]\n"
    "ld1sb { z18.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z16.h, z16.h, z26.h\n"
    "sub z17.h, z17.h, z28.h\n"
    "ld1sb { z22.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z0.h, z0.h, z28.h\n"
    "sub z18.h, z18.h, z28.h\n"
    "ld1sb { z1.s }, p2/Z, [x20]\n"
    "sub z22.h, z22.h, z28.h\n"
    "sub z1.h, z1.h, z28.h\n"
    "ldr x20, [%x[qp], %[offsetof_Requantize32_per_channel_muls]]\n"
    "addvl x21, x21, #-3\n"
    "st1h { z19.h }, p2, [x21]\n"
    "mov z13.d, z12.d\n"
    "mov z14.d, z12.d\n"
    "st1h { z31.h }, p2, [x21, #1, MUL VL]\n"
    "mov z15.d, z12.d\n"
    "trn1 z8.h, z17.h, z0.h\n"
    "st1h { z16.h }, p2, [x21, #2, MUL VL]\n"
    "addvl x21, x21, #-3\n"
    "trn1 z31.h, z18.h, z22.h\n"
    "trn1 z29.h, z1.h, z26.h\n"
    "st1h { z8.h }, p2, [x21]\n"
    "st1h { z31.h }, p2, [x21, #1, MUL VL]\n"
    "st1h { z29.h }, p2, [x21, #2, MUL VL]\n"
    "cbz x20, 3f\n"
    "ld1w { z6.s }, p1/Z, [x20, x6, LSL #2]\n"
    "3:"  // Load mul: End
    "ldr x20, [%x[qp], %[offsetof_Requantize32_per_channel_right_shifts]]\n"
    "cbz x20, 4f\n"
    "ld1w { z4.s }, p1/Z, [x20, x6, LSL #2]\n"
    "4:"  // Load right_shift: End
    "ldr x7, [%x[args], %[offsetof_Args_input_cols]]\n"
    "sub x20, x7, #0x1\n"
    "orr x23, x20, %x[ld_in_col], LSL #16\n"
    "ldr x17, [%x[args], %[offsetof_Args_inptr]]\n"
    "orr x23, x5, x23, LSL #22\n"
    "mov x22, #0xb\n"
    "add x21, x4, x3\n"
    "lsl x20, %x[ld_in_row], #0x0\n"
    "ldr x16, [%x[args], %[offsetof_Args_output_cols]]\n"
    "mov x8, #0x0\n"
    "lsl x23, x23, #0x0\n"
    "sub x22, x22, x21\n"
    "madd x20, x20, x4, x17\n"
    "5:"  // Issue prefetches
    "subs x22, x22, #0x1\n"
    ".inst 0xf8b74a9c  // rprfm pldstrm, x23, [x20]\n"
    "add x20, x20, %x[ld_in_col]\n"
    "bgt 5b\n"
    "ldr x23, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x20, %x[ld_in_row], #0x0\n"
    "msub x17, x4, x20, x17\n"
    ".inst 0xc0040d80  // mova za.d[x8, #0], { z12.d-z15.d }\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    ".inst 0xc0040d81  // mova za.d[x8, #1], { z12.d-z15.d }\n"
    "mov x22, #0x4\n"
    "ldp x15, x14, [x23], #0x10\n"
    ".inst 0xc0040d82  // mova za.d[x8, #2], { z12.d-z15.d }\n"
    "ldp x13, x11, [x20], #0x10\n"
    ".inst 0xc0040d83  // mova za.d[x8, #3], { z12.d-z15.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0040d84  // mova za.d[x8, #4], { z12.d-z15.d }\n"
    "ldp x10, x9, [x23], #0x10\n"
    "ldp x28, x27, [x20], #0x10\n"
    "cbz x21, 7f\n"
    "cmp x21, x22\n"
    "csel x20, x21, x22, LT\n"
    "sub x21, x21, x20\n"
    "sub x22, x22, x20\n"
    "cbz x21, 7f\n"
    ".inst 0xc0060c1c  // mova { z28.d-z31.d }, za.d[x8, #0]\n"
    ".inst 0xc1a6ac1c  // sqdmulh { z28.s-z31.s }, { z28.s-z31.s }, z6.s\n"
    "and x22, x21, #0x1\n"
    ".inst 0xc1a4aa3c  // srshl { z28.s-z31.s }, { z28.s-z31.s }, z4.s\n"
    "add x21, x21, #0x1\n"
    "lsr x21, x21, #0x1\n"
    ".inst 0xc1aaab1c  // add { z28.s-z31.s }, { z28.s-z31.s }, z10.s\n"
    "sub x16, x16, x21\n"
    ".inst 0xc1b5ccbc  // sclamp { z28.s-z31.s }, z5.s, z21.s\n"
    "6:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1b { z28.s }, p1, [x15]\n"
    "add x15, x15, x13\n"
    "st1b { z29.s }, p1, [x14]\n"
    "add x14, x14, x11\n"
    "st1b { z30.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "st1b { z31.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "bgt 6b\n"
    "7:"  // Left padding: End
    "adds XZR, x4, x3\n"
    "bne 14f\n"
    "cbz x22, 12f\n"
    "cmp x22, #0x1\n"
    "sub x7, x7, x22\n"
    "beq 11f\n"
    "cmp x22, #0x2\n"
    "beq 10f\n"
    "cmp x22, #0x3\n"
    "beq 9f\n"
    "8:"  // Unpadded: 4 priming loads
    "add x21, x17, %x[ld_in_row]\n"
    "ld1sb { z27.s }, p1/Z, [x17]\n"
    "addvl x20, SP, #12\n"
    "ld1sb { z0.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z27.h, z27.h, z0.h\n"
    "add z27.h, z27.h, z7.h\n"
    "ld1sb { z28.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add x17, x17, %x[ld_in_col]\n"
    "ld1sb { z11.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z28.h, z28.h, z11.h\n"
    "add z28.h, z28.h, z7.h\n"
    "ld1sb { z29.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1sb { z8.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z29.h, z29.h, z8.h\n"
    "add z29.h, z29.h, z7.h\n"
    "ld1sb { z30.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1sb { z17.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z30.h, z30.h, z17.h\n"
    "add z30.h, z30.h, z7.h\n"
    "ld1sb { z31.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1sb { z26.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z31.h, z31.h, z26.h\n"
    "add z31.h, z31.h, z7.h\n"
    ".inst 0xa1402a80  // ld1h { z0.h, z8.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc1701768  // sdot za.s[x8, 0], { z27.h-z30.h }, z0.h\n"
    "ld1sb { z20.s }, p1/Z, [x21]\n"
    "mov z0.d, z20.d\n"
    "add z0.h, z0.h, z7.h\n"
    ".inst 0xc1781788  // sdot za.s[x8, 0], { z28.h-z31.h }, z8.h\n"
    "ld1h { z8.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc17817a8  // sdot za.s[x8, 0], { z29.h-z0.h }, z8.h\n"
    "9:"  // Unpadded: 3 priming loads
    "add x21, x17, %x[ld_in_row]\n"
    "ld1sb { z29.s }, p1/Z, [x17]\n"
    "addvl x20, SP, #9\n"
    "ld1sb { z17.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z29.h, z29.h, z17.h\n"
    "add z29.h, z29.h, z7.h\n"
    "ld1sb { z30.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add x17, x17, %x[ld_in_col]\n"
    "ld1sb { z0.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z30.h, z30.h, z0.h\n"
    "add z30.h, z30.h, z7.h\n"
    "ld1sb { z31.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1sb { z16.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z31.h, z31.h, z16.h\n"
    "add z31.h, z31.h, z7.h\n"
    "ld1sb { z0.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1sb { z16.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z0.h, z0.h, z16.h\n"
    "add z0.h, z0.h, z7.h\n"
    "ld1sb { z1.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1sb { z16.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z1.h, z1.h, z16.h\n"
    "add z1.h, z1.h, z7.h\n"
    ".inst 0xa0402a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc17217a8  // sdot za.s[x8, 0], { z29.h-z0.h }, z2.h\n"
    "ld1sb { z16.s }, p1/Z, [x21]\n"
    "mov z2.d, z16.d\n"
    "add z2.h, z2.h, z7.h\n"
    ".inst 0xc17317c8  // sdot za.s[x8, 0], { z30.h-z1.h }, z3.h\n"
    "ld1h { z8.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc17817e8  // sdot za.s[x8, 0], { z31.h-z2.h }, z8.h\n"
    "10:"  // Unpadded: 2 priming loads
    "add x22, x17, %x[ld_in_row]\n"
    "ld1sb { z26.s }, p1/Z, [x17]\n"
    "addvl x21, SP, #6\n"
    "ld1sb { z16.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z26.h, z26.h, z16.h\n"
    "add z26.h, z26.h, z7.h\n"
    "ld1sb { z27.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "addvl x20, SP, #12\n"
    "ld1sb { z16.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z27.h, z27.h, z16.h\n"
    "add z27.h, z27.h, z7.h\n"
    "ld1sb { z28.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add x17, x17, %x[ld_in_col]\n"
    "ld1sb { z29.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z28.h, z28.h, z29.h\n"
    "add z28.h, z28.h, z7.h\n"
    "ld1sb { z29.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "ld1sb { z19.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z29.h, z29.h, z19.h\n"
    "add z29.h, z29.h, z7.h\n"
    "ld1sb { z30.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "ld1sb { z23.s }, p1/Z, [x22]\n"
    "trn1 z30.h, z30.h, z23.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add z30.h, z30.h, z7.h\n"
    ".inst 0xa0402aa2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc1721748  // sdot za.s[x8, 0], { z26.h-z29.h }, z2.h\n"
    "ld1sb { z22.s }, p1/Z, [x22]\n"
    "mov z31.d, z22.d\n"
    ".inst 0xc1731768  // sdot za.s[x8, 0], { z27.h-z30.h }, z3.h\n"
    ".inst 0xa1402a83  // ld1h { z3.h, z11.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc1731749  // sdot za.s[x8, 1], { z26.h-z29.h }, z3.h\n"
    "add z31.h, z31.h, z7.h\n"
    "ld1h { z3.h }, p2/Z, [x21, #2, MUL VL]\n"
    ".inst 0xc17b1769  // sdot za.s[x8, 1], { z27.h-z30.h }, z11.h\n"
    ".inst 0xc1731788  // sdot za.s[x8, 0], { z28.h-z31.h }, z3.h\n"
    "ld1h { z0.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc1701789  // sdot za.s[x8, 1], { z28.h-z31.h }, z0.h\n"
    "11:"  // Unpadded: 1 priming loads
    "add x22, x17, %x[ld_in_row]\n"
    "ld1sb { z29.s }, p1/Z, [x17]\n"
    "addvl x21, SP, #3\n"
    "ld1sb { z22.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z29.h, z29.h, z22.h\n"
    "add z29.h, z29.h, z7.h\n"
    "ld1sb { z30.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "addvl x20, SP, #9\n"
    "ld1sb { z25.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z30.h, z30.h, z25.h\n"
    "add z30.h, z30.h, z7.h\n"
    "ld1sb { z31.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add x17, x17, %x[ld_in_col]\n"
    "ld1sb { z16.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z31.h, z31.h, z16.h\n"
    "add z31.h, z31.h, z7.h\n"
    "ld1sb { z0.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "ld1sb { z16.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z0.h, z0.h, z16.h\n"
    "add z0.h, z0.h, z7.h\n"
    "ld1sb { z1.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "ld1sb { z2.s }, p1/Z, [x22]\n"
    "trn1 z1.h, z1.h, z2.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add z1.h, z1.h, z7.h\n"
    ".inst 0xa0402aa2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc17217a8  // sdot za.s[x8, 0], { z29.h-z0.h }, z2.h\n"
    "ld1sb { z24.s }, p1/Z, [x22]\n"
    "mov z2.d, z24.d\n"
    ".inst 0xc17317c8  // sdot za.s[x8, 0], { z30.h-z1.h }, z3.h\n"
    ".inst 0xa0402a88  // ld1h { z8.h-z9.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc17817a9  // sdot za.s[x8, 1], { z29.h-z0.h }, z8.h\n"
    "add z2.h, z2.h, z7.h\n"
    "ld1h { z3.h }, p2/Z, [x21, #2, MUL VL]\n"
    ".inst 0xc17917c9  // sdot za.s[x8, 1], { z30.h-z1.h }, z9.h\n"
    ".inst 0xc17317e8  // sdot za.s[x8, 0], { z31.h-z2.h }, z3.h\n"
    "ld1h { z3.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc17317e9  // sdot za.s[x8, 1], { z31.h-z2.h }, z3.h\n"
    "12:"  // Unpadded: 0 priming loads
    "cmp x7, #0x2\n"
    ".inst 0xa1402be3  // ld1h { z3.h, z11.h }, pn10.b/Z, [SP]\n"
    "ld1h { z2.h }, p2/Z, [SP, #2, MUL VL]\n"
    "blt 22f\n"
    "add x21, x17, %x[ld_in_row]\n"
    "ld1sb { z23.s }, p1/Z, [x17]\n"
    "sub x7, x7, #0x2\n"
    "ld1sb { z25.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z23.h, z23.h, z25.h\n"
    "sub x16, x16, #0x1\n"
    "ld1sb { z24.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "lsr x20, x7, #0x1\n"
    "add z23.h, z23.h, z7.h\n"
    "ld1sb { z30.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z24.h, z24.h, z30.h\n"
    "cmp x20, x16\n"
    "ld1sb { z25.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "csel x26, x20, x16, LT\n"
    "add z24.h, z24.h, z7.h\n"
    "ld1sb { z22.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z25.h, z25.h, z22.h\n"
    "add z25.h, z25.h, z7.h\n"
    "ld1sb { z26.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add x17, x17, %x[ld_in_col]\n"
    "ld1sb { z22.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z26.h, z26.h, z22.h\n"
    "add z26.h, z26.h, z7.h\n"
    "ld1sb { z27.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "and x7, x7, #0x1\n"
    "ld1sb { z30.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z27.h, z27.h, z30.h\n"
    "add z27.h, z27.h, z7.h\n"
    "ld1sb { z28.s }, p1/Z, [x21]\n"
    "mov z28.d, z28.d\n"
    "add z28.h, z28.h, z7.h\n"
    "sub x16, x16, x26\n"
    "cbz x26, 21f\n"
    "13:"  // Unpadded: Main loop
    ".inst 0xc17316e8  // sdot za.s[x8, 0], { z23.h-z26.h }, z3.h\n"
    "addvl x25, SP, #6\n"
    "addvl x24, SP, #12\n"
    ".inst 0xc17b1708  // sdot za.s[x8, 0], { z24.h-z27.h }, z11.h\n"
    ".inst 0xa0402b20  // ld1h { z0.h-z1.h }, pn10.b/Z, [x25]\n"
    "add x23, x17, %x[ld_in_row]\n"
    "addvl x22, SP, #3\n"
    ".inst 0xc17016e9  // sdot za.s[x8, 1], { z23.h-z26.h }, z0.h\n"
    "addvl x21, SP, #9\n"
    "subs x26, x26, #0x1\n"
    ".inst 0xc1711709  // sdot za.s[x8, 1], { z24.h-z27.h }, z1.h\n"
    ".inst 0xa0402b08  // ld1h { z8.h-z9.h }, pn10.b/Z, [x24]\n"
    ".inst 0xc17816ea  // sdot za.s[x8, 2], { z23.h-z26.h }, z8.h\n"
    "ld1sb { z23.s }, p1/Z, [x17]\n"
    "add x17, x17, %x[ld_in_col]\n"
    "add x20, x17, %x[ld_in_row]\n"
    ".inst 0xc1721728  // sdot za.s[x8, 0], { z25.h-z28.h }, z2.h\n"
    "ld1h { z0.h }, p2/Z, [x25, #2, MUL VL]\n"
    ".inst 0xc179170a  // sdot za.s[x8, 2], { z24.h-z27.h }, z9.h\n"
    "ld1sb { z16.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z23.h, z23.h, z16.h\n"
    ".inst 0xc1701729  // sdot za.s[x8, 1], { z25.h-z28.h }, z0.h\n"
    "ld1h { z9.h }, p2/Z, [x24, #2, MUL VL]\n"
    "add z23.h, z23.h, z7.h\n"
    "ld1sb { z24.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    ".inst 0xc179172a  // sdot za.s[x8, 2], { z25.h-z28.h }, z9.h\n"
    "ld1sb { z18.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z24.h, z24.h, z18.h\n"
    "add z24.h, z24.h, z7.h\n"
    "ld1sb { z25.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    ".inst 0xc0060c10  // mova { z16.d-z19.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    "ld1sb { z8.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z25.h, z25.h, z8.h\n"
    "add z25.h, z25.h, z7.h\n"
    "ld1sb { z26.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    ".inst 0xc1a6ac10  // sqdmulh { z16.s-z19.s }, { z16.s-z19.s }, z6.s\n"
    "ld1sb { z28.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z26.h, z26.h, z28.h\n"
    "add z26.h, z26.h, z7.h\n"
    "ld1sb { z27.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    ".inst 0xc1a4aa30  // srshl { z16.s-z19.s }, { z16.s-z19.s }, z4.s\n"
    "ld1sb { z28.s }, p1/Z, [x23]\n"
    "trn1 z27.h, z27.h, z28.h\n"
    "add x23, x23, %x[ld_in_row]\n"
    "add z27.h, z27.h, z7.h\n"
    ".inst 0xa0402ac2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x22]\n"
    ".inst 0xc17216e8  // sdot za.s[x8, 0], { z23.h-z26.h }, z2.h\n"
    ".inst 0xc1aaab10  // add { z16.s-z19.s }, { z16.s-z19.s }, z10.s\n"
    "ld1sb { z20.s }, p1/Z, [x23]\n"
    "mov z28.d, z20.d\n"
    ".inst 0xc1731708  // sdot za.s[x8, 0], { z24.h-z27.h }, z3.h\n"
    ".inst 0xa0402aa0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc17016e9  // sdot za.s[x8, 1], { z23.h-z26.h }, z0.h\n"
    "add z28.h, z28.h, z7.h\n"
    "ld1h { z0.h }, p2/Z, [x22, #2, MUL VL]\n"
    ".inst 0xc1711709  // sdot za.s[x8, 1], { z24.h-z27.h }, z1.h\n"
    ".inst 0xc1b5ccb0  // sclamp { z16.s-z19.s }, z5.s, z21.s\n"
    ".inst 0xc1701728  // sdot za.s[x8, 0], { z25.h-z28.h }, z0.h\n"
    "ld1h { z0.h }, p2/Z, [x21, #2, MUL VL]\n"
    "st1b { z16.s }, p1, [x15]\n"
    "add x15, x15, x13\n"
    "ld1sb { z23.s }, p1/Z, [x17]\n"
    ".inst 0xc1701729  // sdot za.s[x8, 1], { z25.h-z28.h }, z0.h\n"
    "st1b { z17.s }, p1, [x14]\n"
    "add x14, x14, x11\n"
    "ld1sb { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z23.h, z23.h, z16.h\n"
    "st1b { z18.s }, p1, [x10]\n"
    "ld1sb { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add x10, x10, x28\n"
    "st1b { z19.s }, p1, [x9]\n"
    "ld1sb { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z24.h, z24.h, z16.h\n"
    "add x9, x9, x27\n"
    "ld1sb { z25.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc0040d84  // mova za.d[x8, #4], { z12.d-z15.d }\n"
    "add z23.h, z23.h, z7.h\n"
    "ld1sb { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z25.h, z25.h, z16.h\n"
    "add z24.h, z24.h, z7.h\n"
    "ld1sb { z26.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z25.h, z25.h, z7.h\n"
    "add x17, x17, %x[ld_in_col]\n"
    "ld1sb { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z26.h, z26.h, z16.h\n"
    "add z26.h, z26.h, z7.h\n"
    "ld1sb { z27.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "ld1sb { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z27.h, z27.h, z16.h\n"
    "add z27.h, z27.h, z7.h\n"
    "ld1sb { z16.s }, p1/Z, [x20]\n"
    "mov z28.d, z16.d\n"
    "add z28.h, z28.h, z7.h\n"
    ".inst 0xa1402be3  // ld1h { z3.h, z11.h }, pn10.b/Z, [SP]\n"
    "ld1h { z2.h }, p2/Z, [SP, #2, MUL VL]\n"
    "bgt 13b\n"
    "b 21f\n"
    "14:"  // Padded
    "cbz x22, 19f\n"
    "cmp x22, #0x1\n"
    "sub x7, x7, x22\n"
    "beq 18f\n"
    "cmp x22, #0x2\n"
    "beq 17f\n"
    "cmp x22, #0x3\n"
    "beq 16f\n"
    "15:"  // Padded: 4 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z27.s }, p0/Z, [x17]\n"
    "add z27.h, p0/M, z27.h, z7.h\n"
    "add x21, x17, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z17.s }, p0/Z, [x21]\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z28.s }, p0/Z, [x21]\n"
    "add z28.h, p0/M, z28.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z16.s }, p0/Z, [x21]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "mov x12, #0x4\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z27.h, z27.h, z17.h\n"
    "trn1 z28.h, z28.h, z16.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z29.s }, p0/Z, [x21]\n"
    "add z29.h, p0/M, z29.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z18.s }, p0/Z, [x21]\n"
    "add z18.h, p0/M, z18.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z30.s }, p0/Z, [x21]\n"
    "add z30.h, p0/M, z30.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z17.s }, p0/Z, [x21]\n"
    "mov x12, #0x8\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z31.s }, p0/Z, [x21]\n"
    "add z31.h, p0/M, z31.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x21]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "addvl x20, SP, #12\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z29.h, z29.h, z18.h\n"
    "trn1 z30.h, z30.h, z17.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xa0402a80  // ld1h { z0.h-z1.h }, pn10.b/Z, [x20]\n"
    "trn1 z31.h, z31.h, z16.h\n"
    ".inst 0xc1701768  // sdot za.s[x8, 0], { z27.h-z30.h }, z0.h\n"
    "ld1sb { z20.s }, p0/Z, [x21]\n"
    "add z20.h, p0/M, z20.h, z7.h\n"
    "mov z0.d, z20.d\n"
    "add x17, x17, %x[ld_in_col]\n"
    ".inst 0xc1711788  // sdot za.s[x8, 0], { z28.h-z31.h }, z1.h\n"
    "ld1h { z1.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc17117a8  // sdot za.s[x8, 0], { z29.h-z0.h }, z1.h\n"
    "16:"  // Padded: 3 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z24.s }, p0/Z, [x17]\n"
    "add z24.h, p0/M, z24.h, z7.h\n"
    "add x21, x17, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z17.s }, p0/Z, [x21]\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z25.s }, p0/Z, [x21]\n"
    "add z25.h, p0/M, z25.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z16.s }, p0/Z, [x21]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "mov x12, #0x4\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z24.h, z24.h, z17.h\n"
    "trn1 z25.h, z25.h, z16.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z26.s }, p0/Z, [x21]\n"
    "add z26.h, p0/M, z26.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z18.s }, p0/Z, [x21]\n"
    "add z18.h, p0/M, z18.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z27.s }, p0/Z, [x21]\n"
    "add z27.h, p0/M, z27.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z17.s }, p0/Z, [x21]\n"
    "mov x12, #0x8\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z28.s }, p0/Z, [x21]\n"
    "add z28.h, p0/M, z28.h, z7.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x21]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "addvl x20, SP, #9\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z26.h, z26.h, z18.h\n"
    "trn1 z27.h, z27.h, z17.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xa0402a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20]\n"
    "trn1 z28.h, z28.h, z16.h\n"
    ".inst 0xc1721708  // sdot za.s[x8, 0], { z24.h-z27.h }, z2.h\n"
    "ld1sb { z11.s }, p0/Z, [x21]\n"
    "add z11.h, p0/M, z11.h, z7.h\n"
    "mov z29.d, z11.d\n"
    "add x17, x17, %x[ld_in_col]\n"
    ".inst 0xc1731728  // sdot za.s[x8, 0], { z25.h-z28.h }, z3.h\n"
    "ld1h { z0.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc1701748  // sdot za.s[x8, 0], { z26.h-z29.h }, z0.h\n"
    "17:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z25.s }, p0/Z, [x17]\n"
    "add z25.h, p0/M, z25.h, z7.h\n"
    "add x20, x17, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z26.s }, p0/Z, [x20]\n"
    "add z26.h, p0/M, z26.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z25.h, z25.h, z17.h\n"
    "trn1 z26.h, z26.h, z16.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z27.s }, p0/Z, [x20]\n"
    "add z27.h, p0/M, z27.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z18.s }, p0/Z, [x20]\n"
    "add z18.h, p0/M, z18.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z28.s }, p0/Z, [x20]\n"
    "add z28.h, p0/M, z28.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "mov x12, #0x8\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z29.s }, p0/Z, [x20]\n"
    "add z29.h, p0/M, z29.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "addvl x21, SP, #6\n"
    "trn1 z27.h, z27.h, z18.h\n"
    "trn1 z28.h, z28.h, z17.h\n"
    ".inst 0xa1402aa1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x21]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "trn1 z29.h, z29.h, z16.h\n"
    ".inst 0xc1711728  // sdot za.s[x8, 0], { z25.h-z28.h }, z1.h\n"
    "ld1sb { z1.s }, p0/Z, [x20]\n"
    "addvl x20, SP, #12\n"
    "add z1.h, p0/M, z1.h, z7.h\n"
    ".inst 0xc1791748  // sdot za.s[x8, 0], { z26.h-z29.h }, z9.h\n"
    ".inst 0xa0402a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc1721729  // sdot za.s[x8, 1], { z25.h-z28.h }, z2.h\n"
    "mov z30.d, z1.d\n"
    "add x17, x17, %x[ld_in_col]\n"
    "ld1h { z9.h }, p2/Z, [x21, #2, MUL VL]\n"
    ".inst 0xc1731749  // sdot za.s[x8, 1], { z26.h-z29.h }, z3.h\n"
    ".inst 0xc1791768  // sdot za.s[x8, 0], { z27.h-z30.h }, z9.h\n"
    "ld1h { z0.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc1701769  // sdot za.s[x8, 1], { z27.h-z30.h }, z0.h\n"
    "18:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z25.s }, p0/Z, [x17]\n"
    "add z25.h, p0/M, z25.h, z7.h\n"
    "add x20, x17, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z26.s }, p0/Z, [x20]\n"
    "add z26.h, p0/M, z26.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z25.h, z25.h, z17.h\n"
    "trn1 z26.h, z26.h, z16.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z27.s }, p0/Z, [x20]\n"
    "add z27.h, p0/M, z27.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z18.s }, p0/Z, [x20]\n"
    "add z18.h, p0/M, z18.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z28.s }, p0/Z, [x20]\n"
    "add z28.h, p0/M, z28.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "mov x12, #0x8\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z29.s }, p0/Z, [x20]\n"
    "add z29.h, p0/M, z29.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "addvl x21, SP, #3\n"
    "trn1 z27.h, z27.h, z18.h\n"
    "trn1 z28.h, z28.h, z17.h\n"
    ".inst 0xa1402aa3  // ld1h { z3.h, z11.h }, pn10.b/Z, [x21]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "trn1 z29.h, z29.h, z16.h\n"
    ".inst 0xc1731728  // sdot za.s[x8, 0], { z25.h-z28.h }, z3.h\n"
    "ld1sb { z0.s }, p0/Z, [x20]\n"
    "addvl x20, SP, #9\n"
    "add z0.h, p0/M, z0.h, z7.h\n"
    ".inst 0xc17b1748  // sdot za.s[x8, 0], { z26.h-z29.h }, z11.h\n"
    ".inst 0xa0402a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc1721729  // sdot za.s[x8, 1], { z25.h-z28.h }, z2.h\n"
    "mov z30.d, z0.d\n"
    "add x17, x17, %x[ld_in_col]\n"
    "ld1h { z0.h }, p2/Z, [x21, #2, MUL VL]\n"
    ".inst 0xc1731749  // sdot za.s[x8, 1], { z26.h-z29.h }, z3.h\n"
    ".inst 0xc1701768  // sdot za.s[x8, 0], { z27.h-z30.h }, z0.h\n"
    "ld1h { z0.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc1701769  // sdot za.s[x8, 1], { z27.h-z30.h }, z0.h\n"
    "19:"  // Padded: 0 priming loads
    "cmp x7, #0x2\n"
    ".inst 0xa1402be3  // ld1h { z3.h, z11.h }, pn10.b/Z, [SP]\n"
    "ld1h { z2.h }, p2/Z, [SP, #2, MUL VL]\n"
    "blt 22f\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z23.s }, p0/Z, [x17]\n"
    "add z23.h, p0/M, z23.h, z7.h\n"
    "add x20, x17, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z24.s }, p0/Z, [x20]\n"
    "add z24.h, p0/M, z24.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z23.h, z23.h, z17.h\n"
    "trn1 z24.h, z24.h, z16.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z25.s }, p0/Z, [x20]\n"
    "add z25.h, p0/M, z25.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z19.s }, p0/Z, [x20]\n"
    "add z19.h, p0/M, z19.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z26.s }, p0/Z, [x20]\n"
    "add z26.h, p0/M, z26.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z18.s }, p0/Z, [x20]\n"
    "mov x12, #0x8\n"
    "add z18.h, p0/M, z18.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z27.s }, p0/Z, [x20]\n"
    "add z27.h, p0/M, z27.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "sub x7, x7, #0x2\n"
    "sub x16, x16, #0x1\n"
    "trn1 z25.h, z25.h, z19.h\n"
    "trn1 z26.h, z26.h, z18.h\n"
    "lsr x20, x7, #0x1\n"
    "cmp x20, x16\n"
    "trn1 z27.h, z27.h, z17.h\n"
    "mov z28.d, z16.d\n"
    "csel x25, x20, x16, LT\n"
    "add x17, x17, %x[ld_in_col]\n"
    "and x7, x7, #0x1\n"
    "sub x16, x16, x25\n"
    "cbz x25, 21f\n"
    "20:"  // Padded: Main loop
    ".inst 0xc17316e8  // sdot za.s[x8, 0], { z23.h-z26.h }, z3.h\n"
    "addvl x24, SP, #6\n"
    "addvl x23, SP, #12\n"
    ".inst 0xc17b1708  // sdot za.s[x8, 0], { z24.h-z27.h }, z11.h\n"
    ".inst 0xa1402b00  // ld1h { z0.h, z8.h }, pn10.b/Z, [x24]\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc17016e9  // sdot za.s[x8, 1], { z23.h-z26.h }, z0.h\n"
    "add x20, x17, %x[ld_in_row]\n"
    "addvl x22, SP, #3\n"
    ".inst 0xc1781709  // sdot za.s[x8, 1], { z24.h-z27.h }, z8.h\n"
    ".inst 0xa1402ae3  // ld1h { z3.h, z11.h }, pn10.b/Z, [x23]\n"
    "addvl x21, SP, #9\n"
    "subs x25, x25, #0x1\n"
    ".inst 0xc17316ea  // sdot za.s[x8, 2], { z23.h-z26.h }, z3.h\n"
    "ld1sb { z23.s }, p0/Z, [x17]\n"
    "add z23.h, p0/M, z23.h, z7.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc17b170a  // sdot za.s[x8, 2], { z24.h-z27.h }, z11.h\n"
    "ld1sb { z24.s }, p0/Z, [x20]\n"
    "add z24.h, p0/M, z24.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc1721728  // sdot za.s[x8, 0], { z25.h-z28.h }, z2.h\n"
    "ld1h { z0.h }, p2/Z, [x24, #2, MUL VL]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1sb { z1.s }, p0/Z, [x20]\n"
    ".inst 0xc1701729  // sdot za.s[x8, 1], { z25.h-z28.h }, z0.h\n"
    "add z1.h, p0/M, z1.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    "ld1h { z3.h }, p2/Z, [x23, #2, MUL VL]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc173172a  // sdot za.s[x8, 2], { z25.h-z28.h }, z3.h\n"
    "trn1 z23.h, z23.h, z16.h\n"
    "ld1sb { z25.s }, p0/Z, [x20]\n"
    "add z25.h, p0/M, z25.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z3.s }, p0/Z, [x20]\n"
    "add z3.h, p0/M, z3.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z26.s }, p0/Z, [x20]\n"
    "add z26.h, p0/M, z26.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z30.s }, p0/Z, [x20]\n"
    "mov x12, #0x8\n"
    "add z30.h, p0/M, z30.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z27.s }, p0/Z, [x20]\n"
    "add z27.h, p0/M, z27.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z29.s }, p0/Z, [x20]\n"
    "add z29.h, p0/M, z29.h, z7.h\n"
    "trn1 z24.h, z24.h, z1.h\n"
    "trn1 z25.h, z25.h, z3.h\n"
    "trn1 z26.h, z26.h, z30.h\n"
    ".inst 0xa0402ac2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x22]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc0060c10  // mova { z16.d-z19.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    "trn1 z27.h, z27.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc17216e8  // sdot za.s[x8, 0], { z23.h-z26.h }, z2.h\n"
    "ld1sb { z20.s }, p0/Z, [x20]\n"
    "mov x12, #0x0\n"
    "add z20.h, p0/M, z20.h, z7.h\n"
    ".inst 0xc1731708  // sdot za.s[x8, 0], { z24.h-z27.h }, z3.h\n"
    ".inst 0xa0402aa2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x21]\n"
    "add x17, x17, %x[ld_in_col]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc17216e9  // sdot za.s[x8, 1], { z23.h-z26.h }, z2.h\n"
    "ld1sb { z23.s }, p0/Z, [x17]\n"
    "add z23.h, p0/M, z23.h, z7.h\n"
    "add x20, x17, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z8.s }, p0/Z, [x20]\n"
    "add z8.h, p0/M, z8.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1731709  // sdot za.s[x8, 1], { z24.h-z27.h }, z3.h\n"
    "ld1sb { z24.s }, p0/Z, [x20]\n"
    "mov z28.d, z20.d\n"
    "ld1h { z1.h }, p2/Z, [x22, #2, MUL VL]\n"
    "add z24.h, p0/M, z24.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z22.s }, p0/Z, [x20]\n"
    ".inst 0xc1711728  // sdot za.s[x8, 0], { z25.h-z28.h }, z1.h\n"
    "mov x12, #0x4\n"
    "add z22.h, p0/M, z22.h, z7.h\n"
    "ld1h { z1.h }, p2/Z, [x21, #2, MUL VL]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1711729  // sdot za.s[x8, 1], { z25.h-z28.h }, z1.h\n"
    "ld1sb { z25.s }, p0/Z, [x20]\n"
    "add z25.h, p0/M, z25.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z28.s }, p0/Z, [x20]\n"
    "add z28.h, p0/M, z28.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z26.s }, p0/Z, [x20]\n"
    "add z26.h, p0/M, z26.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z20.s }, p0/Z, [x20]\n"
    "mov x12, #0x8\n"
    "add z20.h, p0/M, z20.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z27.s }, p0/Z, [x20]\n"
    ".inst 0xc1a6ac10  // sqdmulh { z16.s-z19.s }, { z16.s-z19.s }, z6.s\n"
    "add z27.h, p0/M, z27.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z31.s }, p0/Z, [x20]\n"
    ".inst 0xc1a4aa30  // srshl { z16.s-z19.s }, { z16.s-z19.s }, z4.s\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc0040d84  // mova za.d[x8, #4], { z12.d-z15.d }\n"
    "add z31.h, p0/M, z31.h, z7.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z1.s }, p0/Z, [x20]\n"
    "add z1.h, p0/M, z1.h, z7.h\n"
    ".inst 0xc1aaab10  // add { z16.s-z19.s }, { z16.s-z19.s }, z10.s\n"
    ".inst 0xa1402be3  // ld1h { z3.h, z11.h }, pn10.b/Z, [SP]\n"
    "add x17, x17, %x[ld_in_col]\n"
    ".inst 0xc1b5ccb0  // sclamp { z16.s-z19.s }, z5.s, z21.s\n"
    "st1b { z16.s }, p1, [x15]\n"
    "add x15, x15, x13\n"
    "ld1h { z2.h }, p2/Z, [SP, #2, MUL VL]\n"
    "st1b { z17.s }, p1, [x14]\n"
    "add x14, x14, x11\n"
    "trn1 z23.h, z23.h, z8.h\n"
    "trn1 z24.h, z24.h, z22.h\n"
    "st1b { z18.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "trn1 z25.h, z25.h, z28.h\n"
    "trn1 z26.h, z26.h, z20.h\n"
    "st1b { z19.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "trn1 z27.h, z27.h, z31.h\n"
    "mov z28.d, z1.d\n"
    "bgt 20b\n"
    "21:"  // Main loop tail
    ".inst 0xc17316e8  // sdot za.s[x8, 0], { z23.h-z26.h }, z3.h\n"
    "addvl x24, SP, #6\n"
    "addvl x23, SP, #12\n"
    ".inst 0xc17b1708  // sdot za.s[x8, 0], { z24.h-z27.h }, z11.h\n"
    ".inst 0xa0402b08  // ld1h { z8.h-z9.h }, pn10.b/Z, [x24]\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc17816e9  // sdot za.s[x8, 1], { z23.h-z26.h }, z8.h\n"
    "add x22, x17, %x[ld_in_row]\n"
    "addvl x21, SP, #3\n"
    ".inst 0xc1791709  // sdot za.s[x8, 1], { z24.h-z27.h }, z9.h\n"
    ".inst 0xa1402ae3  // ld1h { z3.h, z11.h }, pn10.b/Z, [x23]\n"
    "addvl x20, SP, #9\n"
    ".inst 0xc17316ea  // sdot za.s[x8, 2], { z23.h-z26.h }, z3.h\n"
    "ld1sb { z29.s }, p0/Z, [x17]\n"
    "add z29.h, p0/M, z29.h, z7.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z8.s }, p0/Z, [x22]\n"
    "add z8.h, p0/M, z8.h, z7.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc17b170a  // sdot za.s[x8, 2], { z24.h-z27.h }, z11.h\n"
    "ld1sb { z30.s }, p0/Z, [x22]\n"
    "add z30.h, p0/M, z30.h, z7.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc1721728  // sdot za.s[x8, 0], { z25.h-z28.h }, z2.h\n"
    "ld1h { z0.h }, p2/Z, [x24, #2, MUL VL]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1sb { z20.s }, p0/Z, [x22]\n"
    ".inst 0xc1701729  // sdot za.s[x8, 1], { z25.h-z28.h }, z0.h\n"
    "add z20.h, p0/M, z20.h, z7.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    "ld1h { z2.h }, p2/Z, [x23, #2, MUL VL]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc172172a  // sdot za.s[x8, 2], { z25.h-z28.h }, z2.h\n"
    "trn1 z29.h, z29.h, z8.h\n"
    "ld1sb { z31.s }, p0/Z, [x22]\n"
    "add z31.h, p0/M, z31.h, z7.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z25.s }, p0/Z, [x22]\n"
    "add z25.h, p0/M, z25.h, z7.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z0.s }, p0/Z, [x22]\n"
    "add z0.h, p0/M, z0.h, z7.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z17.s }, p0/Z, [x22]\n"
    "mov x12, #0x8\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z1.s }, p0/Z, [x22]\n"
    "add z1.h, p0/M, z1.h, z7.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z28.s }, p0/Z, [x22]\n"
    "add z28.h, p0/M, z28.h, z7.h\n"
    "trn1 z30.h, z30.h, z20.h\n"
    "trn1 z31.h, z31.h, z25.h\n"
    "trn1 z0.h, z0.h, z17.h\n"
    ".inst 0xa1402aa3  // ld1h { z3.h, z11.h }, pn10.b/Z, [x21]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc0060c18  // mova { z24.d-z27.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "trn1 z1.h, z1.h, z28.h\n"
    ".inst 0xc17317a8  // sdot za.s[x8, 0], { z29.h-z0.h }, z3.h\n"
    "ld1sb { z22.s }, p0/Z, [x22]\n"
    ".inst 0xc1a6ac18  // sqdmulh { z24.s-z27.s }, { z24.s-z27.s }, z6.s\n"
    "add z22.h, p0/M, z22.h, z7.h\n"
    ".inst 0xc17b17c8  // sdot za.s[x8, 0], { z30.h-z1.h }, z11.h\n"
    ".inst 0xa1402a83  // ld1h { z3.h, z11.h }, pn10.b/Z, [x20]\n"
    "add x17, x17, %x[ld_in_col]\n"
    ".inst 0xc1a4aa38  // srshl { z24.s-z27.s }, { z24.s-z27.s }, z4.s\n"
    ".inst 0xc17317a9  // sdot za.s[x8, 1], { z29.h-z0.h }, z3.h\n"
    "mov z2.d, z22.d\n"
    "ld1h { z9.h }, p2/Z, [x21, #2, MUL VL]\n"
    ".inst 0xc17b17c9  // sdot za.s[x8, 1], { z30.h-z1.h }, z11.h\n"
    ".inst 0xc1aaab18  // add { z24.s-z27.s }, { z24.s-z27.s }, z10.s\n"
    ".inst 0xc17917e8  // sdot za.s[x8, 0], { z31.h-z2.h }, z9.h\n"
    "ld1h { z8.h }, p2/Z, [x20, #2, MUL VL]\n"
    ".inst 0xc1b5ccb8  // sclamp { z24.s-z27.s }, z5.s, z21.s\n"
    "st1b { z24.s }, p1, [x15]\n"
    "add x15, x15, x13\n"
    "st1b { z25.s }, p1, [x14]\n"
    "add x14, x14, x11\n"
    ".inst 0xc0040d84  // mova za.d[x8, #4], { z12.d-z15.d }\n"
    ".inst 0xa1402be3  // ld1h { z3.h, z11.h }, pn10.b/Z, [SP]\n"
    "st1b { z26.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    ".inst 0xc17817e9  // sdot za.s[x8, 1], { z31.h-z2.h }, z8.h\n"
    "ld1h { z2.h }, p2/Z, [SP, #2, MUL VL]\n"
    "st1b { z27.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "22:"  // Main loop skip tail
    "cbz x7, 23f\n"  // Skip remainder inputs
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z24.s }, p0/Z, [x17]\n"
    "add z24.h, p0/M, z24.h, z7.h\n"
    "add x20, x17, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z25.s }, p0/Z, [x20]\n"
    "add z25.h, p0/M, z25.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z24.h, z24.h, z17.h\n"
    "trn1 z25.h, z25.h, z16.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z26.s }, p0/Z, [x20]\n"
    "add z26.h, p0/M, z26.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "add z17.h, p0/M, z17.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z27.s }, p0/Z, [x20]\n"
    "add z27.h, p0/M, z27.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "mov x12, #0x8\n"
    "add z16.h, p0/M, z16.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z28.s }, p0/Z, [x20]\n"
    "add z28.h, p0/M, z28.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z31.s }, p0/Z, [x20]\n"
    "add z31.h, p0/M, z31.h, z7.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "trn1 z26.h, z26.h, z17.h\n"
    "trn1 z27.h, z27.h, z16.h\n"
    "ld1sb { z0.s }, p0/Z, [x20]\n"
    "add z0.h, p0/M, z0.h, z7.h\n"
    "trn1 z28.h, z28.h, z31.h\n"
    "addvl x21, SP, #6\n"
    ".inst 0xc1731708  // sdot za.s[x8, 0], { z24.h-z27.h }, z3.h\n"
    "mov z29.d, z0.d\n"
    "addvl x20, SP, #12\n"
    "sub x16, x16, #0x1\n"
    ".inst 0xc17b1728  // sdot za.s[x8, 0], { z25.h-z28.h }, z11.h\n"
    ".inst 0xa0402aa8  // ld1h { z8.h-z9.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc1721748  // sdot za.s[x8, 0], { z26.h-z29.h }, z2.h\n"
    "ld1h { z2.h }, p2/Z, [x21, #2, MUL VL]\n"
    ".inst 0xc1781709  // sdot za.s[x8, 1], { z24.h-z27.h }, z8.h\n"
    ".inst 0xc0060c10  // mova { z16.d-z19.d }, za.d[x8, #0]\n"
    ".inst 0xc1a6ac10  // sqdmulh { z16.s-z19.s }, { z16.s-z19.s }, z6.s\n"
    ".inst 0xc1791729  // sdot za.s[x8, 1], { z25.h-z28.h }, z9.h\n"
    ".inst 0xa1402a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc1a4aa30  // srshl { z16.s-z19.s }, { z16.s-z19.s }, z4.s\n"
    ".inst 0xc171170a  // sdot za.s[x8, 2], { z24.h-z27.h }, z1.h\n"
    ".inst 0xc1aaab10  // add { z16.s-z19.s }, { z16.s-z19.s }, z10.s\n"
    ".inst 0xc179172a  // sdot za.s[x8, 2], { z25.h-z28.h }, z9.h\n"
    ".inst 0xc1b5ccb0  // sclamp { z16.s-z19.s }, z5.s, z21.s\n"
    "st1b { z16.s }, p1, [x15]\n"
    "add x15, x15, x13\n"
    ".inst 0xc1721749  // sdot za.s[x8, 1], { z26.h-z29.h }, z2.h\n"
    "ld1h { z3.h }, p2/Z, [x20, #2, MUL VL]\n"
    "st1b { z17.s }, p1, [x14]\n"
    "add x14, x14, x11\n"
    ".inst 0xc173174a  // sdot za.s[x8, 2], { z26.h-z29.h }, z3.h\n"
    "add x8, x8, #0x1\n"
    "st1b { z18.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "st1b { z19.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    ".inst 0xc0040d84  // mova za.d[x8, #4], { z12.d-z15.d }\n"
    "23:"  // Tail input: End
    "cbz x16, 25f\n"
    "24:"  // Right padding loop
    ".inst 0xc0060c1c  // mova { z28.d-z31.d }, za.d[x8, #0]\n"
    ".inst 0xc1a6ac1c  // sqdmulh { z28.s-z31.s }, { z28.s-z31.s }, z6.s\n"
    "add x8, x8, #0x1\n"
    ".inst 0xc1a4aa3c  // srshl { z28.s-z31.s }, { z28.s-z31.s }, z4.s\n"
    "subs x16, x16, #0x1\n"
    ".inst 0xc0040d84  // mova za.d[x8, #4], { z12.d-z15.d }\n"
    ".inst 0xc1aaab1c  // add { z28.s-z31.s }, { z28.s-z31.s }, z10.s\n"
    ".inst 0xc1b5ccbc  // sclamp { z28.s-z31.s }, z5.s, z21.s\n"
    "st1b { z28.s }, p1, [x15]\n"
    "add x15, x15, x13\n"
    "st1b { z29.s }, p1, [x14]\n"
    "add x14, x14, x11\n"
    "st1b { z30.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "st1b { z31.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "bgt 24b\n"
    "25:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incw x20, ALL, MUL #16\n"
    "incw x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "incw x6\n"
    "whilelt p1.s, x6, x5\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21\n"
    "add x22, x22, x20\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21\n"
    "add x22, x22, x20\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    "addvl SP, SP, #15\n"
    ".inst 0xd503467f  // SMSTOP\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights)), [offsetof_Requantize32_a_offset] "I" (offsetof(arm_gemm::Requantize32, a_offset)), [offsetof_Requantize32_b_offset] "I" (offsetof(arm_gemm::Requantize32, b_offset)), [offsetof_Requantize32_bias] "I" (offsetof(arm_gemm::Requantize32, bias)), [offsetof_Requantize32_c_offset] "I" (offsetof(arm_gemm::Requantize32, c_offset)), [offsetof_Requantize32_maxval] "I" (offsetof(arm_gemm::Requantize32, maxval)), [offsetof_Requantize32_minval] "I" (offsetof(arm_gemm::Requantize32, minval)), [offsetof_Requantize32_per_channel_muls] "I" (offsetof(arm_gemm::Requantize32, per_channel_muls)), [offsetof_Requantize32_per_channel_right_shifts] "I" (offsetof(arm_gemm::Requantize32, per_channel_right_shifts)), [offsetof_Requantize32_per_layer_mul] "I" (offsetof(arm_gemm::Requantize32, per_layer_mul)), [offsetof_Requantize32_per_layer_right_shift] "I" (offsetof(arm_gemm::Requantize32, per_layer_right_shift)), [qp] "r" (&qp)
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
