# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from gtk.gdk import Pixbuf as gPixbuf
from gobject import TYPE_STRING as gString
from gtk import (ListStore, TreeViewColumn, CellRendererPixbuf, STOCK_ABOUT,
                 CellRendererText, ICON_SIZE_MENU, STOCK_REFRESH, Dialog,
                 STOCK_OK, RESPONSE_OK, STOCK_CANCEL, RESPONSE_CANCEL, HBox,
                 image_new_from_stock, STOCK_DIALOG_QUESTION, Label,
                 ICON_SIZE_DIALOG, FileFilter, FileChooserDialog,
                 FILE_CHOOSER_ACTION_OPEN, FILE_CHOOSER_ACTION_SAVE,
                 STOCK_OPEN, STOCK_SAVE_AS)
from gtk.glade import XML as glade_XML
from random import randint
from os.path import join, exists
import tagpy

from common.statistics import Statistics
from common.functions import Functions
from common.playlists import Playlists
playlists_management = Playlists()

from modules.explorer.playlist import Playlist
from modules.explorer.musicdb import MusicDatabase

class PlaylistsView(object):
    ref = None
    ref2 = None

    def __new__(cls, *args, **kws):
        # Singleton
        if cls.ref is None:
            cls.ref = super(PlaylistsView, cls).__new__(cls, args, kws)
        return cls.ref
	 	     
    def __init__(self, glade_file, config, userconf): 
        self.playlist = Playlist(glade_file, config, userconf)
        self.glade_file = glade_file
        self.config = config
        self.userconf = userconf

        self.functions = Functions()
        self.statistics = Statistics()
        self.musicdb = MusicDatabase(self.config, None)

        # Show widgets
        self.glade_file.get_widget('vbox2').show()
        self.glade_file.get_widget('frame4').show()

        if PlaylistsView.ref2 is None: 
            PlaylistsView.ref2 = 42 
            self.launch()

    def reload_database(self, force_scan=False):
        pass

    def launch(self):
        # Start GUI
        label_automatic = self.glade_file.get_widget('label12')
        label_saved = self.glade_file.get_widget('label13')
        label_automatic.set_markup('<big><b>Automatic</b></big>')
        label_saved.set_markup('<big><b>Playlists</b></big>')

        treestore = ListStore(gPixbuf, gString, gString)
        auto_tview = self.glade_file.get_widget('treeview5')
        auto_tview.set_model(treestore)
        auto_tview.set_rules_hint(True)

        render_pixbuf = CellRendererPixbuf()
        render_text = CellRendererText()
        auto_column = TreeViewColumn()
        
        auto_column.pack_start(render_pixbuf, expand=False)
        auto_column.add_attribute(render_pixbuf, 'pixbuf', 0)
        auto_column.pack_start(render_text, expand=True)
        auto_column.add_attribute(render_text, 'markup', 1)
        auto_tview.append_column(auto_column)

        logo_t = auto_tview.render_icon(STOCK_ABOUT, ICON_SIZE_MENU)
        logo_r = auto_tview.render_icon(STOCK_REFRESH, ICON_SIZE_MENU)            
        treestore.append((logo_t, _('Top 50 songs'), 's50'))
        treestore.append((logo_t, _('Top 100 songs'), 's100'))
        treestore.append((logo_t, _('Top 10 albums'), 'a10'))
        treestore.append((logo_r, _('Random 50 songs'), 'r50'))
        treestore.append((logo_r, _('Random 100 songs'), 'r100'))

        _treestore = ListStore(gString)
        saved_tview = self.glade_file.get_widget('treeview6')
        saved_tview.set_model(_treestore)
        saved_tview.set_rules_hint(True)

        saved_column = TreeViewColumn()
        saved_column.pack_start(render_text, expand=True)
        saved_column.add_attribute(render_text, 'markup', 0)
        saved_tview.append_column(saved_column)

        playlists_list = playlists_management.get_playlists_list()
        for playlist_name in playlists_list:
            _treestore.append((playlist_name,))

        # Connect GTK signals
        auto_tview.get_selection().connect('changed', self.load_playlist, 'a')
        saved_tview.get_selection().connect('changed', self.load_playlist, 's')

        tool_add = self.glade_file.get_widget('tool-playlists-add')
        tool_add.connect('clicked', self.add_playlist, _treestore)

        tool_remove = self.glade_file.get_widget('tool-playlists-remove')
        tool_remove.connect('clicked', self.remove_playlist,
                                       _treestore, saved_tview)

        tool_import = self.glade_file.get_widget('tool-playlists-import')
        tool_import.connect('clicked', self.import_playlist, _treestore)

        tool_export = self.glade_file.get_widget('tool-playlists-export')
        tool_export.connect('clicked', self.export_playlist, saved_tview)

    def load_playlist(self, selection, pltype):
        """Load a selected playlist."""
        if selection:
            (mod, iter_) = selection.get_selected()
            if not iter_:
                return

            self.playlist.clear_playlist(None)
            songs = []

            if pltype == 'a':
                # Automatic playlist
                pls = mod.get_value(iter_, 2)
                songs_list = self.musicdb.load()

                if pls == 's50':
                    # Top 50 songs
                    songs = self.statistics.get_top()
                elif pls == 's100':
                    # Top 100 songs
                    songs = self.statistics.get_top('filename', 100)
                elif pls == 'a10':
                    # Top 10 albums
                    albums = self.statistics.get_top('album', 10)

                    songs_list.sort(lambda x,y:cmp(x[6], y[6]))

                    for album in albums:
                        for sg in songs_list:
                            if sg[2] == album[0]:
                                songs.append(sg[8])
                elif pls in ('r50', 'r100'):
                    # Random playlists

                    if pls == 'r50':
                        # Random 50 songs
                        maxsongs = 50
                    elif pls == 'r100':
                        # Random 100 songs
                        maxsongs = 100

                    randoms = []
                    i = 0
                    while i <= maxsongs:
                        randoms.append(randint(0, len(songs_list)))
                        i += 1

                    for randid in randoms:
                        songs.append(songs_list[randid][8])
            else:
                # Saved playlist
                pls = mod.get_value(iter_, 0)
                songs = playlists_management.load_playlist(pls)

            # Get audio tags
            new_playlist = []
            for song in songs:
                if type(song) == tuple:
                    song = song[0]

                if exists(song):
                    tagpy_file = tagpy.FileRef(str(song))
                    tags = tagpy_file.tag()
                    audio = tagpy_file.audioProperties()

                    title = tags.title
                    artist = tags.artist
                    album = tags.album
                    comment = tags.comment
                    genre = tags.genre
                    year = tags.year
                    track = tags.track
                    length = audio.length

                    new_playlist.append((title, artist, album, comment,
                                         genre, year, track, length, song))

            if len(new_playlist) > 0:
                # Add songs to playlist
                self.playlist.add_songs(new_playlist, False, pls)

    def add_playlist(self, widget, treestore):
        """Create a new playlist."""
        newwin = glade_XML(join(self.functions.datadir, 'glade',
                 'playlistmenu.glade'), 'dialog1', domain='bluemindo')
        new_win = newwin.get_widget('dialog1')
        response = new_win.run()

        text = newwin.get_widget('entry1').get_text()
        if response and text:
           playlists_management.create_new_playlist(text)
           treestore.append((text,))

        new_win.destroy()

    def remove_playlist(self, widget, treestore, treeview):
        """Remove a playlist."""
        selection = treeview.get_selection()
        if selection:
            (mod, iter_) = selection.get_selected()
            if not iter_:
                return

            playlist = mod.get_value(iter_, 0)

            title = _('Delete the playlist?')
            dialog = Dialog(title, None, 0,(STOCK_OK, RESPONSE_OK,
                                            STOCK_CANCEL, RESPONSE_CANCEL))
            hbox = HBox(False, 8)
            hbox.set_border_width(8)
            dialog.vbox.pack_start(hbox, False, False, 0)
            stock = image_new_from_stock(STOCK_DIALOG_QUESTION,
                                         ICON_SIZE_DIALOG)
            hbox.pack_start(stock, False, False, 0)
            label = Label(_('Do you really want to delete '
                            'the %s playlist?') % playlist)
            hbox.pack_start(label, False, False, 0)

            dialog.show_all()
            reponse = dialog.run()

            if reponse == RESPONSE_OK:
                playlists_management.delete_playlist(playlist)
                del treestore[iter_]

            dialog.destroy()

    def import_playlist(self, widget, treestore):
        """Import a playlist into Bluemindo."""
        fcd_filter = FileFilter()
        fcd_filter.add_mime_type('audio/x-mpegurl')
        fcd_filter.set_name(_('Winamp-like playlists'))

        fcd_playlist = FileChooserDialog(_('Choose a playlist'), None,
                       FILE_CHOOSER_ACTION_OPEN,
                       (STOCK_CANCEL, RESPONSE_CANCEL, STOCK_OPEN, RESPONSE_OK
        ))

        fcd_playlist.add_filter(fcd_filter)
        playlist = fcd_playlist.run()

        if playlist == RESPONSE_OK:
            filename = fcd_playlist.get_filename()
            playlist_name = playlists_management.import_playlist(filename)
            treestore.append((playlist_name,))

        fcd_playlist.destroy()

    def export_playlist(self, widget, treeview):
        """Export a playlist from Bluemindo."""
        selection = treeview.get_selection()
        if selection:
            (mod, iter_) = selection.get_selected()
            if not iter_:
                return

            playlist = mod.get_value(iter_, 0)

            fcd_filter = FileFilter()
            fcd_filter.add_mime_type('audio/x-mpegurl')
            fcd_filter.set_name(_('Winamp-like playlists'))

            fcd_playlist = FileChooserDialog(_('Export playlist as…'), None,
            FILE_CHOOSER_ACTION_SAVE,(STOCK_CANCEL, RESPONSE_CANCEL,
                                      STOCK_SAVE_AS, RESPONSE_OK))

            fcd_playlist.add_filter(fcd_filter)
            _playlist = fcd_playlist.run()

            if _playlist == RESPONSE_OK:
                filename = fcd_playlist.get_filename()
                playlists_management.export_playlist(playlist, filename)

            fcd_playlist.destroy()