/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
package jalview.workers;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jalview.analysis.AAFrequency;
import jalview.analysis.AlignmentUtils;
import jalview.api.AlignViewportI;
import jalview.api.AlignmentViewPanel;
import jalview.datamodel.AlignmentAnnotation;
import jalview.datamodel.AlignmentI;
import jalview.datamodel.Annotation;
import jalview.datamodel.ProfilesI;
import jalview.datamodel.SequenceI;
import jalview.renderer.ResidueShaderI;
import jalview.util.Constants;
import jalview.util.MessageManager;

public class SecondaryStructureConsensusThread extends AlignCalcWorker
{
  public SecondaryStructureConsensusThread(AlignViewportI alignViewport,
          AlignmentViewPanel alignPanel)
  {
    super(alignViewport, alignPanel);
  }

  @Override
  public void run()
  {
    if (calcMan.isPending(this))
    {
      return;
    }
    calcMan.notifyStart(this);
    // long started = System.currentTimeMillis();
    try
    {
      List<AlignmentAnnotation> ssConsensus = getSSConsensusAnnotation();
      if ((ssConsensus == null) || calcMan.isPending(this))
      {
        calcMan.workerComplete(this);
        return;
      }
      while (!calcMan.notifyWorking(this))
      {
        try
        {
          if (ap != null)
          {
            ap.paintAlignment(false, false);
          }
          Thread.sleep(200);
        } catch (Exception ex)
        {
          ex.printStackTrace();
        }
      }
      if (alignViewport.isClosed())
      {
        abortAndDestroy();
        return;
      }
      AlignmentI alignment = alignViewport.getAlignment();

      int aWidth = -1;

      if (alignment == null || (aWidth = alignment.getWidth()) < 0)
      {
        calcMan.workerComplete(this);
        return;
      }

      setSecondaryStructureSources();
      eraseSSConsensus(aWidth);
      computeSSConsensus(alignment);
      updateResultAnnotation(true);

      if (ap != null)
      {
        ap.paintAlignment(true, true);
      }
    } catch (OutOfMemoryError error)
    {
      calcMan.disableWorker(this);
      ap.raiseOOMWarning("calculating consensus", error);
    } finally
    {
      /*
       * e.g. ArrayIndexOutOfBoundsException can happen due to a race condition
       * - alignment was edited at same time as calculation was running
       */
      calcMan.workerComplete(this);
    }
  }

  /**
   * Clear out any existing consensus annotations
   * 
   * @param aWidth
   *          the width (number of columns) of the annotated alignment
   */
  protected void eraseSSConsensus(int aWidth)
  {
    List<AlignmentAnnotation> ssConsensuses = getSSConsensusAnnotation();
    for (AlignmentAnnotation ssConsensus : ssConsensuses)
    {
      if (ssConsensus != null)
      {
        ssConsensus.annotations = new Annotation[aWidth];
      }
    }
  }

  /**
   * @param alignment
   */
  protected void computeSSConsensus(AlignmentI alignment)
  {

    SequenceI[] aseqs = getSequences();
    int width = alignment.getWidth();
    Map<String, ProfilesI> hSSConsensusProfileMap = new HashMap<String, ProfilesI>();
    List<String> ssSources = getSecondaryStructureSources();
    for (String ssSource : ssSources)
    {
      ProfilesI hSSConsensus = AAFrequency.calculateSS(aseqs, width, 0,
              width, true, ssSource, null);
      hSSConsensusProfileMap.put(ssSource, hSSConsensus);
    }

    alignViewport.setSequenceSSConsensusHash(hSSConsensusProfileMap);
    setColourSchemeConsensus(hSSConsensusProfileMap);
  }

  /**
   * @return
   */
  protected SequenceI[] getSequences()
  {
    return alignViewport.getAlignment().getSequencesArray();
  }

  /**
   * @param hconsensus
   */
  protected void setColourSchemeConsensus(
          Map<String, ProfilesI> ssConsensusProfileMap)
  {
    ResidueShaderI cs = alignViewport.getResidueShading();
    if (cs != null)
    {
      cs.setSSConsensusProfileMap(ssConsensusProfileMap);
    }
  }

  /**
   * Get the Consensus annotation for the alignment
   * 
   * @return
   */
  protected List<AlignmentAnnotation> getSSConsensusAnnotation()
  {
    return alignViewport
            .getAlignmentSecondaryStructureConsensusAnnotation();
  }

  /**
   * Get the Consensus annotation for the alignment
   * 
   * @return
   */
  protected void setSecondaryStructureSources()
  {
    List<String> sources = null;
    AlignmentAnnotation[] aa = alignViewport.getAlignment()
            .getAlignmentAnnotation();
    if (aa != null)
    {
      sources = AlignmentUtils.extractSSSourceInAlignmentAnnotation(aa);
      if (sources != null)
      {
        sources.add(0, Constants.SS_ALL_PROVIDERS);
        alignViewport.setSecondaryStructureSources(sources);
      }
    }
  }

  protected List<String> getSecondaryStructureSources()
  {
    return alignViewport.getSecondaryStructureSources();
  }

  /**
   * Get the Gap annotation for the alignment
   * 
   * @return
   */
  protected AlignmentAnnotation getGapAnnotation()
  {
    return alignViewport.getAlignmentGapAnnotation();
  }

  /**
   * update the consensus annotation from the sequence profile data using
   * current visualization settings.
   */
  @Override
  public void updateAnnotation()
  {
    updateResultAnnotation(false);
  }

  public void updateResultAnnotation(boolean immediate)
  {
    List<AlignmentAnnotation> ssConsensuses = getSSConsensusAnnotation();
    Map<String, ProfilesI> ssConsensusProfileMap = getViewportSSConsensus();
    for (AlignmentAnnotation ssConsensus : ssConsensuses)
    {
      ProfilesI ssConsensusProfile = null;
      for (String source : ssConsensusProfileMap.keySet())
      {
        if (ssConsensus.description.startsWith(source))
        {
          ssConsensusProfile = ssConsensusProfileMap.get(source);
          break;
        }
      }
      if (ssConsensusProfile == null)
      {
        continue;
      }
      if (immediate || !calcMan.isWorking(this) && ssConsensus != null
              && ssConsensusProfile != null)
      {        
        deriveSSConsensus(ssConsensus, ssConsensusProfile);
        
        if (ssConsensusProfile.get(1) != null)
        {
          ssConsensus.setNoOfSequencesIncluded(
                ssConsensusProfile.get(1).getSeqWithSSCount());
        }
        ssConsensus.setNoOfTracksIncluded(ssConsensusProfile.getCount());
        ssConsensus.hasData=ssConsensusProfile.getCount()>0;
      }
      
      
    }
  }

  /**
   * Convert the computed consensus data into the desired annotation for
   * display.
   * 
   * @param consensusAnnotation
   *          the annotation to be populated
   * @param hconsensus
   *          the computed consensus data
   */
  protected void deriveSSConsensus(AlignmentAnnotation ssConsensus,
          ProfilesI hSSConsensus)
  {

    long nseq = getSequences().length;
    AAFrequency.completeSSConsensus(ssConsensus, hSSConsensus,
            hSSConsensus.getStartColumn(), hSSConsensus.getEndColumn() + 1,
            alignViewport.isIgnoreGapsConsensus(),
            alignViewport.isShowSequenceLogo(), nseq);
  }
  
  /**
   * Get the consensus data stored on the viewport.
   * 
   * @return
   */
  protected Map<String, ProfilesI> getViewportSSConsensus()
  {
    // TODO convert ComplementConsensusThread to use Profile
    return alignViewport.getSequenceSSConsensusHash();
  }
}
