/*
 * GCM test vectors.
 */
#ifndef GCM_KATS_H
#define GCM_KATS_H

#include <stddef.h>
#include <string.h>

#define AES_BLOCK_SIZE 16
#define MAX_ARRAY_SIZE 128
#define NUM_GCM_TESTS		(sizeof(gcm_kats)/sizeof(gcm_kat_t))

typedef struct {
    unsigned char key[MAX_ARRAY_SIZE];
    unsigned int keylen;
    unsigned char data[MAX_ARRAY_SIZE];
    unsigned int datalen;
    unsigned char aad[MAX_ARRAY_SIZE];
    unsigned int aadlen;
    unsigned char iv[MAX_ARRAY_SIZE];
    unsigned int ivlen;
    unsigned char tag[MAX_ARRAY_SIZE];
    unsigned int taglen;
    unsigned char result[MAX_ARRAY_SIZE];
    unsigned int chunks[MAX_ARRAY_SIZE];
    unsigned int num_chunks;
} gcm_kat_t;

static gcm_kat_t gcm_kats[] = {
	/* GCM data - 0 */
	{{0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,},
	 .keylen = 16,
	 {0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
	  0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
	  0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
	  0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,0x1a,0xaf,0xd2,0x55,},
	 .datalen = 64,
	 {},
	 .aadlen = 0,
	 {0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,},
	 .ivlen = 12,
	 {0x4d,0x5c,0x2a,0xf3,0x27,0xcd,0x64,0xa6,0x2c,0xf3,0x5a,0xbd,0x2b,0xa6,0xfa,0xb4,},
	 .taglen = 16,
	 {0x42,0x83,0x1e,0xc2,0x21,0x77,0x74,0x24,0x4b,0x72,0x21,0xb7,0x84,0xd0,0xd4,0x9c,
	  0xe3,0xaa,0x21,0x2f,0x2c,0x02,0xa4,0xe0,0x35,0xc1,0x7e,0x23,0x29,0xac,0xa1,0x2e,
	  0x21,0xd5,0x14,0xb2,0x54,0x66,0x93,0x1c,0x7d,0x8f,0x6a,0x5a,0xac,0x84,0xaa,0x05,
	  0x1b,0xa3,0x0b,0x39,0x6a,0x0a,0xac,0x97,0x3d,0x58,0xe0,0x91,0x47,0x3f,0x59,0x85,},
	 {16,16,16,16},
	 4,
	},
	/* GCM data - 1 */
	{{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .keylen = 16,
	 {},
	 .datalen = 0,
	 {},
	 .aadlen = 0,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .ivlen = 12,
	 {0x58,0xe2,0xfc,0xce,0xfa,0x7e,0x30,0x61,0x36,0x7f,0x1d,0x57,0xa4,0xe7,0x45,0x5a},
	 .taglen = 16,
	 {},
	 {},
	 .num_chunks = 0,
	},
	/* GCM data - 2 */
	{{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .keylen = 16,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .datalen = 16,
	 {},
	 .aadlen = 0,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .ivlen = 12,
	 {0xab,0x6e,0x47,0xd4,0x2c,0xec,0x13,0xbd,0xf5,0x3a,0x67,0xb2,0x12,0x57,0xbd,0xdf,},
	 .taglen = 16,
	 {0x03,0x88,0xda,0xce,0x60,0xb6,0xa3,0x92,0xf3,0x28,0xc2,0xb9,0x71,0xb2,0xfe,0x78,},
	 {16},
	 .num_chunks = 1,
	},
	/* GCM data - 3 */
	{{0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,},
	 .keylen = 16,
	 {0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
	  0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
	  0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
	  0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,},
	 .datalen = 60,
	 {0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
	  0xab,0xad,0xda,0xd2,},
	 .aadlen = 20,
	 {0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,},
	 .ivlen = 12,
	 {0x5b,0xc9,0x4f,0xbc,0x32,0x21,0xa5,0xdb,0x94,0xfa,0xe9,0x5a,0xe7,0x12,0x1a,0x47,},
	 .taglen = 16,
	 {0x42,0x83,0x1e,0xc2,0x21,0x77,0x74,0x24,0x4b,0x72,0x21,0xb7,0x84,0xd0,0xd4,0x9c,
	  0xe3,0xaa,0x21,0x2f,0x2c,0x02,0xa4,0xe0,0x35,0xc1,0x7e,0x23,0x29,0xac,0xa1,0x2e,
	  0x21,0xd5,0x14,0xb2,0x54,0x66,0x93,0x1c,0x7d,0x8f,0x6a,0x5a,0xac,0x84,0xaa,0x05,
	  0x1b,0xa3,0x0b,0x39,0x6a,0x0a,0xac,0x97,0x3d,0x58,0xe0,0x91,},
	 {16,16,16,12},
	 .num_chunks = 4,
	},
	/* GCM data - 4 - AES 192 - Test Case 7 */
	{{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .keylen = 24,
	 {},
	 .datalen = 0,
	 {},
	 .aadlen = 0,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .ivlen = 12,
	 {0xcd,0x33,0xb2,0x8a,0xc7,0x73,0xf7,0x4b,0xa0,0x0e,0xd1,0xf3,0x12,0x57,0x24,0x35,},
	 .taglen = 16,
	 {},
	 {},
	 .num_chunks = 0,
	},
	/* GCM data - 5 - AES 192 - Test Case 8 */
	{{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .keylen = 24,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .datalen = 16,
	 {},
	 .aadlen = 0,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .ivlen = 12,
	 {0x2f,0xf5,0x8d,0x80,0x03,0x39,0x27,0xab,0x8e,0xf4,0xd4,0x58,0x75,0x14,0xf0,0xfb,},
	 .taglen = 16,
	 {0x98,0xe7,0x24,0x7c,0x07,0xf0,0xfe,0x41,0x1c,0x26,0x7e,0x43,0x84,0xb0,0xf6,0x00,},
	 {16},
	 .num_chunks = 1,
	},
	/* GCM data - 6 - AES 192 - Test Case 9 */
	{{0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
	  0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,},
	 .keylen = 24,
	 {0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
	  0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
	  0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
	  0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,0x1a,0xaf,0xd2,0x55,},
	 .datalen = 64,
	 {},
	 .aadlen = 0,
	 {0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,},
	 .ivlen = 12,
	 {0x99,0x24,0xa7,0xc8,0x58,0x73,0x36,0xbf,0xb1,0x18,0x02,0x4d,0xb8,0x67,0x4a,0x14,},
	 .taglen = 16,
	 {0x39,0x80,0xca,0x0b,0x3c,0x00,0xe8,0x41,0xeb,0x06,0xfa,0xc4,0x87,0x2a,0x27,0x57,
	  0x85,0x9e,0x1c,0xea,0xa6,0xef,0xd9,0x84,0x62,0x85,0x93,0xb4,0x0c,0xa1,0xe1,0x9c,
	  0x7d,0x77,0x3d,0x00,0xc1,0x44,0xc5,0x25,0xac,0x61,0x9d,0x18,0xc8,0x4a,0x3f,0x47,
	  0x18,0xe2,0x44,0x8b,0x2f,0xe3,0x24,0xd9,0xcc,0xda,0x27,0x10,0xac,0xad,0xe2,0x56,},
	 {16,16,16,16},
	 .num_chunks = 4,
	},
	/* GCM data - 7 - AES 192 - Test Case 10 */
	{{0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
	  0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,},
	 .keylen = 24,
	 {0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
	  0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
	  0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
	  0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,},
	 .datalen = 60,
	 {0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
	  0xab,0xad,0xda,0xd2,},
	 .aadlen = 20,
	 {0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,},
	 .ivlen = 12,
	 {0x25,0x19,0x49,0x8e,0x80,0xf1,0x47,0x8f,0x37,0xba,0x55,0xbd,0x6d,0x27,0x61,0x8c,},
	 .taglen = 16,
	 {0x39,0x80,0xca,0x0b,0x3c,0x00,0xe8,0x41,0xeb,0x06,0xfa,0xc4,0x87,0x2a,0x27,0x57,
	  0x85,0x9e,0x1c,0xea,0xa6,0xef,0xd9,0x84,0x62,0x85,0x93,0xb4,0x0c,0xa1,0xe1,0x9c,
	  0x7d,0x77,0x3d,0x00,0xc1,0x44,0xc5,0x25,0xac,0x61,0x9d,0x18,0xc8,0x4a,0x3f,0x47,
	  0x18,0xe2,0x44,0x8b,0x2f,0xe3,0x24,0xd9,0xcc,0xda,0x27,0x10,},
	 {16,16,16,12},
	 .num_chunks = 4,
	},
	/* GCM data - 8 - AES 256 - Test Case 13 */
	{{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .keylen = 32,
	 {},
	 .datalen = 0,
	 {},
	 .aadlen = 0,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .ivlen = 12,
	 {0x53,0x0f,0x8a,0xfb,0xc7,0x45,0x36,0xb9,0xa9,0x63,0xb4,0xf1,0xc4,0xcb,0x73,0x8b,},
	 .taglen = 16,
	 {},
	 {},
	 .num_chunks = 0,
	},
	/* GCM data - 9 - AES 256 - Test Case 14 */
	{{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .keylen = 32,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .datalen = 16,
	 {},
	 .aadlen = 0,
	 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,},
	 .ivlen = 12,
	 {0xd0,0xd1,0xc8,0xa7,0x99,0x99,0x6b,0xf0,0x26,0x5b,0x98,0xb5,0xd4,0x8a,0xb9,0x19,},
	 .taglen = 16,
	 {0xce,0xa7,0x40,0x3d,0x4d,0x60,0x6b,0x6e,0x07,0x4e,0xc5,0xd3,0xba,0xf3,0x9d,0x18,},
	 {16},
	 .num_chunks = 1,
	},
	/* GCM data - 10 - AES 256 - Test Case 15 */
	{{0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
	  0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,},
	 .keylen = 32,
	 {0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
	  0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
	  0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
	  0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,0x1a,0xaf,0xd2,0x55,},
	 .datalen = 64,
	 {},
	 .aadlen = 0,
	 {0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,},
	 .ivlen = 12,
	 {0xb0,0x94,0xda,0xc5,0xd9,0x34,0x71,0xbd,0xec,0x1a,0x50,0x22,0x70,0xe3,0xcc,0x6c,},
	 .taglen = 16,
	 {0x52,0x2d,0xc1,0xf0,0x99,0x56,0x7d,0x07,0xf4,0x7f,0x37,0xa3,0x2a,0x84,0x42,0x7d,
	  0x64,0x3a,0x8c,0xdc,0xbf,0xe5,0xc0,0xc9,0x75,0x98,0xa2,0xbd,0x25,0x55,0xd1,0xaa,
	  0x8c,0xb0,0x8e,0x48,0x59,0x0d,0xbb,0x3d,0xa7,0xb0,0x8b,0x10,0x56,0x82,0x88,0x38,
	  0xc5,0xf6,0x1e,0x63,0x93,0xba,0x7a,0x0a,0xbc,0xc9,0xf6,0x62,0x89,0x80,0x15,0xad,},
	 {16,16,16,16},
	 .num_chunks = 4,
	},
	/* GCM data - 11 */
	{{0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
	  0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,},
	 .keylen = 32,
	 {0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
	  0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
	  0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
	  0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,},
	 .datalen = 60,
	 {0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
	  0xab,0xad,0xda,0xd2,},
	  .aadlen = 20,
	 {0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,},
	 .ivlen = 12,
	 {0x76,0xfc,0x6e,0xce,0x0f,0x4e,0x17,0x68,0xcd,0xdf,0x88,0x53,0xbb,0x2d,0x55,0x1b,},
	 .taglen = 16,
	 {0x52,0x2d,0xc1,0xf0,0x99,0x56,0x7d,0x07,0xf4,0x7f,0x37,0xa3,0x2a,0x84,0x42,0x7d,
	  0x64,0x3a,0x8c,0xdc,0xbf,0xe5,0xc0,0xc9,0x75,0x98,0xa2,0xbd,0x25,0x55,0xd1,0xaa,
	  0x8c,0xb0,0x8e,0x48,0x59,0x0d,0xbb,0x3d,0xa7,0xb0,0x8b,0x10,0x56,0x82,0x88,0x38,
	  0xc5,0xf6,0x1e,0x63,0x93,0xba,0x7a,0x0a,0xbc,0xc9,0xf6,0x62,},
	 {16,16,16,12},
	 .num_chunks = 4,
	},
	/**
	 * GCM data - 12
	 *
	 * [Keylen = 128]
	 * [IVlen = 96]
	 * [PTlen = 0]
	 * [AADlen = 0]
	 * [Taglen = 128]
	 */
	{{0x11,0x75,0x4c,0xd7,0x2a,0xec,0x30,0x9b,0xf5,0x2f,0x76,0x87,0x21,0x2e,0x89,0x57,},
	 .keylen = 16,
	 {},
	 .datalen = 0,
	 {},
	 .aadlen = 0,
	 {0x3c,0x81,0x9d,0x9a,0x9b,0xed,0x08,0x76,0x15,0x03,0x0b,0x65,},
	 .ivlen = 12,
	 {0x25,0x03,0x27,0xc6,0x74,0xaa,0xf4,0x77,0xae,0xf2,0x67,0x57,0x48,0xcf,0x69,0x71},
	 .taglen = 16,
	 {},
	 {},
	 .num_chunks = 0,
	},
	/**
	 * GCM data - 13
	 *
	 * [Keylen = 128]
	 * [IVlen = 96]
	 * [PTlen = 0]
	 * [AADlen = 0]
	 * [Taglen = 120]
	 */
	{{0x27,0x2f,0x16,0xed,0xb8,0x1a,0x7a,0xbb,0xea,0x88,0x73,0x57,0xa5,0x8c,0x19,0x17,},
	 .keylen = 16,
	 {},
	 .datalen = 0,
	 {},
	 .aadlen = 0,
	 {0x79,0x4e,0xc5,0x88,0x17,0x6c,0x70,0x3d,0x3d,0x2a,0x7a,0x07,},
	 .ivlen = 12,
	 {0xb6,0xe6,0xf1,0x97,0x16,0x8f,0x50,0x49,0xae,0xda,0x32,0xda,0xfb,0xda,0xeb,},
	 .taglen = 15,
	 {},
	 {},
	 .num_chunks = 0,
	},
	/**
	 * GCM data - 14
	 *
	 * [Keylen = 128]
	 * [IVlen = 96]
	 * [PTlen = 0]
	 * [AADlen = 0]
	 * [Taglen = 32]
	 */
	{{0x88,0x4e,0xb6,0x36,0xa2,0x77,0xec,0xc4,0xc3,0x4a,0x43,0xcc,0x59,0x07,0x5c,0xa9,},
	 .keylen = 16,
	 {},
	 .datalen = 0,
	 {},
	 .aadlen = 0,
	 {0x44,0xe8,0x01,0x9f,0x40,0x26,0xff,0x0c,0x1f,0x8b,0x3e,0x2f,},
	 .ivlen = 12,
	 {0x05,0x4d,0x06,0x3f,},
	 .taglen = 4,
	 {},
	 {},
	 .num_chunks = 0,
	},
	/**
	 * GCM data - 15
	 *
	 * [Keylen = 128]
	 * [IVlen = 1024]
	 * [PTlen = 104]
	 * [AADlen = 128]
	 * [Taglen = 64]
	 */
	{{0x7e,0x95,0x06,0x6b,0x60,0x09,0x3f,0x66,0x17,0x54,0x93,0xd1,0x41,0x35,0x9d,0xbd,},
	 .keylen = 16,
	 {0x5e,0xa1,0x31,0x2e,0x26,0xc9,0x5b,0xcf,0x00,0x5b,0x61,0x74,0x23,},
	 .datalen = 13,
	 {0x66,0x0c,0x28,0xa4,0x60,0xfa,0x93,0xe1,0x12,0xaa,0xc6,0xce,0xb5,0x4a,0x80,0xa2,},
	 .aadlen = 16,
	 {0x40,0x57,0xda,0x04,0xc7,0x73,0x36,0x1c,0x33,0xbe,0x7f,0x10,0xd7,0xba,0x70,0x8b,
	  0x22,0x78,0x50,0x3f,0xd7,0xb0,0xa6,0xdd,0x13,0x0a,0x96,0x29,0x52,0xb8,0x88,0x7d,
	  0x6a,0x41,0x20,0x74,0xc1,0x57,0x2e,0xb0,0xc5,0x3e,0xdf,0x81,0xee,0x70,0x1c,0xab,
	  0xc6,0x05,0x52,0xac,0xeb,0x0f,0x66,0x26,0x97,0xd3,0xb2,0xac,0xc0,0x37,0xea,0xb9,
	  0x44,0x52,0x42,0xbf,0xf4,0x49,0x66,0x06,0xb8,0xcf,0xbf,0x2d,0x3c,0x72,0x87,0x4b,
	  0x76,0x9b,0x9b,0x63,0x23,0x4b,0x64,0xd4,0x29,0x82,0x9f,0x46,0x73,0x05,0xac,0xab,
	  0x4a,0xe8,0xd4,0x5c,0x8f,0x7c,0x4f,0x5b,0x57,0x71,0xcb,0x7c,0xbd,0xcc,0xc8,0xc7,
	  0x27,0x3a,0x4a,0x20,0x38,0x46,0x4f,0xad,0xfd,0xf7,0x33,0xb6,0x31,0x17,0x90,0x17,},
	 .ivlen = 128,
	 {0x4f,0xd9,0x46,0x71,0xab,0xde,0x61,0x6e,},
	 .taglen = 8,
	 {0x55,0x17,0xfb,0xa3,0x76,0xab,0x16,0xc7,0xe2,0xad,0x16,0xc1,0xe2,},
	 {13},
	 .num_chunks = 1,
	},
	/**
	 * GCM data - 16
	 *
	 * [Keylen = 128]
	 * [IVlen = 8]
	 * [PTlen = 0]
	 * [AADlen = 160]
	 * [Taglen = 104]
	 */
	{{0xf8,0x41,0x3d,0x62,0x4e,0x64,0xc2,0x08,0x1b,0xc7,0xc2,0x42,0x37,0xed,0x8a,0x8d,},
	 .keylen = 16,
	 {},
	 .datalen = 0,
	 {0x9a,0x24,0xf2,0xb8,0xed,0xe8,0x25,0xe7,0xd3,0x13,0xd3,0x92,0x3d,0x21,0x71,0x34,
	  0xe2,0x04,0x64,0x8e,},
	 .aadlen = 20,
	 {0x53,},
	 .ivlen = 1,
	 {0xbf,0x0d,0x41,0x82,0xf1,0x2e,0x33,0x99,0x9e,0x8f,0x0e,0xf4,0xd4,},
	 .taglen = 13,
	 {},
	 {},
	 .num_chunks = 0,
	},
	/**
	 * GCM data - 17
	 *
	 * [Keylen = 128]
	 * [IVlen = 8]
	 * [PTlen = 104]
	 * [AADlen = 0]
	 * [Taglen = 64]
	 */
	{{0xc8,0xed,0x6f,0x0c,0x45,0xa9,0xcb,0xff,0x8d,0x44,0x76,0x02,0xa0,0x7c,0x53,0xcb,},
	 .keylen = 16,
	 {0xc0,0xaf,0xd5,0xca,0xe5,0x5d,0xf1,0xd3,0x4c,0x5f,0xcc,0xb2,0xa4,},
	 .datalen = 13,
	 {},
	 .aadlen = 0,
	 {0xb5,},
	 .ivlen = 1,
	 {0x9b,0xc7,0x8d,0x50,0x93,0x7a,0x38,0x57,},
	 .taglen = 8,
	 {0xf2,0x2c,0x87,0x72,0xaf,0x42,0x17,0x9e,0xc1,0xb3,0xfe,0x08,0x69,},
	 {13},
	 .num_chunks = 1,
	},
	/**
	 * GCM data - 18
	 *
	 * [Keylen = 128]
	 * [IVlen = 1024]
	 * [PTlen = 128]
	 * [AADlen = 720]
	 * [Taglen = 104]
	 */
	{{0x67,0x8d,0xb3,0xb4,0x19,0xa5,0xfd,0xb0,0x9d,0x24,0x2d,0x10,0x6c,0xdb,0xfd,0xd8,},
	 .keylen = 16,
	 {0x94,0xca,0xf0,0x0f,0xa4,0xcd,0xba,0x64,0xc0,0x3f,0x15,0x8f,0xa2,0x5a,0x64,0xc7,},
	 .datalen = 16,
	 {0x62,0xfa,0x1b,0xe1,0x03,0x69,0xe1,0x96,0xe4,0x09,0xd0,0x79,0xce,0x92,0xd0,0x21,
	  0x46,0x76,0xaf,0x43,0x43,0x08,0xce,0x73,0x1c,0x98,0xcc,0xaa,0x08,0x1d,0x4d,0x46,
	  0x0a,0x78,0x5c,0xcb,0xed,0xfa,0x5e,0x90,0x48,0x21,0x4f,0x2a,0x49,0x0d,0x5d,0x62,
	  0x49,0x50,0xc2,0xf8,0x76,0x85,0x8d,0xe3,0x44,0xbe,0xce,0x62,0x91,0x18,0x48,0x25,
	  0x04,0x02,0xa5,0xf8,0x54,0xfc,0xe8,0x9c,0x47,0x39,0x79,0x39,0xc3,0xac,0x82,0xe3,
	  0x5b,0xd7,0x20,0xc6,0x7a,0xa8,0x65,0x9f,0x54,0x63,},
	 .aadlen = 90,
	 {0x8b,0x75,0xc4,0x66,0x77,0x1b,0x86,0xde,0xa8,0xa0,0x23,0x4b,0x59,0xc3,0xf9,0xd3,
	  0x76,0xe7,0x86,0x21,0x0e,0x94,0x5c,0x75,0x1b,0x9f,0x7a,0xc9,0x35,0x4d,0x4b,0x69,
	  0xf4,0xa2,0x88,0x86,0xd9,0x0a,0xb2,0xa0,0x77,0xf5,0x86,0x02,0x0f,0x8c,0x27,0x7d,
	  0x1b,0x6d,0x52,0x56,0xc9,0x8b,0x17,0xf8,0x4c,0x55,0x38,0x62,0x07,0xb6,0x9d,0x4c,
	  0xd3,0xd1,0x4e,0x83,0xbc,0x22,0xd7,0x4a,0xfb,0xfc,0xca,0x47,0x08,0x74,0xeb,0x42,
	  0xb9,0xf1,0xc6,0xa7,0xf8,0xaa,0x1c,0x36,0x30,0x33,0x95,0x04,0x97,0x71,0x5a,0xb6,
	  0x93,0xb1,0xdb,0xca,0xf7,0x7f,0x00,0xad,0x45,0x80,0x2c,0x27,0x03,0x74,0x1c,0x2d,
	  0x9e,0x6e,0x18,0x69,0x1b,0xf9,0x2d,0x28,0x46,0xbd,0xa1,0xc4,0xd5,0x7a,0x31,0xf0,},
	 .ivlen = 128,
	 {0xc4,0x80,0x65,0x98,0x81,0x7e,0x31,0xa0,0xe3,0x3a,0x96,0x79,0x66,},
	 .taglen = 13,
	 {0xb0,0xf8,0x1c,0xbc,0xc7,0x55,0x46,0xc6,0xcf,0xc3,0x3a,0x28,0x86,0x7b,0x26,0x57,},
	 {16},
	 .num_chunks = 1,
	},
	/**
	 * GCM data - 19
	 *
	 * [Keylen = 128]
	 * [IVlen = 8]
	 * [PTlen = 408]
	 * [AADlen = 720]
	 * [Taglen = 104]
	 */
	{{0xc1,0xff,0x59,0x80,0x55,0xf8,0xb3,0xb8,0x3f,0xdf,0xd7,0xba,0xe5,0x7e,0x10,0x18,},
	 .keylen = 16,
	 {0x97,0xc4,0x68,0xcf,0x16,0xfe,0x00,0x9a,0xb8,0x1c,0x32,0xfc,0x91,0x1a,0xd9,0xda,
	  0xe9,0x68,0xf5,0x9e,0x2f,0x77,0x34,0xbb,0xbc,0x8e,0x63,0xf4,0xb1,0x8a,0x5c,0x61,
	  0x36,0x7d,0x30,0x0e,0xd2,0x45,0xfc,0x45,0xfe,0x08,0x6d,0x6b,0xe5,0x6b,0x35,0xf6,
	  0x9d,0xe4,0x89,},
	 .datalen = 51,
	 {0xd9,0x94,0x86,0xf5,0xc7,0xf2,0xc5,0x34,0x11,0x82,0x7b,0xa3,0xb5,0x14,0x55,0x66,
	  0x17,0xf7,0x9b,0x16,0x73,0x6f,0x93,0x70,0x7f,0xdb,0x16,0xa2,0x36,0xd4,0x1e,0x84,
	  0xf6,0x8d,0x55,0x5d,0x11,0x6b,0x25,0x42,0xbc,0xf0,0x27,0xa0,0x4c,0x70,0x00,0xbc,
	  0x7e,0xd5,0xb1,0xf2,0x83,0x92,0x2d,0x0c,0x1f,0x5d,0xbf,0x2b,0x72,0xf4,0x58,0x2f,
	  0xac,0x0e,0x3f,0x20,0x1d,0x2d,0x08,0x0f,0x27,0xfa,0xcb,0x06,0xc5,0x6b,0x9c,0x93,
	  0x84,0x96,0x13,0x1c,0x3f,0xb6,0x49,0xf5,0xf4,0x64,},
	 .aadlen = 90,
	 {0x0a,},
	 .ivlen = 1,
	 {0xc2,0x9d,0xdd,0xc9,0x48,0x00,0xfb,0xf7,0xae,0x43,0x0d,0x59,0x01,},
	 .taglen = 13,
	 {0x4e,0x0e,0x6a,0x45,0x39,0xf9,0x4d,0xc1,0x9c,0xaa,0x15,0xd1,0x4f,0x8f,0xb7,0x2f,
	  0x0b,0xc8,0xa4,0xd5,0xea,0xf0,0xe1,0x73,0xfa,0x04,0x0d,0xe7,0x8c,0x06,0x39,0x2c,
	  0xb4,0x51,0x70,0xde,0xef,0xd3,0x8e,0x02,0x53,0xdd,0xb7,0xda,0xc9,0x3b,0xc4,0x9e,
	  0xd4,0x19,0x04,},
	 {16,32,3},
	 .num_chunks = 3,
	},
	/**
	 * GCM data - 20 : same as 19, but different chunks
	 *
	 * [Keylen = 128]
	 * [IVlen = 8]
	 * [PTlen = 408]
	 * [AADlen = 720]
	 * [Taglen = 104]
	 */
	{{0xc1,0xff,0x59,0x80,0x55,0xf8,0xb3,0xb8,0x3f,0xdf,0xd7,0xba,0xe5,0x7e,0x10,0x18,},
	 .keylen = 16,
	 {0x97,0xc4,0x68,0xcf,0x16,0xfe,0x00,0x9a,0xb8,0x1c,0x32,0xfc,0x91,0x1a,0xd9,0xda,
	  0xe9,0x68,0xf5,0x9e,0x2f,0x77,0x34,0xbb,0xbc,0x8e,0x63,0xf4,0xb1,0x8a,0x5c,0x61,
	  0x36,0x7d,0x30,0x0e,0xd2,0x45,0xfc,0x45,0xfe,0x08,0x6d,0x6b,0xe5,0x6b,0x35,0xf6,
	  0x9d,0xe4,0x89,},
	 .datalen = 51,
	 {0xd9,0x94,0x86,0xf5,0xc7,0xf2,0xc5,0x34,0x11,0x82,0x7b,0xa3,0xb5,0x14,0x55,0x66,
	  0x17,0xf7,0x9b,0x16,0x73,0x6f,0x93,0x70,0x7f,0xdb,0x16,0xa2,0x36,0xd4,0x1e,0x84,
	  0xf6,0x8d,0x55,0x5d,0x11,0x6b,0x25,0x42,0xbc,0xf0,0x27,0xa0,0x4c,0x70,0x00,0xbc,
	  0x7e,0xd5,0xb1,0xf2,0x83,0x92,0x2d,0x0c,0x1f,0x5d,0xbf,0x2b,0x72,0xf4,0x58,0x2f,
	  0xac,0x0e,0x3f,0x20,0x1d,0x2d,0x08,0x0f,0x27,0xfa,0xcb,0x06,0xc5,0x6b,0x9c,0x93,
	  0x84,0x96,0x13,0x1c,0x3f,0xb6,0x49,0xf5,0xf4,0x64,},
	 .aadlen = 90,
	 {0x0a,},
	 .ivlen = 1,
	 {0xc2,0x9d,0xdd,0xc9,0x48,0x00,0xfb,0xf7,0xae,0x43,0x0d,0x59,0x01,},
	 .taglen = 13,
	 {0x4e,0x0e,0x6a,0x45,0x39,0xf9,0x4d,0xc1,0x9c,0xaa,0x15,0xd1,0x4f,0x8f,0xb7,0x2f,
	  0x0b,0xc8,0xa4,0xd5,0xea,0xf0,0xe1,0x73,0xfa,0x04,0x0d,0xe7,0x8c,0x06,0x39,0x2c,
	  0xb4,0x51,0x70,0xde,0xef,0xd3,0x8e,0x02,0x53,0xdd,0xb7,0xda,0xc9,0x3b,0xc4,0x9e,
	  0xd4,0x19,0x04,},
	 {16,35},
	 .num_chunks = 2,
	},
	/* GCM data - 21 : same as 11, but different chunks */
	{{0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
	  0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,},
	 .keylen = 32,
	 {0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
	  0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
	  0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
	  0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,},
	 .datalen = 60,
	 {0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
	  0xab,0xad,0xda,0xd2,},
	 .aadlen = 20,
	 {0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,},
	 .ivlen = 12,
	 {0x76,0xfc,0x6e,0xce,0x0f,0x4e,0x17,0x68,0xcd,0xdf,0x88,0x53,0xbb,0x2d,0x55,0x1b,},
	 .taglen = 16,
	 {0x52,0x2d,0xc1,0xf0,0x99,0x56,0x7d,0x07,0xf4,0x7f,0x37,0xa3,0x2a,0x84,0x42,0x7d,
	  0x64,0x3a,0x8c,0xdc,0xbf,0xe5,0xc0,0xc9,0x75,0x98,0xa2,0xbd,0x25,0x55,0xd1,0xaa,
	  0x8c,0xb0,0x8e,0x48,0x59,0x0d,0xbb,0x3d,0xa7,0xb0,0x8b,0x10,0x56,0x82,0x88,0x38,
	  0xc5,0xf6,0x1e,0x63,0x93,0xba,0x7a,0x0a,0xbc,0xc9,0xf6,0x62,},
	 {32,28},
	 .num_chunks = 2,
	},
};

static inline void dump_gcm_data(unsigned char *iv, unsigned int iv_length,
		   unsigned char *aad, unsigned int aad_length,
		   unsigned char *key, unsigned int key_length,
		   unsigned char *input_data, unsigned int data_length,
		   unsigned char *output_data,
		   unsigned char *t, unsigned int t_size)
{
	VV_(printf("IV \n"));
	dump_array(iv, iv_length);
	VV_(printf("AAD \n"));
	dump_array(aad, aad_length);
	VV_(printf("Key \n"));
	dump_array(key, key_length);
	VV_(printf("Input Data\n"));
	dump_array(input_data, data_length);
	VV_(printf("Output Data\n"));
	dump_array(output_data, data_length);
	VV_(printf("T\n"));
	dump_array(t, t_size);
}

#endif /* GCM_KATS_H */
