/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2016-2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::string

Description
    A class for handling character strings derived from std::string.

    Strings may contain any characters and therefore are delimited by quotes
    for IO : "any list of characters".

    Used as a base class for word and fileName.

See also
    Foam::findEtcFile() for information about the site/user OpenFOAM
    configuration directory

SourceFiles
    stringI.H
    string.C
    stringIO.C
    stringTemplates.C

\*---------------------------------------------------------------------------*/

#ifndef string_H
#define string_H

#include "char.H"
#include "Hasher.H"
#include <string>
#include <cstring>
#include <cstdlib>

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward Declarations
class word;
class wordRe;
class Istream;
class Ostream;

/*---------------------------------------------------------------------------*\
                           Class string Declaration
\*---------------------------------------------------------------------------*/

class string
:
    public std::string
{
protected:

    // Protected Member Functions

        //- Find position of a file extension dot, return npos on failure.
        //  A wrapped version of find_last_of("./") with additional logic.
        inline static std::string::size_type find_ext(const std::string& str);

        //- Find position of a file extension dot, return npos on failure.
        //  A wrapped version of find_last_of("./") with additional logic.
        inline std::string::size_type find_ext() const;

        //- A printf-style formatter for a primitive.
        template<class PrimitiveType>
        static std::string::size_type string_printf
        (
            std::string& output,
            const char* fmt,
            const PrimitiveType& val
        );

        //- A printf-style formatter for a primitive.
        template<class PrimitiveType>
        static std::string::size_type string_printf
        (
            std::string& output,
            const std::string& fmt,
            const PrimitiveType& val
        );

        //- Return file name extension (part after last .)
        word ext() const;

        //- Append a '.' and the ending.
        //  The '.' and ending will not be added when the ending is empty,
        //  or when the object was or ended with a '/'.
        //
        //  \return True if append occurred.
        bool ext(const word& ending);

        //- Return true if it contains a '/' character
        inline bool hasPath() const;

        //- Return true if it has an extension or simply ends with a '.'
        inline bool hasExt() const;

        //- Return true if the extension is the same as the given ending.
        bool hasExt(const word& ending) const;

        //- Return true if the extension matches the given ending.
        bool hasExt(const wordRe& ending) const;

        //- Remove extension, returning true if string changed.
        inline bool removeExt();

        //- Remove leading path, returning true if string changed.
        inline bool removePath();


public:

    // Static Data Members

        //- The type name "string"
        static const char* const typeName;

        //- The debug flag
        static int debug;

        //- An empty string
        static const string null;


    //- Hashing function for string and derived string classes
    struct hash
    {
        inline unsigned operator()
        (
            const std::string& str,
            unsigned seed = 0
        ) const;
    };


    // Constructors

        //- Construct null
        string() = default;

        //- Copy construct from std::string
        inline string(const std::string& str);

        //- Move construct from std::string
        inline string(std::string&& str);

        //- Construct as copy of character array
        inline string(const char* str);

        //- Construct as copy with a maximum number of characters
        inline string(const char* str, const size_type len);

        //- Construct from a single character
        inline explicit string(const char c);

        //- Construct fill copies of a single character
        inline string(const size_type len, const char c);

        //- Construct from Istream
        explicit string(Istream& is);


    // Static Member Functions

        //- Does the string contain valid characters only?
        template<class String>
        static inline bool valid(const std::string& str);

        //- Does this string contain meta-characters?
        //  The meta characters can be optionally quoted.
        template<class String>
        static inline bool meta(const std::string& str, const char quote='\\');

        //- Strip invalid characters from the given string
        template<class String>
        static inline bool stripInvalid(std::string& str);

        //- Return a valid String from the given string
        template<class String>
        static inline String validate(const std::string& str);

        //- Return a String with quoted meta-characters from the given string
        template<class String>
        static inline string quotemeta
        (
            const std::string& str,
            const char quote = '\\'
        );


    // Member Functions

        //- Test for equality.
        //  \return True when strings match literally.
        inline bool match(const std::string& text) const;

        //- Avoid masking the normal std::string replace
        using std::string::replace;

        //- Replace first occurrence of sub-string s1 with s2,
        //- beginning at pos
        string& replace
        (
            const std::string& s1,
            const std::string& s2,
            size_type pos = 0
        );

        //- Replace all occurrences of sub-string s1 with s2,
        //- beginning at pos in the string.
        //  This is a no-op if s1 is empty.
        string& replaceAll
        (
            const std::string& s1,
            const std::string& s2,
            size_type pos = 0
        );

        //- Inplace expand initial tags, tildes, and all occurrences of
        //- environment variables as per stringOps::expand
        //
        //  Any unknown entries are removed silently if allowEmpty is true
        //  \sa
        //  Foam::findEtcFile
        string& expand(const bool allowEmpty = false);

        //- Remove repeated characters
        //  \return True if string changed
        bool removeRepeated(const char character);

        //- Remove the given text from the start of the string.
        //  \return True if the removal occurred
        bool removeStart(const std::string& text);

        //- Remove leading character, unless string is a single character
        //  \return True if the removal occurred
        bool removeStart(const char c);

        //- Remove the given text from the end of the string.
        //  \return True if the removal occurred
        bool removeEnd(const std::string& text);

        //- Remove trailing character, unless string is a single character
        //  \return True if the removal occurred
        bool removeEnd(const char c);


    // Editing

        //- Swap contents. Self-swapping is a no-op.
        inline void swap(std::string& str);


    // Member Operators

        //- Test for equality. Allows use as a predicate.
        //  \return True when strings match literally.
        inline bool operator()(const std::string& text) const;


    // Housekeeping

        //- True if string starts with the given prefix (cf. C++20)
        bool starts_with(const std::string& s) const
        {
            return (size() >= s.size() && !compare(0, s.size(), s));
        }

        //- True if string starts with the given character (cf. C++20)
        bool starts_with(const char c) const
        {
            return (!empty() && front() == c);
        }

        //- True if string ends with the given suffix (cf. C++20)
        bool ends_with(const std::string& s) const
        {
            return (size() >= s.size() && !compare(size()-s.size(), npos, s));
        }

        //- True if string ends with the given character (cf. C++20)
        bool ends_with(const char c) const
        {
            return (!empty() && back() == c);
        }

        //- Count the number of occurrences of the specified character
        //- in the string
        //  Partially deprecated (NOV-2017) in favour of stringOps::count
        size_type count(const char c) const;

        //- Deprecated(2019-11)
        //  \deprecated(2019-11) use starts_with instead
        bool startsWith(const std::string& s) const { return starts_with(s); }

        //- Deprecated(2019-11)
        //  \deprecated(2019-11) use ends_with instead
        bool endsWith(const std::string& s) const { return ends_with(s); }

        //- Deprecated(2019-11)
        //  \deprecated(2019-11) use removeEnd instead
        bool removeTrailing(const char c) { return removeEnd(c); }
};


// IOstream Operators

//- Read operator
Istream& operator>>(Istream& is, string& val);

//- Write operator
Ostream& operator<<(Ostream& os, const string& val);

//- Write operator
Ostream& operator<<(Ostream& os, const std::string& val);


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "stringI.H"

#ifdef NoRepository
    #include "stringTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
