/*
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test 6948803
 * @summary CertPath validation regression caused by SHA1 replacement root
 *  and MD2 disable feature
 */

import java.io.ByteArrayInputStream;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.List;
import sun.security.validator.Validator;

public class CertReplace {

    public static void main(String[] args) throws Exception {

        KeyStore ks = KeyStore.getInstance("JKS");
        ks.load(new ByteArrayInputStream(ca), "changeit".toCharArray());
        Validator v = Validator.getInstance
            (Validator.TYPE_PKIX, Validator.VAR_GENERIC, ks);
        X509Certificate[] chain = createPath();
        for (X509Certificate c: v.validate(chain)) {
            System.out.println("   " + c.getSubjectX500Principal() +
                    " issued by " + c.getIssuerX500Principal());
        }
    }

    public static X509Certificate[] createPath() throws Exception {
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        List list = new ArrayList();
        for (Certificate c: cf.generateCertificates(
                new ByteArrayInputStream(certs.getBytes()))) {
            list.add((X509Certificate)c);
        }
        return (X509Certificate[]) list.toArray(new X509Certificate[0]);
    }

    // The following data are generated using JDK 7 keytool:
    // ca is a JKS keystore including ca's new-refreshed cert whose
    // signature is SHA256withRSA.
    // certs includes 3 certs (user, int, ca) as the cert chain
    // for user, where ca's signature is MD2withRSA,
    // The two certs for ca have the same public keys.

    private static final byte[] ca = {
        (byte)0xFE, (byte)0xED, (byte)0xFE, (byte)0xED,
        (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x02,
        (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x01,
        (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x01,
        (byte)0x00, (byte)0x02, (byte)0x63, (byte)0x61,
        (byte)0x00, (byte)0x00, (byte)0x01, (byte)0x28,
        (byte)0xBA, (byte)0x0C, (byte)0x6D, (byte)0x65,
        (byte)0x00, (byte)0x00, (byte)0x05, (byte)0x01,
        (byte)0x30, (byte)0x82, (byte)0x04, (byte)0xFD,
        (byte)0x30, (byte)0x0E, (byte)0x06, (byte)0x0A,
        (byte)0x2B, (byte)0x06, (byte)0x01, (byte)0x04,
        (byte)0x01, (byte)0x2A, (byte)0x02, (byte)0x11,
        (byte)0x01, (byte)0x01, (byte)0x05, (byte)0x00,
        (byte)0x04, (byte)0x82, (byte)0x04, (byte)0xE9,
        (byte)0x86, (byte)0x0B, (byte)0x08, (byte)0x21,
        (byte)0xF7, (byte)0xB7, (byte)0x4F, (byte)0x64,
        (byte)0x25, (byte)0x06, (byte)0x4D, (byte)0xF3,
        (byte)0xCC, (byte)0x75, (byte)0xBE, (byte)0x9C,
        (byte)0x46, (byte)0x14, (byte)0x71, (byte)0x83,
        (byte)0xDC, (byte)0xE4, (byte)0xD3, (byte)0x24,
        (byte)0xFD, (byte)0x25, (byte)0xFE, (byte)0x99,
        (byte)0x94, (byte)0x61, (byte)0x7A, (byte)0xC2,
        (byte)0x64, (byte)0x32, (byte)0x4D, (byte)0x94,
        (byte)0x3A, (byte)0x3C, (byte)0x0C, (byte)0xF7,
        (byte)0x24, (byte)0xD4, (byte)0x66, (byte)0xED,
        (byte)0xAD, (byte)0x02, (byte)0xB0, (byte)0xAB,
        (byte)0x7C, (byte)0x78, (byte)0x81, (byte)0xD1,
        (byte)0x44, (byte)0xD0, (byte)0xD6, (byte)0x7A,
        (byte)0xF5, (byte)0xE1, (byte)0x9E, (byte)0xBA,
        (byte)0xDC, (byte)0x0D, (byte)0x22, (byte)0x90,
        (byte)0x26, (byte)0xE6, (byte)0xE9, (byte)0xB4,
        (byte)0x74, (byte)0xA8, (byte)0x4C, (byte)0xBA,
        (byte)0x47, (byte)0x67, (byte)0x8E, (byte)0x70,
        (byte)0xA3, (byte)0xAB, (byte)0xED, (byte)0xEC,
        (byte)0x22, (byte)0xF7, (byte)0x43, (byte)0xDD,
        (byte)0xE9, (byte)0xE9, (byte)0x05, (byte)0x08,
        (byte)0x17, (byte)0xA1, (byte)0xB7, (byte)0x64,
        (byte)0x1A, (byte)0xBF, (byte)0x7C, (byte)0xA5,
        (byte)0xF6, (byte)0x61, (byte)0xB1, (byte)0xE9,
        (byte)0x2D, (byte)0x6F, (byte)0xCA, (byte)0x4A,
        (byte)0x20, (byte)0x69, (byte)0x60, (byte)0x67,
        (byte)0xC8, (byte)0xB5, (byte)0x2A, (byte)0xF5,
        (byte)0x4C, (byte)0x56, (byte)0x9C, (byte)0xFD,
        (byte)0xC1, (byte)0xF3, (byte)0x4C, (byte)0x18,
        (byte)0x12, (byte)0x09, (byte)0xC4, (byte)0x13,
        (byte)0xE7, (byte)0xB7, (byte)0xE4, (byte)0xA9,
        (byte)0x5B, (byte)0xF4, (byte)0x25, (byte)0x64,
        (byte)0xC4, (byte)0x8E, (byte)0x4C, (byte)0xA4,
        (byte)0xA6, (byte)0xC2, (byte)0x0B, (byte)0x96,
        (byte)0xDB, (byte)0x71, (byte)0x4C, (byte)0xCB,
        (byte)0xF8, (byte)0x94, (byte)0x7C, (byte)0x10,
        (byte)0xE1, (byte)0xD5, (byte)0xB9, (byte)0x62,
        (byte)0xA4, (byte)0x12, (byte)0x48, (byte)0x81,
        (byte)0x4E, (byte)0x14, (byte)0xA9, (byte)0xD8,
        (byte)0xA5, (byte)0x9D, (byte)0xAC, (byte)0x0A,
        (byte)0xC1, (byte)0x90, (byte)0x06, (byte)0xD8,
        (byte)0x4F, (byte)0xB0, (byte)0x53, (byte)0xF1,
        (byte)0x34, (byte)0xA3, (byte)0x4B, (byte)0xA6,
        (byte)0x06, (byte)0x3D, (byte)0x5C, (byte)0xAE,
        (byte)0x41, (byte)0x34, (byte)0xD1, (byte)0x3B,
        (byte)0x7A, (byte)0xF4, (byte)0x38, (byte)0x04,
        (byte)0xB9, (byte)0x18, (byte)0xF2, (byte)0xF0,
        (byte)0xFB, (byte)0xB2, (byte)0x50, (byte)0x54,
        (byte)0xD4, (byte)0x05, (byte)0x89, (byte)0xD7,
        (byte)0x09, (byte)0xB5, (byte)0xC8, (byte)0x74,
        (byte)0xF0, (byte)0x25, (byte)0xFF, (byte)0x9D,
        (byte)0xE2, (byte)0x96, (byte)0x6B, (byte)0x6A,
        (byte)0x72, (byte)0x0B, (byte)0x19, (byte)0x07,
        (byte)0x9B, (byte)0xBA, (byte)0xE7, (byte)0x8A,
        (byte)0x07, (byte)0x23, (byte)0xFB, (byte)0xBF,
        (byte)0x95, (byte)0xC6, (byte)0x6D, (byte)0xA2,
        (byte)0xF1, (byte)0x78, (byte)0x90, (byte)0x60,
        (byte)0x20, (byte)0x8C, (byte)0xFF, (byte)0x5E,
        (byte)0x77, (byte)0x67, (byte)0x08, (byte)0x27,
        (byte)0x84, (byte)0x7F, (byte)0x1B, (byte)0x30,
        (byte)0xB7, (byte)0xBC, (byte)0x4A, (byte)0x05,
        (byte)0x82, (byte)0x57, (byte)0x14, (byte)0x6F,
        (byte)0xE0, (byte)0x27, (byte)0xD3, (byte)0x79,
        (byte)0x1D, (byte)0x65, (byte)0xF9, (byte)0x22,
        (byte)0x01, (byte)0x52, (byte)0xC1, (byte)0x62,
        (byte)0xB7, (byte)0xDB, (byte)0xB7, (byte)0x92,
        (byte)0x32, (byte)0x3E, (byte)0x71, (byte)0xB7,
        (byte)0x98, (byte)0x1B, (byte)0x40, (byte)0x0B,
        (byte)0x16, (byte)0x0A, (byte)0x23, (byte)0xBC,
        (byte)0xB0, (byte)0x5C, (byte)0x29, (byte)0x49,
        (byte)0x61, (byte)0x26, (byte)0x48, (byte)0x75,
        (byte)0x8D, (byte)0x90, (byte)0xFF, (byte)0x67,
        (byte)0xF7, (byte)0x4A, (byte)0xAC, (byte)0x72,
        (byte)0xCE, (byte)0xAC, (byte)0x9F, (byte)0xD8,
        (byte)0x9C, (byte)0x82, (byte)0x46, (byte)0xB9,
        (byte)0x54, (byte)0xC8, (byte)0x89, (byte)0xE6,
        (byte)0x4C, (byte)0xA3, (byte)0x71, (byte)0xCA,
        (byte)0xB0, (byte)0xF0, (byte)0xFF, (byte)0xEC,
        (byte)0x03, (byte)0x5F, (byte)0x59, (byte)0x6C,
        (byte)0x46, (byte)0x4C, (byte)0x43, (byte)0x60,
        (byte)0xEA, (byte)0x25, (byte)0x58, (byte)0xED,
        (byte)0x73, (byte)0xCB, (byte)0x78, (byte)0xB2,
        (byte)0x88, (byte)0x94, (byte)0x32, (byte)0xF6,
        (byte)0x94, (byte)0x2C, (byte)0x35, (byte)0x1E,
        (byte)0xDA, (byte)0x4C, (byte)0xBE, (byte)0x11,
        (byte)0xF6, (byte)0x1D, (byte)0x7E, (byte)0x76,
        (byte)0x8D, (byte)0x54, (byte)0xA1, (byte)0xB2,
        (byte)0x99, (byte)0x8C, (byte)0xF0, (byte)0x53,
        (byte)0xF5, (byte)0xF4, (byte)0x1A, (byte)0x0F,
        (byte)0x0F, (byte)0x14, (byte)0x99, (byte)0x82,
        (byte)0xA9, (byte)0x15, (byte)0xF8, (byte)0x8B,
        (byte)0x84, (byte)0x02, (byte)0x47, (byte)0x6B,
        (byte)0xA3, (byte)0xBE, (byte)0x48, (byte)0xC9,
        (byte)0x8D, (byte)0x86, (byte)0x67, (byte)0x63,
        (byte)0x84, (byte)0xCA, (byte)0x92, (byte)0x6E,
        (byte)0xE7, (byte)0xA0, (byte)0xA6, (byte)0x0A,
        (byte)0x48, (byte)0x6F, (byte)0x62, (byte)0x9F,
        (byte)0x09, (byte)0x60, (byte)0xC0, (byte)0x02,
        (byte)0x93, (byte)0x4F, (byte)0xD5, (byte)0x5D,
        (byte)0xDA, (byte)0x61, (byte)0x64, (byte)0x7C,
        (byte)0x82, (byte)0x7F, (byte)0x1A, (byte)0xAB,
        (byte)0xB6, (byte)0xE3, (byte)0x09, (byte)0xAC,
        (byte)0x90, (byte)0x08, (byte)0x10, (byte)0x1C,
        (byte)0x43, (byte)0xFB, (byte)0x0C, (byte)0xB8,
        (byte)0xF4, (byte)0x69, (byte)0xC5, (byte)0xF3,
        (byte)0x17, (byte)0xF8, (byte)0xF6, (byte)0x88,
        (byte)0xAC, (byte)0x6B, (byte)0xD6, (byte)0x29,
        (byte)0x86, (byte)0x5A, (byte)0x20, (byte)0x4F,
        (byte)0xEA, (byte)0x9B, (byte)0xC2, (byte)0x4B,
        (byte)0x60, (byte)0x1A, (byte)0xDE, (byte)0x86,
        (byte)0x4C, (byte)0xE0, (byte)0xF9, (byte)0x76,
        (byte)0x1F, (byte)0xC2, (byte)0xE9, (byte)0x9B,
        (byte)0x44, (byte)0x5C, (byte)0xE5, (byte)0x6D,
        (byte)0x25, (byte)0xDC, (byte)0x38, (byte)0x75,
        (byte)0x14, (byte)0x19, (byte)0xD1, (byte)0x59,
        (byte)0x99, (byte)0xCA, (byte)0x88, (byte)0x2F,
        (byte)0x2B, (byte)0xDA, (byte)0xA1, (byte)0x8B,
        (byte)0x54, (byte)0x7E, (byte)0xBA, (byte)0xB8,
        (byte)0x91, (byte)0xDD, (byte)0xF5, (byte)0x27,
        (byte)0x34, (byte)0xCA, (byte)0x0B, (byte)0xC1,
        (byte)0x94, (byte)0x1B, (byte)0x39, (byte)0x23,
        (byte)0x4E, (byte)0x97, (byte)0xE7, (byte)0x7F,
        (byte)0x6A, (byte)0x75, (byte)0xEB, (byte)0xAF,
        (byte)0x6E, (byte)0x47, (byte)0xE4, (byte)0x51,
        (byte)0xE4, (byte)0xB6, (byte)0x29, (byte)0xC9,
        (byte)0x82, (byte)0x3E, (byte)0x5A, (byte)0xAF,
        (byte)0x30, (byte)0x78, (byte)0x77, (byte)0xB2,
        (byte)0x26, (byte)0x6B, (byte)0xE2, (byte)0x2A,
        (byte)0xE2, (byte)0xA6, (byte)0x01, (byte)0xE5,
        (byte)0x70, (byte)0x07, (byte)0xFD, (byte)0x71,
        (byte)0xC6, (byte)0x87, (byte)0x4A, (byte)0xE9,
        (byte)0x1F, (byte)0xC3, (byte)0x56, (byte)0x49,
        (byte)0x3D, (byte)0xFD, (byte)0xCE, (byte)0xD1,
        (byte)0x7D, (byte)0x3C, (byte)0x86, (byte)0x12,
        (byte)0x12, (byte)0x09, (byte)0xA1, (byte)0x46,
        (byte)0xE7, (byte)0xEC, (byte)0x1B, (byte)0xD0,
        (byte)0xC1, (byte)0xEB, (byte)0x63, (byte)0xB1,
        (byte)0xAD, (byte)0x5E, (byte)0xEB, (byte)0x09,
        (byte)0x4A, (byte)0xDA, (byte)0x8D, (byte)0xBA,
        (byte)0xCF, (byte)0xE6, (byte)0xFE, (byte)0x5B,
        (byte)0x7C, (byte)0x59, (byte)0xCE, (byte)0x02,
        (byte)0x84, (byte)0x95, (byte)0x78, (byte)0x04,
        (byte)0xAD, (byte)0x8F, (byte)0x23, (byte)0x3B,
        (byte)0xCB, (byte)0x0A, (byte)0x14, (byte)0x85,
        (byte)0xC7, (byte)0x18, (byte)0xEB, (byte)0xDF,
        (byte)0xE0, (byte)0x7D, (byte)0xA0, (byte)0x6F,
        (byte)0x35, (byte)0xDF, (byte)0xD1, (byte)0x44,
        (byte)0xD5, (byte)0x9B, (byte)0x56, (byte)0xDA,
        (byte)0xFF, (byte)0xD4, (byte)0x13, (byte)0xB0,
        (byte)0x7E, (byte)0x95, (byte)0x2A, (byte)0xB7,
        (byte)0x5D, (byte)0x5A, (byte)0xDF, (byte)0xD7,
        (byte)0x6D, (byte)0xF2, (byte)0x4F, (byte)0xB6,
        (byte)0x13, (byte)0x06, (byte)0xC8, (byte)0x56,
        (byte)0x10, (byte)0x70, (byte)0xEA, (byte)0xE2,
        (byte)0x31, (byte)0x09, (byte)0x18, (byte)0xC7,
        (byte)0x00, (byte)0x45, (byte)0x9B, (byte)0x8F,
        (byte)0xDF, (byte)0x2B, (byte)0xFA, (byte)0x81,
        (byte)0x0D, (byte)0xEF, (byte)0x7A, (byte)0x49,
        (byte)0x6C, (byte)0x14, (byte)0x39, (byte)0xE5,
        (byte)0xDE, (byte)0x66, (byte)0xBE, (byte)0x61,
        (byte)0xD1, (byte)0x74, (byte)0x61, (byte)0x25,
        (byte)0xEB, (byte)0xA7, (byte)0x16, (byte)0x08,
        (byte)0x46, (byte)0x90, (byte)0xF3, (byte)0x9A,
        (byte)0x54, (byte)0x33, (byte)0x61, (byte)0xEF,
        (byte)0x76, (byte)0xE7, (byte)0xDB, (byte)0x56,
        (byte)0xF9, (byte)0x74, (byte)0xC7, (byte)0x6A,
        (byte)0x88, (byte)0x7C, (byte)0xC9, (byte)0xA6,
        (byte)0x2E, (byte)0x9F, (byte)0x95, (byte)0x3D,
        (byte)0x62, (byte)0x12, (byte)0x98, (byte)0xDC,
        (byte)0x2D, (byte)0x38, (byte)0x30, (byte)0xCA,
        (byte)0x9F, (byte)0x09, (byte)0x86, (byte)0x46,
        (byte)0xB1, (byte)0x50, (byte)0xD7, (byte)0x63,
        (byte)0x7A, (byte)0xC3, (byte)0x71, (byte)0x08,
        (byte)0x73, (byte)0x7B, (byte)0x48, (byte)0xDE,
        (byte)0x69, (byte)0x51, (byte)0x88, (byte)0x50,
        (byte)0xD7, (byte)0x01, (byte)0x12, (byte)0xC2,
        (byte)0x3F, (byte)0xCF, (byte)0xFD, (byte)0x6A,
        (byte)0x7B, (byte)0xA7, (byte)0xAA, (byte)0x2D,
        (byte)0x24, (byte)0x6A, (byte)0x9C, (byte)0x14,
        (byte)0x27, (byte)0x30, (byte)0xDA, (byte)0xAE,
        (byte)0xAD, (byte)0x6F, (byte)0xE7, (byte)0xC3,
        (byte)0x62, (byte)0xCF, (byte)0x37, (byte)0x61,
        (byte)0xE9, (byte)0x78, (byte)0x8E, (byte)0xB6,
        (byte)0x83, (byte)0x1F, (byte)0xD4, (byte)0x2F,
        (byte)0x67, (byte)0x8E, (byte)0x6E, (byte)0x50,
        (byte)0x52, (byte)0x5B, (byte)0x5F, (byte)0x67,
        (byte)0x14, (byte)0xC7, (byte)0xF0, (byte)0xA9,
        (byte)0xD1, (byte)0x0B, (byte)0x8F, (byte)0x85,
        (byte)0xE8, (byte)0xD5, (byte)0x9D, (byte)0xAC,
        (byte)0x98, (byte)0xB1, (byte)0x95, (byte)0x86,
        (byte)0x5F, (byte)0x98, (byte)0xF4, (byte)0x2F,
        (byte)0x5A, (byte)0xC9, (byte)0x66, (byte)0x23,
        (byte)0x3B, (byte)0xAC, (byte)0xD4, (byte)0xD7,
        (byte)0xA0, (byte)0x70, (byte)0xB4, (byte)0xD4,
        (byte)0x8D, (byte)0x60, (byte)0xAB, (byte)0xCC,
        (byte)0x0D, (byte)0x99, (byte)0xD4, (byte)0x52,
        (byte)0xBF, (byte)0x52, (byte)0xF9, (byte)0x0F,
        (byte)0x73, (byte)0x5C, (byte)0xB8, (byte)0x49,
        (byte)0x81, (byte)0xD8, (byte)0x32, (byte)0xA1,
        (byte)0x44, (byte)0xC7, (byte)0x3F, (byte)0x96,
        (byte)0xBD, (byte)0xD5, (byte)0xE9, (byte)0x0D,
        (byte)0xEF, (byte)0x88, (byte)0xCA, (byte)0xB0,
        (byte)0x09, (byte)0x29, (byte)0x7D, (byte)0x67,
        (byte)0x13, (byte)0x45, (byte)0x19, (byte)0xD6,
        (byte)0x3F, (byte)0xD2, (byte)0xB7, (byte)0x20,
        (byte)0xEE, (byte)0x03, (byte)0x41, (byte)0x3A,
        (byte)0x4F, (byte)0xC9, (byte)0xFD, (byte)0xE7,
        (byte)0xA1, (byte)0x8E, (byte)0xEE, (byte)0x80,
        (byte)0x78, (byte)0xF3, (byte)0x62, (byte)0x9B,
        (byte)0xB8, (byte)0x9D, (byte)0x0F, (byte)0xC8,
        (byte)0x00, (byte)0x74, (byte)0x6D, (byte)0x92,
        (byte)0xFC, (byte)0x23, (byte)0x8D, (byte)0xFE,
        (byte)0x89, (byte)0x12, (byte)0xF6, (byte)0xC6,
        (byte)0x3F, (byte)0x4E, (byte)0x10, (byte)0xAE,
        (byte)0xDC, (byte)0x26, (byte)0x50, (byte)0x46,
        (byte)0x8D, (byte)0x8A, (byte)0xA4, (byte)0xE5,
        (byte)0x6E, (byte)0xFF, (byte)0x6A, (byte)0x6D,
        (byte)0x71, (byte)0x35, (byte)0x16, (byte)0xD5,
        (byte)0xE3, (byte)0x94, (byte)0xBF, (byte)0x18,
        (byte)0x7D, (byte)0x85, (byte)0x2C, (byte)0xC9,
        (byte)0x33, (byte)0x84, (byte)0x7F, (byte)0x43,
        (byte)0xED, (byte)0xE8, (byte)0xFE, (byte)0xA5,
        (byte)0xF8, (byte)0x2F, (byte)0xD3, (byte)0x46,
        (byte)0xBF, (byte)0x69, (byte)0xE1, (byte)0x03,
        (byte)0xA5, (byte)0x29, (byte)0x87, (byte)0xD3,
        (byte)0xCC, (byte)0x9E, (byte)0xDC, (byte)0xDE,
        (byte)0x1E, (byte)0xC8, (byte)0x31, (byte)0x75,
        (byte)0xE8, (byte)0xF0, (byte)0xA5, (byte)0xB4,
        (byte)0x18, (byte)0x79, (byte)0x70, (byte)0x7B,
        (byte)0x06, (byte)0x5D, (byte)0xC3, (byte)0x5C,
        (byte)0x7B, (byte)0x54, (byte)0x31, (byte)0xB8,
        (byte)0xAE, (byte)0x79, (byte)0x6C, (byte)0x33,
        (byte)0xCD, (byte)0x4C, (byte)0xA7, (byte)0x4B,
        (byte)0xE4, (byte)0xF3, (byte)0xD2, (byte)0xE8,
        (byte)0xD8, (byte)0x65, (byte)0xE9, (byte)0xB0,
        (byte)0x08, (byte)0xF9, (byte)0x45, (byte)0x4B,
        (byte)0x65, (byte)0xED, (byte)0x9A, (byte)0xBE,
        (byte)0x17, (byte)0xF9, (byte)0x7C, (byte)0xBE,
        (byte)0x65, (byte)0xCB, (byte)0x07, (byte)0x87,
        (byte)0xFC, (byte)0xFE, (byte)0xBF, (byte)0xDC,
        (byte)0x2E, (byte)0xBE, (byte)0x46, (byte)0x75,
        (byte)0x99, (byte)0x0E, (byte)0x49, (byte)0xB7,
        (byte)0xA9, (byte)0xA5, (byte)0x49, (byte)0x24,
        (byte)0xBD, (byte)0xDB, (byte)0xAC, (byte)0xE4,
        (byte)0x99, (byte)0xB6, (byte)0x01, (byte)0xC6,
        (byte)0xC1, (byte)0xEC, (byte)0xFA, (byte)0xDD,
        (byte)0x0D, (byte)0x20, (byte)0x89, (byte)0x68,
        (byte)0x87, (byte)0x46, (byte)0x6E, (byte)0xFF,
        (byte)0xCE, (byte)0x2A, (byte)0x2A, (byte)0x7B,
        (byte)0xC5, (byte)0x91, (byte)0xC1, (byte)0xDE,
        (byte)0x19, (byte)0x20, (byte)0x6E, (byte)0xB2,
        (byte)0x42, (byte)0x2F, (byte)0x92, (byte)0xC0,
        (byte)0x21, (byte)0xED, (byte)0x88, (byte)0xD7,
        (byte)0xE9, (byte)0x51, (byte)0xB8, (byte)0x48,
        (byte)0xF8, (byte)0x72, (byte)0xC9, (byte)0x69,
        (byte)0xBB, (byte)0xC2, (byte)0x14, (byte)0x58,
        (byte)0xA4, (byte)0xA8, (byte)0xA8, (byte)0xE7,
        (byte)0x5B, (byte)0x82, (byte)0x1A, (byte)0x40,
        (byte)0x25, (byte)0x80, (byte)0x60, (byte)0x6E,
        (byte)0x85, (byte)0xD4, (byte)0x6C, (byte)0xBA,
        (byte)0x68, (byte)0xED, (byte)0x8A, (byte)0x47,
        (byte)0x5B, (byte)0x56, (byte)0xA5, (byte)0x4C,
        (byte)0xE5, (byte)0x78, (byte)0x93, (byte)0xB1,
        (byte)0xFF, (byte)0xA2, (byte)0xDA, (byte)0x5A,
        (byte)0xA4, (byte)0x1E, (byte)0xC1, (byte)0x5A,
        (byte)0x44, (byte)0xA0, (byte)0x4F, (byte)0x05,
        (byte)0x11, (byte)0x3F, (byte)0xC4, (byte)0xE3,
        (byte)0xB9, (byte)0xB4, (byte)0xF8, (byte)0x68,
        (byte)0xF9, (byte)0x50, (byte)0x77, (byte)0x65,
        (byte)0xAF, (byte)0x64, (byte)0xB6, (byte)0xBD,
        (byte)0x30, (byte)0xC0, (byte)0x65, (byte)0x58,
        (byte)0xFB, (byte)0xBC, (byte)0x66, (byte)0xD9,
        (byte)0x5A, (byte)0x65, (byte)0x0C, (byte)0x28,
        (byte)0xCD, (byte)0x82, (byte)0xB6, (byte)0xF1,
        (byte)0x7D, (byte)0x33, (byte)0x63, (byte)0x32,
        (byte)0xBC, (byte)0x01, (byte)0xDA, (byte)0x88,
        (byte)0x78, (byte)0x40, (byte)0x3C, (byte)0xD2,
        (byte)0xC1, (byte)0x31, (byte)0xE8, (byte)0x83,
        (byte)0x68, (byte)0xC5, (byte)0x51, (byte)0x51,
        (byte)0xD6, (byte)0xE2, (byte)0xA3, (byte)0x63,
        (byte)0x55, (byte)0x27, (byte)0x8F, (byte)0x42,
        (byte)0xF6, (byte)0x52, (byte)0xA5, (byte)0x11,
        (byte)0x1F, (byte)0x0B, (byte)0x6F, (byte)0xD1,
        (byte)0xD9, (byte)0xF1, (byte)0x03, (byte)0x55,
        (byte)0x67, (byte)0xB6, (byte)0x24, (byte)0x9E,
        (byte)0xD3, (byte)0x44, (byte)0xC2, (byte)0xB6,
        (byte)0xD6, (byte)0xF9, (byte)0x99, (byte)0x9A,
        (byte)0x5D, (byte)0x2E, (byte)0x2E, (byte)0x09,
        (byte)0xB2, (byte)0x0B, (byte)0x03, (byte)0xA1,
        (byte)0xF5, (byte)0x7B, (byte)0x33, (byte)0x65,
        (byte)0x68, (byte)0x1A, (byte)0xDD, (byte)0x8E,
        (byte)0x71, (byte)0xF6, (byte)0xFE, (byte)0x96,
        (byte)0x86, (byte)0xC0, (byte)0xAF, (byte)0x71,
        (byte)0x25, (byte)0x17, (byte)0x75, (byte)0x25,
        (byte)0x6B, (byte)0xEB, (byte)0xE1, (byte)0x1B,
        (byte)0xDB, (byte)0x34, (byte)0x34, (byte)0xAF,
        (byte)0x63, (byte)0xC3, (byte)0xEF, (byte)0xF6,
        (byte)0xEF, (byte)0x1B, (byte)0xE1, (byte)0x65,
        (byte)0xEF, (byte)0x96, (byte)0x49, (byte)0x90,
        (byte)0xD2, (byte)0x43, (byte)0x7F, (byte)0x0F,
        (byte)0x7D, (byte)0x98, (byte)0x60, (byte)0x2F,
        (byte)0xE1, (byte)0x09, (byte)0x95, (byte)0x7B,
        (byte)0x98, (byte)0x1C, (byte)0x6B, (byte)0x35,
        (byte)0x8A, (byte)0x59, (byte)0x99, (byte)0x32,
        (byte)0xE5, (byte)0x19, (byte)0x60, (byte)0x54,
        (byte)0x17, (byte)0x06, (byte)0x4E, (byte)0xDB,
        (byte)0x7D, (byte)0x0C, (byte)0x8C, (byte)0x13,
        (byte)0xFD, (byte)0xFD, (byte)0x1F, (byte)0xEE,
        (byte)0xCC, (byte)0x4B, (byte)0xD5, (byte)0x85,
        (byte)0xCF, (byte)0x35, (byte)0x92, (byte)0xF8,
        (byte)0x49, (byte)0x9C, (byte)0xA9, (byte)0x3F,
        (byte)0xCE, (byte)0x39, (byte)0x04, (byte)0x38,
        (byte)0x96, (byte)0x70, (byte)0x08, (byte)0xDB,
        (byte)0x64, (byte)0x00, (byte)0x00, (byte)0x00,
        (byte)0x01, (byte)0x00, (byte)0x05, (byte)0x58,
        (byte)0x2E, (byte)0x35, (byte)0x30, (byte)0x39,
        (byte)0x00, (byte)0x00, (byte)0x02, (byte)0xCB,
        (byte)0x30, (byte)0x82, (byte)0x02, (byte)0xC7,
        (byte)0x30, (byte)0x82, (byte)0x01, (byte)0xAF,
        (byte)0xA0, (byte)0x03, (byte)0x02, (byte)0x01,
        (byte)0x02, (byte)0x02, (byte)0x04, (byte)0x05,
        (byte)0xC4, (byte)0x26, (byte)0x8E, (byte)0x30,
        (byte)0x0D, (byte)0x06, (byte)0x09, (byte)0x2A,
        (byte)0x86, (byte)0x48, (byte)0x86, (byte)0xF7,
        (byte)0x0D, (byte)0x01, (byte)0x01, (byte)0x0B,
        (byte)0x05, (byte)0x00, (byte)0x30, (byte)0x0D,
        (byte)0x31, (byte)0x0B, (byte)0x30, (byte)0x09,
        (byte)0x06, (byte)0x03, (byte)0x55, (byte)0x04,
        (byte)0x03, (byte)0x13, (byte)0x02, (byte)0x43,
        (byte)0x41, (byte)0x30, (byte)0x1E, (byte)0x17,
        (byte)0x0D, (byte)0x31, (byte)0x30, (byte)0x30,
        (byte)0x35, (byte)0x32, (byte)0x31, (byte)0x30,
        (byte)0x38, (byte)0x34, (byte)0x38, (byte)0x31,
        (byte)0x36, (byte)0x5A, (byte)0x17, (byte)0x0D,
        (byte)0x31, (byte)0x30, (byte)0x30, (byte)0x38,
        (byte)0x31, (byte)0x39, (byte)0x30, (byte)0x38,
        (byte)0x34, (byte)0x38, (byte)0x31, (byte)0x36,
        (byte)0x5A, (byte)0x30, (byte)0x0D, (byte)0x31,
        (byte)0x0B, (byte)0x30, (byte)0x09, (byte)0x06,
        (byte)0x03, (byte)0x55, (byte)0x04, (byte)0x03,
        (byte)0x13, (byte)0x02, (byte)0x43, (byte)0x41,
        (byte)0x30, (byte)0x82, (byte)0x01, (byte)0x22,
        (byte)0x30, (byte)0x0D, (byte)0x06, (byte)0x09,
        (byte)0x2A, (byte)0x86, (byte)0x48, (byte)0x86,
        (byte)0xF7, (byte)0x0D, (byte)0x01, (byte)0x01,
        (byte)0x01, (byte)0x05, (byte)0x00, (byte)0x03,
        (byte)0x82, (byte)0x01, (byte)0x0F, (byte)0x00,
        (byte)0x30, (byte)0x82, (byte)0x01, (byte)0x0A,
        (byte)0x02, (byte)0x82, (byte)0x01, (byte)0x01,
        (byte)0x00, (byte)0x9C, (byte)0x3D, (byte)0x44,
        (byte)0x51, (byte)0xF1, (byte)0x0B, (byte)0x58,
        (byte)0x1B, (byte)0x88, (byte)0xBB, (byte)0x17,
        (byte)0xC7, (byte)0x8C, (byte)0x16, (byte)0x3A,
        (byte)0xCF, (byte)0x1D, (byte)0x2E, (byte)0xFC,
        (byte)0x78, (byte)0x20, (byte)0xA2, (byte)0x1E,
        (byte)0x33, (byte)0xD0, (byte)0x6A, (byte)0xF5,
        (byte)0x11, (byte)0x9E, (byte)0x5E, (byte)0x60,
        (byte)0x63, (byte)0x18, (byte)0x63, (byte)0x63,
        (byte)0x3F, (byte)0xE9, (byte)0x06, (byte)0x17,
        (byte)0xA2, (byte)0xC8, (byte)0x0B, (byte)0x03,
        (byte)0x27, (byte)0x0B, (byte)0x6D, (byte)0xA7,
        (byte)0x02, (byte)0x17, (byte)0x76, (byte)0x3C,
        (byte)0x52, (byte)0xA2, (byte)0x98, (byte)0xFD,
        (byte)0x48, (byte)0x5E, (byte)0xCC, (byte)0xA0,
        (byte)0x62, (byte)0xD7, (byte)0x92, (byte)0xC5,
        (byte)0xF0, (byte)0xDF, (byte)0x61, (byte)0x39,
        (byte)0x78, (byte)0xA7, (byte)0x2F, (byte)0x01,
        (byte)0x8F, (byte)0x37, (byte)0x65, (byte)0xE8,
        (byte)0x86, (byte)0xFE, (byte)0x71, (byte)0x40,
        (byte)0x33, (byte)0x88, (byte)0x1B, (byte)0x90,
        (byte)0x4D, (byte)0x43, (byte)0x90, (byte)0x0E,
        (byte)0x82, (byte)0x19, (byte)0x1C, (byte)0xE2,
        (byte)0xA2, (byte)0x70, (byte)0x31, (byte)0x0E,
        (byte)0xBF, (byte)0x58, (byte)0xD5, (byte)0xF8,
        (byte)0x45, (byte)0x5E, (byte)0xB9, (byte)0x92,
        (byte)0x97, (byte)0x20, (byte)0xDB, (byte)0xD4,
        (byte)0x2C, (byte)0xB4, (byte)0x0F, (byte)0xCF,
        (byte)0xC6, (byte)0x9B, (byte)0xA2, (byte)0x91,
        (byte)0xD4, (byte)0x29, (byte)0x0F, (byte)0xB4,
        (byte)0x52, (byte)0x04, (byte)0xD7, (byte)0xFD,
        (byte)0x7D, (byte)0xAF, (byte)0x40, (byte)0xA7,
        (byte)0x73, (byte)0xDB, (byte)0xF1, (byte)0xA5,
        (byte)0xB5, (byte)0xFA, (byte)0xC0, (byte)0x8B,
        (byte)0x2A, (byte)0x59, (byte)0x36, (byte)0xAC,
        (byte)0x34, (byte)0xD1, (byte)0x5B, (byte)0x5E,
        (byte)0xB9, (byte)0x73, (byte)0xA8, (byte)0xCB,
        (byte)0x11, (byte)0x80, (byte)0x59, (byte)0x6F,
        (byte)0x95, (byte)0xE8, (byte)0x54, (byte)0xB1,
        (byte)0xFB, (byte)0x51, (byte)0xD9, (byte)0x8D,
        (byte)0xC4, (byte)0x41, (byte)0xC1, (byte)0x61,
        (byte)0x0E, (byte)0xD6, (byte)0x62, (byte)0x0E,
        (byte)0x25, (byte)0x39, (byte)0xA5, (byte)0x51,
        (byte)0x09, (byte)0xEF, (byte)0x47, (byte)0x06,
        (byte)0x57, (byte)0x37, (byte)0x91, (byte)0x9F,
        (byte)0xDF, (byte)0x4E, (byte)0xD7, (byte)0xFB,
        (byte)0xE1, (byte)0x17, (byte)0x16, (byte)0x7D,
        (byte)0x3A, (byte)0x0E, (byte)0x73, (byte)0x01,
        (byte)0x66, (byte)0x11, (byte)0xFE, (byte)0x05,
        (byte)0x42, (byte)0xB1, (byte)0xBA, (byte)0x02,
        (byte)0x8A, (byte)0x4F, (byte)0x9A, (byte)0x4F,
        (byte)0xA7, (byte)0x5E, (byte)0x36, (byte)0xF1,
        (byte)0x10, (byte)0xEB, (byte)0x61, (byte)0x46,
        (byte)0xD2, (byte)0x51, (byte)0x9C, (byte)0x34,
        (byte)0x89, (byte)0x15, (byte)0x54, (byte)0x6C,
        (byte)0x85, (byte)0x1F, (byte)0x23, (byte)0xCF,
        (byte)0xE7, (byte)0xC1, (byte)0x2E, (byte)0x02,
        (byte)0xDA, (byte)0xD6, (byte)0xCF, (byte)0x5B,
        (byte)0xCC, (byte)0x53, (byte)0xCC, (byte)0xFA,
        (byte)0x60, (byte)0x70, (byte)0x5A, (byte)0x4C,
        (byte)0xDA, (byte)0x10, (byte)0xC5, (byte)0xFF,
        (byte)0xDA, (byte)0x04, (byte)0x44, (byte)0x35,
        (byte)0x66, (byte)0x18, (byte)0x38, (byte)0x1B,
        (byte)0xFE, (byte)0xB6, (byte)0x1F, (byte)0x25,
        (byte)0xD1, (byte)0x02, (byte)0x03, (byte)0x01,
        (byte)0x00, (byte)0x01, (byte)0xA3, (byte)0x2F,
        (byte)0x30, (byte)0x2D, (byte)0x30, (byte)0x1D,
        (byte)0x06, (byte)0x03, (byte)0x55, (byte)0x1D,
        (byte)0x0E, (byte)0x04, (byte)0x16, (byte)0x04,
        (byte)0x14, (byte)0xE6, (byte)0x8B, (byte)0x5C,
        (byte)0x87, (byte)0xCC, (byte)0x5E, (byte)0xB7,
        (byte)0xBD, (byte)0x0C, (byte)0x93, (byte)0x21,
        (byte)0x1A, (byte)0xBA, (byte)0x3A, (byte)0xB9,
        (byte)0x62, (byte)0x4F, (byte)0xF6, (byte)0x74,
        (byte)0x7C, (byte)0x30, (byte)0x0C, (byte)0x06,
        (byte)0x03, (byte)0x55, (byte)0x1D, (byte)0x13,
        (byte)0x04, (byte)0x05, (byte)0x30, (byte)0x03,
        (byte)0x01, (byte)0x01, (byte)0xFF, (byte)0x30,
        (byte)0x0D, (byte)0x06, (byte)0x09, (byte)0x2A,
        (byte)0x86, (byte)0x48, (byte)0x86, (byte)0xF7,
        (byte)0x0D, (byte)0x01, (byte)0x01, (byte)0x0B,
        (byte)0x05, (byte)0x00, (byte)0x03, (byte)0x82,
        (byte)0x01, (byte)0x01, (byte)0x00, (byte)0x8C,
        (byte)0x7F, (byte)0x24, (byte)0x01, (byte)0xEC,
        (byte)0x15, (byte)0x42, (byte)0x24, (byte)0x2D,
        (byte)0x36, (byte)0xE4, (byte)0xAE, (byte)0x2C,
        (byte)0x1B, (byte)0xA3, (byte)0xBF, (byte)0xB8,
        (byte)0xD1, (byte)0x65, (byte)0x47, (byte)0x87,
        (byte)0x3C, (byte)0x53, (byte)0x9A, (byte)0x08,
        (byte)0xC8, (byte)0x57, (byte)0x2F, (byte)0x16,
        (byte)0x95, (byte)0x58, (byte)0x57, (byte)0x30,
        (byte)0xAC, (byte)0x4B, (byte)0x40, (byte)0x8A,
        (byte)0x49, (byte)0x16, (byte)0x0F, (byte)0xF7,
        (byte)0xDA, (byte)0xFA, (byte)0xCA, (byte)0xC4,
        (byte)0x20, (byte)0x28, (byte)0xB5, (byte)0x20,
        (byte)0xF1, (byte)0xB8, (byte)0xCA, (byte)0x18,
        (byte)0x8F, (byte)0x5D, (byte)0x6C, (byte)0x86,
        (byte)0x52, (byte)0xE1, (byte)0xAD, (byte)0x2B,
        (byte)0x9D, (byte)0x03, (byte)0xD2, (byte)0xF2,
        (byte)0x5F, (byte)0x31, (byte)0xB6, (byte)0xDE,
        (byte)0x72, (byte)0x70, (byte)0x00, (byte)0x1B,
        (byte)0xEA, (byte)0xDE, (byte)0x69, (byte)0x12,
        (byte)0x12, (byte)0x0F, (byte)0xC1, (byte)0xF0,
        (byte)0xE0, (byte)0x33, (byte)0xFE, (byte)0x42,
        (byte)0x26, (byte)0x0E, (byte)0x14, (byte)0x3B,
        (byte)0xBA, (byte)0x3A, (byte)0x89, (byte)0x2D,
        (byte)0x71, (byte)0xE1, (byte)0xD3, (byte)0x66,
        (byte)0x82, (byte)0x6D, (byte)0xB9, (byte)0xA0,
        (byte)0xC3, (byte)0x34, (byte)0xA2, (byte)0x73,
        (byte)0xDF, (byte)0x28, (byte)0x2D, (byte)0x78,
        (byte)0xCF, (byte)0xA1, (byte)0xA0, (byte)0x12,
        (byte)0x43, (byte)0x94, (byte)0x66, (byte)0x37,
        (byte)0x3D, (byte)0x7F, (byte)0x6B, (byte)0xC9,
        (byte)0x4D, (byte)0xDE, (byte)0x96, (byte)0x20,
        (byte)0x44, (byte)0x91, (byte)0x64, (byte)0x8A,
        (byte)0xBE, (byte)0x24, (byte)0x4D, (byte)0xD7,
        (byte)0x26, (byte)0x6A, (byte)0xE5, (byte)0xB1,
        (byte)0x1C, (byte)0xA0, (byte)0x26, (byte)0xC1,
        (byte)0x95, (byte)0xC1, (byte)0x97, (byte)0xE1,
        (byte)0x2A, (byte)0x57, (byte)0xFC, (byte)0x50,
        (byte)0x54, (byte)0xF6, (byte)0x92, (byte)0x35,
        (byte)0xF1, (byte)0x55, (byte)0x0B, (byte)0x13,
        (byte)0xC3, (byte)0xE7, (byte)0x7A, (byte)0x15,
        (byte)0xA7, (byte)0x00, (byte)0x13, (byte)0xB9,
        (byte)0x17, (byte)0x15, (byte)0x9F, (byte)0x86,
        (byte)0x68, (byte)0xCE, (byte)0x03, (byte)0x16,
        (byte)0x89, (byte)0xD9, (byte)0x4A, (byte)0xF8,
        (byte)0xEF, (byte)0x81, (byte)0xEE, (byte)0x63,
        (byte)0x6F, (byte)0x2A, (byte)0x3A, (byte)0xD5,
        (byte)0x3F, (byte)0xC3, (byte)0xCF, (byte)0x15,
        (byte)0xBA, (byte)0x52, (byte)0x99, (byte)0x9D,
        (byte)0x94, (byte)0x68, (byte)0xCA, (byte)0x08,
        (byte)0xB1, (byte)0xFB, (byte)0xDE, (byte)0xB7,
        (byte)0x8A, (byte)0x2C, (byte)0x70, (byte)0xC6,
        (byte)0x56, (byte)0x91, (byte)0x7C, (byte)0xF8,
        (byte)0x09, (byte)0x68, (byte)0x0B, (byte)0xA2,
        (byte)0x9E, (byte)0x6C, (byte)0xD6, (byte)0x95,
        (byte)0x9B, (byte)0x90, (byte)0x2A, (byte)0xF7,
        (byte)0x25, (byte)0x93, (byte)0x83, (byte)0x87,
        (byte)0x68, (byte)0x56, (byte)0x17, (byte)0x11,
        (byte)0x75, (byte)0x6F, (byte)0x4E, (byte)0x1B,
        (byte)0xE4, (byte)0x7B, (byte)0x12, (byte)0xD8,
        (byte)0x72, (byte)0x2B, (byte)0x1E, (byte)0xD1,
        (byte)0x57, (byte)0xB1, (byte)0x94, (byte)0xD6,
        (byte)0x07, (byte)0x1A, (byte)0xFE, (byte)0xB8,
        (byte)0x3D, (byte)0x2E, (byte)0x96, (byte)0xE0,
        (byte)0xE0, (byte)0xF7, (byte)0x33, (byte)0x3B,
        (byte)0x72, (byte)0xE2, (byte)0xD7, (byte)0x40,
        (byte)0xD5, (byte)0xF8, (byte)0xFE, (byte)0x79,
        (byte)0x23, (byte)0x11, (byte)0xF4, (byte)0x1A,
        (byte)0x76, (byte)0xCF, (byte)0x7D, (byte)0xFA,
        (byte)0x1D, (byte)0xCD, (byte)0xA6,
    };

    private static final String certs =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIIC3TCCAcWgAwIBAgIEXtLE/zANBgkqhkiG9w0BAQsFADAOMQwwCgYDVQQDEwNJbnQwHhcNMTAw\n" +
        "NTIxMDg0ODE1WhcNMTAwODE5MDg0ODE1WjAPMQ0wCwYDVQQDEwRVc2VyMIIBIjANBgkqhkiG9w0B\n" +
        "AQEFAAOCAQ8AMIIBCgKCAQEAztzpQkfz2m68N+0z1o4SAttA4zwNSRMSB8wPfM0RHS4LSAbTGeHy\n" +
        "Lhc4OpTL9ABaMt275W3uhbQiwXcgDH4mbVl3QtRUAfTfka1oKt0eYCzVgcZHb+I/2HGUge8ihKem\n" +
        "zvdZWngtQ6jYG1GH0J+W6E7y1Wk6pJ+ohJps4wCwMrkQwhEyZZALxegeLKWRh7jb2gJkd5MU0Fn1\n" +
        "l+0hDBJVEu4B5GaUC9GO/JPo3jaA0EkacKCYB313kJkEJBklc1dUzYux5vcLSikxp0xz4YvyonMU\n" +
        "Q7PBMavRDp/ZO0kptSV/EI76H6Qc6rtLzxNbBUIm//QcsJP8wVPBNgpA/Mj36wIDAQABo0IwQDAd\n" +
        "BgNVHQ4EFgQUR/dUteOOD5pcanOZtjrnRp37cGYwHwYDVR0jBBgwFoAU4RRA2KgWg3n7jY8JmtOG\n" +
        "FJNxvicwDQYJKoZIhvcNAQELBQADggEBAAVDMSIW4FhqiYj2bKj4G0zzCQERS+UYG/x2gqD0Sr2e\n" +
        "aR8GOM0w+emh+TBf6UVTwR6QMeB+mq0Qhmxi67TwwbmKlO32fSGBP9eCEJBIiWIHcaFOKTZe2TQR\n" +
        "olhI1yLpCVU+TTFlnu66YsZqWtOeceUWI1TM+V4l+7UYm614AgMo0FTEcPq1XN8VKJQA1MP8FuLh\n" +
        "QAFH7BNNdl5+hMd1PkXiRUskpHoP9RDL0GkxyWxYwldd1/gtYK88X6kqsuy2uigysg2OIssUxtgp\n" +
        "XdVlOdf1eBTTD4nUQQJ9VrUoMwNitggvUOmnprKnX6gNPTFlbj8NueVAMcuXcatxJ3lIDGE=\n" +
        "-----END CERTIFICATE-----\n" +
        "-----BEGIN CERTIFICATE-----\n" +
        "MIIC6TCCAdGgAwIBAgIEKhBU0TANBgkqhkiG9w0BAQsFADANMQswCQYDVQQDEwJDQTAeFw0xMDA1\n" +
        "MjEwODQ4MTRaFw0xMDA4MTkwODQ4MTRaMA4xDDAKBgNVBAMTA0ludDCCASIwDQYJKoZIhvcNAQEB\n" +
        "BQADggEPADCCAQoCggEBAIWLINX6KokLcnlpThE5tYMFRuzFLdcPlQ0DFvku1csVuVz2bcVhsUVa\n" +
        "w18KfAc7vRthux+ZD7QNYiXZYT71lkBUPset66mp6OaizrHMf2hMiRd4154Ztn6Ebb0wA6MKcB/C\n" +
        "2qOd7BC4xe7LZ2dNobH/eqmQHUld0XW3wKL13pXlhIEAlJxhcoKKANtfTFMwEKMnQmgOUBtAzGle\n" +
        "NvH+sXO2iZjPQQwmQAkmxYfR5odWHOL6HHhhMunGQZWGl3YJ8DngtQdzHbjeLoo2nV66gNKRwd2w\n" +
        "4o4V7abfibFUkiBbtQSY60MieTt1fkGEaDLkvZHNYAeXCP4VwKstM6N80WMCAwEAAaNQME4wHQYD\n" +
        "VR0OBBYEFOEUQNioFoN5+42PCZrThhSTcb4nMB8GA1UdIwQYMBaAFOaLXIfMXre9DJMhGro6uWJP\n" +
        "9nR8MAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQELBQADggEBAIvJJbXisDs16PAESL9d1D6jd2/v\n" +
        "N6jRy5VSGH8C7qD26KvFxuw+/dM8q2LE5tCjPUQN9Vfj1QVQ4bi5BF00iL8oAboL0EYJUbB6oKyF\n" +
        "AUxW5WHJgudysgJvSJuNUsSuFSg3QenPi8gFJTdGZu1I0Yd/kelu4/Q8gzq9Htq/H7Sw2oDvXa5u\n" +
        "z6C2AUdewCqiGspPskU/neXc0XfOmJ9YmqNAtyMUir05mYHPAOSkKwCjkQ1z8y3G5ECR995ECxC6\n" +
        "DXTuNv35DwjD1TlhbauBs4KJG7/+mN5yVLkqKhfqxAsFjsEfe6X5I+NfFiPK5upQyjw7K7M5TGbo\n" +
        "+QUYZgSXE94=\n" +
        "-----END CERTIFICATE-----\n" +
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICxzCCAa+gAwIBAgIEGpy06jANBgkqhkiG9w0BAQIFADANMQswCQYDVQQDEwJDQTAeFw0xMDA1\n" +
        "MjEwODQ4MTBaFw0xMDA4MTkwODQ4MTBaMA0xCzAJBgNVBAMTAkNBMIIBIjANBgkqhkiG9w0BAQEF\n" +
        "AAOCAQ8AMIIBCgKCAQEAnD1EUfELWBuIuxfHjBY6zx0u/Hggoh4z0Gr1EZ5eYGMYY2M/6QYXosgL\n" +
        "AycLbacCF3Y8UqKY/UhezKBi15LF8N9hOXinLwGPN2Xohv5xQDOIG5BNQ5AOghkc4qJwMQ6/WNX4\n" +
        "RV65kpcg29QstA/PxpuikdQpD7RSBNf9fa9Ap3Pb8aW1+sCLKlk2rDTRW165c6jLEYBZb5XoVLH7\n" +
        "UdmNxEHBYQ7WYg4lOaVRCe9HBlc3kZ/fTtf74RcWfToOcwFmEf4FQrG6AopPmk+nXjbxEOthRtJR\n" +
        "nDSJFVRshR8jz+fBLgLa1s9bzFPM+mBwWkzaEMX/2gRENWYYOBv+th8l0QIDAQABoy8wLTAdBgNV\n" +
        "HQ4EFgQU5otch8xet70MkyEaujq5Yk/2dHwwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQIFAAOC\n" +
        "AQEACq0bBDsOYZrxahZx0VQEsKZn2kSBKDFqYvalydXhG+z+uhcDifg07Xb7kUTSw+Qxdd+mOZU3\n" +
        "l+GBY7PpiEqvGB6RVPkOWBKXlc7WV6CpRbZCXfi1QbJWUEQINbrTqg2C1TsqL+DQAZ6H+ceLEYsR\n" +
        "SFsA9hFHB+8pBMDk922dSn90ZBcYmKtceQWit4pp3WkZKGUieUVv4DSPx04slZ1t1q0b8/yJejTo\n" +
        "JveDXJys9lNXSwC86EBzyr+eaxoBGBRnaJ2fP+Gs4N1Rwoc7n4NfSDVr+3V0P5z+3GpIR7P4kxbF\n" +
        "uXkGaN/2ox5+ZlrQ1RX9c9sdTpD032iTS96egS0vAQ==\n" +
        "-----END CERTIFICATE-----\n";
}

