/*                                                                            */
/* CDDL HEADER START                                                          */
/*                                                                            */
/* The contents of this file are subject to the terms of the Common           */
/* Development and Distribution License Version 1.0 (the "License").          */
/*                                                                            */
/* You can obtain a copy of the license at                                    */
/* http://www.opensource.org/licenses/CDDL-1.0.  See the License for the      */
/* specific language governing permissions and limitations under the License. */
/*                                                                            */
/* When distributing Covered Code, include this CDDL HEADER in each file and  */
/* include the License file in a prominent location with the name             */
/* LICENSE.CDDL.  If applicable, add the following below this CDDL HEADER,    */
/* with the fields enclosed by brackets "[]" replaced with your own           */
/* identifying information:                                                   */
/*                                                                            */
/* Portions Copyright (c) [yyyy] [name of copyright owner].                   */
/* All rights reserved.                                                       */
/*                                                                            */
/* CDDL HEADER END                                                            */
/*                                                                            */

/*                                                                            */
/* <REPLACE: Change copyright information as needed>                          */
/* Copyright (c) 2019, Regents of the University of Minnesota.                */
/* All rights reserved.                                                       */
/* </REPLACE>                                                                 */
/*                                                                            */
/* <REPLACE: Change contributors as needed>                                   */
/* Contributors:                                                              */
/*    Daniel S. Karls                                                         */
/*    Ellad B. Tadmor                                                         */
/* </REPLACE>                                                                 */
/*                                                                            */

#include <math.h>

/*                                                                            */
/* Auxiliary files for the Stillinger-Weber model driver                      */
/*                                                                            */
/* Functions in this file are only used by cluster.inc. They are not required */
/* by ThreeBodyCluster.c.                                                     */
/*                                                                            */

/*                                                                            */
/* Define functions used in two-body calculations                             */
/*                                                                            */
static double fc2(double const * params, double R)
{
  /* Unpack parameters */
  const double a = params[PARAM_a];
  const double sigma = params[PARAM_sigma];

  if (R < a) { return exp(sigma / (R - a) + (sigma / a)); }
  else
  {
    return 0.0;
  }
}

static void
fc2_dfc2(double const * params, double R, double * fc2, double * dfc2_dR)
{
  /* Unpack parameters */
  const double a = params[PARAM_a];
  const double sigma = params[PARAM_sigma];

  if (R < a)
  {
    *fc2 = exp(sigma / (R - a) + (sigma / a));
    *dfc2_dR = -sigma * *fc2 / ((R - a) * (R - a));
  }
  else
  {
    *fc2 = 0.0;
    *dfc2_dR = 0.0;
  }
}

/*                                                                            */
/* Define functions used in three-body calculations                             */
/*                                                                            */
static double g(double const * params, double theta)
{
  /* Unpack parameters */
  const double costheta0 = params[PARAM_costheta0];

  double costheta;

  costheta = cos(theta);
  return (costheta - costheta0) * (costheta - costheta0);
}

static double dg_dcostheta(double const * params, double theta)
{
  /* Unpack parameters */
  const double costheta0 = params[PARAM_costheta0];

  double costheta;

  costheta = cos(theta);
  return 2 * (costheta - costheta0);
}

static double fc3(double const * params, double R)
{
  /* Unpack parameters */
  const double a = params[PARAM_a];
  const double gamma = params[PARAM_gamma];

  if (R < a) { return exp(gamma / (R - a) + (gamma / a)); }
  else
  {
    return 0.0;
  }
}

static void
fc3_dfc3(double const * params, double R, double * fc3, double * dfc3_dR)
{
  /* Unpack parameters */
  const double a = params[PARAM_a];
  const double gamma = params[PARAM_gamma];

  if (R < a)
  {
    *fc3 = exp(gamma / (R - a) + (gamma / a));
    *dfc3_dR = -gamma * *fc3 / ((R - a) * (R - a));
  }
  else
  {
    *fc3 = 0.0;
    *dfc3_dR = 0.0;
  }
}

static double h(double const * params, double r, double s, double theta)
{
  return fc3(params, r) * fc3(params, s) * g(params, theta);
}

static void dh_drdsdtheta(double const * params,
                          double r,
                          double s,
                          double theta,
                          double * dh_dr,
                          double * dh_ds,
                          double * dh_dtheta)
{
  double fc3_r;
  double fc3_s;
  double dfc3_dr;
  double dfc3_ds;
  double costheta;
  double t;
  double gval;
  double dgdcos;

  double dcostheta_dr;
  double dcostheta_ds;
  double dcostheta_dt;

  /* Compute third distance using law of cosines */
  costheta = cos(theta);
  t = sqrt(r * r + s * s - 2 * r * s * costheta);

  fc3_dfc3(params, r, &fc3_r, &dfc3_dr);
  fc3_dfc3(params, s, &fc3_s, &dfc3_ds);

  dcostheta_dr = (r * r - s * s + t * t) / (2 * r * r * s);
  dcostheta_ds = (s * s - r * r + t * t) / (2 * r * s * s);
  dcostheta_dt = -t / (r * s);

  gval = g(params, theta);
  dgdcos = dg_dcostheta(params, theta);

  *dh_dr = fc3_s * (fc3_r * dgdcos * dcostheta_dr + dfc3_dr * gval);
  *dh_ds = fc3_r * (fc3_s * dgdcos * dcostheta_ds + dfc3_ds * gval);
  *dh_dtheta = fc3_r * fc3_s * dgdcos * dcostheta_dt;
}
