///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __COLUMN_CHANNEL_MAPPING_H
#define __COLUMN_CHANNEL_MAPPING_H

#include <core/Core.h>
#include <atomviz/atoms/datachannels/DataChannel.h>

namespace AtomViz {

/// The maximum number of data columns that can be parsed.
#define MAXIMUM_ATOM_COLUMN_COUNT  64

/**
 * \brief Defines the mapping between data columns in a data file
 *        and the data channels in the AtomsObjects.
 *
 * This class is used by data file parsers derived from AbstractFileColumnParser to
 * store a user-definable mapping of data columns.
 *
 * \sa AbstractFileColumnParser, DataChannel, AtomsObject, ChannelColumnMapping
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT ColumnChannelMapping : public QObject
{
public:
	/// \brief Creates a new mapping object with zero columns.
	ColumnChannelMapping() : QObject(), remapAtomIndices(true) {}

	/// \brief Copy constructor.
	ColumnChannelMapping(const ColumnChannelMapping& other) : QObject() { *this = other; }

	/// \brief Returns the number of columns that have been given a mapping to a DataChannel.
	int columnCount() const { return columns.size(); }

	/// \brief Resizes the internal mapping array to include the specified number of data columns.
	void setColumnCount(int numberOfColumns, const QStringList& columnNames = QStringList());

	/// \brief Reduces the number of columns as much as possible.
	/// All data columns at the end of the mapping that are not associated with a data channel are removed  by this method.
	void shrink() {
		while(columnCount() > 0 && getChannelType(columnCount()-1) == QMetaType::Void)
			setColumnCount(columnCount()-1);
	}

	/// \brief Associates a column in the data file with a custom DataChannel in the AtomsObject.
	/// \param columnIndex The column number starting at 0.
	/// \param channelId The identifier of the DataChannel to be associated with the given column.
	/// \param channelName Specifies the name of DataChannel to be created in the AtomsObject during import.
	/// \param dataType The data type of the DataChannel to create. This must be the Id of the type registered with the QMetaType class.
	/// \param vectorComponent The component of the vector.
	/// \param columnName Specifies the name of the column in the input if known.
	/// \sa defineStandardColumn(), ignoreColumn()
	void defineColumn(int columnIndex, DataChannel::DataChannelIdentifier channelId, const QString& channelName, int dataType, size_t vectorComponent = 0, const QString& columnName = QString());

	/// \brief Associates a column in the data file with a standard DataChannel in the AtomsObject.
	/// \param columnIndex The column number starting at 0.
	/// \param channel The indentifier of the standard channel to be associated with the column.
	/// \param vectorComponent The component in the per-atom vector.
	/// \param columnName Specifies the name of the column in the input if known.
	/// \sa defineColumn()
	void defineStandardColumn(int columnIndex, DataChannel::DataChannelIdentifier channel, size_t vectorComponent = 0, const QString& columnName = QString());

	/// \brief Ignores a column in the data file.
	/// \param columnIndex The column number starting at 0.
	/// \param columnName Specifies the name of the column in the input if known.
	/// \sa defineColumn()
	void ignoreColumn(int columnIndex, const QString& columnName = QString());

	/// \brief Returns the name of a column in the input file.
	/// \return The name of the column or an empty string if the input file does not contain column name information.
	QString getColumnName(int columnIndex) const { return (columnIndex < columns.size()) ? columns[columnIndex].columnName : QString(); }

	/// \brief Resets the column names read from the input file.
	void clearColumnNames() {
		for(QVector<MapEntry>::iterator i = columns.begin(); i != columns.end(); ++i)
			i->columnName = QString();
	}

	/// \brief Returns the identifier of the DataChannel that is associated with the given column of the input file.
	/// \return The identifier of the DataChannel that will be created in the destination AtomsObject for the given column.
	///         If the column is not associated with a data channel then DataChannel::UserDataChannel is returned.
	DataChannel::DataChannelIdentifier getChannelId(int columnIndex) const { return (columnIndex < columns.size()) ? columns[columnIndex].dataChannelId : DataChannel::UserDataChannel; }

	/// \brief Returns the name of the DataChannel that is associated with the given column of the input file.
	/// \return The name of the DataChannel that will be created in the destination AtomsObject for the given column.
	///         If the column is not associated with a data channel then an empty string is returned.
	QString getChannelName(int columnIndex) const { return (columnIndex < columns.size()) ? columns[columnIndex].dataChannelName : QString(); }

	/// \brief Returns the data type of the DataChannel that is associated with the given column of the input file.
	/// \return The data type of the DataChannel that will be created in the destination AtomsObject for the given column.
	///         If the column is not associated with a data channel then DataChannel::TypeNone is returned.
	int getChannelType(int columnIndex) const { return (columnIndex < columns.size()) ? columns[columnIndex].type : QMetaType::Void; }

	/// \brief Returns the vector component for a column when it is associated with a multi-valued DataChannel.
	size_t getVectorComponent(int columnIndex) const { return (columnIndex < columns.size()) ? columns[columnIndex].vectorComponent : 0; }

	/// \brief Returns whether automatic remapping of atom indices is turned on.
	/// \sa enableAtomIndexRemapping()
	bool atomIndexRemappingEnabled() const { return remapAtomIndices; }

	/// \brief Controls automatic remapping of atom indices.
	///
	/// If automatic remapping of atom indices is turned on and there is a data column in the input file
	/// that has been mapped to the DataChannel::AtomIndexChannel standard data channel then the atom index
	/// stored in that data column in the input file will be used as storage index for the atom.
	/// Otherwise the atoms will be stored in the same order as they are listed in the input file.
	///
	/// Automatic index remapping is turned on by default.
	///
	/// \sa atomIndexRemappingEnabled()
	void enableAtomIndexRemapping(bool enable) { remapAtomIndices = enable; }

	/// \brief Saves the mapping to the given stream.
	void saveToStream(SaveStream& stream) const;
	/// \brief Loads the mapping from the given stream.
	void loadFromStream(LoadStream& stream);

	/// \brief Saves the mapping into a byte array.
	QByteArray toByteArray() const;
	/// \brief Loads the mapping from a byte array.
	void fromByteArray(const QByteArray& array);

	/// \brief Saves the mapping the application's settings store.
	/// \param presetName The name under which the mapping is saved.
	/// \sa loadPreset(), listPresets()
	void savePreset(const QString& presetName) const;

	/// \brief Loads a mapping from the application's settings store.
	/// \param presetName The name of the mapping to load.
	/// \throws Exception if there is no preset with the given name.
	/// \sa savePreset(), listPresets()
	void loadPreset(const QString& presetName);

	/// \brief Returns a list of all presets found in the
	///        application's settings store.
	/// \sa loadPreset(), deletePreset()
	static QStringList listPresets();

	/// \brief Deletes a mapping from the application's settings store.
	/// \param presetName The name of the mapping to delete.
	/// \throws Exception if there is no preset with the given name.
	/// \sa savePreset(), loadPreset(), listPresets()
	static void deletePreset(const QString& presetName);

	/// Makes a copy of the mapping object.
	ColumnChannelMapping& operator=(const ColumnChannelMapping& other);

private:

	/** Stores information about a single column in the data file. */
	struct MapEntry {
		QString columnName;					///< The name of the column in the input file.
		DataChannel::DataChannelIdentifier dataChannelId;		///< The identifier of the corresponding DataChannel in the AtomsObject.
		QString dataChannelName;			///< The name of the DataChannel if it is a user-defined channel.
		int type;							///< The data type of the DataChannel if this is a user-defined channel. If this is QMetaType::Void the column will be ignored when parsing.
		size_t vectorComponent;				///< The vector component if the channel is multi-valued.
	};

	/// Contains one entry for each column in the data file.
	QVector<MapEntry> columns;

	/// Controls whether the atoms are renumbered according ot the
	/// atom index column in the input file if there is one.
	bool remapAtomIndices;

	Q_OBJECT
};


/**
 * \brief Parses the data fields in the input file and stores them in the data channels.
 *
 * This helper class can be used by a AbstractFileColumnParser derived class to convert
 * the strings in one row of the text file to their final data type and feed them
 * into the data channels of the destination AtomsObject according to the
 * ColumnChannelMapping.
 *
 * Note that this helper object is only used during the actual parsing process.
 * It creates the necessary data channels in the destination object as defined
 * by the column to channel mapping.
 *
 * \sa ColumnChannelMapping, AbstractFileColumnParser
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT DataRecordParserHelper : public QObject
{
public:
	/// \brief Initializes the helper object.
	/// \param mapping Defines the mapping between the columns in the input file
	///        and the data channels in the destination AtomsObject.
	/// \param destination The helper object will store the parsed data in this object.
	/// \throws Exception if the mapping is not valid.
	///
	/// This constructor creates all necessary data channels in the destination object as defined
 	/// by the column to channel mapping.
	DataRecordParserHelper(const ColumnChannelMapping* mapping, AtomsObject* destination);

	/// \brief Parses the string tokens from one line of the input file and stores the values
	///        in the data channels of the destination AtomsObject.
	/// \param atomIndex The line index starting at 0 that specifies the atom whose properties
	///                  are read from the input file. Please note that this method may
	///                  ignore this index if the atom index is determined from one of the columns
	///                  in the input data.
	/// \param dataLine The text line read from the input file that contains the data field values.
	///                 The contents of this string are detroyed by this parsing method.
	void storeAtom(int atomIndex, char* dataLine);

	/// \brief Parses the string tokens from one line of the input file and stores the values
	///        in the data channels of the destination AtomsObject.
	/// \param atomIndex The line index starting at 0 that specifies the atom whose properties
	///                  are read from the input file. Please note that this method may
	///                  ignore this index if the atom index is determined from one of the columns
	///                  in the input data.
	/// \param ntokens The number ok tokens parsed from the input file line.
	/// \param tokens The list of parsed tokens.
	void storeAtom(int atomIndex, int ntokens, const char* tokens[]);

	/// \brief Stores the columns values for one atom in the data channels of the destination AtomsObject.
	/// \param atomIndex The line index starting at 0 that specifies the atom whose properties
	///                  are read from the input file. Please note that this method may
	///                  ignore this index if the atom index is determined from one of the columns
	///                  in the input data.
	/// \param values A pointer to an array of values. It must contain one value per column in the input file.
	/// \param nvalues The number of columns in the input file.
	void storeAtom(int atomIndex, const double* values, int nvalues);

	/// \brief Indicates that at least some of the atom coordinates are out of range.
	/// \return \c true if at least one parsed coordinate of one atom was out of range;
	///         \c false if all coordinates were valid.
	bool coordinatesOutOfRange() const { return _coordinatesOutOfRange; }

	/// \brief Returns the bounding box of all loaded atoms.
	/// \return The bounding box containing all atoms.
	const Box3& boundingBox() const { return _boundingBox; }

private:

	/// Determines which input data columns are stored in which data channels.
	const ColumnChannelMapping* mapping;

	/// The destination object.
	AtomsObject* destination;

	/// Stores the destination data channels. One for each column of the input file.
	QVector<DataChannel*> channels;

	/// Stores the index of the column that contains the atom indices.
	int atomIndexColumn;

	/// The Qt data type identifiers.
	int intMetaTypeId, floatMetaTypeId;

	/// Stores the bounding box of all loaded atoms.
	Box3 _boundingBox;

	/// Indicates that at least some of the atom coordinates are out of range.
	bool _coordinatesOutOfRange;
};

};	// End of namespace AtomViz

#endif // __COLUMN_CHANNEL_MAPPING_H
