/***************************************************************************
 *   Copyright (C) 2000-2008 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/*!
  The dispatcher is the central system that routes all messages from the different threads.
It also starts, stops and synchronizes the threads.

*/
#include "dispatcher.h"
#include "qsstvglobal.h"
#include "configparams.h"
#include "rxwidget.h"
#include "txwidget.h"
#include "gallerywidget.h"
#include <QSettings>
#include "widgets/spectrumwidget.h"
#include "widgets/vumeter.h"

#include "rig/rigcontrol.h"
#include "editor/editor.h"
#include "sound/soundio.h"
#include "sound/waterfalltext.h"
#include "mainwindow.h"
#include "drmrx/drm.h"
#include "utils/ftp.h"
#include "utils/hybridcrypt.h"





/*!
creates dispatcher instance
 */

dispatcher::dispatcher()
{
  mbox = new QMessageBox(mainWindowPtr);
  progressFTP=NULL;
  serialP=0;
}

/*!
delete dispatcher instance
 */

dispatcher::~dispatcher()
{
  if(serialP!=0)
    {
      close(serialP);
      serialP=0;
    }
}

QString dispatcher::init()
{
  QString ret;
  editorActive=false;
  rxWidgetPtr->fftDisplayPtr()->init(RXSTRIPE,1,BASESAMPLERATE/SUBSAMPLINGRATIO);
  galleryWidgetPtr->init();
  rxWidgetPtr->init();
  txWidgetPtr->init();
  waterfallPtr=new waterfallText;
  waterfallPtr->init();
  if(!rigController->init())
    {
      ret=rigController->initError;
    }
  restartRXFlag=false;
  infoTextPtr=new textDisplay(mainWindowPtr);
  infoTextPtr->hide();
  return ret;
}



void dispatcher::stopRX()
{
  rxWidgetPtr->start(false);
}

void dispatcher::stopTX()
{
  rigController->activatePTT(false);
  txWidgetPtr->start(false);
}

void dispatcher::stopRXTX()
{
  stopRX();
  stopTX();
}

void dispatcher::startRX(bool st)
{
  stopTX();
  rxWidgetPtr->start(st);
  if(st)
    {
      addToLog("dispatcher: starting RX",LOGDISPAT);
    }
  else
    {
      addToLog("dispatcher: stopping RX",LOGDISPAT);
    }
}

void dispatcher::startTX(bool st)
{
  stopRX();
  if(st)
    {
      addToLog("dispatcher: starting TX",LOGDISPAT);
    }
  else
    {
      addToLog("dispatcher: stopping TX",LOGDISPAT);
    }
  rigController->activatePTT(st);
  txWidgetPtr->start(st);
}



void dispatcher::readSettings()
{
  QSettings qSettings;
  logfile->readSettings(qSettings);
}

void dispatcher::writeSettings()
{
  QSettings qSettings;
  logfile->writeSettings(qSettings);
}

/*!
  All communication between the threads are passed via this eventhandler.
*/

void dispatcher::customEvent( QEvent * e )
{
  dispatchEventType type;
  QString fn;
  type=(dispatchEventType)e->type();
  addToLog(((baseEvent*)e)->description,LOGDISPAT);
  switch(type)
    {
    case displayFFT:
      addToLog("dispatcher: displayFFT",LOGDISPAT);
      rxWidgetPtr->fftDisplayPtr()->realFFT(((displayFFTEvent*)e)->data());
      rxWidgetPtr->fftDisplayPtr()->repaint();
      break;
    case displaySync:
      // addToLog("dispatcher: displaySync",LOGDISPAT);
      uint s;double v;
      ((displaySyncEvent*)e)->getInfo(s,v);
      rxWidgetPtr->sMeterPtr()->setValue((double)s);
      rxWidgetPtr->vMeterPtr()->setValue(v);
      break;
    case rxSSTVStatus:
      rxWidgetPtr->setSSTVStatusText(((statusMsgEvent*)e)->getStr());
      break;
    case rxDRMStatus:
      rxWidgetPtr->setDRMStatusText(((statusMsgEvent*)e)->getStr());
      break;
    case startImageRX:
      addToLog("dispatcher: clearing RxImage",LOGDISPAT);
      rxWidgetPtr->getImageViewerPtr()->createImage( ((startImageRXEvent*)e)->getSize(),QColor(255,255,0));
      rxWidgetPtr->getImageViewerPtr()->clear();
      break;
    case lineDisplay:
      {
        rxWidgetPtr->getImageViewerPtr()->displayImage(false);
      }
      break;
      //    case syncLost:
      //    case verticalRetrace:
      //      addToLog("dispatcher: verticalRetrace",LOGDISPAT);

      //      if(autoSave)
      //        {
      //          addToLog("dispatcher: verticalRetrace savingRxImage",LOGDISPAT);
      //          saveRxSSTVImage();
      //        }
      //      rxWidgetPtr->functionsPtr()->retraceVertical();
      //    break;
    case endImageRX:
      if(autoSave)
        {
          addToLog("dispatcher:endImage savingRxImage",LOGDISPAT);
          saveRxSSTVImage();
        }
      break;
    case callEditor:
      if(editorActive) break;
      editorActive=true;
      ed=new editor();
      ed->show();
      iv=((callEditorEvent*)e)->getImageViewer();
      addToLog (QString(" callEditorEvent imageViewPtr: %1").arg(QString::number((ulong)iv,16)),LOGDISPAT);
      addToLog(QString("editor: filename %1").arg(((callEditorEvent*)e)->getFilename()),LOGDISPAT);
      ed->openFile(((callEditorEvent*)e)->getFilename());
      break;

    case editorFinished:
      if(!editorActive) break;
      if(((editorFinishedEvent*)e)->isOK())
        {
          addToLog (QString(" editorFinishedEvent imageViewPtr: %1").arg(QString::number((ulong)iv,16)),LOGDISPAT);
          iv->reload();
        }
      editorActive=false;
      delete ed;
      break;

    case templatesChanged:
      txWidgetPtr->setupTemplatesComboBox();
      break;
    case progressTX:
      txTimeCounter=0;
      prTimerIndex=startTimer(((progressTXEvent*)e)->getInfo()*10); // time in seconds -> times 1000 for msec,divide by 100 for progress
      break;
    case stoppingTX:
      addToLog("dispatcher: endTXImage",LOGDISPAT);
      while(!soundIOPtr->stoppedPlaying())
        {
          qApp->processEvents();
        }
      rigController->activatePTT(false);
      break;

    case endImageTX:
      addToLog("dispatcher: endTXImage",LOGDISPAT);
      while(!soundIOPtr->stoppedPlaying())
        {
          qApp->processEvents();
        }
      rigController->activatePTT(false);
      restartRX();
      break;
    case displayDRMInfo:
      //      rxWidgetPtr->psdWdg()->setPSD();
      rxWidgetPtr->mscWdg()->setConstellation(MSC);
      rxWidgetPtr->facWdg()->setConstellation(FAC);
      rxWidgetPtr->statusWdg()->setStatus();
      break;

    case displayDRMStat:
      DSPFLOAT s1;DSPFLOAT v1;
      ((displayDRMStatEvent*)e)->getInfo(s1,v1);
      rxWidgetPtr->sMeterPtr()->setValue(s1);
      rxWidgetPtr->vMeterPtr()->setValue(v1);
      break;

    case loadRXImage:
      {
        ((loadRXImageEvent*)e)->getFilename(fn);
        rxWidgetPtr->getImageViewerPtr()->openImage(fn,true,false);
      }
      break;

    case saveDRMImage:
      {
        ((saveDRMImageEvent*)e)->getFilename(fn);
        rxWidgetPtr->getImageViewerPtr()->openImage(fn,true,false);
        saveImage(fn);
      }
      break;
    case prepareFix:
      addToLog("prepareFix",LOGDISPAT);
      startDRMFIXTx( ((prepareFixEvent*)e)->getData());
      break;
    case displayText:
      infoTextPtr->clear();
      infoTextPtr->setWindowTitle(QString("Received from %1").arg(drmCallsign));
      infoTextPtr->append(((displayTextEvent*)e)->getStr());
      infoTextPtr->show();

      break;

    case displayMBox:
      mbox->setWindowTitle(((displayMBoxEvent*)e)->getTitle());
      mbox->setText(((displayMBoxEvent*)e)->getStr());
      mbox->show();
      QTimer::singleShot(4000, mbox, SLOT(hide()));
      break;
    case displayProgressFTP:
      {
        if(((displayProgressFTPEvent*)e)->getTotal()==0)
          {
            delete progressFTP;
            progressFTP=NULL;
            break;
          }
        if(progressFTP==NULL)
          {
            progressFTP=new QProgressDialog("FTP Transfer","Cancel",0,0,mainWindowPtr);
          }
        progressFTP->show();
        progressFTP->setMaximum(((displayProgressFTPEvent*)e)->getTotal());
        progressFTP->setValue(((displayProgressFTPEvent*)e)->getBytes());
      }
      break;
    default:
      addToLog(QString("unsupported event: %1").arg(((baseEvent*)e)->description), LOGALL);
      break;
    }
  ((baseEvent *)e)->setDone();
}



void dispatcher::receiveImage()
{
}

void dispatcher::restartRX()
{
  txTimeCounter=9999999; //force stop timer
  startRX(true);
}

void dispatcher::startSSTVTx()
{
  stopRX();
  txWidgetPtr->start(true,false);
  if(txWidgetPtr->functionsPtr()->getTXState()!=txFunctions::TXIDLE) return;
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSSTVIMAGE);
  addToLog("send SSTV Image",LOGDISPAT);

}

void dispatcher::sendTone(double duration,double freq)
{
  stopRX();
  txWidgetPtr->start(true,false);
  if(txWidgetPtr->functionsPtr()->getTXState()!=txFunctions::TXIDLE) return;
  txWidgetPtr->functionsPtr()->setToneParam(duration,freq);
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSENDTONE);
  addToLog("sendTone",LOGDISPAT);
}

void dispatcher::sendWF(QString txt)
{
  stopRX();
  txWidgetPtr->start(true,false);
  if(txWidgetPtr->functionsPtr()->getTXState()!=txFunctions::TXIDLE) return;
  waterfallPtr->setText(txt);
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSENDID);
  addToLog("sendID",LOGDISPAT);
}

void dispatcher::startDRMTx()
{
  stopRX();

  txWidgetPtr->start(true,false);
  if(txWidgetPtr->functionsPtr()->getTXState()!=txFunctions::TXIDLE) return;
  //  waterfallPtr->setText(pictureWF);
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSENDDRM);
  addToLog("sendDRM",LOGDISPAT);
}

void dispatcher::startDRMBSRTx(QByteArray *ba)
{
  if(ba==NULL) return;
  stopRX();
  txWidgetPtr->start(true,false);
  if(txWidgetPtr->functionsPtr()->getTXState()!=txFunctions::TXIDLE) return;
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->initDRMBSR(ba);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSENDDRMBSR);
  addToLog("sendDRMBSR",LOGDISPAT);
}

void dispatcher::startDRMFIXTx(QByteArray ba)
{
  if(!txWidgetPtr->prepareFIX(ba)) return;
  stopRX();
  txWidgetPtr->start(true,false);
  if(txWidgetPtr->functionsPtr()->getTXState()!=txFunctions::TXIDLE) return;
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSENDDRMFIX);
  addToLog("sendDRMFIX",LOGDISPAT);
}

void dispatcher::startDRMHybridTx(QString fn)
{
  eftpError ftpResult;
  QByteArray ba;
  QTemporaryFile ftmp;
  ftpInterface ftpIntf("HybridTX");
  hybridCrypt hc;
  ftpIntf.setupConnection(hc.host(),hc.port(),hc.user(),hc.passwd(),hc.dir()+"/"+hybridFtpHybridFilesDirectory);
  //  txWidgetPtr->getImageViewerPtr()->getFilename();
  txWidgetPtr->getImageViewerPtr()->copyToBuffer(&ba);
  if(!ftmp.open()) return;
  ftmp.write(ba);
  ftmp.close();
  ftpResult=ftpIntf.uploadFile(ftmp.fileName(),fn,true);
  switch(ftpResult)
    {
    case FTPCANCELED:
      QMessageBox::information(mainWindowPtr, tr("FTP Info"),"Connection Canceled");
      return;
      break;
    case FTPOK:
      break;
    case FTPERROR:
      QMessageBox::critical(mainWindowPtr, tr("FTP Error"),ftpIntf.getLastError());
      return;
      break;
    case FTPNAMEERROR:
      QMessageBox::critical(mainWindowPtr, tr("FTP Error"),"Error in filename");
      return;
      break;
    case FTPTIMEOUT:
      QMessageBox::critical(mainWindowPtr, tr("FTP Error"),"FTP timed out");
      return;
      break;
    }
  stopRX();
  txWidgetPtr->start(true,false);
  if(txWidgetPtr->functionsPtr()->getTXState()!=txFunctions::TXIDLE) return;
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSENDDRM);
  addToLog("sendDRMHybrid",LOGDISPAT);
}

void dispatcher::startDRMHybridText(QString txt)
{
  if(!txWidgetPtr->prepareText(txt)) return;
  stopRX();
  txWidgetPtr->start(true,false);
  if(txWidgetPtr->functionsPtr()->getTXState()!=txFunctions::TXIDLE) return;
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSENDDRMTXT);
  addToLog("sendDRMTxt",LOGDISPAT);
}



void dispatcher::sendSweepTone(double duration,double lowerFreq,double upperFreq)
{
  stopRX();
  txWidgetPtr->start(true,false);
  txWidgetPtr->functionsPtr()->setToneParam(duration,lowerFreq,upperFreq);
  rigController->activatePTT(true);
  txWidgetPtr->functionsPtr()->setTXState(txFunctions::TXSENDTONE);
  addToLog("sendSweepTone",LOGDISPAT);
}




void dispatcher::saveRxSSTVImage()
{
  if (rxWidgetPtr->functionsPtr()->getModeString().isEmpty()) return;
  if(!rxWidgetPtr->functionsPtr()->saveOK()) return;
  QString s,fileName;
  QDateTime dt(QDateTime::currentDateTime().toUTC()); //this is compatible with QT 4.6
  dt.setTimeSpec(Qt::UTC);
  fileName=QString("%1/%2_%3.%4").arg(rxImagesPath).arg(rxWidgetPtr->functionsPtr()->getModeString()).arg(dt.toString("yyyyMMdd_HHmmss")).arg(defaultImageFormat);
  addToLog(QString("dispatcher: saveRxImage():%1 ").arg(fileName),LOGDISPAT);
  rxWidgetPtr->getImageViewerPtr()->save(fileName,defaultImageFormat,true);
  saveImage(fileName);
}

void dispatcher::saveImage(QString fileName)
{
  QImage im;
  QFileInfo info(fileName);
  eftpError ftpResult;
  displayMBoxEvent *stmb=0;
  QString fn="/tmp/"+info.baseName()+"."+ftpDefaultImageFormat;
  galleryWidgetPtr->putRxImage(fileName);
  txWidgetPtr->setPreviewWidget(fileName);
  if(enableFTP)
    {
      ftpInterface ftpIntf("Save RX Image");
      rxWidgetPtr->getImageViewerPtr()->save(fn,ftpDefaultImageFormat,true);
      ftpIntf.setupConnection(ftpRemoteHost,ftpPort,ftpLogin,ftpPassword,ftpRemoteDirectory);
      ftpResult=ftpIntf.uploadToRXServer(fn);
      switch(ftpResult)
        {
        case FTPOK:
          break;
        case FTPERROR:
          stmb= new displayMBoxEvent("FTP Error",QString("Host: %1: %2").arg(ftpRemoteHost).arg(ftpIntf.getLastError()));
          break;
        case FTPNAMEERROR:
          stmb= new displayMBoxEvent("FTP Error",QString("Host: %1, Error in filename").arg(ftpRemoteHost));
          break;
        case FTPCANCELED:
          stmb= new displayMBoxEvent("FTP Error",QString("Connection to %1 Canceled").arg(ftpRemoteHost));
          break;
        case FTPTIMEOUT:
          stmb= new displayMBoxEvent("FTP Error",QString("Connection to %1 timed out").arg(ftpRemoteHost));
          break;
        }
      if(ftpResult!=FTPOK)
        {
          QApplication::postEvent( dispatcherPtr, stmb );  // Qt will delete it when done
          return;
        }

    }
}



void dispatcher::timerEvent(QTimerEvent *event)
{
  if(event->timerId()==prTimerIndex)
    {
      txWidgetPtr->setProgress(++txTimeCounter);
      if(txTimeCounter>=100)
        {
          if(prTimerIndex>=0)
            {
              killTimer(prTimerIndex);
              prTimerIndex=-1;
              txWidgetPtr->setProgress(0);
            }
        }
      txWidgetPtr->setProgress(txTimeCounter);
    }
  else if(event->timerId()==logTimerIndex)
    {
      //      addToLog(QString("dumping dispatcher status"),LOGALL);
      //      if( rxFuncPtr) rxFuncPtr->logStatus();
      //      if( txFuncPtr)txFuncPtr->logStatus();
      //      if( sndIO)sndIO->logStatus();
    }
}










