/*
 * Copyright (C) 2004-2020 by the Widelands Development Team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "ui_basic/progressbar.h"

#include <memory>

#include "graphic/font_handler.h"
#include "graphic/graphic.h"
#include "graphic/rendertarget.h"
#include "graphic/text_layout.h"

namespace UI {
/**
 * Initialize the progress bar.
 */
ProgressBar::ProgressBar(Panel* const parent,
                         int32_t const x,
                         int32_t const y,
                         int32_t const w,
                         int32_t const h,
                         uint32_t const orientation)
   : Panel(parent, x, y, w, h),
     orientation_(orientation),
     state_(0),
     total_(100),
     style_(g_gr->styles().progressbar_style(UI::PanelStyle::kWui)) {
}

/**
 * Set the current state of progress.
 */
void ProgressBar::set_state(uint32_t state) {
	state_ = state;
}

/**
 * Set the maximum state
 */
void ProgressBar::set_total(uint32_t total) {
	assert(total);
	total_ = total;
}

/**
 * Draw the progressbar.
 */
void ProgressBar::draw(RenderTarget& dst) {
	assert(0 < get_w());
	assert(0 < get_h());
	assert(total_);
	const float fraction = state_ < total_ ? static_cast<float>(state_) / total_ : 1.0f;
	assert(0 <= fraction);
	assert(fraction <= 1);

	const RGBColor& color = fraction <= 0.33f ? style_.low_color() : fraction <= 0.67f ?
	                                            style_.medium_color() :
	                                            style_.high_color();

	// Draw the actual bar
	if (orientation_ == Horizontal) {
		const float w = get_w() * fraction;
		assert(w <= get_w());

		dst.fill_rect(Recti(0, 0, w, get_h()), color);
		dst.fill_rect(Recti(w, 0, get_w() - w, get_h()), RGBColor(0, 0, 0));
	} else {
		const uint32_t h = static_cast<uint32_t>(get_h() * (1.0f - fraction));

		dst.fill_rect(Recti(0, 0, get_w(), h), RGBColor(0, 0, 0));
		dst.fill_rect(Recti(0, h, get_w(), get_h() - h), color);
	}

	// Print the state in percent without decimal points.
	std::shared_ptr<const UI::RenderedText> rendered_text = UI::g_fh->render(as_richtext_paragraph(
	   (boost::format("%u%%") % floorf(fraction * 100.f)).str(), style_.font()));
	Vector2i pos(get_w() / 2, get_h() / 2);
	UI::center_vertically(rendered_text->height(), &pos);
	rendered_text->draw(dst, pos, UI::Align::kCenter);
}
}  // namespace UI
