/* -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the GNOME-keyring signond extension
 *
 * Copyright (C) 2011 Canonical Ltd.
 *
 * Contact: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "debug.h"
#include "secrets-storage.h"

#include <QDataStream>
#include <libsecret/secret.h>

using namespace SignOn;

static const SecretSchema signonSchema = {
    "com.ubuntu.OnlineAccounts.Secrets", SECRET_SCHEMA_DONT_MATCH_NAME,
    {
        { "signon-type", SECRET_SCHEMA_ATTRIBUTE_INTEGER },
        { "signon-id", SECRET_SCHEMA_ATTRIBUTE_INTEGER },
        { "signon-method", SECRET_SCHEMA_ATTRIBUTE_INTEGER },
    }
};

SecretsStorage::SecretsStorage(QObject *parent) :
    AbstractSecretsStorage(parent),
    m_keyringName()
{
    TRACE() << "Constructed";
}

SecretsStorage::~SecretsStorage()
{
    TRACE() << "Destroyed";
}

bool SecretsStorage::initialize(const QVariantMap &configuration)
{
    if (configuration.contains(QLatin1String("KeyringName"))) {
        m_keyringName = configuration.value(QLatin1String("KeyringName")).toByteArray();
    } else {
        m_keyringName = QByteArray(SECRET_COLLECTION_DEFAULT);
    }
    TRACE() << "Using keyring:" << m_keyringName;

    setIsOpen(true);
    return true;
}

bool SecretsStorage::clear()
{
    if (!removeSecrets(Password, 0, 0, TypeField))
        return false;

    if (!removeSecrets(Username, 0, 0, TypeField))
        return false;

    if (!removeSecrets(Data, 0, 0, TypeField))
        return false;

    return true;
}

bool SecretsStorage::updateCredentials(const quint32 id,
                                       const QString &username,
                                       const QString &password)
{
    if (!username.isEmpty() &&
        !storeSecret(Username, id, 0, username.toUtf8()))
        return false;

    if (!password.isEmpty() &&
        !storeSecret(Password, id, 0, password.toUtf8()))
        return false;

    return true;
}

bool SecretsStorage::removeCredentials(const quint32 id)
{
    return removeSecrets(NoType, id, 0, IdField);
}

bool SecretsStorage::loadCredentials(const quint32 id,
                                     QString &username,
                                     QString &password)
{
    QByteArray baUsername, baPassword;
    bool found = false;

    if (loadSecret(Username, id, 0, baUsername)) {
        username = QString::fromUtf8(baUsername.constData());
        found = true;
    } else {
        username = QString();
    }

    if (loadSecret(Password, id, 0, baPassword)) {
        password = QString::fromUtf8(baPassword.constData());
        found = true;
    } else {
        password = QString();
    }

    return found;
}

QVariantMap SecretsStorage::loadData(quint32 id, quint32 method)
{
    QByteArray serializedData;
    bool ret;

    ret = loadSecret(Data, id, method, serializedData);
    if (!ret) return QVariantMap();

    QDataStream stream(QByteArray::fromPercentEncoding(serializedData));
    QVariantMap data;
    stream >> data;
    return data;
}

bool SecretsStorage::storeData(quint32 id, quint32 method,
                               const QVariantMap &data)
{
    QByteArray serializedData;
    QDataStream stream(&serializedData, QIODevice::WriteOnly);
    stream << data;
    return storeSecret(Data, id, method, serializedData.toPercentEncoding());
}

bool SecretsStorage::removeData(quint32 id, quint32 method)
{
    return (method != 0) ?
        removeSecrets(Data, id, method, IdField | MethodField | TypeField) :
        removeCredentials(id);
}

bool SecretsStorage::storeSecret(SignonSecretType type,
                                 quint32 id,
                                 quint32 method,
                                 const QByteArray &secret)
{
    TRACE() << "Storing secret:" << id <<
        "type:" << type <<
        "method:" << method;

    QString displayName =
        QString::fromLatin1("Ubuntu Web Account: id %1-%2").arg(id).arg(type);
    QByteArray ba = displayName.toUtf8();

    const gchar *signonMethod = (type == Data) ? "signon-method" : NULL;

    gboolean ok;
    GError *error = NULL;
    ok = secret_password_store_sync(&signonSchema,
                                    keyring(),
                                    ba.constData(),
                                    secret.constData(),
                                    NULL,
                                    &error,
                                    "signon-type", type,
                                    "signon-id", id,
                                    signonMethod, method,
                                    NULL);


    if (!ok) {
        TRACE() << "Got error from GNOME keyring:" << error->message;
        g_error_free(error);
        return false;
    }

    return true;
}

bool SecretsStorage::loadSecret(SignonSecretType type,
                                quint32 id,
                                quint32 method,
                                QByteArray &secret)
{
    TRACE() << "id:" << id << "type:" << type << "method:" << method;

    const gchar *signonMethod = (type == Data) ? "signon-method" : NULL;

    GError *error = NULL;
    gchar *data = secret_password_lookup_sync(&signonSchema,
                                              NULL,
                                              &error,
                                              "signon-type", type,
                                              "signon-id", id,
                                              signonMethod, method,
                                              NULL);

    if (error != NULL) {
        TRACE() << "Credentials loading failed:" << error->message;
        g_error_free(error);
        return false;
    }

    secret = QByteArray(data);
    g_free(data);

    return data != NULL;
}

bool SecretsStorage::removeSecrets(SignonSecretType type,
                                   quint32 id,
                                   quint32 method,
                                   QueryFields fields)
{
    GHashTable *attributes = g_hash_table_new(g_str_hash, g_str_equal);
    gchar id_str[16];
    gchar method_str[16];
    gchar type_str[16];

    if (fields & IdField) {
        snprintf(id_str, sizeof(id_str), "%d", id);
        g_hash_table_insert(attributes, gpointer("signon-id"), id_str);
    }
    if (fields & MethodField) {
        snprintf(method_str, sizeof(method_str), "%d", method);
        g_hash_table_insert(attributes, gpointer("signon-method"), method_str);
    }
    if (fields & TypeField) {
        snprintf(type_str, sizeof(type_str), "%d", type);
        g_hash_table_insert(attributes, gpointer("signon-type"), type_str);
    }

    GError *error = NULL;
    secret_password_clearv_sync(&signonSchema,
                                attributes,
                                NULL,
                                &error);
    if (error != NULL) {
        TRACE() << "Credentials search failed:" << error->message;
        g_error_free(error);
        return false;
    }

    g_hash_table_unref(attributes);

    return true;
}

const char *SecretsStorage::keyring() const
{
    return m_keyringName.isEmpty() ? 0 : m_keyringName.constData();
}
