/* xoreos-tools - Tools to help with xoreos development
 *
 * xoreos-tools is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos-tools is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos-tools is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos-tools. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *   Resolve a hash found in hashed Sonic archives back to the filename.
 */

#include "src/common/util.h"
#include "src/common/binsearch.h"

#include "src/files_sonic.h"

typedef Common::BinSearchValue<uint32, const char *> SonicFileHash;

/** All currently known Sonic file names, together with their DJB2 hashes.
 *
 *  Note: For the binary search to work, this list needs to stay sorted by hash value!
 */
static const SonicFileHash kSonicFilesHashes[] = {
	{0x00021EC9, "prtl_gglgen_1.ncgr.small"            },
	{0x0009B65D, "itm_glvgrpcl_0.ncgr.small"           },
	{0x001109AD, "chao_ca_g.nsbtx.small"               },
	{0x0011EACC, "itm_ace_immun_2.ncgr.small"          },
	{0x00125256, "pl_pz_vox_swc.nsbtx.small"           },
	{0x00171B0E, "cmbtbg_a1_s3.nsbtx.small"            },
	{0x00215BBC, "pl_ob_blk_brk.nsbca"                 },
	{0x00263AB0, "prtl_nrrgal.nclr.small"              },
	{0x002D4500, "prtl_amydrm_2.ncgr.small"            },
	{0x003047CE, "conv_bark_bs01.ncgr.small"           },
	{0x00335B49, "sco_ex_walk.nsbca"                   },
	{0x0033C252, "prtl_tailsexc_1.ncgr.small"          },
	{0x003FC555, "a1c3_swatbot.dlg"                    },
	{0x00430080, "a1c5_p9_ixruns.gff"                  },
	{0x0048F874, "egb_cb_blockidle.nsbca"              },
	{0x004C96DE, "prtl_sdwgen_2.ncgr.small"            },
	{0x0051650B, "prtl_txrhap_0.ncgr.small"            },
	{0x0055C62E, "cfx_rfgp1.nsbmd.small"               },
	{0x0060D697, "chao_ha_1.ncgr.small"                },
	{0x006734B3, "pl_ob_blkb_sr.nsbtx.small"           },
	{0x006E5894, "prtl_hczgen_1.ncgr.small"            },
	{0x007A52FC, "a1sq_official.dlg"                   },
	{0x007A5EF8, "a1sq_official.gff"                   },
	{0x007C6A09, "cbt_chasegauge_01.ncgr.small"        },
	{0x008BDFE0, "fx_ome_beam.nsbmd.small"             },
	{0x0090CAF2, "man_tsvin_00.ncgr.small"             },
	{0x00A2B1D7, "man_tsvst_of00.ncgr.small"           },
	{0x00A69252, "pl_pz_pod_dn.nsbca"                  },
	{0x00B187D6, "prtl_sonicsle_2.ncgr.small"          },
	{0x00B2BDEA, "prtl_tailssca_2.ncgr.small"          },
	{0x00BC325E, "prtl_rougeinq_0.ncgr.small"          },
	{0x00D68825, "mut_circl_ms00.ncgr.small"           },
	{0x00D7EBEE, "armboar_aa.nsbtx.small"              },
	{0x00E2B0E5, "prtl_nrsge2_0.ncgr.small"            },
	{0x00E6DCDF, "armdilr_aa_p.ncgr.small"             },
	{0x00EC7224, "lower_blue_ring_02.ncgr.small"       },
	{0x00ED4A49, "kro_cb_counter.nsbca"                },
	{0x00EF0F33, "chao_ca_2.ncgr.small"                },
	{0x00F51992, "cbt_elem_earth.ncgr.small"           },
	{0x00FF4EB2, "fx_dummyobj.nsbmd.small"             },
	{0x0108D151, "itm_glvblncl_1.ncgr.small"           },
	{0x011B61C4, "cfx_spbp1_shb.emit"                  },
	{0x012C8F8B, "endscreen_bs1.tga"                   },
	{0x013070D4, "itm_fotslplt_0.ncgr.small"           },
	{0x0132FDFB, "wmp_l_redpn00.ncgr.small"            },
	{0x013598B6, "man_tsvin_22.ncgr.small"             },
	{0x013A0330, "gzc_cb_kd.nsbca"                     },
	{0x013EA80C, "endscreen_bs2.tga"                   },
	{0x01417EF6, "prtl_salegen_3.ncgr.small"           },
	{0x01450DBD, "a1_w2_mt_00.ncgr.small"              },
	{0x014871B4, "hnpc_tdad_aa.nsbmd.small"            },
	{0x01499B12, "prt_tp_e_big.ncgr.small"             },
	{0x01548C30, "chao_ca_r.nsbtp.small"               },
	{0x0156B1E4, "prtl_krscon_0.ncgr.small"            },
	{0x0157E673, "pl_am_noc_gch.nsbca"                 },
	{0x015AF08B, "spell_stun.spl"                      },
	{0x01652FBA, "lvl_tdbld_ms00.ncgr.small"           },
	{0x01677CC1, "a1c5_p8_transition.gff"              },
	{0x0181336D, "lvl_stat_bspnl02.ncgr.small"         },
	{0x0194F0A2, "a1_s6_ex.walk.tga"                   },
	{0x019834B9, "charpro_rou1.ncgr.small"             },
	{0x019FBA3D, "bsl_cb_damage.nsbca"                 },
	{0x01A0261E, "a2_s4_card_b2.nbfp"                  },
	{0x01A02621, "a2_s4_card_b2.nbfs"                  },
	{0x01A5A5BF, "a1c5_p6_approachcannon.gff"          },
	{0x01A70A8D, "prtl_eixlaf_0.ncgr.small"            },
	{0x01AB5833, "prtl_sdwdef_2.ncgr.small"            },
	{0x01AE9ABE, "adv_shade.gda"                       },
	{0x01C0D194, "btn_back_on.ncgr.small"              },
	{0x01C59C47, "upper_portrait_04.ncgr.small"        },
	{0x01CD4A4A, "fx_a2_s7_env.nsbtx.small"            },
	{0x01E9A684, "pfx_puring.emit"                     },
	{0x01F02D6F, "a2_s2_i04.walk.tga"                  },
	{0x01F3CF46, "a1c5_omega_party_select.gff"         },
	{0x01FC372D, "pro_sidbx_00.ncgr.small"             },
	{0x01FFE7B9, "pl_am_noc_gslr.nsbca"                },
	{0x0209FD7E, "lvl_tdbld_ms22.ncgr.small"           },
	{0x021BBEA8, "itm_act2_chaos5_0.ncgr.small"        },
	{0x0223FD9A, "spell_lucky_2.spl"                   },
	{0x0236161B, "spell_lucky_3.spl"                   },
	{0x023B466D, "icn_s_patk_tai1.ncgr.small"          },
	{0x0248B12A, "chapter2ruinswarp.gff"               },
	{0x024BE463, "cfx_rbep1.nsbtx.small"               },
	{0x024CFA44, "prtl_gnagen_0.ncgr.small"            },
	{0x024DDEBE, "gui_profile.nclr.small"              },
	{0x02545D3F, "prtl_krsgen_1.ncgr.small"            },
	{0x0255EF16, "mut_sonic_ms11.ncgr.small"           },
	{0x025FFB1A, "wmp_trv_03_04.gff"                   },
	{0x0263D2A6, "cbt_black_lower.nanr.small"          },
	{0x0268B478, "worldmap_traveliconnopin.gui"        },
	{0x026D6300, "itm_glv_cloth_2.ncgr.small"          },
	{0x0272A469, "a1c4_reset_party_puzzle.gff"         },
	{0x027A4EDA, "poison.spl"                          },
	{0x027D899B, "fx_vox_b_ea_s.nsbtx.small"           },
	{0x02930A64, "prtl_bigsnz_3.ncgr.small"            },
	{0x0293D197, "prtl_guncosmg_0.ncgr.small"          },
	{0x02984E35, "cfx_crvp1.nsbtx.small"               },
	{0x029DEBA0, "lower_bubblebox_00.ncgr.small"       },
	{0x02A1E47A, "pfx_bub3.nsbtp.small"                },
	{0x02A65984, "itm_fotslpdk_1.ncgr.small"           },
	{0x02A85D1E, "wmp_trv_03_08.gff"                   },
	{0x02AD50BF, "a2_w0_nt_11.ncgr.small"              },
	{0x02AE5205, "fx_drn_lzr_muz.nsbta.small"          },
	{0x02AE62EA, "she_aa.nsbtx.small"                  },
	{0x02B32825, "_typhoonnavigator.dlg"               },
	{0x02BAC5C3, "a1c1_dock_face_sw.gff"               },
	{0x02BAEEC5, "prtl_syraxgen_0.ncgr.small"          },
	{0x02C3C5ED, "chao_jj_r.nsbmd.small"               },
	{0x02C82778, "icn_s_patk_amy3.ncgr.small"          },
	{0x02C89A20, "a2c8_scy_raxosdefeated.gff"          },
	{0x02CE79B0, "itm_ace_shld_1.ncgr.small"           },
	{0x02CF5D00, "prtl_txcgen_3.ncgr.small"            },
	{0x02DA9A02, "prtl_dadrel_2.ncgr.small"            },
	{0x02E0A9A3, "pfx_nicktest03.nsbmd.small"          },
	{0x02E3711D, "exl_qcirc_ms00.ncgr.small"           },
	{0x02E39B29, "itm_ace_rocks_3.ncgr.small"          },
	{0x02EAE43D, "gzd_cb_attackc.nsbca"                },
	{0x02FD3B3C, "chao_je_0.ncgr.small"                },
	{0x02FE0AE0, "swtsbot_aa.nsbtx.small"              },
	{0x03038ECB, "charpro_bs_c11.ncgr.small"           },
	{0x03072A1A, "pl_mv_cyc_prk.nsbta.small"           },
	{0x030AF4D1, "lvl_tlgbx_ms00.ncgr.small"           },
	{0x03134C55, "spell_int_super_sonic.spl"           },
	{0x03273D88, "itm_ace_shipm_3.ncgr.small"          },
	{0x032E6F88, "cfx_bugp1.nsbtx.small"               },
	{0x03387E05, "itm_fotlgtbt_3.ncgr.small"           },
	{0x033A69B4, "cfx_watp1_drp.emit"                  },
	{0x033DCA90, "kron_store.dlg"                      },
	{0x035039A8, "a2c10_gizoidshimmer1.gff"            },
	{0x0350A21C, "icn_l_snglatk.ncgr.small"            },
	{0x03545F31, "cfx_rtip1.nsbtp.small"               },
	{0x03596B5C, "pl_pz_vox_lb.nsbmd.small"            },
	{0x035B2950, "jurnl_arrownlit.ncgr.small"          },
	{0x03625229, "a2c10_gizoidshimmer2.gff"            },
	{0x036CB6B0, "prtl_shdgen_3.ncgr.small"            },
	{0x036D70CA, "chao_fc_g.nsbtp.small"               },
	{0x036E15DA, "itm_ace_nrgal_2.ncgr.small"          },
	{0x03746AAA, "a2c10_gizoidshimmer3.gff"            },
	{0x037BC3A4, "lvl_but_bg.ncgr.small"               },
	{0x03825C15, "prf_tmbar_of00.ncgr.small"           },
	{0x03841E86, "prtl_aeqhap_2.ncgr.small"            },
	{0x038BE160, "prtl_fwrgen_3.ncgr.small"            },
	{0x038D738E, "itm_fotslprg_1.ncgr.small"           },
	{0x03A58965, "chao_jl_2.ncgr.small"                },
	{0x03A5AABE, "cbt_focus_button_failure.ncgr.small" },
	{0x03AB6AC9, "bmp_ex_run.nsbca"                    },
	{0x03AFC295, "lvl_tlgbx_ms22.ncgr.small"           },
	{0x03B3A5D1, "prtl_ffmrel_1.ncgr.small"            },
	{0x03B8C23A, "cmbtbg_a1_s6.nsbta.small"            },
	{0x03B9902E, "pl_mv_boa_flt.nsbca"                 },
	{0x03C081E4, "pl_am_noc_gcsn.nsbta.small"          },
	{0x03C4D668, "kro_cb_damage.nsbca"                 },
	{0x03CA92A2, "itm_glv_midnt_2.ncgr.small"          },
	{0x03CFC2C8, "fx_a2_s4_ex_env.nsbmd.small"         },
	{0x03DA82C4, "pl_ob_spk_dn.nsbca"                  },
	{0x03DFB97A, "spell_empowered_2.spl"               },
	{0x03F1D1FB, "spell_empowered_3.spl"               },
	{0x03F3AA96, "credits_reset.gff"                   },
	{0x03F9607E, "fx_vox_g_ea_l.nsbmd.small"           },
	{0x04036AFF, "prtl_wsgsca_3.ncgr.small"            },
	{0x041BF694, "jrl_buttn_on00.ncgr.small"           },
	{0x0422FCFC, "pfx_glo1_red.emit"                   },
	{0x042AD64E, "chap7scr_bot.tga"                    },
	{0x042BC599, "omg_cb_rund.nsbca"                   },
	{0x042FEF0C, "a1c4_bathroom_reset.gff"             },
	{0x043A6241, "a1_w0_cc_01.ncgr.small"              },
	{0x044FABA9, "a2c8_sonic_approach_guide.gff"       },
	{0x0453D8B6, "a1_s6_underlay.nbfp"                 },
	{0x0453D8B9, "a1_s6_underlay.nbfs"                 },
	{0x0454F1DD, "itm_ace_staplr_3.ncgr.small"         },
	{0x045BAA2C, "cbt_focusringb.ncer.small"           },
	{0x045DDBD1, "cfx_ertp1_roc.emit"                  },
	{0x045E82E4, "fx_shason_nuke.nsbtx.small"          },
	{0x04669285, "cho_ts_02.ncgr.small"                },
	{0x046B979F, "itm_ace_spdbr_0.ncgr.small"          },
	{0x0479DBBF, "fx_gnp_beam.nsbtx.small"             },
	{0x048D3979, "chao_gc_2.ncgr.small"                },
	{0x0492E7CA, "crm_cb_pattack03.nsbca"              },
	{0x04A01B8C, "lvl_tmstr_ms20.ncgr.small"           },
	{0x04A17445, "exp_ghz1_mm11.ncgr.small"            },
	{0x04A22038, "prtl_amysus_2.ncgr.small"            },
	{0x04B9572E, "prtl_timcal_0.ncgr.small"            },
	{0x04C094DC, "itm_glv_light_0.ncgr.small"          },
	{0x04CB7805, "prtl_shddef_3.ncgr.small"            },
	{0x04D91B17, "icn_l_wmp_tornado.nclr.small"        },
	{0x04DC20E5, "egm_cb_attackend.nsbca"              },
	{0x04DCFE5E, "icn_s_str_move.ncgr.small"           },
	{0x04DE5EA7, "cfx_nflp1_fls.emit"                  },
	{0x04E53954, "c_omro_om.nsbca"                     },
	{0x04ECAE28, "enter_twilight.gda"                  },
	{0x04FA2E31, "loc_victr_fr1.ncgr.small"            },
	{0x04FD85BF, "wmp_trv_03_18.gff"                   },
	{0x050682C2, "tws1_minimap.nclr.small"             },
	{0x0509C08D, "main_pnl_off10.ncgr.small"           },
	{0x050E671C, "str_tmdbx_ms11.ncgr.small"           },
	{0x0514C745, "cfx_ixmp1_ixt.emit"                  },
	{0x05251F78, "cfx_empp1_bem.emit"                  },
	{0x052956E1, "chao_jc_r.nsbtx.small"               },
	{0x0533938D, "pfx_lef1_lef.emit"                   },
	{0x05402869, "chao_jr_0.ncgr.small"                },
	{0x054208CC, "itm_chao_egg_3.ncgr.small"           },
	{0x05460BE7, "exp_ghz2_mm30.ncgr.small"            },
	{0x05464209, "exp_ghz1_mm33.ncgr.small"            },
	{0x055053D4, "mmp_lgbar_00.ncgr.small"             },
	{0x05542131, "upper_green_ring_03.ncgr.small"      },
	{0x055ABF82, "sco_cb_counter.nsbca"                },
	{0x055C64FD, "prtl_riadne.nclr.small"              },
	{0x05604D71, "conv_portr_20.ncgr.small"            },
	{0x056E0A6E, "crm_cb_blockd.nsbca"                 },
	{0x056E0E6F, "itm_ace_prune_1.ncgr.small"          },
	{0x0573BCB3, "loc_ambsh_dt0.ncgr.small"            },
	{0x05789AAA, "crm_ex_twitch.nsbca"                 },
	{0x057E6BE6, "fx_gas.nsbmd.small"                  },
	{0x058365F0, "charpro_stat11.ncgr.small"           },
	{0x0587D73A, "egb_cb_kd.nsbca"                     },
	{0x058B6603, "stronghold_tut_bs.tga"               },
	{0x058C50FB, "fx_amy_whirl.nsbta.small"            },
	{0x058ED407, "prtl_fsgmad_1.ncgr.small"            },
	{0x058EFF1A, "chao_ia_3.ncgr.small"                },
	{0x0596EE05, "cfx_gfir_prj.nsbta.small"            },
	{0x0598F413, "main_pnl_on14.ncgr.small"            },
	{0x05A715AB, "str_tsmbx_ms11.ncgr.small"           },
	{0x05BAA3CF, "spell_blowkiss.spl"                  },
	{0x05E3E873, "prtl_rougeoff_2.ncgr.small"          },
	{0x05E43C68, "a2c10_partyjump.gff"                 },
	{0x05E66BAA, "itm_accangam_3.ncgr.small"           },
	{0x05E6768B, "cfx_bub1.nsbtp.small"                },
	{0x05F8B1F9, "kdr_cb_attackc.nsbca"                },
	{0x05FABE90, "a2_s2_i05.walk.tga"                  },
	{0x05FEB83D, "reward.gui"                          },
	{0x06041646, "prtl_wspsad_0.ncgr.small"            },
	{0x061EA4C0, "son_ex_off.nsbca"                    },
	{0x062DDCCE, "fx_ix_beamlg.nsbtx.small"            },
	{0x0630CF67, "fx_gren_rou.nsbtx.small"             },
	{0x06351BC8, "cfx_rbep1_exp.emit"                  },
	{0x063BA619, "lower_textbox_00.ncgr.small"         },
	{0x063C2829, "a2c8_scy_nrrgalswarmattack.gff"      },
	{0x0647512C, "prtl_shdhel_0.ncgr.small"            },
	{0x0649A53F, "icn_l_catk_at.ncgr.small"            },
	{0x064EE88B, "reward_scrollbg.tga"                 },
	{0x06609FCE, "itm_fotstlbt_0.ncgr.small"           },
	{0x066604A2, "fx_els_en_aa.nsbmd.small"            },
	{0x0672504F, "swa_ex_walk.nsbca"                   },
	{0x067A6C8B, "omg_cb_pattack03.nsbca"              },
	{0x0685D7E8, "cbt_focus_button_correct.ncgr.small" },
	{0x06903E79, "hnpc_for_aa.nsbtx.small"             },
	{0x069F9589, "exl_menui_ms00.ncgr.small"           },
	{0x06A39E82, "prt_tp_m_cre.ncgr.small"             },
	{0x06C08C1A, "pfx_spk1.nsbmd.small"                },
	{0x06CBECD5, "cfx_slpp1_zee.emit"                  },
	{0x06D7CBE0, "profile_bottom_equip.gui"            },
	{0x06DB165E, "pl_pz_vox_pth.nsbta.small"           },
	{0x06E0E9EC, "conv_bubble_ts01.ncgr.small"         },
	{0x06E33B5E, "tut_touch.gda"                       },
	{0x06EB9E59, "upper_red_ring_01.ncgr.small"        },
	{0x06EE891D, "btn_s_lftarw_of.ncgr.small"          },
	{0x06F636E5, "invent_scroll_00.ncgr.small"         },
	{0x06F6DB5E, "prtl_scygen_1.ncgr.small"            },
	{0x07214A88, "itm_ace_delux_2.ncgr.small"          },
	{0x0722B758, "cfx_rtip1_rta.emit"                  },
	{0x07325057, "nrg_ab.nsbmd.small"                  },
	{0x073C8A95, "plt_fpa2_knuckles.plo"               },
	{0x0750E51D, "itm_ace_cural_2.ncgr.small"          },
	{0x07541A91, "explorehudmenumin.gui"               },
	{0x0757A845, "fx_steam.nsbmd.small"                },
	{0x07651FD2, "rou_cb_support02.nsbca"              },
	{0x077498AB, "a2c10_noct_stop_move.gff"            },
	{0x0786B8E4, "ecm_aa.nsbmd.small"                  },
	{0x078AEA30, "jrl_tsmbr_on01.ncgr.small"           },
	{0x07988C12, "pl_pu_emrld1.nsbmd.small"            },
	{0x079CFC23, "cbt_rndnum3.ncgr.small"              },
	{0x07A25080, "jurnl_mis_line02.ncgr.small"         },
	{0x07A572A5, "fx_vox_r_ea_m.nsbtx.small"           },
	{0x07A60A02, "adv_omega.gda"                       },
	{0x07A8093D, "cfx_basp1_fls.emit"                  },
	{0x07ABA985, "fx_a2_s7_i03.nsbtp.small"            },
	{0x07B36769, "fx_vox_p_ea_s.nsbtx.small"           },
	{0x07C0FBDE, "exp_ghq1_mm13.ncgr.small"            },
	{0x07C15268, "fx_a1_s5_ex_env.nsbmd.small"         },
	{0x07C268CD, "prtl_tdadhap_3.ncgr.small"           },
	{0x07C9EBE9, "buz_cb_attackc.nsbca"                },
	{0x07CC076F, "profile_bottom_detail.gui"           },
	{0x07D1D79D, "cbt_cmdframe11.ncgr.small"           },
	{0x07D4DDF3, "itm_ace_beaca_3.ncgr.small"          },
	{0x07D8E3A4, "itm_fotkevbt_3.ncgr.small"           },
	{0x07DAEC86, "a2c10_campanemerald.gff"             },
	{0x07DD0046, "prtl_vxcleugen_1.ncgr.small"         },
	{0x07ED47ED, "knu_cb_blockin.nsbca"                },
	{0x07EE9AD4, "chao_ji_g.nsbtp.small"               },
	{0x07F85107, "plt_fp_shade.plo"                    },
	{0x08059C03, "fx_leaf.nsbmd.small"                 },
	{0x0813A131, "fx_a2_s3_ex_nrg.nsbca"               },
	{0x081CAFFF, "prtl_vxcrblgen_0.ncgr.small"         },
	{0x0824F5F2, "prtl_sbtmad_3.ncgr.small"            },
	{0x08372CBF, "chao_ea_r.nsbmd.small"               },
	{0x0838EC90, "pl_pz_rc_sw.nsbmd.small"             },
	{0x083F0DBD, "cbt_info_slide.nanr.small"           },
	{0x0859EB03, "main_action_on01.ncgr.small"         },
	{0x085BFCA6, "rwd_ttxtb_ms01.ncgr.small"           },
	{0x08629DF5, "prtl_fmncon_0.ncgr.small"            },
	{0x0868C3D2, "cfx_poip1.nsbtx.small"               },
	{0x086B31E9, "prtl_vxcbadgen_3.ncgr.small"         },
	{0x0872C0B0, "bif_amy.nbfp"                        },
	{0x0872C0B3, "bif_amy.nbfs"                        },
	{0x0873BB12, "pfx_sms1.nsbtp.small"                },
	{0x0874C5B8, "wmp_airp_of10.ncgr.small"            },
	{0x0879F4D9, "sco_ex_fall.nsbca"                   },
	{0x08975742, "nsw_ex_run.nsbca"                    },
	{0x08B99C5B, "chao_fe_3.ncgr.small"                },
	{0x08C1A3A1, "vox_ex_walk.nsbca"                   },
	{0x08C8CBFA, "cfx_sson_flare.nsbmd.small"          },
	{0x08CC2963, "son_bluebomber_0.gff"                },
	{0x08CEC05C, "itm_fotttnhy_0.ncgr.small"           },
	{0x08D9AC94, "str_tab_02.ncgr.small"               },
	{0x08DA329E, "pl_pz_ppd_up.nsbta.small"            },
	{0x08DE41E4, "son_bluebomber_1.gff"                },
	{0x08EE72CA, "pro_tab_00.ncgr.small"               },
	{0x08F26A40, "fx_egg_wrc_prj.nsbca"                },
	{0x0904FE12, "itm_fotgrshy_0.ncgr.small"           },
	{0x090980A8, "prf_crtbb_hi00.ncgr.small"           },
	{0x0910654D, "pl_mv_bop_aa.nsbmd.small"            },
	{0x09193581, "ch8_scylla_tyxisfadeintoarena.gff"   },
	{0x092677BD, "bal_ex_spinfall.nsbca"               },
	{0x09303D35, "pfx_flyby.emit"                      },
	{0x093AF506, "fx_noc_jav.nsbtx.small"              },
	{0x093B0E2C, "chao_jn_r.nsbtx.small"               },
	{0x093E24AE, "pfx_lef2_lef.emit"                   },
	{0x094DF47C, "a2c8_scy_gateguardhold.gff"          },
	{0x095348BD, "prtl_egg.nclr.small"                 },
	{0x09715C6B, "btn_s_bar_on01.ncgr.small"           },
	{0x0977DFF9, "cfx_rtip1_rtb.emit"                  },
	{0x097F03A8, "sha_cb_attackend.nsbca"              },
	{0x09871191, "a1_w0_br_12.ncgr.small"              },
	{0x09926345, "a2c8_scy_tyxischamberwarpin.gff"     },
	{0x09928D6B, "a1_s3_i01_1to2.gff"                  },
	{0x09A6B392, "cmbtbg_a1_i01.nsbtx.small"           },
	{0x09A7DC46, "she_cb_pattack01.nsbca"              },
	{0x09ABF1D7, "prtl_sbtgen_0.ncgr.small"            },
	{0x09BB2AA9, "dft_scr_01.ncgr.small"               },
	{0x09BB97BA, "chap9_cha_scr_bot.tga"               },
	{0x09C19709, "lvl_pow_bspnl13.ncgr.small"          },
	{0x09C66753, "pl_pz_mem_end.nsbca"                 },
	{0x09C87F7E, "c_ambicr_cr.nsbca"                   },
	{0x09CD35D5, "cbt_victory_star2.ncer.small"        },
	{0x09CDC376, "prtl_rougedis_0.ncgr.small"          },
	{0x09D1C73F, "pl_ob_spk_hup.nsbca"                 },
	{0x09E457BD, "str_txtbr_di01.ncgr.small"           },
	{0x09ED0122, "plt_ch3_eggbot.plo"                  },
	{0x09EDB037, "a1c1_woodsman.dlg"                   },
	{0x09F1B887, "chapter10.gda"                       },
	{0x09FC6D1D, "jurnl_scroll.ncgr.small"             },
	{0x0A04C814, "prtl_sdwmad_0.ncgr.small"            },
	{0x0A0E3FB1, "itm_fotirnhy_2.ncgr.small"           },
	{0x0A0F82D6, "c_rokn_ro.nsbca"                     },
	{0x0A138D90, "exl_alttp_ms00.ncgr.small"           },
	{0x0A235D51, "icn_l_ontalk.ncgr.small"             },
	{0x0A26A20E, "egm_cb_attack.nsbca"                 },
	{0x0A3195E7, "jillcan13rg.nftr"                    },
	{0x0A33F94C, "cfx_snry_mf.nsbmd.small"             },
	{0x0A375652, "prtl_gnogen_0.ncgr.small"            },
	{0x0A4BB54E, "itm_ace_voxai_3.ncgr.small"          },
	{0x0A501159, "chao_hc_g.nsbmd.small"               },
	{0x0A532C56, "she_stb.nsbmd.small"                 },
	{0x0A57C202, "fx_giz_lzr_muz.nsbtx.small"          },
	{0x0A67E323, "plt_cod_ix.plo"                      },
	{0x0A721993, "pl_pu_chst_aa.nsbtp.small"           },
	{0x0A769CA3, "pl_ob_blkb_sb.nsbtx.small"           },
	{0x0A77C1C4, "chao_gd_r.nsbmd.small"               },
	{0x0A8493DD, "prt_tp_ome.ncgr.small"               },
	{0x0A8C1652, "prtl_voxai01.nclr.small"             },
	{0x0A9453D2, "wmp_indydot.ncgr.small"              },
	{0x0A95CA21, "roundstarttop.gui"                   },
	{0x0A9AD65D, "spell_int_zoah.spl"                  },
	{0x0AA3D275, "prt_se_prixb.ncgr.small"             },
	{0x0AA66C9D, "invent_stat_tab4.ncgr.small"         },
	{0x0AADB2D7, "chao_hb_0.ncgr.small"                },
	{0x0AB6A714, "cmbtbg_ts_ghz111.ncgr.small"         },
	{0x0AB982EC, "prtl_txrgen_0.ncgr.small"            },
	{0x0ABA49B4, "jurnl_cod_linelit00.ncgr.small"      },
	{0x0AC6AC6D, "itm_ace_fire_3.ncgr.small"           },
	{0x0ACF20B6, "a2cha_voxaib_teleport_fail.gff"      },
	{0x0AD1AE9F, "pl_pz_vox_lr.nsbtp.small"            },
	{0x0AE473B1, "itm_fotslpnm_2.ncgr.small"           },
	{0x0AE7CA43, "icn_l_oninv1.ncgr.small"             },
	{0x0AF1B5F2, "debugchao1.gda"                      },
	{0x0AF83693, "icn_s_wsl2bcho.ncgr.small"           },
	{0x0B03CE73, "debugchao2.gda"                      },
	{0x0B043F96, "prtl_amyhap_2.ncgr.small"            },
	{0x0B0B39A8, "cfx_mg_prj.nsbtx.small"              },
	{0x0B0D6EBA, "pop_tggle_of01.ncgr.small"           },
	{0x0B13D6FB, "icn_s_catk_kwh.ncgr.small"           },
	{0x0B1B18E2, "amy_cb_idle.nsbca"                   },
	{0x0B23921B, "cfx_pfbp2.nsbtp.small"               },
	{0x0B3E63CC, "charpro_bs_b04.ncgr.small"           },
	{0x0B4A3CC4, "convo2.nftr"                         },
	{0x0B502BDD, "cfx_ghlp1.nsbtp.small"               },
	{0x0B686E7E, "a1_s3_giantworm2_appear.gff"         },
	{0x0B70D3A9, "prt_sp_sdw.ncgr.small"               },
	{0x0B71C52B, "pl_am_noc_gcsf.nsbtp.small"          },
	{0x0B764A0D, "itm_act2_chaos7_3.ncgr.small"        },
	{0x0B794092, "hnpc_con_aa.nsbtx.small"             },
	{0x0B841C51, "prtl_soniccon_1.ncgr.small"          },
	{0x0B8AB837, "a1c4_tailstalk.gff"                  },
	{0x0B93FF1A, "chap5scr_top.tga"                    },
	{0x0B97DC40, "man_tsvst_on01.ncgr.small"           },
	{0x0BA68971, "plots.gda"                           },
	{0x0BAC65BC, "nrg_cb_attack.nsbca"                 },
	{0x0BB245A7, "pfx_aur1.nsbtp.small"                },
	{0x0BB7D2C2, "chao_aa_gard.nsbtx.small"            },
	{0x0BCAC210, "itm_ace_boy_2.ncgr.small"            },
	{0x0BCD089A, "cfx_rtip1_rtc.emit"                  },
	{0x0BDC80B9, "exp_tws1_mm10.ncgr.small"            },
	{0x0BE00F23, "invent_stat_tablit1.ncgr.small"      },
	{0x0BE68153, "invent_list00.ncgr.small"            },
	{0x0BE7DAA0, "prtl_shdint_2.ncgr.small"            },
	{0x0BE87613, "prtl_rougegen_3.ncgr.small"          },
	{0x0BF18054, "itm_ace_power_3.ncgr.small"          },
	{0x0BF7DCC3, "a2_s7_ex.are"                        },
	{0x0BF9ED66, "chao_ab_turn.nsbtp.small"            },
	{0x0C00521F, "chao_jt_g.nsbtp.small"               },
	{0x0C06BB8D, "c_soam_so.nsbca"                     },
	{0x0C14907E, "pl_pz_mem_trd.nsbta.small"           },
	{0x0C207816, "charpro_son4.ncgr.small"             },
	{0x0C224A44, "prtl_nrqge2_1.ncgr.small"            },
	{0x0C255631, "prtl_omggen_0.ncgr.small"            },
	{0x0C29CA38, "upper_bg_10.ncgr.small"              },
	{0x0C29EF40, "a2_s3_i01_ts.nbfp"                   },
	{0x0C29EF43, "a2_s3_i01_ts.nbfs"                   },
	{0x0C2D993C, "conv_bubble_bs12.ncgr.small"         },
	{0x0C30B4A7, "charpro_bs_a00.ncgr.small"           },
	{0x0C33BF6A, "pig_ex_run.nsbca"                    },
	{0x0C397644, "rhk_ex_liftoff.nsbca"                },
	{0x0C41E216, "prtl_knumad_2.ncgr.small"            },
	{0x0C462F8E, "chao_da_g.nsbtx.small"               },
	{0x0C48FB26, "egm_cb_victory.nsbca"                },
	{0x0C4C40EF, "cmbtbg_a2_s3.nsbtx.small"            },
	{0x0C57C6A1, "chao_jg_3.ncgr.small"                },
	{0x0C5C2012, "chao_aa_turn.nsbmd.small"            },
	{0x0C7BFB2F, "cfx_aslt_mf.nsbtx.small"             },
	{0x0C81C7AC, "prtl_sonicgen_2.ncgr.small"          },
	{0x0C90A65E, "chao_jf_g.nsbmd.small"               },
	{0x0C945ECA, "cfx_aurp1.nsbtp.small"               },
	{0x0C990E10, "cfx_glw_rlfp1.emit"                  },
	{0x0CB84B91, "amy_ex_run.nsbca"                    },
	{0x0CBDF46B, "rou_cb_attack.nsbca"                 },
	{0x0CD1CD0C, "cbt_btn_success.gui"                 },
	{0x0CD5F117, "gup_cb_dead.nsbca"                   },
	{0x0CE7B506, "pfx_arnietest01.nsbtx.small"         },
	{0x0CF5314C, "pig_cb_kd.nsbca"                     },
	{0x0D0B101D, "cfx_flap1.nsbtx.small"               },
	{0x0D24AC61, "a2c8_scy_arenacomplete.gff"          },
	{0x0D24E7E6, "prtl_shdmad_1.ncgr.small"            },
	{0x0D2AD6A2, "prt_sp_amy.ncgr.small"               },
	{0x0D2D2678, "drnl_cb_attacken.nsbca"              },
	{0x0D379AE6, "charpro_ome3.ncgr.small"             },
	{0x0D3807AA, "icn_s_patk_cre2.ncgr.small"          },
	{0x0D441296, "prtl_fwrmad_1.ncgr.small"            },
	{0x0D589CD3, "prtl_eixthrngen_3.ncgr.small"        },
	{0x0D5D032F, "crm_cb_victory.nsbca"                },
	{0x0D5DF1B1, "pop_lgbox_ms04.ncgr.small"           },
	{0x0D5F4BB5, "she_cb_attack.nsbca"                 },
	{0x0D618E5C, "pl_pz_cav_scrn.nsbtx.small"          },
	{0x0D89B211, "chao_da_r.nsbtp.small"               },
	{0x0D8ADD0E, "cfx_glw_ghsp1.emit"                  },
	{0x0D8B34DB, "adv_rouge.gda"                       },
	{0x0DAC28D5, "drnlaz_aa.nsbtx.small"               },
	{0x0DC1371A, "prtl_tailsinq_1.ncgr.small"          },
	{0x0DC4FB97, "conv_bubble_amb03.ncgr.small"        },
	{0x0DCA1683, "a2_s6_ex.walk.tga"                   },
	{0x0DCD4CD2, "pl_pu_emrld6.nsbtx.small"            },
	{0x0DD2B14F, "cbt_focusringc.ncgr.small"           },
	{0x0DEC3C67, "prtl_aeqgen_2.ncgr.small"            },
	{0x0DEF72AB, "cbt_cmdhlthfat0.ncgr.small"          },
	{0x0DEFC04B, "plt_fp_omega.plo"                    },
	{0x0DF265A5, "chao_jm_1.ncgr.small"                },
	{0x0DF2864D, "jrl_tlgbx_ms20.ncgr.small"           },
	{0x0DF79557, "jurnl_mis00.ncgr.small"              },
	{0x0DFF86E7, "nsw_aa.nsbmd.small"                  },
	{0x0E0A0D07, "usehealth.ncgr.small"                },
	{0x0E0B12E1, "a1_s2_i04_ts.nbfp"                   },
	{0x0E0B12E4, "a1_s2_i04_ts.nbfs"                   },
	{0x0E1503B8, "ghz_demo.are"                        },
	{0x0E1A5EF7, "fx_tal_spn.nsbtx.small"              },
	{0x0E1A874A, "fx_sonicring.nsbtp.small"            },
	{0x0E22313B, "cfx_rtip1_rtd.emit"                  },
	{0x0E22B169, "pl_pz_ppv_aa.nsbmd.small"            },
	{0x0E235C0B, "knu_uppercut.gff"                    },
	{0x0E297A54, "loc_ambsh_fr1.ncgr.small"            },
	{0x0E2CFF53, "pfx_grs1.nsbtx.small"                },
	{0x0E331130, "bel_cb_counter.nsbca"                },
	{0x0E37EBA2, "rou_cb_kd.nsbca"                     },
	{0x0E426CD1, "sco_cb_kd.nsbca"                     },
	{0x0E44380B, "prtl_sdwsmg_2.ncgr.small"            },
	{0x0E4C1F52, "gwr_cb_getup.nsbca"                  },
	{0x0E66E404, "icn_s_catk_hk.ncgr.small"            },
	{0x0E6DAAED, "creatures.gda"                       },
	{0x0E6EA340, "lvl_stat_tspnl12.ncgr.small"         },
	{0x0E734F7D, "ix_cb_damage.nsbca"                  },
	{0x0E77867E, "prf_tmbar_on01.ncgr.small"           },
	{0x0E7B4CB4, "son_bluebomber.gff"                  },
	{0x0E86C493, "chao_fd_g.nsbtx.small"               },
	{0x0E975411, "jrl_tlgbx_ms42.ncgr.small"           },
	{0x0E99FF15, "pig_cb_attack.nsbca"                 },
	{0x0E9AB3CE, "chao_jt_3.ncgr.small"                },
	{0x0EA55D45, "wmp_s_starb.ncgr.small"              },
	{0x0EC45CE5, "prt_se_ecm.ncgr.small"               },
	{0x0EC7A2EF, "prf_chrbx_di10.ncgr.small"           },
	{0x0ED6F66D, "gwr_cb_counter.nsbca"                },
	{0x0ED78FB0, "a1c1_first_enemy.gff"                },
	{0x0EDA15B9, "chao_gd_1.ncgr.small"                },
	{0x0EDBB286, "a1_w0_ai_11.ncgr.small"              },
	{0x0EF40466, "bsl_cb_pattack.nsbca"                },
	{0x0F086D4A, "cfx_nocp1.nsbtx.small"               },
	{0x0F2311A9, "exp_ghz1_fw12.ncgr.small"            },
	{0x0F2ABB75, "jrl_tmbox_ms01.ncgr.small"           },
	{0x0F2E977D, "lower_sonic_portrait_02.ncgr.small"  },
	{0x0F392023, "prtl_salesmg_3.ncgr.small"           },
	{0x0F481D1A, "swt_cb_damage.nsbca"                 },
	{0x0F4EC2DF, "itm_ace_drink_3.ncgr.small"          },
	{0x0F557019, "cbt_follow_transition.ncgr.small"    },
	{0x0F67CFD7, "prtl_amyoff_1.ncgr.small"            },
	{0x0F68B427, "mmp_globe_2.ncgr.small"              },
	{0x0F6A0369, "chap2scr_bot.tga"                    },
	{0x0F6BBF81, "pl_pz_ppd_rg.nsbtp.small"            },
	{0x0F706802, "cbt_black_upper.ncer.small"          },
	{0x0F79E094, "cfx_spt_slm.nsbta.small"             },
	{0x0F7D4B5A, "a1_w0_mr_01.ncgr.small"              },
	{0x0F9D725A, "shp_aa.nsbmd.small"                  },
	{0x0FA296AB, "chao_gc_g.nsbtp.small"               },
	{0x0FAA4ABE, "itm_ace_jammr_3.ncgr.small"          },
	{0x0FACFC51, "pfx_orb.nsbtp.small"                 },
	{0x0FB9F144, "int_drone.spl"                       },
	{0x0FBC8D26, "gzc_cb_getup.nsbca"                  },
	{0x0FCA4716, "chao_fd_r.nsbtp.small"               },
	{0x0FD60BCD, "prtl_fmnang_2.ncgr.small"            },
	{0x0FDBDB5A, "chao_ib_2.ncgr.small"                },
	{0x0FDDEAAF, "a2ch10_councilenter.gff"             },
	{0x0FEDE81B, "cmbtbg_a2_s6.nsbta.small"            },
	{0x0FEF630A, "a1sq_official2.gff"                  },
	{0x0FFDC42C, "icn_l_onclimb.ncgr.small"            },
	{0x10017B8B, "a1sq_official3.gff"                  },
	{0x10254B3C, "prtl_txrhum_0.ncgr.small"            },
	{0x1025C55B, "hud_lvlnum9.ncgr.small"              },
	{0x10293A10, "lizscor_aa.nsbtx.small"              },
	{0x102D7703, "charpro_she1.ncgr.small"             },
	{0x1032C97F, "itm_fotrunsp_2.ncgr.small"           },
	{0x103D22FF, "cfx_weap1.nsbtp.small"               },
	{0x1054E142, "cmbtbg_a2_s7_i02.nsbmd.small"        },
	{0x105633DC, "conversationlower.gui"               },
	{0x1067773A, "cfx_focp1.nsbtp.small"               },
	{0x106F2FF8, "effectspalette.nclr.small"           },
	{0x107759DC, "cfx_rtip1_rte.emit"                  },
	{0x10783BA2, "prtl_knudet_3.ncgr.small"            },
	{0x108B9C3B, "pl_pz_fan_spn.nsbta.small"           },
	{0x108DDD19, "icn_l_barwonprf.ncgr.small"          },
	{0x1091EB10, "ccw_nw_sw.gff"                       },
	{0x1094CDE9, "cfx_nrg_slsh.nsbtx.small"            },
	{0x10A25DA9, "chao_jq_g.nsbmd.small"               },
	{0x10A4C828, "plt_cod_masteremerald.plo"           },
	{0x10A4FBC4, "a2_w0_ul2.nbfp"                      },
	{0x10A4FBC7, "a2_w0_ul2.nbfs"                      },
	{0x10AD056B, "a1_w2_gh_00.ncgr.small"              },
	{0x10BA1195, "pl_pu_rng_aa.nsbmd.small"            },
	{0x10BC3082, "a1_w1_cc_10.ncgr.small"              },
	{0x10C85CFC, "charpro_ts_a12.ncgr.small"           },
	{0x10CAA1F7, "froggy.nsbtx.small"                  },
	{0x10D1240B, "strongholdbuttons.gui"               },
	{0x10DE1B72, "pl_mv_tor_oi.nsbtx.small"            },
	{0x10E6D051, "fx_vox_y_ea_m.nsbmd.small"           },
	{0x11128A9A, "exl_bchao_ms2.ncgr.small"            },
	{0x11181EE5, "prtl_hangen_0.ncgr.small"            },
	{0x111D6716, "fx_sha_jav_prj.nsbta.small"          },
	{0x1122878E, "c_soknamta_sb.nsbca"                 },
	{0x112436A9, "plt_storysofar.plo"                  },
	{0x112D1E7D, "tai_cb_blockin.nsbca"                },
	{0x11360486, "exp_wel1_mm13.ncgr.small"            },
	{0x11499AAF, "a1_s5_card_t1.nbfp"                  },
	{0x11499AB2, "a1_s5_card_t1.nbfs"                  },
	{0x115B4172, "prtl_shddet_2.ncgr.small"            },
	{0x116B17E6, "cfx_pwup1.nsbtx.small"               },
	{0x11894C0E, "egb_cb_blockout.nsbca"               },
	{0x1195A7A8, "exl_barow_ms11.ncgr.small"           },
	{0x11968E86, "jurnl_bs03.ncgr.small"               },
	{0x11A30124, "spell_coverd.spl"                    },
	{0x11A7DBF1, "power.ncgr.small"                    },
	{0x11CF3F83, "pfx_glo1_glo.emit"                   },
	{0x11D60A80, "prtl_scyclm_1.ncgr.small"            },
	{0x11D8BE47, "exl_globe_ms3.ncgr.small"            },
	{0x11DC9409, "journalgui_ts_blur.tga"              },
	{0x11DD7161, "cfx_rfgp1.nsbtp.small"               },
	{0x11DF97BB, "fx_vox_b_es.nsbtx.small"             },
	{0x11E8986A, "big_cb_attackc.nsbca"                },
	{0x11EDE1EF, "cfx_pjkp1.nsbtx.small"               },
	{0x11F3CF18, "arm_cb_attackc.nsbca"                },
	{0x11F62D54, "pl_pz_pod_pd.nsbca"                  },
	{0x11F79B7A, "pl_pu_chst_oa.nsbca"                 },
	{0x12086E46, "combatguib.nclr.small"               },
	{0x120C9B54, "rou_cb_stunned.nsbca"                },
	{0x12111888, "prtl_knuexh_1.ncgr.small"            },
	{0x1212AEC3, "charpro_back.ncgr.small"             },
	{0x12144FE1, "pl_pz_rc_cdh.nsbca"                  },
	{0x1223503A, "cfx_fgrp3_fgj.emit"                  },
	{0x122A86AE, "prtl_tdadgen_3.ncgr.small"           },
	{0x1235454A, "itm_ace_scan_1.ncgr.small"           },
	{0x123B97F3, "a2cha_voxaib_teleport_success.gff"   },
	{0x123BDF0B, "man_strbt_on10.ncgr.small"           },
	{0x123C5118, "fx_she_shkwv.nsbtx.small"            },
	{0x124A1830, "pl_ob_blkb_so.nsbtx.small"           },
	{0x124CECC6, "lvl_stat_dash_off.ncgr.small"        },
	{0x124E8185, "icn_l_utdoor.ncgr.small"             },
	{0x12631FCC, "exl_smbtn_on00.ncgr.small"           },
	{0x126A0EAF, "spell_covero.spl"                    },
	{0x126D79D5, "charpro_big1.ncgr.small"             },
	{0x126DC5A0, "cfx_rezp1.nsbmd.small"               },
	{0x1272094F, "rax_ex_walk.nsbca"                   },
	{0x12735E1D, "a2cha_aequi.dlg"                     },
	{0x127DCACC, "prtl_thbsad_2.ncgr.small"            },
	{0x128A624C, "icn_s_fly2.ncgr.small"               },
	{0x128B122F, "cfx_mas.nsbmd.small"                 },
	{0x12924588, "cw_s_w.gff"                          },
	{0x129A6087, "cfx_tabp1_spk.emit"                  },
	{0x12A265A6, "pro_medbx_30.ncgr.small"             },
	{0x12A37F1B, "prtl_skphap_0.ncgr.small"            },
	{0x12BB9E75, "hnpc_wdmn_idle.nsbca"                },
	{0x12BC171E, "prt_tp_e_tai.ncgr.small"             },
	{0x12BDDB9E, "prtl_egbgen_2.ncgr.small"            },
	{0x12C5E2BF, "gzc_cb_idle.nsbca"                   },
	{0x12C7ACBF, "exl_shoes_ms2.ncgr.small"            },
	{0x12D448DA, "vox_cb_counter.nsbca"                },
	{0x12FE5FA9, "c_sotaamkn_1_am2.nsbca"              },
	{0x13025ABD, "icn_s_catk_ka.ncgr.small"            },
	{0x13029F22, "a2_chb_postthelxesonicsnap.gff"      },
	{0x1307EE9D, "chao_jj_g.nsbtx.small"               },
	{0x1309B4F5, "pro_lrgbx_21.ncgr.small"             },
	{0x130A9BAD, "a2c8_scy_queenspawnhide_talk.gff"    },
	{0x1311415A, "mut_shoes_ms00.ncgr.small"           },
	{0x13129B4D, "itm_ace_medem_1.ncgr.small"          },
	{0x1315C13A, "prtl_gnasht_1.ncgr.small"            },
	{0x131BF3DA, "gwr_aa.nsbtx.small"                  },
	{0x1323B0FF, "fx_ix_bhol_prj.nsbta.small"          },
	{0x13244A3F, "c_sheso_so.nsbca"                    },
	{0x132EEB9C, "a1_w1_mt_00.ncgr.small"              },
	{0x13317055, "charpro_knu5.ncgr.small"             },
	{0x1344ABA7, "fx_spark.nsbtx.small"                },
	{0x134AF879, "cfx_slpp1.nsbtx.small"               },
	{0x1354FB0B, "c_sotaamkn_1_so.nsbca"               },
	{0x1359AB87, "fx_tal_sbot.nsbta.small"             },
	{0x1369F021, "wmp_airp_on11.ncgr.small"            },
	{0x13724E5E, "upper_bg_07.ncgr.small"              },
	{0x1377D41F, "a1_s3_ex.walk.tga"                   },
	{0x137C0258, "she_cb_attackc.nsbca"                },
	{0x138ABE87, "icn_s_patk_rou3.ncgr.small"          },
	{0x1399A1D9, "jurnl_pnl10.ncgr.small"              },
	{0x139EC350, "a1_s5_card_t2.nbfp"                  },
	{0x139EC353, "a1_s5_card_t2.nbfs"                  },
	{0x13A05A0C, "camerapanblueridge_intro.gff"        },
	{0x13A255B1, "cfx_disp1_pop.emit"                  },
	{0x13A8B256, "mut_chaob_ms01.ncgr.small"           },
	{0x13BEDC0B, "itm_ace_kevlr_3.ncgr.small"          },
	{0x13C172B2, "prtl_agzgen_2.ncgr.small"            },
	{0x13D41421, "a1_s2_ex_la.smp"                     },
	{0x13D4EB24, "plt_fp_rouge.plo"                    },
	{0x13DBFB54, "prtl_pilgen_1.ncgr.small"            },
	{0x13DCD319, "bfl_cb_pattack.nsbca"                },
	{0x13F1E56D, "pfx_bfy1.nsbmd.small"                },
	{0x13FADAC0, "itm_fotrsthy_1.ncgr.small"           },
	{0x14083E3C, "chao_hd_3.ncgr.small"                },
	{0x1408F6B6, "prtl_tailsugh_1.ncgr.small"          },
	{0x14093248, "vom_cb_pattack1.nsbca"               },
	{0x14137BFF, "wmp_trv_18_03.gff"                   },
	{0x141DE56B, "a1_s3_i01.are"                       },
	{0x14213292, "lowertextbubble.gui"                 },
	{0x14254E01, "pl_ob_blk_nb.nsbmd.small"            },
	{0x14259480, "wmp_trv_18_04.gff"                   },
	{0x142AF141, "icn_s_dash2.ncgr.small"              },
	{0x142FFDEC, "a1_s3_i02.are"                       },
	{0x1434C296, "cfx_levp1.nsbmd.small"               },
	{0x143D3541, "cfx_scap1.nsbtx.small"               },
	{0x1441EF86, "gen_cb_attackend.nsbca"              },
	{0x1442166D, "a1_s3_i03.are"                       },
	{0x144B7120, "chao_jj_r.nsbtp.small"               },
	{0x144C34F3, "prt_tp_e_egg.ncgr.small"             },
	{0x144EA1F7, "gwr_cb_pattack02.nsbca"              },
	{0x145F720E, "prtl_credet_3.ncgr.small"            },
	{0x14669E86, "pfx_alm1.nsbmd.small"                },
	{0x146854D6, "pfx_nicktest03.nsbtp.small"          },
	{0x146C52A0, "chao_fa_r.nsbmd.small"               },
	{0x146DF684, "wmp_trv_18_08.gff"                   },
	{0x148C61AE, "gzc_aa.nsbtx.small"                  },
	{0x148CCC10, "loc_abshd_it1.ncgr.small"            },
	{0x1491AEBF, "lvl_tdbld_ms14.ncgr.small"           },
	{0x149275D8, "prtl_egb.nclr.small"                 },
	{0x1498AD71, "plt_lo_gun.plo"                      },
	{0x149D6C9A, "son_ex_stealth.nsbca"                },
	{0x14A3E007, "omg_ex_walk.nsbca"                   },
	{0x14A78908, "prtl_aeqsec_2.ncgr.small"            },
	{0x14AC22C8, "pfx_smk1_col.emit"                   },
	{0x14B744EE, "a1_s5_i01.walk.tga"                  },
	{0x14CDE660, "itm_glv_spike_0.ncgr.small"          },
	{0x14D513B8, "itm_ace_tools_3.ncgr.small"          },
	{0x14DA636F, "gui_iconsdkgrey.nclr.small"          },
	{0x14DC8278, "charpro_ome.nclr.small"              },
	{0x14E1168F, "pl_pz_vox_lb.nsbtp.small"            },
	{0x14F66F0B, "pl_mv_cyc_aa.nsbmd.small"            },
	{0x150C1C7A, "prtl_gizgen_2.ncgr.small"            },
	{0x150E1DDE, "cct1_minimap.nclr.small"             },
	{0x150EDBB2, "realtime.gda"                        },
	{0x1515FD13, "a2_w0_va_10.ncgr.small"              },
	{0x151AD824, "itm_act2_chaos1_0.ncgr.small"        },
	{0x151FF252, "prtl_scylla.nclr.small"              },
	{0x15306DF5, "kdr_ex_damage.nsbca"                 },
	{0x153171DF, "a2c10_noc_doors_right.gff"           },
	{0x1533B042, "cfx_expp1_rkv.emit"                  },
	{0x15440701, "pfx_flyby.nsbmd.small"               },
	{0x1545B56B, "prt_se_wasp.ncgr.small"              },
	{0x1554647E, "pl_pz_crn_hang.nsbca"                },
	{0x1559F56A, "a1c5_p5_end.gff"                     },
	{0x156C5D77, "prtl_amygen_2.ncgr.small"            },
	{0x1583A9CC, "a2c10_remove_phase2_ix.gff"          },
	{0x158B400C, "charpro_bs_c03.ncgr.small"           },
	{0x158E769D, "egm_cb_victoryin.nsbca"              },
	{0x159BA5B7, "cmbtbg_a1_s3.nsbta.small"            },
	{0x159BE709, "gen_aa.nsbmd.small"                  },
	{0x159DDCF8, "icn_l_utclimb.ncgr.small"            },
	{0x15A84AF8, "loc_trppd_it1.ncgr.small"            },
	{0x15ADF87F, "knu_ex_jump.nsbca"                   },
	{0x15B22EF8, "prtl_timsca_3.ncgr.small"            },
	{0x15C70767, "itm_ace_commu_2.ncgr.small"          },
	{0x15D5418F, "a2c10_shadeknockdown.gff"            },
	{0x15D7B21A, "nrg_ex_idle.nsbca"                   },
	{0x15E39A66, "amy_cb_blockin.nsbca"                },
	{0x15F1276B, "prtl_wkrrel_0.ncgr.small"            },
	{0x15F4ACC2, "se_nw.gff"                           },
	{0x15FBBF8C, "cfx_ferp1.nsbmd.small"               },
	{0x15FC54B8, "chao_ja_0.ncgr.small"                },
	{0x15FCECE5, "charpro_tai5.ncgr.small"             },
	{0x16029923, "pop_tggle_on02.ncgr.small"           },
	{0x160A3918, "lvl_pow_tspnl01.ncgr.small"          },
	{0x1611FD2E, "cfx_clup1.nsbtx.small"               },
	{0x1612BA1E, "buzwasn_aa_p.ncgr.small"             },
	{0x16182158, "omg_cb_attack.nsbca"                 },
	{0x162F462C, "wel1_minimap.nclr.small"             },
	{0x16346CFB, "gui_portraits06.nclr.small"          },
	{0x1639F8E2, "prtl_sonicmad_0.ncgr.small"          },
	{0x164E58D6, "itm_ace_sturd_0.ncgr.small"          },
	{0x16509483, "pfx_briantest01.nsbtx.small"         },
	{0x165D30BC, "a2cha_overmind_buddy1_leave.gff"     },
	{0x1664EAFE, "wmp_l_airpo_00b.ncgr.small"          },
	{0x16746A68, "lvl_lrgbt_on00.ncgr.small"           },
	{0x167B4E0F, "areamove.gda"                        },
	{0x167B7E95, "gen_cb_idle.nsbca"                   },
	{0x167C74D1, "prtl_nessca_0.ncgr.small"            },
	{0x1684575C, "rpg_uparrow.ncgr.small"              },
	{0x1685373A, "chao_ic_g.nsbmd.small"               },
	{0x168A8D54, "chao_ea_1.ncgr.small"                },
	{0x168D3A32, "exl_circe_ms10.ncgr.small"           },
	{0x169644D3, "bmp_ex_damage.nsbca"                 },
	{0x169DECBB, "itm_ace_tablt_0.ncgr.small"          },
	{0x169E03F2, "slg_cb_attackend.nsbca"              },
	{0x16A4A2E1, "chao_jh_2.ncgr.small"                },
	{0x16A6209B, "a1c2_mugger_camera.gff"              },
	{0x16ACE7A5, "chao_hd_r.nsbmd.small"               },
	{0x16AF06DC, "lvl_pow_tspnl23.ncgr.small"          },
	{0x16B49F73, "a2cha_overmind_defeated_thebes.gff"  },
	{0x16C65F4F, "prtl_egginq_0.ncgr.small"            },
	{0x16CB7549, "tai_cb_dead.nsbca"                   },
	{0x16E36590, "pil_cb_dead.nsbca"                   },
	{0x16E559FC, "prtl_csxgen_0.ncgr.small"            },
	{0x16E594D1, "a2_ending.dlg"                       },
	{0x16E88F76, "bal_cb_pattack02.nsbca"              },
	{0x16ECF93C, "partyselgui_bs.tga"                  },
	{0x16F68E82, "camerapantyxsis.gff"                 },
	{0x16FD0541, "fx_a1_s4_ics.nsbmd.small"            },
	{0x17011F41, "itm_ace_lghtn_0.ncgr.small"          },
	{0x170306D0, "a2_s7_i01.are"                       },
	{0x170526BD, "wm_trndo_up.tga"                     },
	{0x17061719, "fx_gas.nsbtp.small"                  },
	{0x170B500A, "she_ex_run.nsbca"                    },
	{0x17151F51, "a2_s7_i02.are"                       },
	{0x171B2318, "egb_ex_run.nsbca"                    },
	{0x171B9111, "fx_vox_y_ea_r.nsbtx.small"           },
	{0x171EEF64, "a1c4_shadow_sonicrun.gff"            },
	{0x172737D2, "a2_s7_i03.are"                       },
	{0x1727CCCD, "lvl_tmstr_ms12.ncgr.small"           },
	{0x172847A1, "fx_ix_proj.nsbtx.small"              },
	{0x1728EF64, "exp_ghz2_mm00.ncgr.small"            },
	{0x17292586, "exp_ghz1_mm03.ncgr.small"            },
	{0x172C9CB8, "a2_w0_kr_00.ncgr.small"              },
	{0x172D5C74, "icn_s_wsl3bcho.ncgr.small"           },
	{0x173207F6, "exl_circe_ms32.ncgr.small"           },
	{0x1732ECFB, "cbt_cmdhlthfat.ncgr.small"           },
	{0x17473344, "a1_s1_i05_mm.nclr.small"             },
	{0x1747D82A, "itm_glv_gold_0.ncgr.small"           },
	{0x17487A3E, "ccz_mrz.gff"                         },
	{0x1751D4F3, "fx_a2_s7_env.nsbta.small"            },
	{0x17633AB3, "itm_glv_rug_1.ncgr.small"            },
	{0x1774AE38, "camerapanmysticruins.gff"            },
	{0x177E0ECE, "cfx_ch5.nsbmd.small"                 },
	{0x1783465A, "debugaccessories.gda"                },
	{0x179171CE, "main_pnl_off02.ncgr.small"           },
	{0x1794AC92, "animations.gda"                      },
	{0x1795C24C, "cfx_empp1.nsbtx.small"               },
	{0x179AEFE7, "cho_lgbtn_hi03.ncgr.small"           },
	{0x179CD7EB, "itm_accrfrsh_3.ncgr.small"           },
	{0x17A1F98F, "hnpc_kid_aa.nsbmd.small"             },
	{0x17A3442D, "fx_egb_beam.nsbmd.small"             },
	{0x17C3C2FB, "chasechapter1.gda"                   },
	{0x17CDBD28, "exp_ghz2_mm22.ncgr.small"            },
	{0x17D0BD52, "cfx_acdp1_acd.emit"                  },
	{0x17DCA30A, "journaltab.gui"                      },
	{0x17E3BFD7, "tut_flee.gda"                        },
	{0x17E7FEB2, "conv_portr_12.ncgr.small"            },
	{0x17EDAFD5, "fx_els_en_aa.nsbtp.small"            },
	{0x17EFCF74, "prtl_tailscon_3.ncgr.small"          },
	{0x17F23E35, "cfx_ixdp2.nsbmd.small"               },
	{0x17F528F3, "rou_cb_counter.nsbca"                },
	{0x17FF93E9, "a1c5_p5_shipattacked_b.gff"          },
	{0x17FFE4FA, "gwr_ex_run.nsbca"                    },
	{0x18021444, "fx_vox_b_ea_s.nsbta.small"           },
	{0x180E96AB, "pl_pz_rc_trd.nsbtx.small"            },
	{0x1811AC6A, "a1c5_p5_shipattacked_c.gff"          },
	{0x181A7861, "a2c10_gizoidtalks.gff"               },
	{0x18259089, "a2_s7_ex_la_soundmap.gda"            },
	{0x1835AE13, "prtl_guncodis_3.ncgr.small"          },
	{0x18389CDB, "fx_tyx_prj.nsbmd.small"              },
	{0x183F41E5, "chao_jn_0.ncgr.small"                },
	{0x1848374D, "pfx_spk1.nsbtp.small"                },
	{0x184C428C, "jurnl_cod_line01.ncgr.small"         },
	{0x18520C46, "a1c3_roadblock_gone.gff"             },
	{0x1856F618, "fx_wtrfl_strk.nsbtx.small"           },
	{0x186308E7, "vox_ex_run.nsbca"                    },
	{0x18645F92, "plt_a1sq_timmy.plo"                  },
	{0x187616AA, "prf_tboxa_ms11.ncgr.small"           },
	{0x187B556F, "chao_ea_g.nsbtx.small"               },
	{0x187F9FF0, "stotitle.gui"                        },
	{0x187FB5A3, "fleeduo.gda"                         },
	{0x188A26D2, "cfx_ch7_spkp1.emit"                  },
	{0x188FBD98, "largeportrait.gui"                   },
	{0x189C7E94, "itm_ace_candy_1.ncgr.small"          },
	{0x189D67F9, "charpro_egg4.ncgr.small"             },
	{0x18B25454, "a2scy_drone1.dlg"                    },
	{0x18B3C8E0, "fx_a2_s1_ex_env.nsbtx.small"         },
	{0x18B7DBAD, "item10.itm"                          },
	{0x18C1D60F, "a1_s5_i02.walk.tga"                  },
	{0x18C204F2, "cfx_crsp1.nsbtx.small"               },
	{0x18C46CD5, "a2scy_drone2.dlg"                    },
	{0x18C9F42E, "item11.itm"                          },
	{0x18D60249, "prf_tpanl_ms00.ncgr.small"           },
	{0x18DC0CAF, "item12.itm"                          },
	{0x18E12597, "itm_ace_ice_2.ncgr.small"            },
	{0x18E5B6D1, "pl_pz_pod_aa.nsbtx.small"            },
	{0x18ED7CAA, "chao_jg_r.nsbmd.small"               },
	{0x18EE2530, "item13.itm"                          },
	{0x19003DB1, "item14.itm"                          },
	{0x19125632, "item15.itm"                          },
	{0x1913B416, "itm_glv_grip_0.ncgr.small"           },
	{0x19151858, "pl_pz_ppv_up.nsbca"                  },
	{0x1918FE11, "loc_victr_es1.ncgr.small"            },
	{0x191ECB2C, "cbt_focusring.ncgr.small"            },
	{0x191FDCE2, "crm_ex_fly.nsbca"                    },
	{0x1922ABDD, "a2_s5_ex_slide1.gff"                 },
	{0x19246EB3, "item16.itm"                          },
	{0x1926F1F9, "chao_ge_0.ncgr.small"                },
	{0x192876B9, "prtl_fmnrel_1.ncgr.small"            },
	{0x192BCE92, "rwd_tlgbx_ms11.ncgr.small"           },
	{0x1934C45E, "a2_s5_ex_slide2.gff"                 },
	{0x19368734, "item17.itm"                          },
	{0x19376E05, "bsl_ex_walk.nsbca"                   },
	{0x193AD8E9, "itm_glv_work_2.ncgr.small"           },
	{0x193B7CE5, "itm_ace_gunof_2.ncgr.small"          },
	{0x193CBBB7, "a2c10_hideix.gff"                    },
	{0x1946DCDF, "a2_s5_ex_slide3.gff"                 },
	{0x19489FB5, "item18.itm"                          },
	{0x1948FD9B, "fx_a1_s5_ex_env.nsbtp.small"         },
	{0x195AB836, "item19.itm"                          },
	{0x1963FE3A, "cfx_gtpp1.nsbmd.small"               },
	{0x1970F99A, "itm_ace_wpnp1_0.ncgr.small"          },
	{0x197AD00D, "prf_tpanl_ms22.ncgr.small"           },
	{0x198B6A04, "pl_ob_blkn_sr.nsbmd.small"           },
	{0x198D4736, "fx_leaf.nsbtp.small"                 },
	{0x199364E8, "prt_se_bmbpw.ncgr.small"             },
	{0x19BED7F2, "chao_ea_r.nsbtp.small"               },
	{0x19C097C3, "pl_pz_rc_sw.nsbtp.small"             },
	{0x19C1F653, "charpro_sdw3.ncgr.small"             },
	{0x19CA0372, "ehd_fow_huv.ncgr.small"              },
	{0x19CE65E3, "man_strbt_of01.ncgr.small"           },
	{0x19D11020, "journalbackbutton.gui"               },
	{0x19DCBA66, "a2c8_scy_queenspawn_pre_appear.gff"  },
	{0x19E01740, "prtl_rougesmg_3.ncgr.small"          },
	{0x19E30D8D, "fx_shason_nuke.nsbta.small"          },
	{0x19E575B8, "itm_glv_ice_3.ncgr.small"            },
	{0x19F79591, "ehd_circl_hi00.ncgr.small"           },
	{0x19FA1F72, "prtl_salesca_2.ncgr.small"           },
	{0x19FBE251, "a1c3_scorpion_attack.gff"            },
	{0x19FD2D82, "itm_accspkam_3.ncgr.small"           },
	{0x19FE6668, "fx_gnp_beam.nsbta.small"             },
	{0x1A0188F9, "prtl_outhos_3.ncgr.small"            },
	{0x1A05D5F9, "white6464.ncgr.small"                },
	{0x1A068083, "prtl_caphap_2.ncgr.small"            },
	{0x1A098391, "ssf_tlgbx_ms11.ncgr.small"           },
	{0x1A0B1576, "a2c10_ix_winning.gff"                },
	{0x1A19C837, "movement09_bs.tga"                   },
	{0x1A1FA28A, "plt_cod_chaosemerald.plo"            },
	{0x1A23CAAC, "cho_tsmbx_ms02.ncgr.small"           },
	{0x1A28B79A, "chao_ic_1.ncgr.small"                },
	{0x1A315960, "sha_cb_victoryin.nsbca"              },
	{0x1A31E77E, "prtl_vxcliggen_3.ncgr.small"         },
	{0x1A3D3E64, "gzg_cb_damage.nsbca"                 },
	{0x1A4EBBDC, "gui_items_chaoegg.nclr.small"        },
	{0x1A4FC7BA, "cfx_crvp1_rev.emit"                  },
	{0x1A50772D, "cfx_sson_flare.nsbtp.small"          },
	{0x1A592B13, "prtl_sonicamu_0.ncgr.small"          },
	{0x1A5D7521, "knu_ex_idle.nsbca"                   },
	{0x1A641886, "itm_ace_pring_2.ncgr.small"          },
	{0x1A6423AD, "commandselectionring.gui"            },
	{0x1A70526E, "prtl_sonicdet_1.ncgr.small"          },
	{0x1A719BBC, "itm_place64.ncgr.small"              },
	{0x1A7968D9, "prtl_sonicsmg_2.ncgr.small"          },
	{0x1A7AA748, "wmp_trv_75_72.gff"                   },
	{0x1A826839, "exl_circl_ms10.ncgr.small"           },
	{0x1A830E1D, "str_tlgbx_ms10.ncgr.small"           },
	{0x1A89B008, "sso_ex_idle.nsbca"                   },
	{0x1A8CBFC9, "wmp_trv_75_73.gff"                   },
	{0x1A94A63B, "camerapankronforcefield.gff"         },
	{0x1A9AF837, "exp_cct1_mm21.ncgr.small"            },
	{0x1A9CD9D8, "a2c10_advgizoid_attack_begin.gff"    },
	{0x1A9ED84A, "wmp_trv_75_74.gff"                   },
	{0x1AA3EFF2, "a2_s7_card_t1.nbfp"                  },
	{0x1AA3EFF5, "a2_s7_card_t1.nbfs"                  },
	{0x1AA97A50, "slg_ex_run.nsbca"                    },
	{0x1AAB01E5, "cbt_victory_star.ncgr.small"         },
	{0x1ABBEA74, "chao_gd_g.nsbtx.small"               },
	{0x1AC8B3F7, "charpro_stat_tab3.ncgr.small"        },
	{0x1ACCE3CB, "a2cha_betavoxai.dlg"                 },
	{0x1ACDD4D2, "itm_fotsnklt_1.ncgr.small"           },
	{0x1AD385CE, "icn_l_mm_cyclone.ncgr.small"         },
	{0x1AD5787C, "icn_l_wmp_airport.ncgr.small"        },
	{0x1ADC96E1, "gnp_cb_damage.nsbca"                 },
	{0x1AE39ADF, "chao_fe_r.nsbtx.small"               },
	{0x1B050870, "jrl_tsmbr_on_11.ncgr.small"          },
	{0x1B0D044E, "item20.itm"                          },
	{0x1B14C101, "pfx_bub2.nsbtx.small"                },
	{0x1B15393C, "charpro_ts_b11.ncgr.small"           },
	{0x1B1E9873, "invent_stat00.ncgr.small"            },
	{0x1B1F1CCF, "item21.itm"                          },
	{0x1B2710EF, "fx_vox_r_ea_r.nsbmd.small"           },
	{0x1B2735FD, "exl_circl_ms32.ncgr.small"           },
	{0x1B313550, "item22.itm"                          },
	{0x1B32C487, "cfx_fatp1_glw.emit"                  },
	{0x1B35C138, "invent_ts_panel.tga"                 },
	{0x1B434DD1, "item23.itm"                          },
	{0x1B54D6EF, "a1_s1_ex_playerwinchsouth.gff"       },
	{0x1B556652, "item24.itm"                          },
	{0x1B61A8A4, "a2_s2_ex_la_soundmap.gda"            },
	{0x1B64F276, "hud_lvlnum4.ncgr.small"              },
	{0x1B677ED3, "item25.itm"                          },
	{0x1B70684C, "jurnl_mis_linelit05.ncgr.small"      },
	{0x1B755DFE, "prtl_eixfm1_0.ncgr.small"            },
	{0x1B799754, "item26.itm"                          },
	{0x1B7FF9D6, "a1c1_first_enemy_hold.gff"           },
	{0x1B85AE90, "itm_ace_rejuv_0.ncgr.small"          },
	{0x1B89A3B1, "a1c1_rouge.dlg"                      },
	{0x1B8BAFD5, "item27.itm"                          },
	{0x1B8FA89B, "conv_ambient_bkgd.tga"               },
	{0x1B96508D, "cbt_player_ambush.nanr.small"        },
	{0x1B9DC856, "item28.itm"                          },
	{0x1BA4F5F0, "a1c1_rouge_stronghold.gff"           },
	{0x1BAB2017, "chapter8charyb.gda"                  },
	{0x1BAB205E, "chapter8charyb.gff"                  },
	{0x1BAFE0D7, "item29.itm"                          },
	{0x1BB26777, "fx_ix_beamlg.nsbta.small"            },
	{0x1BB5389C, "itm_ace_leaf_0.ncgr.small"           },
	{0x1BB8B5D7, "chao_fa_3.ncgr.small"                },
	{0x1BBBA47F, "cfx_snry_mf.nsbtp.small"             },
	{0x1BBF786F, "fx_cheese_prj.nsbca"                 },
	{0x1BD7BC8C, "chao_hc_g.nsbtp.small"               },
	{0x1BDDEE3B, "knu_cb_victory.nsbca"                },
	{0x1BE89D86, "itm_glvspncl_1.ncgr.small"           },
	{0x1BEA1327, "pil_cb_getup.nsbca"                  },
	{0x1BF1B307, "gzd_cb_getup.nsbca"                  },
	{0x1BFF6CF7, "chao_gd_r.nsbtp.small"               },
	{0x1C0EFC5E, "itm_fotsnkkb_0.ncgr.small"           },
	{0x1C11A87A, "a2c7_factory_door_right.gff"         },
	{0x1C11ACB5, "hnpc_bcap_idle.nsbca"                },
	{0x1C252F4A, "lights.gda"                          },
	{0x1C2C63EE, "ghz_ccz.gff"                         },
	{0x1C2F8FBA, "pl_am_noc_gco.nsbca"                 },
	{0x1C304B62, "gzd_cb_counter.nsbca"                },
	{0x1C41FB7D, "c_sotaamkn_1_so2.nsbca"              },
	{0x1C505D35, "fx_els_wt_aa.nsbtx.small"            },
	{0x1C5098C6, "prt_se_armdr.ncgr.small"             },
	{0x1C5CA89A, "ending_movie.gff"                    },
	{0x1C66E2A2, "jrl_tlgbr_on00.ncgr.small"           },
	{0x1C87BC6C, "gzg_ex_walk.nsbca"                   },
	{0x1C99BA19, "pl_pu_scr_ghs_bg.nsbtx.small"        },
	{0x1CA4CF05, "cfx_wptp1.nsbtx.small"               },
	{0x1CA55880, "fx_a1_s2_ex_env.nsbtx.small"         },
	{0x1CC5F375, "vox_ove_aa.nsbmd.small"              },
	{0x1CC887E2, "prtl_krwgen_0.ncgr.small"            },
	{0x1CCC6730, "a1_s5_i03.walk.tga"                  },
	{0x1CCD2DEC, "she_cb_kd.nsbca"                     },
	{0x1CEA2C10, "c_sotaamkn_1_sb2.nsbca"              },
	{0x1CEA42D7, "pfx_smk1.nsbmd.small"                },
	{0x1CEA7997, "omg_ex_fall.nsbca"                   },
	{0x1CF5A3F5, "min_circl_ms11.ncgr.small"           },
	{0x1CF91893, "a2_s7_card_t2.nbfp"                  },
	{0x1CF91896, "a2_s7_card_t2.nbfs"                  },
	{0x1CFF33F5, "chao_jr_r.nsbmd.small"               },
	{0x1D0B9CFC, "prtl_skpgen_0.ncgr.small"            },
	{0x1D0ED126, "exl_jurnl_ms0.ncgr.small"            },
	{0x1D126338, "prtl_boysca_3.ncgr.small"            },
	{0x1D29FD4E, "fx_vox_r_ea_m.nsbta.small"           },
	{0x1D2AD78C, "itm_fotsnkrg_2.ncgr.small"           },
	{0x1D2C2E40, "prtl_grxgen_1.ncgr.small"            },
	{0x1D32ED66, "icn_l_patkknu.ncgr.small"            },
	{0x1D34256B, "a1c4_shadow_shadowfight.gff"         },
	{0x1D37F212, "fx_vox_p_ea_s.nsbta.small"           },
	{0x1D39D552, "pl_am_noc_gco.nsbta.small"           },
	{0x1D3E5855, "cmbtbg_ts_ghz103.ncgr.small"         },
	{0x1D457FA9, "rhk_ex_walk.nsbca"                   },
	{0x1D4920C6, "egm_ex_jumpair.nsbca"                },
	{0x1D49FB05, "a1c1_swatbot_move.gff"               },
	{0x1D6012FF, "pfx_mot1_tlp.emit"                   },
	{0x1D622CEF, "item30.itm"                          },
	{0x1D6454ED, "kdr_cb_kd.nsbca"                     },
	{0x1D64A1B5, "armdilr_aa.nsbtx.small"              },
	{0x1D6A48DD, "conv_bubble_tab02.ncgr.small"        },
	{0x1D6FD3E4, "arm_ex_walk.nsbca"                   },
	{0x1D729194, "plt_cod_big.plo"                     },
	{0x1D744570, "item31.itm"                          },
	{0x1D865DF1, "item32.itm"                          },
	{0x1D8CA0A1, "ehd_fow_mu.ncgr.small"               },
	{0x1D987672, "item33.itm"                          },
	{0x1D9E7D2A, "gui_levelup.nclr.small"              },
	{0x1D9F32CE, "a2c8_scy_queenmorphout1.gff"         },
	{0x1DAA8EF3, "item34.itm"                          },
	{0x1DB14B4F, "a2c8_scy_queenmorphout2.gff"         },
	{0x1DB3FEBD, "ch8_scylla_arenafadein.gff"          },
	{0x1DBCA774, "item35.itm"                          },
	{0x1DBD8739, "knu_cb_blockc.nsbca"                 },
	{0x1DC363D0, "a2c8_scy_queenmorphout3.gff"         },
	{0x1DCEBFF5, "item36.itm"                          },
	{0x1DD57C51, "a2c8_scy_queenmorphout4.gff"         },
	{0x1DDBD225, "a1_w3_cc_11.ncgr.small"              },
	{0x1DE0D876, "item37.itm"                          },
	{0x1DE3CB45, "chao_aa_turn.nsbtp.small"            },
	{0x1DE3CF7B, "pl_mv_wnc_on.nsbca"                  },
	{0x1DE794D2, "a2c8_scy_queenmorphout5.gff"         },
	{0x1DECA4D4, "itm_ace_antid_0.ncgr.small"          },
	{0x1DEFC106, "jillcan12rg.nftr"                    },
	{0x1DF2F0F7, "item38.itm"                          },
	{0x1DF9AD53, "a2c8_scy_queenmorphout6.gff"         },
	{0x1DFD4F8E, "pl_pz_mem_bp.nsbca"                  },
	{0x1E01542D, "a2c10_shadedisappears.gff"           },
	{0x1E03D621, "charpro_stat_tablit4.ncgr.small"     },
	{0x1E050978, "item39.itm"                          },
	{0x1E076F39, "pl_mv_zip.nsbtx.small"               },
	{0x1E119A31, "a2_s7_i01.walk.tga"                  },
	{0x1E185191, "chao_jf_g.nsbtp.small"               },
	{0x1E202178, "cfx_rtcp1.nsbmd.small"               },
	{0x1E329420, "levelup_tech_lower.gui"              },
	{0x1E32F9B9, "a1c5_p6_begin.gff"                   },
	{0x1E476F27, "main_pnl_big13.ncgr.small"           },
	{0x1E551A7C, "chao_he_2.ncgr.small"                },
	{0x1E5D28CF, "crm_ex_jumpair.nsbca"                },
	{0x1E5F972F, "gui_combatmenugrey.nclr.small"       },
	{0x1E6431FA, "exp_tws1_mm02.ncgr.small"            },
	{0x1E66982E, "bmp_aa.nsbmd.small"                  },
	{0x1E6F5E02, "cfx_rckp1_rck.emit"                  },
	{0x1E709D70, "charpro_knu0.ncgr.small"             },
	{0x1E756389, "itm_act2_chaos3_3.ncgr.small"        },
	{0x1E759B7A, "lower_bg_06.ncgr.small"              },
	{0x1E7C3CED, "cfx_shcp1.nsbmd.small"               },
	{0x1E7DEAB9, "nes_ex_idle.nsbca"                   },
	{0x1E871168, "a2c10_reposition_ix.gff"             },
	{0x1E8D7EA3, "a1_s4_ex_la.smp"                     },
	{0x1EB17B79, "upper_bg_02.ncgr.small"              },
	{0x1EC20ECF, "chapter10nocturne3.gff"              },
	{0x1EED0BD4, "prtl_telgen_1.ncgr.small"            },
	{0x1F0D30BE, "big_cb_kd.nsbca"                     },
	{0x1F195466, "a2c8_scy_raxosattack.gff"            },
	{0x1F201839, "cfx_spk_cr4p1.emit"                  },
	{0x1F248EAD, "prtl_amymad_0.ncgr.small"            },
	{0x1F2987C8, "prtl_cresca_3.ncgr.small"            },
	{0x1F2B3E3B, "cmbtbg_a1_i01.nsbta.small"           },
	{0x1F2BBF7E, "cbt_btn_left_of.ncgr.small"          },
	{0x1F2F51A0, "pl_am_met_gbp.nsbca"                 },
	{0x1F33B019, "cbt_menubg_00.ncgr.small"            },
	{0x1F3E191E, "kdr_cb_counter.nsbca"                },
	{0x1F4C983B, "combatguigrey.nclr.small"            },
	{0x1F533844, "a1_s3_ex_la_soundmap.gda"            },
	{0x1F56E01D, "chao_jc_3.ncgr.small"                },
	{0x1F5893FE, "fx_wtrfl_splsh.nsbtx.small"          },
	{0x1F636492, "a2_w0_ul.nbfp"                       },
	{0x1F636495, "a2_w0_ul.nbfs"                       },
	{0x1F64C4E9, "chao_jk_r.nsbtx.small"               },
	{0x1F86BC4E, "nrg_cb_attack2.nsbca"                },
	{0x1F89BA55, "buz_ex_walk.nsbca"                   },
	{0x1F8DD9B7, "rwd_lgbtn_hi00.ncgr.small"           },
	{0x1F9A6217, "cfx_dusp2.nsbtx.small"               },
	{0x1F9C8ABC, "icn_s_moonwmp.ncgr.small"            },
	{0x1FAA5692, "itm_fotwrkbt_3.ncgr.small"           },
	{0x1FAC7AFE, "wmp_l_airpo_02.ncgr.small"           },
	{0x1FACFA00, "a2_s3_ex.walk.tga"                   },
	{0x1FB0304F, "pl_pu_emrld3.nsbtx.small"            },
	{0x1FB607E2, "a2_s6_ex.are"                        },
	{0x1FB75590, "item40.itm"                          },
	{0x1FBC16DA, "gennpc3b_aa.nsbtx.small"             },
	{0x1FBC85D5, "exu_tbbox_ms11.ncgr.small"           },
	{0x1FC0AA05, "jurnl_cod01.ncgr.small"              },
	{0x1FC96E11, "item41.itm"                          },
	{0x1FCDA2AF, "fx_noc_tele.nsbtx.small"             },
	{0x1FD4C594, "tyx_aa.nsbmd.small"                  },
	{0x1FDB8692, "item42.itm"                          },
	{0x1FDC4CAB, "fx_giz_lzr_muz.nsbta.small"          },
	{0x1FE59B17, "fx_frcfld.nsbtx.small"               },
	{0x1FE70209, "mut_circl_ms31.ncgr.small"           },
	{0x1FE713FD, "amy_cb_pattack1.nsbca"               },
	{0x1FECDF8D, "lvl_stat_bspnl11.ncgr.small"         },
	{0x1FED9F13, "item43.itm"                          },
	{0x1FFC6502, "bbutton.ncgr.small"                  },
	{0x1FFFB794, "item44.itm"                          },
	{0x20000449, "icn_l_catk_kt.ncgr.small"            },
	{0x20083DAA, "levelupportrait.gui"                 },
	{0x200EBA4A, "fx_sha_rift_prj.nsbta.small"         },
	{0x2011D015, "item45.itm"                          },
	{0x201577EF, "itm_glv_rub_2.ncgr.small"            },
	{0x201AC87E, "tai_cb_support06.nsbca"              },
	{0x201DC533, "prtl_krggen_1.ncgr.small"            },
	{0x2023E896, "item46.itm"                          },
	{0x202F76ED, "plt_ch1_main.plo"                    },
	{0x20360117, "item47.itm"                          },
	{0x2041894D, "gui_explorea.nclr.small"             },
	{0x20481998, "item48.itm"                          },
	{0x204B894C, "charpro_bs_a03b.ncgr.small"          },
	{0x204E8D3F, "a1_w3_mt_01.ncgr.small"              },
	{0x205A3219, "item49.itm"                          },
	{0x205CB827, "gup_cb_idle.nsbca"                   },
	{0x206B6DFF, "itm_ace_kron_1.ncgr.small"           },
	{0x20742988, "itm_glv_spark_3.ncgr.small"          },
	{0x207A378E, "jrl_tlgbx_ms12.ncgr.small"           },
	{0x20818370, "kro_ex_walk.nsbca"                   },
	{0x2083A2B8, "prtl_gunhap_1.ncgr.small"            },
	{0x20939E89, "itm_glv_crystal2.ncgr.small"         },
	{0x209774D9, "itm_fotswtbt_0.ncgr.small"           },
	{0x20A17881, "chao_ga_r.nsbmd.small"               },
	{0x20A6CC03, "spinball.nsbmd.small"                },
	{0x20B9E7AF, "pil_aa.nsbtx.small"                  },
	{0x20C07799, "cfx_curp1.nsbmd.small"               },
	{0x20C1878F, "gzd_aa.nsbtx.small"                  },
	{0x20C7A8A0, "a1c1_swatbot_defeated.gff"           },
	{0x20C9048F, "pl_pu_rngshad.nsbtx.small"           },
	{0x20D10425, "prtl_wdmnhap_2.ncgr.small"           },
	{0x20D6F851, "a1_s5_i04.walk.tga"                  },
	{0x20D8C597, "exl_sonic_ms0.ncgr.small"            },
	{0x20F17F21, "chao_ji_1.ncgr.small"                },
	{0x20F75DAE, "a2c10_ix_disappears_team.gff"        },
	{0x20F822DF, "exl_menub_on00.ncgr.small"           },
	{0x20FE1EC8, "swt_ex_run.nsbca"                    },
	{0x2106FD8C, "prtl_vxmgen_3.ncgr.small"            },
	{0x210F530E, "buz_cb_counter.nsbca"                },
	{0x21114EAC, "mut_ocrna_ms01.ncgr.small"           },
	{0x21169B3E, "a2cha_overmind_buddy3_leave.gff"     },
	{0x2118FCDF, "a2_w0_nt_20.ncgr.small"              },
	{0x2126A26B, "slg_cb_pattack.nsbca"                },
	{0x213C1A00, "charpro_tai0.ncgr.small"             },
	{0x21414D65, "prtl_oldman.nclr.small"              },
	{0x2149082D, "cfx_glw_ghrp1.emit"                  },
	{0x21526B9A, "prtl_krs.nclr.small"                 },
	{0x21739A16, "gui_portraits01.nclr.small"          },
	{0x21979013, "prtl_ncsgen_1.ncgr.small"            },
	{0x2199CD4A, "chao_jp_3.ncgr.small"                },
	{0x21B383FD, "itm_ace_goldr_1.ncgr.small"          },
	{0x21B9D630, "cbt_elem_an_fire.ncer.small"         },
	{0x21B9DE11, "a1c3_eggman_bot1.gff"                },
	{0x21C1666C, "cfx_jwl1_jwl.emit"                   },
	{0x21C71C02, "appearances.gda"                     },
	{0x21C8FB0B, "charpro_amy2.ncgr.small"             },
	{0x21CBF692, "a1c3_eggman_bot2.gff"                },
	{0x21D0CB98, "cmbtbg_a2_s3.nsbta.small"            },
	{0x21DE0F13, "a1c3_eggman_bot3.gff"                },
	{0x21E0942F, "combo.gda"                           },
	{0x21EE942C, "a1_w2_ai.nclr.small"                 },
	{0x21F2A304, "prtl_egggen_3.ncgr.small"            },
	{0x21F418A1, "adv_cream.gda"                       },
	{0x21FCC662, "lowertextbutton.gui"                 },
	{0x220085D8, "cfx_aslt_mf.nsbta.small"             },
	{0x2202E244, "a2c10_switch_hack.gff"               },
	{0x220C7E31, "item50.itm"                          },
	{0x220E13EE, "pl_pu_memrld.nsbmd.small"            },
	{0x221C2B52, "a2_s7_i02.walk.tga"                  },
	{0x221CC268, "cfx_spk1.nsbtx.small"                },
	{0x221E96B2, "item51.itm"                          },
	{0x2221A029, "prtl_thbovr_0.ncgr.small"            },
	{0x2227147D, "pl_pz_wod_aa.nsbmd.small"            },
	{0x222A08DC, "chao_jq_g.nsbtp.small"               },
	{0x22301A96, "fx_sha_rift.nsbtx.small"             },
	{0x2230AF33, "item52.itm"                          },
	{0x2230E56D, "cfx_ffrp1.nsbmd.small"               },
	{0x2239077A, "que_aa.nsbmd.small"                  },
	{0x2240B473, "a2c6_generator_sonic.gff"            },
	{0x2242C7B4, "item53.itm"                          },
	{0x2244804C, "prtl_thbhap_1.ncgr.small"            },
	{0x2245BBEC, "pl_pz_rc_cd3.nsbca"                  },
	{0x22472B89, "savegui_ts_01.tga"                   },
	{0x22484A34, "loc_ambsh_es1.ncgr.small"            },
	{0x2251CE05, "spell_medibot.spl"                   },
	{0x2254E035, "item54.itm"                          },
	{0x2265481A, "spell_temporalfield_2.spl"           },
	{0x2266F8B6, "item55.itm"                          },
	{0x22769C63, "ch5_ixescapes.gda"                   },
	{0x2277609B, "spell_temporalfield_3.spl"           },
	{0x22791137, "item56.itm"                          },
	{0x2279EF23, "cfx_rckp2_rck.emit"                  },
	{0x2287434C, "a2c10_noc_doors_left.gff"            },
	{0x22890C4F, "gen_cb_attack.nsbca"                 },
	{0x228B29B8, "item57.itm"                          },
	{0x228DF7B5, "fx_shason_nuke.nsbca"                },
	{0x2296E34A, "a1_w1_gh_00.ncgr.small"              },
	{0x2297F690, "prtl_grxhum_1.ncgr.small"            },
	{0x229AAEC0, "cfx_exbp1_exb.emit"                  },
	{0x229D4239, "item58.itm"                          },
	{0x22A17D02, "prtl_nrmge2_3.ncgr.small"            },
	{0x22A184AA, "itm_ace_eggmd_2.ncgr.small"          },
	{0x22A60E61, "a1_w0_cc_10.ncgr.small"              },
	{0x22AF5ABA, "item59.itm"                          },
	{0x22B15EB4, "camerahacknocturne.gff"              },
	{0x22BA5D1B, "chao_jc_g.nsbmd.small"               },
	{0x22D23EA5, "cho_ts_11.ncgr.small"                },
	{0x22E20D86, "chao_id_r.nsbmd.small"               },
	{0x22F14DF4, "pl_pz_brd_3hd.nsbca"                 },
	{0x230D2065, "exp_ghz1_mm20.ncgr.small"            },
	{0x2322ECCD, "debugfootwear.gda"                   },
	{0x2331E2FB, "itm_fotrunsl_2.ncgr.small"           },
	{0x2351A5C0, "c_soam_sb.nsbca"                     },
	{0x2359B571, "spell_int_nrrgal.spl"                },
	{0x235AE508, "plt_fp_shadow.plo"                   },
	{0x235AE839, "prtl_amydet_1.ncgr.small"            },
	{0x235FD26F, "main_pnl_on01.ncgr.small"            },
	{0x236143DF, "wmp_space_on00.ncgr.small"           },
	{0x2363FEA4, "prtl_amysmg_2.ncgr.small"            },
	{0x236F3570, "prt_tp_e_son.ncgr.small"             },
	{0x2373357C, "wmp_txtbr_tx06.ncgr.small"           },
	{0x2373751A, "sco_cb_attack.nsbca"                 },
	{0x23739CD6, "spell_int_nocturn_water.spl"         },
	{0x237E48F5, "debuglevelup2.gda"                   },
	{0x238C49B2, "pl_ob_blkb_mr.nsbmd.small"           },
	{0x23966EA2, "cfx_pfbp1.nsbtx.small"               },
	{0x239885CC, "attack1_bs.tga"                      },
	{0x239AD1F4, "pl_ob_blkn_sb.nsbmd.small"           },
	{0x239EE9A0, "fx_tal_spn.nsbta.small"              },
	{0x23BC3BB0, "itm_accearng_0.ncgr.small"           },
	{0x23C953ED, "cfx_ch2_spkp1.emit"                  },
	{0x23E12A6E, "prtl_ffmcon_1.ncgr.small"            },
	{0x23E38C54, "hnpc_cop_aa.nsbtx.small"             },
	{0x23EEB08F, "shpkron_aa.nsbtx.small"              },
	{0x23EF1210, "charpro_stat20.ncgr.small"           },
	{0x23F570D3, "cfx_rezp1.nsbtp.small"               },
	{0x24029CC0, "prf_tlbar_ms01.ncgr.small"           },
	{0x240660BE, "a2c10_were_moving.gff"               },
	{0x240A2314, "itm_glv_power_3.ncgr.small"          },
	{0x240DE3AA, "cbt_sonichead.ncgr.small"            },
	{0x2412BD62, "cfx_mas.nsbtp.small"                 },
	{0x2412C1CB, "str_tsmbx_ms20.ncgr.small"           },
	{0x2412F33A, "prtl_krngen_1.ncgr.small"            },
	{0x242853D7, "pfx_grs1_grs.emit"                   },
	{0x242979EE, "ch10_ix_explains.gda"                },
	{0x242A4CDE, "movementtutorial.gda"                },
	{0x243A080E, "plt_ch2_main.plo"                    },
	{0x243BE7FD, "cho_smbtn_on00.ncgr.small"           },
	{0x243FD871, "cbt_menu_up_on.ncgr.small"           },
	{0x2448B42B, "a1_s1_card_t1.nbfp"                  },
	{0x2448B42E, "a1_s1_card_t1.nbfs"                  },
	{0x244A3408, "itm_ace_clovej_1.ncgr.small"         },
	{0x24590482, "prtl_eixthrnvar_3.ncgr.small"        },
	{0x246003A3, "icn_s_patk_big3.ncgr.small"          },
	{0x2461A6D2, "item60.itm"                          },
	{0x246C9249, "cbt_healt_00.ncgr.small"             },
	{0x246C9607, "itm_ace_tyxis_2.ncgr.small"          },
	{0x2472EA4A, "itm_glvttncl_0.ncgr.small"           },
	{0x2473BF53, "item61.itm"                          },
	{0x247593DA, "chao_id_0.ncgr.small"                },
	{0x24854923, "pro_medbx_00.ncgr.small"             },
	{0x248590B5, "loc_victr_it0.ncgr.small"            },
	{0x2485D7D4, "item62.itm"                          },
	{0x248B010E, "itm_ace_gum_1.ncgr.small"            },
	{0x248D67FB, "prtl_knusad_1.ncgr.small"            },
	{0x2497F055, "item63.itm"                          },
	{0x24A6B685, "ch2_islandgone_st.gda"               },
	{0x24AA08D6, "item64.itm"                          },
	{0x24B07B50, "chao_fa_g.nsbtx.small"               },
	{0x24B21620, "a2_s2_ex_puzzle_walk_back.gff"       },
	{0x24B6B861, "upper_textbox_05.ncgr.small"         },
	{0x24BC2157, "item65.itm"                          },
	{0x24BF75A2, "swt_ex_walk.nsbca"                   },
	{0x24C3904C, "man_strbt_on02.ncgr.small"           },
	{0x24CE39D8, "item66.itm"                          },
	{0x24CFE963, "prtl_eixfm3_3.ncgr.small"            },
	{0x24D61734, "ccw_e_n.gff"                         },
	{0x24DA2BC4, "a1_s2_i04_mm.nclr.small"             },
	{0x24E05259, "item67.itm"                          },
	{0x24E345F3, "pl_pz_vox_ly.nsbmd.small"            },
	{0x24F26ADA, "item68.itm"                          },
	{0x24F3852C, "cfx_shgp1.nsbtx.small"               },
	{0x25012876, "inventorygui_bs_01.tga"              },
	{0x2504835B, "item69.itm"                          },
	{0x251230DF, "a1_s2_i02_ts.nbfp"                   },
	{0x251230E2, "a1_s2_i02_ts.nbfs"                   },
	{0x251340F7, "inventorygui_bs_02.tga"              },
	{0x2515E5F9, "pfx_lzg1.nsbmd.small"                },
	{0x2518C97B, "a1_w0_mt_00.ncgr.small"              },
	{0x251DC4CB, "tai_cb_victory.nsbca"                },
	{0x252A16E7, "pro_medbx_22.ncgr.small"             },
	{0x252B97DE, "icn_s_teleport.ncgr.small"           },
	{0x25321752, "pfx_rip1.nsbtx.small"                },
	{0x2537E143, "mut_globe_ms11.ncgr.small"           },
	{0x253E9EB2, "pfx_mst1_dsh.emit"                   },
	{0x25403774, "gzg_cb_attackend.nsbca"              },
	{0x25431120, "pl_mv_boa_row.nsbca"                 },
	{0x254571F0, "big_cb_stunned.nsbca"                },
	{0x254B02FD, "exp_tws1_mm.nclr.small"              },
	{0x254C960C, "conv_bubble_ts10.ncgr.small"         },
	{0x255E0E22, "itm_ace_echid_1.ncgr.small"          },
	{0x255F7319, "bif_son.nbfp"                        },
	{0x255F731C, "bif_son.nbfs"                        },
	{0x2570CB47, "bif_tia.nbfp"                        },
	{0x2570CB4A, "bif_tia.nbfs"                        },
	{0x2574D71D, "main_pnl_backoff.ncgr.small"         },
	{0x257990A0, "pfx_bfy1.nsbtp.small"                },
	{0x257B9CD5, "cmbtbg_a1_s5.nsbmd.small"            },
	{0x257DB54A, "charpro_bar03.ncgr.small"            },
	{0x2587DA3A, "exp_ghq1_mm00.ncgr.small"            },
	{0x258C1939, "rhk_ex_fall.nsbca"                   },
	{0x25903378, "spell_flashbang_2.spl"               },
	{0x259325CC, "exl_flash_ms3.ncgr.small"            },
	{0x259B461B, "exl_lrgbt_of00.ncgr.small"           },
	{0x25A083F5, "spell_int_final_ix.spl"              },
	{0x25A24BF9, "spell_flashbang_3.spl"               },
	{0x25A96FF6, "a1c1_amy.dlg"                        },
	{0x25B2699F, "itm_glvirncl_2.ncgr.small"           },
	{0x25B41FA4, "a1c5_p7_approachix.gff"              },
	{0x25B71B6E, "ghz2_minimap.nclr.small"             },
	{0x25BC6DC9, "cfx_levp1.nsbtp.small"               },
	{0x25D2D485, "a2scy_shop.dlg"                      },
	{0x25DE7586, "str_arrow_00.ncgr.small"             },
	{0x25EE49B9, "pfx_alm1.nsbtp.small"                },
	{0x25F3FDD3, "chao_fa_r.nsbtp.small"               },
	{0x25F8AEEB, "man_tdlcp_of01.ncgr.small"           },
	{0x25FCB0EC, "cfx_spt_acd.nsbmd.small"             },
	{0x26043A70, "swa_cb_pattack.nsbca"                },
	{0x26055F83, "sha_cb_pattack02.nsbca"              },
	{0x260588EC, "prtl_vecgen_0.ncgr.small"            },
	{0x26059217, "chao_fb_2.ncgr.small"                },
	{0x260873DC, "rhk_cb_pattack2.nsbca"               },
	{0x26190AE0, "attack3_ts.tga"                      },
	{0x26195892, "cfx_nrg_slsh.nsbta.small"            },
	{0x261A3AE3, "lvl_smlbt_hi00.ncgr.small"           },
	{0x261DA3EC, "son_ex_climb.nsbca"                  },
	{0x261FAB78, "itm_glv_metal_1.ncgr.small"          },
	{0x2621531A, "jurnl_pnl02.ncgr.small"              },
	{0x2626BC73, "a2_s7_i03.walk.tga"                  },
	{0x262CA7FE, "exp_ghq1_mm22.ncgr.small"            },
	{0x263739CD, "son_ex_slidein.nsbca"                },
	{0x264045C6, "itm_glv_dark_0.ncgr.small"           },
	{0x26435095, "fx_vox_b_ea_m.nsbtx.small"           },
	{0x2645C945, "nrg_aa_slm.nsbta.small"              },
	{0x264E6C84, "cbt_flcght_1.ncgr.small"             },
	{0x265918C0, "cfx_rfgp1_rfg.emit"                  },
	{0x26602A1F, "wmp_trv_02_73.gff"                   },
	{0x2663487F, "icn_l_ongen.ncgr.small"              },
	{0x266B8AAA, "prt_se_gentl.ncgr.small"             },
	{0x266F177D, "pl_mv_cyc_crs.nsbca"                 },
	{0x26719D62, "prtl_shdstg_1.ncgr.small"            },
	{0x2676D3B6, "hnpc_gunof_aa.nsbmd.small"           },
	{0x267E8CD9, "ixa_cb_pattack2.nsbca"               },
	{0x2690434E, "tut_follow.gda"                      },
	{0x2692A755, "debugcreatechao.gda"                 },
	{0x269387B0, "prtl_krgang_2.ncgr.small"            },
	{0x269390C3, "prtl_eixprd_3.ncgr.small"            },
	{0x269DDCCC, "a1_s1_card_t2.nbfp"                  },
	{0x269DDCCF, "a1_s1_card_t2.nbfs"                  },
	{0x26A16BE1, "prtl_vxccitgen_2.ncgr.small"         },
	{0x26A53FE1, "cfx_exbp2_exb.emit"                  },
	{0x26A60747, "conv_icon05.ncgr.small"              },
	{0x26AD18B8, "a1c1_swatbot_arrive_b.gff"           },
	{0x26AF9567, "jurnl_mis_linelit00.ncgr.small"      },
	{0x26B4CCEA, "a2c10_noflyzone_block.gff"           },
	{0x26B5F7B4, "itm_fotrunrg_1.ncgr.small"           },
	{0x26B6CF73, "item70.itm"                          },
	{0x26C8E7F4, "item71.itm"                          },
	{0x26CBB234, "pfx_flyby.nsbtp.small"               },
	{0x26CC1466, "chao_jn_g.nsbmd.small"               },
	{0x26D8DBDE, "she_cb_stunned.nsbca"                },
	{0x26DB0075, "item72.itm"                          },
	{0x26DC0591, "prtl_tailsdet_3.ncgr.small"          },
	{0x26E67147, "tai_cb_deadidle.nsbca"               },
	{0x26E76171, "spell_antidote_all.spl"              },
	{0x26E8FC2A, "cfx_clup1_luc.emit"                  },
	{0x26ED18F6, "item73.itm"                          },
	{0x26F11055, "chao_hd_g.nsbtx.small"               },
	{0x26F859F2, "icn_l_catk_sat.ncgr.small"           },
	{0x26FF3177, "item74.itm"                          },
	{0x271149F8, "item75.itm"                          },
	{0x2718C0C0, "chao_ge_r.nsbtx.small"               },
	{0x271E675D, "ch1_panintrocam.gff"                 },
	{0x27236279, "item76.itm"                          },
	{0x27255BE4, "cfx_pswp1.nsbtx.small"               },
	{0x2727C93F, "icn_s_comboatk.ncgr.small"           },
	{0x2728B451, "icn_l_utinv3.ncgr.small"             },
	{0x272EB972, "cbt_round_slide.ncer.small"          },
	{0x27357AFA, "item77.itm"                          },
	{0x273D1908, "hnpc_kid_walk.nsbca"                 },
	{0x2747937B, "item78.itm"                          },
	{0x2751434A, "drnsui_cb_attack.nsbca"              },
	{0x27514816, "camerapanvoxaia.gff"                 },
	{0x2759ABFC, "item79.itm"                          },
	{0x275B12A7, "intro_rainforest.gda"                },
	{0x275E6B54, "prtl_egghum_3.ncgr.small"            },
	{0x27636097, "camerapanvoxaib.gff"                 },
	{0x27642264, "fx_vox_b_es.nsbta.small"             },
	{0x2771403B, "pfx_ash1_noc.emit"                   },
	{0x2772AC57, "pfx_tlp1.nsbtx.small"                },
	{0x27743537, "walkmesh_sized.raw"                  },
	{0x27779052, "itm_ace_fragm_1.ncgr.small"          },
	{0x27836ABF, "cfx_ferp1.nsbtp.small"               },
	{0x2785F237, "icn_s_taphrexl.ncgr.small"           },
	{0x27887565, "lvl_pow_bspnl00.ncgr.small"          },
	{0x278956DD, "item1.itm"                           },
	{0x279B6F5E, "item2.itm"                           },
	{0x27AD87DF, "item3.itm"                           },
	{0x27B3E882, "cmra_aim.nsbtx.small"                },
	{0x27B65E6A, "a1_s1_i01.walk.tga"                  },
	{0x27BA89E1, "cfx_chlp1.nsbtx.small"               },
	{0x27BFA060, "item4.itm"                           },
	{0x27C0DBC1, "fx_she_shkwv.nsbta.small"            },
	{0x27D053E5, "buz_ex_fall.nsbca"                   },
	{0x27D1B8E1, "item5.itm"                           },
	{0x27E3D162, "item6.itm"                           },
	{0x27EBDAA4, "flee05_bs.tga"                       },
	{0x27F5E9E3, "item7.itm"                           },
	{0x28080264, "item8.itm"                           },
	{0x280CE26D, "chao_ic_g.nsbtp.small"               },
	{0x28104C55, "worldmap_airbutton.gui"              },
	{0x281A1AE5, "item9.itm"                           },
	{0x281D430E, "icn_s_catk_rk.ncgr.small"            },
	{0x2826D6C9, "dft_scr_10.ncgr.small"               },
	{0x282D4329, "lvl_pow_bspnl22.ncgr.small"          },
	{0x283492D8, "chao_hd_r.nsbtp.small"               },
	{0x283D9E15, "ch5_airbase.gda"                     },
	{0x283DCEEA, "plt_fp_cream.plo"                    },
	{0x283F2C65, "a1_s3_ex_mm.nbfp"                    },
	{0x283F2C68, "a1_s3_ex_mm.nbfs"                    },
	{0x283F3500, "cfx_bub1_pt1.emit"                   },
	{0x2841691B, "a2c6_entervault.gff"                 },
	{0x2844992F, "plt_ch3_main.plo"                    },
	{0x286785B3, "a2cha_overmindattacks.gff"           },
	{0x28749E67, "prtl_cresad_1.ncgr.small"            },
	{0x2876E92A, "levels.gda"                          },
	{0x287B5D7A, "cfx_gfir_pu.nsbmd.small"             },
	{0x2883CA20, "fx_son_whirl.nsbmd.small"            },
	{0x288557DC, "itm_fotrunkv_1.ncgr.small"           },
	{0x28967E21, "cho_lgbtn_di01.ncgr.small"           },
	{0x28B70D67, "prt_mp_big.ncgr.small"               },
	{0x28C23FC9, "itm_act2_chaos4_2.ncgr.small"        },
	{0x28C339F0, "ix_aa.nsbmd.small"                   },
	{0x28C81D00, "kro_ex_fall.nsbca"                   },
	{0x28D839C4, "prtl_gnahap_1.ncgr.small"            },
	{0x28DA9927, "cfx_atkp1_fls.emit"                  },
	{0x28DF9CBF, "prtl_krshap_2.ncgr.small"            },
	{0x28EB48C3, "spell10.spl"                         },
	{0x28FD6144, "spell11.spl"                         },
	{0x2905BA01, "cfx_ch5.nsbtp.small"                 },
	{0x290BF814, "item80.itm"                          },
	{0x290F79C5, "spell12.spl"                         },
	{0x291C5707, "swa_cb_damage.nsbca"                 },
	{0x291C86B5, "main_slt_off00.ncgr.small"           },
	{0x291E1095, "item81.itm"                          },
	{0x29219246, "spell13.spl"                         },
	{0x29225334, "cmbtbg_ts_ghz120.ncgr.small"         },
	{0x2929B550, "cbt_elem_wind.ncgr.small"            },
	{0x29302916, "item82.itm"                          },
	{0x2931A55A, "chao_jg_g.nsbtx.small"               },
	{0x2933AAC7, "spell14.spl"                         },
	{0x293551F0, "fx_arm_whirl.nsbmd.small"            },
	{0x293B6EF8, "invent_barlit03.ncgr.small"          },
	{0x29424197, "item83.itm"                          },
	{0x2945C348, "spell15.spl"                         },
	{0x2946E925, "a1_s6_ex_la.smp"                     },
	{0x29522EFF, "icn_s_patk_son1.ncgr.small"          },
	{0x29545A18, "item84.itm"                          },
	{0x2957DBC9, "spell16.spl"                         },
	{0x29586DB0, "fx_blobshadow.nsbmd.small"           },
	{0x295AAF6B, "statsinfoprojectile.gda"             },
	{0x295CBD40, "conv_arrow_on00.ncgr.small"          },
	{0x295D22D7, "itm_accwtrng_2.ncgr.small"           },
	{0x2960F24B, "cfx_ch2.nsbmd.small"                 },
	{0x29667299, "item85.itm"                          },
	{0x2969F44A, "spell17.spl"                         },
	{0x296C23DF, "cbt_xpball.ncgr.small"               },
	{0x29788B1A, "item86.itm"                          },
	{0x2979E968, "cfx_ixdp2.nsbtp.small"               },
	{0x297C0CCB, "spell18.spl"                         },
	{0x298AA39B, "item87.itm"                          },
	{0x298E1A4A, "tyxis_arena_reset_sonic.gff"         },
	{0x298E254C, "spell19.spl"                         },
	{0x298ED19A, "a2sq_briefhistory.dlg"               },
	{0x299CBC1C, "item88.itm"                          },
	{0x29A3BC5D, "chao_jd_2.ncgr.small"                },
	{0x29AED49D, "item89.itm"                          },
	{0x29B4C895, "lower_bg_01.ncgr.small"              },
	{0x29D22618, "a1c3_reunion_father2.gff"            },
	{0x29D440B4, "amy_cb_victory.nsbca"                },
	{0x29DBE616, "a1_w2_br.nclr.small"                 },
	{0x29EE0210, "a1_w1_gh.nclr.small"                 },
	{0x29F22CFC, "btn_puzz_on.ncgr.small"              },
	{0x29F2854A, "cbt_focus_rpdbg.ncgr.small"          },
	{0x29F86D19, "spell_refresh_all_2.spl"             },
	{0x2A038860, "man_tsvst_on10.ncgr.small"           },
	{0x2A09EE4D, "pl_pz_cav_snow.nsbtx.small"          },
	{0x2A0A366D, "a1_big.dlg"                          },
	{0x2A0A6986, "sha_ex_run.nsbca"                    },
	{0x2A0A859A, "spell_refresh_all_3.spl"             },
	{0x2A0B3440, "cfx_flmp1_flm.emit"                  },
	{0x2A1A298C, "jrl_buttn_of00.ncgr.small"           },
	{0x2A1E4200, "icn_s_statbs.ncgr.small"             },
	{0x2A20B98E, "itm_glv_lucky_2.ncgr.small"          },
	{0x2A249B4D, "lower_green_ring_02.ncgr.small"      },
	{0x2A31F11F, "pfx_lzr1.nsbtx.small"                },
	{0x2A35D54F, "prtl_eixmad_3.ncgr.small"            },
	{0x2A4B485E, "pl_pz_cav_snow.nsbca"                },
	{0x2A4F2B79, "splashlists.gda"                     },
	{0x2A523C59, "tai_cb_idle.nsbca"                   },
	{0x2A5D5B42, "fx_shason_atom.nsbtx.small"          },
	{0x2A63A9E1, "cfx_rfgp2_rfg.emit"                  },
	{0x2A6A2CA0, "pil_cb_idle.nsbca"                   },
	{0x2A6F3472, "c_sheso_sb.nsbca"                    },
	{0x2A7020E1, "armored_tutorial01_bs.tga"           },
	{0x2A74AEF0, "gui_chao_i.nclr.small"               },
	{0x2A7527DD, "chao_jg_r.nsbtp.small"               },
	{0x2A7DC03D, "ch1_rainforest_st.gda"               },
	{0x2A7E7D06, "prtl_wkrstu_2.ncgr.small"            },
	{0x2A87572B, "kdr_ex_run.nsbca"                    },
	{0x2A87C909, "rou_ex_idle.nsbca"                   },
	{0x2A8EA35B, "gennpc1_aa.nsbmd.small"              },
	{0x2A945DA1, "cfx_bub1_pt2.emit"                   },
	{0x2A98E417, "mappins.gda"                         },
	{0x2A9FE53E, "c_sotaamkn_1_sb.nsbca"               },
	{0x2AA31183, "pl_pz_ppd_pdn.nsbca"                 },
	{0x2AA5FEF2, "fx_vox_y_es.nsbtx.small"             },
	{0x2AAFD102, "cfx_exbp3_exb.emit"                  },
	{0x2AB2C0EF, "a1c1_captain.dlg"                    },
	{0x2AB4E99E, "vom_cb_attackend.nsbca"              },
	{0x2AB58B56, "jurnl_ts01.ncgr.small"               },
	{0x2ABB2BE5, "fx_bubbles.nsbtx.small"              },
	{0x2ABDF0D3, "main_pnl_off16.ncgr.small"           },
	{0x2AE3272A, "prtl_tdadsca_2.ncgr.small"           },
	{0x2AEACB41, "storelist.gui"                       },
	{0x2AEBA96D, "cfx_gtpp1.nsbtp.small"               },
	{0x2AEBC099, "prtl_gungen_1.ncgr.small"            },
	{0x2AF11D02, "prtl_txmmad_3.ncgr.small"            },
	{0x2AF4B892, "big_cb_blockout.nsbca"               },
	{0x2AF5CA86, "cbt_focusbreak.nanr.small"           },
	{0x2AFD91C9, "prf_tboxb_ms00.ncgr.small"           },
	{0x2B0C89BC, "a1_s5_ex_slide1.gff"                 },
	{0x2B14E091, "combatsounds.gda"                    },
	{0x2B1EA23D, "a1_s5_ex_slide2.gff"                 },
	{0x2B250369, "a1_s1_i01.are"                       },
	{0x2B2DFF8F, "big_cb_counter.nsbca"                },
	{0x2B30BABE, "a1_s5_ex_slide3.gff"                 },
	{0x2B371BEA, "a1_s1_i02.are"                       },
	{0x2B39363D, "arm_cb_counter.nsbca"                },
	{0x2B3E5B61, "chao_jj_0.ncgr.small"                },
	{0x2B407164, "spell20.spl"                         },
	{0x2B42D33F, "a1_s5_ex_slide4.gff"                 },
	{0x2B49346B, "a1_s1_i03.are"                       },
	{0x2B5289E5, "spell21.spl"                         },
	{0x2B54EBC0, "a1_s5_ex_slide5.gff"                 },
	{0x2B583570, "cfx_fltp1.nsbtx.small"               },
	{0x2B5A591A, "jurnl_ts23.ncgr.small"               },
	{0x2B5B4CEC, "a1_s1_i04.are"                       },
	{0x2B6120B5, "item90.itm"                          },
	{0x2B6298D4, "invent_prt_blank.ncgr.small"         },
	{0x2B64A266, "spell22.spl"                         },
	{0x2B670441, "a1_s5_ex_slide6.gff"                 },
	{0x2B6D656D, "a1_s1_i05.are"                       },
	{0x2B6E4D81, "pl_ob_blkn_so.nsbmd.small"           },
	{0x2B733936, "item91.itm"                          },
	{0x2B76BAE7, "spell23.spl"                         },
	{0x2B7821CE, "hnpc_gunco_aa.nsbtx.small"           },
	{0x2B791CC2, "a1_s5_ex_slide7.gff"                 },
	{0x2B792E63, "fx_vox_p_ea_m.nsbtx.small"           },
	{0x2B7EFAA5, "gwr_cb_attack.nsbca"                 },
	{0x2B8551B7, "item92.itm"                          },
	{0x2B88D368, "spell24.spl"                         },
	{0x2B8B3543, "a1_s5_ex_slide8.gff"                 },
	{0x2B976A38, "item93.itm"                          },
	{0x2B9AEBE9, "spell25.spl"                         },
	{0x2B9C6778, "btn_back_e_on.ncgr.small"            },
	{0x2B9D4DC4, "a1_s5_ex_slide9.gff"                 },
	{0x2BA982B9, "item94.itm"                          },
	{0x2BAAD43A, "prtl_eggmad_1.ncgr.small"            },
	{0x2BAD046A, "spell26.spl"                         },
	{0x2BB53D30, "son_cb_blockout.nsbca"               },
	{0x2BB6038A, "cfx_fatp1.nsbmd.small"               },
	{0x2BBB9B3A, "item95.itm"                          },
	{0x2BBEC245, "itm_fotsnkel_3.ncgr.small"           },
	{0x2BBF1CEB, "spell27.spl"                         },
	{0x2BC0EF8B, "a1_s1_i02.walk.tga"                  },
	{0x2BCA4169, "spell_reverse_engineer.spl"          },
	{0x2BCDB3BB, "item96.itm"                          },
	{0x2BD1356C, "spell28.spl"                         },
	{0x2BDFCC3C, "item97.itm"                          },
	{0x2BE34DED, "spell29.spl"                         },
	{0x2BE6A98A, "chao_jq_2.ncgr.small"                },
	{0x2BEC1DC8, "itm_ace_bugsp_3.ncgr.small"          },
	{0x2BEFD65F, "lower_textbox_06.ncgr.small"         },
	{0x2BF00938, "cfx_ddsp1_dds.emit"                  },
	{0x2BF1E4BD, "item98.itm"                          },
	{0x2C03FD3E, "item99.itm"                          },
	{0x2C055048, "prtl_guncoang_2.ncgr.small"          },
	{0x2C068ABA, "pl_pz_mem_xtnd.nsbca"                },
	{0x2C07B3CE, "prologueb.gda"                       },
	{0x2C167531, "_blank.dlg"                          },
	{0x2C24B6FA, "cfx_srkp1.nsbtx.small"               },
	{0x2C260B75, "chao_ga_0.ncgr.small"                },
	{0x2C2E326F, "a2c7_tailstalkship.gff"              },
	{0x2C2F5F85, "egm_cb_blockd.nsbca"                 },
	{0x2C339D81, "items.gda"                           },
	{0x2C38DB3D, "charpro_cre1.ncgr.small"             },
	{0x2C38ECAD, "cfx_glw_drkp1.emit"                  },
	{0x2C39EFC1, "egm_ex_twitch.nsbca"                 },
	{0x2C4103C5, "pl_am_met_gun.nsbmd.small"           },
	{0x2C467016, "cfx_mexp1_mex.emit"                  },
	{0x2C4F2A50, "plt_ch4_main.plo"                    },
	{0x2C4F7C8A, "cbt_follow_button.ncgr.small"        },
	{0x2C5D0389, "nrg_cb_getup.nsbca"                  },
	{0x2C5EA287, "cbt_sonneutral.ncgr.small"           },
	{0x2C72E737, "btn_scrll_hi.ncgr.small"             },
	{0x2C7818E7, "prtl_txmgen_0.ncgr.small"            },
	{0x2C8189EE, "prtl_ncs.nclr.small"                 },
	{0x2C86706B, "ch5_islandfalls.gda"                 },
	{0x2CA01BBA, "fx_vox_y_ea_r.nsbta.small"           },
	{0x2CA5DC4F, "itm_fotsnksh_3.ncgr.small"           },
	{0x2CAA6FA2, "itm_glv_strong_1.ncgr.small"         },
	{0x2CAC3B6E, "prt_tp_big.ncgr.small"               },
	{0x2CAC9E2D, "prtl_thbgen_1.ncgr.small"            },
	{0x2CACD24A, "fx_ix_proj.nsbta.small"              },
	{0x2CB402E0, "icn_s_bash3.ncgr.small"              },
	{0x2CB44411, "chao_ba_1.ncgr.small"                },
	{0x2CBE9426, "pl_am_noc_gct.nsbtp.small"           },
	{0x2CC1697D, "she_cb_counter.nsbca"                },
	{0x2CC3BE13, "gui_buttonsgrey.nclr.small"          },
	{0x2CC52C58, "spell_sluggish_2.spl"                },
	{0x2CC96B75, "itm_ace_dimbu_0.ncgr.small"          },
	{0x2CCF44C1, "pl_pu_chst_oh.nsbca"                 },
	{0x2CD69E62, "chao_ha_r.nsbmd.small"               },
	{0x2CD744D9, "spell_sluggish_3.spl"                },
	{0x2CD75055, "cbt_pathdotanim.ncgr.small"          },
	{0x2CDA4F60, "lvl_stat_tspnl21.ncgr.small"         },
	{0x2CDB315D, "pfx_nicktest02.nsbtx.small"          },
	{0x2CF9607A, "guianim_switch.gda"                  },
	{0x2D060F32, "swt_ex_fall.nsbca"                   },
	{0x2D0ABF5E, "str_tlgbx_ms02.ncgr.small"           },
	{0x2D22A978, "exp_cct1_mm13.ncgr.small"            },
	{0x2D243244, "a1s3_sonic_wait_worm3.gff"           },
	{0x2D26715E, "cfx_hstp1.nsbmd.small"               },
	{0x2D37CDF1, "loc_victr_dt1.ncgr.small"            },
	{0x2D435CA5, "chao_jr_g.nsbtx.small"               },
	{0x2D4CC505, "prtl_thxgen_3.ncgr.small"            },
	{0x2D512C69, "cbt_rndnum9.ncgr.small"              },
	{0x2D6BA752, "ssb_aa.nsbtx.small"                  },
	{0x2D6D03B1, "prtl_eggaha_0.ncgr.small"            },
	{0x2D6D54E8, "a2_s5_ex_mm.nbfp"                    },
	{0x2D6D54EB, "a2_s5_ex_mm.nbfs"                    },
	{0x2D70EA01, "a1_s1_i02_mm.nclr.small"             },
	{0x2D7BC00C, "son_cb_damage.nsbca"                 },
	{0x2D7C720E, "vci_cb_attackend.nsbca"              },
	{0x2D862FE4, "a1c2_sonic_meets_knuckles.gff"       },
	{0x2D891D28, "jrl_tsmbr_of01.ncgr.small"           },
	{0x2D8EBDC9, "exp_ghz1_fw21.ncgr.small"            },
	{0x2D959A05, "spell30.spl"                         },
	{0x2D966795, "jrl_tmbox_ms10.ncgr.small"           },
	{0x2D969FDE, "plt_cod_sonic.plo"                   },
	{0x2D9BB1A2, "pl_ob_blkb_mb.nsbmd.small"           },
	{0x2DA3096E, "a2_s3_card_t1.nbfp"                  },
	{0x2DA30971, "a2_s3_card_t1.nbfs"                  },
	{0x2DA7B286, "spell31.spl"                         },
	{0x2DAE7030, "nes_ex_run.nsbca"                    },
	{0x2DB4DCD8, "loc_ambsh_it0.ncgr.small"            },
	{0x2DB9CB07, "spell32.spl"                         },
	{0x2DC29838, "a2c7_kronapproach_pan.gff"           },
	{0x2DCBE388, "spell33.spl"                         },
	{0x2DD482A8, "cbt_fatiguebar_s.ncgr.small"         },
	{0x2DD7BB77, "chasetutorial.gda"                   },
	{0x2DD7F356, "movement08_bs.tga"                   },
	{0x2DDB80C1, "fx_wtrfl_strk.nsbta.small"           },
	{0x2DE21E7B, "gui_inventory.nclr.small"            },
	{0x2DFC5831, "a2c10_charybarrives.gff"             },
	{0x2E0155EF, "icn_s_statluc.ncgr.small"            },
	{0x2E09D0AD, "c_soknta_so.nsbca"                   },
	{0x2E0A24CE, "a2_s5_i01.are"                       },
	{0x2E1A2941, "gzc_cb_counter.nsbca"                },
	{0x2E1C3D4F, "a2_s5_i02.are"                       },
	{0x2E24FDF3, "prtl_amysca_1.ncgr.small"            },
	{0x2E26335B, "fx_exclamation.nsbtx.small"          },
	{0x2E2E55D0, "a2_s5_i03.are"                       },
	{0x2E317A28, "prtl_dronegen_2.ncgr.small"          },
	{0x2E385389, "fx_a2_s1_ex_env.nsbta.small"         },
	{0x2E38D267, "wmp_trv_74_72.gff"                   },
	{0x2E406E51, "a2_s5_i04.are"                       },
	{0x2E40D4CE, "prtl_knustb_3.ncgr.small"            },
	{0x2E4AEAE8, "wmp_trv_74_73.gff"                   },
	{0x2E4B46EC, "pfx_fls1.nsbtx.small"                },
	{0x2E4F198A, "prtl_knuckgen_0.ncgr.small"          },
	{0x2E5286D2, "a2_s5_i05.are"                       },
	{0x2E53DC60, "itm_glvsercl_2.ncgr.small"           },
	{0x2E550780, "prtl_eixamu_3.ncgr.small"            },
	{0x2E660B4E, "cfx_fgrp1.nsbmd.small"               },
	{0x2E6F1BEA, "wmp_trv_74_75.gff"                   },
	{0x2E71EE0A, "pfx_smk1.nsbtp.small"                },
	{0x2E7AE0D9, "a1c1_journal_tutorial.gff"           },
	{0x2E86DF28, "chao_jr_r.nsbtp.small"               },
	{0x2E943625, "pl_pz_pod_pu.nsbca"                  },
	{0x2E9706FF, "prtl_ffmmad_0.ncgr.small"            },
	{0x2EA27F6A, "egb_cb_damage.nsbca"                 },
	{0x2EA8647F, "shp_ex_damage.nsbca"                 },
	{0x2EA8D1B2, "nrg_cb_dead.nsbca"                   },
	{0x2EAB865A, "prtl_thxdef_3.ncgr.small"            },
	{0x2EB26297, "wmp_txtbr_tx01.ncgr.small"           },
	{0x2EB731B6, "prtl_nesgen_2.ncgr.small"            },
	{0x2EBD44AA, "prf_crtbd_hi00.ncgr.small"           },
	{0x2EBE2359, "stronghold_ts.tga"                   },
	{0x2ECD1ED0, "lvl_tsstr_ms00.ncgr.small"           },
	{0x2ED1421E, "rou_ex_twitch.nsbca"                 },
	{0x2EDA170A, "cfx_ccup1.nsbmd.small"               },
	{0x2EEACE37, "lvl_stat_smpnl01.ncgr.small"         },
	{0x2EECAAC9, "jillcan13ol.nftr"                    },
	{0x2EF45B2D, "chaos.gda"                           },
	{0x2EF717EB, "son_ex_death.nsbca"                  },
	{0x2EFA79FC, "cfx_ficp1.nsbtx.small"               },
	{0x2EFCE2E2, "exp_wel1_mm00.ncgr.small"            },
	{0x2EFFC31E, "cho_tlgbx_ms01.ncgr.small"           },
	{0x2F173367, "chao_jd_r.nsbmd.small"               },
	{0x2F383967, "cbt_btn_pathdot.gui"                 },
	{0x2F3D3518, "itm_ace_resur_3.ncgr.small"          },
	{0x2F4C629A, "a2_s2_ex_slide1.gff"                 },
	{0x2F4ECBAC, "fx_drn_lzr_prj.nsbtx.small"          },
	{0x2F63A179, "pl_ob_blkn_mr.nsbtx.small"           },
	{0x2F6F8F63, "btn_s_bar_of01.ncgr.small"           },
	{0x2F729968, "she_ex_twitch.nsbca"                 },
	{0x2F7C41BC, "pfx_mot1.nsbmd.small"                },
	{0x2F7FECD0, "gentain_aa.nsbtx.small"              },
	{0x2F884155, "stronghold_tut_ts.tga"               },
	{0x2F88C974, "cfx_crzbns.emit"                     },
	{0x2F8CD94A, "pfx_smk2_dsh.emit"                   },
	{0x2F92A305, "fx_vox_g_ea_s.nsbmd.small"           },
	{0x2F95B30B, "cfx_ch7.nsbtx.small"                 },
	{0x2FA1B0A6, "exp_wel1_mm22.ncgr.small"            },
	{0x2FA7CCAB, "cfx_rtcp1.nsbtp.small"               },
	{0x2FA8DCBB, "fx_prjpulse.nsbtx.small"             },
	{0x2FB27741, "cfx_frgy.nsbtx.small"                },
	{0x2FB684ED, "a1_w3_gh_01.ncgr.small"              },
	{0x2FB6FA7D, "item100.itm"                         },
	{0x2FC5B004, "a1_w2_cc_11.ncgr.small"              },
	{0x2FC912FE, "item101.itm"                         },
	{0x2FCB80AC, "a1_s1_i03.walk.tga"                  },
	{0x2FDB2B7F, "item102.itm"                         },
	{0x2FEA4431, "prtl_eggsmg_3.ncgr.small"            },
	{0x2FECD286, "mut_flash_ms00.ncgr.small"           },
	{0x2FED4400, "item103.itm"                         },
	{0x2FEE0178, "fleesolo.gda"                        },
	{0x2FF28777, "a1c1_cream.dlg"                      },
	{0x2FF5E57C, "upper_textbox_00.ncgr.small"         },
	{0x2FFF5C81, "item104.itm"                         },
	{0x3003E820, "cfx_shcp1.nsbtp.small"               },
	{0x300A94F8, "icn_s_catk_st.ncgr.small"            },
	{0x30113C73, "combatqueue.gui"                     },
	{0x30117502, "item105.itm"                         },
	{0x30238D83, "item106.itm"                         },
	{0x302A6314, "a1_s1_i05_v2.are"                    },
	{0x303188B5, "itm_ace_timmy_1.ncgr.small"          },
	{0x3035A604, "item107.itm"                         },
	{0x303D78AD, "c_knshe_kn.nsbca"                    },
	{0x304098AF, "amy_cb_block.nsbca"                  },
	{0x3047BE85, "item108.itm"                         },
	{0x30510137, "cfx_mexp2_mex.emit"                  },
	{0x30526E57, "chao_fc_1.ncgr.small"                },
	{0x3059BB71, "plt_ch5_main.plo"                    },
	{0x3059D706, "item109.itm"                         },
	{0x3063273E, "icn_s_patk_knu2.ncgr.small"          },
	{0x3067266E, "a1c2_guncommandermission.gff"        },
	{0x306D7D10, "prtl_txsgen_3.ncgr.small"            },
	{0x30803FAB, "jrl_scrll_tk20.ncgr.small"           },
	{0x308E8AF2, "charpro_barlit02.ncgr.small"         },
	{0x30919664, "cfx_gyzp1.nsbtx.small"               },
	{0x309E8E4D, "plt_gbl_misc.plo"                    },
	{0x30A6C82A, "icn_l_victory2.ncgr.small"           },
	{0x30A7501F, "gui_itm_fot.nclr.small"              },
	{0x30AEEAA2, "cfx_firp1_flm.emit"                  },
	{0x30B0591E, "cfx_glw_canp1.emit"                  },
	{0x30B3B2DD, "plt_a2sq_interference.plo"           },
	{0x30B61FB3, "exl_arwut_of.ncgr.small"             },
	{0x30B85A9A, "health.ncgr.small"                   },
	{0x30D7582F, "prtl_scihap_3.ncgr.small"            },
	{0x30DA6D90, "fx_ix_bhol.nsbmd.small"              },
	{0x30E5A131, "chao_ga_g.nsbtx.small"               },
	{0x30EDD954, "man_tdlcp_on02.ncgr.small"           },
	{0x30EFDD0A, "cbt_flgetm_0.ncgr.small"             },
	{0x30FD13BE, "itm_glvgrpcl_1.ncgr.small"           },
	{0x3105482D, "itm_ace_immun_3.ncgr.small"          },
	{0x310D519C, "chao_fb_r.nsbtx.small"               },
	{0x3110B3AD, "a2_s3_i01.walk.tga"                  },
	{0x3120A261, "prtl_amydrm_3.ncgr.small"            },
	{0x3120EDA8, "fx_wtrfl_wtrfl.nsbmd.small"          },
	{0x3123A52F, "conv_bark_bs02.ncgr.small"           },
	{0x31271FB3, "prtl_tailsexc_2.ncgr.small"          },
	{0x312CD811, "nrg_aa.nsbtx.small"                  },
	{0x3130E1DB, "a1_s3_card_b1.nbfp"                  },
	{0x3130E1DE, "a1_s3_card_b1.nbfs"                  },
	{0x3136577E, "fx_a1_s1_ex_wtrf.nsbmd.small"        },
	{0x313FF43F, "prtl_sdwgen_3.ncgr.small"            },
	{0x3144C26C, "prtl_txrhap_1.ncgr.small"            },
	{0x31517B21, "a1c1_scoutattackson.gff"             },
	{0x315433F8, "chao_ha_2.ncgr.small"                },
	{0x3161B5F5, "prtl_hczgen_2.ncgr.small"            },
	{0x31680937, "cfx_acdp1.nsbmd.small"               },
	{0x316FC76A, "cbt_chasegauge_02.ncgr.small"        },
	{0x3172FE00, "a2c10_moveplayer.gff"                },
	{0x317F0A85, "son_ex_dash.nsbca"                   },
	{0x31812126, "prtl_sonic.nclr.small"               },
	{0x31841107, "bal_cb_attackc.nsbca"                },
	{0x31842853, "man_tsvin_01.ncgr.small"             },
	{0x3184891D, "invent_icon_eml.ncgr.small"          },
	{0x3187D450, "plt_lo_sectorcharyb.plo"             },
	{0x318CA64B, "cfx_rckp2.nsbtx.small"               },
	{0x31960F38, "man_tsvst_of01.ncgr.small"           },
	{0x3197B1F3, "a1c1_tails_woodsman.dlg"             },
	{0x31A4E537, "prtl_sonicsle_3.ncgr.small"          },
	{0x31A61B4B, "prtl_tailssca_3.ncgr.small"          },
	{0x31AF8FBF, "prtl_rougeinq_1.ncgr.small"          },
	{0x31B0C2B6, "cmbtbg_a2_s5.nsbmd.small"            },
	{0x31C9E586, "mut_circl_ms01.ncgr.small"           },
	{0x31D1639A, "pl_mv_boa_bp.nsbca"                  },
	{0x31D4E7DE, "fx_els_wt_aa.nsbta.small"            },
	{0x31D60E46, "prtl_nrsge2_1.ncgr.small"            },
	{0x31DFCF85, "lower_blue_ring_03.ncgr.small"       },
	{0x31E26C94, "chao_ca_3.ncgr.small"                },
	{0x31E642E1, "abutton.ncgr.small"                  },
	{0x31E799C4, "exclamation.tga"                     },
	{0x31EE4528, "a1c5_p7_3_ix.gff"                    },
	{0x31F43941, "cfx_tabp1.nsbtx.small"               },
	{0x31FC2EB2, "itm_glvblncl_2.ncgr.small"           },
	{0x31FE2566, "pl_pu_cha_egg.nsbtx.small"           },
	{0x320AA49A, "a1sq_gangster.dlg"                   },
	{0x320C231E, "item110.itm"                         },
	{0x3211DA5F, "pfx_invin.nsbmd.small"               },
	{0x32134B04, "bsc_cb_kd.nsbca"                     },
	{0x32145E76, "mut_chaob_ms10.ncgr.small"           },
	{0x3217776E, "ccz_ghz.gff"                         },
	{0x321E3B9F, "item111.itm"                         },
	{0x3223CE35, "itm_fotslplt_1.ncgr.small"           },
	{0x322923B4, "chao_ga_r.nsbtp.small"               },
	{0x3229E329, "fx_a1_s2_ex_env.nsbta.small"         },
	{0x322E7736, "spinball.nsbtp.small"                },
	{0x32305420, "item112.itm"                         },
	{0x32386B1E, "a1_w2_mt_01.ncgr.small"              },
	{0x323D6DA7, "fx_ix_bhol_prj.nsbca"                },
	{0x32426CA1, "item113.itm"                         },
	{0x324822CC, "cfx_curp1.nsbtp.small"               },
	{0x3248E158, "prtl_sonicinq_0.ncgr.small"          },
	{0x324A0F45, "prtl_krscon_1.ncgr.small"            },
	{0x324C36CD, "pfx_arnietest03.nsbmd.small"         },
	{0x324F26C8, "itm_ace_root_0.ncgr.small"           },
	{0x32548522, "item114.itm"                         },
	{0x32588D1B, "lvl_tdbld_ms01.ncgr.small"           },
	{0x326074FA, "gup_cb_shieldout.nsbca"              },
	{0x32660797, "pl_pz_ppd_aa.nsbmd.small"            },
	{0x32669DA3, "item115.itm"                         },
	{0x3272CA5D, "chapter6warp.gff"                    },
	{0x32786384, "cfx_hstp1_slt.emit"                  },
	{0x3278B624, "item116.itm"                         },
	{0x32871E20, "str_tsmbr_on00.ncgr.small"           },
	{0x328ACEA5, "item117.itm"                         },
	{0x328B921A, "charpro_rou2.ncgr.small"             },
	{0x32938C8E, "cfx_lucp1.nsbtx.small"               },
	{0x329A67EE, "prtl_eixlaf_1.ncgr.small"            },
	{0x329CC28B, "act1_world0.are"                     },
	{0x329CE726, "item118.itm"                         },
	{0x329EB594, "prtl_sdwdef_3.ncgr.small"            },
	{0x32AB7D78, "crm_cb_rund.nsbca"                   },
	{0x32AEDB0C, "act1_world1.are"                     },
	{0x32AEFFA7, "item119.itm"                         },
	{0x32B17329, "gzd_ex_walk.nsbca"                   },
	{0x32B8F9A8, "upper_portrait_05.ncgr.small"        },
	{0x32C00506, "combatactions.gda"                   },
	{0x32C0F38D, "act1_world2.are"                     },
	{0x32D30C0E, "act1_world3.are"                     },
	{0x32D9007C, "shp_ex_jump.nsbca"                   },
	{0x32DD4EDA, "pl_am_rdr_aa.nsbtx.small"            },
	{0x32DE4358, "cfx_spk_cr3p1.emit"                  },
	{0x32E29607, "spell_cursed.spl"                    },
	{0x32EF948E, "pro_sidbx_01.ncgr.small"             },
	{0x32FD5ADF, "lvl_tdbld_ms23.ncgr.small"           },
	{0x3306A04D, "icn_s_medkit.ncgr.small"             },
	{0x330F1C09, "itm_act2_chaos5_1.ncgr.small"        },
	{0x33132E24, "pl_am_rdr_dct.nsbca"                 },
	{0x33162CDC, "spell_cloak_2.spl"                   },
	{0x3324EC33, "itm_ace_ironw_0.ncgr.small"          },
	{0x3325A1B3, "exp_charfrm_ns.ncgr.small"           },
	{0x33263636, "chao_id_g.nsbtx.small"               },
	{0x3328455D, "spell_cloak_3.spl"                   },
	{0x33287A61, "a1_s1_i05_ts.nbfp"                   },
	{0x33287A64, "a1_s1_i05_ts.nbfs"                   },
	{0x3328EAB2, "chao_jo_r.nsbmd.small"               },
	{0x332E94B9, "knu_cb_dead.nsbca"                   },
	{0x332EA3CE, "icn_s_patk_tai2.ncgr.small"          },
	{0x33326DA3, "bif_cre.nbfp"                        },
	{0x33326DA6, "bif_cre.nbfs"                        },
	{0x333D8538, "credits.dlg"                         },
	{0x333EB4BF, "pl_am_tyx_gbp.nsbca"                 },
	{0x333FAB26, "cfx_vulp1.nsbmd.small"               },
	{0x334057A5, "prtl_gnagen_1.ncgr.small"            },
	{0x3346CAFA, "fx_noc_beam.nsbtx.small"             },
	{0x3347BAA0, "prtl_krsgen_2.ncgr.small"            },
	{0x3347FF57, "cfx_spkp1_per.emit"                  },
	{0x334DE6A1, "chao_he_r.nsbtx.small"               },
	{0x3353E81C, "pro_smbox_00.ncgr.small"             },
	{0x3359436D, "levelup_bonus_top.gui"               },
	{0x335B474E, "fx_ix_orb.nsbmd.small"               },
	{0x335EE355, "pl_mv_cyc_crs.nsbta.small"           },
	{0x3360C061, "itm_glv_cloth_3.ncgr.small"          },
	{0x33617834, "icn_l_onteleport.ncgr.small"         },
	{0x336A9AC1, "fx_ix_bhol_imp.nsbca"                },
	{0x33743301, "a2_s5_ex.are"                        },
	{0x3376C5D3, "she_ex_idle.nsbca"                   },
	{0x337A79ED, "pl_am_noc_gcb.nsbca"                 },
	{0x337E188D, "prtl_guncodet_0.ncgr.small"          },
	{0x33872EF8, "prtl_guncosmg_1.ncgr.small"          },
	{0x338BF9E2, "pl_mv_zip.nsbta.small"               },
	{0x33914901, "lower_bubblebox_01.ncgr.small"       },
	{0x3399B6E5, "itm_fotslpdk_2.ncgr.small"           },
	{0x33AE3795, "guianim_exp.gda"                     },
	{0x33AE4C26, "prtl_syraxgen_1.ncgr.small"          },
	{0x33B890A0, "cfx_ffrp1.nsbtp.small"               },
	{0x33BB84D9, "icn_s_patk_amy4.ncgr.small"          },
	{0x33C1D711, "itm_ace_shld_2.ncgr.small"           },
	{0x33CDF763, "prtl_dadrel_3.ncgr.small"            },
	{0x33D611CD, "a1_s1_i04.walk.tga"                  },
	{0x33DD9D40, "prtl_rou.nclr.small"                 },
	{0x33DF527F, "a1sq_factoryforeman.dlg"             },
	{0x33EE5007, "rax_cb_damage.nsbca"                 },
	{0x33F0989D, "chao_je_1.ncgr.small"                },
	{0x33F6EC2C, "charpro_bs_c12.ncgr.small"           },
	{0x33F8A113, "cfx_curp1_glt.emit"                  },
	{0x33FE5232, "lvl_tlgbx_ms01.ncgr.small"           },
	{0x340BF233, "vel_cb_attackend.nsbca"              },
	{0x342E3DC9, "prtl_tdadsad_0.ncgr.small"           },
	{0x3442084E, "chao_jc_g.nsbtp.small"               },
	{0x344DB410, "she_sta.nsbtx.small"                 },
	{0x344F12BB, "rewardchest_hold.nbfp"               },
	{0x344F12BE, "rewardchest_hold.nbfs"               },
	{0x345B9258, "cfx_mexp3_mex.emit"                  },
	{0x34614BBF, "item120.itm"                         },
	{0x3461733B, "itm_ace_nrgal_3.ncgr.small"          },
	{0x34644C92, "plt_ch6_main.plo"                    },
	{0x34671BF1, "pl_pz_crn_aa.nsbtx.small"            },
	{0x3469B8B9, "chao_id_r.nsbtp.small"               },
	{0x34736440, "item121.itm"                         },
	{0x3475B976, "prf_tmbar_of01.ncgr.small"           },
	{0x3475E538, "lvl_pow_tspnl10.ncgr.small"          },
	{0x34777BE7, "prtl_aeqhap_3.ncgr.small"            },
	{0x347841C8, "prt_se_nrrwo.ncgr.small"             },
	{0x347DEB3A, "a1c1_block_fall_hold.gff"            },
	{0x3480C129, "a1_w0_gh_00.ncgr.small"              },
	{0x3480D0EF, "itm_fotslprg_2.ncgr.small"           },
	{0x34857CC1, "item122.itm"                         },
	{0x3494CA46, "big_ex_damage.nsbca"                 },
	{0x34979542, "item123.itm"                         },
	{0x3497A1D5, "a1c5_p5_camera_pan_sonic.gff"        },
	{0x3498E6C6, "chao_jl_3.ncgr.small"                },
	{0x34A3D347, "vox_the_aa.nsbtx.small"              },
	{0x34A7B449, "a1c5_p7_5_knuckles.gff"              },
	{0x34A9ADC3, "item124.itm"                         },
	{0x34BBC644, "item125.itm"                         },
	{0x34BDF003, "itm_glv_midnt_3.ncgr.small"          },
	{0x34BFD7B6, "a2c10_sewerflood.gff"                },
	{0x34C10AF8, "debugqa.gda"                         },
	{0x34CDDEC5, "item126.itm"                         },
	{0x34D40154, "cfx_wndp1_wnd.emit"                  },
	{0x34DFF746, "item127.itm"                         },
	{0x34E3354A, "egm_ex_jump.nsbca"                   },
	{0x34F20FC7, "item128.itm"                         },
	{0x34F3B766, "drn_cb_getup.nsbca"                  },
	{0x35042848, "item129.itm"                         },
	{0x350E1D06, "knu_ex_flyout.nsbca"                 },
	{0x3510E58C, "cbt_victory.ncgr.small"              },
	{0x35135F05, "cbt_focusgood.ncgr.small"            },
	{0x3537C052, "prtl_foreman.nclr.small"             },
	{0x354F078D, "cbt_focusringc.ncer.small"           },
	{0x35519B1F, "a2c10_sonicknockback.gff"            },
	{0x3559EFE6, "cho_ts_03.ncgr.small"                },
	{0x355EF500, "itm_ace_spdbr_1.ncgr.small"          },
	{0x356A25C0, "fx_frcfld.nsbta.small"               },
	{0x356F2D2F, "pl_ob_blkb_mo.nsbmd.small"           },
	{0x358096DA, "chao_gc_3.ncgr.small"                },
	{0x3580FAA1, "egb_ex_idle.nsbca"                   },
	{0x358D1B91, "tyx_cb_attackend.nsbca"              },
	{0x358E7BA6, "chao_jh_r.nsbtx.small"               },
	{0x35902EF6, "ehd_blbar_ms00.ncgr.small"           },
	{0x359378ED, "lvl_tmstr_ms21.ncgr.small"           },
	{0x3594D1A6, "exp_ghz1_mm12.ncgr.small"            },
	{0x35957D99, "prtl_amysus_3.ncgr.small"            },
	{0x35ACB48F, "prtl_timcal_1.ncgr.small"            },
	{0x35B3F23D, "itm_glv_light_1.ncgr.small"          },
	{0x35CDC480, "rou_ex_run.nsbca"                    },
	{0x35CF1EE2, "icn_s_patk_egg1.ncgr.small"          },
	{0x35D05718, "sha_cb_damage.nsbca"                 },
	{0x35EFBEF5, "uppertextbubble.gui"                 },
	{0x35F389FB, "pfx_mst1.nsbtx.small"                },
	{0x35FB984E, "cfx_sson_flare.emit"                 },
	{0x35FD1DEE, "main_pnl_off11.ncgr.small"           },
	{0x3601C47D, "str_tmdbx_ms12.ncgr.small"           },
	{0x3615ECC0, "cyclone_nocturn.are"                 },
	{0x36242F57, "prtl_boyhap_0.ncgr.small"            },
	{0x363385CA, "chao_jr_1.ncgr.small"                },
	{0x3638BC62, "a1_w3_mr.nclr.small"                 },
	{0x36396948, "exp_ghz2_mm31.ncgr.small"            },
	{0x3643B135, "mmp_lgbar_01.ncgr.small"             },
	{0x3645F7C8, "rax_cb_counter.nsbca"                },
	{0x3647AA65, "itm_ace_seed_0.ncgr.small"           },
	{0x364E7135, "black6464.ncgr.small"                },
	{0x3653AAD2, "conv_portr_21.ncgr.small"            },
	{0x3654E58D, "hnpc_bcap_aa.nsbmd.small"            },
	{0x365C2B06, "tai_cb_block.nsbca"                  },
	{0x36616BD0, "itm_ace_prune_2.ncgr.small"          },
	{0x36671A14, "loc_ambsh_dt1.ncgr.small"            },
	{0x366AF126, "pl_pz_vox_ly.nsbtp.small"            },
	{0x366EB582, "pfx_gas1.nsbtx.small"                },
	{0x366F9DB0, "c_sharoom_om.nsbca"                  },
	{0x3673CFFD, "btn_s_rtarw_of.ncgr.small"           },
	{0x368AB982, "gui_tutorial_in_1bs.tga"             },
	{0x368C5174, "main_pnl_on15.ncgr.small"            },
	{0x36957948, "spell_cure_all.spl"                  },
	{0x369A730C, "str_tsmbx_ms12.ncgr.small"           },
	{0x369CD12E, "cfx_exbp1.nsbmd.small"               },
	{0x369D912C, "pfx_lzg1.nsbtp.small"                },
	{0x36B67460, "item130.itm"                         },
	{0x36BBB5CA, "a1c1_sonic_mainland.gff"             },
	{0x36C3C93C, "gennpc2_aa.nsbmd.small"              },
	{0x36C88CE1, "item131.itm"                         },
	{0x36D1C657, "cbt_follow_transition.ncer.small"    },
	{0x36D745D4, "prtl_rougeoff_3.ncgr.small"          },
	{0x36DAA562, "item132.itm"                         },
	{0x36ECBDE3, "item133.itm"                         },
	{0x36F35251, "gui_tutorial_in_bs.tga"              },
	{0x36F773A7, "prtl_wspsad_1.ncgr.small"            },
	{0x36FED664, "item134.itm"                         },
	{0x37042A09, "charpro_big.nclr.small"              },
	{0x3705F84B, "cfx_javp1.nsbtx.small"               },
	{0x3710EEE5, "item135.itm"                         },
	{0x3711DEA2, "icn_s_statvul.ncgr.small"            },
	{0x37230766, "item136.itm"                         },
	{0x37286B8F, "sco_ex_run.nsbca"                    },
	{0x372DE3BD, "pfx_dop1_tox.emit"                   },
	{0x372F037A, "lower_textbox_01.ncgr.small"         },
	{0x37323332, "cfx_rfpp1.nsbtx.small"               },
	{0x373480DF, "icn_s_mm_tornado.ncgr.small"         },
	{0x37351FE7, "item137.itm"                         },
	{0x373A9C26, "fx_els_er_aa.nsbmd.small"            },
	{0x373AAE8D, "prtl_shdhel_1.ncgr.small"            },
	{0x373EE930, "cfx_nocp1_spk.emit"                  },
	{0x37458268, "prtl_wsgrel_0.ncgr.small"            },
	{0x37473868, "item138.itm"                         },
	{0x374852AE, "pl_mv_tor_to.nsbca"                  },
	{0x3753FD2F, "itm_fotstlbt_1.ncgr.small"           },
	{0x375950E9, "item139.itm"                         },
	{0x375E8052, "cmbtbg_a1_s2.nsbmd.small"            },
	{0x3767AB4E, "pl_mov_arw_yw.nsbtx.small"           },
	{0x37706BA9, "spell_hyper.spl"                     },
	{0x3781539F, "numericaltest.gda"                   },
	{0x37887D1E, "shp_ex_idle.nsbca"                   },
	{0x3789E95D, "son_whirlwind_1.gff"                 },
	{0x3792A64A, "gui_popupbox.nclr.small"             },
	{0x3792F2EA, "exl_menui_ms01.ncgr.small"           },
	{0x37ABE76D, "pl_pu_emrld.nsbca"                   },
	{0x37AE1A5F, "son_whirlwind_3.gff"                 },
	{0x37B6E83D, "big_cb_blockc.nsbca"                 },
	{0x37BA5710, "a2_w0_ng_00.ncgr.small"              },
	{0x37CCE0CD, "pfx_drp1.nsbtx.small"                },
	{0x37D4474D, "conv_bubble_ts02.ncgr.small"         },
	{0x37DB43E1, "prt_se_gunpw.ncgr.small"             },
	{0x37DEFBBA, "upper_red_ring_02.ncgr.small"        },
	{0x37E0A2EE, "a1_s1_i05.walk.tga"                  },
	{0x37E99446, "invent_scroll_01.ncgr.small"         },
	{0x37EA38BF, "prtl_scygen_2.ncgr.small"            },
	{0x37EACDCD, "que_trfm_1.nsbca"                    },
	{0x38042AE1, "star_and_check.gui"                  },
	{0x3814A7E9, "itm_ace_delux_3.ncgr.small"          },
	{0x3820DECF, "omg_cb_blockd.nsbca"                 },
	{0x382B6F0B, "omg_ex_twitch.nsbca"                 },
	{0x382E2312, "tai_ex_land.nsbca"                   },
	{0x383A1203, "man_strbt_of10.ncgr.small"           },
	{0x383B53E7, "prtl_crehap_0.ncgr.small"            },
	{0x383D7FA0, "pfx_glf1.nsbtx.small"                },
	{0x3844427E, "itm_ace_cural_3.ncgr.small"          },
	{0x38467925, "cfx_smk1.nsbtx.small"                },
	{0x384744E0, "cfx_fwtp1.nsbmd.small"               },
	{0x3853BF99, "chao_jn_g.nsbtp.small"               },
	{0x385E75D8, "cfx_pksp1.nsbtx.small"               },
	{0x386152C4, "exl_smbtn_of00.ncgr.small"           },
	{0x386EDDB3, "plt_ch7_main.plo"                    },
	{0x3874A119, "chao_fe_g.nsbmd.small"               },
	{0x38752FB1, "ssf_tlgbx_ms20.ncgr.small"           },
	{0x387E4791, "jrl_tsmbr_on02.ncgr.small"           },
	{0x38905984, "cbt_rndnum4.ncgr.small"              },
	{0x3895ADE1, "jurnl_mis_line03.ncgr.small"         },
	{0x38A1070F, "str_resrv_ms20.ncgr.small"           },
	{0x38C3E487, "jurnl_arrown.ncgr.small"             },
	{0x38C534FE, "cbt_cmdframe12.ncgr.small"           },
	{0x38CB5806, "debug.gda"                           },
	{0x38D05DA7, "prtl_vxcleugen_2.ncgr.small"         },
	{0x38E772CD, "leveluplowerbuttons.gui"             },
	{0x390466F3, "a1c5_p7_9_gizoid.gff"                },
	{0x3906A457, "exp_cct1_mm30.ncgr.small"            },
	{0x390B9D01, "item140.itm"                         },
	{0x390BC443, "chao_ia_r.nsbmd.small"               },
	{0x39100D60, "prtl_vxcrblgen_1.ncgr.small"         },
	{0x391DB582, "item141.itm"                         },
	{0x392C123B, "tai_cb_support03.nsbca"              },
	{0x392C18F8, "prtl_timgen_0.ncgr.small"            },
	{0x392FCE03, "item142.itm"                         },
	{0x39330897, "icn_s_catk_cab.ncgr.small"           },
	{0x393BF84C, "conversation_lower.nclr.small"       },
	{0x394104F1, "ixa_aa.nsbmd.small"                  },
	{0x3941E684, "item143.itm"                         },
	{0x3942E609, "a2c10_scylla_appears_knuckeshold.gff"},
	{0x394F5A07, "rwd_ttxtb_ms02.ncgr.small"           },
	{0x3953FF05, "item144.itm"                         },
	{0x3955FB56, "prtl_fmncon_1.ncgr.small"            },
	{0x39587D67, "vom_cb_attack.nsbca"                 },
	{0x3961D708, "cbt_menu_button_on00.ncgr.small"     },
	{0x39661786, "item145.itm"                         },
	{0x39682319, "wmp_airp_of11.ncgr.small"            },
	{0x396C56EA, "a1c1_sonicwaitsmash2.gff"            },
	{0x39730969, "pl_ob_blkn_mb.nsbtx.small"           },
	{0x39783007, "item146.itm"                         },
	{0x398A4888, "item147.itm"                         },
	{0x398A53EC, "itm_act2_mastr_0.ncgr.small"         },
	{0x3992B1EC, "egm_ex_idle.nsbca"                   },
	{0x3996EB1A, "realtimebuttons.gui"                 },
	{0x399C6109, "item148.itm"                         },
	{0x39A032F1, "chao_js_r.nsbtx.small"               },
	{0x39AE798A, "item149.itm"                         },
	{0x39C21DBD, "itm_fotttnhy_1.ncgr.small"           },
	{0x39C4EEDF, "fx_vox_b_ea_r.nsbmd.small"           },
	{0x39C53BDF, "drnsui_aa.nsbtx.small"               },
	{0x39D735FF, "attack2_ts.tga"                      },
	{0x39DFCB5A, "buzwasn_ab.nsbmd.small"              },
	{0x39F85B73, "itm_fotgrshy_1.ncgr.small"           },
	{0x39F86BF2, "pl_pz_brd_1hd.nsbca"                 },
	{0x3A0308AD, "cfx_gfir_pu.nsbtp.small"             },
	{0x3A0B58B4, "itm_ace_goggl_0.ncgr.small"          },
	{0x3A1BBD2C, "she_cb_attackend.nsbca"              },
	{0x3A228C76, "cfx_ltgp1.nsbmd.small"               },
	{0x3A28858F, "a1c2_bigentrance_panto.gff"          },
	{0x3A298973, "prt_mp_tai.ncgr.small"               },
	{0x3A4F68FC, "cmbtbg_a2_s7_i01.nsbtx.small"        },
	{0x3A527778, "btn_arrowdn_of.ncgr.small"           },
	{0x3A64B9CC, "btn_s_bar_on02.ncgr.small"           },
	{0x3A833921, "fx_ome_missile.nsbmd.small"          },
	{0x3A8B371E, "a2_s5_card_b1.nbfp"                  },
	{0x3A8B3721, "a2_s5_card_b1.nbfs"                  },
	{0x3A9530C3, "cbt_explo_ms00.ncgr.small"           },
	{0x3A976F65, "knu_ex_flyin.nsbca"                  },
	{0x3A9F4A97, "chao_fd_0.ncgr.small"                },
	{0x3A9F4F38, "prtl_sbtgen_1.ncgr.small"            },
	{0x3AA86F64, "crm_cb_getup.nsbca"                  },
	{0x3AAE880A, "dft_scr_02.ncgr.small"               },
	{0x3AB23E53, "pfx_invin.emit"                      },
	{0x3AC120D7, "prtl_rougedis_1.ncgr.small"          },
	{0x3AC2DF45, "spell_int_kron.spl"                  },
	{0x3ACEB1CE, "wmptitle.gui"                        },
	{0x3AD46654, "amy_ex_jumpair.nsbca"                },
	{0x3AD7B51E, "str_txtbr_di02.ncgr.small"           },
	{0x3AD9F3D1, "realtimetutone.gda"                  },
	{0x3AE1580B, "fx_vox_y_ea_l.nsbtx.small"           },
	{0x3AE89D7E, "cfx_ch2.nsbtp.small"                 },
	{0x3AF82575, "prtl_sdwmad_1.ncgr.small"            },
	{0x3B019D12, "itm_fotirnhy_3.ncgr.small"           },
	{0x3B03E281, "a1_s2_i01_mm.nclr.small"             },
	{0x3B0AA3EE, "cfx_flmp1.nsbmd.small"               },
	{0x3B12C0A2, "cfx_strp1_rng.emit"                  },
	{0x3B2AB3B3, "prtl_gnogen_1.ncgr.small"            },
	{0x3B3CAC5D, "fx_tal_mbot.nsbmd.small"             },
	{0x3B420D6D, "gzg_cb_pattack.nsbca"                },
	{0x3B5A7270, "prtl_sonicdis_0.ncgr.small"          },
	{0x3B5AE8DB, "prtl_vxcriagen_0.ncgr.small"         },
	{0x3B60C5A2, "item150.itm"                         },
	{0x3B615015, "min_circl_ms20.ncgr.small"           },
	{0x3B72DE23, "item151.itm"                         },
	{0x3B77729F, "exl_lrgbt_hl.nanr.small"             },
	{0x3B7D8896, "cbt_trapd_ms.ncgr.small"             },
	{0x3B7F73B3, "prtl_voxai02.nclr.small"             },
	{0x3B84F6A4, "item152.itm"                         },
	{0x3B8DFA5E, "drnlaz_ex_idle.nsbca"                },
	{0x3B970F25, "item153.itm"                         },
	{0x3B9A2407, "fx_a1_s4_ex_env.nsbmd.small"         },
	{0x3B9AC825, "fx_a2_s3_ex_nrg.nsbmd.small"         },
	{0x3BA11038, "chao_hb_1.ncgr.small"                },
	{0x3BA2CC05, "ehd_fow_f.ncgr.small"                },
	{0x3BA927A6, "item154.itm"                         },
	{0x3BAA0475, "cmbtbg_ts_ghz112.ncgr.small"         },
	{0x3BAA05C3, "flee04_bs.tga"                       },
	{0x3BACE04D, "prtl_txrgen_1.ncgr.small"            },
	{0x3BADA715, "jurnl_cod_linelit01.ncgr.small"      },
	{0x3BB5164A, "pfx_briantest03.nsbmd.small"         },
	{0x3BB9A748, "prt_mp_egg.ncgr.small"               },
	{0x3BBB4027, "item155.itm"                         },
	{0x3BC7DB3E, "fx_vox_b_ea_m.nsbta.small"           },
	{0x3BCD58A8, "item156.itm"                         },
	{0x3BD7D112, "itm_fotslpnm_3.ncgr.small"           },
	{0x3BDB27A4, "icn_l_oninv2.ncgr.small"             },
	{0x3BDCE88E, "pl_ob_blk_aa.nsbtx.small"            },
	{0x3BDE04ED, "genroun_aa.nsbmd.small"              },
	{0x3BDF7129, "item157.itm"                         },
	{0x3BF189AA, "item158.itm"                         },
	{0x3BF298FF, "gen_cb_pattack.nsbca"                },
	{0x3BF79CF7, "prtl_amyhap_3.ncgr.small"            },
	{0x3BFBBED5, "prt_se_rhk.ncgr.small"               },
	{0x3C00CC1B, "pop_tggle_of02.ncgr.small"           },
	{0x3C03A22B, "item159.itm"                         },
	{0x3C0D21B8, "pig_cb_attackc.nsbca"                },
	{0x3C0E4D83, "main_pnl_big00.ncgr.small"           },
	{0x3C1B074A, "pl_mov_arw_ab.nsbta.small"           },
	{0x3C221CEA, "swt_cb_kd.nsbca"                     },
	{0x3C30B886, "a1c1_sonicholddrone.gff"             },
	{0x3C31BBA5, "omg_cb_getup.nsbca"                  },
	{0x3C31C12D, "charpro_bs_b05.ncgr.small"           },
	{0x3C4A1B09, "a2_chb_thebes_zap.gff"               },
	{0x3C5BCBDF, "a1_s3_giantworm3_appear.gff"         },
	{0x3C61E14B, "exp_fogofwar.ncgr.small"             },
	{0x3C729D60, "lvl_lrgbt_of00.ncgr.small"           },
	{0x3C7779B2, "prtl_soniccon_2.ncgr.small"          },
	{0x3C80F51B, "big_cb_pattack04.nsbca"              },
	{0x3C875C7B, "chap6scr_top.tga"                    },
	{0x3C8B39A1, "man_tsvst_on02.ncgr.small"           },
	{0x3C935D9B, "icn_s_catk_ws.ncgr.small"            },
	{0x3C936DCD, "ix_cb_finale.nsbca"                  },
	{0x3C965137, "a2_chb_thelxethug1.gff"              },
	{0x3CA869B8, "a2_chb_thelxethug2.gff"              },
	{0x3CADEC99, "plt_cod_perfectchaos.plo"            },
	{0x3CBE1F71, "itm_ace_boy_3.ncgr.small"            },
	{0x3CC07E7C, "a2c10_ix_firstattack.gff"            },
	{0x3CCFDE1A, "exp_tws1_mm11.ncgr.small"            },
	{0x3CD36C84, "invent_stat_tablit2.ncgr.small"      },
	{0x3CD9DEB4, "invent_list01.ncgr.small"            },
	{0x3CDB3801, "prtl_shdint_3.ncgr.small"            },
	{0x3CE05FBF, "a2_s5_card_b2.nbfp"                  },
	{0x3CE05FC2, "a2_s5_card_b2.nbfs"                  },
	{0x3CE2F823, "a2cha_thebes.dlg"                    },
	{0x3CE9950E, "a2cha_teleporter.dlg"                },
	{0x3CF18517, "cbt_focus_button.ncgr.small"         },
	{0x3CF5755F, "lower_small_ball.ncgr.small"         },
	{0x3CF5CB23, "chao_jk_g.nsbmd.small"               },
	{0x3CFC899A, "cfx_mexp2.nsbmd.small"               },
	{0x3CFFEBD6, "prtl_krsmad_0.ncgr.small"            },
	{0x3D070DC5, "cfx_basp1.nsbmd.small"               },
	{0x3D124D98, "knu_ex_run.nsbca"                    },
	{0x3D13D577, "charpro_son5.ncgr.small"             },
	{0x3D15A7A5, "prtl_nrqge2_2.ncgr.small"            },
	{0x3D17AE82, "a2c6_vaultbot.gff"                   },
	{0x3D18B392, "prtl_omggen_1.ncgr.small"            },
	{0x3D1AC712, "chao_ha_g.nsbtx.small"               },
	{0x3D1D2799, "upper_bg_11.ncgr.small"              },
	{0x3D241208, "charpro_bs_a01.ncgr.small"           },
	{0x3D353F77, "prtl_knumad_3.ncgr.small"            },
	{0x3D38767B, "prtl_chbgen_0.ncgr.small"            },
	{0x3D3DAEBD, "cfx_fatp1.nsbtp.small"               },
	{0x3D3F605C, "a1c4_sonic_send_gun.gff"             },
	{0x3D42777D, "chao_gb_r.nsbtx.small"               },
	{0x3D45A8F9, "hnpc_gen_idle.nsbca"                 },
	{0x3D533F89, "fx_leafdn.emit"                      },
	{0x3D5BF849, "itm_act2_chaos6_0.ncgr.small"        },
	{0x3D5DD50F, "a1_s5_ex_escalator_down.gff"         },
	{0x3D75250D, "prtl_sonicgen_3.ncgr.small"          },
	{0x3D7EFB33, "c_sosh_sh.nsbca"                     },
	{0x3D832DD5, "prt_sp_cre.ncgr.small"               },
	{0x3D892956, "cw_nw_ne.gff"                        },
	{0x3DB5EE43, "item160.itm"                         },
	{0x3DB73EBF, "plt_cod_tails.plo"                   },
	{0x3DBD920A, "cfx_disp1.nsbtx.small"               },
	{0x3DC806C4, "item161.itm"                         },
	{0x3DC8AEF8, "pl_am_met_gun.nsbtp.small"           },
	{0x3DC9E7D8, "chao_ch_0.ncgr.small"                },
	{0x3DD93698, "exp_playerframe0.ncgr.small"         },
	{0x3DDA1F45, "item162.itm"                         },
	{0x3DEC37C6, "item163.itm"                         },
	{0x3DED5EA7, "ch6_egglab_st.gda"                   },
	{0x3DF40152, "pl_pu_cao_egg.nsbmd.small"           },
	{0x3DF826C3, "upper_portrait_00.ncgr.small"        },
	{0x3DFE5047, "item164.itm"                         },
	{0x3E0EE2F8, "plt_lo_nocturnus.plo"                },
	{0x3E1068C8, "item165.itm"                         },
	{0x3E184547, "prtl_shdmad_2.ncgr.small"            },
	{0x3E1EB77A, "prt_tp_tai.ncgr.small"               },
	{0x3E228149, "item166.itm"                         },
	{0x3E2AF847, "charpro_ome4.ncgr.small"             },
	{0x3E2B650B, "icn_s_patk_cre3.ncgr.small"          },
	{0x3E318043, "pl_pz_fan_aa.nsbtx.small"            },
	{0x3E3499CA, "item167.itm"                         },
	{0x3E3D74DD, "chao_jf_0.ncgr.small"                },
	{0x3E3FE4A9, "a1_w3_cc.nclr.small"                 },
	{0x3E46B24B, "item168.itm"                         },
	{0x3E514F12, "pop_lgbox_ms05.ncgr.small"           },
	{0x3E52AE29, "mut_circl_ms40.ncgr.small"           },
	{0x3E588BAD, "lvl_stat_bspnl20.ncgr.small"         },
	{0x3E58CACC, "item169.itm"                         },
	{0x3E5E4995, "chao_ha_r.nsbtp.small"               },
	{0x3E961D9F, "bel_cb_attackend.nsbca"              },
	{0x3EA1176B, "pl_pz_pod_bat.nsbmd.small"           },
	{0x3EAE1C91, "cfx_hstp1.nsbtp.small"               },
	{0x3EB4947B, "prtl_tailsinq_2.ncgr.small"          },
	{0x3EB62934, "gzg_cb_kd.nsbca"                     },
	{0x3EB8B6C6, "gwr_hf.nsbtx.small"                  },
	{0x3EBC3B5C, "prtl_knugen_0.ncgr.small"            },
	{0x3EC17583, "plt_a2_stronghold.plo"               },
	{0x3ED245D9, "a2_chb_thebesexit.gff"               },
	{0x3ED68976, "prt_tp_e_rou.ncgr.small"             },
	{0x3EDCFDB2, "sha_cb_rund.nsbca"                   },
	{0x3EDF99C8, "prtl_aeqgen_3.ncgr.small"            },
	{0x3EE0B5D7, "a1c5_p9_scoutremoved_a.gff"          },
	{0x3EE2D00C, "cbt_cmdhlthfat1.ncgr.small"          },
	{0x3EE5C306, "chao_jm_2.ncgr.small"                },
	{0x3EE5E3AE, "jrl_tlgbx_ms21.ncgr.small"           },
	{0x3EEAF2B8, "jurnl_mis01.ncgr.small"              },
	{0x3EECD7E9, "fx_vox_r_ea_l.nsbmd.small"           },
	{0x3EF2CE58, "a1c5_p9_scoutremoved_b.gff"          },
	{0x3EFACCAD, "fx_vox_p_ea_r.nsbmd.small"           },
	{0x3F0B0D1F, "fx_a2_s3_ex_flt.nsbtx.small"         },
	{0x3F165DED, "ix_cb_attackend.nsbca"               },
	{0x3F2261D3, "c_sotaamkn_1_kn1.nsbca"              },
	{0x3F240E97, "explorehudmenupuzzle.gui"            },
	{0x3F283C06, "a1c1_exitdock.gff"                   },
	{0x3F2D22BA, "egb_cb_attackend.nsbca"              },
	{0x3F37956C, "prtl_sdwsmg_3.ncgr.small"            },
	{0x3F47AA80, "prt_tp_e_she.ncgr.small"             },
	{0x3F5B5C17, "chao_jd_g.nsbtx.small"               },
	{0x3F5DA042, "a2_s1_i05_ts.nbfp"                   },
	{0x3F5DA045, "a2_s1_i05_ts.nbfs"                   },
	{0x3F5FBC48, "pl_pz_tyx_ptn.nsbmd.small"           },
	{0x3F6A194B, "ch5_ixintro.gda"                     },
	{0x3F6AE3DF, "prf_tmbar_on02.ncgr.small"           },
	{0x3F7CFACC, "mut_ocrna_ms10.ncgr.small"           },
	{0x3F7E35CC, "vox_aa.nsbmd.small"                  },
	{0x3F81E6DB, "amy_aa_p.ncgr.small"                 },
	{0x3F830C82, "chao_ie_r.nsbtx.small"               },
	{0x3F849E77, "storeequipmentdata.gui"              },
	{0x3F879562, "journalbuttons.gui"                  },
	{0x3F8E09F5, "a2c10_ix_final_battle_wait.gff"      },
	{0x3F9B3046, "cfx_glw_gyzp1.emit"                  },
	{0x3FAED54F, "prt_tp_egg.ncgr.small"               },
	{0x3FBB0050, "prf_chrbx_di11.ncgr.small"           },
	{0x3FC91EF9, "a2c8_scy_shiptotyxis.gff"            },
	{0x3FCD731A, "chao_gd_2.ncgr.small"                },
	{0x3FDA7613, "a2_s2_ex_puzzle100.gff"              },
	{0x3FE1E5EB, "fx_shason_atom.nsbta.small"          },
	{0x3FE36623, "a1c4_shadow_runby.gff"               },
	{0x3FE8FE04, "a1c5_p7_4_sonic.gff"                 },
	{0x3FE99818, "gui_combatmenu.nclr.small"           },
	{0x3FEDB681, "cfx_fgrp1.nsbtp.small"               },
	{0x400B16E4, "item170.itm"                         },
	{0x400C4939, "invent_title00.ncgr.small"           },
	{0x40100D28, "hnpc_mug_idle.nsbca"                 },
	{0x40166F0A, "exp_ghz1_fw13.ncgr.small"            },
	{0x401D2F65, "item171.itm"                         },
	{0x401E18D6, "jrl_tmbox_ms02.ncgr.small"           },
	{0x401F97C4, "cfx_pfbp2_srd.emit"                  },
	{0x4021F4DE, "lower_sonic_portrait_03.ncgr.small"  },
	{0x402A899B, "fx_vox_y_es.nsbta.small"             },
	{0x402F47E6, "item172.itm"                         },
	{0x4039CE3C, "a1c2_hideoutenter.gff"               },
	{0x40416067, "item173.itm"                         },
	{0x4044AF17, "cfx_stngr.nsbtx.small"               },
	{0x4045BD4F, "movement10_bs.tga"                   },
	{0x40517108, "plt_cod_nrrgal.plo"                  },
	{0x405378E8, "item174.itm"                         },
	{0x4055356E, "pfx_rng1.nsbtx.small"                },
	{0x405B2D38, "prtl_amyoff_2.ncgr.small"            },
	{0x405C1188, "mmp_globe_3.ncgr.small"              },
	{0x405D60CA, "chap3scr_bot.tga"                    },
	{0x405F6DC8, "rpg_crosscircle.ncgr.small"          },
	{0x4061C23D, "cfx_ccup1.nsbtp.small"               },
	{0x40659169, "item175.itm"                         },
	{0x4067F7B9, "a1c5_p9_scene2b.gff"                 },
	{0x4071775D, "fleestandard1.gda"                   },
	{0x4077A9EA, "item176.itm"                         },
	{0x407A103A, "a1c5_p9_scene2c.gff"                 },
	{0x4080620A, "chao_js_0.ncgr.small"                },
	{0x40838FDE, "fleestandard2.gda"                   },
	{0x40844EE2, "lvl_txbar_hi00.ncgr.small"           },
	{0x4089C26B, "item177.itm"                         },
	{0x408D21F4, "profilecreaturepanel.gui"            },
	{0x409124AA, "gzg_cb_getup.nsbca"                  },
	{0x4095A85F, "fleestandard3.gda"                   },
	{0x409B216A, "cbt_focusring.ncer.small"            },
	{0x409BDAEC, "item178.itm"                         },
	{0x409EDE9A, "chao_jd_r.nsbtp.small"               },
	{0x40A0E618, "cfx_heap1_crs.emit"                  },
	{0x40A5A6E8, "a2c8_scy_queenspawn.gff"             },
	{0x40ADF36D, "item179.itm"                         },
	{0x40C48AD3, "itm_ace_radio_0.ncgr.small"          },
	{0x40C9692E, "prtl_fmnang_3.ncgr.small"            },
	{0x40CB2289, "prt_tp_m_ome.ncgr.small"             },
	{0x40CF38BB, "chao_ib_3.ncgr.small"                },
	{0x40FDB90C, "fx_vox_p_ea_m.nsbta.small"           },
	{0x4101902D, "omg_aa.nsbtx.small"                  },
	{0x4103ECEF, "pfx_mot1.nsbtp.small"                },
	{0x410BF5A1, "pfx_bfy1_bfy.emit"                   },
	{0x4118A89D, "prtl_txrhum_1.ncgr.small"            },
	{0x411F191F, "icn_l_patkamy.ncgr.small"            },
	{0x4120D464, "charpro_she2.ncgr.small"             },
	{0x41234847, "vox_ria_aa.nsbmd.small"              },
	{0x412626E0, "itm_fotrunsp_3.ncgr.small"           },
	{0x412F5CF0, "egb_cb_getup.nsbca"                  },
	{0x41364079, "a1_s2_ex_slide1.gff"                 },
	{0x413DEAC5, "cho_ts_20.ncgr.small"                },
	{0x414684F6, "pl_ob_blkn_mo.nsbtx.small"           },
	{0x414858FA, "a1_s2_ex_slide2.gff"                 },
	{0x415A717B, "a1_s2_ex_slide3.gff"                 },
	{0x41654A9C, "reward_bg.tga"                       },
	{0x416C89FC, "a1_s2_ex_slide4.gff"                 },
	{0x416D116F, "spell_int_robot_land.spl"            },
	{0x4172159F, "debugarea.gda"                       },
	{0x41789688, "cfx_ch4.nsbtx.small"                 },
	{0x4178C4E1, "rhk_cb_damage.nsbca"                 },
	{0x41893F38, "inventoryequipmentdata.gui"          },
	{0x4190BAFE, "a1_s2_ex_slide6.gff"                 },
	{0x41961E75, "movement07_bs.tga"                   },
	{0x419F0AAB, "store_ts.gui"                        },
	{0x419F8D07, "prt_se_nctfl.ncgr.small"             },
	{0x41A062CC, "a1_w2_gh_01.ncgr.small"              },
	{0x41A2D37F, "a1_s2_ex_slide7.gff"                 },
	{0x41A97599, "nintendosplash.tga"                  },
	{0x41AEBB44, "itm_ace_life_0.ncgr.small"           },
	{0x41AF8DE3, "a1_w1_cc_11.ncgr.small"              },
	{0x41BBBA5D, "charpro_ts_a13.ncgr.small"           },
	{0x41BE2E21, "son_ex_jump.nsbca"                   },
	{0x41CB7E8F, "main_pnl_on10.ncgr.small"            },
	{0x41ECC5EF, "cfx_ixdp1.nsbtx.small"               },
	{0x41F5BEDD, "cmbtbg_a1_s1.nsbca"                  },
	{0x41F6FD86, "wmp_trv_73_72.gff"                   },
	{0x4205E7FB, "exl_bchao_ms3.ncgr.small"            },
	{0x420ACB31, "gnp_cb_kd.nsbca"                     },
	{0x420B7C46, "prtl_hangen_1.ncgr.small"            },
	{0x4211B912, "c_soknam_am.nsbca"                   },
	{0x42180E70, "bsc_cb_attackc.nsbca"                },
	{0x421B2E88, "wmp_trv_73_74.gff"                   },
	{0x42275823, "cbt_victory_star.ncer.small"         },
	{0x422798A8, "invent_icon_glv.ncgr.small"          },
	{0x422864A7, "prt_tp_e_rng.ncgr.small"             },
	{0x422971C8, "cfx_strp1.nsbmd.small"               },
	{0x422D4709, "wmp_trv_73_75.gff"                   },
	{0x422F98C2, "nrg_cb_idle.nsbca"                   },
	{0x4235C0B3, "fleechapter1.gda"                    },
	{0x424498D9, "gup_cb_shieldin.nsbca"               },
	{0x424E9ED3, "prtl_shddet_3.ncgr.small"            },
	{0x424FC232, "chao_da_0.ncgr.small"                },
	{0x42603F85, "item180.itm"                         },
	{0x4265159A, "jrl_tlgbr_of00.ncgr.small"           },
	{0x42725806, "item181.itm"                         },
	{0x42847087, "item182.itm"                         },
	{0x4291CE1F, "cfx_ffrp1_flm.emit"                  },
	{0x42968908, "item183.itm"                         },
	{0x42A174B7, "pfx_spark.nsbtx.small"               },
	{0x42A39B3B, "wm_cav_drft.tga"                     },
	{0x42A898DB, "fx_wtrfl_wtrfl.nsbtp.small"          },
	{0x42A8A189, "item184.itm"                         },
	{0x42AAD5E8, "jillcan12ol.nftr"                    },
	{0x42AFE178, "cfx_crtp1.nsbmd.small"               },
	{0x42BABA0A, "item185.itm"                         },
	{0x42BE02B1, "fx_a1_s1_ex_wtrf.nsbtp.small"        },
	{0x42C967E1, "prtl_scyclm_2.ncgr.small"            },
	{0x42CA839A, "spell_hyper_2.spl"                   },
	{0x42CCD28B, "item186.itm"                         },
	{0x42DC9C1B, "spell_hyper_3.spl"                   },
	{0x42DEEB0C, "item187.itm"                         },
	{0x42EFB46A, "cfx_acdp1.nsbtp.small"               },
	{0x42F031FB, "ssb_aa.nsbta.small"                  },
	{0x42F1038D, "item188.itm"                         },
	{0x42F8EF1D, "gennpc3_aa.nsbmd.small"              },
	{0x42FB9084, "guianim_sha.gda"                     },
	{0x42FBCBA7, "combatguic.nclr.small"               },
	{0x43031C0E, "item189.itm"                         },
	{0x430475E9, "prtl_knuexh_2.ncgr.small"            },
	{0x432397E6, "crm_ex_walk.nsbca"                   },
	{0x432480AD, "a2c10_advgizoid_1.gff"               },
	{0x4328A2AB, "itm_ace_scan_2.ncgr.small"           },
	{0x432F3C6C, "man_strbt_on11.ncgr.small"           },
	{0x43306E26, "combo2.ncgr.small"                   },
	{0x43349A09, "chaseshadow.gda"                     },
	{0x4336992E, "a2c10_advgizoid_2.gff"               },
	{0x433E4C72, "fx_gnp_bullet.nsbtx.small"           },
	{0x43440B36, "kdr_cb_attack.nsbca"                 },
	{0x4348B1AF, "a2c10_advgizoid_3.gff"               },
	{0x434EBE4B, "icn_l_utgen.ncgr.small"              },
	{0x43567D2D, "exl_smbtn_on01.ncgr.small"           },
	{0x43584492, "pro_lrgbx_00.ncgr.small"             },
	{0x4358E8A1, "rou_cb_dead.nsbca"                   },
	{0x435ACA30, "a2c10_advgizoid_4.gff"               },
	{0x4360D736, "charpro_big2.ncgr.small"             },
	{0x4367EEE9, "pl_pz_cav_bp.nsbca"                  },
	{0x436992CF, "pl_pz_wod_neu.nsbca"                 },
	{0x436D1362, "chao_jo_g.nsbtx.small"               },
	{0x436E0F62, "fx_els_fr_aa.nsbtx.small"            },
	{0x436E18CE, "fx_vox_p_es.nsbmd.small"             },
	{0x43708602, "pl_pz_vox_ppt.nsbmd.small"           },
	{0x4371282D, "prtl_thbsad_3.ncgr.small"            },
	{0x43788096, "prtl_txmvar_0.ncgr.small"            },
	{0x437DBFAD, "icn_s_fly3.ncgr.small"               },
	{0x438B0703, "exp_underbar.ncgr.small"             },
	{0x4393A633, "cmbtbg_a2_s2.nsbmd.small"            },
	{0x4395C307, "pro_medbx_31.ncgr.small"             },
	{0x4396DC7C, "prtl_skphap_1.ncgr.small"            },
	{0x43998592, "pfx_invin.nsbtp.small"               },
	{0x439AA346, "icn_l_chkjrl.ncgr.small"             },
	{0x43A46115, "prtl_gunsca_0.ncgr.small"            },
	{0x43A69D6F, "pl_pu_cao_nst.nsbtx.small"           },
	{0x43A778B4, "prtl_boy.nclr.small"                 },
	{0x43B138FF, "prtl_egbgen_3.ncgr.small"            },
	{0x43BB0A20, "exl_shoes_ms3.ncgr.small"            },
	{0x43BDEC9D, "statusbar.ncgr.small"                },
	{0x43CA29E5, "buz_ex_damage.nsbca"                 },
	{0x43D3E200, "pfx_arnietest03.nsbtp.small"         },
	{0x43FD1256, "pro_lrgbx_22.ncgr.small"             },
	{0x44049EBB, "mut_shoes_ms01.ncgr.small"           },
	{0x4405F8AE, "itm_ace_medem_2.ncgr.small"          },
	{0x442248FD, "a1_w1_mt_01.ncgr.small"              },
	{0x4424F6D5, "introscr_top.tga"                    },
	{0x4438713C, "mut_btton_on00.ncgr.small"           },
	{0x4454A49C, "dungeons.nbfp"                       },
	{0x4454A49F, "dungeons.nbfs"                       },
	{0x4457EAC9, "cyclone_zoah.are"                    },
	{0x44603595, "debuguseitems.gda"                   },
	{0x4465ABBF, "upper_bg_08.ncgr.small"              },
	{0x446BC274, "fx_ix_flash.nsbtx.small"             },
	{0x447E1BE8, "icn_s_patk_rou4.ncgr.small"          },
	{0x4481DAC3, "a2_s4_i03_ts.nbfp"                   },
	{0x4481DAC6, "a2_s4_i03_ts.nbfs"                   },
	{0x448C9DDC, "plt_fpa2_amy.plo"                    },
	{0x448CFF3A, "jurnl_pnl11.ncgr.small"              },
	{0x44A9C6FA, "chao_ge_g.nsbmd.small"               },
	{0x44A9E214, "bmp_cb_attack.nsbca"                 },
	{0x44B095E5, "chao_jo_r.nsbtp.small"               },
	{0x44B4D013, "prtl_agzgen_3.ncgr.small"            },
	{0x44B56826, "item190.itm"                         },
	{0x44C02BD6, "prtl_bigtht_0.ncgr.small"            },
	{0x44C20084, "profile_bottom_static.gui"           },
	{0x44C75659, "cfx_vulp1.nsbtp.small"               },
	{0x44C76817, "a1c3_swatbot_hold.gff"               },
	{0x44C780A7, "item191.itm"                         },
	{0x44CF58B5, "prtl_pilgen_2.ncgr.small"            },
	{0x44D35655, "fx_drn_lzr_prj.nsbta.small"          },
	{0x44D8BDE5, "exp_charfrm_s.ncgr.small"            },
	{0x44D99928, "item192.itm"                         },
	{0x44DA31FD, "spell_fortified.spl"                 },
	{0x44DE7D36, "fx_els_ic_aa.nsbtx.small"            },
	{0x44E0A938, "a2_sonic.dlg"                        },
	{0x44E52E92, "fx_gren_tal.nsbtx.small"             },
	{0x44EBB1A9, "item193.itm"                         },
	{0x44EE3821, "itm_fotrsthy_2.ncgr.small"           },
	{0x44F1503E, "intromovies.gda"                     },
	{0x44FC5417, "prtl_tailsugh_2.ncgr.small"          },
	{0x44FDCA2A, "item194.itm"                         },
	{0x45067595, "ch10_plan.gda"                       },
	{0x450BE88A, "tut_item.gda"                        },
	{0x450FE2AB, "item195.itm"                         },
	{0x451142CC, "a2_s3_i01.are"                       },
	{0x4514B216, "pl_pu_emrld5.nsbmd.small"            },
	{0x451C3070, "cfx_scap1_sca.emit"                  },
	{0x451E4EA2, "icn_s_dash3.ncgr.small"              },
	{0x4521FB2C, "item196.itm"                         },
	{0x453413AD, "item197.itm"                         },
	{0x4540EA24, "chao_ja_r.nsbmd.small"               },
	{0x45462C2E, "item198.itm"                         },
	{0x4554BAE0, "c_soknta_sb.nsbca"                   },
	{0x455844AF, "item199.itm"                         },
	{0x455B8AD6, "cfx_pjkp1_lps.emit"                  },
	{0x4560F932, "gzg_aa.nsbtx.small"                  },
	{0x45762AD2, "ixb_aa.nsbmd.small"                  },
	{0x459AE669, "prtl_aeqsec_3.ncgr.small"            },
	{0x45AD1FF9, "cfx_srkp1_rng.emit"                  },
	{0x45AF2E0F, "journallist.gui"                     },
	{0x45B06138, "c_sokn_so.nsbca"                     },
	{0x45B2A766, "gup_cb_shieldd.nsbca"                },
	{0x45C143C1, "itm_glv_spike_1.ncgr.small"          },
	{0x45C65843, "chaogarden_newlyhatched.gui"         },
	{0x45CB5C2A, "a1c5_p7_1_shade.gff"                 },
	{0x45D74485, "a2c8_scy_queenspawn_pause.gff"       },
	{0x45D792AB, "pfx_puring.nsbtx.small"              },
	{0x45D8202B, "itm_ace_psych_0.ncgr.small"          },
	{0x45D85C03, "cw_w_n.gff"                          },
	{0x45EDEC78, "chao_hc_0.ncgr.small"                },
	{0x45FF3178, "egb_aa.nsbtx.small"                  },
	{0x45FF79DB, "prtl_gizgen_3.ncgr.small"            },
	{0x46095A74, "a2_w0_va_11.ncgr.small"              },
	{0x460E3585, "itm_act2_chaos1_1.ncgr.small"        },
	{0x461593FD, "plt_a2sq_tyxcivs.plo"                },
	{0x461B3BFA, "fx_noc_tele_to.nsbta.small"          },
	{0x463912CC, "prt_se_wasq.ncgr.small"              },
	{0x464CCC89, "lvl_stat_rblit.ncgr.small"           },
	{0x465FBAD8, "prtl_amygen_3.ncgr.small"            },
	{0x466CD294, "c_knshe_she.nsbca"                   },
	{0x466DAAC3, "son_ex_idle.nsbca"                   },
	{0x46743EC3, "gwr_ex_enter.nsbca"                  },
	{0x467DAF8B, "cfx_fgrp3.nsbtx.small"               },
	{0x4684A898, "statsinfosquad.gda"                  },
	{0x4688EC74, "prtl_tim.nclr.small"                 },
	{0x468FB14F, "universaltool.gui"                   },
	{0x4694079B, "cfx_crzbns.nsbtx.small"              },
	{0x469C6E77, "cfx_spk_cr2p1.emit"                  },
	{0x46AC65E1, "omg_cb_stunned.nsbca"                },
	{0x46B55BC9, "knu_cb_idle.nsbca"                   },
	{0x46BA64C8, "itm_ace_commu_3.ncgr.small"          },
	{0x46C77C27, "exp_wel1_mm.nclr.small"              },
	{0x46D05710, "cfx_firp1.nsbmd.small"               },
	{0x46DB8C26, "ch1_intro.gda"                       },
	{0x46E196B0, "sso_cb_idle.nsbca"                   },
	{0x46E484CC, "prtl_wkrrel_1.ncgr.small"            },
	{0x46EFB219, "chao_ja_1.ncgr.small"                },
	{0x46F5F684, "pop_tggle_on03.ncgr.small"           },
	{0x46F655D7, "exl_menub_of00.ncgr.small"           },
	{0x46FD9679, "lvl_pow_tspnl02.ncgr.small"          },
	{0x4721F578, "big_cb_blockidle.nsbca"              },
	{0x4727CA5C, "gui_portraits07.nclr.small"          },
	{0x472D5643, "prtl_sonicmad_1.ncgr.small"          },
	{0x47325E20, "a2_s4_ex.are"                        },
	{0x4741B637, "itm_ace_sturd_1.ncgr.small"          },
	{0x474873EE, "c_rokn_kn.nsbca"                     },
	{0x475057B6, "cfx_gwtr_pu.nsbmd.small"             },
	{0x47601095, "vox_cb_pattack3.nsbca"               },
	{0x4767C7C9, "lvl_lrgbt_on01.ncgr.small"           },
	{0x476C04B6, "cfx_nrgp1.nsbmd.small"               },
	{0x476FD232, "prtl_nessca_1.ncgr.small"            },
	{0x4770EE48, "charpro_bs_b00.ncgr.small"           },
	{0x477DEAB5, "chao_ea_2.ncgr.small"                },
	{0x47809793, "exl_circe_ms11.ncgr.small"           },
	{0x478CD1A9, "gnp_ex_walk.nsbca"                   },
	{0x47914A1C, "itm_ace_tablt_1.ncgr.small"          },
	{0x47980042, "chao_jh_3.ncgr.small"                },
	{0x479A1A56, "cbt_sidebar.ncgr.small"              },
	{0x479BEBE1, "ch1_tutorialislandcam.gff"           },
	{0x479BF6D0, "fx_footprint.nsbmd.small"            },
	{0x47AC524B, "a1c2_startknucklesescape.gff"        },
	{0x47AF065A, "a1c1_drones.dlg"                     },
	{0x47B9BCB0, "prtl_egginq_1.ncgr.small"            },
	{0x47C205E2, "ch10_cavalry_st.gda"                 },
	{0x47C68996, "chap1scr_top.tga"                    },
	{0x47D8B75D, "prtl_csxgen_1.ncgr.small"            },
	{0x47E345D3, "itm_glv_rock_0.ncgr.small"           },
	{0x47EC32A1, "pzl_metroestablish.gff"              },
	{0x47EC6113, "icn_s_catk_kw.ncgr.small"            },
	{0x47F47CA2, "itm_ace_lghtn_1.ncgr.small"          },
	{0x481C4CC5, "exp_ghz2_mm01.ncgr.small"            },
	{0x481FFA19, "a2_w0_kr_01.ncgr.small"              },
	{0x482074B5, "lower_bg_10.ncgr.small"              },
	{0x48247C61, "cfx_exbp1.nsbtp.small"               },
	{0x48257ED9, "ix_cb_attackc.nsbca"                 },
	{0x482A75D1, "prt_se_swatb.ncgr.small"             },
	{0x483B358B, "itm_glv_gold_1.ncgr.small"           },
	{0x483C0148, "son_cb_attackc.nsbca"                },
	{0x484D43D1, "icn_l_barwon.ncgr.small"             },
	{0x48530292, "charpro_son0.ncgr.small"             },
	{0x48569814, "itm_glv_rug_2.ncgr.small"            },
	{0x4876A615, "charpro_tai.nclr.small"              },
	{0x48798AD7, "cfx_ibep1_fls.emit"                  },
	{0x48816B9A, "fx_drn_lzr_beam.nsbmd.small"         },
	{0x4883CB64, "a1c3_forcefield.dlg"                 },
	{0x4884CF2F, "main_pnl_off03.ncgr.small"           },
	{0x4887FFC4, "a1c1_block_fall_1.gff"               },
	{0x48980A61, "e_w.gff"                             },
	{0x489A1845, "a1c1_block_fall_2.gff"               },
	{0x48AC30C6, "a1c1_block_fall_3.gff"               },
	{0x48ADE490, "fx_shadtele.nsbtx.small"             },
	{0x48AF0683, "knu_cb_attack.nsbca"                 },
	{0x48BE4947, "a1c1_block_fall_4.gff"               },
	{0x48C11A89, "exp_ghz2_mm23.ncgr.small"            },
	{0x48C24759, "fx_els_er_aa.nsbtp.small"            },
	{0x48CB55A3, "fx_noc_beam.nsbta.small"             },
	{0x48D061C8, "a1c1_block_fall_5.gff"               },
	{0x48D7A00D, "icn_l_utjump.ncgr.small"             },
	{0x48EC3CF0, "tyx_cb_blockc.nsbca"                 },
	{0x48F5F8D9, "cbt_btn_right_on.ncgr.small"         },
	{0x49213776, "jurnl_ts10.ncgr.small"               },
	{0x4930D934, "drnsen_aa.nsbtx.small"               },
	{0x49329F46, "chao_jn_1.ncgr.small"                },
	{0x493CB207, "plt_a2sq_stronghold.plo"             },
	{0x493F9FED, "jurnl_cod_line02.ncgr.small"         },
	{0x494FECF3, "chao_ia_g.nsbtx.small"               },
	{0x4952A16F, "chao_jl_r.nsbmd.small"               },
	{0x495F52F9, "plt_scylla_main.plo"                 },
	{0x495F76D7, "wmp_space_of00.ncgr.small"           },
	{0x4969740B, "prf_tboxa_ms12.ncgr.small"           },
	{0x4971BA9F, "bif_rou.nbfp"                        },
	{0x4971BAA2, "bif_rou.nbfs"                        },
	{0x49730894, "prt_se_nctsl.ncgr.small"             },
	{0x49779D5E, "chao_hb_r.nsbtx.small"               },
	{0x4979E85B, "chapter4warp.gff"                    },
	{0x4987856C, "pl_pz_brd_2up.nsbca"                 },
	{0x498FDBF5, "itm_ace_candy_2.ncgr.small"          },
	{0x49907C2D, "pop_lgbox_ms00.ncgr.small"           },
	{0x4990C55A, "charpro_egg5.ncgr.small"             },
	{0x49926CAF, "a2_chb_thebes_enters.gff"            },
	{0x49932D96, "guianim_lvl.gda"                     },
	{0x499459A1, "wmp_ballo.ncgr.small"                },
	{0x49A09F32, "prt_ce_gry.ncgr.small"               },
	{0x49C0A199, "a1c3_hide_forcefield.gff"            },
	{0x49C866F3, "cmbtbg_a2_s7.nsbtx.small"            },
	{0x49C8837D, "tyx_cb_attackc.nsbca"                },
	{0x49C95FAA, "prf_tpanl_ms01.ncgr.small"           },
	{0x49CEF013, "cfx_fwtp1.nsbtp.small"               },
	{0x49D482F8, "itm_ace_ice_3.ncgr.small"            },
	{0x49EE638A, "rhk_cb_pattack.nsbca"                },
	{0x49FC4C4C, "chao_fe_g.nsbtp.small"               },
	{0x4A06C3EA, "charpro_egg.nclr.small"              },
	{0x4A06D649, "a1_bigjoinpartyselect.gff"           },
	{0x4A071177, "itm_glv_grip_1.ncgr.small"           },
	{0x4A08930E, "ch10_observationcam.gff"             },
	{0x4A1A4F5A, "chao_ge_1.ncgr.small"                },
	{0x4A1BD41A, "prtl_fmnrel_2.ncgr.small"            },
	{0x4A1BDF0A, "itm_glv_fire_0.ncgr.small"           },
	{0x4A1F2BF3, "rwd_tlgbx_ms12.ncgr.small"           },
	{0x4A2CE2CC, "act1_world0.walk.tga"                },
	{0x4A2E364A, "itm_glv_work_3.ncgr.small"           },
	{0x4A2EDA46, "itm_ace_gunof_3.ncgr.small"          },
	{0x4A2F985F, "a1_s1_i03_ts.nbfp"                   },
	{0x4A2F9862, "a1_s1_i03_ts.nbfs"                   },
	{0x4A376811, "effectspalette.nclr"                 },
	{0x4A3A1AF5, "cho_smbtn_of00.ncgr.small"           },
	{0x4A6456FB, "itm_ace_wpnp1_1.ncgr.small"          },
	{0x4A6FC6D2, "genroun_st.nsbmd.small"              },
	{0x4A936F76, "chao_ia_r.nsbtp.small"               },
	{0x4A97AC1D, "worldmap_maptitle.gui"               },
	{0x4A992744, "a1c5_p7_badguys.gff"                 },
	{0x4A9F8949, "a2c10_noflyzone_cross.gff"           },
	{0x4AB553B4, "charpro_sdw4.ncgr.small"             },
	{0x4AB96045, "a1_s2_ex_tails_fly_se_1.gff"         },
	{0x4AC1C344, "man_strbt_of02.ncgr.small"           },
	{0x4AC6DDB8, "prtl_sciexh_0.ncgr.small"            },
	{0x4ACE12E4, "she_cb_victoryin.nsbca"              },
	{0x4ACEB992, "c_soknamta_kn.nsbca"                 },
	{0x4AD3EF36, "chaoportait.gui"                     },
	{0x4ADCA7C5, "prt_mp_son.ncgr.small"               },
	{0x4AE7E19C, "a1_s2_ex_door1.gff"                  },
	{0x4AE987D4, "exp_cct1_mm00.ncgr.small"            },
	{0x4AED7CD3, "prtl_salesca_3.ncgr.small"           },
	{0x4AF9DDE4, "prtl_caphap_3.ncgr.small"            },
	{0x4AF9FA1D, "a1_s2_ex_door2.gff"                  },
	{0x4AFA2EC5, "pl_pz_pod_up.nsbca"                  },
	{0x4AFCE0F2, "ssf_tlgbx_ms12.ncgr.small"           },
	{0x4B001C19, "a2c10_reposition_ix_2.gff"           },
	{0x4B06F4EB, "fx_a1_s4_icb.nsbtx.small"            },
	{0x4B1C14FB, "chao_ic_2.ncgr.small"                },
	{0x4B4C8874, "prtl_sonicamu_1.ncgr.small"          },
	{0x4B5775E7, "itm_ace_pring_3.ncgr.small"          },
	{0x4B63AFCF, "prtl_sonicdet_2.ncgr.small"          },
	{0x4B6A3176, "crm_ex_fall.nsbca"                   },
	{0x4B6CC63A, "prtl_sonicsmg_3.ncgr.small"          },
	{0x4B75C59A, "exl_circl_ms11.ncgr.small"           },
	{0x4B766B7E, "str_tlgbx_ms11.ncgr.small"           },
	{0x4B8544A5, "spell_adrenaline_2.spl"              },
	{0x4B8DE9F9, "cbt_enemy_ambush.ncgr.small"         },
	{0x4B8E5598, "exp_cct1_mm22.ncgr.small"            },
	{0x4B975D26, "spell_adrenaline_3.spl"              },
	{0x4BAA37A9, "cfx_ltgp1.nsbtp.small"               },
	{0x4BB83263, "chao_je_r.nsbtx.small"               },
	{0x4BBC1158, "charpro_stat_tab4.ncgr.small"        },
	{0x4BC13233, "itm_fotsnklt_2.ncgr.small"           },
	{0x4BCB30D3, "cbt_title_slide.nanr.small"          },
	{0x4BD04E65, "worldmap_spacebutton.gui"            },
	{0x4BD1AA14, "a1c4_shadowsonicchase.gff"           },
	{0x4BE55A34, "a1_w3_ul.nbfp"                       },
	{0x4BE55A37, "a1_w3_ul.nbfs"                       },
	{0x4BEEB670, "lvl_stat_but_auto01.ncgr.small"      },
	{0x4BF4C948, "jrl_tsmbr_of10.ncgr.small"           },
	{0x4BF865D1, "jrl_tsmbr_on_12.ncgr.small"          },
	{0x4BFA69E9, "exp_ghz1_fw30.ncgr.small"            },
	{0x4BFE6228, "a2_s6_ex_la_soundmap.gda"            },
	{0x4C08969D, "charpro_ts_b12.ncgr.small"           },
	{0x4C1B5CA9, "cfx_hdsp1.nsbtx.small"               },
	{0x4C47E56B, "she_cb_dead.nsbca"                   },
	{0x4C584FD7, "hud_lvlnum5.ncgr.small"              },
	{0x4C62731F, "a1_w0_mr.nclr.small"                 },
	{0x4C639AA3, "pl_ob_blkb_cr.nsbtx.small"           },
	{0x4C67DF98, "journalcodexentry.gui"               },
	{0x4C68BB5F, "prtl_eixfm1_1.ncgr.small"            },
	{0x4C790BF1, "itm_ace_rejuv_1.ncgr.small"          },
	{0x4C79A31E, "cfx_pksp1_pks.emit"                  },
	{0x4C7CBB70, "prtl_shadow.nclr.small"              },
	{0x4C7CC652, "fx_giz_lzr_prj.nsbtx.small"          },
	{0x4C825248, "splash_version.gui"                  },
	{0x4C8857AA, "fx_gun_scrn.nsbmd.small"             },
	{0x4C89D450, "a1_s6_card_t1.nbfp"                  },
	{0x4C89D453, "a1_s6_card_t1.nbfs"                  },
	{0x4C8E7291, "pfx_glf1_glf.emit"                   },
	{0x4C924F21, "cfx_flmp1.nsbtp.small"               },
	{0x4C94F380, "omg_cb_counter.nsbca"                },
	{0x4CA895FD, "itm_ace_leaf_1.ncgr.small"           },
	{0x4CB3DC89, "prtl_knusmg_0.ncgr.small"            },
	{0x4CBE39E8, "prtl_krwcon_0.ncgr.small"            },
	{0x4CDBFAE7, "itm_glvspncl_2.ncgr.small"           },
	{0x4CE5E808, "str_vertl_ms00.ncgr.small"           },
	{0x4CF2B626, "itm_ace_noctn_0.ncgr.small"          },
	{0x4CF68CEA, "icn_l_catk_kat.ncgr.small"           },
	{0x4D014F02, "prtl_skpcon_0.ncgr.small"            },
	{0x4D0259BF, "itm_fotsnkkb_1.ncgr.small"           },
	{0x4D08EFC1, "cbt_rdring_ms00.ncgr.small"          },
	{0x4D098436, "conversationicons.gda"               },
	{0x4D21CF3A, "fx_a1_s4_ex_env.nsbtp.small"         },
	{0x4D227358, "fx_a2_s3_ex_nrg.nsbtp.small"         },
	{0x4D3CC17D, "pfx_briantest03.nsbtp.small"         },
	{0x4D421204, "fx_a1_s6_ex_env.nsbtx.small"         },
	{0x4D4D8A16, "guianim_slo.gda"                     },
	{0x4D51AF34, "a2c6_vaultbotdefeated.gff"           },
	{0x4D567A57, "lvl_stat_smpnl10.ncgr.small"         },
	{0x4D5A4003, "jrl_tlgbr_on01.ncgr.small"           },
	{0x4D5ECBBC, "bmp_cb_counter.nsbca"                },
	{0x4D6B6F3E, "cho_tlgbx_ms10.ncgr.small"           },
	{0x4D7855F8, "a2sq_pirates.dlg"                    },
	{0x4D7A2D3A, "knu_cb_attackend.nsbca"              },
	{0x4D7F8649, "pfx_glo1.nsbtx.small"                },
	{0x4D8A509A, "a2_s1_card_b1.nbfp"                  },
	{0x4D8A509D, "a2_s1_card_b1.nbfs"                  },
	{0x4D95611E, "attack1_ts.tga"                      },
	{0x4DA1F8F8, "combatfinished.gui"                  },
	{0x4DB2463F, "pfx_lef2.nsbtx.small"                },
	{0x4DBBE543, "prtl_krwgen_1.ncgr.small"            },
	{0x4DC59012, "convo.nftr"                          },
	{0x4DD9F2D5, "rou_cb_blockin.nsbca"                },
	{0x4DF6978A, "pfx_dop1.nsbtx.small"                },
	{0x4DFEFA5D, "prtl_skpgen_1.ncgr.small"            },
	{0x4E022E87, "exl_jurnl_ms1.ncgr.small"            },
	{0x4E033846, "gui_explorealtgr.nclr.small"         },
	{0x4E04BBC9, "tyxis_arena_exit_stronghold.gff"     },
	{0x4E1E34ED, "itm_fotsnkrg_3.ncgr.small"           },
	{0x4E22310D, "a1_w3_gh_10.ncgr.small"              },
	{0x4E2E1F37, "pl_ob_spk_up.nsbca"                  },
	{0x4E3773ED, "act1_world1.walk.tga"                },
	{0x4E4D7E7F, "prtl_bighap_0.ncgr.small"            },
	{0x4E521A39, "egb_cb_dead.nsbca"                   },
	{0x4E559986, "icn_s_ds.ncgr.small"                 },
	{0x4E5DA63E, "conv_bubble_tab03.ncgr.small"        },
	{0x4E5F582C, "gameattract.gda"                     },
	{0x4E6F79C3, "shp_cb_attackc.nsbca"                },
	{0x4E76B713, "pfx_spk1_shk.emit"                   },
	{0x4E7D7656, "chao_jk_g.nsbtp.small"               },
	{0x4E8434CD, "cfx_mexp2.nsbtp.small"               },
	{0x4E8E8611, "a1c5_p8_eggmantalk.gff"              },
	{0x4E8EB8F8, "cfx_basp1.nsbtp.small"               },
	{0x4E9A2490, "nes_aa.nsbtx.small"                  },
	{0x4E9E57D6, "chao_fb_g.nsbmd.small"               },
	{0x4EB4756B, "cfx_exbp3.nsbtx.small"               },
	{0x4ED1D5CC, "prt_tp_son.ncgr.small"               },
	{0x4EDEFCF1, "a1_s6_card_t2.nbfp"                  },
	{0x4EDEFCF4, "a1_s6_card_t2.nbfs"                  },
	{0x4EE00235, "itm_ace_antid_1.ncgr.small"          },
	{0x4EE95520, "fx_vox_g_es.nsbtx.small"             },
	{0x4F144DC2, "a2_s2_ex_la.smp"                     },
	{0x4F174577, "cmbtbg_ts_ghz1.nclr.small"           },
	{0x4F1AB67A, "cbtmenubg_00.ncgr.small"             },
	{0x4F32BA85, "a2c10_ix_gone.gff"                   },
	{0x4F3A7A43, "a2_s1_ex_la_soundmap.gda"            },
	{0x4F3ACC88, "main_pnl_big14.ncgr.small"           },
	{0x4F42E2CD, "briantestfx01.emit"                  },
	{0x4F4877DD, "chao_he_3.ncgr.small"                },
	{0x4F53D2E0, "a1_s4_i01_ts.nbfp"                   },
	{0x4F53D2E3, "a1_s4_i01_ts.nbfs"                   },
	{0x4F551820, "sha_ex_walk.nsbca"                   },
	{0x4F578F5B, "exp_tws1_mm03.ncgr.small"            },
	{0x4F5AF038, "conv_arrow_of00.ncgr.small"          },
	{0x4F5ED25F, "fx_a1_s4_ics_bob.nsbca"              },
	{0x4F63FAD1, "charpro_knu1.ncgr.small"             },
	{0x4F6830E2, "flee03_bs.tga"                       },
	{0x4F68F8DB, "lower_bg_07.ncgr.small"              },
	{0x4F72D536, "spell_weakened_2.spl"                },
	{0x4F76248F, "cmbtbg_a1_s4.nsbtx.small"            },
	{0x4F774E5E, "bsc_cb_pattack01.nsbca"              },
	{0x4F839B96, "chaogarden_wireicons.gui"            },
	{0x4F84EDB7, "spell_weakened_3.spl"                },
	{0x4F8CAD18, "plt_a1sq_scientist.plo"              },
	{0x4F94DB2B, "a1c1_tails_test.dlg"                 },
	{0x4F9FCE47, "ch5_ixpissed.gda"                    },
	{0x4FA4D8DA, "upper_bg_03.ncgr.small"              },
	{0x4FA80F4C, "actimaginesplash.tga"                },
	{0x4FBC88C5, "cfx_ficp1_ice.emit"                  },
	{0x4FC004A0, "swt_cb_getup.nsbca"                  },
	{0x4FC15208, "prt_se_leuco.ncgr.small"             },
	{0x4FC9E9AE, "chao_jp_r.nsbtx.small"               },
	{0x4FCF5D06, "chapter8scylla.gda"                  },
	{0x4FCF5D4D, "chapter8scylla.gff"                  },
	{0x4FD36B39, "gnp_ex_fall.nsbca"                   },
	{0x4FD3F3A5, "cmbtbg_a2_s7_i01.nsbta.small"        },
	{0x4FD4F7DF, "prtl_wkrsca_0.ncgr.small"            },
	{0x4FD56DBC, "a1_s5_i02_ul.nbfp"                   },
	{0x4FD56DBF, "a1_s5_i02_ul.nbfs"                   },
	{0x4FEFD473, "man_tsvin_10.ncgr.small"             },
	{0x5001BB58, "man_tsvst_of10.ncgr.small"           },
	{0x500C404D, "rou_ex_flyin.nsbca"                  },
	{0x5017EC0E, "prtl_amymad_1.ncgr.small"            },
	{0x50270D7A, "cbt_menubg_01.ncgr.small"            },
	{0x5028C999, "fx_ix_bhol_imp.nsbta.small"          },
	{0x503591A6, "mut_circl_ms10.ncgr.small"           },
	{0x50365BD6, "pfx_mst1_blu.emit"                   },
	{0x5036F56F, "armboar_ab.nsbtx.small"              },
	{0x504537C6, "exl_altft_ms00.ncgr.small"           },
	{0x5047A981, "a1c3_shadowknockdown.gff"            },
	{0x504E3004, "prtl_kron.nclr.small"                },
	{0x50556C75, "jrl_arrow_00.ncgr.small"             },
	{0x50599CB6, "shp_cb_dead.nsbca"                   },
	{0x505B11C5, "itm_act2_chaos2_0.ncgr.small"        },
	{0x505B1D9F, "pl_pz_fanlt_rg.nsbmd.small"          },
	{0x505C32D1, "a1c2_guardiandefeated_camera.gff"    },
	{0x5062C4D1, "debuggamespeed.gda"                  },
	{0x5065E2B4, "fx_vox_y_ea_l.nsbta.small"           },
	{0x50680822, "prtl_shdsht_0.ncgr.small"            },
	{0x5068D6EB, "btn_arrowup_of.ncgr.small"           },
	{0x50700696, "ixb_ex_idle.nsbca"                   },
	{0x507ACF69, "mut_circl_ms21a.ncgr.small"          },
	{0x507E2A1F, "fx_a1_s1_ex_env.nsbtx.small"         },
	{0x50813718, "rwd_lgbtn_hi01.ncgr.small"           },
	{0x50901029, "spell_temporalfield.spl"             },
	{0x50971F5E, "wm_crn_stc.tga"                      },
	{0x50990CB1, "gzd_cb_kd.nsbca"                     },
	{0x509BDFB0, "itm_accnewsp_0.ncgr.small"           },
	{0x50B02755, "guianim_inv.gda"                     },
	{0x50B0C38D, "ssb_cb_pattack.nsbca"                },
	{0x50B40766, "jurnl_cod02.ncgr.small"              },
	{0x50B5EC8E, "prt_tp_e_knu.ncgr.small"             },
	{0x50BC2676, "chao_aa_r.nsbtx.small"               },
	{0x50C4393B, "lvl_tdbld_ms10.ncgr.small"           },
	{0x50DA5F6A, "mut_circl_ms32.ncgr.small"           },
	{0x50DE6337, "gens_walkmesh.tga"                   },
	{0x50DEECDB, "chao_he_g.nsbmd.small"               },
	{0x50E03CEE, "lvl_stat_bspnl12.ncgr.small"         },
	{0x50E3B4F3, "pl_pz_tyx_ans.nsbtx.small"           },
	{0x50F4193A, "btn_puzz_off.ncgr.small"             },
	{0x50FB3A06, "chap10scr_top.tga"                   },
	{0x5108D550, "itm_glv_rub_3.ncgr.small"            },
	{0x51112294, "prtl_krggen_2.ncgr.small"            },
	{0x5134E6AE, "gui_exploreb.nclr.small"             },
	{0x513C8E59, "chao_jb_0.ncgr.small"                },
	{0x513E7053, "plt_a1sq_omgwasps.plo"               },
	{0x514BB012, "gtest.are"                           },
	{0x515B40AE, "pro_sidbx_10.ncgr.small"             },
	{0x515EB816, "rpg_whiterightarrow.ncgr.small"      },
	{0x515ECB60, "itm_ace_kron_2.ncgr.small"           },
	{0x516EDB88, "cbt_focus_rpdbg.ncer.small"          },
	{0x5186FBEA, "itm_glv_crystal3.ncgr.small"         },
	{0x518AD23A, "itm_fotswtbt_1.ncgr.small"           },
	{0x518D248B, "crm_cb_pattack04.nsbca"              },
	{0x51909BA2, "a1_w1_mt.nclr.small"                 },
	{0x51AACF4E, "icn_s_rarwoflvl.ncgr.small"          },
	{0x51BF2EEE, "spell_fortified_2.spl"               },
	{0x51C46186, "prtl_wdmnhap_3.ncgr.small"           },
	{0x51CB49C8, "plt_cod_shadow.plo"                  },
	{0x51CC22F8, "exl_sonic_ms1.ncgr.small"            },
	{0x51CD4402, "a1c3_shadow_runby.gff"               },
	{0x51D1476F, "spell_fortified_3.spl"               },
	{0x51DDC277, "debugrecoveryitems.gda"              },
	{0x51E4DC82, "chao_ji_2.ncgr.small"                },
	{0x51E809F3, "syr_cb_attack.nsbca"                 },
	{0x51EA87FD, "a1_s1_ex_scientist1.gff"             },
	{0x51F32D1E, "pl_mv_boa_aa.nsbmd.small"            },
	{0x51FCA07E, "a1_s1_ex_scientist2.gff"             },
	{0x51FD659B, "pirates_player_entry.gff"            },
	{0x51FE42C8, "a1c3_reunion_boy2.gff"               },
	{0x520C5A40, "a2_w0_nt_21.ncgr.small"              },
	{0x520D6C6B, "she_ab.nsbtx.small"                  },
	{0x520EB8FF, "a1_s1_ex_scientist3.gff"             },
	{0x5220D180, "a1_s1_ex_scientist4.gff"             },
	{0x5226B1A9, "exl_arwin_on.ncgr.small"             },
	{0x522F7761, "charpro_tai1.ncgr.small"             },
	{0x5232EA01, "a1_s1_ex_scientist5.gff"             },
	{0x5242050E, "act1_world2.walk.tga"                },
	{0x52450282, "a1_s1_ex_scientist6.gff"             },
	{0x52467490, "a1c4_shade_enter.gff"                },
	{0x5256989D, "cfx_glw_rezp1.emit"                  },
	{0x525C2645, "pfx_bub1.nsbmd.small"                },
	{0x525CE0E2, "a2_shadow.dlg"                       },
	{0x5263D184, "egm_cb_dead.nsbca"                   },
	{0x5266F777, "gui_portraits02.nclr.small"          },
	{0x5269FE52, "lvl_tlgbx_ms10.ncgr.small"           },
	{0x526B53BD, "pl_pz_vox_wll.nsbmd.small"           },
	{0x5277C61C, "icn_l_catk_stk.ncgr.small"           },
	{0x5287D556, "sco_ex_jump.nsbca"                   },
	{0x528AED74, "prtl_ncsgen_2.ncgr.small"            },
	{0x52943178, "a1_s6_ex_mm.nclr.small"              },
	{0x52A33AEB, "fx_noc_tele_from.nsbta.small"        },
	{0x52A6E15E, "itm_ace_goldr_2.ncgr.small"          },
	{0x52BC586C, "charpro_amy3.ncgr.small"             },
	{0x52C50284, "sso_aa.nsbmd.small"                  },
	{0x52E3EDCB, "pl_pz_brd_aa.nsbmd.small"            },
	{0x52EE40E8, "egm_aa.nsbmd.small"                  },
	{0x52FDF8C3, "gwr_ex_idle.nsbca"                   },
	{0x5306B090, "rpg_opencircle.ncgr.small"           },
	{0x5308598D, "icn_l_utfly1.ncgr.small"             },
	{0x5313F632, "fx_dustpuff.ncgr.small"              },
	{0x5314FD8A, "prtl_thbovr_1.ncgr.small"            },
	{0x5315DD56, "cbt_sonsatisf.ncgr.small"            },
	{0x531F81E0, "chao_jh_g.nsbmd.small"               },
	{0x53267264, "knu_cb_pattack1.nsbca"               },
	{0x532C09E3, "a1_s2_ex_la_soundmap.gda"            },
	{0x5334583A, "bal_ex_spinjump.nsbca"               },
	{0x5337DDAD, "prtl_thbhap_2.ncgr.small"            },
	{0x534F0621, "pl_pz_cav_dr.nsbmd.small"            },
	{0x535869FF, "fx_vox_g_ea_m.nsbmd.small"           },
	{0x53594F41, "a2c9_chest_move.gff"                 },
	{0x535B7A05, "cfx_ch1.nsbtx.small"                 },
	{0x5370319E, "armdiln_icon.tga"                    },
	{0x537217DC, "a2_knuckles.dlg"                     },
	{0x5374A94C, "omg_cb_pattack04.nsbca"              },
	{0x5374CE5C, "pl_ob_blk_fall.nsbca"                },
	{0x53771112, "a1_w3_br_01.ncgr.small"              },
	{0x537F7B86, "chao_jo_0.ncgr.small"                },
	{0x538A40AB, "a1_w1_gh_01.ncgr.small"              },
	{0x539007D9, "prt_se_sbt.ncgr.small"               },
	{0x5394E20B, "itm_ace_eggmd_3.ncgr.small"          },
	{0x539634C0, "a1c1_sonicwalkgun.gff"               },
	{0x53996BC2, "a1_w0_cc_11.ncgr.small"              },
	{0x539A15EF, "fx_els_lt_aa.nsbmd.small"            },
	{0x539CE334, "arm_ex_damage.nsbca"                 },
	{0x53AC5077, "fx_leaf.emit"                        },
	{0x53B11CFB, "cfx_strp1.nsbtp.small"               },
	{0x53C139F6, "prtl_sonicimp_0.ncgr.small"          },
	{0x53C59C06, "cho_ts_12.ncgr.small"                },
	{0x53CBD2C8, "cbt_follow_button.ncer.small"        },
	{0x53D0A4ED, "icn_s_inv1.ncgr.small"               },
	{0x53D6422A, "slg_ex_walk.nsbca"                   },
	{0x53D94D9A, "ecm_cb_pattack.nsbca"                },
	{0x53DDDDBE, "itm_ace_grapp_0.ncgr.small"          },
	{0x53DF517F, "adv_shadow.gda"                      },
	{0x53DF5D1E, "howtoinventory.gda"                  },
	{0x54007DC6, "exp_ghz1_mm21.ncgr.small"            },
	{0x540A8DA9, "c_knshashe_kn.nsbca"                 },
	{0x541E8E72, "exu_circl_hi00.ncgr.small"           },
	{0x5423E466, "cfx_fthr_drt.nsbtx.small"            },
	{0x5425405C, "itm_fotrunsl_3.ncgr.small"           },
	{0x54378CAB, "cfx_crtp1.nsbtp.small"               },
	{0x543E8FBD, "pl_pz_crn_tu.nsbmd.small"            },
	{0x54417BF6, "hnpc_sci_aa.nsbmd.small"             },
	{0x544E459A, "prtl_amydet_2.ncgr.small"            },
	{0x544EDC8E, "mer_aa.nsbtx.small"                  },
	{0x54532FD0, "main_pnl_on02.ncgr.small"            },
	{0x5453A693, "cbt_pathdotanim.ncer.small"          },
	{0x5454A140, "wmp_space_on01.ncgr.small"           },
	{0x54575C05, "prtl_amysmg_3.ncgr.small"            },
	{0x545F5C93, "rou_cb_support03.nsbca"              },
	{0x546692DD, "wmp_txtbr_tx07.ncgr.small"           },
	{0x54699B66, "a1_w0_cc.nclr.small"                 },
	{0x546CFC33, "armored_tutorial01_ts.tga"           },
	{0x54720F6D, "a1c1_nocturnappear.gff"              },
	{0x54785289, "ixb_cb_attack1.nsbca"                },
	{0x547E4B62, "prtl_nesinq_0.ncgr.small"            },
	{0x54891758, "pl_pz_brg_rg.nsbtp.small"            },
	{0x548D63D9, "itm_ace_medkt_0.ncgr.small"          },
	{0x54A78C5A, "lvl_stat_but_auto02lit.ncgr.small"   },
	{0x54AC7592, "icn_l_rarwofstr.ncgr.small"          },
	{0x54AF9911, "itm_accearng_1.ncgr.small"           },
	{0x54B1333B, "itm_ace_wpnp2_0.ncgr.small"          },
	{0x54BCB14E, "cfx_ch3_spkp1.emit"                  },
	{0x54CFF275, "charpro_egg0.ncgr.small"             },
	{0x54D487CF, "prtl_ffmcon_2.ncgr.small"            },
	{0x54E26F71, "charpro_stat21.ncgr.small"           },
	{0x54E58594, "cfx_poip1_skl.emit"                  },
	{0x54F5FA21, "prf_tlbar_ms02.ncgr.small"           },
	{0x54F83135, "pl_pz_vox_ppt.nsbtp.small"           },
	{0x55014819, "a2_tails.dlg"                        },
	{0x55061F2C, "str_tsmbx_ms21.ncgr.small"           },
	{0x5506509B, "prtl_krngen_2.ncgr.small"            },
	{0x5520054B, "prtl_tailsgen_0.ncgr.small"          },
	{0x5521BEC3, "btn_buton_of.ncgr.small"             },
	{0x553D9169, "itm_ace_clovej_2.ncgr.small"         },
	{0x555288DD, "pig_cb_counter.nsbca"                },
	{0x55536104, "icn_s_patk_big4.ncgr.small"          },
	{0x55544994, "movement06_bs.tga"                   },
	{0x555AA284, "exploreportraitdisabled.gui"         },
	{0x555DE911, "vox_leu_aa.nsbmd.small"              },
	{0x555FF368, "itm_ace_tyxis_3.ncgr.small"          },
	{0x5565361C, "fx_shadteleto.nsbta.small"           },
	{0x556647AB, "itm_glvttncl_1.ncgr.small"           },
	{0x5568F13B, "chao_id_1.ncgr.small"                },
	{0x5578A684, "pro_medbx_01.ncgr.small"             },
	{0x5578EE16, "loc_victr_it1.ncgr.small"            },
	{0x557E5E6F, "itm_ace_gum_2.ncgr.small"            },
	{0x5580C55C, "prtl_knusad_2.ncgr.small"            },
	{0x558512D4, "chao_ja_g.nsbtx.small"               },
	{0x55852AA1, "shaball_aa.nsbtx.small"              },
	{0x559139EB, "gnp_cb_pattack01.nsbca"              },
	{0x55923ED8, "big_cb_pattack01.nsbca"              },
	{0x55A4B345, "ix_ex_walk.nsbca"                    },
	{0x55AA15C2, "upper_textbox_06.ncgr.small"         },
	{0x55ACC33F, "chao_ib_r.nsbtx.small"               },
	{0x55B6EDAD, "man_strbt_on03.ncgr.small"           },
	{0x55BA6703, "ccw_sw_se.gff"                       },
	{0x55BCA5F1, "kro_cb_pattack.nsbca"                },
	{0x55BE1FC7, "pfx_mst1_stm.emit"                   },
	{0x55BFF931, "rhk_cb_kd.nsbca"                     },
	{0x55C11A88, "inventorygui_ts_01.tga"              },
	{0x55C74126, "wmp_trv_72_73.gff"                   },
	{0x55C81CB0, "cfx_slgp1.nsbtx.small"               },
	{0x55D319E7, "a2c10_sonicwalk.gff"                 },
	{0x55D4A374, "plt_ch3_parts.plo"                   },
	{0x55D959A7, "wmp_trv_72_74.gff"                   },
	{0x55EB7228, "wmp_trv_72_75.gff"                   },
	{0x55FE9F0A, "exl_menui_ms10.ncgr.small"           },
	{0x560C26DC, "a1_w0_mt_01.ncgr.small"              },
	{0x5614D8B4, "pzl_kronestablish.gff"               },
	{0x5624C960, "prtl_shopkeeper.nclr.small"          },
	{0x5631722D, "chao_ge_g.nsbtp.small"               },
	{0x5633653B, "spell_int_robot_fly.spl"             },
	{0x56378F6F, "cfx_spbp1.nsbtx.small"               },
	{0x563FF36D, "conv_bubble_ts11.ncgr.small"         },
	{0x564595CD, "ix_cb_kd.nsbca"                      },
	{0x564C962F, "act1_world3.walk.tga"                },
	{0x56516B83, "itm_ace_echid_2.ncgr.small"          },
	{0x565B1C0F, "itm_accwnrng_0.ncgr.small"           },
	{0x5664BE40, "a2_s1_i03_ts.nbfp"                   },
	{0x5664BE43, "a2_s1_i03_ts.nbfs"                   },
	{0x566A15F8, "cfx_wndp1.nsbmd.small"               },
	{0x56730293, "pl_ob_blkb_cb.nsbtx.small"           },
	{0x5674621D, "a2c8_scy_raxosdelay.gff"             },
	{0x567B379B, "exp_ghq1_mm01.ncgr.small"            },
	{0x567F7F80, "fx_wtrfl_sprkl.nsbtx.small"          },
	{0x5683B40B, "nsw_ex_idle.nsbca"                   },
	{0x56854A40, "pl_pu_cha_nst.nsbtp.small"           },
	{0x568D79E7, "itm_fotrunkb_0.ncgr.small"           },
	{0x569159D8, "nrg_ac.nsbmd.small"                  },
	{0x5694F5BD, "pl_pz_ppv_dn.nsbta.small"            },
	{0x569D31C9, "fatigue.ncgr.small"                  },
	{0x56A21907, "she_cb_pattack02.nsbca"              },
	{0x56A5C700, "itm_glvirncl_3.ncgr.small"           },
	{0x56AEAA0F, "omg_cb_kd.nsbca"                     },
	{0x56BCB4C3, "pl_ob_spk_aa.nsbtx.small"            },
	{0x56C2F0A2, "bel_aa.nsbmd.small"                  },
	{0x56C89557, "chao_ja_r.nsbtp.small"               },
	{0x56DFAFB1, "rou_cb_idle.nsbca"                   },
	{0x56E1C75B, "cfx_dusp1.nsbmd.small"               },
	{0x56EC0C4C, "man_tdlcp_of02.ncgr.small"           },
	{0x56F79593, "pl_pu_emrld2.nsbmd.small"            },
	{0x56F8E64D, "prtl_vecgen_1.ncgr.small"            },
	{0x56F8EF78, "chao_fb_3.ncgr.small"                },
	{0x56FCCF1A, "statsinfotest.gda"                   },
	{0x56FF55E5, "fx_flash_layout.ncgr"                },
	{0x571308D9, "itm_glv_metal_2.ncgr.small"          },
	{0x5714B07B, "jurnl_pnl03.ncgr.small"              },
	{0x571D812B, "pfx_gul1_cro.emit"                   },
	{0x5720055F, "exp_ghq1_mm23.ncgr.small"            },
	{0x5730FE13, "c_amcr_cr.nsbca"                     },
	{0x5731392B, "chao_js_g.nsbmd.small"               },
	{0x57314B23, "syr_cb_pattack.nsbca"                },
	{0x5733A327, "itm_glv_dark_1.ncgr.small"           },
	{0x573CFB1A, "drnass_aa.nsbmd.small"               },
	{0x573DE05E, "crm_cb_damage.nsbca"                 },
	{0x5743F878, "pfx_rok1.nsbmd.small"                },
	{0x57445976, "fx_wtrfl_mist.nsbmd.small"           },
	{0x575A67A0, "a1c5_shadow_soniccamerahold.gff"     },
	{0x5762DEBF, "cfx_egip1.nsbtx.small"               },
	{0x5764FAC3, "prtl_shdstg_2.ncgr.small"            },
	{0x5769EBA6, "icn_s_wsl0wcho.ncgr.small"           },
	{0x5786E511, "prtl_krgang_3.ncgr.small"            },
	{0x5794C942, "prtl_vxccitgen_3.ncgr.small"         },
	{0x57977CF2, "hud_lvlnum0.ncgr.small"              },
	{0x579964A8, "conv_icon06.ncgr.small"              },
	{0x579A1BF0, "a1c5_stronghold_enter.gff"           },
	{0x579BB1B0, "sha_ex_fall.nsbca"                   },
	{0x57A2F2C8, "jurnl_mis_linelit01.ncgr.small"      },
	{0x57A7CC3D, "cfx_prj_min.nsbmd.small"             },
	{0x57A95515, "itm_fotrunrg_2.ncgr.small"           },
	{0x57AF24D4, "gnp_aa.nsbmd.small"                  },
	{0x57B615FB, "cfx_mst1_tor.emit"                   },
	{0x57C445EF, "profile_ts.gui"                      },
	{0x57DC76CE, "cfx_kipp1.nsbtx.small"               },
	{0x57DE8DA8, "c_taeg_eg.nsbca"                     },
	{0x57F45239, "icn_l_ondoor.ncgr.small"             },
	{0x5812FE8C, "gui_explore.nclr.small"              },
	{0x58146857, "swa_cb_kd.nsbca"                     },
	{0x582A8B6F, "c_sotaamkn_1_ta1.nsbca"              },
	{0x5830806A, "crm_cb_block.nsbca"                  },
	{0x584D7C4B, "pro_tab_10.ncgr.small"               },
	{0x58539128, "chapter10warp.gff"                   },
	{0x5856A47A, "c_sosh_so.nsbca"                     },
	{0x58580243, "cfx_firp1.nsbtp.small"               },
	{0x5862A89C, "pl_mv_tor_aa.nsbtx.small"            },
	{0x586AEDB3, "itm_ace_fragm_2.ncgr.small"          },
	{0x5870A46D, "pfx_haz1_smk.emit"                   },
	{0x587978EF, "chao_aa_0.ncgr.small"                },
	{0x587BD2C6, "lvl_pow_bspnl01.ncgr.small"          },
	{0x587CF4BE, "fx_sha_shock.nsbmd.small"            },
	{0x58832342, "prt_se_armdn.ncgr.small"             },
	{0x58855118, "str_tsmbr_of00.ncgr.small"           },
	{0x58A41431, "rpg_buttonbg.ncgr.small"             },
	{0x58A983FC, "rewardanims.gda"                     },
	{0x58B59C60, "prtl_biggen_0.ncgr.small"            },
	{0x58B78427, "a2_w0_tx_00.ncgr.small"              },
	{0x58C1B7BC, "a1_s5_ex_escalator_up.gff"           },
	{0x58C2D71B, "fx_gnp_bullet.nsbta.small"           },
	{0x58C6CA5C, "swa_ex_jump.nsbca"                   },
	{0x58CBB207, "a1_w3_ai_00.ncgr.small"              },
	{0x58D802E9, "cfx_gwtr_pu.nsbtp.small"             },
	{0x58E00700, "fx_ix_beam.nsbmd.small"              },
	{0x58E2BDE3, "cmbtbg_a1_i01.nsbca"                 },
	{0x58E92C3D, "cbt_btn_right.ncgr.small"            },
	{0x58ECEF7F, "cfx_calp1.nsbmd.small"               },
	{0x58F29A0B, "fx_els_fr_aa.nsbta.small"            },
	{0x58F3AFE9, "cfx_nrgp1.nsbtp.small"               },
	{0x5905BD13, "cmbtbg_a1_i02.nsbtx.small"           },
	{0x59074F87, "rhk_cb_getup.nsbca"                  },
	{0x591A342A, "dft_scr_11.ncgr.small"               },
	{0x591EA304, "texcache.gda"                        },
	{0x5929C467, "charpro_son.nclr.small"              },
	{0x593289C6, "a1_s4_ex_mm.nbfp"                    },
	{0x593289C9, "a1_s4_ex_mm.nbfs"                    },
	{0x5941B52F, "pfx_gul1.nsbtx.small"                },
	{0x5951412D, "fx_greenpuff.nsbmd.small"            },
	{0x5967FBC8, "prtl_cresad_2.ncgr.small"            },
	{0x59762243, "front_selectaslot.gui"               },
	{0x5978B53D, "itm_fotrunkv_2.ncgr.small"           },
	{0x5980FB49, "jillcan13sh.nftr"                    },
	{0x5989DB82, "cho_lgbtn_di02.ncgr.small"           },
	{0x598D2ABF, "fx_vox_g_ea_r.nsbtx.small"           },
	{0x598FED0D, "ghz1_minimap.nclr.small"             },
	{0x5996CA1F, "chao_jl_g.nsbtx.small"               },
	{0x59989A6B, "prtl_telmad_0.ncgr.small"            },
	{0x599B7FE0, "pl_am_noc_gslr.nsbtp.small"          },
	{0x59A1461C, "vel_cb_attack.nsbca"                 },
	{0x59B59D2A, "itm_act2_chaos4_3.ncgr.small"        },
	{0x59B7E8B4, "fx_els_wn_aa.nsbmd.small"            },
	{0x59B9CCAB, "omg_cb_block.nsbca"                  },
	{0x59CDB844, "a2_s4_ex_la.smp"                     },
	{0x59D2FA20, "prtl_krshap_3.ncgr.small"            },
	{0x59DE010C, "itm_fotrnfhy_0.ncgr.small"           },
	{0x59E28D39, "ccw_s_e.gff"                         },
	{0x59E91CCC, "briantestfx01.nsbtp.small"           },
	{0x59F04D1D, "fx_ix_flash.nsbta.small"             },
	{0x5A0FE416, "main_slt_off01.ncgr.small"           },
	{0x5A15B095, "cmbtbg_ts_ghz121.ncgr.small"         },
	{0x5A2A1B2A, "sco_cb_pattack.nsbca"                },
	{0x5A458C60, "icn_s_patk_son2.ncgr.small"          },
	{0x5A4A651D, "pfx_mst1_yel.emit"                   },
	{0x5A4AD8CD, "itm_fotslpgr_0.ncgr.small"           },
	{0x5A508038, "itm_accwtrng_3.ncgr.small"           },
	{0x5A5A9996, "cfx_spk_cr1p1.emit"                  },
	{0x5A6307DF, "fx_els_ic_aa.nsbta.small"            },
	{0x5A633B40, "a1c3_shadow_branch.gff"              },
	{0x5A6E4DC4, "ehd_tmana_ms00.ncgr.small"           },
	{0x5A9719BE, "chao_jd_3.ncgr.small"                },
	{0x5A9BC18D, "a2c10_scylla_appears_pan.gff"        },
	{0x5AA825F6, "lower_bg_02.ncgr.small"              },
	{0x5AAC8C07, "spell_flashbang.spl"                 },
	{0x5AB978C8, "cfx_fglp1.nsbmd.small"               },
	{0x5AD37DB7, "chao_gb_g.nsbmd.small"               },
	{0x5AD7C65B, "a2c10_advgizoid1_attack_end.gff"     },
	{0x5ADA4CA2, "chao_jl_r.nsbtp.small"               },
	{0x5AE7D4F9, "conv_bubble_bs00.ncgr.small"         },
	{0x5AF0893F, "a2_s3_ex.are"                        },
	{0x5AF09636, "gup_ex_run.nsbca"                    },
	{0x5AF6E5C1, "man_tsvst_on11.ncgr.small"           },
	{0x5AFB2E22, "chao_fc_r.nsbmd.small"               },
	{0x5AFCBD7F, "exp_ghq1_mm.nclr.small"              },
	{0x5AFD22F9, "gzc_cb_attack.nsbca"                 },
	{0x5B08C6A5, "ehd_fow_qu.ncgr.small"               },
	{0x5B10BD98, "cbt_btn_break.gui"                   },
	{0x5B1416EF, "itm_glv_lucky_3.ncgr.small"          },
	{0x5B17F8AE, "lower_green_ring_03.ncgr.small"      },
	{0x5B187A95, "cfx_dstp1.nsbtx.small"               },
	{0x5B1F12D5, "a1c1_amy_test.dlg"                   },
	{0x5B2379D4, "sha_cb_attackc.nsbca"                },
	{0x5B28A882, "plt_fp_knuckles.plo"                 },
	{0x5B2D2D69, "ch8_scylla_entertyxisbase.gff"       },
	{0x5B397F5B, "cbt_charpointer.ncgr.small"          },
	{0x5B3B8A3A, "exp_tws1_mm20.ncgr.small"            },
	{0x5B458AD4, "invent_list10.ncgr.small"            },
	{0x5B5B9FB7, "pl_pu_rng_spn.nsbtp.small"           },
	{0x5B5CAF30, "icn_s_patk_ome1.ncgr.small"          },
	{0x5B5D7595, "bsc_cb_counter.nsbca"                },
	{0x5B71DA67, "prtl_wkrstu_3.ncgr.small"            },
	{0x5B98DACE, "son_cb_stunned.nsbca"                },
	{0x5B9BE7E2, "fx_gust.emit"                        },
	{0x5BA452DB, "gup_aa.nsbmd.small"                  },
	{0x5BA8E8B7, "jurnl_ts02.ncgr.small"               },
	{0x5BAB4A70, "cmbtbg_a2_s4.nsbtx.small"            },
	{0x5BB14E34, "main_pnl_off17.ncgr.small"           },
	{0x5BBACD7F, "cfx_fwtp1_wat.emit"                  },
	{0x5BD6848B, "prtl_tdadsca_3.ncgr.small"           },
	{0x5BD6B027, "fx_sha_jav.nsbmd.small"              },
	{0x5BD86490, "fx_vox_r_es.nsbmd.small"             },
	{0x5BDF1DFA, "prtl_gungen_2.ncgr.small"            },
	{0x5BEBD428, "drnsui_ex_idle.nsbca"                },
	{0x5BF0EF2A, "prf_tboxb_ms01.ncgr.small"           },
	{0x5C020444, "jurnl_arrowblit.ncgr.small"          },
	{0x5C14CBC9, "tai_cb_blockc.nsbca"                 },
	{0x5C1860CA, "a2_s1_i01.are"                       },
	{0x5C266BC7, "pzl_bridgeestablish.gff"             },
	{0x5C26C643, "a2_amy.dlg"                          },
	{0x5C270CDB, "fx_shadbeam.nsbtx.small"             },
	{0x5C2A794B, "a2_s1_i02.are"                       },
	{0x5C31B8C2, "chao_jj_1.ncgr.small"                },
	{0x5C3C91CC, "a2_s1_i03.are"                       },
	{0x5C46BE87, "pfx_arnietest02.nsbtx.small"         },
	{0x5C4EAA4D, "a2_s1_i04.are"                       },
	{0x5C54458D, "fx_shadtelefrom.nsbta.small"         },
	{0x5C60C2CE, "a2_s1_i05.are"                       },
	{0x5C8B2007, "ch10_playmovie.gff"                  },
	{0x5C8D3BCA, "cbt_victory.ncer.small"              },
	{0x5C8FB543, "cbt_focusgood.ncer.small"            },
	{0x5C921E49, "gui_man.nclr.small"                  },
	{0x5C9D2E03, "prtl_shdtht_0.ncgr.small"            },
	{0x5C9E319B, "prtl_eggmad_2.ncgr.small"            },
	{0x5CBC01C0, "shp_cb_attack.nsbca"                 },
	{0x5CCB2551, "cbt_menu_backbutton_of.ncgr.small"   },
	{0x5CCE8B71, "spell_distracted_all_2.spl"          },
	{0x5CD21608, "fx_windsock.nsbtx.small"             },
	{0x5CD32229, "cbt_sonsurprise.ncgr.small"          },
	{0x5CDA06EB, "chao_jq_3.ncgr.small"                },
	{0x5CE0A3F2, "spell_distracted_all_3.spl"          },
	{0x5CE333C0, "lower_textbox_07.ncgr.small"         },
	{0x5CED1A3F, "inv_blank_8x8.ncgr.small"            },
	{0x5CF14C57, "chao_ba_r.nsbtx.small"               },
	{0x5CF8ADA9, "prtl_guncoang_3.ncgr.small"          },
	{0x5CF8BA2A, "a1c5_p9_sonicchases.gff"             },
	{0x5CFB4B6B, "c_sokn_sb.nsbca"                     },
	{0x5D045572, "prtl_gglugh_0.ncgr.small"            },
	{0x5D0DD486, "icn_l_medkit.ncgr.small"             },
	{0x5D13D2E4, "fx_sha_rift_imp.nsbta.small"         },
	{0x5D1412BC, "chao_ie_g.nsbmd.small"               },
	{0x5D1641C3, "bmp_ex_walk.nsbca"                   },
	{0x5D1968D6, "chao_ga_1.ncgr.small"                },
	{0x5D215A69, "tyx_ex_walk.nsbca"                   },
	{0x5D28DEFD, "lvl_stat_tspnl00.ncgr.small"         },
	{0x5D292004, "a2_s7_ex.walk.tga"                   },
	{0x5D2C389E, "charpro_cre2.ncgr.small"             },
	{0x5D2C5653, "pl_pu_emrld7.nsbtx.small"            },
	{0x5D518FCE, "jrl_tlgbx_ms30.ncgr.small"           },
	{0x5D620E93, "ixb_cb_rise.nsbca"                   },
	{0x5D6B7648, "prtl_txmgen_1.ncgr.small"            },
	{0x5D7646FE, "swa_ex_idle.nsbca"                   },
	{0x5D8AB5D9, "fx_vox_b_ea_l.nsbmd.small"           },
	{0x5D8C651A, "healthbar.tga"                       },
	{0x5D923F71, "spell_vulnerable.spl"                },
	{0x5D9DCD03, "itm_glv_strong_2.ncgr.small"         },
	{0x5D9FFB8E, "prtl_thbgen_2.ncgr.small"            },
	{0x5DA112DC, "hnpc_tim_aa.nsbtx.small"             },
	{0x5DA7A172, "chao_ba_2.ncgr.small"                },
	{0x5DB58C3F, "cfx_omga_mf.nsbtx.small"             },
	{0x5DBCC8D6, "itm_ace_dimbu_1.ncgr.small"          },
	{0x5DC92FDA, "plt_cod_precursors.plo"              },
	{0x5DCDACC1, "lvl_stat_tspnl22.ncgr.small"         },
	{0x5DDD4D66, "exp_ghz1_fw00.ncgr.small"            },
	{0x5DDDE1E5, "pl_mov_arw_aa.nsbmd.small"           },
	{0x5DE490C5, "buzwasr_icon.tga"                    },
	{0x5DF05A72, "fx_gren_sha.nsbmd.small"             },
	{0x5E0BA221, "pl_pz_tyx_num.nsbtx.small"           },
	{0x5E1002DD, "fx_gun_scrn.nsbtp.small"             },
	{0x5E22A99A, "ghz_hintguy.dlg"                     },
	{0x5E268F56, "bfl_cb_getup.nsbca"                  },
	{0x5E26AC70, "prf_chrbx_di20.ncgr.small"           },
	{0x5E2C82F2, "knu_cb_victoryin.nsbca"              },
	{0x5E467E20, "pl_ob_blkb_co.nsbtx.small"           },
	{0x5E46B408, "big_cb_rund.nsbca"                   },
	{0x5E533379, "icn_s_catk_sat.ncgr.small"           },
	{0x5E5999EE, "prt_se_astdr.ncgr.small"             },
	{0x5E606112, "prtl_eggaha_1.ncgr.small"            },
	{0x5E60B249, "a2_s6_ex_mm.nbfp"                    },
	{0x5E60B24C, "a2_s6_ex_mm.nbfs"                    },
	{0x5E7C2B22, "arm_cb_getup.nsbca"                  },
	{0x5E7C7A89, "jrl_tsmbr_of02.ncgr.small"           },
	{0x5E821B2A, "exp_ghz1_fw22.ncgr.small"            },
	{0x5E852E83, "gzs_aa.nsbmd.small"                  },
	{0x5E89C4F6, "jrl_tmbox_ms11.ncgr.small"           },
	{0x5E9D950F, "pl_pz_swc_dm.nsbmd.small"            },
	{0x5EA61EA0, "a1_s1_ex_boatwinchnorth.gff"         },
	{0x5EA83A39, "loc_ambsh_it1.ncgr.small"            },
	{0x5EACF42A, "cfx_nflp1.nsbtx.small"               },
	{0x5EAD4CB0, "bsl_aa.nsbmd.small"                  },
	{0x5EC846FB, "icn_s_grnchcho.ncgr.small"           },
	{0x5ECC6A91, "cfx_pfbp1_bng.emit"                  },
	{0x5ED20605, "tyx_cb_blockout.nsbca"               },
	{0x5EF56040, "she_cb_blockout.nsbca"               },
	{0x5EFEBDF3, "prtl_dad.nclr.small"                 },
	{0x5F124A01, "prt_se_gtwrm.ncgr.small"             },
	{0x5F185B54, "prtl_amysca_2.ncgr.small"            },
	{0x5F24D789, "prtl_dronegen_3.ncgr.small"          },
	{0x5F3905BE, "statusframe_l.ncgr.small"            },
	{0x5F3ECA5B, "son_cb_dead.nsbca"                   },
	{0x5F40E622, "wstest2.gff"                         },
	{0x5F4276EB, "prtl_knuckgen_1.ncgr.small"          },
	{0x5F4410B5, "chaogarden_expbot.gui"               },
	{0x5F442BA2, "slg_cb_damage.nsbca"                 },
	{0x5F442BCC, "fx_a2_s3_ex_cld.nsbtx.small"         },
	{0x5F44C5FB, "gup_cb_shieldidl.nsbca"              },
	{0x5F4739C1, "itm_glvsercl_3.ncgr.small"           },
	{0x5F49611F, "cfx_ibep1.nsbmd.small"               },
	{0x5F4CF19C, "cmbtbg_a2_s7.nsbta.small"            },
	{0x5F50E074, "cfx_crup1.nsbtx.small"               },
	{0x5F546394, "pl_mv_wnc_aa.nsbmd.small"            },
	{0x5F564EAE, "plt_lo_voxai.plo"                    },
	{0x5F5D096A, "prtl_sal.nclr.small"                 },
	{0x5F600A00, "cbt_menu_button_of00.ncgr.small"     },
	{0x5F6FBA65, "kdr_ex_walk.nsbca"                   },
	{0x5F7C582C, "chao_ji_r.nsbmd.small"               },
	{0x5F8298B9, "charpro_ts_a00.ncgr.small"           },
	{0x5F88EDCC, "a1_s2_card_t1.nbfp"                  },
	{0x5F88EDCF, "a1_s2_card_t1.nbfs"                  },
	{0x5F8A6460, "prtl_ffmmad_1.ncgr.small"            },
	{0x5F976CBE, "endpart1.gda"                        },
	{0x5FA5BFF8, "wmp_txtbr_tx02.ncgr.small"           },
	{0x5FA9853F, "endpart2.gda"                        },
	{0x5FAA8F17, "prtl_nesgen_3.ncgr.small"            },
	{0x5FAB9061, "a2_s2_i05_2.walk.tga"                },
	{0x5FB1875F, "a2_s7_ex_slide1.gff"                 },
	{0x5FB3EAC3, "cmbtbg_a2_s7_i03.nsbmd.small"        },
	{0x5FB5CD7B, "chao_ie_0.ncgr.small"                },
	{0x5FC07C31, "lvl_tsstr_ms01.ncgr.small"           },
	{0x5FC39FE0, "a2_s7_ex_slide2.gff"                 },
	{0x5FC59A50, "vox_ex_idle.nsbca"                   },
	{0x5FC9F373, "wmp_s_radar.ncgr.small"              },
	{0x5FCEAC7B, "she_cb_idle.nsbca"                   },
	{0x5FD5B861, "a2_s7_ex_slide3.gff"                 },
	{0x5FE16DF4, "spell_adrenaline.spl"                },
	{0x5FF04043, "exp_wel1_mm01.ncgr.small"            },
	{0x5FF3207F, "cho_tlgbx_ms02.ncgr.small"           },
	{0x5FF6CF1E, "guianim_sto.gda"                     },
	{0x600512ED, "pfx_rng1_rng.emit"                   },
	{0x600694BC, "prt_se_sudrn.ncgr.small"             },
	{0x6006E31E, "gui_tutorial_ts_2.tga"               },
	{0x600B87E5, "str_buttn_on00.ncgr.small"           },
	{0x600C0EEC, "a1_w2_gh_10.ncgr.small"              },
	{0x601178DD, "a1_shadowjoinpartyselect.gff"        },
	{0x601EA17A, "chaogarden.nsbmd.small"              },
	{0x60260309, "chao_fb_g.nsbtp.small"               },
	{0x60380D81, "icn_s_statsc.ncgr.small"             },
	{0x6046DFFB, "a1c3_boy.dlg"                        },
	{0x604D572E, "exl_shoes.ncgr.small"                },
	{0x6062ECC4, "btn_s_bar_of02.ncgr.small"           },
	{0x606750A5, "pfx_dop1_dop.emit"                   },
	{0x606AE280, "fx_gnp_missile.nsbtx.small"          },
	{0x60724C3B, "lizkdra_aa.nsbtx.small"              },
	{0x60810659, "chapter2warp.gff"                    },
	{0x60950E07, "exp_wel1_mm23.ncgr.small"            },
	{0x6096CD82, "gui_chao_j1.nclr.small"              },
	{0x60A74A46, "fx_mflash.nsbmd.small"               },
	{0x60A9E24E, "a1_w3_gh_02.ncgr.small"              },
	{0x60AF71AF, "fx_egg_wrc_prj.nsbtx.small"          },
	{0x60B3FC81, "pil_cb_damage.nsbca"                 },
	{0x60B5935F, "pl_am_tyx_gun.nsbtx.small"           },
	{0x60BF532C, "strongholdparty.gui"                 },
	{0x60C306AF, "tyx_ex_run.nsbca"                    },
	{0x60CB56F0, "pl_mv_cyc_to.nsbta.small"            },
	{0x60E02FE7, "mut_flash_ms01.ncgr.small"           },
	{0x60E11DEC, "invent_bar00.ncgr.small"             },
	{0x60E942DD, "upper_textbox_01.ncgr.small"         },
	{0x60EA3EDF, "pow_base.tga"                        },
	{0x60EB79D0, "cbt_cmdovricon.ncgr.small"           },
	{0x60EE7471, "prtl_wsprel_0.ncgr.small"            },
	{0x60FB56EB, "overmind_sluggish.spl"               },
	{0x60FF6562, "cbt_buttonleft_00.ncgr.small"        },
	{0x61036F09, "vom_cb_pattack2.nsbca"               },
	{0x610A8F4D, "lvl_but_bglit.ncgr.small"            },
	{0x6112C971, "big_cb_blockin.nsbca"                },
	{0x61142CA8, "cbt_focusbad.nanr.small"             },
	{0x6124E616, "itm_ace_timmy_2.ncgr.small"          },
	{0x6136B65D, "a1_s1_i01_ts.nbfp"                   },
	{0x6136B660, "a1_s1_i01_ts.nbfs"                   },
	{0x6145CBB8, "chao_fc_2.ncgr.small"                },
	{0x614BAD1C, "big_ex_run.nsbca"                    },
	{0x614BFE06, "gui_items_chsemerald.nclr.small"     },
	{0x614E8903, "pl_am_noc_gnf.nsbtp.small"           },
	{0x6159080C, "cmbtbg_a1_s1.nsbtx.small"            },
	{0x616AE5FE, "ix_cb_counter.nsbca"                 },
	{0x617A4222, "buzwasr_aa_p.ncgr.small"             },
	{0x617E2F28, "journalcredits.gda"                  },
	{0x6181686D, "son_cb_counter.nsbca"                },
	{0x6181813D, "plt_cod_chao.plo"                    },
	{0x6181E853, "charpro_barlit03.ncgr.small"         },
	{0x6189511F, "cbt_cornr_ms00.ncgr.small"           },
	{0x619A258B, "icn_l_victory3.ncgr.small"           },
	{0x619EDF92, "a2_chb_thelxe_dies.gff"              },
	{0x61AF9861, "gzd_cb_damage.nsbca"                 },
	{0x61B06228, "slg_cb_getup.nsbca"                  },
	{0x61B70654, "pl_pz_gen_on.nsbca"                  },
	{0x61BB40E1, "plt_ch3_roboanimals.plo"             },
	{0x61C13C78, "a1sq_gunagent.dlg"                   },
	{0x61C2294D, "exl_smbtn_on10.ncgr.small"           },
	{0x61C9ED1A, "itm_glv_stealth0.ncgr.small"         },
	{0x61CA9923, "rou_cb_victory.nsbca"                },
	{0x61D17537, "a1c1_forcetails.gff"                 },
	{0x61D8A5C0, "kro_ex_damage.nsbca"                 },
	{0x61D8E149, "egb_cb_idle.nsbca"                   },
	{0x61D95168, "syr_aa.nsbtx.small"                  },
	{0x61DC9A51, "c_taeg_ta.nsbca"                     },
	{0x61DE166D, "a1_s2_card_t2.nbfp"                  },
	{0x61DE1670, "a1_s2_card_t2.nbfs"                  },
	{0x61E1E920, "chao_jb_r.nsbtx.small"               },
	{0x61E2C8D2, "pl_pz_fanlt_rg.nsbtp.small"          },
	{0x61E33A6B, "cbt_flgetm_1.ncgr.small"             },
	{0x61E457AD, "fx_impactflash.ncgr.small"           },
	{0x61EC7D08, "icn_s_statwea.ncgr.small"            },
	{0x61EF774D, "icn_l_wmp_cyclone.nclr.small"        },
	{0x61F0711F, "itm_glvgrpcl_2.ncgr.small"           },
	{0x61FFFBED, "fx_son_shock.nsbtx.small"            },
	{0x620150FB, "fx_giz_lzr_prj.nsbta.small"          },
	{0x62170290, "conv_bark_bs03.ncgr.small"           },
	{0x621A7D14, "prtl_tailsexc_3.ncgr.small"          },
	{0x621CCE62, "gensonn_aa.nsbtx.small"              },
	{0x622F15A5, "exl_altet_ms00.ncgr.small"           },
	{0x62381FCD, "prtl_txrhap_2.ncgr.small"            },
	{0x62479159, "chao_ha_3.ncgr.small"                },
	{0x6253D155, "pfx_smk1_krn.emit"                   },
	{0x62551356, "prtl_hczgen_3.ncgr.small"            },
	{0x6266980E, "chao_he_g.nsbtp.small"               },
	{0x62704ADB, "mut_shoes_ms10.ncgr.small"           },
	{0x627785B4, "man_tsvin_02.ncgr.small"             },
	{0x62842E43, "str_helpb_on00.ncgr.small"           },
	{0x62896C99, "man_tsvst_of02.ncgr.small"           },
	{0x62A12EDB, "cfx_hypp1.nsbtx.small"               },
	{0x62A2ED20, "prtl_rougeinq_2.ncgr.small"          },
	{0x62A6335F, "she_cb_blockin.nsbca"                },
	{0x62A86787, "big_cb_attack.nsbca"                 },
	{0x62BC37F8, "cfx_hazp1_fal.emit"                  },
	{0x62BFF50D, "fx_a2_s5_env.nsbmd.small"            },
	{0x62C093A7, "fx_vox_p_ea_l.nsbmd.small"           },
	{0x62C69CAD, "fx_a1_s6_ex_env.nsbta.small"         },
	{0x62C81955, "profilegui_tut_bs.tga"               },
	{0x62C96BA7, "prtl_nrsge2_2.ncgr.small"            },
	{0x62CD6578, "exl_buttn_ms00.ncgr.small"           },
	{0x62D1AE3B, "btn_bar_00.ncgr.small"               },
	{0x62D32CE6, "lower_blue_ring_04.ncgr.small"       },
	{0x62D6DDA0, "a1_s4_ex.walk.tga"                   },
	{0x62D891C3, "conv_icon01.ncgr.small"              },
	{0x62E86D29, "cbt_rttarget.ncgr.small"             },
	{0x62EC7832, "a1_w1_ul.nbfp"                       },
	{0x62EC7835, "a1_w1_ul.nbfs"                       },
	{0x62EF8C13, "itm_glvblncl_3.ncgr.small"           },
	{0x62F2CA1A, "prt_tp_m_big.ncgr.small"             },
	{0x62F663DE, "bfl_aa.nsbtx.small"                  },
	{0x62F6980B, "a1_s2_i01.walk.tga"                  },
	{0x62F9DED4, "cbt_trapd_ms.ncer.small"             },
	{0x62FBD7DD, "a2c7_kron_soldier.gff"               },
	{0x63026CD8, "bel_cb_pattack.nsbca"                },
	{0x6307BBD7, "mut_chaob_ms11.ncgr.small"           },
	{0x630DEAA2, "tyx_cb_counter.nsbca"                },
	{0x63172B96, "itm_fotslplt_2.ncgr.small"           },
	{0x63197EA8, "pfx_ash1.nsbmd.small"                },
	{0x63265C01, "flee02_bs.tga"                       },
	{0x633C3EB9, "prtl_sonicinq_1.ncgr.small"          },
	{0x633D6CA6, "prtl_krscon_2.ncgr.small"            },
	{0x63428429, "itm_ace_root_1.ncgr.small"           },
	{0x634336C6, "pfx_ash1_krn.emit"                   },
	{0x634706A6, "a1c2_commander.dlg"                  },
	{0x634B76DE, "prt_se_raidn.ncgr.small"             },
	{0x634BEA7C, "lvl_tdbld_ms02.ncgr.small"           },
	{0x637A7B81, "str_tsmbr_on01.ncgr.small"           },
	{0x637EEF7B, "charpro_rou3.ncgr.small"             },
	{0x638DC54F, "prtl_eixlaf_2.ncgr.small"            },
	{0x638E0F77, "chao_jt_r.nsbmd.small"               },
	{0x63901FEC, "act2_world0.are"                     },
	{0x63940016, "swt_cb_attackc.nsbca"                },
	{0x6395A79E, "plt_cod_twilightcage.plo"            },
	{0x6399BC0D, "c_ombi_bi.nsbca"                     },
	{0x639A7622, "armdilr_icon.tga"                    },
	{0x639E42E6, "a2c8_scy_arrestedtransition.gff"     },
	{0x639F1187, "icn_s_wsl1wcho.ncgr.small"           },
	{0x639F3179, "c_omro_ro.nsbca"                     },
	{0x63B83D47, "a2c10_reposition_knuckles.gff"       },
	{0x63DEADF1, "amy_crush.gff"                       },
	{0x63E063C6, "shp_cb_idle.nsbca"                   },
	{0x63E3D178, "pfx_bub1.nsbtp.small"                },
	{0x63F0B840, "lvl_tdbld_ms24.ncgr.small"           },
	{0x63F139BD, "cfx_heap1.nsbmd.small"               },
	{0x6402796A, "itm_act2_chaos5_2.ncgr.small"        },
	{0x6406CBFE, "a1c1_enemytype_tutorial.gff"         },
	{0x64184994, "itm_ace_ironw_1.ncgr.small"          },
	{0x641E34CD, "bsc_cb_attack.nsbca"                 },
	{0x6422012F, "icn_s_patk_tai3.ncgr.small"          },
	{0x642296A1, "pfx_nicktest01.nsbmd.small"          },
	{0x64248E0B, "cfx_ccup1_spk.emit"                  },
	{0x64272512, "a2c8_exit_nrrgal.gff"                },
	{0x643B1801, "prtl_krsgen_3.ncgr.small"            },
	{0x64432837, "a1c3_sentryhint2.gff"                },
	{0x6447457D, "pro_smbox_01.ncgr.small"             },
	{0x644BA6F9, "gzs_cb_pattack.nsbca"                },
	{0x644C1742, "c_sotaamkn_1_kn.nsbca"               },
	{0x6466F7E7, "brdrhk_aa.nsbtx.small"               },
	{0x646A8345, "fx_gnp_mflash.nsbtx.small"           },
	{0x646C7883, "projectiles.gda"                     },
	{0x646DDFC9, "fx_vox_g_es.nsbta.small"             },
	{0x647175EE, "prtl_guncodet_1.ncgr.small"          },
	{0x647A8C59, "prtl_guncosmg_2.ncgr.small"          },
	{0x647C33B3, "icn_s_statpoi.ncgr.small"            },
	{0x647E0F57, "a1_s5_ex_mm.nclr.small"              },
	{0x648722C6, "a2_s6_ex_la.smp"                     },
	{0x648D1446, "itm_fotslpdk_3.ncgr.small"           },
	{0x648DCCE7, "spell_sluggish.spl"                  },
	{0x6494A3CF, "cfx_wptp1_trl.emit"                  },
	{0x649AB25A, "cfx_fwnp1.nsbmd.small"               },
	{0x64A1A987, "prtl_syraxgen_2.ncgr.small"          },
	{0x64A72D13, "chao_jh_g.nsbtp.small"               },
	{0x64AEC9E3, "spell_big_regen.spl"                 },
	{0x64B53472, "itm_ace_shld_3.ncgr.small"           },
	{0x64C24EBE, "mrz_ccz.gff"                         },
	{0x64C91834, "icn_l_onbash1.ncgr.small"            },
	{0x64E2F7DF, "spell_tinker_2.spl"                  },
	{0x64E3F5FE, "chao_je_2.ncgr.small"                },
	{0x64EA498D, "charpro_bs_c13.ncgr.small"           },
	{0x64F1AF93, "lvl_tlgbx_ms02.ncgr.small"           },
	{0x64F51060, "spell_tinker_3.spl"                  },
	{0x64FAAF38, "cmbtbg_a1_s4.nsbta.small"            },
	{0x64FBDA7C, "a2c10_emeraldfreed_ix.gff"           },
	{0x6502C43A, "prtl_egglaf_0.ncgr.small"            },
	{0x65129BB9, "creaturetypetut.gda"                 },
	{0x6520BB41, "big_aa.nsbmd.small"                  },
	{0x65219B2A, "prtl_tdadsad_1.ncgr.small"           },
	{0x6521C122, "fx_els_lt_aa.nsbtp.small"            },
	{0x652CC064, "icn_s_gstarwmp.ncgr.small"           },
	{0x65432612, "ehd_tbbox_ms00.ncgr.small"           },
	{0x6555A288, "a2c10_ix_screenshake.gff"            },
	{0x6558B47A, "bsc_cb_getup.nsbca"                  },
	{0x6560EEF1, "a1_w2_br_01.ncgr.small"              },
	{0x6561511C, "wmp_trv_14_13.gff"                   },
	{0x656916D7, "prf_tmbar_of02.ncgr.small"           },
	{0x65694299, "lvl_pow_tspnl11.ncgr.small"          },
	{0x65741E8A, "a1_w0_gh_01.ncgr.small"              },
	{0x65742E50, "itm_fotslprg_3.ncgr.small"           },
	{0x65776846, "gup_aa_shid.nsbta.small"             },
	{0x65779E78, "_resetinput.gff"                     },
	{0x657C5734, "pl_pz_vox_lts.nsbtp.small"           },
	{0x65845991, "prtl_ligaia.nclr.small"              },
	{0x658796D6, "cfx_watp1.nsbtx.small"               },
	{0x6587CFF4, "pl_ob_blkn_cr.nsbmd.small"           },
	{0x659B1CF6, "fx_gnp_himissile.nsbmd.small"        },
	{0x65AC113F, "plt_lo_sectorscylla.plo"             },
	{0x65AF9E04, "pfx_briantest02.nsbtx.small"         },
	{0x65B1B013, "prtl_generic_2.ncgr.small"           },
	{0x65C3F47F, "wmp_l_airpo_01b.ncgr.small"          },
	{0x65C63AF0, "pl_pz_crn_tu.nsbtp.small"            },
	{0x65D39FE6, "gencrm_aa.nsbtx.small"               },
	{0x65E2088A, "lvl_tmstr_ms00.ncgr.small"           },
	{0x65EA9894, "egm_cb_idle.nsbca"                   },
	{0x65EC43B3, "exl_circe_ms20.ncgr.small"           },
	{0x65F006C8, "pfx_nrg1_chw.emit"                   },
	{0x65F20F99, "genamyn_aa.nsbtx.small"              },
	{0x65F3A06B, "chao_jm_r.nsbtx.small"               },
	{0x6602B4C8, "fx_a1_s1_ex_env.nsbta.small"         },
	{0x660EB160, "str_txtbr_hi00.ncgr.small"           },
	{0x661E72D6, "prtl_nrdge2_0.ncgr.small"            },
	{0x663A4F2D, "cbt_chaseplayer.ncgr.small"          },
	{0x6643FBCB, "prt_mp_rou.ncgr.small"               },
	{0x66525261, "itm_ace_spdbr_2.ncgr.small"          },
	{0x6672A767, "cfx_jwl1.nsbtx.small"                },
	{0x667A9A92, "fx_vox_y_ea_s.nsbtx.small"           },
	{0x667E9502, "chao_jk_0.ncgr.small"                },
	{0x668036B0, "slg_aa.nsbtx.small"                  },
	{0x6686D64E, "lvl_tmstr_ms22.ncgr.small"           },
	{0x6687F8E5, "exp_ghz2_mm10.ncgr.small"            },
	{0x66882F07, "exp_ghz1_mm13.ncgr.small"            },
	{0x669383AA, "gui_conv_portraits.nclr.small"       },
	{0x6697D423, "levelup_bonus_lower.gui"             },
	{0x669F2F16, "a1c5_p9_scene1.gff"                  },
	{0x66A011F0, "prtl_timcal_2.ncgr.small"            },
	{0x66A23A6F, "conv_portr_00.ncgr.small"            },
	{0x66A43410, "fx_leafdn.nsbtx.small"               },
	{0x66A58908, "a1c5_p5_shipattacked.gff"            },
	{0x66A74F9E, "itm_glv_light_2.ncgr.small"          },
	{0x66AEAAD6, "kro_ex_run.nsbca"                    },
	{0x66B14797, "a1c5_p9_scene2.gff"                  },
	{0x66C36018, "a1c5_p9_scene3.gff"                  },
	{0x66DD184F, "cfx_ch6.nsbmd.small"                 },
	{0x66F07B4F, "main_pnl_off12.ncgr.small"           },
	{0x66F71154, "cfx_mexp1.nsbtx.small"               },
	{0x66F85A14, "omg_ex_jump.nsbca"                   },
	{0x6708A398, "chao_hb_g.nsbmd.small"               },
	{0x67178CB8, "prtl_boyhap_1.ncgr.small"            },
	{0x6726E32B, "chao_jr_2.ncgr.small"                },
	{0x672CC6A9, "exp_ghz2_mm32.ncgr.small"            },
	{0x67305403, "chao_gc_r.nsbmd.small"               },
	{0x67370E96, "mmp_lgbar_02.ncgr.small"             },
	{0x673B07C6, "itm_ace_seed_1.ncgr.small"           },
	{0x67406BBF, "pl_pz_ppv_up.nsbtp.small"            },
	{0x67455588, "plt_gbl_inparty.plo"                 },
	{0x67470833, "conv_portr_22.ncgr.small"            },
	{0x6754C931, "itm_ace_prune_3.ncgr.small"          },
	{0x67636CF1, "pl_am_noc_gcf.nsbca"                 },
	{0x67664516, "chao_gb_0.ncgr.small"                },
	{0x676C541F, "son_ex_loop.nsbca"                   },
	{0x67758D35, "exp_mappin.ncgr.small"               },
	{0x677FAED5, "main_pnl_on16.ncgr.small"            },
	{0x678FF2B2, "icn_l_wmp_tornado.ncgr.small"        },
	{0x67B4E0E8, "shp_cb_counter.nsbca"                },
	{0x67B653F5, "kdr_ex_fall.nsbca"                   },
	{0x67CE34ED, "swa_cb_getup.nsbca"                  },
	{0x67D07F3A, "sso_cb_pattack.nsbca"                },
	{0x67E55333, "minimapgui_bs.tga"                   },
	{0x67F1C12B, "cfx_wndp1.nsbtp.small"               },
	{0x680BDD8F, "pfx_haz1.nsbmd.small"                },
	{0x682260DB, "lower_textbox_02.ncgr.small"         },
	{0x682DE8D4, "animationevents.gda"                 },
	{0x682E0BEE, "prtl_shdhel_2.ncgr.small"            },
	{0x68350BCA, "prf_tpanl_ms10.ncgr.small"           },
	{0x6838DFC9, "prtl_wsgrel_1.ncgr.small"            },
	{0x68472819, "ccw_ne_nw.gff"                       },
	{0x68475A90, "itm_fotstlbt_2.ncgr.small"           },
	{0x68488448, "a1c2_robot2_dies.gff"                },
	{0x684D9463, "itm_ace_crzbn_0.ncgr.small"          },
	{0x68565564, "icn_s_chaocho.ncgr.small"            },
	{0x686371F3, "prtl_amysad_0.ncgr.small"            },
	{0x6863A57A, "cfx_ertp1.nsbmd.small"               },
	{0x6869728E, "cfx_dusp1.nsbtp.small"               },
	{0x6869A91B, "worldmap_tut.nbfp"                   },
	{0x6869A91E, "worldmap_tut.nbfs"                   },
	{0x688C599F, "a2sq_thebeshouse.dlg"                },
	{0x68979183, "a2c7_foremankrag.dlg"                },
	{0x68AFA06C, "prt_se_scyla.ncgr.small"             },
	{0x68B8E45E, "chao_js_g.nsbtp.small"               },
	{0x68BA87D4, "c_ambicr_bi.nsbca"                   },
	{0x68BFD59F, "cbt_cmddnon.ncgr.small"              },
	{0x68C7A4AE, "conv_bubble_ts03.ncgr.small"         },
	{0x68C9D750, "amy_cb_pattack04.nsbca"              },
	{0x68CBA3AB, "pfx_rok1.nsbtp.small"                },
	{0x68CC04A9, "fx_wtrfl_mist.nsbtp.small"           },
	{0x68D2591B, "upper_red_ring_03.ncgr.small"        },
	{0x68D2C13E, "a2c10_supersonic_revert.gff"         },
	{0x68D40B8F, "cfx_rckp1.nsbmd.small"               },
	{0x68DCF1A7, "invent_scroll_02.ncgr.small"         },
	{0x68DD9620, "prtl_scygen_3.ncgr.small"            },
	{0x68E069CA, "exp_fatiguebar.ncgr.small"           },
	{0x68E3430F, "a2_s4_card_t1.nbfp"                  },
	{0x68E34312, "a2_s4_card_t1.nbfs"                  },
	{0x68ED47CC, "cfx_spbp1_spb.emit"                  },
	{0x68FC8AAE, "prtl_fsggen_0.ncgr.small"            },
	{0x68FE6153, "a2_w0_vb_00.ncgr.small"              },
	{0x6917CCE1, "a1c1_sonic_chase_drone.gff"          },
	{0x69267238, "chao_ca_r.nsbtx.small"               },
	{0x692D6F64, "man_strbt_of11.ncgr.small"           },
	{0x692EB148, "prtl_crehap_1.ncgr.small"            },
	{0x692F7770, "cfx_prj_min.nsbtp.small"             },
	{0x69316557, "fx_ix_bhol_muz.nsbca"                },
	{0x6941262E, "ixb_cb_dead.nsbca"                   },
	{0x6946D98D, "ch5_ixpissed_st.gda"                 },
	{0x6949389D, "chao_je_g.nsbmd.small"               },
	{0x6954B025, "exl_smbtn_of01.ncgr.small"           },
	{0x6961404C, "overmind_distracted.spl"             },
	{0x69688D12, "ssf_tlgbx_ms21.ncgr.small"           },
	{0x6975FFFE, "rax_ex_idle.nsbca"                   },
	{0x69760776, "cbt_fatiguebar.ncgr.small"           },
	{0x6976D63D, "cbt_cmddnoff.ncgr.small"             },
	{0x6980BD3F, "pl_am_noc_gun.nsbmd.small"           },
	{0x6983B6E5, "cbt_rndnum5.ncgr.small"              },
	{0x69890B42, "jurnl_mis_line04.ncgr.small"         },
	{0x69898A6F, "ex_bloop_right.nsbca"                },
	{0x6995F946, "worldmap_soniciconnopin.gui"         },
	{0x699FDF64, "a2_s1_ex_mm.nbfp"                    },
	{0x699FDF67, "a2_s1_ex_mm.nbfs"                    },
	{0x69A92C11, "prtl_amy.nclr.small"                 },
	{0x69B34413, "cbt_focusringb.nanr.small"           },
	{0x69B8925F, "cbt_cmdframe13.ncgr.small"           },
	{0x69C3BB08, "prtl_vxcleugen_3.ncgr.small"         },
	{0x69E171BA, "exl_circl_ms20.ncgr.small"           },
	{0x69E2179E, "str_tlgbx_ms20.ncgr.small"           },
	{0x69FA01B8, "exp_cct1_mm31.ncgr.small"            },
	{0x6A036AC1, "prtl_vxcrblgen_2.ncgr.small"         },
	{0x6A108975, "icn_s_defend.ncgr.small"             },
	{0x6A1384D0, "loc_round_fr0.ncgr.small"            },
	{0x6A1F7659, "prtl_timgen_1.ncgr.small"            },
	{0x6A22CE35, "prt_se_thelx.ncgr.small"             },
	{0x6A24DC44, "cbt_menu_down_on.ncgr.small"         },
	{0x6A2659FF, "prt_se_gizda.ncgr.small"             },
	{0x6A288902, "bsc_aa.nsbtx.small"                  },
	{0x6A2B1EA6, "pl_am_met_gfr.nsbca"                 },
	{0x6A311432, "spell_int_voxai.spl"                 },
	{0x6A36A434, "mut_btton_of00.ncgr.small"           },
	{0x6A3929D2, "prt_tp_rou.ncgr.small"               },
	{0x6A438577, "prt_tp_chao.ncgr.small"              },
	{0x6A47E9C8, "main_pnl_big04a.ncgr.small"          },
	{0x6A4958B7, "prtl_fmncon_2.ncgr.small"            },
	{0x6A553469, "cbt_menu_button_on01.ncgr.small"     },
	{0x6A693100, "icn_l_utbash1.ncgr.small"            },
	{0x6A73CA82, "pfx_bub3.nsbtx.small"                },
	{0x6A749AB2, "cfx_calp1.nsbtp.small"               },
	{0x6A7DA1F4, "invent_stat10.ncgr.small"            },
	{0x6A7DB14D, "itm_act2_mastr_1.ncgr.small"         },
	{0x6A80271E, "fx_cheese_prj.nsbtx.small"           },
	{0x6A861A70, "fx_vox_r_ea_s.nsbmd.small"           },
	{0x6A9E60A7, "spell_int_animal_fly.spl"            },
	{0x6A9E6E48, "crm_cb_pattack01.nsbca"              },
	{0x6AAA4ADC, "prt_tp_she.ncgr.small"               },
	{0x6AB57B1E, "itm_fotttnhy_2.ncgr.small"           },
	{0x6AB58FE6, "a1_w2_ai_00.ncgr.small"              },
	{0x6AB7C3FA, "knu_cb_blockd.nsbca"                 },
	{0x6AB9B03E, "a2c10_sonic_face_gizoids.gff"        },
	{0x6AC25436, "knu_ex_twitch.nsbca"                 },
	{0x6AD8EC60, "fx_greenpuff.nsbtp.small"            },
	{0x6AEBB8D4, "itm_fotgrshy_2.ncgr.small"           },
	{0x6AF2EAC3, "chaopickuptut.gda"                   },
	{0x6AFEB615, "itm_ace_goggl_1.ncgr.small"          },
	{0x6B0ADC16, "fx_gren_noc.nsbmd.small"             },
	{0x6B0BBA4D, "a1_s2_i03.walk.tga"                  },
	{0x6B264539, "cfx_rtip1.nsbtx.small"               },
	{0x6B386BB0, "a2_s4_card_t2.nbfp"                  },
	{0x6B386BB3, "a2_s4_card_t2.nbfs"                  },
	{0x6B3F56D2, "chao_fc_g.nsbtx.small"               },
	{0x6B3F93E7, "fx_els_wn_aa.nsbtp.small"            },
	{0x6B59E42B, "tut_inven.tga"                       },
	{0x6B690746, "a1_s2_i06_mm.nclr.small"             },
	{0x6B780F4B, "cfx_fgrp1_frr.emit"                  },
	{0x6B7E8700, "cfx_icep1.nsbmd.small"               },
	{0x6B816606, "fx_a2_s2_ex_env.nsbmd.small"         },
	{0x6B8DECF6, "spell_medibot_2.spl"                 },
	{0x6B8F8AA6, "main_action00.ncgr.small"            },
	{0x6B9189AB, "a2scy_tyxwar1.dlg"                   },
	{0x6B92A7F8, "chao_fd_1.ncgr.small"                },
	{0x6B92AC99, "prtl_sbtgen_2.ncgr.small"            },
	{0x6B9598EC, "fx_bird.nsbmd.small"                 },
	{0x6B9F479F, "sso_ex_run.nsbca"                    },
	{0x6BA00577, "spell_medibot_3.spl"                 },
	{0x6BA3A22C, "a2scy_tyxwar2.dlg"                   },
	{0x6BA7D6B6, "omg_ex_idle.nsbca"                   },
	{0x6BB3E70F, "prtl_wdmnexh_0.ncgr.small"           },
	{0x6BB4542A, "worldmap_backbutton.gui"             },
	{0x6BB47E38, "prtl_rougedis_2.ncgr.small"          },
	{0x6BB61CD5, "adv_sonic.gda"                       },
	{0x6BC9D5F5, "camerapanangel.gff"                  },
	{0x6BCF185B, "gwr_cb_dead.nsbca"                   },
	{0x6BD369BE, "btn_l_lftarw_on.ncgr.small"          },
	{0x6BE63D56, "icn_s_rarwonlvl.ncgr.small"          },
	{0x6BEB82D6, "prtl_sdwmad_2.ncgr.small"            },
	{0x6BF68A23, "swan_aa.nsbtx.small"                 },
	{0x6BF767E1, "a1c1_droneenter.gff"                 },
	{0x6BFBB6AE, "pl_pz_brd_idl.nsbca"                 },
	{0x6BFCCB5B, "prt_se_shp.ncgr.small"               },
	{0x6C06FFA2, "icn_s_str_chao.ncgr.small"           },
	{0x6C11B2CF, "fadetest.gda"                        },
	{0x6C1E1114, "prtl_gnogen_2.ncgr.small"            },
	{0x6C1EA504, "cho_tmbt2_on02.ncgr.small"           },
	{0x6C2AC1AA, "rax_cb_blockin.nsbca"                },
	{0x6C3B1683, "reward_ts.gui"                       },
	{0x6C3DC1E5, "a2scy_tyxcivs.dlg"                   },
	{0x6C4123FB, "cfx_fglp1.nsbtp.small"               },
	{0x6C494C58, "pfx_smk2.nsbmd.small"                },
	{0x6C4DCFD1, "prtl_sonicdis_1.ncgr.small"          },
	{0x6C4E463C, "prtl_vxcriagen_1.ncgr.small"         },
	{0x6C509F7C, "itm_glvcrscl_0.ncgr.small"           },
	{0x6C50ED4E, "a2_s4_i01.walk.tga"                  },
	{0x6C520E00, "a2c10_ix_attack_sonic_2.gff"         },
	{0x6C54AD76, "min_circl_ms21.ncgr.small"           },
	{0x6C5B28EA, "chao_gb_g.nsbtp.small"               },
	{0x6C711B7C, "a1_s4_card_b1.nbfp"                  },
	{0x6C711B7F, "a1_s4_card_b1.nbfs"                  },
	{0x6C74C49F, "a2_chb_thebesleaves.gff"             },
	{0x6C7AE824, "ehd_fow_huh.ncgr.small"              },
	{0x6C81DED7, "icn_l_barwoflvl.ncgr.small"          },
	{0x6C82D955, "chao_fc_r.nsbtp.small"               },
	{0x6C85F309, "omg_cb_pattack01.nsbca"              },
	{0x6C8AAC7F, "fx_tyx_beam.nsbtx.small"             },
	{0x6C946D99, "chao_hb_2.ncgr.small"                },
	{0x6C9D61D6, "cmbtbg_ts_ghz113.ncgr.small"         },
	{0x6CA03DAE, "prtl_txrgen_2.ncgr.small"            },
	{0x6CA10476, "jurnl_cod_linelit02.ncgr.small"      },
	{0x6CCA5D6C, "pl_pz_brd_end.nsbca"                 },
	{0x6CCA5F38, "kro_cb_kd.nsbca"                     },
	{0x6CCE8505, "icn_l_oninv3.ncgr.small"             },
	{0x6CD60C36, "chapter0.gda"                        },
	{0x6CE113C3, "journalgui_bs_01.tga"                },
	{0x6CE150BE, "amy_cb_pattack2.nsbca"               },
	{0x6CE824B7, "chapter1.gda"                        },
	{0x6CF32C44, "journalgui_bs_02.tga"                },
	{0x6CF4297C, "pop_tggle_of03.ncgr.small"           },
	{0x6CF8CD79, "exl_qcrce_ms00.ncgr.small"           },
	{0x6CFA3D38, "chapter2.gda"                        },
	{0x6D01AAE4, "main_pnl_big01.ncgr.small"           },
	{0x6D03E234, "zz_greenhillzone.are"                },
	{0x6D0C55B9, "chapter3.gda"                        },
	{0x6D1E6E3A, "chapter4.gda"                        },
	{0x6D30415F, "runaway.gui"                         },
	{0x6D3086BB, "chapter5.gda"                        },
	{0x6D3F2668, "jillcan12sh.nftr"                    },
	{0x6D4202AC, "arm_cb_kd.nsbca"                     },
	{0x6D429F3C, "chapter6.gda"                        },
	{0x6D4583F6, "cfx_ccap1.nsbmd.small"               },
	{0x6D4F73FB, "a1sq_outhouse.dlg"                   },
	{0x6D54B7BD, "chapter7.gda"                        },
	{0x6D5AEFE8, "chao_jp_g.nsbmd.small"               },
	{0x6D5BEC72, "swt_ex_damage.nsbca"                 },
	{0x6D65FAC1, "lvl_lrgbt_of01.ncgr.small"           },
	{0x6D6AD713, "prtl_soniccon_3.ncgr.small"          },
	{0x6D788423, "cho_tradeoverlay.gui"                },
	{0x6D7AB9DC, "chap7scr_top.tga"                    },
	{0x6D884A2E, "pl_pz_rc_crs.nsbmd.small"            },
	{0x6D8B0503, "prt_tp_rng.ncgr.small"               },
	{0x6D8E2DED, "cmbtbg_a2_s1.nsbtx.small"            },
	{0x6D919806, "randitems.gda"                       },
	{0x6D935EF5, "gui_combat2.nclr.small"              },
	{0x6D9A40E6, "a1c3_shadowavoid.gff"                },
	{0x6DA9EFDC, "icn_s_replay.ncgr.small"             },
	{0x6DB85C93, "cfx_bub1.nsbtx.small"                },
	{0x6DC2075B, "pl_pz_swc_aa.nsbtx.small"            },
	{0x6DC33B7B, "exp_tws1_mm12.ncgr.small"            },
	{0x6DC3F80A, "prt_ce_cyn.ncgr.small"               },
	{0x6DC6C9E5, "invent_stat_tablit3.ncgr.small"      },
	{0x6DCD3C15, "invent_list02.ncgr.small"            },
	{0x6DD43F9C, "a2sq_haniman.dlg"                    },
	{0x6DE4BED4, "pirates_hack.gff"                    },
	{0x6DF34937, "prtl_krsmad_1.ncgr.small"            },
	{0x6E02B1C4, "vox_rbe_aa.nsbmd.small"              },
	{0x6E090506, "prtl_nrqge2_3.ncgr.small"            },
	{0x6E0C10F3, "prtl_omggen_2.ncgr.small"            },
	{0x6E176F69, "charpro_bs_a02.ncgr.small"           },
	{0x6E2BD3DC, "prtl_chbgen_1.ncgr.small"            },
	{0x6E4A6525, "a2c10_advgizoid1_attack.gff"         },
	{0x6E4D2CB0, "chao_aa_g.nsbmd.small"               },
	{0x6E4F55AA, "itm_act2_chaos6_1.ncgr.small"        },
	{0x6E4F5BB7, "bel_ex_walk.nsbca"                   },
	{0x6E519CD5, "cfx_glw_gump1.emit"                  },
	{0x6E5783EF, "rhk_ex_run.nsbca"                    },
	{0x6E635736, "son_cb_pattack01.nsbca"              },
	{0x6E6766D3, "prtl_salegen_0.ncgr.small"           },
	{0x6E69191B, "buz_ex_run.nsbca"                    },
	{0x6E6A1189, "icn_l_catk_sak.ncgr.small"           },
	{0x6E7B2DD6, "c_sosh_shab.nsbca"                   },
	{0x6E7E297B, "cfx_sson_burn.emit"                  },
	{0x6E80535A, "sha_cb_stunned.nsbca"                },
	{0x6E8A47BC, "cmbtbg_a1_i02.nsbta.small"           },
	{0x6E92B99A, "cbt_menubg_10.ncgr.small"            },
	{0x6E9BBDEF, "chao_ie_g.nsbtp.small"               },
	{0x6E9D3FC5, "egm_cb_kd.nsbca"                     },
	{0x6EAEB45E, "a2_s2_ex.are"                        },
	{0x6EBD4539, "chao_ch_1.ncgr.small"                },
	{0x6EBECE76, "big_ex_walk.nsbca"                   },
	{0x6EC6FE86, "plt_cod_argusevent.plo"              },
	{0x6ECC93F9, "exp_playerframe1.ncgr.small"         },
	{0x6EE7E39A, "icn_l_rarwonstr.ncgr.small"          },
	{0x6EEB8424, "upper_portrait_01.ncgr.small"        },
	{0x6EFC85F3, "pfx_smk1_bst.emit"                   },
	{0x6F0BA2A8, "prtl_shdmad_3.ncgr.small"            },
	{0x6F0C0381, "a2_s4_ex.walk.tga"                   },
	{0x6F0F39D0, "pl_pu_emrld4.nsbtx.small"            },
	{0x6F11B568, "fx_vox_g_ea_r.nsbta.small"           },
	{0x6F164B6E, "a1_s2_i04.walk.tga"                  },
	{0x6F167094, "cbt_sidebar.ncer.small"              },
	{0x6F1E55A8, "charpro_ome5.ncgr.small"             },
	{0x6F1EC26C, "icn_s_patk_cre4.ncgr.small"          },
	{0x6F24CFBC, "cbt_victory_star2.nanr.small"        },
	{0x6F26E4DE, "prtl_cap.nclr.small"                 },
	{0x6F2EBB38, "btn_menu_on.ncgr.small"              },
	{0x6F30D23E, "chao_jf_1.ncgr.small"                },
	{0x6F34734B, "jrl_tlgbx_ms00.ncgr.small"           },
	{0x6F36F002, "vox_lig_aa.nsbtx.small"              },
	{0x6F3DEEF8, "a1c1_rougetalkcommand.gff"           },
	{0x6F44AC73, "pop_lgbox_ms06.ncgr.small"           },
	{0x6F460B8A, "mut_circl_ms41.ncgr.small"           },
	{0x6F4BE90E, "lvl_stat_bspnl21.ncgr.small"         },
	{0x6F629B96, "a2_w0_ng.nclr.small"                 },
	{0x6F63EECA, "icn_s_baseatk.ncgr.small"            },
	{0x6F72F5A6, "fx_a1_s3_ex_env.nsbmd.small"         },
	{0x6F77B4C2, "creaturepositions.gda"               },
	{0x6F7D8F8D, "fx_a2_s7_i03.nsbtx.small"            },
	{0x6F9737E4, "pl_ob_blkn_cb.nsbmd.small"           },
	{0x6F99F9B6, "rhk_ex_jump.nsbca"                   },
	{0x6FA18EAD, "c_sosh_sb.nsbca"                     },
	{0x6FA3B285, "a1c3_sentryhint.gff"                 },
	{0x6FA7F1DC, "prtl_tailsinq_3.ncgr.small"          },
	{0x6FAD96C0, "a1_w3_mt_11.ncgr.small"              },
	{0x6FAF98BD, "prtl_knugen_1.ncgr.small"            },
	{0x6FB21152, "son_cb_getup.nsbca"                  },
	{0x6FB727D7, "exu_blbar_ms00.ncgr.small"           },
	{0x6FB8F241, "prtl_bigsnz_0.ncgr.small"            },
	{0x6FBCAB7F, "a1c5_p9_quakeescape.gff"             },
	{0x6FC080DC, "chao_ji_g.nsbtx.small"               },
	{0x6FC44DF1, "arm_ex_jump.nsbca"                   },
	{0x6FCDEC61, "mmp_smbar_00.ncgr.small"             },
	{0x6FD21285, "cfx_fglp1_glw.emit"                  },
	{0x6FD43768, "icn_s_wsl2wcho.ncgr.small"           },
	{0x6FD92067, "chao_jm_3.ncgr.small"                },
	{0x6FD9410F, "jrl_tlgbx_ms22.ncgr.small"           },
	{0x6FDDF771, "debugarea2.gda"                      },
	{0x6FDE5019, "jurnl_mis02.ncgr.small"              },
	{0x6FF00FF2, "debugarea3.gda"                      },
	{0x6FF544DD, "prtl_txcgen_0.ncgr.small"            },
	{0x70017E6F, "voxai_alpha_climb_down.gff"          },
	{0x70087190, "sco_cb_dead.nsbca"                   },
	{0x70098306, "itm_ace_rocks_0.ncgr.small"          },
	{0x700AFB4E, "exp_topframe.ncgr.small"             },
	{0x701FA809, "pl_pu_scr_bg.nsbca"                  },
	{0x7021B74D, "hnpc_wdmn_aa.nsbmd.small"            },
	{0x70254042, "pl_pz_swc_dm.nsbtp.small"            },
	{0x702D13DD, "cfx_gwtr_prj.nsbmd.small"            },
	{0x70390462, "cfx_bub1_zon.emit"                   },
	{0x703B64B4, "bsl_ex_idle.nsbca"                   },
	{0x703EBBE7, "kme_aa.nsbtx.small"                  },
	{0x7045A11A, "pfx_sms1.nsbtx.small"                },
	{0x704D2565, "itm_ace_shipm_0.ncgr.small"          },
	{0x705C29CD, "gui_chaogardts.nclr.small"           },
	{0x705E65E2, "itm_fotlgtbt_0.ncgr.small"           },
	{0x7070C9AC, "ghz1_fowmap.nclr.small"              },
	{0x707CDD92, "a1_w3_gh.nclr.small"                 },
	{0x70929E8D, "prtl_shdgen_0.ncgr.small"            },
	{0x70AD50A5, "cbt_inproftbar_s.ncgr.small"         },
	{0x70B16C9B, "plt_credits.plo"                     },
	{0x70B1C93D, "prtl_fwrgen_0.ncgr.small"            },
	{0x70C0D07B, "chao_gd_3.ncgr.small"                },
	{0x70C9769E, "hnpc_sci_idle.nsbca"                 },
	{0x70D10C52, "cfx_ibep1.nsbtp.small"               },
	{0x70D706E3, "textureanimations.gda"               },
	{0x70FD0E51, "pfx_bub1_bub.emit"                   },
	{0x70FF9562, "cfx_egbt_mf.nsbmd.small"             },
	{0x70FFA69A, "invent_title01.ncgr.small"           },
	{0x7102AF62, "icn_l_mm_airport.ncgr.small"         },
	{0x7104035F, "chao_ji_r.nsbtp.small"               },
	{0x711A4906, "upper_textbox_arrow_00.ncgr.small"   },
	{0x712952DC, "prtl_wsgsca_0.ncgr.small"            },
	{0x712FD519, "cmbtbg_a2_s4.nsbta.small"            },
	{0x71433FD8, "pl_ob_blk_bkn.nsbca"                 },
	{0x71487FA4, "pl_mv_cyc_sm.nsbtx.small"            },
	{0x714E8A99, "prtl_amyoff_3.ncgr.small"            },
	{0x714F4B23, "gui_portraitsskp.nclr.small"         },
	{0x7150BE2B, "chap4scr_bot.tga"                    },
	{0x7155CAE5, "pro_bar_on00.ncgr.small"             },
	{0x7157CD3A, "rwd_tsmbx_on00.ncgr.small"           },
	{0x715DB5C9, "c_shakn_kn.nsbca"                    },
	{0x7167F660, "charpro_backon.ncgr.small"           },
	{0x7173BF6B, "chao_js_1.ncgr.small"                },
	{0x7177AC43, "lvl_txbar_hi01.ncgr.small"           },
	{0x717AD9BA, "itm_ace_staplr_0.ncgr.small"         },
	{0x719275F9, "chapters.gda"                        },
	{0x71A64CAD, "chaogarden.nsbtp.small"              },
	{0x71A9732B, "icn_s_useitem.ncgr.small"            },
	{0x71AB9784, "fx_shadbeam.nsbta.small"             },
	{0x71ADAE86, "cfx_pmip1_pma.emit"                  },
	{0x71B7E834, "itm_ace_radio_1.ncgr.small"          },
	{0x71CDE4FF, "pl_am_tyx_gdn.nsbca"                 },
	{0x71D706A2, "gui_items_accessories.nclr.small"    },
	{0x71DDC726, "buz_cb_attack.nsbca"                 },
	{0x71DE3462, "buz_ex_jump.nsbca"                   },
	{0x71E2BD32, "a1_w3_br_10.ncgr.small"              },
	{0x71F15FE2, "prtl_shddef_0.ncgr.small"            },
	{0x71F5ECCB, "a1_w1_gh_10.ncgr.small"              },
	{0x720C05FE, "prtl_txrhum_2.ncgr.small"            },
	{0x721431C5, "charpro_she3.ncgr.small"             },
	{0x72217843, "prtl_charyb.nclr.small"              },
	{0x722EF579, "fx_mflash.nsbtp.small"               },
	{0x72314826, "cho_ts_21.ncgr.small"                },
	{0x7239FC9C, "vox_ovl_aa.nsbmd.small"              },
	{0x7243FF9B, "pl_pu_chst_aa.nsbtx.small"           },
	{0x7267F0A9, "itm_chao_egg_0.ncgr.small"           },
	{0x726ADB05, "pfx_haz1_blu.emit"                   },
	{0x726C29E6, "exp_ghz1_mm30.ncgr.small"            },
	{0x726C47CC, "man_bkbtn_on00.ncgr.small"           },
	{0x727A090E, "upper_green_ring_00.ncgr.small"      },
	{0x727D7CE6, "gui_combatelem.nclr.small"           },
	{0x7293C02D, "a1_w2_gh_02.ncgr.small"              },
	{0x72A218A5, "itm_ace_life_1.ncgr.small"           },
	{0x72A394A7, "pl_pz_vox_lr.nsbtx.small"            },
	{0x72AF17BE, "charpro_ts_a14.ncgr.small"           },
	{0x72AF5706, "fx_a2_s4_ex_ff.nsbtx.small"          },
	{0x72B4E6F7, "chao_ia_0.ncgr.small"                },
	{0x72BEDBF0, "main_pnl_on11.ncgr.small"            },
	{0x72C04D60, "wmp_space_on10.ncgr.small"           },
	{0x72C5916B, "son_cb_idle.nsbca"                   },
	{0x72CF486D, "a1c5_p8_end.gff"                     },
	{0x72D20310, "a1c2_knuckles.dlg"                   },
	{0x72D20F0C, "a1c2_knuckles.gff"                   },
	{0x72D5FD7D, "kro_ex_jump.nsbca"                   },
	{0x72D90F19, "cbt_healthbar_s.ncgr.small"          },
	{0x72E67098, "cbt_rndround0.ncgr.small"            },
	{0x72E8154E, "prt_tp_e_sdw.ncgr.small"             },
	{0x72F57823, "cfx_pfbp2.nsbtx.small"               },
	{0x72FF9C44, "sha_cb_pattack03.nsbca"              },
	{0x730A4037, "cbt_enemy_ambush.ncer.small"         },
	{0x730C5387, "itm_accangam_0.ncgr.small"           },
	{0x7320DC8F, "a1_s2_i05.walk.tga"                  },
	{0x7320FCC7, "son_ex_happy.nsbca"                  },
	{0x732211E5, "cfx_ghlp1.nsbtx.small"               },
	{0x732DFF60, "exl_smbtn_hi00.ncgr.small"           },
	{0x733678F6, "armdiln_aa.nsbmd.small"              },
	{0x733A16E8, "cfx_omga_mf.nsbta.small"             },
	{0x733A33FA, "plt_fpa2_shade.plo"                  },
	{0x733BC9B1, "spell_stun_distracted.spl"           },
	{0x733DC979, "chao_ib_g.nsbmd.small"               },
	{0x733E532A, "cfx_hypp1_twl.emit"                  },
	{0x73431F93, "chao_da_1.ncgr.small"                },
	{0x734E1B91, "charpro_stat30.ncgr.small"           },
	{0x7357C474, "guianim_cha.gda"                     },
	{0x735872FB, "jrl_tlgbr_of01.ncgr.small"           },
	{0x736579E4, "chao_hc_r.nsbmd.small"               },
	{0x7368464A, "prtl_gnggen_0.ncgr.small"            },
	{0x73766AF1, "pl_mov_arw_aa.nsbca"                 },
	{0x73842BAF, "pfx_aur1.nsbtx.small"                },
	{0x7388249F, "icn_s_statslu.ncgr.small"            },
	{0x738BB31B, "gzg_ex_idle.nsbca"                   },
	{0x73B059FA, "pl_pz_brd_3dn.nsbca"                 },
	{0x73B957DB, "plt_a2sq_telra.plo"                  },
	{0x73BCC542, "prtl_scyclm_3.ncgr.small"            },
	{0x73CAC1AD, "drna_cb_attacken.nsbca"              },
	{0x73CBD36E, "chao_ab_turn.nsbtx.small"            },
	{0x73CBE697, "strongholdtutorial.gda"              },
	{0x73D03621, "fx_drn_lzr_muz.nsbmd.small"          },
	{0x73D23827, "chao_jt_g.nsbtx.small"               },
	{0x73DDBC3A, "tyx_cb_attack.nsbca"                 },
	{0x73E3DB3D, "cfx_expp2.nsbmd.small"               },
	{0x73E452A4, "pro_medbx_10.ncgr.small"             },
	{0x73F7D34A, "prtl_knuexh_3.ncgr.small"            },
	{0x73F863D5, "spell_coverd_2.spl"                  },
	{0x73FCE07A, "a1c2_waitforbig.gff"                 },
	{0x73FDD8BB, "hnpc_mug_aa.nsbtx.small"             },
	{0x73FFC583, "a1_s1_i04_mm.nclr.small"             },
	{0x740A7C56, "spell_coverd_3.spl"                  },
	{0x740D74C6, "kdr_cb_pattack.nsbca"                },
	{0x740D82EC, "fx_son_whirl.nsbca"                  },
	{0x741C000C, "itm_ace_scan_3.ncgr.small"           },
	{0x742299CD, "man_strbt_on12.ncgr.small"           },
	{0x7423CB87, "combo3.ncgr.small"                   },
	{0x7436C90E, "fx_a1_s4_icb_bob.nsbca"              },
	{0x7441FF59, "son_cb_deadidle.nsbca"               },
	{0x74497658, "rhk_ex_idle.nsbca"                   },
	{0x744B5C9D, "icn_s_patktai.ncgr.small"            },
	{0x744BA1F3, "pro_lrgbx_01.ncgr.small"             },
	{0x744E3A1E, "a1_s2_ex_slide5_down_left.gff"       },
	{0x744F6305, "a1c5_p5_captaindefeated.gff"         },
	{0x74543497, "charpro_big3.ncgr.small"             },
	{0x745DC799, "guianim_pro.gda"                     },
	{0x746221E1, "cfx_spcp1_crd.emit"                  },
	{0x74654626, "prt_tp_m_tai.ncgr.small"             },
	{0x74660F90, "a2_s4_i03.walk.tga"                  },
	{0x746644D2, "cfx_aurp1.nsbtx.small"               },
	{0x7468E0F9, "sha_cb_counter.nsbca"                },
	{0x746BDDF7, "prtl_txmvar_1.ncgr.small"            },
	{0x74713A60, "a1_s3_i02_ts.nbfp"                   },
	{0x74713A63, "a1_s3_i02_ts.nbfs"                   },
	{0x7473CA93, "arm_ex_idle.nsbca"                   },
	{0x74791F50, "plt_a1sq_occupado.plo"               },
	{0x747967A0, "a1c2_mugger_approach.gff"            },
	{0x747FF3B1, "cyclone_kron.are"                    },
	{0x74892068, "pro_medbx_32.ncgr.small"             },
	{0x748A39DD, "prtl_skphap_2.ncgr.small"            },
	{0x7491186D, "charpro_rou.nclr.small"              },
	{0x7497BE76, "prtl_gunsca_1.ncgr.small"            },
	{0x74A129DB, "pfx_ash1.nsbtp.small"                },
	{0x74A21847, "prt_tp_e_amy.ncgr.small"             },
	{0x74B9C11D, "a1_s1_ex.walk.tga"                   },
	{0x74C2E400, "cbt_rndnum0.ncgr.small"              },
	{0x74C801E9, "cbt_black_upper.nanr.small"          },
	{0x74C8B675, "fx_a2_s3_ex_cld.nsbta.small"         },
	{0x74DAA656, "cmbtbg_a1_s6.nsbmd.small"            },
	{0x74E1A8B0, "plt_ch1_woodsman.plo"                },
	{0x74E6E3BB, "exp_ghq1_mm10.ncgr.small"            },
	{0x74E850AA, "prtl_tdadhap_0.ncgr.small"           },
	{0x74F6ADF6, "pl_pz_ppd_pup.nsbca"                 },
	{0x74F9560F, "itm_ace_medem_3.ncgr.small"          },
	{0x74FAC5D0, "itm_ace_beaca_0.ncgr.small"          },
	{0x74FECB81, "itm_fotkevbt_0.ncgr.small"           },
	{0x75023977, "charpro_she.nclr.small"              },
	{0x75036FBF, "big_cb_victory.nsbca"                },
	{0x7515BAAA, "chao_jt_r.nsbtp.small"               },
	{0x75182A02, "worldmap_sonicicon.gui"              },
	{0x7540B64C, "pirates_teleport.gff"                },
	{0x7548E27E, "cho_tmbtn_on00.ncgr.small"           },
	{0x754ADDCF, "prtl_sbtmad_0.ncgr.small"            },
	{0x75590920, "upper_bg_09.ncgr.small"              },
	{0x755B9819, "chao_da_r.nsbtx.small"               },
	{0x75717949, "icn_s_patk_rou5.ncgr.small"          },
	{0x7578E4F0, "cfx_heap1.nsbtp.small"               },
	{0x75805C9B, "jurnl_pnl12.ncgr.small"              },
	{0x758FE281, "cbt_black_lower.ncgr.small"          },
	{0x759119C6, "prtl_vxcbadgen_0.ncgr.small"         },
	{0x759C47D3, "vfx.gda"                             },
	{0x75A60EE9, "chao_jf_r.nsbmd.small"               },
	{0x75AA41D4, "pfx_nicktest01.nsbtp.small"          },
	{0x75B38937, "prtl_bigtht_1.ncgr.small"            },
	{0x75C2B616, "prtl_pilgen_3.ncgr.small"            },
	{0x75D8654B, "cfx_cgrp1.nsbmd.small"               },
	{0x75D86F54, "gui_items_accessories2.nclr.small"   },
	{0x75DB3E1C, "a2_s4_ex_slide1.gff"                 },
	{0x75DEAEB6, "buz_cb_pattack.nsbca"                },
	{0x75DF8438, "chao_fe_0.ncgr.small"                },
	{0x75E19582, "itm_fotrsthy_3.ncgr.small"           },
	{0x75ED53D0, "prtl_gizoid.nclr.small"              },
	{0x75ED569D, "a2_s4_ex_slide2.gff"                 },
	{0x75EF6D29, "fx_gnp_missile.nsbta.small"          },
	{0x75EFB178, "prtl_tailsugh_3.ncgr.small"          },
	{0x75F2C2B7, "pfx_mot1_sta.emit"                   },
	{0x75F5561D, "store1.gda"                          },
	{0x75F563FB, "prt_tp_m_egg.ncgr.small"             },
	{0x75FF6F1E, "a2_s4_ex_slide3.gff"                 },
	{0x7601F416, "icn_s_climb.ncgr.small"              },
	{0x76076E9E, "store2.gda"                          },
	{0x7611879F, "a2_s4_ex_slide4.gff"                 },
	{0x7613B452, "gzg_ex_run.nsbca"                    },
	{0x7619871F, "store3.gda"                          },
	{0x761ABCCA, "arm_ex_run.nsbca"                    },
	{0x76225D8D, "cfx_fwnp1.nsbtp.small"               },
	{0x762B9FA0, "store4.gda"                          },
	{0x762F5AB1, "targetting.gui"                      },
	{0x762F9EA7, "prf_crtba_hi00.ncgr.small"           },
	{0x76395E19, "spell_poison.spl"                    },
	{0x763DB821, "store5.gda"                          },
	{0x76476696, "swa_cb_dead.nsbca"                   },
	{0x766094B2, "levelup2_bs.tga"                     },
	{0x7667ED36, "a1_s4_ex_mm.nclr.small"              },
	{0x768489BA, "fx_gas.emit"                         },
	{0x768B81A2, "nw_se.gff"                           },
	{0x768DB104, "buz_ex_idle.nsbca"                   },
	{0x7696D9AD, "she_cb_victory.nsbca"                },
	{0x769D539E, "prt_ce_gre.ncgr.small"               },
	{0x769D9760, "ixb_cb_counter.nsbca"                },
	{0x76A4E6D5, "cfx_pmip1.nsbmd.small"               },
	{0x76A76B29, "bfl_cb_attack.nsbca"                 },
	{0x76A92A84, "pig_ex_walk.nsbca"                   },
	{0x76AE3517, "fx_amy_whirl.nsbmd.small"            },
	{0x76B4A122, "itm_glv_spike_2.ncgr.small"          },
	{0x76B8D221, "cfx_gfir_prj.nsbmd.small"            },
	{0x76CB7D8C, "itm_ace_psych_1.ncgr.small"          },
	{0x76DD92B5, "cmbtbg_a1_s1.nsbta.small"            },
	{0x76E149D9, "chao_hc_1.ncgr.small"                },
	{0x76E1B5B9, "chap8_cha_scr_bot.tga"               },
	{0x76E48720, "flee01_bs.tga"                       },
	{0x76E77EE6, "lvl_pow_bspnl10.ncgr.small"          },
	{0x76F02350, "icn_s_catk_ksa.ncgr.small"           },
	{0x76F1CD38, "store_5_bg.tga"                      },
	{0x76FFFD86, "tyxis_arena_gentleman1_teleport.gff" },
	{0x770192E6, "itm_act2_chaos1_2.ncgr.small"        },
	{0x77056806, "big_ex_fall.nsbca"                   },
	{0x77109C9A, "cbt_dgrnt_ms00.ncgr.small"           },
	{0x7713EFAF, "swt_ex_jump.nsbca"                   },
	{0x771CA894, "pfx_bub2_tox.emit"                   },
	{0x772B6DB0, "a1_s2_i06.walk.tga"                  },
	{0x7734EE6B, "front_slotselected.gui"              },
	{0x7747211D, "amy_cb_rund.nsbca"                   },
	{0x774ACCD0, "a1_w1_br_01.ncgr.small"              },
	{0x77656946, "invent_icon_ass.ncgr.small"          },
	{0x776AB371, "pl_ob_blkn_co.nsbmd.small"           },
	{0x77719D2B, "itm_ace_voxai_0.ncgr.small"          },
	{0x77747CB3, "chao_gc_g.nsbtx.small"               },
	{0x777C4C8F, "icn_s_catk_sk.ncgr.small"            },
	{0x777EE259, "pfx_orb.nsbtx.small"                 },
	{0x77848696, "fx_son_shock.nsbta.small"            },
	{0x77857A1F, "kro_ex_idle.nsbca"                   },
	{0x779C2D1E, "chao_fd_r.nsbtx.small"               },
	{0x77AAB1DC, "prt_sp_ome.ncgr.small"               },
	{0x77ABC02C, "c_shakn_sha.nsbca"                   },
	{0x77AF2A68, "pl_pz_gen_aa.nsbtx.small"            },
	{0x77CC547A, "invent_stat_tab1.ncgr.small"         },
	{0x77CF1705, "genknun_aa.nsbmd.small"              },
	{0x77D7E22D, "prtl_wkrrel_2.ncgr.small"            },
	{0x77E30F7A, "chao_ja_2.ncgr.small"                },
	{0x77E38D6F, "cfx_atkp1.nsbmd.small"               },
	{0x77E4D699, "spell_tough_2.spl"                   },
	{0x77EC944A, "itm_ace_fire_0.ncgr.small"           },
	{0x77F0F3DA, "lvl_pow_tspnl03.ncgr.small"          },
	{0x77F6EF1A, "spell_tough_3.spl"                   },
	{0x77FBE427, "c_sota_ta.nsbca"                     },
	{0x77FC82F9, "fx_steam.emit"                       },
	{0x77FCFA7A, "pl_pz_vox_pth.nsbmd.small"           },
	{0x77FF5CD8, "a2c6_intro_sonic.gff"                },
	{0x7806BA0A, "knu_ex_climb.nsbca"                  },
	{0x78099283, "icn_s_catk_skt.ncgr.small"           },
	{0x780E50BD, "a2c10_nocstop_shake_1.gff"           },
	{0x780F0907, "cfx_weap1.nsbtx.small"               },
	{0x7815CF00, "drnlaz_cb_attack.nsbca"              },
	{0x781B27BD, "gui_portraits08.nclr.small"          },
	{0x7820693E, "a2c10_nocstop_shake_2.gff"           },
	{0x7820B3A4, "prtl_sonicmad_2.ncgr.small"          },
	{0x78235EE3, "prt_mp_knu.ncgr.small"               },
	{0x78273F28, "a1intro.dlg"                         },
	{0x783281BF, "a2c10_nocstop_shake_3.gff"           },
	{0x78351398, "itm_ace_sturd_2.ncgr.small"          },
	{0x78395D42, "cfx_focp1.nsbtx.small"               },
	{0x783FAFE2, "defeat.nbfp"                         },
	{0x783FAFE5, "defeat.nbfs"                         },
	{0x785B252A, "lvl_lrgbt_on02.ncgr.small"           },
	{0x78632F93, "prtl_nessca_2.ncgr.small"            },
	{0x78644BA9, "charpro_bs_b01.ncgr.small"           },
	{0x7864C382, "cfx_ch6.nsbtp.small"                 },
	{0x7870A0B1, "a2_s4_i04.walk.tga"                  },
	{0x78714816, "chao_ea_3.ncgr.small"                },
	{0x78715CE4, "a1c3_shadowescape.gff"               },
	{0x7873F4F4, "exl_circe_ms12.ncgr.small"           },
	{0x787E21C1, "plt_lo_kron.plo"                     },
	{0x7884A77D, "itm_ace_tablt_2.ncgr.small"          },
	{0x78904ECB, "chao_hb_g.nsbtp.small"               },
	{0x7896B9E8, "vox_cb_dead.nsbca"                   },
	{0x789C31EA, "itm_act2_chaos7_0.ncgr.small"        },
	{0x78A72A66, "c_soknam_so.nsbca"                   },
	{0x78AD1A11, "prtl_egginq_2.ncgr.small"            },
	{0x78B06962, "spell_vulnerable_2.spl"              },
	{0x78B7FF36, "chao_gc_r.nsbtp.small"               },
	{0x78B9E6F7, "chap2scr_top.tga"                    },
	{0x78BFFBCC, "cfx_ch3.nsbmd.small"                 },
	{0x78C281E3, "spell_vulnerable_3.spl"              },
	{0x78C5E817, "icn_s_catk_wo.ncgr.small"            },
	{0x78D6A334, "itm_glv_rock_1.ncgr.small"           },
	{0x78E7DA03, "itm_ace_lghtn_2.ncgr.small"          },
	{0x78F06639, "gnp_ex_damage.nsbca"                 },
	{0x79065A46, "camerahacknocturne2.gff"             },
	{0x790E5DF0, "prtl_rougegen_0.ncgr.small"          },
	{0x790FAA26, "exp_ghz2_mm02.ncgr.small"            },
	{0x7913D216, "lower_bg_11.ncgr.small"              },
	{0x79176831, "itm_ace_power_0.ncgr.small"          },
	{0x7923B806, "spell_int_charyb_water.spl"          },
	{0x7928CAF2, "3x3panel64x64.gui"                   },
	{0x792E92EC, "itm_glv_gold_2.ncgr.small"           },
	{0x7931A33E, "plt_fpa2_omega.plo"                  },
	{0x793A2AA3, "omg_cb_blockout.nsbca"               },
	{0x7944E7A8, "drn_ex_walk.nsbca"                   },
	{0x79465FF3, "charpro_son1.ncgr.small"             },
	{0x7949F575, "itm_glv_rug_3.ncgr.small"            },
	{0x79715028, "a1c5_p2_pathblock.gff"               },
	{0x79748FD5, "demo_loop_2.gff"                     },
	{0x79782C90, "main_pnl_off04.ncgr.small"           },
	{0x797DAE7E, "chao_jg_0.ncgr.small"                },
	{0x7986A856, "demo_loop_3.gff"                     },
	{0x798A4EFD, "buz_cb_attackend.nsbca"              },
	{0x79923555, "str_box_hi00.ncgr.small"             },
	{0x799388C2, "pfx_haz1.nsbtp.small"                },
	{0x799C0F52, "voa_vob.gff"                         },
	{0x79AF5769, "cfx_rfgp1.nsbtx.small"               },
	{0x79B7C634, "chao_jq_r.nsbmd.small"               },
	{0x79B91049, "a2c10_shaderuns.gff"                 },
	{0x79DFA827, "amy_cb_blockout.nsbca"               },
	{0x79E5711B, "fx_ome_beam.nsbtx.small"             },
	{0x79EB50AD, "cfx_ertp1.nsbtp.small"               },
	{0x79EF0DEE, "fx_gnp_mflash.nsbta.small"           },
	{0x7A0B1DC0, "useitem.ncgr.small"                  },
	{0x7A0EBDC5, "pfx_sms1_sms.emit"                   },
	{0x7A1494D7, "jurnl_ts11.ncgr.small"               },
	{0x7A25E68C, "btn_back_of.ncgr.small"              },
	{0x7A25FCA7, "chao_jn_2.ncgr.small"                },
	{0x7A32FD4E, "jurnl_cod_line03.ncgr.small"         },
	{0x7A332151, "plt_a2sq_nrrgalshipment.plo"         },
	{0x7A3BF9E0, "a2c7_exitship.gff"                   },
	{0x7A52D438, "wmp_space_of01.ncgr.small"           },
	{0x7A58DFED, "fx_dummyobj.nsbtx.small"             },
	{0x7A5BB6C2, "cfx_rckp1.nsbtp.small"               },
	{0x7A5C9B4A, "prf_tboxb_ms10.ncgr.small"           },
	{0x7A5D82C3, "charpro_ome0.ncgr.small"             },
	{0x7A7E84B0, "prtl_eixthrngen_0.ncgr.small"        },
	{0x7A825291, "chao_ba_g.nsbmd.small"               },
	{0x7A833956, "itm_ace_candy_3.ncgr.small"          },
	{0x7A83D98E, "pop_lgbox_ms01.ncgr.small"           },
	{0x7A848198, "bfl_ex_walk.nsbca"                   },
	{0x7A904C70, "fx_dustpuff.ncer.small"              },
	{0x7A95A069, "plt_cod_voxai.plo"                   },
	{0x7A9FA75F, "stores.gda"                          },
	{0x7AA202EF, "hnpc_tdad_aa.nsbtx.small"            },
	{0x7ABCBD0B, "prf_tpanl_ms02.ncgr.small"           },
	{0x7ABDFFFC, "prt_se_kro.ncgr.small"               },
	{0x7AD04ACC, "pzl_zoahestablish.gff"               },
	{0x7AD0E3D0, "chao_je_g.nsbtp.small"               },
	{0x7AD84BEB, "pl_pz_ppd_dn.nsbta.small"            },
	{0x7AD8B3B7, "cfx_rtbp1.nsbmd.small"               },
	{0x7ADD32C0, "gwr_ex_off.nsbca"                    },
	{0x7AEAE374, "conv_bubble_amb00.ncgr.small"        },
	{0x7AF82BA1, "icn_l_rarwofjrl.ncgr.small"          },
	{0x7AF934EE, "pl_pz_crn_start.nsbca"               },
	{0x7AFA6ED8, "itm_glv_grip_2.ncgr.small"           },
	{0x7B0831A3, "cfx_vulp1_glw.emit"                  },
	{0x7B0DACBB, "chao_ge_2.ncgr.small"                },
	{0x7B0F317B, "prtl_fmnrel_3.ncgr.small"            },
	{0x7B0F3C6B, "itm_glv_fire_1.ncgr.small"           },
	{0x7B24C2E4, "adv_vfxtester.gda"                   },
	{0x7B28F120, "plt_ch3_shopkeeper.plo"              },
	{0x7B35FED1, "a1_s2_i07.walk.tga"                  },
	{0x7B57B45C, "itm_ace_wpnp1_2.ncgr.small"          },
	{0x7B60F649, "a1c5_p5_captainsoundalarm.gff"       },
	{0x7B677ACA, "a1c2_knucklespunch.gff"              },
	{0x7B6B1D98, "cbt_info_slide.ncgr.small"           },
	{0x7BA8B115, "charpro_sdw5.ncgr.small"             },
	{0x7BAC2749, "ch8_scylla_gentlemanpan.gff"         },
	{0x7BB520A5, "man_strbt_of03.ncgr.small"           },
	{0x7BBA3B19, "prtl_sciexh_1.ncgr.small"            },
	{0x7BC09BAB, "chao_jt_0.ncgr.small"                },
	{0x7BC36C51, "swt_ex_idle.nsbca"                   },
	{0x7BD6BBB6, "adv_tails.gda"                       },
	{0x7BDCE535, "exp_cct1_mm01.ncgr.small"            },
	{0x7BDD68C5, "plt_a2_worldmap.plo"                 },
	{0x7BEA782F, "startupscrbot.tga"                   },
	{0x7BF15E1F, "a2c8_scy_gentleman1defeated.gff"     },
	{0x7BFF253B, "fx_vox_y_ea_s.nsbta.small"           },
	{0x7C095D49, "icn_s_wsl3wcho.ncgr.small"           },
	{0x7C0F725C, "chao_ic_3.ncgr.small"                },
	{0x7C188CEA, "prt_tp_knu.ncgr.small"               },
	{0x7C1D5728, "chao_jj_r.nsbtx.small"               },
	{0x7C3A3ADE, "pfx_nicktest03.nsbtx.small"          },
	{0x7C3FE5D5, "prtl_sonicamu_2.ncgr.small"          },
	{0x7C536B60, "que_transform.nsbca"                 },
	{0x7C570D30, "prtl_sonicdet_3.ncgr.small"          },
	{0x7C5F0800, "prtl_salesmg_0.ncgr.small"           },
	{0x7C6922FB, "exl_circl_ms12.ncgr.small"           },
	{0x7C69C8DF, "str_tlgbx_ms12.ncgr.small"           },
	{0x7C723956, "prt_se_bboar.ncgr.small"             },
	{0x7C74AABC, "itm_ace_drink_0.ncgr.small"          },
	{0x7C7B31D2, "a2_s4_i05.walk.tga"                  },
	{0x7C81B2F9, "exp_cct1_mm23.ncgr.small"            },
	{0x7C837B75, "amb_convo_bg.nbfp"                   },
	{0x7C837B78, "amb_convo_bg.nbfs"                   },
	{0x7C8CF09B, "fx_smokepuff.nsbmd.small"            },
	{0x7CB1373E, "item200.itm"                         },
	{0x7CB26ABE, "cbt_usfat_00.ncgr.small"             },
	{0x7CB2FC97, "pl_pz_vox_lb.nsbtx.small"            },
	{0x7CB48F94, "itm_fotsnklt_3.ncgr.small"           },
	{0x7CB70ADD, "prtl_hanmad_0.ncgr.small"            },
	{0x7CC34FBF, "item201.itm"                         },
	{0x7CC7ED3E, "ixb_cb_idle.nsbca"                   },
	{0x7CD0329B, "itm_ace_jammr_0.ncgr.small"          },
	{0x7CD09FD2, "movement04_bs.tga"                   },
	{0x7CD56840, "item202.itm"                         },
	{0x7CD9673B, "swt_cb_counter.nsbca"                },
	{0x7CE213D1, "lvl_stat_but_auto02.ncgr.small"      },
	{0x7CE780C1, "item203.itm"                         },
	{0x7CE826A9, "jrl_tsmbr_of11.ncgr.small"           },
	{0x7CEAA9EE, "chapter5warp2.gff"                   },
	{0x7CEDC74A, "exp_ghz1_fw31.ncgr.small"            },
	{0x7CF57116, "jrl_tmbox_ms20.ncgr.small"           },
	{0x7CF99942, "item204.itm"                         },
	{0x7CFCC26F, "chapter5warp3.gff"                   },
	{0x7D063233, "cfx_icep1.nsbtp.small"               },
	{0x7D0BB1C3, "item205.itm"                         },
	{0x7D0EDAF0, "chapter5warp4.gff"                   },
	{0x7D1D441F, "fx_bird.nsbtp.small"                 },
	{0x7D1DCA44, "item206.itm"                         },
	{0x7D20F371, "chapter5warp5.gff"                   },
	{0x7D295403, "fx_a2_s4_ex_env.nsbtx.small"         },
	{0x7D2D81AA, "fullscreenimagepanel64x64.gui"       },
	{0x7D2FE2C5, "item207.itm"                         },
	{0x7D32E836, "cbt_lredt_ms00.ncgr.small"           },
	{0x7D330BF2, "chapter5warp6.gff"                   },
	{0x7D3363F6, "icn_l_ondash1.ncgr.small"            },
	{0x7D36749A, "pl_pz_mem_trd.nsbmd.small"           },
	{0x7D378973, "journalbackground.gui"               },
	{0x7D41FB46, "item208.itm"                         },
	{0x7D452473, "chapter5warp7.gff"                   },
	{0x7D4BAD38, "hud_lvlnum6.ncgr.small"              },
	{0x7D52F1B9, "fx_vox_g_ea_l.nsbtx.small"           },
	{0x7D5413C7, "item209.itm"                         },
	{0x7D55893F, "pfx_arnietest01.emit"                },
	{0x7D573CF4, "chapter5warp8.gff"                   },
	{0x7D5C18C0, "prtl_eixfm1_2.ncgr.small"            },
	{0x7D695575, "chapter5warp9.gff"                   },
	{0x7D6C6952, "itm_ace_rejuv_2.ncgr.small"          },
	{0x7D9BF35E, "itm_ace_leaf_2.ncgr.small"           },
	{0x7D9E237F, "prtl_knudet_0.ncgr.small"            },
	{0x7DA48066, "s_n.gff"                             },
	{0x7DA739EA, "prtl_knusmg_1.ncgr.small"            },
	{0x7DB19749, "prtl_krwcon_1.ncgr.small"            },
	{0x7DBAB08D, "ehd_ylbar_ms00.ncgr.small"           },
	{0x7DC4E381, "drn_cb_counter.nsbca"                },
	{0x7DC514CF, "cfx_fgrp2.nsbmd.small"               },
	{0x7DCF5848, "itm_glvspncl_3.ncgr.small"           },
	{0x7DD0F78B, "pfx_smk2.nsbtp.small"                },
	{0x7DE61387, "itm_ace_noctn_1.ncgr.small"          },
	{0x7DF01616, "icn_l_catk_ka.ncgr.small"            },
	{0x7DF4AC63, "prtl_skpcon_1.ncgr.small"            },
	{0x7DF5B720, "itm_fotsnkkb_2.ncgr.small"           },
	{0x7DFAB58D, "plt_scylla_or_charyb.plo"            },
	{0x7DFF3575, "egm_cb_damage.nsbca"                 },
	{0x7E183019, "arena_act_1.are"                     },
	{0x7E2365A9, "spell_lucky.spl"                     },
	{0x7E27601C, "shadball.nsbtp.small"                },
	{0x7E28AF70, "pfx_smd1.nsbmd.small"                },
	{0x7E2A489A, "arena_act_2.are"                     },
	{0x7E2C2851, "lvl_tsstr_ms10.ncgr.small"           },
	{0x7E2EBC49, "pow_base.nbfp"                       },
	{0x7E2EBC4C, "pow_base.nbfs"                       },
	{0x7E3254B0, "loc_round_es0.ncgr.small"            },
	{0x7E3A81C5, "pl_am_tyx_gfr.nsbca"                 },
	{0x7E3C4596, "cfx_acdp1_slm.emit"                  },
	{0x7E3FC51F, "brz_hintguy.dlg"                     },
	{0x7E410FB7, "hnpc_kid_idle.nsbca"                 },
	{0x7E49D7B8, "lvl_stat_smpnl11.ncgr.small"         },
	{0x7E4D9D64, "jrl_tlgbr_on02.ncgr.small"           },
	{0x7E5BEC63, "exp_wel1_mm10.ncgr.small"            },
	{0x7E5ECC9F, "cho_tlgbx_ms11.ncgr.small"           },
	{0x7E6BD461, "a2c10_ix_final_battle.gff"           },
	{0x7E6CDC4B, "prtl_factwork.nclr.small"            },
	{0x7EA1C78F, "icn_s_select.ncgr.small"             },
	{0x7EA9B75B, "pfx_gus1.nsbmd.small"                },
	{0x7EAAA8D1, "a1sq_factoryworker.dlg"              },
	{0x7EAF42A4, "prtl_krwgen_2.ncgr.small"            },
	{0x7EBBCD3B, "a1c5_p9_sonicemerges.gff"            },
	{0x7EBC7663, "jurnl_bs00.ncgr.small"               },
	{0x7EC88D87, "realtimetutthree.gda"                },
	{0x7ECD2F29, "cfx_ccap1.nsbtp.small"               },
	{0x7ED7FD21, "fx_gas.nsbtx.small"                  },
	{0x7EDB9625, "gensha_aa.nsbmd.small"               },
	{0x7EDFDA06, "waterslides.gda"                     },
	{0x7EE29B1B, "chao_jp_g.nsbtp.small"               },
	{0x7EEFC414, "pig_ex_fall.nsbca"                   },
	{0x7EF257BE, "prtl_skpgen_2.ncgr.small"            },
	{0x7EF58BE8, "exl_jurnl_ms2.ncgr.small"            },
	{0x7EFEA624, "exl_globe_ms0.ncgr.small"            },
	{0x7F00ECE4, "c_soknta_kn.nsbca"                   },
	{0x7F065FDF, "item210.itm"                         },
	{0x7F0FF561, "pl_pz_rc_crs.nsbtp.small"            },
	{0x7F158E6E, "a1_w3_gh_11.ncgr.small"              },
	{0x7F16CE17, "plt_fpa2_rouge.plo"                  },
	{0x7F187860, "item211.itm"                         },
	{0x7F2A90E1, "item212.itm"                         },
	{0x7F2DC750, "gup_aa_shin.nsbta.small"             },
	{0x7F3B513A, "rax_aa.nsbmd.small"                  },
	{0x7F3CA962, "item213.itm"                         },
	{0x7F40DBE0, "prtl_bighap_1.ncgr.small"            },
	{0x7F4BDC07, "mut_flash_ms10.ncgr.small"           },
	{0x7F4EC1E3, "item214.itm"                         },
	{0x7F506E8B, "prtl_tdadgen_0.ncgr.small"           },
	{0x7F51039F, "conv_bubble_tab04.ncgr.small"        },
	{0x7F55DF6B, "gwr_cb_idle.nsbca"                   },
	{0x7F60DA64, "item215.itm"                         },
	{0x7F72EF5A, "chao_jb_g.nsbmd.small"               },
	{0x7F72F2E5, "item216.itm"                         },
	{0x7F7AE74C, "prtl_pilmad_1.ncgr.small"            },
	{0x7F838458, "a1c1_rougeblock.gff"                 },
	{0x7F84F923, "nrg_cb_damage.nsbca"                 },
	{0x7F850B66, "item217.itm"                         },
	{0x7F8D35B0, "cbt_cmdupoff.ncgr.small"             },
	{0x7F901AA4, "bal_ex_dash.nsbca"                   },
	{0x7F90B40B, "ix_ex_run.nsbca"                     },
	{0x7F9723E7, "item218.itm"                         },
	{0x7F9A9FC5, "chao_ic_r.nsbmd.small"               },
	{0x7FA93C68, "item219.itm"                         },
	{0x7FAAB1E0, "pfx_arnietest02.emit"                },
	{0x7FBF95DD, "fx_els_en_aa.nsbtx.small"            },
	{0x7FD35F96, "itm_ace_antid_2.ncgr.small"          },
	{0x7FD4D7E3, "chao_aa_g.nsbtp.small"               },
	{0x7FD733C7, "a2_s5_ex_la_soundmap.gda"            },
	{0x7FEB2F44, "bsc_cb_attackend.nsbca"              },
	{0x7FF0A00C, "btn_help_on.ncgr.small"              },
	{0x800891E5, "arm_cb_pattack.nsbca"                },
	{0x800D0863, "pl_pz_mem_lts.nsbmd.small"           },
	{0x800E13DB, "cbtmenubg_01.ncgr.small"             },
	{0x801A1D55, "pfx_spk1.nsbtx.small"                },
	{0x801B1E84, "ghq1_minimap.nclr.small"             },
	{0x80256BF5, "cfx_spcp1.nsbmd.small"               },
	{0x802BEF47, "invent_icon_sta.ncgr.small"          },
	{0x802E29E9, "main_pnl_big15.ncgr.small"           },
	{0x80575832, "charpro_knu2.ncgr.small"             },
	{0x805BCD97, "a2c10_emeraldfreed_sonic.gff"        },
	{0x805C563C, "lower_bg_08.ncgr.small"              },
	{0x806EDCFB, "cfx_ftpp1_pls.emit"                  },
	{0x807463A8, "chao_ab_gard.nsbmd.small"            },
	{0x8079F6A6, "prtl_fwrrel_0.ncgr.small"            },
	{0x8085C2F3, "a2_s4_i06.walk.tga"                  },
	{0x808BE192, "nrg_ab.nsbtx.small"                  },
	{0x809687D2, "rou_cb_damage.nsbca"                 },
	{0x8098363B, "upper_bg_04.ncgr.small"              },
	{0x809BC4B0, "cfx_spt_slm.nsbmd.small"             },
	{0x80AEC19D, "gennpc1b_aa.nsbmd.small"             },
	{0x80B13980, "fx_steam.nsbtx.small"                },
	{0x80C85540, "prtl_wkrsca_1.ncgr.small"            },
	{0x80CFD6EB, "cbt_menu_resetbutton_on.ncgr.small"  },
	{0x80E04A1F, "ecm_aa.nsbtx.small"                  },
	{0x80E1B39B, "a1c3_sentryattack1.gff"              },
	{0x80E2ADDE, "pfx_ppr1.nsbmd.small"                },
	{0x80E4C3E8, "itm_ace_kevlr_0.ncgr.small"          },
	{0x80EEE6FE, "a2_s1_ex.walk.tga"                   },
	{0x80F21D4D, "pl_pu_emrld1.nsbtx.small"            },
	{0x80F3CC1C, "a1c3_sentryattack2.gff"              },
	{0x80F49742, "prt_se_brdrh.ncgr.small"             },
	{0x80F518B9, "man_tsvst_of11.ncgr.small"           },
	{0x80FAA0D9, "fx_a1_s3_ex_env.nsbtp.small"         },
	{0x810B496F, "prtl_amymad_2.ncgr.small"            },
	{0x810FCC37, "cmbtbg_a2_s6.nsbmd.small"            },
	{0x811AE3A3, "fx_a1_s5_ex_env.nsbtx.small"         },
	{0x81285072, "pl_ob_bdr_aa.nsbmd.small"            },
	{0x8128EF07, "mut_circl_ms11.ncgr.small"           },
	{0x81294D0E, "fx_gust.nsbmd.small"                 },
	{0x812E2619, "chao_hd_0.ncgr.small"                },
	{0x8137DF1C, "she_cb_damage.nsbca"                 },
	{0x814AFEE6, "cbt_fatig_00.ncgr.small"             },
	{0x814E6F26, "itm_act2_chaos2_1.ncgr.small"        },
	{0x8157D083, "prt_ce_blu.ncgr.small"               },
	{0x815B6583, "prtl_shdsht_1.ncgr.small"            },
	{0x815F2D3E, "fx_leaf.nsbtx.small"                 },
	{0x816E2CCA, "mut_circl_ms21b.ncgr.small"          },
	{0x8171DE22, "icn_s_catk_ob.ncgr.small"            },
	{0x8179C386, "a1c5_p9_gizoiddefeat.gff"            },
	{0x817AC479, "compoundevents.gda"                  },
	{0x817D24D9, "a2c8_scy_sonapproachemerald.gff"     },
	{0x818559EB, "prtl_credet_0.ncgr.small"            },
	{0x818F3D11, "itm_accnewsp_1.ncgr.small"           },
	{0x8190BDFA, "chao_ea_r.nsbtx.small"               },
	{0x81927DCB, "pl_pz_rc_sw.nsbtx.small"             },
	{0x8192BE03, "a1_s2_i03_mm.nclr.small"             },
	{0x8197749F, "a1_w2_mt_11.ncgr.small"              },
	{0x819858D9, "cfx_revp1_glw.emit"                  },
	{0x81A764C7, "jurnl_cod03.ncgr.small"              },
	{0x81B02EDC, "pl_mv_boa_pd.nsbca"                  },
	{0x81B08075, "arm_cb_attack.nsbca"                 },
	{0x81B4BF10, "cfx_gwtr_prj.nsbtp.small"            },
	{0x81B7969C, "lvl_tdbld_ms11.ncgr.small"           },
	{0x81C94B5E, "exp_topframe0.ncgr.small"            },
	{0x81CAEC8E, "screenwipewhite.gui"                 },
	{0x81E961EF, "ixb_cb_kd.nsbca"                     },
	{0x81EA89C2, "invent_ts_panel.nbfp"                },
	{0x81EA89C5, "invent_ts_panel.nbfs"                },
	{0x81FAFB95, "itm_ace_tools_0.ncgr.small"          },
	{0x81FFDA81, "pfx_arnietest03.emit"                },
	{0x82038834, "mut_sonic_ms00.ncgr.small"           },
	{0x82047FF5, "prtl_krggen_3.ncgr.small"            },
	{0x820F3728, "fx_tyx_beam.nsbta.small"             },
	{0x82155828, "a2c10_knucklesdisappears.gff"        },
	{0x822071FF, "plt_fp_tails.plo"                    },
	{0x82225D35, "cfx_sson_flare.nsbtx.small"          },
	{0x8228440F, "gui_explorec.nclr.small"             },
	{0x822FEBBA, "chao_jb_1.ncgr.small"                },
	{0x8243E7BD, "spell_focussed.spl"                  },
	{0x82471F96, "rax_cb_dead.nsbca"                   },
	{0x824DFF1A, "lvl_but_next.ncgr.small"             },
	{0x824E9E0F, "pro_sidbx_11.ncgr.small"             },
	{0x82517C6A, "mrz_hintguy.dlg"                     },
	{0x825228C1, "itm_ace_kron_3.ncgr.small"           },
	{0x825AE9DD, "a2_w0_nt_00.ncgr.small"              },
	{0x8269F688, "pl_mv_bop_aa.nsbtx.small"            },
	{0x826CDF7D, "a2_s1_ex.are"                        },
	{0x82714531, "a1c1_scoutattack.gff"                },
	{0x8272927C, "pig_cb_damage.nsbca"                 },
	{0x8279BD62, "omg_cb_blockin.nsbca"                },
	{0x827E2F9B, "itm_fotswtbt_2.ncgr.small"           },
	{0x828445F4, "prtl_csxmad_0.ncgr.small"            },
	{0x82850AE4, "pfx_glo1_blu.emit"                   },
	{0x82874095, "cfx_egbt_mf.nsbtp.small"             },
	{0x828783E2, "a1_s2_i05_ts.nbfp"                   },
	{0x828783E5, "a1_s2_i05_ts.nbfs"                   },
	{0x82B127E9, "charpro_bs_c00.ncgr.small"           },
	{0x82B2F19D, "pro_smbox_10.ncgr.small"             },
	{0x82BD200B, "icn_s_patk_sha1.ncgr.small"          },
	{0x82BF8059, "exl_sonic_ms2.ncgr.small"            },
	{0x82D37CC2, "icn_l_utdash1.ncgr.small"            },
	{0x82D816D5, "prtl_timsca_0.ncgr.small"            },
	{0x82D839E3, "chao_ji_3.ncgr.small"                },
	{0x82E984E9, "gzc_cb_pattack.nsbca"                },
	{0x82F63DE5, "cbt_lyelt_ms00.ncgr.small"           },
	{0x82F6BEF1, "a1c3_eggman_warp_workshop.gff"       },
	{0x82FC1F6C, "tutorial.gui"                        },
	{0x8312B896, "cmbtbg_a2_s1.nsbta.small"            },
	{0x8322D4C2, "charpro_tai2.ncgr.small"             },
	{0x834E2EDC, "krn_hintguy.dlg"                     },
	{0x835A54D8, "gui_portraits03.nclr.small"          },
	{0x835D5BB3, "lvl_tlgbx_ms11.ncgr.small"           },
	{0x8364B6FB, "a1c5_p3_hijack.gff"                  },
	{0x837E4AD5, "prtl_ncsgen_3.ncgr.small"            },
	{0x8384A6A5, "chao_jm_g.nsbmd.small"               },
	{0x838D8A87, "cfx_snry_mf.nsbtx.small"             },
	{0x838F38A0, "sco_cb_idle.nsbca"                   },
	{0x8393CC77, "a1c3_shadow.dlg"                     },
	{0x839A3EBF, "itm_ace_goldr_3.ncgr.small"          },
	{0x839DBE3C, "guianim_wmp.gda"                     },
	{0x83A446BB, "plt_a2sq_pirates.plo"                },
	{0x83A9A294, "chao_hc_g.nsbtx.small"               },
	{0x83ACBD91, "she_stb.nsbtx.small"                 },
	{0x83AFB5CD, "charpro_amy4.ncgr.small"             },
	{0x83B8AA60, "lvl_stat_rb.ncgr.small"              },
	{0x83C8C367, "a1_s6_ex_la_soundmap.gda"            },
	{0x83CC9B11, "a1_w2_br_10.ncgr.small"              },
	{0x83D152FF, "chao_gd_r.nsbtx.small"               },
	{0x83D4EEB9, "lvl_pow_tspnl20.ncgr.small"          },
	{0x83D6767C, "cbt_menu_info.ncgr.small"            },
	{0x83DCCE39, "pl_pz_mem_box.nsbmd.small"           },
	{0x83DFCAAA, "a1_w0_gh_10.ncgr.small"              },
	{0x83E45E85, "prtl_nestor.nclr.small"              },
	{0x83F975F9, "ybutton.ncgr.small"                  },
	{0x83FBB6EE, "icn_l_utfly2.ncgr.small"             },
	{0x84085AEB, "prtl_thbovr_2.ncgr.small"            },
	{0x841EEB8F, "icn_s_catk_ks.ncgr.small"            },
	{0x842536C7, "sbg.are"                             },
	{0x84272F0D, "shp_cb_getup.nsbca"                  },
	{0x842B3B0E, "prtl_thbhap_3.ncgr.small"            },
	{0x842E0B67, "a1c3_shopkeeper.dlg"                 },
	{0x8439B205, "cfx_pwup1_lgt.emit"                  },
	{0x843C672D, "drnsui_cb_off.nsbca"                 },
	{0x844F0D63, "exp_ghz1_mm00.ncgr.small"            },
	{0x84516671, "icn_s_catk_kat.ncgr.small"           },
	{0x846A6E73, "a1_w3_br_02.ncgr.small"              },
	{0x8472D8E7, "chao_jo_1.ncgr.small"                },
	{0x8478F64E, "omg_cb_dead.nsbca"                   },
	{0x847B8FA3, "fx_tal_sbot.nsbmd.small"             },
	{0x847D9E0C, "a1_w1_gh_02.ncgr.small"              },
	{0x848506E7, "guianim_com.gda"                     },
	{0x84905414, "a2_s4_i07.walk.tga"                  },
	{0x84971F68, "guianim_con.gda"                     },
	{0x84B124FE, "big_cb_blockd.nsbca"                 },
	{0x84B49757, "prtl_sonicimp_1.ncgr.small"          },
	{0x84B8F967, "cho_ts_13.ncgr.small"                },
	{0x84BBB53A, "big_ex_twitch.nsbca"                 },
	{0x84BC003A, "str_tmdbx_ms00.ncgr.small"           },
	{0x84C0D7C4, "cho_lgbtn_hi00.ncgr.small"           },
	{0x84C2BFC8, "itm_accrfrsh_0.ncgr.small"           },
	{0x84C4024E, "icn_s_inv2.ncgr.small"               },
	{0x84C574AC, "chao_ib_g.nsbtp.small"               },
	{0x84C90264, "a1c4_shadow_turn_around.gff"         },
	{0x84D13B1F, "itm_ace_grapp_1.ncgr.small"          },
	{0x84E50A8E, "que_trfm_2.nsbca"                    },
	{0x84ED2517, "chao_hc_r.nsbtp.small"               },
	{0x84ED3604, "statusframe_r.ncgr.small"            },
	{0x84F3DB27, "exp_ghz1_mm22.ncgr.small"            },
	{0x84F9AB83, "pfx_lef1.nsbmd.small"                },
	{0x84FE7AEF, "icn_s_patkson.ncgr.small"            },
	{0x85021A36, "fx_a2_s7_i03.nsbta.small"            },
	{0x8515B751, "prtl_tailscon_0.ncgr.small"          },
	{0x85186478, "prt_tp_m_son.ncgr.small"             },
	{0x85216563, "a1c5_p7_10_scout.gff"                },
	{0x8530FF0E, "charpro_stat00.ncgr.small"           },
	{0x8536FCBC, "icn_s_redxcho.ncgr.small"            },
	{0x853BBF79, "cfx_acdp1_slp.emit"                  },
	{0x853D176D, "cw_sw_nw.gff"                        },
	{0x8541A2FB, "prtl_amydet_3.ncgr.small"            },
	{0x85468D31, "main_pnl_on03.ncgr.small"            },
	{0x85500E9C, "cfx_revp1.nsbmd.small"               },
	{0x8554AEC9, "str_tsmbx_ms00.ncgr.small"           },
	{0x8557E154, "fx_drn_lzr_muz.nsbtp.small"          },
	{0x855B95F0, "prtl_guncodis_0.ncgr.small"          },
	{0x856180FF, "knu_ex_smash.nsbca"                  },
	{0x8571A8C3, "prtl_nesinq_1.ncgr.small"            },
	{0x8580C13A, "itm_ace_medkt_1.ncgr.small"          },
	{0x85A2F672, "itm_accearng_2.ncgr.small"           },
	{0x85A4909C, "itm_ace_wpnp2_1.ncgr.small"          },
	{0x85B00EAF, "cfx_ch4_spkp1.emit"                  },
	{0x85B5B14D, "chao_aa_turn.nsbtx.small"            },
	{0x85C34FD6, "charpro_egg1.ncgr.small"             },
	{0x85C7E530, "prtl_ffmcon_3.ncgr.small"            },
	{0x85DB1D75, "exp_ringfrm0.ncgr.small"             },
	{0x85E29677, "shp_cb_kd.nsbca"                     },
	{0x85E616DC, "son_cb_kd.nsbca"                     },
	{0x85EA3799, "chao_jf_g.nsbtx.small"               },
	{0x85F306A2, "pl_pz_wod_chp.nsbca"                 },
	{0x85F97C8D, "str_tsmbx_ms22.ncgr.small"           },
	{0x85F9ADFC, "prtl_krngen_3.ncgr.small"            },
	{0x86034793, "pl_pz_ppd_dn.nsbca"                  },
	{0x8603955F, "big_ex_jumpair.nsbca"                },
	{0x86051AD1, "gnp_cb_attackend.nsbca"              },
	{0x86061FBE, "big_cb_attackend.nsbca"              },
	{0x8609BADD, "str_buttn_of00.ncgr.small"           },
	{0x861362AC, "prtl_tailsgen_1.ncgr.small"          },
	{0x8616107D, "cfx_ixmp1.nsbmd.small"               },
	{0x86281B3D, "a1_s3_ex_door1.gff"                  },
	{0x8630EECA, "itm_ace_clovej_3.ncgr.small"         },
	{0x863A33BE, "a1_s3_ex_door2.gff"                  },
	{0x8646BE65, "icn_s_patk_big5.ncgr.small"          },
	{0x8659A50C, "itm_glvttncl_2.ncgr.small"           },
	{0x865C4E9C, "chao_id_2.ncgr.small"                },
	{0x866A1133, "pl_am_noc_gcsn.nsbtp.small"          },
	{0x866C03E5, "pro_medbx_02.ncgr.small"             },
	{0x86707B85, "charpro_knu.nclr.small"              },
	{0x8671BBD0, "itm_ace_gum_3.ncgr.small"            },
	{0x867422BD, "prtl_knusad_3.ncgr.small"            },
	{0x8675B217, "ch8_scylla_approachemerald.gff"      },
	{0x867A4D18, "realtime_tutorial_smash_bg.tga"      },
	{0x8682B742, "zoh_hintguy.dlg"                     },
	{0x868A1C39, "cfx_ftpp1.nsbmd.small"               },
	{0x868DB3C6, "cfx_shgp1_shg.emit"                  },
	{0x86947855, "a1_w1_br.nclr.small"                 },
	{0x869D7323, "upper_textbox_07.ncgr.small"         },
	{0x86A2BD87, "pfx_smd1_smd.emit"                   },
	{0x86A6944F, "a1_w0_gh.nclr.small"                 },
	{0x86AAABB7, "bif_knu.nbfp"                        },
	{0x86AAABBA, "bif_knu.nbfs"                        },
	{0x86B3E17E, "btn_l_rtarw_on.ncgr.small"           },
	{0x86B77872, "chao_ca_g.nsbmd.small"               },
	{0x86B8C11B, "pl_pz_vox_swc.nsbmd.small"           },
	{0x86BD4CDF, "icn_l_barwonlvl.ncgr.small"          },
	{0x86BD89D3, "cmbtbg_a1_s3.nsbmd.small"            },
	{0x86CAEA48, "gui_combatport.nclr.small"           },
	{0x86D5D9B6, "a2c8_tyxisdoneboardship.gff"         },
	{0x86DA37BD, "chap9_scy_scr_bot.tga"               },
	{0x86E7DE30, "charpro_sdw0.ncgr.small"             },
	{0x86EC7F07, "plt_fpa2_big.plo"                    },
	{0x86F1FC6B, "exl_menui_ms11.ncgr.small"           },
	{0x8704DB82, "a1_s1_ex_la_soundmap.gda"            },
	{0x8705FF1D, "prtl_rougesmg_0.ncgr.small"          },
	{0x87064B13, "tai_cb_attack.nsbca"                 },
	{0x870B5D95, "itm_glv_ice_0.ncgr.small"            },
	{0x870DA378, "pl_ob_blkb_sr.nsbmd.small"           },
	{0x87117017, "cbt_elem_an_fire.nanr.small"         },
	{0x8723155F, "itm_accspkam_0.ncgr.small"           },
	{0x872770D6, "prtl_outhos_0.ncgr.small"            },
	{0x8728AA69, "spell_steal_ring.spl"                },
	{0x8729267A, "prt_se_nrrwa.ncgr.small"             },
	{0x8729B764, "cfx_icep1_ice.emit"                  },
	{0x872DBA1C, "chao_jf_r.nsbtp.small"               },
	{0x8730265F, "swa_ex_damage.nsbca"                 },
	{0x873350CE, "conv_bubble_ts12.ncgr.small"         },
	{0x8744C8E4, "itm_ace_echid_3.ncgr.small"          },
	{0x874649B1, "wmp_s_radar.ncer.small"              },
	{0x874E7970, "itm_accwnrng_1.ncgr.small"           },
	{0x8757CF5B, "prtl_vxcliggen_0.ncgr.small"         },
	{0x87590DC6, "plt_cod_marauders.plo"               },
	{0x87591822, "nsw_aa.nsbtx.small"                  },
	{0x8760107E, "cfx_cgrp1.nsbtp.small"               },
	{0x876E94FC, "exp_ghq1_mm02.ncgr.small"            },
	{0x877C42A4, "pl_pz_ppv_aa.nsbtx.small"            },
	{0x877E5AB3, "armboar_aa.nsbmd.small"              },
	{0x877F70BB, "cbt_cmdframe00.ncgr.small"           },
	{0x8780138B, "egb_cb_counter.nsbca"                },
	{0x8780D748, "itm_fotrunkb_1.ncgr.small"           },
	{0x87987A4F, "icn_l_automatic.ncgr.small"          },
	{0x87ABBE63, "a1_s5_i03_ts.nbfp"                   },
	{0x87ABBE66, "a1_s5_i03_ts.nbfs"                   },
	{0x87BF3B8B, "amy_ex_walk.nsbca"                   },
	{0x87C05C45, "exl_smbtn_of10.ncgr.small"           },
	{0x87C22030, "squads.gda"                          },
	{0x87C51BFB, "a1_s4_ex_slide1.gff"                 },
	{0x87D43932, "ssf_tlgbx_ms30.ncgr.small"           },
	{0x87D7347C, "a1_s4_ex_slide2.gff"                 },
	{0x87D820F9, "spell_distracted_2.spl"              },
	{0x87DE51D1, "a1c1_scout_sonic_backs_off.gff"      },
	{0x87E94CFD, "a1_s4_ex_slide3.gff"                 },
	{0x87EA397A, "spell_distracted_3.spl"              },
	{0x87EC43AE, "prtl_vecgen_2.ncgr.small"            },
	{0x87FB657E, "a1_s4_ex_slide4.gff"                 },
	{0x8806663A, "itm_glv_metal_3.ncgr.small"          },
	{0x880D7DFF, "a1_s4_ex_slide5.gff"                 },
	{0x881F9680, "a1_s4_ex_slide6.gff"                 },
	{0x88223D06, "prt_se_robar.ncgr.small"             },
	{0x88270088, "itm_glv_dark_2.ncgr.small"           },
	{0x882C9208, "cfx_pmip1.nsbtp.small"               },
	{0x8831AF01, "a1_s4_ex_slide7.gff"                 },
	{0x8833E1AF, "fx_a2_s4_ex_ff.nsbta.small"          },
	{0x88407D54, "cfx_gfir_prj.nsbtp.small"            },
	{0x8843C782, "a1_s4_ex_slide8.gff"                 },
	{0x8851426A, "pl_pz_rc_cd1.nsbca"                  },
	{0x8851CB15, "a1_s3_ex_mm.nclr.small"              },
	{0x88585824, "prtl_shdstg_3.ncgr.small"            },
	{0x8859314D, "shp_cb_blockexit.nsbca"              },
	{0x8862C293, "cbt_menu_arrowdown.ncgr.small"       },
	{0x8873AD18, "cfx_dstp1_dst.emit"                  },
	{0x8873B90F, "fx_a2_s7_env.nsbmd.small"            },
	{0x8876CC25, "a2c10_tauntedbygizoids.gff"          },
	{0x8882613B, "str_helpb_of00.ncgr.small"           },
	{0x888ADA53, "hud_lvlnum1.ncgr.small"              },
	{0x888C8D0E, "defeattip.gui"                       },
	{0x888CC209, "conv_icon07.ncgr.small"              },
	{0x88914908, "bel_cb_attack.nsbca"                 },
	{0x88965029, "jurnl_mis_linelit02.ncgr.small"      },
	{0x889A566C, "a1_s4_i01.are"                       },
	{0x889AE535, "a2_s4_i08.walk.tga"                  },
	{0x889CB276, "itm_fotrunrg_3.ncgr.small"           },
	{0x88ACAE70, "pzl_cranerotend.gff"                 },
	{0x88BFB908, "icn_s_larwoflvl.ncgr.small"          },
	{0x88C3D5A4, "portrait.tga"                        },
	{0x88C835DC, "pl_am_noc_gcsf.nsbta.small"          },
	{0x88CA8A3B, "a2_s2_card_b1.nbfp"                  },
	{0x88CA8A3E, "a2_s2_card_b1.nbfs"                  },
	{0x88DE9DB4, "chao_fa_0.ncgr.small"                },
	{0x88F1A68F, "a1sq_timmy.dlg"                      },
	{0x88F25328, "cfx_rbep1.nsbmd.small"               },
	{0x88F70395, "shp_aa.nsbtx.small"                  },
	{0x890C844C, "bsl_cb_dead.nsbca"                   },
	{0x890CC027, "conversationreplies.gui"             },
	{0x8912CDCE, "gzc_ex_run.nsbca"                    },
	{0x8916FAB4, "a2cha_overmind_sonic1.gff"           },
	{0x8923F860, "fx_vox_b_ea_s.nsbmd.small"           },
	{0x89291335, "a2cha_overmind_sonic2.gff"           },
	{0x8934AAAF, "a1_w0_br_01.ncgr.small"              },
	{0x893EBCFA, "cfx_crvp1.nsbmd.small"               },
	{0x8954D1AF, "she_aa.nsbmd.small"                  },
	{0x89567691, "prf_smarw_on00.ncgr.small"           },
	{0x895E4B14, "itm_ace_fragm_3.ncgr.small"          },
	{0x8960ADEB, "fx_impactflash.ncer.small"           },
	{0x896B38A2, "cfx_atkp1.nsbtp.small"               },
	{0x896CD650, "chao_aa_1.ncgr.small"                },
	{0x896F3027, "lvl_pow_bspnl02.ncgr.small"          },
	{0x8972BF29, "vox_cb_kd.nsbca"                     },
	{0x8978AE79, "str_tsmbr_of01.ncgr.small"           },
	{0x8984A5AD, "pl_pz_vox_pth.nsbtp.small"           },
	{0x89A479A5, "swtsbot_aa.nsbmd.small"              },
	{0x89A8F9C1, "prtl_biggen_1.ncgr.small"            },
	{0x89AAE188, "a2_w0_tx_01.ncgr.small"              },
	{0x89AE727D, "cmbtbg_a2_s7_i02.nsbtx.small"        },
	{0x89B569D8, "gzd_ex_idle.nsbca"                   },
	{0x89BF0F68, "a1_w3_ai_01.ncgr.small"              },
	{0x89CE2DA6, "swa_cb_idle.nsbca"                   },
	{0x89D29FA3, "icn_s_catk_stk.ncgr.small"           },
	{0x89D4DE4D, "cfx_bugp1.nsbmd.small"               },
	{0x89D52A8C, "a1c1_tails.dlg"                      },
	{0x89F0B4BF, "omg_cb_damage.nsbca"                 },
	{0x89F55C5B, "a2sq_nestor.dlg"                     },
	{0x89FBEEE4, "chao_jq_g.nsbtx.small"               },
	{0x8A0D918B, "dft_scr_12.ncgr.small"               },
	{0x8A13A2D0, "pl_pu_rng_aa.nsbtx.small"            },
	{0x8A1EBFD1, "levelup1_bs.tga"                     },
	{0x8A25E727, "a1_s5_ex_mm.nbfp"                    },
	{0x8A25E72A, "a1_s5_ex_mm.nbfs"                    },
	{0x8A297C40, "a1_s1_i01_mm.nclr.small"             },
	{0x8A384B15, "prtl_boysca_0.ncgr.small"            },
	{0x8A40618C, "fx_vox_y_ea_m.nsbtx.small"           },
	{0x8A47A6FF, "cfx_ch3.nsbtp.small"                 },
	{0x8A55A390, "a1c4_amytrysmash.gff"                },
	{0x8A5B5929, "prtl_cresad_3.ncgr.small"            },
	{0x8A644032, "cmbtbg_ts_ghz100.ncgr.small"         },
	{0x8A6C129E, "itm_fotrunkv_3.ncgr.small"           },
	{0x8A7D38E3, "cho_lgbtn_di03.ncgr.small"           },
	{0x8A8BF7CC, "prtl_telmad_1.ncgr.small"            },
	{0x8AAFF857, "store_4_bg.tga"                      },
	{0x8AB777F8, "pl_pz_brd_1dn.nsbca"                 },
	{0x8AC05996, "min_circl_ms30.ncgr.small"           },
	{0x8AD15E6D, "itm_fotrnfhy_1.ncgr.small"           },
	{0x8AD42720, "cbt_menu_arrowup.ncgr.small"         },
	{0x8ADC947F, "storedatabackground.gui"             },
	{0x8AF75150, "pl_pu_scr_bn.nsbca"                  },
	{0x8B01D921, "gui_combatloc.nclr.small"            },
	{0x8B04F1A9, "fx_shason_nuke.nsbmd.small"          },
	{0x8B090DF6, "cmbtbg_ts_ghz122.ncgr.small"         },
	{0x8B1159D6, "a1c3_shadow_enter_hold.gff"          },
	{0x8B155370, "rax_cb_pattack.nsbca"                },
	{0x8B204A84, "fx_gnp_beam.nsbmd.small"             },
	{0x8B22487D, "a2c10_hologram_1_appear.gff"         },
	{0x8B285A55, "son_axekick.gff"                     },
	{0x8B29BDFE, "charpro_stat_tablit1.ncgr.small"     },
	{0x8B3E362E, "itm_fotslpgr_1.ncgr.small"           },
	{0x8B3F7167, "chao_jq_r.nsbtp.small"               },
	{0x8B441070, "prt_tp_e_che.ncgr.small"             },
	{0x8B4999CC, "cfx_gtpp1_brk.emit"                  },
	{0x8B5FA645, "spell_weakened.spl"                  },
	{0x8B6D5704, "main_pnl_big10.ncgr.small"           },
	{0x8B83C1ED, "pl_pz_ppd_up.nsbtp.small"            },
	{0x8B9B4B66, "itm_act2_chaos3_0.ncgr.small"        },
	{0x8B9B8357, "lower_bg_03.ncgr.small"              },
	{0x8BB45DFF, "spell_shieldbot.spl"                 },
	{0x8BC15F4C, "icn_l_larwofstr.ncgr.small"          },
	{0x8BC6080D, "a2_w0_tx.nclr.small"                 },
	{0x8BC756DB, "cfx_rezp1.nsbtx.small"               },
	{0x8BCFC5A6, "chao_jc_r.nsbmd.small"               },
	{0x8BDB325A, "conv_bubble_bs01.ncgr.small"         },
	{0x8BE4A36A, "cfx_mas.nsbtx.small"                 },
	{0x8BEA4322, "man_tsvst_on12.ncgr.small"           },
	{0x8C0309E8, "a1_s1_ex_boatwinchsouth.gff"         },
	{0x8C04F4D9, "a2_s1_ex_slide1.gff"                 },
	{0x8C09FDC4, "chao_ba_g.nsbtp.small"               },
	{0x8C170D5A, "a2_s1_ex_slide2.gff"                 },
	{0x8C1C9E94, "c_sotaamkn_1_kn2.nsbca"              },
	{0x8C1D80F8, "vox_cb_idle.nsbca"                   },
	{0x8C2925DB, "a2_s1_ex_slide3.gff"                 },
	{0x8C2EE79B, "exp_tws1_mm21.ncgr.small"            },
	{0x8C38E835, "invent_list11.ncgr.small"            },
	{0x8C4F6FA5, "prtl_cresca_0.ncgr.small"            },
	{0x8C500C91, "icn_s_patk_ome2.ncgr.small"          },
	{0x8C5CD2B3, "gzg_cb_dead.nsbca"                   },
	{0x8C5CFFF6, "gui_defeatscreen.nclr.small"         },
	{0x8C605EEA, "cfx_rtbp1.nsbtp.small"               },
	{0x8C68E652, "pfx_mot1_blu.emit"                   },
	{0x8C7CC7FA, "chao_jc_0.ncgr.small"                },
	{0x8C865359, "cbt_round_slide.nanr.small"          },
	{0x8C9C4618, "jurnl_ts03.ncgr.small"               },
	{0x8CA2B54C, "a1_s5_ex_slide10.gff"                },
	{0x8CB4CDCD, "a1_s5_ex_slide11.gff"                },
	{0x8CC6E64E, "a1_s5_ex_slide12.gff"                },
	{0x8CCCF185, "pl_pz_cav_dr_01.nsbta.small"         },
	{0x8CD03E6F, "itm_fotwrkbt_0.ncgr.small"           },
	{0x8CD27B5B, "prtl_gungen_3.ncgr.small"            },
	{0x8CD44B93, "fx_ix_beamlg.nsbmd.small"            },
	{0x8CD73E2C, "fx_gren_rou.nsbmd.small"             },
	{0x8CD8FECF, "a1_s5_ex_slide13.gff"                },
	{0x8CDCE82C, "prologue.gda"                        },
	{0x8CE44C8B, "prf_tboxb_ms02.ncgr.small"           },
	{0x8CEB1750, "a1_s5_ex_slide14.gff"                },
	{0x8CEC1372, "gup_cb_pattack01.nsbca"              },
	{0x8CFD2FD1, "a1_s5_ex_slide15.gff"                },
	{0x8D066C9A, "conv_icons.nclr.small"               },
	{0x8D1A95F0, "rhk_cb_dead.nsbca"                   },
	{0x8D1C508D, "omg_ex_run.nsbca"                    },
	{0x8D233753, "flee02_ts.tga"                       },
	{0x8D251623, "chao_jj_2.ncgr.small"                },
	{0x8D36AD3E, "hnpc_for_aa.nsbmd.small"             },
	{0x8D3A2258, "son_cb_block.nsbca"                  },
	{0x8D44EA2B, "arm_cb_dead.nsbca"                   },
	{0x8D4B76A8, "pfx_bfy1.nsbtx.small"                },
	{0x8D55BEEF, "prtl_shade.nclr.small"               },
	{0x8D7EDF3C, "pl_ob_blk_nb.nsbtx.small"            },
	{0x8D815D85, "conv_ambient_bkgd.nbfp"              },
	{0x8D815D88, "conv_ambient_bkgd.nbfs"              },
	{0x8D8E53D1, "cfx_levp1.nsbtx.small"               },
	{0x8D908B64, "prtl_shdtht_1.ncgr.small"            },
	{0x8D918EFC, "prtl_eggmad_3.ncgr.small"            },
	{0x8D9A1165, "itm_glv_spark_0.ncgr.small"          },
	{0x8DB03655, "icn_l_bugspray.ncgr.small"           },
	{0x8DB7824D, "ecm_fx_mflash.nsbtp.small"           },
	{0x8DC02FC1, "pfx_alm1.nsbtx.small"                },
	{0x8DC5E3DB, "chao_fa_r.nsbtx.small"               },
	{0x8DC726D6, "bsl_cb_attack.nsbca"                 },
	{0x8DF7B2D3, "prtl_gglugh_1.ncgr.small"            },
	{0x8DFAA6C5, "icn_s_catk_sa.ncgr.small"            },
	{0x8E0478B9, "icn_s_rbtnwmp.ncgr.small"            },
	{0x8E0CC637, "chao_ga_2.ncgr.small"                },
	{0x8E149BCE, "fx_smokepuff.nsbtp.small"            },
	{0x8E1C3C5E, "lvl_stat_tspnl01.ncgr.small"         },
	{0x8E1F95FF, "charpro_cre3.ncgr.small"             },
	{0x8E2CE569, "prtl_vxmgen_0.ncgr.small"            },
	{0x8E3C5622, "prtl_drone.nclr.small"               },
	{0x8E44ED2F, "jrl_tlgbx_ms31.ncgr.small"           },
	{0x8E4BE16A, "fx_vox_r_ea_m.nsbmd.small"           },
	{0x8E500046, "pl_mv_cyc_aa.nsbtx.small"            },
	{0x8E59D62E, "fx_vox_p_ea_s.nsbmd.small"           },
	{0x8E5ED3A9, "prtl_txmgen_2.ncgr.small"            },
	{0x8E775EE0, "a1_s1_ex_la.smp"                     },
	{0x8E7D70C1, "icn_l_onjump.ncgr.small"             },
	{0x8E85088C, "crm_ex_run.nsbca"                    },
	{0x8E894BA4, "a1_w0_ai_00.ncgr.small"              },
	{0x8E897425, "btn_s_lftarw_on.ncgr.small"          },
	{0x8E912A64, "itm_glv_strong_3.ncgr.small"         },
	{0x8E9358EF, "prtl_thbgen_3.ncgr.small"            },
	{0x8E99239C, "popupdialog.gui"                     },
	{0x8E9AFED3, "chao_ba_3.ncgr.small"                },
	{0x8E9D983C, "pfx_flyby.nsbtx.small"               },
	{0x8EB02637, "itm_ace_dimbu_2.ncgr.small"          },
	{0x8EB1AD5B, "son_cb_blockc.nsbca"                 },
	{0x8EB39DA0, "a2c8_scy_tyxisarenaentercbt.gff"     },
	{0x8EBCA9C3, "a2_s2_i05_ts.nbfp"                   },
	{0x8EBCA9C6, "a2_s2_i05_ts.nbfs"                   },
	{0x8EBFB527, "chao_jp_0.ncgr.small"                },
	{0x8EC26068, "cbt_player_ambush.ncgr.small"        },
	{0x8EC8D74B, "ccw_w_s.gff"                         },
	{0x8ED0AAC7, "exp_ghz1_fw01.ncgr.small"            },
	{0x8EDEB02C, "endscreen_ts.tga"                    },
	{0x8EEFFB9E, "cmbtbg_a1_s2.nsbca"                  },
	{0x8EF57844, "gen_aa.nsbtx.small"                  },
	{0x8F0F3297, "cfx_poip1.nsbmd.small"               },
	{0x8F188AE1, "prtl_egggen_0.ncgr.small"            },
	{0x8F1A09D1, "prf_chrbx_di21.ncgr.small"           },
	{0x8F1A8CED, "nrg_cb_pattack01.nsbca"              },
	{0x8F25B683, "a1_s3_i02_mm.nclr.small"             },
	{0x8F2760AF, "a1c5_sonic.dlg"                      },
	{0x8F4CC002, "cfx_fgrp2.nsbtp.small"               },
	{0x8F53BE73, "prtl_eggaha_2.ncgr.small"            },
	{0x8F540FAA, "a2_s7_ex_mm.nbfp"                    },
	{0x8F540FAD, "a2_s7_ex_mm.nbfs"                    },
	{0x8F5550C7, "cfx_ferp1.nsbtx.small"               },
	{0x8F61D2D7, "pzl_voxestablish.gff"                },
	{0x8F69FBC4, "fx_ome_beam.nsbta.small"             },
	{0x8F6A0568, "gzc_ex_walk.nsbca"                   },
	{0x8F75788B, "exp_ghz1_fw23.ncgr.small"            },
	{0x8F7C250B, "bmp_cb_reload.nsbta.small"           },
	{0x8F7D2257, "jrl_tmbox_ms12.ncgr.small"           },
	{0x8F80F811, "tai_ex_flight.nsbca"                 },
	{0x8F86C577, "ehd_thlth_ms00.ncgr.small"           },
	{0x8F8D0227, "pfx_mst1_mst.emit"                   },
	{0x8F99BA27, "prtl_tdad.nclr.small"                },
	{0x8F99D3FC, "gui_charpro.nclr.small"              },
	{0x8FA31820, "bfl_cb_attackend.nsbca"              },
	{0x8FB05AA3, "pfx_smd1.nsbtp.small"                },
	{0x8FC764DF, "prtl_nrmge2_0.ncgr.small"            },
	{0x8FCAF180, "fx_bird.emit"                        },
	{0x8FD86CB9, "egb_cb_blockc.nsbca"                 },
	{0x8FD98F7B, "cfx_fwnp1_win.emit"                  },
	{0x8FDEC875, "chao_ic_g.nsbtx.small"               },
	{0x8FE163CB, "fx_noc_jav.nsbmd.small"              },
	{0x8FE17CF1, "chao_jn_r.nsbmd.small"               },
	{0x8FEC4301, "kro_cb_attack.nsbca"                 },
	{0x8FF8E327, "spell_death.spl"                     },
	{0x9005B13B, "lootgui.gui"                         },
	{0x9005D51B, "amy_ex_fall.nsbca"                   },
	{0x900678E0, "chao_hd_r.nsbtx.small"               },
	{0x900BB8B5, "prtl_amysca_3.ncgr.small"            },
	{0x9025ECFC, "gui_buttons.nclr.small"              },
	{0x90268E68, "cbt_flescp_0.ncgr.small"             },
	{0x90306A8A, "spell_int_gizoidshield.spl"          },
	{0x9031628E, "pfx_gus1.nsbtp.small"                },
	{0x9035D44C, "prtl_knuckgen_2.ncgr.small"          },
	{0x904D2257, "cmbtbg_a1_i01.nsbmd.small"           },
	{0x904FBBAA, "conv_iconblank.ncgr.small"           },
	{0x905291C0, "spell_distracted_all.spl"            },
	{0x90536761, "cbt_menu_button_of01.ncgr.small"     },
	{0x9056967C, "fx_a1_s4_ics.nsbtx.small"            },
	{0x905699B7, "kro_cb_dead.nsbca"                   },
	{0x905A88D1, "rhk_cb_attackend.nsbca"              },
	{0x90695CB1, "cw_n_e.gff"                          },
	{0x906D54AE, "spell_focussed_2.spl"                },
	{0x9075F61A, "charpro_ts_a01.ncgr.small"           },
	{0x907DC1C1, "prtl_ffmmad_2.ncgr.small"            },
	{0x907F6D2F, "spell_focussed_3.spl"                },
	{0x9081A325, "journalrules.gda"                    },
	{0x908ECAF1, "movement03_bs.tga"                   },
	{0x90991D59, "wmp_txtbr_tx03.ncgr.small"           },
	{0x909DE097, "pfx_mst1_skf.emit"                   },
	{0x90A3346D, "icn_s_statemp.ncgr.small"            },
	{0x90A7B432, "icn_s_catk_or.ncgr.small"            },
	{0x90A92ADC, "chao_ie_1.ncgr.small"                },
	{0x90B3D992, "lvl_tsstr_ms02.ncgr.small"           },
	{0x90C9513C, "rou_cb_pattack01.nsbca"              },
	{0x90D7A009, "cfx_ch5.nsbtx.small"                 },
	{0x90E39DA4, "exp_wel1_mm02.ncgr.small"            },
	{0x90E4C02B, "prt_se_nctsc.ncgr.small"             },
	{0x90EF777C, "a1_s2_ex_pipe_down.gff"              },
	{0x90FA9A8D, "chao_jb_g.nsbtp.small"               },
	{0x90FB8ACA, "hnpc_kid_aa.nsbtx.small"             },
	{0x90FCD568, "fx_egb_beam.nsbtx.small"             },
	{0x90FE30C7, "fx_giz_lzr_muz.nsbmd.small"          },
	{0x90FF6C4D, "a1_w2_gh_11.ncgr.small"              },
	{0x9104448A, "pzl_ventsestablish.gff"              },
	{0x911613A9, "plt_fp_amy.plo"                      },
	{0x911D0B68, "pl_ob_blkb_sb.nsbmd.small"           },
	{0x91224AF8, "chao_ic_r.nsbtp.small"               },
	{0x912788B7, "tut_jump.gda"                        },
	{0x91300AF1, "itm_glv_power_0.ncgr.small"          },
	{0x914340C6, "exl_barow_ms00.ncgr.small"           },
	{0x914AE281, "worldmap_top1.gui"                   },
	{0x914BCF70, "cfx_ixdp2.nsbtx.small"               },
	{0x9161D17B, "gui_main_pnl.nclr.small"             },
	{0x9165CC8C, "bridge.gda"                          },
	{0x917EEC5F, "prtl_eixthrnvar_0.ncgr.small"        },
	{0x918A2AE3, "gui_chao_j2.nclr.small"              },
	{0x91922E16, "fx_tyx_prj.nsbtx.small"              },
	{0x9194B396, "pl_pz_mem_lts.nsbtp.small"           },
	{0x91AD1728, "cfx_spcp1.nsbtp.small"               },
	{0x91B1A86D, "cfx_mg_prj.nsbmd.small"              },
	{0x91B74E1A, "a2_s7_ex_mm.nclr.small"              },
	{0x91BC9B41, "a1c2_gunblock.gff"                   },
	{0x91C22CA9, "jrl_scrll_tk00.ncgr.small"           },
	{0x91D47B4D, "invent_bar01.ncgr.small"             },
	{0x91DCA03E, "upper_textbox_02.ncgr.small"         },
	{0x91E1D1D2, "prtl_wsprel_1.ncgr.small"            },
	{0x91F396C5, "cbt_ambushed.ncgr.small"             },
	{0x91F5D140, "prtl_eixfm3_0.ncgr.small"            },
	{0x91FC0EDB, "chao_ab_gard.nsbtp.small"            },
	{0x92184377, "itm_ace_timmy_3.ncgr.small"          },
	{0x921CD7FC, "cho_tmbt2_of02.ncgr.small"           },
	{0x921FAF57, "hnpc_con_aa.nsbmd.small"             },
	{0x9233616B, "chaogarden.gda"                      },
	{0x92392919, "chao_fc_3.ncgr.small"                },
	{0x92398F82, "icn_l_barwofstr.ncgr.small"          },
	{0x92470DE5, "chao_jg_r.nsbtx.small"               },
	{0x92512490, "loc_round_dt0.ncgr.small"            },
	{0x925E4187, "chao_aa_gard.nsbmd.small"            },
	{0x92616193, "fx_shason_atom.nsbca"                },
	{0x926A5911, "pfx_ppr1.nsbtp.small"                },
	{0x927022B3, "rwd_lgbtn_di00.ncgr.small"           },
	{0x9286EE92, "amy_cb_blockc.nsbca"                 },
	{0x92A39D27, "charpro_bar00.ncgr.small"            },
	{0x92ADDEAC, "fx_a2_s4_ex_env.nsbta.small"         },
	{0x92AE2156, "chaogarden_top.gui"                  },
	{0x92AED4AB, "cbt_xpbar_s.ncgr.small"              },
	{0x92B586AE, "exl_smbtn_on11.ncgr.small"           },
	{0x92B74E13, "pro_lrgbx_10.ncgr.small"             },
	{0x92B90DA9, "exl_flash_ms0.ncgr.small"            },
	{0x92BD4A7B, "itm_glv_stealth1.ncgr.small"         },
	{0x92BD8F75, "cfx_gtpp1.nsbtx.small"               },
	{0x92D77C62, "fx_vox_g_ea_l.nsbta.small"           },
	{0x92DBE724, "a2c7_ksoldier1.dlg"                  },
	{0x92E3CE80, "itm_glvgrpcl_3.ncgr.small"           },
	{0x92E4FB3F, "pl_ob_blkn_sr.nsbtx.small"           },
	{0x92E76B43, "icn_l_barwofcho.ncgr.small"          },
	{0x92EC9E53, "chao_da_g.nsbmd.small"               },
	{0x92F054E1, "bmp_cb_getup.nsbca"                  },
	{0x92F14EA1, "a1_w2_mr.nclr.small"                 },
	{0x92F2AFB4, "cmbtbg_a2_s3.nsbmd.small"            },
	{0x930B4FEE, "a2c10_ix_after_phase_1.gff"          },
	{0x9317893F, "wmp_airp_on00.ncgr.small"            },
	{0x931AAECA, "loc_abshd_fr0.ncgr.small"            },
	{0x931FA13E, "gen_ex_walk.nsbca"                   },
	{0x932269F4, "cfx_aslt_mf.nsbmd.small"             },
	{0x932B7D2E, "prtl_txrhap_3.ncgr.small"            },
	{0x93568EF1, "plt_fp_eggman.plo"                   },
	{0x9363A83C, "mut_shoes_ms11.ncgr.small"           },
	{0x936623F1, "tyx_cb_kd.nsbca"                     },
	{0x9368921A, "cbt_elem_lgtng.ncgr.small"           },
	{0x936AE315, "man_tsvin_03.ncgr.small"             },
	{0x937B1EE5, "a2_s2_ex_puzzle_fail_camera.gff"     },
	{0x937FB1DD, "plt_fpa2_cream.plo"                  },
	{0x9381527E, "a1_w1_mt_11.ncgr.small"              },
	{0x938E23CB, "pfx_arnietest01.nsbmd.small"         },
	{0x93964A81, "prtl_rougeinq_3.ncgr.small"          },
	{0x93A21D10, "a1_s2_ex_slide4_down_right.gff"      },
	{0x93B17EE2, "cfx_flap1.nsbmd.small"               },
	{0x93B978A0, "prtl_eixprd_0.ncgr.small"            },
	{0x93BCC908, "prtl_nrsge2_3.ncgr.small"            },
	{0x93C0C2D9, "exl_buttn_ms01.ncgr.small"           },
	{0x93C50B9C, "btn_bar_01.ncgr.small"               },
	{0x93CB7695, "gui_portraitssal.nclr.small"         },
	{0x93CBEF24, "conv_icon02.ncgr.small"              },
	{0x93CD7111, "plt_cod_fgciv.plo"                   },
	{0x93CE842B, "areas.gda"                           },
	{0x93F316B0, "cfx_spkp1_gud.emit"                  },
	{0x93F42A10, "rpg_redrightarrow.ncgr.small"        },
	{0x93FDF7AD, "a2c6_eggbot.dlg"                     },
	{0x9401ED6E, "prtl_tailsdet_0.ncgr.small"          },
	{0x9407FD21, "pl_pz_cav_scrn.nsbmd.small"          },
	{0x940A88F7, "itm_fotslplt_3.ncgr.small"           },
	{0x9417BDAE, "icn_l_defend.ncgr.small"             },
	{0x942F9C1A, "prtl_sonicinq_2.ncgr.small"          },
	{0x9430CA07, "prtl_krscon_3.ncgr.small"            },
	{0x9435E18A, "itm_ace_root_2.ncgr.small"           },
	{0x94362DB2, "loc_trppd_fr0.ncgr.small"            },
	{0x943F47DD, "lvl_tdbld_ms03.ncgr.small"           },
	{0x944ADD15, "icn_s_mm_cyclone.ncgr.small"         },
	{0x944C1310, "cfx_smgp1_smg.emit"                  },
	{0x9452979A, "drnlaz_aa.nsbmd.small"               },
	{0x945A4D56, "vox_cb_pattack4.nsbca"               },
	{0x945E8247, "tyx_cb_getup.nsbca"                  },
	{0x946DD8E2, "str_tsmbr_on02.ncgr.small"           },
	{0x94724CDC, "charpro_rou4.ncgr.small"             },
	{0x9473BB97, "pl_pu_emrld6.nsbmd.small"            },
	{0x9480A22A, "fx_vox_r_ea_r.nsbtx.small"           },
	{0x948122B0, "prtl_eixlaf_3.ncgr.small"            },
	{0x94845331, "prtl_egghum_0.ncgr.small"            },
	{0x9490360C, "main_icon_grey.nclr.small"           },
	{0x94948BE9, "swt_cb_dead.nsbca"                   },
	{0x949546A5, "exu_tmana_ms00.ncgr.small"           },
	{0x94A7346E, "son_ex_slideout.nsbca"               },
	{0x94C0CDBC, "fx_tal_spn.nsbmd.small"              },
	{0x94D36E18, "pfx_grs1.nsbmd.small"                },
	{0x94F5D6CB, "itm_act2_chaos5_3.ncgr.small"        },
	{0x950BA6F5, "itm_ace_ironw_2.ncgr.small"          },
	{0x950C51D8, "chao_jm_g.nsbtp.small"               },
	{0x950C932C, "a2c10_knuckles_activate.gff"         },
	{0x95155E90, "icn_s_patk_tai4.ncgr.small"          },
	{0x952995D5, "c_sharoom_ro.nsbca"                  },
	{0x952D3358, "chao_fd_g.nsbmd.small"               },
	{0x953399A9, "icn_l_rarwonjrl.ncgr.small"          },
	{0x953AA2DE, "pro_smbox_02.ncgr.small"             },
	{0x95442086, "fx_els_en_aa.nsbta.small"            },
	{0x95495B5C, "pl_pu_chst_bp.nsbca"                 },
	{0x9564796C, "pl_pz_mem_box.nsbtp.small"           },
	{0x9564D34F, "prtl_guncodet_2.ncgr.small"          },
	{0x956DE9BA, "prtl_guncosmg_3.ncgr.small"          },
	{0x957811F3, "crm_ex_jump.nsbca"                   },
	{0x9579DD55, "scrollingbg.gda"                     },
	{0x958E4861, "a1c5_p3_airbase_cutscene.gff"        },
	{0x959506E8, "prtl_syraxgen_3.ncgr.small"          },
	{0x9596D880, "prtl_scientist.nclr.small"           },
	{0x95AEDC0F, "cfx_nocp1.nsbmd.small"               },
	{0x95B16898, "a2_s7_underlay.nbfp"                 },
	{0x95B1689B, "a2_s7_underlay.nbfs"                 },
	{0x95B678F0, "a1_w1_br_10.ncgr.small"              },
	{0x95BC7595, "icn_l_onbash2.ncgr.small"            },
	{0x95C8F1FA, "cfx_sson_blast.emit"                 },
	{0x95CDE6A6, "rax_cb_idle.nsbca"                   },
	{0x95D7535F, "chao_je_3.ncgr.small"                },
	{0x95D8E4DA, "a2_chb_thelxeintro.gff"              },
	{0x95DDA6EE, "charpro_bs_c14.ncgr.small"           },
	{0x95E353D8, "xbutton.ncgr.small"                  },
	{0x95E679B1, "tyx_cb_blockd.nsbca"                 },
	{0x95EA8E70, "a1c1_firstcombathold.gff"            },
	{0x95EE0BD8, "pl_mv_cyc_fly.nsbta.small"           },
	{0x95F2B1CE, "prtl_swatbot.nclr.small"             },
	{0x95F6219B, "prtl_egglaf_1.ncgr.small"            },
	{0x9601CF0C, "icn_s_catk_hs.ncgr.small"            },
	{0x9614F88B, "prtl_tdadsad_2.ncgr.small"           },
	{0x961B00B1, "btn_arrow_00.ncgr.small"             },
	{0x961F84B0, "vox_ove_aa.nsbtx.small"              },
	{0x96224DE1, "cbt_elem_water.ncgr.small"           },
	{0x96368373, "ehd_tbbox_ms01.ncgr.small"           },
	{0x9643D412, "pfx_smk1.nsbtx.small"                },
	{0x96544C52, "a1_w2_br_02.ncgr.small"              },
	{0x9658C530, "chao_jr_r.nsbtx.small"               },
	{0x965C9FFA, "lvl_pow_tspnl12.ncgr.small"          },
	{0x966156D5, "invent_barlit00.ncgr.small"          },
	{0x96619FB6, "gen_cb_damage.nsbca"                 },
	{0x96677BEB, "a1_w0_gh_02.ncgr.small"              },
	{0x966A63B0, "omg_cb_victory.nsbca"                },
	{0x967E4494, "tai_cb_rund.nsbca"                   },
	{0x968156B6, "pfx_lef1.nsbtp.small"                },
	{0x9682E6EA, "a1sq_projectors.dlg"                 },
	{0x968BAC3C, "a2c10_charyb_attack_1.gff"           },
	{0x968F82F1, "worldmap_pathdot.gui"                },
	{0x969F6E4C, "fx_a1_s5_ex_env.nsbta.small"         },
	{0x96A50D74, "prtl_generic_3.ncgr.small"           },
	{0x96A77D6F, "c_sokn_kn.nsbca"                     },
	{0x96B0A65B, "statuses.gda"                        },
	{0x96B87576, "big_cb_victoryin.nsbca"              },
	{0x96CEC73B, "bif_big.nbfp"                        },
	{0x96CEC73E, "bif_big.nbfs"                        },
	{0x96CFA8D5, "lizscor_aa.nsbmd.small"              },
	{0x96D565EB, "lvl_tmstr_ms01.ncgr.small"           },
	{0x96D70EF8, "a1c2_prisonescape.gff"               },
	{0x96D7B9CF, "cfx_revp1.nsbtp.small"               },
	{0x96DA7E7E, "c_ombi_om.nsbca"                     },
	{0x96DFA114, "exl_circe_ms21.ncgr.small"           },
	{0x96E249D4, "camerapannrrgal.gff"                 },
	{0x96EECCE1, "zz_jaytest.dlg"                      },
	{0x96F6E9AD, "test.nftr"                           },
	{0x97020EC1, "str_txtbr_hi01.ncgr.small"           },
	{0x97064467, "sha_cb_blockc.nsbca"                 },
	{0x970C76DD, "a1c1_explore_tutorial.gff"           },
	{0x9711D037, "prtl_nrdge2_1.ncgr.small"            },
	{0x972E1160, "camerapanangel_intro.gff"            },
	{0x97311FE0, "adv_amy.gda"                         },
	{0x973B3CAE, "cfx_nrg_slsh.nsbmd.small"            },
	{0x9745AFC2, "itm_ace_spdbr_3.ncgr.small"          },
	{0x974C0881, "sco_cb_damage.nsbca"                 },
	{0x9753FDDD, "pro_bar_of00.ncgr.small"             },
	{0x97560032, "rwd_tsmbx_of00.ncgr.small"           },
	{0x975BBD2C, "prtl_eixmad_0.ncgr.small"            },
	{0x9766324F, "son_cb_blockin.nsbca"                },
	{0x977110BC, "froggy.nsbmd.small"                  },
	{0x9771DC05, "a1c4_brzarrive.gff"                  },
	{0x9771F263, "chao_jk_1.ncgr.small"                },
	{0x9779B2B3, "cfx_rtcp1.nsbtx.small"               },
	{0x977B5646, "exp_ghz2_mm11.ncgr.small"            },
	{0x977F039A, "a2_w0_kr_11.ncgr.small"              },
	{0x97848A37, "pl_mv_tor_oi.nsbmd.small"            },
	{0x978C04BB, "wmp_s_ringb.ncgr.small"              },
	{0x97936F51, "prtl_timcal_3.ncgr.small"            },
	{0x979597D0, "conv_portr_01.ncgr.small"            },
	{0x979A96CD, "gui_chao_f.nclr.small"               },
	{0x979AACFF, "itm_glv_light_3.ncgr.small"          },
	{0x979DBBB0, "cfx_ixmp1.nsbtp.small"               },
	{0x97AB73A8, "pl_pz_crn_stopb.nsbca"               },
	{0x97B12415, "icn_l_replay.ncgr.small"             },
	{0x97BE0AE7, "cfx_glw_ghlp1.emit"                  },
	{0x97C02969, "bmp_aa.nsbtx.small"                  },
	{0x97C66D31, "prtl_pilsad_0.ncgr.small"            },
	{0x97C75716, "icn_l_barwofrwd.ncgr.small"          },
	{0x97D5CE28, "cfx_shcp1.nsbtx.small"               },
	{0x97E3D8B0, "main_pnl_off13.ncgr.small"           },
	{0x97FFBD5E, "omg_cb_idle.nsbca"                   },
	{0x980A0EB5, "icn_s_catk_ca.ncgr.small"            },
	{0x980AEA19, "prtl_boyhap_2.ncgr.small"            },
	{0x981186AB, "cfx_pwup1.nsbmd.small"               },
	{0x9811C76C, "cfx_ftpp1.nsbtp.small"               },
	{0x981704DF, "prtl_txmmad_0.ncgr.small"            },
	{0x981A408C, "chao_jr_3.ncgr.small"                },
	{0x981F7724, "a1_w3_mt.nclr.small"                 },
	{0x9820240A, "exp_ghz2_mm33.ncgr.small"            },
	{0x9823AFC8, "prf_tboxa_ms00.ncgr.small"           },
	{0x982E6527, "itm_ace_seed_2.ncgr.small"           },
	{0x983F23A5, "chao_ca_g.nsbtp.small"               },
	{0x98406C4E, "pl_pz_vox_swc.nsbtp.small"           },
	{0x984CA280, "drnass_cb_attack.nsbca"              },
	{0x9859A277, "chao_gb_1.ncgr.small"                },
	{0x9862E6C3, "icn_s_snglatk.ncgr.small"            },
	{0x986A7AC4, "man_bkbtn_of00.ncgr.small"           },
	{0x9870AD01, "gwr_lt.nsbta.small"                  },
	{0x98730C36, "main_pnl_on17.ncgr.small"            },
	{0x987AFE10, "pfx_spark.emit"                      },
	{0x988040F7, "jurnl_ts20.ncgr.small"               },
	{0x98860680, "fx_vox_b_es.nsbmd.small"             },
	{0x9889738E, "ch4_shadeappears.gda"                },
	{0x989450B4, "cfx_pjkp1.nsbmd.small"               },
	{0x98A117A6, "gui_exploreagr.nclr.small"           },
	{0x98AE2A41, "icn_l_onfly1.ncgr.small"             },
	{0x98BE8058, "wmp_space_of10.ncgr.small"           },
	{0x98D78C14, "prtl_humcit.nclr.small"              },
	{0x98D967B0, "rwd_tlgbx_ms00.ncgr.small"           },
	{0x98E2BFDD, "fx_she_shkwv.nsbmd.small"            },
	{0x98E4AA22, "itm_fotsnkel_0.ncgr.small"           },
	{0x98F086F5, "pl_ob_blkb_so.nsbmd.small"           },
	{0x98F2B484, "tyx_cb_blockin.nsbca"                },
	{0x98FC83DF, "lvl_smlbt_di00.ncgr.small"           },
	{0x990BB7EB, "rax_cb_blockout.nsbca"               },
	{0x991205A5, "itm_ace_bugsp_0.ncgr.small"          },
	{0x99121530, "cfx_snry_mf.nsbta.small"             },
	{0x9915BE3C, "lower_textbox_03.ncgr.small"         },
	{0x9921694F, "prtl_shdhel_3.ncgr.small"            },
	{0x9928692B, "prf_tpanl_ms11.ncgr.small"           },
	{0x992C3D2A, "prtl_wsgrel_2.ncgr.small"            },
	{0x992E56CF, "tyx_aa.nsbtx.small"                  },
	{0x992EBA2F, "a1c3_shadowbattle.gff"               },
	{0x9930C962, "a1_s3_ex_la.smp"                     },
	{0x993AB7F1, "itm_fotstlbt_3.ncgr.small"           },
	{0x9940F1C4, "itm_ace_crzbn_1.ncgr.small"          },
	{0x994643F7, "questlogselectiongui.gui"            },
	{0x994D5C28, "tai_ex_run.nsbca"                    },
	{0x9956CF54, "prtl_amysad_1.ncgr.small"            },
	{0x99707F8A, "supersonic.gda"                      },
	{0x99840728, "victory.gui"                         },
	{0x998EA1E0, "a1_s2_i03_ts.nbfp"                   },
	{0x998EA1E3, "a1_s2_i03_ts.nbfs"                   },
	{0x99AE5D62, "chao_jj_g.nsbmd.small"               },
	{0x99AEABCF, "cfx_fltp1_lgt.emit"                  },
	{0x99B09F27, "sco_cb_getup.nsbca"                  },
	{0x99B71CAF, "ssf_tlgbx_ms00.ncgr.small"           },
	{0x99C2629F, "gwr_aa.nsbmd.small"                  },
	{0x99C5B67C, "upper_red_ring_04.ncgr.small"        },
	{0x99CBC42C, "itm_fotsnksh_0.ncgr.small"           },
	{0x99D04F08, "invent_scroll_03.ncgr.small"         },
	{0x99D060D1, "pil_cb_kd.nsbca"                     },
	{0x99D2596D, "prt_sp_big.ncgr.small"               },
	{0x99E14BB6, "gnp_ex_jump.nsbca"                   },
	{0x99E2F40D, "str_resrv_ms00.ncgr.small"           },
	{0x99EB1A6C, "fx_spark.nsbmd.small"                },
	{0x99EC1A9D, "tabbedcharacterdisplay.gui"          },
	{0x99EFE80F, "prtl_fsggen_1.ncgr.small"            },
	{0x99F1673E, "cfx_slpp1.nsbmd.small"               },
	{0x99F1BEB4, "a2_w0_vb_01.ncgr.small"              },
	{0x99FB09BC, "chao_ga_r.nsbtx.small"               },
	{0x99FB6442, "fx_a2_s7_env.nsbtp.small"            },
	{0x9A005D3E, "spinball.nsbtx.small"                },
	{0x9A018344, "prtl_cre.nclr.small"                 },
	{0x9A1A08D4, "cfx_curp1.nsbtx.small"               },
	{0x9A20CCC5, "man_strbt_of12.ncgr.small"           },
	{0x9A220EA9, "prtl_crehap_2.ncgr.small"            },
	{0x9A278E95, "crm_ex_idle.nsbca"                   },
	{0x9A3BA8F4, "a1_s2_ex_mm.nclr.small"              },
	{0x9A489155, "exp_cct1_mm10.ncgr.small"            },
	{0x9A4B6887, "a1c3_shadow_avoid_reset.gff"         },
	{0x9A5587A3, "prt_mp_sdw.ncgr.small"               },
	{0x9A5952A1, "a1_s6_ex.are"                        },
	{0x9A5BEA73, "ssf_tlgbx_ms22.ncgr.small"           },
	{0x9A651BAC, "ch10_councilapproach.gff"            },
	{0x9A72ACE2, "prtl_thxgen_0.ncgr.small"            },
	{0x9A771446, "cbt_rndnum6.ncgr.small"              },
	{0x9A79FE5B, "cfx_rbep1.nsbtp.small"               },
	{0x9A7C68A3, "jurnl_mis_line05.ncgr.small"         },
	{0x9A8EB6DC, "gui_levelup2.nclr.small"             },
	{0x9A933CC5, "a2_s2_ex_mm.nbfp"                    },
	{0x9A933CC8, "a2_s2_ex_mm.nbfs"                    },
	{0x9A943E87, "party_switch.gff"                    },
	{0x9A96BB2B, "a2c10_knucklesapproachix.gff"        },
	{0x9A9ABC16, "ix_cb_attack.nsbca"                  },
	{0x9A9D430B, "prtl_tal.nclr.small"                 },
	{0x9AA6A174, "cbt_focusringc.nanr.small"           },
	{0x9AABEFC0, "cbt_cmdframe14.ncgr.small"           },
	{0x9ABED099, "version.txt"                         },
	{0x9AC2D25A, "charpro_ts_b00.ncgr.small"           },
	{0x9AC6682D, "cfx_crvp1.nsbtp.small"               },
	{0x9AC9276D, "a1_s3_card_t1.nbfp"                  },
	{0x9AC92770, "a1_s3_card_t1.nbfs"                  },
	{0x9AD244E5, "prtl_wspsca_0.ncgr.small"            },
	{0x9AD4CF1B, "exl_circl_ms21.ncgr.small"           },
	{0x9AD574FF, "str_tlgbx_ms21.ncgr.small"           },
	{0x9ADC9B20, "wmp_trv_12_73.gff"                   },
	{0x9ADCDC6B, "spell_cloak.spl"                     },
	{0x9AE3A406, "cfx_scap1.nsbmd.small"               },
	{0x9AED5F19, "exp_cct1_mm32.ncgr.small"            },
	{0x9AF6C822, "prtl_vxcrblgen_3.ncgr.small"         },
	{0x9AF85051, "icn_l_catk_st.ncgr.small"            },
	{0x9AF876E8, "a1_w2_cc.nclr.small"                 },
	{0x9B06E231, "loc_round_fr1.ncgr.small"            },
	{0x9B102180, "combatrules.gda"                     },
	{0x9B124F2E, "crm_cb_attackend.nsbca"              },
	{0x9B12D3BA, "prtl_timgen_2.ncgr.small"            },
	{0x9B137CD0, "a1c4_shadowflee.gff"                 },
	{0x9B19B760, "prt_se_gizdb.ncgr.small"             },
	{0x9B32D073, "gzc_aa.nsbmd.small"                  },
	{0x9B3B4729, "main_pnl_big04b.ncgr.small"          },
	{0x9B3CB618, "prtl_fmncon_3.ncgr.small"            },
	{0x9B471576, "cho_tmbtn_of00.ncgr.small"           },
	{0x9B4891CA, "cbt_menu_button_on02.ncgr.small"     },
	{0x9B5BE9E6, "icn_s_catk_at.ncgr.small"            },
	{0x9B5C8980, "cfx_bugp1.nsbtp.small"               },
	{0x9B5C8E61, "icn_l_utbash2.ncgr.small"            },
	{0x9B66BCAB, "prtl_knustb_0.ncgr.small"            },
	{0x9B67A529, "pl_pu_memrld.nsbtx.small"            },
	{0x9B692548, "spell_distracted.spl"                },
	{0x9B6F89B3, "swt_cb_attack.nsbca"                 },
	{0x9B710EAE, "itm_act2_mastr_2.ncgr.small"         },
	{0x9B795AEB, "lose.gui"                            },
	{0x9B7AEF5D, "prtl_eixamu_0.ncgr.small"            },
	{0x9B80A5B8, "pl_pz_wod_aa.nsbtx.small"            },
	{0x9B8A76A8, "cfx_ffrp1.nsbtx.small"               },
	{0x9B8BC73C, "sha_cb_blockout.nsbca"               },
	{0x9B9298B5, "que_aa.nsbtx.small"                  },
	{0x9BA41084, "icn_l_flee.ncgr.small"               },
	{0x9BA8D87F, "itm_fotttnhy_3.ncgr.small"           },
	{0x9BA8ED47, "a1_w2_ai_01.ncgr.small"              },
	{0x9BB9E5FD, "a2cha_overmind_buddy2_leave.gff"     },
	{0x9BBA823E, "a2_w0_kr.nclr.small"                 },
	{0x9BD16E37, "prtl_thxdef_0.ncgr.small"            },
	{0x9BDF1635, "itm_fotgrshy_3.ncgr.small"           },
	{0x9BE2E8A9, "icn_s_catk_wro.ncgr.small"           },
	{0x9BE362A9, "prf_crtbc_hi00.ncgr.small"           },
	{0x9BE7E54D, "a1c4_resetsonic.gff"                 },
	{0x9BF21376, "itm_ace_goggl_2.ncgr.small"          },
	{0x9BFEA989, "spell_empowered.spl"                 },
	{0x9C0BE046, "a1c2_soldiers_2.dlg"                 },
	{0x9C0F8A9C, "prt_mp_amy.ncgr.small"               },
	{0x9C1399BC, "chapter10nocturne.gff"               },
	{0x9C13EE56, "chao_jc_g.nsbtx.small"               },
	{0x9C29603E, "cbt_follow_transition.nanr.small"    },
	{0x9C367802, "a1c3_eggman_hold.gff"                },
	{0x9C3B9EC1, "chao_id_r.nsbtx.small"               },
	{0x9C3C3950, "a1c2_swampsounds.gff"                },
	{0x9C429E13, "str_vertw_ms00.ncgr.small"           },
	{0x9C44F189, "str_vertl_ms10.ncgr.small"           },
	{0x9C4DB220, "pl_am_met_gfl.nsbca"                 },
	{0x9C5B2593, "worldmap_areaindicator.gui"          },
	{0x9C631CF5, "itm_ace_resur_0.ncgr.small"          },
	{0x9C69F846, "a1c3_reunion_father.gff"             },
	{0x9C768CF5, "ehd_tbbx2_ms00.ncgr.small"           },
	{0x9C82E807, "main_action01.ncgr.small"            },
	{0x9C860559, "chao_fd_2.ncgr.small"                },
	{0x9C8609FA, "prtl_sbtgen_3.ncgr.small"            },
	{0x9C934B5C, "bsl_cb_idle.nsbca"                   },
	{0x9CA33D13, "min_circl_ms00.ncgr.small"           },
	{0x9CA4ED69, "cbt_menu_up_of.ncgr.small"           },
	{0x9CA64D73, "pfx_alm1_alm.emit"                   },
	{0x9CA74470, "prtl_wdmnexh_1.ncgr.small"           },
	{0x9CA7DB99, "prtl_rougedis_3.ncgr.small"          },
	{0x9CA8EF2B, "a2c6_amyshout.gff"                   },
	{0x9CB86BF3, "cfx_clup1.nsbmd.small"               },
	{0x9CC39064, "lower_textbox_arrow.ncgr.small"      },
	{0x9CDDA2CB, "nsw_aa.nsbta.small"                  },
	{0x9CDEE037, "prtl_sdwmad_3.ncgr.small"            },
	{0x9CE5DAED, "pl_ob_blkb_mr.nsbtx.small"           },
	{0x9CF4632F, "pl_ob_blkn_sb.nsbtx.small"           },
	{0x9CF70348, "pfx_briantest01.nsbmd.small"         },
	{0x9CF9D3EF, "omg_cb_attackend.nsbca"              },
	{0x9D00DAD0, "gup_ex_walk.nsbca"                   },
	{0x9D09D092, "spell_lucky_all_2.spl"               },
	{0x9D0C38BF, "cbt_black_lower.ncer.small"          },
	{0x9D102C0E, "prtl_eggsmg_0.ncgr.small"            },
	{0x9D116E75, "prtl_gnogen_3.ncgr.small"            },
	{0x9D12BF7F, "pl_am_noc_gct.nsbca"                 },
	{0x9D1BE913, "spell_lucky_all_3.spl"               },
	{0x9D1E500E, "a1_s3_card_t2.nbfp"                  },
	{0x9D1E5011, "a1_s3_card_t2.nbfs"                  },
	{0x9D350F46, "infoframe_l.ncgr.small"              },
	{0x9D36E811, "plt_cod_nocturnusclan.plo"           },
	{0x9D412D32, "prtl_sonicdis_2.ncgr.small"          },
	{0x9D41A39D, "prtl_vxcriagen_2.ncgr.small"         },
	{0x9D43FCDD, "itm_glvcrscl_1.ncgr.small"           },
	{0x9D4CF6A3, "ix_cb_getup.nsbca"                   },
	{0x9D5770D9, "chao_jc_r.nsbtp.small"               },
	{0x9D7785C8, "gup_cb_damage.nsbca"                 },
	{0x9D87CAFA, "chao_hb_3.ncgr.small"                },
	{0x9D896B43, "a1_w3_cc_00.ncgr.small"              },
	{0x9D9364ED, "prtl_txsgen_0.ncgr.small"            },
	{0x9D939B0F, "prtl_txrgen_3.ncgr.small"            },
	{0x9D9461D7, "jurnl_cod_linelit03.ncgr.small"      },
	{0x9D9F4400, "wmptutorial.gui"                     },
	{0x9DA105D5, "journalgui_ts_01.tga"                },
	{0x9DA3B381, "a2c10_ix_transform_final.gff"        },
	{0x9DB0F979, "gen_buttns_on00.ncgr.small"          },
	{0x9DC1FFD6, "fx_vox_y_ea_r.nsbmd.small"           },
	{0x9DCEB666, "fx_ix_proj.nsbmd.small"              },
	{0x9DEED2B8, "a1_s1_ex_slide1.gff"                 },
	{0x9DF1223D, "pfx_gus1_gus.emit"                   },
	{0x9DF50845, "main_pnl_big02.ncgr.small"           },
	{0x9DF83727, "debugqastats.gda"                    },
	{0x9DFD400C, "prtl_scihap_0.ncgr.small"            },
	{0x9E00EB39, "a1_s1_ex_slide2.gff"                 },
	{0x9E049856, "pro_circle_hi00.ncgr.small"          },
	{0x9E1303BA, "a1_s1_ex_slide3.gff"                 },
	{0x9E21DA61, "cbt_focusbreak.ncgr.small"           },
	{0x9E251C3B, "a1_s1_ex_slide4.gff"                 },
	{0x9E2B300A, "itm_ace_immun_0.ncgr.small"          },
	{0x9E335590, "cfx_levp1_lev.emit"                  },
	{0x9E36D1AC, "a1_s3_i01.walk.tga"                  },
	{0x9E3734BC, "a1_s1_ex_slide5.gff"                 },
	{0x9E38F719, "pro_help_on00.ncgr.small"            },
	{0x9E3C3111, "cfx_empp1.nsbmd.small"               },
	{0x9E3CD72E, "pl_pz_vox_ly.nsbtx.small"            },
	{0x9E468A3E, "prtl_amydrm_0.ncgr.small"            },
	{0x9E494D3D, "a1_s1_ex_slide6.gff"                 },
	{0x9E4AB5AA, "prt_tp_sdw.ncgr.small"               },
	{0x9E595822, "lvl_lrgbt_of02.ncgr.small"           },
	{0x9E5B65BE, "a1_s1_ex_slide7.gff"                 },
	{0x9E62A871, "sco_cb_attackend.nsbca"              },
	{0x9E65DC1C, "prtl_sdwgen_0.ncgr.small"            },
	{0x9E6E173D, "chap8scr_top.tga"                    },
	{0x9E6E2376, "store_3_bg.tga"                      },
	{0x9E6F7734, "pfx_lzg1.nsbtx.small"                },
	{0x9E79BFFB, "cbtmenubg_10.ncgr.small"             },
	{0x9E87614C, "crm_cb_pattack05.nsbca"              },
	{0x9E90C858, "gnp_ex_idle.nsbca"                   },
	{0x9EB50570, "pl_pz_rc_trd.nsbmd.small"            },
	{0x9EB698DC, "exp_tws1_mm13.ncgr.small"            },
	{0x9EBA2746, "invent_stat_tablit4.ncgr.small"      },
	{0x9ECACD14, "prtl_sonicsle_0.ncgr.small"          },
	{0x9ECC0328, "prtl_tailssca_0.ncgr.small"          },
	{0x9ED52E10, "cmbtbg_a1_s5.nsbtx.small"            },
	{0x9ED7381C, "son_cb_attackend.nsbca"              },
	{0x9EDF8B5C, "pfx_briantest01.emit"                },
	{0x9EE6298B, "a2c8_scy_gateguardlead.gff"          },
	{0x9EE6A698, "prtl_krsmad_2.ncgr.small"            },
	{0x9EF06962, "cbt_pathdot2.ncgr.small"             },
	{0x9EFD64DD, "fx_wtrfl_strk.nsbmd.small"           },
	{0x9EFF6E54, "prtl_omggen_3.ncgr.small"            },
	{0x9F05B762, "lower_blue_ring_00.ncgr.small"       },
	{0x9F07BA07, "a2c8_scy_tyxisarenaentercit.gff"     },
	{0x9F085471, "chao_ca_0.ncgr.small"                },
	{0x9F1F313D, "prtl_chbgen_2.ncgr.small"            },
	{0x9F21C434, "chao_ea_g.nsbmd.small"               },
	{0x9F32FD26, "cmbtbg_a2_s7_i02.nsbta.small"        },
	{0x9F3798B9, "tai_cb_support01.nsbca"              },
	{0x9F3DBF00, "a2c10_disableinput.gff"              },
	{0x9F42B30B, "itm_act2_chaos6_2.ncgr.small"        },
	{0x9F4EDDF4, "man_tsvin_20.ncgr.small"             },
	{0x9F564227, "cfx_spt_acd.nsbtx.small"             },
	{0x9F578E0C, "gwr_cb_damage.nsbca"                 },
	{0x9F5A37A5, "fx_a2_s1_ex_env.nsbmd.small"         },
	{0x9F5AC434, "prtl_salegen_1.ncgr.small"           },
	{0x9F6873B7, "cfx_crsp1.nsbmd.small"               },
	{0x9F6974F9, "spell_cure_all_2.spl"                },
	{0x9F6A1EF3, "exu_tbbox_ms00.ncgr.small"           },
	{0x9F7B8D7A, "spell_cure_all_3.spl"                },
	{0x9F8616FB, "cbt_menubg_11.ncgr.small"            },
	{0x9F8C2596, "pl_pz_pod_aa.nsbmd.small"            },
	{0x9F8C9439, "rhk_ex_damage.nsbca"                 },
	{0x9F949B27, "mut_circl_ms20.ncgr.small"           },
	{0x9F970AFE, "wmp_trv_08_03.gff"                   },
	{0x9F993E8B, "icn_s_flee.ncgr.small"               },
	{0x9F9A78AB, "lvl_stat_bspnl00.ncgr.small"         },
	{0x9FA1746A, "a1_s2_i01.are"                       },
	{0x9FA44147, "exl_altft_ms10.ncgr.small"           },
	{0x9FA9237F, "wmp_trv_08_04.gff"                   },
	{0x9FB0A29A, "chao_ch_2.ncgr.small"                },
	{0x9FB656DA, "a2c8_messenger_pan.gff"              },
	{0x9FB683FE, "bfl_ex_run.nsbca"                    },
	{0x9FC32957, "a1c4_shade.dlg"                      },
	{0x9FC49D71, "prtl_sdwdef_0.ncgr.small"            },
	{0x9FC4EC35, "fx_vox_y_ea_m.nsbta.small"           },
	{0x9FC5A56C, "a1_s2_i03.are"                       },
	{0x9FD064F1, "hnpc_gunof_aa.nsbtx.small"           },
	{0x9FD7BDED, "a1_s2_i04.are"                       },
	{0x9FDA4C24, "plt_a2sq_spawningpool.plo"           },
	{0x9FDEE185, "upper_portrait_02.ncgr.small"        },
	{0x9FE364A1, "pl_am_noc_gco.nsbtp.small"           },
	{0x9FE399C3, "gzg_cb_idle.nsbca"                   },
	{0x9FE9D66E, "a1_s2_i05.are"                       },
	{0x9FFBEEEF, "a1_s2_i06.are"                       },
	{0xA004B8A3, "prt_tp_amy.ncgr.small"               },
	{0xA008FA90, "cfx_spkp1_exc.emit"                  },
	{0xA00E0770, "a1_s2_i07.are"                       },
	{0xA0121FCD, "icn_s_patk_cre5.ncgr.small"          },
	{0xA020AF25, "knu_cb_pattack2.nsbca"               },
	{0xA02342BC, "lvl_tdbld_ms20.ncgr.small"           },
	{0xA0242F9F, "chao_jf_2.ncgr.small"                },
	{0xA025A5A1, "chao_jn_g.nsbtx.small"               },
	{0xA027D0AC, "jrl_tlgbx_ms01.ncgr.small"           },
	{0xA03809D4, "pop_lgbox_ms07.ncgr.small"           },
	{0xA03F466F, "lvl_stat_bspnl22.ncgr.small"         },
	{0xA055E3F5, "a1c2_knuckles_moves.gff"             },
	{0xA05D7004, "levelup2_ts.tga"                     },
	{0xA0676E62, "pl_am_noc_ghr.nsbca"                 },
	{0xA07FB87E, "prt_tp_m_rou.ncgr.small"             },
	{0xA086A83E, "itm_glv_cloth_0.ncgr.small"          },
	{0xA096DDCA, "cfx_poip1.nsbtp.small"               },
	{0xA0A15D00, "rhk_cb_idle.nsbca"                   },
	{0xA0A2F61E, "prtl_knugen_2.ncgr.small"            },
	{0xA0AC4FA2, "prtl_bigsnz_1.ncgr.small"            },
	{0xA0AC685F, "pfx_tlp1_tlp.emit"                   },
	{0xA0BA4A2F, "pro_sidbx_20.ncgr.small"             },
	{0xA0C149C2, "mmp_smbar_01.ncgr.small"             },
	{0xA0CBB13B, "arm_cb_idle.nsbca"                   },
	{0xA0D1AD7A, "jurnl_mis03.ncgr.small"              },
	{0xA0D2C6DD, "son_ex_sad.nsbca"                    },
	{0xA0D6BA5F, "howtosave.gda"                       },
	{0xA0E16272, "flee01_ts.tga"                       },
	{0xA0E16CB6, "icn_s_patk_amy1.ncgr.small"          },
	{0xA0E8A23E, "prtl_txcgen_1.ncgr.small"            },
	{0xA0F0D988, "prt_tp_m_she.ncgr.small"             },
	{0xA0F3DF40, "prtl_dadrel_0.ncgr.small"            },
	{0xA0FCE067, "itm_ace_rocks_1.ncgr.small"          },
	{0xA107D558, "chapter9charyb.gda"                  },
	{0xA1232583, "btn_l_bar_on00.ncgr.small"           },
	{0xA134B3FD, "pfx_briantest02.emit"                },
	{0xA14082C6, "itm_ace_shipm_1.ncgr.small"          },
	{0xA14E161B, "gui_rewardbox.nclr.small"            },
	{0xA151C343, "itm_fotlgtbt_1.ncgr.small"           },
	{0xA1625939, "chao_gd_g.nsbmd.small"               },
	{0xA1692824, "chao_jn_r.nsbtp.small"               },
	{0xA16E4A0A, "profile_bottom_summary.gui"          },
	{0xA1728F4A, "ixb_cb_attack2.nsbca"                },
	{0xA185FBEE, "prtl_shdgen_1.ncgr.small"            },
	{0xA1875B18, "itm_ace_nrgal_0.ncgr.small"          },
	{0xA18A09A4, "chao_fe_r.nsbmd.small"               },
	{0xA19D63C4, "prtl_aeqhap_0.ncgr.small"            },
	{0xA1A5269E, "prtl_fwrgen_1.ncgr.small"            },
	{0xA1A87F13, "a2c8_scy_tyxisarrival.gff"           },
	{0xA1A9922D, "sha_ex_jump.nsbca"                   },
	{0xA1B73101, "chasetrio.gda"                       },
	{0xA1BB2FC6, "pfx_bub2.nsbmd.small"                },
	{0xA1BECEA3, "chao_jl_0.ncgr.small"                },
	{0xA1C907D3, "lvl_tlgbx_ms20.ncgr.small"           },
	{0xA1CBCC26, "a2c8_scy_syraxapproach.gff"          },
	{0xA1D4EEB5, "cfx_gfir_pu.nsbtx.small"             },
	{0xA1DD5B5B, "fx_son_whirl.nsbtx.small"            },
	{0xA1DFA79C, "gui_charprofile.nclr.small"          },
	{0xA1E3D7E0, "itm_glv_midnt_0.ncgr.small"          },
	{0xA20DFE3A, "statinfos.gda"                       },
	{0xA21CB03D, "prtl_wsgsca_1.ncgr.small"            },
	{0xA21CCB2B, "ix_aa.nsbtx.small"                   },
	{0xA22E2764, "bmp_cb_pattack.nsbca"                },
	{0xA24162CD, "a1_s3_i02.walk.tga"                  },
	{0xA2441B8C, "chap5scr_bot.tga"                    },
	{0xA246AE24, "wmp_trv_08_18.gff"                   },
	{0xA2492846, "pro_bar_on01.ncgr.small"             },
	{0xA24B2A9B, "rwd_tsmbx_on01.ncgr.small"           },
	{0xA250D00C, "invent_icon_rev.ncgr.small"          },
	{0xA2589B2D, "victorygui.gui"                      },
	{0xA2671CCC, "chao_js_2.ncgr.small"                },
	{0xA26E371B, "itm_ace_staplr_1.ncgr.small"         },
	{0xA27FD7C3, "cho_ts_00.ncgr.small"                },
	{0xA285DBFA, "fx_giz_lzr_muz.nsbtp.small"          },
	{0xA2868970, "gzd_cb_dead.nsbca"                   },
	{0xA28B76AC, "gnp_cb_pattack02.nsbca"              },
	{0xA28C7B99, "big_cb_pattack02.nsbca"              },
	{0xA28EE32B, "fx_arm_whirl.nsbtx.small"            },
	{0xA2A67EB7, "chao_gc_0.ncgr.small"                },
	{0xA2AB4595, "itm_ace_radio_2.ncgr.small"          },
	{0xA2B1FEEB, "fx_blobshadow.nsbtx.small"           },
	{0xA2BA8386, "cfx_ch2.nsbtx.small"                 },
	{0xA2BB6576, "prtl_amysus_0.ncgr.small"            },
	{0xA2CDC9F5, "shp_ex_run.nsbca"                    },
	{0xA2D61A93, "a1_w3_br_11.ncgr.small"              },
	{0xA2D9C3E8, "gui_chao_a.nclr.small"               },
	{0xA2DD9D03, "egm_ex_run.nsbca"                    },
	{0xA2E4BD43, "prtl_shddef_1.ncgr.small"            },
	{0xA2E597AC, "buz_cb_idle.nsbca"                   },
	{0xA2E773D6, "cbt_info_slide.ncer.small"           },
	{0xA2E94A2C, "a1_w1_gh_11.ncgr.small"              },
	{0xA2EE163A, "cfx_dusp1_slm.emit"                  },
	{0xA2F6CBFA, "fx_els_wt_aa.nsbmd.small"            },
	{0xA2FB2710, "icn_s_larwonlvl.ncgr.small"          },
	{0xA2FF635F, "prtl_txrhum_3.ncgr.small"            },
	{0xA3078F26, "charpro_she4.ncgr.small"             },
	{0xA309705D, "youarenotdoneyet.gda"                },
	{0xA324A587, "cho_ts_22.ncgr.small"                },
	{0xA34028DE, "pl_pu_scr_ghs_bg.nsbmd.small"        },
	{0xA34156FA, "son_ex_run.nsbca"                    },
	{0xA342B3AB, "cbt_btn_pathbut.gui"                 },
	{0xA34B3DCA, "cfx_wptp1.nsbmd.small"               },
	{0xA34BC745, "fx_a1_s2_ex_env.nsbmd.small"         },
	{0xA35B4E0A, "itm_chao_egg_1.ncgr.small"           },
	{0xA35F8747, "exp_ghz1_mm31.ncgr.small"            },
	{0xA36D666F, "upper_green_ring_01.ncgr.small"      },
	{0xA37C4D8B, "howtojournal.gda"                    },
	{0xA38071AF, "a1_s5_i01b.are"                      },
	{0xA389DC9E, "pfx_briantest03.emit"                },
	{0xA3957606, "itm_ace_life_2.ncgr.small"           },
	{0xA3A12BF9, "a2_s6_ex_mm.nclr.small"              },
	{0xA3A84458, "chao_ia_1.ncgr.small"                },
	{0xA3B23951, "main_pnl_on12.ncgr.small"            },
	{0xA3B3AAC1, "wmp_space_on11.ncgr.small"           },
	{0xA3C1AB42, "a2c8_scy_queenspawnarrive1.gff"      },
	{0xA3D2DFB3, "buz_cb_getup.nsbca"                  },
	{0xA3D3C3C3, "a2c8_scy_queenspawnarrive2.gff"      },
	{0xA3D9CDF9, "cbt_rndround1.ncgr.small"            },
	{0xA3DD60C7, "kro_cb_idle.nsbca"                   },
	{0xA3E5DC44, "a2c8_scy_queenspawnarrive3.gff"      },
	{0xA3E5ECBA, "chao_aa_gard.nsbtp.small"            },
	{0xA3E83496, "gennpc1_aa.nsbtx.small"              },
	{0xA3EA33E4, "a1_s5_ex_la.smp"                     },
	{0xA3EC7C71, "a2c10_taunt_exit_1.gff"              },
	{0xA3F7F4C5, "a2c8_scy_queenspawnarrive4.gff"      },
	{0xA3F973F7, "pl_pz_pod_bat.nsbca"                 },
	{0xA3FD2DB1, "prtl_rougeoff_0.ncgr.small"          },
	{0xA3FE94F2, "a2c10_taunt_exit_2.gff"              },
	{0xA3FFB0E8, "itm_accangam_1.ncgr.small"           },
	{0xA400305F, "plt_a2sq_haniman.plo"                },
	{0xA4017C70, "btn_back_e_of.ncgr.small"            },
	{0xA40A0D46, "a2c8_scy_queenspawnarrive5.gff"      },
	{0xA40B107A, "armdilr_aa.nsbmd.small"              },
	{0xA410AD73, "a2c10_taunt_exit_3.gff"              },
	{0xA4160B02, "a2scy_warrior1.dlg"                  },
	{0xA41C25C7, "a2c8_scy_queenspawnarrive6.gff"      },
	{0xA4215CC1, "exl_smbtn_hi01.ncgr.small"           },
	{0xA4237CB0, "a2_s5_card_t1.nbfp"                  },
	{0xA4237CB3, "a2_s5_card_t1.nbfs"                  },
	{0xA4282383, "a2scy_warrior2.dlg"                  },
	{0xA4367CF4, "chao_da_2.ncgr.small"                },
	{0xA436D9B3, "prtl_omega.nclr.small"               },
	{0xA44178F2, "charpro_stat31.ncgr.small"           },
	{0xA44BD05C, "jrl_tlgbr_of02.ncgr.small"           },
	{0xA44CF610, "movement02_bs.tga"                   },
	{0xA45BA3AB, "prtl_gnggen_1.ncgr.small"            },
	{0xA4744986, "chao_da_g.nsbtp.small"               },
	{0xA474B501, "storelistbackground.gui"             },
	{0xA496B3B5, "icn_s_mm_store.ncgr.small"           },
	{0xA4A129DC, "plt_a1sq_muggers.plo"                },
	{0xA4AA1527, "cfx_aslt_mf.nsbtp.small"             },
	{0xA4ADDDFE, "pl_mv_zip.nsbmd.small"               },
	{0xA4C7DEBC, "pl_ob_blkn_so.nsbtx.small"           },
	{0xA4D7B005, "pro_medbx_11.ncgr.small"             },
	{0xA4E57A61, "mut_globe_ms00.ncgr.small"           },
	{0xA4FA6F7A, "prt_tp_e_cre.ncgr.small"             },
	{0xA5032004, "miles_test.are"                      },
	{0xA50F94C5, "cfx_fatp1.nsbtx.small"               },
	{0xA515CEFE, "pfx_arnietest01.nsbtp.small"         },
	{0xA515F72E, "man_strbt_on13.ncgr.small"           },
	{0xA51728E8, "combo4.ncgr.small"                   },
	{0xA524C830, "c_sotaamkn_1_ta2.nsbca"              },
	{0xA5392A15, "cfx_flap1.nsbtp.small"               },
	{0xA53A8FC6, "itm_ace_delux_0.ncgr.small"          },
	{0xA53EFF54, "pro_lrgbx_02.ncgr.small"             },
	{0xA5477460, "gup_ex_fall.nsbca"                   },
	{0xA54791F8, "charpro_big4.ncgr.small"             },
	{0xA55F3B58, "prtl_txmvar_2.ncgr.small"            },
	{0xA56A2A5B, "itm_ace_cural_0.ncgr.small"          },
	{0xA56B305D, "a1_w0_mt_11.ncgr.small"              },
	{0xA57D973E, "prtl_skphap_3.ncgr.small"            },
	{0xA58FA854, "pl_pz_cav_scrn.nsbtp.small"          },
	{0xA59A9500, "pl_am_met_gun.nsbtx.small"           },
	{0xA5B64161, "cbt_rndnum1.ncgr.small"              },
	{0xA5BB95BE, "jurnl_mis_line00.ncgr.small"         },
	{0xA5C4EB10, "icn_s_catk_sak.ncgr.small"           },
	{0xA5C8104C, "bfl_cb_attackc.nsbca"                },
	{0xA5DA411C, "exp_ghq1_mm11.ncgr.small"            },
	{0xA5DBAE0B, "prtl_tdadhap_1.ncgr.small"           },
	{0xA5DCFAA4, "cmbtbg_a1_i02.nsbca"                 },
	{0xA5E2D009, "cfx_weap1_glw.emit"                  },
	{0xA5EE2331, "itm_ace_beaca_1.ncgr.small"          },
	{0xA5F228E2, "itm_fotkevbt_1.ncgr.small"           },
	{0xA5F2BB51, "cbt_focusring.nanr.small"            },
	{0xA5F721E5, "leveluptechupper.gui"                },
	{0xA5FCCD54, "icn_l_larwonstr.ncgr.small"          },
	{0xA5FF02C3, "fx_wtrfl_splsh.nsbmd.small"          },
	{0xA604634B, "pl_pz_crn_turnb.nsbca"               },
	{0xA609E7CA, "tai_cb_attackend.nsbca"              },
	{0xA60B33AE, "chao_jk_r.nsbmd.small"               },
	{0xA6302F9D, "chao_ha_r.nsbtx.small"               },
	{0xA63C3FDF, "cho_tmbtn_on01.ncgr.small"           },
	{0xA63E3B30, "prtl_sbtmad_1.ncgr.small"            },
	{0xA640D0DC, "cfx_dusp2.nsbmd.small"               },
	{0xA64A1190, "guianim_jrl.gda"                     },
	{0xA64BF3EE, "a1_s3_i03.walk.tga"                  },
	{0xA6569F14, "pl_pu_emrld3.nsbmd.small"            },
	{0xA6590ECF, "sha_ex_idle.nsbca"                   },
	{0xA65B194B, "pfx_grs1.nsbtp.small"                },
	{0xA662859F, "gennpc3b_aa.nsbmd.small"             },
	{0xA664AAFD, "gzg_cb_attack.nsbca"                 },
	{0xA673507F, "a2cha_overmind_buddy4_leave.gff"     },
	{0xA673B9FC, "jurnl_pnl13.ncgr.small"              },
	{0xA6741174, "fx_noc_tele.nsbmd.small"             },
	{0xA674FACF, "prtl_place.nclr.small"               },
	{0xA6775570, "drn_cb_kd.nsbca"                     },
	{0xA678A551, "a2_s5_card_t2.nbfp"                  },
	{0xA678A554, "a2_s5_card_t2.nbfs"                  },
	{0xA6800299, "cfx_hstp1.nsbtx.small"               },
	{0xA6816011, "fx_egb_beam.nsbta.small"             },
	{0xA6847727, "prtl_vxcbadgen_1.ncgr.small"         },
	{0xA68C09DC, "fx_frcfld.nsbmd.small"               },
	{0xA69D9287, "a2c7_factory_door_left.gff"          },
	{0xA6A6E698, "prtl_bigtht_2.ncgr.small"            },
	{0xA6B4DE8B, "chao_fd_g.nsbtp.small"               },
	{0xA6C36A36, "wmp_s_bridge.ncgr.small"             },
	{0xA6C6AA86, "cbt_btn_left_on.ncgr.small"          },
	{0xA6CBCCB5, "gui_items_accessories3.nclr.small"   },
	{0xA6D2E199, "chao_fe_1.ncgr.small"                },
	{0xA6EF3B5E, "chapter7warp.gff"                    },
	{0xA6F2F1D2, "str_tab_00.ncgr.small"               },
	{0xA6F542DD, "pl_ob_blkb_mb.nsbtx.small"           },
	{0xA6F65F02, "tai_ex_walk.nsbca"                   },
	{0xA6F8A75B, "cfx_dusp2_slm.emit"                  },
	{0xA6FFD8C1, "a1c2_knuckles_bot1.gff"              },
	{0xA6FFE6D2, "a1_s3_i01_6to4.gff"                  },
	{0xA704037A, "gnp_cb_attack.nsbca"                 },
	{0xA711F142, "a1c2_knuckles_bot2.gff"              },
	{0xA7152183, "str_blank_ms00.ncgr.small"           },
	{0xA716B8BF, "fx_tyx_prj.nsbta.small"              },
	{0xA71F2094, "ambientbutton.gui"                   },
	{0xA7368742, "cfx_nocp1.nsbtp.small"               },
	{0xA7397EAA, "loc_abshd_es0.ncgr.small"            },
	{0xA7605674, "pil_aa.nsbmd.small"                  },
	{0xA767F654, "gzd_aa.nsbmd.small"                  },
	{0xA76A8950, "omg_ex_jumpair.nsbca"                },
	{0xA76F7354, "pl_pu_rngshad.nsbmd.small"           },
	{0xA77E715A, "lvl_stat_lb.ncgr.small"              },
	{0xA77EF20A, "cbt_victory_star.nanr.small"         },
	{0xA79126EF, "a2_s5_i01.walk.tga"                  },
	{0xA794DB61, "a2c10_ix.dlg"                        },
	{0xA7A056CF, "a1_w0_br_10.ncgr.small"              },
	{0xA7A7FE83, "itm_glv_spike_3.ncgr.small"          },
	{0xA7AC85CC, "pro_tab_20.ncgr.small"               },
	{0xA7AD8DA8, "levelupbonusupper.gui"               },
	{0xA7B1551D, "a1_s5_card_b1.nbfp"                  },
	{0xA7B15520, "a1_s5_card_b1.nbfs"                  },
	{0xA7BEDAED, "itm_ace_psych_2.ncgr.small"          },
	{0xA7BF9C89, "cfx_fgrp1.nsbtx.small"               },
	{0xA7C7E353, "ch6_shadesave.gda"                   },
	{0xA7C88D31, "tails_aa_p.ncgr.small"               },
	{0xA7D4A73A, "chao_hc_2.ncgr.small"                },
	{0xA7DADC47, "lvl_pow_bspnl11.ncgr.small"          },
	{0xA7F4F047, "itm_act2_chaos1_3.ncgr.small"        },
	{0xA8121E53, "pl_am_met_gup.nsbca"                 },
	{0xA8168DA8, "a2_w0_tx_10.ncgr.small"              },
	{0xA81B52F9, "swt_cb_idle.nsbca"                   },
	{0xA8241F3D, "pl_pz_pod_dr.nsbtp.small"            },
	{0xA82784EF, "itm_fotirnhy_0.ncgr.small"           },
	{0xA82ABB88, "a1_w3_ai_10.ncgr.small"              },
	{0xA833A845, "cfx_ccup1.nsbtx.small"               },
	{0xA83E2A31, "a1_w1_br_02.ncgr.small"              },
	{0xA84BD370, "a1c1_rouge_test.dlg"                 },
	{0xA8514C34, "a1c4_brz_snowmelt.gff"               },
	{0xA854FD92, "loc_trppd_es0.ncgr.small"            },
	{0xA859E02A, "levelup_tech_top.gui"                },
	{0xA864FA8C, "itm_ace_voxai_1.ncgr.small"          },
	{0xA870C4A2, "chao_jd_r.nsbtx.small"               },
	{0xA88C2772, "slg_cb_kd.nsbca"                     },
	{0xA8A2A445, "charpro_sdw.nclr.small"              },
	{0xA8A8FBC8, "icn_l_select.ncgr.small"             },
	{0xA8ACBE45, "a1c5_p6_guardian_hold.gff"           },
	{0xA8BAC6F9, "prt_se_eggbt.ncgr.small"             },
	{0xA8BFB1DB, "invent_stat_tab2.ncgr.small"         },
	{0xA8C3312D, "cfx_spk1.nsbmd.small"                },
	{0xA8C3D271, "ccz_hintguy.dlg"                     },
	{0xA8CB3F8E, "prtl_wkrrel_3.ncgr.small"            },
	{0xA8CC545C, "a1_w3_mr_00.ncgr.small"              },
	{0xA8D16DD5, "slg_walk.nsbtp.small"                },
	{0xA8D5D2F7, "pfx_mot1.nsbtx.small"                },
	{0xA8D66CDB, "chao_ja_3.ncgr.small"                },
	{0xA8D6895B, "fx_sha_rift.nsbmd.small"             },
	{0xA8DFF1AB, "itm_ace_fire_1.ncgr.small"           },
	{0xA8EC3440, "fx_vox_g_ea_s.nsbtx.small"           },
	{0xA8EEEE82, "pl_mv_cyc_prk.nsbca"                 },
	{0xA8EFAB13, "cbt_flruna_0.ncgr.small"             },
	{0xA8F4ADA2, "chapter10nocturne_p7.gff"            },
	{0xA8F7F8DA, "a2c8_scy_nrrgalsurround1.gff"        },
	{0xA8FDB8EF, "itm_fotslpnm_0.ncgr.small"           },
	{0xA9085219, "rewardlist.gui"                      },
	{0xA90A115B, "a2c8_scy_nrrgalsurround2.gff"        },
	{0xA90E851E, "gui_portraits09.nclr.small"          },
	{0xA90F088A, "tai_cb_blockd.nsbca"                 },
	{0xA9141105, "prtl_sonicmad_3.ncgr.small"          },
	{0xA91998C6, "tai_ex_twitch.nsbca"                 },
	{0xA91C29DC, "a2c8_scy_nrrgalsurround3.gff"        },
	{0xA91D84D4, "prtl_amyhap_0.ncgr.small"            },
	{0xA92300F2, "cbt_cmdslot.ncgr.small"              },
	{0xA92870F9, "itm_ace_sturd_3.ncgr.small"          },
	{0xA92E425D, "a2c8_scy_nrrgalsurround4.gff"        },
	{0xA9405ADE, "a2c8_scy_nrrgalsurround5.gff"        },
	{0xA94B4A06, "a1c1_rougewalkgun.gff"               },
	{0xA954A8C6, "a2c8_scy_messengerarrives.gff"       },
	{0xA9568CF4, "prtl_nessca_3.ncgr.small"            },
	{0xA957A90A, "charpro_bs_b02.ncgr.small"           },
	{0xA96B493D, "spell_weakened_all.spl"              },
	{0xA97804DE, "itm_ace_tablt_3.ncgr.small"          },
	{0xA98F8F4B, "itm_act2_chaos7_1.ncgr.small"        },
	{0xA99931DE, "cfx_pwup1.nsbtp.small"               },
	{0xA99D1690, "movementtutorial2.gda"               },
	{0xA9A07772, "prtl_egginq_3.ncgr.small"            },
	{0xA9AD4458, "chap3scr_top.tga"                    },
	{0xA9CA0095, "itm_glv_rock_2.ncgr.small"           },
	{0xA9DB3764, "itm_ace_lghtn_3.ncgr.small"          },
	{0xA9DE1F6A, "exploreportraitlayout.gui"           },
	{0xA9E4074E, "itm_ace_boy_0.ncgr.small"            },
	{0xAA011FDE, "prtl_shdint_0.ncgr.small"            },
	{0xAA01BB51, "prtl_rougegen_1.ncgr.small"          },
	{0xAA030787, "exp_ghz2_mm03.ncgr.small"            },
	{0xAA052CD3, "fx_vox_r_ea_r.nsbta.small"           },
	{0xAA067DBE, "a1_s5_card_b2.nbfp"                  },
	{0xAA067DC1, "a1_s5_card_b2.nbfs"                  },
	{0xAA0AC592, "itm_ace_power_1.ncgr.small"          },
	{0xAA1504D7, "pl_am_noc_gct.nsbta.small"           },
	{0xAA1BFBE7, "cfx_pjkp1.nsbtp.small"               },
	{0xAA21E485, "pig_cb_pattack.nsbca"                },
	{0xAA21F04D, "itm_glv_gold_3.ncgr.small"           },
	{0xAA33FECB, "fx_ix_bhol.nsbtx.small"              },
	{0xAA39BD54, "charpro_son2.ncgr.small"             },
	{0xAA3CDD67, "cfx_pfbp1.nsbmd.small"               },
	{0xAA46DE7A, "conv_bubble_bs10.ncgr.small"         },
	{0xAA4DAADB, "sha_cb_blockin.nsbca"                },
	{0xAA4E84B6, "icn_s_automatic.ncgr.small"          },
	{0xAA5B2754, "prtl_knumad_0.ncgr.small"            },
	{0xAA5CA73E, "charpro_amy.nclr.small"              },
	{0xAA710BDF, "chao_jg_1.ncgr.small"                },
	{0xAA7A7EE3, "fx_wtrfl_wtrfl.nsbtx.small"          },
	{0xAA89FB19, "hnpc_cop_aa.nsbmd.small"             },
	{0xAA8D9063, "pzl_craneestablish.gff"              },
	{0xAA8FE8B9, "fx_a1_s1_ex_wtrf.nsbtx.small"        },
	{0xAA93E3E3, "plt_a2sq_knicknacks.plo"             },
	{0xAA951F54, "shpkron_aa.nsbmd.small"              },
	{0xAA9B0CEA, "prtl_sonicgen_0.ncgr.small"          },
	{0xAABC486A, "fx_noc_jav_muz.nsbta.small"          },
	{0xAAC0ECEB, "icn_s_statdis.ncgr.small"            },
	{0xAAC19A72, "cfx_acdp1.nsbtx.small"               },
	{0xAADA38D9, "slg_ex_idle.nsbca"                   },
	{0xAADD63CD, "prt_se_tyxwa.ncgr.small"             },
	{0xAAE80242, "a2_s5_i01_ts.nbfp"                   },
	{0xAAE80245, "a2_s5_i01_ts.nbfs"                   },
	{0xAAEBB0D9, "egm_cb_blockout.nsbca"               },
	{0xAAF11C2A, "a1c5_p5_camera_pan_marauder.gff"     },
	{0xAB06E9E9, "prt_se_mllpd.ncgr.small"             },
	{0xAB07F238, "jurnl_ts12.ncgr.small"               },
	{0xAB0A53F1, "cmbtbg_a2_s5.nsbtx.small"            },
	{0xAB0BC98B, "str_party_ms00.ncgr.small"           },
	{0xAB0E6FBE, "gui_items_gloves.nclr.small"         },
	{0xAB10FAED, "conversations.gda"                   },
	{0xAB195A08, "chao_jn_3.ncgr.small"                },
	{0xAB265AAF, "jurnl_cod_line04.ncgr.small"         },
	{0xAB360895, "chao_jj_g.nsbtp.small"               },
	{0xAB44D579, "prt_sp_tai.ncgr.small"               },
	{0xAB4FF8AB, "prf_tboxb_ms11.ncgr.small"           },
	{0xAB50E024, "charpro_ome1.ncgr.small"             },
	{0xAB56D89D, "son_cb_victory.nsbca"                },
	{0xAB56EA15, "chao_fa_g.nsbmd.small"               },
	{0xAB685FA8, "fx_puff_layout.ncgr"                 },
	{0xAB6B6B9A, "pfx_invin.nsbtx.small"               },
	{0xAB71E211, "prtl_eixthrngen_1.ncgr.small"        },
	{0xAB71F181, "combattopscreen.gui"                 },
	{0xAB72C59F, "fx_spark.nsbtp.small"                },
	{0xAB7736EF, "pop_lgbox_ms02.ncgr.small"           },
	{0xAB791271, "cfx_slpp1.nsbtp.small"               },
	{0xAB891CE7, "cfx_chlp1_hea.emit"                  },
	{0xAB963940, "a2c10_noflyzone.gff"                 },
	{0xAB99F3F1, "cfx_shgp1.nsbmd.small"               },
	{0xAB9BB810, "a2_s5_i02.walk.tga"                  },
	{0xABA0278F, "conversation_upper.nclr.small"       },
	{0xABA5C808, "pfx_arnietest03.nsbtx.small"         },
	{0xABA65CDF, "hnpc_bcap_twit.nsbca"                },
	{0xABB437E8, "camerapankron.gff"                   },
	{0xABB5F62F, "a2c8_scy_exitship.gff"               },
	{0xABBF98D2, "pl_pz_ppd_aa.nsbtx.small"            },
	{0xABC4A4E6, "crm_cb_victoryin.nsbca"              },
	{0xABD88617, "pfx_rip1.nsbmd.small"                },
	{0xABDE40D5, "conv_bubble_amb01.ncgr.small"        },
	{0xABE158CA, "ehd_blbox_ms00.ncgr.small"           },
	{0xABE825D3, "spell1.spl"                          },
	{0xABEDCC39, "itm_glv_grip_3.ncgr.small"           },
	{0xABFA3E54, "spell2.spl"                          },
	{0xAC008517, "chaodescription.gui"                 },
	{0xAC010A1C, "chao_ge_3.ncgr.small"                },
	{0xAC0299CC, "itm_glv_fire_2.ncgr.small"           },
	{0xAC0581A5, "prtl_aeqgen_0.ncgr.small"            },
	{0xAC0C56D5, "spell3.spl"                          },
	{0xAC1E6F56, "spell4.spl"                          },
	{0xAC2586D3, "a1_s1_ex_mm.nclr.small"              },
	{0xAC3087D7, "spell5.spl"                          },
	{0xAC3D89D5, "a1_s3_ex_vent.gff"                   },
	{0xAC42A058, "spell6.spl"                          },
	{0xAC4B11BD, "itm_ace_wpnp1_3.ncgr.small"          },
	{0xAC54B8D9, "spell7.spl"                          },
	{0xAC5D7D49, "prtl_sdwsmg_0.ncgr.small"            },
	{0xAC66D15A, "spell8.spl"                          },
	{0xAC6B4F39, "cfx_scap1.nsbtp.small"               },
	{0xAC74FD8A, "icn_l_barwonstr.ncgr.small"          },
	{0xAC75EE3E, "que_ex_idle.nsbca"                   },
	{0xAC78E9DB, "spell9.spl"                          },
	{0xAC827BED, "chao_jo_r.nsbtx.small"               },
	{0xAC87E87E, "lvl_stat_tspnl10.ncgr.small"         },
	{0xAC91FCA9, "cw_ne_se.gff"                        },
	{0xAC993C61, "cfx_vulp1.nsbtx.small"               },
	{0xACA8A9F4, "ix_ex_idle.nsbca"                    },
	{0xACAD6A63, "prtl_knuk.nclr.small"                },
	{0xACAD987A, "prtl_sciexh_2.ncgr.small"            },
	{0xACB0994F, "jrl_tlgbx_ms40.ncgr.small"           },
	{0xACB3F90C, "chao_jt_1.ncgr.small"                },
	{0xACB4D889, "fx_ix_orb.nsbtx.small"               },
	{0xACD04296, "exp_cct1_mm02.ncgr.small"            },
	{0xACD4F34E, "prt_sp_egg.ncgr.small"               },
	{0xACD55916, "debugparty.gda"                      },
	{0xACDCB284, "amb_uppertextbubble.gui"             },
	{0xACE9BF5A, "fx_vox_b_ea_m.nsbmd.small"           },
	{0xAD1674EF, "ch10_ixexplains.gda"                 },
	{0xAD1DF978, "a2scy_raxos.dlg"                     },
	{0xAD1F068E, "knu_cb_blockout.nsbca"               },
	{0xAD22D94B, "icn_l_barwoncho.ncgr.small"          },
	{0xAD30790D, "fx_a1_s4_icb_bp.nsbca"               },
	{0xAD334336, "prtl_sonicamu_3.ncgr.small"          },
	{0xAD3C56E7, "exp_ghz1_fw10.ncgr.small"            },
	{0xAD47DCBB, "lower_sonic_portrait_00.ncgr.small"  },
	{0xAD4BD0AD, "bsl_cb_attackend.nsbca"              },
	{0xAD526561, "prtl_salesmg_1.ncgr.small"           },
	{0xAD5BCB1D, "plt_ch10_main.plo"                   },
	{0xAD64A35A, "a2c7_kronapproach.gff"               },
	{0xAD68081D, "itm_ace_drink_1.ncgr.small"          },
	{0xAD6B9713, "vox_cb_pattack1.nsbca"               },
	{0xAD6FA40C, "ccw_se_ne.gff"                       },
	{0xAD7C21B7, "icn_s_shipwmp.ncgr.small"            },
	{0xAD81F965, "mmp_globe_0.ncgr.small"              },
	{0xAD8348A7, "chap0scr_bot.tga"                    },
	{0xAD977F1A, "chao_hd_g.nsbmd.small"               },
	{0xADAA683E, "prtl_hanmad_1.ncgr.small"            },
	{0xADAC29A7, "omg_cb_victoryin.nsbca"              },
	{0xADBE82B4, "cfx_kipp1_hrt.emit"                  },
	{0xADBF2F85, "chao_ge_r.nsbmd.small"               },
	{0xADC38FFC, "itm_ace_jammr_1.ncgr.small"          },
	{0xADCBCAA9, "cfx_pswp1.nsbmd.small"               },
	{0xADDB840A, "jrl_tsmbr_of12.ncgr.small"           },
	{0xADE124AB, "exp_ghz1_fw32.ncgr.small"            },
	{0xADE26EAB, "plotflags.gda"                       },
	{0xADE8CE77, "jrl_tmbox_ms21.ncgr.small"           },
	{0xADEF510B, "prtl_fmnang_0.ncgr.small"            },
	{0xADF52098, "chao_ib_0.ncgr.small"                },
	{0xADFDABCA, "vom_cb_pattack3.nsbca"               },
	{0xAE177DC0, "a1_s5_ex.are"                        },
	{0xAE191B1C, "pfx_tlp1.nsbmd.small"                },
	{0xAE26C157, "icn_l_ondash2.ncgr.small"            },
	{0xAE3F0A99, "hud_lvlnum7.ncgr.small"              },
	{0xAE401726, "cfx_clup1.nsbtp.small"               },
	{0xAE492DE1, "a1_w0_mt.nclr.small"                 },
	{0xAE4C0EBD, "itm_fotrunsp_0.ncgr.small"           },
	{0xAE4F7621, "prtl_eixfm1_3.ncgr.small"            },
	{0xAE5A5747, "cmra_aim.nsbmd.small"                },
	{0xAE5E390D, "ehd_tftge_ms00.ncgr.small"           },
	{0xAE5FC6B3, "itm_ace_rejuv_3.ncgr.small"          },
	{0xAE60F8A6, "cfx_chlp1.nsbmd.small"               },
	{0xAE74DCF3, "pl_pz_brd_2hd.nsbca"                 },
	{0xAE7EAE7B, "pfx_briantest01.nsbtp.small"         },
	{0xAE8D59AD, "jillcan13bdrg.nftr"                  },
	{0xAE8F50BF, "itm_ace_leaf_3.ncgr.small"           },
	{0xAE90C91D, "storelowerbuttons.gui"               },
	{0xAE9180E0, "prtl_knudet_1.ncgr.small"            },
	{0xAE96B155, "a2cha_alphavoxai.dlg"                },
	{0xAE9A974B, "prtl_knusmg_2.ncgr.small"            },
	{0xAE9E695D, "front_jurnl_top.gui"                 },
	{0xAEA16D4A, "a2c7_kron_emerald_collect.gff"       },
	{0xAEA3827A, "exl_lrgbt_hl.ncgr.small"             },
	{0xAEA4F4AA, "prtl_krwcon_2.ncgr.small"            },
	{0xAEA797F4, "arm_ex_land.nsbca"                   },
	{0xAEB9A69E, "a2c10_gizoidshutdownatk.gff"         },
	{0xAEC8BE6A, "pl_ob_blkb_mo.nsbtx.small"           },
	{0xAED970E8, "itm_ace_noctn_2.ncgr.small"          },
	{0xAEE1A23A, "charpro_ts_a10.ncgr.small"           },
	{0xAEE809C4, "prtl_skpcon_2.ncgr.small"            },
	{0xAEE91481, "itm_fotsnkkb_3.ncgr.small"           },
	{0xAF1F85B2, "lvl_tsstr_ms11.ncgr.small"           },
	{0xAF25B211, "loc_round_es1.ncgr.small"            },
	{0xAF2BCFD8, "exl_bchao_ms0.ncgr.small"            },
	{0xAF3CF892, "tai_ex_fall.nsbca"                   },
	{0xAF4F49C4, "exp_wel1_mm11.ncgr.small"            },
	{0xAF522A00, "cho_tlgbx_ms12.ncgr.small"           },
	{0xAF54A989, "prf_smarw_of00.ncgr.small"           },
	{0xAF734922, "a1_w2_cc_00.ncgr.small"              },
	{0xAF7486B0, "prtl_shddet_0.ncgr.small"            },
	{0xAF7765A9, "invent_icon_dow.ncgr.small"          },
	{0xAF776A88, "a2_intro.dlg"                        },
	{0xAF84E7E7, "spinball.nsbta.small"                },
	{0xAF8878E2, "exp_qststar.ncgr.small"              },
	{0xAF898DD4, "son_cb_victoryin.nsbca"              },
	{0xAF99C524, "a1c2_robot_flies.gff"                },
	{0xAFA1DD35, "pfx_rip1_rip.emit"                   },
	{0xAFA2A005, "prtl_krwgen_3.ncgr.small"            },
	{0xAFA64931, "a2_s5_i03.walk.tga"                  },
	{0xAFAE76C8, "hnpc_bcap_aa.nsbtx.small"            },
	{0xAFAFD3C4, "jurnl_bs01.ncgr.small"               },
	{0xAFC3DC44, "cfx_empp1.nsbtp.small"               },
	{0xAFCAA2B9, "brz_met.gff"                         },
	{0xAFCB8711, "a2scy_tyxgate.dlg"                   },
	{0xAFD8141F, "chao_jg_g.nsbmd.small"               },
	{0xAFE5B51F, "prtl_skpgen_3.ncgr.small"            },
	{0xAFE8E949, "exl_jurnl_ms3.ncgr.small"            },
	{0xAFF20385, "exl_globe_ms1.ncgr.small"            },
	{0xAFF3940C, "c_knshashe_sha.nsbca"                },
	{0xAFF66269, "cfx_exbp1.nsbtx.small"               },
	{0xAFF946E4, "gen_ex_run.nsbca"                    },
	{0xB008EBCF, "a1_w3_gh_12.ncgr.small"              },
	{0xB01D5A77, "gennpc2_aa.nsbtx.small"              },
	{0xB0343941, "prtl_bighap_2.ncgr.small"            },
	{0xB03F3968, "mut_flash_ms11.ncgr.small"           },
	{0xB043CBEC, "prtl_tdadgen_1.ncgr.small"           },
	{0xB0583990, "a2c8_scy_nrrgalattack1.gff"          },
	{0xB05A1AB2, "fx_sha_rift_prj.nsbca"               },
	{0xB061A5F7, "c_ambicr_am.nsbca"                   },
	{0xB06A1FAE, "c_soknamta_ta.nsbca"                 },
	{0xB06A5211, "a2c8_scy_nrrgalattack2.gff"          },
	{0xB072147A, "a2c6_vaultbotattack.gff"             },
	{0xB07A0E0F, "pfx_aur1_aur.emit"                   },
	{0xB07C6A92, "a2c8_scy_nrrgalattack3.gff"          },
	{0xB08FA320, "gui_conv_arrows.nclr.small"          },
	{0xB0924272, "spells.gda"                          },
	{0xB0942D61, "fx_els_er_aa.nsbtx.small"            },
	{0xB095BFDE, "a1_s2_i01_ts.nbfp"                   },
	{0xB095BFE1, "a1_s2_i01_ts.nbfs"                   },
	{0xB097100A, "prtl_thbsad_0.ncgr.small"            },
	{0xB0A10E76, "rou_cb_blockout.nsbca"               },
	{0xB0A96F67, "chao_ea_g.nsbtp.small"               },
	{0xB0ACFB2A, "subtitle_test.gda"                   },
	{0xB0B05D12, "pl_pz_cav_snow.nsbmd.small"          },
	{0xB0B1963F, "well.are"                            },
	{0xB0B8118D, "cmbtbg_a1_s2.nsbtx.small"            },
	{0xB0C42289, "storecharactertab.gui"               },
	{0xB0C6BCF7, "itm_ace_antid_3.ncgr.small"          },
	{0xB0D720DC, "prtl_egbgen_0.ncgr.small"            },
	{0xB0D85FE4, "pfx_lzr1.nsbmd.small"                },
	{0xB0E0F1FD, "exl_shoes_ms0.ncgr.small"            },
	{0xB0E1E2D8, "fx_a2_s1_ex_env.nsbtp.small"         },
	{0xB0E84A2F, "cfx_mst1_fiz.emit"                   },
	{0xB0E92AEB, "chaseduo.gda"                        },
	{0xB0EA96E1, "w_e.gff"                             },
	{0xB0EB5B8D, "icn_s_manual.ncgr.small"             },
	{0xB0F01EEA, "cfx_crsp1.nsbtp.small"               },
	{0xB103CA07, "fx_shason_atom.nsbmd.small"          },
	{0xB10BAA60, "a2_s7_card_b1.nbfp"                  },
	{0xB10BAA63, "a2_s7_card_b1.nbfs"                  },
	{0xB117235E, "que_aa.nsbta.small"                  },
	{0xB11999D5, "statsinfocreature.gda"               },
	{0xB11FBDD9, "chao_ff_0.ncgr.small"                },
	{0xB121874A, "main_pnl_big16.ncgr.small"           },
	{0xB14AB593, "charpro_knu3.ncgr.small"             },
	{0xB14C4F71, "nrg_ex_run.nsbca"                    },
	{0xB14C6DB7, "fx_vox_y_es.nsbmd.small"             },
	{0xB14FB39D, "lower_bg_09.ncgr.small"              },
	{0xB14FEB82, "sw_ne.gff"                           },
	{0xB157D577, "journalgui_bs_blur.tga"              },
	{0xB15C075F, "invent_icon_boo.ncgr.small"          },
	{0xB1601E99, "lvl_stat_but_auto01lit.ncgr.small"   },
	{0xB1619AAA, "fx_bubbles.nsbmd.small"              },
	{0xB16D5407, "prtl_fwrrel_1.ncgr.small"            },
	{0xB18747CC, "fx_sha_rift_imp.nsbca"               },
	{0xB18B939C, "upper_bg_05.ncgr.small"              },
	{0xB18E1F26, "exl_altet_ms10.ncgr.small"           },
	{0xB1A0D61B, "cfx_fwtp1.nsbtx.small"               },
	{0xB1A403C5, "icn_s_patk_rou1.ncgr.small"          },
	{0xB1BBB2A1, "prtl_wkrsca_2.ncgr.small"            },
	{0xB1C43472, "kdr_ex_jump.nsbca"                   },
	{0xB1C8D788, "spell_medibot_timed_2.spl"           },
	{0xB1CE3254, "chao_fe_g.nsbtx.small"               },
	{0xB1D5CD14, "realtime_tutorial_bg_invert.tga"     },
	{0xB1D82149, "itm_ace_kevlr_1.ncgr.small"          },
	{0xB1DAB7F0, "prtl_agzgen_0.ncgr.small"            },
	{0xB1DAF009, "spell_medibot_timed_3.spl"           },
	{0xB1E8761A, "man_tsvst_of12.ncgr.small"           },
	{0xB1F30FE8, "scrollbg01.tga"                      },
	{0xB1FEA435, "cfx_fltp1.nsbmd.small"               },
	{0xB1FEA6D0, "prtl_amymad_3.ncgr.small"            },
	{0xB202C51E, "icn_l_barwonrwd.ncgr.small"          },
	{0xB20D155B, "icn_l_larwofjrl.ncgr.small"          },
	{0xB215078D, "a1c8_swatbot_attack.gff"             },
	{0xB21C4C68, "mut_circl_ms12.ncgr.small"           },
	{0xB21CCFD0, "genl_walkmesh.tga"                   },
	{0xB21DA561, "a2c7_tailstalkship2.gff"             },
	{0xB21E9093, "hnpc_gunco_aa.nsbmd.small"           },
	{0xB21F9D28, "fx_vox_p_ea_m.nsbmd.small"           },
	{0xB221837A, "chao_hd_1.ncgr.small"                },
	{0xB22C4E95, "store_2_bg.tga"                      },
	{0xB22D416D, "gui_tutorial_ts.tga"                 },
	{0xB22FBDE2, "a2c7_tailstalkship3.gff"             },
	{0xB235E721, "a1_s5_ex.walk.tga"                   },
	{0xB2370ADA, "a2c10_ix_runn_battle.gff"            },
	{0xB2403727, "tyxis_arena_gentleman2_teleport.gff" },
	{0xB241CC87, "itm_act2_chaos2_2.ncgr.small"        },
	{0xB245320D, "icn_s_patkknu.ncgr.small"            },
	{0xB24EC2E4, "prtl_shdsht_2.ncgr.small"            },
	{0xB24F01F3, "amy_cb_attackend.nsbca"              },
	{0xB25F1B96, "prt_tp_m_knu.ncgr.small"             },
	{0xB265557E, "chao_ia_r.nsbtx.small"               },
	{0xB278B74C, "prtl_credet_1.ncgr.small"            },
	{0xB2829A72, "itm_accnewsp_2.ncgr.small"           },
	{0xB285DB68, "prt_ce_org.ncgr.small"               },
	{0xB2904C42, "a2sq_nrrgspawn.dlg"                  },
	{0xB299E38C, "lvl_but_back.ncgr.small"             },
	{0xB29A962C, "ixa_aa.nsbtx.small"                  },
	{0xB2AAF3FD, "lvl_tdbld_ms12.ncgr.small"           },
	{0xB2BCA8BF, "exp_topframe1.ncgr.small"            },
	{0xB2C0CE46, "prtl_aeqsec_0.ncgr.small"            },
	{0xB2C46659, "agile_tutorial01_bs.tga"             },
	{0xB2CB25BF, "cfx_srkp1.nsbmd.small"               },
	{0xB2E16404, "knu_cb_rund.nsbca"                   },
	{0xB2EA046C, "chao_gd_g.nsbtp.small"               },
	{0xB2EE58F6, "itm_ace_tools_1.ncgr.small"          },
	{0xB2F6E595, "mut_sonic_ms01.ncgr.small"           },
	{0xB2F8AE2D, "crm_cb_dead.nsbca"                   },
	{0xB311B4D7, "chao_fe_r.nsbtp.small"               },
	{0xB3125E91, "jurnl_arrowcodb.ncgr.small"          },
	{0xB31E801A, "fx_vox_b_ea_r.nsbtx.small"           },
	{0xB323491B, "chao_jb_2.ncgr.small"                },
	{0xB32561B8, "prtl_gizgen_0.ncgr.small"            },
	{0xB32847BF, "a2c10_sonichold.gff"                 },
	{0xB3395C95, "buzwasn_ab.nsbtx.small"              },
	{0xB342DAF9, "pfx_bub2.nsbtp.small"                },
	{0xB34E473E, "a2_w0_nt_01.ncgr.small"              },
	{0xB35EFAA9, "prt_se_wasn.ncgr.small"              },
	{0xB360D301, "a2_s7_card_b2.nbfp"                  },
	{0xB360D304, "a2_s7_card_b2.nbfs"                  },
	{0xB36697DE, "lvl_txbar_di00.ncgr.small"           },
	{0xB3717C4E, "a2c10_scylla_attacks.gff"            },
	{0xB3718CFC, "itm_fotswtbt_3.ncgr.small"           },
	{0xB377A355, "prtl_csxmad_1.ncgr.small"            },
	{0xB37C1DB1, "cfx_ltgp1.nsbtx.small"               },
	{0xB381A022, "pfx_nicktest02.nsbmd.small"          },
	{0xB385A2B5, "prtl_amygen_0.ncgr.small"            },
	{0xB3A4854A, "charpro_bs_c01.ncgr.small"           },
	{0xB3A4B759, "crm_ex_flyin.nsbca"                  },
	{0xB3A64EFE, "pro_smbox_11.ncgr.small"             },
	{0xB3AFBACE, "puzzlemode.gda"                      },
	{0xB3B00566, "a2_s4_ex_la_soundmap.gda"            },
	{0xB3B07D6C, "icn_s_patk_sha2.ncgr.small"          },
	{0xB3B0DA52, "a2_s5_i04.walk.tga"                  },
	{0xB3B2DDBA, "exl_sonic_ms3.ncgr.small"            },
	{0xB3C6DA23, "icn_l_utdash2.ncgr.small"            },
	{0xB3CB7436, "prtl_timsca_1.ncgr.small"            },
	{0xB3CD6748, "exp_playerframe.ncgr.small"          },
	{0xB3D0DE2E, "crm_cb_kd.nsbca"                     },
	{0xB3DCCA5C, "fx_ome_missile.nsbtx.small"          },
	{0xB3E04CA5, "itm_ace_commu_0.ncgr.small"          },
	{0xB3E9CB6A, "chao_jr_g.nsbmd.small"               },
	{0xB407F27F, "vox_cb_getup.nsbca"                  },
	{0xB4121617, "ssb_aa.nsbmd.small"                  },
	{0xB4163223, "charpro_tai3.ncgr.small"             },
	{0xB41A3872, "bmp_ex_idle.nsbca"                   },
	{0xB41B9B23, "levelup1_ts.tga"                     },
	{0xB41BDE61, "pop_tggle_on00.ncgr.small"           },
	{0xB4255118, "tyx_ex_idle.nsbca"                   },
	{0xB450B914, "lvl_tlgbx_ms12.ncgr.small"           },
	{0xB459B8B9, "cmbtbg_a1_s5.nsbta.small"            },
	{0xB4643529, "cfx_flmp1.nsbtx.small"               },
	{0xB4698AA2, "prt_se_pylon.ncgr.small"             },
	{0xB48E9872, "gui_tutorial_out_bs.tga"             },
	{0xB48F8221, "wmp_s_ceggb.ncgr.small"              },
	{0xB4963D98, "fx_tal_mbot.nsbtx.small"             },
	{0xB4A3132E, "charpro_amy5.ncgr.small"             },
	{0xB4BDE81F, "chao_jh_0.ncgr.small"                },
	{0xB4BFF872, "a1_w2_br_11.ncgr.small"              },
	{0xB4C496D5, "cbt_buttonright_00.ncgr.small"       },
	{0xB4C84C1A, "lvl_pow_tspnl21.ncgr.small"          },
	{0xB4CCA220, "fx_exclamation.nsbmd.small"          },
	{0xB4D2E8FD, "cfx_wptp1.nsbtp.small"               },
	{0xB4D3280B, "a1_w0_gh_11.ncgr.small"              },
	{0xB4D37278, "fx_a1_s2_ex_env.nsbtp.small"         },
	{0xB4D98574, "a1c5_p6_repaircannon.gff"            },
	{0xB4DACCD0, "cfx_spt_acd.nsbta.small"             },
	{0xB4EF144F, "icn_l_utfly3.ncgr.small"             },
	{0xB4F0C95A, "spell_reverse_engineer_2.spl"        },
	{0xB4F1B5B1, "pfx_fls1.nsbmd.small"                },
	{0xB4F3B542, "fx_a1_s4_ex_env.nsbtx.small"         },
	{0xB4F45960, "fx_a2_s3_ex_nrg.nsbtx.small"         },
	{0xB4FBB84C, "prtl_thbovr_3.ncgr.small"            },
	{0xB502E1DB, "spell_reverse_engineer_3.spl"        },
	{0xB50EA785, "pfx_briantest03.nsbtx.small"         },
	{0xB51248F0, "icn_s_catk_kt.ncgr.small"            },
	{0xB522FE00, "wmp_l_airpo_02b.ncgr.small"          },
	{0xB5379628, "genroun_aa.nsbtx.small"              },
	{0xB541120B, "lvl_tmstr_ms10.ncgr.small"           },
	{0xB5426AC4, "exp_ghz1_mm01.ncgr.small"            },
	{0xB543C3A0, "swa_cb_attack.nsbca"                 },
	{0xB547781A, "a1c1_swatboy_camera_pan.gff"         },
	{0xB54B4D34, "exl_circe_ms30.ncgr.small"           },
	{0xB551AFB6, "crm_ex_damage.nsbca"                 },
	{0xB564D68F, "realtime_tutorial_follow_bg.tga"     },
	{0xB5663648, "chao_jo_2.ncgr.small"                },
	{0xB57F54DB, "armdiln_aa_p.ncgr.small"             },
	{0xB58B09D8, "a2_s5_ex_mm.nclr.small"              },
	{0xB5A0E8C1, "cfx_ficp1.nsbmd.small"               },
	{0xB5A7F4B8, "prtl_sonicimp_2.ncgr.small"          },
	{0xB5AAB70C, "main_pnl_off00.ncgr.small"           },
	{0xB5AC594D, "a2c8_scy_gateguardattack.gff"        },
	{0xB5AF5D9B, "str_tmdbx_ms01.ncgr.small"           },
	{0xB5B43525, "cho_lgbtn_hi01.ncgr.small"           },
	{0xB5B61D29, "itm_accrfrsh_1.ncgr.small"           },
	{0xB5B75D97, "guis.gda"                            },
	{0xB5B75FAF, "icn_s_inv3.ncgr.small"               },
	{0xB5B9B9EC, "arm_cb_attackend.nsbca"              },
	{0xB5C41411, "amy_cb_pattack05.nsbca"              },
	{0xB5C49880, "itm_ace_grapp_2.ncgr.small"          },
	{0xB5DAB378, "shp_cb_blockhit.nsbca"               },
	{0xB5DCBDA6, "wmp_s_chaob.ncgr.small"              },
	{0xB5E70266, "exp_ghz2_mm20.ncgr.small"            },
	{0xB5E73888, "exp_ghz1_mm23.ncgr.small"            },
	{0xB5F4360F, "a2c6_generatorquake.gff"             },
	{0xB5F53A71, "fx_drn_lzr_prj.nsbmd.small"          },
	{0xB60143F0, "conv_portr_10.ncgr.small"            },
	{0xB6079EEF, "pfx_lzr1_lzr.emit"                   },
	{0xB60914B2, "prtl_tailscon_1.ncgr.small"          },
	{0xB60A103E, "pl_ob_blkn_mr.nsbmd.small"           },
	{0xB60D5080, "gzd_cb_idle.nsbca"                   },
	{0xB61C806F, "c_shaom_sha.nsbca"                   },
	{0xB6245C6F, "charpro_stat01.ncgr.small"           },
	{0xB6265B95, "gentain_aa.nsbmd.small"              },
	{0xB639EA92, "main_pnl_on04.ncgr.small"            },
	{0xB63C21D0, "cfx_ch7.nsbmd.small"                 },
	{0xB63EB6A6, "cbt_player_ambush.ncer.small"        },
	{0xB6457E08, "jrl_check_00.ncgr.small"             },
	{0xB6480C2A, "str_tsmbx_ms01.ncgr.small"           },
	{0xB6489D1B, "pfx_spk1_spk.emit"                   },
	{0xB64B8432, "a2c7_kronciv.dlg"                    },
	{0xB64EF351, "prtl_guncodis_1.ncgr.small"          },
	{0xB64F4B80, "fx_prjpulse.nsbmd.small"             },
	{0xB64F5C5E, "chao_jk_g.nsbtx.small"               },
	{0xB6561AD5, "cfx_mexp2.nsbtx.small"               },
	{0xB658E606, "cfx_frgy.nsbmd.small"                },
	{0xB6609F00, "cfx_basp1.nsbtx.small"               },
	{0xB6650624, "prtl_nesinq_2.ncgr.small"            },
	{0xB673B114, "kdr_ex_idle.nsbca"                   },
	{0xB6741E9B, "itm_ace_medkt_2.ncgr.small"          },
	{0xB67D0751, "cfx_mas_spkp1.emit"                  },
	{0xB6833AC4, "a2sq_stalker2.gff"                   },
	{0xB6910AB4, "knuckles_aa_p.ncgr.small"            },
	{0xB69653D3, "itm_accearng_3.ncgr.small"           },
	{0xB697EDFD, "itm_ace_wpnp2_2.ncgr.small"          },
	{0xB6A36C10, "cfx_ch5_spkp1.emit"                  },
	{0xB6A6B0CC, "prt_tp_che.ncgr.small"               },
	{0xB6AE9021, "nrg_cb_attacken2.nsbca"              },
	{0xB6B033FD, "str_trigl_00.ncgr.small"             },
	{0xB6B6AD37, "charpro_egg2.ncgr.small"             },
	{0xB6BC3D82, "tai_cb_victoryin.nsbca"              },
	{0xB6C6756F, "cbt_focus_rpdbg.nanr.small"          },
	{0xB6CE7AD6, "exp_ringfrm1.ncgr.small"             },
	{0xB6ECD9EE, "str_tsmbx_ms23.ncgr.small"           },
	{0xB6FA6AD5, "itm_ace_ice_0.ncgr.small"            },
	{0xB6FD0A37, "rax_cb_attackend.nsbca"              },
	{0xB6FE92B2, "cbt_cmdupon.ncgr.small"              },
	{0xB706C00D, "prtl_tailsgen_2.ncgr.small"          },
	{0xB7083EDB, "buz_cb_kdidle.nsbca"                 },
	{0xB70B70C9, "a2_s2_ex_puzzle25.gff"               },
	{0xB71C9E9D, "kdr_cb_damage.nsbca"                 },
	{0xB71E3F7E, "cfx_egip1_lgt.emit"                  },
	{0xB722962C, "a2_s1_card_t1.nbfp"                  },
	{0xB722962F, "a2_s1_card_t1.nbfs"                  },
	{0xB728FB50, "cfx_nrgp1_glt.emit"                  },
	{0xB7380529, "cfx_gyzp1.nsbmd.small"               },
	{0xB73A1BC6, "icn_s_patk_big6.ncgr.small"          },
	{0xB74D026D, "itm_glvttncl_3.ncgr.small"           },
	{0xB74D928D, "pl_pu_cao_egg.nsbtx.small"           },
	{0xB74FABFD, "chao_id_3.ncgr.small"                },
	{0xB7541E27, "itm_glv_work_0.ncgr.small"           },
	{0xB754C223, "itm_ace_gunof_0.ncgr.small"          },
	{0xB759795E, "cfx_gfir_pu.nsbta.small"             },
	{0xB761E604, "fx_son_whirl.nsbta.small"            },
	{0xB761E7F0, "gnp_cb_dead.nsbca"                   },
	{0xB76360D6, "a1c5_p5_captainsonichold.gff"        },
	{0xB781712E, "gui_fleetext.nclr.small"             },
	{0xB786ADF6, "fx_wtrfl_splsh.nsbtp.small"          },
	{0xB78C0FF6, "chao_ga_g.nsbmd.small"               },
	{0xB792DEE1, "chao_jk_r.nsbtp.small"               },
	{0xB794154B, "prf_tpanl_ms20.ncgr.small"           },
	{0xB798AB09, "crm_cb_pattack02.nsbca"              },
	{0xB7A19506, "a1_s5_ex_la_soundmap.gda"            },
	{0xB7B3C061, "chao_fb_r.nsbmd.small"               },
	{0xB7BB6B73, "a2_s5_i05.walk.tga"                  },
	{0xB7C87C0F, "cfx_dusp2.nsbtp.small"               },
	{0xB7D346D6, "nrg_aa.nsbmd.small"                  },
	{0xB7D46370, "spell_shieldbot_2.spl"               },
	{0xB7D9FBFD, "a2c10_congress_enter.gff"            },
	{0xB7DB3B91, "charpro_sdw1.ncgr.small"             },
	{0xB7DD464A, "tyx_cb_pattack.nsbca"                },
	{0xB7E1A96E, "exu_ylbar_ms00.ncgr.small"           },
	{0xB7E67BF1, "spell_shieldbot_3.spl"               },
	{0xB7E720D5, "pfx_nicktest01.emit"                 },
	{0xB7EF77A3, "swa_cb_attackc.nsbca"                },
	{0xB7F95C7E, "prtl_rougesmg_1.ncgr.small"          },
	{0xB7FAA8A6, "pl_pz_pod_bat.nsbtx.small"           },
	{0xB7FEBAF6, "itm_glv_ice_1.ncgr.small"            },
	{0xB80B212F, "movement01_bs.tga"                   },
	{0xB81364B0, "prtl_salesca_0.ncgr.small"           },
	{0xB8136DD4, "fx_arm_whirl.nsbta.small"            },
	{0xB81672C0, "itm_accspkam_1.ncgr.small"           },
	{0xB81ACE37, "prtl_outhos_1.ncgr.small"            },
	{0xB81EA0E1, "icn_l_wmp_airport.nclr.small"        },
	{0xB81FC5C1, "prtl_caphap_0.ncgr.small"            },
	{0xB826AE2F, "conv_bubble_ts13.ncgr.small"         },
	{0xB8331510, "cfx_rckp2.nsbmd.small"               },
	{0xB8381CBD, "spell_immunity.spl"                  },
	{0xB83D0FEA, "cho_tsmbx_ms00.ncgr.small"           },
	{0xB841D6D1, "itm_accwnrng_2.ncgr.small"           },
	{0xB8466924, "fx_vox_r_ea_l.nsbtx.small"           },
	{0xB84B2CBC, "prtl_vxcliggen_1.ncgr.small"         },
	{0xB8510ABB, "exl_arwut_on.ncgr.small"             },
	{0xB8545DE8, "fx_vox_p_ea_r.nsbtx.small"           },
	{0xB85D6AD4, "a2_w0_vb_10.ncgr.small"              },
	{0xB861F25D, "exp_ghq1_mm03.ncgr.small"            },
	{0xB872CE1C, "cbt_cmdframe01.ncgr.small"           },
	{0xB873CDAD, "crm_cb_blockc.nsbca"                 },
	{0xB87434A9, "itm_fotrunkb_2.ncgr.small"           },
	{0xB87D5DC4, "itm_ace_pring_0.ncgr.small"          },
	{0xB8807990, "pl_mv_wnc_bp.nsbca"                  },
	{0xB8854591, "icn_l_barwofjrl.ncgr.small"          },
	{0xB892AE17, "prtl_sonicsmg_0.ncgr.small"          },
	{0xB89AA806, "cfx_tabp1.nsbmd.small"               },
	{0xB8A4942B, "pl_pu_cha_egg.nsbmd.small"           },
	{0xB8A82795, "a1c2_approachgas.gff"                },
	{0xB8AFDD1D, "ringsitems.gui"                      },
	{0xB8B3B9A6, "exl_smbtn_of11.ncgr.small"           },
	{0xB8B94D83, "pl_pz_tyx_ptn.nsbtx.small"           },
	{0xB8C316E6, "knu_cb_attackc.nsbca"                },
	{0xB8C7456F, "a1sq_waspguy.dlg"                    },
	{0xB8C79693, "ssf_tlgbx_ms31.ncgr.small"           },
	{0xB8D7C707, "vox_aa.nsbtx.small"                  },
	{0xB8DFA10F, "prtl_vecgen_3.ncgr.small"            },
	{0xB8E1F935, "charpro_stat_tab1.ncgr.small"        },
	{0xB8F4C1ED, "son_whirlwind.gff"                   },
	{0xB8F71E87, "pl_pu_rngshad.nsbtp.small"           },
	{0xB915BC37, "wmp_airp_of00.ncgr.small"            },
	{0xB9195854, "spell_big_regen_2.spl"               },
	{0xB91A5DE9, "itm_glv_dark_3.ncgr.small"           },
	{0xB91FAB46, "nrg_ab_slm.nsbta.small"              },
	{0xB9236CAF, "cbt_follow_button.nanr.small"        },
	{0xB92B41D0, "cfx_crtp1_fls.emit"                  },
	{0xB92B70D5, "spell_big_regen_3.spl"               },
	{0xB92B724E, "roundstart.gui"                      },
	{0xB939FB53, "cfx_lucp1.nsbmd.small"               },
	{0xB9407B3B, "exl_circl_ms30.ncgr.small"           },
	{0xB9431323, "pfx_ppr1_ppr.emit"                   },
	{0xB96B9BBE, "cfx_shcp1_shc.emit"                  },
	{0xB977BECD, "a2_s1_card_t2.nbfp"                  },
	{0xB977BED0, "a2_s1_card_t2.nbfs"                  },
	{0xB97E37B4, "hud_lvlnum2.ncgr.small"              },
	{0xB9801F6A, "conv_icon08.ncgr.small"              },
	{0xB9802FCA, "omg_cb_pattack02.nsbca"              },
	{0xB983BD9F, "pl_am_rdr_aa.nsbmd.small"            },
	{0xB989AD8A, "jurnl_mis_linelit03.ncgr.small"      },
	{0xB98DB3CD, "a2_s4_i01.are"                       },
	{0xB99AB1F1, "ix_cb_rise.nsbca"                    },
	{0xB9A0ABC5, "cbt_enemyframe.ncgr.small"           },
	{0xB9A32CA5, "son_cb_attack.nsbca"                 },
	{0xB9AB407A, "cbt_pathdotanim.nanr.small"          },
	{0xB9B1E4CF, "a2_s4_i03.are"                       },
	{0xB9C3FD50, "a2_s4_i04.are"                       },
	{0xB9CCA4FB, "chao_id_g.nsbmd.small"               },
	{0xB9D1FB15, "chao_fa_1.ncgr.small"                },
	{0xB9D615D1, "a2_s4_i05.are"                       },
	{0xB9E82E52, "a2_s4_i06.are"                       },
	{0xB9E96F0E, "a1c2_bigentrance.gff"                },
	{0xB9ED39BF, "fx_noc_beam.nsbmd.small"             },
	{0xB9F45566, "chao_he_r.nsbmd.small"               },
	{0xB9FA46D3, "a2_s4_i07.are"                       },
	{0xB9FFE785, "ixb_cb_getup.nsbca"                  },
	{0xBA0C5F54, "a2_s4_i08.are"                       },
	{0xBA149967, "a1_w2_ai_10.ncgr.small"              },
	{0xBA280810, "a1_w0_br_02.ncgr.small"              },
	{0xBA36251F, "combatai.gda"                        },
	{0xBA3C4976, "pfx_nicktest02.emit"                 },
	{0xBA4ADC60, "cfx_spk1.nsbtp.small"                },
	{0xBA6033B1, "chao_aa_2.ncgr.small"                },
	{0xBA628D88, "lvl_pow_bspnl03.ncgr.small"          },
	{0xBA6AE311, "rou_cb_support01.nsbca"              },
	{0xBA6C0BDA, "str_tsmbr_of02.ncgr.small"           },
	{0xBA7CD982, "vox_ria_aa.nsbtx.small"              },
	{0xBA90406B, "a2_s1_i01.walk.tga"                  },
	{0xBA9244B5, "loc_round_it0.ncgr.small"            },
	{0xBA99E0EF, "a1_s3_i01_5to2.gff"                  },
	{0xBA9C5722, "prtl_biggen_2.ncgr.small"            },
	{0xBA9E61B3, "a2c10_turret_removed.gff"            },
	{0xBAAB0931, "a2sq_voxnoct.dlg"                    },
	{0xBAB06E99, "a1_s1_card_b1.nbfp"                  },
	{0xBAB06E9C, "a1_s1_card_b1.nbfs"                  },
	{0xBAB6323B, "a1_w2_mr_00.ncgr.small"              },
	{0xBAC7FA27, "prt_se_chary.ncgr.small"             },
	{0xBAC9EC03, "egb_cb_attack.nsbca"                 },
	{0xBAD2546C, "plt_cod_gun.plo"                     },
	{0xBAF422D5, "she_sta.nsbmd.small"                 },
	{0xBAF5DC7B, "debugloadslots.gda"                  },
	{0xBB0D8AB6, "pl_pz_crn_aa.nsbmd.small"            },
	{0xBB194488, "a1_s6_ex_mm.nbfp"                    },
	{0xBB19448B, "a1_s6_ex_mm.nbfs"                    },
	{0xBB24EB14, "prt_se_boar.ncgr.small"              },
	{0xBB2BA876, "prtl_boysca_1.ncgr.small"            },
	{0xBB39D7D5, "swa_ex_run.nsbca"                    },
	{0xBB3FD31C, "a2c8_scy_queenspawnvfx.gff"          },
	{0xBB441CCA, "itm_fotsnkrg_0.ncgr.small"           },
	{0xBB45F5D8, "gui_jurnl.nclr.small"                },
	{0xBB4A420C, "vox_the_aa.nsbmd.small"              },
	{0xBB579D93, "cmbtbg_ts_ghz101.ncgr.small"         },
	{0xBB584E8A, "loc_abshd_dt0.ncgr.small"            },
	{0xBB5D93F7, "son_cb_pattack02.nsbca"              },
	{0xBB830303, "cfx_strp1.nsbtx.small"               },
	{0xBBB3B6F7, "min_circl_ms31.ncgr.small"           },
	{0xBBB4A8AA, "plt_charyb_main.plo"                 },
	{0xBBC4889A, "cfx_pfbp1.nsbtp.small"               },
	{0xBBC4BBCE, "itm_fotrnfhy_2.ncgr.small"           },
	{0xBBC78113, "a2c10_sewer_redirect.gff"            },
	{0xBBD92651, "combatguilayout.gui"                 },
	{0xBBF7F3CB, "prt_sp_son.ncgr.small"               },
	{0xBBFC6B57, "cmbtbg_ts_ghz123.ncgr.small"         },
	{0xBC0616BA, "vci_cb_attackc.nsbca"                },
	{0xBC0972B3, "cfx_crtp1.nsbtx.small"               },
	{0xBC1D1B5F, "charpro_stat_tablit2.ncgr.small"     },
	{0xBC2CFD42, "leveluptechlower.gui"                },
	{0xBC31938F, "itm_fotslpgr_2.ncgr.small"           },
	{0xBC34EA6B, "chao_jh_r.nsbmd.small"               },
	{0xBC4AB3A3, "ehd_fow_ou.ncgr.small"               },
	{0xBC528058, "gennpc3_aa.nsbtx.small"              },
	{0xBC56FE3D, "son_ex_jumpair.nsbca"                },
	{0xBC60B465, "main_pnl_big11.ncgr.small"           },
	{0xBC6E5FBA, "chao_he_0.ncgr.small"                },
	{0xBC73CD72, "loc_trppd_dt0.ncgr.small"            },
	{0xBC7D7738, "exp_tws1_mm00.ncgr.small"            },
	{0xBC843C90, "shp_cb_pattack.nsbca"                },
	{0xBC8AFD57, "egm_cb_stunned.nsbca"                },
	{0xBC8EA8C7, "itm_act2_chaos3_1.ncgr.small"        },
	{0xBC8EE0B8, "lower_bg_04.ncgr.small"              },
	{0xBC917217, "pfx_nicktest03.emit"                 },
	{0xBC99F8C0, "pfx_mst1.nsbmd.small"                },
	{0xBCBDDAE9, "a1_s2_ex_pipe_up.gff"                },
	{0xBCBF26FB, "prt_se_stdrn.ncgr.small"             },
	{0xBCC7AA09, "fx_vox_p_es.nsbtx.small"             },
	{0xBCCA173D, "pl_pz_vox_ppt.nsbtx.small"           },
	{0xBCCAC0B7, "upper_bg_00.ncgr.small"              },
	{0xBCCAE5BF, "a2_s2_i01_ts.nbfp"                   },
	{0xBCCAE5C2, "a2_s2_i01_ts.nbfs"                   },
	{0xBCCE8FBB, "conv_bubble_bs02.ncgr.small"         },
	{0xBCDCEB72, "icn_s_taphrmin.ncgr.small"           },
	{0xBCDE9548, "chao_fa_g.nsbtp.small"               },
	{0xBCEC8EF2, "a2c10_emeraldfreed_end.gff"          },
	{0xBCED376E, "cmbtbg_a2_s2.nsbtx.small"            },
	{0xBD07FBA1, "combatportraitlayout.gui"            },
	{0xBD152447, "pfx_gas1.nsbmd.small"                },
	{0xBD219F24, "cfx_shgp1.nsbtp.small"               },
	{0xBD2244FC, "exp_tws1_mm22.ncgr.small"            },
	{0xBD2C4596, "invent_list12.ncgr.small"            },
	{0xBD42CD06, "prtl_cresca_1.ncgr.small"            },
	{0xBD4369F2, "icn_s_patk_ome3.ncgr.small"          },
	{0xBD4CEC17, "spell_medibot_timed.spl"             },
	{0xBD60314A, "pfx_rip1.nsbtp.small"                },
	{0xBD64DD6D, "egb_cb_blockin.nsbca"                },
	{0xBD70255B, "chao_jc_1.ncgr.small"                },
	{0xBD786DDC, "amy_cb_attack.nsbca"                 },
	{0xBD83F17F, "rhk_ex_glide.nsbca"                  },
	{0xBD9F0560, "crm_cb_stunned.nsbca"                },
	{0xBDAC6710, "cfx_javp1.nsbmd.small"               },
	{0xBDAF84B6, "pl_pz_mem_tmr.nsbtp.small"           },
	{0xBDB07027, "a2_s2_ex_puzzle50.gff"               },
	{0xBDB1A614, "a1c5_ix.dlg"                         },
	{0xBDB1E7A1, "a2c8_tyxisarenadone.gff"             },
	{0xBDBD2998, "icn_l_mm_tornado.ncgr.small"         },
	{0xBDC39BD0, "itm_fotwrkbt_1.ncgr.small"           },
	{0xBDC5C03C, "wmp_l_airpo_00.ncgr.small"           },
	{0xBDD8A1F7, "cfx_rfpp1.nsbmd.small"               },
	{0xBDE36905, "levelupbonuslower.gui"               },
	{0xBDF6595C, "chapter5warp.gff"                    },
	{0xBE035835, "chao_ge_g.nsbtx.small"               },
	{0xBE03F66D, "pl_pz_brd_3up.nsbca"                 },
	{0xBE0E1A13, "pl_mov_arw_yw.nsbmd.small"           },
	{0xBE0EBB05, "btn_s_rtarw_on.ncgr.small"           },
	{0xBE187384, "chao_jj_3.ncgr.small"                },
	{0xBE2EBD2D, "itm_glv_rub_0.ncgr.small"            },
	{0xBE3A27C6, "pl_mov_arw_in.nsbca"                 },
	{0xBE3E5129, "sha_cb_victory.nsbca"                },
	{0xBE44C4E2, "a1sq_laddie.dlg"                     },
	{0xBE6B0D02, "a2_s5_ex.walk.tga"                   },
	{0xBE6E4351, "pl_pu_emrld5.nsbtx.small"            },
	{0xBE70BEE9, "fx_vox_g_ea_s.nsbta.small"           },
	{0xBE70E35C, "pfx_mst1_pnk.emit"                   },
	{0xBE734F92, "pfx_drp1.nsbmd.small"                },
	{0xBE82D90D, "kdr_cb_attackend.nsbca"              },
	{0xBE83E8C5, "prtl_shdtht_2.ncgr.small"            },
	{0xBE8D6EC6, "itm_glv_spark_1.ncgr.small"          },
	{0xBE937CCC, "jrl_tlgbx_ms10.ncgr.small"           },
	{0xBE9A7B5F, "chao_ja_r.nsbtx.small"               },
	{0xBE9AD18C, "a2_s1_i02.walk.tga"                  },
	{0xBE9B2CCB, "cfx_lucp1_clv.emit"                  },
	{0xBE9B6A4A, "vom_cb_attackc.nsbca"                },
	{0xBEAC0960, "a1_s1_i04_ts.nbfp"                   },
	{0xBEAC0963, "a1_s1_i04_ts.nbfs"                   },
	{0xBEACE3C7, "itm_glv_crystal0.ncgr.small"         },
	{0xBECCAE69, "realtimewidgets.gda"                 },
	{0xBECD49FB, "fleeobstacles.gda"                   },
	{0xBECFBC0D, "ixb_aa.nsbtx.small"                  },
	{0xBED0C5F0, "cfx_crup1_roc.emit"                  },
	{0xBED1E157, "rax_cb_kd.nsbca"                     },
	{0xBED3BB6B, "nrg_ex_walk.nsbca"                   },
	{0xBEE3EE65, "pfx_glf1.nsbmd.small"                },
	{0xBEE779BF, "bel_cb_kd.nsbca"                     },
	{0xBEE95A1C, "gup_aa_shot.nsbta.small"             },
	{0xBEEA4963, "prtl_wdmnhap_0.ncgr.small"           },
	{0xBEECE7EA, "cfx_smk1.nsbmd.small"                },
	{0xBEF740AE, "cbt_title_slide.ncgr.small"          },
	{0xBF002398, "chao_ga_3.ncgr.small"                },
	{0xBF0473BD, "cbt_menu_backbutton.ncgr.small"      },
	{0xBF04E49D, "cfx_pksp1.nsbmd.small"               },
	{0xBF0574E2, "cfx_spkp1_trl.emit"                  },
	{0xBF0D7771, "bfl_cb_counter.nsbca"                },
	{0xBF0F99BF, "lvl_stat_tspnl02.ncgr.small"         },
	{0xBF12F360, "charpro_cre4.ncgr.small"             },
	{0xBF1F2A4D, "chao_hd_g.nsbtp.small"               },
	{0xBF2042CA, "prtl_vxmgen_1.ncgr.small"            },
	{0xBF216279, "adv_knuckles.gda"                    },
	{0xBF2A2E67, "sha_cb_dead.nsbca"                   },
	{0xBF2D19B6, "guiframeelements.nclr.small"         },
	{0xBF384A90, "jrl_tlgbx_ms32.ncgr.small"           },
	{0xBF46DAB8, "chao_ge_r.nsbtp.small"               },
	{0xBF52310A, "prtl_txmgen_3.ncgr.small"            },
	{0xBF5375DC, "cfx_pswp1.nsbtp.small"               },
	{0xBF563AD6, "a2c10_scylladefeated.gff"            },
	{0xBF590E50, "son_ex_give.nsbca"                   },
	{0xBF5D6379, "cfx_smk1_hismk.emit"                 },
	{0xBF68996E, "prf_chrbx_di00.ncgr.small"           },
	{0xBF7CA905, "a1_w0_ai_01.ncgr.small"              },
	{0xBF7E2ACC, "pl_ob_spk_hdn.nsbca"                 },
	{0xBF8E6DD3, "nrg_cb_attackend.nsbca"              },
	{0xBF9E7397, "main_slt_on00.ncgr.small"            },
	{0xBFA0C64F, "pfx_tlp1.nsbtp.small"                },
	{0xBFA10488, "froggy.nsbca"                        },
	{0xBFA38398, "itm_ace_dimbu_3.ncgr.small"          },
	{0xBFA7E101, "chaogarden_tut.nbfp"                 },
	{0xBFA7E104, "chaogarden_tut.nbfs"                 },
	{0xBFB31288, "chao_jp_1.ncgr.small"                },
	{0xBFB564B5, "itm_ace_life.ncgr.small"             },
	{0xBFC40828, "exp_ghz1_fw02.ncgr.small"            },
	{0xBFDAEEE4, "cbt_elem_fire.ncgr.small"            },
	{0xBFE24049, "charpro_amy0.ncgr.small"             },
	{0xBFE8A3D9, "cfx_chlp1.nsbtp.small"               },
	{0xBFFF098C, "fx_wtrfl_wtrfl.nsbta.small"          },
	{0xC00BE842, "prtl_egggen_1.ncgr.small"            },
	{0xC0147362, "fx_a1_s1_ex_wtrf.nsbta.small"        },
	{0xC015BCA0, "rax_cb_attack.nsbca"                 },
	{0xC019782E, "pl_ob_blkn_mb.nsbmd.small"           },
	{0xC029E84B, "cfx_firp1.nsbtx.small"               },
	{0xC039460F, "icn_s_patk_she1.ncgr.small"          },
	{0xC046A1B6, "chao_js_r.nsbmd.small"               },
	{0xC0471BD4, "prtl_eggaha_3.ncgr.small"            },
	{0xC065B5B6, "plt_cod_tyxsis.plo"                  },
	{0xC06898D6, "rou_ex_fly.nsbca"                    },
	{0xC06BAAA4, "drnsui_aa.nsbmd.small"               },
	{0xC0887FB1, "gui_portraitsouthouse.nclr.small"    },
	{0xC08A2B00, "wmp_s_pin0.ncgr.small"               },
	{0xC08EDE9A, "cmbtbg_a2_s5.nsbta.small"            },
	{0xC0A9E8F1, "cfx_gwtr_pu.nsbtx.small"             },
	{0xC0B258C9, "icn_l_barwof.ncgr.small"             },
	{0xC0BAC240, "prtl_nrmge2_1.ncgr.small"            },
	{0xC0BAC9E8, "itm_ace_eggmd_0.ncgr.small"          },
	{0xC0BEF913, "a1c3_reunion_hold.gff"               },
	{0xC0C595F1, "cfx_nrgp1.nsbtx.small"               },
	{0xC0CAAC51, "gzd_cb_attackend.nsbca"              },
	{0xC0E1AE31, "attack6_bs.tga"                      },
	{0xC0E69111, "shp_cb_blockidle.nsbca"              },
	{0xC0F5880B, "fx_footprint.nsbtx.small"            },
	{0xC0F5D7C1, "cmbtbg_a2_s7_i01.nsbmd.small"        },
	{0xC102FE79, "invent_icon_cho.ncgr.small"          },
	{0xC1134883, "big_ex_jump.nsbca"                   },
	{0xC119EBC9, "cbt_flescp_1.ncgr.small"             },
	{0xC11F8FF1, "a2c10_knuckles_face_scylla.gff"      },
	{0xC12931AD, "prtl_knuckgen_3.ncgr.small"          },
	{0xC13D3222, "rou_cb_attackend.nsbca"              },
	{0xC146C4C2, "cbt_menu_button_of02.ncgr.small"     },
	{0xC14B2839, "itm_fotrunsl_0.ncgr.small"           },
	{0xC15B0DD1, "cbt_btn_right_of.ncgr.small"         },
	{0xC15D2701, "a1_w1_cc_00.ncgr.small"              },
	{0xC15FBF52, "chao_jg_g.nsbtp.small"               },
	{0xC169537B, "charpro_ts_a02.ncgr.small"           },
	{0xC16961D4, "c_sota_so.nsbca"                     },
	{0xC16ADB02, "invent_titlelit00.ncgr.small"        },
	{0xC1711F22, "prtl_ffmmad_3.ncgr.small"            },
	{0xC17D43E2, "prtl_amysmg_0.ncgr.small"            },
	{0xC1845C9F, "ixb_cb_damage.nsbca"                 },
	{0xC187C6D0, "fx_vox_y_ea_l.nsbmd.small"           },
	{0xC18C7ABA, "wmp_txtbr_tx04.ncgr.small"           },
	{0xC19C883D, "chao_ie_2.ncgr.small"                },
	{0xC19FCC5C, "bif_egg.nbfp"                        },
	{0xC19FCC5F, "bif_egg.nbfs"                        },
	{0xC1A24A05, "a2c10_repulsor_dropped.gff"          },
	{0xC1C6F3B1, "plt_cod_eggman.plo"                  },
	{0xC1D5A8DF, "a1_s4_ex.are"                        },
	{0xC1D6FB05, "exp_wel1_mm03.ncgr.small"            },
	{0xC1DAFCD5, "fx_drn_lzr_beam.nsbtx.small"         },
	{0xC1E0AB95, "cfx_crsp1_smk.emit"                  },
	{0xC1E4D5B1, "cbt_victory.nanr.small"              },
	{0xC1E74F2A, "cbt_focusgood.nanr.small"            },
	{0xC1ED6280, "btn_arrowdn_on.ncgr.small"           },
	{0xC1F2069D, "cyclone_nrrgal.are"                  },
	{0xC1F2C9AE, "a1_w2_gh_12.ncgr.small"              },
	{0xC1F7C3B1, "sha_cb_attack.nsbca"                 },
	{0xC202ED76, "tai_cb_attackc.nsbca"                },
	{0xC210F54A, "cbt_cmdslothighlight.ncgr.small"     },
	{0xC21CAEC8, "a1c5_p6_repaircannontrigger.gff"     },
	{0xC2236852, "itm_glv_power_1.ncgr.small"          },
	{0xC2369E27, "exl_barow_ms01.ncgr.small"           },
	{0xC2396332, "fx_ix_orb.nsbta.small"               },
	{0xC2600B17, "pfx_lzr1.nsbtp.small"                },
	{0xC26C6A50, "amy_cb_deadidle.nsbca"               },
	{0xC27249C0, "prtl_eixthrnvar_1.ncgr.small"        },
	{0xC2738AF6, "egm_cb_counter.nsbca"                },
	{0xC27948E1, "icn_s_patk_big1.ncgr.small"          },
	{0xC27D8844, "gui_chao_j3.nclr.small"              },
	{0xC2835753, "pl_ob_blk_aa.nsbmd.small"            },
	{0xC285DB45, "itm_ace_tyxis_0.ncgr.small"          },
	{0xC28D08E1, "exl_smbtn_hi10.ncgr.small"           },
	{0xC2A4EC28, "cfx_flap1_fla.emit"                  },
	{0xC2A562AD, "a2_s1_i03.walk.tga"                  },
	{0xC2AC32AA, "chao_jl_r.nsbtx.small"               },
	{0xC2B53BEE, "a2_s2_ex_puzzle75.gff"               },
	{0xC2B90020, "spell_covero_2.spl"                  },
	{0xC2BFBF0C, "ch1_intro_st.gda"                    },
	{0xC2C7D8AE, "invent_bar02.ncgr.small"             },
	{0xC2CB18A1, "spell_covero_3.spl"                  },
	{0xC2CFFD9F, "upper_textbox_03.ncgr.small"         },
	{0xC2DB3E95, "prt_se_kobed.ncgr.small"             },
	{0xC2DCD58A, "man_strbt_on00.ncgr.small"           },
	{0xC2E92EA1, "prtl_eixfm3_1.ncgr.small"            },
	{0xC2E93F8C, "infoframe_r.ncgr.small"              },
	{0xC2E945DD, "fx_bubbles.nsbtp.small"              },
	{0xC30157AB, "amy_cb_victoryin.nsbca"              },
	{0xC30BB7EC, "rou_cb_rund.nsbca"                   },
	{0xC30E5EE1, "wm_cav_dr.tga"                       },
	{0xC3422C07, "c_soknamta_am.nsbca"                 },
	{0xC3435C25, "pro_medbx_20.ncgr.small"             },
	{0xC34481F1, "loc_round_dt1.ncgr.small"            },
	{0xC34AFE50, "cbt_pathdot.ncgr.small"              },
	{0xC3597E72, "knu_ex_walk.nsbca"                   },
	{0xC3638014, "rwd_lgbtn_di01.ncgr.small"           },
	{0xC3823544, "pl_am_rdr_off.nsbca"                 },
	{0xC385B959, "sso_ex_walk.nsbca"                   },
	{0xC3864F68, "cfx_fltp1.nsbtp.small"               },
	{0xC3866364, "chaseix.gda"                         },
	{0xC38792FF, "crm_cb_counter.nsbca"                },
	{0xC38F1DC7, "plt_cod_shade.plo"                   },
	{0xC396FA88, "charpro_bar01.ncgr.small"            },
	{0xC3AAAB74, "pro_lrgbx_11.ncgr.small"             },
	{0xC3AB5871, "slg_cb_dead.nsbca"                   },
	{0xC3AC6B0A, "exl_flash_ms1.ncgr.small"            },
	{0xC3AF2C71, "gen_buttns_of00.ncgr.small"          },
	{0xC3B0A7DC, "itm_glv_stealth2.ncgr.small"         },
	{0xC3C135D7, "chao_ha_g.nsbmd.small"               },
	{0xC3C3CC2E, "kro_cb_getup.nsbca"                  },
	{0xC3C9580D, "genroun_st.nsbtx.small"              },
	{0xC3CBAEDD, "itm_glvirncl_0.ncgr.small"           },
	{0xC3DAFB68, "adv_eggman.gda"                      },
	{0xC3E3F2C4, "plt_cod_argus.plo"                   },
	{0xC3E8E642, "chao_gb_r.nsbmd.small"               },
	{0xC3F586BC, "attack.ncgr.small"                   },
	{0xC3FA093C, "a2c6_generator_bot.gff"              },
	{0xC40AC3DC, "a2_s3_card_b1.nbfp"                  },
	{0xC40AC3DF, "a2_s3_card_b1.nbfs"                  },
	{0xC40AE6A0, "wmp_airp_on01.ncgr.small"            },
	{0xC40E0C2B, "loc_abshd_fr1.ncgr.small"            },
	{0xC418CA9E, "a1_s2_ex.walk.tga"                   },
	{0xC41ED755, "chao_fb_0.ncgr.small"                },
	{0xC4372A11, "pro_help_of00.ncgr.small"            },
	{0xC43A9858, "jurnl_pnl00.ncgr.small"              },
	{0xC445ED3C, "exp_ghq1_mm20.ncgr.small"            },
	{0xC452D0F2, "cfx_srkp1.nsbtp.small"               },
	{0xC453076F, "gzd_ex_run.nsbca"                    },
	{0xC46400CF, "cfx_disp1.nsbmd.small"               },
	{0xC46F9588, "front_slotclick.gui"                 },
	{0xC47D8C9D, "icn_l_uttalk.ncgr.small"             },
	{0xC49BA554, "plt_ch3_boy.plo"                     },
	{0xC49D4813, "gen_titlebar_00.ncgr.small"          },
	{0xC4A64EE8, "icn_l_wmp_cyclone.ncgr.small"        },
	{0xC4A6B758, "a2_w0_va.nclr.small"                 },
	{0xC4ACCCEE, "prtl_krgang_0.ncgr.small"            },
	{0xC4ACD601, "prtl_eixprd_1.ncgr.small"            },
	{0xC4BAB11F, "prtl_vxccitgen_0.ncgr.small"         },
	{0xC4BF4C85, "conv_icon03.ncgr.small"              },
	{0xC4D4043C, "wmp_s_redx.ncgr.small"               },
	{0xC4D7EF08, "pl_pz_fan_aa.nsbmd.small"            },
	{0xC4E0D39C, "ex_bloop_left.nsbca"                 },
	{0xC4E75BD7, "vci_cb_attack.nsbca"                 },
	{0xC4E9823A, "a2c10_knucklesdisappears2.gff"       },
	{0xC4ED9A27, "ch5_permission.gda"                  },
	{0xC4F54ACF, "prtl_tailsdet_1.ncgr.small"          },
	{0xC501749F, "plt_a2sq_clues.plo"                  },
	{0xC5094B55, "pfx_nicktest02.nsbtp.small"          },
	{0xC50B0FDC, "icn_s_bugspray.ncgr.small"           },
	{0xC5109691, "cbt_elem_redx.ncgr.small"            },
	{0xC522F97B, "prtl_sonicinq_3.ncgr.small"          },
	{0xC5293EEB, "itm_ace_root_3.ncgr.small"           },
	{0xC5298B13, "loc_trppd_fr1.ncgr.small"            },
	{0xC52C9AB2, "a2c7_kronbreakthrough.gff"           },
	{0xC532A53E, "lvl_tdbld_ms04.ncgr.small"           },
	{0xC53601ED, "a1_w3_ai.nclr.small"                 },
	{0xC541F98F, "icn_l_utinv1.ncgr.small"             },
	{0xC54F0C23, "cbt_btn_extra.ncgr.small"            },
	{0xC54F5B4F, "lvl_smlbt_on00.ncgr.small"           },
	{0xC5535266, "bel_ex_idle.nsbca"                   },
	{0xC55F258B, "gwr_hf.nsbmd.small"                  },
	{0xC565386A, "bif_ome.nbfp"                        },
	{0xC565386D, "bif_ome.nbfs"                        },
	{0xC565AA3D, "charpro_rou5.ncgr.small"             },
	{0xC571769D, "chao_jr_g.nsbtp.small"               },
	{0xC577B092, "prtl_egghum_1.ncgr.small"            },
	{0xC579C98C, "ix_cb_dead.nsbca"                    },
	{0xC599C14A, "ssb_aa.nsbtp.small"                  },
	{0xC59E309F, "cbt_focusbreak.ncer.small"           },
	{0xC59F0B5C, "ch8_scylla_tyxisshipattack.gff"      },
	{0xC5B6F392, "a2_w0_va_00.ncgr.small"              },
	{0xC5BC994F, "lowertextselected.gui"               },
	{0xC5C2C525, "big_ex_idle.nsbca"                   },
	{0xC5DCCC9F, "cbt_focus_button_pathfail.ncgr.small"},
	{0xC5E1E8E5, "fx_gun_scrn.nsbtx.small"             },
	{0xC5EA79B4, "store_1_bg.tga"                      },
	{0xC5F5EE23, "prtl_hansad_0.ncgr.small"            },
	{0xC5F7F69F, "egm_cb_blockidle.nsbca"              },
	{0xC5FF0456, "itm_ace_ironw_3.ncgr.small"          },
	{0xC601CADC, "chao_jd_g.nsbmd.small"               },
	{0xC60422E8, "c_sotaamkn_1_am1.nsbca"              },
	{0xC607D45C, "worldmap_wslideend.gui"              },
	{0xC608BBF1, "icn_s_patk_tai5.ncgr.small"          },
	{0xC60CEFB0, "a1c3_shadow_enter.gff"               },
	{0xC613A6B2, "a1c1_sonictalkcommand.gff"           },
	{0xC618B80A, "fx_els_er_aa.nsbta.small"            },
	{0xC61CD75B, "sonic.gda"                           },
	{0xC6297B47, "chao_ie_r.nsbmd.small"               },
	{0xC62E9950, "cfx_ltgp1_ltg.emit"                  },
	{0xC6399721, "a1c1_first_enemy_defeated.gff"       },
	{0xC63C9C36, "cmbtbg_a1_s2.nsbta.small"            },
	{0xC63EFE7B, "a2sq_dragon.dlg"                     },
	{0xC6468867, "lvl_pow_bspnl20.ncgr.small"          },
	{0xC6544D53, "fx_exclamation.nsbtp.small"          },
	{0xC65830B0, "prtl_guncodet_3.ncgr.small"          },
	{0xC65871A3, "a1_s1_ex_mm.nbfp"                    },
	{0xC65871A6, "a1_s1_ex_mm.nbfs"                    },
	{0xC65FEC7D, "a2_s3_card_b2.nbfp"                  },
	{0xC65FEC80, "a2_s3_card_b2.nbfs"                  },
	{0xC674DF04, "cbt_btn_standard.gui"                },
	{0xC67960E4, "pfx_fls1.nsbtp.small"                },
	{0xC67F753D, "crm_cb_idle.nsbca"                   },
	{0xC6A9D651, "a1_w1_br_11.ncgr.small"              },
	{0xC6AA1F07, "fadeout.gda"                         },
	{0xC6AFD2F6, "icn_l_onbash3.ncgr.small"            },
	{0xC6AFF3CE, "a2_s1_i04.walk.tga"                  },
	{0xC6B9695F, "amy_cb_attackc.nsbca"                },
	{0xC6D1044F, "charpro_bs_c15.ncgr.small"           },
	{0xC6DB8507, "itm_act2_chaos4_0.ncgr.small"        },
	{0xC6E97EFC, "prtl_egglaf_2.ncgr.small"            },
	{0xC6EB1DDC, "cfx_stngr.nsbmd.small"               },
	{0xC6F8E1FD, "prtl_krshap_0.ncgr.small"            },
	{0xC6FBA433, "pfx_rng1.nsbmd.small"                },
	{0xC70855EC, "prtl_tdadsad_3.ncgr.small"           },
	{0xC716BA74, "cbt_playerframe.ncgr.small"          },
	{0xC721587B, "btn_l_bar_of00.ncgr.small"           },
	{0xC72893F4, "cfx_ficp1.nsbtp.small"               },
	{0xC729E0D4, "ehd_tbbox_ms02.ncgr.small"           },
	{0xC72E30B1, "exl_circe_ms00.ncgr.small"           },
	{0xC73B940D, "pl_am_noc_gsr.nsbca"                 },
	{0xC74FFD5B, "lvl_pow_tspnl13.ncgr.small"          },
	{0xC7511F19, "cbt_btn_mid.ncgr.small"              },
	{0xC7546536, "gwr_cb_pattack01.nsbca"              },
	{0xC754B436, "invent_barlit01.ncgr.small"          },
	{0xC75623A6, "a1c3_beforeshadow.gff"               },
	{0xC7611795, "pfx_gas1_gas.emit"                   },
	{0xC76F54A8, "cfx_smk1_smk.emit"                   },
	{0xC774E7B7, "a2_s4_ex_mm.nclr.small"              },
	{0xC7766815, "itm_accwtrng_0.ncgr.small"           },
	{0xC777FD9B, "egm_cb_getup.nsbca"                  },
	{0xC779F40A, "nes_ex_walk.nsbca"                   },
	{0xC79D27B0, "plt_ch3_shadow.plo"                  },
	{0xC7A7FEF2, "omg_aa.nsbmd.small"                  },
	{0xC7AD21EE, "knu_ex_fly.nsbca"                    },
	{0xC7B2AE2E, "a2c10_arrivenocturn.gff"             },
	{0xC7BD019B, "chao_jd_0.ncgr.small"                },
	{0xC7BFC2A5, "chasestandard1.gda"                  },
	{0xC7C3CD03, "cfx_ch7.nsbtp.small"                 },
	{0xC7C8C34C, "lvl_tmstr_ms02.ncgr.small"           },
	{0xC7D1DB26, "chasestandard2.gda"                  },
	{0xC7D6F6B3, "fx_prjpulse.nsbtp.small"             },
	{0xC7DA1EDD, "a1c1_blockfall_camera.gff"           },
	{0xC7E3F3A7, "chasestandard3.gda"                  },
	{0xC7ECF3BB, "pl_ob_blkn_mo.nsbmd.small"           },
	{0xC7F56C22, "str_txtbr_hi02.ncgr.small"           },
	{0xC7F60C28, "chasestandard4.gda"                  },
	{0xC7F7E911, "chao_fb_g.nsbtx.small"               },
	{0xC7FAAC62, "pzl_monitorestablish.gff"            },
	{0xC8052D98, "prtl_nrdge2_2.ncgr.small"            },
	{0xC80824A9, "chasestandard5.gda"                  },
	{0xC81A3D2A, "chasestandard6.gda"                  },
	{0xC81F054D, "cfx_ch4.nsbmd.small"                 },
	{0xC8219985, "a1_s2_i05_mm.nclr.small"             },
	{0xC8361C71, "pil_cb_attackend.nsbca"              },
	{0xC836D9C0, "pl_mv_cyc_fly.nsbca"                 },
	{0xC839FECC, "itm_glv_lucky_0.ncgr.small"          },
	{0xC83DE08B, "lower_green_ring_00.ncgr.small"      },
	{0xC8475B3E, "pro_bar_of01.ncgr.small"             },
	{0xC8495D93, "rwd_tsmbx_of01.ncgr.small"           },
	{0xC84A9CDD, "conv_bubbles.nclr.small"             },
	{0xC84F1A8D, "prtl_eixmad_1.ncgr.small"            },
	{0xC85178BB, "cbt_trapd_ms.nanr.small"             },
	{0xC85504F0, "prt_se_arm.ncgr.small"               },
	{0xC863F78C, "debugstatusitems.gda"                },
	{0xC8654FC4, "chao_jk_2.ncgr.small"                },
	{0xC86EB3A7, "exp_ghz2_mm12.ncgr.small"            },
	{0xC88B7763, "guianim_linear.gda"                  },
	{0xC88DF42E, "gui_chao_g.nclr.small"               },
	{0xC89334B4, "cfx_ixdp1.nsbmd.small"               },
	{0xC897C244, "prtl_wkrstu_0.ncgr.small"            },
	{0xC8A30AC3, "fx_vox_b_ea_r.nsbta.small"           },
	{0xC8AEEE23, "spell_poison_dart.spl"               },
	{0xC8B8E831, "cavalry.gda"                         },
	{0xC8BD1802, "cfx_gwtr_pu.nsbca"                   },
	{0xC8BD8EFF, "pfx_mot1_sno.emit"                   },
	{0xC8BFB05C, "cfx_gyzp1.nsbtp.small"               },
	{0xC8C300F8, "kro_cb_attackend.nsbca"              },
	{0xC8D73611, "main_pnl_off14.ncgr.small"           },
	{0xC8D8181C, "floaty.nftr"                         },
	{0xC8DB1268, "exp_ghz1_mm.nclr.small"              },
	{0xC8DD7BAD, "ch5_teleport.gda"                    },
	{0xC8FC6C68, "prtl_tdadsca_0.ncgr.small"           },
	{0xC8FDA491, "pig_ex_jump.nsbca"                   },
	{0xC8FE477A, "prtl_boyhap_3.ncgr.small"            },
	{0xC90A6240, "prtl_txmmad_1.ncgr.small"            },
	{0xC913BB29, "chao_ga_g.nsbtp.small"               },
	{0xC9170D29, "prf_tboxa_ms01.ncgr.small"           },
	{0xC921C288, "itm_ace_seed_3.ncgr.small"           },
	{0xC9383CA1, "sha_cb_pattack.nsbca"                },
	{0xC938834D, "lvl_pow_pointon.ncgr.small"          },
	{0xC93B6B94, "chao_fb_r.nsbtp.small"               },
	{0xC941B137, "swa_cb_attackend.nsbca"              },
	{0xC947E37C, "pfx_spark.nsbmd.small"               },
	{0xC94CFFD8, "chao_gb_2.ncgr.small"                },
	{0xC956D477, "pfx_air1_air.emit"                   },
	{0xC9615505, "fx_ome_missile.nsbta.small"          },
	{0xC9739E58, "jurnl_ts21.ncgr.small"               },
	{0xC9769CE6, "movementsounds.gda"                  },
	{0xC9868D0B, "plt_cod_omega.plo"                   },
	{0xC998F49B, "prt_ce_red.ncgr.small"               },
	{0xC99E469D, "c_soknam_kn.nsbca"                   },
	{0xC9A187A2, "icn_l_onfly2.ncgr.small"             },
	{0xC9ADBE58, "exu_thlth_ms00.ncgr.small"           },
	{0xC9B1DDB9, "wmp_space_of11.ncgr.small"           },
	{0xC9B4AEDA, "pl_pz_fanlt_rg.nsbtx.small"          },
	{0xC9B79881, "a2_s1_ex_la.smp"                     },
	{0xC9BAC043, "cfx_rckp2.nsbtp.small"               },
	{0xC9CCC511, "rwd_tlgbx_ms01.ncgr.small"           },
	{0xC9D2524E, "jay_test.are"                        },
	{0xC9D80783, "itm_fotsnkel_1.ncgr.small"           },
	{0xC9E4BB37, "fx_gnp_bullet.nsbmd.small"           },
	{0xC9E77D5E, "c_sotaamkn_1_ta.nsbca"               },
	{0xC9EE52B5, "bal_cb_pattack01.nsbca"              },
	{0xC9F84340, "cfx_hdsp1_hds.emit"                  },
	{0xC9FFEEC8, "chao_jq_0.ncgr.small"                },
	{0xCA056306, "itm_ace_bugsp_1.ncgr.small"          },
	{0xCA07FBE0, "pfx_drp1_tox.emit"                   },
	{0xCA091B9D, "lower_textbox_04.ncgr.small"         },
	{0xCA0C758F, "cbt_elem_ice.ncgr.small"             },
	{0xCA0E0DA1, "camerapanblueridge.gff"              },
	{0xCA138227, "chao_jo_g.nsbmd.small"               },
	{0xCA147E27, "fx_els_fr_aa.nsbmd.small"            },
	{0xCA1AC841, "fx_tal_mbot.nsbta.small"             },
	{0xCA1BC68C, "prf_tpanl_ms12.ncgr.small"           },
	{0xCA1D98DD, "buz_cb_kd.nsbca"                     },
	{0xCA1E9586, "prtl_guncoang_0.ncgr.small"          },
	{0xCA1F9A8B, "prtl_wsgrel_3.ncgr.small"            },
	{0xCA225339, "cfx_tabp1.nsbtp.small"               },
	{0xCA2C3F5E, "pl_pu_cha_egg.nsbtp.small"           },
	{0xCA344F25, "itm_ace_crzbn_2.ncgr.small"          },
	{0xCA387E16, "chao_he_g.nsbtx.small"               },
	{0xCA4907AE, "a1c1_amysmash.gff"                   },
	{0xCA4A2CB5, "prtl_amysad_2.ncgr.small"            },
	{0xCA4D0C34, "pl_pu_cao_nst.nsbmd.small"           },
	{0xCA751F45, "worldmap_tut_bs.tga"                 },
	{0xCA78E409, "fx_a2_s3_ex_nrg.nsbta.small"         },
	{0xCA870748, "a2c7_landonkron.gff"                 },
	{0xCA8BC6A1, "exl_arwin_of.ncgr.small"             },
	{0xCA96DD37, "rewards.gda"                         },
	{0xCAAA7A10, "ssf_tlgbx_ms01.ncgr.small"           },
	{0xCABA84EF, "a2_s1_i05.walk.tga"                  },
	{0xCABED986, "bridgeplank_wm1.tga"                 },
	{0xCABF218D, "itm_fotsnksh_1.ncgr.small"           },
	{0xCAC1A686, "cfx_lucp1.nsbtp.small"               },
	{0xCACD481E, "icn_s_bash1.ncgr.small"              },
	{0xCAD0F207, "bridgeplank_wm2.tga"                 },
	{0xCADEDA17, "ch10_cyclone.gda"                    },
	{0xCAE12F41, "a2_s1_i04_ts.nbfp"                   },
	{0xCAE12F44, "a2_s1_i04_ts.nbfs"                   },
	{0xCAE30A88, "bridgeplank_wm3.tga"                 },
	{0xCAE8AF00, "gnp_cb_idle.nsbca"                   },
	{0xCAED050E, "prtl_voxmerc.nclr.small"             },
	{0xCAF52309, "bridgeplank_wm4.tga"                 },
	{0xCAFCC5F3, "gui_worldmap.nclr.small"             },
	{0xCB0B68D2, "pl_am_rdr_aa.nsbtp.small"            },
	{0xCB123139, "fx_ix_flash.nsbmd.small"             },
	{0xCB142A26, "man_strbt_of13.ncgr.small"           },
	{0xCB156C0A, "prtl_crehap_3.ncgr.small"            },
	{0xCB1828DF, "a1_creamjoinpartyselect.gff"         },
	{0xCB235EB8, "exl_circl_ms00.ncgr.small"           },
	{0xCB24049C, "str_tlgbx_ms00.ncgr.small"           },
	{0xCB28116B, "a1c5_p7_2_shade.gff"                 },
	{0xCB335B7F, "realtime_tutorial_touch_bg.tga"      },
	{0xCB3BEEB6, "exp_cct1_mm11.ncgr.small"            },
	{0xCB4CAD55, "bel_cb_getup.nsbca"                  },
	{0xCB4CBE59, "pl_mv_boa_aa.nsbtx.small"            },
	{0xCB4CC953, "cfx_expp1_min.emit"                  },
	{0xCB5067A0, "a2c8_scy_gentleman2defeated.gff"     },
	{0xCB54502E, "chao_id_g.nsbtp.small"               },
	{0xCB660A43, "prtl_thxgen_1.ncgr.small"            },
	{0xCB6A71A7, "cbt_rndnum7.ncgr.small"              },
	{0xCB72348C, "pig_cb_attackend.nsbca"              },
	{0xCB7C0099, "chao_he_r.nsbtp.small"               },
	{0xCB84EBFB, "fx_els_ic_aa.nsbmd.small"            },
	{0xCB869A26, "a2_s3_ex_mm.nbfp"                    },
	{0xCB869A29, "a2_s3_ex_mm.nbfs"                    },
	{0xCB89A883, "bsc_cb_dead.nsbca"                   },
	{0xCB8B9D57, "fx_gren_tal.nsbmd.small"             },
	{0xCB8E17B8, "cfx_spk_cu0p1.emit"                  },
	{0xCB977F93, "conv_check_on00.ncgr.small"          },
	{0xCBA01802, "knu_ex_fall.nsbca"                   },
	{0xCBB5B780, "pfx_bub1.nsbtx.small"                },
	{0xCBB62FBB, "charpro_ts_b01.ncgr.small"           },
	{0xCBC4E4F8, "pl_pz_vox_wll.nsbtx.small"           },
	{0xCBC5A246, "prtl_wspsca_1.ncgr.small"            },
	{0xCBC5D6CC, "a1c5_p9_after_shade_movie.gff"       },
	{0xCBC8D260, "str_tlgbx_ms22.ncgr.small"           },
	{0xCBC94C4E, "movement00_bs.tga"                   },
	{0xCBD5B07B, "cbt_btn_failure.gui"                 },
	{0xCBE0BC7A, "exp_cct1_mm33.ncgr.small"            },
	{0xCBE1DC41, "hnpc_cop_idle.nsbca"                 },
	{0xCBEC2B0A, "swtsbot_eye.nsbta.small"             },
	{0xCBFAB4B6, "she_cb_rund.nsbca"                   },
	{0xCC06311B, "prtl_timgen_3.ncgr.small"            },
	{0xCC0767F7, "gzg_aa.nsbmd.small"                  },
	{0xCC1E93BF, "sso_aa.nsbtx.small"                  },
	{0xCC2D3EF6, "a1c3_reunion_boy.gff"                },
	{0xCC2DAD0F, "party.gda"                           },
	{0xCC38E187, "gnp_cb_getup.nsbca"                  },
	{0xCC3A72D7, "cho_tmbtn_of01.ncgr.small"           },
	{0xCC3D7F06, "pl_pz_brd_aa.nsbtx.small"            },
	{0xCC40FF92, "splashbuttons.gui"                   },
	{0xCC47D223, "egm_aa.nsbtx.small"                  },
	{0xCC488363, "icn_l_larwonjrl.ncgr.small"          },
	{0xCC4ABF66, "prtl_dronegen_0.ncgr.small"          },
	{0xCC4FEBC2, "icn_l_utbash3.ncgr.small"            },
	{0xCC5746C3, "slg_cb_counter.nsbca"                },
	{0xCC5A1A0C, "prtl_knustb_1.ncgr.small"            },
	{0xCC646C0F, "itm_act2_mastr_3.ncgr.small"         },
	{0xCC6D219E, "itm_glvsercl_0.ncgr.small"           },
	{0xCC6E4CBE, "prtl_eixamu_1.ncgr.small"            },
	{0xCC79131B, "chao_jh_g.nsbtx.small"               },
	{0xCC7E0170, "pfx_puring.nsbmd.small"              },
	{0xCC835D40, "spell_blowkiss_2.spl"                },
	{0xCC883BD2, "icn_s_statfor.ncgr.small"            },
	{0xCC9535E9, "pl_pz_crn_aa.nsbtp.small"            },
	{0xCC9575C1, "spell_blowkiss_3.spl"                },
	{0xCCA0101A, "a1_w1_mr_00.ncgr.small"              },
	{0xCCA5A03D, "egb_aa.nsbmd.small"                  },
	{0xCCA8975C, "pl_pz_cav_dr.nsbtx.small"            },
	{0xCCB1FB3A, "fx_vox_g_ea_m.nsbtx.small"           },
	{0xCCB2C1F2, "a1c5_captainsonichold.gff"           },
	{0xCCC4CB98, "prtl_thxdef_1.ncgr.small"            },
	{0xCCD076F4, "prtl_nesgen_0.ncgr.small"            },
	{0xCCE570D7, "itm_ace_goggl_3.ncgr.small"          },
	{0xCCEB580A, "bmp_cb_dead.nsbca"                   },
	{0xCCF3A72A, "fx_els_lt_aa.nsbtx.small"            },
	{0xCCF670B0, "tyx_cb_dead.nsbca"                   },
	{0xCCFB4850, "cfx_ccap1_cca.emit"                  },
	{0xCD081AA1, "spell_lucky_all.spl"                 },
	{0xCD09F6DE, "sha_cb_getup.nsbca"                  },
	{0xCD0D85A3, "wmp_s_star.ncgr.small"               },
	{0xCD1DDCCD, "prt_se_lazdr.ncgr.small"             },
	{0xCD2353D7, "a1_w3_br.nclr.small"                 },
	{0xCD241E50, "cfx_fgrp3.nsbmd.small"               },
	{0xCD273819, "gui_ssf.nclr.small"                  },
	{0xCD2D4AA0, "gzg_cb_attackc.nsbca"                },
	{0xCD356FD1, "a1_w2_gh.nclr.small"                 },
	{0xCD3A7660, "cfx_crzbns.nsbmd.small"              },
	{0xCD567A56, "itm_ace_resur_1.ncgr.small"          },
	{0xCD62815E, "cbt_chaseenemy.ncgr.small"           },
	{0xCD69EA56, "ehd_tbbx2_ms01.ncgr.small"           },
	{0xCD6A61F8, "cfx_gasp1_gas.emit"                  },
	{0xCD6C9760, "a1c5_p2_end_warp.gff"                },
	{0xCD754D60, "inventorydatabackground.gui"         },
	{0xCD775284, "ambientupper.gui"                    },
	{0xCD7962BA, "chao_fd_3.ncgr.small"                },
	{0xCD915859, "gui_fow.nclr.small"                  },
	{0xCD981AC6, "cfx_gfir_pu.nsbca"                   },
	{0xCD9820F8, "pl_pz_crn_tu.nsbtx.small"            },
	{0xCD9AA1D1, "prtl_wdmnexh_2.ncgr.small"           },
	{0xCD9B0D31, "hnpc_sci_aa.nsbtx.small"             },
	{0xCD9F2CB4, "demo_loop_1_a.gff"                   },
	{0xCDA0317A, "rhk_cb_attack.nsbca"                 },
	{0xCDAD2133, "pig_ex_idle.nsbca"                   },
	{0xCDB14535, "demo_loop_1_b.gff"                   },
	{0xCDBAF5E4, "exp_wel1_mm20.ncgr.small"            },
	{0xCDBC959E, "chao_jh_r.nsbtp.small"               },
	{0xCDCAF3CD, "fx_vox_r_ea_l.nsbta.small"           },
	{0xCDD8E891, "fx_vox_p_ea_r.nsbta.small"           },
	{0xCDF14BCA, "prt_se_evshd.ncgr.small"             },
	{0xCE03896F, "prtl_eggsmg_1.ncgr.small"            },
	{0xCE21A3F3, "pfx_mst1.nsbtp.small"                },
	{0xCE2B10CC, "a1c3_sentinels_appear2.gff"          },
	{0xCE348A93, "prtl_sonicdis_3.ncgr.small"          },
	{0xCE3500FE, "prtl_vxcriagen_3.ncgr.small"         },
	{0xCE375A3E, "itm_glvcrscl_2.ncgr.small"           },
	{0xCE428D1C, "camerapannocturne.gff"               },
	{0xCE46FFF6, "prt_se_ligia.ncgr.small"             },
	{0xCE53F77D, "a1_s6_ex_slide1.gff"                 },
	{0xCE580C0E, "a1_s3_i01_4to2.gff"                  },
	{0xCE59B0C1, "son_ex_bounce.nsbca"                 },
	{0xCE7CC8A4, "a1_w3_cc_01.ncgr.small"              },
	{0xCE80BA02, "crm_cb_blockout.nsbca"               },
	{0xCE86C24E, "prtl_txsgen_1.ncgr.small"            },
	{0xCE87BF38, "jurnl_cod_linelit04.ncgr.small"      },
	{0xCE9CCF7A, "pfx_gas1.nsbtp.small"                },
	{0xCEA02D90, "a2c10_submerged_hack.gff"            },
	{0xCEA06E12, "a1_s3_i01_4to6.gff"                  },
	{0xCEA7D030, "charpro_barlit00.ncgr.small"         },
	{0xCEB77A4C, "vox_leu_aa.nsbtx.small"              },
	{0xCEC00D68, "icn_l_victory0.ncgr.small"           },
	{0xCEC0F81C, "pfx_mst1_wfl.emit"                   },
	{0xCEC17BE5, "pl_pz_ppv_dn.nsbca"                  },
	{0xCEC91322, "a1c3_tails_events.dlg"               },
	{0xCED5B660, "gzc_cb_damage.nsbca"                 },
	{0xCED5D8B7, "ix_cb_pattack1.nsbca"                },
	{0xCEE865A6, "main_pnl_big03.ncgr.small"           },
	{0xCEF09D6D, "prtl_scihap_1.ncgr.small"            },
	{0xCEF862C6, "fx_sha_jav_muz.nsbta.small"          },
	{0xCF071E92, "man_tdlcp_on00.ncgr.small"           },
	{0xCF0BB036, "gui_portraitstal.nclr.small"         },
	{0xCF0EC168, "prtl_gglgen_0.ncgr.small"            },
	{0xCF1E8D6B, "itm_ace_immun_1.ncgr.small"          },
	{0xCF31C03B, "a2_s5_ex_teleport1.gff"              },
	{0xCF341243, "cfx_javp1.nsbtp.small"               },
	{0xCF39E79F, "prtl_amydrm_1.ncgr.small"            },
	{0xCF3CEA6D, "conv_bark_bs00.ncgr.small"           },
	{0xCF3E76C9, "sha_ex_jumpair.nsbca"                },
	{0xCF4064F1, "prtl_tailsexc_0.ncgr.small"          },
	{0xCF43D8BC, "a2_s5_ex_teleport2.gff"              },
	{0xCF44D0AC, "kdr_cb_dead.nsbca"                   },
	{0xCF47BEBC, "c_sotaamkn_1_so1.nsbca"              },
	{0xCF545355, "fx_shadtele.nsbmd.small"             },
	{0xCF55F13D, "a2_s5_ex_teleport3.gff"              },
	{0xCF575A74, "cfx_expp2_min.emit"                  },
	{0xCF59397D, "prtl_sdwgen_1.ncgr.small"            },
	{0xCF604D2A, "cfx_rfpp1.nsbtp.small"               },
	{0xCF61749E, "chap9scr_top.tga"                    },
	{0xCF6809BE, "a2_s5_ex_teleport4.gff"              },
	{0xCF6BB7E4, "plt_cod_rouge.plo"                   },
	{0xCF6D1D5C, "cbtmenubg_11.ncgr.small"             },
	{0xCF6D7936, "chao_ha_0.ncgr.small"                },
	{0xCF7A223F, "a2_s5_ex_teleport5.gff"              },
	{0xCF7AFB33, "prtl_hczgen_0.ncgr.small"            },
	{0xCF890CA8, "cbt_chasegauge_00.ncgr.small"        },
	{0xCF8C3AC0, "a2_s5_ex_teleport6.gff"              },
	{0xCF9E5341, "a2_s5_ex_teleport7.gff"              },
	{0xCFBE2A75, "prtl_sonicsle_1.ncgr.small"          },
	{0xCFBF6089, "prtl_tailssca_1.ncgr.small"          },
	{0xCFC371B6, "debuggloves.gda"                     },
	{0xCFC3A733, "cfx_wndp1.nsbtx.small"               },
	{0xCFC4637A, "crm_ex_flyout.nsbca"                 },
	{0xCFD747F9, "drnsen_aa.nsbmd.small"               },
	{0xCFDA03F9, "prtl_krsmad_3.ncgr.small"            },
	{0xCFE80F35, "ch8_scylla_tyxisshipcrash.gff"       },
	{0xCFEAEB13, "nrg_ac.nsbtx.small"                  },
	{0xCFF65BB8, "chao_ia_g.nsbmd.small"               },
	{0xCFF914C3, "lower_blue_ring_01.ncgr.small"       },
	{0xCFFAFAC5, "pfx_drp1.nsbtp.small"                },
	{0xCFFBB1D2, "chao_ca_1.ncgr.small"                },
	{0xD008B41D, "plt_plot1.plo"                       },
	{0xD0128E9E, "prtl_chbgen_3.ncgr.small"            },
	{0xD01573F0, "itm_glvblncl_0.ncgr.small"           },
	{0xD01ACC9E, "plt_plot2.plo"                       },
	{0xD01C81DD, "bel_aa.nsbtx.small"                  },
	{0xD01E0C23, "chao_hb_r.nsbmd.small"               },
	{0xD01FE940, "flee.ncgr.small"                     },
	{0xD02CE51F, "plt_plot3.plo"                       },
	{0xD02E0F8E, "gup_cb_getup.nsbca"                  },
	{0xD036106C, "itm_act2_chaos6_3.ncgr.small"        },
	{0xD038C084, "bal_cb_attack.nsbca"                 },
	{0xD03B5896, "cfx_dusp1.nsbtx.small"               },
	{0xD0423B55, "man_tsvin_21.ncgr.small"             },
	{0xD0482A47, "bif_tai.nbfp"                        },
	{0xD0482A4A, "bif_tai.nbfs"                        },
	{0xD04DF07F, "a2_s2_ex.walk.tga"                   },
	{0xD04E2195, "prtl_salegen_2.ncgr.small"           },
	{0xD05126CE, "pl_pu_emrld2.nsbtx.small"            },
	{0xD0533574, "cbt_floatybg.ncgr.small"             },
	{0xD056E406, "pl_pz_ppd_up.nsbca"                  },
	{0xD05D0FD6, "prt_tp_cre.ncgr.small"               },
	{0xD06B9998, "pfx_glf1.nsbtp.small"                },
	{0xD06C8CFE, "fx_bubbles.emit"                     },
	{0xD06ED5B8, "cmbtbg_a2_s7.nsbmd.small"            },
	{0xD08ACA66, "chao_js_g.nsbtx.small"               },
	{0xD08C8FD0, "cfx_pksp1.nsbtp.small"               },
	{0xD08DD60C, "lvl_stat_bspnl01.ncgr.small"         },
	{0xD0949527, "shp_cb_damage.nsbca"                 },
	{0xD094D1CB, "a2_s2_i01.are"                       },
	{0xD0968C55, "drnass_aa.nsbtx.small"               },
	{0xD09D89B3, "pfx_rok1.nsbtx.small"                },
	{0xD09DEAB1, "fx_wtrfl_mist.nsbtx.small"           },
	{0xD0A3FFFB, "chao_ch_3.ncgr.small"                },
	{0xD0A40471, "prt_ce_pul.ncgr.small"               },
	{0xD0A4D758, "charpro_rou0.ncgr.small"             },
	{0xD0B7FAD2, "prtl_sdwdef_1.ncgr.small"            },
	{0xD0B82F7C, "a1c2_hideoutenemy.gff"               },
	{0xD0B857C2, "a1_s1_i03_mm.nclr.small"             },
	{0xD0CA9E6F, "wmp_l_bridge.ncgr.small"             },
	{0xD0CB1B4E, "a2_s2_i04.are"                       },
	{0xD0D23EE6, "upper_portrait_03.ncgr.small"        },
	{0xD0DD33CF, "a2_s2_i05.are"                       },
	{0xD0EE750A, "cbt_btn_left.ncgr.small"             },
	{0xD0F5E5A9, "gup_cb_counter.nsbca"                },
	{0xD0FEACAF, "fx_ix_bhol_muz.nsbta.small"          },
	{0xD1015D78, "cfx_prj_min.nsbtx.small"             },
	{0xD1057D2E, "icn_s_patk_cre6.ncgr.small"          },
	{0xD108B60F, "gnp_aa.nsbtx.small"                  },
	{0xD116A01D, "lvl_tdbld_ms21.ncgr.small"           },
	{0xD1178D00, "chao_jf_3.ncgr.small"                },
	{0xD11B2E0D, "jrl_tlgbx_ms02.ncgr.small"           },
	{0xD1209624, "prt_se_voxct.ncgr.small"             },
	{0xD134DEC8, "swa_cb_counter.nsbca"                },
	{0xD13C973B, "targetring.ncgr.small"               },
	{0xD160FFDE, "prtl_krsgen_0.ncgr.small"            },
	{0xD16291B5, "mut_sonic_ms10.ncgr.small"           },
	{0xD16E2E0A, "pl_pz_swc_rg.nsbtp.small"            },
	{0xD1708649, "realtimetuttwo.gda"                  },
	{0xD17A059F, "itm_glv_cloth_1.ncgr.small"          },
	{0xD1887847, "bfl_ex_idle.nsbca"                   },
	{0xD196537F, "prtl_knugen_3.ncgr.small"            },
	{0xD19FAD03, "prtl_bigsnz_2.ncgr.small"            },
	{0xD1A629EA, "fx_a2_s7_i03_wv.nsbca"               },
	{0xD1A8C2E3, "swt_cb_pattack.nsbca"                },
	{0xD1AD63B0, "fx_a1_s4_icb.nsbmd.small"            },
	{0xD1B2FC23, "itm_fotslpdk_0.ncgr.small"           },
	{0xD1B9F35E, "a2_w0_nt_10.ncgr.small"              },
	{0xD1CE4CE9, "chao_js_r.nsbtp.small"               },
	{0xD1D4CA17, "icn_s_patk_amy2.ncgr.small"          },
	{0xD1D685F9, "fx_sha_shock.nsbtx.small"            },
	{0xD1DB1C4F, "itm_ace_shld_0.ncgr.small"           },
	{0xD1DBFF9F, "prtl_txcgen_2.ncgr.small"            },
	{0xD1E73CA1, "prtl_dadrel_1.ncgr.small"            },
	{0xD1ED217A, "tyx_aa_sld.nsbta.small"              },
	{0xD1EF87DA, "rou_cb_victoryin.nsbca"              },
	{0xD1F03DC8, "itm_ace_rocks_2.ncgr.small"          },
	{0xD1F102B0, "portraitguilayout.gui"               },
	{0xD1FD74D4, "front_readingcard.gui"               },
	{0xD2087E0B, "knu_cb_counter.nsbca"                },
	{0xD210316A, "charpro_bs_c10.ncgr.small"           },
	{0xD21682E4, "btn_l_bar_on01.ncgr.small"           },
	{0xD216A0CF, "egm_cb_rund.nsbca"                   },
	{0xD22EACFF, "bal_ex_off.nsbca"                    },
	{0xD233E027, "itm_ace_shipm_2.ncgr.small"          },
	{0xD2362018, "strongholdreserve.gui"               },
	{0xD239983B, "fx_ix_beam.nsbtx.small"              },
	{0xD2416EC5, "plt_a1_amy.plo"                      },
	{0xD24520A4, "itm_fotlgtbt_2.ncgr.small"           },
	{0xD24680BA, "cfx_calp1.nsbtx.small"               },
	{0xD247F324, "combatgui.nclr.small"                },
	{0xD24C34B2, "fx_vox_p_es.nsbta.small"             },
	{0xD256EA7D, "cfx_spkp1_grt.emit"                  },
	{0xD25EA128, "chao_je_r.nsbmd.small"               },
	{0xD271C217, "cmbtbg_a2_s2.nsbta.small"            },
	{0xD279594F, "prtl_shdgen_2.ncgr.small"            },
	{0xD27AB879, "itm_ace_nrgal_1.ncgr.small"          },
	{0xD28A01DC, "pl_am_noc_ghl.nsbca"                 },
	{0xD28CD99B, "prt_tp_e_chao.ncgr.small"            },
	{0xD290C125, "prtl_aeqhap_1.ncgr.small"            },
	{0xD293D05B, "a2_s3_ex_slide1.gff"                 },
	{0xD2943F29, "drn_cb_pattack.nsbca"                },
	{0xD29883FF, "prtl_fwrgen_2.ncgr.small"            },
	{0xD29A162D, "itm_fotslprg_0.ncgr.small"           },
	{0xD2A5E8DC, "a2_s3_ex_slide2.gff"                 },
	{0xD2AA11D7, "prt_se_scorp.ncgr.small"             },
	{0xD2AAD268, "fx_greenpuff.nsbtx.small"            },
	{0xD2B0F577, "sha_cb_idle.nsbca"                   },
	{0xD2B22C04, "chao_jl_1.ncgr.small"                },
	{0xD2BC6534, "lvl_tlgbx_ms21.ncgr.small"           },
	{0xD2C04870, "prtl_ffmrel_0.ncgr.small"            },
	{0xD2C0B399, "icn_l_barwonjrl.ncgr.small"          },
	{0xD2C1CB6E, "cfx_hdsp1.nsbmd.small"               },
	{0xD2C880EB, "a2c10_noflyzone_intro.gff"           },
	{0xD2D66D4A, "a2scy_tyxshop.dlg"                   },
	{0xD2D73541, "itm_glv_midnt_1.ncgr.small"          },
	{0xD2DEDD8E, "a2cha_overmind_thebes_leave.gff"     },
	{0xD30A0968, "pl_ob_blkb_cr.nsbmd.small"           },
	{0xD3100D9E, "prtl_wsgsca_2.ncgr.small"            },
	{0xD31179EF, "fx_els_wn_aa.nsbtx.small"            },
	{0xD3208BBD, "tai_cb_support05.nsbca"              },
	{0xD3233517, "fx_giz_lzr_prj.nsbmd.small"          },
	{0xD3370963, "bsl_cb_getup.nsbca"                  },
	{0xD33778ED, "chap6scr_bot.tga"                    },
	{0xD33C85A7, "pro_bar_on02.ncgr.small"             },
	{0xD33E87FC, "rwd_tsmbx_on02.ncgr.small"           },
	{0xD34704E0, "a1_w0_cc_00.ncgr.small"              },
	{0xD3486E8F, "jillcan13bdol.nftr"                  },
	{0xD34F9E27, "plt_teatest.plo"                     },
	{0xD35A7A2D, "chao_js_3.ncgr.small"                },
	{0xD361947C, "itm_ace_staplr_2.ncgr.small"         },
	{0xD3733524, "cho_ts_01.ncgr.small"                },
	{0xD375F4D4, "plt_fp_big.plo"                      },
	{0xD383D25A, "rou_ex_walk.nsbca"                   },
	{0xD399DC18, "chao_gc_1.ncgr.small"                },
	{0xD39EA2F6, "itm_ace_radio_3.ncgr.small"          },
	{0xD3AE16E4, "exp_ghz1_mm10.ncgr.small"            },
	{0xD3AEC2D7, "prtl_amysus_1.ncgr.small"            },
	{0xD3AF6F32, "cbt_portr_00.ncgr.small"             },
	{0xD3C977F4, "a1_w3_br_12.ncgr.small"              },
	{0xD3D81AA4, "prtl_shddef_2.ncgr.small"            },
	{0xD3D8A28E, "cbt_ystar_ms00.ncgr.small"           },
	{0xD3DCA78D, "a1_w1_gh_12.ncgr.small"              },
	{0xD3E880C9, "fx_a1_s6_ex_env.nsbmd.small"         },
	{0xD3FAEC87, "charpro_she5.ncgr.small"             },
	{0xD406D0D0, "loc_victr_fr0.ncgr.small"            },
	{0xD4130A03, "cfx_fglp1.nsbtx.small"               },
	{0xD41802E8, "cho_ts_23.ncgr.small"                },
	{0xD41B09BB, "str_tmdbx_ms10.ncgr.small"           },
	{0xD425F50E, "pfx_glo1.nsbmd.small"                },
	{0xD42D0EF2, "chao_gb_g.nsbtx.small"               },
	{0xD4403C83, "cbt_focusbad.ncgr.small"             },
	{0xD44EAB6B, "itm_chao_egg_2.ncgr.small"           },
	{0xD450CFC6, "exp_mappinq.ncgr.small"              },
	{0xD452E4A8, "exp_ghz1_mm32.ncgr.small"            },
	{0xD454BF5D, "chao_fc_r.nsbtx.small"               },
	{0xD458B504, "pfx_lef2.nsbmd.small"                },
	{0xD460C3D0, "upper_green_ring_02.ncgr.small"      },
	{0xD4683FFF, "egm_cb_pattack01.nsbca"              },
	{0xD46E0A7B, "cbt_sidebar.nanr.small"              },
	{0xD4710303, "a2_s3_ex_la.smp"                     },
	{0xD47AB10E, "itm_ace_prune_0.ncgr.small"          },
	{0xD4872E43, "gui_jrnl.nclr.small"                 },
	{0xD488D367, "itm_ace_life_3.ncgr.small"           },
	{0xD490088F, "charpro_stat10.ncgr.small"           },
	{0xD4914456, "prt_tp_m_sdw.ncgr.small"             },
	{0xD497372B, "spell_weakened_distracted.spl"       },
	{0xD49B76A6, "prtl_fsgmad_0.ncgr.small"            },
	{0xD49BA1B9, "chao_ia_2.ncgr.small"                },
	{0xD49D064F, "pfx_dop1.nsbmd.small"                },
	{0xD49FD950, "attack5_bs.tga"                      },
	{0xD4A596B2, "main_pnl_on13.ncgr.small"            },
	{0xD4B3B84A, "str_tsmbx_ms10.ncgr.small"           },
	{0xD4D0D5A6, "exp_qstcheck.ncgr.small"             },
	{0xD4D4A18B, "icn_s_str_equip.ncgr.small"          },
	{0xD4DB938F, "prtl_gunoff.nclr.small"              },
	{0xD4DBAAA1, "pro_back_on00.ncgr.small"            },
	{0xD4F08B12, "prtl_rougeoff_1.ncgr.small"          },
	{0xD4F30E49, "itm_accangam_2.ncgr.small"           },
	{0xD4FD775A, "chapter3warp.gff"                    },
	{0xD4FDE416, "gup_aa.nsbtx.small"                  },
	{0xD50B146B, "pl_pz_brd_1up.nsbca"                 },
	{0xD512C10A, "movement.gda"                        },
	{0xD529DA55, "chao_da_3.ncgr.small"                },
	{0xD52B8918, "pl_mv_tor_prk.nsbca"                 },
	{0xD52C1247, "chapter9scylla.gda"                  },
	{0xD5304162, "fx_sha_jav.nsbtx.small"              },
	{0xD531F5CB, "fx_vox_r_es.nsbtx.small"             },
	{0xD5409355, "nes_aa.nsbmd.small"                  },
	{0xD548E10A, "chao_ha_g.nsbtp.small"               },
	{0xD54B7F2B, "pl_pz_rc_cd2.nsbca"                  },
	{0xD55AE430, "cfx_exbp3.nsbmd.small"               },
	{0xD5709175, "chao_gb_r.nsbtp.small"               },
	{0xD5714A39, "c_soam_am.nsbca"                     },
	{0xD5861FC0, "chasesolo.gda"                       },
	{0xD58FC3E5, "fx_vox_g_es.nsbmd.small"             },
	{0xD593D3FE, "a1_s3_ex.are"                        },
	{0xD59DDB21, "pl_am_rdr_act.nsbca"                 },
	{0xD5B3275E, "a1_s1_i02_ts.nbfp"                   },
	{0xD5B32761, "a1_s1_i02_ts.nbfs"                   },
	{0xD5CB0D66, "pro_medbx_12.ncgr.small"             },
	{0xD5CD8E32, "zz_well.are"                         },
	{0xD5D4610D, "jrl_circl_tb00.ncgr.small"           },
	{0xD5D8D7C2, "mut_globe_ms01.ncgr.small"           },
	{0xD5E3DF23, "cfx_ferp1_roc.emit"                  },
	{0xD5EBAC02, "cfx_disp1.nsbtp.small"               },
	{0xD5ED8C8B, "conv_bubble_ts00.ncgr.small"         },
	{0xD5F840F8, "upper_red_ring_00.ncgr.small"        },
	{0xD5FA680E, "drn_ex_run.nsbca"                    },
	{0xD6037DFD, "prtl_scygen_0.ncgr.small"            },
	{0xD609610E, "a1_s4_card_t1.nbfp"                  },
	{0xD6096111, "a1_s4_card_t1.nbfs"                  },
	{0xD60E69B7, "a1c5_p5_shipcannondestroyed.gff"     },
	{0xD61C9354, "cmbtbg_a1_s4.nsbmd.small"            },
	{0xD61FD8B8, "exl_lrgbt_hl.ncer.small"             },
	{0xD62DED27, "itm_ace_delux_1.ncgr.small"          },
	{0xD62E739A, "cfx_gwtr_pu.nsbta.small"             },
	{0xD6315DC6, "icn_s_patkamy.ncgr.small"            },
	{0xD63AEF59, "charpro_big5.ncgr.small"             },
	{0xD64B474F, "prt_tp_m_amy.ncgr.small"             },
	{0xD65298B9, "prtl_txmvar_3.ncgr.small"            },
	{0xD65BCF6E, "prt_se_swt.ncgr.small"               },
	{0xD65D87BC, "itm_ace_cural_1.ncgr.small"          },
	{0xD6656CCB, "bmp_cb_kd.nsbca"                     },
	{0xD66DA3F7, "chao_ie_g.nsbtx.small"               },
	{0xD6705873, "chao_jp_r.nsbmd.small"               },
	{0xD6756E9D, "big_cb_pattack06.nsbca"              },
	{0xD67EE138, "amy_ex_smash.nsbca"                  },
	{0xD680FAEE, "big_cb_damage.nsbca"                 },
	{0xD6978CCF, "jrl_tsmbr_on00.ncgr.small"           },
	{0xD6A99EC2, "cbt_rndnum2.ncgr.small"              },
	{0xD6AEF31F, "jurnl_mis_line01.ncgr.small"         },
	{0xD6CD9E7D, "exp_ghq1_mm12.ncgr.small"            },
	{0xD6CF0B6C, "prtl_tdadhap_2.ncgr.small"           },
	{0xD6D3F3A5, "cbt_verti_ms00.ncgr.small"           },
	{0xD6DD6434, "armboar_ab.nsbmd.small"              },
	{0xD6E18092, "itm_ace_beaca_2.ncgr.small"          },
	{0xD6E44714, "fx_vox_b_ea_l.nsbtx.small"           },
	{0xD6E58643, "itm_fotkevbt_2.ncgr.small"           },
	{0xD6E9A2E5, "prtl_vxcleugen_0.ncgr.small"         },
	{0xD6ECB56A, "defend.ncgr.small"                   },
	{0xD6EFF0B7, "ch4_shadeblockade.gda"               },
	{0xD6F1F091, "pl_am_noc_gslr.nsbta.small"          },
	{0xD704C8F5, "block_walkmesh.tga"                  },
	{0xD70FF858, "rou_cb_getup.nsbca"                  },
	{0xD71B153E, "plt_gbl_puzzles.plo"                 },
	{0xD72498E4, "fx_a1_s1_ex_env.nsbmd.small"         },
	{0xD72F9D40, "cho_tmbtn_on02.ncgr.small"           },
	{0xD7319891, "prtl_sbtmad_2.ncgr.small"            },
	{0xD7321F81, "slg_cb_idle.nsbca"                   },
	{0xD7377320, "pl_mov_arw_aa.nsbtx.small"           },
	{0xD749EBAD, "fx_gren_sha.nsbtx.small"             },
	{0xD75F47D1, "prt_sp_rou.ncgr.small"               },
	{0xD75F877E, "fx_drn_lzr_beam.nsbta.small"         },
	{0xD762953B, "chao_aa_r.nsbmd.small"               },
	{0xD7664B6B, "icn_s_jump.ncgr.small"               },
	{0xD7668DA2, "main_action_on00.ncgr.small"         },
	{0xD7689F45, "rwd_ttxtb_ms00.ncgr.small"           },
	{0xD768E933, "a1_w2_ul.nbfp"                       },
	{0xD768E936, "a1_w2_ul.nbfs"                       },
	{0xD76F9A81, "exp_qstexclm.ncgr.small"             },
	{0xD777D488, "prtl_vxcbadgen_2.ncgr.small"         },
	{0xD7823CB8, "lower_big_ball.ncgr.small"           },
	{0xD789760F, "chao_jd_g.nsbtp.small"               },
	{0xD78A23B8, "pl_pz_tyx_ans.nsbmd.small"           },
	{0xD78FC4B8, "a2c8_scy_arrestedtransition2.gff"    },
	{0xD79A43F9, "prtl_bigtht_3.ncgr.small"            },
	{0xD7B1267A, "chao_ie_r.nsbtp.small"               },
	{0xD7BF2A16, "gui_items_accessories4.nclr.small"   },
	{0xD7C63EFA, "chao_fe_2.ncgr.small"                },
	{0xD7D068DB, "prt_sp_she.ncgr.small"               },
	{0xD7D1B980, "a2c10_sewer_drain_2.gff"             },
	{0xD7DEBFBE, "gzs_aa.nsbtx.small"                  },
	{0xD7E418C0, "gui_iconsgrey.nclr.small"            },
	{0xD7E64F33, "str_tab_01.ncgr.small"               },
	{0xD7F6C834, "bsc_cb_damage.nsbca"                 },
	{0xD7F7264A, "pl_pz_swc_dm.nsbtx.small"            },
	{0xD803664A, "a2cha_overmind_thebes_buddy1.gff"    },
	{0xD803C1F3, "btn_arrowup_on.ncgr.small"           },
	{0xD806DDEB, "bsl_aa.nsbtx.small"                  },
	{0xD8157ECB, "a2cha_overmind_thebes_buddy2.gff"    },
	{0xD827974C, "a2cha_overmind_thebes_buddy3.gff"    },
	{0xD82CDC0B, "loc_abshd_es1.ncgr.small"            },
	{0xD839AFCD, "a2cha_overmind_thebes_buddy4.gff"    },
	{0xD83F6F39, "a2c10_charyb.dlg"                    },
	{0xD85E89AF, "a1_s4_card_t2.nbfp"                  },
	{0xD85E89B2, "a1_s4_card_t2.nbfs"                  },
	{0xD861DA1E, "cbt_enemy_ambush.nanr.small"         },
	{0xD87DFF0A, "btn_s_bar_on00.ncgr.small"           },
	{0xD8834F66, "pfx_rng1.nsbtp.small"                },
	{0xD893B430, "a1_w0_br_11.ncgr.small"              },
	{0xD8A2F25A, "cfx_ibep1.nsbtx.small"               },
	{0xD8ADF4CF, "pl_mv_wnc_aa.nsbtx.small"            },
	{0xD8B2384E, "itm_ace_psych_3.ncgr.small"          },
	{0xD8B3DB30, "she_ab.nsbmd.small"                  },
	{0xD8B44C07, "c_sota_sb.nsbca"                     },
	{0xD8B73145, "worldmap.gda"                        },
	{0xD8B8854C, "emeralds.gda"                        },
	{0xD8C74085, "a1c3_reunion_hold2.gff"              },
	{0xD8C7CD48, "dft_scr_00.ncgr.small"               },
	{0xD8C8049B, "chao_hc_3.ncgr.small"                },
	{0xD8CE39A8, "lvl_pow_bspnl12.ncgr.small"          },
	{0xD8D5E967, "chao_ji_r.nsbtx.small"               },
	{0xD8D80FEA, "icn_s_catk_et.ncgr.small"            },
	{0xD8D95906, "a1c3_reunion_hold3.gff"              },
	{0xD8F0FA5C, "str_txtbr_di00.ncgr.small"           },
	{0xD8F1F929, "gui_icons.nclr.small"                },
	{0xD900909C, "ix_cb_idle.nsbca"                    },
	{0xD90539AB, "lvl_pow_pointoff.ncgr.small"         },
	{0xD909EB09, "a2_w0_tx_11.ncgr.small"              },
	{0xD90B22C2, "sha_cb_pattack01.nsbca"              },
	{0xD90D7BFE, "cmbtbg_a2_s7_i03.nsbtx.small"        },
	{0xD91AE250, "itm_fotirnhy_1.ncgr.small"           },
	{0xD91E18E9, "a1_w3_ai_11.ncgr.small"              },
	{0xD93E850C, "pl_pz_ppv_dn.nsbtp.small"            },
	{0xD94047A8, "a2c10_sonicenterchamber.gff"         },
	{0xD9485AF3, "loc_trppd_es1.ncgr.small"            },
	{0xD95857ED, "itm_ace_voxai_2.ncgr.small"          },
	{0xD95EC596, "a2_s3_ex_mm.nclr.small"              },
	{0xD9770B4D, "a1_s4_i01.walk.tga"                  },
	{0xD97832B5, "chaogarden.nsbtx.small"              },
	{0xD9845018, "ixa_cb_pattack1.nsbca"               },
	{0xD991010B, "adv_big.gda"                         },
	{0xD9A6B080, "cfx_ch4.nsbtp.small"                 },
	{0xD9AA77F4, "big_cb_getup.nsbca"                  },
	{0xD9B07514, "prt_se_prixa.ncgr.small"             },
	{0xD9B30F3C, "invent_stat_tab3.ncgr.small"         },
	{0xD9BFB1BD, "a1_w3_mr_01.ncgr.small"              },
	{0xD9C349B3, "cmbtbg_ts_ghz110.ncgr.small"         },
	{0xD9CF897C, "camerapanmetropolis.gff"             },
	{0xD9D34F0C, "itm_ace_fire_2.ncgr.small"           },
	{0xD9E30874, "cbt_flruna_1.ncgr.small"             },
	{0xD9EB8205, "a1c3_salesman.dlg"                   },
	{0xD9ED564F, "pl_pz_crn_turnf.nsbca"               },
	{0xD9F11650, "itm_fotslpnm_1.ncgr.small"           },
	{0xDA00DB81, "fx_mflash.nsbtx.small"               },
	{0xDA01E8CA, "cfx_ch1.nsbmd.small"                 },
	{0xDA02930C, "placeablestates.gda"                 },
	{0xDA10E235, "prtl_amyhap_1.ncgr.small"            },
	{0xDA13B598, "amy_ex_jump.nsbca"                   },
	{0xDA1A1159, "pop_tggle_of00.ncgr.small"           },
	{0xDA1ADFE7, "cfx_ixdp1.nsbtp.small"               },
	{0xDA1B21EB, "icn_s_str_tech.ncgr.small"           },
	{0xDA20F1D4, "cbt_inprofatbar.ncgr.small"          },
	{0xDA301F0B, "pfx_drp1_drp.emit"                   },
	{0xDA451686, "ccw_n_w.gff"                         },
	{0xDA4B066B, "charpro_bs_b03.ncgr.small"           },
	{0xDA51AA23, "icn_l_baseatk.ncgr.small"            },
	{0xDA75111D, "a1_s3_giantworm1_appear.gff"         },
	{0xDA78BF08, "lvl_stat_dash_on.ncgr.small"         },
	{0xDA7DA2DB, "plt_lo_warpbelts.plo"                },
	{0xDA82ECAC, "itm_act2_chaos7_2.ncgr.small"        },
	{0xDA8EFF1D, "journaloverhangpanel.gui"            },
	{0xDA90BEF0, "prtl_soniccon_0.ncgr.small"          },
	{0xDAA0A1B9, "chap4scr_top.tga"                    },
	{0xDAA47EDF, "man_tsvst_on00.ncgr.small"           },
	{0xDAB26316, "bsl_ex_counter.nsbca"                },
	{0xDAB4FE71, "charpro_cre.nclr.small"              },
	{0xDABD5DF6, "itm_glv_rock_3.ncgr.small"           },
	{0xDACA532B, "cfx_fthr_drt.nsbmd.small"            },
	{0xDACF8EAF, "pfx_spark.nsbtp.small"               },
	{0xDAD51347, "introscr_bot.tga"                    },
	{0xDAD764AF, "itm_ace_boy_1.ncgr.small"            },
	{0xDAD92721, "a2_shade.dlg"                        },
	{0xDAEAC9E2, "a2c10_p4_meet_gizoids.gff"           },
	{0xDAF28FC6, "icn_l_manual.ncgr.small"             },
	{0xDAF47D3F, "prtl_shdint_1.ncgr.small"            },
	{0xDAF518B2, "prtl_rougegen_2.ncgr.small"          },
	{0xDAF54B53, "mer_aa.nsbmd.small"                  },
	{0xDAFE22F3, "itm_ace_power_2.ncgr.small"          },
	{0xDB12B069, "plt_cod_amy.plo"                     },
	{0xDB1997BB, "movielist.gda"                       },
	{0xDB2D1AB5, "charpro_son3.ncgr.small"             },
	{0xDB2EECE3, "prtl_nrqge2_0.ncgr.small"            },
	{0xDB3042E2, "ne_sw.gff"                           },
	{0xDB3A3BDB, "conv_bubble_bs11.ncgr.small"         },
	{0xDB48549B, "tai_cb_counter.nsbca"                },
	{0xDB4E84B5, "prtl_knumad_1.ncgr.small"            },
	{0xDB52D1DD, "gui_victory.nclr.small"              },
	{0xDB5FB8AA, "a1_w0_ai.nclr.small"                 },
	{0xDB61E344, "plt_cod_angelisland.plo"             },
	{0xDB646940, "chao_jg_2.ncgr.small"                },
	{0xDB66738E, "fx_gun_scrn.nsbta.small"             },
	{0xDB6C5D3F, "nrg_cb_attackc.nsbca"                },
	{0xDB763224, "invent_icon_up.ncgr.small"           },
	{0xDB8E6A4B, "prtl_sonicgen_1.ncgr.small"          },
	{0xDB9A40FE, "cyclone_voxai.are"                   },
	{0xDB9B2D5A, "chao_jo_g.nsbtp.small"               },
	{0xDB9C295A, "fx_els_fr_aa.nsbtp.small"            },
	{0xDB9DCC14, "a1c1_combat_tutorial.gff"            },
	{0xDBA85C73, "nrg_cb_kd.nsbca"                     },
	{0xDBA8EB97, "tut_smash.gda"                       },
	{0xDBABEA1C, "son_cb_blockd.nsbca"                 },
	{0xDBB67A58, "son_ex_twitch.nsbca"                 },
	{0xDBC5FB42, "inventorymaintab.gui"                },
	{0xDBCA6BEA, "rou_ex_fall.nsbca"                   },
	{0xDBD9A0BD, "rhk_cb_attackc.nsbca"                },
	{0xDBEA385F, "cmbtbg_a1_s3.nsbca"                  },
	{0xDBFB4F99, "jurnl_ts13.ncgr.small"               },
	{0xDC09B1BC, "cbt_ambush.ncgr.small"               },
	{0xDC1304CD, "egm_ex_damage.nsbca"                 },
	{0xDC14C9AE, "nrg_cb_pattack02.nsbca"              },
	{0xDC198648, "fx_a2_s5_env.nsbtx.small"            },
	{0xDC1A24E2, "fx_vox_p_ea_l.nsbtx.small"           },
	{0xDC2B8199, "chao_ja_g.nsbmd.small"               },
	{0xDC2B9966, "shaball_aa.nsbmd.small"              },
	{0xDC318A85, "prtl_shdmad_0.ncgr.small"            },
	{0xDC43560C, "prf_tboxb_ms12.ncgr.small"           },
	{0xDC443D85, "charpro_ome2.ncgr.small"             },
	{0xDC44AA49, "icn_s_patk_cre1.ncgr.small"          },
	{0xDC4F6F33, "egb_cb_pattack.nsbca"                },
	{0xDC50B535, "prtl_fwrmad_0.ncgr.small"            },
	{0xDC533204, "chao_ib_r.nsbmd.small"               },
	{0xDC5C8CCB, "ch5_wormhole.gda"                    },
	{0xDC653F72, "prtl_eixthrngen_2.ncgr.small"        },
	{0xDC6A9450, "pop_lgbox_ms03.ncgr.small"           },
	{0xDC6E8B75, "cfx_slgp1.nsbmd.small"               },
	{0xDC72CF24, "she_ex_walk.nsbca"                   },
	{0xDC730FE3, "pfx_ash1.nsbtx.small"                },
	{0xDC84F20E, "pl_pz_cav_end.nsbca"                 },
	{0xDC972E84, "icn_l_useitem.ncgr.small"            },
	{0xDC9A90F6, "cw_e_s.gff"                          },
	{0xDCB0A688, "startupscreenwotxt.tga"              },
	{0xDCBCA9CB, "btn_buton_on.ncgr.small"             },
	{0xDCBF89B7, "c_sotaamkn_1_am.nsbca"               },
	{0xDCC141AB, "agile_tutorial01_ts.tga"             },
	{0xDCC8B411, "debugsplash.gda"                     },
	{0xDCCDD9B9, "prtl_tailsinq_0.ncgr.small"          },
	{0xDCD19E36, "conv_bubble_amb02.ncgr.small"        },
	{0xDCD2A97A, "egb_cb_blockd.nsbca"                 },
	{0xDCDDFE34, "cfx_spbp1.nsbmd.small"               },
	{0xDCDF53EE, "cbt_focusringb.ncgr.small"           },
	{0xDCE7A0B2, "chao_jt_r.nsbtx.small"               },
	{0xDCF2BAF2, "movies.gda"                          },
	{0xDCF5F72D, "itm_glv_fire_3.ncgr.small"           },
	{0xDCF8DF06, "prtl_aeqgen_1.ncgr.small"            },
	{0xDCFF0844, "chao_jm_0.ncgr.small"                },
	{0xDD06C4CD, "act2_world0.walk.tga"                },
	{0xDD197158, "pl_ob_blkb_cb.nsbmd.small"           },
	{0xDD239D4B, "charpip.ncgr.small"                  },
	{0xDD25EE45, "fx_wtrfl_sprkl.nsbmd.small"          },
	{0xDD361CF3, "loc_ambsh_fr0.ncgr.small"            },
	{0xDD4ACAF8, "cfx_heap1.nsbtx.small"               },
	{0xDD50DAAA, "prtl_sdwsmg_1.ncgr.small"            },
	{0xDD54C26D, "screenwipe.gui"                      },
	{0xDD5D02AC, "exploreminimapradar.gui"             },
	{0xDD5EDAAA, "plt_cod_robotnik.plo"                },
	{0xDD632388, "pl_ob_spk_aa.nsbmd.small"            },
	{0xDD7B45DF, "lvl_stat_tspnl11.ncgr.small"         },
	{0xDD7C27DC, "pfx_nicktest01.nsbtx.small"          },
	{0xDD84291D, "prf_tmbar_on00.ncgr.small"           },
	{0xDDA0F5DB, "prtl_sciexh_3.ncgr.small"            },
	{0xDDA3F6B0, "jrl_tlgbx_ms41.ncgr.small"           },
	{0xDDA7566D, "chao_jt_2.ncgr.small"                },
	{0xDDC38DFD, "rou_cb_pattack02.nsbca"              },
	{0xDDC39FF7, "exp_cct1_mm03.ncgr.small"            },
	{0xDDCA1F84, "cfx_javp1_jav.emit"                  },
	{0xDDDDDC7A, "plt_a1sq_factory.plo"                },
	{0xDDE6B858, "chao_gd_0.ncgr.small"                },
	{0xDDE85525, "a1_w0_ai_10.ncgr.small"              },
	{0xDDE9B5FA, "fx_sha_rift_muz.nsbta.small"         },
	{0xDDF44395, "cfx_fwnp1.nsbtx.small"               },
	{0xDDF6687D, "a2_shadejoinpartyselect.gff"         },
	{0xDE05ACA3, "pfx_puring.nsbtp.small"              },
	{0xDE094D84, "cfx_egip1.nsbmd.small"               },
	{0xDE2471FE, "bel_cb_dead.nsbca"                   },
	{0xDE2FB448, "exp_ghz1_fw11.ncgr.small"            },
	{0xDE320623, "gui_combat.nclr.small"               },
	{0xDE375E14, "jrl_tmbox_ms00.ncgr.small"           },
	{0xDE3B3A1C, "lower_sonic_portrait_01.ncgr.small"  },
	{0xDE3CA13C, "explorehudroot.gui"                  },
	{0xDE3E10A6, "a2c8_nrrgal_guide.gff"               },
	{0xDE436447, "cfx_bugp1_glt.emit"                  },
	{0xDE448AC0, "button_pressed.tga"                  },
	{0xDE45C2C2, "prtl_salesmg_2.ncgr.small"           },
	{0xDE4B5042, "a1_s2_i02_mm.nclr.small"             },
	{0xDE54FEDF, "a2c10_shadedisappears2.gff"          },
	{0xDE59AF1E, "tai_cb_blockout.nsbca"               },
	{0xDE5B657E, "itm_ace_drink_2.ncgr.small"          },
	{0xDE747276, "prtl_amyoff_0.ncgr.small"            },
	{0xDE7556C6, "mmp_globe_1.ncgr.small"              },
	{0xDE76A608, "chap1scr_bot.tga"                    },
	{0xDE7A4C7C, "big_aa.nsbtx.small"                  },
	{0xDE7D03F2, "egb_ex_walk.nsbca"                   },
	{0xDE80B718, "a2tyx_enemy_enter.gff"               },
	{0xDE82E593, "cfx_kipp1.nsbmd.small"               },
	{0xDE89EDF9, "a1_w0_mr_00.ncgr.small"              },
	{0xDE93E4BD, "big_cb_dead.nsbca"                   },
	{0xDEA4F9B4, "pl_am_noc_gnf.nsbta.small"           },
	{0xDEA9AB45, "a1_findcheese.gff"                   },
	{0xDEABC983, "cfx_fgrp3.nsbtp.small"               },
	{0xDEB6ED5D, "itm_ace_jammr_2.ncgr.small"          },
	{0xDEBEEEDD, "pfx_bub3_lav.emit"                   },
	{0xDEC22193, "cfx_crzbns.nsbtp.small"              },
	{0xDEC3323A, "amy_ex_idle.nsbca"                   },
	{0xDED4820C, "exp_ghz1_fw33.ncgr.small"            },
	{0xDEDC2BD8, "jrl_tmbox_ms22.ncgr.small"           },
	{0xDEE055C2, "pig_cb_getup.nsbca"                  },
	{0xDEE1612F, "pl_ob_blkn_cr.nsbtx.small"           },
	{0xDEE2AE6C, "prtl_fmnang_1.ncgr.small"            },
	{0xDEE87DF9, "chao_ib_1.ncgr.small"                },
	{0xDEF199A6, "son_cb_rund.nsbca"                   },
	{0xDEF4AE31, "fx_gnp_himissile.nsbtx.small"        },
	{0xDF091761, "pl_mv_tor_aa.nsbmd.small"            },
	{0xDF106F93, "bsc_cb_idle.nsbca"                   },
	{0xDF1A1EB8, "icn_l_ondash3.ncgr.small"            },
	{0xDF21F381, "prt_tp_e_ome.ncgr.small"             },
	{0xDF233876, "knu_cb_block.nsbca"                  },
	{0xDF2A6D85, "a2_s5_ex_la.smp"                     },
	{0xDF2D6370, "a2cha_overmind_thug1.gff"            },
	{0xDF3267FA, "hud_lvlnum8.ncgr.small"              },
	{0xDF3522C4, "egm_cb_blockc.nsbca"                 },
	{0xDF3917F6, "icn_l_patktai.ncgr.small"            },
	{0xDF3A19A2, "charpro_she0.ncgr.small"             },
	{0xDF3F6C1E, "itm_fotrunsp_1.ncgr.small"           },
	{0xDF3F7BF1, "a2cha_overmind_thug2.gff"            },
	{0xDF5107E8, "sha_cb_kd.nsbca"                     },
	{0xDF7A9495, "a1sq_scientist.dlg"                  },
	{0xDF7FC6E2, "inventorylist.gui"                   },
	{0xDF812B53, "amy_cb_blockd.nsbca"                 },
	{0xDF84DE41, "prtl_knudet_2.ncgr.small"            },
	{0xDF8596E9, "a2scy_queen.dlg"                     },
	{0xDF8BBB8F, "amy_ex_twitch.nsbca"                 },
	{0xDF8DF4AC, "prtl_knusmg_3.ncgr.small"            },
	{0xDF98520B, "prtl_krwcon_3.ncgr.small"            },
	{0xDFAC2BD8, "cmbtbg_a1_i02.nsbmd.small"           },
	{0xDFC23D35, "cfx_ixmp1_ixd.emit"                  },
	{0xDFCCCE49, "itm_ace_noctn_3.ncgr.small"          },
	{0xDFD4FF9B, "charpro_ts_a11.ncgr.small"           },
	{0xDFDB6725, "prtl_skpcon_3.ncgr.small"            },
	{0xDFDD6DAA, "she_ex_stealth.nsbca"                },
	{0xDFE7E657, "fx_dustpuff.nanr.small"              },
	{0xDFE823F4, "pfx_gul1.nsbmd.small"                },
	{0xDFFED084, "amy_cb_counter.nsbca"                },
	{0xE012E313, "lvl_tsstr_ms12.ncgr.small"           },
	{0xE01F2D39, "exl_bchao_ms1.ncgr.small"            },
	{0xE0339984, "fx_vox_g_ea_r.nsbmd.small"           },
	{0xE036A98A, "cfx_ch6.nsbtx.small"                 },
	{0xE03D38E4, "chao_jl_g.nsbmd.small"               },
	{0xE042A725, "exp_wel1_mm12.ncgr.small"            },
	{0xE06234D3, "chao_hb_g.nsbtx.small"               },
	{0xE066A683, "a1_w2_cc_01.ncgr.small"              },
	{0xE067E411, "prtl_shddet_1.ncgr.small"            },
	{0xE06ECDFB, "plt_cod_gizoids.plo"                 },
	{0xE0721F1A, "bmp_cb_idle.nsbca"                   },
	{0xE078E1C3, "kdr_cb_getup.nsbca"                  },
	{0xE07D37C0, "tyx_cb_idle.nsbca"                   },
	{0xE084866F, "shp_ex_walk.nsbca"                   },
	{0xE08661D4, "pig_ex_damage.nsbca"                 },
	{0xE089E53E, "chao_gc_r.nsbtx.small"               },
	{0xE09AE66F, "a1c5_p7_6_shade.gff"                 },
	{0xE0A24A47, "exl_barow_ms10.ncgr.small"           },
	{0xE0A33125, "jurnl_bs02.ncgr.small"               },
	{0xE0A5628B, "pfx_haz1_pnk.emit"                   },
	{0xE0A93F6E, "a1c5_p9_shaderemoved.gff"            },
	{0xE0CC5E98, "gup_cb_kd.nsbca"                     },
	{0xE0D14902, "pl_mv_boa_aa.nsbta.small"            },
	{0xE0E2AD1F, "prtl_scyclm_0.ncgr.small"            },
	{0xE0E560E6, "exl_globe_ms2.ncgr.small"            },
	{0xE11DBB27, "prtl_knuexh_0.ncgr.small"            },
	{0xE121362A, "jrl_scrll_tk10.ncgr.small"           },
	{0xE12796A2, "prtl_bighap_3.ncgr.small"            },
	{0xE137294D, "prtl_tdadgen_2.ncgr.small"           },
	{0xE141E7E9, "itm_ace_scan_0.ncgr.small"           },
	{0xE14490E4, "a2c8_scy_syraxappears.gff"           },
	{0xE1496FA1, "pl_pz_vox_wll.nsbta.small"           },
	{0xE1507D95, "storegui_bs.tga"                     },
	{0xE1631639, "pfx_lzg1_lzg.emit"                   },
	{0xE1656ECA, "pfx_haz1.nsbtx.small"                },
	{0xE17A1C74, "charpro_big0.ncgr.small"             },
	{0xE17E06EB, "chao_ia_g.nsbtp.small"               },
	{0xE18A6D6B, "prtl_thbsad_1.ncgr.small"            },
	{0xE19704EB, "icn_s_fly1.ncgr.small"               },
	{0xE1A5B756, "chao_hb_r.nsbtp.small"               },
	{0xE1BD36B5, "cfx_ertp1.nsbtx.small"               },
	{0xE1BEE95A, "cfx_dstp1.nsbmd.small"               },
	{0xE1CA7E3D, "prtl_egbgen_1.ncgr.small"            },
	{0xE1CC7521, "rou_cb_blockc.nsbca"                 },
	{0xE1D44F5E, "exl_shoes_ms1.ncgr.small"            },
	{0xE1D72C9C, "cfx_calp1_glw.emit"                  },
	{0xE1E84D3F, "a2_s1_i02_ts.nbfp"                   },
	{0xE1E84D42, "a2_s1_i02_ts.nbfs"                   },
	{0xE1F58EE3, "cbt_focus_button_path.ncgr.small"    },
	{0xE207FC81, "movement01_ts.tga"                   },
	{0xE214E4AB, "main_pnl_big17.ncgr.small"           },
	{0xE216372D, "a1_s3_i01_3to2.gff"                  },
	{0xE2165794, "pro_lrgbx_20.ncgr.small"             },
	{0xE218A855, "ch6_vaultentrycam.gff"               },
	{0xE21F3DEC, "itm_ace_medem_0.ncgr.small"          },
	{0xE22263D9, "prtl_gnasht_0.ncgr.small"            },
	{0xE22D9CCA, "cfx_rckp1.nsbtx.small"               },
	{0xE23685E3, "fx_vox_g_ea_m.nsbta.small"           },
	{0xE23E12F4, "charpro_knu4.ncgr.small"             },
	{0xE250DF97, "cbt_victory_star2.ncgr.small"        },
	{0xE251B935, "cmbtbg_a2_s4.nsbmd.small"            },
	{0xE25AC3A7, "cbt_healthbar.ncgr.small"            },
	{0xE26A07E8, "icn_l_catk_sk.ncgr.small"            },
	{0xE26DCC6B, "she_cb_blockc.nsbca"                 },
	{0xE27692C0, "wmp_airp_on10.ncgr.small"            },
	{0xE27831D3, "fx_els_lt_aa.nsbta.small"            },
	{0xE27EF0FD, "upper_bg_06.ncgr.small"              },
	{0xE289F13C, "cbt_menu_down_of.ncgr.small"         },
	{0xE28EBB3D, "egm_ex_walk.nsbca"                   },
	{0xE2976126, "icn_s_patk_rou2.ncgr.small"          },
	{0xE29A6D80, "icn_l_utteleport.ncgr.small"         },
	{0xE2A2C9D8, "chao_je_g.nsbtx.small"               },
	{0xE2AF1002, "prtl_wkrsca_3.ncgr.small"            },
	{0xE2B554F5, "mut_chaob_ms00.ncgr.small"           },
	{0xE2CB7EAA, "itm_ace_kevlr_2.ncgr.small"          },
	{0xE2CB97BC, "kdr_cb_idle.nsbca"                   },
	{0xE2CD7BA0, "fx_shadbeam.nsbmd.small"             },
	{0xE2CE1551, "prtl_agzgen_1.ncgr.small"            },
	{0xE2D7E7E3, "a2c8_scy_sonicfollowguard.gff"       },
	{0xE2DA4E7A, "pl_am_noc_gun.nsbtx.small"           },
	{0xE2DCA6CB, "a1_s3_giantworm1_hold.gff"           },
	{0xE2E89DF3, "prtl_pilgen_0.ncgr.small"            },
	{0xE2ED2D4C, "pfx_arnietest02.nsbmd.small"         },
	{0xE2F18EBE, "a1_s6_card_b1.nbfp"                  },
	{0xE2F18EC1, "a1_s6_card_b1.nbfs"                  },
	{0xE2F598AC, "a1c5_p7_7_ix.gff"                    },
	{0xE3077D5F, "itm_fotrsthy_0.ncgr.small"           },
	{0xE314E0DB, "chao_hd_2.ncgr.small"                },
	{0xE3155D8F, "a2c10_ix_attack_sonic.gff"           },
	{0xE3159955, "prtl_tailsugh_0.ncgr.small"          },
	{0xE31F547F, "journalinfo.gui"                     },
	{0xE3261BE6, "a2sq_voxdad.dlg"                     },
	{0xE3324692, "a1c3_campan_shadow_1.gff"            },
	{0xE33529E8, "itm_act2_chaos2_3.ncgr.small"        },
	{0xE3377A5B, "spell_int_animal_land.spl"           },
	{0xE33793E0, "icn_s_dash1.ncgr.small"              },
	{0xE3422045, "prtl_shdsht_3.ncgr.small"            },
	{0xE34D0A94, "a1_w0_br.nclr.small"                 },
	{0xE3543746, "exl_altbx_ms00.ncgr.small"           },
	{0xE35BAAC7, "a2c8_scy_returnnrrgal.gff"           },
	{0xE363B2FC, "a1c5_p6_guardian_enter.gff"          },
	{0xE3654CF7, "crm_cb_attack.nsbca"                 },
	{0xE36C14AD, "prtl_credet_2.ncgr.small"            },
	{0xE370BD78, "spell_cursed_2.spl"                  },
	{0xE375F7D3, "itm_accnewsp_3.ncgr.small"           },
	{0xE37884CD, "fx_windsock.nsbmd.small"             },
	{0xE382D5F9, "spell_cursed_3.spl"                  },
	{0xE3898B2A, "a2c8_scy_sonicwalkqueen.gff"         },
	{0xE39256BA, "cfx_glw_wavp1.emit"                  },
	{0xE397BB1C, "chao_ba_r.nsbmd.small"               },
	{0xE3996EAF, "loc_abshd_it0.ncgr.small"            },
	{0xE39E515E, "lvl_tdbld_ms13.ncgr.small"           },
	{0xE3A046A5, "health.tga"                          },
	{0xE3AD2DE1, "ambientlower.gui"                    },
	{0xE3B00620, "exp_topframe2.ncgr.small"            },
	{0xE3B42BA7, "prtl_aeqsec_1.ncgr.small"            },
	{0xE3CCCE56, "cfx_ixmp2_ixd.emit"                  },
	{0xE3D2C518, "pl_pu_emrld7.nsbmd.small"            },
	{0xE3D49BAA, "plt_cod_cream.plo"                   },
	{0xE3DC8710, "c_knshashe_she.nsbca"                },
	{0xE3DFABAB, "fx_vox_r_ea_s.nsbtx.small"           },
	{0xE3E1B657, "itm_ace_tools_2.ncgr.small"          },
	{0xE3E64C5B, "chao_je_r.nsbtp.small"               },
	{0xE3FC95EF, "gnp_ex_run.nsbca"                    },
	{0xE4008128, "sha_cb_blockd.nsbca"                 },
	{0xE40A04AE, "a2sq_sonicstalker.gff"               },
	{0xE40B1164, "sha_ex_twitch.nsbca"                 },
	{0xE40F0212, "cfx_aurp1_aur.emit"                  },
	{0xE416A67C, "chao_jb_3.ncgr.small"                },
	{0xE418BF19, "prtl_gizgen_1.ncgr.small"            },
	{0xE41965D5, "a1c4_shade_sonichold.gff"            },
	{0xE4387EB6, "btn_l_lftarw_of.ncgr.small"          },
	{0xE4469ED8, "a2sq_telra.dlg"                      },
	{0xE44781A1, "hnpc_tim_aa.nsbmd.small"             },
	{0xE44976A1, "cfx_hdsp1.nsbtp.small"               },
	{0xE452BC3E, "cbt_test.gff"                        },
	{0xE459F53F, "lvl_txbar_di01.ncgr.small"           },
	{0xE45BFB04, "cfx_omga_mf.nsbmd.small"             },
	{0xE4646D51, "fx_gren_noc.nsbtx.small"             },
	{0xE4661059, "cbt_menu_backbutton_on.ncgr.small"   },
	{0xE46C43EB, "commandselectionringmanual.gui"      },
	{0xE46CB31E, "a1c5_p2_start.gff"                   },
	{0xE4790016, "prtl_amygen_1.ncgr.small"            },
	{0xE47DAE3A, "a1_s3_ex_slide1.gff"                 },
	{0xE48FC6BB, "a1_s3_ex_slide2.gff"                 },
	{0xE496DC70, "pl_pz_ppv_up.nsbta.small"            },
	{0xE497E2AB, "charpro_bs_c02.ncgr.small"           },
	{0xE499AC5F, "pro_smbox_12.ncgr.small"             },
	{0xE49C5300, "c_soknta_ta.nsbca"                   },
	{0xE4A1DF3C, "a1_s3_ex_slide3.gff"                 },
	{0xE4A3DACD, "icn_s_patk_sha3.ncgr.small"          },
	{0xE4A6C6DD, "infoframe_c.ncgr.small"              },
	{0xE4B210E6, "pl_pz_tyx_num.nsbmd.small"           },
	{0xE4B3F7BD, "a1_s3_ex_slide4.gff"                 },
	{0xE4B4ED97, "loc_trppd_it0.ncgr.small"            },
	{0xE4B968B4, "she_ex_fall.nsbca"                   },
	{0xE4BA3784, "icn_l_utdash3.ncgr.small"            },
	{0xE4BED197, "prtl_timsca_2.ncgr.small"            },
	{0xE4C6103E, "a1_s3_ex_slide5.gff"                 },
	{0xE4D3AA06, "itm_ace_commu_1.ncgr.small"          },
	{0xE4D8183B, "cfx_icep1.nsbtx.small"               },
	{0xE4D828BF, "a1_s3_ex_slide6.gff"                 },
	{0xE4DAF741, "fx_a2_s2_ex_env.nsbtx.small"         },
	{0xE4EA4140, "a1_s3_ex_slide7.gff"                 },
	{0xE4EA9DF4, "worldmap_travelicon.gui"             },
	{0xE4ECECE5, "pl_ob_blkb_co.nsbmd.small"           },
	{0xE4EE4683, "spell_stun_all.spl"                  },
	{0xE4EF2A27, "fx_bird.nsbtx.small"                 },
	{0xE4F11554, "a2c10_ix_attack_sonic_wait_2.gff"    },
	{0xE5098F84, "charpro_tai4.ncgr.small"             },
	{0xE50F3BC2, "pop_tggle_on01.ncgr.small"           },
	{0xE516DBB7, "lvl_pow_tspnl00.ncgr.small"          },
	{0xE51EDA25, "ch10_ending_st.gda"                  },
	{0xE55362EF, "cfx_nflp1.nsbmd.small"               },
	{0xE5702BFC, "fx_a1_s6_ex_env.nsbtp.small"         },
	{0xE5972FF3, "chao_ea_0.ncgr.small"                },
	{0xE5A2DD93, "pfx_smk2.nsbtx.small"                },
	{0xE5ADA041, "pfx_glo1.nsbtp.small"                },
	{0xE5B14580, "chao_jh_1.ncgr.small"                },
	{0xE5B355D3, "a1_w2_br_12.ncgr.small"              },
	{0xE5B65A8D, "buz_cb_damage.nsbca"                 },
	{0xE5B7F436, "cbt_buttonright_01.ncgr.small"       },
	{0xE5BBA97B, "lvl_pow_tspnl22.ncgr.small"          },
	{0xE5C6856C, "a1_w0_gh_12.ncgr.small"              },
	{0xE5D04ADC, "a2c8_arenaentry_1.gff"               },
	{0xE5E06037, "pfx_lef2.nsbtp.small"                },
	{0xE5E2635D, "a2c8_arenaentry_2.gff"               },
	{0xE5EA9184, "icn_s_stealth.ncgr.small"            },
	{0xE5EA9A91, "fx_a2_s3_ex_cld.nsbmd.small"         },
	{0xE5ECA81C, "plt_cod_kron.plo"                    },
	{0xE5F47BDE, "a2c8_arenaentry_3.gff"               },
	{0xE5F74F39, "cfx_crup1.nsbmd.small"               },
	{0xE606945F, "a2c8_arenaentry_4.gff"               },
	{0xE60851AB, "exu_blbox_ms00.ncgr.small"           },
	{0xE622755A, "fx_wtrfl_mist.nsbta.small"           },
	{0xE624B182, "pfx_dop1.nsbtp.small"                },
	{0xE6346F6C, "lvl_tmstr_ms11.ncgr.small"           },
	{0xE635C825, "exp_ghz1_mm02.ncgr.small"            },
	{0xE63C927D, "prtl_vector.nclr.small"              },
	{0xE63EAA95, "exl_circe_ms31.ncgr.small"           },
	{0xE641B24A, "hnpc_gen_walk.nsbca"                 },
	{0xE65993A9, "chao_jo_3.ncgr.small"                },
	{0xE66A659B, "gui_main_icodk.nclr.small"           },
	{0xE66DFC17, "gzc_ex_idle.nsbca"                   },
	{0xE66FDD52, "itm_glv_rug_0.ncgr.small"            },
	{0xE672B1C5, "gzg_cb_counter.nsbca"                },
	{0xE67396EC, "cbt_title_slide.ncer.small"          },
	{0xE67E40CB, "pig_cb_dead.nsbca"                   },
	{0xE69B2D95, "a2c7_arrivekron.gff"                 },
	{0xE69B5219, "prtl_sonicimp_3.ncgr.small"          },
	{0xE69E146D, "main_pnl_off01.ncgr.small"           },
	{0xE69F1531, "cfx_ccap1.nsbtx.small"               },
	{0xE6A2BAFC, "str_tmdbx_ms02.ncgr.small"           },
	{0xE6A79286, "cho_lgbtn_hi02.ncgr.small"           },
	{0xE6A97A8A, "itm_accrfrsh_2.ncgr.small"           },
	{0xE6B48123, "chao_jp_g.nsbtx.small"               },
	{0xE6B5C13E, "a2_rouge.dlg"                        },
	{0xE6B7F5E1, "itm_ace_grapp_3.ncgr.small"          },
	{0xE6D3106F, "c_shaom_om.nsbca"                    },
	{0xE6D3345C, "guianims.gda"                        },
	{0xE6DA5FC7, "exp_ghz2_mm21.ncgr.small"            },
	{0xE6E1A3E7, "minimapgui02_ts.tga"                 },
	{0xE6E1DB69, "pl_pz_rc_crs.nsbtx.small"            },
	{0xE6E28F63, "cfx_exbp3.nsbtp.small"               },
	{0xE6E737EC, "a1_s3_giantworm2_hold.gff"           },
	{0xE6F4A151, "conv_portr_11.ncgr.small"            },
	{0xE6FC7213, "prtl_tailscon_2.ncgr.small"          },
	{0xE70D6DE2, "inventorylistbackground.gui"         },
	{0xE7115145, "fx_gnp_missile.nsbmd.small"          },
	{0xE718BB00, "lizkdra_aa.nsbmd.small"              },
	{0xE723ECD4, "son_ex_slideidle.nsbca"              },
	{0xE726DA20, "walkmeshdebug.ncgr"                  },
	{0xE72A9B07, "cbt_horiz_ms00.ncgr.small"           },
	{0xE7365C7A, "a1c3_sentinels_appear.gff"           },
	{0xE73B698B, "str_tsmbx_ms02.ncgr.small"           },
	{0xE74250B2, "prtl_guncodis_2.ncgr.small"          },
	{0xE7453D42, "plt_cod_knuckles.plo"                },
	{0xE745A431, "nrg_cb_attackc2.nsbca"               },
	{0xE74BB96E, "a1c5_p9_end.gff"                     },
	{0xE74E1262, "fx_sha_rift_muz.nsbca"               },
	{0xE755E074, "fx_egg_wrc_prj.nsbmd.small"          },
	{0xE7586385, "prtl_nesinq_3.ncgr.small"            },
	{0xE758E52B, "jurnl_cod_line00.ncgr.small"         },
	{0xE75B10A2, "fx_sha_shock.nsbta.small"            },
	{0xE75C0224, "pl_am_tyx_gun.nsbmd.small"           },
	{0xE75C42FF, "vox_rbe_aa.nsbtx.small"              },
	{0xE75D6E2D, "a1_s1_eggman_lair_enter.gff"         },
	{0xE7677BFC, "itm_ace_medkt_3.ncgr.small"          },
	{0xE76BFD5C, "a2_tails_stronghold.dlg"             },
	{0xE782B949, "prf_tboxa_ms10.ncgr.small"           },
	{0xE788D705, "a2_s3_ex_la_soundmap.gda"            },
	{0xE78B4B5E, "itm_ace_wpnp2_3.ncgr.small"          },
	{0xE793D030, "btn_menu_of.ncgr.small"              },
	{0xE796C971, "cfx_ch6_spkp1.emit"                  },
	{0xE7A6BDEB, "chao_aa_g.nsbtx.small"               },
	{0xE7A7E324, "kro_cb_attackc.nsbca"                },
	{0xE7A92133, "itm_ace_candy_0.ncgr.small"          },
	{0xE7AA0A98, "charpro_egg3.ncgr.small"             },
	{0xE7AB5678, "chap10scr_bot.tga"                   },
	{0xE7B64FA1, "tyx_cb_damage.nsbca"                 },
	{0xE7BE22E4, "fx_ix_beam.nsbta.small"              },
	{0xE7C42570, "debugcreaturestats.gda"              },
	{0xE7D7DA90, "camerapancentralcity.gff"            },
	{0xE7EDC836, "itm_ace_ice_1.ncgr.small"            },
	{0xE7F411C4, "cbt_black_upper.ncgr.small"          },
	{0xE7F803A6, "chao_jp_r.nsbtp.small"               },
	{0xE7FA1D6E, "prtl_tailsgen_3.ncgr.small"          },
	{0xE7FD499F, "vel_cb_attackc.nsbca"                },
	{0xE7FF76D1, "cmbtbg_a1_s1.nsbmd.small"            },
	{0xE8048417, "omg_ex_damage.nsbca"                 },
	{0xE814110F, "pfx_gul1_gul.emit"                   },
	{0xE825A0B0, "loc_victr_es0.ncgr.small"            },
	{0xE82CCAFB, "pl_pz_brd_4dn.nsbca"                 },
	{0xE82FA8C4, "icn_s_catk_skta.ncgr.small"          },
	{0xE8351958, "prtl_fmnrel_0.ncgr.small"            },
	{0xE8387131, "rwd_tlgbx_ms10.ncgr.small"           },
	{0xE843BE1E, "prt_se_voxel.ncgr.small"             },
	{0xE8477B88, "itm_glv_work_1.ncgr.small"           },
	{0xE8481F84, "itm_ace_gunof_1.ncgr.small"          },
	{0xE85E046F, "attack4_bs.tga"                      },
	{0xE87DFC38, "fleetut.gda"                         },
	{0xE87FC02D, "syr_aa.nsbmd.small"                  },
	{0xE88531EE, "exu_tftge_ms00.ncgr.small"           },
	{0xE88772AC, "prf_tpanl_ms21.ncgr.small"           },
	{0xE88857E5, "chao_jb_r.nsbmd.small"               },
	{0xE8960498, "fx_els_wn_aa.nsbta.small"            },
	{0xE89B32E8, "chaogarden_top2.gui"                 },
	{0xE8A384AD, "cfx_fgrp2_dst.emit"                  },
	{0xE8A66AB2, "fx_son_shock.nsbmd.small"            },
	{0xE8AC4417, "fx_a1_s1_ex_env.nsbtp.small"         },
	{0xE8C33D27, "gensonn_aa.nsbmd.small"              },
	{0xE8CB1FFF, "shp_ex_fall.nsbca"                   },
	{0xE8CC86E1, "fx_a1_s3_ex_env.nsbtx.small"         },
	{0xE8CE98F2, "charpro_sdw2.ncgr.small"             },
	{0xE8D5170D, "statredirect.gda"                    },
	{0xE8D6B4A6, "a2c7_firstkrondefeated.gff"          },
	{0xE8DB0882, "man_strbt_of00.ncgr.small"           },
	{0xE8EA406E, "chao_aa_r.nsbtp.small"               },
	{0xE8ECB9DF, "prtl_rougesmg_2.ncgr.small"          },
	{0xE8F0C91F, "pl_ob_blkn_cb.nsbtx.small"           },
	{0xE8F21857, "itm_glv_ice_2.ncgr.small"            },
	{0xE906C211, "prtl_salesca_1.ncgr.small"           },
	{0xE909D021, "itm_accspkam_2.ncgr.small"           },
	{0xE90C1679, "hnpc_mug_walk.nsbca"                 },
	{0xE90C62D4, "a2c7_enter_kron_interior.gff"        },
	{0xE90E2B98, "prtl_outhos_2.ncgr.small"            },
	{0xE9132322, "prtl_caphap_1.ncgr.small"            },
	{0xE9162630, "ssf_tlgbx_ms10.ncgr.small"           },
	{0xE919FDEF, "drn_cb_dead.nsbca"                   },
	{0xE91C8856, "syr_cb_attackc.nsbca"                },
	{0xE9306D4B, "cho_tsmbx_ms01.ncgr.small"           },
	{0xE9353432, "itm_accwnrng_3.ncgr.small"           },
	{0xE9355A39, "chao_ic_0.ncgr.small"                },
	{0xE93E8A1D, "prtl_vxcliggen_2.ncgr.small"         },
	{0xE93EAAE9, "prt_sp_knu.ncgr.small"               },
	{0xE941FD8E, "str_resrv_ms10.ncgr.small"           },
	{0xE9479DA0, "cfx_hypp1.nsbmd.small"               },
	{0xE951FF1D, "a1_s2_ex.are"                        },
	{0xE9651955, "debugsave.gda"                       },
	{0xE967920A, "itm_fotrunkb_3.ncgr.small"           },
	{0xE970BB25, "itm_ace_pring_1.ncgr.small"          },
	{0xE97B4888, "hnpc_wdmn_aa.nsbtx.small"            },
	{0xE97CF50D, "prtl_sonicdet_0.ncgr.small"          },
	{0xE9860B78, "prtl_sonicsmg_1.ncgr.small"          },
	{0xE986A518, "cfx_gwtr_prj.nsbtx.small"            },
	{0xE98C7185, "a1c2_hideouthold.gff"                },
	{0xE99CD2A3, "bfl_aa.nsbmd.small"                  },
	{0xE9A79AD6, "exp_cct1_mm20.ncgr.small"            },
	{0xE9BAF3F4, "ssf_tlgbx_ms32.ncgr.small"           },
	{0xE9BC7A3C, "a2c10_emeraldfreed.gff"              },
	{0xE9CC46A3, "debuglevelup.gda"                    },
	{0xE9D55696, "charpro_stat_tab2.ncgr.small"        },
	{0xE9DA7771, "itm_fotsnklt_0.ncgr.small"           },
	{0xE9E3D807, "a2_s7_ex_la.smp"                     },
	{0xE9E50AAE, "spell_tinker.spl"                    },
	{0xE9EC35D3, "ehd_fow_qc.ncgr.small"               },
	{0xEA091998, "wmp_airp_of01.ncgr.small"            },
	{0xEA11AB0F, "jrl_tsmbr_on_10.ncgr.small"          },
	{0xEA182B7B, "exp_healthbar.ncgr.small"            },
	{0xEA21DBDB, "charpro_ts_b10.ncgr.small"           },
	{0xEA31DE3B, "camerapancentralcity_intro.gff"      },
	{0xEA33D89C, "exl_circl_ms31.ncgr.small"           },
	{0xEA5716CB, "bsl_ex_run.nsbca"                    },
	{0xEA59269D, "cfx_egbt_mf.nsbtx.small"             },
	{0xEA5997DF, "bfl_cb_dead.nsbca"                   },
	{0xEA719515, "hud_lvlnum3.ncgr.small"              },
	{0xEA737CCB, "conv_icon09.ncgr.small"              },
	{0xEA7A06A9, "icn_s_mm_airport.ncgr.small"         },
	{0xEA7D0AEB, "jurnl_mis_linelit04.ncgr.small"      },
	{0xEA7FFE90, "bfl_cb_damage.nsbca"                 },
	{0xEA828DD4, "erf.dict"                            },
	{0xEA8FF47A, "a1c2_approachknuckles.gff"           },
	{0xEA913A7F, "main_pnl_backon.ncgr.small"          },
	{0xEA9207E4, "sha_cb_block.nsbca"                  },
	{0xEAB68074, "fx_vox_r_es.nsbta.small"             },
	{0xEAC55876, "chao_fa_2.ncgr.small"                },
	{0xEAC7E968, "spell_refresh_all.spl"               },
	{0xEAD554CD, "egm_ex_fall.nsbca"                   },
	{0xEADE8983, "attack6_ts.tga"                      },
	{0xEAF1C90D, "a1_s3_giantworm3_hold.gff"           },
	{0xEAF54025, "itm_glvspncl_0.ncgr.small"           },
	{0xEB07F6C8, "a1_w2_ai_11.ncgr.small"              },
	{0xEB0B61D5, "a2c9_attacknocturnmovie.gff"         },
	{0xEB0D66AC, "brdrhk_aa.nsbmd.small"               },
	{0xEB10F20A, "fx_gnp_mflash.nsbmd.small"           },
	{0xEB17932A, "a2c7_kronshop.dlg"                   },
	{0xEB26A20E, "omg_cb_blockc.nsbca"                 },
	{0xEB2D1A5A, "gui_main_icon.nclr.small"            },
	{0xEB48A375, "a2_s2_ex_mm.nclr.small"              },
	{0xEB4D8E47, "lvl_smlbt_of00.ncgr.small"           },
	{0xEB539112, "chao_aa_3.ncgr.small"                },
	{0xEB5450F1, "a1c5_p7_8_shade.gff"                 },
	{0xEB6B983B, "slg_cb_attack.nsbca"                 },
	{0xEB6C0E3F, "ch2_islandgone.gda"                  },
	{0xEB7A66A5, "a1_s4_ex_la_soundmap.gda"            },
	{0xEB819E0D, "crm_cb_pattack06.nsbca"              },
	{0xEB85A216, "loc_round_it1.ncgr.small"            },
	{0xEB8993FD, "cfx_ch1.nsbtp.small"                 },
	{0xEB8FB483, "prtl_biggen_3.ncgr.small"            },
	{0xEB938DD7, "vox_ovl_aa.nsbtx.small"              },
	{0xEBA1A794, "str_vertw_ms10.ncgr.small"           },
	{0xEBA98F9C, "a1_w2_mr_01.ncgr.small"              },
	{0xEBBE5983, "a2_w0_nt.nclr.small"                 },
	{0xEBCE5F00, "a2_s2_i05_2.are"                     },
	{0xEBD60B62, "spell_antidote_all_2.spl"            },
	{0xEBE823E3, "spell_antidote_all_3.spl"            },
	{0xEC00FB10, "a1c5_p6_guardian_defeated.gff"       },
	{0xEC024694, "min_circl_ms10.ncgr.small"           },
	{0xEC15585D, "sco_cb_attackc.nsbca"                },
	{0xEC1F05D7, "prtl_boysca_2.ncgr.small"            },
	{0xEC2E059B, "cfx_watp1.nsbmd.small"               },
	{0xEC31D57A, "tai_cb_support02.nsbca"              },
	{0xEC34B9A5, "bif_sha.nbfp"                        },
	{0xEC34B9A8, "bif_sha.nbfs"                        },
	{0xEC377A2B, "itm_fotsnkrg_1.ncgr.small"           },
	{0xEC38D0DF, "prtl_grxgen_0.ncgr.small"            },
	{0xEC4AFAF4, "cmbtbg_ts_ghz102.ncgr.small"         },
	{0xEC4BABEB, "loc_abshd_dt1.ncgr.small"            },
	{0xEC560CC9, "pfx_briantest02.nsbmd.small"         },
	{0xEC68D1BD, "fx_vox_b_ea_l.nsbta.small"           },
	{0xEC76EB7C, "conv_bubble_tab01.ncgr.small"        },
	{0xEC7A0EAB, "gencrm_aa.nsbmd.small"               },
	{0xEC7DFCB9, "cbt_xpbar.ncgr.small"                },
	{0xEC900A31, "armdiln_aa.nsbtx.small"              },
	{0xEC975AB4, "chao_ib_g.nsbtx.small"               },
	{0xEC987E5E, "genamyn_aa.nsbmd.small"              },
	{0xEC9A0F30, "chao_jm_r.nsbmd.small"               },
	{0xEC9DE398, "wmp_s_radar.nanr.small"              },
	{0xECB8192F, "itm_fotrnfhy_3.ncgr.small"           },
	{0xECBBFDC9, "pl_mov_arw_aa.nsbta.small"           },
	{0xECBF0B1F, "chao_hc_r.nsbtx.small"               },
	{0xECDB691A, "pil_cb_attack.nsbca"                 },
	{0xECE14CBA, "fx_noc_jav_prj.nsbta.small"          },
	{0xECE874C4, "a1_w3_cc_10.ncgr.small"              },
	{0xED1078C0, "charpro_stat_tablit3.ncgr.small"     },
	{0xED19162C, "cfx_jwl1.nsbmd.small"                },
	{0xED1A7B03, "prtl_txm.nclr.small"                 },
	{0xED1AEBE6, "knu_cb_pattack3.nsbca"               },
	{0xED210957, "fx_vox_y_ea_s.nsbmd.small"           },
	{0xED24F0F0, "itm_fotslpgr_3.ncgr.small"           },
	{0xED26A575, "slg_aa.nsbmd.small"                  },
	{0xED29C75C, "fx_drn_lzr_muz.nsbtx.small"          },
	{0xED2CD9A1, "ch6_egglab.gda"                      },
	{0xED3D6C78, "cfx_expp2.nsbtx.small"               },
	{0xED4AA2D5, "fx_leafdn.nsbmd.small"               },
	{0xED506139, "a1c4_warptobrz.gff"                  },
	{0xED5411C6, "main_pnl_big12.ncgr.small"           },
	{0xED5B83FB, "jurnl_arrowb.ncgr.small"             },
	{0xED61BD1B, "chao_he_1.ncgr.small"                },
	{0xED6254F2, "plt_fpa2_tails.plo"                  },
	{0xED672AD3, "loc_trppd_dt1.ncgr.small"            },
	{0xED70D499, "exp_tws1_mm01.ncgr.small"            },
	{0xED820628, "itm_act2_chaos3_2.ncgr.small"        },
	{0xED823E19, "lower_bg_05.ncgr.small"              },
	{0xED9D8019, "cfx_mexp1.nsbmd.small"               },
	{0xEDB32CCC, "chao_ja_g.nsbtp.small"               },
	{0xEDBE1E18, "upper_bg_01.ncgr.small"              },
	{0xEDD3DDA6, "edgecolors.gda"                      },
	{0xEDD704FA, "gzd_cb_attack.nsbca"                 },
	{0xEDDADD37, "chao_ib_r.nsbtp.small"               },
	{0xEDF636A8, "cfx_slgp1.nsbtp.small"               },
	{0xEDF7EBA7, "a1_s1_ex_playerwinchnorth.gff"       },
	{0xEDF9AE73, "prtl_telgen_0.ncgr.small"            },
	{0xEE0667D2, "vob_voa.gff"                         },
	{0xEE15A25D, "exp_tws1_mm23.ncgr.small"            },
	{0xEE327F78, "pl_mv_wnc_aa.nsbta.small"            },
	{0xEE343791, "cmbtbg_a1_s6.nsbtx.small"            },
	{0xEE362A67, "prtl_cresca_2.ncgr.small"            },
	{0xEE36C753, "icn_s_patk_ome4.ncgr.small"          },
	{0xEE45C323, "icn_s_statfoc.ncgr.small"            },
	{0xEE6382BC, "chao_jc_2.ncgr.small"                },
	{0xEE65A967, "cfx_spbp1.nsbtp.small"               },
	{0xEE700731, "a1_w0_ul.nbfp"                       },
	{0xEE700734, "a1_w0_ul.nbfs"                       },
	{0xEE8070A7, "icn_s_statstu.ncgr.small"            },
	{0xEE8D02EF, "rpg_downarrow.ncgr.small"            },
	{0xEE8FB77A, "wmp_trv_04_03.gff"                   },
	{0xEE9206A7, "cmbtbg_a2_s7_i03.nsbta.small"        },
	{0xEE94C330, "cameapantyxsis.gff"                  },
	{0xEE9BD04C, "chaogarden_lower.gui"                },
	{0xEEAD9978, "fx_wtrfl_sprkl.nsbtp.small"          },
	{0xEEB6F931, "itm_fotwrkbt_2.ncgr.small"           },
	{0xEEB847D2, "fx_impactflash.nanr.small"           },
	{0xEEB91D9D, "wmp_l_airpo_01.ncgr.small"           },
	{0xEEC92874, "exu_tbbox_ms10.ncgr.small"           },
	{0xEECD4CA4, "jurnl_cod00.ncgr.small"              },
	{0xEEEA31FF, "wmp_trv_04_08.gff"                   },
	{0xEEF3A4A8, "mut_circl_ms30.ncgr.small"           },
	{0xEEF6C133, "btn_scrll_di.ncgr.small"             },
	{0xEEF9822C, "lvl_stat_bspnl10.ncgr.small"         },
	{0xEEFCBD5E, "chaogarden.nsbta.small"              },
	{0xEEFFA024, "chao_jf_r.nsbtx.small"               },
	{0xEF221A8E, "itm_glv_rub_1.ncgr.small"            },
	{0xEF261DA0, "a1c5_p4_start.gff"                   },
	{0xEF2A67D2, "prtl_krggen_0.ncgr.small"            },
	{0xEF31F686, "cfx_cgrp1.nsbtx.small"               },
	{0xEF39430A, "wm_trndo_dn.tga"                     },
	{0xEF3D8E17, "a1c1_swatbot_arrive.gff"             },
	{0xEF5554DD, "gup_ex_jump.nsbca"                   },
	{0xEF55984E, "vox_hintguy.dlg"                     },
	{0xEF582BEB, "charpro_bs_a03a.ncgr.small"          },
	{0xEF5B2FDE, "a1_w3_mt_00.ncgr.small"              },
	{0xEF69B414, "son_ex_walk.nsbca"                   },
	{0xEF774626, "prtl_shdtht_3.ncgr.small"            },
	{0xEF78109E, "itm_ace_kron_0.ncgr.small"           },
	{0xEF80CC27, "itm_glv_spark_2.ncgr.small"          },
	{0xEF86B85A, "big_cb_pattack03.nsbca"              },
	{0xEF86DA2D, "jrl_tlgbx_ms11.ncgr.small"           },
	{0xEF904557, "prtl_gunhap_0.ncgr.small"            },
	{0xEF90F8B7, "cfx_egip1.nsbtp.small"               },
	{0xEFA04128, "itm_glv_crystal1.ncgr.small"         },
	{0xEFA9E0E0, "a1_w1_mr.nclr.small"                 },
	{0xEFC61BBB, "spell_revive_all_2.spl"              },
	{0xEFC97A16, "a2c10_tauntedbygizoids_2.gff"        },
	{0xEFCCE0FD, "chao_ca_r.nsbmd.small"               },
	{0xEFCCEFB8, "icn_l_comboatk.ncgr.small"           },
	{0xEFD8343C, "spell_revive_all_3.spl"              },
	{0xEFD9CB82, "a2c8_scy_sonicholdmorphout.gff"      },
	{0xEFDDA6C4, "prtl_wdmnhap_1.ncgr.small"           },
	{0xEFEC3648, "icn_l_patkson.ncgr.small"            },
	{0xEFF5730A, "prt_ce_yel.ncgr.small"               },
	{0xEFFE21C0, "chao_ji_0.ncgr.small"                },
	{0xEFFE7810, "cfx_pmip1.nsbtx.small"               },
	{0xF00650C1, "charpro_cre5.ncgr.small"             },
	{0xF007C652, "fx_amy_whirl.nsbtx.small"            },
	{0xF00A90C6, "cfx_kipp1.nsbtp.small"               },
	{0xF012635C, "cfx_gfir_prj.nsbtx.small"            },
	{0xF013A02B, "prtl_vxmgen_2.ncgr.small"            },
	{0xF01953B0, "pro_sidbx_30.ncgr.small"             },
	{0xF01DF14B, "mut_ocrna_ms00.ncgr.small"           },
	{0xF01FB2D8, "pl_mv_cyc_to.nsbca"                  },
	{0xF02A226A, "swt_cb_attackend.nsbca"              },
	{0xF05BF6CF, "prf_chrbx_di01.ncgr.small"           },
	{0xF07445B1, "gui_stronghold.nclr.small"           },
	{0xF07977A2, "cfx_glw_medp1.emit"                  },
	{0xF08FF4F6, "guianim_rew.gda"                     },
	{0xF091D0F8, "main_slt_on01.ncgr.small"            },
	{0xF09202DC, "cfx_expp1_exp.emit"                  },
	{0xF0A031AD, "wmp_s_sonich.ncgr.small"             },
	{0xF0A432B2, "prtl_ncsgen_0.ncgr.small"            },
	{0xF0A66FE9, "chao_jp_2.ncgr.small"                },
	{0xF0ADE48A, "a2c10_delay_attack_nocturn.gff"      },
	{0xF0B34353, "bmp_aa_rld.nsbta.small"              },
	{0xF0B76589, "exp_ghz1_fw03.ncgr.small"            },
	{0xF0C0269C, "itm_ace_goldr_0.ncgr.small"          },
	{0xF0C444AC, "pl_ob_blkn_co.nsbtx.small"           },
	{0xF0CEF7C7, "bsc_aa.nsbmd.small"                  },
	{0xF0D59DAA, "charpro_amy1.ncgr.small"             },
	{0xF0DB9F98, "a2c6_fadein.gff"                     },
	{0xF0FF45A3, "prtl_egggen_2.ncgr.small"            },
	{0xF114B5B4, "basiccombattut.gda"                  },
	{0xF11A3947, "pfx_bub3.nsbmd.small"                },
	{0xF12695E3, "fx_cheese_prj.nsbmd.small"           },
	{0xF128A840, "genknun_aa.nsbtx.small"              },
	{0xF12CA370, "icn_s_patk_she2.ncgr.small"          },
	{0xF13D1EAA, "cfx_atkp1.nsbtx.small"               },
	{0xF13F5AA0, "wmp_trv_04_18.gff"                   },
	{0xF15122EB, "prtl_thbhap_0.ncgr.small"            },
	{0xF154ECD3, "loc_ambsh_es0.ncgr.small"            },
	{0xF1568BB5, "pl_pz_vox_pth.nsbtx.small"           },
	{0xF1636D21, "a1c4_blueridge_ice_1.gff"            },
	{0xF16FCF27, "pfx_gul1.nsbtp.small"                },
	{0xF17585A2, "a1c4_blueridge_ice_2.gff"            },
	{0xF17D8861, "wmp_s_pin1.ncgr.small"               },
	{0xF1873977, "savegui_bs_01.tga"                   },
	{0xF194BB96, "a1c4_shadow_sonichold.gff"           },
	{0xF195B28B, "conv_check_of00.ncgr.small"          },
	{0xF19951F8, "savegui_bs_02.tga"                   },
	{0xF19E10F1, "fx_a2_s5_env.nsbta.small"            },
	{0xF19EAF8B, "fx_vox_p_ea_l.nsbta.small"           },
	{0xF1A4992F, "prtl_grxhum_0.ncgr.small"            },
	{0xF1AB390E, "bel_cb_idle.nsbca"                   },
	{0xF1AE1FA1, "prtl_nrmge2_2.ncgr.small"            },
	{0xF1AE2749, "itm_ace_eggmd_1.ncgr.small"          },
	{0xF1C2962C, "inventory_ts.gui"                    },
	{0xF1C4E417, "chao_jl_g.nsbtp.small"               },
	{0xF1C87B77, "pl_pz_crn_end.nsbca"                 },
	{0xF1CCB3FE, "cfx_rtip1.nsbmd.small"               },
	{0xF1DEE144, "cho_ts_10.ncgr.small"                },
	{0xF1E41C3E, "fx_a1_s4_ics_bp.nsbca"               },
	{0xF1E5C597, "chao_fc_g.nsbmd.small"               },
	{0xF1F35E1A, "ehd_fow_fu.ncgr.small"               },
	{0xF20A80D4, "a2c10_turret_destroyed.gff"          },
	{0xF2198D07, "cfx_ch3.nsbtx.small"                 },
	{0xF21AABCD, "big_cb_idle.nsbca"                   },
	{0xF23856D8, "a1c1_sonicwaitsmash.gff"             },
	{0xF23ABC53, "a1c5_found_omega.gff"                },
	{0xF23E859A, "itm_fotrunsl_1.ncgr.small"           },
	{0xF2408386, "prt_se_nctel.ncgr.small"             },
	{0xF2508462, "a1_w1_cc_01.ncgr.small"              },
	{0xF25E3863, "invent_titlelit01.ncgr.small"        },
	{0xF25E7AA0, "plt_cod_knucklesclan.plo"            },
	{0xF262CFCD, "a2_s2_card_t1.nbfp"                  },
	{0xF262CFD0, "a2_s2_card_t1.nbfs"                  },
	{0xF2678AD8, "prtl_amydet_0.ncgr.small"            },
	{0xF26C750E, "main_pnl_on00.ncgr.small"            },
	{0xF270A143, "prtl_amysmg_1.ncgr.small"            },
	{0xF27FD81B, "wmp_txtbr_tx05.ncgr.small"           },
	{0xF28DEAD1, "icn_s_wsl0bcho.ncgr.small"           },
	{0xF28FE59E, "chao_ie_3.ncgr.small"                },
	{0xF29CF8E8, "swan_aa.nsbmd.small"                 },
	{0xF2BEEBFD, "a2scy_tyxhqdoor.dlg"                 },
	{0xF2CB0E81, "pfx_nrg1_gas.emit"                   },
	{0xF2D5F68C, "cfx_ch1_spkp1.emit"                  },
	{0xF2EDCD0D, "prtl_ffmcon_0.ncgr.small"            },
	{0xF30C1735, "temppartyadd.gff"                    },
	{0xF30F3F5F, "prf_tlbar_ms00.ncgr.small"           },
	{0xF311576F, "chao_jq_r.nsbtx.small"               },
	{0xF316C5B3, "itm_glv_power_2.ncgr.small"          },
	{0xF31F95D9, "prtl_krngen_0.ncgr.small"            },
	{0xF32672FE, "tyx_aa_nrg.nsbta.small"              },
	{0xF3311B44, "fx_tyx_beam.nsbmd.small"             },
	{0xF346948D, "cfx_dstp1.nsbtp.small"               },
	{0xF347B8FE, "icn_s_chkstr.ncgr.small"             },
	{0xF356D6A7, "itm_ace_clovej_0.ncgr.small"         },
	{0xF365A721, "prtl_eixthrnvar_2.ncgr.small"        },
	{0xF36CA642, "icn_s_patk_big2.ncgr.small"          },
	{0xF370E5A5, "gui_chao_j4.nclr.small"              },
	{0xF37938A6, "itm_ace_tyxis_1.ncgr.small"          },
	{0xF3806642, "exl_smbtn_hi11.ncgr.small"           },
	{0xF385EE5C, "a1c1_forcerouge.gff"                 },
	{0xF38F7660, "a1c3_eggman.dlg"                     },
	{0xF38F825C, "a1c3_eggman.gff"                     },
	{0xF397A3AD, "itm_ace_gum_0.ncgr.small"            },
	{0xF39A0A9A, "prtl_knusad_0.ncgr.small"            },
	{0xF3AC22C2, "a2c8_scy_queenspawnhide.gff"         },
	{0xF3BB360F, "invent_bar03.ncgr.small"             },
	{0xF3C35B00, "upper_textbox_04.ncgr.small"         },
	{0xF3C50DED, "rax_cb_getup.nsbca"                  },
	{0xF3D032EB, "man_strbt_on01.ncgr.small"           },
	{0xF3DBE3CC, "chao_ba_g.nsbtx.small"               },
	{0xF3DC8C02, "prtl_eixfm3_2.ncgr.small"            },
	{0xF3E27FE3, "a2c10_ix_attack_sonic_wait.gff"      },
	{0xF3E5CED2, "cfx_rfpp1_rfp.emit"                  },
	{0xF40055BC, "chap8_scy_scr_bot.tga"               },
	{0xF404D17F, "gup_ex_idle.nsbca"                   },
	{0xF43244F2, "cfx_rtbp1.nsbtx.small"               },
	{0xF4349CB2, "cmbtbg_a2_s1.nsbmd.small"            },
	{0xF436B986, "pro_medbx_21.ncgr.small"             },
	{0xF44483E2, "mut_globe_ms10.ncgr.small"           },
	{0xF445EAC0, "cw_se_sw.gff"                        },
	{0xF45ECB58, "cfx_bub1.nsbmd.small"                },
	{0xF4687620, "pl_pz_swc_aa.nsbmd.small"            },
	{0xF46AB0C1, "itm_ace_echid_0.ncgr.small"          },
	{0xF46E4BA3, "ch6_introcam.gff"                    },
	{0xF4737456, "pl_mv_tor_fly.nsbca"                 },
	{0xF474D87F, "pfx_arnietest02.nsbtp.small"         },
	{0xF47938DA, "fx_gnp_himissile.nsbta.small"        },
	{0xF48A57E9, "charpro_bar02.ncgr.small"            },
	{0xF48CC4C0, "a1c5p6_tailsstart.gff"               },
	{0xF49C93FD, "cfx_expp2_exp.emit"                  },
	{0xF49E08D5, "pro_lrgbx_12.ncgr.small"             },
	{0xF49FC86B, "exl_flash_ms2.ncgr.small"            },
	{0xF4A37DF0, "gzc_cb_attackend.nsbca"              },
	{0xF4A4053D, "itm_glv_stealth3.ncgr.small"         },
	{0xF4B1C464, "nrg_cb_counter.nsbca"                },
	{0xF4B7F86E, "a2_s2_card_t2.nbfp"                  },
	{0xF4B7F871, "a2_s2_card_t2.nbfs"                  },
	{0xF4BF0C3E, "itm_glvirncl_1.ncgr.small"           },
	{0xF4C0AADE, "cfx_ixmp1_ixm.emit"                  },
	{0xF4CB142E, "cbt_dredt_ms00.ncgr.small"           },
	{0xF4D80963, "a1_w2_mt.nclr.small"                 },
	{0xF4DB4BB9, "journal_ts.gui"                      },
	{0xF4F5D8A9, "pfx_fls1_fls.emit"                   },
	{0xF5003000, "fx_windsock.nsbtp.small"             },
	{0xF505518A, "man_tdlcp_of00.ncgr.small"           },
	{0xF51234B6, "chao_fb_1.ncgr.small"                },
	{0xF51F07E2, "rhk_cb_counter.nsbca"                },
	{0xF51F664F, "chao_ba_r.nsbtp.small"               },
	{0xF52C4E17, "itm_glv_metal_0.ncgr.small"          },
	{0xF52DF5B9, "jurnl_pnl01.ncgr.small"              },
	{0xF5394A9D, "exp_ghq1_mm21.ncgr.small"            },
	{0xF55B0F23, "cbt_flcght_0.ncgr.small"             },
	{0xF55FCA93, "explorehudmenumax.gui"               },
	{0xF55FD7D9, "sco_ex_damage.nsbca"                 },
	{0xF57E4001, "prtl_shdstg_0.ncgr.small"            },
	{0xF58913DC, "arm_cb_damage.nsbca"                 },
	{0xF5895C29, "bif_she.nbfp"                        },
	{0xF5895C2C, "bif_she.nbfs"                        },
	{0xF590A574, "gen_titlebar_01.ncgr.small"          },
	{0xF5918F48, "gwr_cb_attackc.nsbca"                },
	{0xF59A14B9, "a2_w0_vb.nclr.small"                 },
	{0xF5A02A4F, "prtl_krgang_1.ncgr.small"            },
	{0xF5A03362, "prtl_eixprd_2.ncgr.small"            },
	{0xF5AA26E8, "camerapangreenhill.gff"              },
	{0xF5ABC8B7, "knu_cb_deadidle.nsbca"               },
	{0xF5AE0E80, "prtl_vxccitgen_1.ncgr.small"         },
	{0xF5AE88DA, "shp_cb_blockent.nsbca"               },
	{0xF5B2A9E6, "conv_icon04.ncgr.small"              },
	{0xF5B2C183, "a2c8_scy_gateguardapproach.gff"      },
	{0xF5B5A895, "pl_pu_emrld4.nsbmd.small"            },
	{0xF5B69741, "storemaintab.gui"                    },
	{0xF5C249CB, "a1_s3_i01_2to1.gff"                  },
	{0xF5C29A53, "itm_fotrunrg_0.ncgr.small"           },
	{0xF5D07A0C, "a2_s2_i01.walk.tga"                  },
	{0xF5DD5EC7, "vox_lig_aa.nsbmd.small"              },
	{0xF5E10C39, "plt_a2sq_stalker.plo"                },
	{0xF5E3A637, "cfx_omga_mf.nsbtp.small"             },
	{0xF5E67ACD, "a1_s3_i01_2to3.gff"                  },
	{0xF5E681D6, "fx_smokepuff.nsbtx.small"            },
	{0xF5E8A830, "prtl_tailsdet_2.ncgr.small"          },
	{0xF5F0A83A, "a1_s2_card_b1.nbfp"                  },
	{0xF5F0A83D, "a1_s2_card_b1.nbfs"                  },
	{0xF5F8934E, "a1_s3_i01_2to4.gff"                  },
	{0xF608797D, "c_sheso_she.nsbca"                   },
	{0xF60AABCF, "a1_s3_i01_2to5.gff"                  },
	{0xF60AE4DC, "gwr_cb_kd.nsbca"                     },
	{0xF623FE52, "fx_a2_s7_i03.nsbmd.small"            },
	{0xF63556F0, "icn_l_utinv2.ncgr.small"             },
	{0xF639BC19, "pl_pz_tyx_num.nsbtp.small"           },
	{0xF639C8C0, "icn_l_utstlth.ncgr.small"            },
	{0xF6526F11, "icn_l_barwofprf.ncgr.small"          },
	{0xF666EFA1, "chao_ji_g.nsbmd.small"               },
	{0xF66B0DF3, "prtl_egghum_2.ncgr.small"            },
	{0xF68432F1, "itm_ace_fragm_0.ncgr.small"          },
	{0xF69005D5, "pl_pz_mem_trd.nsbtx.small"           },
	{0xF6A8F14B, "tyx_cb_blockidle.nsbca"              },
	{0xF6AA50F3, "a2_w0_va_01.ncgr.small"              },
	{0xF6B1C7C3, "a1c5_p9_ixattacks.gff"               },
	{0xF6B5C695, "prt_ce_mag.ncgr.small"               },
	{0xF6B79E06, "a2scy_swarm1.dlg"                    },
	{0xF6CDF0F6, "gup_aa_shno.nsbta.small"             },
	{0xF6D842E1, "a1c1_forceamy.gff"                   },
	{0xF6D9265B, "plt_fpa2_shadow.plo"                 },
	{0xF6DB0E22, "cfx_nflp1.nsbtp.small"               },
	{0xF6E52AAC, "kme_aa.nsbmd.small"                  },
	{0xF6E94B84, "prtl_hansad_1.ncgr.small"            },
	{0xF6EC0FDF, "pfx_sms1.nsbmd.small"                },
	{0xF6FC1952, "icn_s_patk_tai6.ncgr.small"          },
	{0xF701C7F5, "icn_s_statcur.ncgr.small"            },
	{0xF703F4E3, "a1_s2_i06_ts.nbfp"                   },
	{0xF703F4E6, "a1_s2_i06_ts.nbfs"                   },
	{0xF70B8F4D, "pro_tab_30.ncgr.small"               },
	{0xF71EA60A, "cfx_fgrp2.nsbtx.small"               },
	{0xF739E5C8, "lvl_pow_bspnl21.ncgr.small"          },
	{0xF74BCF04, "a1_s2_ex_mm.nbfp"                    },
	{0xF74BCF07, "a1_s2_ex_mm.nbfs"                    },
	{0xF77287DC, "plt_ch1_tutorial.plo"                },
	{0xF77EFA6C, "cfx_crup1.nsbtp.small"               },
	{0xF7814106, "prtl_cresad_0.ncgr.small"            },
	{0xF78240AB, "pfx_smd1.nsbtx.small"                },
	{0xF78BF2E8, "spell_tough.spl"                     },
	{0xF791FA7B, "itm_fotrunkv_0.ncgr.small"           },
	{0xF79451D2, "amy_cb_dead.nsbca"                   },
	{0xF79AAB2B, "bmp_cb_attackend.nsbca"              },
	{0xF79D33B2, "a1_w1_br_12.ncgr.small"              },
	{0xF7A320C0, "cho_lgbtn_di00.ncgr.small"           },
	{0xF7B04DA4, "son_ex_fall.nsbca"                   },
	{0xF7B10927, "a1_w1_cc.nclr.small"                 },
	{0xF7B7D028, "invent_icon_att.ncgr.small"          },
	{0xF7C8461C, "gwr_cb_attackend.nsbca"              },
	{0xF7C9F0A0, "spell_target_all.spl"                },
	{0xF7CEE268, "itm_act2_chaos4_1.ncgr.small"        },
	{0xF7DCDC5D, "prtl_egglaf_3.ncgr.small"            },
	{0xF7E07224, "son_atomicstrike.gff"                },
	{0xF7E4DC63, "prtl_gnahap_0.ncgr.small"            },
	{0xF7EC3F5E, "prtl_krshap_1.ncgr.small"            },
	{0xF7EEEE69, "pl_mv_cyc_sm.nsbmd.small"            },
	{0xF7F05FF3, "cfx_rfpp2_rfp.emit"                  },
	{0xF7F2CCCE, "cfx_slgp1_slg.emit"                  },
	{0xF8034896, "pfx_gus1.nsbtx.small"                },
	{0xF814B5DC, "btn_l_bar_of01.ncgr.small"           },
	{0xF8342046, "gui_chaogard.nclr.small"             },
	{0xF8352760, "gensha_aa.nsbtx.small"               },
	{0xF8481197, "invent_barlit02.ncgr.small"          },
	{0xF855B504, "btn_help_of.ncgr.small"              },
	{0xF85854D8, "egm_cb_blockin.nsbca"                },
	{0xF85BF27D, "cfx_cgrp1_brk.emit"                  },
	{0xF869C576, "itm_accwtrng_1.ncgr.small"           },
	{0xF894E275, "rax_aa.nsbtx.small"                  },
	{0xF89E0F74, "gentitle.gui"                        },
	{0xF89F6A8B, "button.tga"                          },
	{0xF8B05EFC, "chao_jd_1.ncgr.small"                },
	{0xF8BC6A2F, "a2c10_scylla_appears.gff"            },
	{0xF8C16B34, "lower_bg_00.ncgr.small"              },
	{0xF8CC8095, "chao_jb_g.nsbtx.small"               },
	{0xF8D1EFAC, "mtz_hintguy.dlg"                     },
	{0xF8E8621E, "icn_l_catk_sa.ncgr.small"            },
	{0xF8EA6E60, "pl_pu_chst_aa.nsbmd.small"           },
	{0xF8F43100, "chao_ic_r.nsbtx.small"               },
	{0xF8F88AF9, "prtl_nrdge2_3.ncgr.small"            },
	{0xF905F7F8, "drn_ex_damage.nsbca"                 },
	{0xF916C504, "n2da.gda"                            },
	{0xF91D0AEE, "rou_ex_flyout.nsbca"                 },
	{0xF91E7D16, "pl_mv_bop_bon.nsbca"                 },
	{0xF92D5C2D, "itm_glv_lucky_1.ncgr.small"          },
	{0xF9313DEC, "lower_green_ring_01.ncgr.small"      },
	{0xF934EBE3, "cbt_menu_resetbutton_of.ncgr.small"  },
	{0xF93AB89F, "pro_bar_of02.ncgr.small"             },
	{0xF93CBAF4, "rwd_tsmbx_of02.ncgr.small"           },
	{0xF94277EE, "prtl_eixmad_2.ncgr.small"            },
	{0xF9452DFA, "a1c3_tracker_device2.gff"            },
	{0xF94A036C, "pl_pz_vox_lr.nsbmd.small"            },
	{0xF94AB5D0, "c_sharoom_sha.nsbca"                 },
	{0xF94AD90F, "tai_ex_jump.nsbca"                   },
	{0xF955C5CB, "fx_a2_s4_ex_ff.nsbmd.small"          },
	{0xF958AD25, "chao_jk_3.ncgr.small"                },
	{0xF95E2787, "pl_am_noc_gsl.nsbca"                 },
	{0xF9621108, "exp_ghz2_mm13.ncgr.small"            },
	{0xF9643654, "fx_vox_r_ea_s.nsbta.small"           },
	{0xF96461D8, "introbridge.gda"                     },
	{0xF966999E, "pl_pz_mem_lts.nsbtx.small"           },
	{0xF96C5CE1, "crm_cb_blockin.nsbca"                },
	{0xF97419CF, "prt_sp_ssn.ncgr.small"               },
	{0xF9763F47, "a1c3_swatbot_move.gff"               },
	{0xF97EFD30, "cfx_spcp1.nsbtx.small"               },
	{0xF981518F, "gui_chao_h.nclr.small"               },
	{0xF98B1FA5, "prtl_wkrstu_1.ncgr.small"            },
	{0xF99BE6E8, "cfx_pfbp2.nsbmd.small"               },
	{0xF9A670D5, "rax_ex_run.nsbca"                    },
	{0xF9C22DF5, "jurnl_ts00.ncgr.small"               },
	{0xF9C880AA, "cfx_ghlp1.nsbmd.small"               },
	{0xF9CA9372, "main_pnl_off15.ncgr.small"           },
	{0xF9CDF4E3, "chao_ab_gard.nsbtx.small"            },
	{0xF9CFB263, "icn_s_statdt.ncgr.small"             },
	{0xF9D79D5B, "c_soknamta_so.nsbca"                 },
	{0xF9EFC9C9, "prtl_tdadsca_1.ncgr.small"           },
	{0xF9F555EB, "cfx_spt_slm.nsbtx.small"             },
	{0xF9F5BAAA, "syr_cb_attackend.nsbca"              },
	{0xF9F86338, "prtl_gungen_0.ncgr.small"            },
	{0xF9FDBFA1, "prtl_txmmad_2.ncgr.small"            },
	{0xFA0507DB, "pig_cb_idle.nsbca"                   },
	{0xFA0852D8, "gennpc1b_aa.nsbtx.small"             },
	{0xFA0A6A8A, "prf_tboxa_ms02.ncgr.small"           },
	{0xFA100318, "chao_jb_r.nsbtp.small"               },
	{0xFA20587F, "conversationupper.gui"               },
	{0xFA27B360, "bfl_cb_kd.nsbca"                     },
	{0xFA2A9A74, "pfx_aur1.nsbmd.small"                },
	{0xFA3C3F19, "pfx_ppr1.nsbtx.small"                },
	{0xFA3D7FF2, "cbt_elem_an_fire.ncgr.small"         },
	{0xFA405D39, "chao_gb_3.ncgr.small"                },
	{0xFA62339B, "bal_cb_attackend.nsbca"              },
	{0xFA65D3D4, "vox_cb_pattack2.nsbca"               },
	{0xFA66FBB9, "jurnl_ts22.ncgr.small"               },
	{0xFA695D72, "cmbtbg_a2_s6.nsbtx.small"            },
	{0xFA6AD30C, "str_party_ms10.ncgr.small"           },
	{0xFA724233, "chao_ab_turn.nsbmd.small"            },
	{0xFA78A6EC, "chao_jt_g.nsbmd.small"               },
	{0xFA81E1AD, "pl_ob_bdr_aa.nsbtx.small"            },
	{0xFA82DE49, "fx_gust.nsbtx.small"                 },
	{0xFA8E69DD, "cbt_dyelt_ms00.ncgr.small"           },
	{0xFA94E503, "icn_l_onfly3.ncgr.small"             },
	{0xFA9FCF06, "minimapgui01_ts.tga"                 },
	{0xFAA16CF8, "convoshadow.nftr"                    },
	{0xFAA36DD8, "pfx_mot1_pnk.emit"                   },
	{0xFAA44780, "hnpc_mug_aa.nsbmd.small"             },
	{0xFAABA9EB, "prtl_ix.nclr.small"                  },
	{0xFAB776D9, "prtl_eggmad_0.ncgr.small"            },
	{0xFAC02272, "rwd_tlgbx_ms02.ncgr.small"           },
	{0xFAC0F14E, "shp_cb_death.nsbca"                  },
	{0xFACB64E4, "itm_fotsnkel_2.ncgr.small"           },
	{0xFACF48D3, "cfx_hypp1.nsbtp.small"               },
	{0xFAD8535C, "icn_s_stathyp.ncgr.small"            },
	{0xFAD9DD99, "pro_back_of00.ncgr.small"            },
	{0xFADEB755, "a1c2_soldiers.dlg"                   },
	{0xFAE83AFC, "debugcreateitems.gda"                },
	{0xFAF34C29, "chao_jq_1.ncgr.small"                },
	{0xFAF8C067, "itm_ace_bugsp_2.ncgr.small"          },
	{0xFAFC78FE, "lower_textbox_05.ncgr.small"         },
	{0xFAFD19E2, "gnp_cb_counter.nsbca"                },
	{0xFB04FCB8, "a2c10_splitparty.gff"                },
	{0xFB0CB397, "cfx_aurp1.nsbmd.small"               },
	{0xFB11F2E7, "prtl_guncoang_1.ncgr.small"          },
	{0xFB1EA504, "a2_cream.dlg"                        },
	{0xFB27AC86, "itm_ace_crzbn_3.ncgr.small"          },
	{0xFB3D8A16, "prtl_amysad_3.ncgr.small"            },
	{0xFB457DDC, "charpro_cre0.ncgr.small"             },
	{0xFB5184EB, "a1c5_partysplit.gff"                 },
	{0xFB5E93AB, "a2c10_flooded_sewer_start.gff"       },
	{0xFB7DD0CD, "shadball.nsbta.small"                },
	{0xFB8B5520, "gup_ex_damage.nsbca"                 },
	{0xFB9B10DB, "a1c4_after_outhouse.gff"             },
	{0xFB9DD771, "ssf_tlgbx_ms02.ncgr.small"           },
	{0xFBB27EEE, "itm_fotsnksh_2.ncgr.small"           },
	{0xFBB71241, "itm_glv_strong_0.ncgr.small"         },
	{0xFBB87AC4, "wmp_s_airpo.ncgr.small"              },
	{0xFBB940CC, "prtl_thbgen_0.ncgr.small"            },
	{0xFBBC92C1, "cbt_focusbad.ncer.small"             },
	{0xFBC0A57F, "icn_s_bash2.ncgr.small"              },
	{0xFBC0E6B0, "chao_ba_0.ncgr.small"                },
	{0xFBE6F1FF, "lvl_stat_tspnl20.ncgr.small"         },
	{0xFBFA0214, "gwr_ex_walk.nsbca"                   },
	{0xFC0206DE, "chao_da_r.nsbmd.small"               },
	{0xFC1761FD, "str_tlgbx_ms01.ncgr.small"           },
	{0xFC1C2F8E, "attack3_bs.tga"                      },
	{0xFC1DF99F, "plt_a1sq_official.plo"               },
	{0xFC1F12B9, "fleetrio.gda"                        },
	{0xFC282F64, "a1_s5_i04_ts.nbfp"                   },
	{0xFC282F67, "a1_s5_i04_ts.nbfs"                   },
	{0xFC2F4C17, "exp_cct1_mm12.ncgr.small"            },
	{0xFC33D5BE, "inventorylowerbuttons.gui"           },
	{0xFC3DCBD2, "a1c3_sentry.gff"                     },
	{0xFC447090, "loc_victr_dt0.ncgr.small"            },
	{0xFC487C30, "_debug.dlg"                          },
	{0xFC50DC48, "prtl_general.nclr.small"             },
	{0xFC53B069, "pl_pz_gen_bp.nsbca"                  },
	{0xFC54DB23, "a1c4_shade_escape.gff"               },
	{0xFC5967A4, "prtl_thxgen_2.ncgr.small"            },
	{0xFC5DCF08, "cbt_rndnum8.ncgr.small"              },
	{0xFC65ED36, "cfx_pswp1_psw.emit"                  },
	{0xFC69DC6F, "bel_cb_damage.nsbca"                 },
	{0xFC6AA08D, "bsl_cb_kd.nsbca"                     },
	{0xFC6F163D, "bel_ex_run.nsbca"                    },
	{0xFC79F787, "a2_s4_ex_mm.nbfp"                    },
	{0xFC79F78A, "a2_s4_ex_mm.nbfs"                    },
	{0xFC8DC6D9, "vox_cb_damage.nsbca"                 },
	{0xFC95BFC7, "jrl_tsmbr_of00.ncgr.small"           },
	{0xFC9B6068, "exp_ghz1_fw20.ncgr.small"            },
	{0xFCA98D1C, "charpro_ts_b02.ncgr.small"           },
	{0xFCBA2497, "pfx_rok1_rok.emit"                   },
	{0xFCD111F8, "test.gui"                            },
	{0xFCD99FB6, "2da.gda"                             },
	{0xFCDE37E0, "chao_jm_g.nsbtx.small"               },
	{0xFD0DE943, "lvl_stat_lblit.ncgr.small"           },
	{0xFD102A3C, "a1_s1_ex.are"                        },
	{0xFD16C76D, "a1_s5_i01.are"                       },
	{0xFD1BCB26, "n_s.gff"                             },
	{0xFD28DFEE, "a1_s5_i02.are"                       },
	{0xFD2DD038, "cho_tmbtn_of02.ncgr.small"           },
	{0xFD31A092, "prtl_amysca_0.ncgr.small"            },
	{0xFD328154, "a2_s1_ex_mm.nclr.small"              },
	{0xFD365F74, "pl_pz_mem_box.nsbtx.small"           },
	{0xFD3AF86F, "a1_s5_i03.are"                       },
	{0xFD3E1CC7, "prtl_dronegen_1.ncgr.small"          },
	{0xFD4394D0, "sonic_aa_p.ncgr.small"               },
	{0xFD4D10F0, "a1_s5_i04.are"                       },
	{0xFD4D776D, "prtl_knustb_2.ncgr.small"            },
	{0xFD607EFF, "itm_glvsercl_1.ncgr.small"           },
	{0xFD61AA1F, "prtl_eixamu_2.ncgr.small"            },
	{0xFD73F9E2, "combatinputs.nclr.small"             },
	{0xFD81DB3A, "pl_pz_ppd_dn.nsbtp.small"            },
	{0xFD82BA54, "prt_se_grdpw.ncgr.small"             },
	{0xFD936D7B, "a1_w1_mr_01.ncgr.small"              },
	{0xFDB5B0CE, "cfx_watp1.nsbtp.small"               },
	{0xFDB828F9, "prtl_thxdef_2.ncgr.small"            },
	{0xFDBF0536, "wmp_txtbr_tx00.ncgr.small"           },
	{0xFDC3D455, "prtl_nesgen_1.ncgr.small"            },
	{0xFDCA248C, "c_amcr_am.nsbca"                     },
	{0xFDD520DE, "fx_tal_sbot.nsbtx.small"             },
	{0xFDDCBF0F, "jillcan13bdsh.nftr"                  },
	{0xFDDDB7FC, "pfx_briantest02.nsbtp.small"         },
	{0xFDDE8E62, "she_cb_getup.nsbca"                  },
	{0xFDE05EEF, "bfl_cb_idle.nsbca"                   },
	{0xFDF770D6, "lvl_stat_smpnl00.ncgr.small"         },
	{0xFDFA55B1, "tai_ex_idle.nsbca"                   },
	{0xFE0C65BD, "cho_tlgbx_ms00.ncgr.small"           },
	{0xFE1AEB78, "chao_gc_g.nsbmd.small"               },
	{0xFE21BA63, "chao_jm_r.nsbtp.small"               },
	{0xFE25511E, "pfx_orb.nsbmd.small"                 },
	{0xFE33113B, "prtl_piratel.nclr.small"             },
	{0xFE38A6EB, "prtl_tyxis.nclr.small"               },
	{0xFE41CAF2, "cfx_focp1_foc.emit"                  },
	{0xFE429BE3, "chao_fd_r.nsbmd.small"               },
	{0xFE49D7B7, "itm_ace_resur_2.ncgr.small"          },
	{0xFE51118A, "fx_a1_s3_ex_env.nsbta.small"         },
	{0xFE533CBE, "pfx_lef1.nsbtx.small"                },
	{0xFE55992D, "pl_pz_gen_aa.nsbmd.small"            },
	{0xFE5D47B7, "ehd_tbbx2_ms02.ncgr.small"           },
	{0xFE5F8AE5, "pfx_orb1_orb.emit"                   },
	{0xFE672F2A, "inventorycharactertab.gui"           },
	{0xFE7ABF1D, "jurnl_arrowcodn.ncgr.small"          },
	{0xFE7C3202, "btn_s_bar_of00.ncgr.small"           },
	{0xFE8DFF32, "prtl_wdmnexh_3.ncgr.small"           },
	{0xFE9AA79F, "prtl_leucosia.nclr.small"            },
	{0xFEA0C15F, "cfx_jwl1.nsbtp.small"                },
	{0xFEA99FD7, "cfx_revp1.nsbtx.small"               },
	{0xFEAE5345, "exp_wel1_mm21.ncgr.small"            },
	{0xFEAFC1CE, "rou_cb_attackc.nsbca"                },
	{0xFEB577CC, "cfx_weap1.nsbmd.small"               },
	{0xFEB5878A, "spell_revive_all.spl"                },
	{0xFEC310B2, "icn_s_wsl1bcho.ncgr.small"           },
	{0xFEC3278C, "a1_w3_gh_00.ncgr.small"              },
	{0xFED24E08, "fx_leafdn.nsbtp.small"               },
	{0xFED252A3, "a1_w2_cc_10.ncgr.small"              },
	{0xFEDFCC07, "cfx_focp1.nsbmd.small"               },
	{0xFEE117C5, "icn_s_str_char.ncgr.small"           },
	{0xFEF6E6D0, "prtl_eggsmg_2.ncgr.small"            },
	{0xFF09BB0B, "a2_chb_thebesintro.gff"              },
	{0xFF0B2FC1, "cfx_gwtr_prj.nsbta.small"            },
	{0xFF0E91F3, "a1sq_well.dlg"                       },
	{0xFF18F676, "btn_l_rtarw_of.ncgr.small"           },
	{0xFF1E5971, "singlescreencamera.gff"              },
	{0xFF252B4C, "cfx_mexp1.nsbtp.small"               },
	{0xFF2AB79F, "itm_glvcrscl_3.ncgr.small"           },
	{0xFF33E8F9, "pl_pz_brd_2dn.nsbca"                 },
	{0xFF389A8D, "exp_xpbar.ncgr.small"                },
	{0xFF3E2B54, "itm_ace_timmy_0.ncgr.small"          },
	{0xFF3F1BAF, "gzc_cb_dead.nsbca"                   },
	{0xFF4AFD7D, "a2_s4_card_b1.nbfp"                  },
	{0xFF4AFD80, "a2_s4_card_b1.nbfs"                  },
	{0xFF522EEE, "statsinfotrigger.gda"                },
	{0xFF5F10F6, "chao_fc_0.ncgr.small"                },
	{0xFF6FA1B8, "cfx_ixmp1.nsbtx.small"               },
	{0xFF6FC9DD, "icn_s_patk_knu1.ncgr.small"          },
	{0xFF7870A2, "cbt_lgrnt_ms00.ncgr.small"           },
	{0xFF7A1FAF, "prtl_txsgen_2.ncgr.small"            },
	{0xFF7FBD5C, "nsw_ex_walk.nsbca"                   },
	{0xFF9B2D91, "charpro_barlit01.ncgr.small"         },
	{0xFF9D1323, "exl_lrgbt_on00.ncgr.small"           },
	{0xFFB0DD6D, "classes.gda"                         },
	{0xFFB26334, "cbt_round_slide.ncgr.small"          },
	{0xFFB36AC9, "icn_l_victory1.ncgr.small"           },
	{0xFFC03709, "a1c2_guardiandefeated.gff"           },
	{0xFFCA8BE8, "a1c3_tracker_device.gff"             },
	{0xFFCABB57, "shp_cb_attackend.nsbca"              },
	{0xFFDDB146, "cfx_egbt_mf.nsbta.small"             },
	{0xFFE3AD74, "cfx_ftpp1.nsbtx.small"               },
	{0xFFE3FACE, "prtl_scihap_2.ncgr.small"            },
	{0xFFFA7BF3, "man_tdlcp_on01.ncgr.small"           }
};

const char *findSonicFile(uint32 hash) {
	const SonicFileHash *file = Common::binarySearch(kSonicFilesHashes, ARRAYSIZE(kSonicFilesHashes), hash);
	if (!file)
		return 0;

	return file->value;
}
