# SPDX-License-Identifier: Apache-2.0
# SPDX-FileCopyrightText: Zygmunt Krynicki
QEMU_IMG ?= $(or $(shell command -v qemu-img),$(error program qemu-img is required))
QEMU_SYSTEM_X86_64 ?= $(or $(shell command -v qemu-system-x86_64),$(error program qemu-system-x86_64 is required))
QEMU_SYSTEM_AARCH64 ?= $(or $(shell command -v qemu-system-aarch64),$(error program qemu-system-aarch64 is required))
QEMU_SYSTEM_VERSION = $(subst ., ,$(word 4,$(shell $(or $(shell command -v $(QEMU_SYSTEM_X86_64),$(shell command -v QEMU_SYSTEM_AARCH64)) --version)))
# Version of qemu as a triplet major, minor, micro.
QEMU_SYSTEM_VERSION = $(subst ., ,$(word 4,$(shell $(QEMU_SYSTEM_X86_64) --version)))
QEMU_SYSTEM_MAJOR_VERSION = $(firstword $(QEMU_SYSTEM_VERSION))

QEMU_SYSTEM_X86_64_MACHINE = $(or $(QEMU_SYSTEM_X86_64_MACHINE_FOR_VERSION_$(QEMU_SYSTEM_MAJOR_VERSION)),$(error please define QEMU_SYSTEM_X86_64_MACHINE_FOR_VERSION_$(QEMU_SYSTEM_MAJOR_VERSION)))
QEMU_SYSTEM_X86_64_MACHINE_FOR_VERSION_6 = q35
QEMU_SYSTEM_X86_64_MACHINE_FOR_VERSION_8 = q35
QEMU_SYSTEM_X86_64_MACHINE_FOR_VERSION_9 = q35

QEMU_SYSTEM_AARCH64_MACHINE = $(or $(QEMU_SYSTEM_AARCH64_MACHINE_FOR_VERSION_$(QEMU_SYSTEM_MAJOR_VERSION)),$(error please define QEMU_SYSTEM_AARCH64_MACHINE_FOR_VERSION_$(QEMU_SYSTEM_MAJOR_VERSION)))
QEMU_SYSTEM_AARCH64_MACHINE_FOR_VERSION_6 = virt
QEMU_SYSTEM_AARCH64_MACHINE_FOR_VERSION_8 = virt
QEMU_SYSTEM_AARCH64_MACHINE_FOR_VERSION_9 = virt

QEMU_SMP_X86_64 = 1
QEMU_SMP_AARCH64 = 1
QEMU_SMP_$(shell uname -m | tr a-z- A-Z_) = 4

QEMU_IMG_SIZE = 64G
QEMU_MEM_SIZE = 2048

# Full command line for running QEMU with a given EFI firmware.
QEMU_SYSTEM_X86_64_EFI_CMDLINE = \
	$(QEMU_SYSTEM_X86_64) \
	-cpu max \
	-machine $(QEMU_SYSTEM_X86_64_MACHINE)$${QEMU_ACCEL_MACHINE_PROP=,accel=`$(QEMU_SYSTEM_X86_64) -accel help | tail -n +2 | sort | paste -sd :`} \
	$${QEMU_MEM_OPTION=-m $(QEMU_MEM_SIZE)} \
	$${QEMU_SMP_OPTION=-smp $(QEMU_SMP_X86_64)} \
	$${QEMU_DISPLAY_OPTION=-nographic} \
	$${QEMU_BOOT_FIRMWARE_OPTION=-drive if=pflash,format=raw,unit=0,file=$(2),readonly=on \
		-drive if=pflash,format=raw,unit=1,file=$(3)} \
	$${QEMU_STORAGE_OPTION=-drive file=$(1),if=none,format=qcow2,id=drive0,media=disk,cache=writeback \
		-device virtio-blk,drive=drive0,bootindex=0} \
	$${QEMU_RNG_OPTION=-device virtio-rng-pci} \
	$${QEMU_NET_OPTION=-netdev user,id=netdev0$${QEMU_NETDEV_USER_EXTRA:-} \
		-device virtio-net-pci,netdev=netdev0,id=net0 \
		-smbios type=41,designation=Virtual-LAN,instance=1,kind=ethernet,pcidev=net0}

# Full command line for running QEMU with legacy BIOS.
QEMU_SYSTEM_X86_64_BIOS_CMDLINE = \
	$(QEMU_SYSTEM_X86_64) \
	-cpu max \
	-machine $(QEMU_SYSTEM_X86_64_MACHINE)$${QEMU_ACCEL_MACHINE_PROP=,accel=`$(QEMU_SYSTEM_X86_64) -accel help | tail -n +2 | sort | paste -sd :`} \
	$${QEMU_MEM_OPTION=-m $(QEMU_MEM_SIZE)} \
	$${QEMU_SMP_OPTION=-smp $(QEMU_SMP_X86_64)} \
	$${QEMU_DISPLAY_OPTION=-nographic} \
	$${QEMU_STORAGE_OPTION=-drive file=$(1),if=none,format=qcow2,id=drive0,media=disk,cache=writeback \
		-device virtio-blk,drive=drive0,bootindex=0} \
	$${QEMU_RNG_OPTION=-device virtio-rng-pci} \
	$${QEMU_NET_OPTION=-netdev user,id=netdev0$${QEMU_NETDEV_USER_EXTRA:-} \
		-device virtio-net-pci,netdev=netdev0,id=net0 \
		-smbios type=41,designation=Virtual-LAN,instance=1,kind=ethernet,pcidev=net0}

QEMU_SYSTEM_AARCH64_EFI_CMDLINE = \
	$(QEMU_SYSTEM_AARCH64) \
	-cpu max \
	-machine $(QEMU_SYSTEM_AARCH64_MACHINE)$${QEMU_ACCEL_MACHINE_PROP=,accel=`$(QEMU_SYSTEM_AARCH64) -accel help | tail -n +2 | sort | paste -sd :`} \
	$${QEMU_MEM_OPTION=-m $(QEMU_MEM_SIZE)} \
	$${QEMU_SMP_OPTION=-smp $(QEMU_SMP_AARCH64)} \
	$${QEMU_DISPLAY_OPTION=-nographic} \
	$${QEMU_BOOT_FIRMWARE_OPTION=-drive if=pflash,format=raw,unit=0,file=$(2),readonly=on \
		-drive if=pflash,format=raw,unit=1,file=$(3)} \
	$${QEMU_STORAGE_OPTION=-drive file=$(1),if=none,format=qcow2,id=drive0,media=disk,cache=writeback \
		-device virtio-blk,drive=drive0,bootindex=0} \
	$${QEMU_RNG_OPTION=-device virtio-rng-pci} \
	$${QEMU_NET_OPTION=-netdev user,id=netdev0$${QEMU_NETDEV_USER_EXTRA:-} \
		-device virtio-net-pci,netdev=netdev0,id=net0}

efi-code.x86_64.img: $(or $(firstword $(wildcard /usr/share/OVMF/OVMF_CODE.fd /usr/share/OVMF/OVMF_CODE_4M.fd)),$(error please install ovmf firmware))
	dd if=$< of=$@ conv=notrunc

efi-vars.x86_64.img: $(or $(firstword $(wildcard /usr/share/OVMF/OVMF_VARS.fd /usr/share/OVMF/OVMF_VARS_4M.fd)),$(error please install ovmf firmware))
	dd if=$< of=$@ conv=notrunc

efi-code.aarch64.img: /usr/share/AAVMF/AAVMF_CODE.fd
	dd if=$< of=$@ conv=notrunc

efi-vars.aarch64.img: /usr/share/AAVMF/AAVMF_VARS.fd
	dd if=$< of=$@ conv=notrunc

# UEFI firmware code and state.
%.aarch64.efi-code.img: | efi-code.aarch64.img
	ln -sf $(firstword $|) $@

%.aarch64.efi-vars.img: efi-vars.aarch64.img
	cp $< $@

%.x86_64.efi-code.img: | efi-code.x86_64.img
	ln -sf $(firstword $|) $@

%.x86_64.efi-vars.img: efi-vars.x86_64.img
	cp $< $@

define CLOUD_INIT_META_DATA_TEMPLATE
#cloud-config
instance-iid: iid-$1
local-hostname: $(subst .,-,$1)
endef

define CLOUD_INIT_USER_DATA_TEMPLATE
#cloud-config
timezone: Europe/Warsaw
power_state:
  delay: now
  mode: poweroff
  message: Powering off
  timeout: 0
  condition: true
hostname: $(subst .,-,$1)
ssh_pwauth: true
users:
- name: $2
  groups: users
  gecos: $2 User
  # Both the username and the password is "$2"
  hashed_passwd: '$(shell echo -n $2 | $(MKPASSWD) --method=SHA-512 --rounds=4096 -s)'
  sudo: ["ALL=(ALL) NOPASSWD:ALL"]
  lock_passwd: false
  shell: /bin/bash
runcmd:
# This fixes ssh authentication on openSUSE Tumbleweed.
- echo 'UsePAM yes' >> /etc/ssh/sshd_config
- touch /etc/cloud/cloud-init.disabled
endef

# Seed ISOs with cloud-init files.
# The switch -graft-points allows us to have different (fixed) name in the ISO
# and different (variable) names in the local file system. See the manual page
# of genisoimage for details.
%.seed.iso: %.user-data %.meta-data
	$(GENISOIMAGE) \
		-input-charset utf-8 \
		-output $@ \
		-volid CIDATA \
		-joliet \
		-rock \
		-graft-points \
		user-data=$(word 1,$^) \
		meta-data=$(word 2,$^)

# Pick one of the three templates, in order of priority:
# - $(DISTRO)_$(VERSION)_CLOUD_INIT_USER_DATA_TEMPLATE
# - $(DISTRO)_CLOUD_INIT_USER_DATA_TEMPLATE
# - CLOUD_INIT_USER_DATA_TEMPLATE
# This evaluates the name of the template. Use $(call $(call ...)...) to
# evaluate the resulting template.
PICK_CLOUD_INIT_USER_DATA_TEMPLATE_FOR = $(or $(if $(value $1_$2_CLOUD_INIT_USER_DATA_TEMPLATE),$1_$2_CLOUD_INIT_USER_DATA_TEMPLATE),$(if $(value $1_CLOUD_INIT_USER_DATA_TEMPLATE),$1_CLOUD_INIT_USER_DATA_TEMPLATE),CLOUD_INIT_USER_DATA_TEMPLATE)

.PHONY: clean
clean::
	rm -f $(foreach a,x86_64 aarch64,efi-code.$a.img efi-vars.$a.img)
	rm -f $(foreach a,x86_64 aarch64,*.$a.qcow2 *.$a.efi-code.img *.$a.efi-vars.img)
	rm -f *.seed.iso *.meta-data *.user-data
