/*################################################################################
# Linux Management Providers (LMP), Provider Common Library
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr, desmons_frederic@yahoo.fr>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by <Author(s)>, <Affiliation>, <Year>
################################################################################*/

#include "Transtype.h"

string CT_CMPIObjectPathToString(const CMPIObjectPath* cop) {
	string result;
	Objectpath _cop(NULL, (CMPIObjectPath*) cop);
	result = _cop.getNamespace() + ":" + _cop.getClassname();
	unsigned long nbKeys = _cop.getKeyCount();
	vector<string> keyNames;
	vector<string> keyValues;
	for (unsigned int i = 0; i < nbKeys; i++) {
		string keyName;
		CMPIData keyValue;
		_cop.getKeyAt(i, keyValue, keyName);
		vector<string>::size_type _index;
		CF_foundInSortedList(keyName, keyNames, _index);
		keyNames.insert(keyNames.begin() + _index, keyName);
		keyValues.insert(keyValues.begin() + _index, CT_CMPIDataToString(keyValue));
	}
	for (unsigned int i = 0; i < keyNames.size(); i++)
		result += "." + keyNames[i] + "=" + keyValues[i];
	return result;
}

string CT_CMPIArrayToString(const CMPIArray* array) {
	string result;
	if (array == NULL)
		return "null";
	for (unsigned int i = 0; i < array->ft->getSize(array, NULL); i++) {
		result += CT_CMPIDataToString(array->ft->getElementAt(array, i, NULL));
		if (i < array->ft->getSize(array, NULL)-1)
			result += ",";
	}
	return result;
}

string CT_CMPIDataToString(const CMPIData& data) {
	
	if ((data.state & CMPI_goodValue) != CMPI_goodValue && (data.state & CMPI_keyValue) != CMPI_keyValue)
		return "null";
	
	switch (data.type) {
		case (CMPI_uint8):
			return CF_intToStr((unsigned char) data.value.uint8);
		case (CMPI_uint16):
			return CF_intToStr((unsigned short) data.value.uint16);
		case (CMPI_uint32):
			return CF_intToStr((unsigned long) data.value.uint32);
		case (CMPI_uint64):
			return CF_intToStr((unsigned long long) data.value.uint64);
		case (CMPI_sint8):
			return CF_intToStr((char) data.value.sint8);
		case (CMPI_sint16):
			return CF_intToStr((short) data.value.sint16);
		case (CMPI_sint32):
			return CF_intToStr((long) data.value.sint32);
		case (CMPI_sint64):
			return CF_intToStr((long long) data.value.sint64);
		case (CMPI_real32):
			return CF_intToStr((float) data.value.real32);
		case (CMPI_real64):
			return CF_intToStr((double) data.value.real64);
		case (CMPI_boolean):
			return CF_boolToStr((bool) data.value.boolean);
		case (CMPI_char16):
			return CF_intToStr((unsigned short) data.value.char16);
		case (CMPI_string):
			return "\"" + CF_quoteString(CMGetCharPtr(data.value.string)) + "\"";
		case (CMPI_dateTime):
			return CMGetCharPtr(data.value.dateTime->ft->getStringFormat(data.value.dateTime, NULL));
		case (CMPI_ref):
			return CT_CMPIObjectPathToString(data.value.ref);
		case (CMPI_ARRAY):
			return CT_CMPIArrayToString(data.value.array);
		default:
			return "unkown";
	}
	
}

//uint8
CMPIValue CT_toCMPI(const unsigned char& value) {
	CMPIValue myValue;
	myValue.uint8 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, unsigned char& value) {
	value = data.value.uint8;
	return OK;
}

//uint16
CMPIValue CT_toCMPI(const unsigned short& value) {
	CMPIValue myValue;
	myValue.uint16 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, unsigned short& value) {
	value = data.value.uint16;
	return OK;
}

//uint32
CMPIValue CT_toCMPI(const unsigned int& value) {
	CMPIValue myValue;
	myValue.uint32 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, unsigned int& value) {
	value = data.value.uint32;
	return OK;
}

//uint32 deprecated
CMPIValue CT_toCMPI(const unsigned long& value) {
	CMPIValue myValue;
	myValue.uint32 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, unsigned long& value) {
	value = data.value.uint32;
	return OK;
}

//uint64
CMPIValue CT_toCMPI(const unsigned long long& value) {
	CMPIValue myValue;
	myValue.uint64 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, unsigned long long& value) {
	value = data.value.uint64;
	return OK;
}

//sint8
CMPIValue CT_toCMPI(const signed char& value) {
	CMPIValue myValue;
	myValue.sint8 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, signed char& value) {
	value = data.value.sint8;
	return OK;
}

//sint16
CMPIValue CT_toCMPI(const short& value) {
	CMPIValue myValue;
	myValue.sint16 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, short& value) {
	value = data.value.sint16;
	return OK;
}

//sint32
CMPIValue CT_toCMPI(const int& value) {
	CMPIValue myValue;
	myValue.sint32 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, signed int& value) {
	value = data.value.sint32;
	return OK;
}

//sint32 deprecated
CMPIValue CT_toCMPI(const long& value) {
	CMPIValue myValue;
	myValue.sint32 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, signed long& value) {
	value = data.value.sint32;
	return OK;
}

//sint64
CMPIValue CT_toCMPI(const long long& value) {
	CMPIValue myValue;
	myValue.sint64 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, signed long long& value) {
	value = data.value.sint64;
	return OK;
}

//real32
CMPIValue CT_toCMPI(const float& value) {
	CMPIValue myValue;
	myValue.real32 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, float& value) {
	value = data.value.real32;
	return OK;
}

//real64
CMPIValue CT_toCMPI(const double& value) {
	CMPIValue myValue;
	myValue.real64 = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, double& value) {
	value = data.value.real64;
	return OK;
}

//char16
CMPIValue CT_toCMPIChar16(const unsigned short & value) {
	CMPIValue myValue;
	myValue.char16 = value;
	return myValue;
}
int CT_ToCChar16(const CMPIData& data, unsigned short& value) {
	value = data.value.char16;
	return OK;
}

//boolean
CMPIValue CT_toCMPI(const bool& value) {
	CMPIValue myValue;
	myValue.boolean = value;
	return myValue;
}
int CT_ToC(const CMPIData& data, bool& value) {
	value = data.value.boolean;
	return OK;
}

//string
CMPIValue CT_toCMPI(const CMPIBroker* broker, const string& value) {
	CMPIValue myValue;
	myValue.string = broker->eft->newString(broker, value.c_str(), NULL);
	return myValue;
}
int CT_ToC(const CMPIData& data, string& value) {
	if (data.value.string == NULL)
		return FAILED;
	value = CMGetCharPtr(data.value.string);
	return OK;
}

//datetime
CMPIValue CT_toCMPIDatetime(const CMPIBroker* broker, const string& value) {
	CMPIValue myValue;
	myValue.dateTime = broker->eft->newDateTimeFromChars(broker, value.c_str(), NULL);
	return myValue;
}
int CT_ToCDatetime(const CMPIData& data, string& value) {
	CMPIDateTime* myTime = data.value.dateTime;
	value = CMGetCharPtr(myTime->ft->getStringFormat(myTime, NULL));
	return OK;
}

//REF
CMPIValue CT_toCMPI(const Objectpath& value) {
	CMPIValue myValue;
	myValue.ref = ((Objectpath&) value).getHdl();
	return myValue;
}
int CT_ToC(const CMPIBroker* broker, const CMPIData& data, Objectpath& value) {
	CMPIObjectPath* myPath = data.value.ref;
	value = Objectpath((const CMPIBroker*) broker, myPath);
	return OK;
}

//uint8[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<unsigned char>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_uint8, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.uint8 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_uint8);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<unsigned char>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.uint8);
	return OK;
}

//uint16[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<unsigned short>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_uint16, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.uint16 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_uint16);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<unsigned short>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.uint16);
	return OK;
}

//uint32[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<unsigned int>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_uint32, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.uint32 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_uint32);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<unsigned int>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.uint32);
	return OK;
}

//uint32[] deprecated
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<unsigned long>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_uint32, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.uint32 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_uint32);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<unsigned long>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.uint32);
	return OK;
}

//uint64[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<unsigned long long>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_uint64, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.uint64 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_uint64);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<unsigned long long>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.uint64);
	return OK;
}

//sint8[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<signed char>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_sint8, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.sint8 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_sint8);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<signed char>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.sint8);
	return OK;
}

//sint16[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<short>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_sint16, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.sint16 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_sint16);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<short>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.sint16);
	return OK;
}

//sint32[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<int>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_sint32, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.sint32 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_sint32);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<int>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.sint32);
	return OK;
}

//sint32[] deprecated
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<long>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_sint32, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.sint32 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_sint32);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<long>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.sint32);
	return OK;
}

//sint64[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<long long>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_sint64, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.sint64 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_sint64);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<long long>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.sint64);
	return OK;
}

//real32[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<float>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_real32, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.real32 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_real32);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<float>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.real32);
	return OK;
}

//real64[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<double>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_real64, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.real64 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_real64);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<double>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.real64);
	return OK;
}

//char16[]
CMPIValue CT_toCMPIChar16(const CMPIBroker* broker, const vector<unsigned short>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_char16, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.char16 = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_char16);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToCChar16(const CMPIData& data, vector<unsigned short>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.char16);
	return OK;
}

//boolean[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<bool>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_boolean, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.boolean = value[i];
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_boolean);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<bool>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(myArray->ft->getElementAt(myArray, i, NULL).value.boolean);
	return OK;
}

//string[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<string>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_string, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.string = broker->eft->newString(broker, value[i].c_str(), NULL);
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_string);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIData& data, vector<string>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++)
		value.push_back(CMGetCharPtr(myArray->ft->getElementAt(myArray, i, NULL).value.string));
	return OK;
}

//datetime[]
CMPIValue CT_toCMPIDatetime(const CMPIBroker* broker, const vector<string>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_dateTime, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.dateTime = broker->eft->newDateTimeFromChars(broker, value[i].c_str(), NULL);
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_dateTime);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToCDatetime(const CMPIData& data, vector<string>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++) {
		CMPIDateTime* myTime = myArray->ft->getElementAt(myArray, i, NULL).value.dateTime;
		value.push_back(CMGetCharPtr(myTime->ft->getStringFormat(myTime, NULL)));
	}
	return OK;
}

//REF[]
CMPIValue CT_toCMPI(const CMPIBroker* broker, const vector<Objectpath>& value) {
	CMPIArray* myArray;
	CMPIValue myValue;
	myArray = broker->eft->newArray(broker, value.size(), CMPI_ref, NULL);
	for (unsigned int i = 0; i < value.size(); i++) {
		myValue.ref = ((Objectpath&) value[i]).getHdl();
		myArray->ft->setElementAt(myArray, i, &myValue, CMPI_ref);
	}
	myValue.array = myArray;
	return myValue;
}
int CT_ToC(const CMPIBroker* broker, const CMPIData& data, vector<Objectpath>& value) {
	CMPIArray* myArray;
	myArray = data.value.array;
	if (myArray == NULL) return FAILED;
	for (unsigned int i = 0; i < myArray->ft->getSize(myArray, NULL); i++) {
		CMPIObjectPath* myPath = myArray->ft->getElementAt(myArray, i, NULL).value.ref;
		value.push_back(Objectpath((const CMPIBroker*) broker, myPath));
	}
	return OK;
}
